from __future__ import (
    absolute_import,
    division,
    print_function,
)

from unittest import TestCase

from pcs import settings
from pcs.common import report_codes
from pcs.lib.commands import stonith
from pcs.test.tools import fixture
from pcs.test.tools.command_env import get_env_tools
from pcs.test.tools.pcs_unittest import mock


crm_mon_rng_with_history = """\
<?xml version="1.0" encoding="UTF-8"?>
<grammar xmlns="http://relaxng.org/ns/structure/1.0" 
         datatypeLibrary="http://www.w3.org/2001/XMLSchema-datatypes">
    <start>
        <ref name="element-crm_mon"/>
    </start>
    <define name="element-crm_mon">
        <element name="crm_mon">
            <optional>
                <element name="fence_history">
                </element>
            </optional>
        </element>
    </define>
</grammar>
"""

crm_mon_rng_without_history = """\
<?xml version="1.0" encoding="UTF-8"?>
<grammar xmlns="http://relaxng.org/ns/structure/1.0" 
         datatypeLibrary="http://www.w3.org/2001/XMLSchema-datatypes">
    <start>
        <ref name="element-crm_mon"/>
    </start>
    <define name="element-crm_mon">
        <element name="crm_mon">
        </element>
    </define>
</grammar>
"""

class HistoryGetText(TestCase):
    def setUp(self):
        self.env_assist, self.config = get_env_tools(test_case=self)
        self.config.fs.open(
            settings.crm_mon_schema,
            mock.mock_open(read_data=crm_mon_rng_with_history)(),
            name="fs.open.crm_mon_rng"
        )

    def test_success_all_nodes(self):
        history = (
            "rh76-node1 was able to reboot node rh76-node2 on behalf of "
            "crmd.835 from rh76-node1 at Tue Jul 16 09:56:47 2019"
            "\n"
            "rh76-node2 was able to reboot node rh76-node1 on behalf of "
            "crmd.835 from rh76-node2 at Tue Jul 16 09:58:47 2019"
            "\n"
        )
        self.config.runner.pcmk.fence_history_get(stdout=history, node="*")
        output = stonith.history_get_text(self.env_assist.get_env())
        self.assertEqual(output, history.strip())

    def test_success_one_node(self):
        history = (
            "rh76-node1 was able to reboot node rh76-node2 on behalf of "
            "crmd.835 from rh76-node1 at Tue Jul 16 09:56:47 2019"
            "\n"
            "\n"
        )
        node = "rh76-node2"
        self.config.runner.pcmk.fence_history_get(stdout=history, node=node)
        output = stonith.history_get_text(self.env_assist.get_env(), node)
        self.assertEqual(output, history.strip())

    def test_error(self):
        stdout = "some output\n"
        stderr = "some error\n"
        self.config.runner.pcmk.fence_history_get(
            stdout=stdout,
            stderr=stderr,
            returncode=1,
            node="*"
        )
        self.env_assist.assert_raise_library_error(
            lambda: stonith.history_get_text(self.env_assist.get_env()),
            [
                fixture.error(
                    report_codes.FENCE_HISTORY_COMMAND_ERROR,
                    command_label="show",
                    reason=(stderr + stdout).strip(),
                )
            ],
            expected_in_processor=False
        )

    def test_history_not_supported(self):
        self.config.fs.open(
            settings.crm_mon_schema,
            mock.mock_open(read_data=crm_mon_rng_without_history)(),
            name="fs.open.crm_mon_rng",
            instead="fs.open.crm_mon_rng"
        )
        self.env_assist.assert_raise_library_error(
            lambda: stonith.history_get_text(self.env_assist.get_env()),
            [
                fixture.error(
                    report_codes.FENCE_HISTORY_NOT_SUPPORTED,
                )
            ],
            expected_in_processor=False
        )


class HistoryCleanup(TestCase):
    def setUp(self):
        self.env_assist, self.config = get_env_tools(test_case=self)
        self.config.fs.open(
            settings.crm_mon_schema,
            mock.mock_open(read_data=crm_mon_rng_with_history)(),
            name="fs.open.crm_mon_rng"
        )

    def test_success_all_nodes(self):
        msg = "cleaning up fencing-history for node *\n"
        self.config.runner.pcmk.fence_history_cleanup(stdout=msg, node="*")
        output = stonith.history_cleanup(self.env_assist.get_env())
        self.assertEqual(output, msg.strip())

    def test_success_one_node(self):
        node = "rh76-node2"
        msg = "cleaning up fencing-history for node {node}\n".format(node=node)
        self.config.runner.pcmk.fence_history_cleanup(stdout=msg, node=node)
        output = stonith.history_cleanup(self.env_assist.get_env(), node)
        self.assertEqual(output, msg.strip())

    def test_error(self):
        stdout = "some output\n"
        stderr = "some error\n"
        self.config.runner.pcmk.fence_history_cleanup(
            stdout=stdout,
            stderr=stderr,
            returncode=1,
            node="*"
        )
        self.env_assist.assert_raise_library_error(
            lambda: stonith.history_cleanup(self.env_assist.get_env()),
            [
                fixture.error(
                    report_codes.FENCE_HISTORY_COMMAND_ERROR,
                    command_label="cleanup",
                    reason=(stderr + stdout).strip(),
                )
            ],
            expected_in_processor=False
        )

    def test_history_not_supported(self):
        self.config.fs.open(
            settings.crm_mon_schema,
            mock.mock_open(read_data=crm_mon_rng_without_history)(),
            name="fs.open.crm_mon_rng",
            instead="fs.open.crm_mon_rng"
        )
        self.env_assist.assert_raise_library_error(
            lambda: stonith.history_cleanup(self.env_assist.get_env()),
            [
                fixture.error(
                    report_codes.FENCE_HISTORY_NOT_SUPPORTED,
                )
            ],
            expected_in_processor=False
        )


class HistoryUpdate(TestCase):
    def setUp(self):
        self.env_assist, self.config = get_env_tools(test_case=self)
        self.config.fs.open(
            settings.crm_mon_schema,
            mock.mock_open(read_data=crm_mon_rng_with_history)(),
            name="fs.open.crm_mon_rng"
        )

    def test_success_all_nodes(self):
        msg = "gather fencing-history from all nodes\n"
        self.config.runner.pcmk.fence_history_update(stdout=msg)
        output = stonith.history_update(self.env_assist.get_env())
        self.assertEqual(output, msg.strip())

    def test_error(self):
        stdout = "some output\n"
        stderr = "some error\n"
        self.config.runner.pcmk.fence_history_update(
            stdout=stdout,
            stderr=stderr,
            returncode=1,
        )
        self.env_assist.assert_raise_library_error(
            lambda: stonith.history_update(self.env_assist.get_env()),
            [
                fixture.error(
                    report_codes.FENCE_HISTORY_COMMAND_ERROR,
                    command_label="update",
                    reason=(stderr + stdout).strip(),
                )
            ],
            expected_in_processor=False
        )

    def test_history_not_supported(self):
        self.config.fs.open(
            settings.crm_mon_schema,
            mock.mock_open(read_data=crm_mon_rng_without_history)(),
            name="fs.open.crm_mon_rng",
            instead="fs.open.crm_mon_rng"
        )
        self.env_assist.assert_raise_library_error(
            lambda: stonith.history_update(self.env_assist.get_env()),
            [
                fixture.error(
                    report_codes.FENCE_HISTORY_NOT_SUPPORTED,
                )
            ],
            expected_in_processor=False
        )
