/*
 * Copyright (c) 2013 Red Hat Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions of source code must retain the above
 *       copyright notice, this list of conditions and the
 *       following disclaimer.
 *     * Redistributions in binary form must reproduce the
 *       above copyright notice, this list of conditions and
 *       the following disclaimer in the documentation and/or
 *       other materials provided with the distribution.
 *     * The names of contributors to this software may not be
 *       used to endorse or promote products derived from this
 *       software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * Author: Stef Walter <stefw@redhat.com>
 */

#include "config.h"
#include "CuTest.h"

#include "base64.h"
#include "debug.h"
#include "message.h"

#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

static void
check_decode_msg (CuTest *tc,
                  const char *file,
                  int line,
                  const char *input,
                  ssize_t input_len,
                  const unsigned char *expected,
                  ssize_t expected_len)
{
	unsigned char decoded[8192];
	int length;

	if (input_len < 0)
		input_len = strlen (input);
	if (expected_len < 0)
		expected_len = strlen ((char *)expected);
	length = p11_b64_pton (input, input_len, decoded, sizeof (decoded));

	if (expected == NULL) {
		CuAssert_Line (tc, file, line, "decoding should have failed", length < 0);

	} else {
		CuAssert_Line (tc, file, line, "decoding failed", length >= 0);
		CuAssertIntEquals_LineMsg (tc, file, line, "wrong length", expected_len, length);
		CuAssert_Line (tc, file, line, "decoded wrong", memcmp (decoded, expected, length) == 0);
	}
}

#define check_decode_success(tc, input, input_len, expected, expected_len) \
	check_decode_msg (tc, __FILE__, __LINE__, input, input_len, expected, expected_len)

#define check_decode_failure(tc, input, input_len) \
	check_decode_msg (tc, __FILE__, __LINE__, input, input_len, NULL, 0)

static void
test_decode_simple (CuTest *tc)
{
	check_decode_success (tc, "", 0, (unsigned char *)"", 0);
	check_decode_success (tc, "MQ==", 0, (unsigned char *)"1", 0);
	check_decode_success (tc, "YmxhaAo=", -1, (unsigned char *)"blah\n", -1);
	check_decode_success (tc, "bGVlbGEK", -1, (unsigned char *)"leela\n", -1);
	check_decode_success (tc, "bGVlbG9vCg==", -1, (unsigned char *)"leeloo\n", -1);
}

static void
test_decode_thawte (CuTest *tc)
{
	const char *input =
		"MIIEKjCCAxKgAwIBAgIQYAGXt0an6rS0mtZLL/eQ+zANBgkqhkiG9w0BAQsFADCB"
		"rjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYGA1UECxMf"
		"Q2VydGlmaWNhdGlvbiBTZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UECxMvKGMpIDIw"
		"MDggdGhhd3RlLCBJbmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNlIG9ubHkxJDAiBgNV"
		"BAMTG3RoYXd0ZSBQcmltYXJ5IFJvb3QgQ0EgLSBHMzAeFw0wODA0MDIwMDAwMDBa"
		"Fw0zNzEyMDEyMzU5NTlaMIGuMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMdGhhd3Rl"
		"LCBJbmMuMSgwJgYDVQQLEx9DZXJ0aWZpY2F0aW9uIFNlcnZpY2VzIERpdmlzaW9u"
		"MTgwNgYDVQQLEy8oYykgMjAwOCB0aGF3dGUsIEluYy4gLSBGb3IgYXV0aG9yaXpl"
		"ZCB1c2Ugb25seTEkMCIGA1UEAxMbdGhhd3RlIFByaW1hcnkgUm9vdCBDQSAtIEcz"
		"MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsr8nLPvb2FvdeHsbnndm"
		"gcs+vHyu86YnmjSjaDFxODNi5PNxZnmxqWWjpYvVj2AtP0LMqmsywCPLLEHd5N/8"
		"YZzic7IilRFDGF/Eth9XbAoFWCLINkw6fKXRz4aviKdEAhN0cXMKQlkC+BsUa0Lf"
		"b1+6a4KinVvnSr0eAXLbS3ToO39/fR8EtCab4LRarEc9VbjXsCZSKAExQGbY2SS9"
		"9irY7CFJXJv2eul/VTV+lmuNk5Mny5K76qxAwJ/C+IDPXfRa3M50hqY+bAtTyr2S"
		"zhkGcuYMXDhpxwTWvGzOW/b3aJzcJRVIiKHpqfiYnODz1TEoYRFsZ5aNOZnLwkUk"
		"OQIDAQABo0IwQDAPBgNVHRMBAf8EBTADAQH/MA4GA1UdDwEB/wQEAwIBBjAdBgNV"
		"HQ4EFgQUrWyqlGCc7eT/+j4KdCtjA/e2Wb8wDQYJKoZIhvcNAQELBQADggEBABpA"
		"2JVlrAmSicY59BDlqQ5mU1143vokkbvnRFHfxhY0Cu9qRFHqKweKA3rD6z8KLFIW"
		"oCtDuSWQP3CpMyVtRRooOyfPqsMpQhvfO0zAMzRbQYi/aytlryjvsvXDqmbOe1bu"
		"t8jLZ8HJnBoYuMTDSQPxYA5QzUbF83d597YV4Djbxy8ooAw/dyZ02SUS2jHaGh7c"
		"KUGRIjxpp7sC8rZcJwOJ9Abqm+RyguOhCcHpABnTPtRwa7pxpqpYrvS76Wy274fM"
		"m7v/OeZWYdMKp8RcTGB7BXcmer/YB1IsYvdwY9k5vG8cwnncdimvzsUsZAReiDZu"
		"MdRAGmI0Nj81Aa6sY6A=";

	const unsigned char output[] = {
		0x30, 0x82, 0x04, 0x2a, 0x30, 0x82, 0x03, 0x12, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x60,
		0x01, 0x97, 0xb7, 0x46, 0xa7, 0xea, 0xb4, 0xb4, 0x9a, 0xd6, 0x4b, 0x2f, 0xf7, 0x90, 0xfb, 0x30,
		0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x81,
		0xae, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x15,
		0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0c, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65, 0x2c,
		0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x28, 0x30, 0x26, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x1f,
		0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x53, 0x65,
		0x72, 0x76, 0x69, 0x63, 0x65, 0x73, 0x20, 0x44, 0x69, 0x76, 0x69, 0x73, 0x69, 0x6f, 0x6e, 0x31,
		0x38, 0x30, 0x36, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x2f, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30,
		0x30, 0x38, 0x20, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x20,
		0x2d, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65, 0x64,
		0x20, 0x75, 0x73, 0x65, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55,
		0x04, 0x03, 0x13, 0x1b, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61,
		0x72, 0x79, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x30,
		0x1e, 0x17, 0x0d, 0x30, 0x38, 0x30, 0x34, 0x30, 0x32, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a,
		0x17, 0x0d, 0x33, 0x37, 0x31, 0x32, 0x30, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30,
		0x81, 0xae, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31,
		0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0c, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65,
		0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x28, 0x30, 0x26, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
		0x1f, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x53,
		0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73, 0x20, 0x44, 0x69, 0x76, 0x69, 0x73, 0x69, 0x6f, 0x6e,
		0x31, 0x38, 0x30, 0x36, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x2f, 0x28, 0x63, 0x29, 0x20, 0x32,
		0x30, 0x30, 0x38, 0x20, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e,
		0x20, 0x2d, 0x20, 0x46, 0x6f, 0x72, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65,
		0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03,
		0x55, 0x04, 0x03, 0x13, 0x1b, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65, 0x20, 0x50, 0x72, 0x69, 0x6d,
		0x61, 0x72, 0x79, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33,
		0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01,
		0x00, 0xb2, 0xbf, 0x27, 0x2c, 0xfb, 0xdb, 0xd8, 0x5b, 0xdd, 0x78, 0x7b, 0x1b, 0x9e, 0x77, 0x66,
		0x81, 0xcb, 0x3e, 0xbc, 0x7c, 0xae, 0xf3, 0xa6, 0x27, 0x9a, 0x34, 0xa3, 0x68, 0x31, 0x71, 0x38,
		0x33, 0x62, 0xe4, 0xf3, 0x71, 0x66, 0x79, 0xb1, 0xa9, 0x65, 0xa3, 0xa5, 0x8b, 0xd5, 0x8f, 0x60,
		0x2d, 0x3f, 0x42, 0xcc, 0xaa, 0x6b, 0x32, 0xc0, 0x23, 0xcb, 0x2c, 0x41, 0xdd, 0xe4, 0xdf, 0xfc,
		0x61, 0x9c, 0xe2, 0x73, 0xb2, 0x22, 0x95, 0x11, 0x43, 0x18, 0x5f, 0xc4, 0xb6, 0x1f, 0x57, 0x6c,
		0x0a, 0x05, 0x58, 0x22, 0xc8, 0x36, 0x4c, 0x3a, 0x7c, 0xa5, 0xd1, 0xcf, 0x86, 0xaf, 0x88, 0xa7,
		0x44, 0x02, 0x13, 0x74, 0x71, 0x73, 0x0a, 0x42, 0x59, 0x02, 0xf8, 0x1b, 0x14, 0x6b, 0x42, 0xdf,
		0x6f, 0x5f, 0xba, 0x6b, 0x82, 0xa2, 0x9d, 0x5b, 0xe7, 0x4a, 0xbd, 0x1e, 0x01, 0x72, 0xdb, 0x4b,
		0x74, 0xe8, 0x3b, 0x7f, 0x7f, 0x7d, 0x1f, 0x04, 0xb4, 0x26, 0x9b, 0xe0, 0xb4, 0x5a, 0xac, 0x47,
		0x3d, 0x55, 0xb8, 0xd7, 0xb0, 0x26, 0x52, 0x28, 0x01, 0x31, 0x40, 0x66, 0xd8, 0xd9, 0x24, 0xbd,
		0xf6, 0x2a, 0xd8, 0xec, 0x21, 0x49, 0x5c, 0x9b, 0xf6, 0x7a, 0xe9, 0x7f, 0x55, 0x35, 0x7e, 0x96,
		0x6b, 0x8d, 0x93, 0x93, 0x27, 0xcb, 0x92, 0xbb, 0xea, 0xac, 0x40, 0xc0, 0x9f, 0xc2, 0xf8, 0x80,
		0xcf, 0x5d, 0xf4, 0x5a, 0xdc, 0xce, 0x74, 0x86, 0xa6, 0x3e, 0x6c, 0x0b, 0x53, 0xca, 0xbd, 0x92,
		0xce, 0x19, 0x06, 0x72, 0xe6, 0x0c, 0x5c, 0x38, 0x69, 0xc7, 0x04, 0xd6, 0xbc, 0x6c, 0xce, 0x5b,
		0xf6, 0xf7, 0x68, 0x9c, 0xdc, 0x25, 0x15, 0x48, 0x88, 0xa1, 0xe9, 0xa9, 0xf8, 0x98, 0x9c, 0xe0,
		0xf3, 0xd5, 0x31, 0x28, 0x61, 0x11, 0x6c, 0x67, 0x96, 0x8d, 0x39, 0x99, 0xcb, 0xc2, 0x45, 0x24,
		0x39, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x42, 0x30, 0x40, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d,
		0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0e, 0x06, 0x03, 0x55,
		0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55,
		0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xad, 0x6c, 0xaa, 0x94, 0x60, 0x9c, 0xed, 0xe4, 0xff, 0xfa,
		0x3e, 0x0a, 0x74, 0x2b, 0x63, 0x03, 0xf7, 0xb6, 0x59, 0xbf, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
		0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x1a, 0x40,
		0xd8, 0x95, 0x65, 0xac, 0x09, 0x92, 0x89, 0xc6, 0x39, 0xf4, 0x10, 0xe5, 0xa9, 0x0e, 0x66, 0x53,
		0x5d, 0x78, 0xde, 0xfa, 0x24, 0x91, 0xbb, 0xe7, 0x44, 0x51, 0xdf, 0xc6, 0x16, 0x34, 0x0a, 0xef,
		0x6a, 0x44, 0x51, 0xea, 0x2b, 0x07, 0x8a, 0x03, 0x7a, 0xc3, 0xeb, 0x3f, 0x0a, 0x2c, 0x52, 0x16,
		0xa0, 0x2b, 0x43, 0xb9, 0x25, 0x90, 0x3f, 0x70, 0xa9, 0x33, 0x25, 0x6d, 0x45, 0x1a, 0x28, 0x3b,
		0x27, 0xcf, 0xaa, 0xc3, 0x29, 0x42, 0x1b, 0xdf, 0x3b, 0x4c, 0xc0, 0x33, 0x34, 0x5b, 0x41, 0x88,
		0xbf, 0x6b, 0x2b, 0x65, 0xaf, 0x28, 0xef, 0xb2, 0xf5, 0xc3, 0xaa, 0x66, 0xce, 0x7b, 0x56, 0xee,
		0xb7, 0xc8, 0xcb, 0x67, 0xc1, 0xc9, 0x9c, 0x1a, 0x18, 0xb8, 0xc4, 0xc3, 0x49, 0x03, 0xf1, 0x60,
		0x0e, 0x50, 0xcd, 0x46, 0xc5, 0xf3, 0x77, 0x79, 0xf7, 0xb6, 0x15, 0xe0, 0x38, 0xdb, 0xc7, 0x2f,
		0x28, 0xa0, 0x0c, 0x3f, 0x77, 0x26, 0x74, 0xd9, 0x25, 0x12, 0xda, 0x31, 0xda, 0x1a, 0x1e, 0xdc,
		0x29, 0x41, 0x91, 0x22, 0x3c, 0x69, 0xa7, 0xbb, 0x02, 0xf2, 0xb6, 0x5c, 0x27, 0x03, 0x89, 0xf4,
		0x06, 0xea, 0x9b, 0xe4, 0x72, 0x82, 0xe3, 0xa1, 0x09, 0xc1, 0xe9, 0x00, 0x19, 0xd3, 0x3e, 0xd4,
		0x70, 0x6b, 0xba, 0x71, 0xa6, 0xaa, 0x58, 0xae, 0xf4, 0xbb, 0xe9, 0x6c, 0xb6, 0xef, 0x87, 0xcc,
		0x9b, 0xbb, 0xff, 0x39, 0xe6, 0x56, 0x61, 0xd3, 0x0a, 0xa7, 0xc4, 0x5c, 0x4c, 0x60, 0x7b, 0x05,
		0x77, 0x26, 0x7a, 0xbf, 0xd8, 0x07, 0x52, 0x2c, 0x62, 0xf7, 0x70, 0x63, 0xd9, 0x39, 0xbc, 0x6f,
		0x1c, 0xc2, 0x79, 0xdc, 0x76, 0x29, 0xaf, 0xce, 0xc5, 0x2c, 0x64, 0x04, 0x5e, 0x88, 0x36, 0x6e,
		0x31, 0xd4, 0x40, 0x1a, 0x62, 0x34, 0x36, 0x3f, 0x35, 0x01, 0xae, 0xac, 0x63, 0xa0,
	};

	check_decode_success (tc, input, -1, output, sizeof (output));
}

int
main (void)
{
	CuString *output = CuStringNew ();
	CuSuite* suite = CuSuiteNew ();
	int ret;

	putenv ("P11_KIT_STRICT=1");
	p11_debug_init ();

	SUITE_ADD_TEST (suite, test_decode_simple);
	SUITE_ADD_TEST (suite, test_decode_thawte);

	CuSuiteRun (suite);
	CuSuiteSummary (suite, output);
	CuSuiteDetails (suite, output);
	printf ("%s\n", output->buffer);
	ret = suite->failCount;
	CuSuiteDelete (suite);
	CuStringDelete (output);
	return ret;
}
