/*
 * Copyright 2015-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j;

/**
 * {@code AssertionFailedError} is a common base class for test-related
 * {@link AssertionError AssertionErrors}.
 *
 * <p>In addition to a message and a cause, this class stores the expected
 * and actual values of an assertion using the {@link ValueWrapper} type.
 *
 * @author Sam Brannen
 * @author Marc Philipp
 * @since 1.0
 */
public class AssertionFailedError extends AssertionError {

	private static final long serialVersionUID = 1L;

	private final ValueWrapper expected;
	private final ValueWrapper actual;

	/**
	 * Constructs an {@code AssertionFailedError} with an empty message, no
	 * cause, and no expected/actual values.
	 */
	public AssertionFailedError() {
		this(null);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message, no cause,
	 * and no expected/actual values.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String} because the {@link AssertionError}
	 * superclass does not support {@code null} messages.
	 */
	public AssertionFailedError(String message) {
		this(message, null);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message and
	 * expected/actual values but without a cause.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String} because the {@link AssertionError}
	 * superclass does not support {@code null} messages.
	 */
	public AssertionFailedError(String message, Object expected, Object actual) {
		this(message, expected, actual, null);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message and a cause
	 * but without expected/actual values.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String} because the {@link AssertionError}
	 * superclass does not support {@code null} messages.
	 */
	public AssertionFailedError(String message, Throwable cause) {
		this(message, null, null, cause);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message,
	 * expected/actual values, and a cause.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String} because the {@link AssertionError}
	 * superclass does not support {@code null} messages.
	 */
	public AssertionFailedError(String message, Object expected, Object actual, Throwable cause) {
		this(message, ValueWrapper.create(expected), ValueWrapper.create(actual), cause);
	}

	private AssertionFailedError(String message, ValueWrapper expected, ValueWrapper actual, Throwable cause) {
		super((message == null || message.trim().length() == 0) ? "" : message);
		this.expected = expected;
		this.actual = actual;
		initCause(cause);
	}

	/**
	 * Returns {@code true} if the expected value is defined, i.e. was passed
	 * to the constructor.
	 *
	 * @see #getExpected()
	 */
	public boolean isExpectedDefined() {
		return this.expected != null;
	}

	/**
	 * Returns {@code true} if the actual value is defined, i.e. was passed
	 * to the constructor.
	 *
	 * @see #getActual()
	 */
	public boolean isActualDefined() {
		return this.actual != null;
	}

	/**
	 * Returns the wrapped expected value if it is defined; otherwise {@code null}.
	 *
	 * @see #isExpectedDefined()
	 */
	public ValueWrapper getExpected() {
		return this.expected;
	}

	/**
	 * Returns the wrapped actual value if it is defined; otherwise {@code null}.
	 *
	 * @see #isActualDefined()
	 */
	public ValueWrapper getActual() {
		return this.actual;
	}

	/**
	 * Returns a short description of this assertion error using the same format
	 * as {@link Throwable#toString()}.
	 *
	 * @return Since the {@link AssertionError} superclass does not support
	 * {@code null} messages and the constructors of this class convert supplied
	 * {@code null} or blank messages to the empty {@code String}, this method
	 * only includes non-empty messages in its return value.
	 */
	@Override
	public String toString() {
		String className = getClass().getName();
		String message = getLocalizedMessage();
		return ("".equals(message) ? className : (className + ": " + message));
	}

}
