/*
 * COPYRIGHT (c) International Business Machines Corp. 2005-2017
 *
 * This program is provided under the terms of the Common Public License,
 * version 1.0 (CPL-1.0). Any use, reproduction or distribution for this
 * software constitutes recipient's acceptance of CPL-1.0 terms which can be
 * found in the file LICENSE file or at
 * https://opensource.org/licenses/cpl1.0.php
 */

#ifndef _DES3_H_
#define _DES3_H_
#endif

#include "pkcs11types.h"

#define MAX_KEY_SIZE  64
#define MAX_TEXT_SIZE 128
#define MAX_IV_SIZE  64
#define DES3_BLOCK_SIZE	8
#define DES3_KEY_SIZE 24
#define DES3_IV_SIZE 8
#define MAX_CHUNKS 8

struct des3_test_vector {
    unsigned char key[MAX_KEY_SIZE];
    unsigned char klen;
    unsigned char iv[MAX_IV_SIZE];
    unsigned char ivlen;
    unsigned char plaintext[MAX_TEXT_SIZE];
    unsigned char plen;
    unsigned char ciphertext[MAX_TEXT_SIZE];
    unsigned char clen;
    int chunks[MAX_CHUNKS];
    int num_chunks;
};

struct published_test_suite_info {
    const char *name;
    unsigned int tvcount;
    struct des3_test_vector *tv;
    unsigned int size;
    unsigned long mechanism;
};

struct generated_test_suite_info {
    const char *name;
    CK_MECHANISM mech;
};

struct CK_MECHANISM des3_keygen = {
    .mechanism = CKM_DES3_KEY_GEN,
    .ulParameterLen = 0,
    .pParameter = NULL,
};

struct cmac_test_vector {
    unsigned char key[MAX_KEY_SIZE];
    unsigned char klen;
    unsigned char msg[MAX_TEXT_SIZE];
    unsigned char mlen;
    unsigned char mac[MAX_KEY_SIZE];
    unsigned char tlen;
    int chunks_msg[MAX_CHUNKS];
    int num_chunks_message;
};

struct published_cmac_test_suite_info {
    const char *name;
    unsigned int tvcount;
    struct cmac_test_vector *tv;
    CK_MECHANISM mech;
    CK_KEY_TYPE key_type;
};

unsigned char des3_cbc_iv[] = {0x67, 0x9f, 0xdb, 0xee, 0x16, 0x6c, 0x2e, 0x0a};

/** http://csrc.nist.gov/groups/STM/cavp/documents/des/tdesmmt.zip
    TECBMMT1.rsp
**/
static struct des3_test_vector des3_ecb_tv[] = {
    {                   // #0
        .key = {0xe9, 0x7c, 0x83, 0x13, 0xba, 0x26, 0x5d, 0x43, 0x25,
                0x4c, 0xbf, 0x9e, 0x8f, 0x7c, 0x2a, 0xa8, 0xa7, 0x54,
                0xd6, 0x5e, 0x8a, 0xe9, 0x97, 0xe3},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x46, 0x56, 0xed, 0x81, 0xa8, 0xbc, 0x58, 0xa9},
        .plen = 8,
        .ciphertext = {0x42, 0x6a, 0x0a, 0xc2, 0x65, 0x86, 0xbf, 0x6c},
        .clen = 8,
    }, {                // #1
        .key = {0x86, 0x6d, 0x38, 0x94, 0xb3, 0x67, 0x0e, 0xfe, 0x37,
                0x54, 0xa2, 0xe5, 0x0e, 0x76, 0x7f, 0xcd, 0x34, 0x51,
                0xfb, 0x68, 0x89, 0xda, 0xfe, 0xb5},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x50, 0xea, 0xab, 0x2a, 0x14, 0x14, 0x07, 0xc4, 0x7d,
                      0x71, 0x6b, 0x70, 0x36, 0x50, 0xf3, 0xa8},
        .plen = 16,
        .ciphertext = {0xb7, 0x84, 0xaf, 0xf1, 0x1d, 0x71, 0x5d, 0x60, 0x18,
                       0x5d, 0x11, 0x1a, 0x29, 0x10, 0xeb, 0xca},
        .clen = 16,
    }, {                // #2
        .key = {0x02, 0x20, 0x04, 0x83, 0xba, 0x6d, 0xf7, 0x67, 0x5d,
                0xea, 0x40, 0x73, 0xb0, 0x2a, 0x1c, 0x5b, 0x85, 0x02,
                0x54, 0x23, 0xe5, 0xc4, 0xa4, 0xb5},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x30, 0x0c, 0xe0, 0x17, 0x8d, 0x4d, 0x54, 0x56, 0x40,
                      0x24, 0x3b, 0x9d, 0xb8, 0x18, 0x90, 0x4a, 0xe9, 0x17,
                      0x19, 0x5d, 0xe7, 0xbb, 0x9a, 0xda},
        .plen = 24,
        .ciphertext = {0x8f, 0x83, 0x4b, 0x37, 0xf7, 0x40, 0x4e, 0x5e, 0xdb,
                       0x32, 0xd1, 0x0b, 0x17, 0x5c, 0x28, 0xac, 0x6d, 0x94,
                       0x3b, 0x19, 0xc2, 0x59, 0x83, 0x49},
        .clen = 24,
        .num_chunks = 3,
        .chunks = {8, 8, 8},
    }, {                // #3
        .key = {0x2c, 0x29, 0x20, 0x2c, 0x10, 0x79, 0x79, 0x85, 0xef,
                0xc2, 0x52, 0xb3, 0xda, 0x37, 0x8a, 0x89, 0xe9, 0xa7,
                0xf8, 0x8c, 0x98, 0xc7, 0x3b, 0x1c},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x40, 0x58, 0x77, 0x1b, 0x9c, 0x80, 0x8e, 0x69, 0x35,
                      0x65, 0x0f, 0x97, 0xdb, 0x27, 0xe9, 0xe6, 0x96, 0x41,
                      0xfc, 0xc5, 0xe7, 0xbc, 0x7f, 0xa5, 0x51, 0xa2, 0x9f,
                      0x09, 0x18, 0xb6, 0x69, 0xdc},
        .plen = 32,
        .ciphertext = {0x94, 0x38, 0xd7, 0xb8, 0xb2, 0x05, 0x7a, 0x62, 0x4a,
                       0x40, 0x71, 0xde, 0x46, 0xc9, 0x86, 0xa3, 0x39, 0x3d,
                       0xa8, 0x68, 0xa2, 0x96, 0x47, 0x04, 0x14, 0x18, 0xcb,
                       0x94, 0x6a, 0x51, 0xd3, 0x68},
        .clen = 32,
        .num_chunks = 2,
        .chunks = {16, 16},
    }, {                // #4
        .key = {0xc4, 0x02, 0x97, 0x20, 0x20, 0xec, 0x0e, 0x4a, 0x64,
                0x83, 0x3d, 0xfd, 0x5b, 0x1a, 0x43, 0xbf, 0x20, 0xf7,
                0x3b, 0xc7, 0x0b, 0xbf, 0x8f, 0x08},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x35, 0x1c, 0xc5, 0xe4, 0xa8, 0xa8, 0xb0, 0xb0, 0xdb,
                      0x89, 0xde, 0xef, 0xfc, 0xc4, 0x91, 0xee, 0xfa, 0x1f,
                      0xc4, 0xcb, 0x45, 0xd8, 0x0e, 0xcf, 0xd5, 0x96, 0xc8,
                      0xfd, 0x6d, 0xa0, 0x72, 0xeb, 0x21, 0x4d, 0x53, 0xd1,
                      0x16, 0xbc, 0x60, 0xe0},
        .plen = 40,
        .ciphertext = {0xc8, 0x50, 0xf2, 0xb1, 0xe1, 0xd3, 0x76, 0xb3, 0xbd,
                       0x08, 0x66, 0xd0, 0xfd, 0x23, 0x51, 0x5c, 0x4c, 0x95,
                       0x60, 0xe9, 0x3d, 0x5e, 0x74, 0xe7, 0x4e, 0xb2, 0xd3,
                       0xd9, 0x04, 0xc7, 0xf2, 0x90, 0xee, 0xc9, 0x53, 0x9d,
                       0xf3, 0xd8, 0xc7, 0x8a},
        .clen = 40,
        .num_chunks = 3,
        .chunks = {20, -1, 20},
    }, {                // #5
        .key = {0x68, 0x26, 0xf8, 0x91, 0x07, 0x9d, 0xda, 0x4c, 0x0e,
                0x57, 0xd0, 0x58, 0x13, 0x62, 0x9d, 0x61, 0x97, 0x89,
                0xbc, 0x68, 0x4a, 0xe9, 0x6d, 0x32},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0xf5, 0xee, 0x26, 0x02, 0x16, 0x50, 0xef, 0x21, 0x8f,
                      0xa4, 0xf4, 0x4a, 0xa5, 0xeb, 0xc5, 0x22, 0x7d, 0x86,
                      0xd3, 0xa5, 0x30, 0xf5, 0x24, 0x0d, 0x21, 0xe5, 0x4b,
                      0xf6, 0x8f, 0x6a, 0x98, 0xbf, 0x66, 0x89, 0x8c, 0x33,
                      0x5b, 0xf9, 0x8f, 0x69, 0x37, 0x2a, 0xe5, 0xe8, 0x73,
                      0x87, 0xdf, 0x0f},
        .plen = 48,
        .ciphertext = {0xa8, 0x19, 0xe2, 0xbc, 0x81, 0x0d, 0xd1, 0xc6, 0xed,
                       0x4e, 0x7e, 0x07, 0x6c, 0x5a, 0x05, 0xc3, 0xce, 0xb6,
                       0xdb, 0x5e, 0x21, 0x74, 0xc2, 0x86, 0x3d, 0x05, 0x24,
                       0x93, 0xca, 0x24, 0x80, 0x3d, 0xce, 0x58, 0x8b, 0x4b,
                       0x92, 0x8b, 0x36, 0x4d, 0xe7, 0x58, 0x8e, 0xb6, 0x62,
                       0xb5, 0x8c, 0xa6},
        .clen = 48,
        .num_chunks = 4,
        .chunks = {28, 13, 0, 7},
    }, {                // #6
        .key = {0x94, 0xec, 0x51, 0xd0, 0x80, 0x4a, 0x15, 0x31, 0x68,
                0x5e, 0x1c, 0x6b, 0x9b, 0xab, 0xc7, 0x08, 0x02, 0x89,
                0x40, 0x6e, 0x51, 0x19, 0x61, 0xda},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x4e, 0xc4, 0x2e, 0xbd, 0x1e, 0x7d, 0xc5, 0xf2, 0xcf,
                      0xba, 0x60, 0x67, 0xda, 0x9f, 0xe7, 0x77, 0xfb, 0x84,
                      0x9d, 0x67, 0x37, 0xed, 0x48, 0xf6, 0x1e, 0xbc, 0x91,
                      0x35, 0x4e, 0xa5, 0xad, 0x99, 0x38, 0xe3, 0x3d, 0x4f,
                      0x27, 0x29, 0x73, 0xcf, 0x7e, 0xdc, 0x77, 0xb4, 0x40,
                      0x7f, 0x09, 0xcb, 0xd6, 0x6f, 0x67, 0x6a, 0x0d, 0xd0,
                      0x71, 0x03},
        .plen = 56,
        .ciphertext = {0xdd, 0x61, 0x15, 0x22, 0x7a, 0x40, 0x90, 0x7c, 0x83,
                       0x5a, 0x98, 0xeb, 0x22, 0x78, 0x6d, 0xd6, 0x77, 0x19,
                       0x09, 0xd0, 0x9c, 0x4d, 0x9e, 0x3a, 0xe6, 0x99, 0xcd,
                       0xbd, 0x11, 0x0a, 0x82, 0xe1, 0xca, 0xc4, 0xd3, 0x2f,
                       0xc3, 0x5a, 0x5b, 0x3d, 0xb5, 0xba, 0x1a, 0xc1, 0x88,
                       0xcc, 0x63, 0x39, 0xc6, 0x63, 0x34, 0xde, 0xde, 0xca,
                       0x7e, 0x4e},
        .clen = 56,
    }, {                // #7
        .key = {0x04, 0xe3, 0xf4, 0x67, 0x02, 0x91, 0x32, 0x13, 0xb0,
                0x07, 0x76, 0xbf, 0x52, 0xc7, 0x8a, 0xcb, 0xfd, 0xab,
                0xa1, 0xf8, 0xb6, 0xd6, 0x58, 0x57},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x9e, 0xc9, 0xa6, 0x9f, 0xf3, 0x69, 0x22, 0x5b, 0x75,
                      0x03, 0x38, 0xaa, 0x11, 0xdb, 0xee, 0x12, 0x7f, 0xda,
                      0x57, 0xe4, 0x24, 0xe1, 0xa1, 0x26, 0x36, 0x5d, 0x56,
                      0xd1, 0x21, 0x1c, 0xdb, 0x3d, 0x29, 0x5f, 0x3d, 0xd9,
                      0x51, 0x30, 0x2c, 0x1f, 0x7c, 0xf5, 0x48, 0xc7, 0xd3,
                      0x46, 0x9c, 0x3b, 0x66, 0xfc, 0x68, 0x4c, 0x16, 0x8a,
                      0x24, 0x23, 0xb2, 0x91, 0xc1, 0x7a, 0x60, 0xae, 0x32,
                      0x0c},
        .plen = 64,
        .ciphertext = {0x85, 0x88, 0xa7, 0x92, 0x64, 0x0c, 0x21, 0x5b, 0xcb,
                       0x3b, 0x93, 0x89, 0x7e, 0xd8, 0xc5, 0x49, 0x63, 0x36,
                       0xca, 0xc5, 0xa4, 0xbd, 0x39, 0x3e, 0x9d, 0x35, 0x67,
                       0xf8, 0x70, 0x45, 0xb3, 0xa1, 0x4c, 0x4b, 0xfd, 0x3e,
                       0x2c, 0x3b, 0x5e, 0x6e, 0xe1, 0x5c, 0xfb, 0xba, 0xd3,
                       0x92, 0xef, 0x92, 0x95, 0x31, 0x21, 0x8f, 0x4e, 0x9a,
                       0x85, 0x81, 0xf0, 0xb5, 0xc0, 0x92, 0xfd, 0x64, 0x3c,
                       0xe2},
        .clen = 64,
    }, {                // #8
        .key = {0xb6, 0xb0, 0xda, 0xc8, 0xa8, 0x38, 0x61, 0x10, 0x32,
                0x75, 0xd0, 0x4f, 0x46, 0x7a, 0x40, 0xb3, 0x92, 0x4c,
                0x3d, 0xc4, 0x0d, 0xbf, 0x7c, 0x79},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x8f, 0x6e, 0x5d, 0x5d, 0xa6, 0xdc, 0x3a, 0x8e, 0xf8,
                      0x7b, 0xe3, 0x47, 0xf5, 0x46, 0x72, 0x05, 0xbb, 0x3c,
                      0x00, 0x4e, 0x42, 0xbe, 0x76, 0x3a, 0xd6, 0x38, 0xa0,
                      0x32, 0xcb, 0x03, 0x83, 0xf8, 0x6c, 0x8a, 0xf1, 0xe7,
                      0x34, 0xac, 0x5a, 0x7d, 0x41, 0x62, 0xe5, 0x16, 0x05,
                      0xa2, 0xbf, 0xfa, 0x3f, 0x83, 0xb3, 0x9a, 0x29, 0xa1,
                      0xfc, 0x17, 0xf3, 0xc3, 0x57, 0x04, 0x71, 0xbb, 0x5c,
                      0x6d, 0x11, 0x8e, 0x0d, 0x18, 0x0f, 0xd5, 0xb0, 0x0a},
        .plen = 72,
        .ciphertext = {0x7b, 0xe0, 0x1e, 0x90, 0x9b, 0x9d, 0x8d, 0x33, 0xd4,
                       0xba, 0x8d, 0x24, 0xee, 0x9a, 0xc2, 0x78, 0x89, 0x8d,
                       0xca, 0xb3, 0xaf, 0xaf, 0x4c, 0xde, 0xf7, 0x3c, 0x53,
                       0x9c, 0x99, 0x6b, 0x97, 0xa9, 0x68, 0x77, 0x9d, 0x27,
                       0x30, 0xac, 0x7d, 0x0c, 0xbf, 0x68, 0x20, 0xee, 0x3d,
                       0xc2, 0x6c, 0xa7, 0x15, 0x8c, 0xeb, 0x37, 0xa0, 0xb4,
                       0x95, 0x65, 0x16, 0xe0, 0xd9, 0x61, 0x8c, 0xce, 0xff,
                       0xa5, 0xfe, 0x83, 0x6c, 0xd5, 0x53, 0xa1, 0x77, 0x8c},
        .clen = 72,
    }, {                // #9
        .key = {0xa8, 0xd9, 0xfb, 0xec, 0xb9, 0x64, 0x67, 0xbc, 0xbf,
                0xdf, 0x57, 0xa1, 0x1c, 0x49, 0xc7, 0x7f, 0x0d, 0x3e,
                0xce, 0x23, 0xcd, 0xd0, 0xf1, 0x79},
        .klen = 24,
        .iv = {0},
        .ivlen = 0,
        .plaintext = {0x89, 0xd6, 0x67, 0xba, 0x5e, 0x88, 0xf5, 0x63, 0xad,
                      0x02, 0x9a, 0x7f, 0x78, 0x87, 0x97, 0x1d, 0x20, 0x44,
                      0x7b, 0x5c, 0xdb, 0xec, 0x3d, 0x2e, 0x62, 0xa0, 0x88,
                      0x5c, 0xea, 0x11, 0x56, 0xca, 0x56, 0xe8, 0xc0, 0xfd,
                      0xab, 0x66, 0x14, 0xda, 0xc4, 0x47, 0x29, 0xb0, 0x47,
                      0xed, 0xef, 0x79, 0xd7, 0xba, 0x4a, 0x5d, 0x8b, 0xbe,
                      0x21, 0x78, 0x9f, 0x92, 0xb2, 0xdd, 0x6a, 0x83, 0x28,
                      0xdb, 0x90, 0xf3, 0x6f, 0x7f, 0x79, 0xfd, 0x47, 0x59,
                      0x02, 0x58, 0x97, 0x83, 0x72, 0x7e, 0x04, 0xe0},
        .plen = 80,
        .ciphertext = {0xc3, 0x69, 0x6e, 0x20, 0xf3, 0x35, 0xb3, 0xf7, 0x62,
                       0x9b, 0xe3, 0x38, 0xd8, 0x70, 0xd2, 0x79, 0x5b, 0x30,
                       0x75, 0xe9, 0xb8, 0x85, 0x2c, 0x3a, 0xae, 0xa0, 0x76,
                       0x97, 0x7f, 0x62, 0xa9, 0xaf, 0xfd, 0x8e, 0x75, 0x68,
                       0x83, 0x1f, 0x1b, 0x99, 0x40, 0x4e, 0x9a, 0xd4, 0xd5,
                       0x26, 0x84, 0x21, 0x7e, 0xe2, 0x7d, 0xb9, 0x85, 0x10,
                       0xc0, 0x99, 0x4f, 0x53, 0xa2, 0x63, 0xd1, 0x0e, 0xba,
                       0x9b, 0xa2, 0x96, 0xd0, 0x90, 0x9a, 0x51, 0x75, 0xa9,
                       0xa7, 0x63, 0x74, 0x41, 0xfd, 0x42, 0x00, 0xcc},
        .clen = 80,
    }
};

/** http://csrc.nist.gov/groups/STM/cavp/documents/des/tdesmmt.zip
    TCBCMMT3.rsp
**/
static struct des3_test_vector des3_cbc_tv[] = {
    {                   // #0
        .key = {0x46, 0x13, 0x3d, 0xcb, 0xf2, 0x32, 0xb5, 0x19, 0x64,
                0xe0, 0xd9, 0x5e, 0x83, 0x20, 0x8f, 0x15, 0x67, 0x32,
                0xbf, 0x75, 0xb6, 0x73, 0xab, 0xf1},
        .klen = 24,
        .iv = {0x34, 0x81, 0x4c, 0x87, 0xf4, 0x7f, 0xd5, 0x9d},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xde, 0x65, 0x5a, 0x0e, 0xa7, 0x71, 0x43, 0x6c},
        .plen = 8,
        .ciphertext = {0x09, 0x23, 0x68, 0x40, 0x52, 0x96, 0x74, 0x4a},
        .clen = 8,
    }, {                // #1
        .key = {0x6d, 0x0d, 0x67, 0xda, 0x68, 0xab, 0x16, 0x6d, 0x1f,
                0x43, 0xc7, 0x20, 0x4c, 0x4c, 0x2a, 0xa4, 0xc8, 0x1a,
                0x52, 0x85, 0x15, 0xf1, 0xdf, 0xf2},
        .klen = 24,
        .iv = {0x68, 0xe6, 0x3a, 0x07, 0xb2, 0x2e, 0x33, 0xeb},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x43, 0x46, 0xc4, 0xe8, 0x13, 0x80, 0x62, 0x6f, 0xa0,
                      0xb2, 0x77, 0x6d, 0x30, 0xa4, 0xfc, 0x05},
        .plen = 16,
        .ciphertext = {0x52, 0x74, 0xbe, 0x18, 0x3f, 0x5d, 0xfb, 0x6b, 0x01,
                       0x8f, 0x22, 0xb3, 0x22, 0xf0, 0x39, 0x2d},
        .clen = 16,
    }, {                // #2
        .key = {0x13, 0x4c, 0xb3, 0xef, 0xe6, 0x2a, 0x4a, 0xd5, 0x52,
                0xcb, 0x85, 0xa1, 0x64, 0xfe, 0xe6, 0xb9, 0x64, 0xa1,
                0x26, 0x9b, 0x19, 0x3d, 0x68, 0xc4},
        .klen = 24,
        .iv = {0x0f, 0xa3, 0x11, 0xf9, 0x9e, 0xc5, 0x7b, 0x86},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x1b, 0x73, 0x5b, 0x05, 0x57, 0x25, 0x5a, 0x0e, 0x6d,
                      0x8d, 0x67, 0x58, 0x79, 0xe7, 0x20, 0x1c, 0xa3, 0x4c,
                      0x87, 0x61, 0xa1, 0x29, 0xa9, 0x14},
        .plen = 24,
        .ciphertext = {0x2f, 0x1a, 0xc7, 0xee, 0x14, 0x14, 0xaf, 0x15, 0x58,
                       0x7c, 0xb2, 0xc5, 0x40, 0x40, 0x12, 0x94, 0x02, 0x8e,
                       0x1e, 0x39, 0xd1, 0xcf, 0x2f, 0x67},
        .clen = 24,
        .num_chunks = 3,
        .chunks = {8, 8, 8},
    }, {                // #3
        .key = {0x20, 0x23, 0x98, 0xb6, 0x15, 0x49, 0x68, 0xc1, 0x68,
                0x20, 0x13, 0x29, 0x91, 0x0e, 0x61, 0x2f, 0x29, 0x61,
                0x89, 0xd3, 0x20, 0x67, 0x01, 0x20},
        .klen = 24,
        .iv = {0x51, 0x42, 0x73, 0xc9, 0x38, 0x06, 0xfd, 0xe6},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x9a, 0x5e, 0xc9, 0x13, 0x87, 0x62, 0x99, 0x49, 0x2d,
                      0xda, 0x39, 0x98, 0xf8, 0x8e, 0x1c, 0x31, 0xa7, 0x54,
                      0x93, 0xb3, 0xad, 0xe1, 0x4e, 0x9e, 0xd7, 0xde, 0x1f,
                      0x0a, 0x30, 0x3f, 0x02, 0x99},
        .plen = 32,
        .ciphertext = {0x9b, 0xc0, 0x22, 0x47, 0xff, 0x5c, 0xef, 0xde, 0x9a,
                       0x03, 0x07, 0xf9, 0x48, 0xf9, 0x43, 0x7e, 0xf2, 0xa2,
                       0x98, 0xcd, 0xd6, 0x95, 0x42, 0x23, 0x6c, 0xba, 0x47,
                       0xe8, 0xc9, 0x54, 0xe8, 0x19},
        .clen = 32,
        .num_chunks = 2,
        .chunks = {16, 16},
    }, {                // #4
        .key = {0x19, 0x94, 0x76, 0xc2, 0x43, 0x02, 0xb6, 0xab, 0x0b,
                0x98, 0xd5, 0xa8, 0x07, 0x07, 0x9e, 0x43, 0x37, 0xe6,
                0x8f, 0x5e, 0xc7, 0x75, 0x61, 0xab},
        .klen = 24,
        .iv = {0x6e, 0x06, 0x87, 0x59, 0x55, 0xb8, 0x7f, 0x87},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x42, 0x0f, 0xdf, 0xfb, 0x70, 0x67, 0x70, 0xba, 0x1a,
                      0xee, 0x85, 0xbb, 0x99, 0x2e, 0xd2, 0xef, 0xe7, 0xc4,
                      0xcb, 0x10, 0xa7, 0x5c, 0xc0, 0x35, 0x65, 0xdd, 0x4d,
                      0xa4, 0x1e, 0xe1, 0xf1, 0xe7, 0x03, 0x68, 0xfd, 0x22,
                      0x7a, 0xcd, 0xb0, 0xa3},
        .plen = 40,
        .ciphertext = {0xf3, 0x90, 0xd5, 0xaf, 0xc5, 0x96, 0x9d, 0x63, 0xd9,
                       0xc0, 0xa9, 0x5e, 0x1f, 0x1c, 0x29, 0x0d, 0x0b, 0xd2,
                       0x95, 0x10, 0xe8, 0xe9, 0x82, 0x50, 0x2f, 0xaa, 0x7e,
                       0xd6, 0x16, 0xcf, 0xe8, 0x73, 0xaf, 0x88, 0xb2, 0x7a,
                       0x7b, 0x62, 0xcb, 0x21},
        .clen = 40,
        .num_chunks = 0,
        .chunks = {20, -1, 20},
    }, {                // #5
        .key = {0x32, 0x3b, 0xf2, 0x4f, 0xe0, 0xad, 0x70, 0x94, 0x3e,
                0x70, 0xbf, 0x1c, 0x5d, 0xf4, 0xd5, 0x31, 0xdc, 0x0d,
                0x92, 0x6e, 0x83, 0x80, 0x4c, 0x4a},
        .klen = 24,
        .iv = {0x29, 0x85, 0x58, 0xd9, 0x55, 0x17, 0xa0, 0x45},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x26, 0xe4, 0x4a, 0xa7, 0x8f, 0xcc, 0x69, 0x06, 0x87,
                      0xe7, 0x4c, 0xfd, 0xfc, 0xbd, 0x6e, 0xf3, 0x46, 0x96,
                      0x01, 0x1e, 0x5a, 0xe1, 0xcb, 0xfe, 0x40, 0xd6, 0x33,
                      0x2b, 0xc7, 0x5b, 0x9c, 0x51, 0x77, 0x24, 0xf1, 0x79,
                      0xc7, 0x1f, 0x81, 0x8a, 0x90, 0x0f, 0x0e, 0x0f, 0xc2,
                      0x76, 0x20, 0x3a},
        .plen = 48,
        .ciphertext = {0xd0, 0x9b, 0x95, 0x68, 0x87, 0x77, 0x69, 0x78, 0x0f,
                       0xda, 0x99, 0x11, 0xc2, 0x9b, 0x30, 0x3b, 0x27, 0xe1,
                       0x5b, 0x5f, 0x29, 0xb2, 0xdd, 0xf8, 0x9c, 0x3b, 0x7e,
                       0xdc, 0xc0, 0x4d, 0xee, 0x78, 0xb7, 0x51, 0xd4, 0x59,
                       0xa5, 0x0d, 0xf3, 0xae, 0x57, 0xbf, 0xfd, 0x4b, 0x4c,
                       0xa4, 0x1f, 0xc4},
        .clen = 48,
        .num_chunks = 4,
        .chunks = {28, 13, 0, 7},
    }, {                // #6
        .key = {0xf7, 0xa4, 0x94, 0xc2, 0x73, 0x26, 0x73, 0x4a, 0x10,
                0x45, 0x25, 0x04, 0x10, 0x52, 0x10, 0x3e, 0xf2, 0x68,
                0x0e, 0x3b, 0xd3, 0x5e, 0xa1, 0x58},
        .klen = 24,
        .iv = {0xa5, 0xfe, 0x08, 0xd9, 0x04, 0x5b, 0xbf, 0x88},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xf9, 0x11, 0x72, 0x49, 0x04, 0x7f, 0xb9, 0xfe, 0x88,
                      0x4c, 0x89, 0x26, 0x7e, 0x5a, 0xc9, 0x63, 0x14, 0xf2,
                      0x33, 0xd2, 0x01, 0xc1, 0x4d, 0x33, 0x15, 0x20, 0x9a,
                      0x4a, 0x14, 0x99, 0x11, 0x84, 0xc4, 0xa6, 0xce, 0xde,
                      0x45, 0xbb, 0x08, 0x03, 0xa1, 0x87, 0x43, 0xb4, 0xa4,
                      0x78, 0xc6, 0x6d, 0xee, 0x7d, 0x50, 0x46, 0x46, 0x0f,
                      0x07, 0x17},
        .plen = 56,
        .ciphertext = {0xc8, 0x9f, 0xf7, 0xbc, 0xc5, 0x55, 0x89, 0x52, 0xe8,
                       0x61, 0x4a, 0x7b, 0xfc, 0xf8, 0xa1, 0xb0, 0x17, 0xf0,
                       0x40, 0xef, 0x95, 0x4f, 0x65, 0x52, 0x69, 0x6a, 0xe5,
                       0x3b, 0x70, 0xe6, 0xf9, 0x53, 0xba, 0x6d, 0x9d, 0x0e,
                       0xfe, 0x38, 0x1c, 0x2b, 0xdd, 0x54, 0xd9, 0x11, 0xad,
                       0x6e, 0xac, 0x5b, 0x0c, 0xe2, 0x84, 0x53, 0x41, 0xae,
                       0x34, 0x58},
        .clen = 56,
    }, {                // #7
        .key = {0xd3, 0x31, 0x89, 0x58, 0xc7, 0x8c, 0xfd, 0x98, 0xc7,
                0xae, 0x31, 0x2a, 0x2c, 0x08, 0xbc, 0xfe, 0xfd, 0x10,
                0x26, 0x97, 0x29, 0x01, 0x3d, 0xe5},
        .klen = 24,
        .iv = {0x01, 0xb6, 0xdd, 0x4a, 0x4c, 0x96, 0x34, 0x9a},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x83, 0x57, 0x73, 0x46, 0x84, 0x1a, 0x1b, 0xbf, 0x08,
                      0xfb, 0x6d, 0x2b, 0xe3, 0x51, 0xf3, 0x39, 0x14, 0xef,
                      0xc8, 0xef, 0x9a, 0xa7, 0x05, 0x4b, 0x89, 0x0d, 0x7c,
                      0x2b, 0x5c, 0x7d, 0xa9, 0x6a, 0x8d, 0x60, 0xe5, 0x4c,
                      0x63, 0xaa, 0xdd, 0x1c, 0x0c, 0xa7, 0x43, 0x13, 0xeb,
                      0xf8, 0xc3, 0x7c, 0x33, 0xd9, 0x39, 0x16, 0xb5, 0xa5,
                      0x21, 0xba, 0xf6, 0x9b, 0x4e, 0x3d, 0x85, 0xe2, 0xe0,
                      0x98},
        .plen = 64,
        .ciphertext = {0x8b, 0xe3, 0x0c, 0xa7, 0x45, 0xa0, 0x6e, 0x2f, 0x67,
                       0xc3, 0xec, 0x2e, 0xd1, 0x8e, 0x87, 0x20, 0x28, 0x44,
                       0x85, 0x97, 0x21, 0x8b, 0x00, 0x77, 0x46, 0xd9, 0x54,
                       0xda, 0x0b, 0xa6, 0xb3, 0xdb, 0x20, 0xf9, 0xa2, 0x8d,
                       0x7f, 0xae, 0xa3, 0x2e, 0xe3, 0xf1, 0x31, 0x03, 0xdc,
                       0x80, 0x18, 0xe8, 0xf9, 0x7d, 0xc2, 0xc3, 0x48, 0xd2,
                       0x96, 0x6c, 0x9e, 0x2b, 0x87, 0xb0, 0xf3, 0x18, 0xe0,
                       0x26},
        .clen = 64,
    }, {                // #8
        .key = {0xa4, 0xe5, 0x0d, 0x8a, 0x1f, 0x64, 0xf4, 0xd9, 0xf4,
                0x1a, 0x8a, 0xd0, 0xa8, 0xc8, 0x68, 0xb3, 0x4a, 0xc8,
                0x61, 0xdc, 0x4c, 0x38, 0xfd, 0x89},
        .klen = 24,
        .iv = {0xf2, 0x10, 0x47, 0xa0, 0x8a, 0x1b, 0xed, 0x96},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xfa, 0x7c, 0x94, 0xa2, 0xeb, 0x67, 0xdb, 0xf0, 0xce,
                      0x8e, 0x30, 0xd5, 0xb9, 0x44, 0xe9, 0x58, 0x5f, 0xfe,
                      0xf8, 0xb9, 0xd8, 0x07, 0xca, 0xbd, 0xa0, 0x75, 0x6f,
                      0x46, 0xe1, 0x5b, 0xfa, 0x26, 0xf5, 0x40, 0x2b, 0x35,
                      0x26, 0xd0, 0x4b, 0xa2, 0xb9, 0xd2, 0x58, 0x86, 0x2c,
                      0x5a, 0x62, 0xbf, 0xbb, 0x40, 0x35, 0xfb, 0x29, 0xd0,
                      0x38, 0x13, 0xbc, 0xc3, 0x95, 0x9b, 0x8a, 0x6f, 0x9a,
                      0xb7, 0xf3, 0x3a, 0x60, 0x2f, 0x6f, 0x54, 0xaa, 0xc6},
        .plen = 72,
        .ciphertext = {0xe4, 0x02, 0x5b, 0x1f, 0x5b, 0xcd, 0xe6, 0xea, 0xe6,
                       0xc7, 0xbb, 0xfd, 0x48, 0x51, 0x25, 0x8a, 0xf6, 0xd0,
                       0x97, 0x79, 0x1a, 0x02, 0xd8, 0xdf, 0x54, 0x83, 0xa8,
                       0xcb, 0xcf, 0xea, 0xaf, 0x52, 0xdd, 0xdb, 0x61, 0xbf,
                       0x1d, 0xdc, 0x52, 0xb9, 0xcb, 0x6d, 0xf9, 0x47, 0xcb,
                       0x04, 0x92, 0x48, 0x90, 0x36, 0x21, 0x25, 0x43, 0xfa,
                       0xb8, 0x26, 0x94, 0x44, 0x9c, 0x8d, 0x3d, 0x32, 0x3d,
                       0xce, 0x01, 0x6e, 0x26, 0x48, 0xac, 0x6f, 0x65, 0x3e},
        .clen = 72,
    }, {                // #9
        .key = {0x08, 0x76, 0x3d, 0xa8, 0x62, 0xad, 0x16, 0xef, 0x58,
                0x15, 0x40, 0x8f, 0x5d, 0x3b, 0x70, 0x54, 0x15, 0xab,
                0x15, 0x43, 0xa4, 0x2c, 0x3e, 0xfb},
        .klen = 24,
        .iv = {0x06, 0x34, 0xd6, 0x9e, 0xaf, 0xf3, 0xae, 0x17},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x10, 0x9a, 0x3d, 0x3d, 0x74, 0x5d, 0x65, 0xb3, 0x8e,
                      0xdb, 0xc7, 0x3d, 0x1d, 0xe8, 0xb2, 0x80, 0x7f, 0x78,
                      0x20, 0x22, 0x1a, 0x6c, 0x39, 0x37, 0xfa, 0xab, 0x19,
                      0xfc, 0xbb, 0x75, 0xd3, 0xc8, 0xaa, 0xf4, 0xb6, 0x3f,
                      0x27, 0x14, 0xcf, 0xc9, 0x4e, 0x95, 0xae, 0x43, 0xd6,
                      0x5f, 0x6d, 0xf4, 0x38, 0x15, 0xef, 0xc2, 0x14, 0xec,
                      0x66, 0xa5, 0xd1, 0xbe, 0x18, 0x5d, 0x85, 0x5a, 0x62,
                      0x60, 0x14, 0x1f, 0xfd, 0x17, 0x9b, 0xc9, 0x80, 0x49,
                      0x0f, 0x8a, 0x26, 0xd8, 0x21, 0x5d, 0xd2, 0xab},
        .plen = 80,
        .ciphertext = {0xe9, 0x51, 0x3e, 0x88, 0x92, 0xa0, 0x90, 0x85, 0xbe,
                       0xe2, 0x9c, 0x35, 0x80, 0x14, 0xaf, 0xd6, 0x0d, 0x75,
                       0x78, 0xd2, 0x1e, 0x00, 0xa3, 0x1e, 0x5d, 0x61, 0xb9,
                       0x65, 0xc1, 0x87, 0x78, 0xeb, 0xe1, 0x84, 0x69, 0x17,
                       0x07, 0x94, 0xe5, 0xdd, 0xf2, 0x4a, 0xa7, 0x77, 0xc8,
                       0xab, 0x0a, 0x2c, 0x62, 0x47, 0x41, 0x09, 0xe6, 0x17,
                       0x97, 0x8b, 0xcc, 0x5c, 0xe3, 0x45, 0x6d, 0xdd, 0x96,
                       0x22, 0x83, 0x34, 0x20, 0x44, 0x3c, 0x2a, 0x26, 0xb1,
                       0xb6, 0xe2, 0x0a, 0x05, 0xc1, 0x89, 0xda, 0x6c},
        .clen = 80,
    }
};

/** http://csrc.nist.gov/groups/STM/cavp/documents/des/tdesmmt.zip
    TCFB8MMT3.rsp
**/
static struct des3_test_vector des3_cfb8_tv[] = {
    {                   // #0
        .key = {0x94, 0x73, 0xba, 0x2c, 0x29, 0xa8, 0x7f, 0xec, 0x4f,
                0x38, 0x29, 0xb3, 0xb6, 0xbc, 0x9e, 0x92, 0x43, 0x4c,
                0xab, 0x51, 0x89, 0x0e, 0xda, 0x8c},
        .klen = 24,
        .iv = {0x34, 0x20, 0xb1, 0x32, 0x81, 0xd3, 0x0b, 0x06},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xcd},
        .plen = 1,
        .ciphertext = {0x22},
        .clen = 1,
    }, {                // #1
        .key = {0xb6, 0x92, 0xea, 0x4a, 0x62, 0x29, 0x9b, 0x32, 0xd9,
                0x04, 0x51, 0x38, 0x8a, 0x25, 0x8f, 0xce, 0x3b, 0x3b,
                0x1c, 0xfd, 0x34, 0xbc, 0x7f, 0x5d},
        .klen = 24,
        .iv = {0x54, 0xc0, 0xb1, 0xef, 0xef, 0xf2, 0x53, 0xa2},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x32, 0xee},
        .plen = 2,
        .ciphertext = {0x7c, 0x92},
        .clen = 2,
    }, {                // #2
        .key = {0x40, 0x49, 0x02, 0xc8, 0x51, 0x19, 0xda, 0xb6, 0x1f,
                0xb5, 0xce, 0xd6, 0xf4, 0x6b, 0x45, 0xf8, 0xe5, 0xe0,
                0x61, 0x34, 0xa8, 0xb5, 0xc1, 0xb9},
        .klen = 24,
        .iv = {0xa3, 0xd6, 0x97, 0x32, 0xf9, 0x4f, 0xff, 0xc7},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x14, 0xc4, 0x5c},
        .plen = 3,
        .ciphertext = {0x35, 0x8d, 0x66},
        .clen = 3,
    }, {                // #3
        .key = {0xc4, 0xd3, 0x45, 0xfd, 0xdc, 0x9d, 0xf7, 0x02, 0x3b,
                0x04, 0xe6, 0x76, 0x52, 0x8a, 0x2f, 0x57, 0xe0, 0x51,
                0xba, 0x86, 0x6b, 0x98, 0x7a, 0xba},
        .klen = 24,
        .iv = {0xd7, 0x6f, 0xbd, 0x43, 0x4d, 0x8b, 0x2e, 0xd9},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x1d, 0x1e, 0x42, 0xcd},
        .plen = 4,
        .ciphertext = {0x44, 0xde, 0xee, 0xe3},
        .clen = 4,
    }, {                // #4
        .key = {0xf1, 0xc4, 0x07, 0xf8, 0x25, 0x0b, 0xcd, 0x10, 0xda,
                0x91, 0x4f, 0x62, 0x6b, 0xf7, 0x15, 0x54, 0x8c, 0x31,
                0xd6, 0xe0, 0xba, 0x70, 0xec, 0x5b},
        .klen = 24,
        .iv = {0x0f, 0x9d, 0x30, 0x71, 0xaf, 0x25, 0xa1, 0x42},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xa4, 0x3e, 0xcd, 0x3d, 0x06},
        .plen = 5,
        .ciphertext = {0x6a, 0x6d, 0xa4, 0x8c, 0x70},
        .clen = 5,
    }, {                // #5
        .key = {0x26, 0x7c, 0xcb, 0xea, 0xa4, 0x91, 0x25, 0x51, 0x2c,
                0x7f, 0x2c, 0x9e, 0x8a, 0xf8, 0x4a, 0xe6, 0x3b, 0x9b,
                0xb9, 0x57, 0x83, 0x16, 0x8a, 0x52},
        .klen = 24,
        .iv = {0x03, 0xf6, 0x56, 0xfb, 0x72, 0xc9, 0xbf, 0x5d},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xab, 0x15, 0x81, 0x70, 0x7d, 0x7d},
        .plen = 6,
        .ciphertext = {0x16, 0x88, 0x1e, 0x43, 0xfc, 0x0c},
        .clen = 6,
    }, {                // #6
        .key = {0x5e, 0xfe, 0x5d, 0x40, 0x80, 0x2f, 0x4f, 0x25, 0xad,
                0xc2, 0x62, 0x4c, 0xdc, 0x29, 0x80, 0x45, 0x2c, 0x0b,
                0x38, 0x91, 0x52, 0x57, 0x52, 0x80},
        .klen = 24,
        .iv = {0x00, 0x16, 0x7d, 0x7f, 0x48, 0x30, 0x9f, 0x0a},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x02, 0x4e, 0x62, 0x6c, 0xea, 0x49, 0x91},
        .plen = 7,
        .ciphertext = {0x2b, 0x23, 0x9e, 0x41, 0xd9, 0x60, 0x69},
        .clen = 7,
    }, {                // #7
        .key = {0xcb, 0x79, 0xb6, 0xdf, 0xad, 0xa1, 0x13, 0x79, 0xb6,
                0x51, 0x32, 0xa7, 0x70, 0x54, 0xd3, 0x08, 0x13, 0xb9,
                0xda, 0x29, 0x54, 0x02, 0x0e, 0x9e},
        .klen = 24,
        .iv = {0x2c, 0x5f, 0x51, 0xcd, 0x90, 0x0f, 0xe7, 0x53},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x58, 0x76, 0xdb, 0xe3, 0xd1, 0xe6, 0x3b, 0xec},
        .plen = 8,
        .ciphertext = {0x73, 0xc1, 0x54, 0x52, 0x4f, 0x4d, 0x57, 0x29},
        .clen = 8,
    }, {                // #8
        .key = {0x04, 0xdf, 0x08, 0x19, 0x29, 0x64, 0xd6, 0xbc, 0xc8,
                0x4c, 0x97, 0xda, 0x94, 0x0b, 0x32, 0xc8, 0x80, 0xdf,
                0xd0, 0xce, 0x8c, 0x91, 0x80, 0xa8},
        .klen = 24,
        .iv = {0xcf, 0x7e, 0x0c, 0xc1, 0xf2, 0x23, 0x34, 0x3e},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x1d, 0xe8, 0x12, 0x27, 0x9b, 0xd5, 0x57, 0x6e, 0xc2},
        .plen = 9,
        .ciphertext = {0x34, 0xd1, 0x06, 0x7d, 0x1a, 0x92, 0xa8, 0xda, 0x19},
        .clen = 9,
        .num_chunks = 3,
        .chunks = {4, -1, 5},
    }, {                // #9
        .key = {0xe3, 0x34, 0x7a, 0x6b, 0x0b, 0xc1, 0x15, 0x2c, 0x64,
                0x2a, 0x25, 0xcb, 0xd3, 0xbc, 0x31, 0xab, 0xfb, 0xa1,
                0x62, 0xa8, 0x1f, 0x19, 0x7c, 0x15},
        .klen = 24,
        .iv = {0xb7, 0x40, 0xcc, 0x21, 0xe9, 0x25, 0xe3, 0xc8},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xdb, 0xe9, 0x15, 0xfc, 0xb3, 0x3b, 0xca, 0x18, 0xef,
                      0x14},
        .plen = 10,
        .ciphertext = {0xf4, 0x80, 0x1a, 0x8d, 0x03, 0x9d, 0xb4, 0xca, 0x8f,
                       0xf6},
        .clen = 10,
        .num_chunks = 3,
        .chunks = {5, 0, 5},
    }
};


/** http://csrc.nist.gov/groups/STM/cavp/documents/des/tdesmmt.zip
    TCFB64MMT3.rsp
**/
static struct des3_test_vector des3_cfb64_tv[] = {
    {                   // #0
        .key = {0xe0, 0xd5, 0x25, 0xe9, 0xec, 0xa2, 0x26, 0xd5, 0x58,
                0x4a, 0x70, 0x2f, 0xdc, 0xd3, 0xdf, 0x23, 0x80, 0x58,
                0xad, 0x4c, 0x15, 0x70, 0x34, 0x8f},
        .klen = 24,
        .iv = {0x8b, 0xf6, 0xfe, 0xbf, 0xde, 0x90, 0xbd, 0x17},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x26, 0x85, 0xa3, 0x86, 0x57, 0xe8, 0xdb, 0xfe},
        .plen = 8,
        .ciphertext = {0xe9, 0xfb, 0xc0, 0x28, 0x10, 0x53, 0x54, 0xed},
        .clen = 8,
    }, {                // #1
        .key = {0x02, 0xf1, 0xb5, 0x64, 0x51, 0xc8, 0x20, 0x7f, 0xce,
                0x70, 0x92, 0x8f, 0x1f, 0xcd, 0xe0, 0xb3, 0x6d, 0xcd,
                0xa8, 0x58, 0xef, 0xe9, 0xb0, 0x6b},
        .klen = 24,
        .iv = {0xee, 0x34, 0x61, 0x53, 0xc0, 0x9d, 0xc8, 0x73},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xa7, 0x2f, 0x0d, 0x3f, 0xb8, 0xd4, 0xd8, 0xf7, 0x84,
                      0xce, 0x12, 0x0f, 0x2b, 0x4b, 0x85, 0x38},
        .plen = 16,
        .ciphertext = {0x3b, 0x6b, 0x32, 0x36, 0x5d, 0xac, 0xe6, 0xc9, 0x9b,
                       0x76, 0x74, 0xfb, 0xbe, 0xa1, 0x7a, 0x9b},
        .clen = 16,
    }, {                // #2
        .key = {0x76, 0xdc, 0xf7, 0xba, 0x76, 0x9b, 0xd3, 0x32, 0x34,
                0x02, 0xc2, 0xae, 0x61, 0x15, 0x02, 0xc8, 0xd5, 0xc2,
                0x26, 0x15, 0xef, 0xb0, 0xe0, 0x9e},
        .klen = 24,
        .iv = {0x5b, 0x48, 0x33, 0x4d, 0x9f, 0x3a, 0x67, 0xda},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xb4, 0xd0, 0x02, 0x4c, 0x7f, 0xa1, 0x40, 0xbe, 0xde,
                      0xa2, 0x19, 0xdc, 0xce, 0x36, 0x7a, 0xa9, 0xd6, 0x10,
                      0x53, 0x27, 0x82, 0xdf, 0xa8, 0x19},
        .plen = 24,
        .ciphertext = {0x26, 0xa9, 0x83, 0x6c, 0xfa, 0xc9, 0x6d, 0xfa, 0x2e,
                       0x45, 0x55, 0x48, 0xd2, 0x6c, 0x09, 0xea, 0x93, 0xe5,
                       0x3b, 0xff, 0x52, 0x37, 0x5c, 0xf0},
        .clen = 24,
        .num_chunks = 3,
        .chunks = {12, 0, 12},
    }, {                // #3
        .key = {0x4f, 0x5d, 0xad, 0xfe, 0x34, 0x43, 0x80, 0x3d, 0x07,
                0xdc, 0xcb, 0x2c, 0x37, 0x73, 0xdc, 0xb5, 0xc4, 0xc2,
                0xd6, 0x97, 0xcd, 0x7f, 0x67, 0xe9},
        .klen = 24,
        .iv = {0x9a, 0x6d, 0x3d, 0x65, 0xd3, 0xca, 0x4d, 0xfb},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x1c, 0x1f, 0x3c, 0xe7, 0x92, 0x7d, 0x25, 0x62, 0xc0,
                      0x53, 0xc6, 0x5d, 0x73, 0x8a, 0x6e, 0xbc, 0x92, 0xbf,
                      0x59, 0x9c, 0x52, 0xa4, 0x82, 0xdb, 0xa8, 0xa3, 0xb8,
                      0xda, 0x04, 0xc5, 0xf4, 0x81},
        .plen = 32,
        .ciphertext = {0xb6, 0x88, 0xdd, 0x16, 0x5b, 0x56, 0x12, 0x8c, 0x8f,
                       0x14, 0xcc, 0xd5, 0x99, 0x12, 0x63, 0x1f, 0x13, 0x0f,
                       0x0d, 0x95, 0x67, 0x7c, 0xae, 0xc3, 0x60, 0xd4, 0x96,
                       0x58, 0x99, 0x2e, 0x09, 0xec},
        .clen = 32,
        .num_chunks = 4,
        .chunks = {8, 8, 8, 8},
    }, {                // #4
        .key = {0x02, 0xc1, 0x64, 0x46, 0xe5, 0x25, 0x4c, 0x97, 0x45,
                0xf1, 0xab, 0xab, 0xd6, 0xc7, 0xcd, 0x9d, 0xf8, 0xcd,
                0x0d, 0x4a, 0x04, 0x20, 0xd3, 0xe5},
        .klen = 24,
        .iv = {0x61, 0x54, 0x2a, 0xc6, 0x07, 0xbb, 0x26, 0x7e},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xf9, 0x07, 0xdd, 0x85, 0x81, 0x65, 0x1f, 0xef, 0xf3,
                      0x3d, 0x54, 0x5d, 0xc4, 0xb9, 0x4a, 0x74, 0xd3, 0xd1,
                      0x71, 0xf5, 0xaa, 0x77, 0x3d, 0x95, 0x01, 0x12, 0xcc,
                      0x09, 0x84, 0x2e, 0xc9, 0x60, 0xd7, 0x72, 0x90, 0x95,
                      0xd6, 0xd2, 0x93, 0x65},
        .plen = 40,
        .ciphertext = {0x45, 0x3f, 0xe0, 0x1c, 0x94, 0xf0, 0xd0, 0x28, 0xac,
                       0xb8, 0xd8, 0x3d, 0x66, 0xca, 0xc3, 0x73, 0x4e, 0xb8,
                       0x43, 0xab, 0x4b, 0x6c, 0xbd, 0x6c, 0x0c, 0xbf, 0x1a,
                       0x1e, 0x8e, 0x55, 0x4c, 0x88, 0x13, 0xfa, 0x42, 0x93,
                       0xa9, 0x3c, 0x6f, 0x8c},
        .clen = 40,
        .num_chunks = 3,
        .chunks = {20, -1, 20},
    }, {                // #5
        .key = {0x70, 0x75, 0x57, 0x15, 0x23, 0xd9, 0x02, 0xc8, 0x9b,
                0x0d, 0xa2, 0x0b, 0x85, 0x70, 0x43, 0x04, 0x70, 0xe5,
                0x61, 0xda, 0x58, 0xfe, 0xe0, 0x38},
        .klen = 24,
        .iv = {0x86, 0xd8, 0xa5, 0xa0, 0x76, 0xce, 0xae, 0x0a},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x18, 0x39, 0xc8, 0xb6, 0xcd, 0xd4, 0x07, 0x0a, 0xdc,
                      0xc4, 0x24, 0x2c, 0x48, 0x4e, 0x5e, 0x31, 0x5d, 0xdb,
                      0x8f, 0x5c, 0xe9, 0x56, 0x01, 0x9c, 0xa9, 0xd6, 0x82,
                      0x55, 0x02, 0x7d, 0xbb, 0x63, 0xf1, 0xa0, 0xbd, 0x5d,
                      0x45, 0x27, 0xdd, 0x16, 0x0c, 0x37, 0xc2, 0x6a, 0x4b,
                      0x0f, 0xbc, 0x37},
        .plen = 48,
        .ciphertext = {0x4f, 0xe9, 0x7b, 0xab, 0x6c, 0x9d, 0xdd, 0xf7, 0xf6,
                       0x7d, 0xa8, 0xf2, 0xc1, 0x67, 0x3c, 0x17, 0xbf, 0x35,
                       0xd7, 0x4d, 0x8c, 0xf4, 0xa1, 0xb1, 0x6a, 0xc1, 0xb2,
                       0x74, 0xf7, 0x12, 0x40, 0xea, 0xef, 0x79, 0x31, 0x81,
                       0x8b, 0xa7, 0x45, 0x3a, 0x09, 0xa7, 0x4e, 0x3a, 0x1c,
                       0xb0, 0xc4, 0x81},
        .clen = 48,
    }, {                // #6
        .key = {0x79, 0x5b, 0x80, 0x98, 0xe5, 0xb3, 0xe5, 0x67, 0xd9,
                0xa8, 0x9d, 0x4a, 0x8c, 0x08, 0x3e, 0x54, 0x0e, 0xf7,
                0x31, 0x16, 0x62, 0x6d, 0xb0, 0x32},
        .klen = 24,
        .iv = {0xe1, 0x92, 0x73, 0x25, 0x82, 0xbf, 0x00, 0x85},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x02, 0xbc, 0x7e, 0xda, 0x94, 0xe8, 0x91, 0x19, 0xa9,
                      0x6e, 0xf5, 0x50, 0x1d, 0x05, 0x91, 0x96, 0x3a, 0x27,
                      0x0a, 0xcf, 0xe3, 0x72, 0x95, 0x2d, 0x11, 0x4e, 0x09,
                      0xf6, 0x84, 0xd0, 0x39, 0x87, 0xb2, 0xde, 0x80, 0x99,
                      0x04, 0x94, 0x9d, 0xc3, 0x41, 0xe7, 0xc1, 0x40, 0x4a,
                      0x88, 0xe7, 0xf3, 0x09, 0x0b, 0x64, 0xa8, 0xa4, 0x49,
                      0xbf, 0x9d},
        .plen = 56,
        .ciphertext = {0xdd, 0x87, 0x24, 0xcd, 0xd9, 0x53, 0xf6, 0x95, 0x10,
                       0x9c, 0xaa, 0x06, 0x8d, 0xd7, 0x1a, 0xa6, 0x58, 0xe7,
                       0x5e, 0x79, 0x64, 0xc2, 0xc6, 0x02, 0x1c, 0x77, 0x1d,
                       0x53, 0x59, 0x05, 0x60, 0xc0, 0x0c, 0x88, 0xcc, 0xc6,
                       0xee, 0x14, 0x8e, 0x28, 0x72, 0xa1, 0x33, 0x70, 0x37,
                       0xe2, 0x79, 0xea, 0xe0, 0x27, 0x60, 0xac, 0x0f, 0xe4,
                       0x83, 0x96},
        .clen = 56,
        .num_chunks = 4,
        .chunks = {16, 16, 16, 8},
    }, {                // #7
        .key = {0x7c, 0x7c, 0xc7, 0xfe, 0x4a, 0xf2, 0x0e, 0x6b, 0x38,
                0xa8, 0x79, 0x5d, 0xab, 0xfd, 0xba, 0x1f, 0xbc, 0x5d,
                0x25, 0x45, 0x2a, 0x94, 0x32, 0x34},
        .klen = 24,
        .iv = {0x84, 0x32, 0xa5, 0x46, 0x2a, 0x16, 0xbc, 0xfb},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x58, 0x8a, 0x36, 0x9d, 0x93, 0xdd, 0x68, 0xbd, 0x51,
                      0x8f, 0xd3, 0x41, 0xba, 0xb4, 0x83, 0x10, 0xea, 0x89,
                      0x14, 0x6a, 0xf4, 0x65, 0xd3, 0x79, 0x27, 0xec, 0x20,
                      0x29, 0x90, 0x86, 0x17, 0x3a, 0xbf, 0x10, 0x6f, 0x94,
                      0xce, 0xd1, 0xa1, 0xda, 0xa0, 0xaf, 0x4d, 0x41, 0x67,
                      0xa2, 0xda, 0xa3, 0x69, 0x6a, 0x5e, 0xcf, 0x03, 0x7d,
                      0xbd, 0x24, 0xee, 0x44, 0x14, 0x5e, 0xc1, 0x86, 0x58,
                      0xaf},
        .plen = 64,
        .ciphertext = {0x2b, 0xfb, 0x5c, 0x7b, 0x2d, 0x9f, 0xbe, 0x4a, 0x51,
                       0x4a, 0x2d, 0x14, 0x0c, 0xaf, 0x8c, 0x46, 0x46, 0xfd,
                       0x13, 0xd9, 0xc2, 0x51, 0xc1, 0x07, 0x3d, 0x07, 0xf8,
                       0x95, 0xea, 0x99, 0xec, 0x90, 0xf4, 0x5a, 0xbb, 0xd4,
                       0x0b, 0xf4, 0x0f, 0xc7, 0x85, 0xaa, 0x9f, 0x9f, 0xf5,
                       0xb9, 0x25, 0xcd, 0x29, 0xdb, 0x70, 0xe5, 0x14, 0x9c,
                       0x16, 0x28, 0x79, 0xde, 0xd9, 0x45, 0xa7, 0x78, 0x09,
                       0xc8},
        .clen = 64,
        .num_chunks = 4,
        .chunks = {32, 0, -1, 32},
    }, {                // #8
        .key = {0x8c, 0x37, 0xbc, 0x49, 0x32, 0x01, 0x64, 0x79, 0xf1,
                0x97, 0xb0, 0xf7, 0x57, 0xcb, 0x16, 0xf1, 0x46, 0x54,
                0xfb, 0x49, 0x92, 0x46, 0xda, 0xba},
        .klen = 24,
        .iv = {0xfd, 0xee, 0x51, 0xb0, 0x6b, 0x61, 0xfc, 0xe8},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xc1, 0x55, 0x07, 0xd7, 0x25, 0xaf, 0xa1, 0x47, 0x55,
                      0xda, 0x2c, 0xc3, 0xcd, 0xf5, 0x9a, 0x83, 0xad, 0x28,
                      0xbd, 0x20, 0x05, 0xdd, 0xaf, 0x5e, 0x50, 0x2c, 0x71,
                      0x48, 0x3e, 0x96, 0xd2, 0x83, 0xb7, 0xf7, 0x92, 0xdc,
                      0x1c, 0xb6, 0xc8, 0x23, 0xd0, 0x3f, 0x81, 0x3c, 0xf7,
                      0xb3, 0xaa, 0x73, 0x7d, 0xdd, 0x30, 0x10, 0x5a, 0x69,
                      0x65, 0x2b, 0x15, 0x50, 0xa6, 0xc2, 0x87, 0x97, 0x3b,
                      0x09, 0xe2, 0x39, 0x7e, 0x01, 0xf2, 0x62, 0x3f, 0x4c},
        .plen = 72,
        .ciphertext = {0xe5, 0x6f, 0x5d, 0x07, 0x80, 0x60, 0x61, 0xcd, 0xa9,
                       0xcc, 0x7a, 0x83, 0x09, 0x64, 0xc1, 0x64, 0x7e, 0x92,
                       0x02, 0x3d, 0xb2, 0xb6, 0x8d, 0xd8, 0x4a, 0xa5, 0x48,
                       0xf4, 0x86, 0xcc, 0xc7, 0x79, 0xd2, 0x7d, 0xfa, 0xef,
                       0x17, 0x1c, 0x52, 0xc6, 0x32, 0x1f, 0x9f, 0x47, 0x73,
                       0x0d, 0x2a, 0x8e, 0x10, 0x9b, 0x67, 0xad, 0x57, 0x59,
                       0xd5, 0x62, 0x44, 0x06, 0x67, 0x69, 0xe4, 0x3c, 0x6c,
                       0x11, 0xe9, 0x05, 0xa7, 0x37, 0x2a, 0x81, 0x45, 0x6b},
        .clen = 72,
        .num_chunks = 5,
        .chunks = {5, 20, -1, 17, 30},
    }, {                // #9
        .key = {0x4f, 0x52, 0x26, 0x01, 0x9d, 0xb3, 0x6b, 0x40, 0x46,
                0x43, 0x92, 0xc8, 0xe5, 0xc4, 0x2a, 0xd9, 0x5e, 0x20,
                0x32, 0x26, 0x3d, 0xce, 0x8f, 0x20},
        .klen = 24,
        .iv = {0xee, 0x24, 0xf2, 0xe7, 0xd6, 0xd1, 0xf6, 0xf9},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x6a, 0x9a, 0xaf, 0xb3, 0x4b, 0x4d, 0xd8, 0xad, 0x40,
                      0x07, 0x21, 0xb5, 0xc2, 0xee, 0x62, 0x50, 0x15, 0xc8,
                      0x9a, 0x92, 0xcb, 0x6d, 0x92, 0x35, 0x8b, 0x7c, 0xbd,
                      0xe8, 0x10, 0x4e, 0x63, 0xe9, 0x74, 0x28, 0xfe, 0xd6,
                      0xaf, 0xb5, 0xe5, 0xc1, 0x5b, 0xb3, 0x27, 0x82, 0xba,
                      0xec, 0xba, 0x51, 0xf9, 0x68, 0xd5, 0x94, 0xb6, 0xe1,
                      0x9a, 0xf5, 0xc1, 0x77, 0xf8, 0xb8, 0x1f, 0xa5, 0xd0,
                      0x4b, 0x55, 0x74, 0x56, 0x1f, 0xe3, 0xee, 0xbb, 0xb7,
                      0x0e, 0xd6, 0x09, 0xff, 0xe2, 0x96, 0x4e, 0xde},
        .plen = 80,
        .ciphertext = {0x34, 0x9d, 0xdf, 0x37, 0xbf, 0xa1, 0x25, 0x30, 0x4f,
                       0x91, 0x03, 0x5a, 0xa0, 0x2b, 0x5f, 0xd5, 0xc4, 0x9c,
                       0x6f, 0xd2, 0x44, 0x37, 0x9c, 0x43, 0xcf, 0x7e, 0xbe,
                       0xca, 0xd4, 0x89, 0xe3, 0x8a, 0xed, 0xeb, 0x74, 0x90,
                       0xa7, 0x7b, 0xb1, 0x77, 0x7d, 0x62, 0x8f, 0x01, 0x8c,
                       0x86, 0x22, 0x2e, 0xbf, 0x4d, 0x12, 0x48, 0x3f, 0x97,
                       0x9d, 0xdf, 0x13, 0x6f, 0x35, 0x15, 0x1c, 0x56, 0x5c,
                       0x54, 0xf8, 0x6f, 0x24, 0x59, 0xad, 0xc7, 0xb1, 0x66,
                       0x80, 0x66, 0xe6, 0x9f, 0xc2, 0x66, 0x17, 0x01},
        .clen = 80,
        .num_chunks = 2,
        .chunks = {40, 40},
    }
};


/** http://csrc.nist.gov/groups/STM/cavp/documents/des/tdesmmt.zip
    TOFBMMT3.rsp
**/
static struct des3_test_vector des3_ofb64_tv[] = {
    {                   // #0
        .key = {0xe9, 0x61, 0x54, 0xa2, 0xa8, 0x75, 0x5b, 0xfd, 0x2c,
                0x51, 0xd6, 0xa1, 0x97, 0x67, 0xd5, 0x26, 0x43, 0x5d,
                0x19, 0xb6, 0x98, 0x2c, 0x6d, 0xa7},
        .klen = 24,
        .iv = {0xd9, 0xb4, 0x23, 0x43, 0x4c, 0xea, 0x15, 0xf3},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x31, 0x93, 0xf8, 0xd9, 0xf6, 0x90, 0x61, 0xbf},
        .plen = 8,
        .ciphertext = {0x0a, 0x7b, 0x1d, 0xa0, 0x7c, 0x99, 0xa3, 0xdd},
        .clen = 8,
    }, {                // #1
        .key = {0x83, 0x86, 0xc4, 0x64, 0xef, 0xbf, 0xf1, 0xa7, 0x1c,
                0xa8, 0x4c, 0xbf, 0xbf, 0x76, 0x0e, 0x32, 0x2a, 0x07,
                0x8a, 0x2f, 0x6d, 0x1f, 0xf2, 0x31},
        .klen = 24,
        .iv = {0x15, 0xed, 0xcd, 0xe1, 0xbd, 0x53, 0xc4, 0xc5},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x32, 0x7d, 0xa4, 0x0b, 0x70, 0xe3, 0x56, 0x86, 0x8d,
                      0x22, 0x15, 0x1c, 0x96, 0xe7, 0x65, 0x8e},
        .plen = 16,
        .ciphertext = {0xda, 0x19, 0x0d, 0x06, 0xdb, 0x62, 0x69, 0xef, 0xa0,
                       0x6a, 0xe7, 0x2a, 0x79, 0xb9, 0x86, 0xb1},
        .clen = 16,
    }, {                // #2
        .key = {0xb9, 0x20, 0x9e, 0xbc, 0x8c, 0xce, 0x62, 0x40, 0x16,
                0xf1, 0x7a, 0x43, 0xa2, 0x8a, 0x25, 0x15, 0x1a, 0xab,
                0x8c, 0xab, 0xb0, 0x6b, 0x08, 0x3b},
        .klen = 24,
        .iv = {0x2c, 0x95, 0x82, 0x73, 0xea, 0x54, 0x96, 0x7f},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x5e, 0x26, 0x18, 0x8f, 0xb4, 0x5d, 0x09, 0x3f, 0x60,
                      0x67, 0xff, 0x5c, 0x4c, 0x14, 0xe0, 0x43, 0x35, 0xc2,
                      0xdd, 0x74, 0x67, 0x19, 0x53, 0xe9},
        .plen = 24,
        .ciphertext = {0x49, 0x2a, 0x36, 0xf8, 0x13, 0xd1, 0x99, 0xd3, 0x11,
                       0x21, 0xe5, 0x37, 0x9a, 0xf9, 0xa8, 0x20, 0x51, 0x90,
                       0xf4, 0x38, 0xbb, 0x18, 0x03, 0x06},
        .clen = 24,
        .num_chunks = 3,
        .chunks = {12, 0, 12},
    }, {                // #3
        .key = {0xbf, 0x13, 0x92, 0x07, 0x86, 0x6b, 0xb6, 0x5d, 0xe9,
                0x76, 0xa8, 0x70, 0x2f, 0x23, 0x64, 0x34, 0xb3, 0xdc,
                0xc8, 0x37, 0x08, 0x23, 0x52, 0xc2},
        .klen = 24,
        .iv = {0x4c, 0xd5, 0xe1, 0xf5, 0x82, 0x2b, 0x32, 0x4c},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xaa, 0xed, 0x11, 0xce, 0x4c, 0x1e, 0x77, 0xee, 0x37,
                      0xbc, 0xbc, 0x01, 0x98, 0x14, 0x88, 0x60, 0x6d, 0xf7,
                      0xf8, 0x06, 0xde, 0xbe, 0x93, 0xaa, 0xd1, 0x8a, 0x80,
                      0x2b, 0xb9, 0xd1, 0xf6, 0x31},
        .plen = 32,
        .ciphertext = {0x84, 0xab, 0xa2, 0x7a, 0xf1, 0x2f, 0x6c, 0xf9, 0x6e,
                       0x1d, 0xc8, 0x85, 0xb2, 0x79, 0xda, 0xb0, 0xba, 0x88,
                       0x7c, 0x96, 0xb3, 0xbb, 0x0d, 0x29, 0x0d, 0x54, 0x0f,
                       0xa8, 0x96, 0x6a, 0x6e, 0x2d},
        .clen = 32,
        .num_chunks = 4,
        .chunks = {8, 8, 8, 8},
    }, {                // #4
        .key = {0x89, 0xb0, 0x9e, 0x9b, 0x8c, 0x73, 0x37, 0x79, 0xe5,
                0xcd, 0x01, 0x94, 0x83, 0x20, 0xb3, 0xb0, 0x75, 0xc4,
                0x3e, 0x3d, 0xe6, 0x32, 0xe3, 0xc4},
        .klen = 24,
        .iv = {0x37, 0x61, 0x25, 0x03, 0x1f, 0x18, 0xf6, 0xf0},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x88, 0x2c, 0x7a, 0x11, 0x39, 0xa5, 0xa8, 0xfd, 0x56,
                      0xf1, 0x1d, 0xd0, 0x86, 0xa7, 0x65, 0xf7, 0x52, 0xc1,
                      0xe5, 0x54, 0xe3, 0x4d, 0xbe, 0x92, 0x0d, 0x65, 0xf4,
                      0xb6, 0x8d, 0x60, 0xd0, 0xf7, 0x8f, 0x4a, 0x82, 0xd0,
                      0x74, 0x92, 0x0b, 0x53},
        .plen = 40,
        .ciphertext = {0xe8, 0xd9, 0x04, 0x39, 0xe3, 0x06, 0xc3, 0xe3, 0xe9,
                       0x46, 0x00, 0x39, 0xfd, 0x21, 0x4f, 0x00, 0xf7, 0xef,
                       0xfc, 0x0b, 0x59, 0xb6, 0x01, 0x93, 0xe6, 0x1c, 0x6e,
                       0x28, 0xa9, 0xf4, 0xfb, 0xd3, 0x39, 0x25, 0x23, 0x1e,
                       0x48, 0x8a, 0x52, 0xd2},
        .clen = 40,
        .num_chunks = 3,
        .chunks = {20, -1, 20},
    }, {                // #5
        .key = {0xa4, 0x40, 0xba, 0x91, 0xba, 0xe5, 0x62, 0x19, 0x1f,
                0x97, 0xdf, 0x58, 0xae, 0x58, 0x7c, 0x19, 0x89, 0x75,
                0xd3, 0x45, 0x73, 0xcb, 0xcb, 0xa4},
        .klen = 24,
        .iv = {0x0b, 0x39, 0x33, 0x67, 0x57, 0x11, 0xb1, 0x15},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x80, 0xee, 0x89, 0xf4, 0x2d, 0x88, 0x11, 0x9b, 0xbc,
                      0x87, 0x4f, 0x70, 0xd3, 0xc7, 0x3e, 0xd2, 0x5d, 0x59,
                      0x3f, 0xf1, 0xdc, 0x83, 0x0b, 0xee, 0x00, 0x07, 0xec,
                      0x3e, 0x78, 0xa8, 0x2d, 0x50, 0x08, 0xeb, 0xce, 0x6f,
                      0xdc, 0x1b, 0x7d, 0xe9, 0x1c, 0x8f, 0xc2, 0x71, 0xc0,
                      0x30, 0x7a, 0xc2},
        .plen = 48,
        .ciphertext = {0xbb, 0x2b, 0x2f, 0xb1, 0x88, 0xba, 0x69, 0xda, 0x7c,
                       0xc5, 0x2b, 0x01, 0x60, 0x7d, 0x3d, 0x15, 0x42, 0x24,
                       0x9d, 0x0b, 0x11, 0xec, 0x27, 0xcc, 0x6c, 0x6f, 0xc7,
                       0x3f, 0x0f, 0x28, 0x23, 0xe0, 0xae, 0xea, 0xc4, 0x8e,
                       0x30, 0xc6, 0x66, 0x74, 0x02, 0x02, 0xfe, 0x93, 0x70,
                       0xa5, 0x19, 0x0d},
        .clen = 48,
    }, {                // #6
        .key = {0x64, 0x13, 0x20, 0x94, 0x8a, 0x29, 0x85, 0xc2, 0xec,
                0xec, 0x0d, 0xfb, 0x34, 0xb9, 0x04, 0x45, 0x75, 0xfb,
                0x2a, 0x0b, 0x10, 0x3b, 0x75, 0x51},
        .klen = 24,
        .iv = {0x9a, 0xc0, 0x63, 0x68, 0xfa, 0x0e, 0x64, 0xe0},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xea, 0x00, 0xb3, 0x66, 0xd3, 0x2d, 0xae, 0x2d, 0x9c,
                      0xba, 0x94, 0x15, 0x54, 0x84, 0x9c, 0x0f, 0x76, 0x2e,
                      0xac, 0xf3, 0xc9, 0x73, 0x64, 0x75, 0x85, 0xba, 0x5e,
                      0x56, 0xb7, 0xeb, 0xb4, 0xd7, 0xe6, 0x82, 0xc3, 0x73,
                      0xf7, 0xe2, 0x2f, 0xcd, 0x4e, 0xeb, 0x28, 0xe8, 0x36,
                      0xce, 0xeb, 0x96, 0x83, 0x20, 0x8b, 0x33, 0x76, 0x2e,
                      0xf1, 0x92},
        .plen = 56,
        .ciphertext = {0xb3, 0x19, 0x40, 0xc4, 0xdd, 0x1d, 0x84, 0xfd, 0x6f,
                       0xda, 0x95, 0x3c, 0x11, 0x7f, 0x1f, 0x20, 0xe5, 0x80,
                       0xc2, 0xd1, 0x8f, 0x4e, 0xfe, 0x39, 0xfc, 0x15, 0xe4,
                       0xc5, 0x5a, 0xee, 0x88, 0x6a, 0x65, 0x15, 0x5b, 0x4b,
                       0x43, 0x4a, 0xb7, 0x29, 0xeb, 0x25, 0xd8, 0x9c, 0x69,
                       0xb4, 0xd5, 0xc9, 0x3a, 0x66, 0x25, 0xf8, 0x80, 0x32,
                       0xae, 0xe3},
        .clen = 56,
        .num_chunks = 4,
        .chunks = {16, 16, 16, 8},
    }, {                // #7
        .key = {0x2c, 0xfb, 0xbc, 0x62, 0x5b, 0xad, 0x3e, 0x54, 0xe6,
                0x01, 0xf2, 0x52, 0xc4, 0x20, 0xa8, 0x34, 0x13, 0xe9,
                0x2a, 0x46, 0xc1, 0xa7, 0xe5, 0x86},
        .klen = 24,
        .iv = {0xba, 0xda, 0x00, 0xb1, 0x23, 0x6c, 0x22, 0xfa},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x70, 0xa8, 0x94, 0x85, 0x40, 0x56, 0x75, 0x66, 0x54,
                      0x36, 0xb9, 0x27, 0xad, 0x9b, 0x7e, 0x3d, 0x7c, 0x3e,
                      0xa8, 0xcf, 0xda, 0x25, 0x30, 0xbd, 0xb9, 0xc9, 0xbe,
                      0xe2, 0x1b, 0x77, 0x7b, 0x96, 0x82, 0xf7, 0x20, 0x77,
                      0xa7, 0x6c, 0x54, 0x6a, 0x6e, 0x7b, 0x34, 0xdc, 0x1a,
                      0xaf, 0x33, 0x4e, 0xbd, 0x1c, 0x24, 0x10, 0xf3, 0x43,
                      0x95, 0x4c, 0xd0, 0x36, 0x5f, 0x47, 0x17, 0xfd, 0x7b,
                      0xe4},
        .plen = 64,
        .ciphertext = {0x51, 0xde, 0x0e, 0x2f, 0xee, 0x09, 0xc6, 0x0d, 0xa1,
                       0x97, 0x13, 0xa6, 0xcd, 0x1b, 0x82, 0x1f, 0xb2, 0xa2,
                       0x3f, 0xd7, 0xb4, 0x80, 0x76, 0xd2, 0xe0, 0x9d, 0xd7,
                       0x5f, 0x69, 0x2a, 0x29, 0x5f, 0x1c, 0xda, 0x91, 0xe2,
                       0x3e, 0x83, 0x52, 0x26, 0x6e, 0x1e, 0x7c, 0x85, 0x38,
                       0x61, 0xe8, 0xf4, 0x0c, 0xf3, 0x72, 0x56, 0x0c, 0x94,
                       0x92, 0x4e, 0xd9, 0xfd, 0x5e, 0xdc, 0x49, 0x7b, 0xf5,
                       0xeb},
        .clen = 64,
        .num_chunks = 4,
        .chunks = {32, 0, -1, 32},
    }, {                // #8
        .key = {0xb3, 0xae, 0x76, 0x13, 0xc4, 0xd0, 0x3b, 0x7a, 0x23,
                0x58, 0x8a, 0x6e, 0x7c, 0x9b, 0x1f, 0xe5, 0x6e, 0x43,
                0x58, 0x8f, 0xf7, 0x4f, 0x8a, 0x32},
        .klen = 24,
        .iv = {0x89, 0x62, 0x65, 0xe4, 0x3e, 0x5a, 0x7b, 0xea},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0x01, 0xa7, 0x68, 0xf8, 0x00, 0x2e, 0x80, 0xe8, 0xc7,
                      0x5b, 0xbc, 0x2d, 0x48, 0x1d, 0xdb, 0xdc, 0x5b, 0xc6,
                      0x7d, 0x61, 0xc9, 0xc7, 0xdb, 0x66, 0xa3, 0xbb, 0xb4,
                      0xf7, 0xd9, 0xa8, 0x8b, 0xe3, 0x07, 0x81, 0x0f, 0x65,
                      0x92, 0xef, 0xdb, 0x03, 0xf4, 0x50, 0x5f, 0xb7, 0x54,
                      0x01, 0x1e, 0x4d, 0x5a, 0xd8, 0x1b, 0x0c, 0x41, 0xf5,
                      0x8d, 0x98, 0x2e, 0xbb, 0xda, 0x17, 0xc0, 0x19, 0xc3,
                      0x93, 0xe8, 0x0e, 0xdc, 0x9f, 0xfb, 0x6c, 0x40, 0x3e},
        .plen = 72,
        .ciphertext = {0x92, 0xfb, 0xd9, 0x0d, 0xc8, 0x61, 0x5a, 0xef, 0x01,
                       0x6a, 0x1a, 0x43, 0x3f, 0xd8, 0xc5, 0x6a, 0xfa, 0x8d,
                       0x25, 0x68, 0x25, 0xae, 0x50, 0xad, 0x85, 0x59, 0xff,
                       0x5d, 0xcc, 0xfc, 0xe4, 0x2e, 0x56, 0xc3, 0xef, 0x8d,
                       0x7e, 0xbf, 0x6a, 0x75, 0x97, 0xfd, 0x0e, 0x95, 0x3b,
                       0x0c, 0x00, 0xa0, 0x22, 0xbc, 0xaa, 0xd9, 0x2c, 0xbc,
                       0x17, 0x2c, 0xc7, 0x9e, 0xf5, 0xf9, 0x41, 0xb6, 0xe0,
                       0x7e, 0xc7, 0xe1, 0xc2, 0x71, 0x6a, 0x93, 0x9a, 0x7d},
        .clen = 72,
        .num_chunks = 5,
        .chunks = {5, 20, -1, 17, 30},
    }, {                // #9
        .key = {0x13, 0xf1, 0x0b, 0xc8, 0xd5, 0x98, 0x7c, 0x6d, 0x97,
                0x0d, 0xe6, 0xd6, 0x01, 0x52, 0xdc, 0xe6, 0xda, 0x3b,
                0x3b, 0x04, 0xb5, 0x2c, 0x20, 0x5d},
        .klen = 24,
        .iv = {0x9d, 0x99, 0x5c, 0x62, 0x11, 0xb4, 0xec, 0x4d},
        .ivlen = DES3_IV_SIZE,
        .plaintext = {0xb6, 0x94, 0xeb, 0x44, 0x89, 0x7b, 0x59, 0xd3, 0x97,
                      0x65, 0x9d, 0x3b, 0x00, 0x81, 0x80, 0x35, 0x6f, 0x64,
                      0x8a, 0x34, 0x54, 0x46, 0x6a, 0x3e, 0x74, 0x6f, 0x99,
                      0xfb, 0xa4, 0xec, 0x3b, 0x0c, 0xd1, 0x86, 0x46, 0x9d,
                      0x44, 0xf5, 0xe5, 0x3e, 0x24, 0x50, 0x1f, 0xee, 0x3d,
                      0xc8, 0xb4, 0x79, 0xff, 0x96, 0x47, 0x05, 0xca, 0x32,
                      0xc2, 0xe7, 0x69, 0xc6, 0xf7, 0xbb, 0xdc, 0xe2, 0x83,
                      0xb3, 0x5a, 0x71, 0xf8, 0x2f, 0x96, 0xec, 0x3b, 0x8e,
                      0x42, 0xcd, 0x2a, 0x44, 0xd1, 0x47, 0x82, 0x2d},
        .plen = 80,
        .ciphertext = {0xa5, 0xf7, 0xee, 0xe8, 0x26, 0xee, 0x1e, 0x8b, 0xcb,
                       0x77, 0xd2, 0xef, 0x6f, 0xfb, 0x56, 0xc9, 0xa3, 0xa5,
                       0x43, 0xc4, 0xfd, 0x89, 0xa0, 0x6b, 0x4f, 0x26, 0x5b,
                       0xda, 0xab, 0x1c, 0x2c, 0x03, 0xb6, 0x6c, 0x51, 0x43,
                       0x9a, 0xd4, 0x14, 0xf5, 0x2a, 0x4f, 0xcb, 0xac, 0xea,
                       0x83, 0x3a, 0x33, 0xd1, 0x61, 0x0f, 0x80, 0xf9, 0x33,
                       0xab, 0x90, 0x0e, 0xc0, 0x47, 0x33, 0x4f, 0xd3, 0x3f,
                       0xaf, 0xc5, 0xc2, 0x59, 0x17, 0x40, 0x9f, 0x01, 0xe0,
                       0x47, 0xbb, 0x83, 0x28, 0x35, 0x2f, 0x7c, 0x45},
        .clen = 80,
        .num_chunks = 2,
        .chunks = {40, 40},
    }
};

/**
 * NIST Special Publication 800-38B
 * https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/
 *                          CAVP-TESTING-BLOCK-CIPHER-MODES#CMAC
 **/
static struct cmac_test_vector des2_cmac_tv[] = {
    {   // 0 (0)
        .key = { 0xdc,0x04,0x73,0x68,0x02,0x32,0x52,0xd9,0x0e,
                 0xbc,0xcd,0xc2,0x16,0x61,0xe3,0xab },
        .klen = 16,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x72,0x61,0x9d,0xc7,0xda },
        .tlen = 5,
    }, {     // 1 (1)
        .key = { 0x76,0xfd,0x92,0x07,0x04,0x29,0xd3,0x3d,0x29,
                 0x02,0x9b,0x46,0x1a,0x8a,0x2f,0x57 },
        .klen = 16,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x63,0x92,0x98,0x77,0xf1 },
        .tlen = 5,
        .chunks_msg = { 0, 0 },
        .num_chunks_message = 2,
    }, {     // 2 (8)
        .key = { 0x3d,0x4c,0xf2,0x64,0x13,0x76,0xae,0xc2,0x13,
                 0xa8,0x26,0x5d,0x52,0x9b,0x7a,0x20 },
        .klen = 16,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0xcd,0xd2,0xe2,0xc8,0x49,0x31,0xc1 },
        .tlen = 7,
    }, {     // 3 (16)
        .key = { 0xd3,0x94,0x8a,0xa4,0x52,0x7a,0xe5,0xba,0x1a,
                 0x54,0x32,0xe9,0x08,0x54,0x10,0xdc },
        .klen = 16,
        .msg = { 0x39,0x2e,0x55,0x79,0x30,0xd3,0x04,0xd5 },
        .mlen = 8,
        .mac = { 0x0c,0x01,0x23,0x3c,0x9a },
        .tlen = 5,
    }, {     // 4 (17)
        .key = { 0xe6,0x2f,0x08,0xd0,0x26,0x04,0xfd,0xcb,0x2c,
                 0x16,0x6d,0x62,0x0e,0x10,0xe5,0x07 },
        .klen = 16,
        .msg = { 0x95,0x84,0x8f,0x94,0x05,0x62,0xae,0x51 },
        .mlen = 8,
        .mac = { 0x87,0x6d,0x6a,0x38,0x69 },
        .tlen = 5,
        .chunks_msg = { 4, 4 },
        .num_chunks_message = 2,
    }, {     // 5 (32)
        .key = { 0x5d,0xb9,0xb0,0xd6,0xb5,0xd9,0x25,0x46,0x86,
                 0x1c,0xbf,0x5b,0xf8,0xf2,0xf1,0xe6 },
        .klen = 16,
        .msg = { 0xeb,0x00,0x22,0x4e,0x03,0x65,0x2e,0xb1,0x84,
                 0x2a,0x47,0x73,0x54,0xae,0x19,0x1f },
        .mlen = 16,
        .mac = { 0x61,0x75,0xa0,0xf3,0x6f },
        .tlen = 5,
    }, {     // 6 (33)
        .key = { 0x51,0x1c,0x68,0x25,0xfb,0x97,0xda,0x70,0x70,
                 0xb5,0xd5,0x19,0x0b,0xec,0x49,0x9e },
        .klen = 16,
        .msg = { 0x25,0xd8,0x7c,0x44,0x02,0xf3,0x4e,0xa5,0x9b,
                 0x38,0x98,0xbe,0xaa,0x40,0x1e,0x96 },
        .mlen = 16,
        .mac = { 0x18,0x76,0xc6,0xa2,0x72 },
        .tlen = 5,
        .chunks_msg = { 5, 5, 6 },
        .num_chunks_message = 3,
    }, {     // 7 (40)
        .key = { 0xb9,0x02,0x6e,0xba,0xda,0x62,0x38,0xd9,0x20,
                 0x7f,0x07,0x3e,0x1c,0x1c,0xb6,0xd5 },
        .klen = 16,
        .msg = { 0x23,0x32,0x82,0xca,0x3e,0x2a,0xfc,0x6f,0x4c,
                 0x47,0x13,0x26,0x75,0x1e,0xd2,0x5a },
        .mlen = 16,
        .mac = { 0xfe,0x6a,0xc1,0xbf,0x3a,0x15,0x89 },
        .tlen = 7,
    }, {     // 8 (48)
        .key = { 0xc2,0x75,0x57,0x76,0xb0,0x6e,0x67,0x97,0x26,
                 0x1a,0xef,0x15,0xa1,0x38,0x38,0x8f },
        .klen = 16,
        .msg = { 0x30,0x94,0xb1,0x6e,0x28,0x99,0x1c },
        .mlen = 7,
        .mac = { 0x15,0xa3,0x7f,0x1a,0x7b },
        .tlen = 5,
    }, {     // 9 (49)
        .key = { 0xf8,0xf1,0x49,0x2c,0xb5,0xae,0x70,0xcd,0x5e,
                 0x97,0x94,0x19,0xc8,0xfb,0xc7,0xad },
        .klen = 16,
        .msg = { 0x53,0x53,0x4f,0x17,0x77,0x5c,0x0e },
        .mlen = 7,
        .mac = { 0x2c,0x2a,0x55,0x32,0x96 },
        .tlen = 5,
        .chunks_msg = { 3, 4 },
        .num_chunks_message = 2,
    }, {     // 10 (64)
        .key = { 0x04,0x13,0xb3,0x67,0xc2,0xbf,0x49,0x1f,0x5e,
                 0xea,0xb5,0x40,0x43,0xfe,0x7f,0x16 },
        .klen = 16,
        .msg = { 0x70,0xb2,0xdb,0xfe,0x6d,0x89,0x2a,0xd8,0xd0,
                 0xd0,0x48,0xec,0x8a,0x04,0xc9 },
        .mlen = 15,
        .mac = { 0xd5,0x25,0xaa,0x36,0x43 },
        .tlen = 5,
    }
};

/**
 * NIST Special Publication 800-38B
 * https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/
 *                          CAVP-TESTING-BLOCK-CIPHER-MODES#CMAC
 **/
static struct cmac_test_vector des3_cmac_tv[] = {
    {   // 0 (0)
        .key = { 0xc4,0x79,0xf8,0x13,0xad,0x1a,0x45,0xd5,0xdc,
                 0x43,0x45,0x9d,0xa4,0xc8,0x5e,0x85,0x1c,0xda,
                 0x51,0x8a,0xf8,0x86,0xbf,0x1f },
        .klen = 24,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0xfe,0xa0,0x1c,0x84,0xa7 },
        .tlen = 5,
    }, {     // 1 (1)
        .key = { 0x73,0x13,0x31,0x86,0x67,0x54,0x58,0x8f,0xd3,
                 0x32,0xef,0x51,0xe0,0xce,0x19,0x25,0xd5,0x86,
                 0xcb,0xa7,0x04,0x40,0xf4,0x4f },
        .klen = 24,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x16,0x4b,0xdb,0x85,0x82 },
        .tlen = 5,
        .chunks_msg = { 0, 0 },
        .num_chunks_message = 2,
    }, {     // 2 (8)
        .key = { 0xf8,0x7c,0x4c,0x3d,0xf7,0x1f,0xd9,0x8c,0xe5,
                 0x32,0xf4,0x1f,0xfe,0x31,0x58,0x20,0x51,0x4f,
                 0xad,0x7c,0x7a,0xae,0x2f,0x46 },
        .klen = 24,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x19,0xfb,0xd3,0xdc,0x15,0x4b,0xc1,0x39 },
        .tlen = 8,
    }, {     // 3 (16)
        .key = { 0xd9,0x61,0xb6,0xd5,0x57,0xae,0xba,0x4a,0x0e,
                 0xc1,0xbf,0xf2,0x79,0x7a,0xdf,0xda,0x04,0x70,
                 0x61,0x68,0x83,0x92,0x76,0xfe },
        .klen = 24,
        .msg = { 0xd7,0x68,0x97,0x5c,0x45,0x4f,0x93,0x90,0x6c,
                 0xd3,0x5e,0x6a,0x6d,0x5f,0x44,0x53 },
        .mlen = 16,
        .mac = { 0x96,0x72,0x52,0xe4,0xf3 },
        .tlen = 5,
    }, {     // 4 (17)
        .key = { 0xdf,0x10,0xfb,0x6b,0x76,0x6e,0xab,0x34,0xd5,
                 0xcb,0x75,0x79,0x58,0x2f,0xb6,0xa8,0x02,0x5e,
                 0x45,0x9b,0xb3,0x8f,0x43,0x37 },
        .klen = 24,
        .msg = { 0x35,0x07,0x23,0xc4,0xf1,0x67,0x7f,0x7b,0xac,
                 0xdb,0x78,0xd1,0x2a,0x22,0xa1,0x9b },
        .mlen = 16,
        .mac = { 0xb4,0xd9,0x1d,0x08,0xe4 },
        .tlen = 5,
        .chunks_msg = { 5, 5, 6 },
        .num_chunks_message = 3,
    }, {     // 5 (24)
        .key = { 0x01,0xc1,0x25,0xc2,0x79,0xcb,0xe9,0x0d,0xdf,
                 0xc2,0xc1,0xb0,0x4c,0xa8,0xae,0x49,0x16,0x16,
                 0xad,0x64,0x43,0x3d,0x83,0x4a },
        .klen = 24,
        .msg = { 0x0a,0x25,0x8b,0x67,0x19,0xe2,0xe0,0x07,0x9b,
                 0x78,0x0a,0x08,0x63,0xff,0x11,0x5b },
        .mlen = 16,
        .mac = { 0xa7,0xbc,0xbe,0x72,0x0a,0xc2,0xa7,0xe5 },
        .tlen = 8,
    }, {     // 6 (32)
        .key = { 0xe6,0xe0,0xd5,0xfd,0x61,0xda,0xe3,0x15,0x5b,
                 0x32,0x98,0x0e,0x70,0xe6,0xd6,0x7f,0xd9,0x9d,
                 0x8f,0xf4,0x34,0x8f,0xe9,0x8f },
        .klen = 24,
        .msg = { 0xe4,0x3b,0xd4,0x9c,0x39,0x76,0xaf,0xa7,0x8a,
                 0x71,0x38,0xdd,0xab,0xfa,0x13,0x2a,0x72,0xe7,
                 0xd1,0xa4,0x22,0xb4,0x0a,0x58,0x2a,0x6d,0xac,
                 0x4a,0x10,0xce,0x92,0xb7 },
        .mlen = 32,
        .mac = { 0xd4,0x4f,0x71,0x0b,0x81 },
        .tlen = 5,
    }, {     // 7 (33)
        .key = { 0x20,0x40,0x40,0xce,0x67,0x10,0x2c,0x3b,0x31,
                 0x29,0x43,0x5d,0x8a,0x85,0x3d,0x91,0x6d,0x37,
                 0x8a,0x2c,0x68,0x76,0x70,0xd3 },
        .klen = 24,
        .msg = { 0x9b,0x75,0x8b,0xc9,0x3d,0x48,0x25,0x05,0xb1,
                 0xb7,0xb6,0xf1,0x47,0x20,0x3a,0x72,0xe8,0x1b,
                 0xf1,0x90,0x04,0xa5,0xfa,0xc1,0xcb,0x7e,0x23,
                 0x6a,0xa0,0xcf,0x00,0xa3 },
        .mlen = 32,
        .mac = { 0x49,0x00,0x1a,0x2a,0x10 },
        .tlen = 5,
        .chunks_msg = { 10, 10, 10, 2 },
        .num_chunks_message = 4,
    }, {     // 8 (48)
        .key = { 0xef,0x0d,0x5e,0x94,0xe5,0xc8,0x89,0x34,0xbf,
                 0xcb,0xb5,0x80,0x67,0x08,0x97,0x4a,0x70,0xc1,
                 0x4a,0x70,0x31,0x02,0x3d,0xb0 },
        .klen = 24,
        .msg = { 0x8d,0x32,0x44,0xbd,0x36,0x4e,0xea,0x1f,0x53 },
        .mlen = 9,
        .mac = { 0x65,0x52,0x7e,0xeb,0x1b },
        .tlen = 5,
    }, {     // 9 (49)
        .key = { 0xdf,0xd6,0xd0,0xb0,0x58,0x75,0x1f,0xa1,0x7a,
                 0x1f,0xd6,0x73,0x45,0x16,0x3d,0xa1,0x37,0xad,
                 0x25,0x1f,0x3e,0x1f,0x70,0x8a },
        .klen = 24,
        .msg = { 0x9a,0xa7,0xd5,0xfb,0x6b,0xa2,0xd8,0x55,0x75 },
        .mlen = 9,
        .mac = { 0x5c,0x08,0x41,0x49,0x31 },
        .tlen = 5,
        .chunks_msg = { 4, 5 },
        .num_chunks_message = 2,
    }, {     // 10 (56)
        .key = { 0x9e,0x98,0xd0,0x2a,0xc8,0xda,0x0d,0xa8,0x38,
                 0x20,0xda,0x43,0x83,0xce,0x6e,0xea,0x16,0x19,
                 0xa4,0x7f,0xe0,0xa1,0xa1,0x13 },
        .klen = 24,
        .msg = { 0xa1,0xb0,0xd3,0x65,0xea,0x1b,0xfb,0xf0,0x84 },
        .mlen = 9,
        .mac = { 0x9c,0xf6,0xad,0x27,0xde,0xb8,0x4e,0xe7 },
        .tlen = 8,
    }, {     // 11 (64)
        .key = { 0xea,0x5e,0xc8,0x6b,0x61,0x9d,0x7a,0x58,0xbf,
                 0x7c,0x75,0xdf,0xdc,0xa7,0xb6,0x25,0x3d,0xf8,
                 0xd5,0x40,0x73,0x23,0x38,0xb6 },
        .klen = 24,
        .msg = { 0x4c,0xdb,0x3b,0x20,0xd6,0x33,0x8b,0x67,0x96,
                 0x71,0x99,0x23,0xf4,0xeb,0xb8,0x6b,0x74 },
        .mlen = 17,
        .mac = { 0x46,0xe2,0xd2,0xa7,0x8c },
        .tlen = 5,
    }, {     // 12 (65)
        .key = { 0xa1,0xad,0xe6,0x13,0x37,0x43,0x5e,0x3b,0xd3,
                 0xd6,0x43,0x49,0x45,0x4c,0xab,0xe5,0x62,0x45,
                 0x25,0x62,0xc4,0x38,0x34,0xae },
        .klen = 24,
        .msg = { 0x35,0x94,0x21,0xe9,0xf7,0x8c,0xc4,0xa3,0x1f,
                 0x4f,0x01,0x99,0x77,0xd7,0xfd,0x29,0x78 },
        .mlen = 17,
        .mac = { 0xff,0x99,0xe6,0xcc,0x7c },
        .tlen = 5,
         .chunks_msg = { 7, 10 },
         .num_chunks_message = 2,
    }, {     // 13 (80)
        .key = { 0x3e,0x7f,0xf2,0xe9,0x3e,0x01,0x26,0x58,0xbf,
                 0x97,0x79,0x0e,0x49,0x8c,0x23,0xad,0xd3,0x52,
                 0xcd,0x26,0xb3,0x64,0xbf,0x19 },
        .klen = 24,
        .msg = { 0x75,0x55,0x98,0x98,0xf4,0xba,0x03,0xc5,0x5a,
                 0xfc,0x25,0xea,0x91,0xaa,0x61,0xa9,0x3c,0x2f,
                 0x82,0x70,0xa5,0xfa,0x51,0xb6,0xf6,0xdc,0x68,
                 0x81,0xad,0xb1,0x41,0x2c },
        .mlen = 32,
        .mac = { 0xe4,0x21,0xb4,0x30,0x74 },
        .tlen = 5,
    }
};


#define NUM_OF_PUBLISHED_TESTSUITES	5

struct published_test_suite_info published_test_suites[] = {
    {
        .name = "DES3_ECB",
        .tvcount = 10,
        .tv = des3_ecb_tv,
        .size = DES3_BLOCK_SIZE,
        .mechanism = CKM_DES3_ECB,
    }, {
        .name = "DES3_CBC",
        .tvcount = 10,
        .tv = des3_cbc_tv,
        .size = DES3_BLOCK_SIZE,
        .mechanism = CKM_DES3_CBC,
    }, {
        .name = "DES3_CFB8",
        .tvcount = 10,
        .tv = des3_cfb8_tv,
        .size = 1,
        .mechanism = CKM_DES_CFB8,
    }, {
        .name = "DES3_CFB64",
        .tvcount = 10,
        .tv = des3_cfb64_tv,
        .size = DES3_BLOCK_SIZE,
        .mechanism = CKM_DES_CFB64,
    }, {
        .name = "DES3_OFB64",
        .tvcount = 10,
        .tv = des3_ofb64_tv,
        .size = DES3_BLOCK_SIZE,
        .mechanism = CKM_DES_OFB64,
    }
};

#define NUM_OF_GENERATED_TESTSUITES 3

static struct generated_test_suite_info generated_test_suites[] = {
    {
        .name = "DES3_ECB",
        .mech = {CKM_DES3_ECB, 0, 0},
    }, {
        .name = "DES3_CBC",
        .mech = {CKM_DES3_CBC, &des3_cbc_iv, DES3_IV_SIZE},
    }, {
        .name = "DES3_CBC_PAD",
        .mech = {CKM_DES3_CBC_PAD, &des3_cbc_iv, DES3_IV_SIZE},
    }
};

#define NUM_OF_PUBLISHED_CMAC_TESTSUITES 5

struct published_cmac_test_suite_info published_cmac_test_suites[] = {
    {
        .name = "2DES_CMAC_GENERAL",
        .tvcount = 11,
        .tv = des2_cmac_tv,
        .mech = {CKM_DES3_CMAC_GENERAL, 0, 0},
        .key_type = CKK_DES2,
    }, {
        .name = "3DES_CMAC_GENERAL",
        .tvcount = 14,
        .tv = des3_cmac_tv,
        .mech = {CKM_DES3_CMAC_GENERAL, 0, 0},
        .key_type = CKK_DES3,
    }, {
        .name = "2DES_CMAC",
        .tvcount = 11,
        .tv = des2_cmac_tv,
        .mech = {CKM_DES3_CMAC, 0, 0},
        .key_type = CKK_DES2,
    }, {
        .name = "3DES_CMAC",
        .tvcount = 14,
        .tv = des3_cmac_tv,
        .mech = {CKM_DES3_CMAC, 0, 0},
        .key_type = CKK_DES3,
    }, {
        .name = "IBM-CMAC",
        .tvcount = 14,
        .tv = des3_cmac_tv,
        .mech = {CKM_IBM_CMAC, 0, 0},
        .key_type = CKK_DES3,
    }
};
