/*
             Common Public License Version 0.5

             THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF
             THIS COMMON PUBLIC LICENSE ("AGREEMENT"). ANY USE,
             REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
             RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.

             1. DEFINITIONS

             "Contribution" means: 
                   a) in the case of the initial Contributor, the
                   initial code and documentation distributed under
                   this Agreement, and 

                   b) in the case of each subsequent Contributor:
                   i) changes to the Program, and
                   ii) additions to the Program;

                   where such changes and/or additions to the Program
                   originate from and are distributed by that
                   particular Contributor. A Contribution 'originates'
                   from a Contributor if it was added to the Program
                   by such Contributor itself or anyone acting on such
                   Contributor's behalf. Contributions do not include
                   additions to the Program which: (i) are separate
                   modules of software distributed in conjunction with
                   the Program under their own license agreement, and
                   (ii) are not derivative works of the Program.


             "Contributor" means any person or entity that distributes
             the Program.

             "Licensed Patents " mean patent claims licensable by a
             Contributor which are necessarily infringed by the use or
             sale of its Contribution alone or when combined with the
             Program. 

             "Program" means the Contributions distributed in
             accordance with this Agreement.

             "Recipient" means anyone who receives the Program under
             this Agreement, including all Contributors.

             2. GRANT OF RIGHTS

                   a) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free copyright
                   license to reproduce, prepare derivative works of,
                   publicly display, publicly perform, distribute and
                   sublicense the Contribution of such Contributor, if
                   any, and such derivative works, in source code and
                   object code form.

                   b) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free patent
                   license under Licensed Patents to make, use, sell,
                   offer to sell, import and otherwise transfer the
                   Contribution of such Contributor, if any, in source
                   code and object code form. This patent license
                   shall apply to the combination of the Contribution
                   and the Program if, at the time the Contribution is
                   added by the Contributor, such addition of the
                   Contribution causes such combination to be covered
                   by the Licensed Patents. The patent license shall
                   not apply to any other combinations which include
                   the Contribution. No hardware per se is licensed
                   hereunder.

                   c) Recipient understands that although each
                   Contributor grants the licenses to its
                   Contributions set forth herein, no assurances are
                   provided by any Contributor that the Program does
                   not infringe the patent or other intellectual
                   property rights of any other entity. Each
                   Contributor disclaims any liability to Recipient
                   for claims brought by any other entity based on
                   infringement of intellectual property rights or
                   otherwise. As a condition to exercising the rights
                   and licenses granted hereunder, each Recipient
                   hereby assumes sole responsibility to secure any
                   other intellectual property rights needed, if any.

                   For example, if a third party patent license is
                   required to allow Recipient to distribute the
                   Program, it is Recipient's responsibility to
                   acquire that license before distributing the
                   Program.

                   d) Each Contributor represents that to its
                   knowledge it has sufficient copyright rights in its
                   Contribution, if any, to grant the copyright
                   license set forth in this Agreement.

             3. REQUIREMENTS

             A Contributor may choose to distribute the Program in
             object code form under its own license agreement, provided
             that:
                   a) it complies with the terms and conditions of
                   this Agreement; and

                   b) its license agreement:
                   i) effectively disclaims on behalf of all
                   Contributors all warranties and conditions, express
                   and implied, including warranties or conditions of
                   title and non-infringement, and implied warranties
                   or conditions of merchantability and fitness for a
                   particular purpose;

                   ii) effectively excludes on behalf of all
                   Contributors all liability for damages, including
                   direct, indirect, special, incidental and
                   consequential damages, such as lost profits;

                   iii) states that any provisions which differ from
                   this Agreement are offered by that Contributor
                   alone and not by any other party; and

                   iv) states that source code for the Program is
                   available from such Contributor, and informs
                   licensees how to obtain it in a reasonable manner
                   on or through a medium customarily used for
                   software exchange.

             When the Program is made available in source code form:
                   a) it must be made available under this Agreement;
                   and
                   b) a copy of this Agreement must be included with
                   each copy of the Program. 

             Contributors may not remove or alter any copyright notices
             contained within the Program.

             Each Contributor must identify itself as the originator of
             its Contribution, if any, in a manner that reasonably
             allows subsequent Recipients to identify the originator of
             the Contribution. 


             4. COMMERCIAL DISTRIBUTION

             Commercial distributors of software may accept certain
             responsibilities with respect to end users, business
             partners and the like. While this license is intended to
             facilitate the commercial use of the Program, the
             Contributor who includes the Program in a commercial
             product offering should do so in a manner which does not
             create potential liability for other Contributors.
             Therefore, if a Contributor includes the Program in a
             commercial product offering, such Contributor ("Commercial
             Contributor") hereby agrees to defend and indemnify every
             other Contributor ("Indemnified Contributor") against any
             losses, damages and costs (collectively "Losses") arising
             from claims, lawsuits and other legal actions brought by a
             third party against the Indemnified Contributor to the
             extent caused by the acts or omissions of such Commercial
             Contributor in connection with its distribution of the
             Program in a commercial product offering. The obligations
             in this section do not apply to any claims or Losses
             relating to any actual or alleged intellectual property
             infringement. In order to qualify, an Indemnified
             Contributor must: a) promptly notify the Commercial
             Contributor in writing of such claim, and b) allow the
             Commercial Contributor to control, and cooperate with the
             Commercial Contributor in, the defense and any related
             settlement negotiations. The Indemnified Contributor may
             participate in any such claim at its own expense.


             For example, a Contributor might include the Program in a
             commercial product offering, Product X. That Contributor
             is then a Commercial Contributor. If that Commercial
             Contributor then makes performance claims, or offers
             warranties related to Product X, those performance claims
             and warranties are such Commercial Contributor's
             responsibility alone. Under this section, the Commercial
             Contributor would have to defend claims against the other
             Contributors related to those performance claims and
             warranties, and if a court requires any other Contributor
             to pay any damages as a result, the Commercial Contributor
             must pay those damages.


             5. NO WARRANTY

             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE
             PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
             WARRANTIES OR CONDITIONS OF ANY KIND, EITHER EXPRESS OR
             IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
             CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR
             FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is solely
             responsible for determining the appropriateness of using
             and distributing the Program and assumes all risks
             associated with its exercise of rights under this
             Agreement, including but not limited to the risks and
             costs of program errors, compliance with applicable laws,
             damage to or loss of data, programs or equipment, and
             unavailability or interruption of operations. 

             6. DISCLAIMER OF LIABILITY
             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER
             RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY
             FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
             OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
             LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF
             LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
             (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
             OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE
             OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE
             POSSIBILITY OF SUCH DAMAGES.

             7. GENERAL

             If any provision of this Agreement is invalid or
             unenforceable under applicable law, it shall not affect
             the validity or enforceability of the remainder of the
             terms of this Agreement, and without further action by the
             parties hereto, such provision shall be reformed to the
             minimum extent necessary to make such provision valid and
             enforceable.


             If Recipient institutes patent litigation against a
             Contributor with respect to a patent applicable to
             software (including a cross-claim or counterclaim in a
             lawsuit), then any patent licenses granted by that
             Contributor to such Recipient under this Agreement shall
             terminate as of the date such litigation is filed. In
             addition, If Recipient institutes patent litigation
             against any entity (including a cross-claim or
             counterclaim in a lawsuit) alleging that the Program
             itself (excluding combinations of the Program with other
             software or hardware) infringes such Recipient's
             patent(s), then such Recipient's rights granted under
             Section 2(b) shall terminate as of the date such
             litigation is filed.

             All Recipient's rights under this Agreement shall
             terminate if it fails to comply with any of the material
             terms or conditions of this Agreement and does not cure
             such failure in a reasonable period of time after becoming
             aware of such noncompliance. If all Recipient's rights
             under this Agreement terminate, Recipient agrees to cease
             use and distribution of the Program as soon as reasonably
             practicable. However, Recipient's obligations under this
             Agreement and any licenses granted by Recipient relating
             to the Program shall continue and survive. 

             Everyone is permitted to copy and distribute copies of
             this Agreement, but in order to avoid inconsistency the
             Agreement is copyrighted and may only be modified in the
             following manner. The Agreement Steward reserves the right
             to publish new versions (including revisions) of this
             Agreement from time to time. No one other than the
             Agreement Steward has the right to modify this Agreement.

             IBM is the initial Agreement Steward. IBM may assign the
             responsibility to serve as the Agreement Steward to a
             suitable separate entity. Each new version of the
             Agreement will be given a distinguishing version number.
             The Program (including Contributions) may always be
             distributed subject to the version of the Agreement under
             which it was received. In addition, after a new version of
             the Agreement is published, Contributor may elect to
             distribute the Program (including its Contributions) under
             the new version. Except as expressly stated in Sections
             2(a) and 2(b) above, Recipient receives no rights or
             licenses to the intellectual property of any Contributor
             under this Agreement, whether expressly, by implication,
             estoppel or otherwise. All rights in the Program not
             expressly granted under this Agreement are reserved.


             This Agreement is governed by the laws of the State of New
             York and the intellectual property laws of the United
             States of America. No party to this Agreement will bring a
             legal action under this Agreement more than one year after
             the cause of action arose. Each party waives its rights to
             a jury trial in any resulting litigation. 



*/

/* (C) COPYRIGHT International Business Machines Corp. 2001,2002          */

/***************************************************************************
                          Change Log
                          ==========
       4/25/03    Kapil Sood (kapil@corrent.com)
                  Added DH key pair generation and DH shared key derivation
                  functions.
 
 
 
****************************************************************************/


// File:  key_mgr.c
//

#include <pthread.h>
#include <stdlib.h>

#include <string.h>  // for memcmp() et al

#include "pkcs11types.h"
#include "defs.h"
#include "host_defs.h"
#include "h_extern.h"
#include "tok_spec_struct.h"


static CK_BBOOL true = TRUE, false = FALSE;

//
//
CK_RV
key_mgr_generate_key( SESSION           * sess,
                      CK_MECHANISM      * mech,
                      CK_ATTRIBUTE      * pTemplate,
                      CK_ULONG            ulCount,
                      CK_OBJECT_HANDLE  * handle )
{
   OBJECT        * key_obj  = NULL;
   CK_ATTRIBUTE  * attr     = NULL;
   CK_ATTRIBUTE  * new_attr = NULL;
   CK_ULONG        i, keyclass, subclass = 0;
   CK_BBOOL        flag;
   CK_RV           rc;


   if (!sess || !mech || !handle){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }
   if (!pTemplate && (ulCount != 0)){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }
   // it's silly but Cryptoki allows the user to specify the CKA_CLASS
   // in the template.  so we have to iterate through the provided template
   // and make sure that if CKA_CLASS is CKO_SECRET_KEY, if it is present.
   //
   // it would have been more logical for Cryptoki to forbid specifying
   // the CKA_CLASS attribute when generating a key
   //
   for (i=0; i < ulCount; i++) {
      if (pTemplate[i].type == CKA_CLASS) {
         keyclass = *(CK_OBJECT_CLASS *)pTemplate[i].pValue;
         if (keyclass != CKO_SECRET_KEY){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }
      }

      if (pTemplate[i].type == CKA_KEY_TYPE)
         subclass = *(CK_ULONG *)pTemplate[i].pValue;
   }


   switch (mech->mechanism) {
      case CKM_DES_KEY_GEN:
         if (subclass != 0 && subclass != CKK_DES){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }

         subclass = CKK_DES;
         break;

      case CKM_DES3_KEY_GEN:
         if (subclass != 0 && subclass != CKK_DES3){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }

         subclass = CKK_DES3;
         break;

#if !(NOCDMF)
      case CKM_CDMF_KEY_GEN:
         if (subclass != 0 && subclass != CKK_CDMF){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }

         subclass = CKK_CDMF;
         break;
#endif

      case CKM_SSL3_PRE_MASTER_KEY_GEN:
         if (subclass != 0 && subclass != CKK_GENERIC_SECRET){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }
         if (mech->ulParameterLen != sizeof(CK_VERSION)){
            OCK_LOG_ERR(ERR_MECHANISM_PARAM_INVALID);
            return CKR_MECHANISM_PARAM_INVALID;
         }
         subclass = CKK_GENERIC_SECRET;
         break;

      case CKM_AES_KEY_GEN:
	 if (subclass != 0 && subclass != CKK_AES){
	    OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
	    return CKR_TEMPLATE_INCONSISTENT;
	 }

	 subclass = CKK_AES;
	 break;

      default:
         OCK_LOG_ERR(ERR_MECHANISM_INVALID);
         return CKR_MECHANISM_INVALID;
   }


   rc = object_mgr_create_skel( sess,
                                pTemplate, ulCount,
                                MODE_KEYGEN,
                                CKO_SECRET_KEY, subclass,
                                &key_obj );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_SKEL);
      goto error;
   }

   // at this point, 'key_obj' should contain a skeleton key.  depending on
   // the key type, we may need to extract one or more attributes from
   // the object prior to generating the key data (ie. variable key length)
   //

   switch (mech->mechanism) {
      case CKM_DES_KEY_GEN:
            rc = ckm_des_key_gen( key_obj->template );
            break;

         case CKM_DES3_KEY_GEN:
            rc = ckm_des3_key_gen( key_obj->template );
            break;

#if !(NOCDMF)
         case CKM_CDMF_KEY_GEN:
            rc = ckm_cdmf_key_gen( key_obj->template );
            break;
#endif

         case CKM_SSL3_PRE_MASTER_KEY_GEN:
            rc = ckm_ssl3_pre_master_key_gen( key_obj->template, mech );
            break;
#ifndef NOAES
	 case CKM_AES_KEY_GEN:
	    rc = ckm_aes_key_gen( key_obj->template );
	    break;
#endif
      default:
         OCK_LOG_ERR(ERR_MECHANISM_INVALID);
         rc = CKR_MECHANISM_INVALID;
   }

   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_KEYGEN);
      goto error;
   }

   // we can now set CKA_ALWAYS_SENSITIVE and CKA_NEVER_EXTRACTABLE
   // to their appropriate values.  this only applies to CKO_SECRET_KEY
   // and CKO_PRIVATE_KEY objects
   //
   flag = template_attribute_find( key_obj->template, CKA_SENSITIVE, &attr );
   if (flag == TRUE) {
      flag = *(CK_BBOOL *)attr->pValue;

      rc = build_attribute( CKA_ALWAYS_SENSITIVE, &flag, sizeof(CK_BBOOL), &new_attr );
      if (rc != CKR_OK){
         OCK_LOG_ERR(ERR_BLD_ATTR);
         goto error;
      }
      template_update_attribute( key_obj->template, new_attr );

   } else {
      rc = CKR_FUNCTION_FAILED;
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      goto error;
   }


   flag = template_attribute_find( key_obj->template, CKA_EXTRACTABLE, &attr );
   if (flag == TRUE) {
      flag = *(CK_BBOOL *)attr->pValue;

      rc = build_attribute( CKA_NEVER_EXTRACTABLE, &true, sizeof(CK_BBOOL), &new_attr );
      if (rc != CKR_OK){
         OCK_LOG_ERR(ERR_BLD_ATTR);
         goto error;
      }
      if (flag == TRUE)
         *(CK_BBOOL *)new_attr->pValue = FALSE;

      template_update_attribute( key_obj->template, new_attr );

   } else {
      rc = CKR_FUNCTION_FAILED;
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      goto error;
   }


   // at this point, the key should be fully constructed...assign
   // an object handle and store the key
   //
   rc = object_mgr_create_final( sess, key_obj, handle );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_FINAL);
      goto error;
   }
   return rc;

error:
   if (key_obj) object_free( key_obj );

   *handle = 0;

   return rc;
}


//
//
CK_RV
key_mgr_generate_key_pair( SESSION           * sess,
                           CK_MECHANISM      * mech,
                           CK_ATTRIBUTE      * publ_tmpl,
                           CK_ULONG            publ_count,
                           CK_ATTRIBUTE      * priv_tmpl,
                           CK_ULONG            priv_count,
                           CK_OBJECT_HANDLE  * publ_key_handle,
                           CK_OBJECT_HANDLE  * priv_key_handle )
{
   OBJECT        * publ_key_obj = NULL;
   OBJECT        * priv_key_obj = NULL;
   CK_ATTRIBUTE  * attr         = NULL;
   CK_ATTRIBUTE  * new_attr     = NULL;
   CK_ULONG        i, keyclass, subclass = 0;
   CK_BBOOL        flag;
   CK_RV           rc;

   if (!sess || !mech || !publ_key_handle || !priv_key_handle){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }
   if (!publ_tmpl && (publ_count != 0)){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }
   if (!priv_tmpl && (priv_count != 0)){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }

   // it's silly but Cryptoki allows the user to specify the CKA_CLASS
   // in the template.  so we have to iterate through the provided template
   // and make sure that if CKA_CLASS is valid, if it is present.
   //
   // it would have been more logical for Cryptoki to forbid specifying
   // the CKA_CLASS attribute when generating a key
   //
   for (i=0; i < publ_count; i++) {
      if (publ_tmpl[i].type == CKA_CLASS) {
         keyclass = *(CK_OBJECT_CLASS *)publ_tmpl[i].pValue;
         if (keyclass != CKO_PUBLIC_KEY){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }
      }

      if (publ_tmpl[i].type == CKA_KEY_TYPE)
         subclass = *(CK_ULONG *)publ_tmpl[i].pValue;
   }


   for (i=0; i < priv_count; i++) {
      if (priv_tmpl[i].type == CKA_CLASS) {
         keyclass = *(CK_OBJECT_CLASS *)priv_tmpl[i].pValue;
         if (keyclass != CKO_PRIVATE_KEY){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }
      }

      if (priv_tmpl[i].type == CKA_KEY_TYPE) {
         CK_ULONG temp = *(CK_ULONG *)priv_tmpl[i].pValue;
         if (temp != subclass){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
         }
      }
   }


   switch (mech->mechanism) {
      case CKM_RSA_PKCS_KEY_PAIR_GEN:
         if (subclass != 0 && subclass != CKK_RSA){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
          }

         subclass = CKK_RSA;
         break;

      case CKM_EC_KEY_PAIR_GEN:
         if (subclass != 0 && subclass != CKK_EC){
            OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
            return CKR_TEMPLATE_INCONSISTENT;
          }

         subclass = CKK_EC;
         break;

#if !(NODSA)
      case CKM_DSA_KEY_PAIR_GEN:
         if (subclass != 0 && subclass != CKK_DSA){
           OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
           return CKR_TEMPLATE_INCONSISTENT;
         }
         subclass = CKK_DSA;
         break;
#endif

/* Begin code contributed by Corrent corp. */
#if !(NODH)
      case CKM_DH_PKCS_KEY_PAIR_GEN:
         if (subclass != 0 && subclass != CKK_DH){
           OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
           return CKR_TEMPLATE_INCONSISTENT;
         }
         subclass = CKK_DH;
         break;
#endif
/* End  code contributed by Corrent corp. */

      default:
         OCK_LOG_ERR(ERR_MECHANISM_INVALID);
         return CKR_MECHANISM_INVALID;
   }


   rc = object_mgr_create_skel( sess,
                                publ_tmpl,       publ_count,
                                MODE_KEYGEN,
                                CKO_PUBLIC_KEY,  subclass,
                                &publ_key_obj );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_SKEL);
      goto error;
   }
   rc = object_mgr_create_skel( sess,
                                priv_tmpl,       priv_count,
                                MODE_KEYGEN,
                                CKO_PRIVATE_KEY, subclass,
                                &priv_key_obj );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_SKEL);
      goto error;
   }

   // at this point, 'key_obj' should contain a skeleton key.  depending on
   // the key type, we may need to extract one or more attributes from
   // the object prior to generating the key data (ie. variable key length)
   //

   switch (mech->mechanism) {
      case CKM_RSA_PKCS_KEY_PAIR_GEN:
         rc = ckm_rsa_key_pair_gen( publ_key_obj->template,
                                    priv_key_obj->template );
         break;

      case CKM_EC_KEY_PAIR_GEN:
         rc = ckm_ec_key_pair_gen( publ_key_obj->template,
                                   priv_key_obj->template );
         break;

#if !(NODSA)
      case CKM_DSA_KEY_PAIR_GEN:
         rc = ckm_dsa_key_pair_gen( publ_key_obj->template,
                                    priv_key_obj->template );
         break;
#endif

/* Begin code contributed by Corrent corp. */
#if !(NODH)
      case CKM_DH_PKCS_KEY_PAIR_GEN:
         rc = ckm_dh_pkcs_key_pair_gen( publ_key_obj->template,
                                        priv_key_obj->template );
         break;
#endif
/* End code contributed by Corrent corp. */

      default:
         OCK_LOG_ERR(ERR_MECHANISM_INVALID);
         rc = CKR_MECHANISM_INVALID;
         break;
   }

   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_KEYGEN);
      goto error;
   }

   // we can now set CKA_ALWAYS_SENSITIVE and CKA_NEVER_EXTRACTABLE
   // to their appropriate values.  this only applies to CKO_SECRET_KEY
   // and CKO_PRIVATE_KEY objects
   //
   flag = template_attribute_find( priv_key_obj->template, CKA_SENSITIVE, &attr );
   if (flag == TRUE) {
      flag = *(CK_BBOOL *)attr->pValue;

      rc = build_attribute( CKA_ALWAYS_SENSITIVE, &flag, sizeof(CK_BBOOL), &new_attr );
      if (rc != CKR_OK){
         OCK_LOG_ERR(ERR_BLD_ATTR);
         goto error;
      }
      template_update_attribute( priv_key_obj->template, new_attr );

   } else {
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      rc = CKR_FUNCTION_FAILED;
      goto error;
   }


   flag = template_attribute_find( priv_key_obj->template, CKA_EXTRACTABLE, &attr );
   if (flag == TRUE) {
      flag = *(CK_BBOOL *)attr->pValue;

      rc = build_attribute( CKA_NEVER_EXTRACTABLE, &true, sizeof(CK_BBOOL), &new_attr );
      if (rc != CKR_OK){
         OCK_LOG_ERR(ERR_BLD_ATTR);
         goto error;
      }
      if (flag == TRUE)
         *(CK_BBOOL *)new_attr->pValue = false;

      template_update_attribute( priv_key_obj->template, new_attr );

   } else {
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      rc = CKR_FUNCTION_FAILED;
      goto error;
   }


   // at this point, the keys should be fully constructed...assign
   // object handles and store the keys
   //
   rc = object_mgr_create_final( sess, publ_key_obj, publ_key_handle );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_FINAL);
      goto error;
   }
   rc = object_mgr_create_final( sess, priv_key_obj, priv_key_handle );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_FINAL);
      // just calling object_free in the error path below would lead to a double
      // free error on session close - KEY 09/26/07
      object_mgr_destroy_object( sess, *publ_key_handle );
      publ_key_obj = NULL;
      goto error;
   }
   return rc;

error:
   if (publ_key_obj) object_free( publ_key_obj );
   if (priv_key_obj) object_free( priv_key_obj );

   *publ_key_handle = 0;
   *priv_key_handle = 0;

   return rc;
}


//
//
CK_RV
key_mgr_wrap_key( SESSION           * sess,
                  CK_BBOOL            length_only,
                  CK_MECHANISM      * mech,
                  CK_OBJECT_HANDLE    h_wrapping_key,
                  CK_OBJECT_HANDLE    h_key,
                  CK_BYTE           * wrapped_key,
                  CK_ULONG          * wrapped_key_len )
{
   ENCR_DECR_CONTEXT * ctx       = NULL;
   OBJECT            * key1_obj  = NULL;
   OBJECT            * key2_obj  = NULL;
   CK_ATTRIBUTE      * attr      = NULL;
   CK_BYTE           * data      = NULL;
   CK_ULONG            data_len;
   CK_OBJECT_CLASS     class;
   CK_KEY_TYPE         keytype;
   CK_BBOOL            flag;
   CK_RV               rc;


   if (!sess || !wrapped_key_len){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }

   rc = object_mgr_find_in_map1( h_wrapping_key, &key1_obj );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_WRAPPING_KEY_HANDLE_INVALID);
      return CKR_WRAPPING_KEY_HANDLE_INVALID;
   }
   rc = object_mgr_find_in_map1( h_key, &key2_obj );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_KEY_HANDLE_INVALID);
      return CKR_KEY_HANDLE_INVALID;
   }

   // is the key-to-be-wrapped EXTRACTABLE?
   //
   rc = template_attribute_find( key2_obj->template, CKA_EXTRACTABLE, &attr );
   if (rc == FALSE){
      OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
      return CKR_KEY_NOT_WRAPPABLE;  // could happen if user tries to wrap a public key
   }
   else {
      flag = *(CK_BBOOL *)attr->pValue;
      if (flag == FALSE){
         OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
         return CKR_KEY_NOT_WRAPPABLE;
      }
   }

   // what kind of key are we trying to wrap?  make sure the mechanism is
   // allowed to wrap this kind of key
   //
   rc = template_attribute_find( key2_obj->template, CKA_CLASS, &attr );
   if (rc == FALSE) {
      OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
      return CKR_KEY_NOT_WRAPPABLE;
   }
   else
      class = *(CK_OBJECT_CLASS *)attr->pValue;

   // pkcs11v2-20rc3, page 178
   // C_WrapKey can be used in following situations:
   // - To wrap any secret key with a public key that supports encryption
   // and decryption.
   // - To wrap any secret key with any other secret key. Consideration
   // must be given to key size and mechanism strength or the token may
   // not allow the operation.
   // - To wrap a private key with any secret key.
   //
   //  These can be deduced to:
   //  A public key or a secret key can be used to wrap a secret key.
   //  A secret key can be used to wrap a private key.

   switch (mech->mechanism) {
#if !(NOCDMF)
      case CKM_CDMF_ECB:
      case CKM_CDMF_CBC:
      case CKM_CDMF_CBC_PAD:
#endif
      case CKM_DES_ECB:
      case CKM_DES_CBC:
      case CKM_DES3_ECB:
      case CKM_DES3_CBC:
      case CKM_AES_ECB:
      case CKM_AES_CBC:
      case CKM_AES_CTR:
      case CKM_DES_CBC_PAD:
      case CKM_DES3_CBC_PAD:
      case CKM_AES_CBC_PAD:
      case CKM_AES_OFB:
      case CKM_AES_CFB8:
      case CKM_AES_CFB64:
      case CKM_AES_CFB128:
         if ((class != CKO_SECRET_KEY) && (class != CKO_PRIVATE_KEY)) {
            OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
            return CKR_KEY_NOT_WRAPPABLE;
         }
	 break;

      case CKM_RSA_PKCS:
      case CKM_RSA_X_509:
         if (class != CKO_SECRET_KEY){
            OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
            return CKR_KEY_NOT_WRAPPABLE;
         }
         break;

      default:
         OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
         return CKR_KEY_NOT_WRAPPABLE;
   }


   // extract the secret data to be wrapped
   //
   rc = template_attribute_find( key2_obj->template, CKA_KEY_TYPE, &attr );
   if (rc == FALSE){
      OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
      return CKR_KEY_NOT_WRAPPABLE;
   }
   else
      keytype = *(CK_KEY_TYPE *)attr->pValue;

   switch (keytype) {
#if !(NOCDMF)
      case CKK_CDMF:
#endif
      case CKK_DES:
         rc = des_wrap_get_data( key2_obj->template, length_only, &data, &data_len );
         if (rc != CKR_OK){
            OCK_LOG_ERR(ERR_DES_WRAP_GETDATA);
            return rc;
         }
         break;

      case CKK_DES3:
         rc = des3_wrap_get_data( key2_obj->template, length_only, &data, &data_len );
         if (rc != CKR_OK){
            OCK_LOG_ERR(ERR_DES3_WRAP_GETDATA);
            return rc;
         }
         break;

      case CKK_RSA:
         rc = rsa_priv_wrap_get_data( key2_obj->template, length_only, &data, &data_len );
         if (rc != CKR_OK){
            OCK_LOG_ERR(ERR_RSA_WRAP_GETDATA);
            return rc;
         }
         break;

#if !(NODSA)
      case CKK_DSA:
         rc = dsa_priv_wrap_get_data( key2_obj->template, length_only, &data, &data_len );
         if (rc != CKR_OK){
            OCK_LOG_ERR(ERR_DSA_WRAP_GETDATA);
            return rc;
         }
         break;
#endif

      case CKK_GENERIC_SECRET:
         rc = generic_secret_wrap_get_data( key2_obj->template, length_only, &data, &data_len );
         if (rc != CKR_OK){
            OCK_LOG_ERR(ERR_GENERIC_WRAP_GETDATA);
            return rc;
         }
         break;
#ifndef NOAES
      case CKK_AES:
	 rc = aes_wrap_get_data( key2_obj->template, length_only, &data, &data_len );
	 if (rc != CKR_OK){
	    OCK_LOG_ERR(ERR_AES_WRAP_GETDATA);
	    return rc;
	 }
	 break;
#endif
      default:
         OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
         return CKR_KEY_NOT_WRAPPABLE;
   }

   // we might need to format the wrapped data based on the mechanism
   //
   switch (mech->mechanism) {
#if !(NOCMF)
      case CKM_CDMF_ECB:
      case CKM_CDMF_CBC:
#endif
      case CKM_DES_ECB:
      case CKM_DES_CBC:
      case CKM_DES3_ECB:
      case CKM_DES3_CBC:
         rc = ckm_des_wrap_format( length_only, &data, &data_len );
         if (rc != CKR_OK) {
            OCK_LOG_ERR(ERR_DES_WRAP_FORMAT);
            if (data) free( data );
            return rc;
         }
         break;
#ifndef NOAES
      case CKM_AES_ECB:
      case CKM_AES_CBC:
      case CKM_AES_CTR:
      case CKM_AES_OFB:
      case CKM_AES_CFB8:
      case CKM_AES_CFB64:
      case CKM_AES_CFB128:
	 rc = ckm_aes_wrap_format( length_only, &data, &data_len );
	 if (rc != CKR_OK) {
	    OCK_LOG_ERR(ERR_AES_WRAP_FORMAT);
	    if (data) free( data );
	    return rc;
	 }
	 break;
#endif
#if !(NOCMF)
      case CKM_CDMF_CBC_PAD:
#endif
      case CKM_DES_CBC_PAD:
      case CKM_DES3_CBC_PAD:
      case CKM_AES_CBC_PAD:
         // these mechanisms pad themselves
         //
         break;

      case CKM_RSA_PKCS:
      case CKM_RSA_X_509:
         break;

      default:
         OCK_LOG_ERR(ERR_KEY_NOT_WRAPPABLE);
         return CKR_KEY_NOT_WRAPPABLE;
   }

   ctx = (ENCR_DECR_CONTEXT *)malloc(sizeof(ENCR_DECR_CONTEXT));
   if (!ctx){
      OCK_LOG_ERR(ERR_HOST_MEMORY);
      return CKR_HOST_MEMORY;
   }
   memset( ctx, 0x0, sizeof(ENCR_DECR_CONTEXT) );

   // prepare to do the encryption
   //
   rc = encr_mgr_init( sess, ctx, OP_WRAP, mech, h_wrapping_key );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_ENCRYPTMGR_INIT);
      return rc;
   }
   // do the encryption and clean up.  at this point, 'value' may or may not
   // be NULL depending on 'length_only'
   //
   rc = encr_mgr_encrypt( sess,        length_only,
                          ctx,
                          data,        data_len,
                          wrapped_key, wrapped_key_len );
   if (data != NULL){
      free( data );
   }
   encr_mgr_cleanup( ctx );
   free( ctx );
   
   return rc;
}


//
//
CK_RV
key_mgr_unwrap_key( SESSION           * sess,
                    CK_MECHANISM      * mech,
                    CK_ATTRIBUTE      * attributes,
                    CK_ULONG            attrib_count,
                    CK_BYTE           * wrapped_key,
                    CK_ULONG            wrapped_key_len,
                    CK_OBJECT_HANDLE    h_unwrapping_key,
                    CK_OBJECT_HANDLE  * h_unwrapped_key )
{
   ENCR_DECR_CONTEXT * ctx = NULL;
   OBJECT            * key_obj = NULL, * tmp_obj = NULL;
   CK_BYTE           * data = NULL;
   CK_ULONG            data_len;
   CK_ULONG            keyclass, keytype;
   CK_ULONG            i;
   CK_BBOOL            found_class, found_type, fromend, isopaque = FALSE;
   CK_RV               rc;
   CK_ATTRIBUTE	     * attr = NULL;


   if (!sess || !wrapped_key || !h_unwrapped_key){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }

   rc = object_mgr_find_in_map1( h_unwrapping_key, &tmp_obj );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_WRAPPING_KEY_HANDLE_INVALID);
      return CKR_WRAPPING_KEY_HANDLE_INVALID;
   }

   if (template_attribute_find(tmp_obj->template, CKA_IBM_OPAQUE, &attr) == TRUE)
	isopaque = TRUE;

   found_class = FALSE;
   found_type = FALSE;

   // pkcs11v2-20rc3, page 178
   // C_WrapKey can be used in following situations:
   // - To wrap any secret key with a public key that supports encryption
   // and decryption.
   // - To wrap any secret key with any other secret key. Consideration
   // must be given to key size and mechanism strength or the token may
   // not allow the operation.
   // - To wrap a private key with any secret key.
   //
   //  These can be deduced to:
   //  A public key or a secret key can be used to wrap a secret key.
   //  A secret key can be used to wrap a private key.

   switch (mech->mechanism) {
      case CKM_RSA_PKCS:
      case CKM_RSA_X_509:
         keyclass = CKO_SECRET_KEY;
         found_class = TRUE;
         break;

#if !(NOCMF)
      case CKM_CDMF_ECB:
      case CKM_CDMF_CBC:
      case CKM_CDMF_CBC_PAD:
#endif
      case CKM_DES_ECB:
      case CKM_DES_CBC:
      case CKM_DES3_ECB:
      case CKM_DES3_CBC:
      case CKM_AES_ECB:
      case CKM_AES_CBC:
      case CKM_AES_CTR:
      case CKM_AES_OFB:
      case CKM_AES_CFB8:
      case CKM_AES_CFB64:
      case CKM_AES_CFB128:
      case CKM_DES_CBC_PAD:
      case CKM_DES3_CBC_PAD:
      case CKM_AES_CBC_PAD:
         // these mechanisms can wrap any type of key so nothing is implied
         //
         break;

      default:
         OCK_LOG_ERR(ERR_MECHANISM_INVALID);
         return CKR_MECHANISM_INVALID;
   }

   // extract key type and key class from the template if they exist.  we
   // have to scan the entire template in case the CKA_CLASS or CKA_KEY_TYPE
   // attributes are duplicated.

   for (i=0; i < attrib_count; i++) {
      switch (attributes[i].type) {
         case CKA_CLASS:
            keyclass = *(CK_OBJECT_CLASS *)attributes[i].pValue;
            found_class = TRUE;
            break;

         case CKA_KEY_TYPE:
            keytype = *(CK_KEY_TYPE *)attributes[i].pValue;
            found_type = TRUE;
            break;
      }
   }

   // if we're unwrapping a private key, we can extract the key type from
   // the BER-encoded information

   if (found_class == FALSE || 
       (found_type == FALSE && keyclass != CKO_PRIVATE_KEY)) {
      OCK_LOG_ERR(ERR_TEMPLATE_INCOMPLETE);
      return CKR_TEMPLATE_INCOMPLETE;
   }

   // Check again that a public key only wraps/unwraps a secret key.

   if ((mech->mechanism == CKM_RSA_PKCS) || (mech->mechanism == CKM_RSA_X_509)){
      if (keyclass != CKO_SECRET_KEY) {
         OCK_LOG_ERR(ERR_TEMPLATE_INCONSISTENT);
         return CKR_TEMPLATE_INCONSISTENT;
      }
   }

   // looks okay... do the decryption
   //
   ctx = (ENCR_DECR_CONTEXT *)malloc(sizeof(ENCR_DECR_CONTEXT));
   if (!ctx){
      OCK_LOG_ERR(ERR_HOST_MEMORY);
      return CKR_HOST_MEMORY;
   }
   memset( ctx, 0x0, sizeof(ENCR_DECR_CONTEXT) );

   rc = decr_mgr_init( sess, ctx, OP_UNWRAP, mech, h_unwrapping_key );
   if (rc != CKR_OK)
      return rc;

   rc = decr_mgr_decrypt( sess,
                          TRUE,
                          ctx,
                          wrapped_key, wrapped_key_len,
                          data,       &data_len );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_DECRYPTMGR_DECRYPT);
      goto error;
   }
   data = (CK_BYTE *)malloc(data_len);
   if (!data) {
      OCK_LOG_ERR(ERR_HOST_MEMORY);
      rc = CKR_HOST_MEMORY;
      goto error;
   }

   rc = decr_mgr_decrypt( sess,
                          FALSE,
                          ctx,
                          wrapped_key, wrapped_key_len,
                          data,       &data_len );

   decr_mgr_cleanup( ctx );
   free( ctx );

   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_DECRYPTMGR_DECRYPT);
      goto error;
   }
   // if we use X.509, the data will be padded from the front with zeros.
   // PKCS #11 specifies that for this mechanism, CK_VALUE is to be read
   // from the end of the data.
   //
   // Note: the PKCS #11 reference implementation gets this wrong.
   //
   if (mech->mechanism == CKM_RSA_X_509)
      fromend = TRUE;
   else
      fromend = FALSE;

   // extract the key type from the PrivateKeyInfo::AlgorithmIndicator
   //
   if (keyclass == CKO_PRIVATE_KEY) {
      rc = key_mgr_get_private_key_type( data, data_len, &keytype );
      if (rc != CKR_OK){
         OCK_LOG_ERR(ERR_KEYMGR_GETPRIVKEY);
         goto error;
      }
   }

   // we have decrypted the wrapped key data.  we also
   // know what type of key it is.  now we need to construct a new key
   // object...
   //

   rc = object_mgr_create_skel( sess,
                                attributes,    attrib_count,
                                MODE_UNWRAP,
                                keyclass,      keytype,
                                &key_obj );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_SKEL);
      goto error;
   }
   // at this point, 'key_obj' should contain a skeleton key.  depending on
   // the key type.  we're now ready to plug in the decrypted key data.
   // in some cases, the data will be BER-encoded so we'll need to decode it.
   //
   // this routine also ensires that CKA_EXTRACTABLE == FALSE,
   // CKA_ALWAYS_SENSITIVE == FALSE and CKA_LOCAL == FALSE
   //
   switch (keyclass) {
      case CKO_SECRET_KEY:
         rc = secret_key_unwrap(key_obj->template, keytype, data, data_len, fromend,isopaque);
         break;

      case CKO_PRIVATE_KEY:
         rc = priv_key_unwrap( key_obj->template, keytype, data, data_len, isopaque );
         break;

      default:
         rc = CKR_WRAPPED_KEY_INVALID;
         break;
   }

   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_KEY_UNWRAP);
      goto error;
   }
   // at this point, the key should be fully constructed...assign
   // an object handle and store the key
   //
   rc = object_mgr_create_final( sess, key_obj, h_unwrapped_key );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_OBJMGR_CREATE_FINAL);
      goto error;
   }
   if (data) free(data);
   return rc;

error:
   if (key_obj) object_free( key_obj );
   if (data)    free(data);

   return rc;
}


CK_RV
key_mgr_get_private_key_type( CK_BYTE     *keydata,
                              CK_ULONG     keylen,
                              CK_KEY_TYPE *keytype )
{
   CK_BYTE  *alg = NULL;
   CK_BYTE  *priv_key = NULL;
   CK_ULONG  alg_len;
   CK_RV    rc;

   rc = ber_decode_PrivateKeyInfo( keydata, keylen, &alg, &alg_len, &priv_key );
   if (rc != CKR_OK){
      OCK_LOG_ERR(ERR_KEYMGR_GETPRIVKEY);
      return rc;
   }
   // check the entire AlgorithmIdentifier for RSA
   //
   if (alg_len >= ber_rsaEncryptionLen) {
      if (memcmp(alg, ber_rsaEncryption, ber_rsaEncryptionLen) == 0) {
         *keytype = CKK_RSA;
         return CKR_OK;
      }
   }

   // Check only the OBJECT IDENTIFIER for DSA
   //
   if (alg_len >= ber_idDSALen) {
      if (memcmp(alg, ber_idDSA, ber_idDSALen) == 0) {
          *keytype = CKK_DSA;
          return CKR_OK;
      }
   }

   OCK_LOG_ERR(ERR_TEMPLATE_INCOMPLETE);
   return CKR_TEMPLATE_INCOMPLETE;
}


//
//
CK_RV
key_mgr_derive_key( SESSION           * sess,
                    CK_MECHANISM      * mech,
                    CK_OBJECT_HANDLE    base_key,
                    CK_OBJECT_HANDLE  * derived_key,
                    CK_ATTRIBUTE      * pTemplate,
                    CK_ULONG            ulCount )
{
   if (!sess || !mech){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }
   if (!pTemplate && (ulCount != 0)){
      OCK_LOG_ERR(ERR_FUNCTION_FAILED);
      return CKR_FUNCTION_FAILED;
   }
   switch (mech->mechanism)
   {
      case CKM_SSL3_MASTER_KEY_DERIVE:
      {
         if (!derived_key){
            OCK_LOG_ERR(ERR_FUNCTION_FAILED);
            return CKR_FUNCTION_FAILED;
         }
         return ssl3_master_key_derive( sess, mech, base_key,
                                        pTemplate, ulCount, derived_key );
      }
      break ;

      case CKM_SSL3_KEY_AND_MAC_DERIVE:
      {
         return ssl3_key_and_mac_derive( sess, mech, base_key,
                                         pTemplate, ulCount );
      }
      break ;

/* Begin code contributed by Corrent corp. */
#ifndef NODH
      case CKM_DH_PKCS_DERIVE:
      {
         if (!derived_key){
            OCK_LOG_ERR(ERR_FUNCTION_FAILED);
            return CKR_FUNCTION_FAILED;
         }
         return dh_pkcs_derive( sess, mech, base_key,
                                pTemplate, ulCount, derived_key );
      }
      break ;
#endif
/* End code contributed by Corrent corp. */
      
      default:
         OCK_LOG_ERR(ERR_MECHANISM_INVALID);
         return CKR_MECHANISM_INVALID;
   }
}

