/*
 * Copyright 2021 Red Hat
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#ifndef _LIBNMSTATE_H_
#define _LIBNMSTATE_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

#define NMSTATE_VERSION_MAJOR        2
#define NMSTATE_VERSION_MINOR        1
#define NMSTATE_VERSION_MICRO        4

#define NMSTATE_VERSION              \
    ((NMSTATE_VERSION_MAJOR * 10000) + \
     (NMSTATE_VERSION_MINOR * 100) + \
     NMSTATE_VERSION_MICRO)

#define NMSTATE_PASS                 0
#define NMSTATE_FAIL                 1

#define NMSTATE_FLAG_NONE                   0
#define NMSTATE_FLAG_KERNEL_ONLY            1 << 1
#define NMSTATE_FLAG_NO_VERIFY              1 << 2
#define NMSTATE_FLAG_INCLUDE_STATUS_DATA    1 << 3
#define NMSTATE_FLAG_INCLUDE_SECRETS        1 << 4
#define NMSTATE_FLAG_NO_COMMIT              1 << 5
#define NMSTATE_FLAG_MEMORY_ONLY            1 << 6
#define NMSTATE_FLAG_RUNNING_CONFIG_ONLY    1 << 7

/**
 * nmstate_net_state_retrieve - Retrieve network state
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Retrieve network state in the format of JSON.
 *
 * @flags:
 *      Flags for special use cases:
 *          * NMSTATE_FLAG_NONE
 *              No flag
 *          * NMSTATE_FLAG_KERNEL_ONLY
 *              Do not use external plugins, show kernel status only.
 * @state:
 *      Output pointer of char array for network state in json format.
 *      The memory should be freed by nmstate_net_state_free().
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_net_state_retrieve(uint32_t flags, char **state, char **log,
                               char **err_kind, char **err_msg);

/**
 * nmstate_net_state_apply - Apply network state
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Apply network state in the format of JSON.
 *
 * @flags:
 *      Flags for special use cases:
 *          * NMSTATE_FLAG_NONE
 *              No flag
 *          * NMSTATE_FLAG_KERNEL_ONLY
 *              Do not use external plugins, apply to kernel only.
 *          * NMSTATE_FLAG_NO_COMMIT
 *              Do not commit new state after verification
 * @state:
 *      Pointer of char array for network state in json format.
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_net_state_apply(uint32_t flags, const char *state, uint32_t rollback_timeout, char **log,
                            char **err_kind, char **err_msg);

/**
 * nmstate_checkpoint_commit - Destroy the checkpoint
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Destroy the checkpoint, if no checkpoint 
 *      is passed it will destroy last active checkpoint 
 *
 * @checkpoint:
 *      Checkpoint to destroy or empty to select the last active one.
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_checkpoint_commit(const char *checkpoint, char **log, char **err_kind,
                          char **err_msg);

/**
 * nmstate_checkpoint_rollback - Rollback the checkpoint
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Rollack the checkpoint, if no checkpoint 
 *      is passed it will rollback last active checkpoint 
 *
 * @checkpoint:
 *      Checkpoint to rollback or empty to select the last active one.
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_checkpoint_rollback(const char *checkpoint, char **log, char **err_kind,
                                 char **err_msg);

/**
 * nmstate_generate_configurations - Generate network configurations
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Generate offline configrations of each backends base on specified
 *      network state. The returned configs is json string for
 *      HashMap/Dictionary with backend name as key, and array of string
 *      as value.
 *
 * @state:
 *      Pointer of char array for network state in json format.
 * @configs:
 *      Output pointer of char array for network configures in json format.
 *      The memory should be freed by nmstate_net_state_free().
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_generate_configurations(const char *state, char **configs,
                                    char **log, char **err_kind,
                                    char **err_msg);

/**
 * nmstate_cstring_free - free the memory of C string
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Free the memory of C string.
 *
 * @cstring:
 *      Pointer of char array for string
 *
 * Return:
 *      void
 */
void nmstate_cstring_free(char *cstring);

#endif // _LIBNMSTATE_H_
