/*  Copyright (c) 2018, 2022, Oracle and/or its affiliates.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2.0,
    as published by the Free Software Foundation.

    This program is also distributed with certain software (including
    but not limited to OpenSSL) that is licensed under separate terms,
    as designated in a particular file or component or in included license
    documentation.  The authors of MySQL hereby grant you an additional
    permission to link the program and your derivative works with the
    separately licensed software that they have included with MySQL.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License, version 2.0, for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA */

#include <regex>
#include <string>

#include "plugin/ddl_rewriter/ddl_rewriter.h"

bool query_rewritten(const std::string &query, std::string *rewritten_query) {
  /*
    This plugin is intended to parse DDL generated by SHOW CREATE TABLE,
    which starts with 'CREATE TABLE'; hence, we can check the first
    character to exit as early as possible.
  */
  if (query.length() == 0 || (query[0] != 'C' && query[0] != 'c')) return false;

  const std::regex create_table("^CREATE\\s+TABLE",
                                std::regex::icase | std::regex::nosubs);
  if (!regex_search(query, create_table)) return false;

  /*
    We now know this is a CREATE TABLE statement. So we can go ahead with
    replacements.

    First, replace DATA|INDEX DIRECTORY = '...', but note:
     - Optional preceding and trailing commas.
     - Optional assignment operator.
     - Path enclosed in single quotes.
     - Use of ungreedy patterns to get correct grouping.
  */
  std::regex directory_option(
      "\\s*,?\\s*(DATA|INDEX)\\s+DIRECTORY\\s*?=?\\s*?[\"'][^\"']+?[\"']\\s*,?"
      "\\s*",
      std::regex::icase | std::regex::nosubs);
  *rewritten_query = std::regex_replace(query, directory_option, " ");

  /*
    Replace ENCRYPTION option.
  */
  std::regex encryption_option(
      "\\s*,?\\s*ENCRYPTION\\s*?=?\\s*?[\"'][NY]?[\"']\\s*,?\\s*",
      std::regex::icase | std::regex::nosubs);
  *rewritten_query =
      std::regex_replace(*rewritten_query, encryption_option, " ");

  return (*rewritten_query != query);
}
