/* Copyright (c) 2020, 2021, Oracle and/or its affiliates.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License, version 2.0,
   as published by the Free Software Foundation.

   This program is also distributed with certain software (including
   but not limited to OpenSSL) that is licensed under separate terms,
   as designated in a particular file or component or in included license
   documentation.  The authors of MySQL hereby grant you an additional
   permission to link the program and your derivative works with the
   separately licensed software that they have included with MySQL.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License, version 2.0, for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA */

#include "sql/join_optimizer/join_optimizer.h"

#include <assert.h>
#include <math.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <algorithm>
#include <array>
#include <bitset>
#include <functional>
#include <initializer_list>
#include <string>
#include <unordered_map>
#include <utility>
#include <vector>

#include "ft_global.h"
#include "map_helpers.h"
#include "mem_root_deque.h"
#include "my_alloc.h"
#include "my_base.h"
#include "my_bitmap.h"
#include "my_inttypes.h"
#include "my_sys.h"
#include "my_table_map.h"
#include "mysql/components/services/bits/psi_bits.h"
#include "mysql/udf_registration_types.h"
#include "mysqld_error.h"
#include "prealloced_array.h"
#include "sql/field.h"
#include "sql/filesort.h"
#include "sql/handler.h"
#include "sql/item.h"
#include "sql/item_cmpfunc.h"
#include "sql/item_func.h"
#include "sql/item_sum.h"
#include "sql/join_optimizer/access_path.h"
#include "sql/join_optimizer/bit_utils.h"
#include "sql/join_optimizer/build_interesting_orders.h"
#include "sql/join_optimizer/cost_model.h"
#include "sql/join_optimizer/estimate_selectivity.h"
#include "sql/join_optimizer/explain_access_path.h"
#include "sql/join_optimizer/find_contained_subqueries.h"
#include "sql/join_optimizer/interesting_orders.h"
#include "sql/join_optimizer/interesting_orders_defs.h"
#include "sql/join_optimizer/make_join_hypergraph.h"
#include "sql/join_optimizer/node_map.h"
#include "sql/join_optimizer/overflow_bitset.h"
#include "sql/join_optimizer/print_utils.h"
#include "sql/join_optimizer/relational_expression.h"
#include "sql/join_optimizer/subgraph_enumeration.h"
#include "sql/join_optimizer/walk_access_paths.h"
#include "sql/join_type.h"
#include "sql/key.h"
#include "sql/mem_root_array.h"
#include "sql/query_options.h"
#include "sql/range_optimizer/range_optimizer.h"
#include "sql/sql_array.h"
#include "sql/sql_bitmap.h"
#include "sql/sql_class.h"
#include "sql/sql_cmd.h"
#include "sql/sql_const.h"
#include "sql/sql_executor.h"
#include "sql/sql_lex.h"
#include "sql/sql_list.h"
#include "sql/sql_opt_exec_shared.h"
#include "sql/sql_optimizer.h"
#include "sql/sql_select.h"
#include "sql/sql_tmp_table.h"
#include "sql/system_variables.h"
#include "sql/table.h"
#include "sql/table_function.h"
#include "sql/temp_table_param.h"
#include "sql/window.h"
#include "template_utils.h"

namespace hypergraph {
struct Hyperedge;
struct Hypergraph;
}  // namespace hypergraph

using hypergraph::Hyperedge;
using hypergraph::Hypergraph;
using hypergraph::NodeMap;
using std::array;
using std::bitset;
using std::min;
using std::pair;
using std::string;
using std::swap;

namespace {

using OrderingSet = std::bitset<kMaxSupportedOrderings>;

AccessPath *CreateMaterializationPath(THD *thd, JOIN *join, AccessPath *path,
                                      TABLE *temp_table,
                                      Temp_table_param *temp_table_param,
                                      bool copy_items);

AccessPath *GetSafePathToSort(THD *thd, JOIN *join, AccessPath *path,
                              bool need_rowid);

/**
  CostingReceiver contains the main join planning logic, selecting access paths
  based on cost. It receives subplans from DPhyp (see enumerate_subgraph.h),
  assigns them costs based on a cost model, and keeps the ones that are
  cheapest. In the end, this means it will be left with a root access path that
  gives the lowest total cost for joining the tables in the query block, ie.,
  without ORDER BY etc.

  Currently, besides the expected number of produced rows (which is the same no
  matter how we access the table) we keep only a single value per subplan
  (total cost), and thus also only a single best access path. In the future,
  we will have more dimensions to worry about, such as initial cost versus total
  cost (relevant for LIMIT), ordering properties, and so on. At that point,
  there is not necessarily a single “best” access path anymore, and we will need
  to keep multiple ones around, and test all of them as candidates when building
  larger subplans.
 */
class CostingReceiver {
 public:
  CostingReceiver(
      THD *thd, Query_block *query_block, const JoinHypergraph &graph,
      const LogicalOrderings *orderings,
      const Mem_root_array<SortAheadOrdering> *sort_ahead_orderings,
      const Mem_root_array<ActiveIndexInfo> *active_indexes,
      const Mem_root_array<FullTextIndexInfo> *fulltext_searches,
      NodeMap fulltext_tables, uint64_t sargable_fulltext_predicates,
      bool need_rowid, SecondaryEngineFlags engine_flags,
      secondary_engine_modify_access_path_cost_t secondary_engine_cost_hook,
      string *trace)
      : m_thd(thd),
        m_query_block(query_block),
        m_graph(graph),
        m_orderings(orderings),
        m_sort_ahead_orderings(sort_ahead_orderings),
        m_active_indexes(active_indexes),
        m_fulltext_searches(fulltext_searches),
        m_fulltext_tables(fulltext_tables),
        m_sargable_fulltext_predicates(sargable_fulltext_predicates),
        m_need_rowid(need_rowid),
        m_engine_flags(engine_flags),
        m_secondary_engine_cost_hook(secondary_engine_cost_hook),
        m_trace(trace) {
    // At least one join type must be supported.
    assert(Overlaps(engine_flags,
                    MakeSecondaryEngineFlags(
                        SecondaryEngineFlag::SUPPORTS_HASH_JOIN,
                        SecondaryEngineFlag::SUPPORTS_NESTED_LOOP_JOIN)));
  }

  bool HasSeen(NodeMap subgraph) const {
    return m_access_paths.count(subgraph) != 0;
  }

  bool FoundSingleNode(int node_idx);

  // Called EmitCsgCmp() in the DPhyp paper.
  bool FoundSubgraphPair(NodeMap left, NodeMap right, int edge_idx);

  const Prealloced_array<AccessPath *, 4> &root_candidates() {
    const auto it = m_access_paths.find(TablesBetween(0, m_graph.nodes.size()));
    assert(it != m_access_paths.end());
    return it->second.paths;
  }

  FunctionalDependencySet active_fds_at_root() const {
    const auto it = m_access_paths.find(TablesBetween(0, m_graph.nodes.size()));
    assert(it != m_access_paths.end());
    return it->second.active_functional_dependencies;
  }

  size_t num_access_paths() const { return m_access_paths.size(); }

  AccessPath *ProposeAccessPath(
      AccessPath *path, Prealloced_array<AccessPath *, 4> *existing_paths,
      OrderingSet obsolete_orderings, const char *description_for_trace) const;

  bool HasSecondaryEngineCostHook() const {
    return m_secondary_engine_cost_hook != nullptr;
  }

 private:
  THD *m_thd;

  /// The query block we are planning.
  Query_block *m_query_block;

  /**
    Besides the access paths for a set of nodes (see m_access_paths),
    AccessPathSet contains information that is common between all access
    paths for that set. One would believe num_output_rows would be such
    a member (a set of tables should produce the same number of ouptut
    rows no matter the join order), but due to parameterized paths,
    different access paths could have different outputs. delayed_predicates
    is another, but currently, it's already efficiently hidden space-wise
    due to the use of a union.
   */
  struct AccessPathSet {
    Prealloced_array<AccessPath *, 4> paths;
    FunctionalDependencySet active_functional_dependencies{0};

    // Once-interesting orderings that we don't care about anymore,
    // e.g. because they were interesting for a semijoin but that semijoin
    // is now done (with or without using the ordering). This reduces
    // the number of access paths we have to keep in play, since they are
    // de-facto equivalent.
    //
    // Note that if orderings were merged, this could falsely prune out
    // orderings that we would actually need, but as long as all of the
    // relevant ones are semijoin orderings (which are never identical,
    // and never merged with the relevant-at-end orderings), this
    // should not happen.
    OrderingSet obsolete_orderings{0};
  };

  /**
    For each subset of tables that are connected in the join hypergraph,
    keeps the current best access paths for producing said subset.
    There can be several that are best in different ways; see comments
    on ProposeAccessPath().

    Also used for communicating connectivity information back to DPhyp
    (in HasSeen()); if there's an entry here, that subset will induce
    a connected subgraph of the join hypergraph.
   */
  std::unordered_map<NodeMap, AccessPathSet> m_access_paths;

  /**
    How many subgraph pairs we've seen so far. Used to give up
    if we end up allocating too many resources (prompting us to
    create a simpler join graph and try again).
   */
  int m_num_seen_subgraph_pairs = 0;

  /// The graph we are running over.
  const JoinHypergraph &m_graph;

  /// Keeps track of interesting orderings in this query block.
  /// See LogicalOrderings for more information.
  const LogicalOrderings *m_orderings;

  /// List of all orderings that are candidates for sort-ahead
  /// (because they are, or may eventually become, an interesting ordering).
  const Mem_root_array<SortAheadOrdering> *m_sort_ahead_orderings;

  /// List of all indexes that are active and that we can apply in this query.
  /// Indexes can be useful in several ways: We can use them for ref access,
  /// for index-only scans, or to get interesting orderings.
  const Mem_root_array<ActiveIndexInfo> *m_active_indexes;

  /// List of all active full-text indexes that we can apply in this query.
  const Mem_root_array<FullTextIndexInfo> *m_fulltext_searches;

  /// A map of tables that are referenced by a MATCH function (those tables that
  /// have TABLE_LIST::is_fulltext_searched() == true). It is used for
  /// preventing hash joins involving tables that are full-text searched.
  const NodeMap m_fulltext_tables = 0;

  /// The set of WHERE predicates which are on a form that can be satisfied by a
  /// full-text index scan. This includes calls to MATCH with no comparison
  /// operator, and predicates on the form MATCH > const or MATCH >= const
  /// (where const must be high enough to make the comparison return false for
  /// documents with zero score).
  const uint64_t m_sargable_fulltext_predicates = 0;

  /// Whether we will be needing row IDs from our tables, typically for
  /// a later sort. If this happens, derived tables cannot use streaming,
  /// but need an actual materialization, since filesort expects to be
  /// able to go back and ask for a given row. (This is different from
  /// when we need row IDs for weedout, which doesn't preclude streaming.
  /// The hypergraph optimizer does not use weedout.)
  bool m_need_rowid;

  /// The flags declared by the secondary engine. In particular, it describes
  /// what kind of access path types should not be created.
  SecondaryEngineFlags m_engine_flags;

  /// Pointer to a function that modifies the cost estimates of an access path
  /// for execution in a secondary storage engine, or nullptr otherwise.
  secondary_engine_modify_access_path_cost_t m_secondary_engine_cost_hook;

  /// If not nullptr, we store human-readable optimizer trace information here.
  string *m_trace;

  /// A map of tables that can never be on the right side of any join,
  /// ie., they have to be leftmost in the tree. This only affects recursive
  /// table references (ie., when WITH RECURSIVE is in use); they work by
  /// continuously tailing new records, which wouldn't work if we were to
  /// scan them multiple times or put them in a hash table. Naturally,
  /// there must be zero or one bit here; the common case is zero.
  NodeMap forced_leftmost_table = 0;

  /// A special MEM_ROOT for allocating OverflowBitsets that we might end up
  /// discarding, ie. for AccessPaths that do not yet live in m_access_paths.
  /// For any AccessPath that is to have a permanent life (ie., not be
  /// immediately discarded as inferior), the OverflowBitset _must_ be taken
  /// out of this MEM_ROOT and onto the regular one, as it is cleared often.
  /// (This significantly reduces the amount of memory used in situations
  /// where lots of AccessPaths are produced and discarded. Of course,
  /// it only matters for queries with >= 64 predicates.)
  ///
  /// The copying is using CommitBitsetsToHeap(). ProposeAccessPath() will
  /// automatically call CommitBitsetsToHeap() for accepted access paths,
  /// but it will not call it on any of their children. Thus, if you've
  /// added more than one AccessPath in the chain (e.g. if you add a join,
  /// then a sort of that join, and then propose the sort), you will need
  /// to make sure there are no stray bitsets left on this MEM_ROOT.
  ///
  /// Because this can be a source of subtle bugs, you should be conservative
  /// about what bitsets you put here; really, only the ones you could be
  /// allocating many of (like joins) should be candidates.
  MEM_ROOT m_overflow_bitset_mem_root;

  /// For trace use only.
  std::string PrintSet(NodeMap x) {
    std::string ret = "{";
    bool first = true;
    for (size_t node_idx : BitsSetIn(x)) {
      if (!first) {
        ret += ",";
      }
      first = false;
      ret += m_graph.nodes[node_idx].table->alias;
    }
    return ret + "}";
  }

  /// Checks whether the given engine flag is active or not.
  bool SupportedEngineFlag(SecondaryEngineFlag flag) const {
    return Overlaps(m_engine_flags, MakeSecondaryEngineFlags(flag));
  }

  void ProposeAccessPathForBaseTable(int node_idx,
                                     const char *description_for_trace,
                                     AccessPath *path);
  void ProposeAccessPathForIndex(int node_idx,
                                 OverflowBitset applied_predicates,
                                 OverflowBitset subsumed_predicates,
                                 const char *description_for_trace,
                                 AccessPath *path);
  void ProposeAccessPathWithOrderings(NodeMap nodes,
                                      FunctionalDependencySet fd_set,
                                      OrderingSet obsolete_orderings,
                                      AccessPath *path,
                                      const char *description_for_trace);
  bool ProposeTableScan(TABLE *table, int node_idx,
                        bool is_recursive_reference);
  bool ProposeIndexScan(TABLE *table, int node_idx, unsigned key_idx,
                        bool reverse, int ordering_idx);
  bool ProposeRefAccess(TABLE *table, int node_idx, unsigned key_idx,
                        bool reverse, table_map allowed_parameter_tables,
                        int ordering_idx);
  bool RedundantThroughSargable(
      OverflowBitset redundant_against_sargable_predicates,
      OverflowBitset left_applied_sargable_join_predicates,
      OverflowBitset right_applied_sargable_join_predicates, NodeMap left,
      NodeMap right);
  inline pair<bool, bool> AlreadyAppliedAsSargable(
      Item *condition, const AccessPath *left_path,
      const AccessPath *right_path);
  bool ProposeAllFullTextIndexScans(TABLE *table, int node_idx);
  bool ProposeFullTextIndexScan(TABLE *table, int node_idx,
                                Item_func_match *match, int predicate_idx,
                                int ordering_idx);
  void ProposeNestedLoopJoin(NodeMap left, NodeMap right, AccessPath *left_path,
                             AccessPath *right_path, const JoinPredicate *edge,
                             bool rewrite_semi_to_inner,
                             FunctionalDependencySet new_fd_set,
                             OrderingSet new_obsolete_orderings);
  void ProposeHashJoin(NodeMap left, NodeMap right, AccessPath *left_path,
                       AccessPath *right_path, const JoinPredicate *edge,
                       FunctionalDependencySet new_fd_set,
                       OrderingSet new_obsolete_orderings,
                       bool rewrite_semi_to_inner, bool *wrote_trace);
  void ApplyPredicatesForBaseTable(int node_idx,
                                   OverflowBitset applied_predicates,
                                   OverflowBitset subsumed_predicates,
                                   bool materialize_subqueries,
                                   AccessPath *path,
                                   FunctionalDependencySet *new_fd_set);
  void ApplyDelayedPredicatesAfterJoin(
      NodeMap left, NodeMap right, const AccessPath *left_path,
      const AccessPath *right_path, int join_predicate_first,
      int join_predicate_last, bool materialize_subqueries,
      AccessPath *join_path, FunctionalDependencySet *new_fd_set);

  void CommitBitsetsToHeap(AccessPath *path) const;
  bool BitsetsAreCommitted(AccessPath *path) const;
};

/// Lists the current secondary engine flags in use. If there is no secondary
/// engine, will use a default set of permissive flags suitable for
/// non-secondary engine use.
SecondaryEngineFlags EngineFlags(const THD *thd) {
  if (thd->lex->m_sql_cmd != nullptr) {
    const handlerton *secondary_engine =
        thd->lex->m_sql_cmd->secondary_engine();
    if (secondary_engine != nullptr) {
      return secondary_engine->secondary_engine_flags;
    }
  }

  return MakeSecondaryEngineFlags(
      SecondaryEngineFlag::SUPPORTS_HASH_JOIN,
      SecondaryEngineFlag::SUPPORTS_NESTED_LOOP_JOIN);
}

/// Gets the secondary storage engine cost modification function, if any.
secondary_engine_modify_access_path_cost_t SecondaryEngineCostHook(
    const THD *thd) {
  if (thd->lex->m_sql_cmd == nullptr) {
    return nullptr;
  }
  const handlerton *secondary_engine = thd->lex->m_sql_cmd->secondary_engine();
  if (secondary_engine == nullptr) {
    return nullptr;
  } else {
    return secondary_engine->secondary_engine_modify_access_path_cost;
  }
}

/// Returns the MATCH function of a predicate that can be pushed down to a
/// full-text index. This can be done if the predicate is a MATCH function,
/// or in some cases (see IsSargableFullTextIndexPredicate() for details)
/// where the predicate is a comparison function which compares the result
/// of MATCH with a constant. For example, predicates on this form could be
/// pushed down to a full-text index:
///
///   WHERE MATCH (x) AGAINST ('search string') AND <more predicates>
///
///   WHERE MATCH (x) AGAINST ('search string') > 0.5 AND <more predicates>
///
/// Since full-text index scans return documents with positive scores only, an
/// index scan can only be used if the predicate excludes negative or zero
/// scores.
Item_func_match *GetSargableFullTextPredicate(const Predicate &predicate) {
  Item_func *func = down_cast<Item_func *>(predicate.condition);
  switch (func->functype()) {
    case Item_func::MATCH_FUNC:
      // The predicate is MATCH (x) AGAINST ('search string'), which can be
      // pushed to the index.
      return down_cast<Item_func_match *>(func->get_arg(0))->get_master();
    case Item_func::LT_FUNC:
    case Item_func::LE_FUNC:
      // The predicate is const < MATCH or const <= MATCH, with a constant value
      // which makes it pushable.
      assert(func->get_arg(0)->const_item());
      return down_cast<Item_func_match *>(func->get_arg(1))->get_master();
    case Item_func::GT_FUNC:
    case Item_func::GE_FUNC:
      // The predicate is MATCH > const or MATCH >= const, with a constant value
      // which makes it pushable.
      assert(func->get_arg(1)->const_item());
      return down_cast<Item_func_match *>(func->get_arg(0))->get_master();
    default:
      // The predicate is not on a form that can be pushed to a full-text index
      // scan. We should not get here.
      assert(false);
      return nullptr;
  }
}

/**
  Called for each table in the query block, at some arbitrary point before we
  start seeing subsets where it's joined to other tables.

  We support table scans and ref access, so we create access paths for both
  (where possible) and cost them. In this context, “tables” in a query block
  also includes virtual tables such as derived tables, so we need to figure out
  if there is a cost for materializing them.
 */
bool CostingReceiver::FoundSingleNode(int node_idx) {
  if (m_thd->is_error()) return true;

  TABLE *table = m_graph.nodes[node_idx].table;
  TABLE_LIST *tl = table->pos_in_table_list;

  // Ask the storage engine to update stats.records, if needed.
  // NOTE: ha_archive breaks without this call! (That is probably a bug in
  // ha_archive, though.)
  tl->fetch_number_of_rows();

  if (ProposeTableScan(table, node_idx, tl->is_recursive_reference())) {
    return true;
  }

  if (!Overlaps(table->file->ha_table_flags(), HA_NO_INDEX_ACCESS) &&
      !tl->is_recursive_reference()) {
    // Propose index scan (for getting interesting orderings).
    // We only consider those that are more interesting than a table scan;
    // for the others, we don't even need to create the access path and go
    // through the tournament.
    for (const ActiveIndexInfo &order_info : *m_active_indexes) {
      if (order_info.table != table) {
        continue;
      }

      const int forward_order =
          m_orderings->RemapOrderingIndex(order_info.forward_order);
      const int reverse_order =
          m_orderings->RemapOrderingIndex(order_info.reverse_order);
      for (bool reverse : {false, true}) {
        if (reverse && reverse_order == 0) {
          continue;
        }
        const int order = reverse ? reverse_order : forward_order;
        if (order != 0) {
          if (ProposeIndexScan(table, node_idx, order_info.key_idx, reverse,
                               order)) {
            return true;
          }
        }

        // Propose ref access using only sargable predicates that reference no
        // other table.
        if (ProposeRefAccess(table, node_idx, order_info.key_idx, reverse,
                             /*allowed_parameter_tables=*/0, order)) {
          return true;
        }

        // Propose ref access using all sargable predicates that also refer to
        // other tables (e.g. t1.x = t2.x). Such access paths can only be used
        // on the inner side of a nested loop join, where all the other
        // referenced tables are among the outer tables of the join. Such path
        // is called a parameterized path.
        //
        // Since indexes can have multiple parts, the access path can also end
        // up being parameterized on multiple outer tables. However, since
        // parameterized paths are less flexible in joining than
        // non-parameterized ones, it can be advantageous to not use all parts
        // of the index; it's impossible to say locally. Thus, we enumerate all
        // possible subsets of table parameters that may be useful, to make sure
        // we don't miss any such paths.
        table_map want_parameter_tables = 0;
        for (unsigned pred_idx = 0;
             pred_idx < m_graph.nodes[node_idx].sargable_predicates.size();
             ++pred_idx) {
          const SargablePredicate &sp =
              m_graph.nodes[node_idx].sargable_predicates[pred_idx];
          if (sp.field->table == table &&
              sp.field->part_of_key.is_set(order_info.key_idx) &&
              !Overlaps(sp.other_side->used_tables(),
                        PSEUDO_TABLE_BITS | table->pos_in_table_list->map())) {
            want_parameter_tables |= sp.other_side->used_tables();
          }
        }
        for (table_map allowed_parameter_tables :
             NonzeroSubsetsOf(want_parameter_tables)) {
          if (ProposeRefAccess(table, node_idx, order_info.key_idx, reverse,
                               allowed_parameter_tables, order)) {
            return true;
          }
        }
      }
    }

    if (tl->is_fulltext_searched()) {
      if (ProposeAllFullTextIndexScans(table, node_idx)) {
        return true;
      }
    }
  }

  return false;
}

// Specifies a mapping in a TABLE_REF between an index keypart and a condition,
// with the intention to satisfy the condition with the index keypart (ref
// access). Roughly comparable to Key_use in the non-hypergraph optimizer.
struct KeypartForRef {
  // The condition we are pushing down (e.g. t1.f1 = 3).
  Item *condition;

  // The field that is to be matched (e.g. t1.f1).
  Field *field;

  // The value we are matching against (e.g. 3). Could be another field.
  Item *val;

  // Whether this condition would never match if either side is NULL.
  bool null_rejecting;

  // Tables used by the condition. Necessarily includes the table “field”
  // is part of.
  table_map used_tables;
};

int WasPushedDownToRef(Item *condition, const KeypartForRef *keyparts,
                       unsigned num_keyparts) {
  for (unsigned keypart_idx = 0; keypart_idx < num_keyparts; keypart_idx++) {
    if (condition->eq(keyparts[keypart_idx].condition,
                      /*binary_cmp=*/true)) {
      return keypart_idx;
    }
  }
  return -1;
}

bool ContainsSubqueries(Item *item_arg) {
  // Nearly the same as item_arg->has_subquery(), but different for
  // Item_func_not_all, which we currently do not support.
  return WalkItem(item_arg, enum_walk::POSTFIX, [](Item *item) {
    return item->type() == Item::SUBSELECT_ITEM;
  });
}

bool CostingReceiver::ProposeRefAccess(TABLE *table, int node_idx,
                                       unsigned key_idx, bool reverse,
                                       table_map allowed_parameter_tables,
                                       int ordering_idx) {
  KEY *key = &table->key_info[key_idx];

  if (key->flags & HA_FULLTEXT) {
    return false;
  }

  // Go through each of the sargable predicates and see how many key parts
  // we can match.
  unsigned matched_keyparts = 0;
  unsigned length = 0;
  const unsigned usable_keyparts = actual_key_parts(key);
  KeypartForRef keyparts[MAX_REF_PARTS];
  table_map parameter_tables = 0;

  if (PopulationCount(allowed_parameter_tables) >
      static_cast<int>(usable_keyparts)) {
    // It is inevitable that we fail the (parameter_tables ==
    // allowed_parameter_tables) test below, so error out earlier.
    return false;
  }

  for (unsigned keypart_idx = 0;
       keypart_idx < usable_keyparts && keypart_idx < MAX_REF_PARTS;
       ++keypart_idx) {
    const KEY_PART_INFO &keyinfo = key->key_part[keypart_idx];
    bool matched_this_keypart = false;

    for (const SargablePredicate &sp :
         m_graph.nodes[node_idx].sargable_predicates) {
      if (!sp.field->part_of_key.is_set(key_idx)) {
        // Quick reject.
        continue;
      }
      Item_func_eq *item = down_cast<Item_func_eq *>(
          m_graph.predicates[sp.predicate_index].condition);
      if (sp.field->eq(keyinfo.field) &&
          comparable_in_index(item, sp.field, Field::itRAW, item->functype(),
                              sp.other_side) &&
          !(sp.field->cmp_type() == STRING_RESULT &&
            sp.field->match_collation_to_optimize_range() &&
            sp.field->charset() != item->compare_collation())) {
        // x = const. (And true const; no execution of queries
        // or stored procedures during optimization.)
        if ((sp.other_side->const_for_execution() &&
             !sp.other_side->has_subquery() &&
             !sp.other_side->is_expensive()) ||
            sp.other_side->used_tables() == OUTER_REF_TABLE_BIT) {
          matched_this_keypart = true;
          keyparts[keypart_idx].field = sp.field;
          keyparts[keypart_idx].condition = item;
          keyparts[keypart_idx].val = sp.other_side;
          keyparts[keypart_idx].null_rejecting = true;
          keyparts[keypart_idx].used_tables = item->used_tables();
          ++matched_keyparts;
          length += keyinfo.store_length;
          break;
        }

        // x = other_table.field.
        if (sp.other_side->type() == Item::FIELD_ITEM &&
            IsSubset(sp.other_side->used_tables(), allowed_parameter_tables)) {
          parameter_tables |= sp.other_side->used_tables();
          matched_this_keypart = true;
          keyparts[keypart_idx].field = sp.field;
          keyparts[keypart_idx].condition = item;
          keyparts[keypart_idx].val = sp.other_side;
          keyparts[keypart_idx].null_rejecting = true;
          keyparts[keypart_idx].used_tables = item->used_tables();
          ++matched_keyparts;
          length += keyinfo.store_length;
          break;
        }
      }
    }
    if (!matched_this_keypart) {
      break;
    }
  }
  if (matched_keyparts == 0) {
    return false;
  }
  if (parameter_tables != allowed_parameter_tables) {
    // We've already seen this before, with a more lenient subset,
    // so don't try it again.
    return false;
  }

  if (matched_keyparts < usable_keyparts &&
      (table->file->index_flags(key_idx, 0, false) & HA_ONLY_WHOLE_INDEX)) {
    if (m_trace != nullptr) {
      *m_trace += StringPrintf(
          " - %s is whole-key only, and we could only match %d/%d "
          "key parts for ref access\n",
          key->name, matched_keyparts, usable_keyparts);
    }
    return false;
  }

  if (m_trace != nullptr) {
    if (matched_keyparts < usable_keyparts) {
      *m_trace += StringPrintf(
          " - %s is applicable for ref access (using %d/%d key parts only)\n",
          key->name, matched_keyparts, usable_keyparts);
    } else {
      *m_trace +=
          StringPrintf(" - %s is applicable for ref access\n", key->name);
    }
  }

  // Create TABLE_REF for this ref, and set it up based on the chosen keyparts.
  TABLE_REF *ref = new (m_thd->mem_root) TABLE_REF;
  if (init_ref(m_thd, matched_keyparts, length, key_idx, ref)) {
    return true;
  }

  uchar *key_buff = ref->key_buff;
  uchar *null_ref_key = nullptr;
  bool null_rejecting_key = true;
  for (unsigned keypart_idx = 0; keypart_idx < matched_keyparts;
       keypart_idx++) {
    KeypartForRef *keypart = &keyparts[keypart_idx];
    const KEY_PART_INFO *keyinfo = &key->key_part[keypart_idx];

    if (init_ref_part(m_thd, keypart_idx, keypart->val, /*cond_guard=*/nullptr,
                      keypart->null_rejecting, /*const_tables=*/0,
                      keypart->used_tables, keyinfo->null_bit, keyinfo,
                      key_buff, ref)) {
      return true;
    }
    // TODO(sgunders): When we get support for REF_OR_NULL,
    // set null_ref_key = key_buff here if appropriate.
    /*
      The selected key will reject matches on NULL values if:
       - the key field is nullable, and
       - predicate rejects NULL values (keypart->null_rejecting is true), or
       - JT_REF_OR_NULL is not effective.
    */
    if ((keyinfo->field->is_nullable() || table->is_nullable()) &&
        (!keypart->null_rejecting || null_ref_key != nullptr)) {
      null_rejecting_key = false;
    }
    key_buff += keyinfo->store_length;
  }

  double num_output_rows = table->file->stats.records;

  MutableOverflowBitset applied_predicates{m_thd->mem_root,
                                           m_graph.predicates.size()};
  MutableOverflowBitset subsumed_predicates{m_thd->mem_root,
                                            m_graph.predicates.size()};
  for (size_t i = 0; i < m_graph.predicates.size(); ++i) {
    int keypart_idx = WasPushedDownToRef(m_graph.predicates[i].condition,
                                         keyparts, matched_keyparts);
    if (keypart_idx == -1) {
      continue;
    }

    if (m_graph.predicates[i].was_join_condition) {
      // This predicate was promoted from a join condition to a WHERE predicate,
      // since it was part of a cycle. For purposes of sargable predicates,
      // we always see all relevant join conditions, so skip it this time
      // so that we don't double-count its selectivity.
      applied_predicates.SetBit(i);
      continue;
    }

    num_output_rows *= m_graph.predicates[i].selectivity;
    applied_predicates.SetBit(i);

    const KeypartForRef &keypart = keyparts[keypart_idx];
    if (ref_lookup_subsumes_comparison(keypart.field, keypart.val)) {
      if (m_trace != nullptr) {
        *m_trace +=
            StringPrintf(" - %s is subsumed by ref access on %s.%s\n",
                         ItemToString(m_graph.predicates[i].condition).c_str(),
                         table->alias, keypart.field->field_name);
      }
      subsumed_predicates.SetBit(i);
    } else {
      if (m_trace != nullptr) {
        *m_trace += StringPrintf(
            " - %s is not fully subsumed by ref access on %s.%s, keeping\n",
            ItemToString(m_graph.predicates[i].condition).c_str(), table->alias,
            keypart.field->field_name);
      }
    }
  }

  // We are guaranteed to get a single row back if all of these hold:
  //
  //  - The index must be unique.
  //  - We can never query it with NULL (ie., no keyparts are nullable,
  //    or our condition is already NULL-rejecting), since NULL is
  //    an exception for unique indexes.
  //  - We use all key parts.
  //
  // This matches the logic in create_ref_for_key().
  const bool single_row = Overlaps(actual_key_flags(key), HA_NOSAME) &&
                          (!Overlaps(actual_key_flags(key), HA_NULL_PART_KEY) ||
                           null_rejecting_key) &&
                          matched_keyparts == usable_keyparts;
  if (single_row) {
    num_output_rows = std::min(num_output_rows, 1.0);
  }

  const double cost =
      EstimateCostForRefAccess(m_thd, table, key_idx, num_output_rows);

  AccessPath path;
  if (single_row) {
    path.type = AccessPath::EQ_REF;
    path.eq_ref().table = table;
    path.eq_ref().ref = ref;
    path.eq_ref().use_order = false;

    // We could set really any ordering here if we wanted to.
    // It's very rare that it should matter, though.
    path.ordering_state = m_orderings->SetOrder(ordering_idx);
  } else {
    path.type = AccessPath::REF;
    path.ref().table = table;
    path.ref().ref = ref;
    path.ref().reverse = reverse;

    // TODO(sgunders): Some storage engines, like NDB, can benefit from
    // use_order = false if we don't actually need the ordering later.
    // Consider adding a cost model for this, and then proposing both
    // with and without order.
    path.ordering_state = m_orderings->SetOrder(ordering_idx);
    path.ref().use_order = (path.ordering_state != 0);
  }

  path.num_output_rows_before_filter = num_output_rows;
  path.cost_before_filter = cost;
  path.init_cost = path.init_once_cost = 0.0;
  path.parameter_tables = GetNodeMapFromTableMap(
      parameter_tables & ~table->pos_in_table_list->map(),
      m_graph.table_num_to_node_num);

  ProposeAccessPathForIndex(node_idx, std::move(applied_predicates),
                            std::move(subsumed_predicates), key->name, &path);
  return false;
}

void CostingReceiver::ProposeAccessPathForIndex(
    int node_idx, OverflowBitset applied_predicates,
    OverflowBitset subsumed_predicates, const char *description_for_trace,
    AccessPath *path) {
  MutableOverflowBitset applied_sargable_join_predicates_tmp =
      applied_predicates.Clone(m_thd->mem_root);
  applied_sargable_join_predicates_tmp.ClearBits(0,
                                                 m_graph.num_where_predicates);
  OverflowBitset applied_sargable_join_predicates =
      std::move(applied_sargable_join_predicates_tmp);

  MutableOverflowBitset subsumed_sargable_join_predicates_tmp =
      subsumed_predicates.Clone(m_thd->mem_root);
  subsumed_sargable_join_predicates_tmp.ClearBits(0,
                                                  m_graph.num_where_predicates);
  OverflowBitset subsumed_sargable_join_predicates =
      std::move(subsumed_sargable_join_predicates_tmp);
  for (bool materialize_subqueries : {false, true}) {
    FunctionalDependencySet new_fd_set;
    ApplyPredicatesForBaseTable(node_idx, applied_predicates,
                                subsumed_predicates, materialize_subqueries,
                                path, &new_fd_set);
    path->ordering_state =
        m_orderings->ApplyFDs(path->ordering_state, new_fd_set);
    path->applied_sargable_join_predicates() = OverflowBitset::Or(
        m_thd->mem_root, path->applied_sargable_join_predicates(),
        applied_sargable_join_predicates);
    path->subsumed_sargable_join_predicates() = OverflowBitset::Or(
        m_thd->mem_root, path->subsumed_sargable_join_predicates(),
        subsumed_sargable_join_predicates);
    ProposeAccessPathWithOrderings(
        TableBitmap(node_idx), new_fd_set, /*new_obsolete_orderings=*/0, path,
        materialize_subqueries ? "mat. subq" : description_for_trace);

    if (!Overlaps(path->filter_predicates, m_graph.materializable_predicates)) {
      // Nothing to try to materialize.
      break;
    }
  }
}

bool CostingReceiver::ProposeTableScan(TABLE *table, int node_idx,
                                       bool is_recursive_reference) {
  AccessPath path;
  if (is_recursive_reference) {
    path.type = AccessPath::FOLLOW_TAIL;
    path.follow_tail().table = table;
    assert(forced_leftmost_table == 0);  // There can only be one, naturally.
    forced_leftmost_table = NodeMap{1} << node_idx;

    // This will obviously grow, and it is zero now, so force a fairly arbitrary
    // minimum.
    // TODO(sgunders): We should probably go into the CTE and look at its number
    // of expected output rows, which is another minimum.
    table->file->stats.records =
        std::max<ha_rows>(table->file->stats.records, 1000);
  } else {
    path.type = AccessPath::TABLE_SCAN;
    path.table_scan().table = table;
  }
  path.count_examined_rows = true;
  path.ordering_state = 0;

  // Doing at least one table scan (this one), so mark the query as such.
  // TODO(sgunders): Move out when we get more types and this access path could
  // be replaced by something else.
  m_thd->set_status_no_index_used();

  double num_output_rows = table->file->stats.records;
  double cost = table->file->table_scan_cost().total_cost();

  path.num_output_rows_before_filter = num_output_rows;
  path.init_cost = path.init_once_cost = 0.0;
  path.cost_before_filter = path.cost = cost;

  if (m_trace != nullptr) {
    *m_trace +=
        StringPrintf("\nFound node %s [rows=%.0f]\n",
                     m_graph.nodes[node_idx].table->alias, num_output_rows);
    for (int pred_idx : BitsSetIn(path.filter_predicates)) {
      *m_trace += StringPrintf(
          " - applied predicate %s\n",
          ItemToString(m_graph.predicates[pred_idx].condition).c_str());
    }
  }

  // See if this is an information schema table that must be filled in before
  // we scan.
  TABLE_LIST *tl = table->pos_in_table_list;
  if (tl->schema_table != nullptr && tl->schema_table->fill_table) {
    // TODO(sgunders): We don't need to allocate materialize_path on the
    // MEM_ROOT.
    AccessPath *new_path = new (m_thd->mem_root) AccessPath(path);
    AccessPath *materialize_path =
        NewMaterializeInformationSchemaTableAccessPath(m_thd, new_path, tl,
                                                       /*condition=*/nullptr);

    materialize_path->num_output_rows = path.num_output_rows;
    materialize_path->num_output_rows_before_filter =
        path.num_output_rows_before_filter;
    materialize_path->init_cost = path.cost;       // Rudimentary.
    materialize_path->init_once_cost = path.cost;  // Rudimentary.
    materialize_path->cost_before_filter = path.cost;
    materialize_path->cost = path.cost;
    materialize_path->filter_predicates = path.filter_predicates;
    materialize_path->delayed_predicates = path.delayed_predicates;
    new_path->filter_predicates.Clear();
    new_path->delayed_predicates.Clear();

    // Some information schema tables have zero as estimate, which can lead
    // to completely wild plans. Add a placeholder to make sure we have
    // _something_ to work with.
    if (materialize_path->num_output_rows_before_filter == 0) {
      new_path->num_output_rows = 1000;
      new_path->num_output_rows_before_filter = 1000;
      materialize_path->num_output_rows = 1000;
      materialize_path->num_output_rows_before_filter = 1000;
    }

    assert(!tl->uses_materialization());
    path = *materialize_path;
    assert(path.cost >= 0.0);
  } else if (tl->uses_materialization()) {
    // Move the path to stable storage, since we'll be referring to it.
    AccessPath *stable_path = new (m_thd->mem_root) AccessPath(path);

    // TODO(sgunders): We don't need to allocate materialize_path on the
    // MEM_ROOT.
    AccessPath *materialize_path;
    if (tl->is_table_function()) {
      materialize_path = NewMaterializedTableFunctionAccessPath(
          m_thd, table, tl->table_function, stable_path);
      CopyBasicProperties(*stable_path, materialize_path);
      materialize_path->cost_before_filter = materialize_path->init_cost =
          materialize_path->init_once_cost = materialize_path->cost;
      materialize_path->num_output_rows_before_filter =
          materialize_path->num_output_rows;

      if (materialize_path->num_output_rows_before_filter <= 0.0) {
        materialize_path->num_output_rows = 1000.0;
        materialize_path->num_output_rows_before_filter = 1000.0;
      }

      materialize_path->parameter_tables = GetNodeMapFromTableMap(
          tl->table_function->used_tables() & ~PSEUDO_TABLE_BITS,
          m_graph.table_num_to_node_num);
      if (Overlaps(tl->table_function->used_tables(),
                   OUTER_REF_TABLE_BIT | RAND_TABLE_BIT)) {
        // Make sure the table function is never hashed, ever.
        materialize_path->parameter_tables |= RAND_TABLE_BIT;
      }
    } else {
      bool rematerialize = tl->derived_query_expression()->uncacheable != 0;
      if (tl->common_table_expr()) {
        // Handled in clear_corr_derived_tmp_tables(), not here.
        rematerialize = false;
      }
      materialize_path = GetAccessPathForDerivedTable(
          m_thd, tl, table, rematerialize,
          /*invalidators=*/nullptr, m_need_rowid, stable_path);
      // Handle LATERAL.
      materialize_path->parameter_tables =
          GetNodeMapFromTableMap(tl->derived_query_expression()->m_lateral_deps,
                                 m_graph.table_num_to_node_num);

      // If we don't need row IDs, we also don't care about row ID safety.
      // This keeps us from retaining many extra unneeded paths.
      if (!m_need_rowid) {
        materialize_path->safe_for_rowid = AccessPath::SAFE;
      }
    }

    materialize_path->filter_predicates = path.filter_predicates;
    materialize_path->delayed_predicates = path.delayed_predicates;
    stable_path->filter_predicates.Clear();
    stable_path->delayed_predicates.Clear();
    path = *materialize_path;
    assert(path.cost >= 0.0);
  }
  assert(path.cost >= 0.0);

  ProposeAccessPathForBaseTable(node_idx,
                                /*description_for_trace=*/"", &path);
  return false;
}

bool CostingReceiver::ProposeIndexScan(TABLE *table, int node_idx,
                                       unsigned key_idx, bool reverse,
                                       int ordering_idx) {
  AccessPath path;
  path.type = AccessPath::INDEX_SCAN;
  path.index_scan().table = table;
  path.index_scan().idx = key_idx;
  path.index_scan().use_order = true;
  path.index_scan().reverse = reverse;
  path.count_examined_rows = true;
  path.ordering_state = m_orderings->SetOrder(ordering_idx);

  double num_output_rows = table->file->stats.records;
  double cost;

  // If a table scan and a primary key scan is the very same thing,
  // they should also have the same cost. However, read_cost()
  // is based on number of rows, and table_scan_cost() is based on
  // on-disk size, so it's complete potluck which one gives the
  // higher number. We force primary scan cost to be table scan cost
  // plus an arbitrary 0.1% factor, so that we will always prefer
  // table scans if we don't need the ordering (both for user experience,
  // and in case there _is_ a performance difference in the storage
  // engine), but primary index scans otherwise.
  //
  // Note that this will give somewhat more access paths than is
  // required in some cases.
  if (table->s->primary_key == key_idx &&
      table->file->primary_key_is_clustered()) {
    cost = table->file->table_scan_cost().total_cost() * 1.001;
  } else if (table->covering_keys.is_set(key_idx)) {
    // The index is covering, so we can do an index-only scan.
    cost =
        table->file->index_scan_cost(key_idx, /*ranges=*/1.0, num_output_rows)
            .total_cost();
  } else {
    cost = table->file->read_cost(key_idx, /*ranges=*/1.0, num_output_rows)
               .total_cost();
  }

  path.num_output_rows_before_filter = num_output_rows;
  path.init_cost = path.init_once_cost = 0.0;
  path.cost_before_filter = path.cost = cost;

  ProposeAccessPathForBaseTable(node_idx, table->key_info[key_idx].name, &path);
  return false;
}

// Checks if a given predicate can be subsumed by a full-text index. It can
// be subsumed if it returns TRUE for all documents returned by the full-text
// index, and FALSE for all other documents. Since a full-text index scan
// returns the documents with a positive score, predicates that are either a
// standalone call to MATCH, a comparison of MATCH > 0, or a comparison of
// 0 < MATCH, are considered subsumable.
//
// We assume that this function is only called on predicates for which
// IsSargableFullTextIndexPredicate() has returned true, so that we
// already know the predicate is a standalone MATCH function or a <, <=, >
// or >= comparing match to a constant.
bool IsSubsumableFullTextPredicate(Item_func *condition) {
  switch (condition->functype()) {
    case Item_func::MATCH_FUNC: {
      // WHERE MATCH (col) AGAINST ('search string') is subsumable.
      return true;
    }
    case Item_func::GT_FUNC: {
      // WHERE MATCH (col) AGAINST ('search string') > 0 is subsumable.
      assert(is_function_of_type(condition->get_arg(0), Item_func::FT_FUNC));
      assert(condition->get_arg(1)->const_item());
      const double value = condition->get_arg(1)->val_real();
      assert(!condition->get_arg(1)->null_value);
      return value == 0;
    }
    case Item_func::LT_FUNC: {
      // WHERE 0 < MATCH (col) AGAINST ('search string') subsumable.
      assert(condition->get_arg(0)->const_item());
      assert(is_function_of_type(condition->get_arg(1), Item_func::FT_FUNC));
      const double value = condition->get_arg(0)->val_real();
      assert(!condition->get_arg(0)->null_value);
      return value == 0;
    }
    case Item_func::GE_FUNC:
      // WHERE MATCH >= const is not subsumable, but assert the predicate is on
      // the expected form.
      assert(is_function_of_type(condition->get_arg(0), Item_func::FT_FUNC));
      assert(condition->get_arg(1)->const_item());
      return false;
    case Item_func::LE_FUNC:
      // WHERE const <= MATCH is not subsumable, but assert the predicate is on
      // the expected form.
      assert(condition->get_arg(0)->const_item());
      assert(is_function_of_type(condition->get_arg(1), Item_func::FT_FUNC));
      return false;
    default:
      // Not a sargable full-text predicate, so we don't expect to be called on
      // it.
      assert(false);
      return false;
  }
}

// Assuming that we have chosen a full-text index scan on the given predicate,
// can we pass the LIMIT of the query block as a hint to the storage engine?
//
// We can do this if we know that the number of rows seen before the LIMIT
// clause is processed, is the same number of rows as returned by the index
// scan. This is the case when:
//
// 1) It is a single-table query. No joins.
//
// 2) There is no aggregation or DISTINCT which could reduce the number of rows.
//
// 3) There is no filtering of the rows returned from the index. That is, there
// is no HAVING clause, and the WHERE clause contains no predicates apart from
// those that can be subsumed by the index.
bool IsLimitHintPushableToFullTextSearch(const Item_func_match *match,
                                         const JoinHypergraph &graph,
                                         uint64_t fulltext_predicates) {
  const Query_block *query_block = graph.query_block();
  assert(query_block->has_ft_funcs());

  // The query has a LIMIT clause.
  if (query_block->join->m_select_limit == HA_POS_ERROR) {
    return false;
  }

  // A single table, no joins.
  if (graph.nodes.size() != 1) {
    return false;
  }

  // No aggregation, DISTINCT or HAVING.
  if (query_block->is_grouped() || query_block->is_distinct() ||
      query_block->join->having_cond != nullptr) {
    return false;
  }

  // The WHERE clause contains full-text predicates only.
  if (fulltext_predicates != BitsBetween(0, graph.predicates.size())) {
    return false;
  }

  // And all the full-text predicates must be subsumed by the index scan.
  for (const Predicate &predicate : graph.predicates) {
    Item_func_match *cond = GetSargableFullTextPredicate(predicate);
    if (cond != match || !IsSubsumableFullTextPredicate(
                             down_cast<Item_func *>(predicate.condition))) {
      return false;
    }
  }

  return true;
}

// Propose full-text index scans for all full-text predicates found in the
// WHERE clause, if any. If an interesting order can be satisfied by an ordered
// full-text index scan using one of the predicates, propose an ordered scan.
// Otherwise, propose an unordered scan. (For completeness, we should have
// proposed both an ordered and an unordered scan when we have an interesting
// order. But we don't have a good estimate for the extra cost of making the
// scan ordered, so we only propose the ordered scan for simplicity. InnoDB, for
// example, uses an ordered scan regardless of whether we request it, so an
// explicitly ordered scan is no more expensive than an implicitly ordered scan,
// and it could potentially avoid a sort higher up in the query plan.)
bool CostingReceiver::ProposeAllFullTextIndexScans(TABLE *table, int node_idx) {
  for (const FullTextIndexInfo &info : *m_fulltext_searches) {
    if (info.match->table_ref != table->pos_in_table_list) {
      continue;
    }

    // Propose a full-text index scan for each predicate that uses the MATCH
    // function given by info.match. Note that several predicates can use the
    // same MATCH function, due to Item_func_match's linking equivalent callers
    // to one canonical Item_func_match object (via set_master()/get_master()).
    //
    // For example, we may have:
    //
    //   WHERE MATCH (col) AGAINST ('string') AND
    //         MATCH (col) AGAINST ('string') > 0.3
    //
    // Both MATCH invocations have the same canonical Item_func_match object,
    // since they have the same set of columns and search for the same string.
    // In this case, we want to propose two index scans, and let the optimizer
    // pick the one that gives the plan with the lowest estimated cost.
    for (size_t i : BitsSetIn(m_sargable_fulltext_predicates)) {
      Item_func_match *match =
          GetSargableFullTextPredicate(m_graph.predicates[i]);
      assert(match != nullptr);
      if (match != info.match) continue;
      if (ProposeFullTextIndexScan(table, node_idx, match, i, info.order)) {
        return true;
      }
    }

    // Even if we have no predicates, we may use a full-text index scan if it is
    // possible to pass the LIMIT clause to the index scan, and the LIMIT is no
    // greater than the number of documents returned by the index scan. We only
    // do this if the index scan produces rows in an interesting order. And only
    // if the storage engine supports the extended full-text API, which is
    // required for counting the matches in the index.
    if (m_graph.predicates.empty() && info.order != 0 &&
        IsLimitHintPushableToFullTextSearch(info.match, m_graph,
                                            m_sargable_fulltext_predicates) &&
        Overlaps(table->file->ha_table_flags(), HA_CAN_FULLTEXT_EXT)) {
      // The full-text function must be initialized before get_count() is
      // called. Even though we call init_search() on it again after the final
      // plan has been chosen, this does not mean the search is performed twice.
      if (info.match->init_search(m_thd)) {
        return true;
      }
      if (m_query_block->join->m_select_limit <= info.match->get_count()) {
        if (ProposeFullTextIndexScan(table, node_idx, info.match,
                                     /*predicate_idx=*/-1, info.order)) {
          return true;
        }
      }
    }
  }

  return false;
}

bool CostingReceiver::ProposeFullTextIndexScan(TABLE *table, int node_idx,
                                               Item_func_match *match,
                                               int predicate_idx,
                                               int ordering_idx) {
  const unsigned key_idx = match->key;
  TABLE_REF *ref = new (m_thd->mem_root) TABLE_REF;
  if (init_ref(m_thd, /*keyparts=*/1, /*length=*/0, key_idx, ref)) {
    return true;
  }
  ref->items[0] = match->key_item();

  const Predicate *predicate =
      predicate_idx == -1 ? nullptr : &m_graph.predicates[predicate_idx];
  assert(predicate_idx == -1 ||
         match == GetSargableFullTextPredicate(*predicate));

  MutableOverflowBitset applied_predicates{m_thd->mem_root,
                                           m_graph.predicates.size()};
  MutableOverflowBitset subsumed_predicates{m_thd->mem_root,
                                            m_graph.predicates.size()};
  double num_output_rows_from_index;
  double num_output_rows_from_filter;
  if (predicate == nullptr) {
    // We have no predicate. The index is used only for ordering. We only do
    // this if we have a limit.
    assert(m_query_block->join->m_select_limit != HA_POS_ERROR);
    num_output_rows_from_index = num_output_rows_from_filter =
        m_query_block->join->m_select_limit;
  } else {
    // The full-text predicate is both passed to the index and in a filter node
    // on top of the index scan. Find out how much of the selectivity to apply
    // on each node.
    double index_selectivity;
    if (IsSubsumableFullTextPredicate(
            down_cast<Item_func *>(predicate->condition))) {
      // The predicate can be fully subsumed by the index. Apply the full
      // selectivity on the index scan and mark the predicate as subsumed.
      index_selectivity = predicate->selectivity;
      subsumed_predicates.SetBit(predicate_idx);
    } else {
      // The predicate uses <, <=, > or >=, and it cannot be subsumed. For
      // example MATCH(...) AGAINST(...) > 0.5. In this case, the selectivity of
      // the MATCH function is used to estimate the number of rows that come out
      // of the index, and the selectivity of the greater-than function is used
      // to estimate the number of rows returned by the filter.
      index_selectivity = EstimateSelectivity(m_thd, match, m_trace);

      // The selectivity of the predicate is applied manually to
      // num_output_rows_from_index and num_output_rows_from_filter below, so
      // mark the predicate as applied to avoid double-counting of the
      // selectivity.
      applied_predicates.SetBit(predicate_idx);
    }

    num_output_rows_from_index = table->file->stats.records * index_selectivity;
    num_output_rows_from_filter =
        std::min(num_output_rows_from_index,
                 table->file->stats.records * predicate->selectivity);
  }

  const double cost = EstimateCostForRefAccess(m_thd, table, key_idx,
                                               num_output_rows_from_index);

  const LogicalOrderings::StateIndex ordering_state =
      m_orderings->SetOrder(ordering_idx);

  const bool use_order = ordering_state != 0;

  AccessPath *path = NewFullTextSearchAccessPath(
      m_thd, table, ref, match, use_order,
      IsLimitHintPushableToFullTextSearch(match, m_graph,
                                          m_sargable_fulltext_predicates),
      /*count_examined_rows=*/true);
  path->num_output_rows_before_filter = num_output_rows_from_index;
  path->num_output_rows = num_output_rows_from_filter;
  path->cost = path->cost_before_filter = cost;
  path->init_cost = path->init_once_cost = 0;
  path->ordering_state = ordering_state;

  ProposeAccessPathForIndex(node_idx, std::move(applied_predicates),
                            std::move(subsumed_predicates),
                            table->key_info[key_idx].name, path);
  return false;
}

void CostingReceiver::ProposeAccessPathForBaseTable(
    int node_idx, const char *description_for_trace, AccessPath *path) {
  for (bool materialize_subqueries : {false, true}) {
    FunctionalDependencySet new_fd_set;
    ApplyPredicatesForBaseTable(
        node_idx,
        /*applied_predicates=*/
        MutableOverflowBitset{m_thd->mem_root, m_graph.predicates.size()},
        /*subsumed_predicates=*/
        MutableOverflowBitset{m_thd->mem_root, m_graph.predicates.size()},
        materialize_subqueries, path, &new_fd_set);
    path->ordering_state =
        m_orderings->ApplyFDs(path->ordering_state, new_fd_set);
    ProposeAccessPathWithOrderings(
        TableBitmap(node_idx), new_fd_set, /*new_obsolete_orderings=*/0, path,
        materialize_subqueries ? "mat. subq" : description_for_trace);

    if (!Overlaps(path->filter_predicates, m_graph.materializable_predicates)) {
      // Nothing to try to materialize.
      return;
    }
  }
}

/**
  See which predicates that apply to this table. Some can be applied
  right away, some require other tables first and must be delayed.

  @param node_idx Index of the base table in the nodes array.
  @param applied_predicates Bitmap of predicates that are already
    applied by means of ref access, and should not be recalculated selectivity
    for.
  @param subsumed_predicates Bitmap of predicates that are applied
    by means of ref access and do not need to rechecked. Overrides
    applied_predicates.
  @param materialize_subqueries If true, any subqueries in the
    predicate should be materialized. (If there are multiple ones,
    this is an all-or-nothing decision, for simplicity.)
  @param [in,out] path The access path to apply the predicates to.
    Note that if materialize_subqueries is true, a FILTER access path
    will be inserted (overwriting "path", although a copy of it will
    be set as a child), as AccessPath::filter_predicates always assumes
    non-materialized subqueries.
 */
void CostingReceiver::ApplyPredicatesForBaseTable(
    int node_idx, OverflowBitset applied_predicates,
    OverflowBitset subsumed_predicates, bool materialize_subqueries,
    AccessPath *path, FunctionalDependencySet *new_fd_set) {
  double materialize_cost = 0.0;

  const NodeMap my_map = TableBitmap(node_idx);
  path->num_output_rows = path->num_output_rows_before_filter;
  path->cost = path->cost_before_filter;
  MutableOverflowBitset filter_predicates{m_thd->mem_root,
                                          m_graph.predicates.size()};
  MutableOverflowBitset delayed_predicates{m_thd->mem_root,
                                           m_graph.predicates.size()};
  new_fd_set->reset();
  for (size_t i = 0; i < m_graph.num_where_predicates; ++i) {
    if (IsBitSet(i, subsumed_predicates)) {
      // Apply functional dependencies for the base table, but no others;
      // this ensures we get the same functional dependencies set no matter what
      // access path we choose. (The ones that refer to multiple tables,
      // which are fairly rare, are not really relevant before the other
      // table(s) have been joined in.)
      if (m_graph.predicates[i].total_eligibility_set == my_map) {
        *new_fd_set |= m_graph.predicates[i].functional_dependencies;
      } else {
        // We have a WHERE predicate that refers to multiple tables,
        // that we can subsume as if it were a join condition
        // (perhaps because it was identical to an actual join condition).
        // The other side of the join will mark it as delayed, so we
        // need to do so, too. Otherwise, we would never apply the
        // associated functional dependency at the right time.
        delayed_predicates.SetBit(i);
      }
      continue;
    }
    if (m_graph.predicates[i].total_eligibility_set == my_map) {
      filter_predicates.SetBit(i);
      FilterCost cost =
          EstimateFilterCost(m_thd, path->num_output_rows,
                             m_graph.predicates[i].contained_subqueries);
      if (materialize_subqueries) {
        path->cost += cost.cost_if_materialized;
        materialize_cost += cost.cost_to_materialize;
      } else {
        path->cost += cost.cost_if_not_materialized;
      }
      if (IsBitSet(i, applied_predicates)) {
        // We already factored in this predicate when calculating
        // the selectivity of the ref access, so don't do it again.
      } else {
        path->num_output_rows *= m_graph.predicates[i].selectivity;
      }
      *new_fd_set |= m_graph.predicates[i].functional_dependencies;
    } else if (Overlaps(m_graph.predicates[i].total_eligibility_set, my_map)) {
      delayed_predicates.SetBit(i);
    }
  }
  path->filter_predicates = std::move(filter_predicates);
  path->delayed_predicates = std::move(delayed_predicates);

  if (materialize_subqueries) {
    CommitBitsetsToHeap(path);
    ExpandSingleFilterAccessPath(m_thd, path, m_query_block->join,
                                 m_graph.predicates,
                                 m_graph.num_where_predicates);
    assert(path->type == AccessPath::FILTER);
    path->filter().materialize_subqueries = true;
    path->cost += materialize_cost;  // Will be subtracted back for rescans.
    path->init_cost += materialize_cost;
    path->init_once_cost += materialize_cost;
  }
}

/**
  Called to signal that it's possible to connect the non-overlapping
  table subsets “left” and “right” through the edge given by “edge_idx”
  (which corresponds to an index in m_graph.edges), ie., we have found
  a legal subplan for joining (left ∪ right). Assign it a cost based on
  the cost of the children and the join method we use. (Currently, there
  is only one -- hash join.)

  There may be multiple such calls for the same subplan; e.g. for
  inner-joining {t1,t2,t3}, we will get calls for both {t1}/{t2,t3}
  and {t1,t2}/{t3}, and need to assign costs to both and keep the
  cheapest one. However, we will not get calls with the two subsets
  in reversed order.
 */
bool CostingReceiver::FoundSubgraphPair(NodeMap left, NodeMap right,
                                        int edge_idx) {
  if (m_thd->is_error()) return true;

  if (++m_num_seen_subgraph_pairs > 100000) {
    // Bail out; we're going to be needing graph simplification
    // (a separate worklog).
    return true;
  }

  assert(left != 0);
  assert(right != 0);
  assert((left & right) == 0);

  const JoinPredicate *edge = &m_graph.edges[edge_idx];
  if (!PassesConflictRules(left | right, edge->expr)) {
    return false;
  }

  bool is_commutative = OperatorIsCommutative(*edge->expr);

  // If we have an equi-semijoin, and the inner side is deduplicated
  // on the group given by the join predicates, we can rewrite it to an
  // inner join, which is commutative. This is a win in some cases
  // where we have an index on the outer side but not the inner side.
  // (It is rarely a significant win in hash join, especially as we
  // don't propagate orders through it, but we propose it anyway for
  // simplicity.)
  //
  // See the comment on OperatorsAreAssociative() for why we don't
  // also need to change the rules about associativity or l-asscom.
  bool can_rewrite_semi_to_inner =
      edge->expr->type == RelationalExpression::SEMIJOIN &&
      edge->ordering_idx_needed_for_semijoin_rewrite != -1;

  // Enforce that recursive references need to be leftmost.
  if (Overlaps(right, forced_leftmost_table)) {
    if (!is_commutative) {
      assert(IsSingleBitSet(forced_leftmost_table));
      const int node_idx = FindLowestBitSet(forced_leftmost_table);
      my_error(ER_CTE_RECURSIVE_FORBIDDEN_JOIN_ORDER, MYF(0),
               m_graph.nodes[node_idx].table->alias);
      return true;
    }
    swap(left, right);
  }
  if (Overlaps(left, forced_leftmost_table)) {
    is_commutative = false;
    can_rewrite_semi_to_inner = false;
  }

  auto left_it = m_access_paths.find(left);
  assert(left_it != m_access_paths.end());
  auto right_it = m_access_paths.find(right);
  assert(right_it != m_access_paths.end());

  const FunctionalDependencySet new_fd_set =
      left_it->second.active_functional_dependencies |
      right_it->second.active_functional_dependencies |
      edge->functional_dependencies;
  OrderingSet new_obsolete_orderings =
      left_it->second.obsolete_orderings | right_it->second.obsolete_orderings;
  if (edge->ordering_idx_needed_for_semijoin_rewrite >= 1 &&
      edge->ordering_idx_needed_for_semijoin_rewrite < kMaxSupportedOrderings) {
    // This ordering won't be needed anymore after the join is done,
    // so mark it as obsolete.
    new_obsolete_orderings.set(edge->ordering_idx_needed_for_semijoin_rewrite);
  }

  bool wrote_trace = false;

  for (AccessPath *right_path : right_it->second.paths) {
    assert(BitsetsAreCommitted(right_path));
    if (edge->expr->join_conditions_reject_all_rows &&
        edge->expr->type != RelationalExpression::FULL_OUTER_JOIN) {
      // If the join condition can never be true, we also don't need to read the
      // right side. For inner joins and semijoins, we can actually just skip
      // reading the left side as well, but if so, the join condition would
      // normally be pulled up into a WHERE condition (or into the join
      // condition of the next higher non-inner join), so we'll never see that
      // in practice, and thus, don't care particularly about the case. We also
      // don't need to care much about the ordering, since we don't propagate
      // the right-hand ordering properties through joins.
      AccessPath *zero_path = NewZeroRowsAccessPath(
          m_thd, right_path, "Join condition rejects all rows");
      MutableOverflowBitset applied_sargable_join_predicates =
          right_path->applied_sargable_join_predicates().Clone(
              &m_overflow_bitset_mem_root);
      applied_sargable_join_predicates.ClearBits(0,
                                                 m_graph.num_where_predicates);
      zero_path->filter_predicates =
          std::move(applied_sargable_join_predicates);
      zero_path->delayed_predicates = right_path->delayed_predicates;
      right_path = zero_path;
      CommitBitsetsToHeap(right_path);
    }
    for (AccessPath *left_path : left_it->second.paths) {
      assert(BitsetsAreCommitted(left_path));
      // For inner joins and full outer joins, the order does not matter.
      // In lieu of a more precise cost model, always keep the one that hashes
      // the fewest amount of rows. (This has lower initial cost, and the same
      // cost.) When cost estimates are supplied by the secondary engine,
      // explore both orders, since the secondary engine might unilaterally
      // decide to prefer or reject one particular order.
      if (is_commutative && m_secondary_engine_cost_hook == nullptr) {
        if (left_path->num_output_rows < right_path->num_output_rows) {
          ProposeHashJoin(right, left, right_path, left_path, edge, new_fd_set,
                          new_obsolete_orderings,
                          /*rewrite_semi_to_inner=*/false, &wrote_trace);
        } else {
          ProposeHashJoin(left, right, left_path, right_path, edge, new_fd_set,
                          new_obsolete_orderings,
                          /*rewrite_semi_to_inner=*/false, &wrote_trace);
        }
      } else {
        ProposeHashJoin(left, right, left_path, right_path, edge, new_fd_set,
                        new_obsolete_orderings,
                        /*rewrite_semi_to_inner=*/false, &wrote_trace);
        if (is_commutative || can_rewrite_semi_to_inner) {
          ProposeHashJoin(right, left, right_path, left_path, edge, new_fd_set,
                          new_obsolete_orderings,
                          /*rewrite_semi_to_inner=*/can_rewrite_semi_to_inner,
                          &wrote_trace);
        }
      }

      ProposeNestedLoopJoin(left, right, left_path, right_path, edge,
                            /*rewrite_semi_to_inner=*/false, new_fd_set,
                            new_obsolete_orderings);
      if (is_commutative || can_rewrite_semi_to_inner) {
        ProposeNestedLoopJoin(
            right, left, right_path, left_path, edge,
            /*rewrite_semi_to_inner=*/can_rewrite_semi_to_inner, new_fd_set,
            new_obsolete_orderings);
      }
      m_overflow_bitset_mem_root.ClearForReuse();
    }
  }
  return false;
}

/**
  Build an access path that deduplicates its input on a certain grouping.
  This is used for converting semijoins to inner joins. If the grouping is
  empty, all rows are the same, and we make a simple LIMIT 1 instead.
 */
AccessPath *DeduplicateForSemijoin(THD *thd, AccessPath *path,
                                   Item **semijoin_group,
                                   int semijoin_group_size) {
  AccessPath *dedup_path;
  if (semijoin_group_size == 0) {
    dedup_path = NewLimitOffsetAccessPath(thd, path, /*limit=*/1, /*offset=*/0,
                                          /*calc_found_rows=*/false,
                                          /*reject_multiple_rows=*/false,
                                          /*send_records_override=*/nullptr);
  } else {
    dedup_path = NewRemoveDuplicatesAccessPath(thd, path, semijoin_group,
                                               semijoin_group_size);
    CopyBasicProperties(*path, dedup_path);
    // TODO(sgunders): Model the actual reduction in rows somehow.
    dedup_path->cost += kAggregateOneRowCost * path->num_output_rows;
  }
  return dedup_path;
}

void CostingReceiver::ProposeHashJoin(
    NodeMap left, NodeMap right, AccessPath *left_path, AccessPath *right_path,
    const JoinPredicate *edge, FunctionalDependencySet new_fd_set,
    OrderingSet new_obsolete_orderings, bool rewrite_semi_to_inner,
    bool *wrote_trace) {
  if (!SupportedEngineFlag(SecondaryEngineFlag::SUPPORTS_HASH_JOIN)) return;

  if (Overlaps(left_path->parameter_tables, right) ||
      right_path->parameter_tables != 0) {
    // Parameterized paths must be solved by nested loop.
    // We can still have parameters from outside the join,
    // but only on the outer side.
    return;
  }

  if (Overlaps(left | right, m_fulltext_tables)) {
    // Evaluation of a full-text function requires that the underlying scan is
    // positioned on the row that contains the value to be searched. It is not
    // enough that table->record[0] contains the row; the handler needs to be
    // actually positioned on the row. This does not work so well with hash
    // joins, since they may return rows in a different order than that of the
    // underlying scan.
    //
    // For now, be conservative and don't propose a hash join if either side of
    // the join contains a full-text searched table. It is possible to be more
    // lenient and allow hash joins if all the full-text search functions on the
    // accessed tables have been fully pushed down to the table/index scan and
    // don't need to be evaluated again outside of the join.
    return;
  }

  assert(BitsetsAreCommitted(left_path));
  assert(BitsetsAreCommitted(right_path));

  AccessPath join_path;
  join_path.type = AccessPath::HASH_JOIN;
  join_path.parameter_tables =
      (left_path->parameter_tables | right_path->parameter_tables) &
      ~(left | right);
  join_path.hash_join().outer = left_path;
  join_path.hash_join().inner = right_path;
  join_path.hash_join().join_predicate = edge;
  join_path.hash_join().store_rowids = false;
  join_path.hash_join().rewrite_semi_to_inner = rewrite_semi_to_inner;
  join_path.hash_join().tables_to_get_rowid_for = 0;
  join_path.hash_join().allow_spill_to_disk = true;

  // See the equivalent code in ProposeNestedLoopJoin().
  if (rewrite_semi_to_inner) {
    int ordering_idx = edge->ordering_idx_needed_for_semijoin_rewrite;
    assert(ordering_idx != -1);
    if (ordering_idx != 0 && !m_orderings->DoesFollowOrder(
                                 left_path->ordering_state, ordering_idx)) {
      return;
    }
    assert(edge->expr->type == RelationalExpression::SEMIJOIN);

    // NOTE: We purposefully don't overwrite left_path here, so that we
    // don't have to worry about copying ordering_state etc.
    CommitBitsetsToHeap(left_path);
    join_path.hash_join().outer = DeduplicateForSemijoin(
        m_thd, left_path, edge->semijoin_group, edge->semijoin_group_size);
  }

  double num_output_rows =
      FindOutputRowsForJoin(left_path, right_path, edge,
                            /*right_path_already_applied_selectivity=*/1.0);

  // TODO(sgunders): Add estimates for spill-to-disk costs.
  const double build_cost =
      right_path->cost + right_path->num_output_rows * kHashBuildOneRowCost;
  double cost = left_path->cost + build_cost +
                left_path->num_output_rows * kHashProbeOneRowCost +
                num_output_rows * kHashReturnOneRowCost;

  // Note: This isn't strictly correct if the non-equijoin conditions
  // have selectivities far from 1.0; the cost should be calculated
  // on the number of rows after the equijoin conditions, but before
  // the non-equijoin conditions.
  cost += num_output_rows * edge->expr->join_conditions.size() *
          kApplyOneFilterCost;

  join_path.num_output_rows_before_filter = num_output_rows;
  join_path.cost_before_filter = cost;
  join_path.num_output_rows = num_output_rows;
  join_path.init_cost = build_cost + left_path->init_cost;

  const double hash_memory_used_bytes =
      edge->estimated_bytes_per_row * right_path->num_output_rows;
  if (hash_memory_used_bytes <= m_thd->variables.join_buff_size * 0.9) {
    // Fits in memory (with 10% estimation margin), so the hash table can be
    // reused.
    join_path.init_once_cost = build_cost + left_path->init_once_cost;
  } else {
    join_path.init_once_cost =
        left_path->init_once_cost + right_path->init_once_cost;
  }
  join_path.cost = cost;

  // For each scan, hash join will read the left side once and the right side
  // once, so we are as safe as the least safe of the two. (This isn't true
  // if we set spill_to_disk = false, but we never do that in the hypergraph
  // optimizer.) Note that if the right side fits entirely in RAM, we don't
  // scan it the second time (so we could make the operation _more_ safe
  // than the right side, and we should consider both ways of doing
  // an inner join), but we cannot know that when planning.
  join_path.safe_for_rowid =
      std::max(left_path->safe_for_rowid, right_path->safe_for_rowid);

  // Only trace once; the rest ought to be identical.
  if (m_trace != nullptr && !*wrote_trace) {
    *m_trace += StringPrintf(
        "\nFound sets %s and %s, connected by condition %s [rows=%.0f]\n",
        PrintSet(left).c_str(), PrintSet(right).c_str(),
        GenerateExpressionLabel(edge->expr).c_str(), join_path.num_output_rows);
    for (int pred_idx : BitsSetIn(join_path.filter_predicates)) {
      *m_trace += StringPrintf(
          " - applied (delayed) predicate %s\n",
          ItemToString(m_graph.predicates[pred_idx].condition).c_str());
    }
    *wrote_trace = true;
  }

  {
    FunctionalDependencySet filter_fd_set;
    ApplyDelayedPredicatesAfterJoin(
        left, right, left_path, right_path, edge->expr->join_predicate_first,
        edge->expr->join_predicate_last,
        /*materialize_subqueries=*/false, &join_path, &filter_fd_set);
    // Hash join destroys all ordering information (even from the left side,
    // since we may have spill-to-disk).
    join_path.ordering_state = m_orderings->ApplyFDs(
        m_orderings->SetOrder(0), new_fd_set | filter_fd_set);
    ProposeAccessPathWithOrderings(left | right, new_fd_set | filter_fd_set,
                                   new_obsolete_orderings, &join_path,
                                   "hash join");
  }

  if (Overlaps(join_path.filter_predicates,
               m_graph.materializable_predicates)) {
    FunctionalDependencySet filter_fd_set;
    ApplyDelayedPredicatesAfterJoin(
        left, right, left_path, right_path, edge->expr->join_predicate_first,
        edge->expr->join_predicate_last,
        /*materialize_subqueries=*/true, &join_path, &filter_fd_set);
    // Hash join destroys all ordering information (even from the left side,
    // since we may have spill-to-disk).
    join_path.ordering_state = m_orderings->ApplyFDs(
        m_orderings->SetOrder(0), new_fd_set | filter_fd_set);
    ProposeAccessPathWithOrderings(left | right, new_fd_set | filter_fd_set,
                                   new_obsolete_orderings, &join_path,
                                   "hash join, mat. subq");
  }
}

// Of all delayed predicates, see which ones we can apply now, and which
// ones that need to be delayed further.
void CostingReceiver::ApplyDelayedPredicatesAfterJoin(
    NodeMap left, NodeMap right, const AccessPath *left_path,
    const AccessPath *right_path, int join_predicate_first,
    int join_predicate_last, bool materialize_subqueries, AccessPath *join_path,
    FunctionalDependencySet *new_fd_set) {
  // We build up a new FD set each time; it should be the same for the same
  // left/right pair, so it is somewhat redundant, but it allows us to verify
  // that property through the assert in ProposeAccessPathWithOrderings().
  new_fd_set->reset();

  // Keep track of which multiple equalities we have created predicates for
  // so far. We use this to avoid applying redundant predicates, ie. predicates
  // that have already been checked. (This is not only to avoid unneeded work,
  // but to avoid double-counting the selectivity.)
  //
  // Avoiding redundant predicates for a multi-equality is equivalent to never
  // applying those that would cause loops in the subgraph induced by the tables
  // involved in the multi-equality. (In other words, we are building spanning
  // trees in the induced subgraph.) In general, every time we connect two
  // subgraphs, we must apply every relevant multi-equality exactly once,
  // and ignore the others. (This is vaguely reminiscent of Kruskal's algorithm
  // for constructing minimum spanning trees.)
  //
  // DPhyp only ever connects subgraphs that are not already connected
  // (ie., it already constructs spanning trees), so we know that the join
  // conditions applied earlier are never redundant wrt. the rest of the graph.
  // Thus, we only need to test the delayed predicates below; they _may_ contain
  // a multiple equality we haven't already applied, but they may also be new,
  // e.g. in this graph:
  //
  //     b
  //    /|\ .
  //   a | d
  //    \|/
  //     c
  //
  // If we have a multiple equality over {b,c,d}, and connect a-b and then a-c,
  // the edge b-c will come into play and contain a multi-equality that was not
  // applied before. We will need to apply that multi-equality (we will
  // only get one of d-b and d-c). However, if we instead connected d-b
  // and d-c, the edge b-c will now be redundant and must be ignored
  // (except for functional dependencies). We simply track which ones have been
  // applied this iteration by keeping a bitmap of them.
  uint64_t multiple_equality_bitmap = 0;
  for (int pred_idx = join_predicate_first; pred_idx < join_predicate_last;
       ++pred_idx) {
    const Predicate &pred = m_graph.predicates[pred_idx];
    if (pred.source_multiple_equality_idx != -1) {
      multiple_equality_bitmap |= uint64_t{1}
                                  << pred.source_multiple_equality_idx;
    }
  }

  double materialize_cost = 0.0;

  // filter_predicates holds both filter_predicates and
  // applied_sargable_join_predicates. Keep the information about the latter,
  // but reset the one pertaining to the former.
  MutableOverflowBitset filter_predicates =
      OverflowBitset::Or(&m_overflow_bitset_mem_root,
                         left_path->applied_sargable_join_predicates(),
                         right_path->applied_sargable_join_predicates());
  filter_predicates.ClearBits(0, m_graph.num_where_predicates);

  // Predicates we are still delaying.
  MutableOverflowBitset delayed_predicates = OverflowBitset::Xor(
      &m_overflow_bitset_mem_root, left_path->delayed_predicates,
      right_path->delayed_predicates);
  delayed_predicates.ClearBits(join_predicate_first, join_predicate_last);

  // Predicates that were delayed, but that we need to check now.
  // (We don't need to allocate a MutableOverflowBitset for this.)
  const NodeMap ready_tables = left | right;
  for (int pred_idx : BitsSetInBoth(left_path->delayed_predicates,
                                    right_path->delayed_predicates)) {
    if (pred_idx >= join_predicate_first && pred_idx < join_predicate_last) {
      continue;
    }
    const Predicate &pred = m_graph.predicates[pred_idx];
    if (IsSubset(pred.total_eligibility_set, ready_tables)) {
      const auto [already_applied_as_sargable, subsumed] =
          AlreadyAppliedAsSargable(pred.condition, left_path, right_path);
      if (pred.source_multiple_equality_idx == -1 ||
          !IsBitSet(pred.source_multiple_equality_idx,
                    multiple_equality_bitmap)) {
        if (!subsumed) {
          FilterCost cost = EstimateFilterCost(
              m_thd, join_path->num_output_rows, pred.contained_subqueries);
          if (materialize_subqueries) {
            join_path->cost += cost.cost_if_materialized;
            materialize_cost += cost.cost_to_materialize;
          } else {
            join_path->cost += cost.cost_if_not_materialized;
          }
          join_path->num_output_rows *= pred.selectivity;
          if (!already_applied_as_sargable) {
            filter_predicates.SetBit(pred_idx);
          }
        }
        if (pred.source_multiple_equality_idx != -1) {
          multiple_equality_bitmap |= uint64_t{1}
                                      << pred.source_multiple_equality_idx;
        }
      }
      *new_fd_set |= pred.functional_dependencies;
    } else {
      delayed_predicates.SetBit(pred_idx);
    }
  }
  join_path->filter_predicates = std::move(filter_predicates);
  join_path->delayed_predicates = std::move(delayed_predicates);

  if (materialize_subqueries) {
    CommitBitsetsToHeap(join_path);
    ExpandSingleFilterAccessPath(m_thd, join_path, m_query_block->join,
                                 m_graph.predicates,
                                 m_graph.num_where_predicates);
    assert(join_path->type == AccessPath::FILTER);
    join_path->filter().materialize_subqueries = true;
    join_path->cost +=
        materialize_cost;  // Will be subtracted back for rescans.
    join_path->init_cost += materialize_cost;
    join_path->init_once_cost += materialize_cost;
  }
}

string PrintCost(const AccessPath &path, const JoinHypergraph &graph,
                 const char *description_for_trace);

/**
  Check if we're about to apply a join condition that would be redundant
  with regards to an already-applied sargable predicate, ie., whether our
  join condition and the sargable predicate applies the same multiple equality.
  E.g. if we try to join {t1,t2} and {t3} along t1=t3, but the access path
  for t3 already has applied the join condition t2=t3, and these are from the
  same multiple equality, return true.

  Even though this is totally _legal_, having such a situation is bad, because

    a) It double-counts the selectivity, causing the overall row estimate
       to become too low.
    b) It causes unneeded work by adding a redundant filter.

  b) would normally cause the path to be pruned out due to cost, except that
  the artifically low row count due to a) could make the path attractive as a
  subplan of a larger join. Thus, we simply reject these joins; we'll see a
  different alternative for this join at some point that is not redundant
  (e.g., in the given example, we'd see the t2=t3 join).
 */
bool CostingReceiver::RedundantThroughSargable(
    OverflowBitset redundant_against_sargable_predicates,
    OverflowBitset left_applied_sargable_join_predicates,
    OverflowBitset right_applied_sargable_join_predicates, NodeMap left,
    NodeMap right) {
  const auto redundant_and_applied = [](uint64_t redundant_sargable,
                                        uint64_t left_applied,
                                        uint64_t right_applied) {
    return redundant_sargable & (left_applied | right_applied);
  };
  for (size_t predicate_idx :
       OverflowBitsetBitsIn<3, decltype(redundant_and_applied)>(
           {redundant_against_sargable_predicates,
            left_applied_sargable_join_predicates,
            right_applied_sargable_join_predicates},
           redundant_and_applied)) {
    // The sargable condition must work as a join condition for this join
    // (not between tables we've already joined in).
    const Predicate &sargable_predicate = m_graph.predicates[predicate_idx];
    if (Overlaps(sargable_predicate.used_nodes, left) &&
        Overlaps(sargable_predicate.used_nodes, right)) {
      return true;
    }
  }
  return false;
}

/**
  Whether the given join condition is already applied as a sargable predicate
  earlier in the tree (presumably on the right side). This is different from
  RedundantThroughSargable() in that this checks whether we have already applied
  this exact join condition earlier, while the former checks whether we are
  trying to apply a different join condition that is redundant against something
  we've applied earlier.

  The first boolean is whether “condition” is a join condition we've applied
  earlier (as sargable; so we should not count its selectivity again),
  and the second argument is whether that sargable also subsumed the entire
  join condition (so we need not apply it as a filter).
 */
pair<bool, bool> CostingReceiver::AlreadyAppliedAsSargable(
    Item *condition, const AccessPath *left_path [[maybe_unused]],
    const AccessPath *right_path) {
  const auto it = m_graph.sargable_join_predicates.find(condition);
  if (it == m_graph.sargable_join_predicates.end()) {
    return {false, false};
  }

  // NOTE: It is rare that join predicates already have been applied as
  // ref access on the outer side, but not impossible if conditions are
  // duplicated; see e.g. bug #33383388.
  const bool applied =
      IsBitSet(it->second, left_path->applied_sargable_join_predicates()) ||
      IsBitSet(it->second, right_path->applied_sargable_join_predicates());
  const bool subsumed =
      IsBitSet(it->second, left_path->subsumed_sargable_join_predicates()) ||
      IsBitSet(it->second, right_path->subsumed_sargable_join_predicates());
  if (subsumed) {
    assert(applied);
  }
  return {applied, subsumed};
}

void CostingReceiver::ProposeNestedLoopJoin(
    NodeMap left, NodeMap right, AccessPath *left_path, AccessPath *right_path,
    const JoinPredicate *edge, bool rewrite_semi_to_inner,
    FunctionalDependencySet new_fd_set, OrderingSet new_obsolete_orderings) {
  if (!SupportedEngineFlag(SecondaryEngineFlag::SUPPORTS_NESTED_LOOP_JOIN))
    return;

  if (Overlaps(left_path->parameter_tables, right)) {
    // The outer table cannot pick up values from the inner,
    // only the other way around.
    return;
  }

  assert(BitsetsAreCommitted(left_path));
  assert(BitsetsAreCommitted(right_path));

  AccessPath join_path;
  join_path.type = AccessPath::NESTED_LOOP_JOIN;
  join_path.parameter_tables =
      (left_path->parameter_tables | right_path->parameter_tables) &
      ~(left | right);
  join_path.nested_loop_join().pfs_batch_mode = false;
  join_path.nested_loop_join().already_expanded_predicates = false;
  join_path.nested_loop_join().outer = left_path;
  join_path.nested_loop_join().inner = right_path;
  if (rewrite_semi_to_inner) {
    // This join is a semijoin (which is non-commutative), but the caller wants
    // us to try to invert it anyway; or to be precise, it has already inverted
    // it for us, and wants us to make sure that's OK. This is only
    // allowed if we can remove the duplicates from the outer (originally inner)
    // side, so check that it is grouped correctly, and then deduplicate on it.
    //
    // Note that in many cases, the grouping/ordering here would be due to an
    // earlier sort-ahead inserted into the tree. (The other case is due to
    // scanning along an index, but then, we'd usually prefer to
    // use that index for lookups instead of inverting the join. It is possible,
    // though.) If so, it would have been nice to just do a deduplicating sort
    // instead, but it would require is to track deduplication information in
    // the access paths (possibly as part of the ordering state somehow) and
    // track them throughout the join tree, which we don't do at the moment.
    // Thus, there may be an inefficiency here.
    assert(edge->expr->type == RelationalExpression::SEMIJOIN);
    int ordering_idx = edge->ordering_idx_needed_for_semijoin_rewrite;
    assert(ordering_idx != -1);
    if (ordering_idx != 0 && !m_orderings->DoesFollowOrder(
                                 left_path->ordering_state, ordering_idx)) {
      return;
    }
    join_path.nested_loop_join().join_type = JoinType::INNER;

    // NOTE: We purposefully don't overwrite left_path here, so that we
    // don't have to worry about copying ordering_state etc.
    join_path.nested_loop_join().outer = DeduplicateForSemijoin(
        m_thd, left_path, edge->semijoin_group, edge->semijoin_group_size);
  } else if (edge->expr->type == RelationalExpression::STRAIGHT_INNER_JOIN) {
    join_path.nested_loop_join().join_type = JoinType::INNER;
  } else {
    join_path.nested_loop_join().join_type =
        static_cast<JoinType>(edge->expr->type);
  }
  join_path.nested_loop_join().join_predicate = edge;

  const AccessPath *inner = join_path.nested_loop_join().inner;
  double inner_rescan_cost = inner->cost - inner->init_once_cost;

  double right_path_already_applied_selectivity = 1.0;
  join_path.nested_loop_join().equijoin_predicates = OverflowBitset{};
  if (edge->expr->join_conditions_reject_all_rows) {
    // We've already taken out all rows from the right-hand side
    // (by means of a ZeroRowsIterator), so no need to add filters;
    // they'd only clutter the EXPLAIN.
    //
    // Note that for obscure cases (inner joins where the join condition
    // was not pulled up due to a pass ordering issue), we might see
    // the left and right path be switched around due to commutativity.
    assert(left_path->type == AccessPath::ZERO_ROWS ||
           right_path->type == AccessPath::ZERO_ROWS);
  } else if (!edge->expr->equijoin_conditions.empty() ||
             !edge->expr->join_conditions.empty()) {
    // Apply join filters. Don't update num_output_rows, as the join's
    // selectivity will already be applied in FindOutputRowsForJoin().
    // NOTE(sgunders): We don't model the effect of short-circuiting filters on
    // the cost here.
    double rows_after_filtering = inner->num_output_rows;

    // num_output_rows is only for cost calculation and display purposes;
    // we hard-code the use of edge->selectivity below, so that we're
    // seeing the same number of rows as for hash join. This might throw
    // the filtering cost off slightly.
    MutableOverflowBitset equijoin_predicates{
        m_thd->mem_root, edge->expr->equijoin_conditions.size()};
    int filter_idx = 0;
    for (size_t join_cond_idx = 0;
         join_cond_idx < edge->expr->equijoin_conditions.size();
         ++join_cond_idx) {
      Item_func_eq *condition = edge->expr->equijoin_conditions[join_cond_idx];
      const CachedPropertiesForPredicate &properties =
          edge->expr->properties_for_equijoin_conditions[join_cond_idx];

      const auto [already_applied_as_sargable, subsumed] =
          AlreadyAppliedAsSargable(condition, left_path, right_path);
      if (already_applied_as_sargable) {
        // This predicate was already applied as a ref access earlier.
        // Make sure not to double-count its selectivity, and also
        // that we don't reapply it if it was subsumed by the ref access.
        const auto it = m_graph.sargable_join_predicates.find(condition);
        right_path_already_applied_selectivity *=
            m_graph.predicates[it->second].selectivity;
      } else if (RedundantThroughSargable(
                     properties.redundant_against_sargable_predicates,
                     left_path->applied_sargable_join_predicates(),
                     right_path->applied_sargable_join_predicates(), left,
                     right)) {
        if (m_trace != nullptr) {
          *m_trace +=
              " - " + PrintCost(*right_path, m_graph, "") +
              " has a sargable predicate that is redundant with our join "
              "predicate, skipping\n";
        }
        return;
      }
      if (!subsumed) {
        equijoin_predicates.SetBit(filter_idx);
        inner_rescan_cost += EstimateFilterCost(m_thd, rows_after_filtering,
                                                properties.contained_subqueries)
                                 .cost_if_not_materialized;
        rows_after_filtering *= properties.selectivity;
      }
      ++filter_idx;
    }
    for (const CachedPropertiesForPredicate &properties :
         edge->expr->properties_for_join_conditions) {
      inner_rescan_cost += EstimateFilterCost(m_thd, rows_after_filtering,
                                              properties.contained_subqueries)
                               .cost_if_not_materialized;
      rows_after_filtering *= properties.selectivity;
    }
    join_path.nested_loop_join().equijoin_predicates =
        std::move(equijoin_predicates);
  }

  // Ignores the row count from filter_path; see above.
  join_path.num_output_rows_before_filter = join_path.num_output_rows =
      FindOutputRowsForJoin(left_path, right_path, edge,
                            right_path_already_applied_selectivity);
  join_path.init_cost = left_path->init_cost;
  join_path.cost_before_filter = join_path.cost =
      left_path->cost + inner->init_cost +
      inner_rescan_cost * left_path->num_output_rows;

  // Nested-loop preserves any ordering from the outer side. Note that actually,
  // the two orders are _concatenated_ (if you nested-loop join something
  // ordered on (a,b) with something joined on (c,d), the order will be
  // (a,b,c,d)), but the state machine has no way of representing that.
  join_path.ordering_state =
      m_orderings->ApplyFDs(left_path->ordering_state, new_fd_set);

  // We may scan the right side several times, but the left side maybe once.
  // So if the right side is not safe to scan for row IDs after multiple scans,
  // neither are we. But if it's safe, we're exactly as safe as the left side.
  if (right_path->safe_for_rowid != AccessPath::SAFE) {
    join_path.safe_for_rowid = AccessPath::UNSAFE;
  } else {
    join_path.safe_for_rowid = left_path->safe_for_rowid;
  }

  {
    FunctionalDependencySet filter_fd_set;
    ApplyDelayedPredicatesAfterJoin(
        left, right, left_path, right_path, edge->expr->join_predicate_first,
        edge->expr->join_predicate_last,
        /*materialize_subqueries=*/false, &join_path, &filter_fd_set);
    join_path.ordering_state = m_orderings->ApplyFDs(
        join_path.ordering_state, new_fd_set | filter_fd_set);
    ProposeAccessPathWithOrderings(
        left | right, new_fd_set | filter_fd_set, new_obsolete_orderings,
        &join_path,
        rewrite_semi_to_inner ? "dedup to inner nested loop" : "nested loop");
  }

  if (Overlaps(join_path.filter_predicates,
               m_graph.materializable_predicates)) {
    FunctionalDependencySet filter_fd_set;
    ApplyDelayedPredicatesAfterJoin(
        left, right, left_path, right_path, edge->expr->join_predicate_first,
        edge->expr->join_predicate_last,
        /*materialize_subqueries=*/true, &join_path, &filter_fd_set);
    join_path.ordering_state = m_orderings->ApplyFDs(
        join_path.ordering_state, new_fd_set | filter_fd_set);
    ProposeAccessPathWithOrderings(left | right, new_fd_set | filter_fd_set,
                                   new_obsolete_orderings, &join_path,
                                   rewrite_semi_to_inner
                                       ? "dedup to inner nested loop, mat. subq"
                                       : "nested loop, mat. subq");
  }
}

enum class PathComparisonResult {
  FIRST_DOMINATES,
  SECOND_DOMINATES,
  DIFFERENT_STRENGTHS,
  IDENTICAL,
};

// See if one access path is better than the other across all cost dimensions
// (if so, we say it dominates the other one). If not, we return
// DIFFERENT_STRENGTHS so that both must be kept.
//
// TODO(sgunders): If one path is better than the other in cost, and only
// slightly worse (e.g. 1%) in a less important metric such as init_cost,
// consider pruning the latter.
//
// TODO(sgunders): Support turning off certain cost dimensions; e.g., init_cost
// only matters if we have a LIMIT or nested loop semijoin somewhere in the
// query, and it might not matter for secondary engine.
static inline PathComparisonResult CompareAccessPaths(
    const LogicalOrderings &orderings, const AccessPath &a, const AccessPath &b,
    OrderingSet obsolete_orderings) {
  bool a_is_better = false, b_is_better = false;
  if (a.cost < b.cost) {
    a_is_better = true;
  } else if (b.cost < a.cost) {
    b_is_better = true;
  }

  if (a.init_cost < b.init_cost) {
    a_is_better = true;
  } else if (b.init_cost < a.init_cost) {
    b_is_better = true;
  }

  if (a.init_once_cost < b.init_once_cost) {
    a_is_better = true;
  } else if (b.init_once_cost < a.init_once_cost) {
    b_is_better = true;
  }

  if (a.parameter_tables != b.parameter_tables) {
    if (!IsSubset(a.parameter_tables, b.parameter_tables)) {
      b_is_better = true;
    }
    if (!IsSubset(b.parameter_tables, a.parameter_tables)) {
      a_is_better = true;
    }
  }

  // If we have a parameterized path, this means that at some point, it _must_
  // be on the right side of a nested-loop join. This destroys ordering
  // information (at least in our implementation -- see comment in
  // NestedLoopJoin()), so in this situation, consider all orderings as equal.
  // (This is a trick borrowed from Postgres to keep the number of unique access
  // paths down in such situations.)
  const int a_ordering_state = (a.parameter_tables == 0) ? a.ordering_state : 0;
  const int b_ordering_state = (b.parameter_tables == 0) ? b.ordering_state : 0;
  if (orderings.MoreOrderedThan(a_ordering_state, b_ordering_state,
                                obsolete_orderings)) {
    a_is_better = true;
  }
  if (orderings.MoreOrderedThan(b_ordering_state, a_ordering_state,
                                obsolete_orderings)) {
    b_is_better = true;
  }

  // Normally, two access paths for the same subplan should have the same
  // number of output rows. However, for parameterized paths, this need not
  // be the case; due to pushdown of sargable conditions into indexes;
  // some filters may be applied earlier, causing fewer rows to be
  // carried around temporarily (until the parameterization is resolved).
  // This can have an advantage in causing less work later even if it's
  // non-optimal now, e.g. by saving on filtering work, or having less work
  // done in other joins. Thus, we need to keep it around as an extra
  // cost dimension.
  if (a.num_output_rows < b.num_output_rows) {
    a_is_better = true;
  } else if (b.num_output_rows < a.num_output_rows) {
    b_is_better = true;
  }

  // If one path is safe for row IDs and another one is not,
  // that is also something we need to take into account.
  // Safer values have lower numerical values, so we can compare them
  // as integers.
  if (a.safe_for_rowid < b.safe_for_rowid) {
    a_is_better = true;
  } else if (b.safe_for_rowid < a.safe_for_rowid) {
    b_is_better = true;
  }

  if (!a_is_better && !b_is_better) {
    return PathComparisonResult::IDENTICAL;
  } else if (a_is_better && !b_is_better) {
    return PathComparisonResult::FIRST_DOMINATES;
  } else if (!a_is_better && b_is_better) {
    return PathComparisonResult::SECOND_DOMINATES;
  } else {
    return PathComparisonResult::DIFFERENT_STRENGTHS;
  }
}

string PrintCost(const AccessPath &path, const JoinHypergraph &graph,
                 const char *description_for_trace) {
  string str =
      StringPrintf("{cost=%.1f, init_cost=%.1f", path.cost, path.init_cost);
  if (path.init_once_cost != 0.0) {
    str += StringPrintf(", init_once_cost=%.1f", path.init_once_cost);
  }
  str += StringPrintf(", rows=%.1f", path.num_output_rows);

  // Print parameter tables, if any.
  if (path.parameter_tables != 0) {
    str += ", parm={";
    bool first = true;
    for (size_t node_idx : BitsSetIn(path.parameter_tables)) {
      if (!first) {
        str += ", ";
      }
      if ((uint64_t{1} << node_idx) == RAND_TABLE_BIT) {
        str += "<random>";
      } else {
        str += graph.nodes[node_idx].table->alias;
      }
      first = false;
    }
    str += "}";
  }

  if (path.ordering_state != 0) {
    str += StringPrintf(", order=%d", path.ordering_state);
  }

  if (path.safe_for_rowid == AccessPath::SAFE_IF_SCANNED_ONCE) {
    str += StringPrintf(", safe_for_rowid_once");
  } else if (path.safe_for_rowid == AccessPath::UNSAFE) {
    str += StringPrintf(", unsafe_for_rowid");
  }

  if (strcmp(description_for_trace, "") == 0) {
    return str + "}";
  } else {
    return str + "} [" + description_for_trace + "]";
  }
}

/// Commit OverflowBitsets in path (but not its children) to
/// stable storage (see m_overflow_bitset_mem_root).
void CostingReceiver::CommitBitsetsToHeap(AccessPath *path) const {
  if (path->filter_predicates.IsContainedIn(&m_overflow_bitset_mem_root)) {
    path->filter_predicates = path->filter_predicates.Clone(m_thd->mem_root);
  }
  if (path->delayed_predicates.IsContainedIn(&m_overflow_bitset_mem_root)) {
    path->delayed_predicates = path->delayed_predicates.Clone(m_thd->mem_root);
  }
}

/// Check if all bitsets under “path” are committed to stable storage
/// (see m_overflow_bitset_mem_root). Only relevant in debug mode,
/// as it is expensive.
[[maybe_unused]] bool CostingReceiver::BitsetsAreCommitted(
    AccessPath *path) const {
  // Verify that there are no uncommitted bitsets forgotten in children.
  bool all_ok = true;
  WalkAccessPaths(path, /*join=*/nullptr,
                  WalkAccessPathPolicy::STOP_AT_MATERIALIZATION,
                  [this, &all_ok](const AccessPath *subpath, const JOIN *) {
                    all_ok &= !subpath->filter_predicates.IsContainedIn(
                        &m_overflow_bitset_mem_root);
                    all_ok &= !subpath->delayed_predicates.IsContainedIn(
                        &m_overflow_bitset_mem_root);
                    return false;
                  });
  return all_ok;
}

/**
  Propose the given access path as an alternative to the existing access paths
  for the same task (assuming any exist at all), and hold a “tournament” to find
  whether it is better than the others. Only the best alternatives are kept,
  as defined by CompareAccessPaths(); a given access path is kept only if
  it is not dominated by any other path in the group (ie., the Pareto frontier
  is computed). This means that the following are all possible outcomes of the
  tournament:

   - The path is discarded, without ever being inserted in the list
     (dominated by at least one existing entry).
   - The path is inserted as a new alternative in the list (dominates none
     but it also not dominated by any -- or the list was empty), leaving it with
     N+1 entries.
   - The path is inserted as a new alternative in the list, but replaces one
     or more entries (dominates them).
   - The path replaces all existing alternatives, and becomes the sole entry
     in the list.

  “description_for_trace” is a short description of the inserted path
  to distinguish it in optimizer trace, if active. For instance, one might
  write “hash join” when proposing a hash join access path. It may be
  the empty string.

  If the access path is discarded, returns nullptr. Otherwise returns
  a pointer to where it was inserted. (This is useful if you need to
  call CommitBitsetsToHeap() on any of its children, or otherwise do
  work only for access paths that were kept.)
 */
AccessPath *CostingReceiver::ProposeAccessPath(
    AccessPath *path, Prealloced_array<AccessPath *, 4> *existing_paths,
    OrderingSet obsolete_orderings, const char *description_for_trace) const {
  if (m_secondary_engine_cost_hook != nullptr) {
    // If an error was raised by a previous invocation of the hook, reject all
    // paths.
    if (m_thd->is_error()) {
      return nullptr;
    }

    if (m_secondary_engine_cost_hook(m_thd, m_graph, path)) {
      // Rejected by the secondary engine.
      return nullptr;
    }
    assert(!m_thd->is_error());
    assert(path->init_cost <= path->cost);
    if (!IsEmpty(path->filter_predicates)) {
      assert(path->num_output_rows <= path->num_output_rows_before_filter);
      assert(path->cost_before_filter <= path->cost);
      assert(path->init_cost <= path->cost_before_filter);
    }
  }

  if (existing_paths->empty()) {
    if (m_trace != nullptr) {
      *m_trace += " - " + PrintCost(*path, m_graph, description_for_trace) +
                  " is first alternative, keeping\n";
    }
    AccessPath *insert_position = new (m_thd->mem_root) AccessPath(*path);
    existing_paths->push_back(insert_position);
    CommitBitsetsToHeap(insert_position);
    return insert_position;
  }

  AccessPath *insert_position = nullptr;
  int num_dominated = 0;
  for (size_t i = 0; i < existing_paths->size(); ++i) {
    PathComparisonResult result = CompareAccessPaths(
        *m_orderings, *path, *((*existing_paths)[i]), obsolete_orderings);
    if (result == PathComparisonResult::DIFFERENT_STRENGTHS) {
      continue;
    }
    if (result == PathComparisonResult::IDENTICAL ||
        result == PathComparisonResult::SECOND_DOMINATES) {
      assert(insert_position == nullptr);
      if (m_trace != nullptr) {
        *m_trace += " - " + PrintCost(*path, m_graph, description_for_trace) +
                    " is not better than existing path " +
                    PrintCost(*(*existing_paths)[i], m_graph, "") +
                    ", discarding\n";
      }
      return nullptr;
    }
    if (result == PathComparisonResult::FIRST_DOMINATES) {
      ++num_dominated;
      if (insert_position == nullptr) {
        // Replace this path by the new, better one. We continue to search for
        // other paths to dominate. Note that we don't overwrite just yet,
        // because we might want to print out the old one in optimizer trace
        // below.
        insert_position = (*existing_paths)[i];
      } else {
        // The new path is better than the old one, but we don't need to insert
        // it again. Delete the old one by moving the last one into its place
        // (this may be a no-op) and then chopping one off the end.
        (*existing_paths)[i] = existing_paths->back();
        existing_paths->pop_back();
        --i;
      }
    }
  }

  if (insert_position == nullptr) {
    if (m_trace != nullptr) {
      *m_trace += " - " + PrintCost(*path, m_graph, description_for_trace) +
                  " is potential alternative, appending to existing list: (";
      bool first = true;
      for (const AccessPath *other_path : *existing_paths) {
        if (!first) {
          *m_trace += ", ";
        }
        *m_trace += PrintCost(*other_path, m_graph, "");
        first = false;
      }
      *m_trace += ")\n";
    }
    insert_position = new (m_thd->mem_root) AccessPath(*path);
    existing_paths->emplace_back(insert_position);
    CommitBitsetsToHeap(insert_position);
    return insert_position;
  }

  if (m_trace != nullptr) {
    if (existing_paths->size() == 1) {  // Only one left.
      if (num_dominated == 1) {
        *m_trace += " - " + PrintCost(*path, m_graph, description_for_trace) +
                    " is better than previous " +
                    PrintCost(*insert_position, m_graph, "") + ", replacing\n";
      } else {
        *m_trace +=
            " - " + PrintCost(*path, m_graph, description_for_trace) +
            " is better than all previous alternatives, replacing all\n";
      }
    } else {
      assert(num_dominated > 0);
      *m_trace += StringPrintf(
          " - %s is better than %d others, replacing them, remaining are: ",
          PrintCost(*path, m_graph, description_for_trace).c_str(),
          num_dominated);
      bool first = true;
      for (const AccessPath *other_path : *existing_paths) {
        if (other_path == insert_position) {
          // Will be replaced by ourselves momentarily, so don't print it.
          continue;
        }
        if (!first) {
          *m_trace += ", ";
        }
        *m_trace += PrintCost(*other_path, m_graph, "");
        first = false;
      }
      *m_trace += ")\n";
    }
  }
  *insert_position = *path;
  CommitBitsetsToHeap(insert_position);
  return insert_position;
}

AccessPath MakeSortPathWithoutFilesort(THD *thd, AccessPath *child,
                                       ORDER *order, int ordering_state,
                                       int num_where_predicates) {
  AccessPath sort_path;
  sort_path.type = AccessPath::SORT;
  sort_path.ordering_state = ordering_state;
  if (!child->applied_sargable_join_predicates()
           .empty()) {  // Will be empty after grouping.
    MutableOverflowBitset applied_sargable_join_predicates =
        child->applied_sargable_join_predicates().Clone(thd->mem_root);
    applied_sargable_join_predicates.ClearBits(0, num_where_predicates);
    sort_path.applied_sargable_join_predicates() =
        std::move(applied_sargable_join_predicates);
  }
  sort_path.delayed_predicates = child->delayed_predicates;
  sort_path.count_examined_rows = false;
  sort_path.sort().child = child;
  sort_path.sort().filesort = nullptr;
  sort_path.sort().tables_to_get_rowid_for = 0;
  sort_path.sort().order = order;
  sort_path.sort().remove_duplicates = false;
  sort_path.sort().unwrap_rollup = true;
  sort_path.sort().use_limit = false;
  EstimateSortCost(&sort_path);
  return sort_path;
}

void CostingReceiver::ProposeAccessPathWithOrderings(
    NodeMap nodes, FunctionalDependencySet fd_set,
    OrderingSet obsolete_orderings, AccessPath *path,
    const char *description_for_trace) {
  // Insert an empty array if none exists.
  auto it_and_inserted = m_access_paths.emplace(
      nodes,
      AccessPathSet{Prealloced_array<AccessPath *, 4>{PSI_NOT_INSTRUMENTED},
                    fd_set, obsolete_orderings});
  if (!it_and_inserted.second) {
    assert(fd_set ==
           it_and_inserted.first->second.active_functional_dependencies);
    assert(obsolete_orderings ==
           it_and_inserted.first->second.obsolete_orderings);
  }

  ProposeAccessPath(path, &it_and_inserted.first->second.paths,
                    obsolete_orderings, description_for_trace);

  // Don't bother trying sort-ahead if we are done joining;
  // there's no longer anything to be ahead of, so the regular
  // sort operations will take care of it.
  if (nodes == TablesBetween(0, m_graph.nodes.size())) {
    return;
  }

  if (!SupportedEngineFlag(SecondaryEngineFlag::SUPPORTS_NESTED_LOOP_JOIN) &&
      SupportedEngineFlag(SecondaryEngineFlag::AGGREGATION_IS_UNORDERED)) {
    // If sortahead cannot propagate through joins to ORDER BY,
    // and also cannot propagate from anything to aggregation or
    // from aggregation to ORDER BY, it is pointless, so don't try.
    // Note that this also removes rewrite to semijoin via duplicate
    // removal, but that's fine, as it is rarely useful without having
    // nested loops against an index on the outer side.
    return;
  }

  // Don't try to sort-ahead parameterized paths; see the comment in
  // CompareAccessPaths for why.
  if (path->parameter_tables != 0) {
    return;
  }

  path = GetSafePathToSort(m_thd, m_query_block->join, path, m_need_rowid);

  // Try sort-ahead for all interesting orderings.
  // (For the final sort, this might not be so much _ahead_, but still
  // potentially useful, if there are multiple orderings where one is a
  // superset of the other.)
  bool path_is_on_heap = false;
  for (const SortAheadOrdering &sort_ahead_ordering : *m_sort_ahead_orderings) {
    if (!IsSubset(sort_ahead_ordering.required_nodes, nodes)) {
      continue;
    }
    if (sort_ahead_ordering.aggregates_required) {
      // For sort-ahead, we don't have any aggregates yet
      // (since we never group-ahead).
      continue;
    }

    LogicalOrderings::StateIndex new_state = m_orderings->ApplyFDs(
        m_orderings->SetOrder(sort_ahead_ordering.ordering_idx), fd_set);
    if (!m_orderings->MoreOrderedThan(new_state, path->ordering_state,
                                      obsolete_orderings)) {
      continue;
    }

    AccessPath sort_path =
        MakeSortPathWithoutFilesort(m_thd, path, sort_ahead_ordering.order,
                                    new_state, m_graph.num_where_predicates);

    char buf[256];
    if (m_trace != nullptr) {
      if (description_for_trace[0] == '\0') {
        snprintf(buf, sizeof(buf), "sort(%d)",
                 sort_ahead_ordering.ordering_idx);
      } else {
        snprintf(buf, sizeof(buf), "%s, sort(%d)", description_for_trace,
                 sort_ahead_ordering.ordering_idx);
      }
    }
    AccessPath *insert_position =
        ProposeAccessPath(&sort_path, &it_and_inserted.first->second.paths,
                          obsolete_orderings, buf);
    if (insert_position != nullptr && !path_is_on_heap) {
      path = new (m_thd->mem_root) AccessPath(*path);
      CommitBitsetsToHeap(path);
      insert_position->sort().child = path;
      assert(BitsetsAreCommitted(insert_position));
      path_is_on_heap = true;
    }
  }
}

bool CheckSupportedQuery(THD *thd) {
  if (thd->lex->m_sql_cmd != nullptr &&
      thd->lex->m_sql_cmd->using_secondary_storage_engine() &&
      !Overlaps(EngineFlags(thd),
                MakeSecondaryEngineFlags(
                    SecondaryEngineFlag::SUPPORTS_HASH_JOIN,
                    SecondaryEngineFlag::SUPPORTS_NESTED_LOOP_JOIN))) {
    my_error(ER_HYPERGRAPH_NOT_SUPPORTED_YET, MYF(0),
             "the secondary engine in use");
    return true;
  }
  return false;
}

/**
  Set up an access path for streaming or materializing through a temporary
  table. If none is needed (because earlier iterators already materialize
  what needs to be done), returns the path itself.

  The actual temporary table will be created and filled out during finalization.
 */
AccessPath *CreateMaterializationOrStreamingPath(THD *thd, JOIN *join,
                                                 AccessPath *path,
                                                 bool need_rowid,
                                                 bool copy_items) {
  // See if later sorts will need row IDs from us or not.
  if (!need_rowid) {
    // The common case; we can use streaming.
    if (!copy_items) {
      // StreamingIterator exists only to copy items, so we don't need an
      // iterator here at all.
      return path;
    }
    AccessPath *stream_path = NewStreamingAccessPath(
        thd, path, join, /*temp_table_param=*/nullptr, /*temp_table=*/nullptr,
        /*ref_slice=*/-1);
    stream_path->num_output_rows = path->num_output_rows;
    stream_path->cost = path->cost;
    stream_path->init_cost = path->init_cost;
    stream_path->init_once_cost =
        0.0;  // Never recoverable across query blocks.
    stream_path->num_output_rows_before_filter = stream_path->num_output_rows;
    stream_path->cost_before_filter = stream_path->cost;
    stream_path->ordering_state = path->ordering_state;
    stream_path->safe_for_rowid = path->safe_for_rowid;
    return stream_path;
  } else {
    // Filesort needs sort by row ID, possibly because large blobs are
    // involved, so we need to actually materialize. (If we wanted a
    // smaller temporary table at the expense of more seeks, we could
    // materialize only aggregate functions and do a multi-table sort
    // by docid, but this situation is rare, so we go for simplicity.)
    return CreateMaterializationPath(thd, join, path, /*temp_table=*/nullptr,
                                     /*temp_table_param=*/nullptr, copy_items);
  }
}

AccessPath *GetSafePathToSort(THD *thd, JOIN *join, AccessPath *path,
                              bool need_rowid) {
  if (need_rowid && path->safe_for_rowid == AccessPath::UNSAFE) {
    // We need to materialize this path before we can sort it,
    // since it might not give us stable row IDs.
    return CreateMaterializationOrStreamingPath(
        thd, join, new (thd->mem_root) AccessPath(*path), need_rowid,
        /*copy_items=*/true);
  } else {
    return path;
  }
}

/**
  Sets up an access path for materializing the results returned from a path in a
  temporary table.
 */
AccessPath *CreateMaterializationPath(THD *thd, JOIN *join, AccessPath *path,
                                      TABLE *temp_table,
                                      Temp_table_param *temp_table_param,
                                      bool copy_items) {
  AccessPath *table_path =
      NewTableScanAccessPath(thd, temp_table, /*count_examined_rows=*/false);
  AccessPath *materialize_path = NewMaterializeAccessPath(
      thd,
      SingleMaterializeQueryBlock(thd, path, /*select_number=*/-1, join,
                                  copy_items, temp_table_param),
      /*invalidators=*/nullptr, temp_table, table_path, /*cte=*/nullptr,
      /*unit=*/nullptr, /*ref_slice=*/-1, /*rematerialize=*/true,
      /*limit_rows=*/HA_POS_ERROR, /*reject_multiple_rows=*/false);

  EstimateMaterializeCost(thd, materialize_path);
  materialize_path->ordering_state = path->ordering_state;
  return materialize_path;
}

bool IsMaterializationPath(const AccessPath *path) {
  switch (path->type) {
    case AccessPath::MATERIALIZE:
    case AccessPath::MATERIALIZED_TABLE_FUNCTION:
    case AccessPath::MATERIALIZE_INFORMATION_SCHEMA_TABLE:
      return true;
    default:
      return false;
  }
}

/**
  Goes through an item tree and collects all the sub-items that should be
  materialized if full-text search is used in combination with sort-based
  aggregation.

  This means all expressions in the SELECT list, GROUP BY clause, ORDER BY
  clause and HAVING clause that are possible to evaluate before aggregation.

  The reason why this materialization is needed, is that
  Item_func_match::val_real() can only be evaluated if the underlying scan is
  positioned on the row for which the full-text search score is to be retrieved.
  In the sort-based aggregation performed by AggregateIterator, the rows are
  returned with the underlying scan positioned on some other row (typically one
  in the next group). Even though AggregateIterator restores the contents of the
  record buffers to what they had when the scan was positioned on that row, it
  is not enough; the handler needs to be repositioned to make Item_func_match
  give the correct result. To avoid this, we materialize the results of the
  MATCH functions before they are seen by AggregateIterator.

  The most important thing to materialize is the MATCH function, but
  AggregateIterator is not currently prepared for reading some data from a
  materialized source and other data directly from the base tables, so we have
  to materialize all expressions that are to be used as input to
  AggregateIterator.

  The old optimizer does not have special code for materializing MATCH
  functions. In most cases it does not need it, because it usually performs
  aggregation by materializing all expressions (not only MATCH) in the SELECT
  list, the GROUP BY clause and the ORDER BY clause anyway. It does not,
  however, materialize the non-aggregated expressions in the HAVING clause, so
  calls to the MATCH function in the HAVING clause may give wrong results with
  the old optimizer.

  The old optimizer uses the same sort-based aggregation as the hypergraph
  optimizer for ROLLUP. The lack of materialization of MATCH expressions leads
  to wrong results also when MATCH is used in the SELECT list or the ORDER BY
  clause of a ROLLUP query.

  The materialization performed by this function makes the hypergraph optimizer
  produce correct results for the above mentioned cases where the old optimizer
  produces wrong results.
 */
void CollectItemsToMaterializeForFullTextAggregation(
    Item *root, mem_root_deque<Item *> *items) {
  // Walk through the item and materialize those sub-expressions that don't
  // depend on aggregation. We use CompileItem instead of WalkItem, because the
  // former allows skipping sub-trees. This is useful, so that we don't need to
  // materialize every sub-expression if a non-leaf item can be materialized.
  // We don't use the transformation capability of CompileItem, hence the
  // transformer argument is a simple identity function.
  CompileItem(
      root,
      [items](Item *item) {
        if (item->has_aggregation()) {
          // Since we materialize before aggregation, we cannot materialize
          // items that depend on an aggregated value. But we can look inside
          // the item and materialize those parts of the tree that don't depend
          // on aggregation.
          return true;
        } else if (item->has_rollup_expr()) {
          // Similarly, we cannot materialize items depending on rollup items,
          // but we can possibly materialize items inside the rollup item, so
          // keep looking.
          return true;
        } else if (item->type() == Item::REF_ITEM) {
          // Skip past Item_ref and look at its "real" item.
          return true;
        } else if (item->const_for_execution()) {
          // Constant items don't need materialization (and often they are
          // materialized/cached in an Item_cache anyways), so we skip such
          // items and also skip their sub-items.
          return false;
        } else {
          // Otherwise, we have an expression that does not depend on
          // aggregation to have happened, so we materialize the full expression
          // and don't look further inside of it.
          items->push_back(item);
          return false;
        }
      },
      [](Item *item) { return item; });
}

/**
  Creates a temporary table which materializes the results of all full-text
  functions that need to be accessible after aggregation. This is needed for
  sort-based aggregation on full-text searched tables if the full-text search
  score is accessed in the SELECT list, GROUP BY clause, ORDER BY clause or
  HAVING clause. See #CollectItemsToMaterializeForFullTextAggregation() for more
  details.

  @param thd the session object
  @param query_block the query block
  @param[out] temp_table the created temporary table,
    or nullptr if there are no MATCH functions that need materialization
  @param[out] temp_table_param the parameters of the created temporary table

  @returns false on success, true if an error was raised
 */
bool CreateTemporaryTableForFullTextFunctions(
    THD *thd, Query_block *query_block, TABLE **temp_table,
    Temp_table_param **temp_table_param) {
  JOIN *join = query_block->join;

  mem_root_deque<Item *> items_to_materialize(thd->mem_root);

  // Materialize all non-aggregate expressions in the SELECT list and
  // GROUP BY clause.
  for (Item *item : *join->fields) {
    CollectItemsToMaterializeForFullTextAggregation(item,
                                                    &items_to_materialize);
  }

  // Materialize all non-aggregate expressions in the HAVING clause.
  if (join->having_cond != nullptr) {
    CollectItemsToMaterializeForFullTextAggregation(join->having_cond,
                                                    &items_to_materialize);
  }

  // Materialize all non-aggregate expressions in the ORDER BY clause.
  for (ORDER *order = query_block->order_list.first; order != nullptr;
       order = order->next) {
    CollectItemsToMaterializeForFullTextAggregation(*order->item,
                                                    &items_to_materialize);
  }

  // If we didn't find any full-text functions that needed materialization, we
  // don't need a temporary table.
  if (std::none_of(items_to_materialize.begin(), items_to_materialize.end(),
                   [](Item *item) {
                     return contains_function_of_type(item, Item_func::FT_FUNC);
                   })) {
    *temp_table = nullptr;
    return false;
  }

  *temp_table_param = new (thd->mem_root) Temp_table_param;
  if (*temp_table_param == nullptr) return true;
  count_field_types(query_block, *temp_table_param, items_to_materialize,
                    /*reset_with_sum_func=*/false, /*save_sum_fields=*/false);

  *temp_table =
      create_tmp_table(thd, *temp_table_param, items_to_materialize,
                       /*group=*/nullptr, /*distinct=*/false,
                       /*save_sum_fields=*/false, query_block->active_options(),
                       /*rows_limit=*/HA_POS_ERROR, "<temporary>");
  if (*temp_table == nullptr) return true;

  // We made a new table, so make sure it gets properly cleaned up
  // at the end of execution.
  join->temp_tables.push_back(
      JOIN::TemporaryTableToCleanup{*temp_table, *temp_table_param});

  return false;
}

// Returns a map containing the node indexes of all tables referenced by a
// full-text MATCH function.
NodeMap FindFullTextSearchedTables(const JoinHypergraph &graph) {
  NodeMap tables = 0;
  for (size_t i = 0; i < graph.nodes.size(); ++i) {
    if (graph.nodes[i].table->pos_in_table_list->is_fulltext_searched()) {
      tables |= TableBitmap(i);
    }
  }
  return tables;
}

// Checks if an item represents a full-text predicate which can be satisfied by
// a full-text index scan. This can be done if the predicate is on one of the
// following forms:
//
//    MATCH(col) AGAINST ('search string')
//    MATCH(col) AGAINST ('search string') > const, where const >= 0
//    MATCH(col) AGAINST ('search string') >= const, where const > 0
//    const < MATCH(col) AGAINST ('search string'), where const >= 0
//    const <= MATCH(col) AGAINST ('search string'), where const > 0
//
// That is, the predicate must return FALSE if MATCH returns zero. The predicate
// cannot be pushed to an index scan if it returns TRUE when MATCH returns zero,
// because a full-text index scan only returns documents with a positive score.
//
// If the item is sargable, the function returns true.
bool IsSargableFullTextIndexPredicate(Item *condition) {
  if (condition->type() != Item::FUNC_ITEM) {
    return false;
  }

  Item_func *func = down_cast<Item_func *>(condition);
  int const_arg_idx = -1;
  bool is_greater_than_op;
  switch (func->functype()) {
    case Item_func::MATCH_FUNC:
      // A standalone MATCH in WHERE is pushable to a full-text index.
      return true;
    case Item_func::GT_FUNC:
      // MATCH > const is pushable to a full-text index if const >= 0. Checked
      // after the switch.
      const_arg_idx = 1;
      is_greater_than_op = true;
      break;
    case Item_func::GE_FUNC:
      // MATCH >= const is pushable to a full-text index if const > 0. Checked
      // after the switch.
      const_arg_idx = 1;
      is_greater_than_op = false;
      break;
    case Item_func::LT_FUNC:
      // Normalize const < MATCH to MATCH > const.
      const_arg_idx = 0;
      is_greater_than_op = true;
      break;
    case Item_func::LE_FUNC:
      // Normalize const <= MATCH to MATCH >= const.
      const_arg_idx = 0;
      is_greater_than_op = false;
      break;
    default:
      // Other kinds of predicates are not pushable to a full-text index.
      return false;
  }

  assert(func->argument_count() == 2);
  assert(const_arg_idx == 0 || const_arg_idx == 1);

  // Only pushable if we have a MATCH function greater-than(-or-equal) a
  // constant value.
  Item *const_arg = func->get_arg(const_arg_idx);
  Item *match_arg = func->get_arg(1 - const_arg_idx);
  if (!is_function_of_type(match_arg, Item_func::FT_FUNC) ||
      !const_arg->const_item()) {
    return false;
  }

  // Evaluate the constant.
  const double value = const_arg->val_real();
  if (const_arg->null_value) {
    // MATCH <op> NULL cannot be pushed to a full-text index.
    return false;
  }

  // Check if the constant is high enough to exclude MATCH = 0, which is the1
  // requirement for being pushable to a full-text index.
  if (is_greater_than_op) {
    return value >= 0;
  } else {
    return value > 0;
  }
}

// Finds all the WHERE predicates that can be satisfied by a full-text index
// scan, and returns a bitmap of those predicates. See
// IsSargableFullTextIndexPredicate() for a description of which predicates are
// sargable.
uint64_t FindSargableFullTextPredicates(const JoinHypergraph &graph) {
  uint64_t fulltext_predicates = 0;
  for (size_t i = 0; i < graph.num_where_predicates; ++i) {
    const Predicate &predicate = graph.predicates[i];
    if (IsSargableFullTextIndexPredicate(predicate.condition)) {
      fulltext_predicates |= uint64_t{1} << i;

      // If the predicate is a standalone MATCH function, flag it as such. This
      // is used by Item_func_match::can_skip_ranking() to determine if ranking
      // is needed. (We could also have set other operation hints here, like
      // FT_OP_GT and FT_OP_GE. These hints are currently not used by any of the
      // storage engines, so we don't set them for now.)
      Item_func *predicate_func = down_cast<Item_func *>(predicate.condition);
      if (predicate_func->functype() == Item_func::MATCH_FUNC) {
        Item_func_match *parent =
            down_cast<Item_func_match *>(predicate_func->get_arg(0))
                ->get_master();
        List<Item_func_match> *funcs =
            parent->table_ref->query_block->ftfunc_list;
        // We only set the hint if this is the only reference to the MATCH
        // function. If it is used other places (for example in the SELECT list
        // or in other predicates) we may still need ranking.
        if (std::none_of(funcs->begin(), funcs->end(),
                         [parent](const Item_func_match &match) {
                           return match.master == parent;
                         })) {
          parent->set_hints_op(FT_OP_NO, 0.0);
        }
      }
    }
  }
  return fulltext_predicates;
}

// Checks if any of the full-text indexes are covering for a table. If the query
// only needs the document ID and the rank, there is no need to access table
// rows. Index-only access can only be used if there is an FTS_DOC_ID column in
// the table, and no other columns must be accessed. All covering full-text
// indexes that are found, are added to TABLE::covering_keys.
void EnableFullTextCoveringIndexes(const Query_block *query_block) {
  for (Item_func_match &match : *query_block->ftfunc_list) {
    TABLE *table = match.table_ref->table;
    if (match.master == nullptr && match.key != NO_SUCH_KEY &&
        table->fts_doc_id_field != nullptr &&
        bitmap_is_set(table->read_set,
                      table->fts_doc_id_field->field_index()) &&
        bitmap_bits_set(table->read_set) == 1) {
      table->covering_keys.set_bit(match.key);
    }
  }
}

// If we are planned using in2exists, and our SELECT list has a window
// function, the HAVING condition may include parts that refer to window
// functions. (This cannot happen in standard SQL, but we add such conditions
// as part of in2exists processing.) Split them here.
void SplitHavingCondition(THD *thd, Item *cond, Item **having_cond,
                          Item **having_cond_wf) {
  if (cond == nullptr || !cond->has_wf()) {
    *having_cond = cond;
    *having_cond_wf = nullptr;
    return;
  }

  // If we have a IN-to-EXISTS with window functions and multiple columns,
  // we cannot safely push even the ones that are not dependent on the
  // window functions, as some of them would come before the window functions
  // and change their input data incorrectly. So if so, we need to delay
  // all of them.
  const bool delay_all_in2exists = cond->has_wf();

  Mem_root_array<Item *> cond_parts(thd->mem_root);
  ExtractConditions(cond, &cond_parts);

  List<Item> cond_parts_wf;
  List<Item> cond_parts_normal;
  for (Item *item : cond_parts) {
    if (item->has_wf() ||
        (delay_all_in2exists && item->created_by_in2exists())) {
      cond_parts_wf.push_back(item);
    } else {
      cond_parts_normal.push_back(item);
    }
  }
  *having_cond = CreateConjunction(&cond_parts_normal);
  *having_cond_wf = CreateConjunction(&cond_parts_wf);
}

void ApplyHavingCondition(THD *thd, Item *having_cond, Query_block *query_block,
                          const char *description_for_trace, string *trace,
                          Prealloced_array<AccessPath *, 4> *root_candidates,
                          CostingReceiver *receiver) {
  if (having_cond == nullptr) {
    return;
  }

  if (trace != nullptr) {
    *trace += description_for_trace;
  }

  Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
  for (AccessPath *root_path : *root_candidates) {
    AccessPath filter_path;
    filter_path.type = AccessPath::FILTER;
    filter_path.filter().child = root_path;
    filter_path.filter().condition = having_cond;
    // We don't currently bother with materializing subqueries
    // in HAVING, as they should be rare.
    filter_path.filter().materialize_subqueries = false;
    filter_path.num_output_rows = root_path->num_output_rows *
                                  EstimateSelectivity(thd, having_cond, trace);
    filter_path.init_cost = root_path->init_cost;
    filter_path.init_once_cost = root_path->init_once_cost;
    filter_path.cost =
        root_path->cost + EstimateFilterCost(thd, root_path->num_output_rows,
                                             having_cond, query_block)
                              .cost_if_not_materialized;
    filter_path.num_output_rows_before_filter = filter_path.num_output_rows;
    filter_path.cost_before_filter = filter_path.cost;
    // TODO(sgunders): Collect and apply functional dependencies from
    // HAVING conditions.
    filter_path.ordering_state = root_path->ordering_state;
    receiver->ProposeAccessPath(&filter_path, &new_root_candidates,
                                /*obsolete_orderings=*/0, "");
  }
  *root_candidates = std::move(new_root_candidates);
}

}  // namespace

JoinHypergraph::Node *FindNodeWithTable(JoinHypergraph *graph, TABLE *table) {
  for (JoinHypergraph::Node &node : graph->nodes) {
    if (node.table == table) {
      return &node;
    }
  }
  return nullptr;
}

Prealloced_array<AccessPath *, 4> ApplyDistinctAndOrder(
    THD *thd, const CostingReceiver &receiver,
    const LogicalOrderings &orderings, bool aggregation_is_unordered,
    int order_by_ordering_idx, int distinct_ordering_idx,
    const Mem_root_array<SortAheadOrdering> &sort_ahead_orderings,
    FunctionalDependencySet fd_set, Query_block *query_block, bool need_rowid,
    Prealloced_array<AccessPath *, 4> root_candidates, string *trace) {
  JOIN *join = query_block->join;
  assert(join->select_distinct || query_block->is_ordered());

  if (root_candidates.empty()) {
    // Nothing to do if the secondary engine has rejected all candidates.
    assert(receiver.HasSecondaryEngineCostHook());
    return root_candidates;
  }

  // If we have both ORDER BY and GROUP BY, we need a materialization step
  // after the grouping (if windowing hasn't already given us one) -- although
  // in most cases, we only need to materialize one row at a time (streaming),
  // so the performance loss should be very slight. This is because when
  // filesort only really deals with fields, not values; when it is to “output”
  // a row, it puts back the contents of the sorted table's (or tables')
  // row buffer(s). For expressions that only depend on the current row, such as
  // (f1 + 1), this is fine, but aggregate functions (Item_sum) depend on
  // multiple rows, so we need a field where filesort can put back its value
  // (and of course, subsequent readers need to read from that field
  // instead of trying to evaluate the Item_sum). A temporary table provides
  // just that, so we create one based on the current field list;
  // StreamingIterator (or MaterializeIterator, if we actually need to
  // materialize) will evaluate all the Items in turn and put their values
  // into the temporary table's fields.
  //
  // For simplicity, we materialize all items in the SELECT list, even those
  // that are not aggregate functions. This is a tiny performance loss,
  // but makes things simpler.
  //
  // The test on join->sum_funcs is mainly to avoid having to create temporary
  // tables in unit tests; the rationale is that if there are no aggregate
  // functions, we also cannot sort on them, and thus, we don't get the
  // problem. Note that we can't do this if sorting by row IDs, as
  // AggregateIterator doesn't preserve them (doing so would probably not be
  // worth it for something that's fairly niche).
  //
  // NOTE: If we elide the sort due to interesting orderings, this might
  // be redundant. It is fairly harmless, though.
  if ((query_block->is_explicitly_grouped() &&
       (*join->sum_funcs != nullptr ||
        join->rollup_state != JOIN::RollupState::NONE || need_rowid)) &&
      join->m_windows.is_empty()) {
    Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
    for (AccessPath *root_path : root_candidates) {
      root_path =
          CreateMaterializationOrStreamingPath(thd, join, root_path, need_rowid,
                                               /*copy_items=*/true);
      receiver.ProposeAccessPath(root_path, &new_root_candidates,
                                 /*obsolete_orderings=*/0, "");
    }
    root_candidates = std::move(new_root_candidates);
  }

  // Now create iterators for DISTINCT, if applicable.
  if (join->select_distinct) {
    if (trace != nullptr) {
      *trace += "Applying sort for DISTINCT\n";
    }

    Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
    for (AccessPath *root_path : root_candidates) {
      Ordering grouping = orderings.ordering(distinct_ordering_idx);
      if (grouping.empty()) {
        // Only const fields.
        AccessPath *limit_path = NewLimitOffsetAccessPath(
            thd, root_path, /*limit=*/1, /*offset=*/0, join->calc_found_rows,
            /*reject_multiple_rows=*/false,
            /*send_records_override=*/nullptr);
        receiver.ProposeAccessPath(limit_path, &new_root_candidates,
                                   /*obsolete_orderings=*/0, "");
        continue;
      }
      if (!aggregation_is_unordered &&
          orderings.DoesFollowOrder(root_path->ordering_state,
                                    distinct_ordering_idx)) {
        // We don't need the sort, and can do with a simpler deduplication.
        // TODO(sgunders): In some cases, we could apply LIMIT 1,
        // which would be slightly more efficient; see e.g. the test for
        // bug #33148369.
        Item **group_items = thd->mem_root->ArrayAlloc<Item *>(grouping.size());
        for (size_t i = 0; i < grouping.size(); ++i) {
          group_items[i] = orderings.item(grouping[i].item);
        }
        AccessPath *dedup_path = NewRemoveDuplicatesAccessPath(
            thd, root_path, group_items, grouping.size());
        CopyBasicProperties(*root_path, dedup_path);
        // TODO(sgunders): Model the actual reduction in rows somehow.
        dedup_path->cost += kAggregateOneRowCost * root_path->num_output_rows;
        receiver.ProposeAccessPath(dedup_path, &new_root_candidates,
                                   /*obsolete_orderings=*/0, "sort elided");
        continue;
      }

      root_path = GetSafePathToSort(thd, join, root_path, need_rowid);

      // We need to sort. Try all sort-ahead, not just the one directly
      // derived from DISTINCT clause, because a broader one might help us
      // elide ORDER BY later.
      for (const SortAheadOrdering &sort_ahead_ordering :
           sort_ahead_orderings) {
        LogicalOrderings::StateIndex ordering_state = orderings.ApplyFDs(
            orderings.SetOrder(sort_ahead_ordering.ordering_idx), fd_set);
        if (!orderings.DoesFollowOrder(ordering_state, distinct_ordering_idx)) {
          continue;
        }
        AccessPath sort_path;
        sort_path.type = AccessPath::SORT;
        sort_path.count_examined_rows = false;
        sort_path.sort().child = root_path;
        sort_path.sort().filesort = nullptr;
        sort_path.sort().remove_duplicates = true;
        sort_path.sort().unwrap_rollup = false;
        sort_path.sort().use_limit = false;

        if (aggregation_is_unordered) {
          // Even though we create a sort node for the distinct operation,
          // the engine does not actually sort the rows. (The deduplication
          // flag is the hint in this case.)
          sort_path.ordering_state = 0;
        } else {
          sort_path.ordering_state = ordering_state;
        }

        ORDER *order_copy = BuildSortAheadOrdering(
            thd, &orderings,
            orderings.ordering(sort_ahead_ordering.ordering_idx));
        sort_path.sort().order = order_copy;

        EstimateSortCost(&sort_path);
        receiver.ProposeAccessPath(&sort_path, &new_root_candidates,
                                   /*obsolete_orderings=*/0, "");
      }
    }
    root_candidates = std::move(new_root_candidates);
  }

  // Apply ORDER BY, if applicable.
  if (query_block->is_ordered()) {
    Mem_root_array<TABLE *> tables = CollectTables(
        thd, root_candidates[0]);  // Should be same for all paths.
    if (trace != nullptr) {
      *trace += "Applying sort for ORDER BY\n";
    }

    // If we have LIMIT or OFFSET, we apply them here. This is done so that we
    // can push the LIMIT clause down to the SORT node in order to let Filesort
    // take advantage of it.
    const Query_expression *query_expression = join->query_expression();
    const ha_rows limit_rows = query_expression->select_limit_cnt;
    const ha_rows offset_rows = query_expression->offset_limit_cnt;

    Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
    for (AccessPath *root_path : root_candidates) {
      if (orderings.DoesFollowOrder(root_path->ordering_state,
                                    order_by_ordering_idx)) {
        if (limit_rows != HA_POS_ERROR || offset_rows != 0) {
          root_path = NewLimitOffsetAccessPath(
              thd, root_path, limit_rows, offset_rows, join->calc_found_rows,
              /*reject_multiple_rows=*/false,
              /*send_records_override=*/nullptr);
        }
        receiver.ProposeAccessPath(root_path, &new_root_candidates,
                                   /*obsolete_orderings=*/0, "sort elided");
      } else {
        const bool push_limit_to_filesort =
            limit_rows != HA_POS_ERROR && !join->calc_found_rows;

        root_path = GetSafePathToSort(thd, join, root_path, need_rowid);

        AccessPath *sort_path = new (thd->mem_root) AccessPath;
        sort_path->type = AccessPath::SORT;
        sort_path->count_examined_rows = false;
        sort_path->sort().child = root_path;
        sort_path->sort().filesort = nullptr;
        sort_path->sort().remove_duplicates = false;
        sort_path->sort().unwrap_rollup = false;
        sort_path->sort().use_limit = push_limit_to_filesort;
        sort_path->sort().order = query_block->order_list.first;
        EstimateSortCost(sort_path,
                         push_limit_to_filesort ? limit_rows : HA_POS_ERROR);

        // If we have a LIMIT clause that is not pushed down to the filesort, or
        // if we have an OFFSET clause, we need to add a LIMIT_OFFSET path on
        // top of the SORT node.
        if ((limit_rows != HA_POS_ERROR && !push_limit_to_filesort) ||
            offset_rows != 0) {
          sort_path = NewLimitOffsetAccessPath(
              thd, sort_path, limit_rows, offset_rows, join->calc_found_rows,
              /*reject_multiple_rows=*/false,
              /*send_records_override=*/nullptr);
        }
        receiver.ProposeAccessPath(sort_path, &new_root_candidates,
                                   /*obsolete_orderings=*/0, "");
      }
    }
    root_candidates = std::move(new_root_candidates);
  }
  return root_candidates;
}

static AccessPath *ApplyWindow(THD *thd, AccessPath *root_path, Window *window,
                               JOIN *join, bool need_rowid_for_window) {
  AccessPath *window_path =
      NewWindowAccessPath(thd, root_path, /*temp_table_param=*/nullptr,
                          /*ref_slice=*/-1, window->needs_buffering());
  window_path->window().temp_table = nullptr;
  window_path->window().window = window;
  CopyBasicProperties(*root_path, window_path);
  window_path->cost += kWindowOneRowCost * window_path->num_output_rows;

  // NOTE: copy_items = false, because the window iterator does the copying
  // itself.
  return CreateMaterializationOrStreamingPath(thd, join, window_path,
                                              need_rowid_for_window,
                                              /*copy_items=*/false);
}

/**
  Find the ordering that allows us to process the most unprocessed windows.
  If specified, we can also demand that the ordering satisfies one or two
  later orderings (for DISTINCT and/or ORDER BY).

  Our priorities are, in strict order:

    1. Satisfying both DISTINCT and ORDER BY (if both are given;
       but see below).
    2. Satisfying the first operation after windowing
       (which is either DISTINCT or ORDER BY).
    3. Satisfying as many windows as possible.
    4. The shortest possible ordering (as a tie-breaker).

  If first_ordering_idx is given, #2 is mandatory. #4 is so that we don't
  get strange situations where the user specifies e.g. OVER (ORDER BY i)
  and we choose an ordering i,j,k,l,... because it happened to be given
  somewhere else.

  Note that normally, it is very hard to satisfy DISTINCT for a window
  function, because generally, it isn't constant for a given input
  (by nature, it also depends on other rows). But it can happen if the
  window frame is static; see main.window_functions_interesting_orders.

  @param join                    Contains the list of windows.
  @param orderings               Logical orderings in the query block.
  @param sort_ahead_orderings    Candidate orderings to consider.
  @param fd_set                  Active functional dependencies.
  @param finished_windows        Windows to ignore.
  @param tmp_buffer              Temporary space for keeping the best list
                                 of windows so far; must be as large as
                                 the number of values.
  @param first_ordering_idx      The first ordering after the query block
                                 that we need to satisfy (-1 if none).
  @param second_ordering_idx     The second ordering after the query block
                                 that we would like to satisfy (-1 if none).
  @param [out] included_windows  Which windows can be sorted using the given
                                 ordering.

  @return An index into sort_ahead_orderings, or -1 if no ordering could
    be found that sorts at least one window (plus, if first_ordering_idx
    is set, follows that ordering).
 */
static int FindBestOrderingForWindow(
    JOIN *join, const LogicalOrderings &orderings,
    FunctionalDependencySet fd_set,
    const Mem_root_array<SortAheadOrdering> &sort_ahead_orderings,
    Bounds_checked_array<bool> finished_windows,
    Bounds_checked_array<bool> tmp_buffer, int first_ordering_idx,
    int second_ordering_idx, Bounds_checked_array<bool> included_windows) {
  if (first_ordering_idx == -1) {
    assert(second_ordering_idx == -1);
  }

  int best_ordering_idx = -1;
  bool best_following_both_orders = false;
  int best_num_matching_windows = 0;
  for (size_t i = 0; i < sort_ahead_orderings.size(); ++i) {
    const int ordering_idx = sort_ahead_orderings[i].ordering_idx;
    LogicalOrderings::StateIndex ordering_state =
        orderings.ApplyFDs(orderings.SetOrder(ordering_idx), fd_set);

    bool following_both_orders = false;
    if (first_ordering_idx != -1) {
      if (!orderings.DoesFollowOrder(ordering_state, first_ordering_idx)) {
        // Following one is mandatory.
        continue;
      }
      if (second_ordering_idx != -1) {
        if (orderings.DoesFollowOrder(ordering_state, second_ordering_idx)) {
          following_both_orders = true;
        } else if (best_following_both_orders) {
          continue;
        }
      }
    }

    // If we are doing sortahead for DISTINCT/ORDER BY:
    // Find windows that are referred to by DISTINCT/ORDER BY,
    // and disallow them. E.g., if we have
    //
    //   SELECT FOO() OVER w1 AS a ... ORDER BY a,
    //
    // we cannot put w1 in the group of windows that are to be sorted
    // together with ORDER BY.
    for (Window &window : join->m_windows) {
      window.m_mark = false;
    }
    Ordering ordering = orderings.ordering(ordering_idx);
    bool any_wf = false;
    for (OrderElement elem : ordering) {
      WalkItem(orderings.item(elem.item), enum_walk::PREFIX,
               [&any_wf](Item *item) {
                 if (item->m_is_window_function) {
                   down_cast<Item_sum *>(item)->window()->m_mark = true;
                   any_wf = true;
                 }
                 return false;
               });
      if (first_ordering_idx == -1 && any_wf) {
        break;
      }
    }

    // If we are doing sorts _before_ DISTINCT/ORDER BY, simply disallow
    // any sorts on window functions. There should be better options
    // available for us.
    if (first_ordering_idx == -1 && any_wf) {
      continue;
    }

    // Now find out which windows can be processed under this order.
    // We use tmp_buffer to hold which one we selected,
    // and then copy it into included_windows if we are the best so far.
    int num_matching_windows = 0;
    for (size_t window_idx = 0; window_idx < join->m_windows.size();
         ++window_idx) {
      Window *window = join->m_windows[window_idx];
      if (window->m_mark || finished_windows[window_idx] ||
          !orderings.DoesFollowOrder(ordering_state, window->m_ordering_idx)) {
        tmp_buffer[window_idx] = false;
        continue;
      }
      tmp_buffer[window_idx] = true;
      ++num_matching_windows;
    }
    if (num_matching_windows == 0) {
      continue;
    }

    bool is_best;
    if (best_ordering_idx == -1) {
      is_best = true;
    } else if (following_both_orders < best_following_both_orders) {
      is_best = false;
    } else if (following_both_orders > best_following_both_orders) {
      is_best = true;
    } else if (num_matching_windows < best_num_matching_windows) {
      is_best = false;
    } else if (num_matching_windows > best_num_matching_windows) {
      is_best = true;
    } else if (orderings.ordering(ordering_idx).size() <
               orderings
                   .ordering(
                       sort_ahead_orderings[best_ordering_idx].ordering_idx)
                   .size()) {
      is_best = true;
    } else {
      is_best = false;
    }
    if (is_best) {
      best_ordering_idx = i;
      best_following_both_orders = following_both_orders;
      best_num_matching_windows = num_matching_windows;
      memcpy(included_windows.array(), tmp_buffer.array(),
             sizeof(bool) * included_windows.size());
    }
  }
  return best_ordering_idx;
}

/**
  Apply window functions.

  Ordering of window functions is a tricky topic. We can apply window functions
  in any order that we'd like, but we would like to do as few sorts as possible.
  In its most general form, this would entail solving an instance of the
  traveling salesman problem (TSP), and although the number of windows is
  typically small (one or two in most queries), this can blow up for large
  numbers of windows.

  Thankfully, window functions never add or remove rows. We also assume that all
  sorts are equally expensive (which isn't really true, as ones on more columns
  take more processing time and buffer, but it's close enough in practice),
  and we also ignore the fact that as we compute more buffers, the temporary
  tables and sort buffers will get more columns. These assumptions, combined
  with some reasonable assumptions about ordering transitivity (if an ordering A
  is more sorted than an ordering B, and B > C, then also A > C -- the only
  thing that can disturb this is groupings, which we ignore for the sake of
  simplicity), mean that we need to care _only_ about the number of sorts, and
  can do them greedily. Thus, at any point, we pick the ordering that allows us
  to process the largest number of windows, process them, remove them from
  consideration, and repeat until there are none left.

  There is one more ordering complication; after windowing, we may have DISTINCT
  and/or ORDER BY, which may also benefit from groupings/orderings we leave
  after the last window. Thus, first of all, we see if there's an ordering that
  can satisfy them (ideally both if possible) _and_ at least one window; if so,
  we save that ordering and those windows for last.

  Temporary tables are set up in FinalizePlanForQueryBlock(). This is so that
  it is easier to have multiple different orderings for the temporary table
  parameters later.
 */
static Prealloced_array<AccessPath *, 4> ApplyWindowFunctions(
    THD *thd, const CostingReceiver &receiver,
    const LogicalOrderings &orderings, FunctionalDependencySet fd_set,
    bool aggregation_is_unordered, int order_by_ordering_idx,
    int distinct_ordering_idx, const JoinHypergraph &graph,
    const Mem_root_array<SortAheadOrdering> &sort_ahead_orderings,
    Query_block *query_block, int num_where_predicates, bool need_rowid,
    Prealloced_array<AccessPath *, 4> root_candidates, string *trace) {
  JOIN *join = query_block->join;

  // Figure out if windows need row IDs or not; we won't create
  // the temporary tables before later (since the optimal ordering
  // of windows is cost-based), so this is a conservative check.
  bool need_rowid_for_window = need_rowid;
  if (!need_rowid) {
    for (Item *item : *join->fields) {
      if (item->m_is_window_function && item->is_blob_field()) {
        need_rowid_for_window = true;
        break;
      }
    }
  }

  // Windows we're done processing, or have reserved for the last block.
  auto finished_windows =
      Bounds_checked_array<bool>::Alloc(thd->mem_root, join->m_windows.size());

  // Windows we've reserved for the last block (see function comment).
  auto reserved_windows =
      Bounds_checked_array<bool>::Alloc(thd->mem_root, join->m_windows.size());

  // Temporary space for FindBestOrderingForWindow().
  auto tmp =
      Bounds_checked_array<bool>::Alloc(thd->mem_root, join->m_windows.size());

  // Windows we're doing in this pass.
  auto included_windows =
      Bounds_checked_array<bool>::Alloc(thd->mem_root, join->m_windows.size());

  if (trace) {
    *trace += "\n";
  }
  Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
  for (AccessPath *root_path : root_candidates) {
    if (trace) {
      *trace += "Considering window order on top of " +
                PrintCost(*root_path, graph, "") + "\n";
    }

    // First, go through and check which windows we can do without
    // any reordering, just based on the input ordering we get.
    int num_windows_left = join->m_windows.size();
    for (size_t window_idx = 0; window_idx < join->m_windows.size();
         ++window_idx) {
      Window *window = join->m_windows[window_idx];
      if (window->m_ordering_idx == -1 || join->implicit_grouping ||
          orderings.DoesFollowOrder(root_path->ordering_state,
                                    window->m_ordering_idx)) {
        if (trace) {
          *trace += std::string(" - window ") + window->printable_name() +
                    " does not need further sorting\n";
        }
        root_path =
            ApplyWindow(thd, root_path, window, join, need_rowid_for_window);
        finished_windows[window_idx] = true;
        --num_windows_left;
      } else {
        finished_windows[window_idx] = false;
      }
    }

    // Now, see if we can find an ordering that allows us to process
    // at least one window _and_ an operation after the windowing
    // (DISTINCT, ORDER BY). If so, that ordering will be our last.
    int final_sort_ahead_ordering_idx = -1;
    if ((!aggregation_is_unordered || distinct_ordering_idx == -1) &&
        (distinct_ordering_idx != -1 || order_by_ordering_idx != -1)) {
      int first_ordering_idx, second_ordering_idx;
      if (distinct_ordering_idx == -1) {
        first_ordering_idx = order_by_ordering_idx;
        second_ordering_idx = -1;
      } else {
        first_ordering_idx = distinct_ordering_idx;
        second_ordering_idx = order_by_ordering_idx;
      }
      final_sort_ahead_ordering_idx = FindBestOrderingForWindow(
          join, orderings, fd_set, sort_ahead_orderings, finished_windows, tmp,
          first_ordering_idx, second_ordering_idx, reserved_windows);
      for (size_t window_idx = 0; window_idx < join->m_windows.size();
           ++window_idx) {
        finished_windows[window_idx] |= reserved_windows[window_idx];
      }
    }

    // Now all the other orderings, eventually reaching all windows.
    while (num_windows_left > 0) {
      int sort_ahead_ordering_idx = FindBestOrderingForWindow(
          join, orderings, fd_set, sort_ahead_orderings, finished_windows, tmp,
          /*first_ordering_idx=*/-1,
          /*second_ordering_idx=*/-1, included_windows);
      Bounds_checked_array<bool> windows_this_iteration = included_windows;
      if (sort_ahead_ordering_idx == -1) {
        // None left, so take the one we've saved for last.
        sort_ahead_ordering_idx = final_sort_ahead_ordering_idx;
        windows_this_iteration = reserved_windows;
        final_sort_ahead_ordering_idx = -1;
      }
      if (sort_ahead_ordering_idx == -1) {
        // Should never happen.
        assert(false);
        break;
      }

      AccessPath sort_path = MakeSortPathWithoutFilesort(
          thd, root_path, sort_ahead_orderings[sort_ahead_ordering_idx].order,
          /*ordering_state=*/0, num_where_predicates);
      sort_path.ordering_state = orderings.ApplyFDs(
          orderings.SetOrder(
              sort_ahead_orderings[sort_ahead_ordering_idx].ordering_idx),
          fd_set);
      root_path = new (thd->mem_root) AccessPath(sort_path);

      for (size_t window_idx = 0; window_idx < join->m_windows.size();
           ++window_idx) {
        if (!windows_this_iteration[window_idx]) {
          continue;
        }
        root_path = ApplyWindow(thd, root_path, join->m_windows[window_idx],
                                join, need_rowid_for_window);
        finished_windows[window_idx] = true;
        --num_windows_left;
      }
    }

    receiver.ProposeAccessPath(root_path, &new_root_candidates,
                               /*obsolete_orderings=*/0, "");
  }
  if (trace) {
    *trace += "\n";
  }
  return new_root_candidates;
}

/**
  Find out whether “item” is a sargable condition; if so, add it to:

   - The list of sargable predicate for the tables (hypergraph nodes)
     the condition touches. For a regular condition, this will typically
     be one table; for a join condition, it will typically be two.
     If “force_table” is non-nullptr, only that table will be considered
     (this is used for join conditions, to ensure that we do not push
     down predicates that cannot, e.g. to the outer side of left joins).

   - The graph's global list of predicates, if it is not already present
     (predicate_index = -1). This will never happen for WHERE conditions,
     only for join conditions.
 */
static void PossiblyAddSargableCondition(THD *thd, Item *item,
                                         TABLE *force_table,
                                         int predicate_index,
                                         bool is_join_condition,
                                         JoinHypergraph *graph, string *trace) {
  if (item->type() != Item::FUNC_ITEM ||
      down_cast<Item_func *>(item)->functype() != Item_bool_func2::EQ_FUNC) {
    return;
  }
  Item_func_eq *eq_item = down_cast<Item_func_eq *>(item);
  if (eq_item->get_comparator()->get_child_comparator_count() >= 2) {
    return;
  }
  for (unsigned arg_idx = 0; arg_idx < 2; ++arg_idx) {
    Item *left = eq_item->arguments()[arg_idx];
    Item *right = eq_item->arguments()[1 - arg_idx];
    if (left->type() != Item::FIELD_ITEM) {
      continue;
    }
    Field *field = down_cast<Item_field *>(left)->field;
    if (force_table != nullptr && force_table != field->table) {
      continue;
    }
    if (field->part_of_key.is_clear_all()) {
      // Not part of any key, so not sargable. (It could be part of a prefix
      // keys, though, but we include them for now.)
      continue;
    }
    JoinHypergraph::Node *node = FindNodeWithTable(graph, field->table);
    if (node == nullptr) {
      // A field in a different query block, so not sargable for us.
      continue;
    }

    if (trace != nullptr) {
      if (is_join_condition) {
        *trace += "Found sargable join condition " + ItemToString(item) +
                  " on " + node->table->alias + "\n";
      } else {
        *trace += "Found sargable condition " + ItemToString(item) + "\n";
      }
    }

    if (predicate_index == -1) {
      // This predicate is not already registered as a predicate
      // (which means in practice that it's a join predicate,
      // not a WHERE predicate), so add it so that we can refer
      // to it in bitmaps.
      Predicate p;
      p.condition = eq_item;
      p.selectivity = EstimateSelectivity(thd, eq_item, trace);
      p.used_nodes =
          GetNodeMapFromTableMap(eq_item->used_tables() & ~PSEUDO_TABLE_BITS,
                                 graph->table_num_to_node_num);
      p.total_eligibility_set =
          ~0;  // Should never be applied as a WHERE predicate.
      p.functional_dependencies_idx.init(thd->mem_root);
      p.contained_subqueries.init(thd->mem_root);  // Empty.
      graph->predicates.push_back(std::move(p));
      predicate_index = graph->predicates.size() - 1;
      graph->sargable_join_predicates.emplace(eq_item, predicate_index);
    }

    node->sargable_predicates.push_back({predicate_index, field, right});
  }
}

// Find sargable predicates, ie., those that we can push down into indexes.
// See add_key_field().
//
// TODO(sgunders): Include x=y OR NULL predicates, <=> and IS NULL predicates,
// and the special case of COLLATION accepted in add_key_field().
//
// TODO(sgunders): Integrate with the range optimizer, or find some other way
// of accepting <, >, <= and >= predicates.
void FindSargablePredicates(THD *thd, string *trace, JoinHypergraph *graph) {
  if (trace != nullptr) {
    *trace += "\n";
  }
  for (unsigned i = 0; i < graph->num_where_predicates; ++i) {
    if (IsSingleBitSet(graph->predicates[i].total_eligibility_set)) {
      PossiblyAddSargableCondition(thd, graph->predicates[i].condition,
                                   /*force_table=*/nullptr, i,
                                   /*is_join_condition=*/false, graph, trace);
    }
  }
  for (JoinHypergraph::Node &node : graph->nodes) {
    for (Item *cond : node.join_conditions_pushable_to_this) {
      const auto it = graph->sargable_join_predicates.find(cond);
      int predicate_index =
          (it == graph->sargable_join_predicates.end()) ? -1 : it->second;
      PossiblyAddSargableCondition(thd, cond, node.table, predicate_index,
                                   /*is_join_condition=*/true, graph, trace);
    }
  }
}

static bool ComesFromSameMultiEquality(Item *cond1, Item_func_eq *cond2) {
  return cond1->type() == Item::FUNC_ITEM &&
         down_cast<Item_func *>(cond1)->functype() == Item_func::EQ_FUNC &&
         down_cast<Item_func_eq *>(cond1)->source_multiple_equality ==
             cond2->source_multiple_equality;
}

/**
  For each edge, cache some information for each of its join conditions.
  This reduces work when repeatedly applying these join conditions later on.
  In particular, FindContainedSubqueries() contains a large amount of
  virtual function calls that we would like to avoid doing every time
  we consider a given join.
 */
static void CacheCostInfoForJoinConditions(THD *thd,
                                           const Query_block *query_block,
                                           JoinHypergraph *graph,
                                           string *trace) {
  for (JoinPredicate &edge : graph->edges) {
    edge.expr->properties_for_equijoin_conditions.init(thd->mem_root);
    edge.expr->properties_for_join_conditions.init(thd->mem_root);
    for (Item_func_eq *cond : edge.expr->equijoin_conditions) {
      CachedPropertiesForPredicate properties;
      properties.selectivity = EstimateSelectivity(thd, cond, trace);
      properties.contained_subqueries.init(thd->mem_root);
      FindContainedSubqueries(
          thd, cond, query_block,
          [&properties](const ContainedSubquery &subquery) {
            properties.contained_subqueries.push_back(subquery);
          });

      // Cache information about what sargable conditions this join condition
      // would be redundant against, for RedundantThroughSargable().
      // But doon't deduplicate against ourselves (in case we're sargable).
      MutableOverflowBitset redundant(thd->mem_root, graph->predicates.size());
      for (unsigned sargable_pred_idx = graph->num_where_predicates;
           sargable_pred_idx < graph->predicates.size(); ++sargable_pred_idx) {
        Item *sargable_condition =
            graph->predicates[sargable_pred_idx].condition;
        if (sargable_condition != cond &&
            ComesFromSameMultiEquality(sargable_condition, cond)) {
          redundant.SetBit(sargable_pred_idx);
        }
      }
      properties.redundant_against_sargable_predicates = std::move(redundant);
      edge.expr->properties_for_equijoin_conditions.push_back(
          std::move(properties));
    }
    for (Item *cond : edge.expr->join_conditions) {
      CachedPropertiesForPredicate properties;
      properties.selectivity = EstimateSelectivity(thd, cond, trace);
      properties.contained_subqueries.init(thd->mem_root);
      FindContainedSubqueries(
          thd, cond, query_block,
          [&properties](const ContainedSubquery &subquery) {
            properties.contained_subqueries.push_back(subquery);
          });
      edge.expr->properties_for_join_conditions.push_back(
          std::move(properties));
    }
  }
}

/**
  Find the lowest-cost plan (which hopefully is also the cheapest to execute)
  of all the legal ways to execute the query. The overall order of operations is
  largely dictated by the standard:

    1. All joined tables, including join predicates.
    2. WHERE predicates (we push these down into #1 where allowed)
    3. GROUP BY (it is sometimes possible to push this down into #1,
       but we don't have the functionality to do so).
    4. HAVING.
    5. Window functions.
    6. DISTINCT.
    7. ORDER BY.
    8. LIMIT.
    9. SQL_BUFFER_RESULT (a MySQL extension).

  The place where we have the most leeway by far is #1, which is why this
  part of the optimizer is generally called the join optimizer (there are
  potentially billions of different join orderings, whereas each of the
  other steps, except windowing, can only be done in one or two ways).
  But the various outputs of #1 can have different properties, that can make
  for higher or lower costs in the other steps. (For instance, LIMIT will
  affect candidates with different init_cost differently, and ordering
  properties can skip sorting in ORDER BY entirely.) Thus, we allow keeping
  multiple candidates in play at every step if they are meaningfully different,
  and only pick out the winning candidate based on cost at the very end.
 */
AccessPath *FindBestQueryPlan(THD *thd, Query_block *query_block,
                              string *trace) {
  JOIN *join = query_block->join;
  if (CheckSupportedQuery(thd)) return nullptr;

  // The hypergraph optimizer does not do const tables,
  // nor does it evaluate subqueries during optimization.
  assert(
      IsSubset(OPTION_NO_CONST_TABLES | OPTION_NO_SUBQUERY_DURING_OPTIMIZATION,
               query_block->active_options()));

  // In the case of rollup (only): After the base slice list was made, we may
  // have modified the field list to add rollup group items and sum switchers.
  // The resolver also takes care to update these in query_block->order_list.
  // However, even though the hypergraph join optimizer doesn't use slices,
  // setup_order() later modifies order->item to point into the base slice,
  // where the rollup group items are _not_ updated. Thus, we need to refresh
  // the base slice before we do anything.
  //
  // It would be better to have rollup resolving update the base slice directly,
  // but this would break HAVING in the old join optimizer (see the other call
  // to refresh_base_slice(), in JOIN::make_tmp_tables_info()).
  if (join->rollup_state != JOIN::RollupState::NONE) {
    join->refresh_base_slice();
  }

  // NOTE: Normally, we'd expect join->temp_tables and
  // join->filesorts_to_cleanup to be empty, but since we can get called twice
  // for materialized subqueries, there may already be data there that we must
  // keep.

  // Convert the join structures into a hypergraph.
  JoinHypergraph graph(thd->mem_root, query_block);
  if (MakeJoinHypergraph(thd, trace, &graph)) {
    return nullptr;
  }
  FindSargablePredicates(thd, trace, &graph);

  // Now that we have all join conditions, cache some properties
  // that we'd like to use many times.
  CacheCostInfoForJoinConditions(thd, query_block, &graph, trace);

  // Figure out if any later sort will need row IDs.
  bool need_rowid = false;
  if (query_block->is_explicitly_grouped() || query_block->is_ordered() ||
      join->select_distinct || !join->m_windows.is_empty()) {
    // NOTE: This is distinct from SortWillBeOnRowId(), as it also checks blob
    // fields arising from blob-generating functions on non-blob fields.
    for (Item *item : *join->fields) {
      if (item->is_blob_field()) {
        need_rowid = true;
        break;
      }
    }
    for (TABLE_LIST *tl = query_block->leaf_tables;
         tl != nullptr && !need_rowid; tl = tl->next_leaf) {
      if (SortWillBeOnRowId(tl->table)) {
        need_rowid = true;
      }
    }
  }

  // Find out which predicates contain subqueries.
  MutableOverflowBitset materializable_predicates{thd->mem_root,
                                                  graph.predicates.size()};
  for (unsigned i = 0; i < graph.predicates.size(); ++i) {
    if (ContainsSubqueries(graph.predicates[i].condition)) {
      materializable_predicates.SetBit(i);
    }
  }
  graph.materializable_predicates = std::move(materializable_predicates);

  NodeMap fulltext_tables = 0;
  uint64_t sargable_fulltext_predicates = 0;
  if (query_block->has_ft_funcs()) {
    fulltext_tables = FindFullTextSearchedTables(graph);

    // Check if we have full-text indexes that can be used.
    sargable_fulltext_predicates = FindSargableFullTextPredicates(graph);
    EnableFullTextCoveringIndexes(query_block);
  }

  // Collect interesting orders from ORDER BY, GROUP BY, semijoins and windows.
  // See BuildInterestingOrders() for more detailed information.
  SecondaryEngineFlags engine_flags = EngineFlags(thd);
  LogicalOrderings orderings(thd);
  Mem_root_array<SortAheadOrdering> sort_ahead_orderings(thd->mem_root);
  Mem_root_array<ActiveIndexInfo> active_indexes(thd->mem_root);
  Mem_root_array<FullTextIndexInfo> fulltext_searches(thd->mem_root);
  int order_by_ordering_idx = -1;
  int group_by_ordering_idx = -1;
  int distinct_ordering_idx = -1;
  BuildInterestingOrders(thd, &graph, query_block, &orderings,
                         &sort_ahead_orderings, &order_by_ordering_idx,
                         &group_by_ordering_idx, &distinct_ordering_idx,
                         &active_indexes, &fulltext_searches, trace);

  // Run the actual join optimizer algorithm. This creates an access path
  // for the join as a whole (with lowest possible cost, and thus also
  // hopefully optimal execution time), with all pushable predicates applied.
  if (trace != nullptr) {
    *trace += "\nEnumerating subplans:\n";
  }
  for (const JoinHypergraph::Node &node : graph.nodes) {
    node.table->init_cost_model(thd->cost_model());
  }
  const secondary_engine_modify_access_path_cost_t secondary_engine_cost_hook =
      SecondaryEngineCostHook(thd);
  CostingReceiver receiver(thd, query_block, graph, &orderings,
                           &sort_ahead_orderings, &active_indexes,
                           &fulltext_searches, fulltext_tables,
                           sargable_fulltext_predicates, need_rowid,
                           EngineFlags(thd), secondary_engine_cost_hook, trace);
  if (EnumerateAllConnectedPartitions(graph.graph, &receiver) &&
      !thd->is_error()) {
    my_error(ER_HYPERGRAPH_NOT_SUPPORTED_YET, MYF(0), "large join graphs");
    return nullptr;
  }
  if (thd->is_error()) return nullptr;

  // Get the root candidates. If there is a secondary engine cost hook, there
  // may be no candidates, as the hook may have rejected so many access paths
  // that we could not build a complete plan. Otherwise, expect at least one
  // candidate.
  if (secondary_engine_cost_hook != nullptr &&
      (!receiver.HasSeen(TablesBetween(0, graph.nodes.size())) ||
       receiver.root_candidates().empty())) {
    my_error(ER_SECONDARY_ENGINE, MYF(0),
             "All plans were rejected by the secondary storage engine.");
    return nullptr;
  }
  Prealloced_array<AccessPath *, 4> root_candidates =
      receiver.root_candidates();
  assert(!root_candidates.empty());
  thd->m_current_query_partial_plans += receiver.num_access_paths();
  if (trace != nullptr) {
    *trace += StringPrintf(
        "\nEnumerated %zu subplans, got %zu candidate(s) to finalize:\n",
        receiver.num_access_paths(), root_candidates.size());
  }

  // Now we have one or more access paths representing joining all the tables
  // together. (There may be multiple ones because they can be better at
  // different metrics.) We apply the post-join operations to all of them in
  // turn, and then finally pick out the one with the lowest total cost,
  // because at the end, other metrics don't really matter any more.
  //
  // We could have stopped caring about e.g. init_cost after LIMIT
  // has been applied (after which it no longer matters), so that we'd get
  // fewer candidates in each step, but this part is so cheap that it's
  // unlikely to be worth it. We go through ProposeAccessPath() mainly
  // because it gives us better tracing.
  if (trace != nullptr) {
    *trace += "Adding final predicates\n";
  }
  FunctionalDependencySet fd_set = receiver.active_fds_at_root();
  for (size_t i = 0; i < graph.num_where_predicates; ++i) {
    // Apply any predicates that don't belong to any
    // specific table, or which are nondeterministic.
    if (!Overlaps(graph.predicates[i].total_eligibility_set,
                  TablesBetween(0, graph.nodes.size())) ||
        Overlaps(graph.predicates[i].total_eligibility_set, RAND_TABLE_BIT)) {
      fd_set |= graph.predicates[i].functional_dependencies;
    }
  }

  {
    Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
    for (const AccessPath *root_path : root_candidates) {
      for (bool materialize_subqueries : {false, true}) {
        AccessPath path = *root_path;
        double init_once_cost = 0.0;

        MutableOverflowBitset filter_predicates =
            path.filter_predicates.Clone(thd->mem_root);

        // Apply any predicates that don't belong to any
        // specific table, or which are nondeterministic.
        for (size_t i = 0; i < graph.num_where_predicates; ++i) {
          if (!Overlaps(graph.predicates[i].total_eligibility_set,
                        TablesBetween(0, graph.nodes.size())) ||
              Overlaps(graph.predicates[i].total_eligibility_set,
                       RAND_TABLE_BIT)) {
            filter_predicates.SetBit(i);
            FilterCost cost =
                EstimateFilterCost(thd, root_path->num_output_rows,
                                   graph.predicates[i].contained_subqueries);
            if (materialize_subqueries) {
              path.cost += cost.cost_if_materialized;
              init_once_cost += cost.cost_to_materialize;
            } else {
              path.cost += cost.cost_if_not_materialized;
            }
            path.num_output_rows *= graph.predicates[i].selectivity;
          }
        }
        path.ordering_state = orderings.ApplyFDs(path.ordering_state, fd_set);

        path.filter_predicates = std::move(filter_predicates);
        const bool contains_subqueries =
            Overlaps(path.filter_predicates, graph.materializable_predicates);

        // Now that we have decided on a full plan, expand all
        // the applied filter maps into proper FILTER nodes
        // for execution. This is a no-op in the second
        // iteration.
        ExpandFilterAccessPaths(thd, &path, join, graph.predicates,
                                graph.num_where_predicates);

        if (materialize_subqueries) {
          assert(path.type == AccessPath::FILTER);
          path.filter().materialize_subqueries = true;
          path.cost += init_once_cost;  // Will be subtracted
                                        // back for rescans.
          path.init_cost += init_once_cost;
          path.init_once_cost += init_once_cost;
        }

        receiver.ProposeAccessPath(&path, &new_root_candidates,
                                   /*obsolete_orderings=*/0,
                                   materialize_subqueries ? "mat. subq" : "");

        if (!contains_subqueries) {
          // Nothing to try to materialize.
          break;
        }
      }
    }
    root_candidates = std::move(new_root_candidates);
  }

  // Apply GROUP BY, if applicable. We currently always do this by sorting
  // first and then using streaming aggregation.
  const bool aggregation_is_unordered = Overlaps(
      engine_flags,
      MakeSecondaryEngineFlags(SecondaryEngineFlag::AGGREGATION_IS_UNORDERED));
  if (query_block->is_grouped()) {
    if (join->make_sum_func_list(*join->fields, /*before_group_by=*/true))
      return nullptr;

    if (trace != nullptr) {
      *trace += "Applying aggregation for GROUP BY\n";
    }

    // Create a temporary table for materializing the results of full-text
    // functions, if needed.
    //
    // The full-text MATCH function requires the handler to be positioned on the
    // row that holds the value to perform the full-text search on. It is not
    // enough to have all the required column values in the record buffer. Since
    // sort-based aggregation has moved off the original row when a group is
    // returned, we add a temporary table which materializes the results of any
    // calls to MATCH that will be needed after aggregation, and stream the rows
    // through this table before aggregation.
    TABLE *fulltext_table = nullptr;
    Temp_table_param *fulltext_param = nullptr;
    if (query_block->has_ft_funcs()) {
      if (CreateTemporaryTableForFullTextFunctions(
              thd, query_block, &fulltext_table, &fulltext_param)) {
        return nullptr;
      }
    }

    Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
    for (AccessPath *root_path : root_candidates) {
      const bool rollup = (join->rollup_state != JOIN::RollupState::NONE);
      const bool group_needs_sort =
          query_block->is_explicitly_grouped() && !aggregation_is_unordered &&
          !orderings.DoesFollowOrder(root_path->ordering_state,
                                     group_by_ordering_idx);

      if (!group_needs_sort) {
        AccessPath *child_path = root_path;
        if (fulltext_table != nullptr) {
          // Add a streaming path for materializing results from full-text
          // functions before aggregation.
          child_path =
              NewStreamingAccessPath(thd, root_path, join, fulltext_param,
                                     fulltext_table, /*ref_slice=*/-1);
          CopyBasicProperties(*root_path, child_path);
        }

        // TODO(sgunders): We don't need to allocate this on the MEM_ROOT.
        AccessPath *aggregate_path =
            NewAggregateAccessPath(thd, child_path, rollup);
        EstimateAggregateCost(aggregate_path);

        receiver.ProposeAccessPath(aggregate_path, &new_root_candidates,
                                   /*obsolete_orderings=*/0, "sort elided");
        continue;
      }

      root_path = GetSafePathToSort(thd, join, root_path, need_rowid);

      // We need to sort. Try all sort-ahead, not just the one directly derived
      // from GROUP BY clause, because a broader one might help us elide ORDER
      // BY or DISTINCT later.
      for (const SortAheadOrdering &sort_ahead_ordering :
           sort_ahead_orderings) {
        LogicalOrderings::StateIndex ordering_state = orderings.ApplyFDs(
            orderings.SetOrder(sort_ahead_ordering.ordering_idx), fd_set);
        if (!orderings.DoesFollowOrder(ordering_state, group_by_ordering_idx)) {
          continue;
        }
        if (sort_ahead_ordering.aggregates_required) {
          // We can't sort by an aggregate before we've aggregated.
          continue;
        }

        Mem_root_array<TABLE *> tables = CollectTables(thd, root_path);
        AccessPath *sort_path = new (thd->mem_root) AccessPath;
        sort_path->type = AccessPath::SORT;
        sort_path->count_examined_rows = false;
        sort_path->sort().child = root_path;
        sort_path->sort().filesort = nullptr;
        sort_path->sort().remove_duplicates = false;
        sort_path->sort().unwrap_rollup = true;
        sort_path->sort().use_limit = false;
        sort_path->sort().order = sort_ahead_ordering.order;
        EstimateSortCost(sort_path);
        assert(!aggregation_is_unordered);
        sort_path->ordering_state = ordering_state;

        AccessPath *child_path = sort_path;
        if (fulltext_table != nullptr) {
          // Add a streaming path for materializing results from full-text
          // functions before aggregation.
          child_path =
              NewStreamingAccessPath(thd, sort_path, join, fulltext_param,
                                     fulltext_table, /*ref_slice=*/-1);
          CopyBasicProperties(*sort_path, child_path);
        }

        // TODO(sgunders): We don't need to allocate this on the MEM_ROOT.
        AccessPath *aggregate_path =
            NewAggregateAccessPath(thd, child_path, rollup);
        EstimateAggregateCost(aggregate_path);

        char description[256];
        if (trace != nullptr) {
          snprintf(description, sizeof(description), "sort(%d)",
                   sort_ahead_ordering.ordering_idx);
        }
        receiver.ProposeAccessPath(aggregate_path, &new_root_candidates,
                                   /*obsolete_orderings=*/0, description);
      }
    }
    root_candidates = std::move(new_root_candidates);

    if (make_group_fields(join, join)) {
      return nullptr;
    }

    // Final setup will be done in FinalizePlanForQueryBlock(),
    // when we have all materialization done.
  }

  // Apply HAVING, if applicable (sans any window-related in2exists parts,
  // which we apply below).
  Item *having_cond;
  Item *having_cond_wf;
  SplitHavingCondition(thd, join->having_cond, &having_cond, &having_cond_wf);
  ApplyHavingCondition(thd, having_cond, query_block,
                       "Applying filter for HAVING\n", trace, &root_candidates,
                       &receiver);

  // If we have GROUP BY followed by a window function (which might include
  // ORDER BY), we might need to materialize before the first ordering -- see
  // the comment near the top of ApplyDistinctAndOrder() for why.
  if (query_block->is_explicitly_grouped() && !join->m_windows.is_empty()) {
    Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
    for (AccessPath *root_path : root_candidates) {
      root_path =
          CreateMaterializationOrStreamingPath(thd, join, root_path, need_rowid,
                                               /*copy_items=*/true);
      receiver.ProposeAccessPath(root_path, &new_root_candidates,
                                 /*obsolete_orderings=*/0, "");
    }
    root_candidates = std::move(new_root_candidates);
  }

  join->m_windowing_steps = !join->m_windows.is_empty();
  if (join->m_windowing_steps) {
    root_candidates = ApplyWindowFunctions(
        thd, receiver, orderings, fd_set, aggregation_is_unordered,
        order_by_ordering_idx, distinct_ordering_idx, graph,
        sort_ahead_orderings, query_block, graph.num_where_predicates,
        need_rowid, std::move(root_candidates), trace);
  }

  ApplyHavingCondition(
      thd, having_cond_wf, query_block,
      "Applying filter for window function in2exists conditions\n", trace,
      &root_candidates, &receiver);

  if (join->select_distinct || query_block->is_ordered()) {
    root_candidates = ApplyDistinctAndOrder(
        thd, receiver, orderings, aggregation_is_unordered,
        order_by_ordering_idx, distinct_ordering_idx, sort_ahead_orderings,
        fd_set, query_block, need_rowid, std::move(root_candidates), trace);
  }

  // Apply LIMIT and OFFSET, if applicable. If the query block is ordered, they
  // are already applied by ApplyDistinctAndOrder().
  Query_expression *query_expression = join->query_expression();
  if (!query_block->is_ordered() &&
      (query_expression->select_limit_cnt != HA_POS_ERROR ||
       query_expression->offset_limit_cnt != 0)) {
    if (trace != nullptr) {
      *trace += "Applying LIMIT\n";
    }
    Prealloced_array<AccessPath *, 4> new_root_candidates(PSI_NOT_INSTRUMENTED);
    for (AccessPath *root_path : root_candidates) {
      AccessPath *limit_path = NewLimitOffsetAccessPath(
          thd, root_path, query_expression->select_limit_cnt,
          query_expression->offset_limit_cnt, join->calc_found_rows,
          /*reject_multiple_rows=*/false,
          /*send_records_override=*/nullptr);
      receiver.ProposeAccessPath(limit_path, &new_root_candidates,
                                 /*obsolete_orderings=*/0, "");
    }
    root_candidates = std::move(new_root_candidates);
  }

  if (thd->is_error()) return nullptr;

  if (root_candidates.empty()) {
    // The secondary engine has rejected so many of the post-processing paths
    // (e.g., sorting, limit, grouping) that we could not build a complete plan.
    assert(secondary_engine_cost_hook != nullptr);
    my_error(ER_SECONDARY_ENGINE, MYF(0),
             "All plans were rejected by the secondary storage engine.");
    return nullptr;
  }

  // TODO(sgunders): If we are part of e.g. a derived table and are streamed,
  // we might want to keep multiple root paths around for future use, e.g.,
  // if there is a LIMIT higher up.
  AccessPath *root_path =
      *std::min_element(root_candidates.begin(), root_candidates.end(),
                        [](const AccessPath *a, const AccessPath *b) {
                          return a->cost < b->cost;
                        });

  // Materialize the result if a top-level query block has the SQL_BUFFER_RESULT
  // option, and the chosen root path isn't already a materialization path.
  if (query_block->active_options() & OPTION_BUFFER_RESULT &&
      query_block->outer_query_block() == nullptr &&
      !IsMaterializationPath(root_path)) {
    if (trace != nullptr) {
      *trace += "Adding temporary table for SQL_BUFFER_RESULT.\n";
    }

    // If we have windows, we may need to add a materialization for the last
    // window here, or create_tmp_table() will not create fields for its window
    // functions. (All other windows have already been materialized.)
    bool copy_items = join->m_windows.is_empty();
    root_path =
        CreateMaterializationPath(thd, join, root_path, /*temp_table=*/nullptr,
                                  /*temp_table_param=*/nullptr, copy_items);
  }

  if (trace != nullptr) {
    *trace += StringPrintf("Final cost is %.1f.\n", root_path->cost);
  }

#ifndef NDEBUG
  WalkAccessPaths(root_path, join, WalkAccessPathPolicy::ENTIRE_QUERY_BLOCK,
                  [&](const AccessPath *path, const JOIN *) {
                    assert(path->cost >= path->init_cost);
                    assert(path->init_cost >= path->init_once_cost);
                    return false;
                  });
#endif

  join->needs_finalize = true;
  join->best_rowcount = lrint(root_path->num_output_rows);
  join->best_read = root_path->cost;

  // 0 or 1 rows has a special meaning; it means a _guarantee_ we have no more
  // than one (so-called “const tables”). Make sure we don't give that
  // guarantee unless we have a LIMIT.
  if (join->best_rowcount <= 1 &&
      query_expression->select_limit_cnt - query_expression->offset_limit_cnt >
          1) {
    join->best_rowcount = PLACEHOLDER_TABLE_ROW_ESTIMATE;
  }

  return root_path;
}
