/*

  This file is provided under a dual BSD/GPLv2 license.  When using or
  redistributing this file, you may do so under either license.

  GPL LICENSE SUMMARY

  Copyright(c) 2015 Intel Corporation.

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  Contact Information:
  Intel Corporation, www.intel.com

  BSD LICENSE

  Copyright(c) 2015 Intel Corporation.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/* Copyright (c) 2003-2014 Intel Corporation. All rights reserved. */

#ifndef _PSMI_IN_USER_H
#error psm_lock.h not meant to be included directly, include psm_user.h instead
#endif

#ifndef _PSMI_LOCK_H
#define _PSMI_LOCK_H

#ifndef PSMI_USE_PTHREAD_SPINLOCKS
#define PSMI_USE_PTHREAD_SPINLOCKS 0
#endif

#if PSMI_USE_PTHREAD_SPINLOCKS
typedef pthread_spinlock_t psmi_spinlock_t;

#define psmi_spin_init(lock)	  pthread_spin_init(lock, \
					PTHREAD_PROCESS_PRIVATE)
#define psmi_spin_lock(lock)	  pthread_spin_lock(lock)
#define psmi_spin_trylock(lock) pthread_spin_trylock(lock)
#define psmi_spin_unlock(lock)  pthread_spin_unlock(lock)
#else
typedef ips_atomic_t psmi_spinlock_t;
#define PSMI_SPIN_LOCKED    1
#define PSMI_SPIN_UNLOCKED  0
#endif

/* psmi_lock_t structure */
typedef struct {

#ifdef PSMI_LOCK_IS_SPINLOCK
	psmi_spinlock_t lock;
#elif defined(PSMI_LOCK_IS_MUTEXLOCK_DEBUG)
	pthread_mutex_t lock;
	pthread_t lock_owner;
#elif defined(PSMI_LOCK_IS_MUTEXLOCK)
	pthread_mutex_t lock;
#endif
} psmi_lock_t;


#if PSMI_USE_PTHREAD_SPINLOCKS
#else
PSMI_ALWAYS_INLINE(int psmi_spin_init(psmi_spinlock_t *lock))
{
	ips_atomic_set(lock, PSMI_SPIN_UNLOCKED);
	return 0;
}

PSMI_ALWAYS_INLINE(int psmi_spin_trylock(psmi_spinlock_t *lock))
{
	if (ips_atomic_cmpxchg(lock, PSMI_SPIN_UNLOCKED, PSMI_SPIN_LOCKED)
			== PSMI_SPIN_UNLOCKED)
		return 0;
	else
		return EBUSY;
}

PSMI_ALWAYS_INLINE(int psmi_spin_lock(psmi_spinlock_t *lock))
{
	while (psmi_spin_trylock(lock) == EBUSY) {
	}
	return 0;
}

PSMI_ALWAYS_INLINE(int psmi_spin_unlock(psmi_spinlock_t *lock))
{
	atomic_set(lock, PSMI_SPIN_UNLOCKED);
	return 0;
}
#endif /* PSMI_USE_PTHREAD_SPINLOCKS */

PSMI_ALWAYS_INLINE(void psmi_init_lock(psmi_lock_t *lock))
{
#ifdef PSMI_LOCK_IS_SPINLOCK
	psmi_spin_init(&(lock->lock));
#elif defined(PSMI_LOCK_IS_MUTEXLOCK)
	pthread_mutex_init(&(lock->lock), NULL);
#elif defined(PSMI_LOCK_IS_MUTEXLOCK_DEBUG)
	pthread_mutexattr_t attr;
	pthread_mutexattr_init(&attr);
	pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_ERRORCHECK_NP);
	pthread_mutex_init(&(lock->lock), &attr);
	pthread_mutexattr_destroy(&attr);
	lock->lock_owner = PSMI_LOCK_NO_OWNER;
#endif
}

PSMI_ALWAYS_INLINE(int psmi_sem_post(sem_t *sem, const char *name))
{
	if (sem_post(sem) == -1) {
		_HFI_VDBG("Semaphore %s: post failed\n", name ? name : "NULL" );
		return -1;
	}

	_HFI_VDBG("Semaphore %s: post succeeded\n", name ? name : "NULL");

	return 0;
}

PSMI_ALWAYS_INLINE(int psmi_sem_timedwait(sem_t *sem, const char *name))
{
	/* Wait 5 seconds for shm read-write lock to open */
	struct timespec ts;
	clock_gettime(CLOCK_REALTIME, &ts);
	ts.tv_sec += 5;

	if (sem_timedwait(sem, &ts) == -1) {
		_HFI_VDBG("Semaphore %s: Timedwait failed\n", name ? name : "NULL" );
		return -1;
	}

	_HFI_VDBG("Semaphore %s: Timedwait succeeded\n", name ? name : "NULL");

	return 0;
}

PSMI_ALWAYS_INLINE(int psmi_init_semaphore(sem_t **sem, const char *name,
					   mode_t mode, int value))
{
	*sem = sem_open(name, O_CREAT | O_EXCL, mode, value);
	if ((*sem == SEM_FAILED) && (errno == EEXIST)) {
		*sem = sem_open(name, O_CREAT, mode, value);
		if (*sem == SEM_FAILED) {
			_HFI_VDBG("Cannot open semaphore %s, errno=%d\n",
				  name, errno);
			return -1;
		}
	} else if (*sem == SEM_FAILED) {
		_HFI_VDBG("Cannot create semaphore %s, errno=%d\n", name, errno);
		return -1;
	}

	return 0;
}

#endif /* _PSMI_LOCK_H */
