//===----------------------------------------------------------------------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is dual licensed under the MIT and the University of Illinois Open
// Source Licenses. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// Copyright 2019, Intel Corporation
//
// Modified to test pmem::obj containers
//

#include "unittest.hpp"

#include <libpmemobj++/experimental/string.hpp>
#include <libpmemobj++/make_persistent.hpp>
#include <libpmemobj++/persistent_ptr.hpp>
#include <libpmemobj++/pool.hpp>
#include <libpmemobj++/transaction.hpp>

namespace pmem_exp = pmem::obj::experimental;
namespace nvobj = pmem::obj;

struct root {
	nvobj::persistent_ptr<pmem_exp::string> s1, s2, s3, s4;
};

int
sign(int x)
{
	if (x == 0)
		return 0;
	if (x < 0)
		return -1;
	return 1;
}

template <class S>
void
test(const S &s, typename S::size_type pos1, typename S::size_type n1,
     const S &str, typename S::size_type pos2, typename S::size_type n2, int x)
{
	if (pos1 <= s.size() && pos2 <= str.size())
		UT_ASSERT(sign(s.compare(pos1, n1, str, pos2, n2)) == sign(x));
	else {
		try {
			s.compare(pos1, n1, str, pos2, n2);
			UT_ASSERT(0);
		} catch (const std::out_of_range &) {
			UT_ASSERT(pos1 > s.size() || pos2 > str.size());
		}
	}
}

template <class S>
void
test_npos(const S &s, typename S::size_type pos1, typename S::size_type n1,
	  const S &str, typename S::size_type pos2, int x)
{
	if (pos1 <= s.size() && pos2 <= str.size())
		UT_ASSERT(sign(s.compare(pos1, n1, str, pos2)) == sign(x));
	else {
		try {
			s.compare(pos1, n1, str, pos2);
			UT_ASSERT(0);
		} catch (const std::out_of_range &) {
			UT_ASSERT(pos1 > s.size() || pos2 > str.size());
		}
	}
}

template <class S>
void
test0(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s1, 0, 0, *r->s1, 0, 0, 0);
	test(*r->s1, 0, 0, *r->s1, 0, 1, 0);
	test(*r->s1, 0, 0, *r->s1, 1, 0, 0);
	test(*r->s1, 0, 0, *r->s2, 0, 0, 0);
	test(*r->s1, 0, 0, *r->s2, 0, 1, -1);
	test(*r->s1, 0, 0, *r->s2, 0, 2, -2);
	test(*r->s1, 0, 0, *r->s2, 0, 4, -4);
	test(*r->s1, 0, 0, *r->s2, 0, 5, -5);
	test(*r->s1, 0, 0, *r->s2, 0, 6, -5);
	test(*r->s1, 0, 0, *r->s2, 1, 0, 0);
	test(*r->s1, 0, 0, *r->s2, 1, 1, -1);
	test(*r->s1, 0, 0, *r->s2, 1, 2, -2);
	test(*r->s1, 0, 0, *r->s2, 1, 3, -3);
	test(*r->s1, 0, 0, *r->s2, 1, 4, -4);
	test(*r->s1, 0, 0, *r->s2, 1, 5, -4);
	test(*r->s1, 0, 0, *r->s2, 2, 0, 0);
	test(*r->s1, 0, 0, *r->s2, 2, 1, -1);
	test(*r->s1, 0, 0, *r->s2, 2, 2, -2);
	test(*r->s1, 0, 0, *r->s2, 2, 3, -3);
	test(*r->s1, 0, 0, *r->s2, 2, 4, -3);
	test(*r->s1, 0, 0, *r->s2, 4, 0, 0);
	test(*r->s1, 0, 0, *r->s2, 4, 1, -1);
	test(*r->s1, 0, 0, *r->s2, 4, 2, -1);
	test(*r->s1, 0, 0, *r->s2, 5, 0, 0);
	test(*r->s1, 0, 0, *r->s2, 5, 1, 0);
	test(*r->s1, 0, 0, *r->s2, 6, 0, 0);
	test(*r->s1, 0, 0, *r->s3, 0, 0, 0);
	test(*r->s1, 0, 0, *r->s3, 0, 1, -1);
	test(*r->s1, 0, 0, *r->s3, 0, 5, -5);
	test(*r->s1, 0, 0, *r->s3, 0, 9, -9);
	test(*r->s1, 0, 0, *r->s3, 0, 10, -10);
	test(*r->s1, 0, 0, *r->s3, 0, 11, -10);
	test(*r->s1, 0, 0, *r->s3, 1, 0, 0);
	test(*r->s1, 0, 0, *r->s3, 1, 1, -1);
	test(*r->s1, 0, 0, *r->s3, 1, 4, -4);
	test(*r->s1, 0, 0, *r->s3, 1, 8, -8);
	test(*r->s1, 0, 0, *r->s3, 1, 9, -9);
	test(*r->s1, 0, 0, *r->s3, 1, 10, -9);
	test(*r->s1, 0, 0, *r->s3, 5, 0, 0);
	test(*r->s1, 0, 0, *r->s3, 5, 1, -1);
	test(*r->s1, 0, 0, *r->s3, 5, 2, -2);
	test(*r->s1, 0, 0, *r->s3, 5, 4, -4);
	test(*r->s1, 0, 0, *r->s3, 5, 5, -5);
	test(*r->s1, 0, 0, *r->s3, 5, 6, -5);
	test(*r->s1, 0, 0, *r->s3, 9, 0, 0);
	test(*r->s1, 0, 0, *r->s3, 9, 1, -1);
	test(*r->s1, 0, 0, *r->s3, 9, 2, -1);
	test(*r->s1, 0, 0, *r->s3, 10, 0, 0);
	test(*r->s1, 0, 0, *r->s3, 10, 1, 0);
	test(*r->s1, 0, 0, *r->s3, 11, 0, 0);
	test(*r->s1, 0, 0, *r->s4, 0, 0, 0);
	test(*r->s1, 0, 0, *r->s4, 0, 1, -1);
	test(*r->s1, 0, 0, *r->s4, 0, 10, -10);
	test(*r->s1, 0, 0, *r->s4, 0, 19, -19);
	test(*r->s1, 0, 0, *r->s4, 0, 20, -20);
	test(*r->s1, 0, 0, *r->s4, 0, 21, -20);
	test(*r->s1, 0, 0, *r->s4, 1, 0, 0);
	test(*r->s1, 0, 0, *r->s4, 1, 1, -1);
	test(*r->s1, 0, 0, *r->s4, 1, 9, -9);
	test(*r->s1, 0, 0, *r->s4, 1, 18, -18);
	test(*r->s1, 0, 0, *r->s4, 1, 19, -19);
	test(*r->s1, 0, 0, *r->s4, 1, 20, -19);
	test(*r->s1, 0, 0, *r->s4, 10, 0, 0);
	test(*r->s1, 0, 0, *r->s4, 10, 1, -1);
	test(*r->s1, 0, 0, *r->s4, 10, 5, -5);
	test(*r->s1, 0, 0, *r->s4, 10, 9, -9);
	test(*r->s1, 0, 0, *r->s4, 10, 10, -10);
	test(*r->s1, 0, 0, *r->s4, 10, 11, -10);
	test(*r->s1, 0, 0, *r->s4, 19, 0, 0);
	test(*r->s1, 0, 0, *r->s4, 19, 1, -1);
	test(*r->s1, 0, 0, *r->s4, 19, 2, -1);
	test(*r->s1, 0, 0, *r->s4, 20, 0, 0);
	test(*r->s1, 0, 0, *r->s4, 20, 1, 0);
	test(*r->s1, 0, 0, *r->s4, 21, 0, 0);
	test(*r->s1, 0, 1, *r->s1, 0, 0, 0);
	test(*r->s1, 0, 1, *r->s1, 0, 1, 0);
	test(*r->s1, 0, 1, *r->s1, 1, 0, 0);
	test(*r->s1, 0, 1, *r->s2, 0, 0, 0);
	test(*r->s1, 0, 1, *r->s2, 0, 1, -1);
	test(*r->s1, 0, 1, *r->s2, 0, 2, -2);
	test(*r->s1, 0, 1, *r->s2, 0, 4, -4);
	test(*r->s1, 0, 1, *r->s2, 0, 5, -5);
	test(*r->s1, 0, 1, *r->s2, 0, 6, -5);
	test(*r->s1, 0, 1, *r->s2, 1, 0, 0);
	test(*r->s1, 0, 1, *r->s2, 1, 1, -1);
	test(*r->s1, 0, 1, *r->s2, 1, 2, -2);
	test(*r->s1, 0, 1, *r->s2, 1, 3, -3);
	test(*r->s1, 0, 1, *r->s2, 1, 4, -4);
	test(*r->s1, 0, 1, *r->s2, 1, 5, -4);
	test(*r->s1, 0, 1, *r->s2, 2, 0, 0);
	test(*r->s1, 0, 1, *r->s2, 2, 1, -1);
	test(*r->s1, 0, 1, *r->s2, 2, 2, -2);
	test(*r->s1, 0, 1, *r->s2, 2, 3, -3);
	test(*r->s1, 0, 1, *r->s2, 2, 4, -3);
	test(*r->s1, 0, 1, *r->s2, 4, 0, 0);
	test(*r->s1, 0, 1, *r->s2, 4, 1, -1);
	test(*r->s1, 0, 1, *r->s2, 4, 2, -1);
	test(*r->s1, 0, 1, *r->s2, 5, 0, 0);
	test(*r->s1, 0, 1, *r->s2, 5, 1, 0);
	test(*r->s1, 0, 1, *r->s2, 6, 0, 0);
}

template <class S>
void
test1(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s1, 0, 1, *r->s3, 0, 0, 0);
	test(*r->s1, 0, 1, *r->s3, 0, 1, -1);
	test(*r->s1, 0, 1, *r->s3, 0, 5, -5);
	test(*r->s1, 0, 1, *r->s3, 0, 9, -9);
	test(*r->s1, 0, 1, *r->s3, 0, 10, -10);
	test(*r->s1, 0, 1, *r->s3, 0, 11, -10);
	test(*r->s1, 0, 1, *r->s3, 1, 0, 0);
	test(*r->s1, 0, 1, *r->s3, 1, 1, -1);
	test(*r->s1, 0, 1, *r->s3, 1, 4, -4);
	test(*r->s1, 0, 1, *r->s3, 1, 8, -8);
	test(*r->s1, 0, 1, *r->s3, 1, 9, -9);
	test(*r->s1, 0, 1, *r->s3, 1, 10, -9);
	test(*r->s1, 0, 1, *r->s3, 5, 0, 0);
	test(*r->s1, 0, 1, *r->s3, 5, 1, -1);
	test(*r->s1, 0, 1, *r->s3, 5, 2, -2);
	test(*r->s1, 0, 1, *r->s3, 5, 4, -4);
	test(*r->s1, 0, 1, *r->s3, 5, 5, -5);
	test(*r->s1, 0, 1, *r->s3, 5, 6, -5);
	test(*r->s1, 0, 1, *r->s3, 9, 0, 0);
	test(*r->s1, 0, 1, *r->s3, 9, 1, -1);
	test(*r->s1, 0, 1, *r->s3, 9, 2, -1);
	test(*r->s1, 0, 1, *r->s3, 10, 0, 0);
	test(*r->s1, 0, 1, *r->s3, 10, 1, 0);
	test(*r->s1, 0, 1, *r->s3, 11, 0, 0);
	test(*r->s1, 0, 1, *r->s4, 0, 0, 0);
	test(*r->s1, 0, 1, *r->s4, 0, 1, -1);
	test(*r->s1, 0, 1, *r->s4, 0, 10, -10);
	test(*r->s1, 0, 1, *r->s4, 0, 19, -19);
	test(*r->s1, 0, 1, *r->s4, 0, 20, -20);
	test(*r->s1, 0, 1, *r->s4, 0, 21, -20);
	test(*r->s1, 0, 1, *r->s4, 1, 0, 0);
	test(*r->s1, 0, 1, *r->s4, 1, 1, -1);
	test(*r->s1, 0, 1, *r->s4, 1, 9, -9);
	test(*r->s1, 0, 1, *r->s4, 1, 18, -18);
	test(*r->s1, 0, 1, *r->s4, 1, 19, -19);
	test(*r->s1, 0, 1, *r->s4, 1, 20, -19);
	test(*r->s1, 0, 1, *r->s4, 10, 0, 0);
	test(*r->s1, 0, 1, *r->s4, 10, 1, -1);
	test(*r->s1, 0, 1, *r->s4, 10, 5, -5);
	test(*r->s1, 0, 1, *r->s4, 10, 9, -9);
	test(*r->s1, 0, 1, *r->s4, 10, 10, -10);
	test(*r->s1, 0, 1, *r->s4, 10, 11, -10);
	test(*r->s1, 0, 1, *r->s4, 19, 0, 0);
	test(*r->s1, 0, 1, *r->s4, 19, 1, -1);
	test(*r->s1, 0, 1, *r->s4, 19, 2, -1);
	test(*r->s1, 0, 1, *r->s4, 20, 0, 0);
	test(*r->s1, 0, 1, *r->s4, 20, 1, 0);
	test(*r->s1, 0, 1, *r->s4, 21, 0, 0);
	test(*r->s1, 1, 0, *r->s1, 0, 0, 0);
	test(*r->s1, 1, 0, *r->s1, 0, 1, 0);
	test(*r->s1, 1, 0, *r->s1, 1, 0, 0);
	test(*r->s1, 1, 0, *r->s2, 0, 0, 0);
	test(*r->s1, 1, 0, *r->s2, 0, 1, 0);
	test(*r->s1, 1, 0, *r->s2, 0, 2, 0);
	test(*r->s1, 1, 0, *r->s2, 0, 4, 0);
	test(*r->s1, 1, 0, *r->s2, 0, 5, 0);
	test(*r->s1, 1, 0, *r->s2, 0, 6, 0);
	test(*r->s1, 1, 0, *r->s2, 1, 0, 0);
	test(*r->s1, 1, 0, *r->s2, 1, 1, 0);
	test(*r->s1, 1, 0, *r->s2, 1, 2, 0);
	test(*r->s1, 1, 0, *r->s2, 1, 3, 0);
	test(*r->s1, 1, 0, *r->s2, 1, 4, 0);
	test(*r->s1, 1, 0, *r->s2, 1, 5, 0);
	test(*r->s1, 1, 0, *r->s2, 2, 0, 0);
	test(*r->s1, 1, 0, *r->s2, 2, 1, 0);
	test(*r->s1, 1, 0, *r->s2, 2, 2, 0);
	test(*r->s1, 1, 0, *r->s2, 2, 3, 0);
	test(*r->s1, 1, 0, *r->s2, 2, 4, 0);
	test(*r->s1, 1, 0, *r->s2, 4, 0, 0);
	test(*r->s1, 1, 0, *r->s2, 4, 1, 0);
	test(*r->s1, 1, 0, *r->s2, 4, 2, 0);
	test(*r->s1, 1, 0, *r->s2, 5, 0, 0);
	test(*r->s1, 1, 0, *r->s2, 5, 1, 0);
	test(*r->s1, 1, 0, *r->s2, 6, 0, 0);
	test(*r->s1, 1, 0, *r->s3, 0, 0, 0);
	test(*r->s1, 1, 0, *r->s3, 0, 1, 0);
	test(*r->s1, 1, 0, *r->s3, 0, 5, 0);
	test(*r->s1, 1, 0, *r->s3, 0, 9, 0);
	test(*r->s1, 1, 0, *r->s3, 0, 10, 0);
	test(*r->s1, 1, 0, *r->s3, 0, 11, 0);
	test(*r->s1, 1, 0, *r->s3, 1, 0, 0);
	test(*r->s1, 1, 0, *r->s3, 1, 1, 0);
	test(*r->s1, 1, 0, *r->s3, 1, 4, 0);
	test(*r->s1, 1, 0, *r->s3, 1, 8, 0);
	test(*r->s1, 1, 0, *r->s3, 1, 9, 0);
	test(*r->s1, 1, 0, *r->s3, 1, 10, 0);
	test(*r->s1, 1, 0, *r->s3, 5, 0, 0);
	test(*r->s1, 1, 0, *r->s3, 5, 1, 0);
	test(*r->s1, 1, 0, *r->s3, 5, 2, 0);
	test(*r->s1, 1, 0, *r->s3, 5, 4, 0);
	test(*r->s1, 1, 0, *r->s3, 5, 5, 0);
	test(*r->s1, 1, 0, *r->s3, 5, 6, 0);
	test(*r->s1, 1, 0, *r->s3, 9, 0, 0);
	test(*r->s1, 1, 0, *r->s3, 9, 1, 0);
	test(*r->s1, 1, 0, *r->s3, 9, 2, 0);
	test(*r->s1, 1, 0, *r->s3, 10, 0, 0);
	test(*r->s1, 1, 0, *r->s3, 10, 1, 0);
	test(*r->s1, 1, 0, *r->s3, 11, 0, 0);
	test(*r->s1, 1, 0, *r->s4, 0, 0, 0);
	test(*r->s1, 1, 0, *r->s4, 0, 1, 0);
}

template <class S>
void
test2(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s1, 1, 0, *r->s4, 0, 10, 0);
	test(*r->s1, 1, 0, *r->s4, 0, 19, 0);
	test(*r->s1, 1, 0, *r->s4, 0, 20, 0);
	test(*r->s1, 1, 0, *r->s4, 0, 21, 0);
	test(*r->s1, 1, 0, *r->s4, 1, 0, 0);
	test(*r->s1, 1, 0, *r->s4, 1, 1, 0);
	test(*r->s1, 1, 0, *r->s4, 1, 9, 0);
	test(*r->s1, 1, 0, *r->s4, 1, 18, 0);
	test(*r->s1, 1, 0, *r->s4, 1, 19, 0);
	test(*r->s1, 1, 0, *r->s4, 1, 20, 0);
	test(*r->s1, 1, 0, *r->s4, 10, 0, 0);
	test(*r->s1, 1, 0, *r->s4, 10, 1, 0);
	test(*r->s1, 1, 0, *r->s4, 10, 5, 0);
	test(*r->s1, 1, 0, *r->s4, 10, 9, 0);
	test(*r->s1, 1, 0, *r->s4, 10, 10, 0);
	test(*r->s1, 1, 0, *r->s4, 10, 11, 0);
	test(*r->s1, 1, 0, *r->s4, 19, 0, 0);
	test(*r->s1, 1, 0, *r->s4, 19, 1, 0);
	test(*r->s1, 1, 0, *r->s4, 19, 2, 0);
	test(*r->s1, 1, 0, *r->s4, 20, 0, 0);
	test(*r->s1, 1, 0, *r->s4, 20, 1, 0);
	test(*r->s1, 1, 0, *r->s4, 21, 0, 0);
	test(*r->s2, 0, 0, *r->s1, 0, 0, 0);
	test(*r->s2, 0, 0, *r->s1, 0, 1, 0);
	test(*r->s2, 0, 0, *r->s1, 1, 0, 0);
	test(*r->s2, 0, 0, *r->s2, 0, 0, 0);
	test(*r->s2, 0, 0, *r->s2, 0, 1, -1);
	test(*r->s2, 0, 0, *r->s2, 0, 2, -2);
	test(*r->s2, 0, 0, *r->s2, 0, 4, -4);
	test(*r->s2, 0, 0, *r->s2, 0, 5, -5);
	test(*r->s2, 0, 0, *r->s2, 0, 6, -5);
	test(*r->s2, 0, 0, *r->s2, 1, 0, 0);
	test(*r->s2, 0, 0, *r->s2, 1, 1, -1);
	test(*r->s2, 0, 0, *r->s2, 1, 2, -2);
	test(*r->s2, 0, 0, *r->s2, 1, 3, -3);
	test(*r->s2, 0, 0, *r->s2, 1, 4, -4);
	test(*r->s2, 0, 0, *r->s2, 1, 5, -4);
	test(*r->s2, 0, 0, *r->s2, 2, 0, 0);
	test(*r->s2, 0, 0, *r->s2, 2, 1, -1);
	test(*r->s2, 0, 0, *r->s2, 2, 2, -2);
	test(*r->s2, 0, 0, *r->s2, 2, 3, -3);
	test(*r->s2, 0, 0, *r->s2, 2, 4, -3);
	test(*r->s2, 0, 0, *r->s2, 4, 0, 0);
	test(*r->s2, 0, 0, *r->s2, 4, 1, -1);
	test(*r->s2, 0, 0, *r->s2, 4, 2, -1);
	test(*r->s2, 0, 0, *r->s2, 5, 0, 0);
	test(*r->s2, 0, 0, *r->s2, 5, 1, 0);
	test(*r->s2, 0, 0, *r->s2, 6, 0, 0);
	test(*r->s2, 0, 0, *r->s3, 0, 0, 0);
	test(*r->s2, 0, 0, *r->s3, 0, 1, -1);
	test(*r->s2, 0, 0, *r->s3, 0, 5, -5);
	test(*r->s2, 0, 0, *r->s3, 0, 9, -9);
	test(*r->s2, 0, 0, *r->s3, 0, 10, -10);
	test(*r->s2, 0, 0, *r->s3, 0, 11, -10);
	test(*r->s2, 0, 0, *r->s3, 1, 0, 0);
	test(*r->s2, 0, 0, *r->s3, 1, 1, -1);
	test(*r->s2, 0, 0, *r->s3, 1, 4, -4);
	test(*r->s2, 0, 0, *r->s3, 1, 8, -8);
	test(*r->s2, 0, 0, *r->s3, 1, 9, -9);
	test(*r->s2, 0, 0, *r->s3, 1, 10, -9);
	test(*r->s2, 0, 0, *r->s3, 5, 0, 0);
	test(*r->s2, 0, 0, *r->s3, 5, 1, -1);
	test(*r->s2, 0, 0, *r->s3, 5, 2, -2);
	test(*r->s2, 0, 0, *r->s3, 5, 4, -4);
	test(*r->s2, 0, 0, *r->s3, 5, 5, -5);
	test(*r->s2, 0, 0, *r->s3, 5, 6, -5);
	test(*r->s2, 0, 0, *r->s3, 9, 0, 0);
	test(*r->s2, 0, 0, *r->s3, 9, 1, -1);
	test(*r->s2, 0, 0, *r->s3, 9, 2, -1);
	test(*r->s2, 0, 0, *r->s3, 10, 0, 0);
	test(*r->s2, 0, 0, *r->s3, 10, 1, 0);
	test(*r->s2, 0, 0, *r->s3, 11, 0, 0);
	test(*r->s2, 0, 0, *r->s4, 0, 0, 0);
	test(*r->s2, 0, 0, *r->s4, 0, 1, -1);
	test(*r->s2, 0, 0, *r->s4, 0, 10, -10);
	test(*r->s2, 0, 0, *r->s4, 0, 19, -19);
	test(*r->s2, 0, 0, *r->s4, 0, 20, -20);
	test(*r->s2, 0, 0, *r->s4, 0, 21, -20);
	test(*r->s2, 0, 0, *r->s4, 1, 0, 0);
	test(*r->s2, 0, 0, *r->s4, 1, 1, -1);
	test(*r->s2, 0, 0, *r->s4, 1, 9, -9);
	test(*r->s2, 0, 0, *r->s4, 1, 18, -18);
	test(*r->s2, 0, 0, *r->s4, 1, 19, -19);
	test(*r->s2, 0, 0, *r->s4, 1, 20, -19);
	test(*r->s2, 0, 0, *r->s4, 10, 0, 0);
	test(*r->s2, 0, 0, *r->s4, 10, 1, -1);
	test(*r->s2, 0, 0, *r->s4, 10, 5, -5);
	test(*r->s2, 0, 0, *r->s4, 10, 9, -9);
	test(*r->s2, 0, 0, *r->s4, 10, 10, -10);
	test(*r->s2, 0, 0, *r->s4, 10, 11, -10);
	test(*r->s2, 0, 0, *r->s4, 19, 0, 0);
	test(*r->s2, 0, 0, *r->s4, 19, 1, -1);
	test(*r->s2, 0, 0, *r->s4, 19, 2, -1);
	test(*r->s2, 0, 0, *r->s4, 20, 0, 0);
	test(*r->s2, 0, 0, *r->s4, 20, 1, 0);
	test(*r->s2, 0, 0, *r->s4, 21, 0, 0);
	test(*r->s2, 0, 1, *r->s1, 0, 0, 1);
	test(*r->s2, 0, 1, *r->s1, 0, 1, 1);
	test(*r->s2, 0, 1, *r->s1, 1, 0, 0);
	test(*r->s2, 0, 1, *r->s2, 0, 0, 1);
}

template <class S>
void
test3(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 0, 1, *r->s2, 0, 1, 0);
	test(*r->s2, 0, 1, *r->s2, 0, 2, -1);
	test(*r->s2, 0, 1, *r->s2, 0, 4, -3);
	test(*r->s2, 0, 1, *r->s2, 0, 5, -4);
	test(*r->s2, 0, 1, *r->s2, 0, 6, -4);
	test(*r->s2, 0, 1, *r->s2, 1, 0, 1);
	test(*r->s2, 0, 1, *r->s2, 1, 1, -1);
	test(*r->s2, 0, 1, *r->s2, 1, 2, -1);
	test(*r->s2, 0, 1, *r->s2, 1, 3, -1);
	test(*r->s2, 0, 1, *r->s2, 1, 4, -1);
	test(*r->s2, 0, 1, *r->s2, 1, 5, -1);
	test(*r->s2, 0, 1, *r->s2, 2, 0, 1);
	test(*r->s2, 0, 1, *r->s2, 2, 1, -2);
	test(*r->s2, 0, 1, *r->s2, 2, 2, -2);
	test(*r->s2, 0, 1, *r->s2, 2, 3, -2);
	test(*r->s2, 0, 1, *r->s2, 2, 4, -2);
	test(*r->s2, 0, 1, *r->s2, 4, 0, 1);
	test(*r->s2, 0, 1, *r->s2, 4, 1, -4);
	test(*r->s2, 0, 1, *r->s2, 4, 2, -4);
	test(*r->s2, 0, 1, *r->s2, 5, 0, 1);
	test(*r->s2, 0, 1, *r->s2, 5, 1, 1);
	test(*r->s2, 0, 1, *r->s2, 6, 0, 0);
	test(*r->s2, 0, 1, *r->s3, 0, 0, 1);
	test(*r->s2, 0, 1, *r->s3, 0, 1, 0);
	test(*r->s2, 0, 1, *r->s3, 0, 5, -4);
	test(*r->s2, 0, 1, *r->s3, 0, 9, -8);
	test(*r->s2, 0, 1, *r->s3, 0, 10, -9);
	test(*r->s2, 0, 1, *r->s3, 0, 11, -9);
	test(*r->s2, 0, 1, *r->s3, 1, 0, 1);
	test(*r->s2, 0, 1, *r->s3, 1, 1, -1);
	test(*r->s2, 0, 1, *r->s3, 1, 4, -1);
	test(*r->s2, 0, 1, *r->s3, 1, 8, -1);
	test(*r->s2, 0, 1, *r->s3, 1, 9, -1);
	test(*r->s2, 0, 1, *r->s3, 1, 10, -1);
	test(*r->s2, 0, 1, *r->s3, 5, 0, 1);
	test(*r->s2, 0, 1, *r->s3, 5, 1, -5);
	test(*r->s2, 0, 1, *r->s3, 5, 2, -5);
	test(*r->s2, 0, 1, *r->s3, 5, 4, -5);
	test(*r->s2, 0, 1, *r->s3, 5, 5, -5);
	test(*r->s2, 0, 1, *r->s3, 5, 6, -5);
	test(*r->s2, 0, 1, *r->s3, 9, 0, 1);
	test(*r->s2, 0, 1, *r->s3, 9, 1, -9);
	test(*r->s2, 0, 1, *r->s3, 9, 2, -9);
	test(*r->s2, 0, 1, *r->s3, 10, 0, 1);
	test(*r->s2, 0, 1, *r->s3, 10, 1, 1);
	test(*r->s2, 0, 1, *r->s3, 11, 0, 0);
	test(*r->s2, 0, 1, *r->s4, 0, 0, 1);
	test(*r->s2, 0, 1, *r->s4, 0, 1, 0);
	test(*r->s2, 0, 1, *r->s4, 0, 10, -9);
	test(*r->s2, 0, 1, *r->s4, 0, 19, -18);
	test(*r->s2, 0, 1, *r->s4, 0, 20, -19);
	test(*r->s2, 0, 1, *r->s4, 0, 21, -19);
	test(*r->s2, 0, 1, *r->s4, 1, 0, 1);
	test(*r->s2, 0, 1, *r->s4, 1, 1, -1);
	test(*r->s2, 0, 1, *r->s4, 1, 9, -1);
	test(*r->s2, 0, 1, *r->s4, 1, 18, -1);
	test(*r->s2, 0, 1, *r->s4, 1, 19, -1);
	test(*r->s2, 0, 1, *r->s4, 1, 20, -1);
	test(*r->s2, 0, 1, *r->s4, 10, 0, 1);
	test(*r->s2, 0, 1, *r->s4, 10, 1, -10);
	test(*r->s2, 0, 1, *r->s4, 10, 5, -10);
	test(*r->s2, 0, 1, *r->s4, 10, 9, -10);
	test(*r->s2, 0, 1, *r->s4, 10, 10, -10);
	test(*r->s2, 0, 1, *r->s4, 10, 11, -10);
	test(*r->s2, 0, 1, *r->s4, 19, 0, 1);
	test(*r->s2, 0, 1, *r->s4, 19, 1, -19);
	test(*r->s2, 0, 1, *r->s4, 19, 2, -19);
	test(*r->s2, 0, 1, *r->s4, 20, 0, 1);
	test(*r->s2, 0, 1, *r->s4, 20, 1, 1);
	test(*r->s2, 0, 1, *r->s4, 21, 0, 0);
	test(*r->s2, 0, 2, *r->s1, 0, 0, 2);
	test(*r->s2, 0, 2, *r->s1, 0, 1, 2);
	test(*r->s2, 0, 2, *r->s1, 1, 0, 0);
	test(*r->s2, 0, 2, *r->s2, 0, 0, 2);
	test(*r->s2, 0, 2, *r->s2, 0, 1, 1);
	test(*r->s2, 0, 2, *r->s2, 0, 2, 0);
	test(*r->s2, 0, 2, *r->s2, 0, 4, -2);
	test(*r->s2, 0, 2, *r->s2, 0, 5, -3);
	test(*r->s2, 0, 2, *r->s2, 0, 6, -3);
	test(*r->s2, 0, 2, *r->s2, 1, 0, 2);
	test(*r->s2, 0, 2, *r->s2, 1, 1, -1);
	test(*r->s2, 0, 2, *r->s2, 1, 2, -1);
	test(*r->s2, 0, 2, *r->s2, 1, 3, -1);
	test(*r->s2, 0, 2, *r->s2, 1, 4, -1);
	test(*r->s2, 0, 2, *r->s2, 1, 5, -1);
	test(*r->s2, 0, 2, *r->s2, 2, 0, 2);
	test(*r->s2, 0, 2, *r->s2, 2, 1, -2);
	test(*r->s2, 0, 2, *r->s2, 2, 2, -2);
	test(*r->s2, 0, 2, *r->s2, 2, 3, -2);
	test(*r->s2, 0, 2, *r->s2, 2, 4, -2);
	test(*r->s2, 0, 2, *r->s2, 4, 0, 2);
	test(*r->s2, 0, 2, *r->s2, 4, 1, -4);
	test(*r->s2, 0, 2, *r->s2, 4, 2, -4);
	test(*r->s2, 0, 2, *r->s2, 5, 0, 2);
	test(*r->s2, 0, 2, *r->s2, 5, 1, 2);
	test(*r->s2, 0, 2, *r->s2, 6, 0, 0);
	test(*r->s2, 0, 2, *r->s3, 0, 0, 2);
	test(*r->s2, 0, 2, *r->s3, 0, 1, 1);
	test(*r->s2, 0, 2, *r->s3, 0, 5, -3);
	test(*r->s2, 0, 2, *r->s3, 0, 9, -7);
}

template <class S>
void
test4(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 0, 2, *r->s3, 0, 10, -8);
	test(*r->s2, 0, 2, *r->s3, 0, 11, -8);
	test(*r->s2, 0, 2, *r->s3, 1, 0, 2);
	test(*r->s2, 0, 2, *r->s3, 1, 1, -1);
	test(*r->s2, 0, 2, *r->s3, 1, 4, -1);
	test(*r->s2, 0, 2, *r->s3, 1, 8, -1);
	test(*r->s2, 0, 2, *r->s3, 1, 9, -1);
	test(*r->s2, 0, 2, *r->s3, 1, 10, -1);
	test(*r->s2, 0, 2, *r->s3, 5, 0, 2);
	test(*r->s2, 0, 2, *r->s3, 5, 1, -5);
	test(*r->s2, 0, 2, *r->s3, 5, 2, -5);
	test(*r->s2, 0, 2, *r->s3, 5, 4, -5);
	test(*r->s2, 0, 2, *r->s3, 5, 5, -5);
	test(*r->s2, 0, 2, *r->s3, 5, 6, -5);
	test(*r->s2, 0, 2, *r->s3, 9, 0, 2);
	test(*r->s2, 0, 2, *r->s3, 9, 1, -9);
	test(*r->s2, 0, 2, *r->s3, 9, 2, -9);
	test(*r->s2, 0, 2, *r->s3, 10, 0, 2);
	test(*r->s2, 0, 2, *r->s3, 10, 1, 2);
	test(*r->s2, 0, 2, *r->s3, 11, 0, 0);
	test(*r->s2, 0, 2, *r->s4, 0, 0, 2);
	test(*r->s2, 0, 2, *r->s4, 0, 1, 1);
	test(*r->s2, 0, 2, *r->s4, 0, 10, -8);
	test(*r->s2, 0, 2, *r->s4, 0, 19, -17);
	test(*r->s2, 0, 2, *r->s4, 0, 20, -18);
	test(*r->s2, 0, 2, *r->s4, 0, 21, -18);
	test(*r->s2, 0, 2, *r->s4, 1, 0, 2);
	test(*r->s2, 0, 2, *r->s4, 1, 1, -1);
	test(*r->s2, 0, 2, *r->s4, 1, 9, -1);
	test(*r->s2, 0, 2, *r->s4, 1, 18, -1);
	test(*r->s2, 0, 2, *r->s4, 1, 19, -1);
	test(*r->s2, 0, 2, *r->s4, 1, 20, -1);
	test(*r->s2, 0, 2, *r->s4, 10, 0, 2);
	test(*r->s2, 0, 2, *r->s4, 10, 1, -10);
	test(*r->s2, 0, 2, *r->s4, 10, 5, -10);
	test(*r->s2, 0, 2, *r->s4, 10, 9, -10);
	test(*r->s2, 0, 2, *r->s4, 10, 10, -10);
	test(*r->s2, 0, 2, *r->s4, 10, 11, -10);
	test(*r->s2, 0, 2, *r->s4, 19, 0, 2);
	test(*r->s2, 0, 2, *r->s4, 19, 1, -19);
	test(*r->s2, 0, 2, *r->s4, 19, 2, -19);
	test(*r->s2, 0, 2, *r->s4, 20, 0, 2);
	test(*r->s2, 0, 2, *r->s4, 20, 1, 2);
	test(*r->s2, 0, 2, *r->s4, 21, 0, 0);
	test(*r->s2, 0, 4, *r->s1, 0, 0, 4);
	test(*r->s2, 0, 4, *r->s1, 0, 1, 4);
	test(*r->s2, 0, 4, *r->s1, 1, 0, 0);
	test(*r->s2, 0, 4, *r->s2, 0, 0, 4);
	test(*r->s2, 0, 4, *r->s2, 0, 1, 3);
	test(*r->s2, 0, 4, *r->s2, 0, 2, 2);
	test(*r->s2, 0, 4, *r->s2, 0, 4, 0);
	test(*r->s2, 0, 4, *r->s2, 0, 5, -1);
	test(*r->s2, 0, 4, *r->s2, 0, 6, -1);
	test(*r->s2, 0, 4, *r->s2, 1, 0, 4);
	test(*r->s2, 0, 4, *r->s2, 1, 1, -1);
	test(*r->s2, 0, 4, *r->s2, 1, 2, -1);
	test(*r->s2, 0, 4, *r->s2, 1, 3, -1);
	test(*r->s2, 0, 4, *r->s2, 1, 4, -1);
	test(*r->s2, 0, 4, *r->s2, 1, 5, -1);
	test(*r->s2, 0, 4, *r->s2, 2, 0, 4);
	test(*r->s2, 0, 4, *r->s2, 2, 1, -2);
	test(*r->s2, 0, 4, *r->s2, 2, 2, -2);
	test(*r->s2, 0, 4, *r->s2, 2, 3, -2);
	test(*r->s2, 0, 4, *r->s2, 2, 4, -2);
	test(*r->s2, 0, 4, *r->s2, 4, 0, 4);
	test(*r->s2, 0, 4, *r->s2, 4, 1, -4);
	test(*r->s2, 0, 4, *r->s2, 4, 2, -4);
	test(*r->s2, 0, 4, *r->s2, 5, 0, 4);
	test(*r->s2, 0, 4, *r->s2, 5, 1, 4);
	test(*r->s2, 0, 4, *r->s2, 6, 0, 0);
	test(*r->s2, 0, 4, *r->s3, 0, 0, 4);
	test(*r->s2, 0, 4, *r->s3, 0, 1, 3);
	test(*r->s2, 0, 4, *r->s3, 0, 5, -1);
	test(*r->s2, 0, 4, *r->s3, 0, 9, -5);
	test(*r->s2, 0, 4, *r->s3, 0, 10, -6);
	test(*r->s2, 0, 4, *r->s3, 0, 11, -6);
	test(*r->s2, 0, 4, *r->s3, 1, 0, 4);
	test(*r->s2, 0, 4, *r->s3, 1, 1, -1);
	test(*r->s2, 0, 4, *r->s3, 1, 4, -1);
	test(*r->s2, 0, 4, *r->s3, 1, 8, -1);
	test(*r->s2, 0, 4, *r->s3, 1, 9, -1);
	test(*r->s2, 0, 4, *r->s3, 1, 10, -1);
	test(*r->s2, 0, 4, *r->s3, 5, 0, 4);
	test(*r->s2, 0, 4, *r->s3, 5, 1, -5);
	test(*r->s2, 0, 4, *r->s3, 5, 2, -5);
	test(*r->s2, 0, 4, *r->s3, 5, 4, -5);
	test(*r->s2, 0, 4, *r->s3, 5, 5, -5);
	test(*r->s2, 0, 4, *r->s3, 5, 6, -5);
	test(*r->s2, 0, 4, *r->s3, 9, 0, 4);
	test(*r->s2, 0, 4, *r->s3, 9, 1, -9);
	test(*r->s2, 0, 4, *r->s3, 9, 2, -9);
	test(*r->s2, 0, 4, *r->s3, 10, 0, 4);
	test(*r->s2, 0, 4, *r->s3, 10, 1, 4);
	test(*r->s2, 0, 4, *r->s3, 11, 0, 0);
	test(*r->s2, 0, 4, *r->s4, 0, 0, 4);
	test(*r->s2, 0, 4, *r->s4, 0, 1, 3);
	test(*r->s2, 0, 4, *r->s4, 0, 10, -6);
	test(*r->s2, 0, 4, *r->s4, 0, 19, -15);
	test(*r->s2, 0, 4, *r->s4, 0, 20, -16);
	test(*r->s2, 0, 4, *r->s4, 0, 21, -16);
}

template <class S>
void
test5(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 0, 4, *r->s4, 1, 0, 4);
	test(*r->s2, 0, 4, *r->s4, 1, 1, -1);
	test(*r->s2, 0, 4, *r->s4, 1, 9, -1);
	test(*r->s2, 0, 4, *r->s4, 1, 18, -1);
	test(*r->s2, 0, 4, *r->s4, 1, 19, -1);
	test(*r->s2, 0, 4, *r->s4, 1, 20, -1);
	test(*r->s2, 0, 4, *r->s4, 10, 0, 4);
	test(*r->s2, 0, 4, *r->s4, 10, 1, -10);
	test(*r->s2, 0, 4, *r->s4, 10, 5, -10);
	test(*r->s2, 0, 4, *r->s4, 10, 9, -10);
	test(*r->s2, 0, 4, *r->s4, 10, 10, -10);
	test(*r->s2, 0, 4, *r->s4, 10, 11, -10);
	test(*r->s2, 0, 4, *r->s4, 19, 0, 4);
	test(*r->s2, 0, 4, *r->s4, 19, 1, -19);
	test(*r->s2, 0, 4, *r->s4, 19, 2, -19);
	test(*r->s2, 0, 4, *r->s4, 20, 0, 4);
	test(*r->s2, 0, 4, *r->s4, 20, 1, 4);
	test(*r->s2, 0, 4, *r->s4, 21, 0, 0);
	test(*r->s2, 0, 5, *r->s1, 0, 0, 5);
	test(*r->s2, 0, 5, *r->s1, 0, 1, 5);
	test(*r->s2, 0, 5, *r->s1, 1, 0, 0);
	test(*r->s2, 0, 5, *r->s2, 0, 0, 5);
	test(*r->s2, 0, 5, *r->s2, 0, 1, 4);
	test(*r->s2, 0, 5, *r->s2, 0, 2, 3);
	test(*r->s2, 0, 5, *r->s2, 0, 4, 1);
	test(*r->s2, 0, 5, *r->s2, 0, 5, 0);
	test(*r->s2, 0, 5, *r->s2, 0, 6, 0);
	test(*r->s2, 0, 5, *r->s2, 1, 0, 5);
	test(*r->s2, 0, 5, *r->s2, 1, 1, -1);
	test(*r->s2, 0, 5, *r->s2, 1, 2, -1);
	test(*r->s2, 0, 5, *r->s2, 1, 3, -1);
	test(*r->s2, 0, 5, *r->s2, 1, 4, -1);
	test(*r->s2, 0, 5, *r->s2, 1, 5, -1);
	test(*r->s2, 0, 5, *r->s2, 2, 0, 5);
	test(*r->s2, 0, 5, *r->s2, 2, 1, -2);
	test(*r->s2, 0, 5, *r->s2, 2, 2, -2);
	test(*r->s2, 0, 5, *r->s2, 2, 3, -2);
	test(*r->s2, 0, 5, *r->s2, 2, 4, -2);
	test(*r->s2, 0, 5, *r->s2, 4, 0, 5);
	test(*r->s2, 0, 5, *r->s2, 4, 1, -4);
	test(*r->s2, 0, 5, *r->s2, 4, 2, -4);
	test(*r->s2, 0, 5, *r->s2, 5, 0, 5);
	test(*r->s2, 0, 5, *r->s2, 5, 1, 5);
	test(*r->s2, 0, 5, *r->s2, 6, 0, 0);
	test(*r->s2, 0, 5, *r->s3, 0, 0, 5);
	test(*r->s2, 0, 5, *r->s3, 0, 1, 4);
	test(*r->s2, 0, 5, *r->s3, 0, 5, 0);
	test(*r->s2, 0, 5, *r->s3, 0, 9, -4);
	test(*r->s2, 0, 5, *r->s3, 0, 10, -5);
	test(*r->s2, 0, 5, *r->s3, 0, 11, -5);
	test(*r->s2, 0, 5, *r->s3, 1, 0, 5);
	test(*r->s2, 0, 5, *r->s3, 1, 1, -1);
	test(*r->s2, 0, 5, *r->s3, 1, 4, -1);
	test(*r->s2, 0, 5, *r->s3, 1, 8, -1);
	test(*r->s2, 0, 5, *r->s3, 1, 9, -1);
	test(*r->s2, 0, 5, *r->s3, 1, 10, -1);
	test(*r->s2, 0, 5, *r->s3, 5, 0, 5);
	test(*r->s2, 0, 5, *r->s3, 5, 1, -5);
	test(*r->s2, 0, 5, *r->s3, 5, 2, -5);
	test(*r->s2, 0, 5, *r->s3, 5, 4, -5);
	test(*r->s2, 0, 5, *r->s3, 5, 5, -5);
	test(*r->s2, 0, 5, *r->s3, 5, 6, -5);
	test(*r->s2, 0, 5, *r->s3, 9, 0, 5);
	test(*r->s2, 0, 5, *r->s3, 9, 1, -9);
	test(*r->s2, 0, 5, *r->s3, 9, 2, -9);
	test(*r->s2, 0, 5, *r->s3, 10, 0, 5);
	test(*r->s2, 0, 5, *r->s3, 10, 1, 5);
	test(*r->s2, 0, 5, *r->s3, 11, 0, 0);
	test(*r->s2, 0, 5, *r->s4, 0, 0, 5);
	test(*r->s2, 0, 5, *r->s4, 0, 1, 4);
	test(*r->s2, 0, 5, *r->s4, 0, 10, -5);
	test(*r->s2, 0, 5, *r->s4, 0, 19, -14);
	test(*r->s2, 0, 5, *r->s4, 0, 20, -15);
	test(*r->s2, 0, 5, *r->s4, 0, 21, -15);
	test(*r->s2, 0, 5, *r->s4, 1, 0, 5);
	test(*r->s2, 0, 5, *r->s4, 1, 1, -1);
	test(*r->s2, 0, 5, *r->s4, 1, 9, -1);
	test(*r->s2, 0, 5, *r->s4, 1, 18, -1);
	test(*r->s2, 0, 5, *r->s4, 1, 19, -1);
	test(*r->s2, 0, 5, *r->s4, 1, 20, -1);
	test(*r->s2, 0, 5, *r->s4, 10, 0, 5);
	test(*r->s2, 0, 5, *r->s4, 10, 1, -10);
	test(*r->s2, 0, 5, *r->s4, 10, 5, -10);
	test(*r->s2, 0, 5, *r->s4, 10, 9, -10);
	test(*r->s2, 0, 5, *r->s4, 10, 10, -10);
	test(*r->s2, 0, 5, *r->s4, 10, 11, -10);
	test(*r->s2, 0, 5, *r->s4, 19, 0, 5);
	test(*r->s2, 0, 5, *r->s4, 19, 1, -19);
	test(*r->s2, 0, 5, *r->s4, 19, 2, -19);
	test(*r->s2, 0, 5, *r->s4, 20, 0, 5);
	test(*r->s2, 0, 5, *r->s4, 20, 1, 5);
	test(*r->s2, 0, 5, *r->s4, 21, 0, 0);
	test(*r->s2, 0, 6, *r->s1, 0, 0, 5);
	test(*r->s2, 0, 6, *r->s1, 0, 1, 5);
	test(*r->s2, 0, 6, *r->s1, 1, 0, 0);
	test(*r->s2, 0, 6, *r->s2, 0, 0, 5);
	test(*r->s2, 0, 6, *r->s2, 0, 1, 4);
	test(*r->s2, 0, 6, *r->s2, 0, 2, 3);
	test(*r->s2, 0, 6, *r->s2, 0, 4, 1);
	test(*r->s2, 0, 6, *r->s2, 0, 5, 0);
}

template <class S>
void
test6(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 0, 6, *r->s2, 0, 6, 0);
	test(*r->s2, 0, 6, *r->s2, 1, 0, 5);
	test(*r->s2, 0, 6, *r->s2, 1, 1, -1);
	test(*r->s2, 0, 6, *r->s2, 1, 2, -1);
	test(*r->s2, 0, 6, *r->s2, 1, 3, -1);
	test(*r->s2, 0, 6, *r->s2, 1, 4, -1);
	test(*r->s2, 0, 6, *r->s2, 1, 5, -1);
	test(*r->s2, 0, 6, *r->s2, 2, 0, 5);
	test(*r->s2, 0, 6, *r->s2, 2, 1, -2);
	test(*r->s2, 0, 6, *r->s2, 2, 2, -2);
	test(*r->s2, 0, 6, *r->s2, 2, 3, -2);
	test(*r->s2, 0, 6, *r->s2, 2, 4, -2);
	test(*r->s2, 0, 6, *r->s2, 4, 0, 5);
	test(*r->s2, 0, 6, *r->s2, 4, 1, -4);
	test(*r->s2, 0, 6, *r->s2, 4, 2, -4);
	test(*r->s2, 0, 6, *r->s2, 5, 0, 5);
	test(*r->s2, 0, 6, *r->s2, 5, 1, 5);
	test(*r->s2, 0, 6, *r->s2, 6, 0, 0);
	test(*r->s2, 0, 6, *r->s3, 0, 0, 5);
	test(*r->s2, 0, 6, *r->s3, 0, 1, 4);
	test(*r->s2, 0, 6, *r->s3, 0, 5, 0);
	test(*r->s2, 0, 6, *r->s3, 0, 9, -4);
	test(*r->s2, 0, 6, *r->s3, 0, 10, -5);
	test(*r->s2, 0, 6, *r->s3, 0, 11, -5);
	test(*r->s2, 0, 6, *r->s3, 1, 0, 5);
	test(*r->s2, 0, 6, *r->s3, 1, 1, -1);
	test(*r->s2, 0, 6, *r->s3, 1, 4, -1);
	test(*r->s2, 0, 6, *r->s3, 1, 8, -1);
	test(*r->s2, 0, 6, *r->s3, 1, 9, -1);
	test(*r->s2, 0, 6, *r->s3, 1, 10, -1);
	test(*r->s2, 0, 6, *r->s3, 5, 0, 5);
	test(*r->s2, 0, 6, *r->s3, 5, 1, -5);
	test(*r->s2, 0, 6, *r->s3, 5, 2, -5);
	test(*r->s2, 0, 6, *r->s3, 5, 4, -5);
	test(*r->s2, 0, 6, *r->s3, 5, 5, -5);
	test(*r->s2, 0, 6, *r->s3, 5, 6, -5);
	test(*r->s2, 0, 6, *r->s3, 9, 0, 5);
	test(*r->s2, 0, 6, *r->s3, 9, 1, -9);
	test(*r->s2, 0, 6, *r->s3, 9, 2, -9);
	test(*r->s2, 0, 6, *r->s3, 10, 0, 5);
	test(*r->s2, 0, 6, *r->s3, 10, 1, 5);
	test(*r->s2, 0, 6, *r->s3, 11, 0, 0);
	test(*r->s2, 0, 6, *r->s4, 0, 0, 5);
	test(*r->s2, 0, 6, *r->s4, 0, 1, 4);
	test(*r->s2, 0, 6, *r->s4, 0, 10, -5);
	test(*r->s2, 0, 6, *r->s4, 0, 19, -14);
	test(*r->s2, 0, 6, *r->s4, 0, 20, -15);
	test(*r->s2, 0, 6, *r->s4, 0, 21, -15);
	test(*r->s2, 0, 6, *r->s4, 1, 0, 5);
	test(*r->s2, 0, 6, *r->s4, 1, 1, -1);
	test(*r->s2, 0, 6, *r->s4, 1, 9, -1);
	test(*r->s2, 0, 6, *r->s4, 1, 18, -1);
	test(*r->s2, 0, 6, *r->s4, 1, 19, -1);
	test(*r->s2, 0, 6, *r->s4, 1, 20, -1);
	test(*r->s2, 0, 6, *r->s4, 10, 0, 5);
	test(*r->s2, 0, 6, *r->s4, 10, 1, -10);
	test(*r->s2, 0, 6, *r->s4, 10, 5, -10);
	test(*r->s2, 0, 6, *r->s4, 10, 9, -10);
	test(*r->s2, 0, 6, *r->s4, 10, 10, -10);
	test(*r->s2, 0, 6, *r->s4, 10, 11, -10);
	test(*r->s2, 0, 6, *r->s4, 19, 0, 5);
	test(*r->s2, 0, 6, *r->s4, 19, 1, -19);
	test(*r->s2, 0, 6, *r->s4, 19, 2, -19);
	test(*r->s2, 0, 6, *r->s4, 20, 0, 5);
	test(*r->s2, 0, 6, *r->s4, 20, 1, 5);
	test(*r->s2, 0, 6, *r->s4, 21, 0, 0);
	test(*r->s2, 1, 0, *r->s1, 0, 0, 0);
	test(*r->s2, 1, 0, *r->s1, 0, 1, 0);
	test(*r->s2, 1, 0, *r->s1, 1, 0, 0);
	test(*r->s2, 1, 0, *r->s2, 0, 0, 0);
	test(*r->s2, 1, 0, *r->s2, 0, 1, -1);
	test(*r->s2, 1, 0, *r->s2, 0, 2, -2);
	test(*r->s2, 1, 0, *r->s2, 0, 4, -4);
	test(*r->s2, 1, 0, *r->s2, 0, 5, -5);
	test(*r->s2, 1, 0, *r->s2, 0, 6, -5);
	test(*r->s2, 1, 0, *r->s2, 1, 0, 0);
	test(*r->s2, 1, 0, *r->s2, 1, 1, -1);
	test(*r->s2, 1, 0, *r->s2, 1, 2, -2);
	test(*r->s2, 1, 0, *r->s2, 1, 3, -3);
	test(*r->s2, 1, 0, *r->s2, 1, 4, -4);
	test(*r->s2, 1, 0, *r->s2, 1, 5, -4);
	test(*r->s2, 1, 0, *r->s2, 2, 0, 0);
	test(*r->s2, 1, 0, *r->s2, 2, 1, -1);
	test(*r->s2, 1, 0, *r->s2, 2, 2, -2);
	test(*r->s2, 1, 0, *r->s2, 2, 3, -3);
	test(*r->s2, 1, 0, *r->s2, 2, 4, -3);
	test(*r->s2, 1, 0, *r->s2, 4, 0, 0);
	test(*r->s2, 1, 0, *r->s2, 4, 1, -1);
	test(*r->s2, 1, 0, *r->s2, 4, 2, -1);
	test(*r->s2, 1, 0, *r->s2, 5, 0, 0);
	test(*r->s2, 1, 0, *r->s2, 5, 1, 0);
	test(*r->s2, 1, 0, *r->s2, 6, 0, 0);
	test(*r->s2, 1, 0, *r->s3, 0, 0, 0);
	test(*r->s2, 1, 0, *r->s3, 0, 1, -1);
	test(*r->s2, 1, 0, *r->s3, 0, 5, -5);
	test(*r->s2, 1, 0, *r->s3, 0, 9, -9);
	test(*r->s2, 1, 0, *r->s3, 0, 10, -10);
	test(*r->s2, 1, 0, *r->s3, 0, 11, -10);
	test(*r->s2, 1, 0, *r->s3, 1, 0, 0);
	test(*r->s2, 1, 0, *r->s3, 1, 1, -1);
}

template <class S>
void
test7(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 1, 0, *r->s3, 1, 4, -4);
	test(*r->s2, 1, 0, *r->s3, 1, 8, -8);
	test(*r->s2, 1, 0, *r->s3, 1, 9, -9);
	test(*r->s2, 1, 0, *r->s3, 1, 10, -9);
	test(*r->s2, 1, 0, *r->s3, 5, 0, 0);
	test(*r->s2, 1, 0, *r->s3, 5, 1, -1);
	test(*r->s2, 1, 0, *r->s3, 5, 2, -2);
	test(*r->s2, 1, 0, *r->s3, 5, 4, -4);
	test(*r->s2, 1, 0, *r->s3, 5, 5, -5);
	test(*r->s2, 1, 0, *r->s3, 5, 6, -5);
	test(*r->s2, 1, 0, *r->s3, 9, 0, 0);
	test(*r->s2, 1, 0, *r->s3, 9, 1, -1);
	test(*r->s2, 1, 0, *r->s3, 9, 2, -1);
	test(*r->s2, 1, 0, *r->s3, 10, 0, 0);
	test(*r->s2, 1, 0, *r->s3, 10, 1, 0);
	test(*r->s2, 1, 0, *r->s3, 11, 0, 0);
	test(*r->s2, 1, 0, *r->s4, 0, 0, 0);
	test(*r->s2, 1, 0, *r->s4, 0, 1, -1);
	test(*r->s2, 1, 0, *r->s4, 0, 10, -10);
	test(*r->s2, 1, 0, *r->s4, 0, 19, -19);
	test(*r->s2, 1, 0, *r->s4, 0, 20, -20);
	test(*r->s2, 1, 0, *r->s4, 0, 21, -20);
	test(*r->s2, 1, 0, *r->s4, 1, 0, 0);
	test(*r->s2, 1, 0, *r->s4, 1, 1, -1);
	test(*r->s2, 1, 0, *r->s4, 1, 9, -9);
	test(*r->s2, 1, 0, *r->s4, 1, 18, -18);
	test(*r->s2, 1, 0, *r->s4, 1, 19, -19);
	test(*r->s2, 1, 0, *r->s4, 1, 20, -19);
	test(*r->s2, 1, 0, *r->s4, 10, 0, 0);
	test(*r->s2, 1, 0, *r->s4, 10, 1, -1);
	test(*r->s2, 1, 0, *r->s4, 10, 5, -5);
	test(*r->s2, 1, 0, *r->s4, 10, 9, -9);
	test(*r->s2, 1, 0, *r->s4, 10, 10, -10);
	test(*r->s2, 1, 0, *r->s4, 10, 11, -10);
	test(*r->s2, 1, 0, *r->s4, 19, 0, 0);
	test(*r->s2, 1, 0, *r->s4, 19, 1, -1);
	test(*r->s2, 1, 0, *r->s4, 19, 2, -1);
	test(*r->s2, 1, 0, *r->s4, 20, 0, 0);
	test(*r->s2, 1, 0, *r->s4, 20, 1, 0);
	test(*r->s2, 1, 0, *r->s4, 21, 0, 0);
	test(*r->s2, 1, 1, *r->s1, 0, 0, 1);
	test(*r->s2, 1, 1, *r->s1, 0, 1, 1);
	test(*r->s2, 1, 1, *r->s1, 1, 0, 0);
	test(*r->s2, 1, 1, *r->s2, 0, 0, 1);
	test(*r->s2, 1, 1, *r->s2, 0, 1, 1);
	test(*r->s2, 1, 1, *r->s2, 0, 2, 1);
	test(*r->s2, 1, 1, *r->s2, 0, 4, 1);
	test(*r->s2, 1, 1, *r->s2, 0, 5, 1);
	test(*r->s2, 1, 1, *r->s2, 0, 6, 1);
	test(*r->s2, 1, 1, *r->s2, 1, 0, 1);
	test(*r->s2, 1, 1, *r->s2, 1, 1, 0);
	test(*r->s2, 1, 1, *r->s2, 1, 2, -1);
	test(*r->s2, 1, 1, *r->s2, 1, 3, -2);
	test(*r->s2, 1, 1, *r->s2, 1, 4, -3);
	test(*r->s2, 1, 1, *r->s2, 1, 5, -3);
	test(*r->s2, 1, 1, *r->s2, 2, 0, 1);
	test(*r->s2, 1, 1, *r->s2, 2, 1, -1);
	test(*r->s2, 1, 1, *r->s2, 2, 2, -1);
	test(*r->s2, 1, 1, *r->s2, 2, 3, -1);
	test(*r->s2, 1, 1, *r->s2, 2, 4, -1);
	test(*r->s2, 1, 1, *r->s2, 4, 0, 1);
	test(*r->s2, 1, 1, *r->s2, 4, 1, -3);
	test(*r->s2, 1, 1, *r->s2, 4, 2, -3);
	test(*r->s2, 1, 1, *r->s2, 5, 0, 1);
	test(*r->s2, 1, 1, *r->s2, 5, 1, 1);
	test(*r->s2, 1, 1, *r->s2, 6, 0, 0);
	test(*r->s2, 1, 1, *r->s3, 0, 0, 1);
	test(*r->s2, 1, 1, *r->s3, 0, 1, 1);
	test(*r->s2, 1, 1, *r->s3, 0, 5, 1);
	test(*r->s2, 1, 1, *r->s3, 0, 9, 1);
	test(*r->s2, 1, 1, *r->s3, 0, 10, 1);
	test(*r->s2, 1, 1, *r->s3, 0, 11, 1);
	test(*r->s2, 1, 1, *r->s3, 1, 0, 1);
	test(*r->s2, 1, 1, *r->s3, 1, 1, 0);
	test(*r->s2, 1, 1, *r->s3, 1, 4, -3);
	test(*r->s2, 1, 1, *r->s3, 1, 8, -7);
	test(*r->s2, 1, 1, *r->s3, 1, 9, -8);
	test(*r->s2, 1, 1, *r->s3, 1, 10, -8);
	test(*r->s2, 1, 1, *r->s3, 5, 0, 1);
	test(*r->s2, 1, 1, *r->s3, 5, 1, -4);
	test(*r->s2, 1, 1, *r->s3, 5, 2, -4);
	test(*r->s2, 1, 1, *r->s3, 5, 4, -4);
	test(*r->s2, 1, 1, *r->s3, 5, 5, -4);
	test(*r->s2, 1, 1, *r->s3, 5, 6, -4);
	test(*r->s2, 1, 1, *r->s3, 9, 0, 1);
	test(*r->s2, 1, 1, *r->s3, 9, 1, -8);
	test(*r->s2, 1, 1, *r->s3, 9, 2, -8);
	test(*r->s2, 1, 1, *r->s3, 10, 0, 1);
	test(*r->s2, 1, 1, *r->s3, 10, 1, 1);
	test(*r->s2, 1, 1, *r->s3, 11, 0, 0);
	test(*r->s2, 1, 1, *r->s4, 0, 0, 1);
	test(*r->s2, 1, 1, *r->s4, 0, 1, 1);
	test(*r->s2, 1, 1, *r->s4, 0, 10, 1);
	test(*r->s2, 1, 1, *r->s4, 0, 19, 1);
	test(*r->s2, 1, 1, *r->s4, 0, 20, 1);
	test(*r->s2, 1, 1, *r->s4, 0, 21, 1);
	test(*r->s2, 1, 1, *r->s4, 1, 0, 1);
	test(*r->s2, 1, 1, *r->s4, 1, 1, 0);
	test(*r->s2, 1, 1, *r->s4, 1, 9, -8);
	test(*r->s2, 1, 1, *r->s4, 1, 18, -17);
}

template <class S>
void
test8(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 1, 1, *r->s4, 1, 19, -18);
	test(*r->s2, 1, 1, *r->s4, 1, 20, -18);
	test(*r->s2, 1, 1, *r->s4, 10, 0, 1);
	test(*r->s2, 1, 1, *r->s4, 10, 1, -9);
	test(*r->s2, 1, 1, *r->s4, 10, 5, -9);
	test(*r->s2, 1, 1, *r->s4, 10, 9, -9);
	test(*r->s2, 1, 1, *r->s4, 10, 10, -9);
	test(*r->s2, 1, 1, *r->s4, 10, 11, -9);
	test(*r->s2, 1, 1, *r->s4, 19, 0, 1);
	test(*r->s2, 1, 1, *r->s4, 19, 1, -18);
	test(*r->s2, 1, 1, *r->s4, 19, 2, -18);
	test(*r->s2, 1, 1, *r->s4, 20, 0, 1);
	test(*r->s2, 1, 1, *r->s4, 20, 1, 1);
	test(*r->s2, 1, 1, *r->s4, 21, 0, 0);
	test(*r->s2, 1, 2, *r->s1, 0, 0, 2);
	test(*r->s2, 1, 2, *r->s1, 0, 1, 2);
	test(*r->s2, 1, 2, *r->s1, 1, 0, 0);
	test(*r->s2, 1, 2, *r->s2, 0, 0, 2);
	test(*r->s2, 1, 2, *r->s2, 0, 1, 1);
	test(*r->s2, 1, 2, *r->s2, 0, 2, 1);
	test(*r->s2, 1, 2, *r->s2, 0, 4, 1);
	test(*r->s2, 1, 2, *r->s2, 0, 5, 1);
	test(*r->s2, 1, 2, *r->s2, 0, 6, 1);
	test(*r->s2, 1, 2, *r->s2, 1, 0, 2);
	test(*r->s2, 1, 2, *r->s2, 1, 1, 1);
	test(*r->s2, 1, 2, *r->s2, 1, 2, 0);
	test(*r->s2, 1, 2, *r->s2, 1, 3, -1);
	test(*r->s2, 1, 2, *r->s2, 1, 4, -2);
	test(*r->s2, 1, 2, *r->s2, 1, 5, -2);
	test(*r->s2, 1, 2, *r->s2, 2, 0, 2);
	test(*r->s2, 1, 2, *r->s2, 2, 1, -1);
	test(*r->s2, 1, 2, *r->s2, 2, 2, -1);
	test(*r->s2, 1, 2, *r->s2, 2, 3, -1);
	test(*r->s2, 1, 2, *r->s2, 2, 4, -1);
	test(*r->s2, 1, 2, *r->s2, 4, 0, 2);
	test(*r->s2, 1, 2, *r->s2, 4, 1, -3);
	test(*r->s2, 1, 2, *r->s2, 4, 2, -3);
	test(*r->s2, 1, 2, *r->s2, 5, 0, 2);
	test(*r->s2, 1, 2, *r->s2, 5, 1, 2);
	test(*r->s2, 1, 2, *r->s2, 6, 0, 0);
	test(*r->s2, 1, 2, *r->s3, 0, 0, 2);
	test(*r->s2, 1, 2, *r->s3, 0, 1, 1);
	test(*r->s2, 1, 2, *r->s3, 0, 5, 1);
	test(*r->s2, 1, 2, *r->s3, 0, 9, 1);
	test(*r->s2, 1, 2, *r->s3, 0, 10, 1);
	test(*r->s2, 1, 2, *r->s3, 0, 11, 1);
	test(*r->s2, 1, 2, *r->s3, 1, 0, 2);
	test(*r->s2, 1, 2, *r->s3, 1, 1, 1);
	test(*r->s2, 1, 2, *r->s3, 1, 4, -2);
	test(*r->s2, 1, 2, *r->s3, 1, 8, -6);
	test(*r->s2, 1, 2, *r->s3, 1, 9, -7);
	test(*r->s2, 1, 2, *r->s3, 1, 10, -7);
	test(*r->s2, 1, 2, *r->s3, 5, 0, 2);
	test(*r->s2, 1, 2, *r->s3, 5, 1, -4);
	test(*r->s2, 1, 2, *r->s3, 5, 2, -4);
	test(*r->s2, 1, 2, *r->s3, 5, 4, -4);
	test(*r->s2, 1, 2, *r->s3, 5, 5, -4);
	test(*r->s2, 1, 2, *r->s3, 5, 6, -4);
	test(*r->s2, 1, 2, *r->s3, 9, 0, 2);
	test(*r->s2, 1, 2, *r->s3, 9, 1, -8);
	test(*r->s2, 1, 2, *r->s3, 9, 2, -8);
	test(*r->s2, 1, 2, *r->s3, 10, 0, 2);
	test(*r->s2, 1, 2, *r->s3, 10, 1, 2);
	test(*r->s2, 1, 2, *r->s3, 11, 0, 0);
	test(*r->s2, 1, 2, *r->s4, 0, 0, 2);
	test(*r->s2, 1, 2, *r->s4, 0, 1, 1);
	test(*r->s2, 1, 2, *r->s4, 0, 10, 1);
	test(*r->s2, 1, 2, *r->s4, 0, 19, 1);
	test(*r->s2, 1, 2, *r->s4, 0, 20, 1);
	test(*r->s2, 1, 2, *r->s4, 0, 21, 1);
	test(*r->s2, 1, 2, *r->s4, 1, 0, 2);
	test(*r->s2, 1, 2, *r->s4, 1, 1, 1);
	test(*r->s2, 1, 2, *r->s4, 1, 9, -7);
	test(*r->s2, 1, 2, *r->s4, 1, 18, -16);
	test(*r->s2, 1, 2, *r->s4, 1, 19, -17);
	test(*r->s2, 1, 2, *r->s4, 1, 20, -17);
	test(*r->s2, 1, 2, *r->s4, 10, 0, 2);
	test(*r->s2, 1, 2, *r->s4, 10, 1, -9);
	test(*r->s2, 1, 2, *r->s4, 10, 5, -9);
	test(*r->s2, 1, 2, *r->s4, 10, 9, -9);
	test(*r->s2, 1, 2, *r->s4, 10, 10, -9);
	test(*r->s2, 1, 2, *r->s4, 10, 11, -9);
	test(*r->s2, 1, 2, *r->s4, 19, 0, 2);
	test(*r->s2, 1, 2, *r->s4, 19, 1, -18);
	test(*r->s2, 1, 2, *r->s4, 19, 2, -18);
	test(*r->s2, 1, 2, *r->s4, 20, 0, 2);
	test(*r->s2, 1, 2, *r->s4, 20, 1, 2);
	test(*r->s2, 1, 2, *r->s4, 21, 0, 0);
	test(*r->s2, 1, 3, *r->s1, 0, 0, 3);
	test(*r->s2, 1, 3, *r->s1, 0, 1, 3);
	test(*r->s2, 1, 3, *r->s1, 1, 0, 0);
	test(*r->s2, 1, 3, *r->s2, 0, 0, 3);
	test(*r->s2, 1, 3, *r->s2, 0, 1, 1);
	test(*r->s2, 1, 3, *r->s2, 0, 2, 1);
	test(*r->s2, 1, 3, *r->s2, 0, 4, 1);
	test(*r->s2, 1, 3, *r->s2, 0, 5, 1);
	test(*r->s2, 1, 3, *r->s2, 0, 6, 1);
	test(*r->s2, 1, 3, *r->s2, 1, 0, 3);
	test(*r->s2, 1, 3, *r->s2, 1, 1, 2);
	test(*r->s2, 1, 3, *r->s2, 1, 2, 1);
}

template <class S>
void
test9(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 1, 3, *r->s2, 1, 3, 0);
	test(*r->s2, 1, 3, *r->s2, 1, 4, -1);
	test(*r->s2, 1, 3, *r->s2, 1, 5, -1);
	test(*r->s2, 1, 3, *r->s2, 2, 0, 3);
	test(*r->s2, 1, 3, *r->s2, 2, 1, -1);
	test(*r->s2, 1, 3, *r->s2, 2, 2, -1);
	test(*r->s2, 1, 3, *r->s2, 2, 3, -1);
	test(*r->s2, 1, 3, *r->s2, 2, 4, -1);
	test(*r->s2, 1, 3, *r->s2, 4, 0, 3);
	test(*r->s2, 1, 3, *r->s2, 4, 1, -3);
	test(*r->s2, 1, 3, *r->s2, 4, 2, -3);
	test(*r->s2, 1, 3, *r->s2, 5, 0, 3);
	test(*r->s2, 1, 3, *r->s2, 5, 1, 3);
	test(*r->s2, 1, 3, *r->s2, 6, 0, 0);
	test(*r->s2, 1, 3, *r->s3, 0, 0, 3);
	test(*r->s2, 1, 3, *r->s3, 0, 1, 1);
	test(*r->s2, 1, 3, *r->s3, 0, 5, 1);
	test(*r->s2, 1, 3, *r->s3, 0, 9, 1);
	test(*r->s2, 1, 3, *r->s3, 0, 10, 1);
	test(*r->s2, 1, 3, *r->s3, 0, 11, 1);
	test(*r->s2, 1, 3, *r->s3, 1, 0, 3);
	test(*r->s2, 1, 3, *r->s3, 1, 1, 2);
	test(*r->s2, 1, 3, *r->s3, 1, 4, -1);
	test(*r->s2, 1, 3, *r->s3, 1, 8, -5);
	test(*r->s2, 1, 3, *r->s3, 1, 9, -6);
	test(*r->s2, 1, 3, *r->s3, 1, 10, -6);
	test(*r->s2, 1, 3, *r->s3, 5, 0, 3);
	test(*r->s2, 1, 3, *r->s3, 5, 1, -4);
	test(*r->s2, 1, 3, *r->s3, 5, 2, -4);
	test(*r->s2, 1, 3, *r->s3, 5, 4, -4);
	test(*r->s2, 1, 3, *r->s3, 5, 5, -4);
	test(*r->s2, 1, 3, *r->s3, 5, 6, -4);
	test(*r->s2, 1, 3, *r->s3, 9, 0, 3);
	test(*r->s2, 1, 3, *r->s3, 9, 1, -8);
	test(*r->s2, 1, 3, *r->s3, 9, 2, -8);
	test(*r->s2, 1, 3, *r->s3, 10, 0, 3);
	test(*r->s2, 1, 3, *r->s3, 10, 1, 3);
	test(*r->s2, 1, 3, *r->s3, 11, 0, 0);
	test(*r->s2, 1, 3, *r->s4, 0, 0, 3);
	test(*r->s2, 1, 3, *r->s4, 0, 1, 1);
	test(*r->s2, 1, 3, *r->s4, 0, 10, 1);
	test(*r->s2, 1, 3, *r->s4, 0, 19, 1);
	test(*r->s2, 1, 3, *r->s4, 0, 20, 1);
	test(*r->s2, 1, 3, *r->s4, 0, 21, 1);
	test(*r->s2, 1, 3, *r->s4, 1, 0, 3);
	test(*r->s2, 1, 3, *r->s4, 1, 1, 2);
	test(*r->s2, 1, 3, *r->s4, 1, 9, -6);
	test(*r->s2, 1, 3, *r->s4, 1, 18, -15);
	test(*r->s2, 1, 3, *r->s4, 1, 19, -16);
	test(*r->s2, 1, 3, *r->s4, 1, 20, -16);
	test(*r->s2, 1, 3, *r->s4, 10, 0, 3);
	test(*r->s2, 1, 3, *r->s4, 10, 1, -9);
	test(*r->s2, 1, 3, *r->s4, 10, 5, -9);
	test(*r->s2, 1, 3, *r->s4, 10, 9, -9);
	test(*r->s2, 1, 3, *r->s4, 10, 10, -9);
	test(*r->s2, 1, 3, *r->s4, 10, 11, -9);
	test(*r->s2, 1, 3, *r->s4, 19, 0, 3);
	test(*r->s2, 1, 3, *r->s4, 19, 1, -18);
	test(*r->s2, 1, 3, *r->s4, 19, 2, -18);
	test(*r->s2, 1, 3, *r->s4, 20, 0, 3);
	test(*r->s2, 1, 3, *r->s4, 20, 1, 3);
	test(*r->s2, 1, 3, *r->s4, 21, 0, 0);
	test(*r->s2, 1, 4, *r->s1, 0, 0, 4);
	test(*r->s2, 1, 4, *r->s1, 0, 1, 4);
	test(*r->s2, 1, 4, *r->s1, 1, 0, 0);
	test(*r->s2, 1, 4, *r->s2, 0, 0, 4);
	test(*r->s2, 1, 4, *r->s2, 0, 1, 1);
	test(*r->s2, 1, 4, *r->s2, 0, 2, 1);
	test(*r->s2, 1, 4, *r->s2, 0, 4, 1);
	test(*r->s2, 1, 4, *r->s2, 0, 5, 1);
	test(*r->s2, 1, 4, *r->s2, 0, 6, 1);
	test(*r->s2, 1, 4, *r->s2, 1, 0, 4);
	test(*r->s2, 1, 4, *r->s2, 1, 1, 3);
	test(*r->s2, 1, 4, *r->s2, 1, 2, 2);
	test(*r->s2, 1, 4, *r->s2, 1, 3, 1);
	test(*r->s2, 1, 4, *r->s2, 1, 4, 0);
	test(*r->s2, 1, 4, *r->s2, 1, 5, 0);
	test(*r->s2, 1, 4, *r->s2, 2, 0, 4);
	test(*r->s2, 1, 4, *r->s2, 2, 1, -1);
	test(*r->s2, 1, 4, *r->s2, 2, 2, -1);
	test(*r->s2, 1, 4, *r->s2, 2, 3, -1);
	test(*r->s2, 1, 4, *r->s2, 2, 4, -1);
	test(*r->s2, 1, 4, *r->s2, 4, 0, 4);
	test(*r->s2, 1, 4, *r->s2, 4, 1, -3);
	test(*r->s2, 1, 4, *r->s2, 4, 2, -3);
	test(*r->s2, 1, 4, *r->s2, 5, 0, 4);
	test(*r->s2, 1, 4, *r->s2, 5, 1, 4);
	test(*r->s2, 1, 4, *r->s2, 6, 0, 0);
	test(*r->s2, 1, 4, *r->s3, 0, 0, 4);
	test(*r->s2, 1, 4, *r->s3, 0, 1, 1);
	test(*r->s2, 1, 4, *r->s3, 0, 5, 1);
	test(*r->s2, 1, 4, *r->s3, 0, 9, 1);
	test(*r->s2, 1, 4, *r->s3, 0, 10, 1);
	test(*r->s2, 1, 4, *r->s3, 0, 11, 1);
	test(*r->s2, 1, 4, *r->s3, 1, 0, 4);
	test(*r->s2, 1, 4, *r->s3, 1, 1, 3);
	test(*r->s2, 1, 4, *r->s3, 1, 4, 0);
	test(*r->s2, 1, 4, *r->s3, 1, 8, -4);
	test(*r->s2, 1, 4, *r->s3, 1, 9, -5);
	test(*r->s2, 1, 4, *r->s3, 1, 10, -5);
}

template <class S>
void
test10(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 1, 4, *r->s3, 5, 0, 4);
	test(*r->s2, 1, 4, *r->s3, 5, 1, -4);
	test(*r->s2, 1, 4, *r->s3, 5, 2, -4);
	test(*r->s2, 1, 4, *r->s3, 5, 4, -4);
	test(*r->s2, 1, 4, *r->s3, 5, 5, -4);
	test(*r->s2, 1, 4, *r->s3, 5, 6, -4);
	test(*r->s2, 1, 4, *r->s3, 9, 0, 4);
	test(*r->s2, 1, 4, *r->s3, 9, 1, -8);
	test(*r->s2, 1, 4, *r->s3, 9, 2, -8);
	test(*r->s2, 1, 4, *r->s3, 10, 0, 4);
	test(*r->s2, 1, 4, *r->s3, 10, 1, 4);
	test(*r->s2, 1, 4, *r->s3, 11, 0, 0);
	test(*r->s2, 1, 4, *r->s4, 0, 0, 4);
	test(*r->s2, 1, 4, *r->s4, 0, 1, 1);
	test(*r->s2, 1, 4, *r->s4, 0, 10, 1);
	test(*r->s2, 1, 4, *r->s4, 0, 19, 1);
	test(*r->s2, 1, 4, *r->s4, 0, 20, 1);
	test(*r->s2, 1, 4, *r->s4, 0, 21, 1);
	test(*r->s2, 1, 4, *r->s4, 1, 0, 4);
	test(*r->s2, 1, 4, *r->s4, 1, 1, 3);
	test(*r->s2, 1, 4, *r->s4, 1, 9, -5);
	test(*r->s2, 1, 4, *r->s4, 1, 18, -14);
	test(*r->s2, 1, 4, *r->s4, 1, 19, -15);
	test(*r->s2, 1, 4, *r->s4, 1, 20, -15);
	test(*r->s2, 1, 4, *r->s4, 10, 0, 4);
	test(*r->s2, 1, 4, *r->s4, 10, 1, -9);
	test(*r->s2, 1, 4, *r->s4, 10, 5, -9);
	test(*r->s2, 1, 4, *r->s4, 10, 9, -9);
	test(*r->s2, 1, 4, *r->s4, 10, 10, -9);
	test(*r->s2, 1, 4, *r->s4, 10, 11, -9);
	test(*r->s2, 1, 4, *r->s4, 19, 0, 4);
	test(*r->s2, 1, 4, *r->s4, 19, 1, -18);
	test(*r->s2, 1, 4, *r->s4, 19, 2, -18);
	test(*r->s2, 1, 4, *r->s4, 20, 0, 4);
	test(*r->s2, 1, 4, *r->s4, 20, 1, 4);
	test(*r->s2, 1, 4, *r->s4, 21, 0, 0);
	test(*r->s2, 1, 5, *r->s1, 0, 0, 4);
	test(*r->s2, 1, 5, *r->s1, 0, 1, 4);
	test(*r->s2, 1, 5, *r->s1, 1, 0, 0);
	test(*r->s2, 1, 5, *r->s2, 0, 0, 4);
	test(*r->s2, 1, 5, *r->s2, 0, 1, 1);
	test(*r->s2, 1, 5, *r->s2, 0, 2, 1);
	test(*r->s2, 1, 5, *r->s2, 0, 4, 1);
	test(*r->s2, 1, 5, *r->s2, 0, 5, 1);
	test(*r->s2, 1, 5, *r->s2, 0, 6, 1);
	test(*r->s2, 1, 5, *r->s2, 1, 0, 4);
	test(*r->s2, 1, 5, *r->s2, 1, 1, 3);
	test(*r->s2, 1, 5, *r->s2, 1, 2, 2);
	test(*r->s2, 1, 5, *r->s2, 1, 3, 1);
	test(*r->s2, 1, 5, *r->s2, 1, 4, 0);
	test(*r->s2, 1, 5, *r->s2, 1, 5, 0);
	test(*r->s2, 1, 5, *r->s2, 2, 0, 4);
	test(*r->s2, 1, 5, *r->s2, 2, 1, -1);
	test(*r->s2, 1, 5, *r->s2, 2, 2, -1);
	test(*r->s2, 1, 5, *r->s2, 2, 3, -1);
	test(*r->s2, 1, 5, *r->s2, 2, 4, -1);
	test(*r->s2, 1, 5, *r->s2, 4, 0, 4);
	test(*r->s2, 1, 5, *r->s2, 4, 1, -3);
	test(*r->s2, 1, 5, *r->s2, 4, 2, -3);
	test(*r->s2, 1, 5, *r->s2, 5, 0, 4);
	test(*r->s2, 1, 5, *r->s2, 5, 1, 4);
	test(*r->s2, 1, 5, *r->s2, 6, 0, 0);
	test(*r->s2, 1, 5, *r->s3, 0, 0, 4);
	test(*r->s2, 1, 5, *r->s3, 0, 1, 1);
	test(*r->s2, 1, 5, *r->s3, 0, 5, 1);
	test(*r->s2, 1, 5, *r->s3, 0, 9, 1);
	test(*r->s2, 1, 5, *r->s3, 0, 10, 1);
	test(*r->s2, 1, 5, *r->s3, 0, 11, 1);
	test(*r->s2, 1, 5, *r->s3, 1, 0, 4);
	test(*r->s2, 1, 5, *r->s3, 1, 1, 3);
	test(*r->s2, 1, 5, *r->s3, 1, 4, 0);
	test(*r->s2, 1, 5, *r->s3, 1, 8, -4);
	test(*r->s2, 1, 5, *r->s3, 1, 9, -5);
	test(*r->s2, 1, 5, *r->s3, 1, 10, -5);
	test(*r->s2, 1, 5, *r->s3, 5, 0, 4);
	test(*r->s2, 1, 5, *r->s3, 5, 1, -4);
	test(*r->s2, 1, 5, *r->s3, 5, 2, -4);
	test(*r->s2, 1, 5, *r->s3, 5, 4, -4);
	test(*r->s2, 1, 5, *r->s3, 5, 5, -4);
	test(*r->s2, 1, 5, *r->s3, 5, 6, -4);
	test(*r->s2, 1, 5, *r->s3, 9, 0, 4);
	test(*r->s2, 1, 5, *r->s3, 9, 1, -8);
	test(*r->s2, 1, 5, *r->s3, 9, 2, -8);
	test(*r->s2, 1, 5, *r->s3, 10, 0, 4);
	test(*r->s2, 1, 5, *r->s3, 10, 1, 4);
	test(*r->s2, 1, 5, *r->s3, 11, 0, 0);
	test(*r->s2, 1, 5, *r->s4, 0, 0, 4);
	test(*r->s2, 1, 5, *r->s4, 0, 1, 1);
	test(*r->s2, 1, 5, *r->s4, 0, 10, 1);
	test(*r->s2, 1, 5, *r->s4, 0, 19, 1);
	test(*r->s2, 1, 5, *r->s4, 0, 20, 1);
	test(*r->s2, 1, 5, *r->s4, 0, 21, 1);
	test(*r->s2, 1, 5, *r->s4, 1, 0, 4);
	test(*r->s2, 1, 5, *r->s4, 1, 1, 3);
	test(*r->s2, 1, 5, *r->s4, 1, 9, -5);
	test(*r->s2, 1, 5, *r->s4, 1, 18, -14);
	test(*r->s2, 1, 5, *r->s4, 1, 19, -15);
	test(*r->s2, 1, 5, *r->s4, 1, 20, -15);
	test(*r->s2, 1, 5, *r->s4, 10, 0, 4);
	test(*r->s2, 1, 5, *r->s4, 10, 1, -9);
}

template <class S>
void
test11(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 1, 5, *r->s4, 10, 5, -9);
	test(*r->s2, 1, 5, *r->s4, 10, 9, -9);
	test(*r->s2, 1, 5, *r->s4, 10, 10, -9);
	test(*r->s2, 1, 5, *r->s4, 10, 11, -9);
	test(*r->s2, 1, 5, *r->s4, 19, 0, 4);
	test(*r->s2, 1, 5, *r->s4, 19, 1, -18);
	test(*r->s2, 1, 5, *r->s4, 19, 2, -18);
	test(*r->s2, 1, 5, *r->s4, 20, 0, 4);
	test(*r->s2, 1, 5, *r->s4, 20, 1, 4);
	test(*r->s2, 1, 5, *r->s4, 21, 0, 0);
	test(*r->s2, 2, 0, *r->s1, 0, 0, 0);
	test(*r->s2, 2, 0, *r->s1, 0, 1, 0);
	test(*r->s2, 2, 0, *r->s1, 1, 0, 0);
	test(*r->s2, 2, 0, *r->s2, 0, 0, 0);
	test(*r->s2, 2, 0, *r->s2, 0, 1, -1);
	test(*r->s2, 2, 0, *r->s2, 0, 2, -2);
	test(*r->s2, 2, 0, *r->s2, 0, 4, -4);
	test(*r->s2, 2, 0, *r->s2, 0, 5, -5);
	test(*r->s2, 2, 0, *r->s2, 0, 6, -5);
	test(*r->s2, 2, 0, *r->s2, 1, 0, 0);
	test(*r->s2, 2, 0, *r->s2, 1, 1, -1);
	test(*r->s2, 2, 0, *r->s2, 1, 2, -2);
	test(*r->s2, 2, 0, *r->s2, 1, 3, -3);
	test(*r->s2, 2, 0, *r->s2, 1, 4, -4);
	test(*r->s2, 2, 0, *r->s2, 1, 5, -4);
	test(*r->s2, 2, 0, *r->s2, 2, 0, 0);
	test(*r->s2, 2, 0, *r->s2, 2, 1, -1);
	test(*r->s2, 2, 0, *r->s2, 2, 2, -2);
	test(*r->s2, 2, 0, *r->s2, 2, 3, -3);
	test(*r->s2, 2, 0, *r->s2, 2, 4, -3);
	test(*r->s2, 2, 0, *r->s2, 4, 0, 0);
	test(*r->s2, 2, 0, *r->s2, 4, 1, -1);
	test(*r->s2, 2, 0, *r->s2, 4, 2, -1);
	test(*r->s2, 2, 0, *r->s2, 5, 0, 0);
	test(*r->s2, 2, 0, *r->s2, 5, 1, 0);
	test(*r->s2, 2, 0, *r->s2, 6, 0, 0);
	test(*r->s2, 2, 0, *r->s3, 0, 0, 0);
	test(*r->s2, 2, 0, *r->s3, 0, 1, -1);
	test(*r->s2, 2, 0, *r->s3, 0, 5, -5);
	test(*r->s2, 2, 0, *r->s3, 0, 9, -9);
	test(*r->s2, 2, 0, *r->s3, 0, 10, -10);
	test(*r->s2, 2, 0, *r->s3, 0, 11, -10);
	test(*r->s2, 2, 0, *r->s3, 1, 0, 0);
	test(*r->s2, 2, 0, *r->s3, 1, 1, -1);
	test(*r->s2, 2, 0, *r->s3, 1, 4, -4);
	test(*r->s2, 2, 0, *r->s3, 1, 8, -8);
	test(*r->s2, 2, 0, *r->s3, 1, 9, -9);
	test(*r->s2, 2, 0, *r->s3, 1, 10, -9);
	test(*r->s2, 2, 0, *r->s3, 5, 0, 0);
	test(*r->s2, 2, 0, *r->s3, 5, 1, -1);
	test(*r->s2, 2, 0, *r->s3, 5, 2, -2);
	test(*r->s2, 2, 0, *r->s3, 5, 4, -4);
	test(*r->s2, 2, 0, *r->s3, 5, 5, -5);
	test(*r->s2, 2, 0, *r->s3, 5, 6, -5);
	test(*r->s2, 2, 0, *r->s3, 9, 0, 0);
	test(*r->s2, 2, 0, *r->s3, 9, 1, -1);
	test(*r->s2, 2, 0, *r->s3, 9, 2, -1);
	test(*r->s2, 2, 0, *r->s3, 10, 0, 0);
	test(*r->s2, 2, 0, *r->s3, 10, 1, 0);
	test(*r->s2, 2, 0, *r->s3, 11, 0, 0);
	test(*r->s2, 2, 0, *r->s4, 0, 0, 0);
	test(*r->s2, 2, 0, *r->s4, 0, 1, -1);
	test(*r->s2, 2, 0, *r->s4, 0, 10, -10);
	test(*r->s2, 2, 0, *r->s4, 0, 19, -19);
	test(*r->s2, 2, 0, *r->s4, 0, 20, -20);
	test(*r->s2, 2, 0, *r->s4, 0, 21, -20);
	test(*r->s2, 2, 0, *r->s4, 1, 0, 0);
	test(*r->s2, 2, 0, *r->s4, 1, 1, -1);
	test(*r->s2, 2, 0, *r->s4, 1, 9, -9);
	test(*r->s2, 2, 0, *r->s4, 1, 18, -18);
	test(*r->s2, 2, 0, *r->s4, 1, 19, -19);
	test(*r->s2, 2, 0, *r->s4, 1, 20, -19);
	test(*r->s2, 2, 0, *r->s4, 10, 0, 0);
	test(*r->s2, 2, 0, *r->s4, 10, 1, -1);
	test(*r->s2, 2, 0, *r->s4, 10, 5, -5);
	test(*r->s2, 2, 0, *r->s4, 10, 9, -9);
	test(*r->s2, 2, 0, *r->s4, 10, 10, -10);
	test(*r->s2, 2, 0, *r->s4, 10, 11, -10);
	test(*r->s2, 2, 0, *r->s4, 19, 0, 0);
	test(*r->s2, 2, 0, *r->s4, 19, 1, -1);
	test(*r->s2, 2, 0, *r->s4, 19, 2, -1);
	test(*r->s2, 2, 0, *r->s4, 20, 0, 0);
	test(*r->s2, 2, 0, *r->s4, 20, 1, 0);
	test(*r->s2, 2, 0, *r->s4, 21, 0, 0);
	test(*r->s2, 2, 1, *r->s1, 0, 0, 1);
	test(*r->s2, 2, 1, *r->s1, 0, 1, 1);
	test(*r->s2, 2, 1, *r->s1, 1, 0, 0);
	test(*r->s2, 2, 1, *r->s2, 0, 0, 1);
	test(*r->s2, 2, 1, *r->s2, 0, 1, 2);
	test(*r->s2, 2, 1, *r->s2, 0, 2, 2);
	test(*r->s2, 2, 1, *r->s2, 0, 4, 2);
	test(*r->s2, 2, 1, *r->s2, 0, 5, 2);
	test(*r->s2, 2, 1, *r->s2, 0, 6, 2);
	test(*r->s2, 2, 1, *r->s2, 1, 0, 1);
	test(*r->s2, 2, 1, *r->s2, 1, 1, 1);
	test(*r->s2, 2, 1, *r->s2, 1, 2, 1);
	test(*r->s2, 2, 1, *r->s2, 1, 3, 1);
	test(*r->s2, 2, 1, *r->s2, 1, 4, 1);
	test(*r->s2, 2, 1, *r->s2, 1, 5, 1);
	test(*r->s2, 2, 1, *r->s2, 2, 0, 1);
}

template <class S>
void
test12(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 2, 1, *r->s2, 2, 1, 0);
	test(*r->s2, 2, 1, *r->s2, 2, 2, -1);
	test(*r->s2, 2, 1, *r->s2, 2, 3, -2);
	test(*r->s2, 2, 1, *r->s2, 2, 4, -2);
	test(*r->s2, 2, 1, *r->s2, 4, 0, 1);
	test(*r->s2, 2, 1, *r->s2, 4, 1, -2);
	test(*r->s2, 2, 1, *r->s2, 4, 2, -2);
	test(*r->s2, 2, 1, *r->s2, 5, 0, 1);
	test(*r->s2, 2, 1, *r->s2, 5, 1, 1);
	test(*r->s2, 2, 1, *r->s2, 6, 0, 0);
	test(*r->s2, 2, 1, *r->s3, 0, 0, 1);
	test(*r->s2, 2, 1, *r->s3, 0, 1, 2);
	test(*r->s2, 2, 1, *r->s3, 0, 5, 2);
	test(*r->s2, 2, 1, *r->s3, 0, 9, 2);
	test(*r->s2, 2, 1, *r->s3, 0, 10, 2);
	test(*r->s2, 2, 1, *r->s3, 0, 11, 2);
	test(*r->s2, 2, 1, *r->s3, 1, 0, 1);
	test(*r->s2, 2, 1, *r->s3, 1, 1, 1);
	test(*r->s2, 2, 1, *r->s3, 1, 4, 1);
	test(*r->s2, 2, 1, *r->s3, 1, 8, 1);
	test(*r->s2, 2, 1, *r->s3, 1, 9, 1);
	test(*r->s2, 2, 1, *r->s3, 1, 10, 1);
	test(*r->s2, 2, 1, *r->s3, 5, 0, 1);
	test(*r->s2, 2, 1, *r->s3, 5, 1, -3);
	test(*r->s2, 2, 1, *r->s3, 5, 2, -3);
	test(*r->s2, 2, 1, *r->s3, 5, 4, -3);
	test(*r->s2, 2, 1, *r->s3, 5, 5, -3);
	test(*r->s2, 2, 1, *r->s3, 5, 6, -3);
	test(*r->s2, 2, 1, *r->s3, 9, 0, 1);
	test(*r->s2, 2, 1, *r->s3, 9, 1, -7);
	test(*r->s2, 2, 1, *r->s3, 9, 2, -7);
	test(*r->s2, 2, 1, *r->s3, 10, 0, 1);
	test(*r->s2, 2, 1, *r->s3, 10, 1, 1);
	test(*r->s2, 2, 1, *r->s3, 11, 0, 0);
	test(*r->s2, 2, 1, *r->s4, 0, 0, 1);
	test(*r->s2, 2, 1, *r->s4, 0, 1, 2);
	test(*r->s2, 2, 1, *r->s4, 0, 10, 2);
	test(*r->s2, 2, 1, *r->s4, 0, 19, 2);
	test(*r->s2, 2, 1, *r->s4, 0, 20, 2);
	test(*r->s2, 2, 1, *r->s4, 0, 21, 2);
	test(*r->s2, 2, 1, *r->s4, 1, 0, 1);
	test(*r->s2, 2, 1, *r->s4, 1, 1, 1);
	test(*r->s2, 2, 1, *r->s4, 1, 9, 1);
	test(*r->s2, 2, 1, *r->s4, 1, 18, 1);
	test(*r->s2, 2, 1, *r->s4, 1, 19, 1);
	test(*r->s2, 2, 1, *r->s4, 1, 20, 1);
	test(*r->s2, 2, 1, *r->s4, 10, 0, 1);
	test(*r->s2, 2, 1, *r->s4, 10, 1, -8);
	test(*r->s2, 2, 1, *r->s4, 10, 5, -8);
	test(*r->s2, 2, 1, *r->s4, 10, 9, -8);
	test(*r->s2, 2, 1, *r->s4, 10, 10, -8);
	test(*r->s2, 2, 1, *r->s4, 10, 11, -8);
	test(*r->s2, 2, 1, *r->s4, 19, 0, 1);
	test(*r->s2, 2, 1, *r->s4, 19, 1, -17);
	test(*r->s2, 2, 1, *r->s4, 19, 2, -17);
	test(*r->s2, 2, 1, *r->s4, 20, 0, 1);
	test(*r->s2, 2, 1, *r->s4, 20, 1, 1);
	test(*r->s2, 2, 1, *r->s4, 21, 0, 0);
	test(*r->s2, 2, 2, *r->s1, 0, 0, 2);
	test(*r->s2, 2, 2, *r->s1, 0, 1, 2);
	test(*r->s2, 2, 2, *r->s1, 1, 0, 0);
	test(*r->s2, 2, 2, *r->s2, 0, 0, 2);
	test(*r->s2, 2, 2, *r->s2, 0, 1, 2);
	test(*r->s2, 2, 2, *r->s2, 0, 2, 2);
	test(*r->s2, 2, 2, *r->s2, 0, 4, 2);
	test(*r->s2, 2, 2, *r->s2, 0, 5, 2);
	test(*r->s2, 2, 2, *r->s2, 0, 6, 2);
	test(*r->s2, 2, 2, *r->s2, 1, 0, 2);
	test(*r->s2, 2, 2, *r->s2, 1, 1, 1);
	test(*r->s2, 2, 2, *r->s2, 1, 2, 1);
	test(*r->s2, 2, 2, *r->s2, 1, 3, 1);
	test(*r->s2, 2, 2, *r->s2, 1, 4, 1);
	test(*r->s2, 2, 2, *r->s2, 1, 5, 1);
	test(*r->s2, 2, 2, *r->s2, 2, 0, 2);
	test(*r->s2, 2, 2, *r->s2, 2, 1, 1);
	test(*r->s2, 2, 2, *r->s2, 2, 2, 0);
	test(*r->s2, 2, 2, *r->s2, 2, 3, -1);
	test(*r->s2, 2, 2, *r->s2, 2, 4, -1);
	test(*r->s2, 2, 2, *r->s2, 4, 0, 2);
	test(*r->s2, 2, 2, *r->s2, 4, 1, -2);
	test(*r->s2, 2, 2, *r->s2, 4, 2, -2);
	test(*r->s2, 2, 2, *r->s2, 5, 0, 2);
	test(*r->s2, 2, 2, *r->s2, 5, 1, 2);
	test(*r->s2, 2, 2, *r->s2, 6, 0, 0);
	test(*r->s2, 2, 2, *r->s3, 0, 0, 2);
	test(*r->s2, 2, 2, *r->s3, 0, 1, 2);
	test(*r->s2, 2, 2, *r->s3, 0, 5, 2);
	test(*r->s2, 2, 2, *r->s3, 0, 9, 2);
	test(*r->s2, 2, 2, *r->s3, 0, 10, 2);
	test(*r->s2, 2, 2, *r->s3, 0, 11, 2);
	test(*r->s2, 2, 2, *r->s3, 1, 0, 2);
	test(*r->s2, 2, 2, *r->s3, 1, 1, 1);
	test(*r->s2, 2, 2, *r->s3, 1, 4, 1);
	test(*r->s2, 2, 2, *r->s3, 1, 8, 1);
	test(*r->s2, 2, 2, *r->s3, 1, 9, 1);
	test(*r->s2, 2, 2, *r->s3, 1, 10, 1);
	test(*r->s2, 2, 2, *r->s3, 5, 0, 2);
	test(*r->s2, 2, 2, *r->s3, 5, 1, -3);
	test(*r->s2, 2, 2, *r->s3, 5, 2, -3);
	test(*r->s2, 2, 2, *r->s3, 5, 4, -3);
}

template <class S>
void
test13(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 2, 2, *r->s3, 5, 5, -3);
	test(*r->s2, 2, 2, *r->s3, 5, 6, -3);
	test(*r->s2, 2, 2, *r->s3, 9, 0, 2);
	test(*r->s2, 2, 2, *r->s3, 9, 1, -7);
	test(*r->s2, 2, 2, *r->s3, 9, 2, -7);
	test(*r->s2, 2, 2, *r->s3, 10, 0, 2);
	test(*r->s2, 2, 2, *r->s3, 10, 1, 2);
	test(*r->s2, 2, 2, *r->s3, 11, 0, 0);
	test(*r->s2, 2, 2, *r->s4, 0, 0, 2);
	test(*r->s2, 2, 2, *r->s4, 0, 1, 2);
	test(*r->s2, 2, 2, *r->s4, 0, 10, 2);
	test(*r->s2, 2, 2, *r->s4, 0, 19, 2);
	test(*r->s2, 2, 2, *r->s4, 0, 20, 2);
	test(*r->s2, 2, 2, *r->s4, 0, 21, 2);
	test(*r->s2, 2, 2, *r->s4, 1, 0, 2);
	test(*r->s2, 2, 2, *r->s4, 1, 1, 1);
	test(*r->s2, 2, 2, *r->s4, 1, 9, 1);
	test(*r->s2, 2, 2, *r->s4, 1, 18, 1);
	test(*r->s2, 2, 2, *r->s4, 1, 19, 1);
	test(*r->s2, 2, 2, *r->s4, 1, 20, 1);
	test(*r->s2, 2, 2, *r->s4, 10, 0, 2);
	test(*r->s2, 2, 2, *r->s4, 10, 1, -8);
	test(*r->s2, 2, 2, *r->s4, 10, 5, -8);
	test(*r->s2, 2, 2, *r->s4, 10, 9, -8);
	test(*r->s2, 2, 2, *r->s4, 10, 10, -8);
	test(*r->s2, 2, 2, *r->s4, 10, 11, -8);
	test(*r->s2, 2, 2, *r->s4, 19, 0, 2);
	test(*r->s2, 2, 2, *r->s4, 19, 1, -17);
	test(*r->s2, 2, 2, *r->s4, 19, 2, -17);
	test(*r->s2, 2, 2, *r->s4, 20, 0, 2);
	test(*r->s2, 2, 2, *r->s4, 20, 1, 2);
	test(*r->s2, 2, 2, *r->s4, 21, 0, 0);
	test(*r->s2, 2, 3, *r->s1, 0, 0, 3);
	test(*r->s2, 2, 3, *r->s1, 0, 1, 3);
	test(*r->s2, 2, 3, *r->s1, 1, 0, 0);
	test(*r->s2, 2, 3, *r->s2, 0, 0, 3);
	test(*r->s2, 2, 3, *r->s2, 0, 1, 2);
	test(*r->s2, 2, 3, *r->s2, 0, 2, 2);
	test(*r->s2, 2, 3, *r->s2, 0, 4, 2);
	test(*r->s2, 2, 3, *r->s2, 0, 5, 2);
	test(*r->s2, 2, 3, *r->s2, 0, 6, 2);
	test(*r->s2, 2, 3, *r->s2, 1, 0, 3);
	test(*r->s2, 2, 3, *r->s2, 1, 1, 1);
	test(*r->s2, 2, 3, *r->s2, 1, 2, 1);
	test(*r->s2, 2, 3, *r->s2, 1, 3, 1);
	test(*r->s2, 2, 3, *r->s2, 1, 4, 1);
	test(*r->s2, 2, 3, *r->s2, 1, 5, 1);
	test(*r->s2, 2, 3, *r->s2, 2, 0, 3);
	test(*r->s2, 2, 3, *r->s2, 2, 1, 2);
	test(*r->s2, 2, 3, *r->s2, 2, 2, 1);
	test(*r->s2, 2, 3, *r->s2, 2, 3, 0);
	test(*r->s2, 2, 3, *r->s2, 2, 4, 0);
	test(*r->s2, 2, 3, *r->s2, 4, 0, 3);
	test(*r->s2, 2, 3, *r->s2, 4, 1, -2);
	test(*r->s2, 2, 3, *r->s2, 4, 2, -2);
	test(*r->s2, 2, 3, *r->s2, 5, 0, 3);
	test(*r->s2, 2, 3, *r->s2, 5, 1, 3);
	test(*r->s2, 2, 3, *r->s2, 6, 0, 0);
	test(*r->s2, 2, 3, *r->s3, 0, 0, 3);
	test(*r->s2, 2, 3, *r->s3, 0, 1, 2);
	test(*r->s2, 2, 3, *r->s3, 0, 5, 2);
	test(*r->s2, 2, 3, *r->s3, 0, 9, 2);
	test(*r->s2, 2, 3, *r->s3, 0, 10, 2);
	test(*r->s2, 2, 3, *r->s3, 0, 11, 2);
	test(*r->s2, 2, 3, *r->s3, 1, 0, 3);
	test(*r->s2, 2, 3, *r->s3, 1, 1, 1);
	test(*r->s2, 2, 3, *r->s3, 1, 4, 1);
	test(*r->s2, 2, 3, *r->s3, 1, 8, 1);
	test(*r->s2, 2, 3, *r->s3, 1, 9, 1);
	test(*r->s2, 2, 3, *r->s3, 1, 10, 1);
	test(*r->s2, 2, 3, *r->s3, 5, 0, 3);
	test(*r->s2, 2, 3, *r->s3, 5, 1, -3);
	test(*r->s2, 2, 3, *r->s3, 5, 2, -3);
	test(*r->s2, 2, 3, *r->s3, 5, 4, -3);
	test(*r->s2, 2, 3, *r->s3, 5, 5, -3);
	test(*r->s2, 2, 3, *r->s3, 5, 6, -3);
	test(*r->s2, 2, 3, *r->s3, 9, 0, 3);
	test(*r->s2, 2, 3, *r->s3, 9, 1, -7);
	test(*r->s2, 2, 3, *r->s3, 9, 2, -7);
	test(*r->s2, 2, 3, *r->s3, 10, 0, 3);
	test(*r->s2, 2, 3, *r->s3, 10, 1, 3);
	test(*r->s2, 2, 3, *r->s3, 11, 0, 0);
	test(*r->s2, 2, 3, *r->s4, 0, 0, 3);
	test(*r->s2, 2, 3, *r->s4, 0, 1, 2);
	test(*r->s2, 2, 3, *r->s4, 0, 10, 2);
	test(*r->s2, 2, 3, *r->s4, 0, 19, 2);
	test(*r->s2, 2, 3, *r->s4, 0, 20, 2);
	test(*r->s2, 2, 3, *r->s4, 0, 21, 2);
	test(*r->s2, 2, 3, *r->s4, 1, 0, 3);
	test(*r->s2, 2, 3, *r->s4, 1, 1, 1);
	test(*r->s2, 2, 3, *r->s4, 1, 9, 1);
	test(*r->s2, 2, 3, *r->s4, 1, 18, 1);
	test(*r->s2, 2, 3, *r->s4, 1, 19, 1);
	test(*r->s2, 2, 3, *r->s4, 1, 20, 1);
	test(*r->s2, 2, 3, *r->s4, 10, 0, 3);
	test(*r->s2, 2, 3, *r->s4, 10, 1, -8);
	test(*r->s2, 2, 3, *r->s4, 10, 5, -8);
	test(*r->s2, 2, 3, *r->s4, 10, 9, -8);
	test(*r->s2, 2, 3, *r->s4, 10, 10, -8);
	test(*r->s2, 2, 3, *r->s4, 10, 11, -8);
}

template <class S>
void
test14(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 2, 3, *r->s4, 19, 0, 3);
	test(*r->s2, 2, 3, *r->s4, 19, 1, -17);
	test(*r->s2, 2, 3, *r->s4, 19, 2, -17);
	test(*r->s2, 2, 3, *r->s4, 20, 0, 3);
	test(*r->s2, 2, 3, *r->s4, 20, 1, 3);
	test(*r->s2, 2, 3, *r->s4, 21, 0, 0);
	test(*r->s2, 2, 4, *r->s1, 0, 0, 3);
	test(*r->s2, 2, 4, *r->s1, 0, 1, 3);
	test(*r->s2, 2, 4, *r->s1, 1, 0, 0);
	test(*r->s2, 2, 4, *r->s2, 0, 0, 3);
	test(*r->s2, 2, 4, *r->s2, 0, 1, 2);
	test(*r->s2, 2, 4, *r->s2, 0, 2, 2);
	test(*r->s2, 2, 4, *r->s2, 0, 4, 2);
	test(*r->s2, 2, 4, *r->s2, 0, 5, 2);
	test(*r->s2, 2, 4, *r->s2, 0, 6, 2);
	test(*r->s2, 2, 4, *r->s2, 1, 0, 3);
	test(*r->s2, 2, 4, *r->s2, 1, 1, 1);
	test(*r->s2, 2, 4, *r->s2, 1, 2, 1);
	test(*r->s2, 2, 4, *r->s2, 1, 3, 1);
	test(*r->s2, 2, 4, *r->s2, 1, 4, 1);
	test(*r->s2, 2, 4, *r->s2, 1, 5, 1);
	test(*r->s2, 2, 4, *r->s2, 2, 0, 3);
	test(*r->s2, 2, 4, *r->s2, 2, 1, 2);
	test(*r->s2, 2, 4, *r->s2, 2, 2, 1);
	test(*r->s2, 2, 4, *r->s2, 2, 3, 0);
	test(*r->s2, 2, 4, *r->s2, 2, 4, 0);
	test(*r->s2, 2, 4, *r->s2, 4, 0, 3);
	test(*r->s2, 2, 4, *r->s2, 4, 1, -2);
	test(*r->s2, 2, 4, *r->s2, 4, 2, -2);
	test(*r->s2, 2, 4, *r->s2, 5, 0, 3);
	test(*r->s2, 2, 4, *r->s2, 5, 1, 3);
	test(*r->s2, 2, 4, *r->s2, 6, 0, 0);
	test(*r->s2, 2, 4, *r->s3, 0, 0, 3);
	test(*r->s2, 2, 4, *r->s3, 0, 1, 2);
	test(*r->s2, 2, 4, *r->s3, 0, 5, 2);
	test(*r->s2, 2, 4, *r->s3, 0, 9, 2);
	test(*r->s2, 2, 4, *r->s3, 0, 10, 2);
	test(*r->s2, 2, 4, *r->s3, 0, 11, 2);
	test(*r->s2, 2, 4, *r->s3, 1, 0, 3);
	test(*r->s2, 2, 4, *r->s3, 1, 1, 1);
	test(*r->s2, 2, 4, *r->s3, 1, 4, 1);
	test(*r->s2, 2, 4, *r->s3, 1, 8, 1);
	test(*r->s2, 2, 4, *r->s3, 1, 9, 1);
	test(*r->s2, 2, 4, *r->s3, 1, 10, 1);
	test(*r->s2, 2, 4, *r->s3, 5, 0, 3);
	test(*r->s2, 2, 4, *r->s3, 5, 1, -3);
	test(*r->s2, 2, 4, *r->s3, 5, 2, -3);
	test(*r->s2, 2, 4, *r->s3, 5, 4, -3);
	test(*r->s2, 2, 4, *r->s3, 5, 5, -3);
	test(*r->s2, 2, 4, *r->s3, 5, 6, -3);
	test(*r->s2, 2, 4, *r->s3, 9, 0, 3);
	test(*r->s2, 2, 4, *r->s3, 9, 1, -7);
	test(*r->s2, 2, 4, *r->s3, 9, 2, -7);
	test(*r->s2, 2, 4, *r->s3, 10, 0, 3);
	test(*r->s2, 2, 4, *r->s3, 10, 1, 3);
	test(*r->s2, 2, 4, *r->s3, 11, 0, 0);
	test(*r->s2, 2, 4, *r->s4, 0, 0, 3);
	test(*r->s2, 2, 4, *r->s4, 0, 1, 2);
	test(*r->s2, 2, 4, *r->s4, 0, 10, 2);
	test(*r->s2, 2, 4, *r->s4, 0, 19, 2);
	test(*r->s2, 2, 4, *r->s4, 0, 20, 2);
	test(*r->s2, 2, 4, *r->s4, 0, 21, 2);
	test(*r->s2, 2, 4, *r->s4, 1, 0, 3);
	test(*r->s2, 2, 4, *r->s4, 1, 1, 1);
	test(*r->s2, 2, 4, *r->s4, 1, 9, 1);
	test(*r->s2, 2, 4, *r->s4, 1, 18, 1);
	test(*r->s2, 2, 4, *r->s4, 1, 19, 1);
	test(*r->s2, 2, 4, *r->s4, 1, 20, 1);
	test(*r->s2, 2, 4, *r->s4, 10, 0, 3);
	test(*r->s2, 2, 4, *r->s4, 10, 1, -8);
	test(*r->s2, 2, 4, *r->s4, 10, 5, -8);
	test(*r->s2, 2, 4, *r->s4, 10, 9, -8);
	test(*r->s2, 2, 4, *r->s4, 10, 10, -8);
	test(*r->s2, 2, 4, *r->s4, 10, 11, -8);
	test(*r->s2, 2, 4, *r->s4, 19, 0, 3);
	test(*r->s2, 2, 4, *r->s4, 19, 1, -17);
	test(*r->s2, 2, 4, *r->s4, 19, 2, -17);
	test(*r->s2, 2, 4, *r->s4, 20, 0, 3);
	test(*r->s2, 2, 4, *r->s4, 20, 1, 3);
	test(*r->s2, 2, 4, *r->s4, 21, 0, 0);
	test(*r->s2, 4, 0, *r->s1, 0, 0, 0);
	test(*r->s2, 4, 0, *r->s1, 0, 1, 0);
	test(*r->s2, 4, 0, *r->s1, 1, 0, 0);
	test(*r->s2, 4, 0, *r->s2, 0, 0, 0);
	test(*r->s2, 4, 0, *r->s2, 0, 1, -1);
	test(*r->s2, 4, 0, *r->s2, 0, 2, -2);
	test(*r->s2, 4, 0, *r->s2, 0, 4, -4);
	test(*r->s2, 4, 0, *r->s2, 0, 5, -5);
	test(*r->s2, 4, 0, *r->s2, 0, 6, -5);
	test(*r->s2, 4, 0, *r->s2, 1, 0, 0);
	test(*r->s2, 4, 0, *r->s2, 1, 1, -1);
	test(*r->s2, 4, 0, *r->s2, 1, 2, -2);
	test(*r->s2, 4, 0, *r->s2, 1, 3, -3);
	test(*r->s2, 4, 0, *r->s2, 1, 4, -4);
	test(*r->s2, 4, 0, *r->s2, 1, 5, -4);
	test(*r->s2, 4, 0, *r->s2, 2, 0, 0);
	test(*r->s2, 4, 0, *r->s2, 2, 1, -1);
	test(*r->s2, 4, 0, *r->s2, 2, 2, -2);
	test(*r->s2, 4, 0, *r->s2, 2, 3, -3);
	test(*r->s2, 4, 0, *r->s2, 2, 4, -3);
}

template <class S>
void
test15(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 4, 0, *r->s2, 4, 0, 0);
	test(*r->s2, 4, 0, *r->s2, 4, 1, -1);
	test(*r->s2, 4, 0, *r->s2, 4, 2, -1);
	test(*r->s2, 4, 0, *r->s2, 5, 0, 0);
	test(*r->s2, 4, 0, *r->s2, 5, 1, 0);
	test(*r->s2, 4, 0, *r->s2, 6, 0, 0);
	test(*r->s2, 4, 0, *r->s3, 0, 0, 0);
	test(*r->s2, 4, 0, *r->s3, 0, 1, -1);
	test(*r->s2, 4, 0, *r->s3, 0, 5, -5);
	test(*r->s2, 4, 0, *r->s3, 0, 9, -9);
	test(*r->s2, 4, 0, *r->s3, 0, 10, -10);
	test(*r->s2, 4, 0, *r->s3, 0, 11, -10);
	test(*r->s2, 4, 0, *r->s3, 1, 0, 0);
	test(*r->s2, 4, 0, *r->s3, 1, 1, -1);
	test(*r->s2, 4, 0, *r->s3, 1, 4, -4);
	test(*r->s2, 4, 0, *r->s3, 1, 8, -8);
	test(*r->s2, 4, 0, *r->s3, 1, 9, -9);
	test(*r->s2, 4, 0, *r->s3, 1, 10, -9);
	test(*r->s2, 4, 0, *r->s3, 5, 0, 0);
	test(*r->s2, 4, 0, *r->s3, 5, 1, -1);
	test(*r->s2, 4, 0, *r->s3, 5, 2, -2);
	test(*r->s2, 4, 0, *r->s3, 5, 4, -4);
	test(*r->s2, 4, 0, *r->s3, 5, 5, -5);
	test(*r->s2, 4, 0, *r->s3, 5, 6, -5);
	test(*r->s2, 4, 0, *r->s3, 9, 0, 0);
	test(*r->s2, 4, 0, *r->s3, 9, 1, -1);
	test(*r->s2, 4, 0, *r->s3, 9, 2, -1);
	test(*r->s2, 4, 0, *r->s3, 10, 0, 0);
	test(*r->s2, 4, 0, *r->s3, 10, 1, 0);
	test(*r->s2, 4, 0, *r->s3, 11, 0, 0);
	test(*r->s2, 4, 0, *r->s4, 0, 0, 0);
	test(*r->s2, 4, 0, *r->s4, 0, 1, -1);
	test(*r->s2, 4, 0, *r->s4, 0, 10, -10);
	test(*r->s2, 4, 0, *r->s4, 0, 19, -19);
	test(*r->s2, 4, 0, *r->s4, 0, 20, -20);
	test(*r->s2, 4, 0, *r->s4, 0, 21, -20);
	test(*r->s2, 4, 0, *r->s4, 1, 0, 0);
	test(*r->s2, 4, 0, *r->s4, 1, 1, -1);
	test(*r->s2, 4, 0, *r->s4, 1, 9, -9);
	test(*r->s2, 4, 0, *r->s4, 1, 18, -18);
	test(*r->s2, 4, 0, *r->s4, 1, 19, -19);
	test(*r->s2, 4, 0, *r->s4, 1, 20, -19);
	test(*r->s2, 4, 0, *r->s4, 10, 0, 0);
	test(*r->s2, 4, 0, *r->s4, 10, 1, -1);
	test(*r->s2, 4, 0, *r->s4, 10, 5, -5);
	test(*r->s2, 4, 0, *r->s4, 10, 9, -9);
	test(*r->s2, 4, 0, *r->s4, 10, 10, -10);
	test(*r->s2, 4, 0, *r->s4, 10, 11, -10);
	test(*r->s2, 4, 0, *r->s4, 19, 0, 0);
	test(*r->s2, 4, 0, *r->s4, 19, 1, -1);
	test(*r->s2, 4, 0, *r->s4, 19, 2, -1);
	test(*r->s2, 4, 0, *r->s4, 20, 0, 0);
	test(*r->s2, 4, 0, *r->s4, 20, 1, 0);
	test(*r->s2, 4, 0, *r->s4, 21, 0, 0);
	test(*r->s2, 4, 1, *r->s1, 0, 0, 1);
	test(*r->s2, 4, 1, *r->s1, 0, 1, 1);
	test(*r->s2, 4, 1, *r->s1, 1, 0, 0);
	test(*r->s2, 4, 1, *r->s2, 0, 0, 1);
	test(*r->s2, 4, 1, *r->s2, 0, 1, 4);
	test(*r->s2, 4, 1, *r->s2, 0, 2, 4);
	test(*r->s2, 4, 1, *r->s2, 0, 4, 4);
	test(*r->s2, 4, 1, *r->s2, 0, 5, 4);
	test(*r->s2, 4, 1, *r->s2, 0, 6, 4);
	test(*r->s2, 4, 1, *r->s2, 1, 0, 1);
	test(*r->s2, 4, 1, *r->s2, 1, 1, 3);
	test(*r->s2, 4, 1, *r->s2, 1, 2, 3);
	test(*r->s2, 4, 1, *r->s2, 1, 3, 3);
	test(*r->s2, 4, 1, *r->s2, 1, 4, 3);
	test(*r->s2, 4, 1, *r->s2, 1, 5, 3);
	test(*r->s2, 4, 1, *r->s2, 2, 0, 1);
	test(*r->s2, 4, 1, *r->s2, 2, 1, 2);
	test(*r->s2, 4, 1, *r->s2, 2, 2, 2);
	test(*r->s2, 4, 1, *r->s2, 2, 3, 2);
	test(*r->s2, 4, 1, *r->s2, 2, 4, 2);
	test(*r->s2, 4, 1, *r->s2, 4, 0, 1);
	test(*r->s2, 4, 1, *r->s2, 4, 1, 0);
	test(*r->s2, 4, 1, *r->s2, 4, 2, 0);
	test(*r->s2, 4, 1, *r->s2, 5, 0, 1);
	test(*r->s2, 4, 1, *r->s2, 5, 1, 1);
	test(*r->s2, 4, 1, *r->s2, 6, 0, 0);
	test(*r->s2, 4, 1, *r->s3, 0, 0, 1);
	test(*r->s2, 4, 1, *r->s3, 0, 1, 4);
	test(*r->s2, 4, 1, *r->s3, 0, 5, 4);
	test(*r->s2, 4, 1, *r->s3, 0, 9, 4);
	test(*r->s2, 4, 1, *r->s3, 0, 10, 4);
	test(*r->s2, 4, 1, *r->s3, 0, 11, 4);
	test(*r->s2, 4, 1, *r->s3, 1, 0, 1);
	test(*r->s2, 4, 1, *r->s3, 1, 1, 3);
	test(*r->s2, 4, 1, *r->s3, 1, 4, 3);
	test(*r->s2, 4, 1, *r->s3, 1, 8, 3);
	test(*r->s2, 4, 1, *r->s3, 1, 9, 3);
	test(*r->s2, 4, 1, *r->s3, 1, 10, 3);
	test(*r->s2, 4, 1, *r->s3, 5, 0, 1);
	test(*r->s2, 4, 1, *r->s3, 5, 1, -1);
	test(*r->s2, 4, 1, *r->s3, 5, 2, -1);
	test(*r->s2, 4, 1, *r->s3, 5, 4, -1);
	test(*r->s2, 4, 1, *r->s3, 5, 5, -1);
	test(*r->s2, 4, 1, *r->s3, 5, 6, -1);
	test(*r->s2, 4, 1, *r->s3, 9, 0, 1);
	test(*r->s2, 4, 1, *r->s3, 9, 1, -5);
}

template <class S>
void
test16(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 4, 1, *r->s3, 9, 2, -5);
	test(*r->s2, 4, 1, *r->s3, 10, 0, 1);
	test(*r->s2, 4, 1, *r->s3, 10, 1, 1);
	test(*r->s2, 4, 1, *r->s3, 11, 0, 0);
	test(*r->s2, 4, 1, *r->s4, 0, 0, 1);
	test(*r->s2, 4, 1, *r->s4, 0, 1, 4);
	test(*r->s2, 4, 1, *r->s4, 0, 10, 4);
	test(*r->s2, 4, 1, *r->s4, 0, 19, 4);
	test(*r->s2, 4, 1, *r->s4, 0, 20, 4);
	test(*r->s2, 4, 1, *r->s4, 0, 21, 4);
	test(*r->s2, 4, 1, *r->s4, 1, 0, 1);
	test(*r->s2, 4, 1, *r->s4, 1, 1, 3);
	test(*r->s2, 4, 1, *r->s4, 1, 9, 3);
	test(*r->s2, 4, 1, *r->s4, 1, 18, 3);
	test(*r->s2, 4, 1, *r->s4, 1, 19, 3);
	test(*r->s2, 4, 1, *r->s4, 1, 20, 3);
	test(*r->s2, 4, 1, *r->s4, 10, 0, 1);
	test(*r->s2, 4, 1, *r->s4, 10, 1, -6);
	test(*r->s2, 4, 1, *r->s4, 10, 5, -6);
	test(*r->s2, 4, 1, *r->s4, 10, 9, -6);
	test(*r->s2, 4, 1, *r->s4, 10, 10, -6);
	test(*r->s2, 4, 1, *r->s4, 10, 11, -6);
	test(*r->s2, 4, 1, *r->s4, 19, 0, 1);
	test(*r->s2, 4, 1, *r->s4, 19, 1, -15);
	test(*r->s2, 4, 1, *r->s4, 19, 2, -15);
	test(*r->s2, 4, 1, *r->s4, 20, 0, 1);
	test(*r->s2, 4, 1, *r->s4, 20, 1, 1);
	test(*r->s2, 4, 1, *r->s4, 21, 0, 0);
	test(*r->s2, 4, 2, *r->s1, 0, 0, 1);
	test(*r->s2, 4, 2, *r->s1, 0, 1, 1);
	test(*r->s2, 4, 2, *r->s1, 1, 0, 0);
	test(*r->s2, 4, 2, *r->s2, 0, 0, 1);
	test(*r->s2, 4, 2, *r->s2, 0, 1, 4);
	test(*r->s2, 4, 2, *r->s2, 0, 2, 4);
	test(*r->s2, 4, 2, *r->s2, 0, 4, 4);
	test(*r->s2, 4, 2, *r->s2, 0, 5, 4);
	test(*r->s2, 4, 2, *r->s2, 0, 6, 4);
	test(*r->s2, 4, 2, *r->s2, 1, 0, 1);
	test(*r->s2, 4, 2, *r->s2, 1, 1, 3);
	test(*r->s2, 4, 2, *r->s2, 1, 2, 3);
	test(*r->s2, 4, 2, *r->s2, 1, 3, 3);
	test(*r->s2, 4, 2, *r->s2, 1, 4, 3);
	test(*r->s2, 4, 2, *r->s2, 1, 5, 3);
	test(*r->s2, 4, 2, *r->s2, 2, 0, 1);
	test(*r->s2, 4, 2, *r->s2, 2, 1, 2);
	test(*r->s2, 4, 2, *r->s2, 2, 2, 2);
	test(*r->s2, 4, 2, *r->s2, 2, 3, 2);
	test(*r->s2, 4, 2, *r->s2, 2, 4, 2);
	test(*r->s2, 4, 2, *r->s2, 4, 0, 1);
	test(*r->s2, 4, 2, *r->s2, 4, 1, 0);
	test(*r->s2, 4, 2, *r->s2, 4, 2, 0);
	test(*r->s2, 4, 2, *r->s2, 5, 0, 1);
	test(*r->s2, 4, 2, *r->s2, 5, 1, 1);
	test(*r->s2, 4, 2, *r->s2, 6, 0, 0);
	test(*r->s2, 4, 2, *r->s3, 0, 0, 1);
	test(*r->s2, 4, 2, *r->s3, 0, 1, 4);
	test(*r->s2, 4, 2, *r->s3, 0, 5, 4);
	test(*r->s2, 4, 2, *r->s3, 0, 9, 4);
	test(*r->s2, 4, 2, *r->s3, 0, 10, 4);
	test(*r->s2, 4, 2, *r->s3, 0, 11, 4);
	test(*r->s2, 4, 2, *r->s3, 1, 0, 1);
	test(*r->s2, 4, 2, *r->s3, 1, 1, 3);
	test(*r->s2, 4, 2, *r->s3, 1, 4, 3);
	test(*r->s2, 4, 2, *r->s3, 1, 8, 3);
	test(*r->s2, 4, 2, *r->s3, 1, 9, 3);
	test(*r->s2, 4, 2, *r->s3, 1, 10, 3);
	test(*r->s2, 4, 2, *r->s3, 5, 0, 1);
	test(*r->s2, 4, 2, *r->s3, 5, 1, -1);
	test(*r->s2, 4, 2, *r->s3, 5, 2, -1);
	test(*r->s2, 4, 2, *r->s3, 5, 4, -1);
	test(*r->s2, 4, 2, *r->s3, 5, 5, -1);
	test(*r->s2, 4, 2, *r->s3, 5, 6, -1);
	test(*r->s2, 4, 2, *r->s3, 9, 0, 1);
	test(*r->s2, 4, 2, *r->s3, 9, 1, -5);
	test(*r->s2, 4, 2, *r->s3, 9, 2, -5);
	test(*r->s2, 4, 2, *r->s3, 10, 0, 1);
	test(*r->s2, 4, 2, *r->s3, 10, 1, 1);
	test(*r->s2, 4, 2, *r->s3, 11, 0, 0);
	test(*r->s2, 4, 2, *r->s4, 0, 0, 1);
	test(*r->s2, 4, 2, *r->s4, 0, 1, 4);
	test(*r->s2, 4, 2, *r->s4, 0, 10, 4);
	test(*r->s2, 4, 2, *r->s4, 0, 19, 4);
	test(*r->s2, 4, 2, *r->s4, 0, 20, 4);
	test(*r->s2, 4, 2, *r->s4, 0, 21, 4);
	test(*r->s2, 4, 2, *r->s4, 1, 0, 1);
	test(*r->s2, 4, 2, *r->s4, 1, 1, 3);
	test(*r->s2, 4, 2, *r->s4, 1, 9, 3);
	test(*r->s2, 4, 2, *r->s4, 1, 18, 3);
	test(*r->s2, 4, 2, *r->s4, 1, 19, 3);
	test(*r->s2, 4, 2, *r->s4, 1, 20, 3);
	test(*r->s2, 4, 2, *r->s4, 10, 0, 1);
	test(*r->s2, 4, 2, *r->s4, 10, 1, -6);
	test(*r->s2, 4, 2, *r->s4, 10, 5, -6);
	test(*r->s2, 4, 2, *r->s4, 10, 9, -6);
	test(*r->s2, 4, 2, *r->s4, 10, 10, -6);
	test(*r->s2, 4, 2, *r->s4, 10, 11, -6);
	test(*r->s2, 4, 2, *r->s4, 19, 0, 1);
	test(*r->s2, 4, 2, *r->s4, 19, 1, -15);
	test(*r->s2, 4, 2, *r->s4, 19, 2, -15);
	test(*r->s2, 4, 2, *r->s4, 20, 0, 1);
}

template <class S>
void
test17(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 4, 2, *r->s4, 20, 1, 1);
	test(*r->s2, 4, 2, *r->s4, 21, 0, 0);
	test(*r->s2, 5, 0, *r->s1, 0, 0, 0);
	test(*r->s2, 5, 0, *r->s1, 0, 1, 0);
	test(*r->s2, 5, 0, *r->s1, 1, 0, 0);
	test(*r->s2, 5, 0, *r->s2, 0, 0, 0);
	test(*r->s2, 5, 0, *r->s2, 0, 1, -1);
	test(*r->s2, 5, 0, *r->s2, 0, 2, -2);
	test(*r->s2, 5, 0, *r->s2, 0, 4, -4);
	test(*r->s2, 5, 0, *r->s2, 0, 5, -5);
	test(*r->s2, 5, 0, *r->s2, 0, 6, -5);
	test(*r->s2, 5, 0, *r->s2, 1, 0, 0);
	test(*r->s2, 5, 0, *r->s2, 1, 1, -1);
	test(*r->s2, 5, 0, *r->s2, 1, 2, -2);
	test(*r->s2, 5, 0, *r->s2, 1, 3, -3);
	test(*r->s2, 5, 0, *r->s2, 1, 4, -4);
	test(*r->s2, 5, 0, *r->s2, 1, 5, -4);
	test(*r->s2, 5, 0, *r->s2, 2, 0, 0);
	test(*r->s2, 5, 0, *r->s2, 2, 1, -1);
	test(*r->s2, 5, 0, *r->s2, 2, 2, -2);
	test(*r->s2, 5, 0, *r->s2, 2, 3, -3);
	test(*r->s2, 5, 0, *r->s2, 2, 4, -3);
	test(*r->s2, 5, 0, *r->s2, 4, 0, 0);
	test(*r->s2, 5, 0, *r->s2, 4, 1, -1);
	test(*r->s2, 5, 0, *r->s2, 4, 2, -1);
	test(*r->s2, 5, 0, *r->s2, 5, 0, 0);
	test(*r->s2, 5, 0, *r->s2, 5, 1, 0);
	test(*r->s2, 5, 0, *r->s2, 6, 0, 0);
	test(*r->s2, 5, 0, *r->s3, 0, 0, 0);
	test(*r->s2, 5, 0, *r->s3, 0, 1, -1);
	test(*r->s2, 5, 0, *r->s3, 0, 5, -5);
	test(*r->s2, 5, 0, *r->s3, 0, 9, -9);
	test(*r->s2, 5, 0, *r->s3, 0, 10, -10);
	test(*r->s2, 5, 0, *r->s3, 0, 11, -10);
	test(*r->s2, 5, 0, *r->s3, 1, 0, 0);
	test(*r->s2, 5, 0, *r->s3, 1, 1, -1);
	test(*r->s2, 5, 0, *r->s3, 1, 4, -4);
	test(*r->s2, 5, 0, *r->s3, 1, 8, -8);
	test(*r->s2, 5, 0, *r->s3, 1, 9, -9);
	test(*r->s2, 5, 0, *r->s3, 1, 10, -9);
	test(*r->s2, 5, 0, *r->s3, 5, 0, 0);
	test(*r->s2, 5, 0, *r->s3, 5, 1, -1);
	test(*r->s2, 5, 0, *r->s3, 5, 2, -2);
	test(*r->s2, 5, 0, *r->s3, 5, 4, -4);
	test(*r->s2, 5, 0, *r->s3, 5, 5, -5);
	test(*r->s2, 5, 0, *r->s3, 5, 6, -5);
	test(*r->s2, 5, 0, *r->s3, 9, 0, 0);
	test(*r->s2, 5, 0, *r->s3, 9, 1, -1);
	test(*r->s2, 5, 0, *r->s3, 9, 2, -1);
	test(*r->s2, 5, 0, *r->s3, 10, 0, 0);
	test(*r->s2, 5, 0, *r->s3, 10, 1, 0);
	test(*r->s2, 5, 0, *r->s3, 11, 0, 0);
	test(*r->s2, 5, 0, *r->s4, 0, 0, 0);
	test(*r->s2, 5, 0, *r->s4, 0, 1, -1);
	test(*r->s2, 5, 0, *r->s4, 0, 10, -10);
	test(*r->s2, 5, 0, *r->s4, 0, 19, -19);
	test(*r->s2, 5, 0, *r->s4, 0, 20, -20);
	test(*r->s2, 5, 0, *r->s4, 0, 21, -20);
	test(*r->s2, 5, 0, *r->s4, 1, 0, 0);
	test(*r->s2, 5, 0, *r->s4, 1, 1, -1);
	test(*r->s2, 5, 0, *r->s4, 1, 9, -9);
	test(*r->s2, 5, 0, *r->s4, 1, 18, -18);
	test(*r->s2, 5, 0, *r->s4, 1, 19, -19);
	test(*r->s2, 5, 0, *r->s4, 1, 20, -19);
	test(*r->s2, 5, 0, *r->s4, 10, 0, 0);
	test(*r->s2, 5, 0, *r->s4, 10, 1, -1);
	test(*r->s2, 5, 0, *r->s4, 10, 5, -5);
	test(*r->s2, 5, 0, *r->s4, 10, 9, -9);
	test(*r->s2, 5, 0, *r->s4, 10, 10, -10);
	test(*r->s2, 5, 0, *r->s4, 10, 11, -10);
	test(*r->s2, 5, 0, *r->s4, 19, 0, 0);
	test(*r->s2, 5, 0, *r->s4, 19, 1, -1);
	test(*r->s2, 5, 0, *r->s4, 19, 2, -1);
	test(*r->s2, 5, 0, *r->s4, 20, 0, 0);
	test(*r->s2, 5, 0, *r->s4, 20, 1, 0);
	test(*r->s2, 5, 0, *r->s4, 21, 0, 0);
	test(*r->s2, 5, 1, *r->s1, 0, 0, 0);
	test(*r->s2, 5, 1, *r->s1, 0, 1, 0);
	test(*r->s2, 5, 1, *r->s1, 1, 0, 0);
	test(*r->s2, 5, 1, *r->s2, 0, 0, 0);
	test(*r->s2, 5, 1, *r->s2, 0, 1, -1);
	test(*r->s2, 5, 1, *r->s2, 0, 2, -2);
	test(*r->s2, 5, 1, *r->s2, 0, 4, -4);
	test(*r->s2, 5, 1, *r->s2, 0, 5, -5);
	test(*r->s2, 5, 1, *r->s2, 0, 6, -5);
	test(*r->s2, 5, 1, *r->s2, 1, 0, 0);
	test(*r->s2, 5, 1, *r->s2, 1, 1, -1);
	test(*r->s2, 5, 1, *r->s2, 1, 2, -2);
	test(*r->s2, 5, 1, *r->s2, 1, 3, -3);
	test(*r->s2, 5, 1, *r->s2, 1, 4, -4);
	test(*r->s2, 5, 1, *r->s2, 1, 5, -4);
	test(*r->s2, 5, 1, *r->s2, 2, 0, 0);
	test(*r->s2, 5, 1, *r->s2, 2, 1, -1);
	test(*r->s2, 5, 1, *r->s2, 2, 2, -2);
	test(*r->s2, 5, 1, *r->s2, 2, 3, -3);
	test(*r->s2, 5, 1, *r->s2, 2, 4, -3);
	test(*r->s2, 5, 1, *r->s2, 4, 0, 0);
	test(*r->s2, 5, 1, *r->s2, 4, 1, -1);
	test(*r->s2, 5, 1, *r->s2, 4, 2, -1);
	test(*r->s2, 5, 1, *r->s2, 5, 0, 0);
}

template <class S>
void
test18(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 5, 1, *r->s2, 5, 1, 0);
	test(*r->s2, 5, 1, *r->s2, 6, 0, 0);
	test(*r->s2, 5, 1, *r->s3, 0, 0, 0);
	test(*r->s2, 5, 1, *r->s3, 0, 1, -1);
	test(*r->s2, 5, 1, *r->s3, 0, 5, -5);
	test(*r->s2, 5, 1, *r->s3, 0, 9, -9);
	test(*r->s2, 5, 1, *r->s3, 0, 10, -10);
	test(*r->s2, 5, 1, *r->s3, 0, 11, -10);
	test(*r->s2, 5, 1, *r->s3, 1, 0, 0);
	test(*r->s2, 5, 1, *r->s3, 1, 1, -1);
	test(*r->s2, 5, 1, *r->s3, 1, 4, -4);
	test(*r->s2, 5, 1, *r->s3, 1, 8, -8);
	test(*r->s2, 5, 1, *r->s3, 1, 9, -9);
	test(*r->s2, 5, 1, *r->s3, 1, 10, -9);
	test(*r->s2, 5, 1, *r->s3, 5, 0, 0);
	test(*r->s2, 5, 1, *r->s3, 5, 1, -1);
	test(*r->s2, 5, 1, *r->s3, 5, 2, -2);
	test(*r->s2, 5, 1, *r->s3, 5, 4, -4);
	test(*r->s2, 5, 1, *r->s3, 5, 5, -5);
	test(*r->s2, 5, 1, *r->s3, 5, 6, -5);
	test(*r->s2, 5, 1, *r->s3, 9, 0, 0);
	test(*r->s2, 5, 1, *r->s3, 9, 1, -1);
	test(*r->s2, 5, 1, *r->s3, 9, 2, -1);
	test(*r->s2, 5, 1, *r->s3, 10, 0, 0);
	test(*r->s2, 5, 1, *r->s3, 10, 1, 0);
	test(*r->s2, 5, 1, *r->s3, 11, 0, 0);
	test(*r->s2, 5, 1, *r->s4, 0, 0, 0);
	test(*r->s2, 5, 1, *r->s4, 0, 1, -1);
	test(*r->s2, 5, 1, *r->s4, 0, 10, -10);
	test(*r->s2, 5, 1, *r->s4, 0, 19, -19);
	test(*r->s2, 5, 1, *r->s4, 0, 20, -20);
	test(*r->s2, 5, 1, *r->s4, 0, 21, -20);
	test(*r->s2, 5, 1, *r->s4, 1, 0, 0);
	test(*r->s2, 5, 1, *r->s4, 1, 1, -1);
	test(*r->s2, 5, 1, *r->s4, 1, 9, -9);
	test(*r->s2, 5, 1, *r->s4, 1, 18, -18);
	test(*r->s2, 5, 1, *r->s4, 1, 19, -19);
	test(*r->s2, 5, 1, *r->s4, 1, 20, -19);
	test(*r->s2, 5, 1, *r->s4, 10, 0, 0);
	test(*r->s2, 5, 1, *r->s4, 10, 1, -1);
	test(*r->s2, 5, 1, *r->s4, 10, 5, -5);
	test(*r->s2, 5, 1, *r->s4, 10, 9, -9);
	test(*r->s2, 5, 1, *r->s4, 10, 10, -10);
	test(*r->s2, 5, 1, *r->s4, 10, 11, -10);
	test(*r->s2, 5, 1, *r->s4, 19, 0, 0);
	test(*r->s2, 5, 1, *r->s4, 19, 1, -1);
	test(*r->s2, 5, 1, *r->s4, 19, 2, -1);
	test(*r->s2, 5, 1, *r->s4, 20, 0, 0);
	test(*r->s2, 5, 1, *r->s4, 20, 1, 0);
	test(*r->s2, 5, 1, *r->s4, 21, 0, 0);
	test(*r->s2, 6, 0, *r->s1, 0, 0, 0);
	test(*r->s2, 6, 0, *r->s1, 0, 1, 0);
	test(*r->s2, 6, 0, *r->s1, 1, 0, 0);
	test(*r->s2, 6, 0, *r->s2, 0, 0, 0);
	test(*r->s2, 6, 0, *r->s2, 0, 1, 0);
	test(*r->s2, 6, 0, *r->s2, 0, 2, 0);
	test(*r->s2, 6, 0, *r->s2, 0, 4, 0);
	test(*r->s2, 6, 0, *r->s2, 0, 5, 0);
	test(*r->s2, 6, 0, *r->s2, 0, 6, 0);
	test(*r->s2, 6, 0, *r->s2, 1, 0, 0);
	test(*r->s2, 6, 0, *r->s2, 1, 1, 0);
	test(*r->s2, 6, 0, *r->s2, 1, 2, 0);
	test(*r->s2, 6, 0, *r->s2, 1, 3, 0);
	test(*r->s2, 6, 0, *r->s2, 1, 4, 0);
	test(*r->s2, 6, 0, *r->s2, 1, 5, 0);
	test(*r->s2, 6, 0, *r->s2, 2, 0, 0);
	test(*r->s2, 6, 0, *r->s2, 2, 1, 0);
	test(*r->s2, 6, 0, *r->s2, 2, 2, 0);
	test(*r->s2, 6, 0, *r->s2, 2, 3, 0);
	test(*r->s2, 6, 0, *r->s2, 2, 4, 0);
	test(*r->s2, 6, 0, *r->s2, 4, 0, 0);
	test(*r->s2, 6, 0, *r->s2, 4, 1, 0);
	test(*r->s2, 6, 0, *r->s2, 4, 2, 0);
	test(*r->s2, 6, 0, *r->s2, 5, 0, 0);
	test(*r->s2, 6, 0, *r->s2, 5, 1, 0);
	test(*r->s2, 6, 0, *r->s2, 6, 0, 0);
	test(*r->s2, 6, 0, *r->s3, 0, 0, 0);
	test(*r->s2, 6, 0, *r->s3, 0, 1, 0);
	test(*r->s2, 6, 0, *r->s3, 0, 5, 0);
	test(*r->s2, 6, 0, *r->s3, 0, 9, 0);
	test(*r->s2, 6, 0, *r->s3, 0, 10, 0);
	test(*r->s2, 6, 0, *r->s3, 0, 11, 0);
	test(*r->s2, 6, 0, *r->s3, 1, 0, 0);
	test(*r->s2, 6, 0, *r->s3, 1, 1, 0);
	test(*r->s2, 6, 0, *r->s3, 1, 4, 0);
	test(*r->s2, 6, 0, *r->s3, 1, 8, 0);
	test(*r->s2, 6, 0, *r->s3, 1, 9, 0);
	test(*r->s2, 6, 0, *r->s3, 1, 10, 0);
	test(*r->s2, 6, 0, *r->s3, 5, 0, 0);
	test(*r->s2, 6, 0, *r->s3, 5, 1, 0);
	test(*r->s2, 6, 0, *r->s3, 5, 2, 0);
	test(*r->s2, 6, 0, *r->s3, 5, 4, 0);
	test(*r->s2, 6, 0, *r->s3, 5, 5, 0);
	test(*r->s2, 6, 0, *r->s3, 5, 6, 0);
	test(*r->s2, 6, 0, *r->s3, 9, 0, 0);
	test(*r->s2, 6, 0, *r->s3, 9, 1, 0);
	test(*r->s2, 6, 0, *r->s3, 9, 2, 0);
	test(*r->s2, 6, 0, *r->s3, 10, 0, 0);
	test(*r->s2, 6, 0, *r->s3, 10, 1, 0);
	test(*r->s2, 6, 0, *r->s3, 11, 0, 0);
}

template <class S>
void
test19(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s2, 6, 0, *r->s4, 0, 0, 0);
	test(*r->s2, 6, 0, *r->s4, 0, 1, 0);
	test(*r->s2, 6, 0, *r->s4, 0, 10, 0);
	test(*r->s2, 6, 0, *r->s4, 0, 19, 0);
	test(*r->s2, 6, 0, *r->s4, 0, 20, 0);
	test(*r->s2, 6, 0, *r->s4, 0, 21, 0);
	test(*r->s2, 6, 0, *r->s4, 1, 0, 0);
	test(*r->s2, 6, 0, *r->s4, 1, 1, 0);
	test(*r->s2, 6, 0, *r->s4, 1, 9, 0);
	test(*r->s2, 6, 0, *r->s4, 1, 18, 0);
	test(*r->s2, 6, 0, *r->s4, 1, 19, 0);
	test(*r->s2, 6, 0, *r->s4, 1, 20, 0);
	test(*r->s2, 6, 0, *r->s4, 10, 0, 0);
	test(*r->s2, 6, 0, *r->s4, 10, 1, 0);
	test(*r->s2, 6, 0, *r->s4, 10, 5, 0);
	test(*r->s2, 6, 0, *r->s4, 10, 9, 0);
	test(*r->s2, 6, 0, *r->s4, 10, 10, 0);
	test(*r->s2, 6, 0, *r->s4, 10, 11, 0);
	test(*r->s2, 6, 0, *r->s4, 19, 0, 0);
	test(*r->s2, 6, 0, *r->s4, 19, 1, 0);
	test(*r->s2, 6, 0, *r->s4, 19, 2, 0);
	test(*r->s2, 6, 0, *r->s4, 20, 0, 0);
	test(*r->s2, 6, 0, *r->s4, 20, 1, 0);
	test(*r->s2, 6, 0, *r->s4, 21, 0, 0);
	test(*r->s3, 0, 0, *r->s1, 0, 0, 0);
	test(*r->s3, 0, 0, *r->s1, 0, 1, 0);
	test(*r->s3, 0, 0, *r->s1, 1, 0, 0);
	test(*r->s3, 0, 0, *r->s2, 0, 0, 0);
	test(*r->s3, 0, 0, *r->s2, 0, 1, -1);
	test(*r->s3, 0, 0, *r->s2, 0, 2, -2);
	test(*r->s3, 0, 0, *r->s2, 0, 4, -4);
	test(*r->s3, 0, 0, *r->s2, 0, 5, -5);
	test(*r->s3, 0, 0, *r->s2, 0, 6, -5);
	test(*r->s3, 0, 0, *r->s2, 1, 0, 0);
	test(*r->s3, 0, 0, *r->s2, 1, 1, -1);
	test(*r->s3, 0, 0, *r->s2, 1, 2, -2);
	test(*r->s3, 0, 0, *r->s2, 1, 3, -3);
	test(*r->s3, 0, 0, *r->s2, 1, 4, -4);
	test(*r->s3, 0, 0, *r->s2, 1, 5, -4);
	test(*r->s3, 0, 0, *r->s2, 2, 0, 0);
	test(*r->s3, 0, 0, *r->s2, 2, 1, -1);
	test(*r->s3, 0, 0, *r->s2, 2, 2, -2);
	test(*r->s3, 0, 0, *r->s2, 2, 3, -3);
	test(*r->s3, 0, 0, *r->s2, 2, 4, -3);
	test(*r->s3, 0, 0, *r->s2, 4, 0, 0);
	test(*r->s3, 0, 0, *r->s2, 4, 1, -1);
	test(*r->s3, 0, 0, *r->s2, 4, 2, -1);
	test(*r->s3, 0, 0, *r->s2, 5, 0, 0);
	test(*r->s3, 0, 0, *r->s2, 5, 1, 0);
	test(*r->s3, 0, 0, *r->s2, 6, 0, 0);
	test(*r->s3, 0, 0, *r->s3, 0, 0, 0);
	test(*r->s3, 0, 0, *r->s3, 0, 1, -1);
	test(*r->s3, 0, 0, *r->s3, 0, 5, -5);
	test(*r->s3, 0, 0, *r->s3, 0, 9, -9);
	test(*r->s3, 0, 0, *r->s3, 0, 10, -10);
	test(*r->s3, 0, 0, *r->s3, 0, 11, -10);
	test(*r->s3, 0, 0, *r->s3, 1, 0, 0);
	test(*r->s3, 0, 0, *r->s3, 1, 1, -1);
	test(*r->s3, 0, 0, *r->s3, 1, 4, -4);
	test(*r->s3, 0, 0, *r->s3, 1, 8, -8);
	test(*r->s3, 0, 0, *r->s3, 1, 9, -9);
	test(*r->s3, 0, 0, *r->s3, 1, 10, -9);
	test(*r->s3, 0, 0, *r->s3, 5, 0, 0);
	test(*r->s3, 0, 0, *r->s3, 5, 1, -1);
	test(*r->s3, 0, 0, *r->s3, 5, 2, -2);
	test(*r->s3, 0, 0, *r->s3, 5, 4, -4);
	test(*r->s3, 0, 0, *r->s3, 5, 5, -5);
	test(*r->s3, 0, 0, *r->s3, 5, 6, -5);
	test(*r->s3, 0, 0, *r->s3, 9, 0, 0);
	test(*r->s3, 0, 0, *r->s3, 9, 1, -1);
	test(*r->s3, 0, 0, *r->s3, 9, 2, -1);
	test(*r->s3, 0, 0, *r->s3, 10, 0, 0);
	test(*r->s3, 0, 0, *r->s3, 10, 1, 0);
	test(*r->s3, 0, 0, *r->s3, 11, 0, 0);
	test(*r->s3, 0, 0, *r->s4, 0, 0, 0);
	test(*r->s3, 0, 0, *r->s4, 0, 1, -1);
	test(*r->s3, 0, 0, *r->s4, 0, 10, -10);
	test(*r->s3, 0, 0, *r->s4, 0, 19, -19);
	test(*r->s3, 0, 0, *r->s4, 0, 20, -20);
	test(*r->s3, 0, 0, *r->s4, 0, 21, -20);
	test(*r->s3, 0, 0, *r->s4, 1, 0, 0);
	test(*r->s3, 0, 0, *r->s4, 1, 1, -1);
	test(*r->s3, 0, 0, *r->s4, 1, 9, -9);
	test(*r->s3, 0, 0, *r->s4, 1, 18, -18);
	test(*r->s3, 0, 0, *r->s4, 1, 19, -19);
	test(*r->s3, 0, 0, *r->s4, 1, 20, -19);
	test(*r->s3, 0, 0, *r->s4, 10, 0, 0);
	test(*r->s3, 0, 0, *r->s4, 10, 1, -1);
	test(*r->s3, 0, 0, *r->s4, 10, 5, -5);
	test(*r->s3, 0, 0, *r->s4, 10, 9, -9);
	test(*r->s3, 0, 0, *r->s4, 10, 10, -10);
	test(*r->s3, 0, 0, *r->s4, 10, 11, -10);
	test(*r->s3, 0, 0, *r->s4, 19, 0, 0);
	test(*r->s3, 0, 0, *r->s4, 19, 1, -1);
	test(*r->s3, 0, 0, *r->s4, 19, 2, -1);
	test(*r->s3, 0, 0, *r->s4, 20, 0, 0);
	test(*r->s3, 0, 0, *r->s4, 20, 1, 0);
	test(*r->s3, 0, 0, *r->s4, 21, 0, 0);
	test(*r->s3, 0, 1, *r->s1, 0, 0, 1);
	test(*r->s3, 0, 1, *r->s1, 0, 1, 1);
}

template <class S>
void
test20(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 0, 1, *r->s1, 1, 0, 0);
	test(*r->s3, 0, 1, *r->s2, 0, 0, 1);
	test(*r->s3, 0, 1, *r->s2, 0, 1, 0);
	test(*r->s3, 0, 1, *r->s2, 0, 2, -1);
	test(*r->s3, 0, 1, *r->s2, 0, 4, -3);
	test(*r->s3, 0, 1, *r->s2, 0, 5, -4);
	test(*r->s3, 0, 1, *r->s2, 0, 6, -4);
	test(*r->s3, 0, 1, *r->s2, 1, 0, 1);
	test(*r->s3, 0, 1, *r->s2, 1, 1, -1);
	test(*r->s3, 0, 1, *r->s2, 1, 2, -1);
	test(*r->s3, 0, 1, *r->s2, 1, 3, -1);
	test(*r->s3, 0, 1, *r->s2, 1, 4, -1);
	test(*r->s3, 0, 1, *r->s2, 1, 5, -1);
	test(*r->s3, 0, 1, *r->s2, 2, 0, 1);
	test(*r->s3, 0, 1, *r->s2, 2, 1, -2);
	test(*r->s3, 0, 1, *r->s2, 2, 2, -2);
	test(*r->s3, 0, 1, *r->s2, 2, 3, -2);
	test(*r->s3, 0, 1, *r->s2, 2, 4, -2);
	test(*r->s3, 0, 1, *r->s2, 4, 0, 1);
	test(*r->s3, 0, 1, *r->s2, 4, 1, -4);
	test(*r->s3, 0, 1, *r->s2, 4, 2, -4);
	test(*r->s3, 0, 1, *r->s2, 5, 0, 1);
	test(*r->s3, 0, 1, *r->s2, 5, 1, 1);
	test(*r->s3, 0, 1, *r->s2, 6, 0, 0);
	test(*r->s3, 0, 1, *r->s3, 0, 0, 1);
	test(*r->s3, 0, 1, *r->s3, 0, 1, 0);
	test(*r->s3, 0, 1, *r->s3, 0, 5, -4);
	test(*r->s3, 0, 1, *r->s3, 0, 9, -8);
	test(*r->s3, 0, 1, *r->s3, 0, 10, -9);
	test(*r->s3, 0, 1, *r->s3, 0, 11, -9);
	test(*r->s3, 0, 1, *r->s3, 1, 0, 1);
	test(*r->s3, 0, 1, *r->s3, 1, 1, -1);
	test(*r->s3, 0, 1, *r->s3, 1, 4, -1);
	test(*r->s3, 0, 1, *r->s3, 1, 8, -1);
	test(*r->s3, 0, 1, *r->s3, 1, 9, -1);
	test(*r->s3, 0, 1, *r->s3, 1, 10, -1);
	test(*r->s3, 0, 1, *r->s3, 5, 0, 1);
	test(*r->s3, 0, 1, *r->s3, 5, 1, -5);
	test(*r->s3, 0, 1, *r->s3, 5, 2, -5);
	test(*r->s3, 0, 1, *r->s3, 5, 4, -5);
	test(*r->s3, 0, 1, *r->s3, 5, 5, -5);
	test(*r->s3, 0, 1, *r->s3, 5, 6, -5);
	test(*r->s3, 0, 1, *r->s3, 9, 0, 1);
	test(*r->s3, 0, 1, *r->s3, 9, 1, -9);
	test(*r->s3, 0, 1, *r->s3, 9, 2, -9);
	test(*r->s3, 0, 1, *r->s3, 10, 0, 1);
	test(*r->s3, 0, 1, *r->s3, 10, 1, 1);
	test(*r->s3, 0, 1, *r->s3, 11, 0, 0);
	test(*r->s3, 0, 1, *r->s4, 0, 0, 1);
	test(*r->s3, 0, 1, *r->s4, 0, 1, 0);
	test(*r->s3, 0, 1, *r->s4, 0, 10, -9);
	test(*r->s3, 0, 1, *r->s4, 0, 19, -18);
	test(*r->s3, 0, 1, *r->s4, 0, 20, -19);
	test(*r->s3, 0, 1, *r->s4, 0, 21, -19);
	test(*r->s3, 0, 1, *r->s4, 1, 0, 1);
	test(*r->s3, 0, 1, *r->s4, 1, 1, -1);
	test(*r->s3, 0, 1, *r->s4, 1, 9, -1);
	test(*r->s3, 0, 1, *r->s4, 1, 18, -1);
	test(*r->s3, 0, 1, *r->s4, 1, 19, -1);
	test(*r->s3, 0, 1, *r->s4, 1, 20, -1);
	test(*r->s3, 0, 1, *r->s4, 10, 0, 1);
	test(*r->s3, 0, 1, *r->s4, 10, 1, -10);
	test(*r->s3, 0, 1, *r->s4, 10, 5, -10);
	test(*r->s3, 0, 1, *r->s4, 10, 9, -10);
	test(*r->s3, 0, 1, *r->s4, 10, 10, -10);
	test(*r->s3, 0, 1, *r->s4, 10, 11, -10);
	test(*r->s3, 0, 1, *r->s4, 19, 0, 1);
	test(*r->s3, 0, 1, *r->s4, 19, 1, -19);
	test(*r->s3, 0, 1, *r->s4, 19, 2, -19);
	test(*r->s3, 0, 1, *r->s4, 20, 0, 1);
	test(*r->s3, 0, 1, *r->s4, 20, 1, 1);
	test(*r->s3, 0, 1, *r->s4, 21, 0, 0);
	test(*r->s3, 0, 5, *r->s1, 0, 0, 5);
	test(*r->s3, 0, 5, *r->s1, 0, 1, 5);
	test(*r->s3, 0, 5, *r->s1, 1, 0, 0);
	test(*r->s3, 0, 5, *r->s2, 0, 0, 5);
	test(*r->s3, 0, 5, *r->s2, 0, 1, 4);
	test(*r->s3, 0, 5, *r->s2, 0, 2, 3);
	test(*r->s3, 0, 5, *r->s2, 0, 4, 1);
	test(*r->s3, 0, 5, *r->s2, 0, 5, 0);
	test(*r->s3, 0, 5, *r->s2, 0, 6, 0);
	test(*r->s3, 0, 5, *r->s2, 1, 0, 5);
	test(*r->s3, 0, 5, *r->s2, 1, 1, -1);
	test(*r->s3, 0, 5, *r->s2, 1, 2, -1);
	test(*r->s3, 0, 5, *r->s2, 1, 3, -1);
	test(*r->s3, 0, 5, *r->s2, 1, 4, -1);
	test(*r->s3, 0, 5, *r->s2, 1, 5, -1);
	test(*r->s3, 0, 5, *r->s2, 2, 0, 5);
	test(*r->s3, 0, 5, *r->s2, 2, 1, -2);
	test(*r->s3, 0, 5, *r->s2, 2, 2, -2);
	test(*r->s3, 0, 5, *r->s2, 2, 3, -2);
	test(*r->s3, 0, 5, *r->s2, 2, 4, -2);
	test(*r->s3, 0, 5, *r->s2, 4, 0, 5);
	test(*r->s3, 0, 5, *r->s2, 4, 1, -4);
	test(*r->s3, 0, 5, *r->s2, 4, 2, -4);
	test(*r->s3, 0, 5, *r->s2, 5, 0, 5);
	test(*r->s3, 0, 5, *r->s2, 5, 1, 5);
	test(*r->s3, 0, 5, *r->s2, 6, 0, 0);
	test(*r->s3, 0, 5, *r->s3, 0, 0, 5);
	test(*r->s3, 0, 5, *r->s3, 0, 1, 4);
}

template <class S>
void
test21(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 0, 5, *r->s3, 0, 5, 0);
	test(*r->s3, 0, 5, *r->s3, 0, 9, -4);
	test(*r->s3, 0, 5, *r->s3, 0, 10, -5);
	test(*r->s3, 0, 5, *r->s3, 0, 11, -5);
	test(*r->s3, 0, 5, *r->s3, 1, 0, 5);
	test(*r->s3, 0, 5, *r->s3, 1, 1, -1);
	test(*r->s3, 0, 5, *r->s3, 1, 4, -1);
	test(*r->s3, 0, 5, *r->s3, 1, 8, -1);
	test(*r->s3, 0, 5, *r->s3, 1, 9, -1);
	test(*r->s3, 0, 5, *r->s3, 1, 10, -1);
	test(*r->s3, 0, 5, *r->s3, 5, 0, 5);
	test(*r->s3, 0, 5, *r->s3, 5, 1, -5);
	test(*r->s3, 0, 5, *r->s3, 5, 2, -5);
	test(*r->s3, 0, 5, *r->s3, 5, 4, -5);
	test(*r->s3, 0, 5, *r->s3, 5, 5, -5);
	test(*r->s3, 0, 5, *r->s3, 5, 6, -5);
	test(*r->s3, 0, 5, *r->s3, 9, 0, 5);
	test(*r->s3, 0, 5, *r->s3, 9, 1, -9);
	test(*r->s3, 0, 5, *r->s3, 9, 2, -9);
	test(*r->s3, 0, 5, *r->s3, 10, 0, 5);
	test(*r->s3, 0, 5, *r->s3, 10, 1, 5);
	test(*r->s3, 0, 5, *r->s3, 11, 0, 0);
	test(*r->s3, 0, 5, *r->s4, 0, 0, 5);
	test(*r->s3, 0, 5, *r->s4, 0, 1, 4);
	test(*r->s3, 0, 5, *r->s4, 0, 10, -5);
	test(*r->s3, 0, 5, *r->s4, 0, 19, -14);
	test(*r->s3, 0, 5, *r->s4, 0, 20, -15);
	test(*r->s3, 0, 5, *r->s4, 0, 21, -15);
	test(*r->s3, 0, 5, *r->s4, 1, 0, 5);
	test(*r->s3, 0, 5, *r->s4, 1, 1, -1);
	test(*r->s3, 0, 5, *r->s4, 1, 9, -1);
	test(*r->s3, 0, 5, *r->s4, 1, 18, -1);
	test(*r->s3, 0, 5, *r->s4, 1, 19, -1);
	test(*r->s3, 0, 5, *r->s4, 1, 20, -1);
	test(*r->s3, 0, 5, *r->s4, 10, 0, 5);
	test(*r->s3, 0, 5, *r->s4, 10, 1, -10);
	test(*r->s3, 0, 5, *r->s4, 10, 5, -10);
	test(*r->s3, 0, 5, *r->s4, 10, 9, -10);
	test(*r->s3, 0, 5, *r->s4, 10, 10, -10);
	test(*r->s3, 0, 5, *r->s4, 10, 11, -10);
	test(*r->s3, 0, 5, *r->s4, 19, 0, 5);
	test(*r->s3, 0, 5, *r->s4, 19, 1, -19);
	test(*r->s3, 0, 5, *r->s4, 19, 2, -19);
	test(*r->s3, 0, 5, *r->s4, 20, 0, 5);
	test(*r->s3, 0, 5, *r->s4, 20, 1, 5);
	test(*r->s3, 0, 5, *r->s4, 21, 0, 0);
	test(*r->s3, 0, 9, *r->s1, 0, 0, 9);
	test(*r->s3, 0, 9, *r->s1, 0, 1, 9);
	test(*r->s3, 0, 9, *r->s1, 1, 0, 0);
	test(*r->s3, 0, 9, *r->s2, 0, 0, 9);
	test(*r->s3, 0, 9, *r->s2, 0, 1, 8);
	test(*r->s3, 0, 9, *r->s2, 0, 2, 7);
	test(*r->s3, 0, 9, *r->s2, 0, 4, 5);
	test(*r->s3, 0, 9, *r->s2, 0, 5, 4);
	test(*r->s3, 0, 9, *r->s2, 0, 6, 4);
	test(*r->s3, 0, 9, *r->s2, 1, 0, 9);
	test(*r->s3, 0, 9, *r->s2, 1, 1, -1);
	test(*r->s3, 0, 9, *r->s2, 1, 2, -1);
	test(*r->s3, 0, 9, *r->s2, 1, 3, -1);
	test(*r->s3, 0, 9, *r->s2, 1, 4, -1);
	test(*r->s3, 0, 9, *r->s2, 1, 5, -1);
	test(*r->s3, 0, 9, *r->s2, 2, 0, 9);
	test(*r->s3, 0, 9, *r->s2, 2, 1, -2);
	test(*r->s3, 0, 9, *r->s2, 2, 2, -2);
	test(*r->s3, 0, 9, *r->s2, 2, 3, -2);
	test(*r->s3, 0, 9, *r->s2, 2, 4, -2);
	test(*r->s3, 0, 9, *r->s2, 4, 0, 9);
	test(*r->s3, 0, 9, *r->s2, 4, 1, -4);
	test(*r->s3, 0, 9, *r->s2, 4, 2, -4);
	test(*r->s3, 0, 9, *r->s2, 5, 0, 9);
	test(*r->s3, 0, 9, *r->s2, 5, 1, 9);
	test(*r->s3, 0, 9, *r->s2, 6, 0, 0);
	test(*r->s3, 0, 9, *r->s3, 0, 0, 9);
	test(*r->s3, 0, 9, *r->s3, 0, 1, 8);
	test(*r->s3, 0, 9, *r->s3, 0, 5, 4);
	test(*r->s3, 0, 9, *r->s3, 0, 9, 0);
	test(*r->s3, 0, 9, *r->s3, 0, 10, -1);
	test(*r->s3, 0, 9, *r->s3, 0, 11, -1);
	test(*r->s3, 0, 9, *r->s3, 1, 0, 9);
	test(*r->s3, 0, 9, *r->s3, 1, 1, -1);
	test(*r->s3, 0, 9, *r->s3, 1, 4, -1);
	test(*r->s3, 0, 9, *r->s3, 1, 8, -1);
	test(*r->s3, 0, 9, *r->s3, 1, 9, -1);
	test(*r->s3, 0, 9, *r->s3, 1, 10, -1);
	test(*r->s3, 0, 9, *r->s3, 5, 0, 9);
	test(*r->s3, 0, 9, *r->s3, 5, 1, -5);
	test(*r->s3, 0, 9, *r->s3, 5, 2, -5);
	test(*r->s3, 0, 9, *r->s3, 5, 4, -5);
	test(*r->s3, 0, 9, *r->s3, 5, 5, -5);
	test(*r->s3, 0, 9, *r->s3, 5, 6, -5);
	test(*r->s3, 0, 9, *r->s3, 9, 0, 9);
	test(*r->s3, 0, 9, *r->s3, 9, 1, -9);
	test(*r->s3, 0, 9, *r->s3, 9, 2, -9);
	test(*r->s3, 0, 9, *r->s3, 10, 0, 9);
	test(*r->s3, 0, 9, *r->s3, 10, 1, 9);
	test(*r->s3, 0, 9, *r->s3, 11, 0, 0);
	test(*r->s3, 0, 9, *r->s4, 0, 0, 9);
	test(*r->s3, 0, 9, *r->s4, 0, 1, 8);
	test(*r->s3, 0, 9, *r->s4, 0, 10, -1);
	test(*r->s3, 0, 9, *r->s4, 0, 19, -10);
}

template <class S>
void
test22(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 0, 9, *r->s4, 0, 20, -11);
	test(*r->s3, 0, 9, *r->s4, 0, 21, -11);
	test(*r->s3, 0, 9, *r->s4, 1, 0, 9);
	test(*r->s3, 0, 9, *r->s4, 1, 1, -1);
	test(*r->s3, 0, 9, *r->s4, 1, 9, -1);
	test(*r->s3, 0, 9, *r->s4, 1, 18, -1);
	test(*r->s3, 0, 9, *r->s4, 1, 19, -1);
	test(*r->s3, 0, 9, *r->s4, 1, 20, -1);
	test(*r->s3, 0, 9, *r->s4, 10, 0, 9);
	test(*r->s3, 0, 9, *r->s4, 10, 1, -10);
	test(*r->s3, 0, 9, *r->s4, 10, 5, -10);
	test(*r->s3, 0, 9, *r->s4, 10, 9, -10);
	test(*r->s3, 0, 9, *r->s4, 10, 10, -10);
	test(*r->s3, 0, 9, *r->s4, 10, 11, -10);
	test(*r->s3, 0, 9, *r->s4, 19, 0, 9);
	test(*r->s3, 0, 9, *r->s4, 19, 1, -19);
	test(*r->s3, 0, 9, *r->s4, 19, 2, -19);
	test(*r->s3, 0, 9, *r->s4, 20, 0, 9);
	test(*r->s3, 0, 9, *r->s4, 20, 1, 9);
	test(*r->s3, 0, 9, *r->s4, 21, 0, 0);
	test(*r->s3, 0, 10, *r->s1, 0, 0, 10);
	test(*r->s3, 0, 10, *r->s1, 0, 1, 10);
	test(*r->s3, 0, 10, *r->s1, 1, 0, 0);
	test(*r->s3, 0, 10, *r->s2, 0, 0, 10);
	test(*r->s3, 0, 10, *r->s2, 0, 1, 9);
	test(*r->s3, 0, 10, *r->s2, 0, 2, 8);
	test(*r->s3, 0, 10, *r->s2, 0, 4, 6);
	test(*r->s3, 0, 10, *r->s2, 0, 5, 5);
	test(*r->s3, 0, 10, *r->s2, 0, 6, 5);
	test(*r->s3, 0, 10, *r->s2, 1, 0, 10);
	test(*r->s3, 0, 10, *r->s2, 1, 1, -1);
	test(*r->s3, 0, 10, *r->s2, 1, 2, -1);
	test(*r->s3, 0, 10, *r->s2, 1, 3, -1);
	test(*r->s3, 0, 10, *r->s2, 1, 4, -1);
	test(*r->s3, 0, 10, *r->s2, 1, 5, -1);
	test(*r->s3, 0, 10, *r->s2, 2, 0, 10);
	test(*r->s3, 0, 10, *r->s2, 2, 1, -2);
	test(*r->s3, 0, 10, *r->s2, 2, 2, -2);
	test(*r->s3, 0, 10, *r->s2, 2, 3, -2);
	test(*r->s3, 0, 10, *r->s2, 2, 4, -2);
	test(*r->s3, 0, 10, *r->s2, 4, 0, 10);
	test(*r->s3, 0, 10, *r->s2, 4, 1, -4);
	test(*r->s3, 0, 10, *r->s2, 4, 2, -4);
	test(*r->s3, 0, 10, *r->s2, 5, 0, 10);
	test(*r->s3, 0, 10, *r->s2, 5, 1, 10);
	test(*r->s3, 0, 10, *r->s2, 6, 0, 0);
	test(*r->s3, 0, 10, *r->s3, 0, 0, 10);
	test(*r->s3, 0, 10, *r->s3, 0, 1, 9);
	test(*r->s3, 0, 10, *r->s3, 0, 5, 5);
	test(*r->s3, 0, 10, *r->s3, 0, 9, 1);
	test(*r->s3, 0, 10, *r->s3, 0, 10, 0);
	test(*r->s3, 0, 10, *r->s3, 0, 11, 0);
	test(*r->s3, 0, 10, *r->s3, 1, 0, 10);
	test(*r->s3, 0, 10, *r->s3, 1, 1, -1);
	test(*r->s3, 0, 10, *r->s3, 1, 4, -1);
	test(*r->s3, 0, 10, *r->s3, 1, 8, -1);
	test(*r->s3, 0, 10, *r->s3, 1, 9, -1);
	test(*r->s3, 0, 10, *r->s3, 1, 10, -1);
	test(*r->s3, 0, 10, *r->s3, 5, 0, 10);
	test(*r->s3, 0, 10, *r->s3, 5, 1, -5);
	test(*r->s3, 0, 10, *r->s3, 5, 2, -5);
	test(*r->s3, 0, 10, *r->s3, 5, 4, -5);
	test(*r->s3, 0, 10, *r->s3, 5, 5, -5);
	test(*r->s3, 0, 10, *r->s3, 5, 6, -5);
	test(*r->s3, 0, 10, *r->s3, 9, 0, 10);
	test(*r->s3, 0, 10, *r->s3, 9, 1, -9);
	test(*r->s3, 0, 10, *r->s3, 9, 2, -9);
	test(*r->s3, 0, 10, *r->s3, 10, 0, 10);
	test(*r->s3, 0, 10, *r->s3, 10, 1, 10);
	test(*r->s3, 0, 10, *r->s3, 11, 0, 0);
	test(*r->s3, 0, 10, *r->s4, 0, 0, 10);
	test(*r->s3, 0, 10, *r->s4, 0, 1, 9);
	test(*r->s3, 0, 10, *r->s4, 0, 10, 0);
	test(*r->s3, 0, 10, *r->s4, 0, 19, -9);
	test(*r->s3, 0, 10, *r->s4, 0, 20, -10);
	test(*r->s3, 0, 10, *r->s4, 0, 21, -10);
	test(*r->s3, 0, 10, *r->s4, 1, 0, 10);
	test(*r->s3, 0, 10, *r->s4, 1, 1, -1);
	test(*r->s3, 0, 10, *r->s4, 1, 9, -1);
	test(*r->s3, 0, 10, *r->s4, 1, 18, -1);
	test(*r->s3, 0, 10, *r->s4, 1, 19, -1);
	test(*r->s3, 0, 10, *r->s4, 1, 20, -1);
	test(*r->s3, 0, 10, *r->s4, 10, 0, 10);
	test(*r->s3, 0, 10, *r->s4, 10, 1, -10);
	test(*r->s3, 0, 10, *r->s4, 10, 5, -10);
	test(*r->s3, 0, 10, *r->s4, 10, 9, -10);
	test(*r->s3, 0, 10, *r->s4, 10, 10, -10);
	test(*r->s3, 0, 10, *r->s4, 10, 11, -10);
	test(*r->s3, 0, 10, *r->s4, 19, 0, 10);
	test(*r->s3, 0, 10, *r->s4, 19, 1, -19);
	test(*r->s3, 0, 10, *r->s4, 19, 2, -19);
	test(*r->s3, 0, 10, *r->s4, 20, 0, 10);
	test(*r->s3, 0, 10, *r->s4, 20, 1, 10);
	test(*r->s3, 0, 10, *r->s4, 21, 0, 0);
	test(*r->s3, 0, 11, *r->s1, 0, 0, 10);
	test(*r->s3, 0, 11, *r->s1, 0, 1, 10);
	test(*r->s3, 0, 11, *r->s1, 1, 0, 0);
	test(*r->s3, 0, 11, *r->s2, 0, 0, 10);
	test(*r->s3, 0, 11, *r->s2, 0, 1, 9);
	test(*r->s3, 0, 11, *r->s2, 0, 2, 8);
}

template <class S>
void
test23(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 0, 11, *r->s2, 0, 4, 6);
	test(*r->s3, 0, 11, *r->s2, 0, 5, 5);
	test(*r->s3, 0, 11, *r->s2, 0, 6, 5);
	test(*r->s3, 0, 11, *r->s2, 1, 0, 10);
	test(*r->s3, 0, 11, *r->s2, 1, 1, -1);
	test(*r->s3, 0, 11, *r->s2, 1, 2, -1);
	test(*r->s3, 0, 11, *r->s2, 1, 3, -1);
	test(*r->s3, 0, 11, *r->s2, 1, 4, -1);
	test(*r->s3, 0, 11, *r->s2, 1, 5, -1);
	test(*r->s3, 0, 11, *r->s2, 2, 0, 10);
	test(*r->s3, 0, 11, *r->s2, 2, 1, -2);
	test(*r->s3, 0, 11, *r->s2, 2, 2, -2);
	test(*r->s3, 0, 11, *r->s2, 2, 3, -2);
	test(*r->s3, 0, 11, *r->s2, 2, 4, -2);
	test(*r->s3, 0, 11, *r->s2, 4, 0, 10);
	test(*r->s3, 0, 11, *r->s2, 4, 1, -4);
	test(*r->s3, 0, 11, *r->s2, 4, 2, -4);
	test(*r->s3, 0, 11, *r->s2, 5, 0, 10);
	test(*r->s3, 0, 11, *r->s2, 5, 1, 10);
	test(*r->s3, 0, 11, *r->s2, 6, 0, 0);
	test(*r->s3, 0, 11, *r->s3, 0, 0, 10);
	test(*r->s3, 0, 11, *r->s3, 0, 1, 9);
	test(*r->s3, 0, 11, *r->s3, 0, 5, 5);
	test(*r->s3, 0, 11, *r->s3, 0, 9, 1);
	test(*r->s3, 0, 11, *r->s3, 0, 10, 0);
	test(*r->s3, 0, 11, *r->s3, 0, 11, 0);
	test(*r->s3, 0, 11, *r->s3, 1, 0, 10);
	test(*r->s3, 0, 11, *r->s3, 1, 1, -1);
	test(*r->s3, 0, 11, *r->s3, 1, 4, -1);
	test(*r->s3, 0, 11, *r->s3, 1, 8, -1);
	test(*r->s3, 0, 11, *r->s3, 1, 9, -1);
	test(*r->s3, 0, 11, *r->s3, 1, 10, -1);
	test(*r->s3, 0, 11, *r->s3, 5, 0, 10);
	test(*r->s3, 0, 11, *r->s3, 5, 1, -5);
	test(*r->s3, 0, 11, *r->s3, 5, 2, -5);
	test(*r->s3, 0, 11, *r->s3, 5, 4, -5);
	test(*r->s3, 0, 11, *r->s3, 5, 5, -5);
	test(*r->s3, 0, 11, *r->s3, 5, 6, -5);
	test(*r->s3, 0, 11, *r->s3, 9, 0, 10);
	test(*r->s3, 0, 11, *r->s3, 9, 1, -9);
	test(*r->s3, 0, 11, *r->s3, 9, 2, -9);
	test(*r->s3, 0, 11, *r->s3, 10, 0, 10);
	test(*r->s3, 0, 11, *r->s3, 10, 1, 10);
	test(*r->s3, 0, 11, *r->s3, 11, 0, 0);
	test(*r->s3, 0, 11, *r->s4, 0, 0, 10);
	test(*r->s3, 0, 11, *r->s4, 0, 1, 9);
	test(*r->s3, 0, 11, *r->s4, 0, 10, 0);
	test(*r->s3, 0, 11, *r->s4, 0, 19, -9);
	test(*r->s3, 0, 11, *r->s4, 0, 20, -10);
	test(*r->s3, 0, 11, *r->s4, 0, 21, -10);
	test(*r->s3, 0, 11, *r->s4, 1, 0, 10);
	test(*r->s3, 0, 11, *r->s4, 1, 1, -1);
	test(*r->s3, 0, 11, *r->s4, 1, 9, -1);
	test(*r->s3, 0, 11, *r->s4, 1, 18, -1);
	test(*r->s3, 0, 11, *r->s4, 1, 19, -1);
	test(*r->s3, 0, 11, *r->s4, 1, 20, -1);
	test(*r->s3, 0, 11, *r->s4, 10, 0, 10);
	test(*r->s3, 0, 11, *r->s4, 10, 1, -10);
	test(*r->s3, 0, 11, *r->s4, 10, 5, -10);
	test(*r->s3, 0, 11, *r->s4, 10, 9, -10);
	test(*r->s3, 0, 11, *r->s4, 10, 10, -10);
	test(*r->s3, 0, 11, *r->s4, 10, 11, -10);
	test(*r->s3, 0, 11, *r->s4, 19, 0, 10);
	test(*r->s3, 0, 11, *r->s4, 19, 1, -19);
	test(*r->s3, 0, 11, *r->s4, 19, 2, -19);
	test(*r->s3, 0, 11, *r->s4, 20, 0, 10);
	test(*r->s3, 0, 11, *r->s4, 20, 1, 10);
	test(*r->s3, 0, 11, *r->s4, 21, 0, 0);
	test(*r->s3, 1, 0, *r->s1, 0, 0, 0);
	test(*r->s3, 1, 0, *r->s1, 0, 1, 0);
	test(*r->s3, 1, 0, *r->s1, 1, 0, 0);
	test(*r->s3, 1, 0, *r->s2, 0, 0, 0);
	test(*r->s3, 1, 0, *r->s2, 0, 1, -1);
	test(*r->s3, 1, 0, *r->s2, 0, 2, -2);
	test(*r->s3, 1, 0, *r->s2, 0, 4, -4);
	test(*r->s3, 1, 0, *r->s2, 0, 5, -5);
	test(*r->s3, 1, 0, *r->s2, 0, 6, -5);
	test(*r->s3, 1, 0, *r->s2, 1, 0, 0);
	test(*r->s3, 1, 0, *r->s2, 1, 1, -1);
	test(*r->s3, 1, 0, *r->s2, 1, 2, -2);
	test(*r->s3, 1, 0, *r->s2, 1, 3, -3);
	test(*r->s3, 1, 0, *r->s2, 1, 4, -4);
	test(*r->s3, 1, 0, *r->s2, 1, 5, -4);
	test(*r->s3, 1, 0, *r->s2, 2, 0, 0);
	test(*r->s3, 1, 0, *r->s2, 2, 1, -1);
	test(*r->s3, 1, 0, *r->s2, 2, 2, -2);
	test(*r->s3, 1, 0, *r->s2, 2, 3, -3);
	test(*r->s3, 1, 0, *r->s2, 2, 4, -3);
	test(*r->s3, 1, 0, *r->s2, 4, 0, 0);
	test(*r->s3, 1, 0, *r->s2, 4, 1, -1);
	test(*r->s3, 1, 0, *r->s2, 4, 2, -1);
	test(*r->s3, 1, 0, *r->s2, 5, 0, 0);
	test(*r->s3, 1, 0, *r->s2, 5, 1, 0);
	test(*r->s3, 1, 0, *r->s2, 6, 0, 0);
	test(*r->s3, 1, 0, *r->s3, 0, 0, 0);
	test(*r->s3, 1, 0, *r->s3, 0, 1, -1);
	test(*r->s3, 1, 0, *r->s3, 0, 5, -5);
	test(*r->s3, 1, 0, *r->s3, 0, 9, -9);
	test(*r->s3, 1, 0, *r->s3, 0, 10, -10);
	test(*r->s3, 1, 0, *r->s3, 0, 11, -10);
}

template <class S>
void
test24(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 1, 0, *r->s3, 1, 0, 0);
	test(*r->s3, 1, 0, *r->s3, 1, 1, -1);
	test(*r->s3, 1, 0, *r->s3, 1, 4, -4);
	test(*r->s3, 1, 0, *r->s3, 1, 8, -8);
	test(*r->s3, 1, 0, *r->s3, 1, 9, -9);
	test(*r->s3, 1, 0, *r->s3, 1, 10, -9);
	test(*r->s3, 1, 0, *r->s3, 5, 0, 0);
	test(*r->s3, 1, 0, *r->s3, 5, 1, -1);
	test(*r->s3, 1, 0, *r->s3, 5, 2, -2);
	test(*r->s3, 1, 0, *r->s3, 5, 4, -4);
	test(*r->s3, 1, 0, *r->s3, 5, 5, -5);
	test(*r->s3, 1, 0, *r->s3, 5, 6, -5);
	test(*r->s3, 1, 0, *r->s3, 9, 0, 0);
	test(*r->s3, 1, 0, *r->s3, 9, 1, -1);
	test(*r->s3, 1, 0, *r->s3, 9, 2, -1);
	test(*r->s3, 1, 0, *r->s3, 10, 0, 0);
	test(*r->s3, 1, 0, *r->s3, 10, 1, 0);
	test(*r->s3, 1, 0, *r->s3, 11, 0, 0);
	test(*r->s3, 1, 0, *r->s4, 0, 0, 0);
	test(*r->s3, 1, 0, *r->s4, 0, 1, -1);
	test(*r->s3, 1, 0, *r->s4, 0, 10, -10);
	test(*r->s3, 1, 0, *r->s4, 0, 19, -19);
	test(*r->s3, 1, 0, *r->s4, 0, 20, -20);
	test(*r->s3, 1, 0, *r->s4, 0, 21, -20);
	test(*r->s3, 1, 0, *r->s4, 1, 0, 0);
	test(*r->s3, 1, 0, *r->s4, 1, 1, -1);
	test(*r->s3, 1, 0, *r->s4, 1, 9, -9);
	test(*r->s3, 1, 0, *r->s4, 1, 18, -18);
	test(*r->s3, 1, 0, *r->s4, 1, 19, -19);
	test(*r->s3, 1, 0, *r->s4, 1, 20, -19);
	test(*r->s3, 1, 0, *r->s4, 10, 0, 0);
	test(*r->s3, 1, 0, *r->s4, 10, 1, -1);
	test(*r->s3, 1, 0, *r->s4, 10, 5, -5);
	test(*r->s3, 1, 0, *r->s4, 10, 9, -9);
	test(*r->s3, 1, 0, *r->s4, 10, 10, -10);
	test(*r->s3, 1, 0, *r->s4, 10, 11, -10);
	test(*r->s3, 1, 0, *r->s4, 19, 0, 0);
	test(*r->s3, 1, 0, *r->s4, 19, 1, -1);
	test(*r->s3, 1, 0, *r->s4, 19, 2, -1);
	test(*r->s3, 1, 0, *r->s4, 20, 0, 0);
	test(*r->s3, 1, 0, *r->s4, 20, 1, 0);
	test(*r->s3, 1, 0, *r->s4, 21, 0, 0);
	test(*r->s3, 1, 1, *r->s1, 0, 0, 1);
	test(*r->s3, 1, 1, *r->s1, 0, 1, 1);
	test(*r->s3, 1, 1, *r->s1, 1, 0, 0);
	test(*r->s3, 1, 1, *r->s2, 0, 0, 1);
	test(*r->s3, 1, 1, *r->s2, 0, 1, 1);
	test(*r->s3, 1, 1, *r->s2, 0, 2, 1);
	test(*r->s3, 1, 1, *r->s2, 0, 4, 1);
	test(*r->s3, 1, 1, *r->s2, 0, 5, 1);
	test(*r->s3, 1, 1, *r->s2, 0, 6, 1);
	test(*r->s3, 1, 1, *r->s2, 1, 0, 1);
	test(*r->s3, 1, 1, *r->s2, 1, 1, 0);
	test(*r->s3, 1, 1, *r->s2, 1, 2, -1);
	test(*r->s3, 1, 1, *r->s2, 1, 3, -2);
	test(*r->s3, 1, 1, *r->s2, 1, 4, -3);
	test(*r->s3, 1, 1, *r->s2, 1, 5, -3);
	test(*r->s3, 1, 1, *r->s2, 2, 0, 1);
	test(*r->s3, 1, 1, *r->s2, 2, 1, -1);
	test(*r->s3, 1, 1, *r->s2, 2, 2, -1);
	test(*r->s3, 1, 1, *r->s2, 2, 3, -1);
	test(*r->s3, 1, 1, *r->s2, 2, 4, -1);
	test(*r->s3, 1, 1, *r->s2, 4, 0, 1);
	test(*r->s3, 1, 1, *r->s2, 4, 1, -3);
	test(*r->s3, 1, 1, *r->s2, 4, 2, -3);
	test(*r->s3, 1, 1, *r->s2, 5, 0, 1);
	test(*r->s3, 1, 1, *r->s2, 5, 1, 1);
	test(*r->s3, 1, 1, *r->s2, 6, 0, 0);
	test(*r->s3, 1, 1, *r->s3, 0, 0, 1);
	test(*r->s3, 1, 1, *r->s3, 0, 1, 1);
	test(*r->s3, 1, 1, *r->s3, 0, 5, 1);
	test(*r->s3, 1, 1, *r->s3, 0, 9, 1);
	test(*r->s3, 1, 1, *r->s3, 0, 10, 1);
	test(*r->s3, 1, 1, *r->s3, 0, 11, 1);
	test(*r->s3, 1, 1, *r->s3, 1, 0, 1);
	test(*r->s3, 1, 1, *r->s3, 1, 1, 0);
	test(*r->s3, 1, 1, *r->s3, 1, 4, -3);
	test(*r->s3, 1, 1, *r->s3, 1, 8, -7);
	test(*r->s3, 1, 1, *r->s3, 1, 9, -8);
	test(*r->s3, 1, 1, *r->s3, 1, 10, -8);
	test(*r->s3, 1, 1, *r->s3, 5, 0, 1);
	test(*r->s3, 1, 1, *r->s3, 5, 1, -4);
	test(*r->s3, 1, 1, *r->s3, 5, 2, -4);
	test(*r->s3, 1, 1, *r->s3, 5, 4, -4);
	test(*r->s3, 1, 1, *r->s3, 5, 5, -4);
	test(*r->s3, 1, 1, *r->s3, 5, 6, -4);
	test(*r->s3, 1, 1, *r->s3, 9, 0, 1);
	test(*r->s3, 1, 1, *r->s3, 9, 1, -8);
	test(*r->s3, 1, 1, *r->s3, 9, 2, -8);
	test(*r->s3, 1, 1, *r->s3, 10, 0, 1);
	test(*r->s3, 1, 1, *r->s3, 10, 1, 1);
	test(*r->s3, 1, 1, *r->s3, 11, 0, 0);
	test(*r->s3, 1, 1, *r->s4, 0, 0, 1);
	test(*r->s3, 1, 1, *r->s4, 0, 1, 1);
	test(*r->s3, 1, 1, *r->s4, 0, 10, 1);
	test(*r->s3, 1, 1, *r->s4, 0, 19, 1);
	test(*r->s3, 1, 1, *r->s4, 0, 20, 1);
	test(*r->s3, 1, 1, *r->s4, 0, 21, 1);
	test(*r->s3, 1, 1, *r->s4, 1, 0, 1);
	test(*r->s3, 1, 1, *r->s4, 1, 1, 0);
}

template <class S>
void
test25(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 1, 1, *r->s4, 1, 9, -8);
	test(*r->s3, 1, 1, *r->s4, 1, 18, -17);
	test(*r->s3, 1, 1, *r->s4, 1, 19, -18);
	test(*r->s3, 1, 1, *r->s4, 1, 20, -18);
	test(*r->s3, 1, 1, *r->s4, 10, 0, 1);
	test(*r->s3, 1, 1, *r->s4, 10, 1, -9);
	test(*r->s3, 1, 1, *r->s4, 10, 5, -9);
	test(*r->s3, 1, 1, *r->s4, 10, 9, -9);
	test(*r->s3, 1, 1, *r->s4, 10, 10, -9);
	test(*r->s3, 1, 1, *r->s4, 10, 11, -9);
	test(*r->s3, 1, 1, *r->s4, 19, 0, 1);
	test(*r->s3, 1, 1, *r->s4, 19, 1, -18);
	test(*r->s3, 1, 1, *r->s4, 19, 2, -18);
	test(*r->s3, 1, 1, *r->s4, 20, 0, 1);
	test(*r->s3, 1, 1, *r->s4, 20, 1, 1);
	test(*r->s3, 1, 1, *r->s4, 21, 0, 0);
	test(*r->s3, 1, 4, *r->s1, 0, 0, 4);
	test(*r->s3, 1, 4, *r->s1, 0, 1, 4);
	test(*r->s3, 1, 4, *r->s1, 1, 0, 0);
	test(*r->s3, 1, 4, *r->s2, 0, 0, 4);
	test(*r->s3, 1, 4, *r->s2, 0, 1, 1);
	test(*r->s3, 1, 4, *r->s2, 0, 2, 1);
	test(*r->s3, 1, 4, *r->s2, 0, 4, 1);
	test(*r->s3, 1, 4, *r->s2, 0, 5, 1);
	test(*r->s3, 1, 4, *r->s2, 0, 6, 1);
	test(*r->s3, 1, 4, *r->s2, 1, 0, 4);
	test(*r->s3, 1, 4, *r->s2, 1, 1, 3);
	test(*r->s3, 1, 4, *r->s2, 1, 2, 2);
	test(*r->s3, 1, 4, *r->s2, 1, 3, 1);
	test(*r->s3, 1, 4, *r->s2, 1, 4, 0);
	test(*r->s3, 1, 4, *r->s2, 1, 5, 0);
	test(*r->s3, 1, 4, *r->s2, 2, 0, 4);
	test(*r->s3, 1, 4, *r->s2, 2, 1, -1);
	test(*r->s3, 1, 4, *r->s2, 2, 2, -1);
	test(*r->s3, 1, 4, *r->s2, 2, 3, -1);
	test(*r->s3, 1, 4, *r->s2, 2, 4, -1);
	test(*r->s3, 1, 4, *r->s2, 4, 0, 4);
	test(*r->s3, 1, 4, *r->s2, 4, 1, -3);
	test(*r->s3, 1, 4, *r->s2, 4, 2, -3);
	test(*r->s3, 1, 4, *r->s2, 5, 0, 4);
	test(*r->s3, 1, 4, *r->s2, 5, 1, 4);
	test(*r->s3, 1, 4, *r->s2, 6, 0, 0);
	test(*r->s3, 1, 4, *r->s3, 0, 0, 4);
	test(*r->s3, 1, 4, *r->s3, 0, 1, 1);
	test(*r->s3, 1, 4, *r->s3, 0, 5, 1);
	test(*r->s3, 1, 4, *r->s3, 0, 9, 1);
	test(*r->s3, 1, 4, *r->s3, 0, 10, 1);
	test(*r->s3, 1, 4, *r->s3, 0, 11, 1);
	test(*r->s3, 1, 4, *r->s3, 1, 0, 4);
	test(*r->s3, 1, 4, *r->s3, 1, 1, 3);
	test(*r->s3, 1, 4, *r->s3, 1, 4, 0);
	test(*r->s3, 1, 4, *r->s3, 1, 8, -4);
	test(*r->s3, 1, 4, *r->s3, 1, 9, -5);
	test(*r->s3, 1, 4, *r->s3, 1, 10, -5);
	test(*r->s3, 1, 4, *r->s3, 5, 0, 4);
	test(*r->s3, 1, 4, *r->s3, 5, 1, -4);
	test(*r->s3, 1, 4, *r->s3, 5, 2, -4);
	test(*r->s3, 1, 4, *r->s3, 5, 4, -4);
	test(*r->s3, 1, 4, *r->s3, 5, 5, -4);
	test(*r->s3, 1, 4, *r->s3, 5, 6, -4);
	test(*r->s3, 1, 4, *r->s3, 9, 0, 4);
	test(*r->s3, 1, 4, *r->s3, 9, 1, -8);
	test(*r->s3, 1, 4, *r->s3, 9, 2, -8);
	test(*r->s3, 1, 4, *r->s3, 10, 0, 4);
	test(*r->s3, 1, 4, *r->s3, 10, 1, 4);
	test(*r->s3, 1, 4, *r->s3, 11, 0, 0);
	test(*r->s3, 1, 4, *r->s4, 0, 0, 4);
	test(*r->s3, 1, 4, *r->s4, 0, 1, 1);
	test(*r->s3, 1, 4, *r->s4, 0, 10, 1);
	test(*r->s3, 1, 4, *r->s4, 0, 19, 1);
	test(*r->s3, 1, 4, *r->s4, 0, 20, 1);
	test(*r->s3, 1, 4, *r->s4, 0, 21, 1);
	test(*r->s3, 1, 4, *r->s4, 1, 0, 4);
	test(*r->s3, 1, 4, *r->s4, 1, 1, 3);
	test(*r->s3, 1, 4, *r->s4, 1, 9, -5);
	test(*r->s3, 1, 4, *r->s4, 1, 18, -14);
	test(*r->s3, 1, 4, *r->s4, 1, 19, -15);
	test(*r->s3, 1, 4, *r->s4, 1, 20, -15);
	test(*r->s3, 1, 4, *r->s4, 10, 0, 4);
	test(*r->s3, 1, 4, *r->s4, 10, 1, -9);
	test(*r->s3, 1, 4, *r->s4, 10, 5, -9);
	test(*r->s3, 1, 4, *r->s4, 10, 9, -9);
	test(*r->s3, 1, 4, *r->s4, 10, 10, -9);
	test(*r->s3, 1, 4, *r->s4, 10, 11, -9);
	test(*r->s3, 1, 4, *r->s4, 19, 0, 4);
	test(*r->s3, 1, 4, *r->s4, 19, 1, -18);
	test(*r->s3, 1, 4, *r->s4, 19, 2, -18);
	test(*r->s3, 1, 4, *r->s4, 20, 0, 4);
	test(*r->s3, 1, 4, *r->s4, 20, 1, 4);
	test(*r->s3, 1, 4, *r->s4, 21, 0, 0);
	test(*r->s3, 1, 8, *r->s1, 0, 0, 8);
	test(*r->s3, 1, 8, *r->s1, 0, 1, 8);
	test(*r->s3, 1, 8, *r->s1, 1, 0, 0);
	test(*r->s3, 1, 8, *r->s2, 0, 0, 8);
	test(*r->s3, 1, 8, *r->s2, 0, 1, 1);
	test(*r->s3, 1, 8, *r->s2, 0, 2, 1);
	test(*r->s3, 1, 8, *r->s2, 0, 4, 1);
	test(*r->s3, 1, 8, *r->s2, 0, 5, 1);
	test(*r->s3, 1, 8, *r->s2, 0, 6, 1);
	test(*r->s3, 1, 8, *r->s2, 1, 0, 8);
}

template <class S>
void
test26(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 1, 8, *r->s2, 1, 1, 7);
	test(*r->s3, 1, 8, *r->s2, 1, 2, 6);
	test(*r->s3, 1, 8, *r->s2, 1, 3, 5);
	test(*r->s3, 1, 8, *r->s2, 1, 4, 4);
	test(*r->s3, 1, 8, *r->s2, 1, 5, 4);
	test(*r->s3, 1, 8, *r->s2, 2, 0, 8);
	test(*r->s3, 1, 8, *r->s2, 2, 1, -1);
	test(*r->s3, 1, 8, *r->s2, 2, 2, -1);
	test(*r->s3, 1, 8, *r->s2, 2, 3, -1);
	test(*r->s3, 1, 8, *r->s2, 2, 4, -1);
	test(*r->s3, 1, 8, *r->s2, 4, 0, 8);
	test(*r->s3, 1, 8, *r->s2, 4, 1, -3);
	test(*r->s3, 1, 8, *r->s2, 4, 2, -3);
	test(*r->s3, 1, 8, *r->s2, 5, 0, 8);
	test(*r->s3, 1, 8, *r->s2, 5, 1, 8);
	test(*r->s3, 1, 8, *r->s2, 6, 0, 0);
	test(*r->s3, 1, 8, *r->s3, 0, 0, 8);
	test(*r->s3, 1, 8, *r->s3, 0, 1, 1);
	test(*r->s3, 1, 8, *r->s3, 0, 5, 1);
	test(*r->s3, 1, 8, *r->s3, 0, 9, 1);
	test(*r->s3, 1, 8, *r->s3, 0, 10, 1);
	test(*r->s3, 1, 8, *r->s3, 0, 11, 1);
	test(*r->s3, 1, 8, *r->s3, 1, 0, 8);
	test(*r->s3, 1, 8, *r->s3, 1, 1, 7);
	test(*r->s3, 1, 8, *r->s3, 1, 4, 4);
	test(*r->s3, 1, 8, *r->s3, 1, 8, 0);
	test(*r->s3, 1, 8, *r->s3, 1, 9, -1);
	test(*r->s3, 1, 8, *r->s3, 1, 10, -1);
	test(*r->s3, 1, 8, *r->s3, 5, 0, 8);
	test(*r->s3, 1, 8, *r->s3, 5, 1, -4);
	test(*r->s3, 1, 8, *r->s3, 5, 2, -4);
	test(*r->s3, 1, 8, *r->s3, 5, 4, -4);
	test(*r->s3, 1, 8, *r->s3, 5, 5, -4);
	test(*r->s3, 1, 8, *r->s3, 5, 6, -4);
	test(*r->s3, 1, 8, *r->s3, 9, 0, 8);
	test(*r->s3, 1, 8, *r->s3, 9, 1, -8);
	test(*r->s3, 1, 8, *r->s3, 9, 2, -8);
	test(*r->s3, 1, 8, *r->s3, 10, 0, 8);
	test(*r->s3, 1, 8, *r->s3, 10, 1, 8);
	test(*r->s3, 1, 8, *r->s3, 11, 0, 0);
	test(*r->s3, 1, 8, *r->s4, 0, 0, 8);
	test(*r->s3, 1, 8, *r->s4, 0, 1, 1);
	test(*r->s3, 1, 8, *r->s4, 0, 10, 1);
	test(*r->s3, 1, 8, *r->s4, 0, 19, 1);
	test(*r->s3, 1, 8, *r->s4, 0, 20, 1);
	test(*r->s3, 1, 8, *r->s4, 0, 21, 1);
	test(*r->s3, 1, 8, *r->s4, 1, 0, 8);
	test(*r->s3, 1, 8, *r->s4, 1, 1, 7);
	test(*r->s3, 1, 8, *r->s4, 1, 9, -1);
	test(*r->s3, 1, 8, *r->s4, 1, 18, -10);
	test(*r->s3, 1, 8, *r->s4, 1, 19, -11);
	test(*r->s3, 1, 8, *r->s4, 1, 20, -11);
	test(*r->s3, 1, 8, *r->s4, 10, 0, 8);
	test(*r->s3, 1, 8, *r->s4, 10, 1, -9);
	test(*r->s3, 1, 8, *r->s4, 10, 5, -9);
	test(*r->s3, 1, 8, *r->s4, 10, 9, -9);
	test(*r->s3, 1, 8, *r->s4, 10, 10, -9);
	test(*r->s3, 1, 8, *r->s4, 10, 11, -9);
	test(*r->s3, 1, 8, *r->s4, 19, 0, 8);
	test(*r->s3, 1, 8, *r->s4, 19, 1, -18);
	test(*r->s3, 1, 8, *r->s4, 19, 2, -18);
	test(*r->s3, 1, 8, *r->s4, 20, 0, 8);
	test(*r->s3, 1, 8, *r->s4, 20, 1, 8);
	test(*r->s3, 1, 8, *r->s4, 21, 0, 0);
	test(*r->s3, 1, 9, *r->s1, 0, 0, 9);
	test(*r->s3, 1, 9, *r->s1, 0, 1, 9);
	test(*r->s3, 1, 9, *r->s1, 1, 0, 0);
	test(*r->s3, 1, 9, *r->s2, 0, 0, 9);
	test(*r->s3, 1, 9, *r->s2, 0, 1, 1);
	test(*r->s3, 1, 9, *r->s2, 0, 2, 1);
	test(*r->s3, 1, 9, *r->s2, 0, 4, 1);
	test(*r->s3, 1, 9, *r->s2, 0, 5, 1);
	test(*r->s3, 1, 9, *r->s2, 0, 6, 1);
	test(*r->s3, 1, 9, *r->s2, 1, 0, 9);
	test(*r->s3, 1, 9, *r->s2, 1, 1, 8);
	test(*r->s3, 1, 9, *r->s2, 1, 2, 7);
	test(*r->s3, 1, 9, *r->s2, 1, 3, 6);
	test(*r->s3, 1, 9, *r->s2, 1, 4, 5);
	test(*r->s3, 1, 9, *r->s2, 1, 5, 5);
	test(*r->s3, 1, 9, *r->s2, 2, 0, 9);
	test(*r->s3, 1, 9, *r->s2, 2, 1, -1);
	test(*r->s3, 1, 9, *r->s2, 2, 2, -1);
	test(*r->s3, 1, 9, *r->s2, 2, 3, -1);
	test(*r->s3, 1, 9, *r->s2, 2, 4, -1);
	test(*r->s3, 1, 9, *r->s2, 4, 0, 9);
	test(*r->s3, 1, 9, *r->s2, 4, 1, -3);
	test(*r->s3, 1, 9, *r->s2, 4, 2, -3);
	test(*r->s3, 1, 9, *r->s2, 5, 0, 9);
	test(*r->s3, 1, 9, *r->s2, 5, 1, 9);
	test(*r->s3, 1, 9, *r->s2, 6, 0, 0);
	test(*r->s3, 1, 9, *r->s3, 0, 0, 9);
	test(*r->s3, 1, 9, *r->s3, 0, 1, 1);
	test(*r->s3, 1, 9, *r->s3, 0, 5, 1);
	test(*r->s3, 1, 9, *r->s3, 0, 9, 1);
	test(*r->s3, 1, 9, *r->s3, 0, 10, 1);
	test(*r->s3, 1, 9, *r->s3, 0, 11, 1);
	test(*r->s3, 1, 9, *r->s3, 1, 0, 9);
	test(*r->s3, 1, 9, *r->s3, 1, 1, 8);
	test(*r->s3, 1, 9, *r->s3, 1, 4, 5);
	test(*r->s3, 1, 9, *r->s3, 1, 8, 1);
}

template <class S>
void
test27(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 1, 9, *r->s3, 1, 9, 0);
	test(*r->s3, 1, 9, *r->s3, 1, 10, 0);
	test(*r->s3, 1, 9, *r->s3, 5, 0, 9);
	test(*r->s3, 1, 9, *r->s3, 5, 1, -4);
	test(*r->s3, 1, 9, *r->s3, 5, 2, -4);
	test(*r->s3, 1, 9, *r->s3, 5, 4, -4);
	test(*r->s3, 1, 9, *r->s3, 5, 5, -4);
	test(*r->s3, 1, 9, *r->s3, 5, 6, -4);
	test(*r->s3, 1, 9, *r->s3, 9, 0, 9);
	test(*r->s3, 1, 9, *r->s3, 9, 1, -8);
	test(*r->s3, 1, 9, *r->s3, 9, 2, -8);
	test(*r->s3, 1, 9, *r->s3, 10, 0, 9);
	test(*r->s3, 1, 9, *r->s3, 10, 1, 9);
	test(*r->s3, 1, 9, *r->s3, 11, 0, 0);
	test(*r->s3, 1, 9, *r->s4, 0, 0, 9);
	test(*r->s3, 1, 9, *r->s4, 0, 1, 1);
	test(*r->s3, 1, 9, *r->s4, 0, 10, 1);
	test(*r->s3, 1, 9, *r->s4, 0, 19, 1);
	test(*r->s3, 1, 9, *r->s4, 0, 20, 1);
	test(*r->s3, 1, 9, *r->s4, 0, 21, 1);
	test(*r->s3, 1, 9, *r->s4, 1, 0, 9);
	test(*r->s3, 1, 9, *r->s4, 1, 1, 8);
	test(*r->s3, 1, 9, *r->s4, 1, 9, 0);
	test(*r->s3, 1, 9, *r->s4, 1, 18, -9);
	test(*r->s3, 1, 9, *r->s4, 1, 19, -10);
	test(*r->s3, 1, 9, *r->s4, 1, 20, -10);
	test(*r->s3, 1, 9, *r->s4, 10, 0, 9);
	test(*r->s3, 1, 9, *r->s4, 10, 1, -9);
	test(*r->s3, 1, 9, *r->s4, 10, 5, -9);
	test(*r->s3, 1, 9, *r->s4, 10, 9, -9);
	test(*r->s3, 1, 9, *r->s4, 10, 10, -9);
	test(*r->s3, 1, 9, *r->s4, 10, 11, -9);
	test(*r->s3, 1, 9, *r->s4, 19, 0, 9);
	test(*r->s3, 1, 9, *r->s4, 19, 1, -18);
	test(*r->s3, 1, 9, *r->s4, 19, 2, -18);
	test(*r->s3, 1, 9, *r->s4, 20, 0, 9);
	test(*r->s3, 1, 9, *r->s4, 20, 1, 9);
	test(*r->s3, 1, 9, *r->s4, 21, 0, 0);
	test(*r->s3, 1, 10, *r->s1, 0, 0, 9);
	test(*r->s3, 1, 10, *r->s1, 0, 1, 9);
	test(*r->s3, 1, 10, *r->s1, 1, 0, 0);
	test(*r->s3, 1, 10, *r->s2, 0, 0, 9);
	test(*r->s3, 1, 10, *r->s2, 0, 1, 1);
	test(*r->s3, 1, 10, *r->s2, 0, 2, 1);
	test(*r->s3, 1, 10, *r->s2, 0, 4, 1);
	test(*r->s3, 1, 10, *r->s2, 0, 5, 1);
	test(*r->s3, 1, 10, *r->s2, 0, 6, 1);
	test(*r->s3, 1, 10, *r->s2, 1, 0, 9);
	test(*r->s3, 1, 10, *r->s2, 1, 1, 8);
	test(*r->s3, 1, 10, *r->s2, 1, 2, 7);
	test(*r->s3, 1, 10, *r->s2, 1, 3, 6);
	test(*r->s3, 1, 10, *r->s2, 1, 4, 5);
	test(*r->s3, 1, 10, *r->s2, 1, 5, 5);
	test(*r->s3, 1, 10, *r->s2, 2, 0, 9);
	test(*r->s3, 1, 10, *r->s2, 2, 1, -1);
	test(*r->s3, 1, 10, *r->s2, 2, 2, -1);
	test(*r->s3, 1, 10, *r->s2, 2, 3, -1);
	test(*r->s3, 1, 10, *r->s2, 2, 4, -1);
	test(*r->s3, 1, 10, *r->s2, 4, 0, 9);
	test(*r->s3, 1, 10, *r->s2, 4, 1, -3);
	test(*r->s3, 1, 10, *r->s2, 4, 2, -3);
	test(*r->s3, 1, 10, *r->s2, 5, 0, 9);
	test(*r->s3, 1, 10, *r->s2, 5, 1, 9);
	test(*r->s3, 1, 10, *r->s2, 6, 0, 0);
	test(*r->s3, 1, 10, *r->s3, 0, 0, 9);
	test(*r->s3, 1, 10, *r->s3, 0, 1, 1);
	test(*r->s3, 1, 10, *r->s3, 0, 5, 1);
	test(*r->s3, 1, 10, *r->s3, 0, 9, 1);
	test(*r->s3, 1, 10, *r->s3, 0, 10, 1);
	test(*r->s3, 1, 10, *r->s3, 0, 11, 1);
	test(*r->s3, 1, 10, *r->s3, 1, 0, 9);
	test(*r->s3, 1, 10, *r->s3, 1, 1, 8);
	test(*r->s3, 1, 10, *r->s3, 1, 4, 5);
	test(*r->s3, 1, 10, *r->s3, 1, 8, 1);
	test(*r->s3, 1, 10, *r->s3, 1, 9, 0);
	test(*r->s3, 1, 10, *r->s3, 1, 10, 0);
	test(*r->s3, 1, 10, *r->s3, 5, 0, 9);
	test(*r->s3, 1, 10, *r->s3, 5, 1, -4);
	test(*r->s3, 1, 10, *r->s3, 5, 2, -4);
	test(*r->s3, 1, 10, *r->s3, 5, 4, -4);
	test(*r->s3, 1, 10, *r->s3, 5, 5, -4);
	test(*r->s3, 1, 10, *r->s3, 5, 6, -4);
	test(*r->s3, 1, 10, *r->s3, 9, 0, 9);
	test(*r->s3, 1, 10, *r->s3, 9, 1, -8);
	test(*r->s3, 1, 10, *r->s3, 9, 2, -8);
	test(*r->s3, 1, 10, *r->s3, 10, 0, 9);
	test(*r->s3, 1, 10, *r->s3, 10, 1, 9);
	test(*r->s3, 1, 10, *r->s3, 11, 0, 0);
	test(*r->s3, 1, 10, *r->s4, 0, 0, 9);
	test(*r->s3, 1, 10, *r->s4, 0, 1, 1);
	test(*r->s3, 1, 10, *r->s4, 0, 10, 1);
	test(*r->s3, 1, 10, *r->s4, 0, 19, 1);
	test(*r->s3, 1, 10, *r->s4, 0, 20, 1);
	test(*r->s3, 1, 10, *r->s4, 0, 21, 1);
	test(*r->s3, 1, 10, *r->s4, 1, 0, 9);
	test(*r->s3, 1, 10, *r->s4, 1, 1, 8);
	test(*r->s3, 1, 10, *r->s4, 1, 9, 0);
	test(*r->s3, 1, 10, *r->s4, 1, 18, -9);
	test(*r->s3, 1, 10, *r->s4, 1, 19, -10);
	test(*r->s3, 1, 10, *r->s4, 1, 20, -10);
}

template <class S>
void
test28(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 1, 10, *r->s4, 10, 0, 9);
	test(*r->s3, 1, 10, *r->s4, 10, 1, -9);
	test(*r->s3, 1, 10, *r->s4, 10, 5, -9);
	test(*r->s3, 1, 10, *r->s4, 10, 9, -9);
	test(*r->s3, 1, 10, *r->s4, 10, 10, -9);
	test(*r->s3, 1, 10, *r->s4, 10, 11, -9);
	test(*r->s3, 1, 10, *r->s4, 19, 0, 9);
	test(*r->s3, 1, 10, *r->s4, 19, 1, -18);
	test(*r->s3, 1, 10, *r->s4, 19, 2, -18);
	test(*r->s3, 1, 10, *r->s4, 20, 0, 9);
	test(*r->s3, 1, 10, *r->s4, 20, 1, 9);
	test(*r->s3, 1, 10, *r->s4, 21, 0, 0);
	test(*r->s3, 5, 0, *r->s1, 0, 0, 0);
	test(*r->s3, 5, 0, *r->s1, 0, 1, 0);
	test(*r->s3, 5, 0, *r->s1, 1, 0, 0);
	test(*r->s3, 5, 0, *r->s2, 0, 0, 0);
	test(*r->s3, 5, 0, *r->s2, 0, 1, -1);
	test(*r->s3, 5, 0, *r->s2, 0, 2, -2);
	test(*r->s3, 5, 0, *r->s2, 0, 4, -4);
	test(*r->s3, 5, 0, *r->s2, 0, 5, -5);
	test(*r->s3, 5, 0, *r->s2, 0, 6, -5);
	test(*r->s3, 5, 0, *r->s2, 1, 0, 0);
	test(*r->s3, 5, 0, *r->s2, 1, 1, -1);
	test(*r->s3, 5, 0, *r->s2, 1, 2, -2);
	test(*r->s3, 5, 0, *r->s2, 1, 3, -3);
	test(*r->s3, 5, 0, *r->s2, 1, 4, -4);
	test(*r->s3, 5, 0, *r->s2, 1, 5, -4);
	test(*r->s3, 5, 0, *r->s2, 2, 0, 0);
	test(*r->s3, 5, 0, *r->s2, 2, 1, -1);
	test(*r->s3, 5, 0, *r->s2, 2, 2, -2);
	test(*r->s3, 5, 0, *r->s2, 2, 3, -3);
	test(*r->s3, 5, 0, *r->s2, 2, 4, -3);
	test(*r->s3, 5, 0, *r->s2, 4, 0, 0);
	test(*r->s3, 5, 0, *r->s2, 4, 1, -1);
	test(*r->s3, 5, 0, *r->s2, 4, 2, -1);
	test(*r->s3, 5, 0, *r->s2, 5, 0, 0);
	test(*r->s3, 5, 0, *r->s2, 5, 1, 0);
	test(*r->s3, 5, 0, *r->s2, 6, 0, 0);
	test(*r->s3, 5, 0, *r->s3, 0, 0, 0);
	test(*r->s3, 5, 0, *r->s3, 0, 1, -1);
	test(*r->s3, 5, 0, *r->s3, 0, 5, -5);
	test(*r->s3, 5, 0, *r->s3, 0, 9, -9);
	test(*r->s3, 5, 0, *r->s3, 0, 10, -10);
	test(*r->s3, 5, 0, *r->s3, 0, 11, -10);
	test(*r->s3, 5, 0, *r->s3, 1, 0, 0);
	test(*r->s3, 5, 0, *r->s3, 1, 1, -1);
	test(*r->s3, 5, 0, *r->s3, 1, 4, -4);
	test(*r->s3, 5, 0, *r->s3, 1, 8, -8);
	test(*r->s3, 5, 0, *r->s3, 1, 9, -9);
	test(*r->s3, 5, 0, *r->s3, 1, 10, -9);
	test(*r->s3, 5, 0, *r->s3, 5, 0, 0);
	test(*r->s3, 5, 0, *r->s3, 5, 1, -1);
	test(*r->s3, 5, 0, *r->s3, 5, 2, -2);
	test(*r->s3, 5, 0, *r->s3, 5, 4, -4);
	test(*r->s3, 5, 0, *r->s3, 5, 5, -5);
	test(*r->s3, 5, 0, *r->s3, 5, 6, -5);
	test(*r->s3, 5, 0, *r->s3, 9, 0, 0);
	test(*r->s3, 5, 0, *r->s3, 9, 1, -1);
	test(*r->s3, 5, 0, *r->s3, 9, 2, -1);
	test(*r->s3, 5, 0, *r->s3, 10, 0, 0);
	test(*r->s3, 5, 0, *r->s3, 10, 1, 0);
	test(*r->s3, 5, 0, *r->s3, 11, 0, 0);
	test(*r->s3, 5, 0, *r->s4, 0, 0, 0);
	test(*r->s3, 5, 0, *r->s4, 0, 1, -1);
	test(*r->s3, 5, 0, *r->s4, 0, 10, -10);
	test(*r->s3, 5, 0, *r->s4, 0, 19, -19);
	test(*r->s3, 5, 0, *r->s4, 0, 20, -20);
	test(*r->s3, 5, 0, *r->s4, 0, 21, -20);
	test(*r->s3, 5, 0, *r->s4, 1, 0, 0);
	test(*r->s3, 5, 0, *r->s4, 1, 1, -1);
	test(*r->s3, 5, 0, *r->s4, 1, 9, -9);
	test(*r->s3, 5, 0, *r->s4, 1, 18, -18);
	test(*r->s3, 5, 0, *r->s4, 1, 19, -19);
	test(*r->s3, 5, 0, *r->s4, 1, 20, -19);
	test(*r->s3, 5, 0, *r->s4, 10, 0, 0);
	test(*r->s3, 5, 0, *r->s4, 10, 1, -1);
	test(*r->s3, 5, 0, *r->s4, 10, 5, -5);
	test(*r->s3, 5, 0, *r->s4, 10, 9, -9);
	test(*r->s3, 5, 0, *r->s4, 10, 10, -10);
	test(*r->s3, 5, 0, *r->s4, 10, 11, -10);
	test(*r->s3, 5, 0, *r->s4, 19, 0, 0);
	test(*r->s3, 5, 0, *r->s4, 19, 1, -1);
	test(*r->s3, 5, 0, *r->s4, 19, 2, -1);
	test(*r->s3, 5, 0, *r->s4, 20, 0, 0);
	test(*r->s3, 5, 0, *r->s4, 20, 1, 0);
	test(*r->s3, 5, 0, *r->s4, 21, 0, 0);
	test(*r->s3, 5, 1, *r->s1, 0, 0, 1);
	test(*r->s3, 5, 1, *r->s1, 0, 1, 1);
	test(*r->s3, 5, 1, *r->s1, 1, 0, 0);
	test(*r->s3, 5, 1, *r->s2, 0, 0, 1);
	test(*r->s3, 5, 1, *r->s2, 0, 1, 5);
	test(*r->s3, 5, 1, *r->s2, 0, 2, 5);
	test(*r->s3, 5, 1, *r->s2, 0, 4, 5);
	test(*r->s3, 5, 1, *r->s2, 0, 5, 5);
	test(*r->s3, 5, 1, *r->s2, 0, 6, 5);
	test(*r->s3, 5, 1, *r->s2, 1, 0, 1);
	test(*r->s3, 5, 1, *r->s2, 1, 1, 4);
	test(*r->s3, 5, 1, *r->s2, 1, 2, 4);
	test(*r->s3, 5, 1, *r->s2, 1, 3, 4);
	test(*r->s3, 5, 1, *r->s2, 1, 4, 4);
}

template <class S>
void
test29(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 5, 1, *r->s2, 1, 5, 4);
	test(*r->s3, 5, 1, *r->s2, 2, 0, 1);
	test(*r->s3, 5, 1, *r->s2, 2, 1, 3);
	test(*r->s3, 5, 1, *r->s2, 2, 2, 3);
	test(*r->s3, 5, 1, *r->s2, 2, 3, 3);
	test(*r->s3, 5, 1, *r->s2, 2, 4, 3);
	test(*r->s3, 5, 1, *r->s2, 4, 0, 1);
	test(*r->s3, 5, 1, *r->s2, 4, 1, 1);
	test(*r->s3, 5, 1, *r->s2, 4, 2, 1);
	test(*r->s3, 5, 1, *r->s2, 5, 0, 1);
	test(*r->s3, 5, 1, *r->s2, 5, 1, 1);
	test(*r->s3, 5, 1, *r->s2, 6, 0, 0);
	test(*r->s3, 5, 1, *r->s3, 0, 0, 1);
	test(*r->s3, 5, 1, *r->s3, 0, 1, 5);
	test(*r->s3, 5, 1, *r->s3, 0, 5, 5);
	test(*r->s3, 5, 1, *r->s3, 0, 9, 5);
	test(*r->s3, 5, 1, *r->s3, 0, 10, 5);
	test(*r->s3, 5, 1, *r->s3, 0, 11, 5);
	test(*r->s3, 5, 1, *r->s3, 1, 0, 1);
	test(*r->s3, 5, 1, *r->s3, 1, 1, 4);
	test(*r->s3, 5, 1, *r->s3, 1, 4, 4);
	test(*r->s3, 5, 1, *r->s3, 1, 8, 4);
	test(*r->s3, 5, 1, *r->s3, 1, 9, 4);
	test(*r->s3, 5, 1, *r->s3, 1, 10, 4);
	test(*r->s3, 5, 1, *r->s3, 5, 0, 1);
	test(*r->s3, 5, 1, *r->s3, 5, 1, 0);
	test(*r->s3, 5, 1, *r->s3, 5, 2, -1);
	test(*r->s3, 5, 1, *r->s3, 5, 4, -3);
	test(*r->s3, 5, 1, *r->s3, 5, 5, -4);
	test(*r->s3, 5, 1, *r->s3, 5, 6, -4);
	test(*r->s3, 5, 1, *r->s3, 9, 0, 1);
	test(*r->s3, 5, 1, *r->s3, 9, 1, -4);
	test(*r->s3, 5, 1, *r->s3, 9, 2, -4);
	test(*r->s3, 5, 1, *r->s3, 10, 0, 1);
	test(*r->s3, 5, 1, *r->s3, 10, 1, 1);
	test(*r->s3, 5, 1, *r->s3, 11, 0, 0);
	test(*r->s3, 5, 1, *r->s4, 0, 0, 1);
	test(*r->s3, 5, 1, *r->s4, 0, 1, 5);
	test(*r->s3, 5, 1, *r->s4, 0, 10, 5);
	test(*r->s3, 5, 1, *r->s4, 0, 19, 5);
	test(*r->s3, 5, 1, *r->s4, 0, 20, 5);
	test(*r->s3, 5, 1, *r->s4, 0, 21, 5);
	test(*r->s3, 5, 1, *r->s4, 1, 0, 1);
	test(*r->s3, 5, 1, *r->s4, 1, 1, 4);
	test(*r->s3, 5, 1, *r->s4, 1, 9, 4);
	test(*r->s3, 5, 1, *r->s4, 1, 18, 4);
	test(*r->s3, 5, 1, *r->s4, 1, 19, 4);
	test(*r->s3, 5, 1, *r->s4, 1, 20, 4);
	test(*r->s3, 5, 1, *r->s4, 10, 0, 1);
	test(*r->s3, 5, 1, *r->s4, 10, 1, -5);
	test(*r->s3, 5, 1, *r->s4, 10, 5, -5);
	test(*r->s3, 5, 1, *r->s4, 10, 9, -5);
	test(*r->s3, 5, 1, *r->s4, 10, 10, -5);
	test(*r->s3, 5, 1, *r->s4, 10, 11, -5);
	test(*r->s3, 5, 1, *r->s4, 19, 0, 1);
	test(*r->s3, 5, 1, *r->s4, 19, 1, -14);
	test(*r->s3, 5, 1, *r->s4, 19, 2, -14);
	test(*r->s3, 5, 1, *r->s4, 20, 0, 1);
	test(*r->s3, 5, 1, *r->s4, 20, 1, 1);
	test(*r->s3, 5, 1, *r->s4, 21, 0, 0);
	test(*r->s3, 5, 2, *r->s1, 0, 0, 2);
	test(*r->s3, 5, 2, *r->s1, 0, 1, 2);
	test(*r->s3, 5, 2, *r->s1, 1, 0, 0);
	test(*r->s3, 5, 2, *r->s2, 0, 0, 2);
	test(*r->s3, 5, 2, *r->s2, 0, 1, 5);
	test(*r->s3, 5, 2, *r->s2, 0, 2, 5);
	test(*r->s3, 5, 2, *r->s2, 0, 4, 5);
	test(*r->s3, 5, 2, *r->s2, 0, 5, 5);
	test(*r->s3, 5, 2, *r->s2, 0, 6, 5);
	test(*r->s3, 5, 2, *r->s2, 1, 0, 2);
	test(*r->s3, 5, 2, *r->s2, 1, 1, 4);
	test(*r->s3, 5, 2, *r->s2, 1, 2, 4);
	test(*r->s3, 5, 2, *r->s2, 1, 3, 4);
	test(*r->s3, 5, 2, *r->s2, 1, 4, 4);
	test(*r->s3, 5, 2, *r->s2, 1, 5, 4);
	test(*r->s3, 5, 2, *r->s2, 2, 0, 2);
	test(*r->s3, 5, 2, *r->s2, 2, 1, 3);
	test(*r->s3, 5, 2, *r->s2, 2, 2, 3);
	test(*r->s3, 5, 2, *r->s2, 2, 3, 3);
	test(*r->s3, 5, 2, *r->s2, 2, 4, 3);
	test(*r->s3, 5, 2, *r->s2, 4, 0, 2);
	test(*r->s3, 5, 2, *r->s2, 4, 1, 1);
	test(*r->s3, 5, 2, *r->s2, 4, 2, 1);
	test(*r->s3, 5, 2, *r->s2, 5, 0, 2);
	test(*r->s3, 5, 2, *r->s2, 5, 1, 2);
	test(*r->s3, 5, 2, *r->s2, 6, 0, 0);
	test(*r->s3, 5, 2, *r->s3, 0, 0, 2);
	test(*r->s3, 5, 2, *r->s3, 0, 1, 5);
	test(*r->s3, 5, 2, *r->s3, 0, 5, 5);
	test(*r->s3, 5, 2, *r->s3, 0, 9, 5);
	test(*r->s3, 5, 2, *r->s3, 0, 10, 5);
	test(*r->s3, 5, 2, *r->s3, 0, 11, 5);
	test(*r->s3, 5, 2, *r->s3, 1, 0, 2);
	test(*r->s3, 5, 2, *r->s3, 1, 1, 4);
	test(*r->s3, 5, 2, *r->s3, 1, 4, 4);
	test(*r->s3, 5, 2, *r->s3, 1, 8, 4);
	test(*r->s3, 5, 2, *r->s3, 1, 9, 4);
	test(*r->s3, 5, 2, *r->s3, 1, 10, 4);
	test(*r->s3, 5, 2, *r->s3, 5, 0, 2);
	test(*r->s3, 5, 2, *r->s3, 5, 1, 1);
}

template <class S>
void
test30(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 5, 2, *r->s3, 5, 2, 0);
	test(*r->s3, 5, 2, *r->s3, 5, 4, -2);
	test(*r->s3, 5, 2, *r->s3, 5, 5, -3);
	test(*r->s3, 5, 2, *r->s3, 5, 6, -3);
	test(*r->s3, 5, 2, *r->s3, 9, 0, 2);
	test(*r->s3, 5, 2, *r->s3, 9, 1, -4);
	test(*r->s3, 5, 2, *r->s3, 9, 2, -4);
	test(*r->s3, 5, 2, *r->s3, 10, 0, 2);
	test(*r->s3, 5, 2, *r->s3, 10, 1, 2);
	test(*r->s3, 5, 2, *r->s3, 11, 0, 0);
	test(*r->s3, 5, 2, *r->s4, 0, 0, 2);
	test(*r->s3, 5, 2, *r->s4, 0, 1, 5);
	test(*r->s3, 5, 2, *r->s4, 0, 10, 5);
	test(*r->s3, 5, 2, *r->s4, 0, 19, 5);
	test(*r->s3, 5, 2, *r->s4, 0, 20, 5);
	test(*r->s3, 5, 2, *r->s4, 0, 21, 5);
	test(*r->s3, 5, 2, *r->s4, 1, 0, 2);
	test(*r->s3, 5, 2, *r->s4, 1, 1, 4);
	test(*r->s3, 5, 2, *r->s4, 1, 9, 4);
	test(*r->s3, 5, 2, *r->s4, 1, 18, 4);
	test(*r->s3, 5, 2, *r->s4, 1, 19, 4);
	test(*r->s3, 5, 2, *r->s4, 1, 20, 4);
	test(*r->s3, 5, 2, *r->s4, 10, 0, 2);
	test(*r->s3, 5, 2, *r->s4, 10, 1, -5);
	test(*r->s3, 5, 2, *r->s4, 10, 5, -5);
	test(*r->s3, 5, 2, *r->s4, 10, 9, -5);
	test(*r->s3, 5, 2, *r->s4, 10, 10, -5);
	test(*r->s3, 5, 2, *r->s4, 10, 11, -5);
	test(*r->s3, 5, 2, *r->s4, 19, 0, 2);
	test(*r->s3, 5, 2, *r->s4, 19, 1, -14);
	test(*r->s3, 5, 2, *r->s4, 19, 2, -14);
	test(*r->s3, 5, 2, *r->s4, 20, 0, 2);
	test(*r->s3, 5, 2, *r->s4, 20, 1, 2);
	test(*r->s3, 5, 2, *r->s4, 21, 0, 0);
	test(*r->s3, 5, 4, *r->s1, 0, 0, 4);
	test(*r->s3, 5, 4, *r->s1, 0, 1, 4);
	test(*r->s3, 5, 4, *r->s1, 1, 0, 0);
	test(*r->s3, 5, 4, *r->s2, 0, 0, 4);
	test(*r->s3, 5, 4, *r->s2, 0, 1, 5);
	test(*r->s3, 5, 4, *r->s2, 0, 2, 5);
	test(*r->s3, 5, 4, *r->s2, 0, 4, 5);
	test(*r->s3, 5, 4, *r->s2, 0, 5, 5);
	test(*r->s3, 5, 4, *r->s2, 0, 6, 5);
	test(*r->s3, 5, 4, *r->s2, 1, 0, 4);
	test(*r->s3, 5, 4, *r->s2, 1, 1, 4);
	test(*r->s3, 5, 4, *r->s2, 1, 2, 4);
	test(*r->s3, 5, 4, *r->s2, 1, 3, 4);
	test(*r->s3, 5, 4, *r->s2, 1, 4, 4);
	test(*r->s3, 5, 4, *r->s2, 1, 5, 4);
	test(*r->s3, 5, 4, *r->s2, 2, 0, 4);
	test(*r->s3, 5, 4, *r->s2, 2, 1, 3);
	test(*r->s3, 5, 4, *r->s2, 2, 2, 3);
	test(*r->s3, 5, 4, *r->s2, 2, 3, 3);
	test(*r->s3, 5, 4, *r->s2, 2, 4, 3);
	test(*r->s3, 5, 4, *r->s2, 4, 0, 4);
	test(*r->s3, 5, 4, *r->s2, 4, 1, 1);
	test(*r->s3, 5, 4, *r->s2, 4, 2, 1);
	test(*r->s3, 5, 4, *r->s2, 5, 0, 4);
	test(*r->s3, 5, 4, *r->s2, 5, 1, 4);
	test(*r->s3, 5, 4, *r->s2, 6, 0, 0);
	test(*r->s3, 5, 4, *r->s3, 0, 0, 4);
	test(*r->s3, 5, 4, *r->s3, 0, 1, 5);
	test(*r->s3, 5, 4, *r->s3, 0, 5, 5);
	test(*r->s3, 5, 4, *r->s3, 0, 9, 5);
	test(*r->s3, 5, 4, *r->s3, 0, 10, 5);
	test(*r->s3, 5, 4, *r->s3, 0, 11, 5);
	test(*r->s3, 5, 4, *r->s3, 1, 0, 4);
	test(*r->s3, 5, 4, *r->s3, 1, 1, 4);
	test(*r->s3, 5, 4, *r->s3, 1, 4, 4);
	test(*r->s3, 5, 4, *r->s3, 1, 8, 4);
	test(*r->s3, 5, 4, *r->s3, 1, 9, 4);
	test(*r->s3, 5, 4, *r->s3, 1, 10, 4);
	test(*r->s3, 5, 4, *r->s3, 5, 0, 4);
	test(*r->s3, 5, 4, *r->s3, 5, 1, 3);
	test(*r->s3, 5, 4, *r->s3, 5, 2, 2);
	test(*r->s3, 5, 4, *r->s3, 5, 4, 0);
	test(*r->s3, 5, 4, *r->s3, 5, 5, -1);
	test(*r->s3, 5, 4, *r->s3, 5, 6, -1);
	test(*r->s3, 5, 4, *r->s3, 9, 0, 4);
	test(*r->s3, 5, 4, *r->s3, 9, 1, -4);
	test(*r->s3, 5, 4, *r->s3, 9, 2, -4);
	test(*r->s3, 5, 4, *r->s3, 10, 0, 4);
	test(*r->s3, 5, 4, *r->s3, 10, 1, 4);
	test(*r->s3, 5, 4, *r->s3, 11, 0, 0);
	test(*r->s3, 5, 4, *r->s4, 0, 0, 4);
	test(*r->s3, 5, 4, *r->s4, 0, 1, 5);
	test(*r->s3, 5, 4, *r->s4, 0, 10, 5);
	test(*r->s3, 5, 4, *r->s4, 0, 19, 5);
	test(*r->s3, 5, 4, *r->s4, 0, 20, 5);
	test(*r->s3, 5, 4, *r->s4, 0, 21, 5);
	test(*r->s3, 5, 4, *r->s4, 1, 0, 4);
	test(*r->s3, 5, 4, *r->s4, 1, 1, 4);
	test(*r->s3, 5, 4, *r->s4, 1, 9, 4);
	test(*r->s3, 5, 4, *r->s4, 1, 18, 4);
	test(*r->s3, 5, 4, *r->s4, 1, 19, 4);
	test(*r->s3, 5, 4, *r->s4, 1, 20, 4);
	test(*r->s3, 5, 4, *r->s4, 10, 0, 4);
	test(*r->s3, 5, 4, *r->s4, 10, 1, -5);
	test(*r->s3, 5, 4, *r->s4, 10, 5, -5);
	test(*r->s3, 5, 4, *r->s4, 10, 9, -5);
}

template <class S>
void
test31(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 5, 4, *r->s4, 10, 10, -5);
	test(*r->s3, 5, 4, *r->s4, 10, 11, -5);
	test(*r->s3, 5, 4, *r->s4, 19, 0, 4);
	test(*r->s3, 5, 4, *r->s4, 19, 1, -14);
	test(*r->s3, 5, 4, *r->s4, 19, 2, -14);
	test(*r->s3, 5, 4, *r->s4, 20, 0, 4);
	test(*r->s3, 5, 4, *r->s4, 20, 1, 4);
	test(*r->s3, 5, 4, *r->s4, 21, 0, 0);
	test(*r->s3, 5, 5, *r->s1, 0, 0, 5);
	test(*r->s3, 5, 5, *r->s1, 0, 1, 5);
	test(*r->s3, 5, 5, *r->s1, 1, 0, 0);
	test(*r->s3, 5, 5, *r->s2, 0, 0, 5);
	test(*r->s3, 5, 5, *r->s2, 0, 1, 5);
	test(*r->s3, 5, 5, *r->s2, 0, 2, 5);
	test(*r->s3, 5, 5, *r->s2, 0, 4, 5);
	test(*r->s3, 5, 5, *r->s2, 0, 5, 5);
	test(*r->s3, 5, 5, *r->s2, 0, 6, 5);
	test(*r->s3, 5, 5, *r->s2, 1, 0, 5);
	test(*r->s3, 5, 5, *r->s2, 1, 1, 4);
	test(*r->s3, 5, 5, *r->s2, 1, 2, 4);
	test(*r->s3, 5, 5, *r->s2, 1, 3, 4);
	test(*r->s3, 5, 5, *r->s2, 1, 4, 4);
	test(*r->s3, 5, 5, *r->s2, 1, 5, 4);
	test(*r->s3, 5, 5, *r->s2, 2, 0, 5);
	test(*r->s3, 5, 5, *r->s2, 2, 1, 3);
	test(*r->s3, 5, 5, *r->s2, 2, 2, 3);
	test(*r->s3, 5, 5, *r->s2, 2, 3, 3);
	test(*r->s3, 5, 5, *r->s2, 2, 4, 3);
	test(*r->s3, 5, 5, *r->s2, 4, 0, 5);
	test(*r->s3, 5, 5, *r->s2, 4, 1, 1);
	test(*r->s3, 5, 5, *r->s2, 4, 2, 1);
	test(*r->s3, 5, 5, *r->s2, 5, 0, 5);
	test(*r->s3, 5, 5, *r->s2, 5, 1, 5);
	test(*r->s3, 5, 5, *r->s2, 6, 0, 0);
	test(*r->s3, 5, 5, *r->s3, 0, 0, 5);
	test(*r->s3, 5, 5, *r->s3, 0, 1, 5);
	test(*r->s3, 5, 5, *r->s3, 0, 5, 5);
	test(*r->s3, 5, 5, *r->s3, 0, 9, 5);
	test(*r->s3, 5, 5, *r->s3, 0, 10, 5);
	test(*r->s3, 5, 5, *r->s3, 0, 11, 5);
	test(*r->s3, 5, 5, *r->s3, 1, 0, 5);
	test(*r->s3, 5, 5, *r->s3, 1, 1, 4);
	test(*r->s3, 5, 5, *r->s3, 1, 4, 4);
	test(*r->s3, 5, 5, *r->s3, 1, 8, 4);
	test(*r->s3, 5, 5, *r->s3, 1, 9, 4);
	test(*r->s3, 5, 5, *r->s3, 1, 10, 4);
	test(*r->s3, 5, 5, *r->s3, 5, 0, 5);
	test(*r->s3, 5, 5, *r->s3, 5, 1, 4);
	test(*r->s3, 5, 5, *r->s3, 5, 2, 3);
	test(*r->s3, 5, 5, *r->s3, 5, 4, 1);
	test(*r->s3, 5, 5, *r->s3, 5, 5, 0);
	test(*r->s3, 5, 5, *r->s3, 5, 6, 0);
	test(*r->s3, 5, 5, *r->s3, 9, 0, 5);
	test(*r->s3, 5, 5, *r->s3, 9, 1, -4);
	test(*r->s3, 5, 5, *r->s3, 9, 2, -4);
	test(*r->s3, 5, 5, *r->s3, 10, 0, 5);
	test(*r->s3, 5, 5, *r->s3, 10, 1, 5);
	test(*r->s3, 5, 5, *r->s3, 11, 0, 0);
	test(*r->s3, 5, 5, *r->s4, 0, 0, 5);
	test(*r->s3, 5, 5, *r->s4, 0, 1, 5);
	test(*r->s3, 5, 5, *r->s4, 0, 10, 5);
	test(*r->s3, 5, 5, *r->s4, 0, 19, 5);
	test(*r->s3, 5, 5, *r->s4, 0, 20, 5);
	test(*r->s3, 5, 5, *r->s4, 0, 21, 5);
	test(*r->s3, 5, 5, *r->s4, 1, 0, 5);
	test(*r->s3, 5, 5, *r->s4, 1, 1, 4);
	test(*r->s3, 5, 5, *r->s4, 1, 9, 4);
	test(*r->s3, 5, 5, *r->s4, 1, 18, 4);
	test(*r->s3, 5, 5, *r->s4, 1, 19, 4);
	test(*r->s3, 5, 5, *r->s4, 1, 20, 4);
	test(*r->s3, 5, 5, *r->s4, 10, 0, 5);
	test(*r->s3, 5, 5, *r->s4, 10, 1, -5);
	test(*r->s3, 5, 5, *r->s4, 10, 5, -5);
	test(*r->s3, 5, 5, *r->s4, 10, 9, -5);
	test(*r->s3, 5, 5, *r->s4, 10, 10, -5);
	test(*r->s3, 5, 5, *r->s4, 10, 11, -5);
	test(*r->s3, 5, 5, *r->s4, 19, 0, 5);
	test(*r->s3, 5, 5, *r->s4, 19, 1, -14);
	test(*r->s3, 5, 5, *r->s4, 19, 2, -14);
	test(*r->s3, 5, 5, *r->s4, 20, 0, 5);
	test(*r->s3, 5, 5, *r->s4, 20, 1, 5);
	test(*r->s3, 5, 5, *r->s4, 21, 0, 0);
	test(*r->s3, 5, 6, *r->s1, 0, 0, 5);
	test(*r->s3, 5, 6, *r->s1, 0, 1, 5);
	test(*r->s3, 5, 6, *r->s1, 1, 0, 0);
	test(*r->s3, 5, 6, *r->s2, 0, 0, 5);
	test(*r->s3, 5, 6, *r->s2, 0, 1, 5);
	test(*r->s3, 5, 6, *r->s2, 0, 2, 5);
	test(*r->s3, 5, 6, *r->s2, 0, 4, 5);
	test(*r->s3, 5, 6, *r->s2, 0, 5, 5);
	test(*r->s3, 5, 6, *r->s2, 0, 6, 5);
	test(*r->s3, 5, 6, *r->s2, 1, 0, 5);
	test(*r->s3, 5, 6, *r->s2, 1, 1, 4);
	test(*r->s3, 5, 6, *r->s2, 1, 2, 4);
	test(*r->s3, 5, 6, *r->s2, 1, 3, 4);
	test(*r->s3, 5, 6, *r->s2, 1, 4, 4);
	test(*r->s3, 5, 6, *r->s2, 1, 5, 4);
	test(*r->s3, 5, 6, *r->s2, 2, 0, 5);
	test(*r->s3, 5, 6, *r->s2, 2, 1, 3);
	test(*r->s3, 5, 6, *r->s2, 2, 2, 3);
}

template <class S>
void
test32(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 5, 6, *r->s2, 2, 3, 3);
	test(*r->s3, 5, 6, *r->s2, 2, 4, 3);
	test(*r->s3, 5, 6, *r->s2, 4, 0, 5);
	test(*r->s3, 5, 6, *r->s2, 4, 1, 1);
	test(*r->s3, 5, 6, *r->s2, 4, 2, 1);
	test(*r->s3, 5, 6, *r->s2, 5, 0, 5);
	test(*r->s3, 5, 6, *r->s2, 5, 1, 5);
	test(*r->s3, 5, 6, *r->s2, 6, 0, 0);
	test(*r->s3, 5, 6, *r->s3, 0, 0, 5);
	test(*r->s3, 5, 6, *r->s3, 0, 1, 5);
	test(*r->s3, 5, 6, *r->s3, 0, 5, 5);
	test(*r->s3, 5, 6, *r->s3, 0, 9, 5);
	test(*r->s3, 5, 6, *r->s3, 0, 10, 5);
	test(*r->s3, 5, 6, *r->s3, 0, 11, 5);
	test(*r->s3, 5, 6, *r->s3, 1, 0, 5);
	test(*r->s3, 5, 6, *r->s3, 1, 1, 4);
	test(*r->s3, 5, 6, *r->s3, 1, 4, 4);
	test(*r->s3, 5, 6, *r->s3, 1, 8, 4);
	test(*r->s3, 5, 6, *r->s3, 1, 9, 4);
	test(*r->s3, 5, 6, *r->s3, 1, 10, 4);
	test(*r->s3, 5, 6, *r->s3, 5, 0, 5);
	test(*r->s3, 5, 6, *r->s3, 5, 1, 4);
	test(*r->s3, 5, 6, *r->s3, 5, 2, 3);
	test(*r->s3, 5, 6, *r->s3, 5, 4, 1);
	test(*r->s3, 5, 6, *r->s3, 5, 5, 0);
	test(*r->s3, 5, 6, *r->s3, 5, 6, 0);
	test(*r->s3, 5, 6, *r->s3, 9, 0, 5);
	test(*r->s3, 5, 6, *r->s3, 9, 1, -4);
	test(*r->s3, 5, 6, *r->s3, 9, 2, -4);
	test(*r->s3, 5, 6, *r->s3, 10, 0, 5);
	test(*r->s3, 5, 6, *r->s3, 10, 1, 5);
	test(*r->s3, 5, 6, *r->s3, 11, 0, 0);
	test(*r->s3, 5, 6, *r->s4, 0, 0, 5);
	test(*r->s3, 5, 6, *r->s4, 0, 1, 5);
	test(*r->s3, 5, 6, *r->s4, 0, 10, 5);
	test(*r->s3, 5, 6, *r->s4, 0, 19, 5);
	test(*r->s3, 5, 6, *r->s4, 0, 20, 5);
	test(*r->s3, 5, 6, *r->s4, 0, 21, 5);
	test(*r->s3, 5, 6, *r->s4, 1, 0, 5);
	test(*r->s3, 5, 6, *r->s4, 1, 1, 4);
	test(*r->s3, 5, 6, *r->s4, 1, 9, 4);
	test(*r->s3, 5, 6, *r->s4, 1, 18, 4);
	test(*r->s3, 5, 6, *r->s4, 1, 19, 4);
	test(*r->s3, 5, 6, *r->s4, 1, 20, 4);
	test(*r->s3, 5, 6, *r->s4, 10, 0, 5);
	test(*r->s3, 5, 6, *r->s4, 10, 1, -5);
	test(*r->s3, 5, 6, *r->s4, 10, 5, -5);
	test(*r->s3, 5, 6, *r->s4, 10, 9, -5);
	test(*r->s3, 5, 6, *r->s4, 10, 10, -5);
	test(*r->s3, 5, 6, *r->s4, 10, 11, -5);
	test(*r->s3, 5, 6, *r->s4, 19, 0, 5);
	test(*r->s3, 5, 6, *r->s4, 19, 1, -14);
	test(*r->s3, 5, 6, *r->s4, 19, 2, -14);
	test(*r->s3, 5, 6, *r->s4, 20, 0, 5);
	test(*r->s3, 5, 6, *r->s4, 20, 1, 5);
	test(*r->s3, 5, 6, *r->s4, 21, 0, 0);
	test(*r->s3, 9, 0, *r->s1, 0, 0, 0);
	test(*r->s3, 9, 0, *r->s1, 0, 1, 0);
	test(*r->s3, 9, 0, *r->s1, 1, 0, 0);
	test(*r->s3, 9, 0, *r->s2, 0, 0, 0);
	test(*r->s3, 9, 0, *r->s2, 0, 1, -1);
	test(*r->s3, 9, 0, *r->s2, 0, 2, -2);
	test(*r->s3, 9, 0, *r->s2, 0, 4, -4);
	test(*r->s3, 9, 0, *r->s2, 0, 5, -5);
	test(*r->s3, 9, 0, *r->s2, 0, 6, -5);
	test(*r->s3, 9, 0, *r->s2, 1, 0, 0);
	test(*r->s3, 9, 0, *r->s2, 1, 1, -1);
	test(*r->s3, 9, 0, *r->s2, 1, 2, -2);
	test(*r->s3, 9, 0, *r->s2, 1, 3, -3);
	test(*r->s3, 9, 0, *r->s2, 1, 4, -4);
	test(*r->s3, 9, 0, *r->s2, 1, 5, -4);
	test(*r->s3, 9, 0, *r->s2, 2, 0, 0);
	test(*r->s3, 9, 0, *r->s2, 2, 1, -1);
	test(*r->s3, 9, 0, *r->s2, 2, 2, -2);
	test(*r->s3, 9, 0, *r->s2, 2, 3, -3);
	test(*r->s3, 9, 0, *r->s2, 2, 4, -3);
	test(*r->s3, 9, 0, *r->s2, 4, 0, 0);
	test(*r->s3, 9, 0, *r->s2, 4, 1, -1);
	test(*r->s3, 9, 0, *r->s2, 4, 2, -1);
	test(*r->s3, 9, 0, *r->s2, 5, 0, 0);
	test(*r->s3, 9, 0, *r->s2, 5, 1, 0);
	test(*r->s3, 9, 0, *r->s2, 6, 0, 0);
	test(*r->s3, 9, 0, *r->s3, 0, 0, 0);
	test(*r->s3, 9, 0, *r->s3, 0, 1, -1);
	test(*r->s3, 9, 0, *r->s3, 0, 5, -5);
	test(*r->s3, 9, 0, *r->s3, 0, 9, -9);
	test(*r->s3, 9, 0, *r->s3, 0, 10, -10);
	test(*r->s3, 9, 0, *r->s3, 0, 11, -10);
	test(*r->s3, 9, 0, *r->s3, 1, 0, 0);
	test(*r->s3, 9, 0, *r->s3, 1, 1, -1);
	test(*r->s3, 9, 0, *r->s3, 1, 4, -4);
	test(*r->s3, 9, 0, *r->s3, 1, 8, -8);
	test(*r->s3, 9, 0, *r->s3, 1, 9, -9);
	test(*r->s3, 9, 0, *r->s3, 1, 10, -9);
	test(*r->s3, 9, 0, *r->s3, 5, 0, 0);
	test(*r->s3, 9, 0, *r->s3, 5, 1, -1);
	test(*r->s3, 9, 0, *r->s3, 5, 2, -2);
	test(*r->s3, 9, 0, *r->s3, 5, 4, -4);
	test(*r->s3, 9, 0, *r->s3, 5, 5, -5);
	test(*r->s3, 9, 0, *r->s3, 5, 6, -5);
}

template <class S>
void
test33(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 9, 0, *r->s3, 9, 0, 0);
	test(*r->s3, 9, 0, *r->s3, 9, 1, -1);
	test(*r->s3, 9, 0, *r->s3, 9, 2, -1);
	test(*r->s3, 9, 0, *r->s3, 10, 0, 0);
	test(*r->s3, 9, 0, *r->s3, 10, 1, 0);
	test(*r->s3, 9, 0, *r->s3, 11, 0, 0);
	test(*r->s3, 9, 0, *r->s4, 0, 0, 0);
	test(*r->s3, 9, 0, *r->s4, 0, 1, -1);
	test(*r->s3, 9, 0, *r->s4, 0, 10, -10);
	test(*r->s3, 9, 0, *r->s4, 0, 19, -19);
	test(*r->s3, 9, 0, *r->s4, 0, 20, -20);
	test(*r->s3, 9, 0, *r->s4, 0, 21, -20);
	test(*r->s3, 9, 0, *r->s4, 1, 0, 0);
	test(*r->s3, 9, 0, *r->s4, 1, 1, -1);
	test(*r->s3, 9, 0, *r->s4, 1, 9, -9);
	test(*r->s3, 9, 0, *r->s4, 1, 18, -18);
	test(*r->s3, 9, 0, *r->s4, 1, 19, -19);
	test(*r->s3, 9, 0, *r->s4, 1, 20, -19);
	test(*r->s3, 9, 0, *r->s4, 10, 0, 0);
	test(*r->s3, 9, 0, *r->s4, 10, 1, -1);
	test(*r->s3, 9, 0, *r->s4, 10, 5, -5);
	test(*r->s3, 9, 0, *r->s4, 10, 9, -9);
	test(*r->s3, 9, 0, *r->s4, 10, 10, -10);
	test(*r->s3, 9, 0, *r->s4, 10, 11, -10);
	test(*r->s3, 9, 0, *r->s4, 19, 0, 0);
	test(*r->s3, 9, 0, *r->s4, 19, 1, -1);
	test(*r->s3, 9, 0, *r->s4, 19, 2, -1);
	test(*r->s3, 9, 0, *r->s4, 20, 0, 0);
	test(*r->s3, 9, 0, *r->s4, 20, 1, 0);
	test(*r->s3, 9, 0, *r->s4, 21, 0, 0);
	test(*r->s3, 9, 1, *r->s1, 0, 0, 1);
	test(*r->s3, 9, 1, *r->s1, 0, 1, 1);
	test(*r->s3, 9, 1, *r->s1, 1, 0, 0);
	test(*r->s3, 9, 1, *r->s2, 0, 0, 1);
	test(*r->s3, 9, 1, *r->s2, 0, 1, 9);
	test(*r->s3, 9, 1, *r->s2, 0, 2, 9);
	test(*r->s3, 9, 1, *r->s2, 0, 4, 9);
	test(*r->s3, 9, 1, *r->s2, 0, 5, 9);
	test(*r->s3, 9, 1, *r->s2, 0, 6, 9);
	test(*r->s3, 9, 1, *r->s2, 1, 0, 1);
	test(*r->s3, 9, 1, *r->s2, 1, 1, 8);
	test(*r->s3, 9, 1, *r->s2, 1, 2, 8);
	test(*r->s3, 9, 1, *r->s2, 1, 3, 8);
	test(*r->s3, 9, 1, *r->s2, 1, 4, 8);
	test(*r->s3, 9, 1, *r->s2, 1, 5, 8);
	test(*r->s3, 9, 1, *r->s2, 2, 0, 1);
	test(*r->s3, 9, 1, *r->s2, 2, 1, 7);
	test(*r->s3, 9, 1, *r->s2, 2, 2, 7);
	test(*r->s3, 9, 1, *r->s2, 2, 3, 7);
	test(*r->s3, 9, 1, *r->s2, 2, 4, 7);
	test(*r->s3, 9, 1, *r->s2, 4, 0, 1);
	test(*r->s3, 9, 1, *r->s2, 4, 1, 5);
	test(*r->s3, 9, 1, *r->s2, 4, 2, 5);
	test(*r->s3, 9, 1, *r->s2, 5, 0, 1);
	test(*r->s3, 9, 1, *r->s2, 5, 1, 1);
	test(*r->s3, 9, 1, *r->s2, 6, 0, 0);
	test(*r->s3, 9, 1, *r->s3, 0, 0, 1);
	test(*r->s3, 9, 1, *r->s3, 0, 1, 9);
	test(*r->s3, 9, 1, *r->s3, 0, 5, 9);
	test(*r->s3, 9, 1, *r->s3, 0, 9, 9);
	test(*r->s3, 9, 1, *r->s3, 0, 10, 9);
	test(*r->s3, 9, 1, *r->s3, 0, 11, 9);
	test(*r->s3, 9, 1, *r->s3, 1, 0, 1);
	test(*r->s3, 9, 1, *r->s3, 1, 1, 8);
	test(*r->s3, 9, 1, *r->s3, 1, 4, 8);
	test(*r->s3, 9, 1, *r->s3, 1, 8, 8);
	test(*r->s3, 9, 1, *r->s3, 1, 9, 8);
	test(*r->s3, 9, 1, *r->s3, 1, 10, 8);
	test(*r->s3, 9, 1, *r->s3, 5, 0, 1);
	test(*r->s3, 9, 1, *r->s3, 5, 1, 4);
	test(*r->s3, 9, 1, *r->s3, 5, 2, 4);
	test(*r->s3, 9, 1, *r->s3, 5, 4, 4);
	test(*r->s3, 9, 1, *r->s3, 5, 5, 4);
	test(*r->s3, 9, 1, *r->s3, 5, 6, 4);
	test(*r->s3, 9, 1, *r->s3, 9, 0, 1);
	test(*r->s3, 9, 1, *r->s3, 9, 1, 0);
	test(*r->s3, 9, 1, *r->s3, 9, 2, 0);
	test(*r->s3, 9, 1, *r->s3, 10, 0, 1);
	test(*r->s3, 9, 1, *r->s3, 10, 1, 1);
	test(*r->s3, 9, 1, *r->s3, 11, 0, 0);
	test(*r->s3, 9, 1, *r->s4, 0, 0, 1);
	test(*r->s3, 9, 1, *r->s4, 0, 1, 9);
	test(*r->s3, 9, 1, *r->s4, 0, 10, 9);
	test(*r->s3, 9, 1, *r->s4, 0, 19, 9);
	test(*r->s3, 9, 1, *r->s4, 0, 20, 9);
	test(*r->s3, 9, 1, *r->s4, 0, 21, 9);
	test(*r->s3, 9, 1, *r->s4, 1, 0, 1);
	test(*r->s3, 9, 1, *r->s4, 1, 1, 8);
	test(*r->s3, 9, 1, *r->s4, 1, 9, 8);
	test(*r->s3, 9, 1, *r->s4, 1, 18, 8);
	test(*r->s3, 9, 1, *r->s4, 1, 19, 8);
	test(*r->s3, 9, 1, *r->s4, 1, 20, 8);
	test(*r->s3, 9, 1, *r->s4, 10, 0, 1);
	test(*r->s3, 9, 1, *r->s4, 10, 1, -1);
	test(*r->s3, 9, 1, *r->s4, 10, 5, -1);
	test(*r->s3, 9, 1, *r->s4, 10, 9, -1);
	test(*r->s3, 9, 1, *r->s4, 10, 10, -1);
	test(*r->s3, 9, 1, *r->s4, 10, 11, -1);
	test(*r->s3, 9, 1, *r->s4, 19, 0, 1);
	test(*r->s3, 9, 1, *r->s4, 19, 1, -10);
}

template <class S>
void
test34(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 9, 1, *r->s4, 19, 2, -10);
	test(*r->s3, 9, 1, *r->s4, 20, 0, 1);
	test(*r->s3, 9, 1, *r->s4, 20, 1, 1);
	test(*r->s3, 9, 1, *r->s4, 21, 0, 0);
	test(*r->s3, 9, 2, *r->s1, 0, 0, 1);
	test(*r->s3, 9, 2, *r->s1, 0, 1, 1);
	test(*r->s3, 9, 2, *r->s1, 1, 0, 0);
	test(*r->s3, 9, 2, *r->s2, 0, 0, 1);
	test(*r->s3, 9, 2, *r->s2, 0, 1, 9);
	test(*r->s3, 9, 2, *r->s2, 0, 2, 9);
	test(*r->s3, 9, 2, *r->s2, 0, 4, 9);
	test(*r->s3, 9, 2, *r->s2, 0, 5, 9);
	test(*r->s3, 9, 2, *r->s2, 0, 6, 9);
	test(*r->s3, 9, 2, *r->s2, 1, 0, 1);
	test(*r->s3, 9, 2, *r->s2, 1, 1, 8);
	test(*r->s3, 9, 2, *r->s2, 1, 2, 8);
	test(*r->s3, 9, 2, *r->s2, 1, 3, 8);
	test(*r->s3, 9, 2, *r->s2, 1, 4, 8);
	test(*r->s3, 9, 2, *r->s2, 1, 5, 8);
	test(*r->s3, 9, 2, *r->s2, 2, 0, 1);
	test(*r->s3, 9, 2, *r->s2, 2, 1, 7);
	test(*r->s3, 9, 2, *r->s2, 2, 2, 7);
	test(*r->s3, 9, 2, *r->s2, 2, 3, 7);
	test(*r->s3, 9, 2, *r->s2, 2, 4, 7);
	test(*r->s3, 9, 2, *r->s2, 4, 0, 1);
	test(*r->s3, 9, 2, *r->s2, 4, 1, 5);
	test(*r->s3, 9, 2, *r->s2, 4, 2, 5);
	test(*r->s3, 9, 2, *r->s2, 5, 0, 1);
	test(*r->s3, 9, 2, *r->s2, 5, 1, 1);
	test(*r->s3, 9, 2, *r->s2, 6, 0, 0);
	test(*r->s3, 9, 2, *r->s3, 0, 0, 1);
	test(*r->s3, 9, 2, *r->s3, 0, 1, 9);
	test(*r->s3, 9, 2, *r->s3, 0, 5, 9);
	test(*r->s3, 9, 2, *r->s3, 0, 9, 9);
	test(*r->s3, 9, 2, *r->s3, 0, 10, 9);
	test(*r->s3, 9, 2, *r->s3, 0, 11, 9);
	test(*r->s3, 9, 2, *r->s3, 1, 0, 1);
	test(*r->s3, 9, 2, *r->s3, 1, 1, 8);
	test(*r->s3, 9, 2, *r->s3, 1, 4, 8);
	test(*r->s3, 9, 2, *r->s3, 1, 8, 8);
	test(*r->s3, 9, 2, *r->s3, 1, 9, 8);
	test(*r->s3, 9, 2, *r->s3, 1, 10, 8);
	test(*r->s3, 9, 2, *r->s3, 5, 0, 1);
	test(*r->s3, 9, 2, *r->s3, 5, 1, 4);
	test(*r->s3, 9, 2, *r->s3, 5, 2, 4);
	test(*r->s3, 9, 2, *r->s3, 5, 4, 4);
	test(*r->s3, 9, 2, *r->s3, 5, 5, 4);
	test(*r->s3, 9, 2, *r->s3, 5, 6, 4);
	test(*r->s3, 9, 2, *r->s3, 9, 0, 1);
	test(*r->s3, 9, 2, *r->s3, 9, 1, 0);
	test(*r->s3, 9, 2, *r->s3, 9, 2, 0);
	test(*r->s3, 9, 2, *r->s3, 10, 0, 1);
	test(*r->s3, 9, 2, *r->s3, 10, 1, 1);
	test(*r->s3, 9, 2, *r->s3, 11, 0, 0);
	test(*r->s3, 9, 2, *r->s4, 0, 0, 1);
	test(*r->s3, 9, 2, *r->s4, 0, 1, 9);
	test(*r->s3, 9, 2, *r->s4, 0, 10, 9);
	test(*r->s3, 9, 2, *r->s4, 0, 19, 9);
	test(*r->s3, 9, 2, *r->s4, 0, 20, 9);
	test(*r->s3, 9, 2, *r->s4, 0, 21, 9);
	test(*r->s3, 9, 2, *r->s4, 1, 0, 1);
	test(*r->s3, 9, 2, *r->s4, 1, 1, 8);
	test(*r->s3, 9, 2, *r->s4, 1, 9, 8);
	test(*r->s3, 9, 2, *r->s4, 1, 18, 8);
	test(*r->s3, 9, 2, *r->s4, 1, 19, 8);
	test(*r->s3, 9, 2, *r->s4, 1, 20, 8);
	test(*r->s3, 9, 2, *r->s4, 10, 0, 1);
	test(*r->s3, 9, 2, *r->s4, 10, 1, -1);
	test(*r->s3, 9, 2, *r->s4, 10, 5, -1);
	test(*r->s3, 9, 2, *r->s4, 10, 9, -1);
	test(*r->s3, 9, 2, *r->s4, 10, 10, -1);
	test(*r->s3, 9, 2, *r->s4, 10, 11, -1);
	test(*r->s3, 9, 2, *r->s4, 19, 0, 1);
	test(*r->s3, 9, 2, *r->s4, 19, 1, -10);
	test(*r->s3, 9, 2, *r->s4, 19, 2, -10);
	test(*r->s3, 9, 2, *r->s4, 20, 0, 1);
	test(*r->s3, 9, 2, *r->s4, 20, 1, 1);
	test(*r->s3, 9, 2, *r->s4, 21, 0, 0);
	test(*r->s3, 10, 0, *r->s1, 0, 0, 0);
	test(*r->s3, 10, 0, *r->s1, 0, 1, 0);
	test(*r->s3, 10, 0, *r->s1, 1, 0, 0);
	test(*r->s3, 10, 0, *r->s2, 0, 0, 0);
	test(*r->s3, 10, 0, *r->s2, 0, 1, -1);
	test(*r->s3, 10, 0, *r->s2, 0, 2, -2);
	test(*r->s3, 10, 0, *r->s2, 0, 4, -4);
	test(*r->s3, 10, 0, *r->s2, 0, 5, -5);
	test(*r->s3, 10, 0, *r->s2, 0, 6, -5);
	test(*r->s3, 10, 0, *r->s2, 1, 0, 0);
	test(*r->s3, 10, 0, *r->s2, 1, 1, -1);
	test(*r->s3, 10, 0, *r->s2, 1, 2, -2);
	test(*r->s3, 10, 0, *r->s2, 1, 3, -3);
	test(*r->s3, 10, 0, *r->s2, 1, 4, -4);
	test(*r->s3, 10, 0, *r->s2, 1, 5, -4);
	test(*r->s3, 10, 0, *r->s2, 2, 0, 0);
	test(*r->s3, 10, 0, *r->s2, 2, 1, -1);
	test(*r->s3, 10, 0, *r->s2, 2, 2, -2);
	test(*r->s3, 10, 0, *r->s2, 2, 3, -3);
	test(*r->s3, 10, 0, *r->s2, 2, 4, -3);
	test(*r->s3, 10, 0, *r->s2, 4, 0, 0);
	test(*r->s3, 10, 0, *r->s2, 4, 1, -1);
}

template <class S>
void
test35(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 10, 0, *r->s2, 4, 2, -1);
	test(*r->s3, 10, 0, *r->s2, 5, 0, 0);
	test(*r->s3, 10, 0, *r->s2, 5, 1, 0);
	test(*r->s3, 10, 0, *r->s2, 6, 0, 0);
	test(*r->s3, 10, 0, *r->s3, 0, 0, 0);
	test(*r->s3, 10, 0, *r->s3, 0, 1, -1);
	test(*r->s3, 10, 0, *r->s3, 0, 5, -5);
	test(*r->s3, 10, 0, *r->s3, 0, 9, -9);
	test(*r->s3, 10, 0, *r->s3, 0, 10, -10);
	test(*r->s3, 10, 0, *r->s3, 0, 11, -10);
	test(*r->s3, 10, 0, *r->s3, 1, 0, 0);
	test(*r->s3, 10, 0, *r->s3, 1, 1, -1);
	test(*r->s3, 10, 0, *r->s3, 1, 4, -4);
	test(*r->s3, 10, 0, *r->s3, 1, 8, -8);
	test(*r->s3, 10, 0, *r->s3, 1, 9, -9);
	test(*r->s3, 10, 0, *r->s3, 1, 10, -9);
	test(*r->s3, 10, 0, *r->s3, 5, 0, 0);
	test(*r->s3, 10, 0, *r->s3, 5, 1, -1);
	test(*r->s3, 10, 0, *r->s3, 5, 2, -2);
	test(*r->s3, 10, 0, *r->s3, 5, 4, -4);
	test(*r->s3, 10, 0, *r->s3, 5, 5, -5);
	test(*r->s3, 10, 0, *r->s3, 5, 6, -5);
	test(*r->s3, 10, 0, *r->s3, 9, 0, 0);
	test(*r->s3, 10, 0, *r->s3, 9, 1, -1);
	test(*r->s3, 10, 0, *r->s3, 9, 2, -1);
	test(*r->s3, 10, 0, *r->s3, 10, 0, 0);
	test(*r->s3, 10, 0, *r->s3, 10, 1, 0);
	test(*r->s3, 10, 0, *r->s3, 11, 0, 0);
	test(*r->s3, 10, 0, *r->s4, 0, 0, 0);
	test(*r->s3, 10, 0, *r->s4, 0, 1, -1);
	test(*r->s3, 10, 0, *r->s4, 0, 10, -10);
	test(*r->s3, 10, 0, *r->s4, 0, 19, -19);
	test(*r->s3, 10, 0, *r->s4, 0, 20, -20);
	test(*r->s3, 10, 0, *r->s4, 0, 21, -20);
	test(*r->s3, 10, 0, *r->s4, 1, 0, 0);
	test(*r->s3, 10, 0, *r->s4, 1, 1, -1);
	test(*r->s3, 10, 0, *r->s4, 1, 9, -9);
	test(*r->s3, 10, 0, *r->s4, 1, 18, -18);
	test(*r->s3, 10, 0, *r->s4, 1, 19, -19);
	test(*r->s3, 10, 0, *r->s4, 1, 20, -19);
	test(*r->s3, 10, 0, *r->s4, 10, 0, 0);
	test(*r->s3, 10, 0, *r->s4, 10, 1, -1);
	test(*r->s3, 10, 0, *r->s4, 10, 5, -5);
	test(*r->s3, 10, 0, *r->s4, 10, 9, -9);
	test(*r->s3, 10, 0, *r->s4, 10, 10, -10);
	test(*r->s3, 10, 0, *r->s4, 10, 11, -10);
	test(*r->s3, 10, 0, *r->s4, 19, 0, 0);
	test(*r->s3, 10, 0, *r->s4, 19, 1, -1);
	test(*r->s3, 10, 0, *r->s4, 19, 2, -1);
	test(*r->s3, 10, 0, *r->s4, 20, 0, 0);
	test(*r->s3, 10, 0, *r->s4, 20, 1, 0);
	test(*r->s3, 10, 0, *r->s4, 21, 0, 0);
	test(*r->s3, 10, 1, *r->s1, 0, 0, 0);
	test(*r->s3, 10, 1, *r->s1, 0, 1, 0);
	test(*r->s3, 10, 1, *r->s1, 1, 0, 0);
	test(*r->s3, 10, 1, *r->s2, 0, 0, 0);
	test(*r->s3, 10, 1, *r->s2, 0, 1, -1);
	test(*r->s3, 10, 1, *r->s2, 0, 2, -2);
	test(*r->s3, 10, 1, *r->s2, 0, 4, -4);
	test(*r->s3, 10, 1, *r->s2, 0, 5, -5);
	test(*r->s3, 10, 1, *r->s2, 0, 6, -5);
	test(*r->s3, 10, 1, *r->s2, 1, 0, 0);
	test(*r->s3, 10, 1, *r->s2, 1, 1, -1);
	test(*r->s3, 10, 1, *r->s2, 1, 2, -2);
	test(*r->s3, 10, 1, *r->s2, 1, 3, -3);
	test(*r->s3, 10, 1, *r->s2, 1, 4, -4);
	test(*r->s3, 10, 1, *r->s2, 1, 5, -4);
	test(*r->s3, 10, 1, *r->s2, 2, 0, 0);
	test(*r->s3, 10, 1, *r->s2, 2, 1, -1);
	test(*r->s3, 10, 1, *r->s2, 2, 2, -2);
	test(*r->s3, 10, 1, *r->s2, 2, 3, -3);
	test(*r->s3, 10, 1, *r->s2, 2, 4, -3);
	test(*r->s3, 10, 1, *r->s2, 4, 0, 0);
	test(*r->s3, 10, 1, *r->s2, 4, 1, -1);
	test(*r->s3, 10, 1, *r->s2, 4, 2, -1);
	test(*r->s3, 10, 1, *r->s2, 5, 0, 0);
	test(*r->s3, 10, 1, *r->s2, 5, 1, 0);
	test(*r->s3, 10, 1, *r->s2, 6, 0, 0);
	test(*r->s3, 10, 1, *r->s3, 0, 0, 0);
	test(*r->s3, 10, 1, *r->s3, 0, 1, -1);
	test(*r->s3, 10, 1, *r->s3, 0, 5, -5);
	test(*r->s3, 10, 1, *r->s3, 0, 9, -9);
	test(*r->s3, 10, 1, *r->s3, 0, 10, -10);
	test(*r->s3, 10, 1, *r->s3, 0, 11, -10);
	test(*r->s3, 10, 1, *r->s3, 1, 0, 0);
	test(*r->s3, 10, 1, *r->s3, 1, 1, -1);
	test(*r->s3, 10, 1, *r->s3, 1, 4, -4);
	test(*r->s3, 10, 1, *r->s3, 1, 8, -8);
	test(*r->s3, 10, 1, *r->s3, 1, 9, -9);
	test(*r->s3, 10, 1, *r->s3, 1, 10, -9);
	test(*r->s3, 10, 1, *r->s3, 5, 0, 0);
	test(*r->s3, 10, 1, *r->s3, 5, 1, -1);
	test(*r->s3, 10, 1, *r->s3, 5, 2, -2);
	test(*r->s3, 10, 1, *r->s3, 5, 4, -4);
	test(*r->s3, 10, 1, *r->s3, 5, 5, -5);
	test(*r->s3, 10, 1, *r->s3, 5, 6, -5);
	test(*r->s3, 10, 1, *r->s3, 9, 0, 0);
	test(*r->s3, 10, 1, *r->s3, 9, 1, -1);
	test(*r->s3, 10, 1, *r->s3, 9, 2, -1);
	test(*r->s3, 10, 1, *r->s3, 10, 0, 0);
}

template <class S>
void
test36(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s3, 10, 1, *r->s3, 10, 1, 0);
	test(*r->s3, 10, 1, *r->s3, 11, 0, 0);
	test(*r->s3, 10, 1, *r->s4, 0, 0, 0);
	test(*r->s3, 10, 1, *r->s4, 0, 1, -1);
	test(*r->s3, 10, 1, *r->s4, 0, 10, -10);
	test(*r->s3, 10, 1, *r->s4, 0, 19, -19);
	test(*r->s3, 10, 1, *r->s4, 0, 20, -20);
	test(*r->s3, 10, 1, *r->s4, 0, 21, -20);
	test(*r->s3, 10, 1, *r->s4, 1, 0, 0);
	test(*r->s3, 10, 1, *r->s4, 1, 1, -1);
	test(*r->s3, 10, 1, *r->s4, 1, 9, -9);
	test(*r->s3, 10, 1, *r->s4, 1, 18, -18);
	test(*r->s3, 10, 1, *r->s4, 1, 19, -19);
	test(*r->s3, 10, 1, *r->s4, 1, 20, -19);
	test(*r->s3, 10, 1, *r->s4, 10, 0, 0);
	test(*r->s3, 10, 1, *r->s4, 10, 1, -1);
	test(*r->s3, 10, 1, *r->s4, 10, 5, -5);
	test(*r->s3, 10, 1, *r->s4, 10, 9, -9);
	test(*r->s3, 10, 1, *r->s4, 10, 10, -10);
	test(*r->s3, 10, 1, *r->s4, 10, 11, -10);
	test(*r->s3, 10, 1, *r->s4, 19, 0, 0);
	test(*r->s3, 10, 1, *r->s4, 19, 1, -1);
	test(*r->s3, 10, 1, *r->s4, 19, 2, -1);
	test(*r->s3, 10, 1, *r->s4, 20, 0, 0);
	test(*r->s3, 10, 1, *r->s4, 20, 1, 0);
	test(*r->s3, 10, 1, *r->s4, 21, 0, 0);
	test(*r->s3, 11, 0, *r->s1, 0, 0, 0);
	test(*r->s3, 11, 0, *r->s1, 0, 1, 0);
	test(*r->s3, 11, 0, *r->s1, 1, 0, 0);
	test(*r->s3, 11, 0, *r->s2, 0, 0, 0);
	test(*r->s3, 11, 0, *r->s2, 0, 1, 0);
	test(*r->s3, 11, 0, *r->s2, 0, 2, 0);
	test(*r->s3, 11, 0, *r->s2, 0, 4, 0);
	test(*r->s3, 11, 0, *r->s2, 0, 5, 0);
	test(*r->s3, 11, 0, *r->s2, 0, 6, 0);
	test(*r->s3, 11, 0, *r->s2, 1, 0, 0);
	test(*r->s3, 11, 0, *r->s2, 1, 1, 0);
	test(*r->s3, 11, 0, *r->s2, 1, 2, 0);
	test(*r->s3, 11, 0, *r->s2, 1, 3, 0);
	test(*r->s3, 11, 0, *r->s2, 1, 4, 0);
	test(*r->s3, 11, 0, *r->s2, 1, 5, 0);
	test(*r->s3, 11, 0, *r->s2, 2, 0, 0);
	test(*r->s3, 11, 0, *r->s2, 2, 1, 0);
	test(*r->s3, 11, 0, *r->s2, 2, 2, 0);
	test(*r->s3, 11, 0, *r->s2, 2, 3, 0);
	test(*r->s3, 11, 0, *r->s2, 2, 4, 0);
	test(*r->s3, 11, 0, *r->s2, 4, 0, 0);
	test(*r->s3, 11, 0, *r->s2, 4, 1, 0);
	test(*r->s3, 11, 0, *r->s2, 4, 2, 0);
	test(*r->s3, 11, 0, *r->s2, 5, 0, 0);
	test(*r->s3, 11, 0, *r->s2, 5, 1, 0);
	test(*r->s3, 11, 0, *r->s2, 6, 0, 0);
	test(*r->s3, 11, 0, *r->s3, 0, 0, 0);
	test(*r->s3, 11, 0, *r->s3, 0, 1, 0);
	test(*r->s3, 11, 0, *r->s3, 0, 5, 0);
	test(*r->s3, 11, 0, *r->s3, 0, 9, 0);
	test(*r->s3, 11, 0, *r->s3, 0, 10, 0);
	test(*r->s3, 11, 0, *r->s3, 0, 11, 0);
	test(*r->s3, 11, 0, *r->s3, 1, 0, 0);
	test(*r->s3, 11, 0, *r->s3, 1, 1, 0);
	test(*r->s3, 11, 0, *r->s3, 1, 4, 0);
	test(*r->s3, 11, 0, *r->s3, 1, 8, 0);
	test(*r->s3, 11, 0, *r->s3, 1, 9, 0);
	test(*r->s3, 11, 0, *r->s3, 1, 10, 0);
	test(*r->s3, 11, 0, *r->s3, 5, 0, 0);
	test(*r->s3, 11, 0, *r->s3, 5, 1, 0);
	test(*r->s3, 11, 0, *r->s3, 5, 2, 0);
	test(*r->s3, 11, 0, *r->s3, 5, 4, 0);
	test(*r->s3, 11, 0, *r->s3, 5, 5, 0);
	test(*r->s3, 11, 0, *r->s3, 5, 6, 0);
	test(*r->s3, 11, 0, *r->s3, 9, 0, 0);
	test(*r->s3, 11, 0, *r->s3, 9, 1, 0);
	test(*r->s3, 11, 0, *r->s3, 9, 2, 0);
	test(*r->s3, 11, 0, *r->s3, 10, 0, 0);
	test(*r->s3, 11, 0, *r->s3, 10, 1, 0);
	test(*r->s3, 11, 0, *r->s3, 11, 0, 0);
	test(*r->s3, 11, 0, *r->s4, 0, 0, 0);
	test(*r->s3, 11, 0, *r->s4, 0, 1, 0);
	test(*r->s3, 11, 0, *r->s4, 0, 10, 0);
	test(*r->s3, 11, 0, *r->s4, 0, 19, 0);
	test(*r->s3, 11, 0, *r->s4, 0, 20, 0);
	test(*r->s3, 11, 0, *r->s4, 0, 21, 0);
	test(*r->s3, 11, 0, *r->s4, 1, 0, 0);
	test(*r->s3, 11, 0, *r->s4, 1, 1, 0);
	test(*r->s3, 11, 0, *r->s4, 1, 9, 0);
	test(*r->s3, 11, 0, *r->s4, 1, 18, 0);
	test(*r->s3, 11, 0, *r->s4, 1, 19, 0);
	test(*r->s3, 11, 0, *r->s4, 1, 20, 0);
	test(*r->s3, 11, 0, *r->s4, 10, 0, 0);
	test(*r->s3, 11, 0, *r->s4, 10, 1, 0);
	test(*r->s3, 11, 0, *r->s4, 10, 5, 0);
	test(*r->s3, 11, 0, *r->s4, 10, 9, 0);
	test(*r->s3, 11, 0, *r->s4, 10, 10, 0);
	test(*r->s3, 11, 0, *r->s4, 10, 11, 0);
	test(*r->s3, 11, 0, *r->s4, 19, 0, 0);
	test(*r->s3, 11, 0, *r->s4, 19, 1, 0);
	test(*r->s3, 11, 0, *r->s4, 19, 2, 0);
	test(*r->s3, 11, 0, *r->s4, 20, 0, 0);
	test(*r->s3, 11, 0, *r->s4, 20, 1, 0);
	test(*r->s3, 11, 0, *r->s4, 21, 0, 0);
}

template <class S>
void
test37(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 0, 0, *r->s1, 0, 0, 0);
	test(*r->s4, 0, 0, *r->s1, 0, 1, 0);
	test(*r->s4, 0, 0, *r->s1, 1, 0, 0);
	test(*r->s4, 0, 0, *r->s2, 0, 0, 0);
	test(*r->s4, 0, 0, *r->s2, 0, 1, -1);
	test(*r->s4, 0, 0, *r->s2, 0, 2, -2);
	test(*r->s4, 0, 0, *r->s2, 0, 4, -4);
	test(*r->s4, 0, 0, *r->s2, 0, 5, -5);
	test(*r->s4, 0, 0, *r->s2, 0, 6, -5);
	test(*r->s4, 0, 0, *r->s2, 1, 0, 0);
	test(*r->s4, 0, 0, *r->s2, 1, 1, -1);
	test(*r->s4, 0, 0, *r->s2, 1, 2, -2);
	test(*r->s4, 0, 0, *r->s2, 1, 3, -3);
	test(*r->s4, 0, 0, *r->s2, 1, 4, -4);
	test(*r->s4, 0, 0, *r->s2, 1, 5, -4);
	test(*r->s4, 0, 0, *r->s2, 2, 0, 0);
	test(*r->s4, 0, 0, *r->s2, 2, 1, -1);
	test(*r->s4, 0, 0, *r->s2, 2, 2, -2);
	test(*r->s4, 0, 0, *r->s2, 2, 3, -3);
	test(*r->s4, 0, 0, *r->s2, 2, 4, -3);
	test(*r->s4, 0, 0, *r->s2, 4, 0, 0);
	test(*r->s4, 0, 0, *r->s2, 4, 1, -1);
	test(*r->s4, 0, 0, *r->s2, 4, 2, -1);
	test(*r->s4, 0, 0, *r->s2, 5, 0, 0);
	test(*r->s4, 0, 0, *r->s2, 5, 1, 0);
	test(*r->s4, 0, 0, *r->s2, 6, 0, 0);
	test(*r->s4, 0, 0, *r->s3, 0, 0, 0);
	test(*r->s4, 0, 0, *r->s3, 0, 1, -1);
	test(*r->s4, 0, 0, *r->s3, 0, 5, -5);
	test(*r->s4, 0, 0, *r->s3, 0, 9, -9);
	test(*r->s4, 0, 0, *r->s3, 0, 10, -10);
	test(*r->s4, 0, 0, *r->s3, 0, 11, -10);
	test(*r->s4, 0, 0, *r->s3, 1, 0, 0);
	test(*r->s4, 0, 0, *r->s3, 1, 1, -1);
	test(*r->s4, 0, 0, *r->s3, 1, 4, -4);
	test(*r->s4, 0, 0, *r->s3, 1, 8, -8);
	test(*r->s4, 0, 0, *r->s3, 1, 9, -9);
	test(*r->s4, 0, 0, *r->s3, 1, 10, -9);
	test(*r->s4, 0, 0, *r->s3, 5, 0, 0);
	test(*r->s4, 0, 0, *r->s3, 5, 1, -1);
	test(*r->s4, 0, 0, *r->s3, 5, 2, -2);
	test(*r->s4, 0, 0, *r->s3, 5, 4, -4);
	test(*r->s4, 0, 0, *r->s3, 5, 5, -5);
	test(*r->s4, 0, 0, *r->s3, 5, 6, -5);
	test(*r->s4, 0, 0, *r->s3, 9, 0, 0);
	test(*r->s4, 0, 0, *r->s3, 9, 1, -1);
	test(*r->s4, 0, 0, *r->s3, 9, 2, -1);
	test(*r->s4, 0, 0, *r->s3, 10, 0, 0);
	test(*r->s4, 0, 0, *r->s3, 10, 1, 0);
	test(*r->s4, 0, 0, *r->s3, 11, 0, 0);
	test(*r->s4, 0, 0, *r->s4, 0, 0, 0);
	test(*r->s4, 0, 0, *r->s4, 0, 1, -1);
	test(*r->s4, 0, 0, *r->s4, 0, 10, -10);
	test(*r->s4, 0, 0, *r->s4, 0, 19, -19);
	test(*r->s4, 0, 0, *r->s4, 0, 20, -20);
	test(*r->s4, 0, 0, *r->s4, 0, 21, -20);
	test(*r->s4, 0, 0, *r->s4, 1, 0, 0);
	test(*r->s4, 0, 0, *r->s4, 1, 1, -1);
	test(*r->s4, 0, 0, *r->s4, 1, 9, -9);
	test(*r->s4, 0, 0, *r->s4, 1, 18, -18);
	test(*r->s4, 0, 0, *r->s4, 1, 19, -19);
	test(*r->s4, 0, 0, *r->s4, 1, 20, -19);
	test(*r->s4, 0, 0, *r->s4, 10, 0, 0);
	test(*r->s4, 0, 0, *r->s4, 10, 1, -1);
	test(*r->s4, 0, 0, *r->s4, 10, 5, -5);
	test(*r->s4, 0, 0, *r->s4, 10, 9, -9);
	test(*r->s4, 0, 0, *r->s4, 10, 10, -10);
	test(*r->s4, 0, 0, *r->s4, 10, 11, -10);
	test(*r->s4, 0, 0, *r->s4, 19, 0, 0);
	test(*r->s4, 0, 0, *r->s4, 19, 1, -1);
	test(*r->s4, 0, 0, *r->s4, 19, 2, -1);
	test(*r->s4, 0, 0, *r->s4, 20, 0, 0);
	test(*r->s4, 0, 0, *r->s4, 20, 1, 0);
	test(*r->s4, 0, 0, *r->s4, 21, 0, 0);
	test(*r->s4, 0, 1, *r->s1, 0, 0, 1);
	test(*r->s4, 0, 1, *r->s1, 0, 1, 1);
	test(*r->s4, 0, 1, *r->s1, 1, 0, 0);
	test(*r->s4, 0, 1, *r->s2, 0, 0, 1);
	test(*r->s4, 0, 1, *r->s2, 0, 1, 0);
	test(*r->s4, 0, 1, *r->s2, 0, 2, -1);
	test(*r->s4, 0, 1, *r->s2, 0, 4, -3);
	test(*r->s4, 0, 1, *r->s2, 0, 5, -4);
	test(*r->s4, 0, 1, *r->s2, 0, 6, -4);
	test(*r->s4, 0, 1, *r->s2, 1, 0, 1);
	test(*r->s4, 0, 1, *r->s2, 1, 1, -1);
	test(*r->s4, 0, 1, *r->s2, 1, 2, -1);
	test(*r->s4, 0, 1, *r->s2, 1, 3, -1);
	test(*r->s4, 0, 1, *r->s2, 1, 4, -1);
	test(*r->s4, 0, 1, *r->s2, 1, 5, -1);
	test(*r->s4, 0, 1, *r->s2, 2, 0, 1);
	test(*r->s4, 0, 1, *r->s2, 2, 1, -2);
	test(*r->s4, 0, 1, *r->s2, 2, 2, -2);
	test(*r->s4, 0, 1, *r->s2, 2, 3, -2);
	test(*r->s4, 0, 1, *r->s2, 2, 4, -2);
	test(*r->s4, 0, 1, *r->s2, 4, 0, 1);
	test(*r->s4, 0, 1, *r->s2, 4, 1, -4);
	test(*r->s4, 0, 1, *r->s2, 4, 2, -4);
	test(*r->s4, 0, 1, *r->s2, 5, 0, 1);
	test(*r->s4, 0, 1, *r->s2, 5, 1, 1);
	test(*r->s4, 0, 1, *r->s2, 6, 0, 0);
}

template <class S>
void
test38(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 0, 1, *r->s3, 0, 0, 1);
	test(*r->s4, 0, 1, *r->s3, 0, 1, 0);
	test(*r->s4, 0, 1, *r->s3, 0, 5, -4);
	test(*r->s4, 0, 1, *r->s3, 0, 9, -8);
	test(*r->s4, 0, 1, *r->s3, 0, 10, -9);
	test(*r->s4, 0, 1, *r->s3, 0, 11, -9);
	test(*r->s4, 0, 1, *r->s3, 1, 0, 1);
	test(*r->s4, 0, 1, *r->s3, 1, 1, -1);
	test(*r->s4, 0, 1, *r->s3, 1, 4, -1);
	test(*r->s4, 0, 1, *r->s3, 1, 8, -1);
	test(*r->s4, 0, 1, *r->s3, 1, 9, -1);
	test(*r->s4, 0, 1, *r->s3, 1, 10, -1);
	test(*r->s4, 0, 1, *r->s3, 5, 0, 1);
	test(*r->s4, 0, 1, *r->s3, 5, 1, -5);
	test(*r->s4, 0, 1, *r->s3, 5, 2, -5);
	test(*r->s4, 0, 1, *r->s3, 5, 4, -5);
	test(*r->s4, 0, 1, *r->s3, 5, 5, -5);
	test(*r->s4, 0, 1, *r->s3, 5, 6, -5);
	test(*r->s4, 0, 1, *r->s3, 9, 0, 1);
	test(*r->s4, 0, 1, *r->s3, 9, 1, -9);
	test(*r->s4, 0, 1, *r->s3, 9, 2, -9);
	test(*r->s4, 0, 1, *r->s3, 10, 0, 1);
	test(*r->s4, 0, 1, *r->s3, 10, 1, 1);
	test(*r->s4, 0, 1, *r->s3, 11, 0, 0);
	test(*r->s4, 0, 1, *r->s4, 0, 0, 1);
	test(*r->s4, 0, 1, *r->s4, 0, 1, 0);
	test(*r->s4, 0, 1, *r->s4, 0, 10, -9);
	test(*r->s4, 0, 1, *r->s4, 0, 19, -18);
	test(*r->s4, 0, 1, *r->s4, 0, 20, -19);
	test(*r->s4, 0, 1, *r->s4, 0, 21, -19);
	test(*r->s4, 0, 1, *r->s4, 1, 0, 1);
	test(*r->s4, 0, 1, *r->s4, 1, 1, -1);
	test(*r->s4, 0, 1, *r->s4, 1, 9, -1);
	test(*r->s4, 0, 1, *r->s4, 1, 18, -1);
	test(*r->s4, 0, 1, *r->s4, 1, 19, -1);
	test(*r->s4, 0, 1, *r->s4, 1, 20, -1);
	test(*r->s4, 0, 1, *r->s4, 10, 0, 1);
	test(*r->s4, 0, 1, *r->s4, 10, 1, -10);
	test(*r->s4, 0, 1, *r->s4, 10, 5, -10);
	test(*r->s4, 0, 1, *r->s4, 10, 9, -10);
	test(*r->s4, 0, 1, *r->s4, 10, 10, -10);
	test(*r->s4, 0, 1, *r->s4, 10, 11, -10);
	test(*r->s4, 0, 1, *r->s4, 19, 0, 1);
	test(*r->s4, 0, 1, *r->s4, 19, 1, -19);
	test(*r->s4, 0, 1, *r->s4, 19, 2, -19);
	test(*r->s4, 0, 1, *r->s4, 20, 0, 1);
	test(*r->s4, 0, 1, *r->s4, 20, 1, 1);
	test(*r->s4, 0, 1, *r->s4, 21, 0, 0);
	test(*r->s4, 0, 10, *r->s1, 0, 0, 10);
	test(*r->s4, 0, 10, *r->s1, 0, 1, 10);
	test(*r->s4, 0, 10, *r->s1, 1, 0, 0);
	test(*r->s4, 0, 10, *r->s2, 0, 0, 10);
	test(*r->s4, 0, 10, *r->s2, 0, 1, 9);
	test(*r->s4, 0, 10, *r->s2, 0, 2, 8);
	test(*r->s4, 0, 10, *r->s2, 0, 4, 6);
	test(*r->s4, 0, 10, *r->s2, 0, 5, 5);
	test(*r->s4, 0, 10, *r->s2, 0, 6, 5);
	test(*r->s4, 0, 10, *r->s2, 1, 0, 10);
	test(*r->s4, 0, 10, *r->s2, 1, 1, -1);
	test(*r->s4, 0, 10, *r->s2, 1, 2, -1);
	test(*r->s4, 0, 10, *r->s2, 1, 3, -1);
	test(*r->s4, 0, 10, *r->s2, 1, 4, -1);
	test(*r->s4, 0, 10, *r->s2, 1, 5, -1);
	test(*r->s4, 0, 10, *r->s2, 2, 0, 10);
	test(*r->s4, 0, 10, *r->s2, 2, 1, -2);
	test(*r->s4, 0, 10, *r->s2, 2, 2, -2);
	test(*r->s4, 0, 10, *r->s2, 2, 3, -2);
	test(*r->s4, 0, 10, *r->s2, 2, 4, -2);
	test(*r->s4, 0, 10, *r->s2, 4, 0, 10);
	test(*r->s4, 0, 10, *r->s2, 4, 1, -4);
	test(*r->s4, 0, 10, *r->s2, 4, 2, -4);
	test(*r->s4, 0, 10, *r->s2, 5, 0, 10);
	test(*r->s4, 0, 10, *r->s2, 5, 1, 10);
	test(*r->s4, 0, 10, *r->s2, 6, 0, 0);
	test(*r->s4, 0, 10, *r->s3, 0, 0, 10);
	test(*r->s4, 0, 10, *r->s3, 0, 1, 9);
	test(*r->s4, 0, 10, *r->s3, 0, 5, 5);
	test(*r->s4, 0, 10, *r->s3, 0, 9, 1);
	test(*r->s4, 0, 10, *r->s3, 0, 10, 0);
	test(*r->s4, 0, 10, *r->s3, 0, 11, 0);
	test(*r->s4, 0, 10, *r->s3, 1, 0, 10);
	test(*r->s4, 0, 10, *r->s3, 1, 1, -1);
	test(*r->s4, 0, 10, *r->s3, 1, 4, -1);
	test(*r->s4, 0, 10, *r->s3, 1, 8, -1);
	test(*r->s4, 0, 10, *r->s3, 1, 9, -1);
	test(*r->s4, 0, 10, *r->s3, 1, 10, -1);
	test(*r->s4, 0, 10, *r->s3, 5, 0, 10);
	test(*r->s4, 0, 10, *r->s3, 5, 1, -5);
	test(*r->s4, 0, 10, *r->s3, 5, 2, -5);
	test(*r->s4, 0, 10, *r->s3, 5, 4, -5);
	test(*r->s4, 0, 10, *r->s3, 5, 5, -5);
	test(*r->s4, 0, 10, *r->s3, 5, 6, -5);
	test(*r->s4, 0, 10, *r->s3, 9, 0, 10);
	test(*r->s4, 0, 10, *r->s3, 9, 1, -9);
	test(*r->s4, 0, 10, *r->s3, 9, 2, -9);
	test(*r->s4, 0, 10, *r->s3, 10, 0, 10);
	test(*r->s4, 0, 10, *r->s3, 10, 1, 10);
	test(*r->s4, 0, 10, *r->s3, 11, 0, 0);
	test(*r->s4, 0, 10, *r->s4, 0, 0, 10);
	test(*r->s4, 0, 10, *r->s4, 0, 1, 9);
}

template <class S>
void
test39(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 0, 10, *r->s4, 0, 10, 0);
	test(*r->s4, 0, 10, *r->s4, 0, 19, -9);
	test(*r->s4, 0, 10, *r->s4, 0, 20, -10);
	test(*r->s4, 0, 10, *r->s4, 0, 21, -10);
	test(*r->s4, 0, 10, *r->s4, 1, 0, 10);
	test(*r->s4, 0, 10, *r->s4, 1, 1, -1);
	test(*r->s4, 0, 10, *r->s4, 1, 9, -1);
	test(*r->s4, 0, 10, *r->s4, 1, 18, -1);
	test(*r->s4, 0, 10, *r->s4, 1, 19, -1);
	test(*r->s4, 0, 10, *r->s4, 1, 20, -1);
	test(*r->s4, 0, 10, *r->s4, 10, 0, 10);
	test(*r->s4, 0, 10, *r->s4, 10, 1, -10);
	test(*r->s4, 0, 10, *r->s4, 10, 5, -10);
	test(*r->s4, 0, 10, *r->s4, 10, 9, -10);
	test(*r->s4, 0, 10, *r->s4, 10, 10, -10);
	test(*r->s4, 0, 10, *r->s4, 10, 11, -10);
	test(*r->s4, 0, 10, *r->s4, 19, 0, 10);
	test(*r->s4, 0, 10, *r->s4, 19, 1, -19);
	test(*r->s4, 0, 10, *r->s4, 19, 2, -19);
	test(*r->s4, 0, 10, *r->s4, 20, 0, 10);
	test(*r->s4, 0, 10, *r->s4, 20, 1, 10);
	test(*r->s4, 0, 10, *r->s4, 21, 0, 0);
	test(*r->s4, 0, 19, *r->s1, 0, 0, 19);
	test(*r->s4, 0, 19, *r->s1, 0, 1, 19);
	test(*r->s4, 0, 19, *r->s1, 1, 0, 0);
	test(*r->s4, 0, 19, *r->s2, 0, 0, 19);
	test(*r->s4, 0, 19, *r->s2, 0, 1, 18);
	test(*r->s4, 0, 19, *r->s2, 0, 2, 17);
	test(*r->s4, 0, 19, *r->s2, 0, 4, 15);
	test(*r->s4, 0, 19, *r->s2, 0, 5, 14);
	test(*r->s4, 0, 19, *r->s2, 0, 6, 14);
	test(*r->s4, 0, 19, *r->s2, 1, 0, 19);
	test(*r->s4, 0, 19, *r->s2, 1, 1, -1);
	test(*r->s4, 0, 19, *r->s2, 1, 2, -1);
	test(*r->s4, 0, 19, *r->s2, 1, 3, -1);
	test(*r->s4, 0, 19, *r->s2, 1, 4, -1);
	test(*r->s4, 0, 19, *r->s2, 1, 5, -1);
	test(*r->s4, 0, 19, *r->s2, 2, 0, 19);
	test(*r->s4, 0, 19, *r->s2, 2, 1, -2);
	test(*r->s4, 0, 19, *r->s2, 2, 2, -2);
	test(*r->s4, 0, 19, *r->s2, 2, 3, -2);
	test(*r->s4, 0, 19, *r->s2, 2, 4, -2);
	test(*r->s4, 0, 19, *r->s2, 4, 0, 19);
	test(*r->s4, 0, 19, *r->s2, 4, 1, -4);
	test(*r->s4, 0, 19, *r->s2, 4, 2, -4);
	test(*r->s4, 0, 19, *r->s2, 5, 0, 19);
	test(*r->s4, 0, 19, *r->s2, 5, 1, 19);
	test(*r->s4, 0, 19, *r->s2, 6, 0, 0);
	test(*r->s4, 0, 19, *r->s3, 0, 0, 19);
	test(*r->s4, 0, 19, *r->s3, 0, 1, 18);
	test(*r->s4, 0, 19, *r->s3, 0, 5, 14);
	test(*r->s4, 0, 19, *r->s3, 0, 9, 10);
	test(*r->s4, 0, 19, *r->s3, 0, 10, 9);
	test(*r->s4, 0, 19, *r->s3, 0, 11, 9);
	test(*r->s4, 0, 19, *r->s3, 1, 0, 19);
	test(*r->s4, 0, 19, *r->s3, 1, 1, -1);
	test(*r->s4, 0, 19, *r->s3, 1, 4, -1);
	test(*r->s4, 0, 19, *r->s3, 1, 8, -1);
	test(*r->s4, 0, 19, *r->s3, 1, 9, -1);
	test(*r->s4, 0, 19, *r->s3, 1, 10, -1);
	test(*r->s4, 0, 19, *r->s3, 5, 0, 19);
	test(*r->s4, 0, 19, *r->s3, 5, 1, -5);
	test(*r->s4, 0, 19, *r->s3, 5, 2, -5);
	test(*r->s4, 0, 19, *r->s3, 5, 4, -5);
	test(*r->s4, 0, 19, *r->s3, 5, 5, -5);
	test(*r->s4, 0, 19, *r->s3, 5, 6, -5);
	test(*r->s4, 0, 19, *r->s3, 9, 0, 19);
	test(*r->s4, 0, 19, *r->s3, 9, 1, -9);
	test(*r->s4, 0, 19, *r->s3, 9, 2, -9);
	test(*r->s4, 0, 19, *r->s3, 10, 0, 19);
	test(*r->s4, 0, 19, *r->s3, 10, 1, 19);
	test(*r->s4, 0, 19, *r->s3, 11, 0, 0);
	test(*r->s4, 0, 19, *r->s4, 0, 0, 19);
	test(*r->s4, 0, 19, *r->s4, 0, 1, 18);
	test(*r->s4, 0, 19, *r->s4, 0, 10, 9);
	test(*r->s4, 0, 19, *r->s4, 0, 19, 0);
	test(*r->s4, 0, 19, *r->s4, 0, 20, -1);
	test(*r->s4, 0, 19, *r->s4, 0, 21, -1);
	test(*r->s4, 0, 19, *r->s4, 1, 0, 19);
	test(*r->s4, 0, 19, *r->s4, 1, 1, -1);
	test(*r->s4, 0, 19, *r->s4, 1, 9, -1);
	test(*r->s4, 0, 19, *r->s4, 1, 18, -1);
	test(*r->s4, 0, 19, *r->s4, 1, 19, -1);
	test(*r->s4, 0, 19, *r->s4, 1, 20, -1);
	test(*r->s4, 0, 19, *r->s4, 10, 0, 19);
	test(*r->s4, 0, 19, *r->s4, 10, 1, -10);
	test(*r->s4, 0, 19, *r->s4, 10, 5, -10);
	test(*r->s4, 0, 19, *r->s4, 10, 9, -10);
	test(*r->s4, 0, 19, *r->s4, 10, 10, -10);
	test(*r->s4, 0, 19, *r->s4, 10, 11, -10);
	test(*r->s4, 0, 19, *r->s4, 19, 0, 19);
	test(*r->s4, 0, 19, *r->s4, 19, 1, -19);
	test(*r->s4, 0, 19, *r->s4, 19, 2, -19);
	test(*r->s4, 0, 19, *r->s4, 20, 0, 19);
	test(*r->s4, 0, 19, *r->s4, 20, 1, 19);
	test(*r->s4, 0, 19, *r->s4, 21, 0, 0);
	test(*r->s4, 0, 20, *r->s1, 0, 0, 20);
	test(*r->s4, 0, 20, *r->s1, 0, 1, 20);
	test(*r->s4, 0, 20, *r->s1, 1, 0, 0);
	test(*r->s4, 0, 20, *r->s2, 0, 0, 20);
}

template <class S>
void
test40(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 0, 20, *r->s2, 0, 1, 19);
	test(*r->s4, 0, 20, *r->s2, 0, 2, 18);
	test(*r->s4, 0, 20, *r->s2, 0, 4, 16);
	test(*r->s4, 0, 20, *r->s2, 0, 5, 15);
	test(*r->s4, 0, 20, *r->s2, 0, 6, 15);
	test(*r->s4, 0, 20, *r->s2, 1, 0, 20);
	test(*r->s4, 0, 20, *r->s2, 1, 1, -1);
	test(*r->s4, 0, 20, *r->s2, 1, 2, -1);
	test(*r->s4, 0, 20, *r->s2, 1, 3, -1);
	test(*r->s4, 0, 20, *r->s2, 1, 4, -1);
	test(*r->s4, 0, 20, *r->s2, 1, 5, -1);
	test(*r->s4, 0, 20, *r->s2, 2, 0, 20);
	test(*r->s4, 0, 20, *r->s2, 2, 1, -2);
	test(*r->s4, 0, 20, *r->s2, 2, 2, -2);
	test(*r->s4, 0, 20, *r->s2, 2, 3, -2);
	test(*r->s4, 0, 20, *r->s2, 2, 4, -2);
	test(*r->s4, 0, 20, *r->s2, 4, 0, 20);
	test(*r->s4, 0, 20, *r->s2, 4, 1, -4);
	test(*r->s4, 0, 20, *r->s2, 4, 2, -4);
	test(*r->s4, 0, 20, *r->s2, 5, 0, 20);
	test(*r->s4, 0, 20, *r->s2, 5, 1, 20);
	test(*r->s4, 0, 20, *r->s2, 6, 0, 0);
	test(*r->s4, 0, 20, *r->s3, 0, 0, 20);
	test(*r->s4, 0, 20, *r->s3, 0, 1, 19);
	test(*r->s4, 0, 20, *r->s3, 0, 5, 15);
	test(*r->s4, 0, 20, *r->s3, 0, 9, 11);
	test(*r->s4, 0, 20, *r->s3, 0, 10, 10);
	test(*r->s4, 0, 20, *r->s3, 0, 11, 10);
	test(*r->s4, 0, 20, *r->s3, 1, 0, 20);
	test(*r->s4, 0, 20, *r->s3, 1, 1, -1);
	test(*r->s4, 0, 20, *r->s3, 1, 4, -1);
	test(*r->s4, 0, 20, *r->s3, 1, 8, -1);
	test(*r->s4, 0, 20, *r->s3, 1, 9, -1);
	test(*r->s4, 0, 20, *r->s3, 1, 10, -1);
	test(*r->s4, 0, 20, *r->s3, 5, 0, 20);
	test(*r->s4, 0, 20, *r->s3, 5, 1, -5);
	test(*r->s4, 0, 20, *r->s3, 5, 2, -5);
	test(*r->s4, 0, 20, *r->s3, 5, 4, -5);
	test(*r->s4, 0, 20, *r->s3, 5, 5, -5);
	test(*r->s4, 0, 20, *r->s3, 5, 6, -5);
	test(*r->s4, 0, 20, *r->s3, 9, 0, 20);
	test(*r->s4, 0, 20, *r->s3, 9, 1, -9);
	test(*r->s4, 0, 20, *r->s3, 9, 2, -9);
	test(*r->s4, 0, 20, *r->s3, 10, 0, 20);
	test(*r->s4, 0, 20, *r->s3, 10, 1, 20);
	test(*r->s4, 0, 20, *r->s3, 11, 0, 0);
	test(*r->s4, 0, 20, *r->s4, 0, 0, 20);
	test(*r->s4, 0, 20, *r->s4, 0, 1, 19);
	test(*r->s4, 0, 20, *r->s4, 0, 10, 10);
	test(*r->s4, 0, 20, *r->s4, 0, 19, 1);
	test(*r->s4, 0, 20, *r->s4, 0, 20, 0);
	test(*r->s4, 0, 20, *r->s4, 0, 21, 0);
	test(*r->s4, 0, 20, *r->s4, 1, 0, 20);
	test(*r->s4, 0, 20, *r->s4, 1, 1, -1);
	test(*r->s4, 0, 20, *r->s4, 1, 9, -1);
	test(*r->s4, 0, 20, *r->s4, 1, 18, -1);
	test(*r->s4, 0, 20, *r->s4, 1, 19, -1);
	test(*r->s4, 0, 20, *r->s4, 1, 20, -1);
	test(*r->s4, 0, 20, *r->s4, 10, 0, 20);
	test(*r->s4, 0, 20, *r->s4, 10, 1, -10);
	test(*r->s4, 0, 20, *r->s4, 10, 5, -10);
	test(*r->s4, 0, 20, *r->s4, 10, 9, -10);
	test(*r->s4, 0, 20, *r->s4, 10, 10, -10);
	test(*r->s4, 0, 20, *r->s4, 10, 11, -10);
	test(*r->s4, 0, 20, *r->s4, 19, 0, 20);
	test(*r->s4, 0, 20, *r->s4, 19, 1, -19);
	test(*r->s4, 0, 20, *r->s4, 19, 2, -19);
	test(*r->s4, 0, 20, *r->s4, 20, 0, 20);
	test(*r->s4, 0, 20, *r->s4, 20, 1, 20);
	test(*r->s4, 0, 20, *r->s4, 21, 0, 0);
	test(*r->s4, 0, 21, *r->s1, 0, 0, 20);
	test(*r->s4, 0, 21, *r->s1, 0, 1, 20);
	test(*r->s4, 0, 21, *r->s1, 1, 0, 0);
	test(*r->s4, 0, 21, *r->s2, 0, 0, 20);
	test(*r->s4, 0, 21, *r->s2, 0, 1, 19);
	test(*r->s4, 0, 21, *r->s2, 0, 2, 18);
	test(*r->s4, 0, 21, *r->s2, 0, 4, 16);
	test(*r->s4, 0, 21, *r->s2, 0, 5, 15);
	test(*r->s4, 0, 21, *r->s2, 0, 6, 15);
	test(*r->s4, 0, 21, *r->s2, 1, 0, 20);
	test(*r->s4, 0, 21, *r->s2, 1, 1, -1);
	test(*r->s4, 0, 21, *r->s2, 1, 2, -1);
	test(*r->s4, 0, 21, *r->s2, 1, 3, -1);
	test(*r->s4, 0, 21, *r->s2, 1, 4, -1);
	test(*r->s4, 0, 21, *r->s2, 1, 5, -1);
	test(*r->s4, 0, 21, *r->s2, 2, 0, 20);
	test(*r->s4, 0, 21, *r->s2, 2, 1, -2);
	test(*r->s4, 0, 21, *r->s2, 2, 2, -2);
	test(*r->s4, 0, 21, *r->s2, 2, 3, -2);
	test(*r->s4, 0, 21, *r->s2, 2, 4, -2);
	test(*r->s4, 0, 21, *r->s2, 4, 0, 20);
	test(*r->s4, 0, 21, *r->s2, 4, 1, -4);
	test(*r->s4, 0, 21, *r->s2, 4, 2, -4);
	test(*r->s4, 0, 21, *r->s2, 5, 0, 20);
	test(*r->s4, 0, 21, *r->s2, 5, 1, 20);
	test(*r->s4, 0, 21, *r->s2, 6, 0, 0);
	test(*r->s4, 0, 21, *r->s3, 0, 0, 20);
	test(*r->s4, 0, 21, *r->s3, 0, 1, 19);
	test(*r->s4, 0, 21, *r->s3, 0, 5, 15);
	test(*r->s4, 0, 21, *r->s3, 0, 9, 11);
}

template <class S>
void
test41(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 0, 21, *r->s3, 0, 10, 10);
	test(*r->s4, 0, 21, *r->s3, 0, 11, 10);
	test(*r->s4, 0, 21, *r->s3, 1, 0, 20);
	test(*r->s4, 0, 21, *r->s3, 1, 1, -1);
	test(*r->s4, 0, 21, *r->s3, 1, 4, -1);
	test(*r->s4, 0, 21, *r->s3, 1, 8, -1);
	test(*r->s4, 0, 21, *r->s3, 1, 9, -1);
	test(*r->s4, 0, 21, *r->s3, 1, 10, -1);
	test(*r->s4, 0, 21, *r->s3, 5, 0, 20);
	test(*r->s4, 0, 21, *r->s3, 5, 1, -5);
	test(*r->s4, 0, 21, *r->s3, 5, 2, -5);
	test(*r->s4, 0, 21, *r->s3, 5, 4, -5);
	test(*r->s4, 0, 21, *r->s3, 5, 5, -5);
	test(*r->s4, 0, 21, *r->s3, 5, 6, -5);
	test(*r->s4, 0, 21, *r->s3, 9, 0, 20);
	test(*r->s4, 0, 21, *r->s3, 9, 1, -9);
	test(*r->s4, 0, 21, *r->s3, 9, 2, -9);
	test(*r->s4, 0, 21, *r->s3, 10, 0, 20);
	test(*r->s4, 0, 21, *r->s3, 10, 1, 20);
	test(*r->s4, 0, 21, *r->s3, 11, 0, 0);
	test(*r->s4, 0, 21, *r->s4, 0, 0, 20);
	test(*r->s4, 0, 21, *r->s4, 0, 1, 19);
	test(*r->s4, 0, 21, *r->s4, 0, 10, 10);
	test(*r->s4, 0, 21, *r->s4, 0, 19, 1);
	test(*r->s4, 0, 21, *r->s4, 0, 20, 0);
	test(*r->s4, 0, 21, *r->s4, 0, 21, 0);
	test(*r->s4, 0, 21, *r->s4, 1, 0, 20);
	test(*r->s4, 0, 21, *r->s4, 1, 1, -1);
	test(*r->s4, 0, 21, *r->s4, 1, 9, -1);
	test(*r->s4, 0, 21, *r->s4, 1, 18, -1);
	test(*r->s4, 0, 21, *r->s4, 1, 19, -1);
	test(*r->s4, 0, 21, *r->s4, 1, 20, -1);
	test(*r->s4, 0, 21, *r->s4, 10, 0, 20);
	test(*r->s4, 0, 21, *r->s4, 10, 1, -10);
	test(*r->s4, 0, 21, *r->s4, 10, 5, -10);
	test(*r->s4, 0, 21, *r->s4, 10, 9, -10);
	test(*r->s4, 0, 21, *r->s4, 10, 10, -10);
	test(*r->s4, 0, 21, *r->s4, 10, 11, -10);
	test(*r->s4, 0, 21, *r->s4, 19, 0, 20);
	test(*r->s4, 0, 21, *r->s4, 19, 1, -19);
	test(*r->s4, 0, 21, *r->s4, 19, 2, -19);
	test(*r->s4, 0, 21, *r->s4, 20, 0, 20);
	test(*r->s4, 0, 21, *r->s4, 20, 1, 20);
	test(*r->s4, 0, 21, *r->s4, 21, 0, 0);
	test(*r->s4, 1, 0, *r->s1, 0, 0, 0);
	test(*r->s4, 1, 0, *r->s1, 0, 1, 0);
	test(*r->s4, 1, 0, *r->s1, 1, 0, 0);
	test(*r->s4, 1, 0, *r->s2, 0, 0, 0);
	test(*r->s4, 1, 0, *r->s2, 0, 1, -1);
	test(*r->s4, 1, 0, *r->s2, 0, 2, -2);
	test(*r->s4, 1, 0, *r->s2, 0, 4, -4);
	test(*r->s4, 1, 0, *r->s2, 0, 5, -5);
	test(*r->s4, 1, 0, *r->s2, 0, 6, -5);
	test(*r->s4, 1, 0, *r->s2, 1, 0, 0);
	test(*r->s4, 1, 0, *r->s2, 1, 1, -1);
	test(*r->s4, 1, 0, *r->s2, 1, 2, -2);
	test(*r->s4, 1, 0, *r->s2, 1, 3, -3);
	test(*r->s4, 1, 0, *r->s2, 1, 4, -4);
	test(*r->s4, 1, 0, *r->s2, 1, 5, -4);
	test(*r->s4, 1, 0, *r->s2, 2, 0, 0);
	test(*r->s4, 1, 0, *r->s2, 2, 1, -1);
	test(*r->s4, 1, 0, *r->s2, 2, 2, -2);
	test(*r->s4, 1, 0, *r->s2, 2, 3, -3);
	test(*r->s4, 1, 0, *r->s2, 2, 4, -3);
	test(*r->s4, 1, 0, *r->s2, 4, 0, 0);
	test(*r->s4, 1, 0, *r->s2, 4, 1, -1);
	test(*r->s4, 1, 0, *r->s2, 4, 2, -1);
	test(*r->s4, 1, 0, *r->s2, 5, 0, 0);
	test(*r->s4, 1, 0, *r->s2, 5, 1, 0);
	test(*r->s4, 1, 0, *r->s2, 6, 0, 0);
	test(*r->s4, 1, 0, *r->s3, 0, 0, 0);
	test(*r->s4, 1, 0, *r->s3, 0, 1, -1);
	test(*r->s4, 1, 0, *r->s3, 0, 5, -5);
	test(*r->s4, 1, 0, *r->s3, 0, 9, -9);
	test(*r->s4, 1, 0, *r->s3, 0, 10, -10);
	test(*r->s4, 1, 0, *r->s3, 0, 11, -10);
	test(*r->s4, 1, 0, *r->s3, 1, 0, 0);
	test(*r->s4, 1, 0, *r->s3, 1, 1, -1);
	test(*r->s4, 1, 0, *r->s3, 1, 4, -4);
	test(*r->s4, 1, 0, *r->s3, 1, 8, -8);
	test(*r->s4, 1, 0, *r->s3, 1, 9, -9);
	test(*r->s4, 1, 0, *r->s3, 1, 10, -9);
	test(*r->s4, 1, 0, *r->s3, 5, 0, 0);
	test(*r->s4, 1, 0, *r->s3, 5, 1, -1);
	test(*r->s4, 1, 0, *r->s3, 5, 2, -2);
	test(*r->s4, 1, 0, *r->s3, 5, 4, -4);
	test(*r->s4, 1, 0, *r->s3, 5, 5, -5);
	test(*r->s4, 1, 0, *r->s3, 5, 6, -5);
	test(*r->s4, 1, 0, *r->s3, 9, 0, 0);
	test(*r->s4, 1, 0, *r->s3, 9, 1, -1);
	test(*r->s4, 1, 0, *r->s3, 9, 2, -1);
	test(*r->s4, 1, 0, *r->s3, 10, 0, 0);
	test(*r->s4, 1, 0, *r->s3, 10, 1, 0);
	test(*r->s4, 1, 0, *r->s3, 11, 0, 0);
	test(*r->s4, 1, 0, *r->s4, 0, 0, 0);
	test(*r->s4, 1, 0, *r->s4, 0, 1, -1);
	test(*r->s4, 1, 0, *r->s4, 0, 10, -10);
	test(*r->s4, 1, 0, *r->s4, 0, 19, -19);
	test(*r->s4, 1, 0, *r->s4, 0, 20, -20);
	test(*r->s4, 1, 0, *r->s4, 0, 21, -20);
}

template <class S>
void
test42(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 1, 0, *r->s4, 1, 0, 0);
	test(*r->s4, 1, 0, *r->s4, 1, 1, -1);
	test(*r->s4, 1, 0, *r->s4, 1, 9, -9);
	test(*r->s4, 1, 0, *r->s4, 1, 18, -18);
	test(*r->s4, 1, 0, *r->s4, 1, 19, -19);
	test(*r->s4, 1, 0, *r->s4, 1, 20, -19);
	test(*r->s4, 1, 0, *r->s4, 10, 0, 0);
	test(*r->s4, 1, 0, *r->s4, 10, 1, -1);
	test(*r->s4, 1, 0, *r->s4, 10, 5, -5);
	test(*r->s4, 1, 0, *r->s4, 10, 9, -9);
	test(*r->s4, 1, 0, *r->s4, 10, 10, -10);
	test(*r->s4, 1, 0, *r->s4, 10, 11, -10);
	test(*r->s4, 1, 0, *r->s4, 19, 0, 0);
	test(*r->s4, 1, 0, *r->s4, 19, 1, -1);
	test(*r->s4, 1, 0, *r->s4, 19, 2, -1);
	test(*r->s4, 1, 0, *r->s4, 20, 0, 0);
	test(*r->s4, 1, 0, *r->s4, 20, 1, 0);
	test(*r->s4, 1, 0, *r->s4, 21, 0, 0);
	test(*r->s4, 1, 1, *r->s1, 0, 0, 1);
	test(*r->s4, 1, 1, *r->s1, 0, 1, 1);
	test(*r->s4, 1, 1, *r->s1, 1, 0, 0);
	test(*r->s4, 1, 1, *r->s2, 0, 0, 1);
	test(*r->s4, 1, 1, *r->s2, 0, 1, 1);
	test(*r->s4, 1, 1, *r->s2, 0, 2, 1);
	test(*r->s4, 1, 1, *r->s2, 0, 4, 1);
	test(*r->s4, 1, 1, *r->s2, 0, 5, 1);
	test(*r->s4, 1, 1, *r->s2, 0, 6, 1);
	test(*r->s4, 1, 1, *r->s2, 1, 0, 1);
	test(*r->s4, 1, 1, *r->s2, 1, 1, 0);
	test(*r->s4, 1, 1, *r->s2, 1, 2, -1);
	test(*r->s4, 1, 1, *r->s2, 1, 3, -2);
	test(*r->s4, 1, 1, *r->s2, 1, 4, -3);
	test(*r->s4, 1, 1, *r->s2, 1, 5, -3);
	test(*r->s4, 1, 1, *r->s2, 2, 0, 1);
	test(*r->s4, 1, 1, *r->s2, 2, 1, -1);
	test(*r->s4, 1, 1, *r->s2, 2, 2, -1);
	test(*r->s4, 1, 1, *r->s2, 2, 3, -1);
	test(*r->s4, 1, 1, *r->s2, 2, 4, -1);
	test(*r->s4, 1, 1, *r->s2, 4, 0, 1);
	test(*r->s4, 1, 1, *r->s2, 4, 1, -3);
	test(*r->s4, 1, 1, *r->s2, 4, 2, -3);
	test(*r->s4, 1, 1, *r->s2, 5, 0, 1);
	test(*r->s4, 1, 1, *r->s2, 5, 1, 1);
	test(*r->s4, 1, 1, *r->s2, 6, 0, 0);
	test(*r->s4, 1, 1, *r->s3, 0, 0, 1);
	test(*r->s4, 1, 1, *r->s3, 0, 1, 1);
	test(*r->s4, 1, 1, *r->s3, 0, 5, 1);
	test(*r->s4, 1, 1, *r->s3, 0, 9, 1);
	test(*r->s4, 1, 1, *r->s3, 0, 10, 1);
	test(*r->s4, 1, 1, *r->s3, 0, 11, 1);
	test(*r->s4, 1, 1, *r->s3, 1, 0, 1);
	test(*r->s4, 1, 1, *r->s3, 1, 1, 0);
	test(*r->s4, 1, 1, *r->s3, 1, 4, -3);
	test(*r->s4, 1, 1, *r->s3, 1, 8, -7);
	test(*r->s4, 1, 1, *r->s3, 1, 9, -8);
	test(*r->s4, 1, 1, *r->s3, 1, 10, -8);
	test(*r->s4, 1, 1, *r->s3, 5, 0, 1);
	test(*r->s4, 1, 1, *r->s3, 5, 1, -4);
	test(*r->s4, 1, 1, *r->s3, 5, 2, -4);
	test(*r->s4, 1, 1, *r->s3, 5, 4, -4);
	test(*r->s4, 1, 1, *r->s3, 5, 5, -4);
	test(*r->s4, 1, 1, *r->s3, 5, 6, -4);
	test(*r->s4, 1, 1, *r->s3, 9, 0, 1);
	test(*r->s4, 1, 1, *r->s3, 9, 1, -8);
	test(*r->s4, 1, 1, *r->s3, 9, 2, -8);
	test(*r->s4, 1, 1, *r->s3, 10, 0, 1);
	test(*r->s4, 1, 1, *r->s3, 10, 1, 1);
	test(*r->s4, 1, 1, *r->s3, 11, 0, 0);
	test(*r->s4, 1, 1, *r->s4, 0, 0, 1);
	test(*r->s4, 1, 1, *r->s4, 0, 1, 1);
	test(*r->s4, 1, 1, *r->s4, 0, 10, 1);
	test(*r->s4, 1, 1, *r->s4, 0, 19, 1);
	test(*r->s4, 1, 1, *r->s4, 0, 20, 1);
	test(*r->s4, 1, 1, *r->s4, 0, 21, 1);
	test(*r->s4, 1, 1, *r->s4, 1, 0, 1);
	test(*r->s4, 1, 1, *r->s4, 1, 1, 0);
	test(*r->s4, 1, 1, *r->s4, 1, 9, -8);
	test(*r->s4, 1, 1, *r->s4, 1, 18, -17);
	test(*r->s4, 1, 1, *r->s4, 1, 19, -18);
	test(*r->s4, 1, 1, *r->s4, 1, 20, -18);
	test(*r->s4, 1, 1, *r->s4, 10, 0, 1);
	test(*r->s4, 1, 1, *r->s4, 10, 1, -9);
	test(*r->s4, 1, 1, *r->s4, 10, 5, -9);
	test(*r->s4, 1, 1, *r->s4, 10, 9, -9);
	test(*r->s4, 1, 1, *r->s4, 10, 10, -9);
	test(*r->s4, 1, 1, *r->s4, 10, 11, -9);
	test(*r->s4, 1, 1, *r->s4, 19, 0, 1);
	test(*r->s4, 1, 1, *r->s4, 19, 1, -18);
	test(*r->s4, 1, 1, *r->s4, 19, 2, -18);
	test(*r->s4, 1, 1, *r->s4, 20, 0, 1);
	test(*r->s4, 1, 1, *r->s4, 20, 1, 1);
	test(*r->s4, 1, 1, *r->s4, 21, 0, 0);
	test(*r->s4, 1, 9, *r->s1, 0, 0, 9);
	test(*r->s4, 1, 9, *r->s1, 0, 1, 9);
	test(*r->s4, 1, 9, *r->s1, 1, 0, 0);
	test(*r->s4, 1, 9, *r->s2, 0, 0, 9);
	test(*r->s4, 1, 9, *r->s2, 0, 1, 1);
	test(*r->s4, 1, 9, *r->s2, 0, 2, 1);
	test(*r->s4, 1, 9, *r->s2, 0, 4, 1);
	test(*r->s4, 1, 9, *r->s2, 0, 5, 1);
}

template <class S>
void
test43(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 1, 9, *r->s2, 0, 6, 1);
	test(*r->s4, 1, 9, *r->s2, 1, 0, 9);
	test(*r->s4, 1, 9, *r->s2, 1, 1, 8);
	test(*r->s4, 1, 9, *r->s2, 1, 2, 7);
	test(*r->s4, 1, 9, *r->s2, 1, 3, 6);
	test(*r->s4, 1, 9, *r->s2, 1, 4, 5);
	test(*r->s4, 1, 9, *r->s2, 1, 5, 5);
	test(*r->s4, 1, 9, *r->s2, 2, 0, 9);
	test(*r->s4, 1, 9, *r->s2, 2, 1, -1);
	test(*r->s4, 1, 9, *r->s2, 2, 2, -1);
	test(*r->s4, 1, 9, *r->s2, 2, 3, -1);
	test(*r->s4, 1, 9, *r->s2, 2, 4, -1);
	test(*r->s4, 1, 9, *r->s2, 4, 0, 9);
	test(*r->s4, 1, 9, *r->s2, 4, 1, -3);
	test(*r->s4, 1, 9, *r->s2, 4, 2, -3);
	test(*r->s4, 1, 9, *r->s2, 5, 0, 9);
	test(*r->s4, 1, 9, *r->s2, 5, 1, 9);
	test(*r->s4, 1, 9, *r->s2, 6, 0, 0);
	test(*r->s4, 1, 9, *r->s3, 0, 0, 9);
	test(*r->s4, 1, 9, *r->s3, 0, 1, 1);
	test(*r->s4, 1, 9, *r->s3, 0, 5, 1);
	test(*r->s4, 1, 9, *r->s3, 0, 9, 1);
	test(*r->s4, 1, 9, *r->s3, 0, 10, 1);
	test(*r->s4, 1, 9, *r->s3, 0, 11, 1);
	test(*r->s4, 1, 9, *r->s3, 1, 0, 9);
	test(*r->s4, 1, 9, *r->s3, 1, 1, 8);
	test(*r->s4, 1, 9, *r->s3, 1, 4, 5);
	test(*r->s4, 1, 9, *r->s3, 1, 8, 1);
	test(*r->s4, 1, 9, *r->s3, 1, 9, 0);
	test(*r->s4, 1, 9, *r->s3, 1, 10, 0);
	test(*r->s4, 1, 9, *r->s3, 5, 0, 9);
	test(*r->s4, 1, 9, *r->s3, 5, 1, -4);
	test(*r->s4, 1, 9, *r->s3, 5, 2, -4);
	test(*r->s4, 1, 9, *r->s3, 5, 4, -4);
	test(*r->s4, 1, 9, *r->s3, 5, 5, -4);
	test(*r->s4, 1, 9, *r->s3, 5, 6, -4);
	test(*r->s4, 1, 9, *r->s3, 9, 0, 9);
	test(*r->s4, 1, 9, *r->s3, 9, 1, -8);
	test(*r->s4, 1, 9, *r->s3, 9, 2, -8);
	test(*r->s4, 1, 9, *r->s3, 10, 0, 9);
	test(*r->s4, 1, 9, *r->s3, 10, 1, 9);
	test(*r->s4, 1, 9, *r->s3, 11, 0, 0);
	test(*r->s4, 1, 9, *r->s4, 0, 0, 9);
	test(*r->s4, 1, 9, *r->s4, 0, 1, 1);
	test(*r->s4, 1, 9, *r->s4, 0, 10, 1);
	test(*r->s4, 1, 9, *r->s4, 0, 19, 1);
	test(*r->s4, 1, 9, *r->s4, 0, 20, 1);
	test(*r->s4, 1, 9, *r->s4, 0, 21, 1);
	test(*r->s4, 1, 9, *r->s4, 1, 0, 9);
	test(*r->s4, 1, 9, *r->s4, 1, 1, 8);
	test(*r->s4, 1, 9, *r->s4, 1, 9, 0);
	test(*r->s4, 1, 9, *r->s4, 1, 18, -9);
	test(*r->s4, 1, 9, *r->s4, 1, 19, -10);
	test(*r->s4, 1, 9, *r->s4, 1, 20, -10);
	test(*r->s4, 1, 9, *r->s4, 10, 0, 9);
	test(*r->s4, 1, 9, *r->s4, 10, 1, -9);
	test(*r->s4, 1, 9, *r->s4, 10, 5, -9);
	test(*r->s4, 1, 9, *r->s4, 10, 9, -9);
	test(*r->s4, 1, 9, *r->s4, 10, 10, -9);
	test(*r->s4, 1, 9, *r->s4, 10, 11, -9);
	test(*r->s4, 1, 9, *r->s4, 19, 0, 9);
	test(*r->s4, 1, 9, *r->s4, 19, 1, -18);
	test(*r->s4, 1, 9, *r->s4, 19, 2, -18);
	test(*r->s4, 1, 9, *r->s4, 20, 0, 9);
	test(*r->s4, 1, 9, *r->s4, 20, 1, 9);
	test(*r->s4, 1, 9, *r->s4, 21, 0, 0);
	test(*r->s4, 1, 18, *r->s1, 0, 0, 18);
	test(*r->s4, 1, 18, *r->s1, 0, 1, 18);
	test(*r->s4, 1, 18, *r->s1, 1, 0, 0);
	test(*r->s4, 1, 18, *r->s2, 0, 0, 18);
	test(*r->s4, 1, 18, *r->s2, 0, 1, 1);
	test(*r->s4, 1, 18, *r->s2, 0, 2, 1);
	test(*r->s4, 1, 18, *r->s2, 0, 4, 1);
	test(*r->s4, 1, 18, *r->s2, 0, 5, 1);
	test(*r->s4, 1, 18, *r->s2, 0, 6, 1);
	test(*r->s4, 1, 18, *r->s2, 1, 0, 18);
	test(*r->s4, 1, 18, *r->s2, 1, 1, 17);
	test(*r->s4, 1, 18, *r->s2, 1, 2, 16);
	test(*r->s4, 1, 18, *r->s2, 1, 3, 15);
	test(*r->s4, 1, 18, *r->s2, 1, 4, 14);
	test(*r->s4, 1, 18, *r->s2, 1, 5, 14);
	test(*r->s4, 1, 18, *r->s2, 2, 0, 18);
	test(*r->s4, 1, 18, *r->s2, 2, 1, -1);
	test(*r->s4, 1, 18, *r->s2, 2, 2, -1);
	test(*r->s4, 1, 18, *r->s2, 2, 3, -1);
	test(*r->s4, 1, 18, *r->s2, 2, 4, -1);
	test(*r->s4, 1, 18, *r->s2, 4, 0, 18);
	test(*r->s4, 1, 18, *r->s2, 4, 1, -3);
	test(*r->s4, 1, 18, *r->s2, 4, 2, -3);
	test(*r->s4, 1, 18, *r->s2, 5, 0, 18);
	test(*r->s4, 1, 18, *r->s2, 5, 1, 18);
	test(*r->s4, 1, 18, *r->s2, 6, 0, 0);
	test(*r->s4, 1, 18, *r->s3, 0, 0, 18);
	test(*r->s4, 1, 18, *r->s3, 0, 1, 1);
	test(*r->s4, 1, 18, *r->s3, 0, 5, 1);
	test(*r->s4, 1, 18, *r->s3, 0, 9, 1);
	test(*r->s4, 1, 18, *r->s3, 0, 10, 1);
	test(*r->s4, 1, 18, *r->s3, 0, 11, 1);
	test(*r->s4, 1, 18, *r->s3, 1, 0, 18);
	test(*r->s4, 1, 18, *r->s3, 1, 1, 17);
}

template <class S>
void
test44(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 1, 18, *r->s3, 1, 4, 14);
	test(*r->s4, 1, 18, *r->s3, 1, 8, 10);
	test(*r->s4, 1, 18, *r->s3, 1, 9, 9);
	test(*r->s4, 1, 18, *r->s3, 1, 10, 9);
	test(*r->s4, 1, 18, *r->s3, 5, 0, 18);
	test(*r->s4, 1, 18, *r->s3, 5, 1, -4);
	test(*r->s4, 1, 18, *r->s3, 5, 2, -4);
	test(*r->s4, 1, 18, *r->s3, 5, 4, -4);
	test(*r->s4, 1, 18, *r->s3, 5, 5, -4);
	test(*r->s4, 1, 18, *r->s3, 5, 6, -4);
	test(*r->s4, 1, 18, *r->s3, 9, 0, 18);
	test(*r->s4, 1, 18, *r->s3, 9, 1, -8);
	test(*r->s4, 1, 18, *r->s3, 9, 2, -8);
	test(*r->s4, 1, 18, *r->s3, 10, 0, 18);
	test(*r->s4, 1, 18, *r->s3, 10, 1, 18);
	test(*r->s4, 1, 18, *r->s3, 11, 0, 0);
	test(*r->s4, 1, 18, *r->s4, 0, 0, 18);
	test(*r->s4, 1, 18, *r->s4, 0, 1, 1);
	test(*r->s4, 1, 18, *r->s4, 0, 10, 1);
	test(*r->s4, 1, 18, *r->s4, 0, 19, 1);
	test(*r->s4, 1, 18, *r->s4, 0, 20, 1);
	test(*r->s4, 1, 18, *r->s4, 0, 21, 1);
	test(*r->s4, 1, 18, *r->s4, 1, 0, 18);
	test(*r->s4, 1, 18, *r->s4, 1, 1, 17);
	test(*r->s4, 1, 18, *r->s4, 1, 9, 9);
	test(*r->s4, 1, 18, *r->s4, 1, 18, 0);
	test(*r->s4, 1, 18, *r->s4, 1, 19, -1);
	test(*r->s4, 1, 18, *r->s4, 1, 20, -1);
	test(*r->s4, 1, 18, *r->s4, 10, 0, 18);
	test(*r->s4, 1, 18, *r->s4, 10, 1, -9);
	test(*r->s4, 1, 18, *r->s4, 10, 5, -9);
	test(*r->s4, 1, 18, *r->s4, 10, 9, -9);
	test(*r->s4, 1, 18, *r->s4, 10, 10, -9);
	test(*r->s4, 1, 18, *r->s4, 10, 11, -9);
	test(*r->s4, 1, 18, *r->s4, 19, 0, 18);
	test(*r->s4, 1, 18, *r->s4, 19, 1, -18);
	test(*r->s4, 1, 18, *r->s4, 19, 2, -18);
	test(*r->s4, 1, 18, *r->s4, 20, 0, 18);
	test(*r->s4, 1, 18, *r->s4, 20, 1, 18);
	test(*r->s4, 1, 18, *r->s4, 21, 0, 0);
	test(*r->s4, 1, 19, *r->s1, 0, 0, 19);
	test(*r->s4, 1, 19, *r->s1, 0, 1, 19);
	test(*r->s4, 1, 19, *r->s1, 1, 0, 0);
	test(*r->s4, 1, 19, *r->s2, 0, 0, 19);
	test(*r->s4, 1, 19, *r->s2, 0, 1, 1);
	test(*r->s4, 1, 19, *r->s2, 0, 2, 1);
	test(*r->s4, 1, 19, *r->s2, 0, 4, 1);
	test(*r->s4, 1, 19, *r->s2, 0, 5, 1);
	test(*r->s4, 1, 19, *r->s2, 0, 6, 1);
	test(*r->s4, 1, 19, *r->s2, 1, 0, 19);
	test(*r->s4, 1, 19, *r->s2, 1, 1, 18);
	test(*r->s4, 1, 19, *r->s2, 1, 2, 17);
	test(*r->s4, 1, 19, *r->s2, 1, 3, 16);
	test(*r->s4, 1, 19, *r->s2, 1, 4, 15);
	test(*r->s4, 1, 19, *r->s2, 1, 5, 15);
	test(*r->s4, 1, 19, *r->s2, 2, 0, 19);
	test(*r->s4, 1, 19, *r->s2, 2, 1, -1);
	test(*r->s4, 1, 19, *r->s2, 2, 2, -1);
	test(*r->s4, 1, 19, *r->s2, 2, 3, -1);
	test(*r->s4, 1, 19, *r->s2, 2, 4, -1);
	test(*r->s4, 1, 19, *r->s2, 4, 0, 19);
	test(*r->s4, 1, 19, *r->s2, 4, 1, -3);
	test(*r->s4, 1, 19, *r->s2, 4, 2, -3);
	test(*r->s4, 1, 19, *r->s2, 5, 0, 19);
	test(*r->s4, 1, 19, *r->s2, 5, 1, 19);
	test(*r->s4, 1, 19, *r->s2, 6, 0, 0);
	test(*r->s4, 1, 19, *r->s3, 0, 0, 19);
	test(*r->s4, 1, 19, *r->s3, 0, 1, 1);
	test(*r->s4, 1, 19, *r->s3, 0, 5, 1);
	test(*r->s4, 1, 19, *r->s3, 0, 9, 1);
	test(*r->s4, 1, 19, *r->s3, 0, 10, 1);
	test(*r->s4, 1, 19, *r->s3, 0, 11, 1);
	test(*r->s4, 1, 19, *r->s3, 1, 0, 19);
	test(*r->s4, 1, 19, *r->s3, 1, 1, 18);
	test(*r->s4, 1, 19, *r->s3, 1, 4, 15);
	test(*r->s4, 1, 19, *r->s3, 1, 8, 11);
	test(*r->s4, 1, 19, *r->s3, 1, 9, 10);
	test(*r->s4, 1, 19, *r->s3, 1, 10, 10);
	test(*r->s4, 1, 19, *r->s3, 5, 0, 19);
	test(*r->s4, 1, 19, *r->s3, 5, 1, -4);
	test(*r->s4, 1, 19, *r->s3, 5, 2, -4);
	test(*r->s4, 1, 19, *r->s3, 5, 4, -4);
	test(*r->s4, 1, 19, *r->s3, 5, 5, -4);
	test(*r->s4, 1, 19, *r->s3, 5, 6, -4);
	test(*r->s4, 1, 19, *r->s3, 9, 0, 19);
	test(*r->s4, 1, 19, *r->s3, 9, 1, -8);
	test(*r->s4, 1, 19, *r->s3, 9, 2, -8);
	test(*r->s4, 1, 19, *r->s3, 10, 0, 19);
	test(*r->s4, 1, 19, *r->s3, 10, 1, 19);
	test(*r->s4, 1, 19, *r->s3, 11, 0, 0);
	test(*r->s4, 1, 19, *r->s4, 0, 0, 19);
	test(*r->s4, 1, 19, *r->s4, 0, 1, 1);
	test(*r->s4, 1, 19, *r->s4, 0, 10, 1);
	test(*r->s4, 1, 19, *r->s4, 0, 19, 1);
	test(*r->s4, 1, 19, *r->s4, 0, 20, 1);
	test(*r->s4, 1, 19, *r->s4, 0, 21, 1);
	test(*r->s4, 1, 19, *r->s4, 1, 0, 19);
	test(*r->s4, 1, 19, *r->s4, 1, 1, 18);
	test(*r->s4, 1, 19, *r->s4, 1, 9, 10);
	test(*r->s4, 1, 19, *r->s4, 1, 18, 1);
}

template <class S>
void
test45(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 1, 19, *r->s4, 1, 19, 0);
	test(*r->s4, 1, 19, *r->s4, 1, 20, 0);
	test(*r->s4, 1, 19, *r->s4, 10, 0, 19);
	test(*r->s4, 1, 19, *r->s4, 10, 1, -9);
	test(*r->s4, 1, 19, *r->s4, 10, 5, -9);
	test(*r->s4, 1, 19, *r->s4, 10, 9, -9);
	test(*r->s4, 1, 19, *r->s4, 10, 10, -9);
	test(*r->s4, 1, 19, *r->s4, 10, 11, -9);
	test(*r->s4, 1, 19, *r->s4, 19, 0, 19);
	test(*r->s4, 1, 19, *r->s4, 19, 1, -18);
	test(*r->s4, 1, 19, *r->s4, 19, 2, -18);
	test(*r->s4, 1, 19, *r->s4, 20, 0, 19);
	test(*r->s4, 1, 19, *r->s4, 20, 1, 19);
	test(*r->s4, 1, 19, *r->s4, 21, 0, 0);
	test(*r->s4, 1, 20, *r->s1, 0, 0, 19);
	test(*r->s4, 1, 20, *r->s1, 0, 1, 19);
	test(*r->s4, 1, 20, *r->s1, 1, 0, 0);
	test(*r->s4, 1, 20, *r->s2, 0, 0, 19);
	test(*r->s4, 1, 20, *r->s2, 0, 1, 1);
	test(*r->s4, 1, 20, *r->s2, 0, 2, 1);
	test(*r->s4, 1, 20, *r->s2, 0, 4, 1);
	test(*r->s4, 1, 20, *r->s2, 0, 5, 1);
	test(*r->s4, 1, 20, *r->s2, 0, 6, 1);
	test(*r->s4, 1, 20, *r->s2, 1, 0, 19);
	test(*r->s4, 1, 20, *r->s2, 1, 1, 18);
	test(*r->s4, 1, 20, *r->s2, 1, 2, 17);
	test(*r->s4, 1, 20, *r->s2, 1, 3, 16);
	test(*r->s4, 1, 20, *r->s2, 1, 4, 15);
	test(*r->s4, 1, 20, *r->s2, 1, 5, 15);
	test(*r->s4, 1, 20, *r->s2, 2, 0, 19);
	test(*r->s4, 1, 20, *r->s2, 2, 1, -1);
	test(*r->s4, 1, 20, *r->s2, 2, 2, -1);
	test(*r->s4, 1, 20, *r->s2, 2, 3, -1);
	test(*r->s4, 1, 20, *r->s2, 2, 4, -1);
	test(*r->s4, 1, 20, *r->s2, 4, 0, 19);
	test(*r->s4, 1, 20, *r->s2, 4, 1, -3);
	test(*r->s4, 1, 20, *r->s2, 4, 2, -3);
	test(*r->s4, 1, 20, *r->s2, 5, 0, 19);
	test(*r->s4, 1, 20, *r->s2, 5, 1, 19);
	test(*r->s4, 1, 20, *r->s2, 6, 0, 0);
	test(*r->s4, 1, 20, *r->s3, 0, 0, 19);
	test(*r->s4, 1, 20, *r->s3, 0, 1, 1);
	test(*r->s4, 1, 20, *r->s3, 0, 5, 1);
	test(*r->s4, 1, 20, *r->s3, 0, 9, 1);
	test(*r->s4, 1, 20, *r->s3, 0, 10, 1);
	test(*r->s4, 1, 20, *r->s3, 0, 11, 1);
	test(*r->s4, 1, 20, *r->s3, 1, 0, 19);
	test(*r->s4, 1, 20, *r->s3, 1, 1, 18);
	test(*r->s4, 1, 20, *r->s3, 1, 4, 15);
	test(*r->s4, 1, 20, *r->s3, 1, 8, 11);
	test(*r->s4, 1, 20, *r->s3, 1, 9, 10);
	test(*r->s4, 1, 20, *r->s3, 1, 10, 10);
	test(*r->s4, 1, 20, *r->s3, 5, 0, 19);
	test(*r->s4, 1, 20, *r->s3, 5, 1, -4);
	test(*r->s4, 1, 20, *r->s3, 5, 2, -4);
	test(*r->s4, 1, 20, *r->s3, 5, 4, -4);
	test(*r->s4, 1, 20, *r->s3, 5, 5, -4);
	test(*r->s4, 1, 20, *r->s3, 5, 6, -4);
	test(*r->s4, 1, 20, *r->s3, 9, 0, 19);
	test(*r->s4, 1, 20, *r->s3, 9, 1, -8);
	test(*r->s4, 1, 20, *r->s3, 9, 2, -8);
	test(*r->s4, 1, 20, *r->s3, 10, 0, 19);
	test(*r->s4, 1, 20, *r->s3, 10, 1, 19);
	test(*r->s4, 1, 20, *r->s3, 11, 0, 0);
	test(*r->s4, 1, 20, *r->s4, 0, 0, 19);
	test(*r->s4, 1, 20, *r->s4, 0, 1, 1);
	test(*r->s4, 1, 20, *r->s4, 0, 10, 1);
	test(*r->s4, 1, 20, *r->s4, 0, 19, 1);
	test(*r->s4, 1, 20, *r->s4, 0, 20, 1);
	test(*r->s4, 1, 20, *r->s4, 0, 21, 1);
	test(*r->s4, 1, 20, *r->s4, 1, 0, 19);
	test(*r->s4, 1, 20, *r->s4, 1, 1, 18);
	test(*r->s4, 1, 20, *r->s4, 1, 9, 10);
	test(*r->s4, 1, 20, *r->s4, 1, 18, 1);
	test(*r->s4, 1, 20, *r->s4, 1, 19, 0);
	test(*r->s4, 1, 20, *r->s4, 1, 20, 0);
	test(*r->s4, 1, 20, *r->s4, 10, 0, 19);
	test(*r->s4, 1, 20, *r->s4, 10, 1, -9);
	test(*r->s4, 1, 20, *r->s4, 10, 5, -9);
	test(*r->s4, 1, 20, *r->s4, 10, 9, -9);
	test(*r->s4, 1, 20, *r->s4, 10, 10, -9);
	test(*r->s4, 1, 20, *r->s4, 10, 11, -9);
	test(*r->s4, 1, 20, *r->s4, 19, 0, 19);
	test(*r->s4, 1, 20, *r->s4, 19, 1, -18);
	test(*r->s4, 1, 20, *r->s4, 19, 2, -18);
	test(*r->s4, 1, 20, *r->s4, 20, 0, 19);
	test(*r->s4, 1, 20, *r->s4, 20, 1, 19);
	test(*r->s4, 1, 20, *r->s4, 21, 0, 0);
	test(*r->s4, 10, 0, *r->s1, 0, 0, 0);
	test(*r->s4, 10, 0, *r->s1, 0, 1, 0);
	test(*r->s4, 10, 0, *r->s1, 1, 0, 0);
	test(*r->s4, 10, 0, *r->s2, 0, 0, 0);
	test(*r->s4, 10, 0, *r->s2, 0, 1, -1);
	test(*r->s4, 10, 0, *r->s2, 0, 2, -2);
	test(*r->s4, 10, 0, *r->s2, 0, 4, -4);
	test(*r->s4, 10, 0, *r->s2, 0, 5, -5);
	test(*r->s4, 10, 0, *r->s2, 0, 6, -5);
	test(*r->s4, 10, 0, *r->s2, 1, 0, 0);
	test(*r->s4, 10, 0, *r->s2, 1, 1, -1);
	test(*r->s4, 10, 0, *r->s2, 1, 2, -2);
}

template <class S>
void
test46(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 10, 0, *r->s2, 1, 3, -3);
	test(*r->s4, 10, 0, *r->s2, 1, 4, -4);
	test(*r->s4, 10, 0, *r->s2, 1, 5, -4);
	test(*r->s4, 10, 0, *r->s2, 2, 0, 0);
	test(*r->s4, 10, 0, *r->s2, 2, 1, -1);
	test(*r->s4, 10, 0, *r->s2, 2, 2, -2);
	test(*r->s4, 10, 0, *r->s2, 2, 3, -3);
	test(*r->s4, 10, 0, *r->s2, 2, 4, -3);
	test(*r->s4, 10, 0, *r->s2, 4, 0, 0);
	test(*r->s4, 10, 0, *r->s2, 4, 1, -1);
	test(*r->s4, 10, 0, *r->s2, 4, 2, -1);
	test(*r->s4, 10, 0, *r->s2, 5, 0, 0);
	test(*r->s4, 10, 0, *r->s2, 5, 1, 0);
	test(*r->s4, 10, 0, *r->s2, 6, 0, 0);
	test(*r->s4, 10, 0, *r->s3, 0, 0, 0);
	test(*r->s4, 10, 0, *r->s3, 0, 1, -1);
	test(*r->s4, 10, 0, *r->s3, 0, 5, -5);
	test(*r->s4, 10, 0, *r->s3, 0, 9, -9);
	test(*r->s4, 10, 0, *r->s3, 0, 10, -10);
	test(*r->s4, 10, 0, *r->s3, 0, 11, -10);
	test(*r->s4, 10, 0, *r->s3, 1, 0, 0);
	test(*r->s4, 10, 0, *r->s3, 1, 1, -1);
	test(*r->s4, 10, 0, *r->s3, 1, 4, -4);
	test(*r->s4, 10, 0, *r->s3, 1, 8, -8);
	test(*r->s4, 10, 0, *r->s3, 1, 9, -9);
	test(*r->s4, 10, 0, *r->s3, 1, 10, -9);
	test(*r->s4, 10, 0, *r->s3, 5, 0, 0);
	test(*r->s4, 10, 0, *r->s3, 5, 1, -1);
	test(*r->s4, 10, 0, *r->s3, 5, 2, -2);
	test(*r->s4, 10, 0, *r->s3, 5, 4, -4);
	test(*r->s4, 10, 0, *r->s3, 5, 5, -5);
	test(*r->s4, 10, 0, *r->s3, 5, 6, -5);
	test(*r->s4, 10, 0, *r->s3, 9, 0, 0);
	test(*r->s4, 10, 0, *r->s3, 9, 1, -1);
	test(*r->s4, 10, 0, *r->s3, 9, 2, -1);
	test(*r->s4, 10, 0, *r->s3, 10, 0, 0);
	test(*r->s4, 10, 0, *r->s3, 10, 1, 0);
	test(*r->s4, 10, 0, *r->s3, 11, 0, 0);
	test(*r->s4, 10, 0, *r->s4, 0, 0, 0);
	test(*r->s4, 10, 0, *r->s4, 0, 1, -1);
	test(*r->s4, 10, 0, *r->s4, 0, 10, -10);
	test(*r->s4, 10, 0, *r->s4, 0, 19, -19);
	test(*r->s4, 10, 0, *r->s4, 0, 20, -20);
	test(*r->s4, 10, 0, *r->s4, 0, 21, -20);
	test(*r->s4, 10, 0, *r->s4, 1, 0, 0);
	test(*r->s4, 10, 0, *r->s4, 1, 1, -1);
	test(*r->s4, 10, 0, *r->s4, 1, 9, -9);
	test(*r->s4, 10, 0, *r->s4, 1, 18, -18);
	test(*r->s4, 10, 0, *r->s4, 1, 19, -19);
	test(*r->s4, 10, 0, *r->s4, 1, 20, -19);
	test(*r->s4, 10, 0, *r->s4, 10, 0, 0);
	test(*r->s4, 10, 0, *r->s4, 10, 1, -1);
	test(*r->s4, 10, 0, *r->s4, 10, 5, -5);
	test(*r->s4, 10, 0, *r->s4, 10, 9, -9);
	test(*r->s4, 10, 0, *r->s4, 10, 10, -10);
	test(*r->s4, 10, 0, *r->s4, 10, 11, -10);
	test(*r->s4, 10, 0, *r->s4, 19, 0, 0);
	test(*r->s4, 10, 0, *r->s4, 19, 1, -1);
	test(*r->s4, 10, 0, *r->s4, 19, 2, -1);
	test(*r->s4, 10, 0, *r->s4, 20, 0, 0);
	test(*r->s4, 10, 0, *r->s4, 20, 1, 0);
	test(*r->s4, 10, 0, *r->s4, 21, 0, 0);
	test(*r->s4, 10, 1, *r->s1, 0, 0, 1);
	test(*r->s4, 10, 1, *r->s1, 0, 1, 1);
	test(*r->s4, 10, 1, *r->s1, 1, 0, 0);
	test(*r->s4, 10, 1, *r->s2, 0, 0, 1);
	test(*r->s4, 10, 1, *r->s2, 0, 1, 10);
	test(*r->s4, 10, 1, *r->s2, 0, 2, 10);
	test(*r->s4, 10, 1, *r->s2, 0, 4, 10);
	test(*r->s4, 10, 1, *r->s2, 0, 5, 10);
	test(*r->s4, 10, 1, *r->s2, 0, 6, 10);
	test(*r->s4, 10, 1, *r->s2, 1, 0, 1);
	test(*r->s4, 10, 1, *r->s2, 1, 1, 9);
	test(*r->s4, 10, 1, *r->s2, 1, 2, 9);
	test(*r->s4, 10, 1, *r->s2, 1, 3, 9);
	test(*r->s4, 10, 1, *r->s2, 1, 4, 9);
	test(*r->s4, 10, 1, *r->s2, 1, 5, 9);
	test(*r->s4, 10, 1, *r->s2, 2, 0, 1);
	test(*r->s4, 10, 1, *r->s2, 2, 1, 8);
	test(*r->s4, 10, 1, *r->s2, 2, 2, 8);
	test(*r->s4, 10, 1, *r->s2, 2, 3, 8);
	test(*r->s4, 10, 1, *r->s2, 2, 4, 8);
	test(*r->s4, 10, 1, *r->s2, 4, 0, 1);
	test(*r->s4, 10, 1, *r->s2, 4, 1, 6);
	test(*r->s4, 10, 1, *r->s2, 4, 2, 6);
	test(*r->s4, 10, 1, *r->s2, 5, 0, 1);
	test(*r->s4, 10, 1, *r->s2, 5, 1, 1);
	test(*r->s4, 10, 1, *r->s2, 6, 0, 0);
	test(*r->s4, 10, 1, *r->s3, 0, 0, 1);
	test(*r->s4, 10, 1, *r->s3, 0, 1, 10);
	test(*r->s4, 10, 1, *r->s3, 0, 5, 10);
	test(*r->s4, 10, 1, *r->s3, 0, 9, 10);
	test(*r->s4, 10, 1, *r->s3, 0, 10, 10);
	test(*r->s4, 10, 1, *r->s3, 0, 11, 10);
	test(*r->s4, 10, 1, *r->s3, 1, 0, 1);
	test(*r->s4, 10, 1, *r->s3, 1, 1, 9);
	test(*r->s4, 10, 1, *r->s3, 1, 4, 9);
	test(*r->s4, 10, 1, *r->s3, 1, 8, 9);
	test(*r->s4, 10, 1, *r->s3, 1, 9, 9);
	test(*r->s4, 10, 1, *r->s3, 1, 10, 9);
}

template <class S>
void
test47(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 10, 1, *r->s3, 5, 0, 1);
	test(*r->s4, 10, 1, *r->s3, 5, 1, 5);
	test(*r->s4, 10, 1, *r->s3, 5, 2, 5);
	test(*r->s4, 10, 1, *r->s3, 5, 4, 5);
	test(*r->s4, 10, 1, *r->s3, 5, 5, 5);
	test(*r->s4, 10, 1, *r->s3, 5, 6, 5);
	test(*r->s4, 10, 1, *r->s3, 9, 0, 1);
	test(*r->s4, 10, 1, *r->s3, 9, 1, 1);
	test(*r->s4, 10, 1, *r->s3, 9, 2, 1);
	test(*r->s4, 10, 1, *r->s3, 10, 0, 1);
	test(*r->s4, 10, 1, *r->s3, 10, 1, 1);
	test(*r->s4, 10, 1, *r->s3, 11, 0, 0);
	test(*r->s4, 10, 1, *r->s4, 0, 0, 1);
	test(*r->s4, 10, 1, *r->s4, 0, 1, 10);
	test(*r->s4, 10, 1, *r->s4, 0, 10, 10);
	test(*r->s4, 10, 1, *r->s4, 0, 19, 10);
	test(*r->s4, 10, 1, *r->s4, 0, 20, 10);
	test(*r->s4, 10, 1, *r->s4, 0, 21, 10);
	test(*r->s4, 10, 1, *r->s4, 1, 0, 1);
	test(*r->s4, 10, 1, *r->s4, 1, 1, 9);
	test(*r->s4, 10, 1, *r->s4, 1, 9, 9);
	test(*r->s4, 10, 1, *r->s4, 1, 18, 9);
	test(*r->s4, 10, 1, *r->s4, 1, 19, 9);
	test(*r->s4, 10, 1, *r->s4, 1, 20, 9);
	test(*r->s4, 10, 1, *r->s4, 10, 0, 1);
	test(*r->s4, 10, 1, *r->s4, 10, 1, 0);
	test(*r->s4, 10, 1, *r->s4, 10, 5, -4);
	test(*r->s4, 10, 1, *r->s4, 10, 9, -8);
	test(*r->s4, 10, 1, *r->s4, 10, 10, -9);
	test(*r->s4, 10, 1, *r->s4, 10, 11, -9);
	test(*r->s4, 10, 1, *r->s4, 19, 0, 1);
	test(*r->s4, 10, 1, *r->s4, 19, 1, -9);
	test(*r->s4, 10, 1, *r->s4, 19, 2, -9);
	test(*r->s4, 10, 1, *r->s4, 20, 0, 1);
	test(*r->s4, 10, 1, *r->s4, 20, 1, 1);
	test(*r->s4, 10, 1, *r->s4, 21, 0, 0);
	test(*r->s4, 10, 5, *r->s1, 0, 0, 5);
	test(*r->s4, 10, 5, *r->s1, 0, 1, 5);
	test(*r->s4, 10, 5, *r->s1, 1, 0, 0);
	test(*r->s4, 10, 5, *r->s2, 0, 0, 5);
	test(*r->s4, 10, 5, *r->s2, 0, 1, 10);
	test(*r->s4, 10, 5, *r->s2, 0, 2, 10);
	test(*r->s4, 10, 5, *r->s2, 0, 4, 10);
	test(*r->s4, 10, 5, *r->s2, 0, 5, 10);
	test(*r->s4, 10, 5, *r->s2, 0, 6, 10);
	test(*r->s4, 10, 5, *r->s2, 1, 0, 5);
	test(*r->s4, 10, 5, *r->s2, 1, 1, 9);
	test(*r->s4, 10, 5, *r->s2, 1, 2, 9);
	test(*r->s4, 10, 5, *r->s2, 1, 3, 9);
	test(*r->s4, 10, 5, *r->s2, 1, 4, 9);
	test(*r->s4, 10, 5, *r->s2, 1, 5, 9);
	test(*r->s4, 10, 5, *r->s2, 2, 0, 5);
	test(*r->s4, 10, 5, *r->s2, 2, 1, 8);
	test(*r->s4, 10, 5, *r->s2, 2, 2, 8);
	test(*r->s4, 10, 5, *r->s2, 2, 3, 8);
	test(*r->s4, 10, 5, *r->s2, 2, 4, 8);
	test(*r->s4, 10, 5, *r->s2, 4, 0, 5);
	test(*r->s4, 10, 5, *r->s2, 4, 1, 6);
	test(*r->s4, 10, 5, *r->s2, 4, 2, 6);
	test(*r->s4, 10, 5, *r->s2, 5, 0, 5);
	test(*r->s4, 10, 5, *r->s2, 5, 1, 5);
	test(*r->s4, 10, 5, *r->s2, 6, 0, 0);
	test(*r->s4, 10, 5, *r->s3, 0, 0, 5);
	test(*r->s4, 10, 5, *r->s3, 0, 1, 10);
	test(*r->s4, 10, 5, *r->s3, 0, 5, 10);
	test(*r->s4, 10, 5, *r->s3, 0, 9, 10);
	test(*r->s4, 10, 5, *r->s3, 0, 10, 10);
	test(*r->s4, 10, 5, *r->s3, 0, 11, 10);
	test(*r->s4, 10, 5, *r->s3, 1, 0, 5);
	test(*r->s4, 10, 5, *r->s3, 1, 1, 9);
	test(*r->s4, 10, 5, *r->s3, 1, 4, 9);
	test(*r->s4, 10, 5, *r->s3, 1, 8, 9);
	test(*r->s4, 10, 5, *r->s3, 1, 9, 9);
	test(*r->s4, 10, 5, *r->s3, 1, 10, 9);
	test(*r->s4, 10, 5, *r->s3, 5, 0, 5);
	test(*r->s4, 10, 5, *r->s3, 5, 1, 5);
	test(*r->s4, 10, 5, *r->s3, 5, 2, 5);
	test(*r->s4, 10, 5, *r->s3, 5, 4, 5);
	test(*r->s4, 10, 5, *r->s3, 5, 5, 5);
	test(*r->s4, 10, 5, *r->s3, 5, 6, 5);
	test(*r->s4, 10, 5, *r->s3, 9, 0, 5);
	test(*r->s4, 10, 5, *r->s3, 9, 1, 1);
	test(*r->s4, 10, 5, *r->s3, 9, 2, 1);
	test(*r->s4, 10, 5, *r->s3, 10, 0, 5);
	test(*r->s4, 10, 5, *r->s3, 10, 1, 5);
	test(*r->s4, 10, 5, *r->s3, 11, 0, 0);
	test(*r->s4, 10, 5, *r->s4, 0, 0, 5);
	test(*r->s4, 10, 5, *r->s4, 0, 1, 10);
	test(*r->s4, 10, 5, *r->s4, 0, 10, 10);
	test(*r->s4, 10, 5, *r->s4, 0, 19, 10);
	test(*r->s4, 10, 5, *r->s4, 0, 20, 10);
	test(*r->s4, 10, 5, *r->s4, 0, 21, 10);
	test(*r->s4, 10, 5, *r->s4, 1, 0, 5);
	test(*r->s4, 10, 5, *r->s4, 1, 1, 9);
	test(*r->s4, 10, 5, *r->s4, 1, 9, 9);
	test(*r->s4, 10, 5, *r->s4, 1, 18, 9);
	test(*r->s4, 10, 5, *r->s4, 1, 19, 9);
	test(*r->s4, 10, 5, *r->s4, 1, 20, 9);
	test(*r->s4, 10, 5, *r->s4, 10, 0, 5);
	test(*r->s4, 10, 5, *r->s4, 10, 1, 4);
}

template <class S>
void
test48(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 10, 5, *r->s4, 10, 5, 0);
	test(*r->s4, 10, 5, *r->s4, 10, 9, -4);
	test(*r->s4, 10, 5, *r->s4, 10, 10, -5);
	test(*r->s4, 10, 5, *r->s4, 10, 11, -5);
	test(*r->s4, 10, 5, *r->s4, 19, 0, 5);
	test(*r->s4, 10, 5, *r->s4, 19, 1, -9);
	test(*r->s4, 10, 5, *r->s4, 19, 2, -9);
	test(*r->s4, 10, 5, *r->s4, 20, 0, 5);
	test(*r->s4, 10, 5, *r->s4, 20, 1, 5);
	test(*r->s4, 10, 5, *r->s4, 21, 0, 0);
	test(*r->s4, 10, 9, *r->s1, 0, 0, 9);
	test(*r->s4, 10, 9, *r->s1, 0, 1, 9);
	test(*r->s4, 10, 9, *r->s1, 1, 0, 0);
	test(*r->s4, 10, 9, *r->s2, 0, 0, 9);
	test(*r->s4, 10, 9, *r->s2, 0, 1, 10);
	test(*r->s4, 10, 9, *r->s2, 0, 2, 10);
	test(*r->s4, 10, 9, *r->s2, 0, 4, 10);
	test(*r->s4, 10, 9, *r->s2, 0, 5, 10);
	test(*r->s4, 10, 9, *r->s2, 0, 6, 10);
	test(*r->s4, 10, 9, *r->s2, 1, 0, 9);
	test(*r->s4, 10, 9, *r->s2, 1, 1, 9);
	test(*r->s4, 10, 9, *r->s2, 1, 2, 9);
	test(*r->s4, 10, 9, *r->s2, 1, 3, 9);
	test(*r->s4, 10, 9, *r->s2, 1, 4, 9);
	test(*r->s4, 10, 9, *r->s2, 1, 5, 9);
	test(*r->s4, 10, 9, *r->s2, 2, 0, 9);
	test(*r->s4, 10, 9, *r->s2, 2, 1, 8);
	test(*r->s4, 10, 9, *r->s2, 2, 2, 8);
	test(*r->s4, 10, 9, *r->s2, 2, 3, 8);
	test(*r->s4, 10, 9, *r->s2, 2, 4, 8);
	test(*r->s4, 10, 9, *r->s2, 4, 0, 9);
	test(*r->s4, 10, 9, *r->s2, 4, 1, 6);
	test(*r->s4, 10, 9, *r->s2, 4, 2, 6);
	test(*r->s4, 10, 9, *r->s2, 5, 0, 9);
	test(*r->s4, 10, 9, *r->s2, 5, 1, 9);
	test(*r->s4, 10, 9, *r->s2, 6, 0, 0);
	test(*r->s4, 10, 9, *r->s3, 0, 0, 9);
	test(*r->s4, 10, 9, *r->s3, 0, 1, 10);
	test(*r->s4, 10, 9, *r->s3, 0, 5, 10);
	test(*r->s4, 10, 9, *r->s3, 0, 9, 10);
	test(*r->s4, 10, 9, *r->s3, 0, 10, 10);
	test(*r->s4, 10, 9, *r->s3, 0, 11, 10);
	test(*r->s4, 10, 9, *r->s3, 1, 0, 9);
	test(*r->s4, 10, 9, *r->s3, 1, 1, 9);
	test(*r->s4, 10, 9, *r->s3, 1, 4, 9);
	test(*r->s4, 10, 9, *r->s3, 1, 8, 9);
	test(*r->s4, 10, 9, *r->s3, 1, 9, 9);
	test(*r->s4, 10, 9, *r->s3, 1, 10, 9);
	test(*r->s4, 10, 9, *r->s3, 5, 0, 9);
	test(*r->s4, 10, 9, *r->s3, 5, 1, 5);
	test(*r->s4, 10, 9, *r->s3, 5, 2, 5);
	test(*r->s4, 10, 9, *r->s3, 5, 4, 5);
	test(*r->s4, 10, 9, *r->s3, 5, 5, 5);
	test(*r->s4, 10, 9, *r->s3, 5, 6, 5);
	test(*r->s4, 10, 9, *r->s3, 9, 0, 9);
	test(*r->s4, 10, 9, *r->s3, 9, 1, 1);
	test(*r->s4, 10, 9, *r->s3, 9, 2, 1);
	test(*r->s4, 10, 9, *r->s3, 10, 0, 9);
	test(*r->s4, 10, 9, *r->s3, 10, 1, 9);
	test(*r->s4, 10, 9, *r->s3, 11, 0, 0);
	test(*r->s4, 10, 9, *r->s4, 0, 0, 9);
	test(*r->s4, 10, 9, *r->s4, 0, 1, 10);
	test(*r->s4, 10, 9, *r->s4, 0, 10, 10);
	test(*r->s4, 10, 9, *r->s4, 0, 19, 10);
	test(*r->s4, 10, 9, *r->s4, 0, 20, 10);
	test(*r->s4, 10, 9, *r->s4, 0, 21, 10);
	test(*r->s4, 10, 9, *r->s4, 1, 0, 9);
	test(*r->s4, 10, 9, *r->s4, 1, 1, 9);
	test(*r->s4, 10, 9, *r->s4, 1, 9, 9);
	test(*r->s4, 10, 9, *r->s4, 1, 18, 9);
	test(*r->s4, 10, 9, *r->s4, 1, 19, 9);
	test(*r->s4, 10, 9, *r->s4, 1, 20, 9);
	test(*r->s4, 10, 9, *r->s4, 10, 0, 9);
	test(*r->s4, 10, 9, *r->s4, 10, 1, 8);
	test(*r->s4, 10, 9, *r->s4, 10, 5, 4);
	test(*r->s4, 10, 9, *r->s4, 10, 9, 0);
	test(*r->s4, 10, 9, *r->s4, 10, 10, -1);
	test(*r->s4, 10, 9, *r->s4, 10, 11, -1);
	test(*r->s4, 10, 9, *r->s4, 19, 0, 9);
	test(*r->s4, 10, 9, *r->s4, 19, 1, -9);
	test(*r->s4, 10, 9, *r->s4, 19, 2, -9);
	test(*r->s4, 10, 9, *r->s4, 20, 0, 9);
	test(*r->s4, 10, 9, *r->s4, 20, 1, 9);
	test(*r->s4, 10, 9, *r->s4, 21, 0, 0);
	test(*r->s4, 10, 10, *r->s1, 0, 0, 10);
	test(*r->s4, 10, 10, *r->s1, 0, 1, 10);
	test(*r->s4, 10, 10, *r->s1, 1, 0, 0);
	test(*r->s4, 10, 10, *r->s2, 0, 0, 10);
	test(*r->s4, 10, 10, *r->s2, 0, 1, 10);
	test(*r->s4, 10, 10, *r->s2, 0, 2, 10);
	test(*r->s4, 10, 10, *r->s2, 0, 4, 10);
	test(*r->s4, 10, 10, *r->s2, 0, 5, 10);
	test(*r->s4, 10, 10, *r->s2, 0, 6, 10);
	test(*r->s4, 10, 10, *r->s2, 1, 0, 10);
	test(*r->s4, 10, 10, *r->s2, 1, 1, 9);
	test(*r->s4, 10, 10, *r->s2, 1, 2, 9);
	test(*r->s4, 10, 10, *r->s2, 1, 3, 9);
	test(*r->s4, 10, 10, *r->s2, 1, 4, 9);
	test(*r->s4, 10, 10, *r->s2, 1, 5, 9);
	test(*r->s4, 10, 10, *r->s2, 2, 0, 10);
}

template <class S>
void
test49(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 10, 10, *r->s2, 2, 1, 8);
	test(*r->s4, 10, 10, *r->s2, 2, 2, 8);
	test(*r->s4, 10, 10, *r->s2, 2, 3, 8);
	test(*r->s4, 10, 10, *r->s2, 2, 4, 8);
	test(*r->s4, 10, 10, *r->s2, 4, 0, 10);
	test(*r->s4, 10, 10, *r->s2, 4, 1, 6);
	test(*r->s4, 10, 10, *r->s2, 4, 2, 6);
	test(*r->s4, 10, 10, *r->s2, 5, 0, 10);
	test(*r->s4, 10, 10, *r->s2, 5, 1, 10);
	test(*r->s4, 10, 10, *r->s2, 6, 0, 0);
	test(*r->s4, 10, 10, *r->s3, 0, 0, 10);
	test(*r->s4, 10, 10, *r->s3, 0, 1, 10);
	test(*r->s4, 10, 10, *r->s3, 0, 5, 10);
	test(*r->s4, 10, 10, *r->s3, 0, 9, 10);
	test(*r->s4, 10, 10, *r->s3, 0, 10, 10);
	test(*r->s4, 10, 10, *r->s3, 0, 11, 10);
	test(*r->s4, 10, 10, *r->s3, 1, 0, 10);
	test(*r->s4, 10, 10, *r->s3, 1, 1, 9);
	test(*r->s4, 10, 10, *r->s3, 1, 4, 9);
	test(*r->s4, 10, 10, *r->s3, 1, 8, 9);
	test(*r->s4, 10, 10, *r->s3, 1, 9, 9);
	test(*r->s4, 10, 10, *r->s3, 1, 10, 9);
	test(*r->s4, 10, 10, *r->s3, 5, 0, 10);
	test(*r->s4, 10, 10, *r->s3, 5, 1, 5);
	test(*r->s4, 10, 10, *r->s3, 5, 2, 5);
	test(*r->s4, 10, 10, *r->s3, 5, 4, 5);
	test(*r->s4, 10, 10, *r->s3, 5, 5, 5);
	test(*r->s4, 10, 10, *r->s3, 5, 6, 5);
	test(*r->s4, 10, 10, *r->s3, 9, 0, 10);
	test(*r->s4, 10, 10, *r->s3, 9, 1, 1);
	test(*r->s4, 10, 10, *r->s3, 9, 2, 1);
	test(*r->s4, 10, 10, *r->s3, 10, 0, 10);
	test(*r->s4, 10, 10, *r->s3, 10, 1, 10);
	test(*r->s4, 10, 10, *r->s3, 11, 0, 0);
	test(*r->s4, 10, 10, *r->s4, 0, 0, 10);
	test(*r->s4, 10, 10, *r->s4, 0, 1, 10);
	test(*r->s4, 10, 10, *r->s4, 0, 10, 10);
	test(*r->s4, 10, 10, *r->s4, 0, 19, 10);
	test(*r->s4, 10, 10, *r->s4, 0, 20, 10);
	test(*r->s4, 10, 10, *r->s4, 0, 21, 10);
	test(*r->s4, 10, 10, *r->s4, 1, 0, 10);
	test(*r->s4, 10, 10, *r->s4, 1, 1, 9);
	test(*r->s4, 10, 10, *r->s4, 1, 9, 9);
	test(*r->s4, 10, 10, *r->s4, 1, 18, 9);
	test(*r->s4, 10, 10, *r->s4, 1, 19, 9);
	test(*r->s4, 10, 10, *r->s4, 1, 20, 9);
	test(*r->s4, 10, 10, *r->s4, 10, 0, 10);
	test(*r->s4, 10, 10, *r->s4, 10, 1, 9);
	test(*r->s4, 10, 10, *r->s4, 10, 5, 5);
	test(*r->s4, 10, 10, *r->s4, 10, 9, 1);
	test(*r->s4, 10, 10, *r->s4, 10, 10, 0);
	test(*r->s4, 10, 10, *r->s4, 10, 11, 0);
	test(*r->s4, 10, 10, *r->s4, 19, 0, 10);
	test(*r->s4, 10, 10, *r->s4, 19, 1, -9);
	test(*r->s4, 10, 10, *r->s4, 19, 2, -9);
	test(*r->s4, 10, 10, *r->s4, 20, 0, 10);
	test(*r->s4, 10, 10, *r->s4, 20, 1, 10);
	test(*r->s4, 10, 10, *r->s4, 21, 0, 0);
	test(*r->s4, 10, 11, *r->s1, 0, 0, 10);
	test(*r->s4, 10, 11, *r->s1, 0, 1, 10);
	test(*r->s4, 10, 11, *r->s1, 1, 0, 0);
	test(*r->s4, 10, 11, *r->s2, 0, 0, 10);
	test(*r->s4, 10, 11, *r->s2, 0, 1, 10);
	test(*r->s4, 10, 11, *r->s2, 0, 2, 10);
	test(*r->s4, 10, 11, *r->s2, 0, 4, 10);
	test(*r->s4, 10, 11, *r->s2, 0, 5, 10);
	test(*r->s4, 10, 11, *r->s2, 0, 6, 10);
	test(*r->s4, 10, 11, *r->s2, 1, 0, 10);
	test(*r->s4, 10, 11, *r->s2, 1, 1, 9);
	test(*r->s4, 10, 11, *r->s2, 1, 2, 9);
	test(*r->s4, 10, 11, *r->s2, 1, 3, 9);
	test(*r->s4, 10, 11, *r->s2, 1, 4, 9);
	test(*r->s4, 10, 11, *r->s2, 1, 5, 9);
	test(*r->s4, 10, 11, *r->s2, 2, 0, 10);
	test(*r->s4, 10, 11, *r->s2, 2, 1, 8);
	test(*r->s4, 10, 11, *r->s2, 2, 2, 8);
	test(*r->s4, 10, 11, *r->s2, 2, 3, 8);
	test(*r->s4, 10, 11, *r->s2, 2, 4, 8);
	test(*r->s4, 10, 11, *r->s2, 4, 0, 10);
	test(*r->s4, 10, 11, *r->s2, 4, 1, 6);
	test(*r->s4, 10, 11, *r->s2, 4, 2, 6);
	test(*r->s4, 10, 11, *r->s2, 5, 0, 10);
	test(*r->s4, 10, 11, *r->s2, 5, 1, 10);
	test(*r->s4, 10, 11, *r->s2, 6, 0, 0);
	test(*r->s4, 10, 11, *r->s3, 0, 0, 10);
	test(*r->s4, 10, 11, *r->s3, 0, 1, 10);
	test(*r->s4, 10, 11, *r->s3, 0, 5, 10);
	test(*r->s4, 10, 11, *r->s3, 0, 9, 10);
	test(*r->s4, 10, 11, *r->s3, 0, 10, 10);
	test(*r->s4, 10, 11, *r->s3, 0, 11, 10);
	test(*r->s4, 10, 11, *r->s3, 1, 0, 10);
	test(*r->s4, 10, 11, *r->s3, 1, 1, 9);
	test(*r->s4, 10, 11, *r->s3, 1, 4, 9);
	test(*r->s4, 10, 11, *r->s3, 1, 8, 9);
	test(*r->s4, 10, 11, *r->s3, 1, 9, 9);
	test(*r->s4, 10, 11, *r->s3, 1, 10, 9);
	test(*r->s4, 10, 11, *r->s3, 5, 0, 10);
	test(*r->s4, 10, 11, *r->s3, 5, 1, 5);
	test(*r->s4, 10, 11, *r->s3, 5, 2, 5);
	test(*r->s4, 10, 11, *r->s3, 5, 4, 5);
}

template <class S>
void
test50(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 10, 11, *r->s3, 5, 5, 5);
	test(*r->s4, 10, 11, *r->s3, 5, 6, 5);
	test(*r->s4, 10, 11, *r->s3, 9, 0, 10);
	test(*r->s4, 10, 11, *r->s3, 9, 1, 1);
	test(*r->s4, 10, 11, *r->s3, 9, 2, 1);
	test(*r->s4, 10, 11, *r->s3, 10, 0, 10);
	test(*r->s4, 10, 11, *r->s3, 10, 1, 10);
	test(*r->s4, 10, 11, *r->s3, 11, 0, 0);
	test(*r->s4, 10, 11, *r->s4, 0, 0, 10);
	test(*r->s4, 10, 11, *r->s4, 0, 1, 10);
	test(*r->s4, 10, 11, *r->s4, 0, 10, 10);
	test(*r->s4, 10, 11, *r->s4, 0, 19, 10);
	test(*r->s4, 10, 11, *r->s4, 0, 20, 10);
	test(*r->s4, 10, 11, *r->s4, 0, 21, 10);
	test(*r->s4, 10, 11, *r->s4, 1, 0, 10);
	test(*r->s4, 10, 11, *r->s4, 1, 1, 9);
	test(*r->s4, 10, 11, *r->s4, 1, 9, 9);
	test(*r->s4, 10, 11, *r->s4, 1, 18, 9);
	test(*r->s4, 10, 11, *r->s4, 1, 19, 9);
	test(*r->s4, 10, 11, *r->s4, 1, 20, 9);
	test(*r->s4, 10, 11, *r->s4, 10, 0, 10);
	test(*r->s4, 10, 11, *r->s4, 10, 1, 9);
	test(*r->s4, 10, 11, *r->s4, 10, 5, 5);
	test(*r->s4, 10, 11, *r->s4, 10, 9, 1);
	test(*r->s4, 10, 11, *r->s4, 10, 10, 0);
	test(*r->s4, 10, 11, *r->s4, 10, 11, 0);
	test(*r->s4, 10, 11, *r->s4, 19, 0, 10);
	test(*r->s4, 10, 11, *r->s4, 19, 1, -9);
	test(*r->s4, 10, 11, *r->s4, 19, 2, -9);
	test(*r->s4, 10, 11, *r->s4, 20, 0, 10);
	test(*r->s4, 10, 11, *r->s4, 20, 1, 10);
	test(*r->s4, 10, 11, *r->s4, 21, 0, 0);
	test(*r->s4, 19, 0, *r->s1, 0, 0, 0);
	test(*r->s4, 19, 0, *r->s1, 0, 1, 0);
	test(*r->s4, 19, 0, *r->s1, 1, 0, 0);
	test(*r->s4, 19, 0, *r->s2, 0, 0, 0);
	test(*r->s4, 19, 0, *r->s2, 0, 1, -1);
	test(*r->s4, 19, 0, *r->s2, 0, 2, -2);
	test(*r->s4, 19, 0, *r->s2, 0, 4, -4);
	test(*r->s4, 19, 0, *r->s2, 0, 5, -5);
	test(*r->s4, 19, 0, *r->s2, 0, 6, -5);
	test(*r->s4, 19, 0, *r->s2, 1, 0, 0);
	test(*r->s4, 19, 0, *r->s2, 1, 1, -1);
	test(*r->s4, 19, 0, *r->s2, 1, 2, -2);
	test(*r->s4, 19, 0, *r->s2, 1, 3, -3);
	test(*r->s4, 19, 0, *r->s2, 1, 4, -4);
	test(*r->s4, 19, 0, *r->s2, 1, 5, -4);
	test(*r->s4, 19, 0, *r->s2, 2, 0, 0);
	test(*r->s4, 19, 0, *r->s2, 2, 1, -1);
	test(*r->s4, 19, 0, *r->s2, 2, 2, -2);
	test(*r->s4, 19, 0, *r->s2, 2, 3, -3);
	test(*r->s4, 19, 0, *r->s2, 2, 4, -3);
	test(*r->s4, 19, 0, *r->s2, 4, 0, 0);
	test(*r->s4, 19, 0, *r->s2, 4, 1, -1);
	test(*r->s4, 19, 0, *r->s2, 4, 2, -1);
	test(*r->s4, 19, 0, *r->s2, 5, 0, 0);
	test(*r->s4, 19, 0, *r->s2, 5, 1, 0);
	test(*r->s4, 19, 0, *r->s2, 6, 0, 0);
	test(*r->s4, 19, 0, *r->s3, 0, 0, 0);
	test(*r->s4, 19, 0, *r->s3, 0, 1, -1);
	test(*r->s4, 19, 0, *r->s3, 0, 5, -5);
	test(*r->s4, 19, 0, *r->s3, 0, 9, -9);
	test(*r->s4, 19, 0, *r->s3, 0, 10, -10);
	test(*r->s4, 19, 0, *r->s3, 0, 11, -10);
	test(*r->s4, 19, 0, *r->s3, 1, 0, 0);
	test(*r->s4, 19, 0, *r->s3, 1, 1, -1);
	test(*r->s4, 19, 0, *r->s3, 1, 4, -4);
	test(*r->s4, 19, 0, *r->s3, 1, 8, -8);
	test(*r->s4, 19, 0, *r->s3, 1, 9, -9);
	test(*r->s4, 19, 0, *r->s3, 1, 10, -9);
	test(*r->s4, 19, 0, *r->s3, 5, 0, 0);
	test(*r->s4, 19, 0, *r->s3, 5, 1, -1);
	test(*r->s4, 19, 0, *r->s3, 5, 2, -2);
	test(*r->s4, 19, 0, *r->s3, 5, 4, -4);
	test(*r->s4, 19, 0, *r->s3, 5, 5, -5);
	test(*r->s4, 19, 0, *r->s3, 5, 6, -5);
	test(*r->s4, 19, 0, *r->s3, 9, 0, 0);
	test(*r->s4, 19, 0, *r->s3, 9, 1, -1);
	test(*r->s4, 19, 0, *r->s3, 9, 2, -1);
	test(*r->s4, 19, 0, *r->s3, 10, 0, 0);
	test(*r->s4, 19, 0, *r->s3, 10, 1, 0);
	test(*r->s4, 19, 0, *r->s3, 11, 0, 0);
	test(*r->s4, 19, 0, *r->s4, 0, 0, 0);
	test(*r->s4, 19, 0, *r->s4, 0, 1, -1);
	test(*r->s4, 19, 0, *r->s4, 0, 10, -10);
	test(*r->s4, 19, 0, *r->s4, 0, 19, -19);
	test(*r->s4, 19, 0, *r->s4, 0, 20, -20);
	test(*r->s4, 19, 0, *r->s4, 0, 21, -20);
	test(*r->s4, 19, 0, *r->s4, 1, 0, 0);
	test(*r->s4, 19, 0, *r->s4, 1, 1, -1);
	test(*r->s4, 19, 0, *r->s4, 1, 9, -9);
	test(*r->s4, 19, 0, *r->s4, 1, 18, -18);
	test(*r->s4, 19, 0, *r->s4, 1, 19, -19);
	test(*r->s4, 19, 0, *r->s4, 1, 20, -19);
	test(*r->s4, 19, 0, *r->s4, 10, 0, 0);
	test(*r->s4, 19, 0, *r->s4, 10, 1, -1);
	test(*r->s4, 19, 0, *r->s4, 10, 5, -5);
	test(*r->s4, 19, 0, *r->s4, 10, 9, -9);
	test(*r->s4, 19, 0, *r->s4, 10, 10, -10);
	test(*r->s4, 19, 0, *r->s4, 10, 11, -10);
}

template <class S>
void
test51(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 19, 0, *r->s4, 19, 0, 0);
	test(*r->s4, 19, 0, *r->s4, 19, 1, -1);
	test(*r->s4, 19, 0, *r->s4, 19, 2, -1);
	test(*r->s4, 19, 0, *r->s4, 20, 0, 0);
	test(*r->s4, 19, 0, *r->s4, 20, 1, 0);
	test(*r->s4, 19, 0, *r->s4, 21, 0, 0);
	test(*r->s4, 19, 1, *r->s1, 0, 0, 1);
	test(*r->s4, 19, 1, *r->s1, 0, 1, 1);
	test(*r->s4, 19, 1, *r->s1, 1, 0, 0);
	test(*r->s4, 19, 1, *r->s2, 0, 0, 1);
	test(*r->s4, 19, 1, *r->s2, 0, 1, 19);
	test(*r->s4, 19, 1, *r->s2, 0, 2, 19);
	test(*r->s4, 19, 1, *r->s2, 0, 4, 19);
	test(*r->s4, 19, 1, *r->s2, 0, 5, 19);
	test(*r->s4, 19, 1, *r->s2, 0, 6, 19);
	test(*r->s4, 19, 1, *r->s2, 1, 0, 1);
	test(*r->s4, 19, 1, *r->s2, 1, 1, 18);
	test(*r->s4, 19, 1, *r->s2, 1, 2, 18);
	test(*r->s4, 19, 1, *r->s2, 1, 3, 18);
	test(*r->s4, 19, 1, *r->s2, 1, 4, 18);
	test(*r->s4, 19, 1, *r->s2, 1, 5, 18);
	test(*r->s4, 19, 1, *r->s2, 2, 0, 1);
	test(*r->s4, 19, 1, *r->s2, 2, 1, 17);
	test(*r->s4, 19, 1, *r->s2, 2, 2, 17);
	test(*r->s4, 19, 1, *r->s2, 2, 3, 17);
	test(*r->s4, 19, 1, *r->s2, 2, 4, 17);
	test(*r->s4, 19, 1, *r->s2, 4, 0, 1);
	test(*r->s4, 19, 1, *r->s2, 4, 1, 15);
	test(*r->s4, 19, 1, *r->s2, 4, 2, 15);
	test(*r->s4, 19, 1, *r->s2, 5, 0, 1);
	test(*r->s4, 19, 1, *r->s2, 5, 1, 1);
	test(*r->s4, 19, 1, *r->s2, 6, 0, 0);
	test(*r->s4, 19, 1, *r->s3, 0, 0, 1);
	test(*r->s4, 19, 1, *r->s3, 0, 1, 19);
	test(*r->s4, 19, 1, *r->s3, 0, 5, 19);
	test(*r->s4, 19, 1, *r->s3, 0, 9, 19);
	test(*r->s4, 19, 1, *r->s3, 0, 10, 19);
	test(*r->s4, 19, 1, *r->s3, 0, 11, 19);
	test(*r->s4, 19, 1, *r->s3, 1, 0, 1);
	test(*r->s4, 19, 1, *r->s3, 1, 1, 18);
	test(*r->s4, 19, 1, *r->s3, 1, 4, 18);
	test(*r->s4, 19, 1, *r->s3, 1, 8, 18);
	test(*r->s4, 19, 1, *r->s3, 1, 9, 18);
	test(*r->s4, 19, 1, *r->s3, 1, 10, 18);
	test(*r->s4, 19, 1, *r->s3, 5, 0, 1);
	test(*r->s4, 19, 1, *r->s3, 5, 1, 14);
	test(*r->s4, 19, 1, *r->s3, 5, 2, 14);
	test(*r->s4, 19, 1, *r->s3, 5, 4, 14);
	test(*r->s4, 19, 1, *r->s3, 5, 5, 14);
	test(*r->s4, 19, 1, *r->s3, 5, 6, 14);
	test(*r->s4, 19, 1, *r->s3, 9, 0, 1);
	test(*r->s4, 19, 1, *r->s3, 9, 1, 10);
	test(*r->s4, 19, 1, *r->s3, 9, 2, 10);
	test(*r->s4, 19, 1, *r->s3, 10, 0, 1);
	test(*r->s4, 19, 1, *r->s3, 10, 1, 1);
	test(*r->s4, 19, 1, *r->s3, 11, 0, 0);
	test(*r->s4, 19, 1, *r->s4, 0, 0, 1);
	test(*r->s4, 19, 1, *r->s4, 0, 1, 19);
	test(*r->s4, 19, 1, *r->s4, 0, 10, 19);
	test(*r->s4, 19, 1, *r->s4, 0, 19, 19);
	test(*r->s4, 19, 1, *r->s4, 0, 20, 19);
	test(*r->s4, 19, 1, *r->s4, 0, 21, 19);
	test(*r->s4, 19, 1, *r->s4, 1, 0, 1);
	test(*r->s4, 19, 1, *r->s4, 1, 1, 18);
	test(*r->s4, 19, 1, *r->s4, 1, 9, 18);
	test(*r->s4, 19, 1, *r->s4, 1, 18, 18);
	test(*r->s4, 19, 1, *r->s4, 1, 19, 18);
	test(*r->s4, 19, 1, *r->s4, 1, 20, 18);
	test(*r->s4, 19, 1, *r->s4, 10, 0, 1);
	test(*r->s4, 19, 1, *r->s4, 10, 1, 9);
	test(*r->s4, 19, 1, *r->s4, 10, 5, 9);
	test(*r->s4, 19, 1, *r->s4, 10, 9, 9);
	test(*r->s4, 19, 1, *r->s4, 10, 10, 9);
	test(*r->s4, 19, 1, *r->s4, 10, 11, 9);
	test(*r->s4, 19, 1, *r->s4, 19, 0, 1);
	test(*r->s4, 19, 1, *r->s4, 19, 1, 0);
	test(*r->s4, 19, 1, *r->s4, 19, 2, 0);
	test(*r->s4, 19, 1, *r->s4, 20, 0, 1);
	test(*r->s4, 19, 1, *r->s4, 20, 1, 1);
	test(*r->s4, 19, 1, *r->s4, 21, 0, 0);
	test(*r->s4, 19, 2, *r->s1, 0, 0, 1);
	test(*r->s4, 19, 2, *r->s1, 0, 1, 1);
	test(*r->s4, 19, 2, *r->s1, 1, 0, 0);
	test(*r->s4, 19, 2, *r->s2, 0, 0, 1);
	test(*r->s4, 19, 2, *r->s2, 0, 1, 19);
	test(*r->s4, 19, 2, *r->s2, 0, 2, 19);
	test(*r->s4, 19, 2, *r->s2, 0, 4, 19);
	test(*r->s4, 19, 2, *r->s2, 0, 5, 19);
	test(*r->s4, 19, 2, *r->s2, 0, 6, 19);
	test(*r->s4, 19, 2, *r->s2, 1, 0, 1);
	test(*r->s4, 19, 2, *r->s2, 1, 1, 18);
	test(*r->s4, 19, 2, *r->s2, 1, 2, 18);
	test(*r->s4, 19, 2, *r->s2, 1, 3, 18);
	test(*r->s4, 19, 2, *r->s2, 1, 4, 18);
	test(*r->s4, 19, 2, *r->s2, 1, 5, 18);
	test(*r->s4, 19, 2, *r->s2, 2, 0, 1);
	test(*r->s4, 19, 2, *r->s2, 2, 1, 17);
	test(*r->s4, 19, 2, *r->s2, 2, 2, 17);
	test(*r->s4, 19, 2, *r->s2, 2, 3, 17);
	test(*r->s4, 19, 2, *r->s2, 2, 4, 17);
}

template <class S>
void
test52(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 19, 2, *r->s2, 4, 0, 1);
	test(*r->s4, 19, 2, *r->s2, 4, 1, 15);
	test(*r->s4, 19, 2, *r->s2, 4, 2, 15);
	test(*r->s4, 19, 2, *r->s2, 5, 0, 1);
	test(*r->s4, 19, 2, *r->s2, 5, 1, 1);
	test(*r->s4, 19, 2, *r->s2, 6, 0, 0);
	test(*r->s4, 19, 2, *r->s3, 0, 0, 1);
	test(*r->s4, 19, 2, *r->s3, 0, 1, 19);
	test(*r->s4, 19, 2, *r->s3, 0, 5, 19);
	test(*r->s4, 19, 2, *r->s3, 0, 9, 19);
	test(*r->s4, 19, 2, *r->s3, 0, 10, 19);
	test(*r->s4, 19, 2, *r->s3, 0, 11, 19);
	test(*r->s4, 19, 2, *r->s3, 1, 0, 1);
	test(*r->s4, 19, 2, *r->s3, 1, 1, 18);
	test(*r->s4, 19, 2, *r->s3, 1, 4, 18);
	test(*r->s4, 19, 2, *r->s3, 1, 8, 18);
	test(*r->s4, 19, 2, *r->s3, 1, 9, 18);
	test(*r->s4, 19, 2, *r->s3, 1, 10, 18);
	test(*r->s4, 19, 2, *r->s3, 5, 0, 1);
	test(*r->s4, 19, 2, *r->s3, 5, 1, 14);
	test(*r->s4, 19, 2, *r->s3, 5, 2, 14);
	test(*r->s4, 19, 2, *r->s3, 5, 4, 14);
	test(*r->s4, 19, 2, *r->s3, 5, 5, 14);
	test(*r->s4, 19, 2, *r->s3, 5, 6, 14);
	test(*r->s4, 19, 2, *r->s3, 9, 0, 1);
	test(*r->s4, 19, 2, *r->s3, 9, 1, 10);
	test(*r->s4, 19, 2, *r->s3, 9, 2, 10);
	test(*r->s4, 19, 2, *r->s3, 10, 0, 1);
	test(*r->s4, 19, 2, *r->s3, 10, 1, 1);
	test(*r->s4, 19, 2, *r->s3, 11, 0, 0);
	test(*r->s4, 19, 2, *r->s4, 0, 0, 1);
	test(*r->s4, 19, 2, *r->s4, 0, 1, 19);
	test(*r->s4, 19, 2, *r->s4, 0, 10, 19);
	test(*r->s4, 19, 2, *r->s4, 0, 19, 19);
	test(*r->s4, 19, 2, *r->s4, 0, 20, 19);
	test(*r->s4, 19, 2, *r->s4, 0, 21, 19);
	test(*r->s4, 19, 2, *r->s4, 1, 0, 1);
	test(*r->s4, 19, 2, *r->s4, 1, 1, 18);
	test(*r->s4, 19, 2, *r->s4, 1, 9, 18);
	test(*r->s4, 19, 2, *r->s4, 1, 18, 18);
	test(*r->s4, 19, 2, *r->s4, 1, 19, 18);
	test(*r->s4, 19, 2, *r->s4, 1, 20, 18);
	test(*r->s4, 19, 2, *r->s4, 10, 0, 1);
	test(*r->s4, 19, 2, *r->s4, 10, 1, 9);
	test(*r->s4, 19, 2, *r->s4, 10, 5, 9);
	test(*r->s4, 19, 2, *r->s4, 10, 9, 9);
	test(*r->s4, 19, 2, *r->s4, 10, 10, 9);
	test(*r->s4, 19, 2, *r->s4, 10, 11, 9);
	test(*r->s4, 19, 2, *r->s4, 19, 0, 1);
	test(*r->s4, 19, 2, *r->s4, 19, 1, 0);
	test(*r->s4, 19, 2, *r->s4, 19, 2, 0);
	test(*r->s4, 19, 2, *r->s4, 20, 0, 1);
	test(*r->s4, 19, 2, *r->s4, 20, 1, 1);
	test(*r->s4, 19, 2, *r->s4, 21, 0, 0);
	test(*r->s4, 20, 0, *r->s1, 0, 0, 0);
	test(*r->s4, 20, 0, *r->s1, 0, 1, 0);
	test(*r->s4, 20, 0, *r->s1, 1, 0, 0);
	test(*r->s4, 20, 0, *r->s2, 0, 0, 0);
	test(*r->s4, 20, 0, *r->s2, 0, 1, -1);
	test(*r->s4, 20, 0, *r->s2, 0, 2, -2);
	test(*r->s4, 20, 0, *r->s2, 0, 4, -4);
	test(*r->s4, 20, 0, *r->s2, 0, 5, -5);
	test(*r->s4, 20, 0, *r->s2, 0, 6, -5);
	test(*r->s4, 20, 0, *r->s2, 1, 0, 0);
	test(*r->s4, 20, 0, *r->s2, 1, 1, -1);
	test(*r->s4, 20, 0, *r->s2, 1, 2, -2);
	test(*r->s4, 20, 0, *r->s2, 1, 3, -3);
	test(*r->s4, 20, 0, *r->s2, 1, 4, -4);
	test(*r->s4, 20, 0, *r->s2, 1, 5, -4);
	test(*r->s4, 20, 0, *r->s2, 2, 0, 0);
	test(*r->s4, 20, 0, *r->s2, 2, 1, -1);
	test(*r->s4, 20, 0, *r->s2, 2, 2, -2);
	test(*r->s4, 20, 0, *r->s2, 2, 3, -3);
	test(*r->s4, 20, 0, *r->s2, 2, 4, -3);
	test(*r->s4, 20, 0, *r->s2, 4, 0, 0);
	test(*r->s4, 20, 0, *r->s2, 4, 1, -1);
	test(*r->s4, 20, 0, *r->s2, 4, 2, -1);
	test(*r->s4, 20, 0, *r->s2, 5, 0, 0);
	test(*r->s4, 20, 0, *r->s2, 5, 1, 0);
	test(*r->s4, 20, 0, *r->s2, 6, 0, 0);
	test(*r->s4, 20, 0, *r->s3, 0, 0, 0);
	test(*r->s4, 20, 0, *r->s3, 0, 1, -1);
	test(*r->s4, 20, 0, *r->s3, 0, 5, -5);
	test(*r->s4, 20, 0, *r->s3, 0, 9, -9);
	test(*r->s4, 20, 0, *r->s3, 0, 10, -10);
	test(*r->s4, 20, 0, *r->s3, 0, 11, -10);
	test(*r->s4, 20, 0, *r->s3, 1, 0, 0);
	test(*r->s4, 20, 0, *r->s3, 1, 1, -1);
	test(*r->s4, 20, 0, *r->s3, 1, 4, -4);
	test(*r->s4, 20, 0, *r->s3, 1, 8, -8);
	test(*r->s4, 20, 0, *r->s3, 1, 9, -9);
	test(*r->s4, 20, 0, *r->s3, 1, 10, -9);
	test(*r->s4, 20, 0, *r->s3, 5, 0, 0);
	test(*r->s4, 20, 0, *r->s3, 5, 1, -1);
	test(*r->s4, 20, 0, *r->s3, 5, 2, -2);
	test(*r->s4, 20, 0, *r->s3, 5, 4, -4);
	test(*r->s4, 20, 0, *r->s3, 5, 5, -5);
	test(*r->s4, 20, 0, *r->s3, 5, 6, -5);
	test(*r->s4, 20, 0, *r->s3, 9, 0, 0);
	test(*r->s4, 20, 0, *r->s3, 9, 1, -1);
}

template <class S>
void
test53(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 20, 0, *r->s3, 9, 2, -1);
	test(*r->s4, 20, 0, *r->s3, 10, 0, 0);
	test(*r->s4, 20, 0, *r->s3, 10, 1, 0);
	test(*r->s4, 20, 0, *r->s3, 11, 0, 0);
	test(*r->s4, 20, 0, *r->s4, 0, 0, 0);
	test(*r->s4, 20, 0, *r->s4, 0, 1, -1);
	test(*r->s4, 20, 0, *r->s4, 0, 10, -10);
	test(*r->s4, 20, 0, *r->s4, 0, 19, -19);
	test(*r->s4, 20, 0, *r->s4, 0, 20, -20);
	test(*r->s4, 20, 0, *r->s4, 0, 21, -20);
	test(*r->s4, 20, 0, *r->s4, 1, 0, 0);
	test(*r->s4, 20, 0, *r->s4, 1, 1, -1);
	test(*r->s4, 20, 0, *r->s4, 1, 9, -9);
	test(*r->s4, 20, 0, *r->s4, 1, 18, -18);
	test(*r->s4, 20, 0, *r->s4, 1, 19, -19);
	test(*r->s4, 20, 0, *r->s4, 1, 20, -19);
	test(*r->s4, 20, 0, *r->s4, 10, 0, 0);
	test(*r->s4, 20, 0, *r->s4, 10, 1, -1);
	test(*r->s4, 20, 0, *r->s4, 10, 5, -5);
	test(*r->s4, 20, 0, *r->s4, 10, 9, -9);
	test(*r->s4, 20, 0, *r->s4, 10, 10, -10);
	test(*r->s4, 20, 0, *r->s4, 10, 11, -10);
	test(*r->s4, 20, 0, *r->s4, 19, 0, 0);
	test(*r->s4, 20, 0, *r->s4, 19, 1, -1);
	test(*r->s4, 20, 0, *r->s4, 19, 2, -1);
	test(*r->s4, 20, 0, *r->s4, 20, 0, 0);
	test(*r->s4, 20, 0, *r->s4, 20, 1, 0);
	test(*r->s4, 20, 0, *r->s4, 21, 0, 0);
	test(*r->s4, 20, 1, *r->s1, 0, 0, 0);
	test(*r->s4, 20, 1, *r->s1, 0, 1, 0);
	test(*r->s4, 20, 1, *r->s1, 1, 0, 0);
	test(*r->s4, 20, 1, *r->s2, 0, 0, 0);
	test(*r->s4, 20, 1, *r->s2, 0, 1, -1);
	test(*r->s4, 20, 1, *r->s2, 0, 2, -2);
	test(*r->s4, 20, 1, *r->s2, 0, 4, -4);
	test(*r->s4, 20, 1, *r->s2, 0, 5, -5);
	test(*r->s4, 20, 1, *r->s2, 0, 6, -5);
	test(*r->s4, 20, 1, *r->s2, 1, 0, 0);
	test(*r->s4, 20, 1, *r->s2, 1, 1, -1);
	test(*r->s4, 20, 1, *r->s2, 1, 2, -2);
	test(*r->s4, 20, 1, *r->s2, 1, 3, -3);
	test(*r->s4, 20, 1, *r->s2, 1, 4, -4);
	test(*r->s4, 20, 1, *r->s2, 1, 5, -4);
	test(*r->s4, 20, 1, *r->s2, 2, 0, 0);
	test(*r->s4, 20, 1, *r->s2, 2, 1, -1);
	test(*r->s4, 20, 1, *r->s2, 2, 2, -2);
	test(*r->s4, 20, 1, *r->s2, 2, 3, -3);
	test(*r->s4, 20, 1, *r->s2, 2, 4, -3);
	test(*r->s4, 20, 1, *r->s2, 4, 0, 0);
	test(*r->s4, 20, 1, *r->s2, 4, 1, -1);
	test(*r->s4, 20, 1, *r->s2, 4, 2, -1);
	test(*r->s4, 20, 1, *r->s2, 5, 0, 0);
	test(*r->s4, 20, 1, *r->s2, 5, 1, 0);
	test(*r->s4, 20, 1, *r->s2, 6, 0, 0);
	test(*r->s4, 20, 1, *r->s3, 0, 0, 0);
	test(*r->s4, 20, 1, *r->s3, 0, 1, -1);
	test(*r->s4, 20, 1, *r->s3, 0, 5, -5);
	test(*r->s4, 20, 1, *r->s3, 0, 9, -9);
	test(*r->s4, 20, 1, *r->s3, 0, 10, -10);
	test(*r->s4, 20, 1, *r->s3, 0, 11, -10);
	test(*r->s4, 20, 1, *r->s3, 1, 0, 0);
	test(*r->s4, 20, 1, *r->s3, 1, 1, -1);
	test(*r->s4, 20, 1, *r->s3, 1, 4, -4);
	test(*r->s4, 20, 1, *r->s3, 1, 8, -8);
	test(*r->s4, 20, 1, *r->s3, 1, 9, -9);
	test(*r->s4, 20, 1, *r->s3, 1, 10, -9);
	test(*r->s4, 20, 1, *r->s3, 5, 0, 0);
	test(*r->s4, 20, 1, *r->s3, 5, 1, -1);
	test(*r->s4, 20, 1, *r->s3, 5, 2, -2);
	test(*r->s4, 20, 1, *r->s3, 5, 4, -4);
	test(*r->s4, 20, 1, *r->s3, 5, 5, -5);
	test(*r->s4, 20, 1, *r->s3, 5, 6, -5);
	test(*r->s4, 20, 1, *r->s3, 9, 0, 0);
	test(*r->s4, 20, 1, *r->s3, 9, 1, -1);
	test(*r->s4, 20, 1, *r->s3, 9, 2, -1);
	test(*r->s4, 20, 1, *r->s3, 10, 0, 0);
	test(*r->s4, 20, 1, *r->s3, 10, 1, 0);
	test(*r->s4, 20, 1, *r->s3, 11, 0, 0);
	test(*r->s4, 20, 1, *r->s4, 0, 0, 0);
	test(*r->s4, 20, 1, *r->s4, 0, 1, -1);
	test(*r->s4, 20, 1, *r->s4, 0, 10, -10);
	test(*r->s4, 20, 1, *r->s4, 0, 19, -19);
	test(*r->s4, 20, 1, *r->s4, 0, 20, -20);
	test(*r->s4, 20, 1, *r->s4, 0, 21, -20);
	test(*r->s4, 20, 1, *r->s4, 1, 0, 0);
	test(*r->s4, 20, 1, *r->s4, 1, 1, -1);
	test(*r->s4, 20, 1, *r->s4, 1, 9, -9);
	test(*r->s4, 20, 1, *r->s4, 1, 18, -18);
	test(*r->s4, 20, 1, *r->s4, 1, 19, -19);
	test(*r->s4, 20, 1, *r->s4, 1, 20, -19);
	test(*r->s4, 20, 1, *r->s4, 10, 0, 0);
	test(*r->s4, 20, 1, *r->s4, 10, 1, -1);
	test(*r->s4, 20, 1, *r->s4, 10, 5, -5);
	test(*r->s4, 20, 1, *r->s4, 10, 9, -9);
	test(*r->s4, 20, 1, *r->s4, 10, 10, -10);
	test(*r->s4, 20, 1, *r->s4, 10, 11, -10);
	test(*r->s4, 20, 1, *r->s4, 19, 0, 0);
	test(*r->s4, 20, 1, *r->s4, 19, 1, -1);
	test(*r->s4, 20, 1, *r->s4, 19, 2, -1);
	test(*r->s4, 20, 1, *r->s4, 20, 0, 0);
}

template <class S>
void
test54(pmem::obj::persistent_ptr<root> &r)
{
	test(*r->s4, 20, 1, *r->s4, 20, 1, 0);
	test(*r->s4, 20, 1, *r->s4, 21, 0, 0);
	test(*r->s4, 21, 0, *r->s1, 0, 0, 0);
	test(*r->s4, 21, 0, *r->s1, 0, 1, 0);
	test(*r->s4, 21, 0, *r->s1, 1, 0, 0);
	test(*r->s4, 21, 0, *r->s2, 0, 0, 0);
	test(*r->s4, 21, 0, *r->s2, 0, 1, 0);
	test(*r->s4, 21, 0, *r->s2, 0, 2, 0);
	test(*r->s4, 21, 0, *r->s2, 0, 4, 0);
	test(*r->s4, 21, 0, *r->s2, 0, 5, 0);
	test(*r->s4, 21, 0, *r->s2, 0, 6, 0);
	test(*r->s4, 21, 0, *r->s2, 1, 0, 0);
	test(*r->s4, 21, 0, *r->s2, 1, 1, 0);
	test(*r->s4, 21, 0, *r->s2, 1, 2, 0);
	test(*r->s4, 21, 0, *r->s2, 1, 3, 0);
	test(*r->s4, 21, 0, *r->s2, 1, 4, 0);
	test(*r->s4, 21, 0, *r->s2, 1, 5, 0);
	test(*r->s4, 21, 0, *r->s2, 2, 0, 0);
	test(*r->s4, 21, 0, *r->s2, 2, 1, 0);
	test(*r->s4, 21, 0, *r->s2, 2, 2, 0);
	test(*r->s4, 21, 0, *r->s2, 2, 3, 0);
	test(*r->s4, 21, 0, *r->s2, 2, 4, 0);
	test(*r->s4, 21, 0, *r->s2, 4, 0, 0);
	test(*r->s4, 21, 0, *r->s2, 4, 1, 0);
	test(*r->s4, 21, 0, *r->s2, 4, 2, 0);
	test(*r->s4, 21, 0, *r->s2, 5, 0, 0);
	test(*r->s4, 21, 0, *r->s2, 5, 1, 0);
	test(*r->s4, 21, 0, *r->s2, 6, 0, 0);
	test(*r->s4, 21, 0, *r->s3, 0, 0, 0);
	test(*r->s4, 21, 0, *r->s3, 0, 1, 0);
	test(*r->s4, 21, 0, *r->s3, 0, 5, 0);
	test(*r->s4, 21, 0, *r->s3, 0, 9, 0);
	test(*r->s4, 21, 0, *r->s3, 0, 10, 0);
	test(*r->s4, 21, 0, *r->s3, 0, 11, 0);
	test(*r->s4, 21, 0, *r->s3, 1, 0, 0);
	test(*r->s4, 21, 0, *r->s3, 1, 1, 0);
	test(*r->s4, 21, 0, *r->s3, 1, 4, 0);
	test(*r->s4, 21, 0, *r->s3, 1, 8, 0);
	test(*r->s4, 21, 0, *r->s3, 1, 9, 0);
	test(*r->s4, 21, 0, *r->s3, 1, 10, 0);
	test(*r->s4, 21, 0, *r->s3, 5, 0, 0);
	test(*r->s4, 21, 0, *r->s3, 5, 1, 0);
	test(*r->s4, 21, 0, *r->s3, 5, 2, 0);
	test(*r->s4, 21, 0, *r->s3, 5, 4, 0);
	test(*r->s4, 21, 0, *r->s3, 5, 5, 0);
	test(*r->s4, 21, 0, *r->s3, 5, 6, 0);
	test(*r->s4, 21, 0, *r->s3, 9, 0, 0);
	test(*r->s4, 21, 0, *r->s3, 9, 1, 0);
	test(*r->s4, 21, 0, *r->s3, 9, 2, 0);
	test(*r->s4, 21, 0, *r->s3, 10, 0, 0);
	test(*r->s4, 21, 0, *r->s3, 10, 1, 0);
	test(*r->s4, 21, 0, *r->s3, 11, 0, 0);
	test(*r->s4, 21, 0, *r->s4, 0, 0, 0);
	test(*r->s4, 21, 0, *r->s4, 0, 1, 0);
	test(*r->s4, 21, 0, *r->s4, 0, 10, 0);
	test(*r->s4, 21, 0, *r->s4, 0, 19, 0);
	test(*r->s4, 21, 0, *r->s4, 0, 20, 0);
	test(*r->s4, 21, 0, *r->s4, 0, 21, 0);
	test(*r->s4, 21, 0, *r->s4, 1, 0, 0);
	test(*r->s4, 21, 0, *r->s4, 1, 1, 0);
	test(*r->s4, 21, 0, *r->s4, 1, 9, 0);
	test(*r->s4, 21, 0, *r->s4, 1, 18, 0);
	test(*r->s4, 21, 0, *r->s4, 1, 19, 0);
	test(*r->s4, 21, 0, *r->s4, 1, 20, 0);
	test(*r->s4, 21, 0, *r->s4, 10, 0, 0);
	test(*r->s4, 21, 0, *r->s4, 10, 1, 0);
	test(*r->s4, 21, 0, *r->s4, 10, 5, 0);
	test(*r->s4, 21, 0, *r->s4, 10, 9, 0);
	test(*r->s4, 21, 0, *r->s4, 10, 10, 0);
	test(*r->s4, 21, 0, *r->s4, 10, 11, 0);
	test(*r->s4, 21, 0, *r->s4, 19, 0, 0);
	test(*r->s4, 21, 0, *r->s4, 19, 1, 0);
	test(*r->s4, 21, 0, *r->s4, 19, 2, 0);
	test(*r->s4, 21, 0, *r->s4, 20, 0, 0);
	test(*r->s4, 21, 0, *r->s4, 20, 1, 0);
	test(*r->s4, 21, 0, *r->s4, 21, 0, 0);
}

template <class S>
void
test55(pmem::obj::persistent_ptr<root> &r)
{
	test_npos(*r->s1, 0, 0, *r->s1, 0, 0);
	test_npos(*r->s1, 0, 0, *r->s2, 0, -5);
	test_npos(*r->s2, 0, 0, *r->s3, 0, -10);
	test_npos(*r->s2, 0, 0, *r->s3, 1, -9);
	test_npos(*r->s2, 0, 0, *r->s3, 5, -5);
}

void
run(pmem::obj::pool<root> &pop)
{
	auto r = pop.root();

	try {
		nvobj::transaction::run(pop, [&] {
			r->s1 = nvobj::make_persistent<pmem_exp::string>("");
			r->s2 = nvobj::make_persistent<pmem_exp::string>(
				"abcde");
			r->s3 = nvobj::make_persistent<pmem_exp::string>(
				"abcdefghij");
			r->s4 = nvobj::make_persistent<pmem_exp::string>(
				"abcdefghijklmnopqrst");
		});

		typedef pmem_exp::string S;
		test0<S>(r);
		test1<S>(r);
		test2<S>(r);
		test3<S>(r);
		test4<S>(r);
		test5<S>(r);
		test6<S>(r);
		test7<S>(r);
		test8<S>(r);
		test9<S>(r);
		test10<S>(r);
		test11<S>(r);
		test12<S>(r);
		test13<S>(r);
		test14<S>(r);
		test15<S>(r);
		test16<S>(r);
		test17<S>(r);
		test18<S>(r);
		test19<S>(r);
		test20<S>(r);
		test21<S>(r);
		test22<S>(r);
		test23<S>(r);
		test24<S>(r);
		test25<S>(r);
		test26<S>(r);
		test27<S>(r);
		test28<S>(r);
		test29<S>(r);
		test30<S>(r);
		test31<S>(r);
		test32<S>(r);
		test33<S>(r);
		test34<S>(r);
		test35<S>(r);
		test36<S>(r);
		test37<S>(r);
		test38<S>(r);
		test39<S>(r);
		test40<S>(r);
		test41<S>(r);
		test42<S>(r);
		test43<S>(r);
		test44<S>(r);
		test45<S>(r);
		test46<S>(r);
		test47<S>(r);
		test48<S>(r);
		test49<S>(r);
		test50<S>(r);
		test51<S>(r);
		test52<S>(r);
		test53<S>(r);
		test54<S>(r);
		test55<S>(r);

		nvobj::transaction::run(pop, [&] {
			nvobj::delete_persistent<pmem_exp::string>(r->s1);
			nvobj::delete_persistent<pmem_exp::string>(r->s2);
			nvobj::delete_persistent<pmem_exp::string>(r->s3);
			nvobj::delete_persistent<pmem_exp::string>(r->s4);
		});
	} catch (std::exception &e) {
		UT_FATALexc(e);
	}
}

int
main(int argc, char *argv[])
{
	START();

	if (argc != 2)
		UT_FATAL("usage: %s file-name", argv[0]);

	const char *path = argv[1];

	pmem::obj::pool<root> pop;

	try {
		pop = pmem::obj::pool<root>::create(path, "basic_string",
						    PMEMOBJ_MIN_POOL,
						    S_IWUSR | S_IRUSR);
	} catch (...) {
		UT_FATAL("!pmemobj_create: %s", path);
	}

	run(pop);

	pop.close();

	return 0;
}
