#include "test.h"
#include "test_utils.h"

#define MAX_LOOPS 5

/*
 * Here there is a small collection of test cases for when inflate must return Z_BUF_ERROR.
 * 1. test_0_avail_in - no progress can be made because avail_in is 0.
 * 2. test_small_avail_out tests - no progress can be made because avail_out is not enough
 * to finish at once.
 * 3. test_scp - replicates an issue found using libnxz with scp.
 */

int test_0_avail_in() {

	z_stream strm = {0};
	Byte compr[10], uncompr[10];
	unsigned int uncomprlen = 10;
	int rc = 0, ret = 0;

	if ((rc = inflateInit(&strm)) != Z_OK){
		printf("*** failed: inflateInt returned %d\n", rc);
		return TEST_ERROR;
	}

	strm.avail_out = uncomprlen;
	strm.next_out = uncompr;
	strm.next_in = compr;
	strm.avail_in = 0;

	if ((rc = inflate(&strm, Z_NO_FLUSH)) != Z_BUF_ERROR) {
		ret = TEST_ERROR;
		printf("*** failed: inflate returned %d\n", rc);
	}

	if ((rc = inflateEnd(&strm)) != Z_OK) {
		ret = TEST_ERROR;
		printf("*** failed: inflateEnd returned %d\n", rc);
	}

	return ret;
}

int test_small_avail_out(){
	int srclen = 128;
	unsigned int comprlen = srclen*2;
	unsigned int uncomprlen = 3;
	z_stream strm = {0};
	Byte *src, compr[comprlen], uncompr[uncomprlen];
	int rc = 0, ret = 0;

	generate_random_data(srclen);
	src = ran_data;

	if (_test_nx_deflate(src, srclen, compr, &comprlen, srclen, NULL) != 0) {
		printf("*** failed: nx_deflate failed");
		return TEST_ERROR;
	}

	if ((rc = inflateInit(&strm)) != Z_OK){
		printf("*** failed: inflateInt returned %d\n", rc);
		return TEST_ERROR;
	}

	strm.avail_out = uncomprlen;
	strm.next_out = uncompr;
	strm.next_in = compr;
	strm.avail_in = comprlen;

	/* avail_out is not enough */
	if ((rc = inflate(&strm, Z_FINISH)) != Z_BUF_ERROR) {
		ret = TEST_ERROR;
		printf("*** failed: inflate returned %d\n", rc);
	}

	if ((rc = inflateEnd(&strm)) != Z_OK) {
		ret = TEST_ERROR;
		printf("*** failed: inflateEnd returned %d\n", rc);
	}

	return ret;
}

/*
 * This test uses compressed data from scp to ensure that bug #74 is reproduced.
 */
int test_scp() {
	Byte uncompr[4096];
	unsigned int uncompr_len = 4096;
	int err;
	z_stream strm;

	strm.zalloc = NULL;
	strm.zfree = NULL;
	strm.opaque = (voidpf) 0;

	if ((err = inflateInit(&strm)) != Z_OK) {
		printf("*** failed: inflateInit returned %d\n", err);
		return TEST_ERROR;
	}

/* This is compressed data extracted from a scp call to reproduce the behavior
  from bug #74. This data will be fully consumed but partially processed leaving
  1 byte on used_in after the second inflate. In this state libnxz was just
  waiting for more data and returning Z_OK on every call. The expected behavior
  is to return Z_BUF_ERROR as no progress was made on the inflate call. */
	Byte compr[611] ={ 0x78, 0x9c, 0x00, 0x5b, 0x02, 0xa4, 0xfd, 0x50,
0x00, 0x00, 0x00, 0x17, 0x68, 0x6f, 0x73, 0x74,
0x6b, 0x65, 0x79, 0x73, 0x2d, 0x30, 0x30, 0x40,
0x6f, 0x70, 0x65, 0x6e, 0x73, 0x73, 0x68, 0x2e,
0x63, 0x6f, 0x6d, 0x00, 0x00, 0x00, 0x01, 0x97,
0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d,
0x72, 0x73, 0x61, 0x00, 0x00, 0x00, 0x03, 0x01,
0x00, 0x01, 0x00, 0x00, 0x01, 0x81, 0x00, 0xca,
0x5f, 0x1f, 0x9f, 0xb8, 0x3a, 0x33, 0xba, 0xaa,
0xcd, 0x63, 0x41, 0xfc, 0xaf, 0xcf, 0x66, 0x8c,
0xc4, 0x33, 0x29, 0x5d, 0x3d, 0xe0, 0xa0, 0x4c,
0x9c, 0xa2, 0x82, 0xd3, 0x6b, 0xc4, 0x4a, 0xb6,
0x78, 0x0b, 0xb2, 0xfa, 0x33, 0xb3, 0xc6, 0x70,
0x1b, 0x3b, 0x46, 0x34, 0xf2, 0xf8, 0x5d, 0xef,
0x7b, 0x6b, 0xe9, 0xe7, 0x61, 0x62, 0x12, 0xe3,
0x35, 0x9d, 0x3e, 0x41, 0x44, 0x0a, 0x2b, 0x18,
0xe4, 0x9b, 0x6c, 0x8b, 0xe2, 0xb9, 0x38, 0x50,
0x82, 0x15, 0x83, 0xb0, 0x8b, 0xb0, 0x5b, 0xd2,
0xd1, 0xa7, 0xff, 0x8e, 0xd8, 0x9e, 0x6f, 0xbe,
0x17, 0xdc, 0xbb, 0xfd, 0x99, 0x5c, 0x3a, 0x96,
0xb2, 0x7d, 0xba, 0xec, 0xea, 0x1c, 0x3d, 0xf5,
0xd8, 0xd6, 0xe2, 0x8e, 0x72, 0x5a, 0xfe, 0x07,
0x0e, 0x1a, 0xc3, 0x97, 0x45, 0x2c, 0x41, 0x17,
0x1c, 0x63, 0xac, 0x19, 0x7c, 0x62, 0x3b, 0x28,
0xe2, 0xc8, 0xb7, 0x4a, 0xa4, 0xb5, 0xb0, 0x0a,
0x1c, 0xec, 0x3b, 0x82, 0xc0, 0xa0, 0x0d, 0x87,
0xfe, 0x60, 0x93, 0x6d, 0x45, 0xe7, 0x20, 0x91,
0x82, 0x62, 0xc8, 0xdf, 0xba, 0x12, 0x7b, 0xf0,
0xde, 0x79, 0x8a, 0xca, 0x4b, 0x6f, 0xd8, 0xa4,
0x66, 0x8e, 0xd7, 0xf6, 0xf5, 0x0c, 0x1b, 0xee,
0x73, 0x90, 0x0a, 0xdb, 0x6e, 0x10, 0x73, 0x88,
0x3e, 0x23, 0x6a, 0xe9, 0xab, 0x16, 0xe9, 0x07,
0x16, 0x77, 0x16, 0x3e, 0x73, 0xe9, 0x8c, 0xed,
0x31, 0xfa, 0x7f, 0x4a, 0x31, 0x01, 0x1a, 0x4f,
0x56, 0x6d, 0x7a, 0x89, 0xdf, 0x9e, 0x3f, 0xc7,
0x55, 0xb0, 0xb3, 0xa5, 0xd6, 0x8f, 0x86, 0xb2,
0xfb, 0x4a, 0x59, 0xd9, 0x53, 0xc3, 0x4e, 0x82,
0x75, 0x82, 0xf0, 0x68, 0x20, 0x6e, 0xae, 0x15,
0xa2, 0x41, 0xca, 0xfd, 0xad, 0x33, 0x4c, 0xb4,
0xc9, 0xa5, 0x2a, 0xc4, 0xa2, 0xfa, 0x8f, 0xc1,
0x78, 0xdf, 0x66, 0xc0, 0x3d, 0xf8, 0x6f, 0x3d,
0x23, 0x85, 0xa5, 0x1d, 0xf6, 0x5c, 0xb5, 0xd2,
0xc1, 0x46, 0x07, 0xbb, 0xb2, 0xe3, 0xb0, 0x96,
0xc0, 0xbf, 0x06, 0xc8, 0xb4, 0xc4, 0xf4, 0x93,
0xe1, 0x4e, 0x79, 0x66, 0x56, 0xd3, 0xc2, 0x23,
0xda, 0x52, 0xd3, 0x20, 0x07, 0xc6, 0x18, 0x2f,
0x53, 0xe5, 0xb7, 0xab, 0x22, 0x5e, 0xb9, 0x46,
0xa0, 0x11, 0x88, 0xd3, 0x6c, 0x13, 0xff, 0x8c,
0x3f, 0x52, 0x7b, 0x7a, 0xb3, 0x96, 0x31, 0x8a,
0x8e, 0xa7, 0xd0, 0xea, 0x55, 0xdc, 0xed, 0xd7,
0x04, 0xbe, 0xb6, 0x94, 0x5f, 0xa5, 0xb2, 0x79,
0x2e, 0x50, 0x2d, 0x51, 0xed, 0xf1, 0xad, 0x37,
0x2f, 0x19, 0x68, 0x31, 0x53, 0xdb, 0xce, 0x0b,
0xb6, 0xed, 0x5b, 0x0f, 0x29, 0x49, 0xbd, 0x59,
0x2a, 0xf4, 0xd7, 0xcb, 0x7b, 0x83, 0x8b, 0xa9,
0xf1, 0xb0, 0x90, 0x70, 0x26, 0x48, 0x1b, 0x00,
0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x13, 0x65,
0x63, 0x64, 0x73, 0x61, 0x2d, 0x73, 0x68, 0x61,
0x32, 0x2d, 0x6e, 0x69, 0x73, 0x74, 0x70, 0x32,
0x35, 0x36, 0x00, 0x00, 0x00, 0x08, 0x6e, 0x69,
0x73, 0x74, 0x70, 0x32, 0x35, 0x36, 0x00, 0x00,
0x00, 0x41, 0x04, 0x43, 0x3d, 0x9e, 0x59, 0x3d,
0xb3, 0xb0, 0x11, 0x71, 0x77, 0xd4, 0x8f, 0xf5,
0x6e, 0xed, 0x5a, 0xcd, 0xba, 0x04, 0xee, 0x8a,
0x17, 0xa7, 0x84, 0xee, 0x71, 0x8c, 0x30, 0x3d,
0x08, 0x58, 0x43, 0x88, 0x83, 0x82, 0x08, 0xce,
0xf4, 0x9d, 0xe9, 0xab, 0xf2, 0xf6, 0x5e, 0x80,
0xb4, 0xd8, 0x84, 0xcc, 0xb3, 0x03, 0x71, 0xca,
0x4c, 0x67, 0x8a, 0x85, 0xe7, 0x1d, 0x50, 0x52,
0x42, 0x32, 0x85, 0x00, 0x00, 0x00, 0x33, 0x00,
0x00, 0x00, 0x0b, 0x73, 0x73, 0x68, 0x2d, 0x65,
0x64, 0x32, 0x35, 0x35, 0x31, 0x39, 0x00, 0x00,
0x00, 0x20, 0x9c, 0xf8, 0x6d, 0x70, 0x35, 0x1e,
0xba, 0xdc, 0x44, 0x47, 0x47, 0x19, 0x05, 0x83,
0x80, 0x23, 0x2a, 0x37, 0x25, 0x5d, 0xe7, 0xd8,
0x65, 0x44, 0x4d, 0x98, 0xdb, 0x02, 0x61, 0xdf,
0x8b, 0xfd, 0x02};

	strm.next_in = compr;
	strm.avail_in = sizeof(compr)/ sizeof(Byte);

	for (int i = MAX_LOOPS; i >= 0; i--) {
		strm.next_out = uncompr;
		strm.avail_out = uncompr_len;

		err = inflate(&strm, Z_NO_FLUSH);
		switch (err) {
		case Z_OK:
			break;
		case Z_BUF_ERROR:
			printf("Success 1\n");
			goto out_loop;
		default:
			printf ("*** failed: inflate returned %d\n", err);
			if((err = inflateEnd(&strm)) != Z_OK)
				printf("*** failed: inflateEnd returned %d\n", err);
			return TEST_ERROR;
		}
	}

out_loop:
	if (err == Z_OK) {
		printf("*** failed: inflate didn't return Z_BUF_ERROR as expected\n");
		if((err = inflateEnd(&strm)) != Z_OK)
			printf("*** failed: inflateEnd returned %d\n", err);
		return TEST_ERROR;
	}

/* This compressed data continues the one before. It will also not progress
   after the second call that must return Z_BUF_ERROR. */
	Byte compr2[92] = {
0x88, 0x05, 0xa8, 0x88, 0x21, 0x5b, 0x3f, 0x23,
0x3f, 0x37, 0x55, 0x3f, 0x27, 0x31, 0x37, 0x57,
0x5f, 0x0f, 0x68, 0x80, 0x7e, 0x62, 0x69, 0x49,
0x46, 0x7e, 0x51, 0x66, 0x55, 0x6a, 0x4a, 0x3c,
0x28, 0xc9, 0x59, 0x19, 0x59, 0x29, 0x00, 0x69,
0x85, 0xfc, 0x82, 0x92, 0xcc, 0xfc, 0xbc, 0x62,
0x2b, 0x85, 0xc4, 0xf4, 0xd4, 0xbc, 0x12, 0xdd,
0xb4, 0xfc, 0xa2, 0xf2, 0xc4, 0xa2, 0x94, 0xcc,
0xbc, 0x74, 0x85, 0x82, 0xfc, 0x22, 0x54, 0x7e,
0x49, 0xa5, 0x42, 0x69, 0x71, 0x6a, 0x91, 0x6e,
0x51, 0xb2, 0x42, 0x85, 0xa1, 0x21, 0x92, 0x14,
0xc8, 0x3e, 0x80, 0x00};

	strm.next_in = compr2;
	strm.avail_in = 92;

	for (int i = MAX_LOOPS; i >= 0; i--) {
		strm.next_out = uncompr;
		strm.avail_out = uncompr_len;

		err = inflate(&strm, Z_NO_FLUSH);
		switch (err) {
		case Z_OK:
			break;
		case Z_BUF_ERROR:
			if ((err = inflateEnd(&strm)) != Z_OK) {
				printf("*** failed: inflateEnd returned %d\n", err);
				return TEST_ERROR;
			}
			printf("Success 2 \n");
			return 0;
		default:
			printf ("*** failed: inflate returned %d\n", err);
			if((err = inflateEnd(&strm)) != Z_OK)
				printf("*** failed: inflateEnd returned %d\n", err);
			return TEST_ERROR;
		}
	}

	printf("*** failed: inflate didn't return Z_BUF_ERROR as expected\n");
	if((err = inflateEnd(&strm)) != Z_OK)
		printf("*** failed: inflateEnd returned %d\n", err);
	return TEST_ERROR;
}

int main( int argc, char* argv[]) {
	printf ("Running test_0_avail_in:\n");
	if (test_0_avail_in()) return TEST_ERROR;
	printf ("Success\n");
	printf ("Running test_small_avail_out:\n");
	if (test_small_avail_out()) return TEST_ERROR;
	printf ("Success\n");
	printf ("Running test_scp:\n");
	if (test_scp()) return TEST_ERROR;
	printf ("Success\n");
	return 0;
}
