#include <stdbool.h>
#include <stddef.h>

#include "s390_drbg_sha512.h"

/*
 * SHA-512 DRBG mechanism test vectors
 * (obtained from NIST)
 */
const drbg_sha512_test_vec_t DRBG_SHA512_TEST_VEC[] = {
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x6b,0x50,0xa7,0xd8,0xf8,0xa5,0x5d,0x7a,0x3d,0xf8,0xbb,0x40,0xbc,0xc3,0xb7,0x22,
0xd8,0x70,0x8d,0xe6,0x7f,0xda,0x01,0x0b,0x03,0xc4,0xc8,0x4d,0x72,0x09,0x6f,0x8c,
},
.inst.nonce = (unsigned char[]){
0x3e,0xc6,0x49,0xcc,0x62,0x56,0xd9,0xfa,0x31,0xdb,0x7a,0x29,0x04,0xaa,0xf0,0x25,
},
.inst.v = (unsigned char[]){
0xad,0xe3,0x6f,0x9a,0x01,0xc7,0x76,0x61,0x34,0x35,0xf5,0x4e,0x24,0x74,0x22,0x21,
0x9a,0x29,0x89,0xc7,0x93,0x2e,0x60,0x1e,0xe8,0x14,0x24,0x8d,0xd5,0x03,0xf1,0x65,
0x5d,0x08,0x22,0x72,0xd5,0xad,0x95,0xe1,0x23,0x1e,0x8a,0xa7,0x13,0xd9,0x2b,0x5e,
0xbc,0xbb,0x80,0xab,0x8d,0xe5,0x79,0xab,0x5b,0x47,0x4e,0xdd,0xee,0x6b,0x03,0x8f,
0x0f,0x5c,0x5e,0xa9,0x1a,0x83,0xdd,0xd3,0x88,0xb2,0x75,0x4b,0xce,0x83,0x36,0x57,
0x4b,0xf1,0x5c,0xca,0x7e,0x09,0xc0,0xd3,0x89,0xc6,0xe0,0xda,0xc4,0x81,0x7e,0x5b,
0xf9,0xe1,0x01,0xc1,0x92,0x05,0xea,0xf5,0x2f,0xc6,0xc6,0xc7,0x8f,0xbc,0xf4,
},
.inst.c = (unsigned char[]){
0xf4,0xa3,0xe5,0xa0,0x72,0x63,0x95,0xc6,0x4f,0x48,0xd0,0x8b,0x5b,0x5f,0x8e,0x6b,
0x96,0x1f,0x16,0xed,0xbc,0x66,0x94,0x45,0x31,0xd7,0x47,0x73,0x22,0xa5,0x86,0xce,
0xc0,0x4c,0xac,0x63,0xb8,0x39,0x50,0xbf,0xe6,0x59,0x6c,0x38,0x58,0x99,0x1f,0x27,
0xa7,0x9d,0x71,0x2a,0xb3,0x7b,0xf9,0xfb,0x17,0x86,0xaa,0x99,0x81,0xaa,0x43,0xe4,
0x37,0xd3,0x1e,0x6e,0xe5,0xe6,0xee,0xc2,0xed,0x95,0x4f,0x53,0x0e,0x46,0x8a,0xcc,
0x45,0xa5,0xdb,0x69,0x0d,0x81,0xc9,0x32,0x92,0xbc,0x8f,0x33,0xe6,0xf6,0x09,0x7c,
0x8e,0x05,0x19,0x0d,0xf1,0xb6,0xcc,0xf3,0x02,0x21,0x90,0x25,0xec,0xed,0x0e,
},
.inst.reseed_ctr = 1,

.gen1.v =(unsigned char[]){
0xa2,0x87,0x55,0x3a,0x74,0x2b,0x0c,0x27,0x83,0x7e,0xc5,0xd9,0x7f,0xd3,0xb0,0x8d,
0x30,0x48,0xa0,0xb5,0x4f,0x94,0xf4,0x64,0x19,0xeb,0x6c,0x00,0xf7,0xa9,0x78,0x34,
0x1d,0x54,0xce,0xd6,0x8d,0xe6,0xe6,0xa1,0x09,0x77,0xf6,0xdf,0x6c,0x72,0x4b,0x26,
0x78,0x01,0xa8,0x2e,0x3a,0x96,0x7a,0x5f,0xb0,0x63,0x6e,0xdf,0xc9,0x16,0xc0,0x93,
0x52,0x6f,0x16,0x68,0xea,0x76,0x95,0x50,0x56,0xc9,0x54,0x09,0x44,0xce,0x67,0x73,
0xd6,0x55,0x30,0xce,0xae,0xa7,0xcf,0xf2,0x70,0xb3,0x57,0x27,0x86,0xe9,0x60,0x09,
0x73,0x23,0x2e,0xa5,0xdc,0xd3,0x75,0x77,0xc0,0xd9,0x84,0x37,0x19,0x9c,0x69,
} ,
.gen1.c = (unsigned char[]){
0xf4,0xa3,0xe5,0xa0,0x72,0x63,0x95,0xc6,0x4f,0x48,0xd0,0x8b,0x5b,0x5f,0x8e,0x6b,
0x96,0x1f,0x16,0xed,0xbc,0x66,0x94,0x45,0x31,0xd7,0x47,0x73,0x22,0xa5,0x86,0xce,
0xc0,0x4c,0xac,0x63,0xb8,0x39,0x50,0xbf,0xe6,0x59,0x6c,0x38,0x58,0x99,0x1f,0x27,
0xa7,0x9d,0x71,0x2a,0xb3,0x7b,0xf9,0xfb,0x17,0x86,0xaa,0x99,0x81,0xaa,0x43,0xe4,
0x37,0xd3,0x1e,0x6e,0xe5,0xe6,0xee,0xc2,0xed,0x95,0x4f,0x53,0x0e,0x46,0x8a,0xcc,
0x45,0xa5,0xdb,0x69,0x0d,0x81,0xc9,0x32,0x92,0xbc,0x8f,0x33,0xe6,0xf6,0x09,0x7c,
0x8e,0x05,0x19,0x0d,0xf1,0xb6,0xcc,0xf3,0x02,0x21,0x90,0x25,0xec,0xed,0x0e,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char[]){
0x97,0x2b,0x3a,0xda,0xe6,0x8e,0xa1,0xed,0xd2,0xc7,0x96,0x64,0xdb,0x33,0x3e,0xf8,
0xc6,0x67,0xb7,0xa3,0x0b,0xfb,0x88,0xa9,0x4b,0xc2,0xb3,0x74,0x1a,0x4e,0xff,0x02,
0xdd,0xa1,0x7b,0x3a,0x46,0x20,0x37,0x60,0xef,0xd1,0x63,0x17,0xc5,0x0b,0x6a,0xb8,
0xbc,0x68,0x49,0x73,0xb3,0xa3,0x83,0x57,0x0e,0x00,0x53,0x78,0xdd,0x89,0xf1,0xf9,
0xa6,0xdc,0xcb,0x83,0xb6,0xe2,0x28,0x21,0x94,0x41,0x33,0x56,0xa7,0x9c,0x52,0x3d,
0x5e,0x38,0x1f,0x78,0xe1,0x22,0xa9,0xa2,0xd2,0xe6,0x00,0xdb,0xc1,0x61,0x85,0xf5,
0x1c,0xca,0x1c,0x70,0xf5,0x5d,0x27,0x6a,0x76,0x63,0x1b,0xac,0x4e,0xee,0x7a,
},
.gen2.c = (unsigned char[]){
0xf4,0xa3,0xe5,0xa0,0x72,0x63,0x95,0xc6,0x4f,0x48,0xd0,0x8b,0x5b,0x5f,0x8e,0x6b,
0x96,0x1f,0x16,0xed,0xbc,0x66,0x94,0x45,0x31,0xd7,0x47,0x73,0x22,0xa5,0x86,0xce,
0xc0,0x4c,0xac,0x63,0xb8,0x39,0x50,0xbf,0xe6,0x59,0x6c,0x38,0x58,0x99,0x1f,0x27,
0xa7,0x9d,0x71,0x2a,0xb3,0x7b,0xf9,0xfb,0x17,0x86,0xaa,0x99,0x81,0xaa,0x43,0xe4,
0x37,0xd3,0x1e,0x6e,0xe5,0xe6,0xee,0xc2,0xed,0x95,0x4f,0x53,0x0e,0x46,0x8a,0xcc,
0x45,0xa5,0xdb,0x69,0x0d,0x81,0xc9,0x32,0x92,0xbc,0x8f,0x33,0xe6,0xf6,0x09,0x7c,
0x8e,0x05,0x19,0x0d,0xf1,0xb6,0xcc,0xf3,0x02,0x21,0x90,0x25,0xec,0xed,0x0e,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x95,0xb7,0xf1,0x7e,0x98,0x02,0xd3,0x57,0x73,0x92,0xc6,0xa9,0xc0,0x80,0x83,0xb6,
0x7d,0xd1,0x29,0x22,0x65,0xb5,0xf4,0x2d,0x23,0x7f,0x1c,0x55,0xbb,0x9b,0x10,0xbf,
0xcf,0xd8,0x2c,0x77,0xa3,0x78,0xb8,0x26,0x6a,0x00,0x99,0x14,0x3b,0x3c,0x2d,0x64,
0x61,0x1e,0xee,0xb6,0x9a,0xcd,0xc0,0x55,0x95,0x7c,0x13,0x9e,0x8b,0x19,0x0c,0x7a,
0x06,0x95,0x5f,0x2c,0x79,0x7c,0x27,0x78,0xde,0x94,0x03,0x96,0xa5,0x01,0xf4,0x0e,
0x91,0x39,0x6a,0xcf,0x8d,0x7e,0x45,0xeb,0xdb,0xb5,0x3b,0xbf,0x8c,0x97,0x52,0x30,
0xd2,0xf0,0xff,0x91,0x06,0xc7,0x61,0x19,0xae,0x49,0x8e,0x7f,0xbc,0x03,0xd9,0x0f,
0x8e,0x4c,0x51,0x62,0x7a,0xed,0x5c,0x8d,0x42,0x63,0xd5,0xd2,0xb9,0x78,0x87,0x3a,
0x0d,0xe5,0x96,0xee,0x6d,0xc7,0xf7,0xc2,0x9e,0x37,0xee,0xe8,0xb3,0x4c,0x90,0xdd,
0x1c,0xf6,0xa9,0xdd,0xb2,0x2b,0x4c,0xbd,0x08,0x6b,0x14,0xb3,0x5d,0xe9,0x3d,0xa2,
0xd5,0xcb,0x18,0x06,0x69,0x8c,0xbd,0x7b,0xbb,0x67,0xbf,0xe3,0xd3,0x1f,0xd2,0xd1,
0xdb,0xd2,0xa1,0xe0,0x58,0xa3,0xeb,0x99,0xd7,0xe5,0x1f,0x1a,0x93,0x8e,0xed,0x5e,
0x1c,0x1d,0xe2,0x3a,0x6b,0x43,0x45,0xd3,0x19,0x14,0x09,0xf9,0x2f,0x39,0xb3,0x67,
0x0d,0x8d,0xbf,0xb6,0x35,0xd8,0xe6,0xa3,0x69,0x32,0xd8,0x10,0x33,0xd1,0x44,0x8d,
0x63,0xb4,0x03,0xdd,0xf8,0x8e,0x12,0x1b,0x6e,0x81,0x9a,0xc3,0x81,0x22,0x6c,0x13,
0x21,0xe4,0xb0,0x86,0x44,0xf6,0x72,0x7c,0x36,0x8c,0x5a,0x9f,0x7a,0x4b,0x3e,0xe2,
},
},
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x9c,0x96,0xa3,0x4f,0x68,0x68,0x9b,0x8a,0xa8,0xd9,0xc1,0xf6,0xcd,0x0f,0xa7,0xc6,
0xf9,0x60,0x71,0xca,0xf1,0xbf,0x55,0x56,0xf4,0x5b,0xdb,0xf4,0x8c,0x6c,0xf0,0xc6,
},
.inst.nonce = (unsigned char[]){
0x88,0x5c,0x25,0x39,0x04,0x6a,0xfb,0x14,0x01,0xeb,0x7a,0x5c,0x84,0xdb,0xd9,0xc2,
},
.inst.v = (unsigned char[]){
0x11,0xb6,0xf9,0xc7,0x4e,0x77,0x32,0xa3,0x93,0xe6,0xd3,0x4f,0x4d,0x8d,0x5b,0xb9,
0x3d,0xf8,0xd0,0x7a,0x52,0xfe,0xb5,0x30,0xe0,0x9e,0x44,0xaa,0xdc,0x90,0x19,0x16,
0x5e,0x9c,0xf9,0x03,0x87,0xf8,0x0f,0x1a,0x41,0x62,0x16,0x90,0xff,0x29,0x08,0xb7,
0x28,0x21,0x37,0x09,0x27,0xcb,0xc1,0xd3,0xd6,0x6d,0x47,0xa9,0xc0,0x4c,0x8e,0xd8,
0x49,0x26,0x39,0xe3,0x8b,0xe7,0x6d,0x30,0x05,0xc1,0xc4,0xa8,0x83,0x73,0x0b,0x55,
0xa4,0x91,0x4a,0x87,0xc9,0x73,0x31,0xbd,0x06,0xba,0xf0,0x79,0x40,0x95,0x7f,0x62,
0x82,0xe3,0xcd,0x28,0xa3,0x7d,0xa7,0x55,0xde,0x25,0x0f,0xf1,0x08,0x38,0x85,},
.inst.c = (unsigned char[]){
0xb6,0xba,0xbd,0x1c,0xa5,0x4b,0x3c,0x3c,0x7c,0x5b,0xec,0xf6,0xc1,0xa0,0x52,0xa5,
0x37,0x6e,0x2a,0x63,0x7a,0xe9,0x25,0x1a,0x8a,0xee,0x90,0xa4,0x6e,0x9b,0x0a,0xbf,
0xe9,0xe1,0xda,0x50,0xd6,0x97,0x17,0x10,0x04,0x3a,0x3f,0xff,0x16,0xef,0x17,0x05,
0xe4,0x98,0x27,0xd7,0xe9,0x5c,0x4e,0xed,0x90,0x9e,0x34,0xee,0x00,0x39,0x9a,0x96,
0x22,0x10,0xcb,0x36,0x91,0x33,0xed,0x04,0xcf,0x50,0x33,0xc2,0xa8,0xda,0x69,0x2d,
0x41,0x8c,0x70,0xec,0x14,0x10,0x62,0x37,0x9c,0xd0,0x84,0x6f,0x1a,0x32,0x72,0x5e,
0x46,0x4d,0x7b,0x39,0xa1,0x69,0x9e,0x49,0xdc,0xc4,0x02,0x6a,0xe2,0xd8,0x7e,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char[]){
0xcb,0x61,0xc4,0xf7,0x5c,0x01,0xb5,0x78,0xaa,0x23,0x3a,0x0b,0xae,0x48,0x81,0xc0,
0xa1,0x15,0x27,0xc2,0x2f,0xe7,0xb3,0x4f,0xb6,0xae,0x62,0xee,0xbc,0xfe,0x60,0x85,
},
.gen1.v = (unsigned char[]){
0xc8,0x71,0xb6,0xe3,0xf3,0xc2,0x6e,0xe0,0x10,0x42,0xc0,0x46,0x0f,0x2d,0xae,0x5e,
0x75,0x66,0xfa,0xdd,0xcd,0xe7,0xda,0x4b,0x6b,0x8c,0xd5,0x4f,0x4b,0x2b,0x23,0xd6,
0x48,0x7e,0xd3,0x54,0x5e,0x8f,0x26,0x2a,0x45,0x9c,0x56,0x90,0x16,0x18,0x20,0xc4,
0xd7,0x95,0xe0,0xb1,0xd2,0xc5,0x25,0x5c,0x61,0xe5,0xa0,0x1b,0xa2,0xee,0x46,0x8e,
0xfe,0x9e,0xdf,0x2a,0x9d,0x68,0x0f,0x37,0x5d,0x37,0xd7,0x2f,0x43,0xf2,0x08,0x3e,
0x92,0x6d,0x6d,0x78,0xb2,0x21,0x95,0xdd,0xb6,0x4f,0x40,0xbc,0x0c,0x56,0xc4,0x5a,
0x0d,0x48,0x14,0x19,0xbf,0x2e,0x29,0xca,0xec,0xfc,0xbd,0xca,0x32,0xbc,0x2e,
},
.gen1.c = (unsigned char[]){
0xb6,0xba,0xbd,0x1c,0xa5,0x4b,0x3c,0x3c,0x7c,0x5b,0xec,0xf6,0xc1,0xa0,0x52,0xa5,
0x37,0x6e,0x2a,0x63,0x7a,0xe9,0x25,0x1a,0x8a,0xee,0x90,0xa4,0x6e,0x9b,0x0a,0xbf,
0xe9,0xe1,0xda,0x50,0xd6,0x97,0x17,0x10,0x04,0x3a,0x3f,0xff,0x16,0xef,0x17,0x05,
0xe4,0x98,0x27,0xd7,0xe9,0x5c,0x4e,0xed,0x90,0x9e,0x34,0xee,0x00,0x39,0x9a,0x96,
0x22,0x10,0xcb,0x36,0x91,0x33,0xed,0x04,0xcf,0x50,0x33,0xc2,0xa8,0xda,0x69,0x2d,
0x41,0x8c,0x70,0xec,0x14,0x10,0x62,0x37,0x9c,0xd0,0x84,0x6f,0x1a,0x32,0x72,0x5e,
0x46,0x4d,0x7b,0x39,0xa1,0x69,0x9e,0x49,0xdc,0xc4,0x02,0x6a,0xe2,0xd8,0x7e,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char[]){
0xc0,0x66,0xfd,0x2e,0xb8,0xe4,0xae,0xa2,0xe7,0x14,0x5e,0xda,0x0c,0xfc,0x8b,0xef,
0x5e,0xed,0xcc,0x36,0x7b,0x1c,0xb4,0xde,0x7e,0xb2,0xc2,0x75,0x9f,0xa7,0x5b,0xf7,
},
.gen2.v = (unsigned char[]){
0x7f,0x2c,0x74,0x00,0x99,0x0d,0xab,0x1c,0x8c,0x9e,0xad,0x3c,0xd0,0xce,0x01,0x03,
0xac,0xd5,0x25,0x41,0x48,0xd0,0xff,0x65,0xf6,0x7b,0x65,0xf3,0xb9,0xc6,0x2e,0x96,
0x32,0x60,0xad,0xa5,0x35,0x26,0x3d,0x3a,0x49,0xd6,0x96,0x8f,0x2d,0x07,0x38,0xf1,
0xff,0x65,0xdf,0x49,0x81,0xe3,0xf6,0x71,0x0c,0x7e,0x69,0xec,0xc0,0x61,0x16,0xa8,
0x27,0xa5,0x38,0x37,0x38,0x52,0x56,0xb8,0x2b,0x25,0x7b,0x70,0x33,0x15,0x79,0xb2,
0x2a,0xbd,0x05,0x55,0x3e,0x2d,0x36,0x14,0x66,0xe4,0x83,0x97,0x2a,0xbe,0x31,0x10,
0xcd,0x4d,0x09,0x35,0x99,0x9a,0x21,0xd1,0xce,0x3d,0x98,0x0d,0x4a,0x31,0xec,
},
.gen2.c = (unsigned char[]){
0xb6,0xba,0xbd,0x1c,0xa5,0x4b,0x3c,0x3c,0x7c,0x5b,0xec,0xf6,0xc1,0xa0,0x52,0xa5,
0x37,0x6e,0x2a,0x63,0x7a,0xe9,0x25,0x1a,0x8a,0xee,0x90,0xa4,0x6e,0x9b,0x0a,0xbf,
0xe9,0xe1,0xda,0x50,0xd6,0x97,0x17,0x10,0x04,0x3a,0x3f,0xff,0x16,0xef,0x17,0x05,
0xe4,0x98,0x27,0xd7,0xe9,0x5c,0x4e,0xed,0x90,0x9e,0x34,0xee,0x00,0x39,0x9a,0x96,
0x22,0x10,0xcb,0x36,0x91,0x33,0xed,0x04,0xcf,0x50,0x33,0xc2,0xa8,0xda,0x69,0x2d,
0x41,0x8c,0x70,0xec,0x14,0x10,0x62,0x37,0x9c,0xd0,0x84,0x6f,0x1a,0x32,0x72,0x5e,
0x46,0x4d,0x7b,0x39,0xa1,0x69,0x9e,0x49,0xdc,0xc4,0x02,0x6a,0xe2,0xd8,0x7e,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x78,0x2c,0x20,0x8e,0xd5,0x80,0x44,0xe7,0x8b,0x5b,0xbb,0xd8,0x77,0x2a,0x3c,0xaf,
0x25,0xb4,0x7d,0x36,0xaf,0xeb,0x0d,0x34,0x93,0xc4,0x3e,0x01,0xcc,0x66,0xa0,0xca,
0x2f,0xac,0xed,0x2a,0xb1,0x86,0xbc,0x46,0x82,0x5d,0x98,0x9c,0xf8,0xee,0x7c,0x95,
0xf8,0xc0,0xb0,0xd2,0xb7,0x6e,0x6c,0x85,0x90,0xe7,0x28,0x34,0xd4,0xc5,0x24,0x45,
0xae,0xce,0xeb,0x7b,0xf5,0xf5,0xd9,0xac,0x44,0xa1,0x2c,0xbd,0x3f,0xa7,0xf4,0x46,
0x2f,0x85,0x64,0x52,0xdc,0x4a,0x92,0x91,0x82,0xd2,0x38,0x8a,0xa7,0x63,0x5b,0x96,
0x98,0xa9,0x12,0x58,0x5d,0xf7,0xf5,0x60,0xad,0xc5,0x08,0x0d,0x53,0xb8,0x2b,0xbd,
0x7e,0x9e,0x48,0x0b,0x00,0xd1,0xda,0x5b,0xb2,0xd4,0x80,0xca,0xe2,0xba,0x8c,0x67,
0xd4,0xbf,0x3b,0xfd,0x14,0x6a,0x91,0xd6,0xaa,0xb3,0x9f,0xaa,0xe1,0x60,0x0a,0xf2,
0xce,0x32,0x04,0xca,0xbf,0x4c,0x1c,0xae,0xe4,0xcf,0xd5,0xe6,0xf8,0xdb,0x19,0x02,
0x03,0x3f,0x7f,0x8d,0x33,0xbc,0x6e,0x0e,0x5d,0x32,0xa3,0x20,0xba,0x73,0x5d,0x09,
0x1f,0x30,0x86,0x7b,0x7c,0xb7,0x88,0x0c,0x2e,0x3c,0xe6,0xaa,0xda,0x79,0x66,0x41,
0x91,0xdf,0x36,0x0d,0x35,0xfe,0x9a,0xe7,0xba,0xbc,0xa4,0x14,0x85,0xb0,0x6a,0xb4,
0x9d,0xff,0x52,0x87,0x82,0xfb,0xe6,0xf2,0xb0,0xe7,0x49,0x96,0xe9,0xce,0x92,0x72,
0xd1,0xef,0x39,0x2b,0xe5,0xc1,0x7c,0xc6,0x2c,0x74,0xbe,0x50,0x4e,0x6a,0x87,0x31,
0xdd,0x95,0x48,0xb0,0xdb,0x27,0xe0,0xb7,0xdb,0x48,0x86,0xf5,0x37,0x88,0x36,0x23,
},
},
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0xb5,0x10,0xa6,0x0a,0x5a,0x19,0x00,0x7b,0xa8,0x08,0xe6,0x0b,0x16,0x28,0xed,0x99,
0x07,0xd1,0xbe,0xf9,0x92,0xd4,0xfc,0x0a,0xbe,0x51,0x50,0x01,0x31,0xcb,0x4c,0xf8,
},
.inst.nonce = (unsigned char[]){
0xeb,0x1b,0xd7,0x83,0xac,0xcc,0x85,0x5e,0xb3,0x20,0xcd,0xe1,0x0b,0x1b,0xe6,0xd0,
},
.inst.pers = (unsigned char[]){
0x7e,0xa1,0x0e,0x96,0xaf,0x90,0x0c,0x25,0xd3,0xbe,0x3b,0x50,0xa0,0xcc,0x71,0xa7,
0x9f,0xe4,0x14,0xbd,0x4c,0x37,0x39,0x80,0x3f,0x02,0xff,0xe5,0xb2,0x60,0xbf,0xbb,
},
.inst.v = (unsigned char[]){
0x6e,0x10,0x09,0xeb,0x74,0x25,0x4e,0x17,0xbc,0xa3,0x2a,0xb7,0xbc,0x01,0x91,0xba,
0x70,0x89,0xb7,0x8c,0x4b,0xa4,0xbb,0xab,0xab,0x03,0xfc,0x6c,0x1e,0xa5,0xb7,0xdb,
0xef,0x37,0xec,0xcd,0x5f,0x70,0x7f,0x99,0x11,0xab,0x57,0xe4,0xd1,0x8d,0x34,0x84,
0x53,0x42,0x42,0x90,0x07,0x33,0x61,0x5d,0x66,0xf2,0x2b,0x76,0x90,0x1e,0x08,0xcc,
0x0d,0x86,0xa3,0x89,0xc8,0x6f,0x27,0x12,0xab,0x34,0x05,0x23,0x66,0x36,0x14,0x1d,
0xdd,0xc8,0xae,0x83,0xaa,0x33,0x77,0x21,0xf5,0x7e,0x23,0x08,0xf8,0x15,0x0a,0x35,
0x0f,0x07,0xef,0x94,0x68,0x26,0xef,0x80,0xba,0x20,0xfe,0x20,0x23,0xe4,0x39,
},
.inst.c = (unsigned char[]){
0x61,0x67,0x19,0xd5,0xd6,0xa4,0xdf,0xd8,0x39,0x9e,0x41,0xf3,0x67,0x7e,0x12,0xcc,
0xbd,0x2e,0x3d,0x1e,0x1d,0xc5,0x7b,0xa0,0xb7,0x1d,0x1c,0x61,0x5a,0x50,0x14,0xd7,
0x33,0x63,0x7f,0x2c,0x6f,0x0a,0x5e,0x97,0xe7,0x78,0x31,0x59,0xbe,0x99,0xf7,0x47,
0x7a,0x31,0x7a,0x5c,0x72,0x56,0x52,0xcc,0xd6,0x31,0x95,0xd9,0x73,0x6d,0xf9,0xa9,
0x7e,0x6f,0x59,0x4e,0x57,0x0b,0xad,0x9f,0x99,0x7a,0xa3,0x8b,0xdf,0x51,0x44,0xda,
0x63,0x61,0xfe,0x9b,0xe2,0x82,0x88,0x20,0x5a,0xbc,0x9d,0x8b,0x41,0x1f,0x5f,0x38,
0x0a,0x56,0x15,0xce,0xfe,0x6d,0x3e,0x82,0x88,0xda,0x5f,0x43,0x01,0xfb,0x08,
},
.inst.reseed_ctr = 1,

.gen1.v = (unsigned char[]){
0xcf,0x77,0x23,0xc1,0x4a,0xca,0x2d,0xef,0xf6,0x41,0x6c,0xab,0x23,0x7f,0xa4,0x87,
0x2d,0xb7,0xf4,0xaa,0x69,0x6a,0x37,0x4c,0x62,0x21,0x18,0xcd,0x78,0xf5,0xcc,0xb3,
0x22,0x9b,0x6b,0xf9,0xce,0x7a,0xde,0x30,0xf9,0x23,0x89,0x3e,0x90,0x27,0x2c,0xa2,
0x36,0x72,0xbc,0x98,0xda,0xea,0x39,0xca,0x18,0x52,0x1a,0xcc,0xc2,0x56,0x41,0xff,
0x36,0x00,0xf4,0x40,0x89,0x45,0x24,0x63,0x29,0x31,0x87,0xf2,0x41,0x98,0xca,0x38,
0x45,0x39,0xdd,0xaa,0xbb,0x5c,0x4b,0x72,0xb5,0x33,0xf1,0x51,0xec,0xcf,0x82,0x53,
0x6d,0xf2,0xcf,0x82,0x9a,0x69,0x41,0xf7,0x0b,0x95,0x49,0xe4,0xc2,0xd9,0x6e,
},
.gen1.c = (unsigned char[]){
0x61,0x67,0x19,0xd5,0xd6,0xa4,0xdf,0xd8,0x39,0x9e,0x41,0xf3,0x67,0x7e,0x12,0xcc,
0xbd,0x2e,0x3d,0x1e,0x1d,0xc5,0x7b,0xa0,0xb7,0x1d,0x1c,0x61,0x5a,0x50,0x14,0xd7,
0x33,0x63,0x7f,0x2c,0x6f,0x0a,0x5e,0x97,0xe7,0x78,0x31,0x59,0xbe,0x99,0xf7,0x47,
0x7a,0x31,0x7a,0x5c,0x72,0x56,0x52,0xcc,0xd6,0x31,0x95,0xd9,0x73,0x6d,0xf9,0xa9,
0x7e,0x6f,0x59,0x4e,0x57,0x0b,0xad,0x9f,0x99,0x7a,0xa3,0x8b,0xdf,0x51,0x44,0xda,
0x63,0x61,0xfe,0x9b,0xe2,0x82,0x88,0x20,0x5a,0xbc,0x9d,0x8b,0x41,0x1f,0x5f,0x38,
0x0a,0x56,0x15,0xce,0xfe,0x6d,0x3e,0x82,0x88,0xda,0x5f,0x43,0x01,0xfb,0x08,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char[]){
0x30,0xde,0x3d,0x97,0x21,0x6f,0x0d,0xc8,0x2f,0xdf,0xae,0x9e,0x8a,0xfd,0xb7,0x53,
0xea,0xe6,0x31,0xc8,0x87,0x2f,0xb2,0xed,0x19,0x3e,0x35,0x2e,0xd3,0x45,0xe1,0x8a,
0x55,0xfe,0xeb,0x26,0x3d,0x85,0x3c,0xc8,0xe0,0x9b,0xba,0x98,0x4e,0xc1,0x24,0x3f,
0x77,0x1a,0x2c,0x1a,0x65,0x07,0x99,0x3f,0x08,0xe9,0x48,0x16,0x77,0x03,0x0d,0xa7,
0x8b,0xc5,0x13,0xcf,0x4d,0xf3,0x3e,0xf5,0xe1,0x61,0x09,0x2e,0xee,0x86,0xfb,0x21,
0x40,0x70,0x1b,0x20,0xdf,0x82,0xde,0x4c,0x3b,0xf0,0xe7,0x67,0x36,0xf6,0x24,0x82,
0xc7,0x94,0xff,0xd2,0x99,0x73,0x5f,0x49,0x13,0x23,0x77,0x8e,0xa4,0x72,0x8f,
},
.gen2.c = (unsigned char[]){
0x61,0x67,0x19,0xd5,0xd6,0xa4,0xdf,0xd8,0x39,0x9e,0x41,0xf3,0x67,0x7e,0x12,0xcc,
0xbd,0x2e,0x3d,0x1e,0x1d,0xc5,0x7b,0xa0,0xb7,0x1d,0x1c,0x61,0x5a,0x50,0x14,0xd7,
0x33,0x63,0x7f,0x2c,0x6f,0x0a,0x5e,0x97,0xe7,0x78,0x31,0x59,0xbe,0x99,0xf7,0x47,
0x7a,0x31,0x7a,0x5c,0x72,0x56,0x52,0xcc,0xd6,0x31,0x95,0xd9,0x73,0x6d,0xf9,0xa9,
0x7e,0x6f,0x59,0x4e,0x57,0x0b,0xad,0x9f,0x99,0x7a,0xa3,0x8b,0xdf,0x51,0x44,0xda,
0x63,0x61,0xfe,0x9b,0xe2,0x82,0x88,0x20,0x5a,0xbc,0x9d,0x8b,0x41,0x1f,0x5f,0x38,
0x0a,0x56,0x15,0xce,0xfe,0x6d,0x3e,0x82,0x88,0xda,0x5f,0x43,0x01,0xfb,0x08,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x5f,0x03,0xab,0x52,0x9b,0x9e,0xda,0xb5,0x5e,0x75,0xe4,0x89,0xbc,0xeb,0x0a,0xb6,
0xcb,0x9e,0xc6,0x25,0x5c,0x61,0x96,0xcd,0x07,0x53,0x97,0xb0,0x91,0xc3,0xeb,0x59,
0x8e,0x31,0x62,0x32,0x0c,0x71,0xe3,0x89,0xe1,0xd9,0xd6,0x2a,0x2d,0xe5,0x99,0x0c,
0x20,0x97,0x56,0x10,0x84,0xe1,0x3a,0xb4,0xb2,0x97,0x7f,0x99,0xc2,0x3d,0xaf,0x21,
0x4b,0xbf,0xeb,0x16,0x21,0x67,0xc9,0x33,0x86,0x0f,0xb2,0xcc,0x9f,0x80,0x07,0x9e,
0x17,0x89,0x39,0x62,0xff,0x03,0xa2,0x6b,0x2b,0x4b,0x83,0x8e,0x6d,0xe9,0xff,0x5e,
0x9d,0xcf,0xba,0x73,0xac,0xc6,0xee,0x76,0xc4,0x77,0xd1,0xf7,0x00,0x22,0x65,0xe6,
0x93,0x85,0x0a,0x26,0x0e,0x68,0x61,0x7f,0x13,0xb2,0x47,0x16,0xd8,0xdb,0x42,0xc9,
0xa8,0xf2,0xe0,0xbd,0x50,0x02,0x28,0x9a,0x86,0xb0,0x2a,0x84,0x5d,0x98,0x86,0xd3,
0xfd,0x05,0xd8,0xd8,0x5a,0xc7,0x7e,0xa6,0x01,0x92,0xa2,0x90,0x47,0x30,0x3a,0x03,
0xb5,0x1e,0x0e,0x70,0x8f,0xa0,0x02,0x05,0xb5,0xc0,0x3b,0x67,0xb7,0x78,0x74,0x05,
0x2c,0xe5,0x6d,0x5c,0xf3,0x8e,0x72,0x6d,0x60,0x8b,0xc0,0xef,0x8f,0xc2,0xa7,0xd2,
0x77,0x78,0xf4,0x7c,0x68,0x91,0x63,0x7b,0x63,0x1c,0xcc,0x6e,0x36,0xeb,0x89,0x3f,
0xe0,0x8a,0x5b,0x9f,0x55,0x71,0xbd,0x43,0x2c,0x82,0x6d,0x99,0x69,0x8f,0xd4,0x17,
0x9b,0xc5,0xcf,0xb0,0xd8,0xea,0x2d,0x55,0x5c,0x36,0xb7,0x05,0x68,0xc2,0x20,0x6b,
0x18,0xf3,0xbe,0xa1,0x4b,0x39,0x8d,0x2b,0x6a,0x00,0x12,0x2d,0x7a,0x44,0x7e,0xa3,
},
},
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0xb1,0x48,0xb3,0x17,0xa2,0x68,0x62,0x8f,0x04,0xc9,0xa8,0x7a,0x0a,0xc5,0xf9,0x14,
0x98,0xb8,0x02,0x0f,0x4e,0x48,0x35,0x72,0x0c,0x1a,0x3c,0xc0,0x7b,0x05,0x0f,0xa2,
},
.inst.nonce = (unsigned char[]){
0x2f,0x35,0xe5,0xb6,0x22,0xae,0xef,0xe7,0x56,0x05,0xc2,0x27,0x4e,0xc8,0xc6,0x96,
},
.inst.pers = (unsigned char[]){
0xfc,0x52,0x7a,0x2f,0x16,0xb5,0x3c,0x51,0x3f,0x94,0x85,0x5b,0x35,0xce,0xa6,0x09,
0x0c,0x30,0x3d,0xcc,0x64,0x2e,0x98,0xed,0x5f,0x32,0x3a,0xba,0x0f,0x35,0xfa,0x27,
},
.inst.v = (unsigned char[]){
0xc0,0xcb,0x7d,0x99,0x92,0xeb,0x7d,0x59,0x7e,0xbe,0x77,0x6d,0x6a,0x88,0x68,0x81,
0xc1,0x79,0x15,0xeb,0x90,0xb7,0x67,0x2d,0xee,0x5c,0x18,0xcb,0x10,0xd4,0xb9,0xfb,
0x7b,0x66,0x1d,0x0b,0xa6,0x8f,0x1b,0xef,0x72,0xbe,0xeb,0xc0,0x40,0x98,0x2c,0xe7,
0xcb,0xb2,0xd0,0xb9,0x0b,0xfd,0x3a,0xce,0x08,0xea,0x37,0xa3,0x3e,0x71,0x41,0x27,
0x18,0xb1,0xc0,0xe6,0x85,0x63,0xad,0x5b,0x35,0x3a,0x20,0xa5,0x84,0xad,0xf0,0x4c,
0xfa,0x4a,0x19,0x91,0x7c,0xf3,0x42,0xa1,0x3f,0x9f,0x51,0xbb,0x12,0xf3,0xe2,0x4b,
0x3b,0x1d,0xf0,0xf7,0x03,0xcb,0xa2,0x09,0x9f,0x63,0x2c,0x63,0x99,0xe7,0x4c,
},
.inst.c = (unsigned char[]){
0x02,0xd6,0x77,0x70,0xe8,0x13,0x7f,0x9b,0x39,0x96,0x09,0x63,0xae,0xa6,0x82,0x73,
0xe2,0x6c,0x75,0x94,0x5b,0xe1,0x13,0x6a,0xc1,0x4a,0xdd,0xdc,0xa6,0x33,0x90,0x43,
0xdc,0xab,0x44,0xe7,0xd1,0x2b,0xf1,0x1d,0xc4,0x30,0xa9,0x67,0x26,0xa7,0xa0,0x29,
0xeb,0x0e,0xc0,0x39,0xe0,0x37,0x6a,0x12,0x45,0x48,0x1a,0xeb,0x7e,0x06,0x8a,0x12,
0xb7,0x5f,0x55,0xd5,0xad,0x5f,0xe6,0xe5,0xb2,0x37,0xad,0x50,0xbc,0x9e,0x2d,0x91,
0xb7,0xce,0xaa,0x3a,0x23,0xfb,0xbb,0x04,0x49,0xe1,0x6f,0xdb,0xab,0x57,0xde,0x0a,
0xef,0xd8,0xd1,0x7f,0x36,0x28,0x1c,0x74,0x25,0x90,0xda,0xf8,0x3c,0x14,0x3d,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char[]){
0x2c,0xc9,0xf1,0x37,0xfc,0xd8,0xc2,0xd5,0x26,0xd7,0x00,0x93,0xfe,0x11,0xf9,0x0a,
0x0a,0x36,0xbc,0x97,0x64,0xa4,0xc5,0x60,0x90,0x72,0xe1,0x81,0xa2,0x49,0x45,0x16,
},
.gen1.v =(unsigned char[]){
0xc3,0xa1,0xf5,0x0a,0x7a,0xfe,0xfc,0xf4,0xb8,0x54,0x80,0xd1,0x19,0x2e,0xea,0xf5,
0xa3,0xe5,0x8b,0x7f,0xec,0x98,0x7a,0x98,0xaf,0xa6,0xf6,0xa7,0xb7,0x08,0x4a,0x3f,
0x58,0x11,0x61,0xf3,0x77,0xbb,0x0d,0x0d,0x36,0xef,0x95,0x27,0x67,0x3f,0xcd,0xf7,
0xc0,0xa6,0x8e,0x23,0x93,0x29,0x5e,0xdc,0x31,0x2d,0x32,0xc1,0x89,0xf7,0x7c,0xbb,
0x60,0xbe,0xdc,0x1e,0x7c,0x7a,0x16,0x77,0x81,0x5f,0x06,0x6f,0x70,0x37,0x43,0xfb,
0xcc,0xbe,0x3b,0x59,0x24,0x35,0x4e,0x7a,0xd0,0x88,0xd1,0xde,0xd4,0x1a,0x70,0x39,
0xbb,0x52,0x6a,0x91,0xef,0x70,0xe1,0xb1,0xdf,0xc9,0x39,0xc3,0xcc,0x04,0x58,
} ,
.gen1.c = (unsigned char[]){
0x02,0xd6,0x77,0x70,0xe8,0x13,0x7f,0x9b,0x39,0x96,0x09,0x63,0xae,0xa6,0x82,0x73,
0xe2,0x6c,0x75,0x94,0x5b,0xe1,0x13,0x6a,0xc1,0x4a,0xdd,0xdc,0xa6,0x33,0x90,0x43,
0xdc,0xab,0x44,0xe7,0xd1,0x2b,0xf1,0x1d,0xc4,0x30,0xa9,0x67,0x26,0xa7,0xa0,0x29,
0xeb,0x0e,0xc0,0x39,0xe0,0x37,0x6a,0x12,0x45,0x48,0x1a,0xeb,0x7e,0x06,0x8a,0x12,
0xb7,0x5f,0x55,0xd5,0xad,0x5f,0xe6,0xe5,0xb2,0x37,0xad,0x50,0xbc,0x9e,0x2d,0x91,
0xb7,0xce,0xaa,0x3a,0x23,0xfb,0xbb,0x04,0x49,0xe1,0x6f,0xdb,0xab,0x57,0xde,0x0a,
0xef,0xd8,0xd1,0x7f,0x36,0x28,0x1c,0x74,0x25,0x90,0xda,0xf8,0x3c,0x14,0x3d,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char[]){
0xe4,0x03,0x61,0x24,0x5b,0x91,0x88,0x0e,0x30,0x8f,0xb7,0x77,0xc2,0x8b,0xbf,0xae,
0xa5,0x98,0x2e,0x45,0xfe,0xcb,0x77,0x57,0xbb,0x1c,0x9d,0xe2,0xdf,0x9d,0xc6,0x12,
},
.gen2.v = (unsigned char[]){
0xc6,0x78,0x6c,0x7b,0x63,0x12,0x7c,0x8f,0xf1,0xea,0x8a,0x34,0xc7,0xd5,0x6d,0x69,
0x86,0x52,0x01,0x14,0x48,0x79,0x8e,0x03,0x70,0xf1,0xd4,0x84,0x5d,0x3b,0xda,0x83,
0x34,0xbc,0xa6,0xdb,0x48,0xe6,0xfe,0x2a,0xfb,0x20,0x3e,0x8e,0x8d,0xe7,0x6e,0xe5,
0x30,0x94,0x36,0x9e,0x68,0xb3,0x2d,0xb7,0xe1,0xf7,0x18,0x53,0xc1,0xa3,0x0b,0x9c,
0xb5,0x5b,0x06,0x7b,0x9a,0x19,0x87,0xe9,0x18,0x80,0xf2,0xc8,0xdc,0x5d,0x77,0xec,
0x26,0x7c,0x12,0xcb,0x99,0xeb,0xf4,0xc3,0xf0,0xe1,0x3d,0xa4,0x74,0x45,0xe0,0xec,
0x23,0x49,0xa7,0x1d,0x34,0x16,0x8e,0x33,0xfe,0xbb,0x65,0x7f,0x8d,0x12,0xc7,
},
.gen2.c = (unsigned char[]){
0x02,0xd6,0x77,0x70,0xe8,0x13,0x7f,0x9b,0x39,0x96,0x09,0x63,0xae,0xa6,0x82,0x73,
0xe2,0x6c,0x75,0x94,0x5b,0xe1,0x13,0x6a,0xc1,0x4a,0xdd,0xdc,0xa6,0x33,0x90,0x43,
0xdc,0xab,0x44,0xe7,0xd1,0x2b,0xf1,0x1d,0xc4,0x30,0xa9,0x67,0x26,0xa7,0xa0,0x29,
0xeb,0x0e,0xc0,0x39,0xe0,0x37,0x6a,0x12,0x45,0x48,0x1a,0xeb,0x7e,0x06,0x8a,0x12,
0xb7,0x5f,0x55,0xd5,0xad,0x5f,0xe6,0xe5,0xb2,0x37,0xad,0x50,0xbc,0x9e,0x2d,0x91,
0xb7,0xce,0xaa,0x3a,0x23,0xfb,0xbb,0x04,0x49,0xe1,0x6f,0xdb,0xab,0x57,0xde,0x0a,
0xef,0xd8,0xd1,0x7f,0x36,0x28,0x1c,0x74,0x25,0x90,0xda,0xf8,0x3c,0x14,0x3d,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x66,0xad,0x04,0x8b,0x4d,0x2d,0x00,0x32,0x23,0xc6,0x4d,0xd9,0x82,0x7c,0xc2,0x2e,
0xd3,0xec,0x8f,0xcb,0x61,0x20,0x9d,0x19,0x96,0x19,0x17,0x75,0x92,0xe9,0xb8,0x92,
0x26,0xbe,0x30,0xb1,0x93,0x0b,0xdd,0x74,0x9f,0x30,0xed,0x09,0xda,0x52,0xab,0xaa,
0x2e,0x59,0x9a,0xfa,0xf9,0x19,0x03,0xe7,0xa2,0xb5,0x9f,0xfb,0x8f,0xd4,0x70,0xe6,
0x60,0x44,0x85,0xa2,0x7c,0x20,0x0d,0x37,0x5f,0xef,0xf6,0x21,0x11,0x85,0x95,0xa7,
0xa3,0x05,0x7b,0x7e,0x31,0xea,0xdc,0x06,0x87,0xb1,0x00,0x8c,0x3c,0xb2,0xc7,0x43,
0x5a,0x57,0x04,0xb1,0xa1,0xa6,0xa3,0x48,0x7d,0x60,0xfd,0x14,0x79,0x3c,0x31,0x48,
0x6a,0xf7,0x65,0xce,0x2c,0xe1,0x82,0xde,0x88,0x11,0x24,0x45,0xdd,0x5f,0xf1,0x1b,
0x25,0x6c,0xfd,0xa0,0x70,0x18,0xb9,0x5f,0x97,0xed,0xba,0xb4,0xe4,0xc3,0x9c,0xa0,
0x97,0xc4,0x2f,0x9d,0xce,0x80,0xcd,0x3f,0x32,0x67,0x7f,0x3c,0x22,0x4a,0x86,0xb3,
0x15,0xd0,0x2e,0x37,0x7d,0xca,0x8f,0x37,0x85,0xe9,0x74,0x8f,0xfd,0xbe,0x3f,0xca,
0xa3,0xb0,0xc6,0xbf,0x00,0x1b,0x63,0xb5,0x74,0x26,0x83,0x63,0x58,0xe9,0xb3,0x15,
0xc6,0x71,0x8e,0x0b,0x74,0xfb,0x82,0xb9,0xbf,0x3d,0xf7,0x00,0xa6,0x41,0xab,0x94,
0x11,0xd1,0xb9,0xfb,0xa4,0x23,0x09,0xa8,0x4b,0xef,0x67,0xa1,0x42,0x04,0xf3,0x16,
0x0e,0xd1,0x6a,0x54,0x97,0xfe,0x21,0x1a,0xa1,0xf5,0xd3,0xae,0x4b,0x85,0x8b,0x6d,
0x44,0x5f,0x1d,0x09,0x45,0x43,0xd0,0x10,0x7c,0xe0,0x4e,0xf1,0xd1,0xba,0x33,0xab,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x73,0xc9,0xb1,0x15,0xb7,0xef,0xb0,0xa6,0x32,0x44,0xd7,0x49,0x3a,0xe5,0x82,0x05,
0x99,0xd7,0xce,0xe5,0xca,0x05,0x4d,0xb2,0xf7,0x26,0x9b,0xa7,0xf6,0x21,0xbd,0xca,
},
.inst.nonce = (unsigned char[]){
0xc2,0x04,0xe6,0xde,0x78,0x9b,0x03,0x94,0xfb,0xbe,0x66,0x63,0x46,0x6e,0xfc,0xea,
},
.inst.v = (unsigned char[]){
0xf7,0xba,0x3c,0xf1,0x65,0xbe,0x38,0x8f,0x1e,0x05,0x02,0xac,0x63,0x92,0x02,0x21,
0xd2,0x7c,0x05,0x1e,0x44,0x7e,0xc3,0xd3,0x08,0x91,0x4d,0xa7,0xfe,0x17,0x6c,0xc7,
0xfb,0x91,0x46,0xdc,0x86,0x0e,0x5a,0x5a,0xda,0x53,0xda,0x7d,0x8b,0xb8,0xf6,0x7a,
0xc8,0xe2,0x5f,0xe9,0x44,0xd4,0x53,0xf0,0x99,0x75,0x6b,0xc5,0xae,0x58,0xef,0xbf,
0x80,0x2d,0x07,0x83,0x01,0xf8,0xdf,0xad,0x9c,0x9e,0xd4,0xb3,0x3c,0x56,0x0f,0x43,
0xe4,0x9b,0xf4,0xfb,0xd3,0x53,0x48,0x00,0xac,0xeb,0xd1,0xf2,0xc3,0x81,0x77,0x2b,
0x21,0x33,0xe4,0xff,0x46,0x9e,0xa4,0x99,0x65,0xd9,0xc2,0x57,0x58,0xf2,0xab,
},
.inst.c = (unsigned char[]){
0x20,0xc0,0xbc,0xf1,0x5a,0x1e,0xd3,0xb5,0x89,0x0f,0xb3,0xc7,0xc7,0x55,0xfe,0x2f,
0x57,0x45,0x93,0x0d,0xa0,0xf6,0x53,0x03,0x04,0xe3,0xed,0xfa,0xec,0x45,0x69,0xc8,
0x79,0x60,0x64,0x7f,0xa1,0xda,0xc7,0x1c,0x31,0x61,0xe4,0xaf,0xde,0x6b,0x87,0xd4,
0x1f,0x1b,0x1b,0x9a,0xbf,0x7f,0x5b,0x5f,0x63,0xa6,0x7c,0xd6,0xa6,0x05,0xaf,0xf2,
0x31,0x1d,0x7b,0x46,0xea,0xdd,0xf9,0x75,0x29,0x9a,0x13,0x2c,0xc3,0xc2,0x72,0xcc,
0x4e,0x6f,0x48,0xe2,0x31,0xcf,0x1c,0x75,0x07,0xbf,0xaf,0x32,0x86,0xcd,0xc9,0x2a,
0xe9,0x67,0xe4,0x31,0xc5,0xef,0x70,0xb1,0x9e,0x06,0x6b,0x51,0x9f,0xbc,0x3a,
},
.inst.reseed_ctr = 1,

.gen1.entropy = (unsigned char[]){
0xcf,0xce,0xf3,0x77,0x6b,0x37,0x64,0x9a,0x7f,0x6d,0x2b,0x48,0xf4,0x43,0xda,0x79,
0xa2,0xf2,0xf8,0x1d,0x04,0xf3,0xaf,0x98,0x53,0xa9,0xe6,0x96,0xc4,0x48,0x74,0x40,
},
.gen1.v = (unsigned char[]){
0xda,0xd2,0x1f,0x2d,0x3f,0xda,0x88,0x4e,0x22,0xda,0x80,0x0f,0xaa,0xee,0x0e,0x2e,
0xb5,0xb0,0x21,0x2f,0x4a,0x37,0x4d,0x7c,0x6f,0x94,0x6e,0x4f,0x7f,0xad,0x8a,0x71,
0xe9,0x3e,0xb0,0xb8,0xd7,0xea,0xeb,0x55,0x44,0x7c,0x51,0x0c,0xe4,0x41,0xa9,0x9c,
0x8b,0xf8,0x2b,0xe1,0xa7,0x70,0x8b,0x43,0x43,0xaf,0x52,0xc3,0xad,0x8f,0xbc,0x2f,
0x29,0x6b,0x29,0xac,0x89,0xeb,0x46,0x9c,0x71,0xa4,0x1f,0x80,0x02,0xdb,0xe4,0x9f,
0xf2,0xac,0x70,0x93,0xb0,0x61,0xb7,0xf7,0x62,0xa3,0x3b,0xf8,0x4c,0xe7,0x19,0x02,
0x40,0xe8,0xa3,0xfd,0x9c,0x02,0x66,0x3b,0xcb,0x80,0xdd,0xac,0x63,0xac,0x98,
},
.gen1.c = (unsigned char[]){
0x5d,0x6d,0x9b,0xb3,0x5c,0xf2,0x29,0x19,0xda,0xc5,0x05,0x76,0x96,0x64,0x23,0xbf,
0xb4,0x62,0x32,0x3d,0x1a,0x04,0x3c,0x00,0xa8,0x56,0x50,0x63,0xfc,0x8a,0x8e,0x2d,
0x75,0x99,0x26,0xf7,0xb4,0xda,0x65,0x33,0xe0,0x9b,0xe2,0xbe,0xd5,0x33,0x51,0xe4,
0x68,0xec,0x32,0x50,0x63,0x0f,0x4d,0xd9,0x4e,0xc8,0xa6,0xbb,0x20,0xa2,0x74,0x9b,
0x18,0x32,0xfc,0x5e,0xa5,0x4f,0xa5,0xed,0xe8,0x70,0x31,0xa6,0xbe,0x21,0xfe,0xa5,
0x65,0x73,0x7f,0xb7,0xf5,0x07,0xac,0x22,0xbd,0xe3,0x83,0xad,0x69,0xcc,0xe9,0x59,
0xbd,0xf8,0xea,0xef,0x21,0xf3,0x9e,0xff,0xa5,0x25,0x59,0x6b,0x89,0xf1,0xda,
},
.gen1.reseed_ctr = 2,

.gen2.entropy = (unsigned char[]){
0xd0,0x63,0x8e,0x28,0xca,0xe8,0xd1,0xc0,0xf5,0x72,0x09,0xd6,0x77,0xd8,0x89,0xd1,
0x95,0xa6,0x72,0x02,0x3c,0xb8,0xad,0xe3,0x9f,0x79,0x49,0x89,0xe1,0xda,0xee,0x34,
},
.gen2.v = (unsigned char[]){
0xb3,0xc0,0x3b,0x5a,0x64,0x2f,0xc9,0x4f,0xb6,0x63,0xc4,0x7c,0xc7,0xbc,0xcf,0x61,
0x81,0x5d,0x8e,0x94,0xf8,0x2a,0x3b,0x59,0x03,0x17,0xed,0x7e,0xa6,0x02,0xb4,0xde,
0x8c,0x27,0xdc,0x9e,0x42,0x66,0xa7,0x98,0x57,0xc8,0xbd,0xb9,0x8c,0x84,0x9e,0xd0,
0x88,0x17,0x86,0x13,0xc4,0xda,0x20,0x99,0x93,0x20,0x42,0x33,0xbb,0xa7,0xc1,0xe8,
0x0a,0xe5,0xcb,0x6a,0xf9,0x0d,0x6b,0x7f,0x09,0x17,0xf8,0x0b,0x5b,0x0c,0xc8,0x54,
0x90,0xe3,0xa1,0xce,0x0e,0xe1,0x0f,0xb9,0x53,0x96,0x2a,0xd6,0x40,0x0f,0xf1,0x54,
0x28,0x6d,0xdf,0x29,0xd9,0xc0,0xa0,0x04,0x7b,0xf0,0x60,0x95,0xc8,0xa1,0x27,
},
.gen2.c = (unsigned char[]){
0x1f,0xff,0x76,0x2e,0x53,0x76,0x82,0x21,0xd1,0xc8,0x64,0xa1,0x8e,0xb4,0x84,0xd4,
0x69,0x30,0x50,0x11,0xf8,0x9d,0x1a,0x6c,0xca,0x4a,0x21,0x6f,0x64,0x01,0x03,0xe2,
0x19,0x65,0xbc,0xb2,0x5f,0xed,0x86,0xd4,0x0d,0xdb,0x2f,0xa9,0xbb,0xee,0x7f,0x81,
0x43,0x1c,0xe4,0xc9,0x98,0x53,0x09,0xd8,0x2a,0xc2,0xf1,0x33,0xa9,0x31,0x6f,0xd3,
0x42,0xb2,0x64,0x96,0xcb,0xe0,0x89,0x3f,0x53,0x6c,0xbc,0x52,0x8a,0x02,0x11,0x6e,
0xf7,0xf4,0x47,0x90,0x29,0x0a,0x4a,0x57,0x41,0x7e,0xfb,0xc5,0xd2,0xc7,0xb7,0x48,
0xad,0x1a,0x4e,0x6e,0x13,0xff,0x32,0xfb,0x83,0x12,0x5b,0x9c,0x0f,0xd6,0x47,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char []){
0x04,0x74,0x4d,0x1d,0x42,0x60,0x19,0x95,0xfa,0x3b,0x10,0x1d,0xed,0x3d,0x25,0x31,
0xcb,0xf4,0x5a,0xfd,0x83,0x12,0x0d,0x58,0xeb,0x26,0x59,0x4a,0x86,0x3b,0xd8,0x31,
0x83,0x11,0xb0,0x8d,0x3d,0xf4,0xc5,0x71,0xa9,0xc2,0x6d,0xff,0x63,0xa3,0xe9,0x91,
0x3a,0x9a,0x17,0xa7,0xc4,0x55,0x18,0x6f,0xdf,0xdd,0x90,0xc6,0x64,0xa8,0x4b,0x73,
0xa1,0x10,0x6a,0x5a,0x82,0xf7,0x41,0xbd,0x4c,0x7a,0x48,0xbd,0x04,0x6c,0x26,0x8d,
0x89,0x19,0xef,0xc9,0x41,0xf8,0xb4,0x5a,0x3c,0x3d,0x89,0xcf,0x37,0x14,0x1b,0x5c,
0x41,0xb1,0x0f,0xf5,0x43,0xa6,0x92,0x62,0x72,0xd6,0x23,0xad,0x8e,0xcc,0xd0,0x26,
0x55,0x20,0x90,0xad,0xcf,0xac,0xb1,0x24,0xf4,0x7c,0x4a,0xd6,0x2b,0xe9,0x0e,0xa5,
0xa0,0xa7,0x08,0x7d,0x81,0x45,0x84,0x45,0x81,0x3a,0xf8,0x8f,0xfb,0x5a,0x8c,0x35,
0x19,0xf9,0x77,0x13,0x1c,0xc8,0x51,0xcb,0x44,0x54,0xb0,0xa7,0x56,0xc8,0x37,0x3f,
0x05,0x23,0x82,0x43,0x5a,0xb9,0x34,0x71,0x8c,0x95,0x51,0x77,0x36,0x33,0x89,0xc0,
0x6b,0x0b,0x50,0x73,0x47,0x8e,0x84,0xd2,0x53,0xff,0x02,0xa3,0xf1,0xbe,0xf1,0xbb,
0xf1,0x33,0x8f,0x77,0xf9,0x2f,0x02,0x9f,0x63,0x8a,0x46,0x91,0xc4,0x8c,0x47,0x0d,
0x30,0xd2,0x30,0xf0,0x07,0xf5,0x45,0xe0,0x22,0xf6,0x6c,0x78,0xa1,0x30,0x69,0x78,
0x14,0xaa,0x55,0xd2,0x00,0x0a,0x49,0x55,0x3b,0xef,0x35,0xfa,0xb5,0x80,0x8e,0x2f,
0x3c,0xbb,0x38,0xc4,0x05,0x61,0x1f,0xa8,0x14,0x44,0x12,0x4e,0x3f,0x89,0xe1,0xe8,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x7b,0xa8,0x43,0x1c,0x16,0xfb,0x16,0xd1,0x12,0x15,0x09,0xf2,0x25,0xab,0xd5,0xb3,
0x7a,0x25,0xb0,0xc5,0xc0,0x69,0xb7,0x45,0x4c,0x22,0x66,0x26,0x9c,0xf2,0x6e,0x1b,
},
.inst.nonce = (unsigned char[]){
0x7f,0x04,0x48,0x2b,0x3a,0x14,0xd8,0xfc,0x7a,0x29,0x81,0x6a,0x27,0xca,0x66,0xa2,
},
.inst.v = (unsigned char[]){
0xa3,0x30,0x40,0xd0,0x3d,0x1a,0x6b,0x5e,0xd9,0x31,0x9a,0x37,0xa0,0xac,0x8b,0x63,
0x91,0x0c,0x2b,0xb0,0x27,0x79,0x5f,0xd7,0x81,0xfb,0x9b,0xbf,0x8a,0xa9,0xdc,0xac,
0x73,0xb3,0x27,0x6c,0xdf,0x04,0x8d,0x05,0xd7,0x33,0x2c,0x65,0xd0,0x63,0xaf,0xac,
0x34,0x55,0x88,0x1e,0xf7,0x78,0xfc,0xc5,0xcf,0x12,0xf2,0x5f,0xcf,0x5f,0xe6,0xb3,
0xd4,0x1c,0xc3,0xf3,0xfb,0x0d,0x9f,0xd9,0x56,0x7d,0xa8,0x10,0x87,0x51,0xe5,0xef,
0x5b,0x54,0xb9,0xae,0x1e,0x2f,0xb3,0x00,0x58,0x6b,0x9f,0xce,0x2c,0x00,0xaf,0xb0,
0xf5,0xb1,0x97,0xd0,0x8d,0x16,0x17,0xf5,0x49,0x6b,0x7a,0xdb,0x2c,0x9f,0x2c,
},
.inst.c = (unsigned char[]){
0x8d,0x06,0xa0,0x3f,0x5d,0x2f,0x9a,0x9c,0x40,0xd1,0x6b,0x34,0xa5,0xd3,0x4e,0xa8,
0xcd,0x00,0x1b,0x79,0x77,0xfc,0xcd,0x67,0x2d,0xde,0x8b,0xff,0xe4,0x87,0xf6,0x3f,
0x60,0x89,0x9a,0xbf,0x3e,0x57,0xc4,0x2d,0x57,0xba,0x3f,0x00,0xdb,0x41,0x1e,0x57,
0x13,0x7f,0x14,0x12,0xf9,0x47,0xf3,0xc3,0xb0,0x31,0x56,0x99,0x27,0x1c,0xbf,0xc3,
0xa8,0xa1,0xf3,0x03,0x26,0x21,0xa0,0xa7,0xdf,0xc4,0x82,0xa9,0xc3,0x6a,0x8c,0x6a,
0x5a,0x15,0xb0,0x89,0xca,0x69,0xea,0x8d,0xe9,0xa9,0xcb,0x86,0x44,0x79,0x72,0x52,
0x19,0x32,0xc0,0xe1,0x47,0x45,0x5f,0xf3,0xcc,0x07,0x06,0x05,0x81,0xa7,0xb2,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char[]){
0x64,0x65,0x3b,0x87,0x53,0x59,0xf3,0xab,0x5d,0x96,0xaa,0x49,0x4d,0x44,0x66,0x90,
0x22,0xb4,0x8c,0xa5,0xc1,0xc1,0xf7,0x6a,0x18,0x14,0xb7,0x65,0x51,0x99,0xdb,0xf3,
},
.gen1.entropy = (unsigned char[]){
0xd0,0x52,0x01,0x36,0x5d,0x2b,0xcc,0xf9,0xa6,0x58,0x04,0xb2,0x3b,0xba,0x82,0xbf,
0xb6,0x55,0x96,0xd4,0x2f,0xc5,0x1c,0xc5,0x29,0x22,0x08,0x1d,0x6e,0x50,0x74,0xa2,
},
.gen1.v = (unsigned char[]){
0x05,0x3b,0x46,0xee,0x13,0x54,0x25,0xf9,0x99,0x32,0xb0,0x06,0x57,0x22,0x11,0xf3,
0xb5,0xa1,0x68,0x11,0x11,0x12,0x76,0x99,0xb9,0x26,0x6e,0xc3,0x73,0xed,0xf7,0x83,
0x60,0xfa,0x5f,0x7e,0xbf,0x3d,0x8b,0x22,0xc3,0xfe,0x78,0x51,0x48,0x83,0x25,0x37,
0xe7,0x33,0x5f,0x02,0x0b,0x59,0x5a,0x97,0x35,0xeb,0xc5,0xb7,0x26,0x66,0xf6,0xbf,
0x2b,0x50,0xa2,0x0a,0xe1,0xc0,0x5a,0x68,0xb7,0xfa,0xd6,0x0f,0x36,0x52,0x1c,0x46,
0x27,0x2b,0xff,0x70,0x7f,0x66,0xbb,0xfa,0x2a,0xb4,0x66,0x6f,0xfc,0x2a,0x56,0x9d,
0x8e,0xfc,0x30,0x27,0xfb,0x3b,0x48,0x83,0x04,0xfb,0x38,0xfb,0xd6,0x2e,0x16,
},
.gen1.c = (unsigned char[]){
0x60,0xdc,0x33,0x8e,0x05,0x60,0xbe,0x50,0x39,0x6b,0x94,0x57,0xd4,0x55,0x83,0xb3,
0x03,0x7e,0x2f,0xd0,0x91,0xd4,0xb2,0x1d,0x25,0xbb,0x2d,0xb4,0xde,0x15,0xc5,0xdf,
0x11,0xe9,0x00,0x84,0x4c,0xca,0x4c,0xe5,0x3f,0x38,0x8e,0x4d,0x9d,0xcb,0xa4,0x3e,
0x91,0xa8,0xc1,0x15,0xdc,0x0f,0x89,0x9a,0xfc,0x15,0xa6,0xa9,0xda,0x9b,0x73,0x4c,
0x52,0x18,0x7c,0xf2,0x6a,0xe9,0xc7,0x60,0x93,0xce,0x1e,0x91,0x27,0xf7,0xb8,0x2a,
0x16,0x42,0x9b,0x4f,0x05,0xde,0xdb,0x89,0x4f,0x1e,0x3f,0x93,0x79,0x52,0x3a,0x6c,
0x0b,0x06,0xd5,0x4d,0xda,0x25,0x65,0x8e,0xe8,0x49,0x5a,0x0e,0x0a,0x9f,0x91,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char[]){
0x1e,0x50,0x04,0xc4,0xc2,0xf8,0x50,0xfa,0x6b,0xe2,0xe0,0x17,0x10,0xa4,0xdf,0x0e,
0x57,0x63,0x00,0xff,0x4f,0xc4,0xde,0x40,0xe9,0xc9,0x13,0x58,0xf2,0x6f,0xd4,0x08,
},
.gen2.entropy = (unsigned char[]){
0x59,0x18,0x5a,0x07,0xb2,0x0d,0xec,0x58,0x57,0xb7,0x9d,0xda,0xb2,0x8d,0x46,0xa5,
0x2e,0x4b,0x0a,0xad,0x9d,0xf9,0xcd,0xb7,0x76,0x13,0x38,0x54,0xaa,0xc0,0xdb,0x3e,
},
.gen2.v = (unsigned char[]){
0x2e,0x54,0xe1,0x54,0x44,0xb8,0x43,0x9e,0x2e,0x06,0x6c,0xf2,0x3d,0x94,0x88,0x0b,
0x72,0xe6,0xfd,0x3e,0xb4,0x3c,0xfd,0xa7,0xd8,0x91,0x0a,0xcb,0x89,0xc0,0xbd,0x76,
0x31,0x21,0x52,0xe6,0x91,0x18,0x9d,0x44,0x75,0x78,0xc2,0x7c,0x7b,0xdf,0xb0,0x7c,
0x2c,0x4a,0x0e,0x49,0x87,0xe3,0xf7,0x96,0xcd,0xa9,0xa0,0x0f,0x99,0xb9,0x5e,0x3d,
0xb9,0xa1,0x9a,0xf2,0xdc,0x24,0x1b,0x71,0x4d,0xba,0x0a,0x86,0x14,0xa4,0xa7,0x28,
0xe5,0xbc,0xb9,0x92,0xf8,0x1e,0x49,0x6c,0xa4,0x68,0x8a,0xdd,0x84,0x37,0x58,0x89,
0x52,0xbf,0x8b,0x16,0x7c,0xc6,0xdb,0x30,0x81,0x00,0xd8,0xf6,0xa3,0x1c,0x77,
},
.gen2.c = (unsigned char[]){
0x06,0x3f,0x41,0xe9,0x7b,0xe2,0x0f,0xcf,0x7f,0x74,0xa3,0xad,0x28,0x3c,0x6c,0x4e,
0x77,0xe3,0x43,0x1f,0x4d,0x36,0x09,0x10,0x54,0xa7,0xd3,0xf6,0x4d,0x45,0x07,0xc5,
0x25,0x66,0x4e,0x2b,0xd4,0x45,0xe2,0xb8,0x2c,0x6a,0xf4,0x37,0xf7,0x4f,0xe8,0x02,
0x91,0x9c,0x4a,0x09,0x30,0xe8,0x65,0x5e,0xe2,0x0d,0x9c,0x6c,0x7d,0x9c,0x7c,0x2f,
0xb1,0xab,0xdc,0xa2,0x58,0xb7,0x5d,0x22,0x3e,0xb0,0xfd,0xbc,0xea,0x7e,0x68,0xb8,
0xa8,0xe6,0x21,0x5f,0xea,0xee,0x03,0xcf,0x1f,0xed,0x7e,0x7c,0x6b,0x2a,0x4f,0xb9,
0x3e,0x1b,0xdb,0x43,0x97,0x0d,0x6c,0xa2,0xfd,0x97,0x03,0x65,0x05,0x51,0x76,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char[]){
0x16,0x23,0x2b,0x52,0x84,0xea,0x1a,0x8a,0xad,0xca,0x65,0x67,0x79,0x08,0xba,0x74,
0xc9,0x70,0xab,0x6c,0x34,0x6b,0xe4,0xe6,0x78,0x96,0x25,0xc5,0xdc,0x10,0xe5,0xac,
0xa1,0xe6,0xb8,0x53,0xfe,0x17,0x4b,0xc1,0x16,0xa0,0x58,0x2b,0xa7,0xe0,0x7b,0x20,
0xf6,0x98,0xea,0xbf,0xb9,0x1c,0x89,0xce,0x5b,0x87,0x23,0x26,0x36,0xbe,0xf1,0x6e,
0x5a,0x96,0x6d,0x00,0x73,0xd4,0xe9,0x2a,0x89,0x73,0x5c,0x39,0x7d,0xbb,0xdf,0x80,
0xbc,0xf9,0xc3,0x10,0xa5,0xae,0xc9,0x8a,0xd0,0x77,0x64,0x3a,0xde,0x1d,0x87,0x37,
0xd4,0x08,0xb7,0x88,0xd9,0x0c,0x6b,0x53,0x8d,0x10,0xda,0x03,0x91,0xd9,0xea,0x87,
0xe7,0x4a,0x7f,0x8d,0x05,0x27,0xb1,0xd0,0xed,0xf7,0x60,0xc6,0x9a,0xcf,0xac,0xca,
0xdd,0x29,0xbc,0x69,0xb2,0x74,0xcf,0x08,0xfd,0x2a,0x50,0x6d,0x8d,0x58,0x8b,0x4d,
0xee,0xfc,0x67,0xa3,0x1a,0xeb,0x43,0x83,0x1c,0x09,0x6e,0x3c,0xa7,0x06,0x6b,0x43,
0x45,0xb2,0x69,0x16,0xd3,0xc5,0xf8,0x31,0x82,0x40,0x6b,0x44,0x8c,0xea,0x0f,0xb0,
0xd1,0xc6,0x20,0xd8,0x75,0xa1,0x07,0xa6,0xd8,0xd0,0xd3,0xb1,0x2e,0xab,0xb4,0x04,
0xe8,0x0c,0x86,0x08,0xe7,0x14,0xac,0x43,0xd6,0xe9,0x62,0xc9,0x7e,0xf2,0x2e,0x88,
0xc9,0xac,0x66,0x2d,0x48,0x4d,0x24,0xe2,0xa8,0xcf,0x03,0x98,0xe0,0x91,0x85,0xe3,
0x36,0x80,0xb1,0x25,0x95,0xe9,0x77,0x62,0x37,0x14,0xb3,0xf7,0x09,0xb6,0xfd,0xdc,
0x18,0x61,0xc5,0x3c,0x4a,0xfd,0xb3,0xfd,0xed,0xf7,0xdc,0x4d,0xcf,0x5b,0xb0,0xe2,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x62,0xba,0x7f,0x80,0xb7,0x78,0xdf,0x98,0x6a,0x17,0x68,0x7f,0xc2,0xd2,0x41,0xc1,
0x7d,0x8a,0xe0,0x9d,0x26,0xef,0x1a,0xfe,0x06,0x95,0x7b,0xfc,0x1e,0x94,0x14,0xcd,
},
.inst.nonce = (unsigned char []){
0x1a,0xcd,0x0e,0xd5,0xaa,0xa6,0xb9,0x6a,0xb1,0x7e,0x23,0x20,0x87,0xb7,0x1d,0xe6,
},
.inst.pers = (unsigned char []){
0x5e,0x44,0x97,0x60,0x37,0xe2,0x92,0x13,0x53,0x14,0x1e,0x0e,0xcf,0x0c,0x56,0x20,
0x6b,0xd8,0x5c,0x05,0x19,0x43,0xe7,0xb8,0x10,0xbd,0x0e,0xf8,0xe5,0xd3,0xa0,0x2a,
},
.inst.v = (unsigned char []){
0x93,0x2b,0xba,0x3b,0x03,0xb0,0x11,0xad,0xef,0x52,0x87,0xc1,0x49,0xc5,0x42,0x57,
0x03,0x15,0x9f,0x64,0x1a,0xad,0x52,0xf9,0xd9,0x44,0xf4,0x25,0xf5,0x4c,0x49,0x72,
0xd8,0x7e,0x01,0xe8,0xf2,0x66,0xcc,0x61,0xdc,0xaa,0x0d,0x9d,0x4e,0xc9,0xc6,0xb9,
0x2b,0x7e,0x6a,0x29,0x63,0xf4,0x53,0x78,0xbf,0x63,0x19,0x27,0x63,0xca,0xcf,0x58,
0x01,0xc3,0x7d,0x47,0x6e,0xf3,0x9a,0x4b,0x7a,0x2f,0x44,0x13,0x44,0xbe,0x96,0xeb,
0x72,0x04,0xb3,0xd7,0xd4,0x20,0x2f,0xbf,0x67,0xae,0x8a,0x57,0x1a,0x66,0x61,0xe2,
0x86,0x34,0x8c,0xaa,0xc9,0x13,0x8b,0x89,0x68,0x7d,0x9f,0x8e,0x94,0x02,0xb6,
},
.inst.c = (unsigned char []){
0x43,0x45,0xf7,0x0e,0x8e,0x44,0x16,0xf0,0x92,0xdd,0xe1,0x9b,0x10,0x74,0x12,0x33,
0xec,0x6b,0xe1,0x57,0x54,0x47,0xb9,0xb6,0xff,0x3e,0x06,0xc9,0x4c,0x61,0x0d,0xfc,
0xf7,0xd8,0xc3,0xc8,0x25,0x23,0x38,0x89,0x0f,0x5d,0xf7,0x05,0x4d,0x2e,0x3e,0xc2,
0xa2,0xf0,0x7f,0x6a,0x01,0x30,0x20,0x6b,0x8b,0x87,0x51,0x7e,0x8f,0x0b,0xec,0xe7,
0x5c,0x31,0x34,0x78,0x9f,0xea,0x34,0x88,0x04,0xa7,0x27,0x27,0xe9,0xe4,0xe1,0x13,
0x9e,0xe5,0xf6,0x0e,0x90,0x3c,0x15,0xbc,0x0f,0xed,0xf9,0x09,0xb3,0x0e,0x08,0x8d,
0xeb,0xa7,0x2f,0x77,0x57,0x21,0xa8,0x15,0x1c,0x30,0x1f,0xda,0xd7,0x01,0x71,
},
.inst.reseed_ctr = 1,

.gen1.entropy = (unsigned char []){
0x7b,0xd0,0xf0,0x37,0x1b,0x81,0x79,0xbc,0x40,0x48,0x10,0xdc,0xcf,0xa9,0xc5,0x53,
0x2f,0xa7,0xae,0xca,0x7b,0x41,0x5d,0xb7,0xee,0x31,0x7d,0x5a,0x0c,0xf6,0x3c,0xb9,
},
.gen1.v = (unsigned char []){
0x95,0x17,0xc5,0xa2,0x6a,0x1d,0x73,0xf3,0x37,0xbe,0x44,0xc7,0x65,0x63,0x49,0x6f,
0x9a,0xf2,0xb6,0x75,0x46,0xf4,0x83,0xb1,0x77,0x67,0x99,0xe4,0xc0,0x08,0xe2,0xcf,
0xfd,0xef,0x37,0x59,0xa8,0x08,0x69,0x76,0xce,0x78,0x13,0x15,0x3c,0x37,0xd3,0x21,
0x8b,0x4c,0xba,0xe2,0xb6,0x90,0x19,0xc9,0x93,0xa7,0x5d,0x52,0x6e,0xff,0xe0,0x5e,
0x0c,0x0d,0x65,0x19,0xea,0x42,0x60,0x48,0xbb,0x39,0x79,0x44,0x64,0x59,0x69,0x5c,
0x53,0x11,0x6b,0x42,0x84,0xa6,0xab,0x41,0xff,0xea,0x18,0xe9,0x7d,0x0d,0xe8,0x7a,
0xee,0xf1,0x08,0xa0,0xe9,0xd8,0x48,0x93,0x85,0x02,0x98,0xd0,0xa8,0xad,0x6b,
},
.gen1.c = (unsigned char []){
0x76,0x3d,0x01,0xa4,0xef,0xc5,0x04,0xf8,0x40,0x8d,0xc6,0x12,0xbc,0x33,0xc2,0x61,
0x40,0x24,0x79,0x6e,0x2e,0x56,0x26,0x11,0x3b,0xae,0x91,0xb2,0xc3,0x76,0x6e,0x64,
0x8a,0x6a,0xac,0x83,0x98,0x89,0xc4,0x8a,0x0c,0xa5,0x1d,0xa8,0xec,0xa3,0x39,0xbb,
0xee,0x80,0xd0,0xc1,0x62,0x78,0xbd,0x85,0x88,0x10,0x74,0x7b,0x3b,0x9d,0xdb,0x2b,
0xe0,0x7a,0x5d,0x1e,0xbe,0xb1,0xf4,0x35,0x20,0x45,0x3b,0xb4,0x00,0x87,0x29,0xf1,
0x69,0x33,0x46,0xd4,0x27,0x27,0x0c,0x37,0x7e,0x09,0x0a,0xb2,0x3f,0x12,0x0a,0xf6,
0x66,0x50,0x65,0x79,0xa8,0x5b,0x6c,0xd1,0x75,0xea,0x7a,0xfa,0x6e,0x1d,0x74,
},
.gen1.reseed_ctr = 2,

.gen2.entropy = (unsigned char []){
0xe1,0x0b,0xed,0x57,0x82,0xd4,0xe9,0xd1,0x51,0x86,0x67,0x14,0xde,0xe9,0x97,0x97,
0xb6,0x40,0x9f,0xec,0x81,0xf3,0x14,0xfa,0x22,0xbb,0xe6,0x46,0xcd,0xb9,0x25,0xa9,
},
.gen2.v = (unsigned char []){
0xdc,0x49,0x58,0x39,0x3b,0x4b,0x3e,0xb0,0xc2,0xdc,0x24,0x87,0x55,0xf0,0x20,0xa0,
0xd7,0xa8,0x9f,0x44,0xd9,0x4e,0xc5,0x3a,0x89,0x1b,0xd4,0x0f,0x2c,0x23,0x22,0xc2,
0xc1,0x7f,0x5c,0x9b,0x34,0x5d,0xd0,0x27,0x21,0xf7,0xe5,0x06,0x09,0xd7,0x27,0x58,
0x8a,0x6b,0x4a,0x0b,0x6b,0x98,0xe5,0x87,0xd6,0x96,0x7c,0xa7,0x5b,0xc3,0x84,0x9a,
0x6b,0xc6,0x3d,0x75,0xe6,0xb5,0x38,0x6c,0x86,0x57,0xf4,0x12,0xc4,0xd1,0x4b,0x9f,
0x00,0x95,0xa6,0x03,0x86,0x89,0x27,0xa0,0xd2,0x07,0x79,0xdb,0xa9,0x1c,0x20,0xf0,
0xb7,0xdb,0xc1,0x11,0x6e,0x5a,0x73,0x03,0x3c,0xae,0xbf,0x6c,0xfd,0xae,0xb8,
},
.gen2.c = (unsigned char []){
0xef,0x08,0x9b,0x11,0xdb,0x37,0xf5,0xa7,0xb9,0x6b,0xb4,0x9f,0x7a,0xff,0x41,0x56,
0x41,0x93,0x33,0x36,0xa9,0x85,0xd4,0x7c,0x3e,0xe6,0x06,0x02,0xbd,0x9f,0xa7,0x0d,
0x6a,0x46,0x05,0x55,0x05,0xcc,0x31,0x6f,0x0f,0x88,0x59,0x12,0x7c,0x81,0x6c,0x77,
0x47,0x23,0x16,0x58,0x5d,0x14,0x5d,0xf3,0xc4,0x6f,0xa8,0x14,0x96,0x56,0x06,0x16,
0x62,0x93,0xd0,0x7f,0x2d,0x1f,0x90,0x63,0x6f,0x68,0xd3,0xe7,0xaa,0x41,0x04,0x7b,
0x61,0x01,0x28,0x74,0x70,0xce,0x14,0xbc,0x15,0xb7,0x22,0xe4,0xcd,0xd0,0xf5,0x37,
0xf0,0x76,0xfd,0x02,0x47,0x6d,0x7e,0x9c,0x70,0x4b,0xf1,0xb7,0x31,0xdc,0xb0,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char []){
0xde,0x3d,0xdd,0xcb,0xfc,0x99,0xc2,0x25,0x8a,0x98,0xbd,0x04,0x2e,0x57,0x15,0x99,
0xa3,0x11,0xd8,0x6b,0x5f,0x23,0xfe,0x50,0xe8,0x9a,0x42,0x58,0x8d,0x51,0x60,0x6f,
0x51,0xb3,0x5e,0xc7,0xd4,0xe7,0x5a,0x49,0x26,0x16,0x35,0x6a,0x63,0x5e,0x7b,0x2d,
0x3f,0x1e,0xea,0xa5,0xdb,0x19,0xb8,0xfc,0xb7,0xa9,0x4e,0x27,0x4f,0x36,0x9e,0x73,
0xa6,0x7b,0xc2,0x8a,0x8b,0xf5,0xd4,0x95,0x59,0xfa,0xdc,0xda,0xfd,0x1f,0xaf,0x30,
0xca,0x52,0x41,0x10,0x12,0xa5,0xe3,0xff,0xb6,0x68,0x65,0xf4,0x1c,0x0c,0x98,0x80,
0x50,0x61,0x44,0x6e,0x62,0xfe,0x69,0x79,0xbc,0x5d,0x09,0x5f,0x1d,0x87,0x2b,0x96,
0x37,0x5e,0x33,0x5c,0xde,0x53,0x80,0x31,0x98,0x8f,0xc4,0xec,0x32,0xab,0xe4,0xda,
0xb5,0xff,0x18,0xf4,0x5f,0xb5,0x12,0x7a,0x14,0x07,0x9f,0x0f,0x0a,0x82,0x51,0x93,
0x20,0x65,0xb1,0x6a,0xf2,0x8d,0xd4,0xb0,0x62,0x07,0x3d,0xcb,0x0b,0xa1,0xb2,0x04,
0x95,0x97,0xdf,0x45,0xc1,0x7c,0x0b,0xc0,0x6a,0x23,0xd2,0x4b,0x8f,0xdd,0x66,0x97,
0x8a,0x24,0x39,0x8b,0x5d,0x42,0x63,0x7c,0xbb,0x53,0xd1,0x98,0xa5,0x6b,0x61,0x93,
0xb3,0x57,0x52,0x1d,0xb7,0x11,0x7b,0x7b,0x44,0x2e,0x75,0xbf,0x16,0x9e,0x42,0xbb,
0x07,0x98,0xfa,0x59,0xfa,0xdc,0x19,0xf5,0xd9,0xe1,0x05,0x0a,0x43,0x45,0x35,0x8b,
0xd7,0xe4,0x0f,0x90,0x96,0x45,0xd8,0x05,0x20,0xa0,0x93,0xb2,0x78,0x90,0xa3,0xd1,
0x6e,0xd2,0x27,0x79,0x90,0xc3,0xa4,0xb3,0x6a,0x3a,0x0c,0x5f,0x37,0x45,0xec,0x59,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x19,0xff,0x49,0x7e,0x90,0x6d,0x08,0x11,0xa0,0xb7,0xf7,0x75,0x86,0x18,0xaf,0xab,
0x44,0xfc,0xe8,0xed,0x35,0x38,0xab,0xfd,0x10,0x1c,0x58,0x6a,0x9c,0xea,0xf6,0x2d,
},
.inst.nonce = (unsigned char []){
0x49,0x80,0xcb,0x99,0x83,0x87,0x0a,0x35,0xd8,0x00,0xff,0xf9,0x32,0x78,0x40,0xc5,
},
.inst.pers = (unsigned char []){
0x2e,0x34,0xbe,0xa4,0xc1,0x28,0x43,0x4a,0xa2,0x85,0x9c,0xc3,0x39,0x13,0x71,0x6b,
0xc8,0x05,0xa1,0x83,0xbb,0x17,0x62,0xfc,0x3a,0xb5,0x44,0x89,0xd9,0xbd,0x1c,0x56,
},
.inst.v = (unsigned char []){
0x6c,0xf5,0x7e,0xdf,0x5f,0xec,0x61,0x34,0x5e,0xf8,0xcc,0x5e,0x39,0x1d,0xcc,0x3f,
0x88,0x0a,0xd5,0x51,0xaa,0xfb,0x16,0x37,0x57,0x4e,0x92,0xc1,0xfa,0x8f,0x3e,0xd8,
0x15,0x19,0xa0,0xf1,0xe9,0x32,0xa1,0xf8,0xb2,0xa5,0x53,0xb0,0xfa,0xb9,0xf9,0x9e,
0x99,0x51,0xee,0x49,0x9c,0x59,0xfd,0x23,0xf3,0x1a,0x1b,0x73,0x8f,0x3a,0x54,0x7d,
0xca,0xb6,0x36,0x8e,0x80,0x2c,0x57,0x89,0x6a,0x4f,0xcf,0xeb,0x3e,0xd0,0xe6,0x26,
0x02,0x97,0xd6,0x16,0x5f,0xb2,0xe4,0x10,0x88,0x6f,0xe2,0xaf,0x76,0xe5,0x0e,0x20,
0x23,0xc2,0xf2,0xcd,0xb3,0xf7,0xa8,0x1f,0x48,0x9a,0x76,0xfd,0x46,0x57,0x0e,
},
.inst.c = (unsigned char []){
0x70,0xc9,0xfc,0xe3,0xd5,0x7e,0x39,0x4b,0x6a,0xc5,0xc0,0x04,0x8b,0xfd,0xae,0x4e,
0xb5,0xa3,0xed,0x6a,0xa2,0xb2,0xfa,0xf5,0xe6,0xc5,0xb7,0xc2,0x9e,0xc6,0x54,0x3c,
0x47,0xf5,0x0e,0x7b,0xe8,0x49,0x66,0xb5,0xa6,0x73,0x9a,0xa3,0x19,0x1c,0xa3,0xb2,
0x1b,0xee,0xb4,0x16,0xaa,0x8f,0x7a,0x10,0xd0,0xee,0x15,0x46,0xf5,0x51,0x20,0x3d,
0xd3,0xca,0xce,0x74,0xb6,0xaf,0x8d,0xce,0xb4,0x82,0xe7,0x53,0x50,0x0f,0xf4,0x17,
0x65,0xdd,0xde,0xee,0x3a,0x72,0x5d,0x83,0xbf,0x5b,0xc0,0xda,0xff,0x35,0x86,0xc3,
0xf5,0x75,0x0f,0x76,0xde,0x15,0x3c,0x24,0xda,0x52,0xfe,0xfe,0x50,0xb4,0x24,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char []){
0x47,0xdd,0x5c,0x72,0xea,0x44,0xdb,0xf1,0xd2,0x35,0x04,0x0b,0x18,0xa8,0x21,0xc2,
0x0f,0xb4,0x0b,0x88,0xc0,0x39,0x1a,0x5c,0x16,0x3f,0x9c,0x90,0x2f,0x6c,0x6e,0xb7,
},
.gen1.entropy = (unsigned char []){
0xbe,0x45,0xca,0x98,0x78,0x24,0x7b,0x00,0x8e,0x5d,0xe8,0xc5,0x88,0xb6,0xcb,0x84,
0x04,0x8c,0x0d,0x90,0x35,0x7d,0xbd,0xfc,0x98,0xe9,0xe2,0x87,0x91,0x2f,0xdb,0x33,
},
.gen1.v = (unsigned char []){
0x2b,0xec,0x9f,0xc2,0x78,0x1d,0xb4,0xcb,0xfa,0x98,0x50,0x18,0x15,0x5b,0xdf,0x51,
0x02,0xe3,0x1b,0x0f,0x34,0x0c,0x32,0x9a,0xa0,0xa5,0xca,0x1a,0x70,0xb6,0xc3,0x90,
0x6a,0x6a,0x8e,0x03,0xba,0x62,0xfd,0x2f,0xd8,0x1d,0xc9,0xc2,0xc0,0xdc,0x86,0xea,
0x8f,0x8d,0xea,0x1f,0xb5,0x38,0xa0,0xdb,0xf5,0x69,0x8e,0x27,0xcc,0x2d,0x83,0xf8,
0x0a,0xfb,0x4a,0xfe,0xfc,0x46,0x8c,0xda,0xf6,0xeb,0xce,0x06,0xee,0x35,0xcd,0xd0,
0x59,0x77,0xb2,0x32,0x53,0x40,0xd1,0x91,0xff,0x03,0xb2,0xc4,0x33,0x90,0x8c,0x31,
0xf2,0xbc,0x28,0xd6,0x23,0x72,0xaa,0xd9,0x67,0x1d,0xe0,0x59,0x8a,0x35,0x09,
},
.gen1.c = (unsigned char []){
0xcf,0xf7,0xfc,0x70,0x9b,0x81,0x74,0x33,0x71,0x24,0x54,0xd6,0xbc,0xd5,0x74,0xde,
0xae,0x3f,0x2b,0xc2,0xc2,0x00,0xf0,0x88,0x38,0x8b,0x07,0x37,0xfa,0x0d,0xfd,0x8f,
0x7c,0x31,0x1b,0x8b,0x67,0xd3,0x7e,0x0a,0x7f,0x2a,0x55,0x1c,0x69,0x86,0x33,0xcd,
0x4e,0x01,0x56,0x4d,0x36,0x72,0xae,0x9c,0x01,0xdb,0x90,0x6d,0xff,0xc2,0xa4,0xc3,
0xf0,0x45,0x56,0x38,0xda,0x06,0x60,0x51,0x80,0x00,0x4b,0xb7,0xae,0x1d,0x01,0x86,
0x19,0xda,0x3d,0xdf,0xab,0xef,0xf7,0x37,0x11,0xe2,0x5c,0xff,0xf0,0x65,0x77,0x2a,
0xcf,0xa0,0xd6,0x79,0x54,0xbe,0xe4,0x02,0xc0,0x41,0xf6,0xc7,0x82,0x60,0xf3,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char []){
0xcf,0x20,0x40,0xe9,0x04,0x6a,0x69,0xdd,0x96,0x38,0xde,0x94,0x1f,0x00,0x90,0xb7,
0x53,0x5c,0x51,0xcf,0xa9,0xf1,0xc7,0xbb,0x2a,0x56,0xa3,0x32,0x32,0x69,0x43,0xd3,
},
.gen2.entropy = (unsigned char []){
0xb8,0x71,0x61,0x1f,0x8f,0xcb,0x8c,0x86,0x0a,0x72,0xc4,0xfd,0x40,0x6d,0x49,0x39,
0x33,0x5a,0x03,0x1e,0x0d,0xe9,0xf2,0xd4,0x36,0xd4,0x73,0x6b,0x6b,0x06,0x0c,0x2d,
},
.gen2.v = (unsigned char []){
0x59,0x44,0xf8,0xec,0xc9,0xa4,0x4c,0x0f,0x05,0x3f,0xfe,0x7e,0xa3,0x64,0x27,0xfc,
0xa5,0x6f,0x26,0xa3,0xd0,0x96,0x68,0xff,0x08,0x52,0xde,0x82,0x2c,0xa1,0x1a,0x88,
0x74,0x68,0xd6,0x36,0x25,0xf1,0xe8,0x6f,0xa8,0xaa,0x2a,0x12,0xc9,0x45,0x59,0xc5,
0xa4,0x01,0xe1,0x9f,0x8c,0xd5,0x8a,0xd9,0x53,0x60,0x12,0x8b,0x85,0x2f,0xef,0xbe,
0xbe,0x6f,0x8b,0x9e,0x96,0x7f,0x80,0xd1,0x84,0x9d,0x8b,0xb7,0x9c,0x0e,0x34,0x9b,
0x74,0x27,0xdf,0x85,0xa5,0x52,0xd5,0x48,0x7f,0xeb,0xd5,0x72,0xc1,0x17,0x8e,0x23,
0x2b,0x2c,0xd6,0x1b,0x2b,0xf9,0x67,0x45,0xba,0xe5,0xd4,0x60,0x64,0x0b,0x56,
},
.gen2.c = (unsigned char []){
0x34,0x0e,0xb7,0x02,0xa2,0x0b,0x75,0xed,0x0a,0x68,0x57,0x39,0x4d,0xcb,0x10,0xfc,
0xc2,0xad,0xd4,0xfa,0xf4,0x98,0x94,0xbe,0x67,0xd6,0xf2,0x03,0xf7,0x7b,0x71,0xd0,
0x35,0xe7,0xa7,0x37,0xee,0x24,0xdc,0x3b,0x6e,0x22,0xb5,0x95,0xf4,0xc0,0x86,0xb9,
0x2d,0x57,0xab,0x21,0x18,0xb9,0xa5,0x0d,0xad,0x6d,0x1f,0xf4,0xb4,0xfd,0x7b,0x3b,
0x51,0x09,0x88,0xed,0x49,0x93,0x9e,0xec,0x82,0x2b,0xf9,0x14,0x93,0x61,0x2b,0xe6,
0x54,0x66,0x5b,0x27,0x8e,0xab,0x9e,0x6d,0x8b,0xeb,0x25,0xf3,0xe2,0x08,0xd8,0xc3,
0x68,0x60,0x5a,0x39,0x9a,0x94,0x7e,0x13,0xa0,0x02,0xe3,0x67,0xed,0x50,0x3c,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char []){
0x2d,0x99,0x0f,0x0d,0xe4,0x3d,0x3a,0x4b,0x29,0x30,0x54,0x2c,0x27,0xad,0x27,0x45,
0x8e,0x88,0x65,0xca,0x6b,0x8f,0x27,0xfd,0x7a,0x96,0x9c,0xf4,0xe2,0xa0,0x32,0x3e,
0x38,0xfe,0x6f,0x50,0x5a,0x2d,0xba,0x48,0x8e,0xa6,0xb0,0x43,0x65,0x20,0x9c,0x6d,
0xb7,0x86,0xcb,0xbf,0x0a,0x7c,0x73,0xb4,0xfd,0x56,0xd2,0x49,0x87,0x71,0x9d,0xb0,
0xfd,0xba,0x1a,0x3f,0x07,0x14,0x95,0x21,0xdc,0xf5,0xb7,0x75,0x9c,0x61,0x0d,0xa2,
0x2d,0x15,0x10,0x57,0xac,0xef,0xe7,0x0d,0xf1,0xcc,0xae,0xb6,0x7a,0x97,0x51,0x59,
0xb8,0x99,0x6a,0xca,0x93,0xd7,0xa4,0x80,0x96,0x92,0x6d,0xb4,0x38,0x1b,0xbc,0xe4,
0x81,0x27,0x7d,0x7a,0xb2,0x7c,0xbc,0x03,0x88,0xf0,0xb7,0xce,0xdb,0xbf,0xb8,0x42,
0x1c,0xb1,0xdc,0x5f,0x2a,0x9c,0x67,0x7f,0x62,0xac,0xf9,0x6a,0xb2,0x5e,0x7e,0x40,
0x6c,0xe8,0x2f,0x5b,0x96,0xbc,0xb4,0x71,0xaf,0xbd,0xf4,0xb3,0xf5,0xa6,0xfb,0xcb,
0x8d,0xa4,0x5d,0x22,0x58,0xe3,0x50,0xe7,0x7d,0x46,0x33,0xb0,0xc1,0xda,0x69,0x16,
0x62,0xdd,0x86,0x99,0x09,0xdc,0xfd,0x7c,0x8e,0xd0,0xf5,0x4b,0xa7,0xaf,0x0f,0x9c,
0x03,0x8e,0xb3,0x2d,0x32,0xb7,0x05,0xe5,0x1b,0x35,0xbb,0x3c,0x2e,0xef,0xf0,0x10,
0xbb,0x47,0xee,0x32,0x6c,0x23,0x18,0xb5,0xbc,0xda,0x96,0x3c,0x2d,0xad,0x41,0x9c,
0x59,0x23,0xe3,0x68,0xd9,0xb2,0x8f,0x25,0xb0,0x48,0xa8,0x7b,0xdb,0xa0,0xa9,0x0d,
0x98,0xc2,0x4c,0x81,0xb6,0xdb,0xde,0x0f,0x58,0x05,0x4a,0x41,0xa8,0x29,0x3a,0x65,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x31,0x44,0xe1,0x7a,0x10,0xc8,0x56,0x12,0x97,0x64,0xf5,0x8f,0xd8,0xe4,0x23,0x10,
0x20,0x54,0x69,0x96,0xc0,0xbf,0x6c,0xff,0x8e,0x91,0xc2,0x4e,0xe0,0x9b,0xe3,0x33,
},
.inst.nonce = (unsigned char []){
0xb1,0x6f,0xcb,0x1c,0xf0,0xc0,0x10,0xf3,0x1f,0xea,0xb7,0x33,0x58,0x8b,0x8e,0x04,
},
.inst.v = (unsigned char []){
0x3a,0x85,0xca,0x10,0xea,0xc6,0x83,0xd6,0xa9,0x27,0x05,0x94,0xd1,0x7f,0x33,0xa2,
0x1d,0xad,0x7b,0x9b,0x25,0x9c,0x2a,0x17,0x44,0x62,0xa5,0xe0,0xc9,0x09,0xa1,0x33,
0xdb,0x84,0xb4,0xee,0x2b,0xdb,0x0f,0x72,0xcd,0xce,0xf7,0xd6,0x28,0x54,0xe5,0x35,
0x46,0x84,0x52,0x28,0x5d,0xbe,0x8e,0x46,0xbe,0xd3,0x96,0x5d,0xc9,0xc6,0x69,0x52,
0xde,0xfa,0x48,0x87,0x94,0x93,0xed,0xc0,0x1b,0xc0,0x7e,0xd4,0x97,0x3c,0x11,0x5c,
0xfd,0xd9,0x94,0x7a,0x70,0x84,0x65,0x35,0x1b,0x78,0xb8,0x04,0x65,0x2e,0xc7,0xcb,
0xe7,0xf6,0xe2,0xa0,0x91,0x93,0xfa,0x35,0x2f,0xf9,0x91,0xd3,0x8c,0x94,0xac,
},
.inst.c = (unsigned char []){
0x74,0xea,0x43,0x7c,0x49,0x12,0x6f,0xf3,0x61,0xfe,0xab,0x56,0x39,0xa8,0xad,0x31,
0x8d,0x45,0x5c,0x94,0xb3,0xf9,0x99,0xff,0x16,0x06,0xf5,0x92,0xc2,0x7f,0x8b,0xf0,
0xbe,0x56,0x2c,0x7b,0xff,0xa2,0x97,0xde,0x85,0x12,0xef,0x44,0xb0,0xdf,0xc8,0xdb,
0x5c,0xb1,0x7c,0x96,0x92,0xac,0x0d,0x80,0xf0,0x66,0x96,0x1e,0x64,0x26,0x08,0x41,
0x08,0x08,0x9e,0xee,0x4a,0x75,0x9d,0x53,0x09,0xec,0x86,0x16,0x68,0xdd,0xeb,0x1c,
0x31,0xce,0xef,0x26,0xed,0xad,0x67,0x8b,0x6f,0x36,0xc3,0xeb,0xcb,0x9c,0x93,0x6c,
0xaf,0xce,0xe3,0xd9,0xa9,0x6a,0xe6,0x55,0x4e,0x22,0xd4,0x28,0x88,0xab,0x07,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0xa0,0xb3,0x58,0x4c,0x2c,0x84,0x12,0xf6,0x18,0x40,0x68,0x34,0x40,0x4d,0x1e,0xb0,
0xce,0x99,0x9b,0xa2,0x89,0x66,0x05,0x4d,0x7e,0x49,0x7e,0x0d,0xb6,0x08,0xb9,0x67,
},
.res.v = (unsigned char []){
0xb3,0x7f,0x9a,0xa3,0x9c,0x5a,0x80,0xdf,0x56,0xc0,0x40,0x40,0x24,0x07,0x96,0x0e,
0xf6,0xf8,0x89,0x2d,0x1a,0x68,0x8f,0xfc,0x93,0xba,0xd6,0xeb,0xe6,0xaf,0x44,0xd5,
0x5c,0xcd,0x66,0xc1,0xf4,0x4e,0xb5,0x31,0xe9,0xda,0xc1,0xc9,0x44,0x76,0x81,0xd7,
0xb2,0x7b,0x2b,0x70,0x3b,0x49,0x00,0x32,0x69,0x6b,0x32,0x33,0x0b,0x5e,0xdd,0x12,
0x3e,0x5e,0xce,0x7c,0x40,0xef,0xe7,0x0a,0x29,0x82,0x2e,0xa8,0xe4,0xe4,0x54,0xbb,
0x72,0x08,0x5c,0x6b,0x03,0x7a,0x86,0x52,0xec,0x22,0x7f,0x89,0x9d,0xd0,0x14,0x55,
0xdb,0x8e,0xe7,0xb6,0xb2,0xe9,0x21,0x14,0xf6,0xf9,0xfb,0x67,0x8e,0x63,0x32,
},
.res.c = (unsigned char []){
0x90,0x8a,0xd8,0x58,0xdb,0x2c,0x5d,0x21,0xfa,0x1c,0xd8,0x60,0x21,0x7b,0xd7,0x5a,
0xd0,0xba,0x1d,0xf2,0xfd,0x24,0xe3,0x03,0x96,0x4c,0x01,0x11,0x3a,0x0b,0x02,0x4a,
0x1e,0x53,0x64,0x0d,0x5a,0xe3,0x39,0x04,0x0b,0x43,0x57,0xc1,0xf3,0xc0,0xbe,0x2f,
0x14,0x60,0x7b,0x13,0x85,0xe9,0x68,0x18,0x3c,0x53,0xec,0xd9,0xa3,0x3d,0xdb,0x04,
0xb3,0xac,0x36,0xdf,0xc1,0x35,0x3d,0x85,0x71,0x15,0x9a,0x0b,0x31,0xb8,0x1b,0x5d,
0x3d,0xe2,0x4b,0x8a,0xe6,0x53,0x0c,0x83,0x8f,0xa8,0x71,0x2e,0xa5,0xd4,0xd5,0x87,
0x63,0xf2,0xbe,0x0a,0xb1,0x98,0x99,0x87,0xc5,0x6b,0xfd,0x31,0x5d,0xf5,0x21,
},
.res.reseed_ctr = 1,

.gen1.v = (unsigned char []){
0x44,0x0a,0x72,0xfc,0x77,0x86,0xde,0x01,0x50,0xdd,0x18,0xa0,0x45,0x83,0x6d,0x69,
0xc7,0xb2,0xa7,0x20,0x17,0x8d,0x73,0x00,0x2a,0x06,0xd7,0xfd,0x20,0xba,0x47,0x1f,
0x7b,0x20,0xca,0xcf,0x4f,0x31,0xee,0x35,0xf5,0x1e,0x19,0x8b,0x38,0x37,0x40,0xfb,
0x34,0x72,0x4a,0x07,0x47,0xe2,0x61,0xc8,0x00,0xfa,0x0f,0x74,0x4b,0xdc,0x84,0x2d,
0x37,0x19,0x9f,0x6a,0xcf,0x5f,0x4a,0xf0,0x41,0xa6,0x60,0x08,0x78,0xcf,0x72,0xa7,
0xce,0xaa,0x75,0x0f,0xa1,0xc2,0x35,0x46,0xf9,0x62,0xaf,0xe9,0x7c,0x05,0x56,0x83,
0xea,0xf5,0x13,0x1d,0x9f,0x9c,0x88,0x2e,0xdb,0x93,0xc5,0x0a,0xdb,0xa9,0x63,
},
.gen1.c = (unsigned char []){
0x90,0x8a,0xd8,0x58,0xdb,0x2c,0x5d,0x21,0xfa,0x1c,0xd8,0x60,0x21,0x7b,0xd7,0x5a,
0xd0,0xba,0x1d,0xf2,0xfd,0x24,0xe3,0x03,0x96,0x4c,0x01,0x11,0x3a,0x0b,0x02,0x4a,
0x1e,0x53,0x64,0x0d,0x5a,0xe3,0x39,0x04,0x0b,0x43,0x57,0xc1,0xf3,0xc0,0xbe,0x2f,
0x14,0x60,0x7b,0x13,0x85,0xe9,0x68,0x18,0x3c,0x53,0xec,0xd9,0xa3,0x3d,0xdb,0x04,
0xb3,0xac,0x36,0xdf,0xc1,0x35,0x3d,0x85,0x71,0x15,0x9a,0x0b,0x31,0xb8,0x1b,0x5d,
0x3d,0xe2,0x4b,0x8a,0xe6,0x53,0x0c,0x83,0x8f,0xa8,0x71,0x2e,0xa5,0xd4,0xd5,0x87,
0x63,0xf2,0xbe,0x0a,0xb1,0x98,0x99,0x87,0xc5,0x6b,0xfd,0x31,0x5d,0xf5,0x21,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char []){
0xd4,0x95,0x4b,0x55,0x52,0xb3,0x3b,0x23,0x4a,0xf9,0xf1,0x00,0x66,0xff,0x44,0xc4,
0x98,0x6c,0xc5,0x13,0x14,0xb2,0x56,0x03,0xc0,0x52,0xd9,0x0e,0x5a,0xc5,0x49,0x69,
0x99,0x74,0x2e,0xdc,0xaa,0x15,0x27,0x3a,0x00,0x61,0x71,0x4d,0x2b,0xf7,0xff,0xb3,
0x2b,0x70,0x00,0xbf,0xde,0xb1,0x06,0x05,0xf3,0x61,0x74,0xeb,0x33,0xa4,0x8a,0x4c,
0xc0,0x07,0xc2,0x3b,0xb0,0x35,0x97,0xb4,0xd8,0xa6,0x37,0x3c,0xa7,0x03,0x7e,0x8a,
0x8f,0xf0,0x8f,0x63,0x77,0x9d,0xa9,0xe6,0x18,0x78,0xb1,0x88,0x6c,0xb0,0x84,0xba,
0x68,0xce,0xef,0x8a,0xd4,0xe5,0xba,0x77,0x20,0xac,0xbd,0x3b,0x26,0x28,0x22,
},
.gen2.c = (unsigned char []){
0x90,0x8a,0xd8,0x58,0xdb,0x2c,0x5d,0x21,0xfa,0x1c,0xd8,0x60,0x21,0x7b,0xd7,0x5a,
0xd0,0xba,0x1d,0xf2,0xfd,0x24,0xe3,0x03,0x96,0x4c,0x01,0x11,0x3a,0x0b,0x02,0x4a,
0x1e,0x53,0x64,0x0d,0x5a,0xe3,0x39,0x04,0x0b,0x43,0x57,0xc1,0xf3,0xc0,0xbe,0x2f,
0x14,0x60,0x7b,0x13,0x85,0xe9,0x68,0x18,0x3c,0x53,0xec,0xd9,0xa3,0x3d,0xdb,0x04,
0xb3,0xac,0x36,0xdf,0xc1,0x35,0x3d,0x85,0x71,0x15,0x9a,0x0b,0x31,0xb8,0x1b,0x5d,
0x3d,0xe2,0x4b,0x8a,0xe6,0x53,0x0c,0x83,0x8f,0xa8,0x71,0x2e,0xa5,0xd4,0xd5,0x87,
0x63,0xf2,0xbe,0x0a,0xb1,0x98,0x99,0x87,0xc5,0x6b,0xfd,0x31,0x5d,0xf5,0x21,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0xef,0xa3,0x5d,0xd0,0x36,0x2a,0xdb,0x76,0x26,0x45,0x6b,0x36,0xfa,0xc7,0x4d,0x3c,
0x28,0xd0,0x1d,0x92,0x64,0x20,0x27,0x5a,0x28,0xbe,0xa9,0xc9,0xdd,0x75,0x47,0xc1,
0x5e,0x79,0x31,0x85,0x2a,0xc1,0x27,0x70,0x76,0x56,0x75,0x35,0x23,0x9c,0x1f,0x42,
0x9c,0x7f,0x75,0xcf,0x74,0xc2,0x26,0x7d,0xeb,0x6a,0x3e,0x59,0x6c,0xf3,0x26,0x15,
0x6c,0x79,0x69,0x41,0x28,0x3b,0x8d,0x58,0x3f,0x17,0x1c,0x2f,0x6e,0x33,0x23,0xf7,
0x55,0x5e,0x1b,0x18,0x1f,0xfd,0xa3,0x05,0x07,0x21,0x0c,0xb1,0xf5,0x89,0xb2,0x3c,
0xd7,0x18,0x80,0xfd,0x44,0x37,0x0c,0xac,0xf4,0x33,0x75,0xb0,0xdb,0x7e,0x33,0x6f,
0x12,0xb3,0x09,0xbf,0xd4,0xf6,0x10,0xbb,0x8f,0x20,0xe1,0xa1,0x5e,0x25,0x3a,0x4f,
0xe5,0x11,0xa0,0x27,0x96,0x8d,0xf0,0xb1,0x05,0xa1,0xd7,0x3a,0xff,0x7c,0x7a,0x82,
0x6d,0x39,0xf6,0x40,0xdf,0xb8,0xf5,0x22,0x25,0x9e,0xd4,0x02,0x28,0x2e,0x2c,0x2e,
0x9d,0x3a,0x49,0x8f,0x51,0x72,0x5f,0xe4,0x14,0x1b,0x06,0xda,0x55,0x98,0xa4,0x2a,
0xc1,0xe0,0x49,0x4e,0x99,0x7d,0x56,0x6a,0x1a,0x39,0xb6,0x76,0xb9,0x6a,0x60,0x03,
0xa4,0xc5,0xdb,0x84,0xf2,0x46,0x58,0x4e,0xe6,0x5a,0xf7,0x0f,0xf2,0x16,0x02,0x78,
0x16,0x6d,0xa1,0x6d,0x91,0xc9,0xb8,0xf2,0xde,0xb0,0x27,0x51,0xa1,0x08,0x8a,0xd6,
0xbe,0x4e,0x80,0xef,0x96,0x6e,0xb7,0x3e,0x66,0xbc,0x87,0xca,0xd8,0x7c,0x77,0xc0,
0xb3,0x4a,0x21,0xba,0x1d,0xa0,0xba,0x6d,0x16,0xca,0x50,0x46,0xdc,0x4a,0xbd,0xa0,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0xc7,0x3a,0x78,0x20,0xf0,0xf5,0x3e,0x8b,0xbf,0xc3,0xb7,0xb7,0x1d,0x99,0x41,0x43,
0xcf,0x6e,0x98,0x64,0x2e,0x9e,0xa6,0xd8,0xdf,0x5d,0xcc,0xbc,0x43,0xdb,0x87,0x20,
},
.inst.nonce = (unsigned char []){
0x20,0xcc,0x98,0x34,0xb5,0x88,0xad,0xcb,0x1b,0xbd,0xe6,0x4f,0x0d,0x2a,0x34,0xcb,
},
.inst.v =(unsigned char []){
0x85,0x2d,0x1b,0x4f,0xdd,0x41,0xcd,0x7e,0x0f,0x59,0x7c,0x45,0xc8,0xe4,0xb4,0x01,
0xa5,0xfe,0xcf,0x92,0x29,0xb6,0x07,0x24,0x51,0xca,0x65,0xb5,0x28,0x98,0x82,0xc6,
0x86,0xe7,0x91,0x99,0x22,0xce,0x82,0xde,0x2f,0xaa,0xc8,0x3c,0xd4,0xc4,0xed,0xdf,
0xa2,0xcd,0xcf,0x62,0x44,0xa4,0xd2,0xac,0xdd,0x34,0xc0,0x23,0x21,0x36,0x40,0x9b,
0xb5,0x0e,0xa2,0x4d,0x0c,0x33,0xfc,0xfd,0x1a,0xaf,0x1c,0xc1,0x10,0xb5,0x35,0x3d,
0x32,0xe4,0xe6,0xdf,0x59,0xae,0x25,0xec,0x12,0x40,0x00,0xde,0x62,0xfc,0xfa,0x8b,
0xb4,0xcb,0x3f,0x3b,0x72,0xe2,0xda,0x20,0x66,0xef,0x00,0xcd,0x66,0xd9,0xe9,
},
.inst.c = (unsigned char []){
0xf7,0xb0,0xc9,0xcf,0x2c,0xcf,0x58,0xfd,0x8c,0x8b,0x69,0xda,0xa4,0xcf,0x24,0xa8,
0x74,0xc9,0x5b,0x57,0xa9,0xf5,0xbe,0x16,0xaa,0xa7,0x1e,0xc3,0x00,0x70,0xac,0x8f,
0x22,0x2f,0xe2,0x17,0x88,0xfe,0xc1,0x4b,0x8a,0x9a,0xd7,0xad,0x20,0x91,0x2c,0x05,
0xa6,0xf9,0x45,0x48,0x64,0x67,0x79,0xa1,0x6c,0x78,0x7b,0x13,0x5c,0xe8,0xd0,0x8c,
0x49,0xf7,0xe2,0x34,0xcb,0xd2,0xc7,0x73,0x35,0x71,0xf5,0xad,0x64,0x79,0xb5,0xfc,
0x50,0x40,0x34,0x96,0x58,0x1b,0x48,0x61,0xef,0x8e,0xc8,0x48,0xaf,0xfb,0xd2,0x07,
0x7a,0xb1,0x64,0xfc,0x6b,0xb2,0xdd,0x7b,0x00,0x8a,0x65,0x05,0x04,0xbf,0xd8,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0x12,0xdd,0x2a,0xca,0x88,0x79,0x04,0x6d,0x23,0x16,0x5c,0x60,0xf8,0xae,0xdc,0x20,
0x41,0x57,0x83,0xe1,0x56,0xd4,0x2a,0x94,0x34,0x68,0x26,0xaa,0xeb,0x02,0xea,0xcf,
},
.res.add = (unsigned char []){
0x9b,0x59,0xff,0x78,0xa3,0x4e,0xab,0xe0,0x06,0x0c,0x27,0x92,0xca,0x9b,0x49,0xe9,
0x78,0x1e,0x6b,0x80,0x2b,0xad,0xf7,0xdb,0xde,0x27,0xca,0xae,0xd3,0x34,0x37,0x06,
},
.res.v = (unsigned char []){
0x18,0x1a,0x30,0x23,0x52,0xd9,0xeb,0xf0,0xb6,0x69,0x73,0x0b,0x24,0x41,0xa9,0xf4,
0xc1,0x6a,0x4b,0x9d,0x25,0xeb,0xc8,0x4e,0xd0,0x1c,0x46,0x0d,0x29,0x3c,0xd3,0xe8,
0xb7,0xbf,0xf1,0xac,0xa3,0x2b,0x0e,0xa8,0xd2,0x81,0xdf,0x0e,0xf8,0xd1,0xae,0x09,
0xd4,0xcf,0x97,0x69,0x0c,0x94,0x4f,0x47,0x13,0xad,0xb9,0xed,0xe9,0x07,0x63,0xf3,
0xed,0x77,0x08,0x1c,0x37,0xc0,0xfc,0x60,0xf8,0xb6,0x0b,0x51,0x08,0xcf,0x62,0x76,
0xc8,0x0d,0xb1,0x4a,0x82,0xaa,0xef,0x1b,0xf8,0xda,0x03,0x78,0x14,0x45,0xcf,0xcc,
0x7c,0xdc,0x02,0xb1,0xc7,0xa2,0x74,0x08,0x74,0xdd,0x94,0x81,0x18,0xf7,0xef,
},
.res.c = (unsigned char []){
0x28,0xb6,0x38,0xd6,0x31,0xf0,0x54,0xeb,0xa5,0x62,0x32,0x0e,0x9d,0x15,0x1f,0x90,
0x58,0x63,0xdd,0x6c,0x04,0xd8,0xba,0x41,0x16,0x7b,0xcf,0x3b,0x02,0x36,0xd4,0xe5,
0xdd,0xe1,0xdc,0x7b,0xf6,0x90,0xe6,0x1b,0x4a,0x65,0x99,0x7b,0xd9,0xc6,0x7f,0xf9,
0x08,0xfe,0x7e,0x24,0x43,0xd0,0x1c,0x8e,0xac,0x15,0xb2,0xea,0x5c,0x80,0xba,0x89,
0xf0,0x9a,0xa9,0xb8,0xa8,0x1d,0x56,0x12,0x4b,0xb7,0x15,0x86,0x81,0x28,0x27,0xf4,
0x63,0xde,0x90,0x31,0x87,0x27,0x10,0x2d,0xbd,0x5e,0x59,0xca,0x5f,0x1a,0xf7,0x8a,
0xb7,0x38,0x44,0x69,0x5e,0xee,0x09,0x77,0xb7,0x54,0x85,0x4e,0x52,0x50,0x97,
},
.res.reseed_ctr = 1,

.gen1.add = (unsigned char []){
0xdc,0x74,0xa9,0xe4,0x80,0xa6,0xff,0x6f,0x6b,0xce,0x53,0xab,0x9c,0x7b,0xdd,0xe4,
0xb1,0x3d,0x70,0xfb,0x51,0x96,0xcd,0xd5,0xe3,0xa0,0x55,0x5c,0xcf,0x06,0xfe,0x91,
},
.gen1.v = (unsigned char []){
0x40,0xd0,0x68,0xf9,0x84,0xca,0x40,0xdc,0x5b,0xcb,0xa5,0x19,0xc1,0x56,0xc9,0x85,
0x19,0xce,0x29,0x09,0x2a,0xc4,0x82,0x8f,0xe6,0x98,0x15,0x48,0x2b,0x73,0xa8,0xce,
0x95,0xa1,0xce,0x28,0x99,0xbb,0xf4,0xc4,0x1c,0xe7,0x78,0x8a,0xd2,0x98,0x2e,0x3c,
0xea,0x32,0x66,0xf4,0xca,0xdc,0x50,0xae,0x52,0x8d,0xc6,0x1a,0xa7,0xc5,0x21,0x48,
0x98,0x69,0xe3,0xef,0xc6,0xc8,0x2c,0xce,0xfb,0xba,0xb4,0x56,0x73,0xe0,0xf5,0x9d,
0x56,0x54,0xcf,0x91,0x0f,0xa1,0x46,0xd9,0x84,0xa4,0x2c,0x5f,0x17,0xfb,0x60,0x34,
0x0c,0x86,0xd0,0xd0,0x7c,0x7e,0x2f,0x2e,0x6d,0xf3,0xcf,0xfd,0x72,0x2a,0x0e,
},
.gen1.c = (unsigned char []){
0x28,0xb6,0x38,0xd6,0x31,0xf0,0x54,0xeb,0xa5,0x62,0x32,0x0e,0x9d,0x15,0x1f,0x90,
0x58,0x63,0xdd,0x6c,0x04,0xd8,0xba,0x41,0x16,0x7b,0xcf,0x3b,0x02,0x36,0xd4,0xe5,
0xdd,0xe1,0xdc,0x7b,0xf6,0x90,0xe6,0x1b,0x4a,0x65,0x99,0x7b,0xd9,0xc6,0x7f,0xf9,
0x08,0xfe,0x7e,0x24,0x43,0xd0,0x1c,0x8e,0xac,0x15,0xb2,0xea,0x5c,0x80,0xba,0x89,
0xf0,0x9a,0xa9,0xb8,0xa8,0x1d,0x56,0x12,0x4b,0xb7,0x15,0x86,0x81,0x28,0x27,0xf4,
0x63,0xde,0x90,0x31,0x87,0x27,0x10,0x2d,0xbd,0x5e,0x59,0xca,0x5f,0x1a,0xf7,0x8a,
0xb7,0x38,0x44,0x69,0x5e,0xee,0x09,0x77,0xb7,0x54,0x85,0x4e,0x52,0x50,0x97,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char []){
0x8f,0x3f,0x22,0x90,0x11,0x20,0x9b,0x2f,0x39,0x90,0x96,0xaf,0xb0,0x54,0xbc,0xcc,
0xa6,0xbc,0x46,0xaa,0xee,0x98,0x84,0x58,0x38,0xfb,0x1f,0xb7,0x8b,0x66,0xf3,0xbd,
},
.gen2.v = (unsigned char []){
0x69,0x86,0xa1,0xcf,0xb6,0xba,0x95,0xc8,0x01,0x2d,0xd7,0x28,0x5e,0x6b,0xe9,0x15,
0x72,0x32,0x06,0x75,0x2f,0x9d,0x3c,0xd0,0xfd,0x13,0xe4,0x83,0x2d,0xaa,0x7d,0xb4,
0x73,0x83,0xaa,0xa4,0x90,0x4c,0xda,0xdf,0x67,0x4d,0x12,0x06,0xac,0x5e,0xaf,0xa9,
0x9d,0xe1,0x30,0x4f,0xc0,0xb6,0xa1,0xb5,0xe3,0x2e,0x34,0xa7,0xf4,0x14,0x1e,0x89,
0x35,0x38,0x78,0xc0,0xd3,0xf6,0xa0,0xba,0x5b,0x9e,0xd4,0x52,0xd6,0x12,0x60,0xde,
0x9e,0x5a,0xcb,0xf8,0x13,0x44,0x85,0xb3,0xb9,0xe9,0x90,0xf5,0x9f,0x34,0xd4,0xd4,
0x33,0x07,0xe4,0x0a,0xd0,0xd0,0xa5,0x05,0xef,0xdb,0x24,0xb7,0x2f,0x80,0x7b,
},
.gen2.c = (unsigned char []){
0x28,0xb6,0x38,0xd6,0x31,0xf0,0x54,0xeb,0xa5,0x62,0x32,0x0e,0x9d,0x15,0x1f,0x90,
0x58,0x63,0xdd,0x6c,0x04,0xd8,0xba,0x41,0x16,0x7b,0xcf,0x3b,0x02,0x36,0xd4,0xe5,
0xdd,0xe1,0xdc,0x7b,0xf6,0x90,0xe6,0x1b,0x4a,0x65,0x99,0x7b,0xd9,0xc6,0x7f,0xf9,
0x08,0xfe,0x7e,0x24,0x43,0xd0,0x1c,0x8e,0xac,0x15,0xb2,0xea,0x5c,0x80,0xba,0x89,
0xf0,0x9a,0xa9,0xb8,0xa8,0x1d,0x56,0x12,0x4b,0xb7,0x15,0x86,0x81,0x28,0x27,0xf4,
0x63,0xde,0x90,0x31,0x87,0x27,0x10,0x2d,0xbd,0x5e,0x59,0xca,0x5f,0x1a,0xf7,0x8a,
0xb7,0x38,0x44,0x69,0x5e,0xee,0x09,0x77,0xb7,0x54,0x85,0x4e,0x52,0x50,0x97,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0xe6,0xc9,0x64,0x42,0x58,0x28,0x11,0xec,0x90,0xe5,0x87,0x52,0x5f,0x36,0xc5,0x55,
0xe2,0xfd,0x63,0x61,0xa0,0xc5,0xb0,0x28,0x49,0x17,0xa4,0xfa,0x6f,0x6e,0x8a,0xce,
0x83,0xf1,0x1a,0x1f,0xb2,0x6c,0xea,0x66,0x92,0xb2,0x25,0xae,0x7c,0x5b,0xe2,0x86,
0xdd,0x27,0x47,0x1f,0x32,0x3d,0x7a,0x2e,0x44,0x31,0x72,0x2b,0xb3,0x37,0xb1,0xba,
0x0e,0x64,0x8e,0xa2,0xe9,0xf0,0x91,0x8b,0x50,0xe9,0x11,0x1f,0x23,0x77,0x63,0x6b,
0xa6,0x9b,0x0e,0x1c,0xb5,0x29,0x50,0x78,0xd7,0x6c,0x54,0x9c,0x86,0x56,0x94,0x0e,
0xb1,0x5c,0xa5,0xad,0xed,0x7a,0xdc,0x46,0xe6,0xfa,0x4b,0x86,0x94,0x8f,0x21,0x2f,
0xea,0x3f,0x3b,0xef,0xde,0xec,0xe8,0xb2,0x0e,0x42,0x0c,0xa8,0x4c,0x76,0x01,0x96,
0xdd,0xf0,0xb0,0x74,0xdf,0x0a,0x9f,0x09,0x7a,0x5d,0xb8,0xf6,0x12,0x58,0x00,0xf5,
0xfe,0x74,0x6a,0x62,0xdf,0x12,0x08,0x04,0x2f,0x12,0x55,0xb5,0x24,0x46,0x5a,0x17,
0xef,0xcf,0x6a,0x53,0x76,0x12,0x96,0x84,0x30,0xe2,0xad,0xcf,0xf3,0x0f,0x74,0x07,
0xa5,0x1e,0xd7,0x30,0x53,0x34,0x38,0x4e,0x51,0x2e,0x00,0x36,0x42,0xcc,0xa1,0x75,
0x63,0x68,0x19,0xf0,0x21,0xc7,0x6a,0x2f,0x44,0xe8,0x9e,0x6f,0xe3,0x9c,0xf1,0x64,
0x47,0x79,0x10,0x37,0x9c,0xd3,0x14,0xf7,0x35,0xc3,0x57,0xf9,0x37,0x9d,0xe2,0x24,
0x95,0x27,0x6b,0x40,0x1c,0x98,0xff,0xb0,0x9a,0x6d,0xc0,0x3e,0x48,0x4b,0x35,0x5a,
0x94,0x64,0x51,0x14,0x01,0xee,0xaa,0x05,0xb4,0x55,0x6e,0x73,0xb5,0x52,0x27,0xf8,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x12,0xc2,0x20,0x31,0xd0,0x3a,0x85,0x0f,0x76,0x96,0xc9,0x27,0xd4,0xdc,0x8d,0xad,
0x4c,0x3d,0xd7,0x17,0xe6,0x0e,0xe5,0x68,0x1b,0x05,0xdb,0x64,0x26,0xa0,0xe1,0x67,
},
.inst.nonce = (unsigned char []){
0xcd,0x32,0xe3,0xeb,0x3f,0x83,0x34,0xd9,0x69,0x8d,0xec,0x62,0x7b,0x2e,0xaa,0xbe,
},
.inst.pers = (unsigned char []){
0x75,0x63,0x63,0xf6,0x81,0x78,0xda,0xc0,0x9a,0x5c,0x8d,0x64,0xef,0xfb,0xab,0x23,
0x87,0x3a,0x3b,0xc2,0xdf,0xba,0xc3,0x9b,0x6d,0x47,0xeb,0xf9,0x29,0xad,0x98,0x54,
},
.inst.v =(unsigned char []){
0x29,0x5e,0x0b,0x18,0x92,0x1d,0x33,0xaa,0x64,0xe3,0xde,0xec,0x80,0x04,0x4b,0x41,
0xac,0x4e,0x49,0x90,0x04,0xc7,0x7b,0x7d,0x1a,0xc0,0x21,0xdd,0x1c,0x4e,0x2e,0x6a,
0x85,0x2d,0x6a,0xd6,0x27,0x6c,0xe2,0xa9,0x01,0x5f,0xa3,0x1e,0x22,0x34,0x96,0x76,
0x14,0x3d,0xa2,0x02,0xb7,0x7c,0xc0,0x08,0x26,0xea,0x17,0xec,0x67,0xc0,0xdb,0x12,
0x75,0xa2,0xd0,0x7c,0x0e,0xae,0x8a,0x87,0xdf,0xbe,0x0f,0xd5,0x34,0x7a,0x2b,0x38,
0x36,0x40,0x29,0xf4,0x38,0x8e,0x6e,0x32,0xe8,0x53,0xad,0xec,0x3a,0xe4,0x79,0xc4,
0x07,0x75,0xf7,0xab,0x17,0x08,0x09,0x90,0x57,0x2b,0xac,0x29,0x47,0x7a,0x31,
},
.inst.c = (unsigned char []){
0x24,0x70,0x96,0x11,0x1b,0xa9,0x0f,0x4f,0x56,0x11,0xe2,0x65,0x84,0xa9,0xf1,0x83,
0xc9,0x60,0xda,0xe8,0x5c,0x57,0x1a,0xa5,0xae,0xfc,0x13,0xe3,0xdd,0x04,0x96,0x8c,
0x53,0xa3,0xd0,0xb9,0xdd,0x69,0x9a,0xa6,0xf8,0x33,0x0c,0x5d,0xa4,0x44,0x7e,0x30,
0x5f,0xf8,0xa3,0xdc,0x21,0xbe,0xf0,0x27,0x07,0xc0,0x64,0xa8,0xeb,0xf4,0xa7,0x34,
0x0c,0xab,0x11,0x33,0x35,0x58,0x97,0xd1,0xea,0xc2,0x0f,0xf7,0xb8,0x10,0x7f,0xce,
0x28,0xf8,0xef,0x64,0xbf,0x6b,0x39,0x0d,0x58,0x9d,0xec,0x19,0xe7,0x47,0x65,0x03,
0x73,0x44,0x88,0x72,0x39,0xf3,0x0f,0x4b,0x2b,0x04,0xc8,0x11,0xe6,0x00,0x69,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0x17,0xed,0x31,0xbd,0xa6,0x4b,0x08,0xec,0xe5,0x0e,0xdf,0x5b,0x91,0xf6,0xa0,0x86,
0x2e,0x56,0x90,0x18,0x17,0x34,0xa0,0x13,0x4e,0x05,0xe3,0x66,0x64,0x0b,0x7e,0x85,
},
.res.v = (unsigned char []){
0xba,0xa7,0xae,0xbc,0x24,0x31,0x32,0x82,0x49,0x0e,0x40,0xd8,0x90,0x3d,0x2b,0x4c,
0x23,0xf3,0x0a,0xfc,0x59,0x8d,0x36,0x7f,0xa8,0xb5,0x70,0xc3,0xf9,0xa7,0x92,0x39,
0x9c,0xeb,0xff,0x96,0xb3,0xf4,0x58,0xd3,0xa8,0xd2,0xd8,0xb9,0xb1,0x40,0xdb,0x40,
0xd9,0x8d,0xcf,0x5a,0x44,0x8e,0xce,0xb7,0x6d,0x25,0x19,0xd2,0xbf,0xd5,0xef,0x0d,
0x1c,0x80,0x5f,0x50,0xbc,0xde,0xdb,0x38,0x24,0x8d,0x2e,0xd7,0xcd,0x66,0xab,0x07,
0xf9,0x7e,0x13,0xd1,0xd2,0x69,0x10,0xb8,0x5f,0x7e,0xec,0x8f,0x1b,0x26,0xdc,0x1f,
0x3a,0x23,0x02,0x91,0xa7,0x32,0x25,0xe3,0xf2,0xcf,0xb7,0xf4,0xde,0x76,0x24,
},
.res.c = (unsigned char []){
0xd2,0x96,0xd0,0x95,0x7e,0x86,0x06,0xb5,0xe2,0x7b,0x74,0xac,0x72,0x98,0x08,0x60,
0x4c,0x48,0x35,0x03,0x52,0x53,0x4c,0x3d,0x8b,0x1a,0xc0,0x55,0x1c,0x9e,0xfc,0x80,
0x8b,0x55,0x7d,0x8f,0x5c,0x7a,0x1b,0x62,0x7b,0x86,0x85,0xb5,0x87,0xa5,0x12,0x1e,
0x73,0xd5,0xbe,0x13,0x78,0xe8,0xea,0x03,0x5f,0xab,0x8a,0xab,0x96,0x2a,0xf8,0x6d,
0xc4,0x1f,0xda,0x58,0x8d,0xcc,0xde,0xfc,0xfa,0xaa,0x78,0xde,0x18,0xfc,0x7a,0x50,
0x63,0xaf,0x35,0xfd,0xb2,0x4e,0xf6,0x04,0xda,0x15,0x3e,0x64,0x87,0x8a,0xaa,0xf6,
0x37,0x9a,0x95,0xe7,0x63,0x57,0xe6,0xd5,0xfd,0xd4,0x7e,0x07,0x3a,0x0f,0xfe,
},
.res.reseed_ctr = 1,

.gen1.v = (unsigned char []){
0x8d,0x3e,0x7f,0x51,0xa2,0xb7,0x39,0x38,0x2b,0x89,0xb5,0x85,0x02,0xd5,0x33,0xac,
0x70,0x3b,0x3f,0xff,0xab,0xe0,0x82,0xbd,0x33,0xd0,0x31,0x19,0x16,0x46,0x8e,0xba,
0x28,0x41,0x7d,0x26,0x10,0x6e,0x74,0x36,0x24,0x59,0x5e,0x6f,0x38,0xe5,0xed,0x9a,
0xac,0x0e,0x71,0xaa,0xb9,0x38,0xce,0xf1,0x40,0x29,0x43,0xa4,0x47,0x38,0xdd,0xf5,
0x7b,0x90,0x74,0x21,0x41,0xda,0x08,0x8a,0x32,0x22,0x5c,0x24,0x96,0x82,0x4f,0x7b,
0xca,0x1d,0xf8,0x3b,0xaa,0x8d,0x04,0xfc,0x7e,0xa7,0x63,0x55,0x22,0x27,0xed,0xec,
0xd9,0x4c,0x71,0x3d,0x64,0x92,0x6b,0x13,0x7b,0x1c,0xa2,0x01,0x21,0x2f,0x2b,
},
.gen1.c = (unsigned char []){
0xd2,0x96,0xd0,0x95,0x7e,0x86,0x06,0xb5,0xe2,0x7b,0x74,0xac,0x72,0x98,0x08,0x60,
0x4c,0x48,0x35,0x03,0x52,0x53,0x4c,0x3d,0x8b,0x1a,0xc0,0x55,0x1c,0x9e,0xfc,0x80,
0x8b,0x55,0x7d,0x8f,0x5c,0x7a,0x1b,0x62,0x7b,0x86,0x85,0xb5,0x87,0xa5,0x12,0x1e,
0x73,0xd5,0xbe,0x13,0x78,0xe8,0xea,0x03,0x5f,0xab,0x8a,0xab,0x96,0x2a,0xf8,0x6d,
0xc4,0x1f,0xda,0x58,0x8d,0xcc,0xde,0xfc,0xfa,0xaa,0x78,0xde,0x18,0xfc,0x7a,0x50,
0x63,0xaf,0x35,0xfd,0xb2,0x4e,0xf6,0x04,0xda,0x15,0x3e,0x64,0x87,0x8a,0xaa,0xf6,
0x37,0x9a,0x95,0xe7,0x63,0x57,0xe6,0xd5,0xfd,0xd4,0x7e,0x07,0x3a,0x0f,0xfe,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char []){
0x5f,0xd5,0x4f,0xe7,0x21,0x3d,0x3f,0xee,0x0e,0x05,0x2a,0x31,0x75,0x6d,0x3c,0x0c,
0xbc,0x83,0x75,0x02,0xfe,0x33,0xce,0xfa,0xbe,0xea,0xf1,0x6e,0x32,0xe5,0x8b,0x3a,
0xb3,0x96,0xfa,0xb5,0x6c,0xe8,0x8f,0x98,0x9f,0xdf,0xe4,0x24,0xc0,0x8b,0x00,0xb6,
0x46,0x6e,0x50,0xc6,0x6f,0x3f,0xa1,0x36,0x6f,0x7e,0xa2,0x0d,0x6d,0x4b,0xec,0xf1,
0x67,0x77,0xff,0x54,0xd6,0xea,0xc0,0xcb,0x8b,0x86,0xb0,0x2f,0x2b,0x88,0x5c,0xc7,
0x80,0x75,0x0a,0x8c,0xef,0x52,0x5f,0x0f,0xe1,0x25,0x8d,0xf1,0xb4,0xc2,0x53,0x45,
0x29,0xe6,0x8b,0xe9,0x3e,0xbb,0x30,0xb1,0x61,0x4d,0xb7,0x97,0x80,0xd9,0x9e,
},
.gen2.c = (unsigned char []){
0xd2,0x96,0xd0,0x95,0x7e,0x86,0x06,0xb5,0xe2,0x7b,0x74,0xac,0x72,0x98,0x08,0x60,
0x4c,0x48,0x35,0x03,0x52,0x53,0x4c,0x3d,0x8b,0x1a,0xc0,0x55,0x1c,0x9e,0xfc,0x80,
0x8b,0x55,0x7d,0x8f,0x5c,0x7a,0x1b,0x62,0x7b,0x86,0x85,0xb5,0x87,0xa5,0x12,0x1e,
0x73,0xd5,0xbe,0x13,0x78,0xe8,0xea,0x03,0x5f,0xab,0x8a,0xab,0x96,0x2a,0xf8,0x6d,
0xc4,0x1f,0xda,0x58,0x8d,0xcc,0xde,0xfc,0xfa,0xaa,0x78,0xde,0x18,0xfc,0x7a,0x50,
0x63,0xaf,0x35,0xfd,0xb2,0x4e,0xf6,0x04,0xda,0x15,0x3e,0x64,0x87,0x8a,0xaa,0xf6,
0x37,0x9a,0x95,0xe7,0x63,0x57,0xe6,0xd5,0xfd,0xd4,0x7e,0x07,0x3a,0x0f,0xfe,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0xbc,0x5d,0x8d,0x11,0xbe,0x22,0x67,0x9e,0x33,0xf4,0x03,0x74,0x17,0x57,0x16,0xe6,
0x7c,0x8b,0x5f,0xe6,0x81,0x9a,0xc5,0x3a,0x9b,0x20,0x8d,0xc0,0x58,0x17,0x04,0x31,
0xff,0xc2,0x9e,0xbd,0x1a,0x81,0x51,0xca,0xa3,0xe9,0x15,0x6d,0x4c,0x7e,0x89,0xe3,
0x9c,0x12,0x4f,0x71,0x94,0x09,0x51,0x02,0xe8,0x69,0x31,0x06,0x74,0x47,0x1a,0x7f,
0x1d,0xad,0x4e,0x58,0xf4,0x78,0x6e,0x96,0xb7,0xaa,0x1a,0xd7,0xa5,0x11,0x59,0x23,
0xba,0x01,0xd4,0xe7,0xa6,0x0a,0x8f,0x11,0xee,0x9c,0x47,0x26,0x6c,0x0f,0x1a,0xe4,
0x34,0x16,0x8b,0x7b,0x1f,0xb6,0x1e,0xc0,0xfc,0x29,0x2c,0x6c,0x2d,0x3a,0x87,0x78,
0xdc,0x7b,0x88,0x16,0x42,0xfb,0x8a,0x3e,0x9f,0xa5,0xff,0x37,0x20,0xf7,0x00,0xff,
0x89,0x00,0x1d,0x21,0xe9,0x7c,0x61,0xc2,0x46,0xdd,0x4f,0x87,0xbd,0x8a,0x64,0xfb,
0xcb,0x92,0x01,0x4d,0x52,0xf6,0xe6,0x41,0x83,0xbd,0xca,0x84,0xec,0x25,0xed,0x35,
0x24,0xb9,0xab,0xc8,0x6d,0xf2,0xbc,0x4d,0xff,0x2b,0x76,0x29,0x98,0x55,0xde,0x61,
0xb7,0xda,0x7e,0xde,0xc0,0x27,0x89,0x3a,0xb4,0xed,0xd0,0xd6,0xea,0x63,0x48,0xe7,
0x61,0x0e,0x6d,0x94,0x0a,0xf4,0x22,0x54,0x63,0x88,0x68,0x59,0xea,0x4f,0x5c,0x53,
0xfe,0xa2,0xc3,0x98,0xac,0x2f,0xe7,0x4a,0x9b,0x31,0x8b,0x11,0x5d,0xd4,0x6b,0xbe,
0xc6,0x88,0x4f,0x07,0x78,0x35,0xaa,0x95,0x15,0x0e,0xf0,0xb3,0xed,0x34,0xd6,0xd5,
0xb1,0x44,0xff,0x1c,0x1e,0x23,0x88,0x48,0x3d,0x9b,0x5f,0xca,0x8a,0x2c,0x5b,0xf3,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x2c,0x48,0x2c,0xd3,0x92,0xd9,0xb8,0x67,0x7f,0xf3,0x19,0xa7,0xad,0x53,0x9c,0xbd,
0x8a,0x24,0xeb,0xdc,0x2b,0x8d,0xbc,0xeb,0xbc,0x1d,0x1d,0x0d,0x0b,0xcb,0x69,0x8a,
},
.inst.nonce = (unsigned char []){
0x27,0xcb,0x9f,0xff,0xcc,0xdd,0x7a,0x3b,0x28,0x56,0xfe,0xdd,0x68,0x17,0xf1,0xaa,
},
.inst.pers = (unsigned char []){
0xbe,0x95,0x71,0x17,0x54,0xbe,0xb1,0xb2,0x5a,0x46,0xd5,0x04,0xd1,0x9b,0xbf,0x44,
0x23,0x2c,0x39,0x53,0x32,0x33,0xf6,0xea,0x84,0x14,0x0c,0xa1,0xd4,0x70,0xfb,0x10,
},
.inst.v = (unsigned char []){
0x36,0xf3,0xe7,0x49,0x11,0x77,0xc5,0xb9,0x1e,0xf4,0xbd,0x81,0x38,0x1b,0xec,0xc9,
0xfe,0x82,0x64,0x72,0x17,0x75,0x3b,0xf1,0xdf,0xc5,0x2d,0x3d,0x8c,0x56,0x38,0x30,
0xf3,0xa5,0x98,0xd7,0xe3,0x0d,0x6d,0xd8,0xf6,0x3e,0x37,0x07,0xb2,0x59,0xa9,0x8d,
0xb5,0x87,0x76,0x79,0x4d,0x32,0xaf,0x6a,0x55,0x5b,0x57,0x72,0x11,0x4d,0xa6,0x22,
0xf3,0xd2,0xa1,0x9d,0xf2,0x49,0xfc,0x88,0xbf,0x8f,0xaf,0x67,0x95,0xed,0x72,0xc5,
0x20,0x26,0xfe,0x22,0x84,0xd6,0x0d,0x91,0x69,0x43,0x12,0xd9,0x89,0xf3,0x16,0xd9,
0xc6,0xe8,0x1b,0xf1,0x91,0x6a,0x00,0x8d,0xcf,0x5f,0x6b,0xe4,0x9b,0x87,0xe3,
},
.inst.c = (unsigned char []){
0xb4,0x92,0x33,0x33,0x00,0x5a,0xb7,0x14,0x0e,0xc1,0x08,0xc1,0x20,0xa9,0x77,0x17,
0x47,0x23,0x57,0xc2,0x56,0x08,0x85,0x33,0xae,0xed,0x55,0x84,0x41,0x17,0xda,0x6c,
0x28,0xd1,0x11,0xd7,0x05,0x1c,0x5f,0x6b,0xda,0xee,0xe0,0x73,0x10,0x15,0x10,0xdb,
0x95,0x14,0xbb,0xb8,0xb0,0xb4,0x36,0x23,0xc2,0xb3,0x7f,0x2f,0x1d,0x3f,0x20,0xa9,
0xc9,0x2e,0xdb,0x2e,0x7b,0xc1,0x86,0x9e,0x43,0xc9,0x14,0xc1,0x0c,0x32,0xc1,0xbf,
0x81,0xbd,0xff,0x58,0x4c,0x9b,0x5c,0x47,0x80,0xfb,0xba,0xfa,0xa0,0xca,0x01,0x0a,
0xfb,0x7e,0x8b,0x05,0x2b,0xe8,0x38,0xf5,0x8c,0x04,0x3e,0x0c,0x2b,0x6a,0xd2,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0x4f,0x39,0xa2,0xb6,0x30,0x9a,0xf3,0x2a,0x1c,0x72,0x1b,0xd2,0xa3,0x1a,0x97,0x81,
0x48,0x07,0x64,0x43,0x03,0xb1,0xd1,0x32,0x23,0x93,0x64,0xa3,0x6d,0x26,0x6f,0x10,
},
.res.add = (unsigned char []){
0x94,0x7d,0x7b,0x37,0x2c,0x90,0x9e,0xae,0x57,0xba,0xe6,0x4d,0x7a,0x9b,0xbf,0x75,
0x23,0x4f,0x5c,0xb0,0x5c,0x09,0x39,0x27,0x03,0x7f,0x63,0x2d,0xd8,0x56,0x76,0x93,
},
.res.v = (unsigned char []){
0xe8,0x80,0xaa,0xd5,0x78,0x88,0xbc,0x81,0xbe,0xb6,0xe1,0x6b,0xdc,0xe7,0x5f,0x6d,
0xbd,0xd0,0xad,0xab,0x7e,0xb0,0x4a,0xeb,0x62,0xc4,0xd6,0xa4,0x27,0x4c,0x8e,0xaa,
0xcb,0x9c,0x3a,0x84,0x22,0xa4,0xdf,0x14,0xe6,0x57,0x0a,0x42,0x7b,0x01,0x35,0x61,
0xfd,0xd6,0x04,0x60,0x29,0x84,0xef,0xa4,0xe3,0xc0,0x17,0xb7,0xa5,0x6e,0x13,0x31,
0xe8,0xca,0xe2,0x54,0xe4,0x69,0xd2,0x54,0x38,0x70,0x99,0x94,0xbb,0x8b,0x37,0x07,
0xa5,0x2f,0xed,0xa8,0x6f,0x1b,0x0c,0xbf,0x3b,0x3c,0xa3,0x4a,0x97,0x35,0x7c,0x10,
0x98,0x6d,0x96,0x71,0x4d,0x2e,0xa2,0x66,0x0b,0xc6,0x71,0x30,0xeb,0xf8,0xe6,
},
.res.c = (unsigned char []){
0xec,0xed,0x12,0x31,0x3f,0x5f,0x03,0xdd,0xd2,0xdc,0x63,0xf0,0x99,0x8e,0xf7,0x23,
0x28,0x39,0xfe,0xd2,0xdd,0x04,0xd4,0x0f,0x43,0xf5,0x1c,0xb1,0x89,0xc5,0x4a,0xa4,
0xa0,0x15,0x24,0x62,0xbb,0x45,0x5e,0x89,0x7f,0xe8,0x15,0x0f,0x32,0x31,0x7d,0x6e,
0xba,0x74,0x84,0xd3,0xb2,0x07,0x89,0xf5,0xd7,0xa7,0x70,0x96,0x80,0xa6,0x90,0xa4,
0x31,0xdf,0x7d,0xe9,0x18,0xc0,0xa5,0x0e,0x8f,0x3f,0x3f,0x1a,0x4c,0xe0,0xdf,0xfa,
0x1e,0x74,0xba,0x70,0x85,0x69,0xde,0x35,0xa0,0xcd,0x52,0xed,0x31,0xd2,0x94,0x29,
0x87,0xd8,0xef,0x52,0xce,0x26,0xa8,0xb2,0x9c,0x0b,0x94,0xa6,0x5a,0xb0,0x65,
},
.res.reseed_ctr = 1,

.gen1.add = (unsigned char []){
0x65,0x1f,0x03,0xa8,0x02,0x34,0x34,0xb2,0xaf,0x28,0xa1,0xfa,0xd9,0x4b,0x31,0x91,
0x01,0x52,0xd4,0x9a,0x1b,0xbc,0xea,0xe3,0x2b,0xcc,0xd7,0x4d,0xde,0x60,0xd9,0x2f,
},
.gen1.v = (unsigned char []){
0xd5,0x6d,0xbd,0x06,0xb7,0xe7,0xc0,0x5f,0x91,0x93,0x45,0x5c,0x76,0x76,0x56,0x90,
0xe6,0x0a,0xac,0x7e,0x5b,0xb5,0x1e,0xfa,0xa6,0xb9,0xf3,0x55,0xb1,0x11,0xd9,0x4f,
0x6b,0xb1,0x5e,0xe6,0xdd,0xea,0x3d,0x9e,0x66,0x3f,0x1f,0x51,0xad,0x32,0xb3,0xb1,
0x2f,0x75,0xf5,0x94,0xe1,0x94,0xa0,0xeb,0xb8,0xc6,0xbd,0xb9,0x9a,0x42,0x88,0x31,
0x91,0x95,0xdb,0x4c,0xa3,0xe7,0x50,0x5f,0xa3,0x3a,0xb4,0x5f,0xa1,0x17,0x8f,0x51,
0xb9,0x66,0x84,0x0b,0x40,0x4e,0xfe,0x35,0xe0,0xbb,0xb3,0x87,0xe4,0x00,0xe8,0xd7,
0x5b,0x43,0x09,0xef,0x6c,0x3d,0xe2,0x8f,0x80,0x48,0x1e,0x10,0x1c,0xbf,0x3f,
},
.gen1.c = (unsigned char []){
0xec,0xed,0x12,0x31,0x3f,0x5f,0x03,0xdd,0xd2,0xdc,0x63,0xf0,0x99,0x8e,0xf7,0x23,
0x28,0x39,0xfe,0xd2,0xdd,0x04,0xd4,0x0f,0x43,0xf5,0x1c,0xb1,0x89,0xc5,0x4a,0xa4,
0xa0,0x15,0x24,0x62,0xbb,0x45,0x5e,0x89,0x7f,0xe8,0x15,0x0f,0x32,0x31,0x7d,0x6e,
0xba,0x74,0x84,0xd3,0xb2,0x07,0x89,0xf5,0xd7,0xa7,0x70,0x96,0x80,0xa6,0x90,0xa4,
0x31,0xdf,0x7d,0xe9,0x18,0xc0,0xa5,0x0e,0x8f,0x3f,0x3f,0x1a,0x4c,0xe0,0xdf,0xfa,
0x1e,0x74,0xba,0x70,0x85,0x69,0xde,0x35,0xa0,0xcd,0x52,0xed,0x31,0xd2,0x94,0x29,
0x87,0xd8,0xef,0x52,0xce,0x26,0xa8,0xb2,0x9c,0x0b,0x94,0xa6,0x5a,0xb0,0x65,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char []){
0x93,0x4a,0x97,0xd6,0xee,0x46,0x85,0x25,0x0a,0xc5,0xe5,0x11,0x59,0xe5,0xbc,0xbc,
0x48,0xa6,0xf9,0x2c,0x7c,0xe7,0xbf,0xd1,0xc6,0x1d,0x0e,0x7b,0x0a,0x0e,0x7f,0x61,
},
.gen2.v = (unsigned char []){
0xc2,0x5a,0xcf,0x37,0xf7,0x46,0xc4,0x3d,0x64,0x6f,0xa9,0x4d,0x10,0x05,0x4d,0xb4,
0x0e,0x44,0xab,0x51,0x38,0xb9,0xf3,0x09,0xea,0xaf,0x10,0x07,0x3a,0xd7,0x23,0xf4,
0x0b,0xc6,0x83,0x49,0x99,0x2f,0x9c,0x27,0xe6,0x27,0x34,0x60,0xdf,0x64,0x31,0xa9,
0xff,0xd8,0x0c,0x15,0xed,0x67,0x40,0xd8,0x72,0xc4,0x64,0x68,0x24,0xee,0x9e,0x36,
0x2d,0x7a,0xce,0x7c,0xf6,0x5d,0x37,0x31,0x8c,0x2d,0x48,0x30,0x19,0x62,0x78,0x04,
0x4a,0xd9,0x8b,0x6e,0x0d,0x87,0x99,0xa0,0x55,0x37,0x4d,0x71,0x19,0x34,0xe5,0xbd,
0xa4,0x51,0xc0,0x82,0x4d,0x51,0xf3,0x83,0x8f,0xf1,0xf7,0x99,0x0e,0xef,0x1a,
},
.gen2.c = (unsigned char []){
0xec,0xed,0x12,0x31,0x3f,0x5f,0x03,0xdd,0xd2,0xdc,0x63,0xf0,0x99,0x8e,0xf7,0x23,
0x28,0x39,0xfe,0xd2,0xdd,0x04,0xd4,0x0f,0x43,0xf5,0x1c,0xb1,0x89,0xc5,0x4a,0xa4,
0xa0,0x15,0x24,0x62,0xbb,0x45,0x5e,0x89,0x7f,0xe8,0x15,0x0f,0x32,0x31,0x7d,0x6e,
0xba,0x74,0x84,0xd3,0xb2,0x07,0x89,0xf5,0xd7,0xa7,0x70,0x96,0x80,0xa6,0x90,0xa4,
0x31,0xdf,0x7d,0xe9,0x18,0xc0,0xa5,0x0e,0x8f,0x3f,0x3f,0x1a,0x4c,0xe0,0xdf,0xfa,
0x1e,0x74,0xba,0x70,0x85,0x69,0xde,0x35,0xa0,0xcd,0x52,0xed,0x31,0xd2,0x94,0x29,
0x87,0xd8,0xef,0x52,0xce,0x26,0xa8,0xb2,0x9c,0x0b,0x94,0xa6,0x5a,0xb0,0x65,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0x77,0xce,0x58,0xaf,0x82,0x2a,0x11,0xf1,0xd6,0xdf,0xb2,0xbe,0x24,0x6a,0xa6,0x54,
0x9e,0x4c,0x08,0x9e,0x15,0x32,0xae,0xba,0x3e,0x6f,0x13,0xd1,0x96,0x35,0x93,0x0f,
0x70,0x24,0xf9,0x43,0x99,0xba,0x6c,0x84,0xdf,0xa3,0xcd,0x9c,0x78,0x06,0xd7,0xb7,
0x69,0xc0,0x96,0xd3,0x34,0x15,0xa5,0x6b,0xe1,0x09,0x66,0x18,0xd5,0xbd,0x80,0xd7,
0xd6,0x49,0x95,0x0e,0xbc,0xfe,0x74,0x18,0xe9,0xf2,0x33,0xac,0xdb,0xb7,0xc2,0xed,
0x36,0x81,0xf5,0xdc,0xd0,0x7e,0x74,0x1e,0xec,0xd2,0xa4,0x67,0x5a,0xc8,0xdb,0x7b,
0xec,0xd2,0xeb,0x77,0xf1,0x64,0xcf,0x5c,0x76,0x72,0xa6,0xe9,0xe6,0x19,0x25,0x4c,
0x5f,0x23,0x1c,0x85,0x34,0x4c,0x34,0x61,0xfa,0xdb,0x5f,0x19,0x17,0x06,0xda,0xe9,
0xfc,0x2c,0xb5,0xd6,0x1c,0x8b,0x36,0xb3,0xe1,0xaf,0x3e,0x82,0xce,0xf8,0xe2,0xa9,
0xca,0x76,0xfa,0xfb,0xaf,0x4d,0x4b,0xf9,0xe0,0xf9,0xb0,0xcd,0x59,0x07,0xd4,0x06,
0x9e,0x0e,0xf4,0xe1,0xf1,0x1a,0x25,0x99,0x63,0xc9,0xd8,0x28,0x2f,0x36,0xb2,0x7b,
0x75,0x68,0x39,0x58,0xa1,0x65,0x07,0x3e,0x69,0x2d,0xd9,0x82,0x08,0xd6,0xdc,0x17,
0xa7,0x91,0xfd,0x9c,0x3d,0xd5,0x45,0x54,0xca,0x6b,0x65,0x67,0x50,0xb7,0x7b,0x19,
0x61,0xc9,0x17,0xf4,0x86,0xba,0x3d,0xcf,0xfe,0x78,0xde,0x2b,0x34,0x6c,0xd3,0x8b,
0x4d,0xe2,0x34,0xaf,0x1a,0x35,0x66,0xcc,0xb8,0x6b,0x76,0x14,0xe9,0x9d,0x96,0x62,
0x4a,0xb3,0x91,0x95,0x23,0xdf,0xe9,0x38,0x1d,0xd5,0x7d,0x4d,0x00,0xf0,0x0b,0x6d,
},
},
};
const size_t DRBG_SHA512_TEST_VEC_LEN = sizeof(DRBG_SHA512_TEST_VEC)
					/ sizeof(DRBG_SHA512_TEST_VEC[0]);
