/*
 * This program is released under the Common Public License V1.0
 *
 * You should have received a copy of Common Public License V1.0 along with
 * with this program.
 *
 * DRBG conforming to NIST SP800-90A
 *
 * Author(s): Patrick Steuer <patrick.steuer@de.ibm.com>
 *
 * Copyright IBM Corp. 2015
 */

#ifndef S390_DRBG_SHA512_H
#define S390_DRBG_SHA512_H

#include <stdint.h>

#include "ica_api.h"

#define DRBG_SHA512_SEED_LEN (888 / 8)

/*
 * SHA-512 DRBG mechanism working state type (see POP)
 */
struct drbg_sha512_ws{
	uint32_t rsvd0;				/* padding */
	uint32_t reseed_ctr;			/* reseed counter */
	uint64_t stream_bytes;			/* no. of generated bytes */
	unsigned char rsvd1;			/* padding */
	unsigned char v[DRBG_SHA512_SEED_LEN];	/* V */
	unsigned char rsvd2;			/* padding */
	unsigned char c[DRBG_SHA512_SEED_LEN];	/* C */
};
typedef struct drbg_sha512_ws drbg_sha512_ws_t;

/*
 * SHA-512 DRBG mechanism test vector type
 */
struct drbg_sha512_test_vec{
	bool no_reseed;
	bool pr;
	size_t entropy_len;
	size_t nonce_len;
	size_t pers_len;
	size_t add_len;
	size_t prnd_len;

	struct{
		/* Input */
		unsigned char *entropy;
		unsigned char *nonce;
		unsigned char *pers;
		/* Output */
		unsigned char *v;
		unsigned char *c;
		int reseed_ctr;
	}inst;

	struct{
		/* Input */
		unsigned char *entropy;
		unsigned char *add;
		/* Output */
		unsigned char *v;
		unsigned char *c;
		int reseed_ctr;
	}res, gen1, gen2;

	unsigned char *prnd;
};
typedef struct drbg_sha512_test_vec drbg_sha512_test_vec_t;

/*
 * Array of SHA-512 DRBG test vectors for health testing
 */
extern const drbg_sha512_test_vec_t DRBG_SHA512_TEST_VEC[];
extern const size_t DRBG_SHA512_TEST_VEC_LEN;

/*
 * SHA-512 DRBG mechanism functions
 */
int drbg_sha512_instantiate(void **ws,
			    int sec_strength,
			    const unsigned char *pers,
			    size_t pers_len,
			    const unsigned char *entropy,
			    size_t entropy_len,
			    const unsigned char *nonce,
			    size_t nonce_len);

int drbg_sha512_instantiate_ppno(void **ws,
				 int sec_strength,
				 const unsigned char *pers,
				 size_t pers_len,
				 const unsigned char *entropy,
				 size_t entropy_len,
				 const unsigned char *nonce,
				 size_t nonce_len);

int drbg_sha512_reseed(void *ws,
		       const unsigned char *add,
		       size_t add_len,
		       const unsigned char *entropy,
		       size_t entropy_len);

int drbg_sha512_reseed_ppno(void *ws,
			    const unsigned char *add,
			    size_t add_len,
			    const unsigned char *entropy,
			    size_t entropy_len);

int drbg_sha512_generate(void *ws,
			 const unsigned char *add,
			 size_t add_len,
			 unsigned char *prnd,
			 size_t prnd_len);

int drbg_sha512_generate_ppno(void *ws,
			      const unsigned char *add,
			      size_t add_len,
			      unsigned char *prnd,
			      size_t prnd_len);

int drbg_sha512_uninstantiate(void **ws,
			      bool test_mode);

int drbg_sha512_health_test(void *func,
			    int sec,
			    bool pr);

#endif
