(* virt-v2v
 * Copyright (C) 2009-2017 Red Hat Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *)

(** This module implements various [virsh]-like commands, but with
    non-broken authentication handling.

    If you do [virsh dumpxml foo] and if the libvirt source (eg. ESX)
    requires an interactive password, then virsh unhelpfully sends the
    password prompt to stdout, which is the same place we would be
    reading the XML from.  This file works around this brokenness. *)

val dumpxml : ?password:string -> ?conn:string -> string -> string
(** [dumpxml ?password ?conn dom] returns the libvirt XML of domain [dom].
    The optional [?conn] parameter is the libvirt connection URI.
    [dom] may be a guest name or UUID. *)

val pool_dumpxml : ?conn:string -> string -> string
(** [pool_dumpxml ?conn pool] returns the libvirt XML of pool [pool].
    The optional [?conn] parameter is the libvirt connection URI.
    [pool] may be a pool name or UUID. *)

val vol_dumpxml : ?conn:string -> string -> string -> string
(** [vol_dumpxml ?conn pool vol] returns the libvirt XML of volume [vol],
    which is part of the pool [pool].
    The optional [?conn] parameter is the libvirt connection URI.
    [pool] may be a pool name or UUID. *)

val capabilities : ?conn:string -> unit -> string
(** [capabilities ?conn ()] returns the libvirt capabilities XML.
    The optional [?conn] parameter is the libvirt connection URI. *)

val domain_exists : ?conn:string -> string -> bool
(** [domain_exists ?conn dom] returns a boolean indicating if the
    the libvirt XML domain [dom] exists.
    The optional [?conn] parameter is the libvirt connection URI.
    [dom] may be a guest name, but not a UUID. *)

val libvirt_get_version : unit -> int * int * int
(** [libvirt_get_version] returns the triple [(major, minor, release)]
    version number of the libvirt library that we are linked against. *)
