/*
 * Copyright (C) 2016-2018 Red Hat, Inc.  All rights reserved.
 *
 * Authors: Fabio M. Di Nitto <fabbione@kronosnet.org>
 *
 * This software licensed under GPL-2.0+, LGPL-2.0+
 */

#include "config.h"

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "libknet.h"
#include "internals.h"

#include "test-common.h"

static void test(void)
{
	knet_handle_t knet_h;
	int logfds[2];

	setup_logpipes(logfds);

	printf("Test knet_handle_free with invalid knet_h\n");

	if ((!knet_handle_free(NULL)) || (errno != EINVAL)) {
		printf("knet_handle_free failed to detect invalid parameter\n");
		exit(FAIL);
	}

	printf("Test knet_handle_free with one host configured\n");

	knet_h = knet_handle_start(logfds, KNET_LOG_DEBUG);

	if (knet_host_add(knet_h, 1) < 0) {
		printf("Unable to add new knet_host: %s\n", strerror(errno));
		knet_handle_free(knet_h);
		flush_logs(logfds[0], stdout);
		close_logpipes(logfds);
		exit(FAIL);
	}

	if ((!knet_handle_free(knet_h)) || (errno != EBUSY)) {
		printf("knet_handle_free didn't return error or correct errno with one host configured: %s\n", strerror(errno));
		flush_logs(logfds[0], stdout);
		close_logpipes(logfds);
		exit(FAIL);
	}

	flush_logs(logfds[0], stdout);

	if (knet_host_remove(knet_h, 1) < 0) {
		printf("Unable to remove knet_host: %s\n", strerror(errno));
		knet_handle_free(knet_h);
		flush_logs(logfds[0], stdout);
		close_logpipes(logfds);
		exit(FAIL);
	}

	if (knet_handle_free(knet_h) < 0) {
		printf("knet_handle_free failed: %s\n", strerror(errno));
		flush_logs(logfds[0], stdout);
		close_logpipes(logfds);
		exit(FAIL);
	}

	flush_logs(logfds[0], stdout);
	close_logpipes(logfds);
}

int main(int argc, char *argv[])
{
	test();

	return PASS;
}
