#! /usr/bin/env bash

set -eu

declare -A HDRS

# shellcheck disable=SC2034
SRCS_pinctrl_core=(
	"pinctrl/core.h"
	"pinctrl/pinctrl-utils.h"
)
HDRS["pinctrl_core"]="pinctrl:"
# shellcheck disable=SC2034
SRCS_pinctrl_msm=(
	"pinctrl/qcom/pinctrl-msm.c"
	"pinctrl/qcom/pinctrl-msm.h"
)
HDRS["pinctrl_msm"]="pinctrl: msm:"
# shellcheck disable=SC2034
SRCS_pinctrl_sa8775p=(
	"pinctrl/qcom/pinctrl-sa8775p.c"
)
HDRS["pinctrl_sa8775p"]="pinctrl: qcom: sa8775p:"
# shellcheck disable=SC2034
SRCS_qcom_geni_se=(
	"soc/qcom/qcom-geni-se.c"
	"include/linux/soc/qcom/geni-se.h"
)
HDRS["qcom_geni_se"]="soc: qcom:"
# shellcheck disable=SC2034
SRCS_qcom_geni_serial=(
	"tty/serial/qcom_geni_serial.c"
)
HDRS["qcom_geni_serial"]="tty: serial: qcom-geni-serial:"
# shellcheck disable=SC2034
SRCS_ufs_qcom=(
	"ufs/host/ufs-qcom.c"
	"ufs/host/ufshcd-pltfrm.h"
	"ufs/host/ufs-qcom.h"
)
HDRS["ufs_qcom"]="scsi: ufs: qcom:"

MODULES_DFL=(
	"pinctrl_core" # Headers only
	"pinctrl_msm"
	"pinctrl_sa8775p"
	"qcom_geni_se"
	"qcom_geni_serial"
	"ufs_qcom"
)

KGIT_DFL="https://gitlab.com/redhat/centos-stream/src/kernel/centos-stream-9.git"
KTAG_DFL="kernel-automotive-5.14.0-602.551.el9iv"
ODIR_DFL="$(pwd)"
usage() {
	cat - <<EOF
$0 [-h] [ -k KERNEL_GIT ] [ -K KERNEL_DIR ] [ -t KERNEL_TAG ] [ -C OUT_OF_TREE_REPOSITORY ] [ -m module[,...] ]

Extract the sources files from the kernel repository at the given tag in preparation for the Qualcomm out-of-tree modules.
A commit for each sub-system extracted will be created in the out-of-tree repository.
Defaults:
	Repository: $KGIT_DFL
	Tag: $KTAG_DFL
	Out-of-tree repository: $ODIR_DFL
EOF
}

KGIT="${KGIT_DFL}"
KTAG="${KTAG_DFL}"
ODIR="${ODIR_DFL}"
parse_cli() {
	local cli_args
	if ! cli_args=$(getopt -o 'hk:K:t:C:m:' --long 'help,kernel:,tag:,dir:,modules:' -- "$@"); then
		usage
		exit 1
	fi
	eval set -- "$cli_args"

	while true; do
		case "$1" in
		'-h' | '--help')
			usage
			exit 0
			;;
		'-k' | '--kernel')
			KGIT="$2"
			shift 2
			continue
			;;
		'-K' | '--kdir')
			KDIR="$2"
			shift 2
			continue
			;;
		'-t' | '--tag')
			KTAG="$2"
			shift 2
			continue
			;;
		'-C' | '--dir')
			ODIR="$2"
			shift 2
			continue
			;;
		'-m' | '--modules')
			IFS=', ' read -r -a MODULES <<<"$2"
			shift 2
			continue
			;;
		'--')
			shift
			break
			;;
		*)
			echo 'getopt error!' >&2
			exit 1
			;;
		esac
	done
}

parse_cli "$@"

if [[ ! -d "$ODIR" ]]; then
	echo "Out-of-tree repository does not exists: '$ODIR'" >&2
	exit 1
fi
if [[ ! -d "$ODIR/.git" ]]; then
	echo "Out-of-tree repository is not a git repository: '$ODIR'" >&2
	exit 1
fi

pushd "$ODIR" >/dev/null

if [[ -n "$KDIR" ]]; then
	git -C "$KDIR" checkout "$KTAG"
else
	KDIR=$(mktemp -d)
	git clone --depth 1 --revision "$KTAG" "$KGIT" "$KDIR"
fi

if [[ "${#MODULES[@]}" -eq 0 ]]; then
	MODULES=("${MODULES_DFL[@]}")
fi

for mod in "${MODULES[@]}"; do
	if ! printf "%s\n" "${MODULES_DFL[@]}" | grep -q -- "$mod"; then
		echo "\"$mod\" is not supported" >&2
		continue
	fi
	echo "$mod:"
	declare -n srcs=SRCS_$mod
	srcdir=$(dirname "${srcs[0]}")
	# Only add to Kbuild if there are sources to build.
	if printf "%s\n" "${srcs[@]}" | grep -q -- '\.c'; then
		echo "obj-m += $srcdir/" >>Kbuild
	fi
	mkdir -p "$srcdir"
	for file in "${srcs[@]}"; do
		if [[ "$file" = include/* ]]; then
			echo "HEADER $file"
			mkdir -p "$(dirname "$file")"
			cp -v "$KDIR/$file" "$file"
			echo "ccflags-y := -I\$(src)/$(realpath --relative-to="$srcdir" include)" >>"$srcdir/Kbuild"
		else
			echo "FILE $file"
			set -x
			cp -v "$KDIR/drivers/$file" "$file"
			if [[ "$file" = *.c ]]; then
				echo "KBUILD ${file##.c}.o: $file"
				echo "obj-m += $(basename "${file%%.c}.o")" >>"$srcdir/Kbuild"
				cat "$srcdir/Kbuild"
			fi
			set +x
		fi
	done

	git add "${srcs[@]}"
	git add "./*Kbuild*"
	git commit -s -m "${HDRS["$mod"]} import $KTAG" -m "Upstream-Status: $KGIT"
done

popd >/dev/null
