/*
 * Copyright (c) 2015 2016, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name of Intel Corporation nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * This file contains the implementation of the configuration database interface.
 *
 * It is auto-generated by the db_schema_gen. To make changes to this file 
 * modify the schema.c.template.
 */
#include "schema.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sqlite3.h>
#ifdef __cplusplus
extern "C" {
#endif
// declared in either the test_driver or common/[ostype]_os.c
extern int delete_file(const char *path);
/*
 * This file contains helper functions and MACROS for schema db generation.
 * It is combined with the schema.c.template code when generating schema.c
 * and therefore it does not contain the copyright header.
 */
#define	SQLITE_PREPARE(db, sql, p_stmt) \
		(sqlite3_prepare_v2((db), (sql), strlen(sql) + 1, (&p_stmt), NULL) == SQLITE_OK)
/*!
 * copy a sqlite int64 column into the destination
 */
#define	INTEGER_COLUMN(p_stmt, col, dest) \
	{ \
		dest = sqlite3_column_int64(p_stmt, col); \
	}
/*
 * Replace strncpy with own version. This string copy does not
 * guarantee an ending '\0' which is more appropriate for loading some
 * DB data into some structs.
 */
char *db_strcpy(char *dst, const char *src, size_t dst_size)
{
	if (dst && src && (dst_size != 0))
	{
		memset(dst, 0, dst_size);
		char * dst_i = dst;
		const char * src_i = src;
		memset(dst, 0, dst_size);
		char *end = &dst[dst_size];
		while ((dst_i < end) && (*src_i != '\0'))
		{
			*dst_i++ = *src_i++;
		}
		if (*src_i == '\0' && dst_i < end)
		{
			*dst_i = '\0';
		}
	}
	return dst;
}
/*!
 * copy a sqlite text column into the destination
 */
#define	TEXT_COLUMN(p_stmt, col, dest, len) \
	{\
		if (sqlite3_column_bytes(p_stmt, col) > 0) \
		{ \
			db_strcpy((char *)dest, (char *)sqlite3_column_text(p_stmt, col), len); \
		}\
	}
/*!
 * bind a string value to a specific parameter in the query
 */
#define	BIND_TEXT(p_stmt, parameter, text) \
		sqlite3_bind_text(p_stmt, \
			sqlite3_bind_parameter_index(p_stmt, parameter), \
			(char *)(text), -1, SQLITE_STATIC)
/*!
 * bind an int value to a specific parameter in the query
 */
#define	BIND_INTEGER(p_stmt, parameter, value) \
		sqlite3_bind_int64(p_stmt, \
			sqlite3_bind_parameter_index(p_stmt, parameter), \
			value)
/*!
 * Macro that will persist the first database error encountered.
 * @param[in,out] rc
 * 		Return code to be returned if indicating an error
 * @param[in] rc_new
 * 		Return code to be returned if @c rc does not indicate error
 */
#define	KEEP_DB_ERROR(rc, rc_new)	rc = (rc < DB_SUCCESS) ? rc : rc_new;
/*!
 * Macro that will persist the first database success encountered.
 * @param[in,out] rc
 * 		Return code to be returned if indicating success
 * @param[in] rc_new
 * 		Return code to be returned if @c rc does not indicate success
 */
#define	KEEP_DB_SUCCESS(rc, rc_new)	rc = (rc >= DB_SUCCESS) ? rc : rc_new;
/*
 *	SQL API
 */
struct persistentStore
{
	sqlite3 *db;
};
/*!
 * Returns the number of rows in the table name provided.  If there is an issue with the
 * query (or the table doesn't exist) will return 0.
 * @param[in] p_ps
 * 		Pointer to the persistent store connection
 * @param[in] table_name
 * 		Table name
 * @param[out] p_count
 *		Count of the table is put in p_count
 * @return
 * 		Whether it was successful or not
 */
enum db_return_codes table_row_count(const PersistentStore *p_ps, const char *table_name, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) from %s", table_name);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 *	Execute some sql with no expected results (INSERT, UPDATE)
 */
enum db_return_codes  run_sql_no_results(sqlite3 *p_db, const char *sql)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_ERR_FAILURE;
	if (SQLITE_PREPARE(p_db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		else
		{
			// uncomment for debugging
			// printf("Error running SQL: \n%s\n", sql);
		}
		sqlite3_finalize(p_stmt);
	}
	else
	{
		// uncomment for debugging
		// printf("Error preparing SQL: \n%s\n", sql);
	}
	return rc;
}
/*
 * Execute some SQL on a sqlite db and expect a single int value as result
 */
enum db_return_codes run_scalar_sql(const PersistentStore *p_ps, const char *sql, int *p_scalar)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_scalar = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * Execute some SQL on a sqlite db and expect a single char* value as result
 */
enum db_return_codes run_text_scalar_sql(const PersistentStore *p_ps, const char *sql, char *p_value, int len)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			TEXT_COLUMN(p_stmt, 0, p_value, len);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * Returns if the table exists or not
 */
int table_exists(sqlite3 *p_db, const char *table)
{
	int exists = 0;  // table does not exist by default (false)
	sqlite3_stmt *p_stmt;
	char buf[1024];
	snprintf(buf, 1024,
			"SELECT name FROM sqlite_master WHERE name = '%s'", table);
	if (SQLITE_PREPARE(p_db, buf, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			exists = 1; // table exists (true)
		}
		sqlite3_finalize(p_stmt);
	}
	return exists;
}
PersistentStore *open_PersistentStore(const char *path)
{
	PersistentStore *result = (PersistentStore *)malloc(sizeof (PersistentStore));
	if (result != NULL)
	{
		if (sqlite3_open_v2(path, &(result->db),
			SQLITE_OPEN_READWRITE|SQLITE_OPEN_CREATE|SQLITE_OPEN_FULLMUTEX, NULL) != SQLITE_OK)
		{
			free_PersistentStore(&result);
		}
		else
		{
			// set a busy timeout to avoid file locking issues
			sqlite3_busy_timeout(result->db, 30000);
		}
	}
	return result;
}
/*
 * Close the DB and release the memory for a PersistentStore object
 */
int free_PersistentStore(PersistentStore **pp_persistentStore)
{
	int rc = DB_SUCCESS;
	if (*pp_persistentStore != NULL && (*pp_persistentStore)->db != NULL)
	{
		if (sqlite3_close((*pp_persistentStore)->db) != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	if (*pp_persistentStore != NULL)
	{
		free(*pp_persistentStore);
	}
	*pp_persistentStore = NULL;
	return rc;
}
/*
 * Add a new history instance.  Return the new instance ID
 */
enum db_return_codes db_add_history(PersistentStore *p_ps,
	const char *history_name,
	int *p_history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_history_id = 0;
	sqlite3_stmt *p_stmt;
	if (run_scalar_sql(p_ps, "SELECT MAX(history_id) FROM history", p_history_id) == DB_SUCCESS)
	{
		(*p_history_id)++;
		char *sql = "INSERT INTO history \
			( history_id,  timestamp,  history_name) VALUES \
			($history_id, datetime('now'), $history_name);";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", *p_history_id);
			BIND_TEXT(p_stmt, "$history_name", history_name);
			if (sqlite3_step(p_stmt) == SQLITE_DONE)
			{
				rc = DB_SUCCESS;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	return rc;
}
enum db_return_codes  db_begin_transaction(PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "BEGIN TRANSACTION");
}
enum db_return_codes  db_end_transaction(PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "END TRANSACTION");
}
enum db_return_codes  db_rollback_transaction(PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "ROLLBACK TRANSACTION");
}
enum db_return_codes db_run_custom_sql(PersistentStore *p_ps, const char *sql)
{
	return run_sql_no_results(p_ps->db, sql);
}
void update_sqlite3_hook(PersistentStore *p_ps, void (*xCallback)(void*,int,char const *,char const *, long long))
{
	sqlite3_update_hook(p_ps->db, xCallback, NULL);
}
/*
 * Get history table entries
 */
int db_get_history_ids(const PersistentStore *p_ps,
	int *p_history_ids,
	int count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_history_ids, 0, sizeof (int) * count);
	char *sql = "SELECT \
		history_id \
		FROM history \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < count)
		{
			INTEGER_COLUMN(p_stmt,
					0,
					p_history_ids[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
// Table count is calculated in CrudSchemaGenerator
#define	TABLE_COUNT (115)
/*
 * Create a PersistentStore object
 */
PersistentStore *create_PersistentStore(const char *path, int force)
{
	PersistentStore *result = (PersistentStore *)malloc(sizeof (PersistentStore));
	if (result != NULL)
	{
		// check if the file exists - delete it if force
		FILE *file;
		if (force && (file = fopen(path, "r")))
		{
			fclose(file);
			delete_file(path); // make sure doesn't already exist
		}
		// build the schema ...
		struct table { char table_name[256]; char create_statement[2048]; };
		struct table *tables = (struct table *)malloc(TABLE_COUNT * sizeof (struct table));
		if (tables)
		{
			int populate_index = 0;
			tables[populate_index++] = ((struct table){"history",
				"CREATE TABLE history(history_id INTEGER PRIMARY KEY NOT NULL UNIQUE, \
					timestamp DATETIME NOT NULL, history_name TEXT )"});
			tables[populate_index++] = ((struct table){"config",
				"CREATE TABLE config (       \
					 key TEXT  PRIMARY KEY  NOT NULL UNIQUE  , \
					 value TEXT   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"config_history",
				"CREATE TABLE config_history (       \
					history_id INTEGER NOT NULL, \
					 key TEXT , \
					 value TEXT  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"log",
				"CREATE TABLE log (       \
					 id INTEGER  PRIMARY KEY  AUTOINCREMENT  NOT NULL UNIQUE  , \
					 thread_id INTEGER  , \
					 time INTEGER  , \
					 level INTEGER  , \
					 file_name TEXT  , \
					 line_number INTEGER  , \
					 message TEXT   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"log_history",
				"CREATE TABLE log_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 thread_id INTEGER , \
					 time INTEGER , \
					 level INTEGER , \
					 file_name TEXT , \
					 line_number INTEGER , \
					 message TEXT  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"event",
				"CREATE TABLE event (       \
					 id INTEGER  PRIMARY KEY  AUTOINCREMENT  NOT NULL UNIQUE  , \
					 type INTEGER  , \
					 severity INTEGER  , \
					 code INTEGER  , \
					 action_required INTEGER  , \
					 uid TEXT  , \
					 time INTEGER  , \
					 arg1 TEXT  , \
					 arg2 TEXT  , \
					 arg3 TEXT  , \
					 diag_result INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"event_history",
				"CREATE TABLE event_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 type INTEGER , \
					 severity INTEGER , \
					 code INTEGER , \
					 action_required INTEGER , \
					 uid TEXT , \
					 time INTEGER , \
					 arg1 TEXT , \
					 arg2 TEXT , \
					 arg3 TEXT , \
					 diag_result INTEGER  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"topology_state",
				"CREATE TABLE topology_state (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 uid TEXT  , \
					 manufacturer INTEGER  , \
					 serial_num INTEGER  , \
					 model_num TEXT  , \
					 current_config_status INTEGER  , \
					 config_goal_status INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"topology_state_history",
				"CREATE TABLE topology_state_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 uid TEXT , \
					 manufacturer INTEGER , \
					 serial_num INTEGER , \
					 model_num TEXT , \
					 current_config_status INTEGER , \
					 config_goal_status INTEGER  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"host",
				"CREATE TABLE host (       \
					 name TEXT  PRIMARY KEY  NOT NULL UNIQUE  , \
					 os_type INTEGER  , \
					 os_name TEXT  , \
					 os_version TEXT   \
					);"});
			tables[populate_index++] = ((struct table){"host_history",
				"CREATE TABLE host_history (       \
					history_id INTEGER NOT NULL, \
					 name TEXT , \
					 os_type INTEGER , \
					 os_name TEXT , \
					 os_version TEXT  \
					);"});
tables[populate_index++] = ((struct table){"sw_inventory",
				"CREATE TABLE sw_inventory (       \
					 name TEXT  PRIMARY KEY  NOT NULL UNIQUE  , \
					 mgmt_sw_rev TEXT  , \
					 vendor_driver_rev TEXT  , \
					 supported_driver_available INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"sw_inventory_history",
				"CREATE TABLE sw_inventory_history (       \
					history_id INTEGER NOT NULL, \
					 name TEXT , \
					 mgmt_sw_rev TEXT , \
					 vendor_driver_rev TEXT , \
					 supported_driver_available INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"socket",
				"CREATE TABLE socket (       \
					 socket_id INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 type INTEGER  , \
					 model INTEGER  , \
					 brand INTEGER  , \
					 family INTEGER  , \
					 stepping INTEGER  , \
					 manufacturer TEXT  , \
					 logical_processor_count INTEGER  , \
					 rapl_limited INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"socket_history",
				"CREATE TABLE socket_history (       \
					history_id INTEGER NOT NULL, \
					 socket_id INTEGER , \
					 type INTEGER , \
					 model INTEGER , \
					 brand INTEGER , \
					 family INTEGER , \
					 stepping INTEGER , \
					 manufacturer TEXT , \
					 logical_processor_count INTEGER , \
					 rapl_limited INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"runtime_config_validation",
				"CREATE TABLE runtime_config_validation (       \
					 id INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 type INTEGER  , \
					 length INTEGER  , \
					 address_space_id INTEGER  , \
					 bit_width INTEGER  , \
					 bit_offset INTEGER  , \
					 access_size INTEGER  , \
					 address INTEGER  , \
					 operation_type_1 INTEGER  , \
					 value INTEGER  , \
					 mask_1 INTEGER  , \
					 gas_structure_0 INTEGER  , \
					 gas_structure_1 INTEGER  , \
					 gas_structure_2 INTEGER  , \
					 gas_structure_3 INTEGER  , \
					 gas_structure_4 INTEGER  , \
					 gas_structure_5 INTEGER  , \
					 gas_structure_6 INTEGER  , \
					 gas_structure_7 INTEGER  , \
					 gas_structure_8 INTEGER  , \
					 gas_structure_9 INTEGER  , \
					 gas_structure_10 INTEGER  , \
					 gas_structure_11 INTEGER  , \
					 operation_type_2 INTEGER  , \
					 mask_2 INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"runtime_config_validation_history",
				"CREATE TABLE runtime_config_validation_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 type INTEGER , \
					 length INTEGER , \
					 address_space_id INTEGER , \
					 bit_width INTEGER , \
					 bit_offset INTEGER , \
					 access_size INTEGER , \
					 address INTEGER , \
					 operation_type_1 INTEGER , \
					 value INTEGER , \
					 mask_1 INTEGER , \
					 gas_structure_0 INTEGER , \
					 gas_structure_1 INTEGER , \
					 gas_structure_2 INTEGER , \
					 gas_structure_3 INTEGER , \
					 gas_structure_4 INTEGER , \
					 gas_structure_5 INTEGER , \
					 gas_structure_6 INTEGER , \
					 gas_structure_7 INTEGER , \
					 gas_structure_8 INTEGER , \
					 gas_structure_9 INTEGER , \
					 gas_structure_10 INTEGER , \
					 gas_structure_11 INTEGER , \
					 operation_type_2 INTEGER , \
					 mask_2 INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"interleave_capability",
				"CREATE TABLE interleave_capability (       \
					 id INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 type INTEGER  , \
					 length INTEGER  , \
					 memory_mode INTEGER  , \
					 interleave_alignment_size INTEGER  , \
					 supported_interleave_count INTEGER  , \
					 interleave_format_list_0 INTEGER  , \
					 interleave_format_list_1 INTEGER  , \
					 interleave_format_list_2 INTEGER  , \
					 interleave_format_list_3 INTEGER  , \
					 interleave_format_list_4 INTEGER  , \
					 interleave_format_list_5 INTEGER  , \
					 interleave_format_list_6 INTEGER  , \
					 interleave_format_list_7 INTEGER  , \
					 interleave_format_list_8 INTEGER  , \
					 interleave_format_list_9 INTEGER  , \
					 interleave_format_list_10 INTEGER  , \
					 interleave_format_list_11 INTEGER  , \
					 interleave_format_list_12 INTEGER  , \
					 interleave_format_list_13 INTEGER  , \
					 interleave_format_list_14 INTEGER  , \
					 interleave_format_list_15 INTEGER  , \
					 interleave_format_list_16 INTEGER  , \
					 interleave_format_list_17 INTEGER  , \
					 interleave_format_list_18 INTEGER  , \
					 interleave_format_list_19 INTEGER  , \
					 interleave_format_list_20 INTEGER  , \
					 interleave_format_list_21 INTEGER  , \
					 interleave_format_list_22 INTEGER  , \
					 interleave_format_list_23 INTEGER  , \
					 interleave_format_list_24 INTEGER  , \
					 interleave_format_list_25 INTEGER  , \
					 interleave_format_list_26 INTEGER  , \
					 interleave_format_list_27 INTEGER  , \
					 interleave_format_list_28 INTEGER  , \
					 interleave_format_list_29 INTEGER  , \
					 interleave_format_list_30 INTEGER  , \
					 interleave_format_list_31 INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"interleave_capability_history",
				"CREATE TABLE interleave_capability_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 type INTEGER , \
					 length INTEGER , \
					 memory_mode INTEGER , \
					 interleave_alignment_size INTEGER , \
					 supported_interleave_count INTEGER , \
					 interleave_format_list_0 INTEGER , \
					 interleave_format_list_1 INTEGER , \
					 interleave_format_list_2 INTEGER , \
					 interleave_format_list_3 INTEGER , \
					 interleave_format_list_4 INTEGER , \
					 interleave_format_list_5 INTEGER , \
					 interleave_format_list_6 INTEGER , \
					 interleave_format_list_7 INTEGER , \
					 interleave_format_list_8 INTEGER , \
					 interleave_format_list_9 INTEGER , \
					 interleave_format_list_10 INTEGER , \
					 interleave_format_list_11 INTEGER , \
					 interleave_format_list_12 INTEGER , \
					 interleave_format_list_13 INTEGER , \
					 interleave_format_list_14 INTEGER , \
					 interleave_format_list_15 INTEGER , \
					 interleave_format_list_16 INTEGER , \
					 interleave_format_list_17 INTEGER , \
					 interleave_format_list_18 INTEGER , \
					 interleave_format_list_19 INTEGER , \
					 interleave_format_list_20 INTEGER , \
					 interleave_format_list_21 INTEGER , \
					 interleave_format_list_22 INTEGER , \
					 interleave_format_list_23 INTEGER , \
					 interleave_format_list_24 INTEGER , \
					 interleave_format_list_25 INTEGER , \
					 interleave_format_list_26 INTEGER , \
					 interleave_format_list_27 INTEGER , \
					 interleave_format_list_28 INTEGER , \
					 interleave_format_list_29 INTEGER , \
					 interleave_format_list_30 INTEGER , \
					 interleave_format_list_31 INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"platform_info_capability",
				"CREATE TABLE platform_info_capability (       \
					 id INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 type INTEGER  , \
					 length INTEGER  , \
					 mgmt_sw_config_support INTEGER  , \
					 mem_mode_capabilities INTEGER  , \
					 current_mem_mode INTEGER  , \
					 pmem_ras_capabilities INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"platform_info_capability_history",
				"CREATE TABLE platform_info_capability_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 type INTEGER , \
					 length INTEGER , \
					 mgmt_sw_config_support INTEGER , \
					 mem_mode_capabilities INTEGER , \
					 current_mem_mode INTEGER , \
					 pmem_ras_capabilities INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"platform_capabilities",
				"CREATE TABLE platform_capabilities (       \
					 signature TEXT  PRIMARY KEY  NOT NULL UNIQUE  , \
					 length INTEGER  , \
					 revision INTEGER  , \
					 checksum INTEGER  , \
					 oem_id TEXT  , \
					 oem_table_id TEXT  , \
					 oem_revision INTEGER  , \
					 creator_id_0 INTEGER  , \
					 creator_id_1 INTEGER  , \
					 creator_id_2 INTEGER  , \
					 creator_id_3 INTEGER  , \
					 creator_revision INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"platform_capabilities_history",
				"CREATE TABLE platform_capabilities_history (       \
					history_id INTEGER NOT NULL, \
					 signature TEXT , \
					 length INTEGER , \
					 revision INTEGER , \
					 checksum INTEGER , \
					 oem_id TEXT , \
					 oem_table_id TEXT , \
					 oem_revision INTEGER , \
					 creator_id_0 INTEGER , \
					 creator_id_1 INTEGER , \
					 creator_id_2 INTEGER , \
					 creator_id_3 INTEGER , \
					 creator_revision INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"driver_capabilities",
				"CREATE TABLE driver_capabilities (       \
					 id INTEGER  PRIMARY KEY  AUTOINCREMENT  NOT NULL UNIQUE  , \
					 min_namespace_size INTEGER  , \
					 max_non_continguous_namespaces INTEGER  , \
					 block_sizes_0 INTEGER  , \
					 block_sizes_1 INTEGER  , \
					 block_sizes_2 INTEGER  , \
					 block_sizes_3 INTEGER  , \
					 block_sizes_4 INTEGER  , \
					 block_sizes_5 INTEGER  , \
					 block_sizes_6 INTEGER  , \
					 block_sizes_7 INTEGER  , \
					 block_sizes_8 INTEGER  , \
					 block_sizes_9 INTEGER  , \
					 block_sizes_10 INTEGER  , \
					 block_sizes_11 INTEGER  , \
					 block_sizes_12 INTEGER  , \
					 block_sizes_13 INTEGER  , \
					 block_sizes_14 INTEGER  , \
					 block_sizes_15 INTEGER  , \
					 num_block_sizes INTEGER  , \
					 namespace_memory_page_allocation_capable INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"driver_capabilities_history",
				"CREATE TABLE driver_capabilities_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 min_namespace_size INTEGER , \
					 max_non_continguous_namespaces INTEGER , \
					 block_sizes_0 INTEGER , \
					 block_sizes_1 INTEGER , \
					 block_sizes_2 INTEGER , \
					 block_sizes_3 INTEGER , \
					 block_sizes_4 INTEGER , \
					 block_sizes_5 INTEGER , \
					 block_sizes_6 INTEGER , \
					 block_sizes_7 INTEGER , \
					 block_sizes_8 INTEGER , \
					 block_sizes_9 INTEGER , \
					 block_sizes_10 INTEGER , \
					 block_sizes_11 INTEGER , \
					 block_sizes_12 INTEGER , \
					 block_sizes_13 INTEGER , \
					 block_sizes_14 INTEGER , \
					 block_sizes_15 INTEGER , \
					 num_block_sizes INTEGER , \
					 namespace_memory_page_allocation_capable INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"driver_features",
				"CREATE TABLE driver_features (       \
					 id INTEGER  PRIMARY KEY  AUTOINCREMENT  NOT NULL UNIQUE  , \
					 get_platform_capabilities INTEGER  , \
					 get_topology INTEGER  , \
					 get_interleave INTEGER  , \
					 get_dimm_detail INTEGER  , \
					 get_namespaces INTEGER  , \
					 get_namespace_detail INTEGER  , \
					 get_address_scrub_data INTEGER  , \
					 get_platform_config_data INTEGER  , \
					 get_boot_status INTEGER  , \
					 get_power_data INTEGER  , \
					 get_security_state INTEGER  , \
					 get_log_page INTEGER  , \
					 get_features INTEGER  , \
					 set_features INTEGER  , \
					 create_namespace INTEGER  , \
					 rename_namespace INTEGER  , \
					 grow_namespace INTEGER  , \
					 shrink_namespace INTEGER  , \
					 delete_namespace INTEGER  , \
					 enable_namespace INTEGER  , \
					 disable_namespace INTEGER  , \
					 set_security_state INTEGER  , \
					 enable_logging INTEGER  , \
					 run_diagnostic INTEGER  , \
					 set_platform_config INTEGER  , \
					 passthrough INTEGER  , \
					 start_address_scrub INTEGER  , \
					 app_direct_mode INTEGER  , \
					 storage_mode INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"driver_features_history",
				"CREATE TABLE driver_features_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 get_platform_capabilities INTEGER , \
					 get_topology INTEGER , \
					 get_interleave INTEGER , \
					 get_dimm_detail INTEGER , \
					 get_namespaces INTEGER , \
					 get_namespace_detail INTEGER , \
					 get_address_scrub_data INTEGER , \
					 get_platform_config_data INTEGER , \
					 get_boot_status INTEGER , \
					 get_power_data INTEGER , \
					 get_security_state INTEGER , \
					 get_log_page INTEGER , \
					 get_features INTEGER , \
					 set_features INTEGER , \
					 create_namespace INTEGER , \
					 rename_namespace INTEGER , \
					 grow_namespace INTEGER , \
					 shrink_namespace INTEGER , \
					 delete_namespace INTEGER , \
					 enable_namespace INTEGER , \
					 disable_namespace INTEGER , \
					 set_security_state INTEGER , \
					 enable_logging INTEGER , \
					 run_diagnostic INTEGER , \
					 set_platform_config INTEGER , \
					 passthrough INTEGER , \
					 start_address_scrub INTEGER , \
					 app_direct_mode INTEGER , \
					 storage_mode INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_topology",
				"CREATE TABLE dimm_topology (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 id INTEGER  , \
					 vendor_id INTEGER  , \
					 device_id INTEGER  , \
					 revision_id INTEGER  , \
					 subsystem_vendor_id INTEGER  , \
					 subsystem_device_id INTEGER  , \
					 subsystem_revision_id INTEGER  , \
					 manufacturing_info_valid INTEGER  , \
					 manufacturing_location INTEGER  , \
					 manufacturing_date INTEGER  , \
					 type INTEGER  , \
					 interface_format_codes_0 INTEGER  , \
					 interface_format_codes_1 INTEGER  , \
					 interface_format_codes_2 INTEGER  , \
					 interface_format_codes_3 INTEGER  , \
					 interface_format_codes_4 INTEGER  , \
					 interface_format_codes_5 INTEGER  , \
					 interface_format_codes_6 INTEGER  , \
					 interface_format_codes_7 INTEGER  , \
					 interface_format_codes_8 INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_topology_history",
				"CREATE TABLE dimm_topology_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 id INTEGER , \
					 vendor_id INTEGER , \
					 device_id INTEGER , \
					 revision_id INTEGER , \
					 subsystem_vendor_id INTEGER , \
					 subsystem_device_id INTEGER , \
					 subsystem_revision_id INTEGER , \
					 manufacturing_info_valid INTEGER , \
					 manufacturing_location INTEGER , \
					 manufacturing_date INTEGER , \
					 type INTEGER , \
					 interface_format_codes_0 INTEGER , \
					 interface_format_codes_1 INTEGER , \
					 interface_format_codes_2 INTEGER , \
					 interface_format_codes_3 INTEGER , \
					 interface_format_codes_4 INTEGER , \
					 interface_format_codes_5 INTEGER , \
					 interface_format_codes_6 INTEGER , \
					 interface_format_codes_7 INTEGER , \
					 interface_format_codes_8 INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"namespace",
				"CREATE TABLE namespace (       \
					 namespace_uid TEXT  PRIMARY KEY  NOT NULL UNIQUE  , \
					 friendly_name TEXT  , \
					 block_size INTEGER  , \
					 block_count INTEGER  , \
					 type INTEGER  , \
					 health INTEGER  , \
					 enabled INTEGER  , \
					 btt INTEGER  , \
					 device_handle INTEGER  , \
					 interleave_set_index INTEGER  , \
					 memory_page_allocation INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"namespace_history",
				"CREATE TABLE namespace_history (       \
					history_id INTEGER NOT NULL, \
					 namespace_uid TEXT , \
					 friendly_name TEXT , \
					 block_size INTEGER , \
					 block_count INTEGER , \
					 type INTEGER , \
					 health INTEGER , \
					 enabled INTEGER , \
					 btt INTEGER , \
					 device_handle INTEGER , \
					 interleave_set_index INTEGER , \
					 memory_page_allocation INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"identify_dimm",
				"CREATE TABLE identify_dimm (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 vendor_id INTEGER  , \
					 device_id INTEGER  , \
					 revision_id INTEGER  , \
					 interface_format_code INTEGER  , \
					 fw_revision TEXT  , \
					 fw_api_version INTEGER  , \
					 fw_sw_mask INTEGER  , \
					 dimm_sku INTEGER  , \
					 block_windows INTEGER  , \
					 write_flush_addresses INTEGER  , \
					 write_flush_address_start INTEGER  , \
					 block_control_region_offset INTEGER  , \
					 raw_cap INTEGER  , \
					 manufacturer INTEGER  , \
					 serial_num INTEGER  , \
					 model_num TEXT   \
					);"});
			tables[populate_index++] = ((struct table){"identify_dimm_history",
				"CREATE TABLE identify_dimm_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 vendor_id INTEGER , \
					 device_id INTEGER , \
					 revision_id INTEGER , \
					 interface_format_code INTEGER , \
					 fw_revision TEXT , \
					 fw_api_version INTEGER , \
					 fw_sw_mask INTEGER , \
					 dimm_sku INTEGER , \
					 block_windows INTEGER , \
					 write_flush_addresses INTEGER , \
					 write_flush_address_start INTEGER , \
					 block_control_region_offset INTEGER , \
					 raw_cap INTEGER , \
					 manufacturer INTEGER , \
					 serial_num INTEGER , \
					 model_num TEXT  \
					);"});
tables[populate_index++] = ((struct table){"device_characteristics",
				"CREATE TABLE device_characteristics (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 controller_temp_shutdown_threshold INTEGER  , \
					 media_temp_shutdown_threshold INTEGER  , \
					 throttling_start_threshold INTEGER  , \
					 throttling_stop_threshold INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"device_characteristics_history",
				"CREATE TABLE device_characteristics_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 controller_temp_shutdown_threshold INTEGER , \
					 media_temp_shutdown_threshold INTEGER , \
					 throttling_start_threshold INTEGER , \
					 throttling_stop_threshold INTEGER  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"dimm_partition",
				"CREATE TABLE dimm_partition (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 volatile_capacity INTEGER  , \
					 volatile_start INTEGER  , \
					 pmem_capacity INTEGER  , \
					 pm_start INTEGER  , \
					 raw_capacity INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_partition_history",
				"CREATE TABLE dimm_partition_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 volatile_capacity INTEGER , \
					 volatile_start INTEGER , \
					 pmem_capacity INTEGER , \
					 pm_start INTEGER , \
					 raw_capacity INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_smart",
				"CREATE TABLE dimm_smart (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 validation_flags INTEGER  , \
					 health_status INTEGER  , \
					 media_temperature INTEGER  , \
					 spare INTEGER  , \
					 alarm_trips INTEGER  , \
					 percentage_used INTEGER  , \
					 lss INTEGER  , \
					 vendor_specific_data_size INTEGER  , \
					 power_cycles INTEGER  , \
					 power_on_seconds INTEGER  , \
					 uptime INTEGER  , \
					 unsafe_shutdowns INTEGER  , \
					 lss_details INTEGER  , \
					 last_shutdown_time INTEGER  , \
					 controller_temperature INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_smart_history",
				"CREATE TABLE dimm_smart_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 validation_flags INTEGER , \
					 health_status INTEGER , \
					 media_temperature INTEGER , \
					 spare INTEGER , \
					 alarm_trips INTEGER , \
					 percentage_used INTEGER , \
					 lss INTEGER , \
					 vendor_specific_data_size INTEGER , \
					 power_cycles INTEGER , \
					 power_on_seconds INTEGER , \
					 uptime INTEGER , \
					 unsafe_shutdowns INTEGER , \
					 lss_details INTEGER , \
					 last_shutdown_time INTEGER , \
					 controller_temperature INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_state",
				"CREATE TABLE dimm_state (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 spare_capacity_state INTEGER  , \
					 wearlevel_state INTEGER  , \
					 mediaerrors_corrected INTEGER  , \
					 mediaerrors_uncorrectable INTEGER  , \
					 mediaerrors_erasurecoded INTEGER  , \
					 health_state INTEGER  , \
					 die_spares_used INTEGER  , \
					 mediatemperature_state INTEGER  , \
					 controllertemperature_state INTEGER  , \
					 viral_state INTEGER  , \
					 newest_error_log_timestamp INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"dimm_state_history",
				"CREATE TABLE dimm_state_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 spare_capacity_state INTEGER , \
					 wearlevel_state INTEGER , \
					 mediaerrors_corrected INTEGER , \
					 mediaerrors_uncorrectable INTEGER , \
					 mediaerrors_erasurecoded INTEGER , \
					 health_state INTEGER , \
					 die_spares_used INTEGER , \
					 mediatemperature_state INTEGER , \
					 controllertemperature_state INTEGER , \
					 viral_state INTEGER , \
					 newest_error_log_timestamp INTEGER  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"namespace_state",
				"CREATE TABLE namespace_state (       \
					 namespace_uid TEXT  PRIMARY KEY  NOT NULL UNIQUE  , \
					 health_state INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"namespace_state_history",
				"CREATE TABLE namespace_state_history (       \
					history_id INTEGER NOT NULL, \
					 namespace_uid TEXT , \
					 health_state INTEGER  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"dimm_alarm_thresholds",
				"CREATE TABLE dimm_alarm_thresholds (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 enable INTEGER  , \
					 media_temperature INTEGER  , \
					 controller_temperature INTEGER  , \
					 spare INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_alarm_thresholds_history",
				"CREATE TABLE dimm_alarm_thresholds_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 enable INTEGER , \
					 media_temperature INTEGER , \
					 controller_temperature INTEGER , \
					 spare INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_power_management",
				"CREATE TABLE dimm_power_management (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 enable INTEGER  , \
					 tdp_power_limit INTEGER  , \
					 peak_power_budget INTEGER  , \
					 avg_power_budget INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_power_management_history",
				"CREATE TABLE dimm_power_management_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 enable INTEGER , \
					 tdp_power_limit INTEGER , \
					 peak_power_budget INTEGER , \
					 avg_power_budget INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_die_sparing",
				"CREATE TABLE dimm_die_sparing (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 enable INTEGER  , \
					 aggressiveness INTEGER  , \
					 supported_by_rank_0 INTEGER  , \
					 supported_by_rank_1 INTEGER  , \
					 supported_by_rank_2 INTEGER  , \
					 supported_by_rank_3 INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_die_sparing_history",
				"CREATE TABLE dimm_die_sparing_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 enable INTEGER , \
					 aggressiveness INTEGER , \
					 supported_by_rank_0 INTEGER , \
					 supported_by_rank_1 INTEGER , \
					 supported_by_rank_2 INTEGER , \
					 supported_by_rank_3 INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_optional_config_data",
				"CREATE TABLE dimm_optional_config_data (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 first_fast_refresh_enable INTEGER  , \
					 viral_policy_enable INTEGER  , \
					 viral_status INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_optional_config_data_history",
				"CREATE TABLE dimm_optional_config_data_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 first_fast_refresh_enable INTEGER , \
					 viral_policy_enable INTEGER , \
					 viral_status INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_err_correction",
				"CREATE TABLE dimm_err_correction (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 unrefreshed_enable INTEGER  , \
					 refreshed_enable INTEGER  , \
					 unrefreshed_force_write INTEGER  , \
					 refreshed_force_write INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_err_correction_history",
				"CREATE TABLE dimm_err_correction_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 unrefreshed_enable INTEGER , \
					 refreshed_enable INTEGER , \
					 unrefreshed_force_write INTEGER , \
					 refreshed_force_write INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_erasure_coding",
				"CREATE TABLE dimm_erasure_coding (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 verify_erc INTEGER  , \
					 unrefreshed_enable INTEGER  , \
					 refreshed_enable INTEGER  , \
					 unrefreshed_force_write INTEGER  , \
					 refreshed_force_write INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_erasure_coding_history",
				"CREATE TABLE dimm_erasure_coding_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 verify_erc INTEGER , \
					 unrefreshed_enable INTEGER , \
					 refreshed_enable INTEGER , \
					 unrefreshed_force_write INTEGER , \
					 refreshed_force_write INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_thermal",
				"CREATE TABLE dimm_thermal (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 throttling_enable INTEGER  , \
					 alerting_enable INTEGER  , \
					 critical_shutdown_enable INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_thermal_history",
				"CREATE TABLE dimm_thermal_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 throttling_enable INTEGER , \
					 alerting_enable INTEGER , \
					 critical_shutdown_enable INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_fw_image",
				"CREATE TABLE dimm_fw_image (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 fw_rev TEXT  , \
					 fw_type INTEGER  , \
					 staged_fw_status INTEGER  , \
					 staged_fw_rev TEXT  , \
					 staged_fw_type INTEGER  , \
					 commit_id TEXT  , \
					 build_configuration TEXT   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_fw_image_history",
				"CREATE TABLE dimm_fw_image_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 fw_rev TEXT , \
					 fw_type INTEGER , \
					 staged_fw_status INTEGER , \
					 staged_fw_rev TEXT , \
					 staged_fw_type INTEGER , \
					 commit_id TEXT , \
					 build_configuration TEXT  \
					);"});
tables[populate_index++] = ((struct table){"dimm_fw_debug_log",
				"CREATE TABLE dimm_fw_debug_log (       \
					 device_handle INTEGER  , \
					 fw_log TEXT  PRIMARY KEY  NOT NULL UNIQUE   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_fw_debug_log_history",
				"CREATE TABLE dimm_fw_debug_log_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 fw_log TEXT  \
					);"});
tables[populate_index++] = ((struct table){"dimm_memory_info_page0",
				"CREATE TABLE dimm_memory_info_page0 (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 bytes_read INTEGER  , \
					 bytes_written INTEGER  , \
					 read_reqs INTEGER  , \
					 write_reqs INTEGER  , \
					 block_read_reqs INTEGER  , \
					 block_write_reqs INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_memory_info_page0_history",
				"CREATE TABLE dimm_memory_info_page0_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 bytes_read INTEGER , \
					 bytes_written INTEGER , \
					 read_reqs INTEGER , \
					 write_reqs INTEGER , \
					 block_read_reqs INTEGER , \
					 block_write_reqs INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_memory_info_page1",
				"CREATE TABLE dimm_memory_info_page1 (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 total_bytes_read INTEGER  , \
					 total_bytes_written INTEGER  , \
					 total_read_reqs INTEGER  , \
					 total_write_reqs INTEGER  , \
					 total_block_read_reqs INTEGER  , \
					 total_block_write_reqs INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_memory_info_page1_history",
				"CREATE TABLE dimm_memory_info_page1_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 total_bytes_read INTEGER , \
					 total_bytes_written INTEGER , \
					 total_read_reqs INTEGER , \
					 total_write_reqs INTEGER , \
					 total_block_read_reqs INTEGER , \
					 total_block_write_reqs INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_memory_info_page2",
				"CREATE TABLE dimm_memory_info_page2 (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 write_count_max INTEGER  , \
					 write_count_average INTEGER  , \
					 uncorrectable_host INTEGER  , \
					 uncorrectable_non_host INTEGER  , \
					 media_errors_uc INTEGER  , \
					 media_errors_ce INTEGER  , \
					 media_errors_ecc INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_memory_info_page2_history",
				"CREATE TABLE dimm_memory_info_page2_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 write_count_max INTEGER , \
					 write_count_average INTEGER , \
					 uncorrectable_host INTEGER , \
					 uncorrectable_non_host INTEGER , \
					 media_errors_uc INTEGER , \
					 media_errors_ce INTEGER , \
					 media_errors_ecc INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_ars_command_specific_data",
				"CREATE TABLE dimm_ars_command_specific_data (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 num_errors INTEGER  , \
					 ars_state INTEGER  , \
					 dpa_error_address_0 INTEGER  , \
					 dpa_error_address_1 INTEGER  , \
					 dpa_error_address_2 INTEGER  , \
					 dpa_error_address_3 INTEGER  , \
					 dpa_error_address_4 INTEGER  , \
					 dpa_error_address_5 INTEGER  , \
					 dpa_error_address_6 INTEGER  , \
					 dpa_error_address_7 INTEGER  , \
					 dpa_error_address_8 INTEGER  , \
					 dpa_error_address_9 INTEGER  , \
					 dpa_error_address_10 INTEGER  , \
					 dpa_error_address_11 INTEGER  , \
					 dpa_error_address_12 INTEGER  , \
					 dpa_error_address_13 INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_ars_command_specific_data_history",
				"CREATE TABLE dimm_ars_command_specific_data_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 num_errors INTEGER , \
					 ars_state INTEGER , \
					 dpa_error_address_0 INTEGER , \
					 dpa_error_address_1 INTEGER , \
					 dpa_error_address_2 INTEGER , \
					 dpa_error_address_3 INTEGER , \
					 dpa_error_address_4 INTEGER , \
					 dpa_error_address_5 INTEGER , \
					 dpa_error_address_6 INTEGER , \
					 dpa_error_address_7 INTEGER , \
					 dpa_error_address_8 INTEGER , \
					 dpa_error_address_9 INTEGER , \
					 dpa_error_address_10 INTEGER , \
					 dpa_error_address_11 INTEGER , \
					 dpa_error_address_12 INTEGER , \
					 dpa_error_address_13 INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_long_op_status",
				"CREATE TABLE dimm_long_op_status (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 opcode INTEGER  , \
					 subopcode INTEGER  , \
					 percent_complete INTEGER  , \
					 etc INTEGER  , \
					 status_code INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_long_op_status_history",
				"CREATE TABLE dimm_long_op_status_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 opcode INTEGER , \
					 subopcode INTEGER , \
					 percent_complete INTEGER , \
					 etc INTEGER , \
					 status_code INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_details",
				"CREATE TABLE dimm_details (       \
					 device_handle INTEGER  , \
					 form_factor INTEGER  , \
					 data_width INTEGER  , \
					 total_width INTEGER  , \
					 size INTEGER  , \
					 speed INTEGER  , \
					 part_number TEXT  , \
					 device_locator TEXT  , \
					 bank_label TEXT  , \
					 manufacturer TEXT  , \
					 type INTEGER  , \
					 type_detail INTEGER  , \
					 id INTEGER  PRIMARY KEY  NOT NULL UNIQUE   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_details_history",
				"CREATE TABLE dimm_details_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 form_factor INTEGER , \
					 data_width INTEGER , \
					 total_width INTEGER , \
					 size INTEGER , \
					 speed INTEGER , \
					 part_number TEXT , \
					 device_locator TEXT , \
					 bank_label TEXT , \
					 manufacturer TEXT , \
					 type INTEGER , \
					 type_detail INTEGER , \
					 id INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_security_info",
				"CREATE TABLE dimm_security_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 security_state INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_security_info_history",
				"CREATE TABLE dimm_security_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 security_state INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_sanitize_info",
				"CREATE TABLE dimm_sanitize_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 sanitize_state INTEGER  , \
					 sanitize_progress INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_sanitize_info_history",
				"CREATE TABLE dimm_sanitize_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 sanitize_state INTEGER , \
					 sanitize_progress INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_media_low_log_entry",
				"CREATE TABLE fw_media_low_log_entry (       \
					 device_handle INTEGER  , \
					 system_timestamp INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 dpa INTEGER  , \
					 pda INTEGER  , \
					 range INTEGER  , \
					 error_type INTEGER  , \
					 error_flags INTEGER  , \
					 transaction_type INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_media_low_log_entry_history",
				"CREATE TABLE fw_media_low_log_entry_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 system_timestamp INTEGER , \
					 dpa INTEGER , \
					 pda INTEGER , \
					 range INTEGER , \
					 error_type INTEGER , \
					 error_flags INTEGER , \
					 transaction_type INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_media_high_log_entry",
				"CREATE TABLE fw_media_high_log_entry (       \
					 device_handle INTEGER  , \
					 system_timestamp INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 dpa INTEGER  , \
					 pda INTEGER  , \
					 range INTEGER  , \
					 error_type INTEGER  , \
					 error_flags INTEGER  , \
					 transaction_type INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_media_high_log_entry_history",
				"CREATE TABLE fw_media_high_log_entry_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 system_timestamp INTEGER , \
					 dpa INTEGER , \
					 pda INTEGER , \
					 range INTEGER , \
					 error_type INTEGER , \
					 error_flags INTEGER , \
					 transaction_type INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_thermal_low_log_entry",
				"CREATE TABLE fw_thermal_low_log_entry (       \
					 device_handle INTEGER  , \
					 system_timestamp INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 host_reported_temp_data INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_thermal_low_log_entry_history",
				"CREATE TABLE fw_thermal_low_log_entry_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 system_timestamp INTEGER , \
					 host_reported_temp_data INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_thermal_high_log_entry",
				"CREATE TABLE fw_thermal_high_log_entry (       \
					 device_handle INTEGER  , \
					 system_timestamp INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 host_reported_temp_data INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_thermal_high_log_entry_history",
				"CREATE TABLE fw_thermal_high_log_entry_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 system_timestamp INTEGER , \
					 host_reported_temp_data INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_media_low_log_info",
				"CREATE TABLE fw_media_low_log_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 max_log_entries INTEGER  , \
					 new_log_entries INTEGER  , \
					 newest_log_entry_timestamp INTEGER  , \
					 oldest_log_entry_timestamp INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_media_low_log_info_history",
				"CREATE TABLE fw_media_low_log_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 max_log_entries INTEGER , \
					 new_log_entries INTEGER , \
					 newest_log_entry_timestamp INTEGER , \
					 oldest_log_entry_timestamp INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_media_high_log_info",
				"CREATE TABLE fw_media_high_log_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 max_log_entries INTEGER  , \
					 new_log_entries INTEGER  , \
					 newest_log_entry_timestamp INTEGER  , \
					 oldest_log_entry_timestamp INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_media_high_log_info_history",
				"CREATE TABLE fw_media_high_log_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 max_log_entries INTEGER , \
					 new_log_entries INTEGER , \
					 newest_log_entry_timestamp INTEGER , \
					 oldest_log_entry_timestamp INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_thermal_low_log_info",
				"CREATE TABLE fw_thermal_low_log_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 max_log_entries INTEGER  , \
					 new_log_entries INTEGER  , \
					 newest_log_entry_timestamp INTEGER  , \
					 oldest_log_entry_timestamp INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_thermal_low_log_info_history",
				"CREATE TABLE fw_thermal_low_log_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 max_log_entries INTEGER , \
					 new_log_entries INTEGER , \
					 newest_log_entry_timestamp INTEGER , \
					 oldest_log_entry_timestamp INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"fw_thermal_high_log_info",
				"CREATE TABLE fw_thermal_high_log_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 max_log_entries INTEGER  , \
					 new_log_entries INTEGER  , \
					 newest_log_entry_timestamp INTEGER  , \
					 oldest_log_entry_timestamp INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"fw_thermal_high_log_info_history",
				"CREATE TABLE fw_thermal_high_log_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 max_log_entries INTEGER , \
					 new_log_entries INTEGER , \
					 newest_log_entry_timestamp INTEGER , \
					 oldest_log_entry_timestamp INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_fw_log_level",
				"CREATE TABLE dimm_fw_log_level (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 log_level INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_fw_log_level_history",
				"CREATE TABLE dimm_fw_log_level_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 log_level INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_fw_time",
				"CREATE TABLE dimm_fw_time (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 time INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_fw_time_history",
				"CREATE TABLE dimm_fw_time_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 time INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_platform_config",
				"CREATE TABLE dimm_platform_config (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 signature TEXT  , \
					 length INTEGER  , \
					 revision INTEGER  , \
					 checksum INTEGER  , \
					 oem_id TEXT  , \
					 oem_table_id TEXT  , \
					 oem_revision INTEGER  , \
					 creator_id INTEGER  , \
					 creator_revision INTEGER  , \
					 current_config_size INTEGER  , \
					 current_config_offset INTEGER  , \
					 config_input_size INTEGER  , \
					 config_input_offset INTEGER  , \
					 config_output_size INTEGER  , \
					 config_output_offset INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_platform_config_history",
				"CREATE TABLE dimm_platform_config_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 signature TEXT , \
					 length INTEGER , \
					 revision INTEGER , \
					 checksum INTEGER , \
					 oem_id TEXT , \
					 oem_table_id TEXT , \
					 oem_revision INTEGER , \
					 creator_id INTEGER , \
					 creator_revision INTEGER , \
					 current_config_size INTEGER , \
					 current_config_offset INTEGER , \
					 config_input_size INTEGER , \
					 config_input_offset INTEGER , \
					 config_output_size INTEGER , \
					 config_output_offset INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_current_config",
				"CREATE TABLE dimm_current_config (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 signature TEXT  , \
					 length INTEGER  , \
					 revision INTEGER  , \
					 checksum INTEGER  , \
					 oem_id TEXT  , \
					 oem_table_id TEXT  , \
					 oem_revision INTEGER  , \
					 creator_id INTEGER  , \
					 creator_revision INTEGER  , \
					 config_status INTEGER  , \
					 mapped_memory_capacity INTEGER  , \
					 mapped_app_direct_capacity INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_current_config_history",
				"CREATE TABLE dimm_current_config_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 signature TEXT , \
					 length INTEGER , \
					 revision INTEGER , \
					 checksum INTEGER , \
					 oem_id TEXT , \
					 oem_table_id TEXT , \
					 oem_revision INTEGER , \
					 creator_id INTEGER , \
					 creator_revision INTEGER , \
					 config_status INTEGER , \
					 mapped_memory_capacity INTEGER , \
					 mapped_app_direct_capacity INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_config_input",
				"CREATE TABLE dimm_config_input (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 signature TEXT  , \
					 length INTEGER  , \
					 revision INTEGER  , \
					 checksum INTEGER  , \
					 oem_id TEXT  , \
					 oem_table_id TEXT  , \
					 oem_revision INTEGER  , \
					 creator_id INTEGER  , \
					 creator_revision INTEGER  , \
					 sequence_number INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_config_input_history",
				"CREATE TABLE dimm_config_input_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 signature TEXT , \
					 length INTEGER , \
					 revision INTEGER , \
					 checksum INTEGER , \
					 oem_id TEXT , \
					 oem_table_id TEXT , \
					 oem_revision INTEGER , \
					 creator_id INTEGER , \
					 creator_revision INTEGER , \
					 sequence_number INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_config_output",
				"CREATE TABLE dimm_config_output (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 signature TEXT  , \
					 length INTEGER  , \
					 revision INTEGER  , \
					 checksum INTEGER  , \
					 oem_id TEXT  , \
					 oem_table_id TEXT  , \
					 oem_revision INTEGER  , \
					 creator_id INTEGER  , \
					 creator_revision INTEGER  , \
					 sequence_number INTEGER  , \
					 validation_status INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_config_output_history",
				"CREATE TABLE dimm_config_output_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 signature TEXT , \
					 length INTEGER , \
					 revision INTEGER , \
					 checksum INTEGER , \
					 oem_id TEXT , \
					 oem_table_id TEXT , \
					 oem_revision INTEGER , \
					 creator_id INTEGER , \
					 creator_revision INTEGER , \
					 sequence_number INTEGER , \
					 validation_status INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_partition_change",
				"CREATE TABLE dimm_partition_change (       \
					 device_handle INTEGER  , \
					 id INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 config_table_type INTEGER  , \
					 extension_table_type INTEGER  , \
					 length INTEGER  , \
					 partition_size INTEGER  , \
					 status INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_partition_change_history",
				"CREATE TABLE dimm_partition_change_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 id INTEGER , \
					 config_table_type INTEGER , \
					 extension_table_type INTEGER , \
					 length INTEGER , \
					 partition_size INTEGER , \
					 status INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"dimm_interleave_set",
				"CREATE TABLE dimm_interleave_set (       \
					 id INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 device_handle INTEGER  , \
					 config_table_type INTEGER  , \
					 extension_table_type INTEGER  , \
					 length INTEGER  , \
					 index_id INTEGER  , \
					 dimm_count INTEGER  , \
					 memory_type INTEGER  , \
					 interleave_format INTEGER  , \
					 mirror_enable INTEGER  , \
					 status INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"dimm_interleave_set_history",
				"CREATE TABLE dimm_interleave_set_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 device_handle INTEGER , \
					 config_table_type INTEGER , \
					 extension_table_type INTEGER , \
					 length INTEGER , \
					 index_id INTEGER , \
					 dimm_count INTEGER , \
					 memory_type INTEGER , \
					 interleave_format INTEGER , \
					 mirror_enable INTEGER , \
					 status INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"interleave_set_dimm_info",
				"CREATE TABLE interleave_set_dimm_info (       \
					 id INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 config_table_type INTEGER  , \
					 index_id INTEGER  , \
					 device_handle INTEGER  , \
					 manufacturer INTEGER  , \
					 serial_num INTEGER  , \
					 model_num TEXT  , \
					 offset INTEGER  , \
					 size INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"interleave_set_dimm_info_history",
				"CREATE TABLE interleave_set_dimm_info_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 config_table_type INTEGER , \
					 index_id INTEGER , \
					 device_handle INTEGER , \
					 manufacturer INTEGER , \
					 serial_num INTEGER , \
					 model_num TEXT , \
					 offset INTEGER , \
					 size INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"enable_error_injection_info",
				"CREATE TABLE enable_error_injection_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 enable INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"enable_error_injection_info_history",
				"CREATE TABLE enable_error_injection_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 enable INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"temperature_error_injection_info",
				"CREATE TABLE temperature_error_injection_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 temperature INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"temperature_error_injection_info_history",
				"CREATE TABLE temperature_error_injection_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 temperature INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"poison_error_injection_info",
				"CREATE TABLE poison_error_injection_info (       \
					 id INTEGER  PRIMARY KEY  AUTOINCREMENT  NOT NULL UNIQUE  , \
					 device_handle INTEGER  , \
					 dpa_address INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"poison_error_injection_info_history",
				"CREATE TABLE poison_error_injection_info_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 device_handle INTEGER , \
					 dpa_address INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"software_trigger_info",
				"CREATE TABLE software_trigger_info (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 die_sparing_trigger INTEGER  , \
					 user_spare_block_alarm_trip_trigger INTEGER  , \
					 fatal_error_trigger INTEGER   \
					);"});
			tables[populate_index++] = ((struct table){"software_trigger_info_history",
				"CREATE TABLE software_trigger_info_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 die_sparing_trigger INTEGER , \
					 user_spare_block_alarm_trip_trigger INTEGER , \
					 fatal_error_trigger INTEGER  \
					);"});
tables[populate_index++] = ((struct table){"performance",
				"CREATE TABLE performance (       \
					 id INTEGER  PRIMARY KEY  AUTOINCREMENT  NOT NULL UNIQUE  , \
					 dimm_uid TEXT  , \
					 time INTEGER  , \
					 bytes_read INTEGER  , \
					 bytes_written INTEGER  , \
					 read_reqs INTEGER  , \
					 host_write_cmds INTEGER  , \
					 block_reads INTEGER  , \
					 block_writes INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"performance_history",
				"CREATE TABLE performance_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 dimm_uid TEXT , \
					 time INTEGER , \
					 bytes_read INTEGER , \
					 bytes_written INTEGER , \
					 read_reqs INTEGER , \
					 host_write_cmds INTEGER , \
					 block_reads INTEGER , \
					 block_writes INTEGER  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"driver_metadata_check_diag_result",
				"CREATE TABLE driver_metadata_check_diag_result (       \
					 id INTEGER  PRIMARY KEY  AUTOINCREMENT  NOT NULL UNIQUE  , \
					 result_type INTEGER  , \
					 ns_uid TEXT  , \
					 device_handle INTEGER  , \
					 health_flag INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"driver_metadata_check_diag_result_history",
				"CREATE TABLE driver_metadata_check_diag_result_history (       \
					history_id INTEGER NOT NULL, \
					 id INTEGER , \
					 result_type INTEGER , \
					 ns_uid TEXT , \
					 device_handle INTEGER , \
					 health_flag INTEGER  \
					);"}
#endif
);
tables[populate_index++] = ((struct table){"boot_status_register",
				"CREATE TABLE boot_status_register (       \
					 device_handle INTEGER  PRIMARY KEY  NOT NULL UNIQUE  , \
					 bsr INTEGER   \
					);"}
#if 0
//NON-HISTORY TABLE
);
			tables[populate_index++] = ((struct table){"boot_status_register_history",
				"CREATE TABLE boot_status_register_history (       \
					history_id INTEGER NOT NULL, \
					 device_handle INTEGER , \
					 bsr INTEGER  \
					);"}
#endif
);
			if (sqlite3_open_v2(path, &(result->db),
				SQLITE_OPEN_READWRITE|SQLITE_OPEN_CREATE|SQLITE_OPEN_FULLMUTEX, NULL) == SQLITE_OK)
			{
				for (int i = 0; i < TABLE_COUNT; i++)
				{
					if (!table_exists(result->db, tables[i].table_name))
					{
						run_sql_no_results(result->db, tables[i].create_statement);
					}
				}
			}
			else
			{
				free_PersistentStore(&result);
			}
			free(tables);
		}
	}
	return result;
}
/*
 * Create an array containing all history table names
 */
char* history_table_names[] = {

#if 0
//NON-HISTORY TABLE

	"config_history",

#endif

#if 0
//NON-HISTORY TABLE

	"log_history",

#endif

#if 0
//NON-HISTORY TABLE

	"event_history",

#endif

#if 0
//NON-HISTORY TABLE

	"topology_state_history",

#endif

	"host_history",

	"sw_inventory_history",

	"socket_history",

	"runtime_config_validation_history",

	"interleave_capability_history",

	"platform_info_capability_history",

	"platform_capabilities_history",

	"driver_capabilities_history",

	"driver_features_history",

	"dimm_topology_history",

	"namespace_history",

	"identify_dimm_history",

#if 0
//NON-HISTORY TABLE

	"device_characteristics_history",

#endif

	"dimm_partition_history",

	"dimm_smart_history",

#if 0
//NON-HISTORY TABLE

	"dimm_state_history",

#endif

#if 0
//NON-HISTORY TABLE

	"namespace_state_history",

#endif

	"dimm_alarm_thresholds_history",

	"dimm_power_management_history",

	"dimm_die_sparing_history",

	"dimm_optional_config_data_history",

	"dimm_err_correction_history",

	"dimm_erasure_coding_history",

	"dimm_thermal_history",

	"dimm_fw_image_history",

	"dimm_fw_debug_log_history",

	"dimm_memory_info_page0_history",

	"dimm_memory_info_page1_history",

	"dimm_memory_info_page2_history",

	"dimm_ars_command_specific_data_history",

	"dimm_long_op_status_history",

	"dimm_details_history",

	"dimm_security_info_history",

	"dimm_sanitize_info_history",

	"fw_media_low_log_entry_history",

	"fw_media_high_log_entry_history",

	"fw_thermal_low_log_entry_history",

	"fw_thermal_high_log_entry_history",

	"fw_media_low_log_info_history",

	"fw_media_high_log_info_history",

	"fw_thermal_low_log_info_history",

	"fw_thermal_high_log_info_history",

	"dimm_fw_log_level_history",

	"dimm_fw_time_history",

	"dimm_platform_config_history",

	"dimm_current_config_history",

	"dimm_config_input_history",

	"dimm_config_output_history",

	"dimm_partition_change_history",

	"dimm_interleave_set_history",

	"interleave_set_dimm_info_history",

	"enable_error_injection_info_history",

	"temperature_error_injection_info_history",

	"poison_error_injection_info_history",

	"software_trigger_info_history",

#if 0
//NON-HISTORY TABLE

	"performance_history",

#endif

#if 0
//NON-HISTORY TABLE

	"driver_metadata_check_diag_result_history",

#endif

#if 0
//NON-HISTORY TABLE

	"boot_status_register_history",

#endif

	"history",
	"\0"
};
/*
 * --- config ----------------
 */
void local_bind_config(sqlite3_stmt *p_stmt, struct db_config *p_config)
{
	BIND_TEXT(p_stmt, "$key", (char *)p_config->key);
	BIND_TEXT(p_stmt, "$value", (char *)p_config->value);
}
void local_get_config_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_config *p_config)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_config_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_config *p_config,
	int history_id)
{
}

#endif

void local_row_to_config(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_config *p_config)
{
	TEXT_COLUMN(p_stmt,
		0,
		p_config->key,
		CONFIG_KEY_LEN);
	TEXT_COLUMN(p_stmt,
		1,
		p_config->value,
		CONFIG_VALUE_LEN);
}
void db_print_config(struct db_config *p_value)
{
	printf("config.key: %s\n", p_value->key);
	printf("config.value: %s\n", p_value->value);
}
enum db_return_codes db_add_config(const PersistentStore *p_ps,
	struct db_config *p_config)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO config \
		(key, value)  \
		VALUES 		\
		($key, \
		$value) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_config(p_stmt, p_config);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_config_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "config", p_count);
}
int db_get_configs(const PersistentStore *p_ps,
	struct db_config *p_config,
	int config_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_config, 0, sizeof (struct db_config) * config_count);
	char *sql = "SELECT \
		key \
		,  value \
		  \
		FROM config \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < config_count)
		{
			local_row_to_config(p_ps, p_stmt, &p_config[index]);
			local_get_config_relationships(p_ps, p_stmt, &p_config[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_configs(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM config");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_config_state(const PersistentStore *p_ps,
	int history_id,
	struct db_config *p_config)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_config temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_config_by_key(p_ps, p_config->key, &temp) == DB_SUCCESS)
	{
		rc = db_update_config_by_key(p_ps,
				p_config->key,
				p_config);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO config \
			( key ,  value )  \
			VALUES 		\
			($key, \
			$value) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_config(p_stmt, p_config);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO config_history \
			(history_id, \
				 key,  value)  \
			VALUES 		($history_id, \
				 $key , \
				 $value )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_config(p_stmt, p_config);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_config_by_key(const PersistentStore *p_ps,
	const char * key,
	struct db_config *p_config)
{
	memset(p_config, 0, sizeof (struct db_config));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		key,  value  \
		FROM config \
		WHERE  key = $key";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$key", (char *)key);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_config(p_ps, p_stmt, p_config);
			local_get_config_relationships(p_ps, p_stmt, p_config);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_config_by_key(const PersistentStore *p_ps,
	const char * key,
	struct db_config *p_config)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE config \
	SET \
	key=$key \
		,  value=$value \
		  \
	WHERE key=$key ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$key", (char *)key);
		local_bind_config(p_stmt, p_config);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_config_by_key(const PersistentStore *p_ps,
	const char * key)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM config \
				 WHERE key = $key";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$key", (char *)key);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_config_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM config_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_config_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM config_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_config_history_by_history_id(const PersistentStore *p_ps,
	struct db_config *p_config,
	int history_id,
	int config_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_config, 0, sizeof (struct db_config) * config_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		key,  value  \
		FROM config_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < config_count)
		{
		rc = DB_SUCCESS;
		local_row_to_config(p_ps, p_stmt, &p_config[index]);
		local_get_config_relationships_history(p_ps, p_stmt, &p_config[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_config_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM config_history");
}

#endif

/*
 * --- END config ----------------
 */
/*
 * --- log ----------------
 */
void local_bind_log(sqlite3_stmt *p_stmt, struct db_log *p_log)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_log->id);
	BIND_INTEGER(p_stmt, "$thread_id", (unsigned long long)p_log->thread_id);
	BIND_INTEGER(p_stmt, "$time", (unsigned long long)p_log->time);
	BIND_INTEGER(p_stmt, "$level", (int)p_log->level);
	BIND_TEXT(p_stmt, "$file_name", (char *)p_log->file_name);
	BIND_INTEGER(p_stmt, "$line_number", (unsigned int)p_log->line_number);
	BIND_TEXT(p_stmt, "$message", (char *)p_log->message);
}
void local_get_log_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_log *p_log)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_log_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_log *p_log,
	int history_id)
{
}

#endif

void local_row_to_log(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_log *p_log)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_log->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_log->thread_id);
	INTEGER_COLUMN(p_stmt,
		2,
		p_log->time);
	INTEGER_COLUMN(p_stmt,
		3,
		p_log->level);
	TEXT_COLUMN(p_stmt,
		4,
		p_log->file_name,
		LOG_FILE_NAME_LEN);
	INTEGER_COLUMN(p_stmt,
		5,
		p_log->line_number);
	TEXT_COLUMN(p_stmt,
		6,
		p_log->message,
		LOG_MESSAGE_LEN);
}
void db_print_log(struct db_log *p_value)
{
	printf("log.id: %d\n", p_value->id);
	printf("log.thread_id: unsigned %lld\n", p_value->thread_id);
	printf("log.time: unsigned %lld\n", p_value->time);
	printf("log.level: %d\n", p_value->level);
	printf("log.file_name: %s\n", p_value->file_name);
	printf("log.line_number: unsigned %d\n", p_value->line_number);
	printf("log.message: %s\n", p_value->message);
}
enum db_return_codes db_add_log(const PersistentStore *p_ps,
	struct db_log *p_log)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO log \
		(thread_id, time, level, file_name, line_number, message)  \
		VALUES 		\
		(\
		$thread_id, \
		$time, \
		$level, \
		$file_name, \
		$line_number, \
		$message) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_log(p_stmt, p_log);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_log_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "log", p_count);
}
int db_get_logs(const PersistentStore *p_ps,
	struct db_log *p_log,
	int log_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_log, 0, sizeof (struct db_log) * log_count);
	char *sql = "SELECT \
		id \
		,  thread_id \
		,  time \
		,  level \
		,  file_name \
		,  line_number \
		,  message \
		  \
		FROM log \
		        \
		 ORDER BY id DESC  \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < log_count)
		{
			local_row_to_log(p_ps, p_stmt, &p_log[index]);
			local_get_log_relationships(p_ps, p_stmt, &p_log[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_logs(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM log");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_log_state(const PersistentStore *p_ps,
	int history_id,
	struct db_log *p_log)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_log temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_log_by_id(p_ps, p_log->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_log_by_id(p_ps,
				p_log->id,
				p_log);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO log \
			( id ,  thread_id ,  time ,  level ,  file_name ,  line_number ,  message )  \
			VALUES 		\
			($id, \
			$thread_id, \
			$time, \
			$level, \
			$file_name, \
			$line_number, \
			$message) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_log(p_stmt, p_log);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO log_history \
			(history_id, \
				 id,  thread_id,  time,  level,  file_name,  line_number,  message)  \
			VALUES 		($history_id, \
				 $id , \
				 $thread_id , \
				 $time , \
				 $level , \
				 $file_name , \
				 $line_number , \
				 $message )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_log(p_stmt, p_log);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_log_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_log *p_log)
{
	memset(p_log, 0, sizeof (struct db_log));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  thread_id,  time,  level,  file_name,  line_number,  message  \
		FROM log \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_log(p_ps, p_stmt, p_log);
			local_get_log_relationships(p_ps, p_stmt, p_log);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_log_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_log *p_log)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE log \
	SET \
	id=$id \
		,  thread_id=$thread_id \
		,  time=$time \
		,  level=$level \
		,  file_name=$file_name \
		,  line_number=$line_number \
		,  message=$message \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_log(p_stmt, p_log);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_log_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM log \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_log_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM log_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_log_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM log_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_log_history_by_history_id(const PersistentStore *p_ps,
	struct db_log *p_log,
	int history_id,
	int log_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_log, 0, sizeof (struct db_log) * log_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  thread_id,  time,  level,  file_name,  line_number,  message  \
		FROM log_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < log_count)
		{
		rc = DB_SUCCESS;
		local_row_to_log(p_ps, p_stmt, &p_log[index]);
		local_get_log_relationships_history(p_ps, p_stmt, &p_log[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_log_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM log_history");
}

#endif

/*
 * --- END log ----------------
 */
/*
 * --- event ----------------
 */
void local_bind_event(sqlite3_stmt *p_stmt, struct db_event *p_event)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_event->id);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_event->type);
	BIND_INTEGER(p_stmt, "$severity", (unsigned int)p_event->severity);
	BIND_INTEGER(p_stmt, "$code", (unsigned int)p_event->code);
	BIND_INTEGER(p_stmt, "$action_required", (unsigned int)p_event->action_required);
	BIND_TEXT(p_stmt, "$uid", (char *)p_event->uid);
	BIND_INTEGER(p_stmt, "$time", (unsigned long long)p_event->time);
	BIND_TEXT(p_stmt, "$arg1", (char *)p_event->arg1);
	BIND_TEXT(p_stmt, "$arg2", (char *)p_event->arg2);
	BIND_TEXT(p_stmt, "$arg3", (char *)p_event->arg3);
	BIND_INTEGER(p_stmt, "$diag_result", (unsigned int)p_event->diag_result);
}
void local_get_event_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_event *p_event)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_event_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_event *p_event,
	int history_id)
{
}

#endif

void local_row_to_event(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_event *p_event)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_event->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_event->type);
	INTEGER_COLUMN(p_stmt,
		2,
		p_event->severity);
	INTEGER_COLUMN(p_stmt,
		3,
		p_event->code);
	INTEGER_COLUMN(p_stmt,
		4,
		p_event->action_required);
	TEXT_COLUMN(p_stmt,
		5,
		p_event->uid,
		EVENT_UID_LEN);
	INTEGER_COLUMN(p_stmt,
		6,
		p_event->time);
	TEXT_COLUMN(p_stmt,
		7,
		p_event->arg1,
		EVENT_ARG1_LEN);
	TEXT_COLUMN(p_stmt,
		8,
		p_event->arg2,
		EVENT_ARG2_LEN);
	TEXT_COLUMN(p_stmt,
		9,
		p_event->arg3,
		EVENT_ARG3_LEN);
	INTEGER_COLUMN(p_stmt,
		10,
		p_event->diag_result);
}
void db_print_event(struct db_event *p_value)
{
	printf("event.id: %d\n", p_value->id);
	printf("event.type: unsigned %d\n", p_value->type);
	printf("event.severity: unsigned %d\n", p_value->severity);
	printf("event.code: unsigned %d\n", p_value->code);
	printf("event.action_required: unsigned %d\n", p_value->action_required);
	printf("event.uid: %s\n", p_value->uid);
	printf("event.time: unsigned %lld\n", p_value->time);
	printf("event.arg1: %s\n", p_value->arg1);
	printf("event.arg2: %s\n", p_value->arg2);
	printf("event.arg3: %s\n", p_value->arg3);
	printf("event.diag_result: unsigned %d\n", p_value->diag_result);
}
enum db_return_codes db_add_event(const PersistentStore *p_ps,
	struct db_event *p_event)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO event \
		(type, severity, code, action_required, uid, time, arg1, arg2, arg3, diag_result)  \
		VALUES 		\
		(\
		$type, \
		$severity, \
		$code, \
		$action_required, \
		$uid, \
		$time, \
		$arg1, \
		$arg2, \
		$arg3, \
		$diag_result) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_event(p_stmt, p_event);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_event_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "event", p_count);
}
int db_get_events(const PersistentStore *p_ps,
	struct db_event *p_event,
	int event_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_event, 0, sizeof (struct db_event) * event_count);
	char *sql = "SELECT \
		id \
		,  type \
		,  severity \
		,  code \
		,  action_required \
		,  uid \
		,  time \
		,  arg1 \
		,  arg2 \
		,  arg3 \
		,  diag_result \
		  \
		FROM event \
		            \
		 ORDER BY id DESC  \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < event_count)
		{
			local_row_to_event(p_ps, p_stmt, &p_event[index]);
			local_get_event_relationships(p_ps, p_stmt, &p_event[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_events(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM event");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_event_state(const PersistentStore *p_ps,
	int history_id,
	struct db_event *p_event)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_event temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_event_by_id(p_ps, p_event->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_event_by_id(p_ps,
				p_event->id,
				p_event);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO event \
			( id ,  type ,  severity ,  code ,  action_required ,  uid ,  time ,  arg1 ,  arg2 ,  arg3 ,  diag_result )  \
			VALUES 		\
			($id, \
			$type, \
			$severity, \
			$code, \
			$action_required, \
			$uid, \
			$time, \
			$arg1, \
			$arg2, \
			$arg3, \
			$diag_result) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_event(p_stmt, p_event);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO event_history \
			(history_id, \
				 id,  type,  severity,  code,  action_required,  uid,  time,  arg1,  arg2,  arg3,  diag_result)  \
			VALUES 		($history_id, \
				 $id , \
				 $type , \
				 $severity , \
				 $code , \
				 $action_required , \
				 $uid , \
				 $time , \
				 $arg1 , \
				 $arg2 , \
				 $arg3 , \
				 $diag_result )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_event(p_stmt, p_event);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_event_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_event *p_event)
{
	memset(p_event, 0, sizeof (struct db_event));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  type,  severity,  code,  action_required,  uid,  time,  arg1,  arg2,  arg3,  diag_result  \
		FROM event \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_event(p_ps, p_stmt, p_event);
			local_get_event_relationships(p_ps, p_stmt, p_event);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_event_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_event *p_event)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE event \
	SET \
	id=$id \
		,  type=$type \
		,  severity=$severity \
		,  code=$code \
		,  action_required=$action_required \
		,  uid=$uid \
		,  time=$time \
		,  arg1=$arg1 \
		,  arg2=$arg2 \
		,  arg3=$arg3 \
		,  diag_result=$diag_result \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_event(p_stmt, p_event);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_event_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM event \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_event_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM event_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_event_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM event_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_event_history_by_history_id(const PersistentStore *p_ps,
	struct db_event *p_event,
	int history_id,
	int event_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_event, 0, sizeof (struct db_event) * event_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  type,  severity,  code,  action_required,  uid,  time,  arg1,  arg2,  arg3,  diag_result  \
		FROM event_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < event_count)
		{
		rc = DB_SUCCESS;
		local_row_to_event(p_ps, p_stmt, &p_event[index]);
		local_get_event_relationships_history(p_ps, p_stmt, &p_event[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_event_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM event_history");
}

#endif

enum db_return_codes db_get_event_count_by_event_type_type(
	const PersistentStore *p_ps,
	const unsigned int type,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM event WHERE type = $type";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$type", (unsigned int)type);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_event_count_by_event_type_type_history(
	const PersistentStore *p_ps,
	const unsigned int type,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM event_history "
		"WHERE type = $type "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$type", (unsigned int)type);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_events_by_event_type_type(const PersistentStore *p_ps,
	unsigned int type,
	struct db_event *p_event,
	int event_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  type ,  severity ,  code ,  action_required ,  uid ,  time ,  arg1 ,  arg2 ,  arg3 ,  diag_result  \
		FROM event \
		WHERE  type = $type";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$type", (unsigned int)type);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < event_count)
		{
			local_row_to_event(p_ps, p_stmt, &p_event[index]);
			local_get_event_relationships(p_ps, p_stmt, &p_event[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_events_by_event_type_type_history(const PersistentStore *p_ps,
	unsigned int type,
	struct db_event *p_event,
	int event_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  type ,  severity ,  code ,  action_required ,  uid ,  time ,  arg1 ,  arg2 ,  arg3 ,  diag_result  \
		FROM event_history \
		WHERE  type = $type AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$type", (unsigned int)type);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < event_count)
		{
			local_row_to_event(p_ps, p_stmt, &p_event[index]);
			local_get_event_relationships(p_ps, p_stmt, &p_event[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_event_by_event_type_type(const PersistentStore *p_ps,
	unsigned int type)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM event \
				 WHERE type = $type";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$type", (unsigned int)type);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END event ----------------
 */
/*
 * --- topology_state ----------------
 */
void local_bind_topology_state(sqlite3_stmt *p_stmt, struct db_topology_state *p_topology_state)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_topology_state->device_handle);
	BIND_TEXT(p_stmt, "$uid", (char *)p_topology_state->uid);
	BIND_INTEGER(p_stmt, "$manufacturer", (unsigned int)p_topology_state->manufacturer);
	BIND_INTEGER(p_stmt, "$serial_num", (unsigned int)p_topology_state->serial_num);
	BIND_TEXT(p_stmt, "$model_num", (char *)p_topology_state->model_num);
	BIND_INTEGER(p_stmt, "$current_config_status", (int)p_topology_state->current_config_status);
	BIND_INTEGER(p_stmt, "$config_goal_status", (int)p_topology_state->config_goal_status);
}
void local_get_topology_state_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_topology_state *p_topology_state)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_topology_state_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_topology_state *p_topology_state,
	int history_id)
{
}

#endif

void local_row_to_topology_state(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_topology_state *p_topology_state)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_topology_state->device_handle);
	TEXT_COLUMN(p_stmt,
		1,
		p_topology_state->uid,
		TOPOLOGY_STATE_UID_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_topology_state->manufacturer);
	INTEGER_COLUMN(p_stmt,
		3,
		p_topology_state->serial_num);
	TEXT_COLUMN(p_stmt,
		4,
		p_topology_state->model_num,
		TOPOLOGY_STATE_MODEL_NUM_LEN);
	INTEGER_COLUMN(p_stmt,
		5,
		p_topology_state->current_config_status);
	INTEGER_COLUMN(p_stmt,
		6,
		p_topology_state->config_goal_status);
}
void db_print_topology_state(struct db_topology_state *p_value)
{
	printf("topology_state.device_handle: unsigned %d\n", p_value->device_handle);
	printf("topology_state.uid: %s\n", p_value->uid);
	printf("topology_state.manufacturer: unsigned %d\n", p_value->manufacturer);
	printf("topology_state.serial_num: unsigned %d\n", p_value->serial_num);
	printf("topology_state.model_num: %s\n", p_value->model_num);
	printf("topology_state.current_config_status: %d\n", p_value->current_config_status);
	printf("topology_state.config_goal_status: %d\n", p_value->config_goal_status);
}
enum db_return_codes db_add_topology_state(const PersistentStore *p_ps,
	struct db_topology_state *p_topology_state)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO topology_state \
		(device_handle, uid, manufacturer, serial_num, model_num, current_config_status, config_goal_status)  \
		VALUES 		\
		($device_handle, \
		$uid, \
		$manufacturer, \
		$serial_num, \
		$model_num, \
		$current_config_status, \
		$config_goal_status) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_topology_state(p_stmt, p_topology_state);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_topology_state_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "topology_state", p_count);
}
int db_get_topology_states(const PersistentStore *p_ps,
	struct db_topology_state *p_topology_state,
	int topology_state_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_topology_state, 0, sizeof (struct db_topology_state) * topology_state_count);
	char *sql = "SELECT \
		device_handle \
		,  uid \
		,  manufacturer \
		,  serial_num \
		,  model_num \
		,  current_config_status \
		,  config_goal_status \
		  \
		FROM topology_state \
		        \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < topology_state_count)
		{
			local_row_to_topology_state(p_ps, p_stmt, &p_topology_state[index]);
			local_get_topology_state_relationships(p_ps, p_stmt, &p_topology_state[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_topology_states(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM topology_state");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_topology_state_state(const PersistentStore *p_ps,
	int history_id,
	struct db_topology_state *p_topology_state)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_topology_state temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_topology_state_by_device_handle(p_ps, p_topology_state->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_topology_state_by_device_handle(p_ps,
				p_topology_state->device_handle,
				p_topology_state);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO topology_state \
			( device_handle ,  uid ,  manufacturer ,  serial_num ,  model_num ,  current_config_status ,  config_goal_status )  \
			VALUES 		\
			($device_handle, \
			$uid, \
			$manufacturer, \
			$serial_num, \
			$model_num, \
			$current_config_status, \
			$config_goal_status) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_topology_state(p_stmt, p_topology_state);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO topology_state_history \
			(history_id, \
				 device_handle,  uid,  manufacturer,  serial_num,  model_num,  current_config_status,  config_goal_status)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $uid , \
				 $manufacturer , \
				 $serial_num , \
				 $model_num , \
				 $current_config_status , \
				 $config_goal_status )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_topology_state(p_stmt, p_topology_state);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_topology_state_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_topology_state *p_topology_state)
{
	memset(p_topology_state, 0, sizeof (struct db_topology_state));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  uid,  manufacturer,  serial_num,  model_num,  current_config_status,  config_goal_status  \
		FROM topology_state \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_topology_state(p_ps, p_stmt, p_topology_state);
			local_get_topology_state_relationships(p_ps, p_stmt, p_topology_state);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_topology_state_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_topology_state *p_topology_state)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE topology_state \
	SET \
	device_handle=$device_handle \
		,  uid=$uid \
		,  manufacturer=$manufacturer \
		,  serial_num=$serial_num \
		,  model_num=$model_num \
		,  current_config_status=$current_config_status \
		,  config_goal_status=$config_goal_status \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_topology_state(p_stmt, p_topology_state);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_topology_state_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM topology_state \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_topology_state_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM topology_state_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_topology_state_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM topology_state_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_topology_state_history_by_history_id(const PersistentStore *p_ps,
	struct db_topology_state *p_topology_state,
	int history_id,
	int topology_state_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_topology_state, 0, sizeof (struct db_topology_state) * topology_state_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  uid,  manufacturer,  serial_num,  model_num,  current_config_status,  config_goal_status  \
		FROM topology_state_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < topology_state_count)
		{
		rc = DB_SUCCESS;
		local_row_to_topology_state(p_ps, p_stmt, &p_topology_state[index]);
		local_get_topology_state_relationships_history(p_ps, p_stmt, &p_topology_state[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_topology_state_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM topology_state_history");
}

#endif

/*
 * --- END topology_state ----------------
 */
/*
 * --- host ----------------
 */
void local_bind_host(sqlite3_stmt *p_stmt, struct db_host *p_host)
{
	BIND_TEXT(p_stmt, "$name", (char *)p_host->name);
	BIND_INTEGER(p_stmt, "$os_type", (int)p_host->os_type);
	BIND_TEXT(p_stmt, "$os_name", (char *)p_host->os_name);
	BIND_TEXT(p_stmt, "$os_version", (char *)p_host->os_version);
}
void local_get_host_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_host *p_host)
{
}

void local_get_host_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_host *p_host,
	int history_id)
{
}

void local_row_to_host(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_host *p_host)
{
	TEXT_COLUMN(p_stmt,
		0,
		p_host->name,
		HOST_NAME_LEN);
	INTEGER_COLUMN(p_stmt,
		1,
		p_host->os_type);
	TEXT_COLUMN(p_stmt,
		2,
		p_host->os_name,
		HOST_OS_NAME_LEN);
	TEXT_COLUMN(p_stmt,
		3,
		p_host->os_version,
		HOST_OS_VERSION_LEN);
}
void db_print_host(struct db_host *p_value)
{
	printf("host.name: %s\n", p_value->name);
	printf("host.os_type: %d\n", p_value->os_type);
	printf("host.os_name: %s\n", p_value->os_name);
	printf("host.os_version: %s\n", p_value->os_version);
}
enum db_return_codes db_add_host(const PersistentStore *p_ps,
	struct db_host *p_host)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO host \
		(name, os_type, os_name, os_version)  \
		VALUES 		\
		($name, \
		$os_type, \
		$os_name, \
		$os_version) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_host(p_stmt, p_host);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_host_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "host", p_count);
}
int db_get_hosts(const PersistentStore *p_ps,
	struct db_host *p_host,
	int host_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_host, 0, sizeof (struct db_host) * host_count);
	char *sql = "SELECT \
		name \
		,  os_type \
		,  os_name \
		,  os_version \
		  \
		FROM host \
		     \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < host_count)
		{
			local_row_to_host(p_ps, p_stmt, &p_host[index]);
			local_get_host_relationships(p_ps, p_stmt, &p_host[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_hosts(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM host");
}

enum db_return_codes db_save_host_state(const PersistentStore *p_ps,
	int history_id,
	struct db_host *p_host)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_host temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_host_by_name(p_ps, p_host->name, &temp) == DB_SUCCESS)
	{
		rc = db_update_host_by_name(p_ps,
				p_host->name,
				p_host);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO host \
			( name ,  os_type ,  os_name ,  os_version )  \
			VALUES 		\
			($name, \
			$os_type, \
			$os_name, \
			$os_version) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_host(p_stmt, p_host);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO host_history \
			(history_id, \
				 name,  os_type,  os_name,  os_version)  \
			VALUES 		($history_id, \
				 $name , \
				 $os_type , \
				 $os_name , \
				 $os_version )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_host(p_stmt, p_host);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_host_by_name(const PersistentStore *p_ps,
	const char * name,
	struct db_host *p_host)
{
	memset(p_host, 0, sizeof (struct db_host));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		name,  os_type,  os_name,  os_version  \
		FROM host \
		WHERE  name = $name";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$name", (char *)name);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_host(p_ps, p_stmt, p_host);
			local_get_host_relationships(p_ps, p_stmt, p_host);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_host_by_name(const PersistentStore *p_ps,
	const char * name,
	struct db_host *p_host)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE host \
	SET \
	name=$name \
		,  os_type=$os_type \
		,  os_name=$os_name \
		,  os_version=$os_version \
		  \
	WHERE name=$name ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$name", (char *)name);
		local_bind_host(p_stmt, p_host);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_host_by_name(const PersistentStore *p_ps,
	const char * name)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM host \
				 WHERE name = $name";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$name", (char *)name);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_host_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM host_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_host_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM host_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_host_history_by_history_id(const PersistentStore *p_ps,
	struct db_host *p_host,
	int history_id,
	int host_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_host, 0, sizeof (struct db_host) * host_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		name,  os_type,  os_name,  os_version  \
		FROM host_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < host_count)
		{
		rc = DB_SUCCESS;
		local_row_to_host(p_ps, p_stmt, &p_host[index]);
		local_get_host_relationships_history(p_ps, p_stmt, &p_host[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_host_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM host_history");
}

/*
 * --- END host ----------------
 */
/*
 * --- sw_inventory ----------------
 */
void local_bind_sw_inventory(sqlite3_stmt *p_stmt, struct db_sw_inventory *p_sw_inventory)
{
	BIND_TEXT(p_stmt, "$name", (char *)p_sw_inventory->name);
	BIND_TEXT(p_stmt, "$mgmt_sw_rev", (char *)p_sw_inventory->mgmt_sw_rev);
	BIND_TEXT(p_stmt, "$vendor_driver_rev", (char *)p_sw_inventory->vendor_driver_rev);
	BIND_INTEGER(p_stmt, "$supported_driver_available", (unsigned int)p_sw_inventory->supported_driver_available);
}
void local_get_sw_inventory_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_sw_inventory *p_sw_inventory)
{
}

void local_get_sw_inventory_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_sw_inventory *p_sw_inventory,
	int history_id)
{
}

void local_row_to_sw_inventory(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_sw_inventory *p_sw_inventory)
{
	TEXT_COLUMN(p_stmt,
		0,
		p_sw_inventory->name,
		SW_INVENTORY_NAME_LEN);
	TEXT_COLUMN(p_stmt,
		1,
		p_sw_inventory->mgmt_sw_rev,
		SW_INVENTORY_MGMT_SW_REV_LEN);
	TEXT_COLUMN(p_stmt,
		2,
		p_sw_inventory->vendor_driver_rev,
		SW_INVENTORY_VENDOR_DRIVER_REV_LEN);
	INTEGER_COLUMN(p_stmt,
		3,
		p_sw_inventory->supported_driver_available);
}
void db_print_sw_inventory(struct db_sw_inventory *p_value)
{
	printf("sw_inventory.name: %s\n", p_value->name);
	printf("sw_inventory.mgmt_sw_rev: %s\n", p_value->mgmt_sw_rev);
	printf("sw_inventory.vendor_driver_rev: %s\n", p_value->vendor_driver_rev);
	printf("sw_inventory.supported_driver_available: unsigned %d\n", p_value->supported_driver_available);
}
enum db_return_codes db_add_sw_inventory(const PersistentStore *p_ps,
	struct db_sw_inventory *p_sw_inventory)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO sw_inventory \
		(name, mgmt_sw_rev, vendor_driver_rev, supported_driver_available)  \
		VALUES 		\
		($name, \
		$mgmt_sw_rev, \
		$vendor_driver_rev, \
		$supported_driver_available) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_sw_inventory(p_stmt, p_sw_inventory);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_sw_inventory_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "sw_inventory", p_count);
}
int db_get_sw_inventorys(const PersistentStore *p_ps,
	struct db_sw_inventory *p_sw_inventory,
	int sw_inventory_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_sw_inventory, 0, sizeof (struct db_sw_inventory) * sw_inventory_count);
	char *sql = "SELECT \
		name \
		,  mgmt_sw_rev \
		,  vendor_driver_rev \
		,  supported_driver_available \
		  \
		FROM sw_inventory \
		     \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < sw_inventory_count)
		{
			local_row_to_sw_inventory(p_ps, p_stmt, &p_sw_inventory[index]);
			local_get_sw_inventory_relationships(p_ps, p_stmt, &p_sw_inventory[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_sw_inventorys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM sw_inventory");
}

enum db_return_codes db_save_sw_inventory_state(const PersistentStore *p_ps,
	int history_id,
	struct db_sw_inventory *p_sw_inventory)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_sw_inventory temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_sw_inventory_by_name(p_ps, p_sw_inventory->name, &temp) == DB_SUCCESS)
	{
		rc = db_update_sw_inventory_by_name(p_ps,
				p_sw_inventory->name,
				p_sw_inventory);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO sw_inventory \
			( name ,  mgmt_sw_rev ,  vendor_driver_rev ,  supported_driver_available )  \
			VALUES 		\
			($name, \
			$mgmt_sw_rev, \
			$vendor_driver_rev, \
			$supported_driver_available) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_sw_inventory(p_stmt, p_sw_inventory);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO sw_inventory_history \
			(history_id, \
				 name,  mgmt_sw_rev,  vendor_driver_rev,  supported_driver_available)  \
			VALUES 		($history_id, \
				 $name , \
				 $mgmt_sw_rev , \
				 $vendor_driver_rev , \
				 $supported_driver_available )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_sw_inventory(p_stmt, p_sw_inventory);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_sw_inventory_by_name(const PersistentStore *p_ps,
	const char * name,
	struct db_sw_inventory *p_sw_inventory)
{
	memset(p_sw_inventory, 0, sizeof (struct db_sw_inventory));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		name,  mgmt_sw_rev,  vendor_driver_rev,  supported_driver_available  \
		FROM sw_inventory \
		WHERE  name = $name";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$name", (char *)name);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_sw_inventory(p_ps, p_stmt, p_sw_inventory);
			local_get_sw_inventory_relationships(p_ps, p_stmt, p_sw_inventory);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_sw_inventory_by_name(const PersistentStore *p_ps,
	const char * name,
	struct db_sw_inventory *p_sw_inventory)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE sw_inventory \
	SET \
	name=$name \
		,  mgmt_sw_rev=$mgmt_sw_rev \
		,  vendor_driver_rev=$vendor_driver_rev \
		,  supported_driver_available=$supported_driver_available \
		  \
	WHERE name=$name ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$name", (char *)name);
		local_bind_sw_inventory(p_stmt, p_sw_inventory);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_sw_inventory_by_name(const PersistentStore *p_ps,
	const char * name)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM sw_inventory \
				 WHERE name = $name";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$name", (char *)name);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_sw_inventory_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM sw_inventory_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_sw_inventory_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM sw_inventory_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_sw_inventory_history_by_history_id(const PersistentStore *p_ps,
	struct db_sw_inventory *p_sw_inventory,
	int history_id,
	int sw_inventory_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_sw_inventory, 0, sizeof (struct db_sw_inventory) * sw_inventory_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		name,  mgmt_sw_rev,  vendor_driver_rev,  supported_driver_available  \
		FROM sw_inventory_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < sw_inventory_count)
		{
		rc = DB_SUCCESS;
		local_row_to_sw_inventory(p_ps, p_stmt, &p_sw_inventory[index]);
		local_get_sw_inventory_relationships_history(p_ps, p_stmt, &p_sw_inventory[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_sw_inventory_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM sw_inventory_history");
}

/*
 * --- END sw_inventory ----------------
 */
/*
 * --- socket ----------------
 */
void local_bind_socket(sqlite3_stmt *p_stmt, struct db_socket *p_socket)
{
	BIND_INTEGER(p_stmt, "$socket_id", (unsigned int)p_socket->socket_id);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_socket->type);
	BIND_INTEGER(p_stmt, "$model", (unsigned int)p_socket->model);
	BIND_INTEGER(p_stmt, "$brand", (unsigned int)p_socket->brand);
	BIND_INTEGER(p_stmt, "$family", (unsigned int)p_socket->family);
	BIND_INTEGER(p_stmt, "$stepping", (unsigned int)p_socket->stepping);
	BIND_TEXT(p_stmt, "$manufacturer", (char *)p_socket->manufacturer);
	BIND_INTEGER(p_stmt, "$logical_processor_count", (unsigned int)p_socket->logical_processor_count);
	BIND_INTEGER(p_stmt, "$rapl_limited", (unsigned int)p_socket->rapl_limited);
}
void local_get_socket_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_socket *p_socket)
{
}

void local_get_socket_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_socket *p_socket,
	int history_id)
{
}

void local_row_to_socket(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_socket *p_socket)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_socket->socket_id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_socket->type);
	INTEGER_COLUMN(p_stmt,
		2,
		p_socket->model);
	INTEGER_COLUMN(p_stmt,
		3,
		p_socket->brand);
	INTEGER_COLUMN(p_stmt,
		4,
		p_socket->family);
	INTEGER_COLUMN(p_stmt,
		5,
		p_socket->stepping);
	TEXT_COLUMN(p_stmt,
		6,
		p_socket->manufacturer,
		SOCKET_MANUFACTURER_LEN);
	INTEGER_COLUMN(p_stmt,
		7,
		p_socket->logical_processor_count);
	INTEGER_COLUMN(p_stmt,
		8,
		p_socket->rapl_limited);
}
void db_print_socket(struct db_socket *p_value)
{
	printf("socket.socket_id: unsigned %d\n", p_value->socket_id);
	printf("socket.type: unsigned %d\n", p_value->type);
	printf("socket.model: unsigned %d\n", p_value->model);
	printf("socket.brand: unsigned %d\n", p_value->brand);
	printf("socket.family: unsigned %d\n", p_value->family);
	printf("socket.stepping: unsigned %d\n", p_value->stepping);
	printf("socket.manufacturer: %s\n", p_value->manufacturer);
	printf("socket.logical_processor_count: unsigned %d\n", p_value->logical_processor_count);
	printf("socket.rapl_limited: unsigned %d\n", p_value->rapl_limited);
}
enum db_return_codes db_add_socket(const PersistentStore *p_ps,
	struct db_socket *p_socket)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO socket \
		(socket_id, type, model, brand, family, stepping, manufacturer, logical_processor_count, rapl_limited)  \
		VALUES 		\
		($socket_id, \
		$type, \
		$model, \
		$brand, \
		$family, \
		$stepping, \
		$manufacturer, \
		$logical_processor_count, \
		$rapl_limited) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_socket(p_stmt, p_socket);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_socket_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "socket", p_count);
}
int db_get_sockets(const PersistentStore *p_ps,
	struct db_socket *p_socket,
	int socket_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_socket, 0, sizeof (struct db_socket) * socket_count);
	char *sql = "SELECT \
		socket_id \
		,  type \
		,  model \
		,  brand \
		,  family \
		,  stepping \
		,  manufacturer \
		,  logical_processor_count \
		,  rapl_limited \
		  \
		FROM socket \
		          \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < socket_count)
		{
			local_row_to_socket(p_ps, p_stmt, &p_socket[index]);
			local_get_socket_relationships(p_ps, p_stmt, &p_socket[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_sockets(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM socket");
}

enum db_return_codes db_save_socket_state(const PersistentStore *p_ps,
	int history_id,
	struct db_socket *p_socket)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_socket temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_socket_by_socket_id(p_ps, p_socket->socket_id, &temp) == DB_SUCCESS)
	{
		rc = db_update_socket_by_socket_id(p_ps,
				p_socket->socket_id,
				p_socket);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO socket \
			( socket_id ,  type ,  model ,  brand ,  family ,  stepping ,  manufacturer ,  logical_processor_count ,  rapl_limited )  \
			VALUES 		\
			($socket_id, \
			$type, \
			$model, \
			$brand, \
			$family, \
			$stepping, \
			$manufacturer, \
			$logical_processor_count, \
			$rapl_limited) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_socket(p_stmt, p_socket);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO socket_history \
			(history_id, \
				 socket_id,  type,  model,  brand,  family,  stepping,  manufacturer,  logical_processor_count,  rapl_limited)  \
			VALUES 		($history_id, \
				 $socket_id , \
				 $type , \
				 $model , \
				 $brand , \
				 $family , \
				 $stepping , \
				 $manufacturer , \
				 $logical_processor_count , \
				 $rapl_limited )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_socket(p_stmt, p_socket);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_socket_by_socket_id(const PersistentStore *p_ps,
	const unsigned int socket_id,
	struct db_socket *p_socket)
{
	memset(p_socket, 0, sizeof (struct db_socket));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		socket_id,  type,  model,  brand,  family,  stepping,  manufacturer,  logical_processor_count,  rapl_limited  \
		FROM socket \
		WHERE  socket_id = $socket_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$socket_id", (unsigned int)socket_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_socket(p_ps, p_stmt, p_socket);
			local_get_socket_relationships(p_ps, p_stmt, p_socket);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_socket_by_socket_id(const PersistentStore *p_ps,
	const unsigned int socket_id,
	struct db_socket *p_socket)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE socket \
	SET \
	socket_id=$socket_id \
		,  type=$type \
		,  model=$model \
		,  brand=$brand \
		,  family=$family \
		,  stepping=$stepping \
		,  manufacturer=$manufacturer \
		,  logical_processor_count=$logical_processor_count \
		,  rapl_limited=$rapl_limited \
		  \
	WHERE socket_id=$socket_id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$socket_id", (unsigned int)socket_id);
		local_bind_socket(p_stmt, p_socket);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_socket_by_socket_id(const PersistentStore *p_ps,
	const unsigned int socket_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM socket \
				 WHERE socket_id = $socket_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$socket_id", (unsigned int)socket_id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_socket_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM socket_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_socket_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM socket_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_socket_history_by_history_id(const PersistentStore *p_ps,
	struct db_socket *p_socket,
	int history_id,
	int socket_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_socket, 0, sizeof (struct db_socket) * socket_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		socket_id,  type,  model,  brand,  family,  stepping,  manufacturer,  logical_processor_count,  rapl_limited  \
		FROM socket_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < socket_count)
		{
		rc = DB_SUCCESS;
		local_row_to_socket(p_ps, p_stmt, &p_socket[index]);
		local_get_socket_relationships_history(p_ps, p_stmt, &p_socket[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_socket_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM socket_history");
}

/*
 * --- END socket ----------------
 */
/*
 * --- runtime_config_validation ----------------
 */
void local_bind_runtime_config_validation(sqlite3_stmt *p_stmt, struct db_runtime_config_validation *p_runtime_config_validation)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_runtime_config_validation->id);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_runtime_config_validation->type);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_runtime_config_validation->length);
	BIND_INTEGER(p_stmt, "$address_space_id", (unsigned int)p_runtime_config_validation->address_space_id);
	BIND_INTEGER(p_stmt, "$bit_width", (unsigned int)p_runtime_config_validation->bit_width);
	BIND_INTEGER(p_stmt, "$bit_offset", (unsigned int)p_runtime_config_validation->bit_offset);
	BIND_INTEGER(p_stmt, "$access_size", (unsigned int)p_runtime_config_validation->access_size);
	BIND_INTEGER(p_stmt, "$address", (unsigned long long)p_runtime_config_validation->address);
	BIND_INTEGER(p_stmt, "$operation_type_1", (unsigned int)p_runtime_config_validation->operation_type_1);
	BIND_INTEGER(p_stmt, "$value", (unsigned long long)p_runtime_config_validation->value);
	BIND_INTEGER(p_stmt, "$mask_1", (unsigned long long)p_runtime_config_validation->mask_1);
	BIND_INTEGER(p_stmt, "$gas_structure_0", (unsigned int)p_runtime_config_validation->gas_structure[0]);
	BIND_INTEGER(p_stmt, "$gas_structure_1", (unsigned int)p_runtime_config_validation->gas_structure[1]);
	BIND_INTEGER(p_stmt, "$gas_structure_2", (unsigned int)p_runtime_config_validation->gas_structure[2]);
	BIND_INTEGER(p_stmt, "$gas_structure_3", (unsigned int)p_runtime_config_validation->gas_structure[3]);
	BIND_INTEGER(p_stmt, "$gas_structure_4", (unsigned int)p_runtime_config_validation->gas_structure[4]);
	BIND_INTEGER(p_stmt, "$gas_structure_5", (unsigned int)p_runtime_config_validation->gas_structure[5]);
	BIND_INTEGER(p_stmt, "$gas_structure_6", (unsigned int)p_runtime_config_validation->gas_structure[6]);
	BIND_INTEGER(p_stmt, "$gas_structure_7", (unsigned int)p_runtime_config_validation->gas_structure[7]);
	BIND_INTEGER(p_stmt, "$gas_structure_8", (unsigned int)p_runtime_config_validation->gas_structure[8]);
	BIND_INTEGER(p_stmt, "$gas_structure_9", (unsigned int)p_runtime_config_validation->gas_structure[9]);
	BIND_INTEGER(p_stmt, "$gas_structure_10", (unsigned int)p_runtime_config_validation->gas_structure[10]);
	BIND_INTEGER(p_stmt, "$gas_structure_11", (unsigned int)p_runtime_config_validation->gas_structure[11]);
	BIND_INTEGER(p_stmt, "$operation_type_2", (unsigned int)p_runtime_config_validation->operation_type_2);
	BIND_INTEGER(p_stmt, "$mask_2", (unsigned long long)p_runtime_config_validation->mask_2);
}
void local_get_runtime_config_validation_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_runtime_config_validation *p_runtime_config_validation)
{
}

void local_get_runtime_config_validation_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_runtime_config_validation *p_runtime_config_validation,
	int history_id)
{
}

void local_row_to_runtime_config_validation(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_runtime_config_validation *p_runtime_config_validation)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_runtime_config_validation->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_runtime_config_validation->type);
	INTEGER_COLUMN(p_stmt,
		2,
		p_runtime_config_validation->length);
	INTEGER_COLUMN(p_stmt,
		3,
		p_runtime_config_validation->address_space_id);
	INTEGER_COLUMN(p_stmt,
		4,
		p_runtime_config_validation->bit_width);
	INTEGER_COLUMN(p_stmt,
		5,
		p_runtime_config_validation->bit_offset);
	INTEGER_COLUMN(p_stmt,
		6,
		p_runtime_config_validation->access_size);
	INTEGER_COLUMN(p_stmt,
		7,
		p_runtime_config_validation->address);
	INTEGER_COLUMN(p_stmt,
		8,
		p_runtime_config_validation->operation_type_1);
	INTEGER_COLUMN(p_stmt,
		9,
		p_runtime_config_validation->value);
	INTEGER_COLUMN(p_stmt,
		10,
		p_runtime_config_validation->mask_1);
	INTEGER_COLUMN(p_stmt,
		11,
		p_runtime_config_validation->gas_structure[0]);
	INTEGER_COLUMN(p_stmt,
		12,
		p_runtime_config_validation->gas_structure[1]);
	INTEGER_COLUMN(p_stmt,
		13,
		p_runtime_config_validation->gas_structure[2]);
	INTEGER_COLUMN(p_stmt,
		14,
		p_runtime_config_validation->gas_structure[3]);
	INTEGER_COLUMN(p_stmt,
		15,
		p_runtime_config_validation->gas_structure[4]);
	INTEGER_COLUMN(p_stmt,
		16,
		p_runtime_config_validation->gas_structure[5]);
	INTEGER_COLUMN(p_stmt,
		17,
		p_runtime_config_validation->gas_structure[6]);
	INTEGER_COLUMN(p_stmt,
		18,
		p_runtime_config_validation->gas_structure[7]);
	INTEGER_COLUMN(p_stmt,
		19,
		p_runtime_config_validation->gas_structure[8]);
	INTEGER_COLUMN(p_stmt,
		20,
		p_runtime_config_validation->gas_structure[9]);
	INTEGER_COLUMN(p_stmt,
		21,
		p_runtime_config_validation->gas_structure[10]);
	INTEGER_COLUMN(p_stmt,
		22,
		p_runtime_config_validation->gas_structure[11]);
	INTEGER_COLUMN(p_stmt,
		23,
		p_runtime_config_validation->operation_type_2);
	INTEGER_COLUMN(p_stmt,
		24,
		p_runtime_config_validation->mask_2);
}
void db_print_runtime_config_validation(struct db_runtime_config_validation *p_value)
{
	printf("runtime_config_validation.id: %d\n", p_value->id);
	printf("runtime_config_validation.type: unsigned %d\n", p_value->type);
	printf("runtime_config_validation.length: unsigned %d\n", p_value->length);
	printf("runtime_config_validation.address_space_id: unsigned %d\n", p_value->address_space_id);
	printf("runtime_config_validation.bit_width: unsigned %d\n", p_value->bit_width);
	printf("runtime_config_validation.bit_offset: unsigned %d\n", p_value->bit_offset);
	printf("runtime_config_validation.access_size: unsigned %d\n", p_value->access_size);
	printf("runtime_config_validation.address: unsigned %lld\n", p_value->address);
	printf("runtime_config_validation.operation_type_1: unsigned %d\n", p_value->operation_type_1);
	printf("runtime_config_validation.value: unsigned %lld\n", p_value->value);
	printf("runtime_config_validation.mask_1: unsigned %lld\n", p_value->mask_1);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[0]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[1]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[2]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[3]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[4]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[5]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[6]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[7]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[8]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[9]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[10]);
	printf("runtime_config_validation.gas_structure: unsigned %d\n", p_value->gas_structure[11]);
	printf("runtime_config_validation.operation_type_2: unsigned %d\n", p_value->operation_type_2);
	printf("runtime_config_validation.mask_2: unsigned %lld\n", p_value->mask_2);
}
enum db_return_codes db_add_runtime_config_validation(const PersistentStore *p_ps,
	struct db_runtime_config_validation *p_runtime_config_validation)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO runtime_config_validation \
		(id, type, length, address_space_id, bit_width, bit_offset, access_size, address, operation_type_1, value, mask_1, gas_structure_0, gas_structure_1, gas_structure_2, gas_structure_3, gas_structure_4, gas_structure_5, gas_structure_6, gas_structure_7, gas_structure_8, gas_structure_9, gas_structure_10, gas_structure_11, operation_type_2, mask_2)  \
		VALUES 		\
		($id, \
		$type, \
		$length, \
		$address_space_id, \
		$bit_width, \
		$bit_offset, \
		$access_size, \
		$address, \
		$operation_type_1, \
		$value, \
		$mask_1, \
		$gas_structure_0, \
		$gas_structure_1, \
		$gas_structure_2, \
		$gas_structure_3, \
		$gas_structure_4, \
		$gas_structure_5, \
		$gas_structure_6, \
		$gas_structure_7, \
		$gas_structure_8, \
		$gas_structure_9, \
		$gas_structure_10, \
		$gas_structure_11, \
		$operation_type_2, \
		$mask_2) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_runtime_config_validation(p_stmt, p_runtime_config_validation);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_runtime_config_validation_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "runtime_config_validation", p_count);
}
int db_get_runtime_config_validations(const PersistentStore *p_ps,
	struct db_runtime_config_validation *p_runtime_config_validation,
	int runtime_config_validation_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_runtime_config_validation, 0, sizeof (struct db_runtime_config_validation) * runtime_config_validation_count);
	char *sql = "SELECT \
		id \
		,  type \
		,  length \
		,  address_space_id \
		,  bit_width \
		,  bit_offset \
		,  access_size \
		,  address \
		,  operation_type_1 \
		,  value \
		,  mask_1 \
		,  gas_structure_0 \
		,  gas_structure_1 \
		,  gas_structure_2 \
		,  gas_structure_3 \
		,  gas_structure_4 \
		,  gas_structure_5 \
		,  gas_structure_6 \
		,  gas_structure_7 \
		,  gas_structure_8 \
		,  gas_structure_9 \
		,  gas_structure_10 \
		,  gas_structure_11 \
		,  operation_type_2 \
		,  mask_2 \
		  \
		FROM runtime_config_validation \
		                          \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < runtime_config_validation_count)
		{
			local_row_to_runtime_config_validation(p_ps, p_stmt, &p_runtime_config_validation[index]);
			local_get_runtime_config_validation_relationships(p_ps, p_stmt, &p_runtime_config_validation[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_runtime_config_validations(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM runtime_config_validation");
}

enum db_return_codes db_save_runtime_config_validation_state(const PersistentStore *p_ps,
	int history_id,
	struct db_runtime_config_validation *p_runtime_config_validation)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_runtime_config_validation temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_runtime_config_validation_by_id(p_ps, p_runtime_config_validation->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_runtime_config_validation_by_id(p_ps,
				p_runtime_config_validation->id,
				p_runtime_config_validation);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO runtime_config_validation \
			( id ,  type ,  length ,  address_space_id ,  bit_width ,  bit_offset ,  access_size ,  address ,  operation_type_1 ,  value ,  mask_1 ,  gas_structure_0 ,  gas_structure_1 ,  gas_structure_2 ,  gas_structure_3 ,  gas_structure_4 ,  gas_structure_5 ,  gas_structure_6 ,  gas_structure_7 ,  gas_structure_8 ,  gas_structure_9 ,  gas_structure_10 ,  gas_structure_11 ,  operation_type_2 ,  mask_2 )  \
			VALUES 		\
			($id, \
			$type, \
			$length, \
			$address_space_id, \
			$bit_width, \
			$bit_offset, \
			$access_size, \
			$address, \
			$operation_type_1, \
			$value, \
			$mask_1, \
			$gas_structure_0, \
			$gas_structure_1, \
			$gas_structure_2, \
			$gas_structure_3, \
			$gas_structure_4, \
			$gas_structure_5, \
			$gas_structure_6, \
			$gas_structure_7, \
			$gas_structure_8, \
			$gas_structure_9, \
			$gas_structure_10, \
			$gas_structure_11, \
			$operation_type_2, \
			$mask_2) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_runtime_config_validation(p_stmt, p_runtime_config_validation);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO runtime_config_validation_history \
			(history_id, \
				 id,  type,  length,  address_space_id,  bit_width,  bit_offset,  access_size,  address,  operation_type_1,  value,  mask_1,  gas_structure_0,  gas_structure_1,  gas_structure_2,  gas_structure_3,  gas_structure_4,  gas_structure_5,  gas_structure_6,  gas_structure_7,  gas_structure_8,  gas_structure_9,  gas_structure_10,  gas_structure_11,  operation_type_2,  mask_2)  \
			VALUES 		($history_id, \
				 $id , \
				 $type , \
				 $length , \
				 $address_space_id , \
				 $bit_width , \
				 $bit_offset , \
				 $access_size , \
				 $address , \
				 $operation_type_1 , \
				 $value , \
				 $mask_1 , \
				 $gas_structure_0 , \
				 $gas_structure_1 , \
				 $gas_structure_2 , \
				 $gas_structure_3 , \
				 $gas_structure_4 , \
				 $gas_structure_5 , \
				 $gas_structure_6 , \
				 $gas_structure_7 , \
				 $gas_structure_8 , \
				 $gas_structure_9 , \
				 $gas_structure_10 , \
				 $gas_structure_11 , \
				 $operation_type_2 , \
				 $mask_2 )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_runtime_config_validation(p_stmt, p_runtime_config_validation);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_runtime_config_validation_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_runtime_config_validation *p_runtime_config_validation)
{
	memset(p_runtime_config_validation, 0, sizeof (struct db_runtime_config_validation));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  type,  length,  address_space_id,  bit_width,  bit_offset,  access_size,  address,  operation_type_1,  value,  mask_1,  gas_structure_0,  gas_structure_1,  gas_structure_2,  gas_structure_3,  gas_structure_4,  gas_structure_5,  gas_structure_6,  gas_structure_7,  gas_structure_8,  gas_structure_9,  gas_structure_10,  gas_structure_11,  operation_type_2,  mask_2  \
		FROM runtime_config_validation \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_runtime_config_validation(p_ps, p_stmt, p_runtime_config_validation);
			local_get_runtime_config_validation_relationships(p_ps, p_stmt, p_runtime_config_validation);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_runtime_config_validation_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_runtime_config_validation *p_runtime_config_validation)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE runtime_config_validation \
	SET \
	id=$id \
		,  type=$type \
		,  length=$length \
		,  address_space_id=$address_space_id \
		,  bit_width=$bit_width \
		,  bit_offset=$bit_offset \
		,  access_size=$access_size \
		,  address=$address \
		,  operation_type_1=$operation_type_1 \
		,  value=$value \
		,  mask_1=$mask_1 \
		,  gas_structure_0=$gas_structure_0 \
		,  gas_structure_1=$gas_structure_1 \
		,  gas_structure_2=$gas_structure_2 \
		,  gas_structure_3=$gas_structure_3 \
		,  gas_structure_4=$gas_structure_4 \
		,  gas_structure_5=$gas_structure_5 \
		,  gas_structure_6=$gas_structure_6 \
		,  gas_structure_7=$gas_structure_7 \
		,  gas_structure_8=$gas_structure_8 \
		,  gas_structure_9=$gas_structure_9 \
		,  gas_structure_10=$gas_structure_10 \
		,  gas_structure_11=$gas_structure_11 \
		,  operation_type_2=$operation_type_2 \
		,  mask_2=$mask_2 \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_runtime_config_validation(p_stmt, p_runtime_config_validation);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_runtime_config_validation_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM runtime_config_validation \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_runtime_config_validation_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM runtime_config_validation_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_runtime_config_validation_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM runtime_config_validation_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_runtime_config_validation_history_by_history_id(const PersistentStore *p_ps,
	struct db_runtime_config_validation *p_runtime_config_validation,
	int history_id,
	int runtime_config_validation_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_runtime_config_validation, 0, sizeof (struct db_runtime_config_validation) * runtime_config_validation_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  type,  length,  address_space_id,  bit_width,  bit_offset,  access_size,  address,  operation_type_1,  value,  mask_1,  gas_structure_0,  gas_structure_1,  gas_structure_2,  gas_structure_3,  gas_structure_4,  gas_structure_5,  gas_structure_6,  gas_structure_7,  gas_structure_8,  gas_structure_9,  gas_structure_10,  gas_structure_11,  operation_type_2,  mask_2  \
		FROM runtime_config_validation_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < runtime_config_validation_count)
		{
		rc = DB_SUCCESS;
		local_row_to_runtime_config_validation(p_ps, p_stmt, &p_runtime_config_validation[index]);
		local_get_runtime_config_validation_relationships_history(p_ps, p_stmt, &p_runtime_config_validation[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_runtime_config_validation_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM runtime_config_validation_history");
}

/*!
 * Roll runtime_config_validations by id to specified max.
 */
enum db_return_codes db_roll_runtime_config_validations_by_id(const PersistentStore *p_ps,
	int max_rows)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char sql[1024];
	snprintf(sql, 1024,
				"DELETE FROM runtime_config_validation "
				"WHERE id NOT IN ("
				"SELECT id "
				"FROM runtime_config_validation "
				"ORDER BY id DESC "
				"LIMIT %d)", max_rows); 
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Get the max id in the runtime_config_validation table.
 */
enum db_return_codes db_get_next_runtime_config_validation_id(const PersistentStore *p_ps, int *p_max)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	if (p_max)
	{
		if ((rc = run_scalar_sql(p_ps, 
			"SELECT MAX(id) FROM runtime_config_validation", p_max)) 
			==  DB_SUCCESS)
		{
			(*p_max)++;
		}
	}
	return rc;
}
/*
 * --- END runtime_config_validation ----------------
 */
/*
 * --- interleave_capability ----------------
 */
void local_bind_interleave_capability(sqlite3_stmt *p_stmt, struct db_interleave_capability *p_interleave_capability)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_interleave_capability->id);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_interleave_capability->type);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_interleave_capability->length);
	BIND_INTEGER(p_stmt, "$memory_mode", (unsigned int)p_interleave_capability->memory_mode);
	BIND_INTEGER(p_stmt, "$interleave_alignment_size", (unsigned int)p_interleave_capability->interleave_alignment_size);
	BIND_INTEGER(p_stmt, "$supported_interleave_count", (unsigned int)p_interleave_capability->supported_interleave_count);
	BIND_INTEGER(p_stmt, "$interleave_format_list_0", (unsigned int)p_interleave_capability->interleave_format_list[0]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_1", (unsigned int)p_interleave_capability->interleave_format_list[1]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_2", (unsigned int)p_interleave_capability->interleave_format_list[2]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_3", (unsigned int)p_interleave_capability->interleave_format_list[3]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_4", (unsigned int)p_interleave_capability->interleave_format_list[4]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_5", (unsigned int)p_interleave_capability->interleave_format_list[5]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_6", (unsigned int)p_interleave_capability->interleave_format_list[6]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_7", (unsigned int)p_interleave_capability->interleave_format_list[7]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_8", (unsigned int)p_interleave_capability->interleave_format_list[8]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_9", (unsigned int)p_interleave_capability->interleave_format_list[9]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_10", (unsigned int)p_interleave_capability->interleave_format_list[10]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_11", (unsigned int)p_interleave_capability->interleave_format_list[11]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_12", (unsigned int)p_interleave_capability->interleave_format_list[12]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_13", (unsigned int)p_interleave_capability->interleave_format_list[13]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_14", (unsigned int)p_interleave_capability->interleave_format_list[14]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_15", (unsigned int)p_interleave_capability->interleave_format_list[15]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_16", (unsigned int)p_interleave_capability->interleave_format_list[16]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_17", (unsigned int)p_interleave_capability->interleave_format_list[17]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_18", (unsigned int)p_interleave_capability->interleave_format_list[18]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_19", (unsigned int)p_interleave_capability->interleave_format_list[19]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_20", (unsigned int)p_interleave_capability->interleave_format_list[20]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_21", (unsigned int)p_interleave_capability->interleave_format_list[21]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_22", (unsigned int)p_interleave_capability->interleave_format_list[22]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_23", (unsigned int)p_interleave_capability->interleave_format_list[23]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_24", (unsigned int)p_interleave_capability->interleave_format_list[24]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_25", (unsigned int)p_interleave_capability->interleave_format_list[25]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_26", (unsigned int)p_interleave_capability->interleave_format_list[26]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_27", (unsigned int)p_interleave_capability->interleave_format_list[27]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_28", (unsigned int)p_interleave_capability->interleave_format_list[28]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_29", (unsigned int)p_interleave_capability->interleave_format_list[29]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_30", (unsigned int)p_interleave_capability->interleave_format_list[30]);
	BIND_INTEGER(p_stmt, "$interleave_format_list_31", (unsigned int)p_interleave_capability->interleave_format_list[31]);
}
void local_get_interleave_capability_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_interleave_capability *p_interleave_capability)
{
}

void local_get_interleave_capability_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_interleave_capability *p_interleave_capability,
	int history_id)
{
}

void local_row_to_interleave_capability(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_interleave_capability *p_interleave_capability)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_interleave_capability->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_interleave_capability->type);
	INTEGER_COLUMN(p_stmt,
		2,
		p_interleave_capability->length);
	INTEGER_COLUMN(p_stmt,
		3,
		p_interleave_capability->memory_mode);
	INTEGER_COLUMN(p_stmt,
		4,
		p_interleave_capability->interleave_alignment_size);
	INTEGER_COLUMN(p_stmt,
		5,
		p_interleave_capability->supported_interleave_count);
	INTEGER_COLUMN(p_stmt,
		6,
		p_interleave_capability->interleave_format_list[0]);
	INTEGER_COLUMN(p_stmt,
		7,
		p_interleave_capability->interleave_format_list[1]);
	INTEGER_COLUMN(p_stmt,
		8,
		p_interleave_capability->interleave_format_list[2]);
	INTEGER_COLUMN(p_stmt,
		9,
		p_interleave_capability->interleave_format_list[3]);
	INTEGER_COLUMN(p_stmt,
		10,
		p_interleave_capability->interleave_format_list[4]);
	INTEGER_COLUMN(p_stmt,
		11,
		p_interleave_capability->interleave_format_list[5]);
	INTEGER_COLUMN(p_stmt,
		12,
		p_interleave_capability->interleave_format_list[6]);
	INTEGER_COLUMN(p_stmt,
		13,
		p_interleave_capability->interleave_format_list[7]);
	INTEGER_COLUMN(p_stmt,
		14,
		p_interleave_capability->interleave_format_list[8]);
	INTEGER_COLUMN(p_stmt,
		15,
		p_interleave_capability->interleave_format_list[9]);
	INTEGER_COLUMN(p_stmt,
		16,
		p_interleave_capability->interleave_format_list[10]);
	INTEGER_COLUMN(p_stmt,
		17,
		p_interleave_capability->interleave_format_list[11]);
	INTEGER_COLUMN(p_stmt,
		18,
		p_interleave_capability->interleave_format_list[12]);
	INTEGER_COLUMN(p_stmt,
		19,
		p_interleave_capability->interleave_format_list[13]);
	INTEGER_COLUMN(p_stmt,
		20,
		p_interleave_capability->interleave_format_list[14]);
	INTEGER_COLUMN(p_stmt,
		21,
		p_interleave_capability->interleave_format_list[15]);
	INTEGER_COLUMN(p_stmt,
		22,
		p_interleave_capability->interleave_format_list[16]);
	INTEGER_COLUMN(p_stmt,
		23,
		p_interleave_capability->interleave_format_list[17]);
	INTEGER_COLUMN(p_stmt,
		24,
		p_interleave_capability->interleave_format_list[18]);
	INTEGER_COLUMN(p_stmt,
		25,
		p_interleave_capability->interleave_format_list[19]);
	INTEGER_COLUMN(p_stmt,
		26,
		p_interleave_capability->interleave_format_list[20]);
	INTEGER_COLUMN(p_stmt,
		27,
		p_interleave_capability->interleave_format_list[21]);
	INTEGER_COLUMN(p_stmt,
		28,
		p_interleave_capability->interleave_format_list[22]);
	INTEGER_COLUMN(p_stmt,
		29,
		p_interleave_capability->interleave_format_list[23]);
	INTEGER_COLUMN(p_stmt,
		30,
		p_interleave_capability->interleave_format_list[24]);
	INTEGER_COLUMN(p_stmt,
		31,
		p_interleave_capability->interleave_format_list[25]);
	INTEGER_COLUMN(p_stmt,
		32,
		p_interleave_capability->interleave_format_list[26]);
	INTEGER_COLUMN(p_stmt,
		33,
		p_interleave_capability->interleave_format_list[27]);
	INTEGER_COLUMN(p_stmt,
		34,
		p_interleave_capability->interleave_format_list[28]);
	INTEGER_COLUMN(p_stmt,
		35,
		p_interleave_capability->interleave_format_list[29]);
	INTEGER_COLUMN(p_stmt,
		36,
		p_interleave_capability->interleave_format_list[30]);
	INTEGER_COLUMN(p_stmt,
		37,
		p_interleave_capability->interleave_format_list[31]);
}
void db_print_interleave_capability(struct db_interleave_capability *p_value)
{
	printf("interleave_capability.id: %d\n", p_value->id);
	printf("interleave_capability.type: unsigned %d\n", p_value->type);
	printf("interleave_capability.length: unsigned %d\n", p_value->length);
	printf("interleave_capability.memory_mode: unsigned %d\n", p_value->memory_mode);
	printf("interleave_capability.interleave_alignment_size: unsigned %d\n", p_value->interleave_alignment_size);
	printf("interleave_capability.supported_interleave_count: unsigned %d\n", p_value->supported_interleave_count);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[0]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[1]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[2]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[3]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[4]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[5]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[6]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[7]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[8]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[9]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[10]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[11]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[12]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[13]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[14]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[15]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[16]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[17]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[18]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[19]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[20]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[21]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[22]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[23]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[24]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[25]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[26]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[27]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[28]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[29]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[30]);
	printf("interleave_capability.interleave_format_list: unsigned %d\n", p_value->interleave_format_list[31]);
}
enum db_return_codes db_add_interleave_capability(const PersistentStore *p_ps,
	struct db_interleave_capability *p_interleave_capability)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO interleave_capability \
		(id, type, length, memory_mode, interleave_alignment_size, supported_interleave_count, interleave_format_list_0, interleave_format_list_1, interleave_format_list_2, interleave_format_list_3, interleave_format_list_4, interleave_format_list_5, interleave_format_list_6, interleave_format_list_7, interleave_format_list_8, interleave_format_list_9, interleave_format_list_10, interleave_format_list_11, interleave_format_list_12, interleave_format_list_13, interleave_format_list_14, interleave_format_list_15, interleave_format_list_16, interleave_format_list_17, interleave_format_list_18, interleave_format_list_19, interleave_format_list_20, interleave_format_list_21, interleave_format_list_22, interleave_format_list_23, interleave_format_list_24, interleave_format_list_25, interleave_format_list_26, interleave_format_list_27, interleave_format_list_28, interleave_format_list_29, interleave_format_list_30, interleave_format_list_31)  \
		VALUES 		\
		($id, \
		$type, \
		$length, \
		$memory_mode, \
		$interleave_alignment_size, \
		$supported_interleave_count, \
		$interleave_format_list_0, \
		$interleave_format_list_1, \
		$interleave_format_list_2, \
		$interleave_format_list_3, \
		$interleave_format_list_4, \
		$interleave_format_list_5, \
		$interleave_format_list_6, \
		$interleave_format_list_7, \
		$interleave_format_list_8, \
		$interleave_format_list_9, \
		$interleave_format_list_10, \
		$interleave_format_list_11, \
		$interleave_format_list_12, \
		$interleave_format_list_13, \
		$interleave_format_list_14, \
		$interleave_format_list_15, \
		$interleave_format_list_16, \
		$interleave_format_list_17, \
		$interleave_format_list_18, \
		$interleave_format_list_19, \
		$interleave_format_list_20, \
		$interleave_format_list_21, \
		$interleave_format_list_22, \
		$interleave_format_list_23, \
		$interleave_format_list_24, \
		$interleave_format_list_25, \
		$interleave_format_list_26, \
		$interleave_format_list_27, \
		$interleave_format_list_28, \
		$interleave_format_list_29, \
		$interleave_format_list_30, \
		$interleave_format_list_31) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_interleave_capability(p_stmt, p_interleave_capability);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_capability_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "interleave_capability", p_count);
}
int db_get_interleave_capabilitys(const PersistentStore *p_ps,
	struct db_interleave_capability *p_interleave_capability,
	int interleave_capability_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_interleave_capability, 0, sizeof (struct db_interleave_capability) * interleave_capability_count);
	char *sql = "SELECT \
		id \
		,  type \
		,  length \
		,  memory_mode \
		,  interleave_alignment_size \
		,  supported_interleave_count \
		,  interleave_format_list_0 \
		,  interleave_format_list_1 \
		,  interleave_format_list_2 \
		,  interleave_format_list_3 \
		,  interleave_format_list_4 \
		,  interleave_format_list_5 \
		,  interleave_format_list_6 \
		,  interleave_format_list_7 \
		,  interleave_format_list_8 \
		,  interleave_format_list_9 \
		,  interleave_format_list_10 \
		,  interleave_format_list_11 \
		,  interleave_format_list_12 \
		,  interleave_format_list_13 \
		,  interleave_format_list_14 \
		,  interleave_format_list_15 \
		,  interleave_format_list_16 \
		,  interleave_format_list_17 \
		,  interleave_format_list_18 \
		,  interleave_format_list_19 \
		,  interleave_format_list_20 \
		,  interleave_format_list_21 \
		,  interleave_format_list_22 \
		,  interleave_format_list_23 \
		,  interleave_format_list_24 \
		,  interleave_format_list_25 \
		,  interleave_format_list_26 \
		,  interleave_format_list_27 \
		,  interleave_format_list_28 \
		,  interleave_format_list_29 \
		,  interleave_format_list_30 \
		,  interleave_format_list_31 \
		  \
		FROM interleave_capability \
		                                       \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_capability_count)
		{
			local_row_to_interleave_capability(p_ps, p_stmt, &p_interleave_capability[index]);
			local_get_interleave_capability_relationships(p_ps, p_stmt, &p_interleave_capability[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_interleave_capabilitys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM interleave_capability");
}

enum db_return_codes db_save_interleave_capability_state(const PersistentStore *p_ps,
	int history_id,
	struct db_interleave_capability *p_interleave_capability)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_interleave_capability temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_interleave_capability_by_id(p_ps, p_interleave_capability->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_interleave_capability_by_id(p_ps,
				p_interleave_capability->id,
				p_interleave_capability);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO interleave_capability \
			( id ,  type ,  length ,  memory_mode ,  interleave_alignment_size ,  supported_interleave_count ,  interleave_format_list_0 ,  interleave_format_list_1 ,  interleave_format_list_2 ,  interleave_format_list_3 ,  interleave_format_list_4 ,  interleave_format_list_5 ,  interleave_format_list_6 ,  interleave_format_list_7 ,  interleave_format_list_8 ,  interleave_format_list_9 ,  interleave_format_list_10 ,  interleave_format_list_11 ,  interleave_format_list_12 ,  interleave_format_list_13 ,  interleave_format_list_14 ,  interleave_format_list_15 ,  interleave_format_list_16 ,  interleave_format_list_17 ,  interleave_format_list_18 ,  interleave_format_list_19 ,  interleave_format_list_20 ,  interleave_format_list_21 ,  interleave_format_list_22 ,  interleave_format_list_23 ,  interleave_format_list_24 ,  interleave_format_list_25 ,  interleave_format_list_26 ,  interleave_format_list_27 ,  interleave_format_list_28 ,  interleave_format_list_29 ,  interleave_format_list_30 ,  interleave_format_list_31 )  \
			VALUES 		\
			($id, \
			$type, \
			$length, \
			$memory_mode, \
			$interleave_alignment_size, \
			$supported_interleave_count, \
			$interleave_format_list_0, \
			$interleave_format_list_1, \
			$interleave_format_list_2, \
			$interleave_format_list_3, \
			$interleave_format_list_4, \
			$interleave_format_list_5, \
			$interleave_format_list_6, \
			$interleave_format_list_7, \
			$interleave_format_list_8, \
			$interleave_format_list_9, \
			$interleave_format_list_10, \
			$interleave_format_list_11, \
			$interleave_format_list_12, \
			$interleave_format_list_13, \
			$interleave_format_list_14, \
			$interleave_format_list_15, \
			$interleave_format_list_16, \
			$interleave_format_list_17, \
			$interleave_format_list_18, \
			$interleave_format_list_19, \
			$interleave_format_list_20, \
			$interleave_format_list_21, \
			$interleave_format_list_22, \
			$interleave_format_list_23, \
			$interleave_format_list_24, \
			$interleave_format_list_25, \
			$interleave_format_list_26, \
			$interleave_format_list_27, \
			$interleave_format_list_28, \
			$interleave_format_list_29, \
			$interleave_format_list_30, \
			$interleave_format_list_31) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_interleave_capability(p_stmt, p_interleave_capability);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO interleave_capability_history \
			(history_id, \
				 id,  type,  length,  memory_mode,  interleave_alignment_size,  supported_interleave_count,  interleave_format_list_0,  interleave_format_list_1,  interleave_format_list_2,  interleave_format_list_3,  interleave_format_list_4,  interleave_format_list_5,  interleave_format_list_6,  interleave_format_list_7,  interleave_format_list_8,  interleave_format_list_9,  interleave_format_list_10,  interleave_format_list_11,  interleave_format_list_12,  interleave_format_list_13,  interleave_format_list_14,  interleave_format_list_15,  interleave_format_list_16,  interleave_format_list_17,  interleave_format_list_18,  interleave_format_list_19,  interleave_format_list_20,  interleave_format_list_21,  interleave_format_list_22,  interleave_format_list_23,  interleave_format_list_24,  interleave_format_list_25,  interleave_format_list_26,  interleave_format_list_27,  interleave_format_list_28,  interleave_format_list_29,  interleave_format_list_30,  interleave_format_list_31)  \
			VALUES 		($history_id, \
				 $id , \
				 $type , \
				 $length , \
				 $memory_mode , \
				 $interleave_alignment_size , \
				 $supported_interleave_count , \
				 $interleave_format_list_0 , \
				 $interleave_format_list_1 , \
				 $interleave_format_list_2 , \
				 $interleave_format_list_3 , \
				 $interleave_format_list_4 , \
				 $interleave_format_list_5 , \
				 $interleave_format_list_6 , \
				 $interleave_format_list_7 , \
				 $interleave_format_list_8 , \
				 $interleave_format_list_9 , \
				 $interleave_format_list_10 , \
				 $interleave_format_list_11 , \
				 $interleave_format_list_12 , \
				 $interleave_format_list_13 , \
				 $interleave_format_list_14 , \
				 $interleave_format_list_15 , \
				 $interleave_format_list_16 , \
				 $interleave_format_list_17 , \
				 $interleave_format_list_18 , \
				 $interleave_format_list_19 , \
				 $interleave_format_list_20 , \
				 $interleave_format_list_21 , \
				 $interleave_format_list_22 , \
				 $interleave_format_list_23 , \
				 $interleave_format_list_24 , \
				 $interleave_format_list_25 , \
				 $interleave_format_list_26 , \
				 $interleave_format_list_27 , \
				 $interleave_format_list_28 , \
				 $interleave_format_list_29 , \
				 $interleave_format_list_30 , \
				 $interleave_format_list_31 )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_interleave_capability(p_stmt, p_interleave_capability);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_interleave_capability_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_interleave_capability *p_interleave_capability)
{
	memset(p_interleave_capability, 0, sizeof (struct db_interleave_capability));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  type,  length,  memory_mode,  interleave_alignment_size,  supported_interleave_count,  interleave_format_list_0,  interleave_format_list_1,  interleave_format_list_2,  interleave_format_list_3,  interleave_format_list_4,  interleave_format_list_5,  interleave_format_list_6,  interleave_format_list_7,  interleave_format_list_8,  interleave_format_list_9,  interleave_format_list_10,  interleave_format_list_11,  interleave_format_list_12,  interleave_format_list_13,  interleave_format_list_14,  interleave_format_list_15,  interleave_format_list_16,  interleave_format_list_17,  interleave_format_list_18,  interleave_format_list_19,  interleave_format_list_20,  interleave_format_list_21,  interleave_format_list_22,  interleave_format_list_23,  interleave_format_list_24,  interleave_format_list_25,  interleave_format_list_26,  interleave_format_list_27,  interleave_format_list_28,  interleave_format_list_29,  interleave_format_list_30,  interleave_format_list_31  \
		FROM interleave_capability \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_interleave_capability(p_ps, p_stmt, p_interleave_capability);
			local_get_interleave_capability_relationships(p_ps, p_stmt, p_interleave_capability);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_interleave_capability_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_interleave_capability *p_interleave_capability)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE interleave_capability \
	SET \
	id=$id \
		,  type=$type \
		,  length=$length \
		,  memory_mode=$memory_mode \
		,  interleave_alignment_size=$interleave_alignment_size \
		,  supported_interleave_count=$supported_interleave_count \
		,  interleave_format_list_0=$interleave_format_list_0 \
		,  interleave_format_list_1=$interleave_format_list_1 \
		,  interleave_format_list_2=$interleave_format_list_2 \
		,  interleave_format_list_3=$interleave_format_list_3 \
		,  interleave_format_list_4=$interleave_format_list_4 \
		,  interleave_format_list_5=$interleave_format_list_5 \
		,  interleave_format_list_6=$interleave_format_list_6 \
		,  interleave_format_list_7=$interleave_format_list_7 \
		,  interleave_format_list_8=$interleave_format_list_8 \
		,  interleave_format_list_9=$interleave_format_list_9 \
		,  interleave_format_list_10=$interleave_format_list_10 \
		,  interleave_format_list_11=$interleave_format_list_11 \
		,  interleave_format_list_12=$interleave_format_list_12 \
		,  interleave_format_list_13=$interleave_format_list_13 \
		,  interleave_format_list_14=$interleave_format_list_14 \
		,  interleave_format_list_15=$interleave_format_list_15 \
		,  interleave_format_list_16=$interleave_format_list_16 \
		,  interleave_format_list_17=$interleave_format_list_17 \
		,  interleave_format_list_18=$interleave_format_list_18 \
		,  interleave_format_list_19=$interleave_format_list_19 \
		,  interleave_format_list_20=$interleave_format_list_20 \
		,  interleave_format_list_21=$interleave_format_list_21 \
		,  interleave_format_list_22=$interleave_format_list_22 \
		,  interleave_format_list_23=$interleave_format_list_23 \
		,  interleave_format_list_24=$interleave_format_list_24 \
		,  interleave_format_list_25=$interleave_format_list_25 \
		,  interleave_format_list_26=$interleave_format_list_26 \
		,  interleave_format_list_27=$interleave_format_list_27 \
		,  interleave_format_list_28=$interleave_format_list_28 \
		,  interleave_format_list_29=$interleave_format_list_29 \
		,  interleave_format_list_30=$interleave_format_list_30 \
		,  interleave_format_list_31=$interleave_format_list_31 \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_interleave_capability(p_stmt, p_interleave_capability);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_interleave_capability_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM interleave_capability \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_interleave_capability_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM interleave_capability_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_capability_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM interleave_capability_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_interleave_capability_history_by_history_id(const PersistentStore *p_ps,
	struct db_interleave_capability *p_interleave_capability,
	int history_id,
	int interleave_capability_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_interleave_capability, 0, sizeof (struct db_interleave_capability) * interleave_capability_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  type,  length,  memory_mode,  interleave_alignment_size,  supported_interleave_count,  interleave_format_list_0,  interleave_format_list_1,  interleave_format_list_2,  interleave_format_list_3,  interleave_format_list_4,  interleave_format_list_5,  interleave_format_list_6,  interleave_format_list_7,  interleave_format_list_8,  interleave_format_list_9,  interleave_format_list_10,  interleave_format_list_11,  interleave_format_list_12,  interleave_format_list_13,  interleave_format_list_14,  interleave_format_list_15,  interleave_format_list_16,  interleave_format_list_17,  interleave_format_list_18,  interleave_format_list_19,  interleave_format_list_20,  interleave_format_list_21,  interleave_format_list_22,  interleave_format_list_23,  interleave_format_list_24,  interleave_format_list_25,  interleave_format_list_26,  interleave_format_list_27,  interleave_format_list_28,  interleave_format_list_29,  interleave_format_list_30,  interleave_format_list_31  \
		FROM interleave_capability_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_capability_count)
		{
		rc = DB_SUCCESS;
		local_row_to_interleave_capability(p_ps, p_stmt, &p_interleave_capability[index]);
		local_get_interleave_capability_relationships_history(p_ps, p_stmt, &p_interleave_capability[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_interleave_capability_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM interleave_capability_history");
}

/*!
 * Roll interleave_capabilitys by id to specified max.
 */
enum db_return_codes db_roll_interleave_capabilitys_by_id(const PersistentStore *p_ps,
	int max_rows)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char sql[1024];
	snprintf(sql, 1024,
				"DELETE FROM interleave_capability "
				"WHERE id NOT IN ("
				"SELECT id "
				"FROM interleave_capability "
				"ORDER BY id DESC "
				"LIMIT %d)", max_rows); 
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Get the max id in the interleave_capability table.
 */
enum db_return_codes db_get_next_interleave_capability_id(const PersistentStore *p_ps, int *p_max)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	if (p_max)
	{
		if ((rc = run_scalar_sql(p_ps, 
			"SELECT MAX(id) FROM interleave_capability", p_max)) 
			==  DB_SUCCESS)
		{
			(*p_max)++;
		}
	}
	return rc;
}
/*
 * --- END interleave_capability ----------------
 */
/*
 * --- platform_info_capability ----------------
 */
void local_bind_platform_info_capability(sqlite3_stmt *p_stmt, struct db_platform_info_capability *p_platform_info_capability)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_platform_info_capability->id);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_platform_info_capability->type);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_platform_info_capability->length);
	BIND_INTEGER(p_stmt, "$mgmt_sw_config_support", (unsigned int)p_platform_info_capability->mgmt_sw_config_support);
	BIND_INTEGER(p_stmt, "$mem_mode_capabilities", (unsigned int)p_platform_info_capability->mem_mode_capabilities);
	BIND_INTEGER(p_stmt, "$current_mem_mode", (unsigned int)p_platform_info_capability->current_mem_mode);
	BIND_INTEGER(p_stmt, "$pmem_ras_capabilities", (unsigned int)p_platform_info_capability->pmem_ras_capabilities);
}
void local_get_platform_info_capability_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_platform_info_capability *p_platform_info_capability)
{
}

void local_get_platform_info_capability_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_platform_info_capability *p_platform_info_capability,
	int history_id)
{
}

void local_row_to_platform_info_capability(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_platform_info_capability *p_platform_info_capability)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_platform_info_capability->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_platform_info_capability->type);
	INTEGER_COLUMN(p_stmt,
		2,
		p_platform_info_capability->length);
	INTEGER_COLUMN(p_stmt,
		3,
		p_platform_info_capability->mgmt_sw_config_support);
	INTEGER_COLUMN(p_stmt,
		4,
		p_platform_info_capability->mem_mode_capabilities);
	INTEGER_COLUMN(p_stmt,
		5,
		p_platform_info_capability->current_mem_mode);
	INTEGER_COLUMN(p_stmt,
		6,
		p_platform_info_capability->pmem_ras_capabilities);
}
void db_print_platform_info_capability(struct db_platform_info_capability *p_value)
{
	printf("platform_info_capability.id: %d\n", p_value->id);
	printf("platform_info_capability.type: unsigned %d\n", p_value->type);
	printf("platform_info_capability.length: unsigned %d\n", p_value->length);
	printf("platform_info_capability.mgmt_sw_config_support: unsigned %d\n", p_value->mgmt_sw_config_support);
	printf("platform_info_capability.mem_mode_capabilities: unsigned %d\n", p_value->mem_mode_capabilities);
	printf("platform_info_capability.current_mem_mode: unsigned %d\n", p_value->current_mem_mode);
	printf("platform_info_capability.pmem_ras_capabilities: unsigned %d\n", p_value->pmem_ras_capabilities);
}
enum db_return_codes db_add_platform_info_capability(const PersistentStore *p_ps,
	struct db_platform_info_capability *p_platform_info_capability)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO platform_info_capability \
		(id, type, length, mgmt_sw_config_support, mem_mode_capabilities, current_mem_mode, pmem_ras_capabilities)  \
		VALUES 		\
		($id, \
		$type, \
		$length, \
		$mgmt_sw_config_support, \
		$mem_mode_capabilities, \
		$current_mem_mode, \
		$pmem_ras_capabilities) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_platform_info_capability(p_stmt, p_platform_info_capability);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_platform_info_capability_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "platform_info_capability", p_count);
}
int db_get_platform_info_capabilitys(const PersistentStore *p_ps,
	struct db_platform_info_capability *p_platform_info_capability,
	int platform_info_capability_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_platform_info_capability, 0, sizeof (struct db_platform_info_capability) * platform_info_capability_count);
	char *sql = "SELECT \
		id \
		,  type \
		,  length \
		,  mgmt_sw_config_support \
		,  mem_mode_capabilities \
		,  current_mem_mode \
		,  pmem_ras_capabilities \
		  \
		FROM platform_info_capability \
		        \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < platform_info_capability_count)
		{
			local_row_to_platform_info_capability(p_ps, p_stmt, &p_platform_info_capability[index]);
			local_get_platform_info_capability_relationships(p_ps, p_stmt, &p_platform_info_capability[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_platform_info_capabilitys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM platform_info_capability");
}

enum db_return_codes db_save_platform_info_capability_state(const PersistentStore *p_ps,
	int history_id,
	struct db_platform_info_capability *p_platform_info_capability)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_platform_info_capability temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_platform_info_capability_by_id(p_ps, p_platform_info_capability->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_platform_info_capability_by_id(p_ps,
				p_platform_info_capability->id,
				p_platform_info_capability);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO platform_info_capability \
			( id ,  type ,  length ,  mgmt_sw_config_support ,  mem_mode_capabilities ,  current_mem_mode ,  pmem_ras_capabilities )  \
			VALUES 		\
			($id, \
			$type, \
			$length, \
			$mgmt_sw_config_support, \
			$mem_mode_capabilities, \
			$current_mem_mode, \
			$pmem_ras_capabilities) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_platform_info_capability(p_stmt, p_platform_info_capability);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO platform_info_capability_history \
			(history_id, \
				 id,  type,  length,  mgmt_sw_config_support,  mem_mode_capabilities,  current_mem_mode,  pmem_ras_capabilities)  \
			VALUES 		($history_id, \
				 $id , \
				 $type , \
				 $length , \
				 $mgmt_sw_config_support , \
				 $mem_mode_capabilities , \
				 $current_mem_mode , \
				 $pmem_ras_capabilities )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_platform_info_capability(p_stmt, p_platform_info_capability);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_platform_info_capability_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_platform_info_capability *p_platform_info_capability)
{
	memset(p_platform_info_capability, 0, sizeof (struct db_platform_info_capability));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  type,  length,  mgmt_sw_config_support,  mem_mode_capabilities,  current_mem_mode,  pmem_ras_capabilities  \
		FROM platform_info_capability \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_platform_info_capability(p_ps, p_stmt, p_platform_info_capability);
			local_get_platform_info_capability_relationships(p_ps, p_stmt, p_platform_info_capability);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_platform_info_capability_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_platform_info_capability *p_platform_info_capability)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE platform_info_capability \
	SET \
	id=$id \
		,  type=$type \
		,  length=$length \
		,  mgmt_sw_config_support=$mgmt_sw_config_support \
		,  mem_mode_capabilities=$mem_mode_capabilities \
		,  current_mem_mode=$current_mem_mode \
		,  pmem_ras_capabilities=$pmem_ras_capabilities \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_platform_info_capability(p_stmt, p_platform_info_capability);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_platform_info_capability_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM platform_info_capability \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_platform_info_capability_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM platform_info_capability_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_platform_info_capability_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM platform_info_capability_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_platform_info_capability_history_by_history_id(const PersistentStore *p_ps,
	struct db_platform_info_capability *p_platform_info_capability,
	int history_id,
	int platform_info_capability_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_platform_info_capability, 0, sizeof (struct db_platform_info_capability) * platform_info_capability_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  type,  length,  mgmt_sw_config_support,  mem_mode_capabilities,  current_mem_mode,  pmem_ras_capabilities  \
		FROM platform_info_capability_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < platform_info_capability_count)
		{
		rc = DB_SUCCESS;
		local_row_to_platform_info_capability(p_ps, p_stmt, &p_platform_info_capability[index]);
		local_get_platform_info_capability_relationships_history(p_ps, p_stmt, &p_platform_info_capability[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_platform_info_capability_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM platform_info_capability_history");
}

/*!
 * Roll platform_info_capabilitys by id to specified max.
 */
enum db_return_codes db_roll_platform_info_capabilitys_by_id(const PersistentStore *p_ps,
	int max_rows)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char sql[1024];
	snprintf(sql, 1024,
				"DELETE FROM platform_info_capability "
				"WHERE id NOT IN ("
				"SELECT id "
				"FROM platform_info_capability "
				"ORDER BY id DESC "
				"LIMIT %d)", max_rows); 
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Get the max id in the platform_info_capability table.
 */
enum db_return_codes db_get_next_platform_info_capability_id(const PersistentStore *p_ps, int *p_max)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	if (p_max)
	{
		if ((rc = run_scalar_sql(p_ps, 
			"SELECT MAX(id) FROM platform_info_capability", p_max)) 
			==  DB_SUCCESS)
		{
			(*p_max)++;
		}
	}
	return rc;
}
/*
 * --- END platform_info_capability ----------------
 */
/*
 * --- platform_capabilities ----------------
 */
void local_bind_platform_capabilities(sqlite3_stmt *p_stmt, struct db_platform_capabilities *p_platform_capabilities)
{
	BIND_TEXT(p_stmt, "$signature", (char *)p_platform_capabilities->signature);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_platform_capabilities->length);
	BIND_INTEGER(p_stmt, "$revision", (unsigned int)p_platform_capabilities->revision);
	BIND_INTEGER(p_stmt, "$checksum", (unsigned int)p_platform_capabilities->checksum);
	BIND_TEXT(p_stmt, "$oem_id", (char *)p_platform_capabilities->oem_id);
	BIND_TEXT(p_stmt, "$oem_table_id", (char *)p_platform_capabilities->oem_table_id);
	BIND_INTEGER(p_stmt, "$oem_revision", (unsigned int)p_platform_capabilities->oem_revision);
	BIND_INTEGER(p_stmt, "$creator_id_0", (unsigned int)p_platform_capabilities->creator_id[0]);
	BIND_INTEGER(p_stmt, "$creator_id_1", (unsigned int)p_platform_capabilities->creator_id[1]);
	BIND_INTEGER(p_stmt, "$creator_id_2", (unsigned int)p_platform_capabilities->creator_id[2]);
	BIND_INTEGER(p_stmt, "$creator_id_3", (unsigned int)p_platform_capabilities->creator_id[3]);
	BIND_INTEGER(p_stmt, "$creator_revision", (unsigned int)p_platform_capabilities->creator_revision);
}
void local_get_platform_capabilities_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_platform_capabilities *p_platform_capabilities)
{
}

void local_get_platform_capabilities_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_platform_capabilities *p_platform_capabilities,
	int history_id)
{
}

void local_row_to_platform_capabilities(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_platform_capabilities *p_platform_capabilities)
{
	TEXT_COLUMN(p_stmt,
		0,
		p_platform_capabilities->signature,
		PLATFORM_CAPABILITIES_SIGNATURE_LEN);
	INTEGER_COLUMN(p_stmt,
		1,
		p_platform_capabilities->length);
	INTEGER_COLUMN(p_stmt,
		2,
		p_platform_capabilities->revision);
	INTEGER_COLUMN(p_stmt,
		3,
		p_platform_capabilities->checksum);
	TEXT_COLUMN(p_stmt,
		4,
		p_platform_capabilities->oem_id,
		PLATFORM_CAPABILITIES_OEM_ID_LEN);
	TEXT_COLUMN(p_stmt,
		5,
		p_platform_capabilities->oem_table_id,
		PLATFORM_CAPABILITIES_OEM_TABLE_ID_LEN);
	INTEGER_COLUMN(p_stmt,
		6,
		p_platform_capabilities->oem_revision);
	INTEGER_COLUMN(p_stmt,
		7,
		p_platform_capabilities->creator_id[0]);
	INTEGER_COLUMN(p_stmt,
		8,
		p_platform_capabilities->creator_id[1]);
	INTEGER_COLUMN(p_stmt,
		9,
		p_platform_capabilities->creator_id[2]);
	INTEGER_COLUMN(p_stmt,
		10,
		p_platform_capabilities->creator_id[3]);
	INTEGER_COLUMN(p_stmt,
		11,
		p_platform_capabilities->creator_revision);
}
void db_print_platform_capabilities(struct db_platform_capabilities *p_value)
{
	printf("platform_capabilities.signature: %s\n", p_value->signature);
	printf("platform_capabilities.length: unsigned %d\n", p_value->length);
	printf("platform_capabilities.revision: unsigned %d\n", p_value->revision);
	printf("platform_capabilities.checksum: unsigned %d\n", p_value->checksum);
	printf("platform_capabilities.oem_id: %s\n", p_value->oem_id);
	printf("platform_capabilities.oem_table_id: %s\n", p_value->oem_table_id);
	printf("platform_capabilities.oem_revision: unsigned %d\n", p_value->oem_revision);
	printf("platform_capabilities.creator_id: unsigned %d\n", p_value->creator_id[0]);
	printf("platform_capabilities.creator_id: unsigned %d\n", p_value->creator_id[1]);
	printf("platform_capabilities.creator_id: unsigned %d\n", p_value->creator_id[2]);
	printf("platform_capabilities.creator_id: unsigned %d\n", p_value->creator_id[3]);
	printf("platform_capabilities.creator_revision: unsigned %d\n", p_value->creator_revision);
}
enum db_return_codes db_add_platform_capabilities(const PersistentStore *p_ps,
	struct db_platform_capabilities *p_platform_capabilities)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO platform_capabilities \
		(signature, length, revision, checksum, oem_id, oem_table_id, oem_revision, creator_id_0, creator_id_1, creator_id_2, creator_id_3, creator_revision)  \
		VALUES 		\
		($signature, \
		$length, \
		$revision, \
		$checksum, \
		$oem_id, \
		$oem_table_id, \
		$oem_revision, \
		$creator_id_0, \
		$creator_id_1, \
		$creator_id_2, \
		$creator_id_3, \
		$creator_revision) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_platform_capabilities(p_stmt, p_platform_capabilities);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_platform_capabilities_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "platform_capabilities", p_count);
}
int db_get_platform_capabilitiess(const PersistentStore *p_ps,
	struct db_platform_capabilities *p_platform_capabilities,
	int platform_capabilities_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_platform_capabilities, 0, sizeof (struct db_platform_capabilities) * platform_capabilities_count);
	char *sql = "SELECT \
		signature \
		,  length \
		,  revision \
		,  checksum \
		,  oem_id \
		,  oem_table_id \
		,  oem_revision \
		,  creator_id_0 \
		,  creator_id_1 \
		,  creator_id_2 \
		,  creator_id_3 \
		,  creator_revision \
		  \
		FROM platform_capabilities \
		             \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < platform_capabilities_count)
		{
			local_row_to_platform_capabilities(p_ps, p_stmt, &p_platform_capabilities[index]);
			local_get_platform_capabilities_relationships(p_ps, p_stmt, &p_platform_capabilities[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_platform_capabilitiess(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM platform_capabilities");
}

enum db_return_codes db_save_platform_capabilities_state(const PersistentStore *p_ps,
	int history_id,
	struct db_platform_capabilities *p_platform_capabilities)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_platform_capabilities temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_platform_capabilities_by_signature(p_ps, p_platform_capabilities->signature, &temp) == DB_SUCCESS)
	{
		rc = db_update_platform_capabilities_by_signature(p_ps,
				p_platform_capabilities->signature,
				p_platform_capabilities);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO platform_capabilities \
			( signature ,  length ,  revision ,  checksum ,  oem_id ,  oem_table_id ,  oem_revision ,  creator_id_0 ,  creator_id_1 ,  creator_id_2 ,  creator_id_3 ,  creator_revision )  \
			VALUES 		\
			($signature, \
			$length, \
			$revision, \
			$checksum, \
			$oem_id, \
			$oem_table_id, \
			$oem_revision, \
			$creator_id_0, \
			$creator_id_1, \
			$creator_id_2, \
			$creator_id_3, \
			$creator_revision) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_platform_capabilities(p_stmt, p_platform_capabilities);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO platform_capabilities_history \
			(history_id, \
				 signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id_0,  creator_id_1,  creator_id_2,  creator_id_3,  creator_revision)  \
			VALUES 		($history_id, \
				 $signature , \
				 $length , \
				 $revision , \
				 $checksum , \
				 $oem_id , \
				 $oem_table_id , \
				 $oem_revision , \
				 $creator_id_0 , \
				 $creator_id_1 , \
				 $creator_id_2 , \
				 $creator_id_3 , \
				 $creator_revision )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_platform_capabilities(p_stmt, p_platform_capabilities);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_platform_capabilities_by_signature(const PersistentStore *p_ps,
	const char * signature,
	struct db_platform_capabilities *p_platform_capabilities)
{
	memset(p_platform_capabilities, 0, sizeof (struct db_platform_capabilities));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id_0,  creator_id_1,  creator_id_2,  creator_id_3,  creator_revision  \
		FROM platform_capabilities \
		WHERE  signature = $signature";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$signature", (char *)signature);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_platform_capabilities(p_ps, p_stmt, p_platform_capabilities);
			local_get_platform_capabilities_relationships(p_ps, p_stmt, p_platform_capabilities);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_platform_capabilities_by_signature(const PersistentStore *p_ps,
	const char * signature,
	struct db_platform_capabilities *p_platform_capabilities)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE platform_capabilities \
	SET \
	signature=$signature \
		,  length=$length \
		,  revision=$revision \
		,  checksum=$checksum \
		,  oem_id=$oem_id \
		,  oem_table_id=$oem_table_id \
		,  oem_revision=$oem_revision \
		,  creator_id_0=$creator_id_0 \
		,  creator_id_1=$creator_id_1 \
		,  creator_id_2=$creator_id_2 \
		,  creator_id_3=$creator_id_3 \
		,  creator_revision=$creator_revision \
		  \
	WHERE signature=$signature ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$signature", (char *)signature);
		local_bind_platform_capabilities(p_stmt, p_platform_capabilities);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_platform_capabilities_by_signature(const PersistentStore *p_ps,
	const char * signature)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM platform_capabilities \
				 WHERE signature = $signature";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$signature", (char *)signature);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_platform_capabilities_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM platform_capabilities_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_platform_capabilities_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM platform_capabilities_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_platform_capabilities_history_by_history_id(const PersistentStore *p_ps,
	struct db_platform_capabilities *p_platform_capabilities,
	int history_id,
	int platform_capabilities_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_platform_capabilities, 0, sizeof (struct db_platform_capabilities) * platform_capabilities_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id_0,  creator_id_1,  creator_id_2,  creator_id_3,  creator_revision  \
		FROM platform_capabilities_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < platform_capabilities_count)
		{
		rc = DB_SUCCESS;
		local_row_to_platform_capabilities(p_ps, p_stmt, &p_platform_capabilities[index]);
		local_get_platform_capabilities_relationships_history(p_ps, p_stmt, &p_platform_capabilities[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_platform_capabilities_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM platform_capabilities_history");
}

/*
 * --- END platform_capabilities ----------------
 */
/*
 * --- driver_capabilities ----------------
 */
void local_bind_driver_capabilities(sqlite3_stmt *p_stmt, struct db_driver_capabilities *p_driver_capabilities)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_driver_capabilities->id);
	BIND_INTEGER(p_stmt, "$min_namespace_size", (unsigned long long)p_driver_capabilities->min_namespace_size);
	BIND_INTEGER(p_stmt, "$max_non_continguous_namespaces", (unsigned long long)p_driver_capabilities->max_non_continguous_namespaces);
	BIND_INTEGER(p_stmt, "$block_sizes_0", (unsigned int)p_driver_capabilities->block_sizes[0]);
	BIND_INTEGER(p_stmt, "$block_sizes_1", (unsigned int)p_driver_capabilities->block_sizes[1]);
	BIND_INTEGER(p_stmt, "$block_sizes_2", (unsigned int)p_driver_capabilities->block_sizes[2]);
	BIND_INTEGER(p_stmt, "$block_sizes_3", (unsigned int)p_driver_capabilities->block_sizes[3]);
	BIND_INTEGER(p_stmt, "$block_sizes_4", (unsigned int)p_driver_capabilities->block_sizes[4]);
	BIND_INTEGER(p_stmt, "$block_sizes_5", (unsigned int)p_driver_capabilities->block_sizes[5]);
	BIND_INTEGER(p_stmt, "$block_sizes_6", (unsigned int)p_driver_capabilities->block_sizes[6]);
	BIND_INTEGER(p_stmt, "$block_sizes_7", (unsigned int)p_driver_capabilities->block_sizes[7]);
	BIND_INTEGER(p_stmt, "$block_sizes_8", (unsigned int)p_driver_capabilities->block_sizes[8]);
	BIND_INTEGER(p_stmt, "$block_sizes_9", (unsigned int)p_driver_capabilities->block_sizes[9]);
	BIND_INTEGER(p_stmt, "$block_sizes_10", (unsigned int)p_driver_capabilities->block_sizes[10]);
	BIND_INTEGER(p_stmt, "$block_sizes_11", (unsigned int)p_driver_capabilities->block_sizes[11]);
	BIND_INTEGER(p_stmt, "$block_sizes_12", (unsigned int)p_driver_capabilities->block_sizes[12]);
	BIND_INTEGER(p_stmt, "$block_sizes_13", (unsigned int)p_driver_capabilities->block_sizes[13]);
	BIND_INTEGER(p_stmt, "$block_sizes_14", (unsigned int)p_driver_capabilities->block_sizes[14]);
	BIND_INTEGER(p_stmt, "$block_sizes_15", (unsigned int)p_driver_capabilities->block_sizes[15]);
	BIND_INTEGER(p_stmt, "$num_block_sizes", (unsigned int)p_driver_capabilities->num_block_sizes);
	BIND_INTEGER(p_stmt, "$namespace_memory_page_allocation_capable", (unsigned int)p_driver_capabilities->namespace_memory_page_allocation_capable);
}
void local_get_driver_capabilities_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_capabilities *p_driver_capabilities)
{
}

void local_get_driver_capabilities_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_capabilities *p_driver_capabilities,
	int history_id)
{
}

void local_row_to_driver_capabilities(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_capabilities *p_driver_capabilities)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_driver_capabilities->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_driver_capabilities->min_namespace_size);
	INTEGER_COLUMN(p_stmt,
		2,
		p_driver_capabilities->max_non_continguous_namespaces);
	INTEGER_COLUMN(p_stmt,
		3,
		p_driver_capabilities->block_sizes[0]);
	INTEGER_COLUMN(p_stmt,
		4,
		p_driver_capabilities->block_sizes[1]);
	INTEGER_COLUMN(p_stmt,
		5,
		p_driver_capabilities->block_sizes[2]);
	INTEGER_COLUMN(p_stmt,
		6,
		p_driver_capabilities->block_sizes[3]);
	INTEGER_COLUMN(p_stmt,
		7,
		p_driver_capabilities->block_sizes[4]);
	INTEGER_COLUMN(p_stmt,
		8,
		p_driver_capabilities->block_sizes[5]);
	INTEGER_COLUMN(p_stmt,
		9,
		p_driver_capabilities->block_sizes[6]);
	INTEGER_COLUMN(p_stmt,
		10,
		p_driver_capabilities->block_sizes[7]);
	INTEGER_COLUMN(p_stmt,
		11,
		p_driver_capabilities->block_sizes[8]);
	INTEGER_COLUMN(p_stmt,
		12,
		p_driver_capabilities->block_sizes[9]);
	INTEGER_COLUMN(p_stmt,
		13,
		p_driver_capabilities->block_sizes[10]);
	INTEGER_COLUMN(p_stmt,
		14,
		p_driver_capabilities->block_sizes[11]);
	INTEGER_COLUMN(p_stmt,
		15,
		p_driver_capabilities->block_sizes[12]);
	INTEGER_COLUMN(p_stmt,
		16,
		p_driver_capabilities->block_sizes[13]);
	INTEGER_COLUMN(p_stmt,
		17,
		p_driver_capabilities->block_sizes[14]);
	INTEGER_COLUMN(p_stmt,
		18,
		p_driver_capabilities->block_sizes[15]);
	INTEGER_COLUMN(p_stmt,
		19,
		p_driver_capabilities->num_block_sizes);
	INTEGER_COLUMN(p_stmt,
		20,
		p_driver_capabilities->namespace_memory_page_allocation_capable);
}
void db_print_driver_capabilities(struct db_driver_capabilities *p_value)
{
	printf("driver_capabilities.id: %d\n", p_value->id);
	printf("driver_capabilities.min_namespace_size: unsigned %lld\n", p_value->min_namespace_size);
	printf("driver_capabilities.max_non_continguous_namespaces: unsigned %lld\n", p_value->max_non_continguous_namespaces);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[0]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[1]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[2]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[3]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[4]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[5]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[6]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[7]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[8]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[9]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[10]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[11]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[12]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[13]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[14]);
	printf("driver_capabilities.block_sizes: unsigned %d\n", p_value->block_sizes[15]);
	printf("driver_capabilities.num_block_sizes: unsigned %d\n", p_value->num_block_sizes);
	printf("driver_capabilities.namespace_memory_page_allocation_capable: unsigned %d\n", p_value->namespace_memory_page_allocation_capable);
}
enum db_return_codes db_add_driver_capabilities(const PersistentStore *p_ps,
	struct db_driver_capabilities *p_driver_capabilities)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO driver_capabilities \
		(min_namespace_size, max_non_continguous_namespaces, block_sizes_0, block_sizes_1, block_sizes_2, block_sizes_3, block_sizes_4, block_sizes_5, block_sizes_6, block_sizes_7, block_sizes_8, block_sizes_9, block_sizes_10, block_sizes_11, block_sizes_12, block_sizes_13, block_sizes_14, block_sizes_15, num_block_sizes, namespace_memory_page_allocation_capable)  \
		VALUES 		\
		(\
		$min_namespace_size, \
		$max_non_continguous_namespaces, \
		$block_sizes_0, \
		$block_sizes_1, \
		$block_sizes_2, \
		$block_sizes_3, \
		$block_sizes_4, \
		$block_sizes_5, \
		$block_sizes_6, \
		$block_sizes_7, \
		$block_sizes_8, \
		$block_sizes_9, \
		$block_sizes_10, \
		$block_sizes_11, \
		$block_sizes_12, \
		$block_sizes_13, \
		$block_sizes_14, \
		$block_sizes_15, \
		$num_block_sizes, \
		$namespace_memory_page_allocation_capable) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_driver_capabilities(p_stmt, p_driver_capabilities);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_driver_capabilities_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "driver_capabilities", p_count);
}
int db_get_driver_capabilitiess(const PersistentStore *p_ps,
	struct db_driver_capabilities *p_driver_capabilities,
	int driver_capabilities_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_driver_capabilities, 0, sizeof (struct db_driver_capabilities) * driver_capabilities_count);
	char *sql = "SELECT \
		id \
		,  min_namespace_size \
		,  max_non_continguous_namespaces \
		,  block_sizes_0 \
		,  block_sizes_1 \
		,  block_sizes_2 \
		,  block_sizes_3 \
		,  block_sizes_4 \
		,  block_sizes_5 \
		,  block_sizes_6 \
		,  block_sizes_7 \
		,  block_sizes_8 \
		,  block_sizes_9 \
		,  block_sizes_10 \
		,  block_sizes_11 \
		,  block_sizes_12 \
		,  block_sizes_13 \
		,  block_sizes_14 \
		,  block_sizes_15 \
		,  num_block_sizes \
		,  namespace_memory_page_allocation_capable \
		  \
		FROM driver_capabilities \
		                      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < driver_capabilities_count)
		{
			local_row_to_driver_capabilities(p_ps, p_stmt, &p_driver_capabilities[index]);
			local_get_driver_capabilities_relationships(p_ps, p_stmt, &p_driver_capabilities[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_driver_capabilitiess(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM driver_capabilities");
}

enum db_return_codes db_save_driver_capabilities_state(const PersistentStore *p_ps,
	int history_id,
	struct db_driver_capabilities *p_driver_capabilities)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_driver_capabilities temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_driver_capabilities_by_id(p_ps, p_driver_capabilities->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_driver_capabilities_by_id(p_ps,
				p_driver_capabilities->id,
				p_driver_capabilities);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO driver_capabilities \
			( id ,  min_namespace_size ,  max_non_continguous_namespaces ,  block_sizes_0 ,  block_sizes_1 ,  block_sizes_2 ,  block_sizes_3 ,  block_sizes_4 ,  block_sizes_5 ,  block_sizes_6 ,  block_sizes_7 ,  block_sizes_8 ,  block_sizes_9 ,  block_sizes_10 ,  block_sizes_11 ,  block_sizes_12 ,  block_sizes_13 ,  block_sizes_14 ,  block_sizes_15 ,  num_block_sizes ,  namespace_memory_page_allocation_capable )  \
			VALUES 		\
			($id, \
			$min_namespace_size, \
			$max_non_continguous_namespaces, \
			$block_sizes_0, \
			$block_sizes_1, \
			$block_sizes_2, \
			$block_sizes_3, \
			$block_sizes_4, \
			$block_sizes_5, \
			$block_sizes_6, \
			$block_sizes_7, \
			$block_sizes_8, \
			$block_sizes_9, \
			$block_sizes_10, \
			$block_sizes_11, \
			$block_sizes_12, \
			$block_sizes_13, \
			$block_sizes_14, \
			$block_sizes_15, \
			$num_block_sizes, \
			$namespace_memory_page_allocation_capable) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_driver_capabilities(p_stmt, p_driver_capabilities);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO driver_capabilities_history \
			(history_id, \
				 id,  min_namespace_size,  max_non_continguous_namespaces,  block_sizes_0,  block_sizes_1,  block_sizes_2,  block_sizes_3,  block_sizes_4,  block_sizes_5,  block_sizes_6,  block_sizes_7,  block_sizes_8,  block_sizes_9,  block_sizes_10,  block_sizes_11,  block_sizes_12,  block_sizes_13,  block_sizes_14,  block_sizes_15,  num_block_sizes,  namespace_memory_page_allocation_capable)  \
			VALUES 		($history_id, \
				 $id , \
				 $min_namespace_size , \
				 $max_non_continguous_namespaces , \
				 $block_sizes_0 , \
				 $block_sizes_1 , \
				 $block_sizes_2 , \
				 $block_sizes_3 , \
				 $block_sizes_4 , \
				 $block_sizes_5 , \
				 $block_sizes_6 , \
				 $block_sizes_7 , \
				 $block_sizes_8 , \
				 $block_sizes_9 , \
				 $block_sizes_10 , \
				 $block_sizes_11 , \
				 $block_sizes_12 , \
				 $block_sizes_13 , \
				 $block_sizes_14 , \
				 $block_sizes_15 , \
				 $num_block_sizes , \
				 $namespace_memory_page_allocation_capable )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_driver_capabilities(p_stmt, p_driver_capabilities);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_driver_capabilities_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_driver_capabilities *p_driver_capabilities)
{
	memset(p_driver_capabilities, 0, sizeof (struct db_driver_capabilities));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  min_namespace_size,  max_non_continguous_namespaces,  block_sizes_0,  block_sizes_1,  block_sizes_2,  block_sizes_3,  block_sizes_4,  block_sizes_5,  block_sizes_6,  block_sizes_7,  block_sizes_8,  block_sizes_9,  block_sizes_10,  block_sizes_11,  block_sizes_12,  block_sizes_13,  block_sizes_14,  block_sizes_15,  num_block_sizes,  namespace_memory_page_allocation_capable  \
		FROM driver_capabilities \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_driver_capabilities(p_ps, p_stmt, p_driver_capabilities);
			local_get_driver_capabilities_relationships(p_ps, p_stmt, p_driver_capabilities);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_driver_capabilities_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_driver_capabilities *p_driver_capabilities)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE driver_capabilities \
	SET \
	id=$id \
		,  min_namespace_size=$min_namespace_size \
		,  max_non_continguous_namespaces=$max_non_continguous_namespaces \
		,  block_sizes_0=$block_sizes_0 \
		,  block_sizes_1=$block_sizes_1 \
		,  block_sizes_2=$block_sizes_2 \
		,  block_sizes_3=$block_sizes_3 \
		,  block_sizes_4=$block_sizes_4 \
		,  block_sizes_5=$block_sizes_5 \
		,  block_sizes_6=$block_sizes_6 \
		,  block_sizes_7=$block_sizes_7 \
		,  block_sizes_8=$block_sizes_8 \
		,  block_sizes_9=$block_sizes_9 \
		,  block_sizes_10=$block_sizes_10 \
		,  block_sizes_11=$block_sizes_11 \
		,  block_sizes_12=$block_sizes_12 \
		,  block_sizes_13=$block_sizes_13 \
		,  block_sizes_14=$block_sizes_14 \
		,  block_sizes_15=$block_sizes_15 \
		,  num_block_sizes=$num_block_sizes \
		,  namespace_memory_page_allocation_capable=$namespace_memory_page_allocation_capable \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_driver_capabilities(p_stmt, p_driver_capabilities);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_driver_capabilities_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM driver_capabilities \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_driver_capabilities_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM driver_capabilities_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_driver_capabilities_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM driver_capabilities_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_driver_capabilities_history_by_history_id(const PersistentStore *p_ps,
	struct db_driver_capabilities *p_driver_capabilities,
	int history_id,
	int driver_capabilities_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_driver_capabilities, 0, sizeof (struct db_driver_capabilities) * driver_capabilities_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  min_namespace_size,  max_non_continguous_namespaces,  block_sizes_0,  block_sizes_1,  block_sizes_2,  block_sizes_3,  block_sizes_4,  block_sizes_5,  block_sizes_6,  block_sizes_7,  block_sizes_8,  block_sizes_9,  block_sizes_10,  block_sizes_11,  block_sizes_12,  block_sizes_13,  block_sizes_14,  block_sizes_15,  num_block_sizes,  namespace_memory_page_allocation_capable  \
		FROM driver_capabilities_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < driver_capabilities_count)
		{
		rc = DB_SUCCESS;
		local_row_to_driver_capabilities(p_ps, p_stmt, &p_driver_capabilities[index]);
		local_get_driver_capabilities_relationships_history(p_ps, p_stmt, &p_driver_capabilities[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_driver_capabilities_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM driver_capabilities_history");
}

/*
 * --- END driver_capabilities ----------------
 */
/*
 * --- driver_features ----------------
 */
void local_bind_driver_features(sqlite3_stmt *p_stmt, struct db_driver_features *p_driver_features)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_driver_features->id);
	BIND_INTEGER(p_stmt, "$get_platform_capabilities", (unsigned int)p_driver_features->get_platform_capabilities);
	BIND_INTEGER(p_stmt, "$get_topology", (unsigned int)p_driver_features->get_topology);
	BIND_INTEGER(p_stmt, "$get_interleave", (unsigned int)p_driver_features->get_interleave);
	BIND_INTEGER(p_stmt, "$get_dimm_detail", (unsigned int)p_driver_features->get_dimm_detail);
	BIND_INTEGER(p_stmt, "$get_namespaces", (unsigned int)p_driver_features->get_namespaces);
	BIND_INTEGER(p_stmt, "$get_namespace_detail", (unsigned int)p_driver_features->get_namespace_detail);
	BIND_INTEGER(p_stmt, "$get_address_scrub_data", (unsigned int)p_driver_features->get_address_scrub_data);
	BIND_INTEGER(p_stmt, "$get_platform_config_data", (unsigned int)p_driver_features->get_platform_config_data);
	BIND_INTEGER(p_stmt, "$get_boot_status", (unsigned int)p_driver_features->get_boot_status);
	BIND_INTEGER(p_stmt, "$get_power_data", (unsigned int)p_driver_features->get_power_data);
	BIND_INTEGER(p_stmt, "$get_security_state", (unsigned int)p_driver_features->get_security_state);
	BIND_INTEGER(p_stmt, "$get_log_page", (unsigned int)p_driver_features->get_log_page);
	BIND_INTEGER(p_stmt, "$get_features", (unsigned int)p_driver_features->get_features);
	BIND_INTEGER(p_stmt, "$set_features", (unsigned int)p_driver_features->set_features);
	BIND_INTEGER(p_stmt, "$create_namespace", (unsigned int)p_driver_features->create_namespace);
	BIND_INTEGER(p_stmt, "$rename_namespace", (unsigned int)p_driver_features->rename_namespace);
	BIND_INTEGER(p_stmt, "$grow_namespace", (unsigned int)p_driver_features->grow_namespace);
	BIND_INTEGER(p_stmt, "$shrink_namespace", (unsigned int)p_driver_features->shrink_namespace);
	BIND_INTEGER(p_stmt, "$delete_namespace", (unsigned int)p_driver_features->delete_namespace);
	BIND_INTEGER(p_stmt, "$enable_namespace", (unsigned int)p_driver_features->enable_namespace);
	BIND_INTEGER(p_stmt, "$disable_namespace", (unsigned int)p_driver_features->disable_namespace);
	BIND_INTEGER(p_stmt, "$set_security_state", (unsigned int)p_driver_features->set_security_state);
	BIND_INTEGER(p_stmt, "$enable_logging", (unsigned int)p_driver_features->enable_logging);
	BIND_INTEGER(p_stmt, "$run_diagnostic", (unsigned int)p_driver_features->run_diagnostic);
	BIND_INTEGER(p_stmt, "$set_platform_config", (unsigned int)p_driver_features->set_platform_config);
	BIND_INTEGER(p_stmt, "$passthrough", (unsigned int)p_driver_features->passthrough);
	BIND_INTEGER(p_stmt, "$start_address_scrub", (unsigned int)p_driver_features->start_address_scrub);
	BIND_INTEGER(p_stmt, "$app_direct_mode", (unsigned int)p_driver_features->app_direct_mode);
	BIND_INTEGER(p_stmt, "$storage_mode", (unsigned int)p_driver_features->storage_mode);
}
void local_get_driver_features_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_features *p_driver_features)
{
}

void local_get_driver_features_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_features *p_driver_features,
	int history_id)
{
}

void local_row_to_driver_features(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_features *p_driver_features)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_driver_features->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_driver_features->get_platform_capabilities);
	INTEGER_COLUMN(p_stmt,
		2,
		p_driver_features->get_topology);
	INTEGER_COLUMN(p_stmt,
		3,
		p_driver_features->get_interleave);
	INTEGER_COLUMN(p_stmt,
		4,
		p_driver_features->get_dimm_detail);
	INTEGER_COLUMN(p_stmt,
		5,
		p_driver_features->get_namespaces);
	INTEGER_COLUMN(p_stmt,
		6,
		p_driver_features->get_namespace_detail);
	INTEGER_COLUMN(p_stmt,
		7,
		p_driver_features->get_address_scrub_data);
	INTEGER_COLUMN(p_stmt,
		8,
		p_driver_features->get_platform_config_data);
	INTEGER_COLUMN(p_stmt,
		9,
		p_driver_features->get_boot_status);
	INTEGER_COLUMN(p_stmt,
		10,
		p_driver_features->get_power_data);
	INTEGER_COLUMN(p_stmt,
		11,
		p_driver_features->get_security_state);
	INTEGER_COLUMN(p_stmt,
		12,
		p_driver_features->get_log_page);
	INTEGER_COLUMN(p_stmt,
		13,
		p_driver_features->get_features);
	INTEGER_COLUMN(p_stmt,
		14,
		p_driver_features->set_features);
	INTEGER_COLUMN(p_stmt,
		15,
		p_driver_features->create_namespace);
	INTEGER_COLUMN(p_stmt,
		16,
		p_driver_features->rename_namespace);
	INTEGER_COLUMN(p_stmt,
		17,
		p_driver_features->grow_namespace);
	INTEGER_COLUMN(p_stmt,
		18,
		p_driver_features->shrink_namespace);
	INTEGER_COLUMN(p_stmt,
		19,
		p_driver_features->delete_namespace);
	INTEGER_COLUMN(p_stmt,
		20,
		p_driver_features->enable_namespace);
	INTEGER_COLUMN(p_stmt,
		21,
		p_driver_features->disable_namespace);
	INTEGER_COLUMN(p_stmt,
		22,
		p_driver_features->set_security_state);
	INTEGER_COLUMN(p_stmt,
		23,
		p_driver_features->enable_logging);
	INTEGER_COLUMN(p_stmt,
		24,
		p_driver_features->run_diagnostic);
	INTEGER_COLUMN(p_stmt,
		25,
		p_driver_features->set_platform_config);
	INTEGER_COLUMN(p_stmt,
		26,
		p_driver_features->passthrough);
	INTEGER_COLUMN(p_stmt,
		27,
		p_driver_features->start_address_scrub);
	INTEGER_COLUMN(p_stmt,
		28,
		p_driver_features->app_direct_mode);
	INTEGER_COLUMN(p_stmt,
		29,
		p_driver_features->storage_mode);
}
void db_print_driver_features(struct db_driver_features *p_value)
{
	printf("driver_features.id: %d\n", p_value->id);
	printf("driver_features.get_platform_capabilities: unsigned %d\n", p_value->get_platform_capabilities);
	printf("driver_features.get_topology: unsigned %d\n", p_value->get_topology);
	printf("driver_features.get_interleave: unsigned %d\n", p_value->get_interleave);
	printf("driver_features.get_dimm_detail: unsigned %d\n", p_value->get_dimm_detail);
	printf("driver_features.get_namespaces: unsigned %d\n", p_value->get_namespaces);
	printf("driver_features.get_namespace_detail: unsigned %d\n", p_value->get_namespace_detail);
	printf("driver_features.get_address_scrub_data: unsigned %d\n", p_value->get_address_scrub_data);
	printf("driver_features.get_platform_config_data: unsigned %d\n", p_value->get_platform_config_data);
	printf("driver_features.get_boot_status: unsigned %d\n", p_value->get_boot_status);
	printf("driver_features.get_power_data: unsigned %d\n", p_value->get_power_data);
	printf("driver_features.get_security_state: unsigned %d\n", p_value->get_security_state);
	printf("driver_features.get_log_page: unsigned %d\n", p_value->get_log_page);
	printf("driver_features.get_features: unsigned %d\n", p_value->get_features);
	printf("driver_features.set_features: unsigned %d\n", p_value->set_features);
	printf("driver_features.create_namespace: unsigned %d\n", p_value->create_namespace);
	printf("driver_features.rename_namespace: unsigned %d\n", p_value->rename_namespace);
	printf("driver_features.grow_namespace: unsigned %d\n", p_value->grow_namespace);
	printf("driver_features.shrink_namespace: unsigned %d\n", p_value->shrink_namespace);
	printf("driver_features.delete_namespace: unsigned %d\n", p_value->delete_namespace);
	printf("driver_features.enable_namespace: unsigned %d\n", p_value->enable_namespace);
	printf("driver_features.disable_namespace: unsigned %d\n", p_value->disable_namespace);
	printf("driver_features.set_security_state: unsigned %d\n", p_value->set_security_state);
	printf("driver_features.enable_logging: unsigned %d\n", p_value->enable_logging);
	printf("driver_features.run_diagnostic: unsigned %d\n", p_value->run_diagnostic);
	printf("driver_features.set_platform_config: unsigned %d\n", p_value->set_platform_config);
	printf("driver_features.passthrough: unsigned %d\n", p_value->passthrough);
	printf("driver_features.start_address_scrub: unsigned %d\n", p_value->start_address_scrub);
	printf("driver_features.app_direct_mode: unsigned %d\n", p_value->app_direct_mode);
	printf("driver_features.storage_mode: unsigned %d\n", p_value->storage_mode);
}
enum db_return_codes db_add_driver_features(const PersistentStore *p_ps,
	struct db_driver_features *p_driver_features)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO driver_features \
		(get_platform_capabilities, get_topology, get_interleave, get_dimm_detail, get_namespaces, get_namespace_detail, get_address_scrub_data, get_platform_config_data, get_boot_status, get_power_data, get_security_state, get_log_page, get_features, set_features, create_namespace, rename_namespace, grow_namespace, shrink_namespace, delete_namespace, enable_namespace, disable_namespace, set_security_state, enable_logging, run_diagnostic, set_platform_config, passthrough, start_address_scrub, app_direct_mode, storage_mode)  \
		VALUES 		\
		(\
		$get_platform_capabilities, \
		$get_topology, \
		$get_interleave, \
		$get_dimm_detail, \
		$get_namespaces, \
		$get_namespace_detail, \
		$get_address_scrub_data, \
		$get_platform_config_data, \
		$get_boot_status, \
		$get_power_data, \
		$get_security_state, \
		$get_log_page, \
		$get_features, \
		$set_features, \
		$create_namespace, \
		$rename_namespace, \
		$grow_namespace, \
		$shrink_namespace, \
		$delete_namespace, \
		$enable_namespace, \
		$disable_namespace, \
		$set_security_state, \
		$enable_logging, \
		$run_diagnostic, \
		$set_platform_config, \
		$passthrough, \
		$start_address_scrub, \
		$app_direct_mode, \
		$storage_mode) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_driver_features(p_stmt, p_driver_features);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_driver_features_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "driver_features", p_count);
}
int db_get_driver_featuress(const PersistentStore *p_ps,
	struct db_driver_features *p_driver_features,
	int driver_features_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_driver_features, 0, sizeof (struct db_driver_features) * driver_features_count);
	char *sql = "SELECT \
		id \
		,  get_platform_capabilities \
		,  get_topology \
		,  get_interleave \
		,  get_dimm_detail \
		,  get_namespaces \
		,  get_namespace_detail \
		,  get_address_scrub_data \
		,  get_platform_config_data \
		,  get_boot_status \
		,  get_power_data \
		,  get_security_state \
		,  get_log_page \
		,  get_features \
		,  set_features \
		,  create_namespace \
		,  rename_namespace \
		,  grow_namespace \
		,  shrink_namespace \
		,  delete_namespace \
		,  enable_namespace \
		,  disable_namespace \
		,  set_security_state \
		,  enable_logging \
		,  run_diagnostic \
		,  set_platform_config \
		,  passthrough \
		,  start_address_scrub \
		,  app_direct_mode \
		,  storage_mode \
		  \
		FROM driver_features \
		                               \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < driver_features_count)
		{
			local_row_to_driver_features(p_ps, p_stmt, &p_driver_features[index]);
			local_get_driver_features_relationships(p_ps, p_stmt, &p_driver_features[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_driver_featuress(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM driver_features");
}

enum db_return_codes db_save_driver_features_state(const PersistentStore *p_ps,
	int history_id,
	struct db_driver_features *p_driver_features)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_driver_features temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_driver_features_by_id(p_ps, p_driver_features->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_driver_features_by_id(p_ps,
				p_driver_features->id,
				p_driver_features);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO driver_features \
			( id ,  get_platform_capabilities ,  get_topology ,  get_interleave ,  get_dimm_detail ,  get_namespaces ,  get_namespace_detail ,  get_address_scrub_data ,  get_platform_config_data ,  get_boot_status ,  get_power_data ,  get_security_state ,  get_log_page ,  get_features ,  set_features ,  create_namespace ,  rename_namespace ,  grow_namespace ,  shrink_namespace ,  delete_namespace ,  enable_namespace ,  disable_namespace ,  set_security_state ,  enable_logging ,  run_diagnostic ,  set_platform_config ,  passthrough ,  start_address_scrub ,  app_direct_mode ,  storage_mode )  \
			VALUES 		\
			($id, \
			$get_platform_capabilities, \
			$get_topology, \
			$get_interleave, \
			$get_dimm_detail, \
			$get_namespaces, \
			$get_namespace_detail, \
			$get_address_scrub_data, \
			$get_platform_config_data, \
			$get_boot_status, \
			$get_power_data, \
			$get_security_state, \
			$get_log_page, \
			$get_features, \
			$set_features, \
			$create_namespace, \
			$rename_namespace, \
			$grow_namespace, \
			$shrink_namespace, \
			$delete_namespace, \
			$enable_namespace, \
			$disable_namespace, \
			$set_security_state, \
			$enable_logging, \
			$run_diagnostic, \
			$set_platform_config, \
			$passthrough, \
			$start_address_scrub, \
			$app_direct_mode, \
			$storage_mode) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_driver_features(p_stmt, p_driver_features);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO driver_features_history \
			(history_id, \
				 id,  get_platform_capabilities,  get_topology,  get_interleave,  get_dimm_detail,  get_namespaces,  get_namespace_detail,  get_address_scrub_data,  get_platform_config_data,  get_boot_status,  get_power_data,  get_security_state,  get_log_page,  get_features,  set_features,  create_namespace,  rename_namespace,  grow_namespace,  shrink_namespace,  delete_namespace,  enable_namespace,  disable_namespace,  set_security_state,  enable_logging,  run_diagnostic,  set_platform_config,  passthrough,  start_address_scrub,  app_direct_mode,  storage_mode)  \
			VALUES 		($history_id, \
				 $id , \
				 $get_platform_capabilities , \
				 $get_topology , \
				 $get_interleave , \
				 $get_dimm_detail , \
				 $get_namespaces , \
				 $get_namespace_detail , \
				 $get_address_scrub_data , \
				 $get_platform_config_data , \
				 $get_boot_status , \
				 $get_power_data , \
				 $get_security_state , \
				 $get_log_page , \
				 $get_features , \
				 $set_features , \
				 $create_namespace , \
				 $rename_namespace , \
				 $grow_namespace , \
				 $shrink_namespace , \
				 $delete_namespace , \
				 $enable_namespace , \
				 $disable_namespace , \
				 $set_security_state , \
				 $enable_logging , \
				 $run_diagnostic , \
				 $set_platform_config , \
				 $passthrough , \
				 $start_address_scrub , \
				 $app_direct_mode , \
				 $storage_mode )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_driver_features(p_stmt, p_driver_features);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_driver_features_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_driver_features *p_driver_features)
{
	memset(p_driver_features, 0, sizeof (struct db_driver_features));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  get_platform_capabilities,  get_topology,  get_interleave,  get_dimm_detail,  get_namespaces,  get_namespace_detail,  get_address_scrub_data,  get_platform_config_data,  get_boot_status,  get_power_data,  get_security_state,  get_log_page,  get_features,  set_features,  create_namespace,  rename_namespace,  grow_namespace,  shrink_namespace,  delete_namespace,  enable_namespace,  disable_namespace,  set_security_state,  enable_logging,  run_diagnostic,  set_platform_config,  passthrough,  start_address_scrub,  app_direct_mode,  storage_mode  \
		FROM driver_features \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_driver_features(p_ps, p_stmt, p_driver_features);
			local_get_driver_features_relationships(p_ps, p_stmt, p_driver_features);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_driver_features_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_driver_features *p_driver_features)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE driver_features \
	SET \
	id=$id \
		,  get_platform_capabilities=$get_platform_capabilities \
		,  get_topology=$get_topology \
		,  get_interleave=$get_interleave \
		,  get_dimm_detail=$get_dimm_detail \
		,  get_namespaces=$get_namespaces \
		,  get_namespace_detail=$get_namespace_detail \
		,  get_address_scrub_data=$get_address_scrub_data \
		,  get_platform_config_data=$get_platform_config_data \
		,  get_boot_status=$get_boot_status \
		,  get_power_data=$get_power_data \
		,  get_security_state=$get_security_state \
		,  get_log_page=$get_log_page \
		,  get_features=$get_features \
		,  set_features=$set_features \
		,  create_namespace=$create_namespace \
		,  rename_namespace=$rename_namespace \
		,  grow_namespace=$grow_namespace \
		,  shrink_namespace=$shrink_namespace \
		,  delete_namespace=$delete_namespace \
		,  enable_namespace=$enable_namespace \
		,  disable_namespace=$disable_namespace \
		,  set_security_state=$set_security_state \
		,  enable_logging=$enable_logging \
		,  run_diagnostic=$run_diagnostic \
		,  set_platform_config=$set_platform_config \
		,  passthrough=$passthrough \
		,  start_address_scrub=$start_address_scrub \
		,  app_direct_mode=$app_direct_mode \
		,  storage_mode=$storage_mode \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_driver_features(p_stmt, p_driver_features);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_driver_features_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM driver_features \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_driver_features_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM driver_features_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_driver_features_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM driver_features_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_driver_features_history_by_history_id(const PersistentStore *p_ps,
	struct db_driver_features *p_driver_features,
	int history_id,
	int driver_features_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_driver_features, 0, sizeof (struct db_driver_features) * driver_features_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  get_platform_capabilities,  get_topology,  get_interleave,  get_dimm_detail,  get_namespaces,  get_namespace_detail,  get_address_scrub_data,  get_platform_config_data,  get_boot_status,  get_power_data,  get_security_state,  get_log_page,  get_features,  set_features,  create_namespace,  rename_namespace,  grow_namespace,  shrink_namespace,  delete_namespace,  enable_namespace,  disable_namespace,  set_security_state,  enable_logging,  run_diagnostic,  set_platform_config,  passthrough,  start_address_scrub,  app_direct_mode,  storage_mode  \
		FROM driver_features_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < driver_features_count)
		{
		rc = DB_SUCCESS;
		local_row_to_driver_features(p_ps, p_stmt, &p_driver_features[index]);
		local_get_driver_features_relationships_history(p_ps, p_stmt, &p_driver_features[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_driver_features_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM driver_features_history");
}

/*
 * --- END driver_features ----------------
 */
/*
 * --- dimm_topology ----------------
 */
void local_bind_dimm_topology(sqlite3_stmt *p_stmt, struct db_dimm_topology *p_dimm_topology)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_topology->device_handle);
	BIND_INTEGER(p_stmt, "$id", (unsigned int)p_dimm_topology->id);
	BIND_INTEGER(p_stmt, "$vendor_id", (unsigned int)p_dimm_topology->vendor_id);
	BIND_INTEGER(p_stmt, "$device_id", (unsigned int)p_dimm_topology->device_id);
	BIND_INTEGER(p_stmt, "$revision_id", (unsigned int)p_dimm_topology->revision_id);
	BIND_INTEGER(p_stmt, "$subsystem_vendor_id", (unsigned int)p_dimm_topology->subsystem_vendor_id);
	BIND_INTEGER(p_stmt, "$subsystem_device_id", (unsigned int)p_dimm_topology->subsystem_device_id);
	BIND_INTEGER(p_stmt, "$subsystem_revision_id", (unsigned int)p_dimm_topology->subsystem_revision_id);
	BIND_INTEGER(p_stmt, "$manufacturing_info_valid", (unsigned int)p_dimm_topology->manufacturing_info_valid);
	BIND_INTEGER(p_stmt, "$manufacturing_location", (unsigned int)p_dimm_topology->manufacturing_location);
	BIND_INTEGER(p_stmt, "$manufacturing_date", (unsigned int)p_dimm_topology->manufacturing_date);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_dimm_topology->type);
	BIND_INTEGER(p_stmt, "$interface_format_codes_0", (unsigned int)p_dimm_topology->interface_format_codes[0]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_1", (unsigned int)p_dimm_topology->interface_format_codes[1]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_2", (unsigned int)p_dimm_topology->interface_format_codes[2]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_3", (unsigned int)p_dimm_topology->interface_format_codes[3]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_4", (unsigned int)p_dimm_topology->interface_format_codes[4]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_5", (unsigned int)p_dimm_topology->interface_format_codes[5]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_6", (unsigned int)p_dimm_topology->interface_format_codes[6]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_7", (unsigned int)p_dimm_topology->interface_format_codes[7]);
	BIND_INTEGER(p_stmt, "$interface_format_codes_8", (unsigned int)p_dimm_topology->interface_format_codes[8]);
}
void local_get_dimm_topology_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_topology *p_dimm_topology)
{
}

void local_get_dimm_topology_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_topology *p_dimm_topology,
	int history_id)
{
}

void local_row_to_dimm_topology(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_topology *p_dimm_topology)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_topology->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_topology->id);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_topology->vendor_id);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_topology->device_id);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_topology->revision_id);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_topology->subsystem_vendor_id);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_topology->subsystem_device_id);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_topology->subsystem_revision_id);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_topology->manufacturing_info_valid);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_topology->manufacturing_location);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_topology->manufacturing_date);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_topology->type);
	INTEGER_COLUMN(p_stmt,
		12,
		p_dimm_topology->interface_format_codes[0]);
	INTEGER_COLUMN(p_stmt,
		13,
		p_dimm_topology->interface_format_codes[1]);
	INTEGER_COLUMN(p_stmt,
		14,
		p_dimm_topology->interface_format_codes[2]);
	INTEGER_COLUMN(p_stmt,
		15,
		p_dimm_topology->interface_format_codes[3]);
	INTEGER_COLUMN(p_stmt,
		16,
		p_dimm_topology->interface_format_codes[4]);
	INTEGER_COLUMN(p_stmt,
		17,
		p_dimm_topology->interface_format_codes[5]);
	INTEGER_COLUMN(p_stmt,
		18,
		p_dimm_topology->interface_format_codes[6]);
	INTEGER_COLUMN(p_stmt,
		19,
		p_dimm_topology->interface_format_codes[7]);
	INTEGER_COLUMN(p_stmt,
		20,
		p_dimm_topology->interface_format_codes[8]);
}
void db_print_dimm_topology(struct db_dimm_topology *p_value)
{
	printf("dimm_topology.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_topology.id: unsigned %d\n", p_value->id);
	printf("dimm_topology.vendor_id: unsigned %d\n", p_value->vendor_id);
	printf("dimm_topology.device_id: unsigned %d\n", p_value->device_id);
	printf("dimm_topology.revision_id: unsigned %d\n", p_value->revision_id);
	printf("dimm_topology.subsystem_vendor_id: unsigned %d\n", p_value->subsystem_vendor_id);
	printf("dimm_topology.subsystem_device_id: unsigned %d\n", p_value->subsystem_device_id);
	printf("dimm_topology.subsystem_revision_id: unsigned %d\n", p_value->subsystem_revision_id);
	printf("dimm_topology.manufacturing_info_valid: unsigned %d\n", p_value->manufacturing_info_valid);
	printf("dimm_topology.manufacturing_location: unsigned %d\n", p_value->manufacturing_location);
	printf("dimm_topology.manufacturing_date: unsigned %d\n", p_value->manufacturing_date);
	printf("dimm_topology.type: unsigned %d\n", p_value->type);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[0]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[1]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[2]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[3]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[4]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[5]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[6]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[7]);
	printf("dimm_topology.interface_format_codes: unsigned %d\n", p_value->interface_format_codes[8]);
}
enum db_return_codes db_add_dimm_topology(const PersistentStore *p_ps,
	struct db_dimm_topology *p_dimm_topology)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_topology \
		(device_handle, id, vendor_id, device_id, revision_id, subsystem_vendor_id, subsystem_device_id, subsystem_revision_id, manufacturing_info_valid, manufacturing_location, manufacturing_date, type, interface_format_codes_0, interface_format_codes_1, interface_format_codes_2, interface_format_codes_3, interface_format_codes_4, interface_format_codes_5, interface_format_codes_6, interface_format_codes_7, interface_format_codes_8)  \
		VALUES 		\
		($device_handle, \
		$id, \
		$vendor_id, \
		$device_id, \
		$revision_id, \
		$subsystem_vendor_id, \
		$subsystem_device_id, \
		$subsystem_revision_id, \
		$manufacturing_info_valid, \
		$manufacturing_location, \
		$manufacturing_date, \
		$type, \
		$interface_format_codes_0, \
		$interface_format_codes_1, \
		$interface_format_codes_2, \
		$interface_format_codes_3, \
		$interface_format_codes_4, \
		$interface_format_codes_5, \
		$interface_format_codes_6, \
		$interface_format_codes_7, \
		$interface_format_codes_8) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_topology(p_stmt, p_dimm_topology);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_topology_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_topology", p_count);
}
int db_get_dimm_topologys(const PersistentStore *p_ps,
	struct db_dimm_topology *p_dimm_topology,
	int dimm_topology_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_topology, 0, sizeof (struct db_dimm_topology) * dimm_topology_count);
	char *sql = "SELECT \
		device_handle \
		,  id \
		,  vendor_id \
		,  device_id \
		,  revision_id \
		,  subsystem_vendor_id \
		,  subsystem_device_id \
		,  subsystem_revision_id \
		,  manufacturing_info_valid \
		,  manufacturing_location \
		,  manufacturing_date \
		,  type \
		,  interface_format_codes_0 \
		,  interface_format_codes_1 \
		,  interface_format_codes_2 \
		,  interface_format_codes_3 \
		,  interface_format_codes_4 \
		,  interface_format_codes_5 \
		,  interface_format_codes_6 \
		,  interface_format_codes_7 \
		,  interface_format_codes_8 \
		  \
		FROM dimm_topology \
		                      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_topology_count)
		{
			local_row_to_dimm_topology(p_ps, p_stmt, &p_dimm_topology[index]);
			local_get_dimm_topology_relationships(p_ps, p_stmt, &p_dimm_topology[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_topologys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_topology");
}

enum db_return_codes db_save_dimm_topology_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_topology *p_dimm_topology)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_topology temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_topology_by_device_handle(p_ps, p_dimm_topology->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_topology_by_device_handle(p_ps,
				p_dimm_topology->device_handle,
				p_dimm_topology);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_topology \
			( device_handle ,  id ,  vendor_id ,  device_id ,  revision_id ,  subsystem_vendor_id ,  subsystem_device_id ,  subsystem_revision_id ,  manufacturing_info_valid ,  manufacturing_location ,  manufacturing_date ,  type ,  interface_format_codes_0 ,  interface_format_codes_1 ,  interface_format_codes_2 ,  interface_format_codes_3 ,  interface_format_codes_4 ,  interface_format_codes_5 ,  interface_format_codes_6 ,  interface_format_codes_7 ,  interface_format_codes_8 )  \
			VALUES 		\
			($device_handle, \
			$id, \
			$vendor_id, \
			$device_id, \
			$revision_id, \
			$subsystem_vendor_id, \
			$subsystem_device_id, \
			$subsystem_revision_id, \
			$manufacturing_info_valid, \
			$manufacturing_location, \
			$manufacturing_date, \
			$type, \
			$interface_format_codes_0, \
			$interface_format_codes_1, \
			$interface_format_codes_2, \
			$interface_format_codes_3, \
			$interface_format_codes_4, \
			$interface_format_codes_5, \
			$interface_format_codes_6, \
			$interface_format_codes_7, \
			$interface_format_codes_8) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_topology(p_stmt, p_dimm_topology);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_topology_history \
			(history_id, \
				 device_handle,  id,  vendor_id,  device_id,  revision_id,  subsystem_vendor_id,  subsystem_device_id,  subsystem_revision_id,  manufacturing_info_valid,  manufacturing_location,  manufacturing_date,  type,  interface_format_codes_0,  interface_format_codes_1,  interface_format_codes_2,  interface_format_codes_3,  interface_format_codes_4,  interface_format_codes_5,  interface_format_codes_6,  interface_format_codes_7,  interface_format_codes_8)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $id , \
				 $vendor_id , \
				 $device_id , \
				 $revision_id , \
				 $subsystem_vendor_id , \
				 $subsystem_device_id , \
				 $subsystem_revision_id , \
				 $manufacturing_info_valid , \
				 $manufacturing_location , \
				 $manufacturing_date , \
				 $type , \
				 $interface_format_codes_0 , \
				 $interface_format_codes_1 , \
				 $interface_format_codes_2 , \
				 $interface_format_codes_3 , \
				 $interface_format_codes_4 , \
				 $interface_format_codes_5 , \
				 $interface_format_codes_6 , \
				 $interface_format_codes_7 , \
				 $interface_format_codes_8 )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_topology(p_stmt, p_dimm_topology);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_topology_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_topology *p_dimm_topology)
{
	memset(p_dimm_topology, 0, sizeof (struct db_dimm_topology));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  id,  vendor_id,  device_id,  revision_id,  subsystem_vendor_id,  subsystem_device_id,  subsystem_revision_id,  manufacturing_info_valid,  manufacturing_location,  manufacturing_date,  type,  interface_format_codes_0,  interface_format_codes_1,  interface_format_codes_2,  interface_format_codes_3,  interface_format_codes_4,  interface_format_codes_5,  interface_format_codes_6,  interface_format_codes_7,  interface_format_codes_8  \
		FROM dimm_topology \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_topology(p_ps, p_stmt, p_dimm_topology);
			local_get_dimm_topology_relationships(p_ps, p_stmt, p_dimm_topology);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_topology_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_topology *p_dimm_topology)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_topology \
	SET \
	device_handle=$device_handle \
		,  id=$id \
		,  vendor_id=$vendor_id \
		,  device_id=$device_id \
		,  revision_id=$revision_id \
		,  subsystem_vendor_id=$subsystem_vendor_id \
		,  subsystem_device_id=$subsystem_device_id \
		,  subsystem_revision_id=$subsystem_revision_id \
		,  manufacturing_info_valid=$manufacturing_info_valid \
		,  manufacturing_location=$manufacturing_location \
		,  manufacturing_date=$manufacturing_date \
		,  type=$type \
		,  interface_format_codes_0=$interface_format_codes_0 \
		,  interface_format_codes_1=$interface_format_codes_1 \
		,  interface_format_codes_2=$interface_format_codes_2 \
		,  interface_format_codes_3=$interface_format_codes_3 \
		,  interface_format_codes_4=$interface_format_codes_4 \
		,  interface_format_codes_5=$interface_format_codes_5 \
		,  interface_format_codes_6=$interface_format_codes_6 \
		,  interface_format_codes_7=$interface_format_codes_7 \
		,  interface_format_codes_8=$interface_format_codes_8 \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_topology(p_stmt, p_dimm_topology);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_topology_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_topology \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_topology_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_topology_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_topology_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_topology_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_topology_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_topology *p_dimm_topology,
	int history_id,
	int dimm_topology_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_topology, 0, sizeof (struct db_dimm_topology) * dimm_topology_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  id,  vendor_id,  device_id,  revision_id,  subsystem_vendor_id,  subsystem_device_id,  subsystem_revision_id,  manufacturing_info_valid,  manufacturing_location,  manufacturing_date,  type,  interface_format_codes_0,  interface_format_codes_1,  interface_format_codes_2,  interface_format_codes_3,  interface_format_codes_4,  interface_format_codes_5,  interface_format_codes_6,  interface_format_codes_7,  interface_format_codes_8  \
		FROM dimm_topology_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_topology_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_topology(p_ps, p_stmt, &p_dimm_topology[index]);
		local_get_dimm_topology_relationships_history(p_ps, p_stmt, &p_dimm_topology[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_topology_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_topology_history");
}

/*
 * --- END dimm_topology ----------------
 */
/*
 * --- namespace ----------------
 */
void local_bind_namespace(sqlite3_stmt *p_stmt, struct db_namespace *p_namespace)
{
	BIND_TEXT(p_stmt, "$namespace_uid", (char *)p_namespace->namespace_uid);
	BIND_TEXT(p_stmt, "$friendly_name", (char *)p_namespace->friendly_name);
	BIND_INTEGER(p_stmt, "$block_size", (unsigned int)p_namespace->block_size);
	BIND_INTEGER(p_stmt, "$block_count", (unsigned long long)p_namespace->block_count);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_namespace->type);
	BIND_INTEGER(p_stmt, "$health", (unsigned int)p_namespace->health);
	BIND_INTEGER(p_stmt, "$enabled", (unsigned int)p_namespace->enabled);
	BIND_INTEGER(p_stmt, "$btt", (unsigned int)p_namespace->btt);
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_namespace->device_handle);
	BIND_INTEGER(p_stmt, "$interleave_set_index", (unsigned int)p_namespace->interleave_set_index);
	BIND_INTEGER(p_stmt, "$memory_page_allocation", (unsigned int)p_namespace->memory_page_allocation);
}
void local_get_namespace_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_namespace *p_namespace)
{
}

void local_get_namespace_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_namespace *p_namespace,
	int history_id)
{
}

void local_row_to_namespace(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_namespace *p_namespace)
{
	TEXT_COLUMN(p_stmt,
		0,
		p_namespace->namespace_uid,
		NAMESPACE_NAMESPACE_UID_LEN);
	TEXT_COLUMN(p_stmt,
		1,
		p_namespace->friendly_name,
		NAMESPACE_FRIENDLY_NAME_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_namespace->block_size);
	INTEGER_COLUMN(p_stmt,
		3,
		p_namespace->block_count);
	INTEGER_COLUMN(p_stmt,
		4,
		p_namespace->type);
	INTEGER_COLUMN(p_stmt,
		5,
		p_namespace->health);
	INTEGER_COLUMN(p_stmt,
		6,
		p_namespace->enabled);
	INTEGER_COLUMN(p_stmt,
		7,
		p_namespace->btt);
	INTEGER_COLUMN(p_stmt,
		8,
		p_namespace->device_handle);
	INTEGER_COLUMN(p_stmt,
		9,
		p_namespace->interleave_set_index);
	INTEGER_COLUMN(p_stmt,
		10,
		p_namespace->memory_page_allocation);
}
void db_print_namespace(struct db_namespace *p_value)
{
	printf("namespace.namespace_uid: %s\n", p_value->namespace_uid);
	printf("namespace.friendly_name: %s\n", p_value->friendly_name);
	printf("namespace.block_size: unsigned %d\n", p_value->block_size);
	printf("namespace.block_count: unsigned %lld\n", p_value->block_count);
	printf("namespace.type: unsigned %d\n", p_value->type);
	printf("namespace.health: unsigned %d\n", p_value->health);
	printf("namespace.enabled: unsigned %d\n", p_value->enabled);
	printf("namespace.btt: unsigned %d\n", p_value->btt);
	printf("namespace.device_handle: unsigned %d\n", p_value->device_handle);
	printf("namespace.interleave_set_index: unsigned %d\n", p_value->interleave_set_index);
	printf("namespace.memory_page_allocation: unsigned %d\n", p_value->memory_page_allocation);
}
enum db_return_codes db_add_namespace(const PersistentStore *p_ps,
	struct db_namespace *p_namespace)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO namespace \
		(namespace_uid, friendly_name, block_size, block_count, type, health, enabled, btt, device_handle, interleave_set_index, memory_page_allocation)  \
		VALUES 		\
		($namespace_uid, \
		$friendly_name, \
		$block_size, \
		$block_count, \
		$type, \
		$health, \
		$enabled, \
		$btt, \
		$device_handle, \
		$interleave_set_index, \
		$memory_page_allocation) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_namespace(p_stmt, p_namespace);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespace_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "namespace", p_count);
}
int db_get_namespaces(const PersistentStore *p_ps,
	struct db_namespace *p_namespace,
	int namespace_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_namespace, 0, sizeof (struct db_namespace) * namespace_count);
	char *sql = "SELECT \
		namespace_uid \
		,  friendly_name \
		,  block_size \
		,  block_count \
		,  type \
		,  health \
		,  enabled \
		,  btt \
		,  device_handle \
		,  interleave_set_index \
		,  memory_page_allocation \
		  \
		FROM namespace \
		            \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_count)
		{
			local_row_to_namespace(p_ps, p_stmt, &p_namespace[index]);
			local_get_namespace_relationships(p_ps, p_stmt, &p_namespace[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_namespaces(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM namespace");
}

enum db_return_codes db_save_namespace_state(const PersistentStore *p_ps,
	int history_id,
	struct db_namespace *p_namespace)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_namespace temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_namespace_by_namespace_uid(p_ps, p_namespace->namespace_uid, &temp) == DB_SUCCESS)
	{
		rc = db_update_namespace_by_namespace_uid(p_ps,
				p_namespace->namespace_uid,
				p_namespace);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO namespace \
			( namespace_uid ,  friendly_name ,  block_size ,  block_count ,  type ,  health ,  enabled ,  btt ,  device_handle ,  interleave_set_index ,  memory_page_allocation )  \
			VALUES 		\
			($namespace_uid, \
			$friendly_name, \
			$block_size, \
			$block_count, \
			$type, \
			$health, \
			$enabled, \
			$btt, \
			$device_handle, \
			$interleave_set_index, \
			$memory_page_allocation) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_namespace(p_stmt, p_namespace);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO namespace_history \
			(history_id, \
				 namespace_uid,  friendly_name,  block_size,  block_count,  type,  health,  enabled,  btt,  device_handle,  interleave_set_index,  memory_page_allocation)  \
			VALUES 		($history_id, \
				 $namespace_uid , \
				 $friendly_name , \
				 $block_size , \
				 $block_count , \
				 $type , \
				 $health , \
				 $enabled , \
				 $btt , \
				 $device_handle , \
				 $interleave_set_index , \
				 $memory_page_allocation )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_namespace(p_stmt, p_namespace);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_namespace_by_namespace_uid(const PersistentStore *p_ps,
	const char * namespace_uid,
	struct db_namespace *p_namespace)
{
	memset(p_namespace, 0, sizeof (struct db_namespace));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		namespace_uid,  friendly_name,  block_size,  block_count,  type,  health,  enabled,  btt,  device_handle,  interleave_set_index,  memory_page_allocation  \
		FROM namespace \
		WHERE  namespace_uid = $namespace_uid";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$namespace_uid", (char *)namespace_uid);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_namespace(p_ps, p_stmt, p_namespace);
			local_get_namespace_relationships(p_ps, p_stmt, p_namespace);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_namespace_by_namespace_uid(const PersistentStore *p_ps,
	const char * namespace_uid,
	struct db_namespace *p_namespace)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE namespace \
	SET \
	namespace_uid=$namespace_uid \
		,  friendly_name=$friendly_name \
		,  block_size=$block_size \
		,  block_count=$block_count \
		,  type=$type \
		,  health=$health \
		,  enabled=$enabled \
		,  btt=$btt \
		,  device_handle=$device_handle \
		,  interleave_set_index=$interleave_set_index \
		,  memory_page_allocation=$memory_page_allocation \
		  \
	WHERE namespace_uid=$namespace_uid ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$namespace_uid", (char *)namespace_uid);
		local_bind_namespace(p_stmt, p_namespace);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_namespace_by_namespace_uid(const PersistentStore *p_ps,
	const char * namespace_uid)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM namespace \
				 WHERE namespace_uid = $namespace_uid";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$namespace_uid", (char *)namespace_uid);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_namespace_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM namespace_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespace_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM namespace_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_namespace_history_by_history_id(const PersistentStore *p_ps,
	struct db_namespace *p_namespace,
	int history_id,
	int namespace_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_namespace, 0, sizeof (struct db_namespace) * namespace_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		namespace_uid,  friendly_name,  block_size,  block_count,  type,  health,  enabled,  btt,  device_handle,  interleave_set_index,  memory_page_allocation  \
		FROM namespace_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_count)
		{
		rc = DB_SUCCESS;
		local_row_to_namespace(p_ps, p_stmt, &p_namespace[index]);
		local_get_namespace_relationships_history(p_ps, p_stmt, &p_namespace[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_namespace_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM namespace_history");
}

enum db_return_codes db_get_namespace_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM namespace WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespace_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM namespace_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespaces_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_namespace *p_namespace,
	int namespace_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 namespace_uid ,  friendly_name ,  block_size ,  block_count ,  type ,  health ,  enabled ,  btt ,  device_handle ,  interleave_set_index ,  memory_page_allocation  \
		FROM namespace \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_count)
		{
			local_row_to_namespace(p_ps, p_stmt, &p_namespace[index]);
			local_get_namespace_relationships(p_ps, p_stmt, &p_namespace[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespaces_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_namespace *p_namespace,
	int namespace_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 namespace_uid ,  friendly_name ,  block_size ,  block_count ,  type ,  health ,  enabled ,  btt ,  device_handle ,  interleave_set_index ,  memory_page_allocation  \
		FROM namespace_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_count)
		{
			local_row_to_namespace(p_ps, p_stmt, &p_namespace[index]);
			local_get_namespace_relationships(p_ps, p_stmt, &p_namespace[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_namespace_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM namespace \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespace_count_by_interleave_set_dimm_info_index_id(
	const PersistentStore *p_ps,
	const unsigned int interleave_set_index,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM namespace WHERE interleave_set_index = $interleave_set_index";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$interleave_set_index", (unsigned int)interleave_set_index);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespace_count_by_interleave_set_dimm_info_index_id_history(
	const PersistentStore *p_ps,
	const unsigned int interleave_set_index,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM namespace_history "
		"WHERE interleave_set_index = $interleave_set_index "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$interleave_set_index", (unsigned int)interleave_set_index);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespaces_by_interleave_set_dimm_info_index_id(const PersistentStore *p_ps,
	unsigned int interleave_set_index,
	struct db_namespace *p_namespace,
	int namespace_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 namespace_uid ,  friendly_name ,  block_size ,  block_count ,  type ,  health ,  enabled ,  btt ,  device_handle ,  interleave_set_index ,  memory_page_allocation  \
		FROM namespace \
		WHERE  interleave_set_index = $interleave_set_index";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$interleave_set_index", (unsigned int)interleave_set_index);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_count)
		{
			local_row_to_namespace(p_ps, p_stmt, &p_namespace[index]);
			local_get_namespace_relationships(p_ps, p_stmt, &p_namespace[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespaces_by_interleave_set_dimm_info_index_id_history(const PersistentStore *p_ps,
	unsigned int interleave_set_index,
	struct db_namespace *p_namespace,
	int namespace_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 namespace_uid ,  friendly_name ,  block_size ,  block_count ,  type ,  health ,  enabled ,  btt ,  device_handle ,  interleave_set_index ,  memory_page_allocation  \
		FROM namespace_history \
		WHERE  interleave_set_index = $interleave_set_index AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$interleave_set_index", (unsigned int)interleave_set_index);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_count)
		{
			local_row_to_namespace(p_ps, p_stmt, &p_namespace[index]);
			local_get_namespace_relationships(p_ps, p_stmt, &p_namespace[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_namespace_by_interleave_set_dimm_info_index_id(const PersistentStore *p_ps,
	unsigned int interleave_set_index)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM namespace \
				 WHERE interleave_set_index = $interleave_set_index";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$interleave_set_index", (unsigned int)interleave_set_index);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END namespace ----------------
 */
/*
 * --- identify_dimm ----------------
 */
void local_bind_identify_dimm(sqlite3_stmt *p_stmt, struct db_identify_dimm *p_identify_dimm)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_identify_dimm->device_handle);
	BIND_INTEGER(p_stmt, "$vendor_id", (unsigned int)p_identify_dimm->vendor_id);
	BIND_INTEGER(p_stmt, "$device_id", (unsigned int)p_identify_dimm->device_id);
	BIND_INTEGER(p_stmt, "$revision_id", (unsigned int)p_identify_dimm->revision_id);
	BIND_INTEGER(p_stmt, "$interface_format_code", (unsigned int)p_identify_dimm->interface_format_code);
	BIND_TEXT(p_stmt, "$fw_revision", (char *)p_identify_dimm->fw_revision);
	BIND_INTEGER(p_stmt, "$fw_api_version", (unsigned int)p_identify_dimm->fw_api_version);
	BIND_INTEGER(p_stmt, "$fw_sw_mask", (unsigned int)p_identify_dimm->fw_sw_mask);
	BIND_INTEGER(p_stmt, "$dimm_sku", (unsigned int)p_identify_dimm->dimm_sku);
	BIND_INTEGER(p_stmt, "$block_windows", (unsigned int)p_identify_dimm->block_windows);
	BIND_INTEGER(p_stmt, "$write_flush_addresses", (unsigned int)p_identify_dimm->write_flush_addresses);
	BIND_INTEGER(p_stmt, "$write_flush_address_start", (unsigned long long)p_identify_dimm->write_flush_address_start);
	BIND_INTEGER(p_stmt, "$block_control_region_offset", (unsigned int)p_identify_dimm->block_control_region_offset);
	BIND_INTEGER(p_stmt, "$raw_cap", (unsigned long long)p_identify_dimm->raw_cap);
	BIND_INTEGER(p_stmt, "$manufacturer", (unsigned int)p_identify_dimm->manufacturer);
	BIND_INTEGER(p_stmt, "$serial_num", (unsigned int)p_identify_dimm->serial_num);
	BIND_TEXT(p_stmt, "$model_num", (char *)p_identify_dimm->model_num);
}
void local_get_identify_dimm_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_identify_dimm *p_identify_dimm)
{
}

void local_get_identify_dimm_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_identify_dimm *p_identify_dimm,
	int history_id)
{
}

void local_row_to_identify_dimm(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_identify_dimm *p_identify_dimm)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_identify_dimm->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_identify_dimm->vendor_id);
	INTEGER_COLUMN(p_stmt,
		2,
		p_identify_dimm->device_id);
	INTEGER_COLUMN(p_stmt,
		3,
		p_identify_dimm->revision_id);
	INTEGER_COLUMN(p_stmt,
		4,
		p_identify_dimm->interface_format_code);
	TEXT_COLUMN(p_stmt,
		5,
		p_identify_dimm->fw_revision,
		IDENTIFY_DIMM_FW_REVISION_LEN);
	INTEGER_COLUMN(p_stmt,
		6,
		p_identify_dimm->fw_api_version);
	INTEGER_COLUMN(p_stmt,
		7,
		p_identify_dimm->fw_sw_mask);
	INTEGER_COLUMN(p_stmt,
		8,
		p_identify_dimm->dimm_sku);
	INTEGER_COLUMN(p_stmt,
		9,
		p_identify_dimm->block_windows);
	INTEGER_COLUMN(p_stmt,
		10,
		p_identify_dimm->write_flush_addresses);
	INTEGER_COLUMN(p_stmt,
		11,
		p_identify_dimm->write_flush_address_start);
	INTEGER_COLUMN(p_stmt,
		12,
		p_identify_dimm->block_control_region_offset);
	INTEGER_COLUMN(p_stmt,
		13,
		p_identify_dimm->raw_cap);
	INTEGER_COLUMN(p_stmt,
		14,
		p_identify_dimm->manufacturer);
	INTEGER_COLUMN(p_stmt,
		15,
		p_identify_dimm->serial_num);
	TEXT_COLUMN(p_stmt,
		16,
		p_identify_dimm->model_num,
		IDENTIFY_DIMM_MODEL_NUM_LEN);
}
void db_print_identify_dimm(struct db_identify_dimm *p_value)
{
	printf("identify_dimm.device_handle: unsigned %d\n", p_value->device_handle);
	printf("identify_dimm.vendor_id: unsigned %d\n", p_value->vendor_id);
	printf("identify_dimm.device_id: unsigned %d\n", p_value->device_id);
	printf("identify_dimm.revision_id: unsigned %d\n", p_value->revision_id);
	printf("identify_dimm.interface_format_code: unsigned %d\n", p_value->interface_format_code);
	printf("identify_dimm.fw_revision: %s\n", p_value->fw_revision);
	printf("identify_dimm.fw_api_version: unsigned %d\n", p_value->fw_api_version);
	printf("identify_dimm.fw_sw_mask: unsigned %d\n", p_value->fw_sw_mask);
	printf("identify_dimm.dimm_sku: unsigned %d\n", p_value->dimm_sku);
	printf("identify_dimm.block_windows: unsigned %d\n", p_value->block_windows);
	printf("identify_dimm.write_flush_addresses: unsigned %d\n", p_value->write_flush_addresses);
	printf("identify_dimm.write_flush_address_start: unsigned %lld\n", p_value->write_flush_address_start);
	printf("identify_dimm.block_control_region_offset: unsigned %d\n", p_value->block_control_region_offset);
	printf("identify_dimm.raw_cap: unsigned %lld\n", p_value->raw_cap);
	printf("identify_dimm.manufacturer: unsigned %d\n", p_value->manufacturer);
	printf("identify_dimm.serial_num: unsigned %d\n", p_value->serial_num);
	printf("identify_dimm.model_num: %s\n", p_value->model_num);
}
enum db_return_codes db_add_identify_dimm(const PersistentStore *p_ps,
	struct db_identify_dimm *p_identify_dimm)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO identify_dimm \
		(device_handle, vendor_id, device_id, revision_id, interface_format_code, fw_revision, fw_api_version, fw_sw_mask, dimm_sku, block_windows, write_flush_addresses, write_flush_address_start, block_control_region_offset, raw_cap, manufacturer, serial_num, model_num)  \
		VALUES 		\
		($device_handle, \
		$vendor_id, \
		$device_id, \
		$revision_id, \
		$interface_format_code, \
		$fw_revision, \
		$fw_api_version, \
		$fw_sw_mask, \
		$dimm_sku, \
		$block_windows, \
		$write_flush_addresses, \
		$write_flush_address_start, \
		$block_control_region_offset, \
		$raw_cap, \
		$manufacturer, \
		$serial_num, \
		$model_num) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_identify_dimm(p_stmt, p_identify_dimm);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_identify_dimm_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "identify_dimm", p_count);
}
int db_get_identify_dimms(const PersistentStore *p_ps,
	struct db_identify_dimm *p_identify_dimm,
	int identify_dimm_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_identify_dimm, 0, sizeof (struct db_identify_dimm) * identify_dimm_count);
	char *sql = "SELECT \
		device_handle \
		,  vendor_id \
		,  device_id \
		,  revision_id \
		,  interface_format_code \
		,  fw_revision \
		,  fw_api_version \
		,  fw_sw_mask \
		,  dimm_sku \
		,  block_windows \
		,  write_flush_addresses \
		,  write_flush_address_start \
		,  block_control_region_offset \
		,  raw_cap \
		,  manufacturer \
		,  serial_num \
		,  model_num \
		  \
		FROM identify_dimm \
		                  \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < identify_dimm_count)
		{
			local_row_to_identify_dimm(p_ps, p_stmt, &p_identify_dimm[index]);
			local_get_identify_dimm_relationships(p_ps, p_stmt, &p_identify_dimm[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_identify_dimms(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM identify_dimm");
}

enum db_return_codes db_save_identify_dimm_state(const PersistentStore *p_ps,
	int history_id,
	struct db_identify_dimm *p_identify_dimm)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_identify_dimm temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_identify_dimm_by_device_handle(p_ps, p_identify_dimm->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_identify_dimm_by_device_handle(p_ps,
				p_identify_dimm->device_handle,
				p_identify_dimm);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO identify_dimm \
			( device_handle ,  vendor_id ,  device_id ,  revision_id ,  interface_format_code ,  fw_revision ,  fw_api_version ,  fw_sw_mask ,  dimm_sku ,  block_windows ,  write_flush_addresses ,  write_flush_address_start ,  block_control_region_offset ,  raw_cap ,  manufacturer ,  serial_num ,  model_num )  \
			VALUES 		\
			($device_handle, \
			$vendor_id, \
			$device_id, \
			$revision_id, \
			$interface_format_code, \
			$fw_revision, \
			$fw_api_version, \
			$fw_sw_mask, \
			$dimm_sku, \
			$block_windows, \
			$write_flush_addresses, \
			$write_flush_address_start, \
			$block_control_region_offset, \
			$raw_cap, \
			$manufacturer, \
			$serial_num, \
			$model_num) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_identify_dimm(p_stmt, p_identify_dimm);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO identify_dimm_history \
			(history_id, \
				 device_handle,  vendor_id,  device_id,  revision_id,  interface_format_code,  fw_revision,  fw_api_version,  fw_sw_mask,  dimm_sku,  block_windows,  write_flush_addresses,  write_flush_address_start,  block_control_region_offset,  raw_cap,  manufacturer,  serial_num,  model_num)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $vendor_id , \
				 $device_id , \
				 $revision_id , \
				 $interface_format_code , \
				 $fw_revision , \
				 $fw_api_version , \
				 $fw_sw_mask , \
				 $dimm_sku , \
				 $block_windows , \
				 $write_flush_addresses , \
				 $write_flush_address_start , \
				 $block_control_region_offset , \
				 $raw_cap , \
				 $manufacturer , \
				 $serial_num , \
				 $model_num )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_identify_dimm(p_stmt, p_identify_dimm);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_identify_dimm_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_identify_dimm *p_identify_dimm)
{
	memset(p_identify_dimm, 0, sizeof (struct db_identify_dimm));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  vendor_id,  device_id,  revision_id,  interface_format_code,  fw_revision,  fw_api_version,  fw_sw_mask,  dimm_sku,  block_windows,  write_flush_addresses,  write_flush_address_start,  block_control_region_offset,  raw_cap,  manufacturer,  serial_num,  model_num  \
		FROM identify_dimm \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_identify_dimm(p_ps, p_stmt, p_identify_dimm);
			local_get_identify_dimm_relationships(p_ps, p_stmt, p_identify_dimm);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_identify_dimm_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_identify_dimm *p_identify_dimm)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE identify_dimm \
	SET \
	device_handle=$device_handle \
		,  vendor_id=$vendor_id \
		,  device_id=$device_id \
		,  revision_id=$revision_id \
		,  interface_format_code=$interface_format_code \
		,  fw_revision=$fw_revision \
		,  fw_api_version=$fw_api_version \
		,  fw_sw_mask=$fw_sw_mask \
		,  dimm_sku=$dimm_sku \
		,  block_windows=$block_windows \
		,  write_flush_addresses=$write_flush_addresses \
		,  write_flush_address_start=$write_flush_address_start \
		,  block_control_region_offset=$block_control_region_offset \
		,  raw_cap=$raw_cap \
		,  manufacturer=$manufacturer \
		,  serial_num=$serial_num \
		,  model_num=$model_num \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_identify_dimm(p_stmt, p_identify_dimm);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_identify_dimm_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM identify_dimm \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_identify_dimm_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM identify_dimm_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_identify_dimm_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM identify_dimm_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_identify_dimm_history_by_history_id(const PersistentStore *p_ps,
	struct db_identify_dimm *p_identify_dimm,
	int history_id,
	int identify_dimm_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_identify_dimm, 0, sizeof (struct db_identify_dimm) * identify_dimm_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  vendor_id,  device_id,  revision_id,  interface_format_code,  fw_revision,  fw_api_version,  fw_sw_mask,  dimm_sku,  block_windows,  write_flush_addresses,  write_flush_address_start,  block_control_region_offset,  raw_cap,  manufacturer,  serial_num,  model_num  \
		FROM identify_dimm_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < identify_dimm_count)
		{
		rc = DB_SUCCESS;
		local_row_to_identify_dimm(p_ps, p_stmt, &p_identify_dimm[index]);
		local_get_identify_dimm_relationships_history(p_ps, p_stmt, &p_identify_dimm[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_identify_dimm_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM identify_dimm_history");
}

/*!
 * Clear identify_dimm.serial_num
 */
enum db_return_codes  db_clear_identify_dimm_serial_num(PersistentStore *p_ps)
{
	enum db_return_codes rc =
		run_sql_no_results(p_ps->db, "UPDATE identify_dimm SET serial_num=''");

	if (rc == DB_SUCCESS)
	{
		rc = run_sql_no_results(p_ps->db, "UPDATE identify_dimm_history SET serial_num=''");
	}

	return rc;
}
/*
 * --- END identify_dimm ----------------
 */
/*
 * --- device_characteristics ----------------
 */
void local_bind_device_characteristics(sqlite3_stmt *p_stmt, struct db_device_characteristics *p_device_characteristics)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_device_characteristics->device_handle);
	BIND_INTEGER(p_stmt, "$controller_temp_shutdown_threshold", (unsigned int)p_device_characteristics->controller_temp_shutdown_threshold);
	BIND_INTEGER(p_stmt, "$media_temp_shutdown_threshold", (unsigned int)p_device_characteristics->media_temp_shutdown_threshold);
	BIND_INTEGER(p_stmt, "$throttling_start_threshold", (unsigned int)p_device_characteristics->throttling_start_threshold);
	BIND_INTEGER(p_stmt, "$throttling_stop_threshold", (unsigned int)p_device_characteristics->throttling_stop_threshold);
}
void local_get_device_characteristics_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_device_characteristics *p_device_characteristics)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_device_characteristics_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_device_characteristics *p_device_characteristics,
	int history_id)
{
}

#endif

void local_row_to_device_characteristics(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_device_characteristics *p_device_characteristics)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_device_characteristics->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_device_characteristics->controller_temp_shutdown_threshold);
	INTEGER_COLUMN(p_stmt,
		2,
		p_device_characteristics->media_temp_shutdown_threshold);
	INTEGER_COLUMN(p_stmt,
		3,
		p_device_characteristics->throttling_start_threshold);
	INTEGER_COLUMN(p_stmt,
		4,
		p_device_characteristics->throttling_stop_threshold);
}
void db_print_device_characteristics(struct db_device_characteristics *p_value)
{
	printf("device_characteristics.device_handle: unsigned %d\n", p_value->device_handle);
	printf("device_characteristics.controller_temp_shutdown_threshold: unsigned %d\n", p_value->controller_temp_shutdown_threshold);
	printf("device_characteristics.media_temp_shutdown_threshold: unsigned %d\n", p_value->media_temp_shutdown_threshold);
	printf("device_characteristics.throttling_start_threshold: unsigned %d\n", p_value->throttling_start_threshold);
	printf("device_characteristics.throttling_stop_threshold: unsigned %d\n", p_value->throttling_stop_threshold);
}
enum db_return_codes db_add_device_characteristics(const PersistentStore *p_ps,
	struct db_device_characteristics *p_device_characteristics)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO device_characteristics \
		(device_handle, controller_temp_shutdown_threshold, media_temp_shutdown_threshold, throttling_start_threshold, throttling_stop_threshold)  \
		VALUES 		\
		($device_handle, \
		$controller_temp_shutdown_threshold, \
		$media_temp_shutdown_threshold, \
		$throttling_start_threshold, \
		$throttling_stop_threshold) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_device_characteristics(p_stmt, p_device_characteristics);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_device_characteristics_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "device_characteristics", p_count);
}
int db_get_device_characteristicss(const PersistentStore *p_ps,
	struct db_device_characteristics *p_device_characteristics,
	int device_characteristics_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_device_characteristics, 0, sizeof (struct db_device_characteristics) * device_characteristics_count);
	char *sql = "SELECT \
		device_handle \
		,  controller_temp_shutdown_threshold \
		,  media_temp_shutdown_threshold \
		,  throttling_start_threshold \
		,  throttling_stop_threshold \
		  \
		FROM device_characteristics \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < device_characteristics_count)
		{
			local_row_to_device_characteristics(p_ps, p_stmt, &p_device_characteristics[index]);
			local_get_device_characteristics_relationships(p_ps, p_stmt, &p_device_characteristics[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_device_characteristicss(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM device_characteristics");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_device_characteristics_state(const PersistentStore *p_ps,
	int history_id,
	struct db_device_characteristics *p_device_characteristics)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_device_characteristics temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_device_characteristics_by_device_handle(p_ps, p_device_characteristics->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_device_characteristics_by_device_handle(p_ps,
				p_device_characteristics->device_handle,
				p_device_characteristics);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO device_characteristics \
			( device_handle ,  controller_temp_shutdown_threshold ,  media_temp_shutdown_threshold ,  throttling_start_threshold ,  throttling_stop_threshold )  \
			VALUES 		\
			($device_handle, \
			$controller_temp_shutdown_threshold, \
			$media_temp_shutdown_threshold, \
			$throttling_start_threshold, \
			$throttling_stop_threshold) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_device_characteristics(p_stmt, p_device_characteristics);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO device_characteristics_history \
			(history_id, \
				 device_handle,  controller_temp_shutdown_threshold,  media_temp_shutdown_threshold,  throttling_start_threshold,  throttling_stop_threshold)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $controller_temp_shutdown_threshold , \
				 $media_temp_shutdown_threshold , \
				 $throttling_start_threshold , \
				 $throttling_stop_threshold )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_device_characteristics(p_stmt, p_device_characteristics);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_device_characteristics_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_device_characteristics *p_device_characteristics)
{
	memset(p_device_characteristics, 0, sizeof (struct db_device_characteristics));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  controller_temp_shutdown_threshold,  media_temp_shutdown_threshold,  throttling_start_threshold,  throttling_stop_threshold  \
		FROM device_characteristics \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_device_characteristics(p_ps, p_stmt, p_device_characteristics);
			local_get_device_characteristics_relationships(p_ps, p_stmt, p_device_characteristics);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_device_characteristics_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_device_characteristics *p_device_characteristics)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE device_characteristics \
	SET \
	device_handle=$device_handle \
		,  controller_temp_shutdown_threshold=$controller_temp_shutdown_threshold \
		,  media_temp_shutdown_threshold=$media_temp_shutdown_threshold \
		,  throttling_start_threshold=$throttling_start_threshold \
		,  throttling_stop_threshold=$throttling_stop_threshold \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_device_characteristics(p_stmt, p_device_characteristics);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_device_characteristics_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM device_characteristics \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_device_characteristics_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM device_characteristics_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_device_characteristics_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM device_characteristics_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_device_characteristics_history_by_history_id(const PersistentStore *p_ps,
	struct db_device_characteristics *p_device_characteristics,
	int history_id,
	int device_characteristics_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_device_characteristics, 0, sizeof (struct db_device_characteristics) * device_characteristics_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  controller_temp_shutdown_threshold,  media_temp_shutdown_threshold,  throttling_start_threshold,  throttling_stop_threshold  \
		FROM device_characteristics_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < device_characteristics_count)
		{
		rc = DB_SUCCESS;
		local_row_to_device_characteristics(p_ps, p_stmt, &p_device_characteristics[index]);
		local_get_device_characteristics_relationships_history(p_ps, p_stmt, &p_device_characteristics[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_device_characteristics_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM device_characteristics_history");
}

#endif

/*
 * --- END device_characteristics ----------------
 */
/*
 * --- dimm_partition ----------------
 */
void local_bind_dimm_partition(sqlite3_stmt *p_stmt, struct db_dimm_partition *p_dimm_partition)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_partition->device_handle);
	BIND_INTEGER(p_stmt, "$volatile_capacity", (int)p_dimm_partition->volatile_capacity);
	BIND_INTEGER(p_stmt, "$volatile_start", (long long)p_dimm_partition->volatile_start);
	BIND_INTEGER(p_stmt, "$pmem_capacity", (int)p_dimm_partition->pmem_capacity);
	BIND_INTEGER(p_stmt, "$pm_start", (long long)p_dimm_partition->pm_start);
	BIND_INTEGER(p_stmt, "$raw_capacity", (int)p_dimm_partition->raw_capacity);
}
void local_get_dimm_partition_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_partition *p_dimm_partition)
{
}

void local_get_dimm_partition_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_partition *p_dimm_partition,
	int history_id)
{
}

void local_row_to_dimm_partition(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_partition *p_dimm_partition)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_partition->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_partition->volatile_capacity);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_partition->volatile_start);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_partition->pmem_capacity);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_partition->pm_start);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_partition->raw_capacity);
}
void db_print_dimm_partition(struct db_dimm_partition *p_value)
{
	printf("dimm_partition.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_partition.volatile_capacity: %d\n", p_value->volatile_capacity);
	printf("dimm_partition.volatile_start: %lld\n", p_value->volatile_start);
	printf("dimm_partition.pmem_capacity: %d\n", p_value->pmem_capacity);
	printf("dimm_partition.pm_start: %lld\n", p_value->pm_start);
	printf("dimm_partition.raw_capacity: %d\n", p_value->raw_capacity);
}
enum db_return_codes db_add_dimm_partition(const PersistentStore *p_ps,
	struct db_dimm_partition *p_dimm_partition)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_partition \
		(device_handle, volatile_capacity, volatile_start, pmem_capacity, pm_start, raw_capacity)  \
		VALUES 		\
		($device_handle, \
		$volatile_capacity, \
		$volatile_start, \
		$pmem_capacity, \
		$pm_start, \
		$raw_capacity) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_partition(p_stmt, p_dimm_partition);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_partition_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_partition", p_count);
}
int db_get_dimm_partitions(const PersistentStore *p_ps,
	struct db_dimm_partition *p_dimm_partition,
	int dimm_partition_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_partition, 0, sizeof (struct db_dimm_partition) * dimm_partition_count);
	char *sql = "SELECT \
		device_handle \
		,  volatile_capacity \
		,  volatile_start \
		,  pmem_capacity \
		,  pm_start \
		,  raw_capacity \
		  \
		FROM dimm_partition \
		       \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_partition_count)
		{
			local_row_to_dimm_partition(p_ps, p_stmt, &p_dimm_partition[index]);
			local_get_dimm_partition_relationships(p_ps, p_stmt, &p_dimm_partition[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_partitions(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition");
}

enum db_return_codes db_save_dimm_partition_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_partition *p_dimm_partition)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_partition temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_partition_by_device_handle(p_ps, p_dimm_partition->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_partition_by_device_handle(p_ps,
				p_dimm_partition->device_handle,
				p_dimm_partition);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_partition \
			( device_handle ,  volatile_capacity ,  volatile_start ,  pmem_capacity ,  pm_start ,  raw_capacity )  \
			VALUES 		\
			($device_handle, \
			$volatile_capacity, \
			$volatile_start, \
			$pmem_capacity, \
			$pm_start, \
			$raw_capacity) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_partition(p_stmt, p_dimm_partition);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_partition_history \
			(history_id, \
				 device_handle,  volatile_capacity,  volatile_start,  pmem_capacity,  pm_start,  raw_capacity)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $volatile_capacity , \
				 $volatile_start , \
				 $pmem_capacity , \
				 $pm_start , \
				 $raw_capacity )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_partition(p_stmt, p_dimm_partition);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_partition_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_partition *p_dimm_partition)
{
	memset(p_dimm_partition, 0, sizeof (struct db_dimm_partition));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  volatile_capacity,  volatile_start,  pmem_capacity,  pm_start,  raw_capacity  \
		FROM dimm_partition \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_partition(p_ps, p_stmt, p_dimm_partition);
			local_get_dimm_partition_relationships(p_ps, p_stmt, p_dimm_partition);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_partition_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_partition *p_dimm_partition)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_partition \
	SET \
	device_handle=$device_handle \
		,  volatile_capacity=$volatile_capacity \
		,  volatile_start=$volatile_start \
		,  pmem_capacity=$pmem_capacity \
		,  pm_start=$pm_start \
		,  raw_capacity=$raw_capacity \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_partition(p_stmt, p_dimm_partition);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_partition_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_partition \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_partition_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_partition_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_partition_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_partition_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_partition_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_partition *p_dimm_partition,
	int history_id,
	int dimm_partition_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_partition, 0, sizeof (struct db_dimm_partition) * dimm_partition_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  volatile_capacity,  volatile_start,  pmem_capacity,  pm_start,  raw_capacity  \
		FROM dimm_partition_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_partition_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_partition(p_ps, p_stmt, &p_dimm_partition[index]);
		local_get_dimm_partition_relationships_history(p_ps, p_stmt, &p_dimm_partition[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_partition_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_history");
}

/*
 * --- END dimm_partition ----------------
 */
/*
 * --- dimm_smart ----------------
 */
void local_bind_dimm_smart(sqlite3_stmt *p_stmt, struct db_dimm_smart *p_dimm_smart)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_smart->device_handle);
	BIND_INTEGER(p_stmt, "$validation_flags", (unsigned int)p_dimm_smart->validation_flags);
	BIND_INTEGER(p_stmt, "$health_status", (unsigned int)p_dimm_smart->health_status);
	BIND_INTEGER(p_stmt, "$media_temperature", (unsigned int)p_dimm_smart->media_temperature);
	BIND_INTEGER(p_stmt, "$spare", (unsigned int)p_dimm_smart->spare);
	BIND_INTEGER(p_stmt, "$alarm_trips", (unsigned int)p_dimm_smart->alarm_trips);
	BIND_INTEGER(p_stmt, "$percentage_used", (unsigned int)p_dimm_smart->percentage_used);
	BIND_INTEGER(p_stmt, "$lss", (unsigned int)p_dimm_smart->lss);
	BIND_INTEGER(p_stmt, "$vendor_specific_data_size", (unsigned int)p_dimm_smart->vendor_specific_data_size);
	BIND_INTEGER(p_stmt, "$power_cycles", (unsigned long long)p_dimm_smart->power_cycles);
	BIND_INTEGER(p_stmt, "$power_on_seconds", (unsigned long long)p_dimm_smart->power_on_seconds);
	BIND_INTEGER(p_stmt, "$uptime", (unsigned long long)p_dimm_smart->uptime);
	BIND_INTEGER(p_stmt, "$unsafe_shutdowns", (unsigned int)p_dimm_smart->unsafe_shutdowns);
	BIND_INTEGER(p_stmt, "$lss_details", (unsigned int)p_dimm_smart->lss_details);
	BIND_INTEGER(p_stmt, "$last_shutdown_time", (unsigned long long)p_dimm_smart->last_shutdown_time);
	BIND_INTEGER(p_stmt, "$controller_temperature", (unsigned int)p_dimm_smart->controller_temperature);
}
void local_get_dimm_smart_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_smart *p_dimm_smart)
{
}

void local_get_dimm_smart_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_smart *p_dimm_smart,
	int history_id)
{
}

void local_row_to_dimm_smart(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_smart *p_dimm_smart)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_smart->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_smart->validation_flags);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_smart->health_status);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_smart->media_temperature);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_smart->spare);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_smart->alarm_trips);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_smart->percentage_used);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_smart->lss);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_smart->vendor_specific_data_size);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_smart->power_cycles);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_smart->power_on_seconds);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_smart->uptime);
	INTEGER_COLUMN(p_stmt,
		12,
		p_dimm_smart->unsafe_shutdowns);
	INTEGER_COLUMN(p_stmt,
		13,
		p_dimm_smart->lss_details);
	INTEGER_COLUMN(p_stmt,
		14,
		p_dimm_smart->last_shutdown_time);
	INTEGER_COLUMN(p_stmt,
		15,
		p_dimm_smart->controller_temperature);
}
void db_print_dimm_smart(struct db_dimm_smart *p_value)
{
	printf("dimm_smart.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_smart.validation_flags: unsigned %d\n", p_value->validation_flags);
	printf("dimm_smart.health_status: unsigned %d\n", p_value->health_status);
	printf("dimm_smart.media_temperature: unsigned %d\n", p_value->media_temperature);
	printf("dimm_smart.spare: unsigned %d\n", p_value->spare);
	printf("dimm_smart.alarm_trips: unsigned %d\n", p_value->alarm_trips);
	printf("dimm_smart.percentage_used: unsigned %d\n", p_value->percentage_used);
	printf("dimm_smart.lss: unsigned %d\n", p_value->lss);
	printf("dimm_smart.vendor_specific_data_size: unsigned %d\n", p_value->vendor_specific_data_size);
	printf("dimm_smart.power_cycles: unsigned %lld\n", p_value->power_cycles);
	printf("dimm_smart.power_on_seconds: unsigned %lld\n", p_value->power_on_seconds);
	printf("dimm_smart.uptime: unsigned %lld\n", p_value->uptime);
	printf("dimm_smart.unsafe_shutdowns: unsigned %d\n", p_value->unsafe_shutdowns);
	printf("dimm_smart.lss_details: unsigned %d\n", p_value->lss_details);
	printf("dimm_smart.last_shutdown_time: unsigned %lld\n", p_value->last_shutdown_time);
	printf("dimm_smart.controller_temperature: unsigned %d\n", p_value->controller_temperature);
}
enum db_return_codes db_add_dimm_smart(const PersistentStore *p_ps,
	struct db_dimm_smart *p_dimm_smart)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_smart \
		(device_handle, validation_flags, health_status, media_temperature, spare, alarm_trips, percentage_used, lss, vendor_specific_data_size, power_cycles, power_on_seconds, uptime, unsafe_shutdowns, lss_details, last_shutdown_time, controller_temperature)  \
		VALUES 		\
		($device_handle, \
		$validation_flags, \
		$health_status, \
		$media_temperature, \
		$spare, \
		$alarm_trips, \
		$percentage_used, \
		$lss, \
		$vendor_specific_data_size, \
		$power_cycles, \
		$power_on_seconds, \
		$uptime, \
		$unsafe_shutdowns, \
		$lss_details, \
		$last_shutdown_time, \
		$controller_temperature) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_smart(p_stmt, p_dimm_smart);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_smart_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_smart", p_count);
}
int db_get_dimm_smarts(const PersistentStore *p_ps,
	struct db_dimm_smart *p_dimm_smart,
	int dimm_smart_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_smart, 0, sizeof (struct db_dimm_smart) * dimm_smart_count);
	char *sql = "SELECT \
		device_handle \
		,  validation_flags \
		,  health_status \
		,  media_temperature \
		,  spare \
		,  alarm_trips \
		,  percentage_used \
		,  lss \
		,  vendor_specific_data_size \
		,  power_cycles \
		,  power_on_seconds \
		,  uptime \
		,  unsafe_shutdowns \
		,  lss_details \
		,  last_shutdown_time \
		,  controller_temperature \
		  \
		FROM dimm_smart \
		                 \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_smart_count)
		{
			local_row_to_dimm_smart(p_ps, p_stmt, &p_dimm_smart[index]);
			local_get_dimm_smart_relationships(p_ps, p_stmt, &p_dimm_smart[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_smarts(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_smart");
}

enum db_return_codes db_save_dimm_smart_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_smart *p_dimm_smart)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_smart temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_smart_by_device_handle(p_ps, p_dimm_smart->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_smart_by_device_handle(p_ps,
				p_dimm_smart->device_handle,
				p_dimm_smart);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_smart \
			( device_handle ,  validation_flags ,  health_status ,  media_temperature ,  spare ,  alarm_trips ,  percentage_used ,  lss ,  vendor_specific_data_size ,  power_cycles ,  power_on_seconds ,  uptime ,  unsafe_shutdowns ,  lss_details ,  last_shutdown_time ,  controller_temperature )  \
			VALUES 		\
			($device_handle, \
			$validation_flags, \
			$health_status, \
			$media_temperature, \
			$spare, \
			$alarm_trips, \
			$percentage_used, \
			$lss, \
			$vendor_specific_data_size, \
			$power_cycles, \
			$power_on_seconds, \
			$uptime, \
			$unsafe_shutdowns, \
			$lss_details, \
			$last_shutdown_time, \
			$controller_temperature) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_smart(p_stmt, p_dimm_smart);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_smart_history \
			(history_id, \
				 device_handle,  validation_flags,  health_status,  media_temperature,  spare,  alarm_trips,  percentage_used,  lss,  vendor_specific_data_size,  power_cycles,  power_on_seconds,  uptime,  unsafe_shutdowns,  lss_details,  last_shutdown_time,  controller_temperature)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $validation_flags , \
				 $health_status , \
				 $media_temperature , \
				 $spare , \
				 $alarm_trips , \
				 $percentage_used , \
				 $lss , \
				 $vendor_specific_data_size , \
				 $power_cycles , \
				 $power_on_seconds , \
				 $uptime , \
				 $unsafe_shutdowns , \
				 $lss_details , \
				 $last_shutdown_time , \
				 $controller_temperature )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_smart(p_stmt, p_dimm_smart);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_smart_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_smart *p_dimm_smart)
{
	memset(p_dimm_smart, 0, sizeof (struct db_dimm_smart));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  validation_flags,  health_status,  media_temperature,  spare,  alarm_trips,  percentage_used,  lss,  vendor_specific_data_size,  power_cycles,  power_on_seconds,  uptime,  unsafe_shutdowns,  lss_details,  last_shutdown_time,  controller_temperature  \
		FROM dimm_smart \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_smart(p_ps, p_stmt, p_dimm_smart);
			local_get_dimm_smart_relationships(p_ps, p_stmt, p_dimm_smart);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_smart_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_smart *p_dimm_smart)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_smart \
	SET \
	device_handle=$device_handle \
		,  validation_flags=$validation_flags \
		,  health_status=$health_status \
		,  media_temperature=$media_temperature \
		,  spare=$spare \
		,  alarm_trips=$alarm_trips \
		,  percentage_used=$percentage_used \
		,  lss=$lss \
		,  vendor_specific_data_size=$vendor_specific_data_size \
		,  power_cycles=$power_cycles \
		,  power_on_seconds=$power_on_seconds \
		,  uptime=$uptime \
		,  unsafe_shutdowns=$unsafe_shutdowns \
		,  lss_details=$lss_details \
		,  last_shutdown_time=$last_shutdown_time \
		,  controller_temperature=$controller_temperature \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_smart(p_stmt, p_dimm_smart);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_smart_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_smart \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_smart_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_smart_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_smart_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_smart_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_smart_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_smart *p_dimm_smart,
	int history_id,
	int dimm_smart_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_smart, 0, sizeof (struct db_dimm_smart) * dimm_smart_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  validation_flags,  health_status,  media_temperature,  spare,  alarm_trips,  percentage_used,  lss,  vendor_specific_data_size,  power_cycles,  power_on_seconds,  uptime,  unsafe_shutdowns,  lss_details,  last_shutdown_time,  controller_temperature  \
		FROM dimm_smart_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_smart_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_smart(p_ps, p_stmt, &p_dimm_smart[index]);
		local_get_dimm_smart_relationships_history(p_ps, p_stmt, &p_dimm_smart[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_smart_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_smart_history");
}

/*
 * --- END dimm_smart ----------------
 */
/*
 * --- dimm_state ----------------
 */
void local_bind_dimm_state(sqlite3_stmt *p_stmt, struct db_dimm_state *p_dimm_state)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_state->device_handle);
	BIND_INTEGER(p_stmt, "$spare_capacity_state", (int)p_dimm_state->spare_capacity_state);
	BIND_INTEGER(p_stmt, "$wearlevel_state", (int)p_dimm_state->wearlevel_state);
	BIND_INTEGER(p_stmt, "$mediaerrors_corrected", (unsigned long long)p_dimm_state->mediaerrors_corrected);
	BIND_INTEGER(p_stmt, "$mediaerrors_uncorrectable", (unsigned long long)p_dimm_state->mediaerrors_uncorrectable);
	BIND_INTEGER(p_stmt, "$mediaerrors_erasurecoded", (unsigned long long)p_dimm_state->mediaerrors_erasurecoded);
	BIND_INTEGER(p_stmt, "$health_state", (int)p_dimm_state->health_state);
	BIND_INTEGER(p_stmt, "$die_spares_used", (unsigned int)p_dimm_state->die_spares_used);
	BIND_INTEGER(p_stmt, "$mediatemperature_state", (int)p_dimm_state->mediatemperature_state);
	BIND_INTEGER(p_stmt, "$controllertemperature_state", (int)p_dimm_state->controllertemperature_state);
	BIND_INTEGER(p_stmt, "$viral_state", (int)p_dimm_state->viral_state);
	BIND_INTEGER(p_stmt, "$newest_error_log_timestamp", (unsigned long long)p_dimm_state->newest_error_log_timestamp);
}
void local_get_dimm_state_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_state *p_dimm_state)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_dimm_state_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_state *p_dimm_state,
	int history_id)
{
}

#endif

void local_row_to_dimm_state(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_state *p_dimm_state)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_state->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_state->spare_capacity_state);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_state->wearlevel_state);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_state->mediaerrors_corrected);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_state->mediaerrors_uncorrectable);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_state->mediaerrors_erasurecoded);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_state->health_state);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_state->die_spares_used);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_state->mediatemperature_state);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_state->controllertemperature_state);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_state->viral_state);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_state->newest_error_log_timestamp);
}
void db_print_dimm_state(struct db_dimm_state *p_value)
{
	printf("dimm_state.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_state.spare_capacity_state: %d\n", p_value->spare_capacity_state);
	printf("dimm_state.wearlevel_state: %d\n", p_value->wearlevel_state);
	printf("dimm_state.mediaerrors_corrected: unsigned %lld\n", p_value->mediaerrors_corrected);
	printf("dimm_state.mediaerrors_uncorrectable: unsigned %lld\n", p_value->mediaerrors_uncorrectable);
	printf("dimm_state.mediaerrors_erasurecoded: unsigned %lld\n", p_value->mediaerrors_erasurecoded);
	printf("dimm_state.health_state: %d\n", p_value->health_state);
	printf("dimm_state.die_spares_used: unsigned %d\n", p_value->die_spares_used);
	printf("dimm_state.mediatemperature_state: %d\n", p_value->mediatemperature_state);
	printf("dimm_state.controllertemperature_state: %d\n", p_value->controllertemperature_state);
	printf("dimm_state.viral_state: %d\n", p_value->viral_state);
	printf("dimm_state.newest_error_log_timestamp: unsigned %lld\n", p_value->newest_error_log_timestamp);
}
enum db_return_codes db_add_dimm_state(const PersistentStore *p_ps,
	struct db_dimm_state *p_dimm_state)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_state \
		(device_handle, spare_capacity_state, wearlevel_state, mediaerrors_corrected, mediaerrors_uncorrectable, mediaerrors_erasurecoded, health_state, die_spares_used, mediatemperature_state, controllertemperature_state, viral_state, newest_error_log_timestamp)  \
		VALUES 		\
		($device_handle, \
		$spare_capacity_state, \
		$wearlevel_state, \
		$mediaerrors_corrected, \
		$mediaerrors_uncorrectable, \
		$mediaerrors_erasurecoded, \
		$health_state, \
		$die_spares_used, \
		$mediatemperature_state, \
		$controllertemperature_state, \
		$viral_state, \
		$newest_error_log_timestamp) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_state(p_stmt, p_dimm_state);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_state_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_state", p_count);
}
int db_get_dimm_states(const PersistentStore *p_ps,
	struct db_dimm_state *p_dimm_state,
	int dimm_state_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_state, 0, sizeof (struct db_dimm_state) * dimm_state_count);
	char *sql = "SELECT \
		device_handle \
		,  spare_capacity_state \
		,  wearlevel_state \
		,  mediaerrors_corrected \
		,  mediaerrors_uncorrectable \
		,  mediaerrors_erasurecoded \
		,  health_state \
		,  die_spares_used \
		,  mediatemperature_state \
		,  controllertemperature_state \
		,  viral_state \
		,  newest_error_log_timestamp \
		  \
		FROM dimm_state \
		             \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_state_count)
		{
			local_row_to_dimm_state(p_ps, p_stmt, &p_dimm_state[index]);
			local_get_dimm_state_relationships(p_ps, p_stmt, &p_dimm_state[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_states(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_state");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_dimm_state_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_state *p_dimm_state)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_state temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_state_by_device_handle(p_ps, p_dimm_state->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_state_by_device_handle(p_ps,
				p_dimm_state->device_handle,
				p_dimm_state);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_state \
			( device_handle ,  spare_capacity_state ,  wearlevel_state ,  mediaerrors_corrected ,  mediaerrors_uncorrectable ,  mediaerrors_erasurecoded ,  health_state ,  die_spares_used ,  mediatemperature_state ,  controllertemperature_state ,  viral_state ,  newest_error_log_timestamp )  \
			VALUES 		\
			($device_handle, \
			$spare_capacity_state, \
			$wearlevel_state, \
			$mediaerrors_corrected, \
			$mediaerrors_uncorrectable, \
			$mediaerrors_erasurecoded, \
			$health_state, \
			$die_spares_used, \
			$mediatemperature_state, \
			$controllertemperature_state, \
			$viral_state, \
			$newest_error_log_timestamp) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_state(p_stmt, p_dimm_state);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_state_history \
			(history_id, \
				 device_handle,  spare_capacity_state,  wearlevel_state,  mediaerrors_corrected,  mediaerrors_uncorrectable,  mediaerrors_erasurecoded,  health_state,  die_spares_used,  mediatemperature_state,  controllertemperature_state,  viral_state,  newest_error_log_timestamp)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $spare_capacity_state , \
				 $wearlevel_state , \
				 $mediaerrors_corrected , \
				 $mediaerrors_uncorrectable , \
				 $mediaerrors_erasurecoded , \
				 $health_state , \
				 $die_spares_used , \
				 $mediatemperature_state , \
				 $controllertemperature_state , \
				 $viral_state , \
				 $newest_error_log_timestamp )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_state(p_stmt, p_dimm_state);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_dimm_state_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_state *p_dimm_state)
{
	memset(p_dimm_state, 0, sizeof (struct db_dimm_state));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  spare_capacity_state,  wearlevel_state,  mediaerrors_corrected,  mediaerrors_uncorrectable,  mediaerrors_erasurecoded,  health_state,  die_spares_used,  mediatemperature_state,  controllertemperature_state,  viral_state,  newest_error_log_timestamp  \
		FROM dimm_state \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_state(p_ps, p_stmt, p_dimm_state);
			local_get_dimm_state_relationships(p_ps, p_stmt, p_dimm_state);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_state_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_state *p_dimm_state)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_state \
	SET \
	device_handle=$device_handle \
		,  spare_capacity_state=$spare_capacity_state \
		,  wearlevel_state=$wearlevel_state \
		,  mediaerrors_corrected=$mediaerrors_corrected \
		,  mediaerrors_uncorrectable=$mediaerrors_uncorrectable \
		,  mediaerrors_erasurecoded=$mediaerrors_erasurecoded \
		,  health_state=$health_state \
		,  die_spares_used=$die_spares_used \
		,  mediatemperature_state=$mediatemperature_state \
		,  controllertemperature_state=$controllertemperature_state \
		,  viral_state=$viral_state \
		,  newest_error_log_timestamp=$newest_error_log_timestamp \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_state(p_stmt, p_dimm_state);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_state_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_state \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_dimm_state_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_state_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_state_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_state_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_state_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_state *p_dimm_state,
	int history_id,
	int dimm_state_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_state, 0, sizeof (struct db_dimm_state) * dimm_state_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  spare_capacity_state,  wearlevel_state,  mediaerrors_corrected,  mediaerrors_uncorrectable,  mediaerrors_erasurecoded,  health_state,  die_spares_used,  mediatemperature_state,  controllertemperature_state,  viral_state,  newest_error_log_timestamp  \
		FROM dimm_state_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_state_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_state(p_ps, p_stmt, &p_dimm_state[index]);
		local_get_dimm_state_relationships_history(p_ps, p_stmt, &p_dimm_state[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_state_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_state_history");
}

#endif

/*
 * --- END dimm_state ----------------
 */
/*
 * --- namespace_state ----------------
 */
void local_bind_namespace_state(sqlite3_stmt *p_stmt, struct db_namespace_state *p_namespace_state)
{
	BIND_TEXT(p_stmt, "$namespace_uid", (char *)p_namespace_state->namespace_uid);
	BIND_INTEGER(p_stmt, "$health_state", (int)p_namespace_state->health_state);
}
void local_get_namespace_state_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_namespace_state *p_namespace_state)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_namespace_state_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_namespace_state *p_namespace_state,
	int history_id)
{
}

#endif

void local_row_to_namespace_state(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_namespace_state *p_namespace_state)
{
	TEXT_COLUMN(p_stmt,
		0,
		p_namespace_state->namespace_uid,
		NAMESPACE_STATE_NAMESPACE_UID_LEN);
	INTEGER_COLUMN(p_stmt,
		1,
		p_namespace_state->health_state);
}
void db_print_namespace_state(struct db_namespace_state *p_value)
{
	printf("namespace_state.namespace_uid: %s\n", p_value->namespace_uid);
	printf("namespace_state.health_state: %d\n", p_value->health_state);
}
enum db_return_codes db_add_namespace_state(const PersistentStore *p_ps,
	struct db_namespace_state *p_namespace_state)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO namespace_state \
		(namespace_uid, health_state)  \
		VALUES 		\
		($namespace_uid, \
		$health_state) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_namespace_state(p_stmt, p_namespace_state);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespace_state_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "namespace_state", p_count);
}
int db_get_namespace_states(const PersistentStore *p_ps,
	struct db_namespace_state *p_namespace_state,
	int namespace_state_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_namespace_state, 0, sizeof (struct db_namespace_state) * namespace_state_count);
	char *sql = "SELECT \
		namespace_uid \
		,  health_state \
		  \
		FROM namespace_state \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_state_count)
		{
			local_row_to_namespace_state(p_ps, p_stmt, &p_namespace_state[index]);
			local_get_namespace_state_relationships(p_ps, p_stmt, &p_namespace_state[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_namespace_states(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM namespace_state");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_namespace_state_state(const PersistentStore *p_ps,
	int history_id,
	struct db_namespace_state *p_namespace_state)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_namespace_state temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_namespace_state_by_namespace_uid(p_ps, p_namespace_state->namespace_uid, &temp) == DB_SUCCESS)
	{
		rc = db_update_namespace_state_by_namespace_uid(p_ps,
				p_namespace_state->namespace_uid,
				p_namespace_state);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO namespace_state \
			( namespace_uid ,  health_state )  \
			VALUES 		\
			($namespace_uid, \
			$health_state) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_namespace_state(p_stmt, p_namespace_state);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO namespace_state_history \
			(history_id, \
				 namespace_uid,  health_state)  \
			VALUES 		($history_id, \
				 $namespace_uid , \
				 $health_state )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_namespace_state(p_stmt, p_namespace_state);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_namespace_state_by_namespace_uid(const PersistentStore *p_ps,
	const char * namespace_uid,
	struct db_namespace_state *p_namespace_state)
{
	memset(p_namespace_state, 0, sizeof (struct db_namespace_state));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		namespace_uid,  health_state  \
		FROM namespace_state \
		WHERE  namespace_uid = $namespace_uid";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$namespace_uid", (char *)namespace_uid);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_namespace_state(p_ps, p_stmt, p_namespace_state);
			local_get_namespace_state_relationships(p_ps, p_stmt, p_namespace_state);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_namespace_state_by_namespace_uid(const PersistentStore *p_ps,
	const char * namespace_uid,
	struct db_namespace_state *p_namespace_state)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE namespace_state \
	SET \
	namespace_uid=$namespace_uid \
		,  health_state=$health_state \
		  \
	WHERE namespace_uid=$namespace_uid ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$namespace_uid", (char *)namespace_uid);
		local_bind_namespace_state(p_stmt, p_namespace_state);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_namespace_state_by_namespace_uid(const PersistentStore *p_ps,
	const char * namespace_uid)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM namespace_state \
				 WHERE namespace_uid = $namespace_uid";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$namespace_uid", (char *)namespace_uid);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_namespace_state_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM namespace_state_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_namespace_state_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM namespace_state_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_namespace_state_history_by_history_id(const PersistentStore *p_ps,
	struct db_namespace_state *p_namespace_state,
	int history_id,
	int namespace_state_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_namespace_state, 0, sizeof (struct db_namespace_state) * namespace_state_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		namespace_uid,  health_state  \
		FROM namespace_state_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < namespace_state_count)
		{
		rc = DB_SUCCESS;
		local_row_to_namespace_state(p_ps, p_stmt, &p_namespace_state[index]);
		local_get_namespace_state_relationships_history(p_ps, p_stmt, &p_namespace_state[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_namespace_state_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM namespace_state_history");
}

#endif

/*
 * --- END namespace_state ----------------
 */
/*
 * --- dimm_alarm_thresholds ----------------
 */
void local_bind_dimm_alarm_thresholds(sqlite3_stmt *p_stmt, struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_alarm_thresholds->device_handle);
	BIND_INTEGER(p_stmt, "$enable", (unsigned int)p_dimm_alarm_thresholds->enable);
	BIND_INTEGER(p_stmt, "$media_temperature", (unsigned int)p_dimm_alarm_thresholds->media_temperature);
	BIND_INTEGER(p_stmt, "$controller_temperature", (unsigned int)p_dimm_alarm_thresholds->controller_temperature);
	BIND_INTEGER(p_stmt, "$spare", (unsigned int)p_dimm_alarm_thresholds->spare);
}
void local_get_dimm_alarm_thresholds_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds)
{
}

void local_get_dimm_alarm_thresholds_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds,
	int history_id)
{
}

void local_row_to_dimm_alarm_thresholds(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_alarm_thresholds->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_alarm_thresholds->enable);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_alarm_thresholds->media_temperature);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_alarm_thresholds->controller_temperature);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_alarm_thresholds->spare);
}
void db_print_dimm_alarm_thresholds(struct db_dimm_alarm_thresholds *p_value)
{
	printf("dimm_alarm_thresholds.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_alarm_thresholds.enable: unsigned %d\n", p_value->enable);
	printf("dimm_alarm_thresholds.media_temperature: unsigned %d\n", p_value->media_temperature);
	printf("dimm_alarm_thresholds.controller_temperature: unsigned %d\n", p_value->controller_temperature);
	printf("dimm_alarm_thresholds.spare: unsigned %d\n", p_value->spare);
}
enum db_return_codes db_add_dimm_alarm_thresholds(const PersistentStore *p_ps,
	struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_alarm_thresholds \
		(device_handle, enable, media_temperature, controller_temperature, spare)  \
		VALUES 		\
		($device_handle, \
		$enable, \
		$media_temperature, \
		$controller_temperature, \
		$spare) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_alarm_thresholds(p_stmt, p_dimm_alarm_thresholds);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_alarm_thresholds_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_alarm_thresholds", p_count);
}
int db_get_dimm_alarm_thresholdss(const PersistentStore *p_ps,
	struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds,
	int dimm_alarm_thresholds_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_alarm_thresholds, 0, sizeof (struct db_dimm_alarm_thresholds) * dimm_alarm_thresholds_count);
	char *sql = "SELECT \
		device_handle \
		,  enable \
		,  media_temperature \
		,  controller_temperature \
		,  spare \
		  \
		FROM dimm_alarm_thresholds \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_alarm_thresholds_count)
		{
			local_row_to_dimm_alarm_thresholds(p_ps, p_stmt, &p_dimm_alarm_thresholds[index]);
			local_get_dimm_alarm_thresholds_relationships(p_ps, p_stmt, &p_dimm_alarm_thresholds[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_alarm_thresholdss(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_alarm_thresholds");
}

enum db_return_codes db_save_dimm_alarm_thresholds_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_alarm_thresholds temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_alarm_thresholds_by_device_handle(p_ps, p_dimm_alarm_thresholds->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_alarm_thresholds_by_device_handle(p_ps,
				p_dimm_alarm_thresholds->device_handle,
				p_dimm_alarm_thresholds);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_alarm_thresholds \
			( device_handle ,  enable ,  media_temperature ,  controller_temperature ,  spare )  \
			VALUES 		\
			($device_handle, \
			$enable, \
			$media_temperature, \
			$controller_temperature, \
			$spare) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_alarm_thresholds(p_stmt, p_dimm_alarm_thresholds);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_alarm_thresholds_history \
			(history_id, \
				 device_handle,  enable,  media_temperature,  controller_temperature,  spare)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $enable , \
				 $media_temperature , \
				 $controller_temperature , \
				 $spare )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_alarm_thresholds(p_stmt, p_dimm_alarm_thresholds);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_alarm_thresholds_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds)
{
	memset(p_dimm_alarm_thresholds, 0, sizeof (struct db_dimm_alarm_thresholds));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  enable,  media_temperature,  controller_temperature,  spare  \
		FROM dimm_alarm_thresholds \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_alarm_thresholds(p_ps, p_stmt, p_dimm_alarm_thresholds);
			local_get_dimm_alarm_thresholds_relationships(p_ps, p_stmt, p_dimm_alarm_thresholds);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_alarm_thresholds_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_alarm_thresholds \
	SET \
	device_handle=$device_handle \
		,  enable=$enable \
		,  media_temperature=$media_temperature \
		,  controller_temperature=$controller_temperature \
		,  spare=$spare \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_alarm_thresholds(p_stmt, p_dimm_alarm_thresholds);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_alarm_thresholds_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_alarm_thresholds \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_alarm_thresholds_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_alarm_thresholds_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_alarm_thresholds_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_alarm_thresholds_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_alarm_thresholds_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_alarm_thresholds *p_dimm_alarm_thresholds,
	int history_id,
	int dimm_alarm_thresholds_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_alarm_thresholds, 0, sizeof (struct db_dimm_alarm_thresholds) * dimm_alarm_thresholds_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  enable,  media_temperature,  controller_temperature,  spare  \
		FROM dimm_alarm_thresholds_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_alarm_thresholds_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_alarm_thresholds(p_ps, p_stmt, &p_dimm_alarm_thresholds[index]);
		local_get_dimm_alarm_thresholds_relationships_history(p_ps, p_stmt, &p_dimm_alarm_thresholds[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_alarm_thresholds_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_alarm_thresholds_history");
}

/*
 * --- END dimm_alarm_thresholds ----------------
 */
/*
 * --- dimm_power_management ----------------
 */
void local_bind_dimm_power_management(sqlite3_stmt *p_stmt, struct db_dimm_power_management *p_dimm_power_management)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_power_management->device_handle);
	BIND_INTEGER(p_stmt, "$enable", (unsigned int)p_dimm_power_management->enable);
	BIND_INTEGER(p_stmt, "$tdp_power_limit", (unsigned int)p_dimm_power_management->tdp_power_limit);
	BIND_INTEGER(p_stmt, "$peak_power_budget", (unsigned int)p_dimm_power_management->peak_power_budget);
	BIND_INTEGER(p_stmt, "$avg_power_budget", (unsigned int)p_dimm_power_management->avg_power_budget);
}
void local_get_dimm_power_management_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_power_management *p_dimm_power_management)
{
}

void local_get_dimm_power_management_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_power_management *p_dimm_power_management,
	int history_id)
{
}

void local_row_to_dimm_power_management(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_power_management *p_dimm_power_management)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_power_management->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_power_management->enable);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_power_management->tdp_power_limit);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_power_management->peak_power_budget);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_power_management->avg_power_budget);
}
void db_print_dimm_power_management(struct db_dimm_power_management *p_value)
{
	printf("dimm_power_management.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_power_management.enable: unsigned %d\n", p_value->enable);
	printf("dimm_power_management.tdp_power_limit: unsigned %d\n", p_value->tdp_power_limit);
	printf("dimm_power_management.peak_power_budget: unsigned %d\n", p_value->peak_power_budget);
	printf("dimm_power_management.avg_power_budget: unsigned %d\n", p_value->avg_power_budget);
}
enum db_return_codes db_add_dimm_power_management(const PersistentStore *p_ps,
	struct db_dimm_power_management *p_dimm_power_management)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_power_management \
		(device_handle, enable, tdp_power_limit, peak_power_budget, avg_power_budget)  \
		VALUES 		\
		($device_handle, \
		$enable, \
		$tdp_power_limit, \
		$peak_power_budget, \
		$avg_power_budget) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_power_management(p_stmt, p_dimm_power_management);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_power_management_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_power_management", p_count);
}
int db_get_dimm_power_managements(const PersistentStore *p_ps,
	struct db_dimm_power_management *p_dimm_power_management,
	int dimm_power_management_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_power_management, 0, sizeof (struct db_dimm_power_management) * dimm_power_management_count);
	char *sql = "SELECT \
		device_handle \
		,  enable \
		,  tdp_power_limit \
		,  peak_power_budget \
		,  avg_power_budget \
		  \
		FROM dimm_power_management \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_power_management_count)
		{
			local_row_to_dimm_power_management(p_ps, p_stmt, &p_dimm_power_management[index]);
			local_get_dimm_power_management_relationships(p_ps, p_stmt, &p_dimm_power_management[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_power_managements(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_power_management");
}

enum db_return_codes db_save_dimm_power_management_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_power_management *p_dimm_power_management)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_power_management temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_power_management_by_device_handle(p_ps, p_dimm_power_management->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_power_management_by_device_handle(p_ps,
				p_dimm_power_management->device_handle,
				p_dimm_power_management);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_power_management \
			( device_handle ,  enable ,  tdp_power_limit ,  peak_power_budget ,  avg_power_budget )  \
			VALUES 		\
			($device_handle, \
			$enable, \
			$tdp_power_limit, \
			$peak_power_budget, \
			$avg_power_budget) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_power_management(p_stmt, p_dimm_power_management);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_power_management_history \
			(history_id, \
				 device_handle,  enable,  tdp_power_limit,  peak_power_budget,  avg_power_budget)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $enable , \
				 $tdp_power_limit , \
				 $peak_power_budget , \
				 $avg_power_budget )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_power_management(p_stmt, p_dimm_power_management);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_power_management_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_power_management *p_dimm_power_management)
{
	memset(p_dimm_power_management, 0, sizeof (struct db_dimm_power_management));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  enable,  tdp_power_limit,  peak_power_budget,  avg_power_budget  \
		FROM dimm_power_management \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_power_management(p_ps, p_stmt, p_dimm_power_management);
			local_get_dimm_power_management_relationships(p_ps, p_stmt, p_dimm_power_management);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_power_management_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_power_management *p_dimm_power_management)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_power_management \
	SET \
	device_handle=$device_handle \
		,  enable=$enable \
		,  tdp_power_limit=$tdp_power_limit \
		,  peak_power_budget=$peak_power_budget \
		,  avg_power_budget=$avg_power_budget \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_power_management(p_stmt, p_dimm_power_management);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_power_management_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_power_management \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_power_management_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_power_management_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_power_management_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_power_management_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_power_management_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_power_management *p_dimm_power_management,
	int history_id,
	int dimm_power_management_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_power_management, 0, sizeof (struct db_dimm_power_management) * dimm_power_management_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  enable,  tdp_power_limit,  peak_power_budget,  avg_power_budget  \
		FROM dimm_power_management_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_power_management_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_power_management(p_ps, p_stmt, &p_dimm_power_management[index]);
		local_get_dimm_power_management_relationships_history(p_ps, p_stmt, &p_dimm_power_management[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_power_management_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_power_management_history");
}

/*
 * --- END dimm_power_management ----------------
 */
/*
 * --- dimm_die_sparing ----------------
 */
void local_bind_dimm_die_sparing(sqlite3_stmt *p_stmt, struct db_dimm_die_sparing *p_dimm_die_sparing)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_die_sparing->device_handle);
	BIND_INTEGER(p_stmt, "$enable", (unsigned int)p_dimm_die_sparing->enable);
	BIND_INTEGER(p_stmt, "$aggressiveness", (unsigned int)p_dimm_die_sparing->aggressiveness);
	BIND_INTEGER(p_stmt, "$supported_by_rank_0", (unsigned int)p_dimm_die_sparing->supported_by_rank[0]);
	BIND_INTEGER(p_stmt, "$supported_by_rank_1", (unsigned int)p_dimm_die_sparing->supported_by_rank[1]);
	BIND_INTEGER(p_stmt, "$supported_by_rank_2", (unsigned int)p_dimm_die_sparing->supported_by_rank[2]);
	BIND_INTEGER(p_stmt, "$supported_by_rank_3", (unsigned int)p_dimm_die_sparing->supported_by_rank[3]);
}
void local_get_dimm_die_sparing_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_die_sparing *p_dimm_die_sparing)
{
}

void local_get_dimm_die_sparing_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_die_sparing *p_dimm_die_sparing,
	int history_id)
{
}

void local_row_to_dimm_die_sparing(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_die_sparing *p_dimm_die_sparing)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_die_sparing->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_die_sparing->enable);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_die_sparing->aggressiveness);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_die_sparing->supported_by_rank[0]);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_die_sparing->supported_by_rank[1]);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_die_sparing->supported_by_rank[2]);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_die_sparing->supported_by_rank[3]);
}
void db_print_dimm_die_sparing(struct db_dimm_die_sparing *p_value)
{
	printf("dimm_die_sparing.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_die_sparing.enable: unsigned %d\n", p_value->enable);
	printf("dimm_die_sparing.aggressiveness: unsigned %d\n", p_value->aggressiveness);
	printf("dimm_die_sparing.supported_by_rank: unsigned %d\n", p_value->supported_by_rank[0]);
	printf("dimm_die_sparing.supported_by_rank: unsigned %d\n", p_value->supported_by_rank[1]);
	printf("dimm_die_sparing.supported_by_rank: unsigned %d\n", p_value->supported_by_rank[2]);
	printf("dimm_die_sparing.supported_by_rank: unsigned %d\n", p_value->supported_by_rank[3]);
}
enum db_return_codes db_add_dimm_die_sparing(const PersistentStore *p_ps,
	struct db_dimm_die_sparing *p_dimm_die_sparing)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_die_sparing \
		(device_handle, enable, aggressiveness, supported_by_rank_0, supported_by_rank_1, supported_by_rank_2, supported_by_rank_3)  \
		VALUES 		\
		($device_handle, \
		$enable, \
		$aggressiveness, \
		$supported_by_rank_0, \
		$supported_by_rank_1, \
		$supported_by_rank_2, \
		$supported_by_rank_3) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_die_sparing(p_stmt, p_dimm_die_sparing);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_die_sparing_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_die_sparing", p_count);
}
int db_get_dimm_die_sparings(const PersistentStore *p_ps,
	struct db_dimm_die_sparing *p_dimm_die_sparing,
	int dimm_die_sparing_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_die_sparing, 0, sizeof (struct db_dimm_die_sparing) * dimm_die_sparing_count);
	char *sql = "SELECT \
		device_handle \
		,  enable \
		,  aggressiveness \
		,  supported_by_rank_0 \
		,  supported_by_rank_1 \
		,  supported_by_rank_2 \
		,  supported_by_rank_3 \
		  \
		FROM dimm_die_sparing \
		        \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_die_sparing_count)
		{
			local_row_to_dimm_die_sparing(p_ps, p_stmt, &p_dimm_die_sparing[index]);
			local_get_dimm_die_sparing_relationships(p_ps, p_stmt, &p_dimm_die_sparing[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_die_sparings(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_die_sparing");
}

enum db_return_codes db_save_dimm_die_sparing_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_die_sparing *p_dimm_die_sparing)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_die_sparing temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_die_sparing_by_device_handle(p_ps, p_dimm_die_sparing->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_die_sparing_by_device_handle(p_ps,
				p_dimm_die_sparing->device_handle,
				p_dimm_die_sparing);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_die_sparing \
			( device_handle ,  enable ,  aggressiveness ,  supported_by_rank_0 ,  supported_by_rank_1 ,  supported_by_rank_2 ,  supported_by_rank_3 )  \
			VALUES 		\
			($device_handle, \
			$enable, \
			$aggressiveness, \
			$supported_by_rank_0, \
			$supported_by_rank_1, \
			$supported_by_rank_2, \
			$supported_by_rank_3) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_die_sparing(p_stmt, p_dimm_die_sparing);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_die_sparing_history \
			(history_id, \
				 device_handle,  enable,  aggressiveness,  supported_by_rank_0,  supported_by_rank_1,  supported_by_rank_2,  supported_by_rank_3)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $enable , \
				 $aggressiveness , \
				 $supported_by_rank_0 , \
				 $supported_by_rank_1 , \
				 $supported_by_rank_2 , \
				 $supported_by_rank_3 )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_die_sparing(p_stmt, p_dimm_die_sparing);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_die_sparing_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_die_sparing *p_dimm_die_sparing)
{
	memset(p_dimm_die_sparing, 0, sizeof (struct db_dimm_die_sparing));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  enable,  aggressiveness,  supported_by_rank_0,  supported_by_rank_1,  supported_by_rank_2,  supported_by_rank_3  \
		FROM dimm_die_sparing \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_die_sparing(p_ps, p_stmt, p_dimm_die_sparing);
			local_get_dimm_die_sparing_relationships(p_ps, p_stmt, p_dimm_die_sparing);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_die_sparing_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_die_sparing *p_dimm_die_sparing)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_die_sparing \
	SET \
	device_handle=$device_handle \
		,  enable=$enable \
		,  aggressiveness=$aggressiveness \
		,  supported_by_rank_0=$supported_by_rank_0 \
		,  supported_by_rank_1=$supported_by_rank_1 \
		,  supported_by_rank_2=$supported_by_rank_2 \
		,  supported_by_rank_3=$supported_by_rank_3 \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_die_sparing(p_stmt, p_dimm_die_sparing);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_die_sparing_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_die_sparing \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_die_sparing_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_die_sparing_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_die_sparing_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_die_sparing_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_die_sparing_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_die_sparing *p_dimm_die_sparing,
	int history_id,
	int dimm_die_sparing_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_die_sparing, 0, sizeof (struct db_dimm_die_sparing) * dimm_die_sparing_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  enable,  aggressiveness,  supported_by_rank_0,  supported_by_rank_1,  supported_by_rank_2,  supported_by_rank_3  \
		FROM dimm_die_sparing_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_die_sparing_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_die_sparing(p_ps, p_stmt, &p_dimm_die_sparing[index]);
		local_get_dimm_die_sparing_relationships_history(p_ps, p_stmt, &p_dimm_die_sparing[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_die_sparing_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_die_sparing_history");
}

/*
 * --- END dimm_die_sparing ----------------
 */
/*
 * --- dimm_optional_config_data ----------------
 */
void local_bind_dimm_optional_config_data(sqlite3_stmt *p_stmt, struct db_dimm_optional_config_data *p_dimm_optional_config_data)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_optional_config_data->device_handle);
	BIND_INTEGER(p_stmt, "$first_fast_refresh_enable", (unsigned int)p_dimm_optional_config_data->first_fast_refresh_enable);
	BIND_INTEGER(p_stmt, "$viral_policy_enable", (unsigned int)p_dimm_optional_config_data->viral_policy_enable);
	BIND_INTEGER(p_stmt, "$viral_status", (unsigned int)p_dimm_optional_config_data->viral_status);
}
void local_get_dimm_optional_config_data_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_optional_config_data *p_dimm_optional_config_data)
{
}

void local_get_dimm_optional_config_data_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_optional_config_data *p_dimm_optional_config_data,
	int history_id)
{
}

void local_row_to_dimm_optional_config_data(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_optional_config_data *p_dimm_optional_config_data)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_optional_config_data->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_optional_config_data->first_fast_refresh_enable);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_optional_config_data->viral_policy_enable);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_optional_config_data->viral_status);
}
void db_print_dimm_optional_config_data(struct db_dimm_optional_config_data *p_value)
{
	printf("dimm_optional_config_data.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_optional_config_data.first_fast_refresh_enable: unsigned %d\n", p_value->first_fast_refresh_enable);
	printf("dimm_optional_config_data.viral_policy_enable: unsigned %d\n", p_value->viral_policy_enable);
	printf("dimm_optional_config_data.viral_status: unsigned %d\n", p_value->viral_status);
}
enum db_return_codes db_add_dimm_optional_config_data(const PersistentStore *p_ps,
	struct db_dimm_optional_config_data *p_dimm_optional_config_data)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_optional_config_data \
		(device_handle, first_fast_refresh_enable, viral_policy_enable, viral_status)  \
		VALUES 		\
		($device_handle, \
		$first_fast_refresh_enable, \
		$viral_policy_enable, \
		$viral_status) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_optional_config_data(p_stmt, p_dimm_optional_config_data);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_optional_config_data_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_optional_config_data", p_count);
}
int db_get_dimm_optional_config_datas(const PersistentStore *p_ps,
	struct db_dimm_optional_config_data *p_dimm_optional_config_data,
	int dimm_optional_config_data_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_optional_config_data, 0, sizeof (struct db_dimm_optional_config_data) * dimm_optional_config_data_count);
	char *sql = "SELECT \
		device_handle \
		,  first_fast_refresh_enable \
		,  viral_policy_enable \
		,  viral_status \
		  \
		FROM dimm_optional_config_data \
		     \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_optional_config_data_count)
		{
			local_row_to_dimm_optional_config_data(p_ps, p_stmt, &p_dimm_optional_config_data[index]);
			local_get_dimm_optional_config_data_relationships(p_ps, p_stmt, &p_dimm_optional_config_data[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_optional_config_datas(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_optional_config_data");
}

enum db_return_codes db_save_dimm_optional_config_data_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_optional_config_data *p_dimm_optional_config_data)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_optional_config_data temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_optional_config_data_by_device_handle(p_ps, p_dimm_optional_config_data->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_optional_config_data_by_device_handle(p_ps,
				p_dimm_optional_config_data->device_handle,
				p_dimm_optional_config_data);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_optional_config_data \
			( device_handle ,  first_fast_refresh_enable ,  viral_policy_enable ,  viral_status )  \
			VALUES 		\
			($device_handle, \
			$first_fast_refresh_enable, \
			$viral_policy_enable, \
			$viral_status) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_optional_config_data(p_stmt, p_dimm_optional_config_data);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_optional_config_data_history \
			(history_id, \
				 device_handle,  first_fast_refresh_enable,  viral_policy_enable,  viral_status)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $first_fast_refresh_enable , \
				 $viral_policy_enable , \
				 $viral_status )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_optional_config_data(p_stmt, p_dimm_optional_config_data);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_optional_config_data_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_optional_config_data *p_dimm_optional_config_data)
{
	memset(p_dimm_optional_config_data, 0, sizeof (struct db_dimm_optional_config_data));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  first_fast_refresh_enable,  viral_policy_enable,  viral_status  \
		FROM dimm_optional_config_data \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_optional_config_data(p_ps, p_stmt, p_dimm_optional_config_data);
			local_get_dimm_optional_config_data_relationships(p_ps, p_stmt, p_dimm_optional_config_data);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_optional_config_data_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_optional_config_data *p_dimm_optional_config_data)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_optional_config_data \
	SET \
	device_handle=$device_handle \
		,  first_fast_refresh_enable=$first_fast_refresh_enable \
		,  viral_policy_enable=$viral_policy_enable \
		,  viral_status=$viral_status \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_optional_config_data(p_stmt, p_dimm_optional_config_data);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_optional_config_data_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_optional_config_data \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_optional_config_data_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_optional_config_data_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_optional_config_data_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_optional_config_data_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_optional_config_data_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_optional_config_data *p_dimm_optional_config_data,
	int history_id,
	int dimm_optional_config_data_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_optional_config_data, 0, sizeof (struct db_dimm_optional_config_data) * dimm_optional_config_data_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  first_fast_refresh_enable,  viral_policy_enable,  viral_status  \
		FROM dimm_optional_config_data_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_optional_config_data_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_optional_config_data(p_ps, p_stmt, &p_dimm_optional_config_data[index]);
		local_get_dimm_optional_config_data_relationships_history(p_ps, p_stmt, &p_dimm_optional_config_data[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_optional_config_data_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_optional_config_data_history");
}

/*
 * --- END dimm_optional_config_data ----------------
 */
/*
 * --- dimm_err_correction ----------------
 */
void local_bind_dimm_err_correction(sqlite3_stmt *p_stmt, struct db_dimm_err_correction *p_dimm_err_correction)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_err_correction->device_handle);
	BIND_INTEGER(p_stmt, "$unrefreshed_enable", (unsigned int)p_dimm_err_correction->unrefreshed_enable);
	BIND_INTEGER(p_stmt, "$refreshed_enable", (unsigned int)p_dimm_err_correction->refreshed_enable);
	BIND_INTEGER(p_stmt, "$unrefreshed_force_write", (unsigned int)p_dimm_err_correction->unrefreshed_force_write);
	BIND_INTEGER(p_stmt, "$refreshed_force_write", (unsigned int)p_dimm_err_correction->refreshed_force_write);
}
void local_get_dimm_err_correction_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_err_correction *p_dimm_err_correction)
{
}

void local_get_dimm_err_correction_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_err_correction *p_dimm_err_correction,
	int history_id)
{
}

void local_row_to_dimm_err_correction(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_err_correction *p_dimm_err_correction)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_err_correction->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_err_correction->unrefreshed_enable);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_err_correction->refreshed_enable);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_err_correction->unrefreshed_force_write);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_err_correction->refreshed_force_write);
}
void db_print_dimm_err_correction(struct db_dimm_err_correction *p_value)
{
	printf("dimm_err_correction.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_err_correction.unrefreshed_enable: unsigned %d\n", p_value->unrefreshed_enable);
	printf("dimm_err_correction.refreshed_enable: unsigned %d\n", p_value->refreshed_enable);
	printf("dimm_err_correction.unrefreshed_force_write: unsigned %d\n", p_value->unrefreshed_force_write);
	printf("dimm_err_correction.refreshed_force_write: unsigned %d\n", p_value->refreshed_force_write);
}
enum db_return_codes db_add_dimm_err_correction(const PersistentStore *p_ps,
	struct db_dimm_err_correction *p_dimm_err_correction)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_err_correction \
		(device_handle, unrefreshed_enable, refreshed_enable, unrefreshed_force_write, refreshed_force_write)  \
		VALUES 		\
		($device_handle, \
		$unrefreshed_enable, \
		$refreshed_enable, \
		$unrefreshed_force_write, \
		$refreshed_force_write) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_err_correction(p_stmt, p_dimm_err_correction);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_err_correction_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_err_correction", p_count);
}
int db_get_dimm_err_corrections(const PersistentStore *p_ps,
	struct db_dimm_err_correction *p_dimm_err_correction,
	int dimm_err_correction_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_err_correction, 0, sizeof (struct db_dimm_err_correction) * dimm_err_correction_count);
	char *sql = "SELECT \
		device_handle \
		,  unrefreshed_enable \
		,  refreshed_enable \
		,  unrefreshed_force_write \
		,  refreshed_force_write \
		  \
		FROM dimm_err_correction \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_err_correction_count)
		{
			local_row_to_dimm_err_correction(p_ps, p_stmt, &p_dimm_err_correction[index]);
			local_get_dimm_err_correction_relationships(p_ps, p_stmt, &p_dimm_err_correction[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_err_corrections(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_err_correction");
}

enum db_return_codes db_save_dimm_err_correction_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_err_correction *p_dimm_err_correction)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_err_correction temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_err_correction_by_device_handle(p_ps, p_dimm_err_correction->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_err_correction_by_device_handle(p_ps,
				p_dimm_err_correction->device_handle,
				p_dimm_err_correction);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_err_correction \
			( device_handle ,  unrefreshed_enable ,  refreshed_enable ,  unrefreshed_force_write ,  refreshed_force_write )  \
			VALUES 		\
			($device_handle, \
			$unrefreshed_enable, \
			$refreshed_enable, \
			$unrefreshed_force_write, \
			$refreshed_force_write) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_err_correction(p_stmt, p_dimm_err_correction);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_err_correction_history \
			(history_id, \
				 device_handle,  unrefreshed_enable,  refreshed_enable,  unrefreshed_force_write,  refreshed_force_write)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $unrefreshed_enable , \
				 $refreshed_enable , \
				 $unrefreshed_force_write , \
				 $refreshed_force_write )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_err_correction(p_stmt, p_dimm_err_correction);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_err_correction_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_err_correction *p_dimm_err_correction)
{
	memset(p_dimm_err_correction, 0, sizeof (struct db_dimm_err_correction));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  unrefreshed_enable,  refreshed_enable,  unrefreshed_force_write,  refreshed_force_write  \
		FROM dimm_err_correction \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_err_correction(p_ps, p_stmt, p_dimm_err_correction);
			local_get_dimm_err_correction_relationships(p_ps, p_stmt, p_dimm_err_correction);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_err_correction_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_err_correction *p_dimm_err_correction)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_err_correction \
	SET \
	device_handle=$device_handle \
		,  unrefreshed_enable=$unrefreshed_enable \
		,  refreshed_enable=$refreshed_enable \
		,  unrefreshed_force_write=$unrefreshed_force_write \
		,  refreshed_force_write=$refreshed_force_write \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_err_correction(p_stmt, p_dimm_err_correction);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_err_correction_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_err_correction \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_err_correction_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_err_correction_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_err_correction_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_err_correction_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_err_correction_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_err_correction *p_dimm_err_correction,
	int history_id,
	int dimm_err_correction_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_err_correction, 0, sizeof (struct db_dimm_err_correction) * dimm_err_correction_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  unrefreshed_enable,  refreshed_enable,  unrefreshed_force_write,  refreshed_force_write  \
		FROM dimm_err_correction_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_err_correction_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_err_correction(p_ps, p_stmt, &p_dimm_err_correction[index]);
		local_get_dimm_err_correction_relationships_history(p_ps, p_stmt, &p_dimm_err_correction[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_err_correction_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_err_correction_history");
}

/*
 * --- END dimm_err_correction ----------------
 */
/*
 * --- dimm_erasure_coding ----------------
 */
void local_bind_dimm_erasure_coding(sqlite3_stmt *p_stmt, struct db_dimm_erasure_coding *p_dimm_erasure_coding)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_erasure_coding->device_handle);
	BIND_INTEGER(p_stmt, "$verify_erc", (unsigned int)p_dimm_erasure_coding->verify_erc);
	BIND_INTEGER(p_stmt, "$unrefreshed_enable", (unsigned int)p_dimm_erasure_coding->unrefreshed_enable);
	BIND_INTEGER(p_stmt, "$refreshed_enable", (unsigned int)p_dimm_erasure_coding->refreshed_enable);
	BIND_INTEGER(p_stmt, "$unrefreshed_force_write", (unsigned int)p_dimm_erasure_coding->unrefreshed_force_write);
	BIND_INTEGER(p_stmt, "$refreshed_force_write", (unsigned int)p_dimm_erasure_coding->refreshed_force_write);
}
void local_get_dimm_erasure_coding_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_erasure_coding *p_dimm_erasure_coding)
{
}

void local_get_dimm_erasure_coding_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_erasure_coding *p_dimm_erasure_coding,
	int history_id)
{
}

void local_row_to_dimm_erasure_coding(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_erasure_coding *p_dimm_erasure_coding)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_erasure_coding->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_erasure_coding->verify_erc);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_erasure_coding->unrefreshed_enable);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_erasure_coding->refreshed_enable);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_erasure_coding->unrefreshed_force_write);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_erasure_coding->refreshed_force_write);
}
void db_print_dimm_erasure_coding(struct db_dimm_erasure_coding *p_value)
{
	printf("dimm_erasure_coding.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_erasure_coding.verify_erc: unsigned %d\n", p_value->verify_erc);
	printf("dimm_erasure_coding.unrefreshed_enable: unsigned %d\n", p_value->unrefreshed_enable);
	printf("dimm_erasure_coding.refreshed_enable: unsigned %d\n", p_value->refreshed_enable);
	printf("dimm_erasure_coding.unrefreshed_force_write: unsigned %d\n", p_value->unrefreshed_force_write);
	printf("dimm_erasure_coding.refreshed_force_write: unsigned %d\n", p_value->refreshed_force_write);
}
enum db_return_codes db_add_dimm_erasure_coding(const PersistentStore *p_ps,
	struct db_dimm_erasure_coding *p_dimm_erasure_coding)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_erasure_coding \
		(device_handle, verify_erc, unrefreshed_enable, refreshed_enable, unrefreshed_force_write, refreshed_force_write)  \
		VALUES 		\
		($device_handle, \
		$verify_erc, \
		$unrefreshed_enable, \
		$refreshed_enable, \
		$unrefreshed_force_write, \
		$refreshed_force_write) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_erasure_coding(p_stmt, p_dimm_erasure_coding);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_erasure_coding_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_erasure_coding", p_count);
}
int db_get_dimm_erasure_codings(const PersistentStore *p_ps,
	struct db_dimm_erasure_coding *p_dimm_erasure_coding,
	int dimm_erasure_coding_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_erasure_coding, 0, sizeof (struct db_dimm_erasure_coding) * dimm_erasure_coding_count);
	char *sql = "SELECT \
		device_handle \
		,  verify_erc \
		,  unrefreshed_enable \
		,  refreshed_enable \
		,  unrefreshed_force_write \
		,  refreshed_force_write \
		  \
		FROM dimm_erasure_coding \
		       \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_erasure_coding_count)
		{
			local_row_to_dimm_erasure_coding(p_ps, p_stmt, &p_dimm_erasure_coding[index]);
			local_get_dimm_erasure_coding_relationships(p_ps, p_stmt, &p_dimm_erasure_coding[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_erasure_codings(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_erasure_coding");
}

enum db_return_codes db_save_dimm_erasure_coding_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_erasure_coding *p_dimm_erasure_coding)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_erasure_coding temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_erasure_coding_by_device_handle(p_ps, p_dimm_erasure_coding->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_erasure_coding_by_device_handle(p_ps,
				p_dimm_erasure_coding->device_handle,
				p_dimm_erasure_coding);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_erasure_coding \
			( device_handle ,  verify_erc ,  unrefreshed_enable ,  refreshed_enable ,  unrefreshed_force_write ,  refreshed_force_write )  \
			VALUES 		\
			($device_handle, \
			$verify_erc, \
			$unrefreshed_enable, \
			$refreshed_enable, \
			$unrefreshed_force_write, \
			$refreshed_force_write) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_erasure_coding(p_stmt, p_dimm_erasure_coding);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_erasure_coding_history \
			(history_id, \
				 device_handle,  verify_erc,  unrefreshed_enable,  refreshed_enable,  unrefreshed_force_write,  refreshed_force_write)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $verify_erc , \
				 $unrefreshed_enable , \
				 $refreshed_enable , \
				 $unrefreshed_force_write , \
				 $refreshed_force_write )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_erasure_coding(p_stmt, p_dimm_erasure_coding);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_erasure_coding_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_erasure_coding *p_dimm_erasure_coding)
{
	memset(p_dimm_erasure_coding, 0, sizeof (struct db_dimm_erasure_coding));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  verify_erc,  unrefreshed_enable,  refreshed_enable,  unrefreshed_force_write,  refreshed_force_write  \
		FROM dimm_erasure_coding \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_erasure_coding(p_ps, p_stmt, p_dimm_erasure_coding);
			local_get_dimm_erasure_coding_relationships(p_ps, p_stmt, p_dimm_erasure_coding);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_erasure_coding_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_erasure_coding *p_dimm_erasure_coding)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_erasure_coding \
	SET \
	device_handle=$device_handle \
		,  verify_erc=$verify_erc \
		,  unrefreshed_enable=$unrefreshed_enable \
		,  refreshed_enable=$refreshed_enable \
		,  unrefreshed_force_write=$unrefreshed_force_write \
		,  refreshed_force_write=$refreshed_force_write \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_erasure_coding(p_stmt, p_dimm_erasure_coding);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_erasure_coding_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_erasure_coding \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_erasure_coding_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_erasure_coding_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_erasure_coding_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_erasure_coding_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_erasure_coding_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_erasure_coding *p_dimm_erasure_coding,
	int history_id,
	int dimm_erasure_coding_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_erasure_coding, 0, sizeof (struct db_dimm_erasure_coding) * dimm_erasure_coding_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  verify_erc,  unrefreshed_enable,  refreshed_enable,  unrefreshed_force_write,  refreshed_force_write  \
		FROM dimm_erasure_coding_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_erasure_coding_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_erasure_coding(p_ps, p_stmt, &p_dimm_erasure_coding[index]);
		local_get_dimm_erasure_coding_relationships_history(p_ps, p_stmt, &p_dimm_erasure_coding[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_erasure_coding_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_erasure_coding_history");
}

/*
 * --- END dimm_erasure_coding ----------------
 */
/*
 * --- dimm_thermal ----------------
 */
void local_bind_dimm_thermal(sqlite3_stmt *p_stmt, struct db_dimm_thermal *p_dimm_thermal)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_thermal->device_handle);
	BIND_INTEGER(p_stmt, "$throttling_enable", (unsigned int)p_dimm_thermal->throttling_enable);
	BIND_INTEGER(p_stmt, "$alerting_enable", (unsigned int)p_dimm_thermal->alerting_enable);
	BIND_INTEGER(p_stmt, "$critical_shutdown_enable", (unsigned int)p_dimm_thermal->critical_shutdown_enable);
}
void local_get_dimm_thermal_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_thermal *p_dimm_thermal)
{
}

void local_get_dimm_thermal_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_thermal *p_dimm_thermal,
	int history_id)
{
}

void local_row_to_dimm_thermal(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_thermal *p_dimm_thermal)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_thermal->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_thermal->throttling_enable);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_thermal->alerting_enable);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_thermal->critical_shutdown_enable);
}
void db_print_dimm_thermal(struct db_dimm_thermal *p_value)
{
	printf("dimm_thermal.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_thermal.throttling_enable: unsigned %d\n", p_value->throttling_enable);
	printf("dimm_thermal.alerting_enable: unsigned %d\n", p_value->alerting_enable);
	printf("dimm_thermal.critical_shutdown_enable: unsigned %d\n", p_value->critical_shutdown_enable);
}
enum db_return_codes db_add_dimm_thermal(const PersistentStore *p_ps,
	struct db_dimm_thermal *p_dimm_thermal)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_thermal \
		(device_handle, throttling_enable, alerting_enable, critical_shutdown_enable)  \
		VALUES 		\
		($device_handle, \
		$throttling_enable, \
		$alerting_enable, \
		$critical_shutdown_enable) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_thermal(p_stmt, p_dimm_thermal);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_thermal_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_thermal", p_count);
}
int db_get_dimm_thermals(const PersistentStore *p_ps,
	struct db_dimm_thermal *p_dimm_thermal,
	int dimm_thermal_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_thermal, 0, sizeof (struct db_dimm_thermal) * dimm_thermal_count);
	char *sql = "SELECT \
		device_handle \
		,  throttling_enable \
		,  alerting_enable \
		,  critical_shutdown_enable \
		  \
		FROM dimm_thermal \
		     \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_thermal_count)
		{
			local_row_to_dimm_thermal(p_ps, p_stmt, &p_dimm_thermal[index]);
			local_get_dimm_thermal_relationships(p_ps, p_stmt, &p_dimm_thermal[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_thermals(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_thermal");
}

enum db_return_codes db_save_dimm_thermal_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_thermal *p_dimm_thermal)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_thermal temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_thermal_by_device_handle(p_ps, p_dimm_thermal->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_thermal_by_device_handle(p_ps,
				p_dimm_thermal->device_handle,
				p_dimm_thermal);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_thermal \
			( device_handle ,  throttling_enable ,  alerting_enable ,  critical_shutdown_enable )  \
			VALUES 		\
			($device_handle, \
			$throttling_enable, \
			$alerting_enable, \
			$critical_shutdown_enable) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_thermal(p_stmt, p_dimm_thermal);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_thermal_history \
			(history_id, \
				 device_handle,  throttling_enable,  alerting_enable,  critical_shutdown_enable)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $throttling_enable , \
				 $alerting_enable , \
				 $critical_shutdown_enable )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_thermal(p_stmt, p_dimm_thermal);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_thermal_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_thermal *p_dimm_thermal)
{
	memset(p_dimm_thermal, 0, sizeof (struct db_dimm_thermal));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  throttling_enable,  alerting_enable,  critical_shutdown_enable  \
		FROM dimm_thermal \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_thermal(p_ps, p_stmt, p_dimm_thermal);
			local_get_dimm_thermal_relationships(p_ps, p_stmt, p_dimm_thermal);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_thermal_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_thermal *p_dimm_thermal)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_thermal \
	SET \
	device_handle=$device_handle \
		,  throttling_enable=$throttling_enable \
		,  alerting_enable=$alerting_enable \
		,  critical_shutdown_enable=$critical_shutdown_enable \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_thermal(p_stmt, p_dimm_thermal);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_thermal_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_thermal \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_thermal_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_thermal_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_thermal_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_thermal_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_thermal_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_thermal *p_dimm_thermal,
	int history_id,
	int dimm_thermal_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_thermal, 0, sizeof (struct db_dimm_thermal) * dimm_thermal_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  throttling_enable,  alerting_enable,  critical_shutdown_enable  \
		FROM dimm_thermal_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_thermal_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_thermal(p_ps, p_stmt, &p_dimm_thermal[index]);
		local_get_dimm_thermal_relationships_history(p_ps, p_stmt, &p_dimm_thermal[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_thermal_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_thermal_history");
}

/*
 * --- END dimm_thermal ----------------
 */
/*
 * --- dimm_fw_image ----------------
 */
void local_bind_dimm_fw_image(sqlite3_stmt *p_stmt, struct db_dimm_fw_image *p_dimm_fw_image)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_fw_image->device_handle);
	BIND_TEXT(p_stmt, "$fw_rev", (char *)p_dimm_fw_image->fw_rev);
	BIND_INTEGER(p_stmt, "$fw_type", (unsigned int)p_dimm_fw_image->fw_type);
	BIND_INTEGER(p_stmt, "$staged_fw_status", (unsigned int)p_dimm_fw_image->staged_fw_status);
	BIND_TEXT(p_stmt, "$staged_fw_rev", (char *)p_dimm_fw_image->staged_fw_rev);
	BIND_INTEGER(p_stmt, "$staged_fw_type", (unsigned int)p_dimm_fw_image->staged_fw_type);
	BIND_TEXT(p_stmt, "$commit_id", (char *)p_dimm_fw_image->commit_id);
	BIND_TEXT(p_stmt, "$build_configuration", (char *)p_dimm_fw_image->build_configuration);
}
void local_get_dimm_fw_image_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_image *p_dimm_fw_image)
{
}

void local_get_dimm_fw_image_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_image *p_dimm_fw_image,
	int history_id)
{
}

void local_row_to_dimm_fw_image(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_image *p_dimm_fw_image)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_fw_image->device_handle);
	TEXT_COLUMN(p_stmt,
		1,
		p_dimm_fw_image->fw_rev,
		DIMM_FW_IMAGE_FW_REV_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_fw_image->fw_type);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_fw_image->staged_fw_status);
	TEXT_COLUMN(p_stmt,
		4,
		p_dimm_fw_image->staged_fw_rev,
		DIMM_FW_IMAGE_STAGED_FW_REV_LEN);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_fw_image->staged_fw_type);
	TEXT_COLUMN(p_stmt,
		6,
		p_dimm_fw_image->commit_id,
		DIMM_FW_IMAGE_COMMIT_ID_LEN);
	TEXT_COLUMN(p_stmt,
		7,
		p_dimm_fw_image->build_configuration,
		DIMM_FW_IMAGE_BUILD_CONFIGURATION_LEN);
}
void db_print_dimm_fw_image(struct db_dimm_fw_image *p_value)
{
	printf("dimm_fw_image.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_fw_image.fw_rev: %s\n", p_value->fw_rev);
	printf("dimm_fw_image.fw_type: unsigned %d\n", p_value->fw_type);
	printf("dimm_fw_image.staged_fw_status: unsigned %d\n", p_value->staged_fw_status);
	printf("dimm_fw_image.staged_fw_rev: %s\n", p_value->staged_fw_rev);
	printf("dimm_fw_image.staged_fw_type: unsigned %d\n", p_value->staged_fw_type);
	printf("dimm_fw_image.commit_id: %s\n", p_value->commit_id);
	printf("dimm_fw_image.build_configuration: %s\n", p_value->build_configuration);
}
enum db_return_codes db_add_dimm_fw_image(const PersistentStore *p_ps,
	struct db_dimm_fw_image *p_dimm_fw_image)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_fw_image \
		(device_handle, fw_rev, fw_type, staged_fw_status, staged_fw_rev, staged_fw_type, commit_id, build_configuration)  \
		VALUES 		\
		($device_handle, \
		$fw_rev, \
		$fw_type, \
		$staged_fw_status, \
		$staged_fw_rev, \
		$staged_fw_type, \
		$commit_id, \
		$build_configuration) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_fw_image(p_stmt, p_dimm_fw_image);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_image_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_fw_image", p_count);
}
int db_get_dimm_fw_images(const PersistentStore *p_ps,
	struct db_dimm_fw_image *p_dimm_fw_image,
	int dimm_fw_image_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_image, 0, sizeof (struct db_dimm_fw_image) * dimm_fw_image_count);
	char *sql = "SELECT \
		device_handle \
		,  fw_rev \
		,  fw_type \
		,  staged_fw_status \
		,  staged_fw_rev \
		,  staged_fw_type \
		,  commit_id \
		,  build_configuration \
		  \
		FROM dimm_fw_image \
		         \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_image_count)
		{
			local_row_to_dimm_fw_image(p_ps, p_stmt, &p_dimm_fw_image[index]);
			local_get_dimm_fw_image_relationships(p_ps, p_stmt, &p_dimm_fw_image[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_fw_images(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_image");
}

enum db_return_codes db_save_dimm_fw_image_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_fw_image *p_dimm_fw_image)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_fw_image temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_fw_image_by_device_handle(p_ps, p_dimm_fw_image->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_fw_image_by_device_handle(p_ps,
				p_dimm_fw_image->device_handle,
				p_dimm_fw_image);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_fw_image \
			( device_handle ,  fw_rev ,  fw_type ,  staged_fw_status ,  staged_fw_rev ,  staged_fw_type ,  commit_id ,  build_configuration )  \
			VALUES 		\
			($device_handle, \
			$fw_rev, \
			$fw_type, \
			$staged_fw_status, \
			$staged_fw_rev, \
			$staged_fw_type, \
			$commit_id, \
			$build_configuration) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_fw_image(p_stmt, p_dimm_fw_image);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_fw_image_history \
			(history_id, \
				 device_handle,  fw_rev,  fw_type,  staged_fw_status,  staged_fw_rev,  staged_fw_type,  commit_id,  build_configuration)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $fw_rev , \
				 $fw_type , \
				 $staged_fw_status , \
				 $staged_fw_rev , \
				 $staged_fw_type , \
				 $commit_id , \
				 $build_configuration )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_fw_image(p_stmt, p_dimm_fw_image);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_image_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_fw_image *p_dimm_fw_image)
{
	memset(p_dimm_fw_image, 0, sizeof (struct db_dimm_fw_image));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  fw_rev,  fw_type,  staged_fw_status,  staged_fw_rev,  staged_fw_type,  commit_id,  build_configuration  \
		FROM dimm_fw_image \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_fw_image(p_ps, p_stmt, p_dimm_fw_image);
			local_get_dimm_fw_image_relationships(p_ps, p_stmt, p_dimm_fw_image);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_fw_image_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_fw_image *p_dimm_fw_image)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_fw_image \
	SET \
	device_handle=$device_handle \
		,  fw_rev=$fw_rev \
		,  fw_type=$fw_type \
		,  staged_fw_status=$staged_fw_status \
		,  staged_fw_rev=$staged_fw_rev \
		,  staged_fw_type=$staged_fw_type \
		,  commit_id=$commit_id \
		,  build_configuration=$build_configuration \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_fw_image(p_stmt, p_dimm_fw_image);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_image_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_fw_image \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_image_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_image_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_image_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_image_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_fw_image_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_fw_image *p_dimm_fw_image,
	int history_id,
	int dimm_fw_image_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_image, 0, sizeof (struct db_dimm_fw_image) * dimm_fw_image_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  fw_rev,  fw_type,  staged_fw_status,  staged_fw_rev,  staged_fw_type,  commit_id,  build_configuration  \
		FROM dimm_fw_image_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_image_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_fw_image(p_ps, p_stmt, &p_dimm_fw_image[index]);
		local_get_dimm_fw_image_relationships_history(p_ps, p_stmt, &p_dimm_fw_image[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_image_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_image_history");
}

/*
 * --- END dimm_fw_image ----------------
 */
/*
 * --- dimm_fw_debug_log ----------------
 */
void local_bind_dimm_fw_debug_log(sqlite3_stmt *p_stmt, struct db_dimm_fw_debug_log *p_dimm_fw_debug_log)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_fw_debug_log->device_handle);
	BIND_TEXT(p_stmt, "$fw_log", (unsigned char *)p_dimm_fw_debug_log->fw_log);
}
void local_get_dimm_fw_debug_log_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_debug_log *p_dimm_fw_debug_log)
{
}

void local_get_dimm_fw_debug_log_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_debug_log *p_dimm_fw_debug_log,
	int history_id)
{
}

void local_row_to_dimm_fw_debug_log(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_debug_log *p_dimm_fw_debug_log)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_fw_debug_log->device_handle);
	TEXT_COLUMN(p_stmt,
		1,
		p_dimm_fw_debug_log->fw_log,
		DIMM_FW_DEBUG_LOG_FW_LOG_LEN);
}
void db_print_dimm_fw_debug_log(struct db_dimm_fw_debug_log *p_value)
{
	printf("dimm_fw_debug_log.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_fw_debug_log.fw_log: unsigned %s\n", p_value->fw_log);
}
enum db_return_codes db_add_dimm_fw_debug_log(const PersistentStore *p_ps,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_fw_debug_log \
		(device_handle, fw_log)  \
		VALUES 		\
		($device_handle, \
		$fw_log) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_fw_debug_log(p_stmt, p_dimm_fw_debug_log);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_debug_log_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_fw_debug_log", p_count);
}
int db_get_dimm_fw_debug_logs(const PersistentStore *p_ps,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log,
	int dimm_fw_debug_log_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_debug_log, 0, sizeof (struct db_dimm_fw_debug_log) * dimm_fw_debug_log_count);
	char *sql = "SELECT \
		device_handle \
		,  fw_log \
		  \
		FROM dimm_fw_debug_log \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_debug_log_count)
		{
			local_row_to_dimm_fw_debug_log(p_ps, p_stmt, &p_dimm_fw_debug_log[index]);
			local_get_dimm_fw_debug_log_relationships(p_ps, p_stmt, &p_dimm_fw_debug_log[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_fw_debug_logs(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_debug_log");
}

enum db_return_codes db_save_dimm_fw_debug_log_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_fw_debug_log temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_fw_debug_log_by_fw_log(p_ps, p_dimm_fw_debug_log->fw_log, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_fw_debug_log_by_fw_log(p_ps,
				p_dimm_fw_debug_log->fw_log,
				p_dimm_fw_debug_log);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_fw_debug_log \
			( device_handle ,  fw_log )  \
			VALUES 		\
			($device_handle, \
			$fw_log) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_fw_debug_log(p_stmt, p_dimm_fw_debug_log);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_fw_debug_log_history \
			(history_id, \
				 device_handle,  fw_log)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $fw_log )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_fw_debug_log(p_stmt, p_dimm_fw_debug_log);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_debug_log_by_fw_log(const PersistentStore *p_ps,
	const unsigned char * fw_log,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log)
{
	memset(p_dimm_fw_debug_log, 0, sizeof (struct db_dimm_fw_debug_log));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  fw_log  \
		FROM dimm_fw_debug_log \
		WHERE  fw_log = $fw_log";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$fw_log", (unsigned char *)fw_log);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_fw_debug_log(p_ps, p_stmt, p_dimm_fw_debug_log);
			local_get_dimm_fw_debug_log_relationships(p_ps, p_stmt, p_dimm_fw_debug_log);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_fw_debug_log_by_fw_log(const PersistentStore *p_ps,
	const unsigned char * fw_log,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_fw_debug_log \
	SET \
	device_handle=$device_handle \
		,  fw_log=$fw_log \
		  \
	WHERE fw_log=$fw_log ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$fw_log", (unsigned char *)fw_log);
		local_bind_dimm_fw_debug_log(p_stmt, p_dimm_fw_debug_log);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_debug_log_by_fw_log(const PersistentStore *p_ps,
	const unsigned char * fw_log)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_fw_debug_log \
				 WHERE fw_log = $fw_log";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_TEXT(p_stmt, "$fw_log", (unsigned char *)fw_log);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_debug_log_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_debug_log_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_debug_log_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_debug_log_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_fw_debug_log_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log,
	int history_id,
	int dimm_fw_debug_log_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_debug_log, 0, sizeof (struct db_dimm_fw_debug_log) * dimm_fw_debug_log_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  fw_log  \
		FROM dimm_fw_debug_log_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_debug_log_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_fw_debug_log(p_ps, p_stmt, &p_dimm_fw_debug_log[index]);
		local_get_dimm_fw_debug_log_relationships_history(p_ps, p_stmt, &p_dimm_fw_debug_log[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_debug_log_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_debug_log_history");
}

enum db_return_codes db_get_dimm_fw_debug_log_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM dimm_fw_debug_log WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_debug_log_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM dimm_fw_debug_log_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_debug_logs_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log,
	int dimm_fw_debug_log_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  fw_log  \
		FROM dimm_fw_debug_log \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_debug_log_count)
		{
			local_row_to_dimm_fw_debug_log(p_ps, p_stmt, &p_dimm_fw_debug_log[index]);
			local_get_dimm_fw_debug_log_relationships(p_ps, p_stmt, &p_dimm_fw_debug_log[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_debug_logs_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_dimm_fw_debug_log *p_dimm_fw_debug_log,
	int dimm_fw_debug_log_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  fw_log  \
		FROM dimm_fw_debug_log_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_debug_log_count)
		{
			local_row_to_dimm_fw_debug_log(p_ps, p_stmt, &p_dimm_fw_debug_log[index]);
			local_get_dimm_fw_debug_log_relationships(p_ps, p_stmt, &p_dimm_fw_debug_log[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_debug_log_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_fw_debug_log \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END dimm_fw_debug_log ----------------
 */
/*
 * --- dimm_memory_info_page0 ----------------
 */
void local_bind_dimm_memory_info_page0(sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_memory_info_page0->device_handle);
	BIND_INTEGER(p_stmt, "$bytes_read", (unsigned long long)p_dimm_memory_info_page0->bytes_read);
	BIND_INTEGER(p_stmt, "$bytes_written", (unsigned long long)p_dimm_memory_info_page0->bytes_written);
	BIND_INTEGER(p_stmt, "$read_reqs", (unsigned long long)p_dimm_memory_info_page0->read_reqs);
	BIND_INTEGER(p_stmt, "$write_reqs", (unsigned long long)p_dimm_memory_info_page0->write_reqs);
	BIND_INTEGER(p_stmt, "$block_read_reqs", (unsigned long long)p_dimm_memory_info_page0->block_read_reqs);
	BIND_INTEGER(p_stmt, "$block_write_reqs", (unsigned long long)p_dimm_memory_info_page0->block_write_reqs);
}
void local_get_dimm_memory_info_page0_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0)
{
}

void local_get_dimm_memory_info_page0_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0,
	int history_id)
{
}

void local_row_to_dimm_memory_info_page0(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_memory_info_page0->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_memory_info_page0->bytes_read);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_memory_info_page0->bytes_written);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_memory_info_page0->read_reqs);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_memory_info_page0->write_reqs);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_memory_info_page0->block_read_reqs);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_memory_info_page0->block_write_reqs);
}
void db_print_dimm_memory_info_page0(struct db_dimm_memory_info_page0 *p_value)
{
	printf("dimm_memory_info_page0.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_memory_info_page0.bytes_read: unsigned %lld\n", p_value->bytes_read);
	printf("dimm_memory_info_page0.bytes_written: unsigned %lld\n", p_value->bytes_written);
	printf("dimm_memory_info_page0.read_reqs: unsigned %lld\n", p_value->read_reqs);
	printf("dimm_memory_info_page0.write_reqs: unsigned %lld\n", p_value->write_reqs);
	printf("dimm_memory_info_page0.block_read_reqs: unsigned %lld\n", p_value->block_read_reqs);
	printf("dimm_memory_info_page0.block_write_reqs: unsigned %lld\n", p_value->block_write_reqs);
}
enum db_return_codes db_add_dimm_memory_info_page0(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_memory_info_page0 \
		(device_handle, bytes_read, bytes_written, read_reqs, write_reqs, block_read_reqs, block_write_reqs)  \
		VALUES 		\
		($device_handle, \
		$bytes_read, \
		$bytes_written, \
		$read_reqs, \
		$write_reqs, \
		$block_read_reqs, \
		$block_write_reqs) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_memory_info_page0(p_stmt, p_dimm_memory_info_page0);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_memory_info_page0_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_memory_info_page0", p_count);
}
int db_get_dimm_memory_info_page0s(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0,
	int dimm_memory_info_page0_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_memory_info_page0, 0, sizeof (struct db_dimm_memory_info_page0) * dimm_memory_info_page0_count);
	char *sql = "SELECT \
		device_handle \
		,  bytes_read \
		,  bytes_written \
		,  read_reqs \
		,  write_reqs \
		,  block_read_reqs \
		,  block_write_reqs \
		  \
		FROM dimm_memory_info_page0 \
		        \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_memory_info_page0_count)
		{
			local_row_to_dimm_memory_info_page0(p_ps, p_stmt, &p_dimm_memory_info_page0[index]);
			local_get_dimm_memory_info_page0_relationships(p_ps, p_stmt, &p_dimm_memory_info_page0[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_memory_info_page0s(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page0");
}

enum db_return_codes db_save_dimm_memory_info_page0_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_memory_info_page0 temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_memory_info_page0_by_device_handle(p_ps, p_dimm_memory_info_page0->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_memory_info_page0_by_device_handle(p_ps,
				p_dimm_memory_info_page0->device_handle,
				p_dimm_memory_info_page0);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_memory_info_page0 \
			( device_handle ,  bytes_read ,  bytes_written ,  read_reqs ,  write_reqs ,  block_read_reqs ,  block_write_reqs )  \
			VALUES 		\
			($device_handle, \
			$bytes_read, \
			$bytes_written, \
			$read_reqs, \
			$write_reqs, \
			$block_read_reqs, \
			$block_write_reqs) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_memory_info_page0(p_stmt, p_dimm_memory_info_page0);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_memory_info_page0_history \
			(history_id, \
				 device_handle,  bytes_read,  bytes_written,  read_reqs,  write_reqs,  block_read_reqs,  block_write_reqs)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $bytes_read , \
				 $bytes_written , \
				 $read_reqs , \
				 $write_reqs , \
				 $block_read_reqs , \
				 $block_write_reqs )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_memory_info_page0(p_stmt, p_dimm_memory_info_page0);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_memory_info_page0_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0)
{
	memset(p_dimm_memory_info_page0, 0, sizeof (struct db_dimm_memory_info_page0));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  bytes_read,  bytes_written,  read_reqs,  write_reqs,  block_read_reqs,  block_write_reqs  \
		FROM dimm_memory_info_page0 \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_memory_info_page0(p_ps, p_stmt, p_dimm_memory_info_page0);
			local_get_dimm_memory_info_page0_relationships(p_ps, p_stmt, p_dimm_memory_info_page0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_memory_info_page0_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_memory_info_page0 \
	SET \
	device_handle=$device_handle \
		,  bytes_read=$bytes_read \
		,  bytes_written=$bytes_written \
		,  read_reqs=$read_reqs \
		,  write_reqs=$write_reqs \
		,  block_read_reqs=$block_read_reqs \
		,  block_write_reqs=$block_write_reqs \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_memory_info_page0(p_stmt, p_dimm_memory_info_page0);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_memory_info_page0_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_memory_info_page0 \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_memory_info_page0_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_memory_info_page0_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_memory_info_page0_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_memory_info_page0_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_memory_info_page0_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page0 *p_dimm_memory_info_page0,
	int history_id,
	int dimm_memory_info_page0_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_memory_info_page0, 0, sizeof (struct db_dimm_memory_info_page0) * dimm_memory_info_page0_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  bytes_read,  bytes_written,  read_reqs,  write_reqs,  block_read_reqs,  block_write_reqs  \
		FROM dimm_memory_info_page0_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_memory_info_page0_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_memory_info_page0(p_ps, p_stmt, &p_dimm_memory_info_page0[index]);
		local_get_dimm_memory_info_page0_relationships_history(p_ps, p_stmt, &p_dimm_memory_info_page0[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_memory_info_page0_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page0_history");
}

/*
 * --- END dimm_memory_info_page0 ----------------
 */
/*
 * --- dimm_memory_info_page1 ----------------
 */
void local_bind_dimm_memory_info_page1(sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_memory_info_page1->device_handle);
	BIND_INTEGER(p_stmt, "$total_bytes_read", (unsigned long long)p_dimm_memory_info_page1->total_bytes_read);
	BIND_INTEGER(p_stmt, "$total_bytes_written", (unsigned long long)p_dimm_memory_info_page1->total_bytes_written);
	BIND_INTEGER(p_stmt, "$total_read_reqs", (unsigned long long)p_dimm_memory_info_page1->total_read_reqs);
	BIND_INTEGER(p_stmt, "$total_write_reqs", (unsigned long long)p_dimm_memory_info_page1->total_write_reqs);
	BIND_INTEGER(p_stmt, "$total_block_read_reqs", (unsigned long long)p_dimm_memory_info_page1->total_block_read_reqs);
	BIND_INTEGER(p_stmt, "$total_block_write_reqs", (unsigned long long)p_dimm_memory_info_page1->total_block_write_reqs);
}
void local_get_dimm_memory_info_page1_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1)
{
}

void local_get_dimm_memory_info_page1_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1,
	int history_id)
{
}

void local_row_to_dimm_memory_info_page1(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_memory_info_page1->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_memory_info_page1->total_bytes_read);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_memory_info_page1->total_bytes_written);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_memory_info_page1->total_read_reqs);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_memory_info_page1->total_write_reqs);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_memory_info_page1->total_block_read_reqs);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_memory_info_page1->total_block_write_reqs);
}
void db_print_dimm_memory_info_page1(struct db_dimm_memory_info_page1 *p_value)
{
	printf("dimm_memory_info_page1.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_memory_info_page1.total_bytes_read: unsigned %lld\n", p_value->total_bytes_read);
	printf("dimm_memory_info_page1.total_bytes_written: unsigned %lld\n", p_value->total_bytes_written);
	printf("dimm_memory_info_page1.total_read_reqs: unsigned %lld\n", p_value->total_read_reqs);
	printf("dimm_memory_info_page1.total_write_reqs: unsigned %lld\n", p_value->total_write_reqs);
	printf("dimm_memory_info_page1.total_block_read_reqs: unsigned %lld\n", p_value->total_block_read_reqs);
	printf("dimm_memory_info_page1.total_block_write_reqs: unsigned %lld\n", p_value->total_block_write_reqs);
}
enum db_return_codes db_add_dimm_memory_info_page1(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_memory_info_page1 \
		(device_handle, total_bytes_read, total_bytes_written, total_read_reqs, total_write_reqs, total_block_read_reqs, total_block_write_reqs)  \
		VALUES 		\
		($device_handle, \
		$total_bytes_read, \
		$total_bytes_written, \
		$total_read_reqs, \
		$total_write_reqs, \
		$total_block_read_reqs, \
		$total_block_write_reqs) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_memory_info_page1(p_stmt, p_dimm_memory_info_page1);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_memory_info_page1_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_memory_info_page1", p_count);
}
int db_get_dimm_memory_info_page1s(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1,
	int dimm_memory_info_page1_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_memory_info_page1, 0, sizeof (struct db_dimm_memory_info_page1) * dimm_memory_info_page1_count);
	char *sql = "SELECT \
		device_handle \
		,  total_bytes_read \
		,  total_bytes_written \
		,  total_read_reqs \
		,  total_write_reqs \
		,  total_block_read_reqs \
		,  total_block_write_reqs \
		  \
		FROM dimm_memory_info_page1 \
		        \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_memory_info_page1_count)
		{
			local_row_to_dimm_memory_info_page1(p_ps, p_stmt, &p_dimm_memory_info_page1[index]);
			local_get_dimm_memory_info_page1_relationships(p_ps, p_stmt, &p_dimm_memory_info_page1[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_memory_info_page1s(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page1");
}

enum db_return_codes db_save_dimm_memory_info_page1_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_memory_info_page1 temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_memory_info_page1_by_device_handle(p_ps, p_dimm_memory_info_page1->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_memory_info_page1_by_device_handle(p_ps,
				p_dimm_memory_info_page1->device_handle,
				p_dimm_memory_info_page1);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_memory_info_page1 \
			( device_handle ,  total_bytes_read ,  total_bytes_written ,  total_read_reqs ,  total_write_reqs ,  total_block_read_reqs ,  total_block_write_reqs )  \
			VALUES 		\
			($device_handle, \
			$total_bytes_read, \
			$total_bytes_written, \
			$total_read_reqs, \
			$total_write_reqs, \
			$total_block_read_reqs, \
			$total_block_write_reqs) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_memory_info_page1(p_stmt, p_dimm_memory_info_page1);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_memory_info_page1_history \
			(history_id, \
				 device_handle,  total_bytes_read,  total_bytes_written,  total_read_reqs,  total_write_reqs,  total_block_read_reqs,  total_block_write_reqs)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $total_bytes_read , \
				 $total_bytes_written , \
				 $total_read_reqs , \
				 $total_write_reqs , \
				 $total_block_read_reqs , \
				 $total_block_write_reqs )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_memory_info_page1(p_stmt, p_dimm_memory_info_page1);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_memory_info_page1_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1)
{
	memset(p_dimm_memory_info_page1, 0, sizeof (struct db_dimm_memory_info_page1));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  total_bytes_read,  total_bytes_written,  total_read_reqs,  total_write_reqs,  total_block_read_reqs,  total_block_write_reqs  \
		FROM dimm_memory_info_page1 \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_memory_info_page1(p_ps, p_stmt, p_dimm_memory_info_page1);
			local_get_dimm_memory_info_page1_relationships(p_ps, p_stmt, p_dimm_memory_info_page1);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_memory_info_page1_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_memory_info_page1 \
	SET \
	device_handle=$device_handle \
		,  total_bytes_read=$total_bytes_read \
		,  total_bytes_written=$total_bytes_written \
		,  total_read_reqs=$total_read_reqs \
		,  total_write_reqs=$total_write_reqs \
		,  total_block_read_reqs=$total_block_read_reqs \
		,  total_block_write_reqs=$total_block_write_reqs \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_memory_info_page1(p_stmt, p_dimm_memory_info_page1);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_memory_info_page1_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_memory_info_page1 \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_memory_info_page1_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_memory_info_page1_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_memory_info_page1_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_memory_info_page1_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_memory_info_page1_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page1 *p_dimm_memory_info_page1,
	int history_id,
	int dimm_memory_info_page1_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_memory_info_page1, 0, sizeof (struct db_dimm_memory_info_page1) * dimm_memory_info_page1_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  total_bytes_read,  total_bytes_written,  total_read_reqs,  total_write_reqs,  total_block_read_reqs,  total_block_write_reqs  \
		FROM dimm_memory_info_page1_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_memory_info_page1_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_memory_info_page1(p_ps, p_stmt, &p_dimm_memory_info_page1[index]);
		local_get_dimm_memory_info_page1_relationships_history(p_ps, p_stmt, &p_dimm_memory_info_page1[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_memory_info_page1_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page1_history");
}

/*
 * --- END dimm_memory_info_page1 ----------------
 */
/*
 * --- dimm_memory_info_page2 ----------------
 */
void local_bind_dimm_memory_info_page2(sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_memory_info_page2->device_handle);
	BIND_INTEGER(p_stmt, "$write_count_max", (unsigned long long)p_dimm_memory_info_page2->write_count_max);
	BIND_INTEGER(p_stmt, "$write_count_average", (unsigned long long)p_dimm_memory_info_page2->write_count_average);
	BIND_INTEGER(p_stmt, "$uncorrectable_host", (unsigned int)p_dimm_memory_info_page2->uncorrectable_host);
	BIND_INTEGER(p_stmt, "$uncorrectable_non_host", (unsigned int)p_dimm_memory_info_page2->uncorrectable_non_host);
	BIND_INTEGER(p_stmt, "$media_errors_uc", (unsigned int)p_dimm_memory_info_page2->media_errors_uc);
	BIND_INTEGER(p_stmt, "$media_errors_ce", (unsigned long long)p_dimm_memory_info_page2->media_errors_ce);
	BIND_INTEGER(p_stmt, "$media_errors_ecc", (unsigned long long)p_dimm_memory_info_page2->media_errors_ecc);
}
void local_get_dimm_memory_info_page2_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2)
{
}

void local_get_dimm_memory_info_page2_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2,
	int history_id)
{
}

void local_row_to_dimm_memory_info_page2(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_memory_info_page2->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_memory_info_page2->write_count_max);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_memory_info_page2->write_count_average);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_memory_info_page2->uncorrectable_host);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_memory_info_page2->uncorrectable_non_host);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_memory_info_page2->media_errors_uc);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_memory_info_page2->media_errors_ce);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_memory_info_page2->media_errors_ecc);
}
void db_print_dimm_memory_info_page2(struct db_dimm_memory_info_page2 *p_value)
{
	printf("dimm_memory_info_page2.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_memory_info_page2.write_count_max: unsigned %lld\n", p_value->write_count_max);
	printf("dimm_memory_info_page2.write_count_average: unsigned %lld\n", p_value->write_count_average);
	printf("dimm_memory_info_page2.uncorrectable_host: unsigned %d\n", p_value->uncorrectable_host);
	printf("dimm_memory_info_page2.uncorrectable_non_host: unsigned %d\n", p_value->uncorrectable_non_host);
	printf("dimm_memory_info_page2.media_errors_uc: unsigned %d\n", p_value->media_errors_uc);
	printf("dimm_memory_info_page2.media_errors_ce: unsigned %lld\n", p_value->media_errors_ce);
	printf("dimm_memory_info_page2.media_errors_ecc: unsigned %lld\n", p_value->media_errors_ecc);
}
enum db_return_codes db_add_dimm_memory_info_page2(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_memory_info_page2 \
		(device_handle, write_count_max, write_count_average, uncorrectable_host, uncorrectable_non_host, media_errors_uc, media_errors_ce, media_errors_ecc)  \
		VALUES 		\
		($device_handle, \
		$write_count_max, \
		$write_count_average, \
		$uncorrectable_host, \
		$uncorrectable_non_host, \
		$media_errors_uc, \
		$media_errors_ce, \
		$media_errors_ecc) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_memory_info_page2(p_stmt, p_dimm_memory_info_page2);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_memory_info_page2_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_memory_info_page2", p_count);
}
int db_get_dimm_memory_info_page2s(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2,
	int dimm_memory_info_page2_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_memory_info_page2, 0, sizeof (struct db_dimm_memory_info_page2) * dimm_memory_info_page2_count);
	char *sql = "SELECT \
		device_handle \
		,  write_count_max \
		,  write_count_average \
		,  uncorrectable_host \
		,  uncorrectable_non_host \
		,  media_errors_uc \
		,  media_errors_ce \
		,  media_errors_ecc \
		  \
		FROM dimm_memory_info_page2 \
		         \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_memory_info_page2_count)
		{
			local_row_to_dimm_memory_info_page2(p_ps, p_stmt, &p_dimm_memory_info_page2[index]);
			local_get_dimm_memory_info_page2_relationships(p_ps, p_stmt, &p_dimm_memory_info_page2[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_memory_info_page2s(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page2");
}

enum db_return_codes db_save_dimm_memory_info_page2_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_memory_info_page2 temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_memory_info_page2_by_device_handle(p_ps, p_dimm_memory_info_page2->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_memory_info_page2_by_device_handle(p_ps,
				p_dimm_memory_info_page2->device_handle,
				p_dimm_memory_info_page2);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_memory_info_page2 \
			( device_handle ,  write_count_max ,  write_count_average ,  uncorrectable_host ,  uncorrectable_non_host ,  media_errors_uc ,  media_errors_ce ,  media_errors_ecc )  \
			VALUES 		\
			($device_handle, \
			$write_count_max, \
			$write_count_average, \
			$uncorrectable_host, \
			$uncorrectable_non_host, \
			$media_errors_uc, \
			$media_errors_ce, \
			$media_errors_ecc) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_memory_info_page2(p_stmt, p_dimm_memory_info_page2);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_memory_info_page2_history \
			(history_id, \
				 device_handle,  write_count_max,  write_count_average,  uncorrectable_host,  uncorrectable_non_host,  media_errors_uc,  media_errors_ce,  media_errors_ecc)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $write_count_max , \
				 $write_count_average , \
				 $uncorrectable_host , \
				 $uncorrectable_non_host , \
				 $media_errors_uc , \
				 $media_errors_ce , \
				 $media_errors_ecc )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_memory_info_page2(p_stmt, p_dimm_memory_info_page2);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_memory_info_page2_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2)
{
	memset(p_dimm_memory_info_page2, 0, sizeof (struct db_dimm_memory_info_page2));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  write_count_max,  write_count_average,  uncorrectable_host,  uncorrectable_non_host,  media_errors_uc,  media_errors_ce,  media_errors_ecc  \
		FROM dimm_memory_info_page2 \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_memory_info_page2(p_ps, p_stmt, p_dimm_memory_info_page2);
			local_get_dimm_memory_info_page2_relationships(p_ps, p_stmt, p_dimm_memory_info_page2);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_memory_info_page2_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_memory_info_page2 \
	SET \
	device_handle=$device_handle \
		,  write_count_max=$write_count_max \
		,  write_count_average=$write_count_average \
		,  uncorrectable_host=$uncorrectable_host \
		,  uncorrectable_non_host=$uncorrectable_non_host \
		,  media_errors_uc=$media_errors_uc \
		,  media_errors_ce=$media_errors_ce \
		,  media_errors_ecc=$media_errors_ecc \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_memory_info_page2(p_stmt, p_dimm_memory_info_page2);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_memory_info_page2_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_memory_info_page2 \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_memory_info_page2_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_memory_info_page2_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_memory_info_page2_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_memory_info_page2_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_memory_info_page2_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_memory_info_page2 *p_dimm_memory_info_page2,
	int history_id,
	int dimm_memory_info_page2_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_memory_info_page2, 0, sizeof (struct db_dimm_memory_info_page2) * dimm_memory_info_page2_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  write_count_max,  write_count_average,  uncorrectable_host,  uncorrectable_non_host,  media_errors_uc,  media_errors_ce,  media_errors_ecc  \
		FROM dimm_memory_info_page2_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_memory_info_page2_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_memory_info_page2(p_ps, p_stmt, &p_dimm_memory_info_page2[index]);
		local_get_dimm_memory_info_page2_relationships_history(p_ps, p_stmt, &p_dimm_memory_info_page2[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_memory_info_page2_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page2_history");
}

/*
 * --- END dimm_memory_info_page2 ----------------
 */
/*
 * --- dimm_ars_command_specific_data ----------------
 */
void local_bind_dimm_ars_command_specific_data(sqlite3_stmt *p_stmt, struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_ars_command_specific_data->device_handle);
	BIND_INTEGER(p_stmt, "$num_errors", (long long)p_dimm_ars_command_specific_data->num_errors);
	BIND_INTEGER(p_stmt, "$ars_state", (long long)p_dimm_ars_command_specific_data->ars_state);
	BIND_INTEGER(p_stmt, "$dpa_error_address_0", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[0]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_1", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[1]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_2", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[2]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_3", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[3]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_4", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[4]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_5", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[5]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_6", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[6]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_7", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[7]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_8", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[8]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_9", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[9]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_10", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[10]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_11", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[11]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_12", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[12]);
	BIND_INTEGER(p_stmt, "$dpa_error_address_13", (unsigned long long)p_dimm_ars_command_specific_data->dpa_error_address[13]);
}
void local_get_dimm_ars_command_specific_data_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data)
{
}

void local_get_dimm_ars_command_specific_data_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data,
	int history_id)
{
}

void local_row_to_dimm_ars_command_specific_data(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_ars_command_specific_data->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_ars_command_specific_data->num_errors);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_ars_command_specific_data->ars_state);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_ars_command_specific_data->dpa_error_address[0]);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_ars_command_specific_data->dpa_error_address[1]);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_ars_command_specific_data->dpa_error_address[2]);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_ars_command_specific_data->dpa_error_address[3]);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_ars_command_specific_data->dpa_error_address[4]);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_ars_command_specific_data->dpa_error_address[5]);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_ars_command_specific_data->dpa_error_address[6]);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_ars_command_specific_data->dpa_error_address[7]);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_ars_command_specific_data->dpa_error_address[8]);
	INTEGER_COLUMN(p_stmt,
		12,
		p_dimm_ars_command_specific_data->dpa_error_address[9]);
	INTEGER_COLUMN(p_stmt,
		13,
		p_dimm_ars_command_specific_data->dpa_error_address[10]);
	INTEGER_COLUMN(p_stmt,
		14,
		p_dimm_ars_command_specific_data->dpa_error_address[11]);
	INTEGER_COLUMN(p_stmt,
		15,
		p_dimm_ars_command_specific_data->dpa_error_address[12]);
	INTEGER_COLUMN(p_stmt,
		16,
		p_dimm_ars_command_specific_data->dpa_error_address[13]);
}
void db_print_dimm_ars_command_specific_data(struct db_dimm_ars_command_specific_data *p_value)
{
	printf("dimm_ars_command_specific_data.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_ars_command_specific_data.num_errors: %lld\n", p_value->num_errors);
	printf("dimm_ars_command_specific_data.ars_state: %lld\n", p_value->ars_state);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[0]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[1]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[2]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[3]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[4]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[5]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[6]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[7]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[8]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[9]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[10]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[11]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[12]);
	printf("dimm_ars_command_specific_data.dpa_error_address: unsigned %lld\n", p_value->dpa_error_address[13]);
}
enum db_return_codes db_add_dimm_ars_command_specific_data(const PersistentStore *p_ps,
	struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_ars_command_specific_data \
		(device_handle, num_errors, ars_state, dpa_error_address_0, dpa_error_address_1, dpa_error_address_2, dpa_error_address_3, dpa_error_address_4, dpa_error_address_5, dpa_error_address_6, dpa_error_address_7, dpa_error_address_8, dpa_error_address_9, dpa_error_address_10, dpa_error_address_11, dpa_error_address_12, dpa_error_address_13)  \
		VALUES 		\
		($device_handle, \
		$num_errors, \
		$ars_state, \
		$dpa_error_address_0, \
		$dpa_error_address_1, \
		$dpa_error_address_2, \
		$dpa_error_address_3, \
		$dpa_error_address_4, \
		$dpa_error_address_5, \
		$dpa_error_address_6, \
		$dpa_error_address_7, \
		$dpa_error_address_8, \
		$dpa_error_address_9, \
		$dpa_error_address_10, \
		$dpa_error_address_11, \
		$dpa_error_address_12, \
		$dpa_error_address_13) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_ars_command_specific_data(p_stmt, p_dimm_ars_command_specific_data);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_ars_command_specific_data_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_ars_command_specific_data", p_count);
}
int db_get_dimm_ars_command_specific_datas(const PersistentStore *p_ps,
	struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data,
	int dimm_ars_command_specific_data_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_ars_command_specific_data, 0, sizeof (struct db_dimm_ars_command_specific_data) * dimm_ars_command_specific_data_count);
	char *sql = "SELECT \
		device_handle \
		,  num_errors \
		,  ars_state \
		,  dpa_error_address_0 \
		,  dpa_error_address_1 \
		,  dpa_error_address_2 \
		,  dpa_error_address_3 \
		,  dpa_error_address_4 \
		,  dpa_error_address_5 \
		,  dpa_error_address_6 \
		,  dpa_error_address_7 \
		,  dpa_error_address_8 \
		,  dpa_error_address_9 \
		,  dpa_error_address_10 \
		,  dpa_error_address_11 \
		,  dpa_error_address_12 \
		,  dpa_error_address_13 \
		  \
		FROM dimm_ars_command_specific_data \
		                  \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_ars_command_specific_data_count)
		{
			local_row_to_dimm_ars_command_specific_data(p_ps, p_stmt, &p_dimm_ars_command_specific_data[index]);
			local_get_dimm_ars_command_specific_data_relationships(p_ps, p_stmt, &p_dimm_ars_command_specific_data[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_ars_command_specific_datas(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_ars_command_specific_data");
}

enum db_return_codes db_save_dimm_ars_command_specific_data_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_ars_command_specific_data temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_ars_command_specific_data_by_device_handle(p_ps, p_dimm_ars_command_specific_data->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_ars_command_specific_data_by_device_handle(p_ps,
				p_dimm_ars_command_specific_data->device_handle,
				p_dimm_ars_command_specific_data);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_ars_command_specific_data \
			( device_handle ,  num_errors ,  ars_state ,  dpa_error_address_0 ,  dpa_error_address_1 ,  dpa_error_address_2 ,  dpa_error_address_3 ,  dpa_error_address_4 ,  dpa_error_address_5 ,  dpa_error_address_6 ,  dpa_error_address_7 ,  dpa_error_address_8 ,  dpa_error_address_9 ,  dpa_error_address_10 ,  dpa_error_address_11 ,  dpa_error_address_12 ,  dpa_error_address_13 )  \
			VALUES 		\
			($device_handle, \
			$num_errors, \
			$ars_state, \
			$dpa_error_address_0, \
			$dpa_error_address_1, \
			$dpa_error_address_2, \
			$dpa_error_address_3, \
			$dpa_error_address_4, \
			$dpa_error_address_5, \
			$dpa_error_address_6, \
			$dpa_error_address_7, \
			$dpa_error_address_8, \
			$dpa_error_address_9, \
			$dpa_error_address_10, \
			$dpa_error_address_11, \
			$dpa_error_address_12, \
			$dpa_error_address_13) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_ars_command_specific_data(p_stmt, p_dimm_ars_command_specific_data);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_ars_command_specific_data_history \
			(history_id, \
				 device_handle,  num_errors,  ars_state,  dpa_error_address_0,  dpa_error_address_1,  dpa_error_address_2,  dpa_error_address_3,  dpa_error_address_4,  dpa_error_address_5,  dpa_error_address_6,  dpa_error_address_7,  dpa_error_address_8,  dpa_error_address_9,  dpa_error_address_10,  dpa_error_address_11,  dpa_error_address_12,  dpa_error_address_13)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $num_errors , \
				 $ars_state , \
				 $dpa_error_address_0 , \
				 $dpa_error_address_1 , \
				 $dpa_error_address_2 , \
				 $dpa_error_address_3 , \
				 $dpa_error_address_4 , \
				 $dpa_error_address_5 , \
				 $dpa_error_address_6 , \
				 $dpa_error_address_7 , \
				 $dpa_error_address_8 , \
				 $dpa_error_address_9 , \
				 $dpa_error_address_10 , \
				 $dpa_error_address_11 , \
				 $dpa_error_address_12 , \
				 $dpa_error_address_13 )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_ars_command_specific_data(p_stmt, p_dimm_ars_command_specific_data);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_ars_command_specific_data_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data)
{
	memset(p_dimm_ars_command_specific_data, 0, sizeof (struct db_dimm_ars_command_specific_data));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  num_errors,  ars_state,  dpa_error_address_0,  dpa_error_address_1,  dpa_error_address_2,  dpa_error_address_3,  dpa_error_address_4,  dpa_error_address_5,  dpa_error_address_6,  dpa_error_address_7,  dpa_error_address_8,  dpa_error_address_9,  dpa_error_address_10,  dpa_error_address_11,  dpa_error_address_12,  dpa_error_address_13  \
		FROM dimm_ars_command_specific_data \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_ars_command_specific_data(p_ps, p_stmt, p_dimm_ars_command_specific_data);
			local_get_dimm_ars_command_specific_data_relationships(p_ps, p_stmt, p_dimm_ars_command_specific_data);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_ars_command_specific_data_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_ars_command_specific_data \
	SET \
	device_handle=$device_handle \
		,  num_errors=$num_errors \
		,  ars_state=$ars_state \
		,  dpa_error_address_0=$dpa_error_address_0 \
		,  dpa_error_address_1=$dpa_error_address_1 \
		,  dpa_error_address_2=$dpa_error_address_2 \
		,  dpa_error_address_3=$dpa_error_address_3 \
		,  dpa_error_address_4=$dpa_error_address_4 \
		,  dpa_error_address_5=$dpa_error_address_5 \
		,  dpa_error_address_6=$dpa_error_address_6 \
		,  dpa_error_address_7=$dpa_error_address_7 \
		,  dpa_error_address_8=$dpa_error_address_8 \
		,  dpa_error_address_9=$dpa_error_address_9 \
		,  dpa_error_address_10=$dpa_error_address_10 \
		,  dpa_error_address_11=$dpa_error_address_11 \
		,  dpa_error_address_12=$dpa_error_address_12 \
		,  dpa_error_address_13=$dpa_error_address_13 \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_ars_command_specific_data(p_stmt, p_dimm_ars_command_specific_data);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_ars_command_specific_data_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_ars_command_specific_data \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_ars_command_specific_data_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_ars_command_specific_data_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_ars_command_specific_data_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_ars_command_specific_data_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_ars_command_specific_data_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_ars_command_specific_data *p_dimm_ars_command_specific_data,
	int history_id,
	int dimm_ars_command_specific_data_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_ars_command_specific_data, 0, sizeof (struct db_dimm_ars_command_specific_data) * dimm_ars_command_specific_data_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  num_errors,  ars_state,  dpa_error_address_0,  dpa_error_address_1,  dpa_error_address_2,  dpa_error_address_3,  dpa_error_address_4,  dpa_error_address_5,  dpa_error_address_6,  dpa_error_address_7,  dpa_error_address_8,  dpa_error_address_9,  dpa_error_address_10,  dpa_error_address_11,  dpa_error_address_12,  dpa_error_address_13  \
		FROM dimm_ars_command_specific_data_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_ars_command_specific_data_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_ars_command_specific_data(p_ps, p_stmt, &p_dimm_ars_command_specific_data[index]);
		local_get_dimm_ars_command_specific_data_relationships_history(p_ps, p_stmt, &p_dimm_ars_command_specific_data[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_ars_command_specific_data_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_ars_command_specific_data_history");
}

/*
 * --- END dimm_ars_command_specific_data ----------------
 */
/*
 * --- dimm_long_op_status ----------------
 */
void local_bind_dimm_long_op_status(sqlite3_stmt *p_stmt, struct db_dimm_long_op_status *p_dimm_long_op_status)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_long_op_status->device_handle);
	BIND_INTEGER(p_stmt, "$opcode", (long long)p_dimm_long_op_status->opcode);
	BIND_INTEGER(p_stmt, "$subopcode", (long long)p_dimm_long_op_status->subopcode);
	BIND_INTEGER(p_stmt, "$percent_complete", (long long)p_dimm_long_op_status->percent_complete);
	BIND_INTEGER(p_stmt, "$etc", (long long)p_dimm_long_op_status->etc);
	BIND_INTEGER(p_stmt, "$status_code", (long long)p_dimm_long_op_status->status_code);
}
void local_get_dimm_long_op_status_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_long_op_status *p_dimm_long_op_status)
{
}

void local_get_dimm_long_op_status_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_long_op_status *p_dimm_long_op_status,
	int history_id)
{
}

void local_row_to_dimm_long_op_status(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_long_op_status *p_dimm_long_op_status)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_long_op_status->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_long_op_status->opcode);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_long_op_status->subopcode);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_long_op_status->percent_complete);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_long_op_status->etc);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_long_op_status->status_code);
}
void db_print_dimm_long_op_status(struct db_dimm_long_op_status *p_value)
{
	printf("dimm_long_op_status.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_long_op_status.opcode: %lld\n", p_value->opcode);
	printf("dimm_long_op_status.subopcode: %lld\n", p_value->subopcode);
	printf("dimm_long_op_status.percent_complete: %lld\n", p_value->percent_complete);
	printf("dimm_long_op_status.etc: %lld\n", p_value->etc);
	printf("dimm_long_op_status.status_code: %lld\n", p_value->status_code);
}
enum db_return_codes db_add_dimm_long_op_status(const PersistentStore *p_ps,
	struct db_dimm_long_op_status *p_dimm_long_op_status)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_long_op_status \
		(device_handle, opcode, subopcode, percent_complete, etc, status_code)  \
		VALUES 		\
		($device_handle, \
		$opcode, \
		$subopcode, \
		$percent_complete, \
		$etc, \
		$status_code) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_long_op_status(p_stmt, p_dimm_long_op_status);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_long_op_status_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_long_op_status", p_count);
}
int db_get_dimm_long_op_statuss(const PersistentStore *p_ps,
	struct db_dimm_long_op_status *p_dimm_long_op_status,
	int dimm_long_op_status_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_long_op_status, 0, sizeof (struct db_dimm_long_op_status) * dimm_long_op_status_count);
	char *sql = "SELECT \
		device_handle \
		,  opcode \
		,  subopcode \
		,  percent_complete \
		,  etc \
		,  status_code \
		  \
		FROM dimm_long_op_status \
		       \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_long_op_status_count)
		{
			local_row_to_dimm_long_op_status(p_ps, p_stmt, &p_dimm_long_op_status[index]);
			local_get_dimm_long_op_status_relationships(p_ps, p_stmt, &p_dimm_long_op_status[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_long_op_statuss(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_long_op_status");
}

enum db_return_codes db_save_dimm_long_op_status_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_long_op_status *p_dimm_long_op_status)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_long_op_status temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_long_op_status_by_device_handle(p_ps, p_dimm_long_op_status->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_long_op_status_by_device_handle(p_ps,
				p_dimm_long_op_status->device_handle,
				p_dimm_long_op_status);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_long_op_status \
			( device_handle ,  opcode ,  subopcode ,  percent_complete ,  etc ,  status_code )  \
			VALUES 		\
			($device_handle, \
			$opcode, \
			$subopcode, \
			$percent_complete, \
			$etc, \
			$status_code) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_long_op_status(p_stmt, p_dimm_long_op_status);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_long_op_status_history \
			(history_id, \
				 device_handle,  opcode,  subopcode,  percent_complete,  etc,  status_code)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $opcode , \
				 $subopcode , \
				 $percent_complete , \
				 $etc , \
				 $status_code )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_long_op_status(p_stmt, p_dimm_long_op_status);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_long_op_status_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_long_op_status *p_dimm_long_op_status)
{
	memset(p_dimm_long_op_status, 0, sizeof (struct db_dimm_long_op_status));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  opcode,  subopcode,  percent_complete,  etc,  status_code  \
		FROM dimm_long_op_status \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_long_op_status(p_ps, p_stmt, p_dimm_long_op_status);
			local_get_dimm_long_op_status_relationships(p_ps, p_stmt, p_dimm_long_op_status);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_long_op_status_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_long_op_status *p_dimm_long_op_status)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_long_op_status \
	SET \
	device_handle=$device_handle \
		,  opcode=$opcode \
		,  subopcode=$subopcode \
		,  percent_complete=$percent_complete \
		,  etc=$etc \
		,  status_code=$status_code \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_long_op_status(p_stmt, p_dimm_long_op_status);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_long_op_status_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_long_op_status \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_long_op_status_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_long_op_status_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_long_op_status_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_long_op_status_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_long_op_status_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_long_op_status *p_dimm_long_op_status,
	int history_id,
	int dimm_long_op_status_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_long_op_status, 0, sizeof (struct db_dimm_long_op_status) * dimm_long_op_status_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  opcode,  subopcode,  percent_complete,  etc,  status_code  \
		FROM dimm_long_op_status_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_long_op_status_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_long_op_status(p_ps, p_stmt, &p_dimm_long_op_status[index]);
		local_get_dimm_long_op_status_relationships_history(p_ps, p_stmt, &p_dimm_long_op_status[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_long_op_status_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_long_op_status_history");
}

/*
 * --- END dimm_long_op_status ----------------
 */
/*
 * --- dimm_details ----------------
 */
void local_bind_dimm_details(sqlite3_stmt *p_stmt, struct db_dimm_details *p_dimm_details)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_details->device_handle);
	BIND_INTEGER(p_stmt, "$form_factor", (unsigned int)p_dimm_details->form_factor);
	BIND_INTEGER(p_stmt, "$data_width", (unsigned long long)p_dimm_details->data_width);
	BIND_INTEGER(p_stmt, "$total_width", (unsigned long long)p_dimm_details->total_width);
	BIND_INTEGER(p_stmt, "$size", (unsigned long long)p_dimm_details->size);
	BIND_INTEGER(p_stmt, "$speed", (unsigned long long)p_dimm_details->speed);
	BIND_TEXT(p_stmt, "$part_number", (char *)p_dimm_details->part_number);
	BIND_TEXT(p_stmt, "$device_locator", (char *)p_dimm_details->device_locator);
	BIND_TEXT(p_stmt, "$bank_label", (char *)p_dimm_details->bank_label);
	BIND_TEXT(p_stmt, "$manufacturer", (char *)p_dimm_details->manufacturer);
	BIND_INTEGER(p_stmt, "$type", (unsigned int)p_dimm_details->type);
	BIND_INTEGER(p_stmt, "$type_detail", (unsigned int)p_dimm_details->type_detail);
	BIND_INTEGER(p_stmt, "$id", (unsigned int)p_dimm_details->id);
}
void local_get_dimm_details_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_details *p_dimm_details)
{
}

void local_get_dimm_details_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_details *p_dimm_details,
	int history_id)
{
}

void local_row_to_dimm_details(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_details *p_dimm_details)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_details->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_details->form_factor);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_details->data_width);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_details->total_width);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_details->size);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_details->speed);
	TEXT_COLUMN(p_stmt,
		6,
		p_dimm_details->part_number,
		DIMM_DETAILS_PART_NUMBER_LEN);
	TEXT_COLUMN(p_stmt,
		7,
		p_dimm_details->device_locator,
		DIMM_DETAILS_DEVICE_LOCATOR_LEN);
	TEXT_COLUMN(p_stmt,
		8,
		p_dimm_details->bank_label,
		DIMM_DETAILS_BANK_LABEL_LEN);
	TEXT_COLUMN(p_stmt,
		9,
		p_dimm_details->manufacturer,
		DIMM_DETAILS_MANUFACTURER_LEN);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_details->type);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_details->type_detail);
	INTEGER_COLUMN(p_stmt,
		12,
		p_dimm_details->id);
}
void db_print_dimm_details(struct db_dimm_details *p_value)
{
	printf("dimm_details.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_details.form_factor: unsigned %d\n", p_value->form_factor);
	printf("dimm_details.data_width: unsigned %lld\n", p_value->data_width);
	printf("dimm_details.total_width: unsigned %lld\n", p_value->total_width);
	printf("dimm_details.size: unsigned %lld\n", p_value->size);
	printf("dimm_details.speed: unsigned %lld\n", p_value->speed);
	printf("dimm_details.part_number: %s\n", p_value->part_number);
	printf("dimm_details.device_locator: %s\n", p_value->device_locator);
	printf("dimm_details.bank_label: %s\n", p_value->bank_label);
	printf("dimm_details.manufacturer: %s\n", p_value->manufacturer);
	printf("dimm_details.type: unsigned %d\n", p_value->type);
	printf("dimm_details.type_detail: unsigned %d\n", p_value->type_detail);
	printf("dimm_details.id: unsigned %d\n", p_value->id);
}
enum db_return_codes db_add_dimm_details(const PersistentStore *p_ps,
	struct db_dimm_details *p_dimm_details)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_details \
		(device_handle, form_factor, data_width, total_width, size, speed, part_number, device_locator, bank_label, manufacturer, type, type_detail, id)  \
		VALUES 		\
		($device_handle, \
		$form_factor, \
		$data_width, \
		$total_width, \
		$size, \
		$speed, \
		$part_number, \
		$device_locator, \
		$bank_label, \
		$manufacturer, \
		$type, \
		$type_detail, \
		$id) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_details(p_stmt, p_dimm_details);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_details_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_details", p_count);
}
int db_get_dimm_detailss(const PersistentStore *p_ps,
	struct db_dimm_details *p_dimm_details,
	int dimm_details_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_details, 0, sizeof (struct db_dimm_details) * dimm_details_count);
	char *sql = "SELECT \
		device_handle \
		,  form_factor \
		,  data_width \
		,  total_width \
		,  size \
		,  speed \
		,  part_number \
		,  device_locator \
		,  bank_label \
		,  manufacturer \
		,  type \
		,  type_detail \
		,  id \
		  \
		FROM dimm_details \
		              \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_details_count)
		{
			local_row_to_dimm_details(p_ps, p_stmt, &p_dimm_details[index]);
			local_get_dimm_details_relationships(p_ps, p_stmt, &p_dimm_details[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_detailss(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_details");
}

enum db_return_codes db_save_dimm_details_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_details *p_dimm_details)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_details temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_details_by_id(p_ps, p_dimm_details->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_details_by_id(p_ps,
				p_dimm_details->id,
				p_dimm_details);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_details \
			( device_handle ,  form_factor ,  data_width ,  total_width ,  size ,  speed ,  part_number ,  device_locator ,  bank_label ,  manufacturer ,  type ,  type_detail ,  id )  \
			VALUES 		\
			($device_handle, \
			$form_factor, \
			$data_width, \
			$total_width, \
			$size, \
			$speed, \
			$part_number, \
			$device_locator, \
			$bank_label, \
			$manufacturer, \
			$type, \
			$type_detail, \
			$id) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_details(p_stmt, p_dimm_details);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_details_history \
			(history_id, \
				 device_handle,  form_factor,  data_width,  total_width,  size,  speed,  part_number,  device_locator,  bank_label,  manufacturer,  type,  type_detail,  id)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $form_factor , \
				 $data_width , \
				 $total_width , \
				 $size , \
				 $speed , \
				 $part_number , \
				 $device_locator , \
				 $bank_label , \
				 $manufacturer , \
				 $type , \
				 $type_detail , \
				 $id )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_details(p_stmt, p_dimm_details);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_details_by_id(const PersistentStore *p_ps,
	const unsigned int id,
	struct db_dimm_details *p_dimm_details)
{
	memset(p_dimm_details, 0, sizeof (struct db_dimm_details));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  form_factor,  data_width,  total_width,  size,  speed,  part_number,  device_locator,  bank_label,  manufacturer,  type,  type_detail,  id  \
		FROM dimm_details \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (unsigned int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_details(p_ps, p_stmt, p_dimm_details);
			local_get_dimm_details_relationships(p_ps, p_stmt, p_dimm_details);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_details_by_id(const PersistentStore *p_ps,
	const unsigned int id,
	struct db_dimm_details *p_dimm_details)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_details \
	SET \
	device_handle=$device_handle \
		,  form_factor=$form_factor \
		,  data_width=$data_width \
		,  total_width=$total_width \
		,  size=$size \
		,  speed=$speed \
		,  part_number=$part_number \
		,  device_locator=$device_locator \
		,  bank_label=$bank_label \
		,  manufacturer=$manufacturer \
		,  type=$type \
		,  type_detail=$type_detail \
		,  id=$id \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (unsigned int)id);
		local_bind_dimm_details(p_stmt, p_dimm_details);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_details_by_id(const PersistentStore *p_ps,
	const unsigned int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_details \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (unsigned int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_details_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_details_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_details_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_details_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_details_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_details *p_dimm_details,
	int history_id,
	int dimm_details_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_details, 0, sizeof (struct db_dimm_details) * dimm_details_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  form_factor,  data_width,  total_width,  size,  speed,  part_number,  device_locator,  bank_label,  manufacturer,  type,  type_detail,  id  \
		FROM dimm_details_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_details_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_details(p_ps, p_stmt, &p_dimm_details[index]);
		local_get_dimm_details_relationships_history(p_ps, p_stmt, &p_dimm_details[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_details_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_details_history");
}

/*
 * --- END dimm_details ----------------
 */
/*
 * --- dimm_security_info ----------------
 */
void local_bind_dimm_security_info(sqlite3_stmt *p_stmt, struct db_dimm_security_info *p_dimm_security_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_security_info->device_handle);
	BIND_INTEGER(p_stmt, "$security_state", (int)p_dimm_security_info->security_state);
}
void local_get_dimm_security_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_security_info *p_dimm_security_info)
{
}

void local_get_dimm_security_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_security_info *p_dimm_security_info,
	int history_id)
{
}

void local_row_to_dimm_security_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_security_info *p_dimm_security_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_security_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_security_info->security_state);
}
void db_print_dimm_security_info(struct db_dimm_security_info *p_value)
{
	printf("dimm_security_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_security_info.security_state: %d\n", p_value->security_state);
}
enum db_return_codes db_add_dimm_security_info(const PersistentStore *p_ps,
	struct db_dimm_security_info *p_dimm_security_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_security_info \
		(device_handle, security_state)  \
		VALUES 		\
		($device_handle, \
		$security_state) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_security_info(p_stmt, p_dimm_security_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_security_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_security_info", p_count);
}
int db_get_dimm_security_infos(const PersistentStore *p_ps,
	struct db_dimm_security_info *p_dimm_security_info,
	int dimm_security_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_security_info, 0, sizeof (struct db_dimm_security_info) * dimm_security_info_count);
	char *sql = "SELECT \
		device_handle \
		,  security_state \
		  \
		FROM dimm_security_info \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_security_info_count)
		{
			local_row_to_dimm_security_info(p_ps, p_stmt, &p_dimm_security_info[index]);
			local_get_dimm_security_info_relationships(p_ps, p_stmt, &p_dimm_security_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_security_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_security_info");
}

enum db_return_codes db_save_dimm_security_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_security_info *p_dimm_security_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_security_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_security_info_by_device_handle(p_ps, p_dimm_security_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_security_info_by_device_handle(p_ps,
				p_dimm_security_info->device_handle,
				p_dimm_security_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_security_info \
			( device_handle ,  security_state )  \
			VALUES 		\
			($device_handle, \
			$security_state) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_security_info(p_stmt, p_dimm_security_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_security_info_history \
			(history_id, \
				 device_handle,  security_state)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $security_state )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_security_info(p_stmt, p_dimm_security_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_security_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_security_info *p_dimm_security_info)
{
	memset(p_dimm_security_info, 0, sizeof (struct db_dimm_security_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  security_state  \
		FROM dimm_security_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_security_info(p_ps, p_stmt, p_dimm_security_info);
			local_get_dimm_security_info_relationships(p_ps, p_stmt, p_dimm_security_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_security_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_security_info *p_dimm_security_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_security_info \
	SET \
	device_handle=$device_handle \
		,  security_state=$security_state \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_security_info(p_stmt, p_dimm_security_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_security_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_security_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_security_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_security_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_security_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_security_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_security_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_security_info *p_dimm_security_info,
	int history_id,
	int dimm_security_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_security_info, 0, sizeof (struct db_dimm_security_info) * dimm_security_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  security_state  \
		FROM dimm_security_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_security_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_security_info(p_ps, p_stmt, &p_dimm_security_info[index]);
		local_get_dimm_security_info_relationships_history(p_ps, p_stmt, &p_dimm_security_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_security_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_security_info_history");
}

/*
 * --- END dimm_security_info ----------------
 */
/*
 * --- dimm_sanitize_info ----------------
 */
void local_bind_dimm_sanitize_info(sqlite3_stmt *p_stmt, struct db_dimm_sanitize_info *p_dimm_sanitize_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_sanitize_info->device_handle);
	BIND_INTEGER(p_stmt, "$sanitize_state", (int)p_dimm_sanitize_info->sanitize_state);
	BIND_INTEGER(p_stmt, "$sanitize_progress", (int)p_dimm_sanitize_info->sanitize_progress);
}
void local_get_dimm_sanitize_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_sanitize_info *p_dimm_sanitize_info)
{
}

void local_get_dimm_sanitize_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_sanitize_info *p_dimm_sanitize_info,
	int history_id)
{
}

void local_row_to_dimm_sanitize_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_sanitize_info *p_dimm_sanitize_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_sanitize_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_sanitize_info->sanitize_state);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_sanitize_info->sanitize_progress);
}
void db_print_dimm_sanitize_info(struct db_dimm_sanitize_info *p_value)
{
	printf("dimm_sanitize_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_sanitize_info.sanitize_state: %d\n", p_value->sanitize_state);
	printf("dimm_sanitize_info.sanitize_progress: %d\n", p_value->sanitize_progress);
}
enum db_return_codes db_add_dimm_sanitize_info(const PersistentStore *p_ps,
	struct db_dimm_sanitize_info *p_dimm_sanitize_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_sanitize_info \
		(device_handle, sanitize_state, sanitize_progress)  \
		VALUES 		\
		($device_handle, \
		$sanitize_state, \
		$sanitize_progress) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_sanitize_info(p_stmt, p_dimm_sanitize_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_sanitize_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_sanitize_info", p_count);
}
int db_get_dimm_sanitize_infos(const PersistentStore *p_ps,
	struct db_dimm_sanitize_info *p_dimm_sanitize_info,
	int dimm_sanitize_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_sanitize_info, 0, sizeof (struct db_dimm_sanitize_info) * dimm_sanitize_info_count);
	char *sql = "SELECT \
		device_handle \
		,  sanitize_state \
		,  sanitize_progress \
		  \
		FROM dimm_sanitize_info \
		    \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_sanitize_info_count)
		{
			local_row_to_dimm_sanitize_info(p_ps, p_stmt, &p_dimm_sanitize_info[index]);
			local_get_dimm_sanitize_info_relationships(p_ps, p_stmt, &p_dimm_sanitize_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_sanitize_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_sanitize_info");
}

enum db_return_codes db_save_dimm_sanitize_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_sanitize_info *p_dimm_sanitize_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_sanitize_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_sanitize_info_by_device_handle(p_ps, p_dimm_sanitize_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_sanitize_info_by_device_handle(p_ps,
				p_dimm_sanitize_info->device_handle,
				p_dimm_sanitize_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_sanitize_info \
			( device_handle ,  sanitize_state ,  sanitize_progress )  \
			VALUES 		\
			($device_handle, \
			$sanitize_state, \
			$sanitize_progress) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_sanitize_info(p_stmt, p_dimm_sanitize_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_sanitize_info_history \
			(history_id, \
				 device_handle,  sanitize_state,  sanitize_progress)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $sanitize_state , \
				 $sanitize_progress )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_sanitize_info(p_stmt, p_dimm_sanitize_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_sanitize_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_sanitize_info *p_dimm_sanitize_info)
{
	memset(p_dimm_sanitize_info, 0, sizeof (struct db_dimm_sanitize_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  sanitize_state,  sanitize_progress  \
		FROM dimm_sanitize_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_sanitize_info(p_ps, p_stmt, p_dimm_sanitize_info);
			local_get_dimm_sanitize_info_relationships(p_ps, p_stmt, p_dimm_sanitize_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_sanitize_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_sanitize_info *p_dimm_sanitize_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_sanitize_info \
	SET \
	device_handle=$device_handle \
		,  sanitize_state=$sanitize_state \
		,  sanitize_progress=$sanitize_progress \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_sanitize_info(p_stmt, p_dimm_sanitize_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_sanitize_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_sanitize_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_sanitize_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_sanitize_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_sanitize_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_sanitize_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_sanitize_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_sanitize_info *p_dimm_sanitize_info,
	int history_id,
	int dimm_sanitize_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_sanitize_info, 0, sizeof (struct db_dimm_sanitize_info) * dimm_sanitize_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  sanitize_state,  sanitize_progress  \
		FROM dimm_sanitize_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_sanitize_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_sanitize_info(p_ps, p_stmt, &p_dimm_sanitize_info[index]);
		local_get_dimm_sanitize_info_relationships_history(p_ps, p_stmt, &p_dimm_sanitize_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_sanitize_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_sanitize_info_history");
}

/*
 * --- END dimm_sanitize_info ----------------
 */
/*
 * --- fw_media_low_log_entry ----------------
 */
void local_bind_fw_media_low_log_entry(sqlite3_stmt *p_stmt, struct db_fw_media_low_log_entry *p_fw_media_low_log_entry)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_media_low_log_entry->device_handle);
	BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)p_fw_media_low_log_entry->system_timestamp);
	BIND_INTEGER(p_stmt, "$dpa", (unsigned long long)p_fw_media_low_log_entry->dpa);
	BIND_INTEGER(p_stmt, "$pda", (unsigned long long)p_fw_media_low_log_entry->pda);
	BIND_INTEGER(p_stmt, "$range", (unsigned int)p_fw_media_low_log_entry->range);
	BIND_INTEGER(p_stmt, "$error_type", (unsigned int)p_fw_media_low_log_entry->error_type);
	BIND_INTEGER(p_stmt, "$error_flags", (unsigned int)p_fw_media_low_log_entry->error_flags);
	BIND_INTEGER(p_stmt, "$transaction_type", (unsigned int)p_fw_media_low_log_entry->transaction_type);
}
void local_get_fw_media_low_log_entry_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_low_log_entry *p_fw_media_low_log_entry)
{
}

void local_get_fw_media_low_log_entry_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_low_log_entry *p_fw_media_low_log_entry,
	int history_id)
{
}

void local_row_to_fw_media_low_log_entry(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_low_log_entry *p_fw_media_low_log_entry)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_media_low_log_entry->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_media_low_log_entry->system_timestamp);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_media_low_log_entry->dpa);
	INTEGER_COLUMN(p_stmt,
		3,
		p_fw_media_low_log_entry->pda);
	INTEGER_COLUMN(p_stmt,
		4,
		p_fw_media_low_log_entry->range);
	INTEGER_COLUMN(p_stmt,
		5,
		p_fw_media_low_log_entry->error_type);
	INTEGER_COLUMN(p_stmt,
		6,
		p_fw_media_low_log_entry->error_flags);
	INTEGER_COLUMN(p_stmt,
		7,
		p_fw_media_low_log_entry->transaction_type);
}
void db_print_fw_media_low_log_entry(struct db_fw_media_low_log_entry *p_value)
{
	printf("fw_media_low_log_entry.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_media_low_log_entry.system_timestamp: unsigned %lld\n", p_value->system_timestamp);
	printf("fw_media_low_log_entry.dpa: unsigned %lld\n", p_value->dpa);
	printf("fw_media_low_log_entry.pda: unsigned %lld\n", p_value->pda);
	printf("fw_media_low_log_entry.range: unsigned %d\n", p_value->range);
	printf("fw_media_low_log_entry.error_type: unsigned %d\n", p_value->error_type);
	printf("fw_media_low_log_entry.error_flags: unsigned %d\n", p_value->error_flags);
	printf("fw_media_low_log_entry.transaction_type: unsigned %d\n", p_value->transaction_type);
}
enum db_return_codes db_add_fw_media_low_log_entry(const PersistentStore *p_ps,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_media_low_log_entry \
		(device_handle, system_timestamp, dpa, pda, range, error_type, error_flags, transaction_type)  \
		VALUES 		\
		($device_handle, \
		$system_timestamp, \
		$dpa, \
		$pda, \
		$range, \
		$error_type, \
		$error_flags, \
		$transaction_type) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_media_low_log_entry(p_stmt, p_fw_media_low_log_entry);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_low_log_entry_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_media_low_log_entry", p_count);
}
int db_get_fw_media_low_log_entrys(const PersistentStore *p_ps,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry,
	int fw_media_low_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_low_log_entry, 0, sizeof (struct db_fw_media_low_log_entry) * fw_media_low_log_entry_count);
	char *sql = "SELECT \
		device_handle \
		,  system_timestamp \
		,  dpa \
		,  pda \
		,  range \
		,  error_type \
		,  error_flags \
		,  transaction_type \
		  \
		FROM fw_media_low_log_entry \
		         \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_low_log_entry_count)
		{
			local_row_to_fw_media_low_log_entry(p_ps, p_stmt, &p_fw_media_low_log_entry[index]);
			local_get_fw_media_low_log_entry_relationships(p_ps, p_stmt, &p_fw_media_low_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_media_low_log_entrys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_entry");
}

enum db_return_codes db_save_fw_media_low_log_entry_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_media_low_log_entry temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_media_low_log_entry_by_system_timestamp(p_ps, p_fw_media_low_log_entry->system_timestamp, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_media_low_log_entry_by_system_timestamp(p_ps,
				p_fw_media_low_log_entry->system_timestamp,
				p_fw_media_low_log_entry);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_media_low_log_entry \
			( device_handle ,  system_timestamp ,  dpa ,  pda ,  range ,  error_type ,  error_flags ,  transaction_type )  \
			VALUES 		\
			($device_handle, \
			$system_timestamp, \
			$dpa, \
			$pda, \
			$range, \
			$error_type, \
			$error_flags, \
			$transaction_type) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_media_low_log_entry(p_stmt, p_fw_media_low_log_entry);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_media_low_log_entry_history \
			(history_id, \
				 device_handle,  system_timestamp,  dpa,  pda,  range,  error_type,  error_flags,  transaction_type)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $system_timestamp , \
				 $dpa , \
				 $pda , \
				 $range , \
				 $error_type , \
				 $error_flags , \
				 $transaction_type )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_media_low_log_entry(p_stmt, p_fw_media_low_log_entry);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_media_low_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry)
{
	memset(p_fw_media_low_log_entry, 0, sizeof (struct db_fw_media_low_log_entry));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  system_timestamp,  dpa,  pda,  range,  error_type,  error_flags,  transaction_type  \
		FROM fw_media_low_log_entry \
		WHERE  system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_media_low_log_entry(p_ps, p_stmt, p_fw_media_low_log_entry);
			local_get_fw_media_low_log_entry_relationships(p_ps, p_stmt, p_fw_media_low_log_entry);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_media_low_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_media_low_log_entry \
	SET \
	device_handle=$device_handle \
		,  system_timestamp=$system_timestamp \
		,  dpa=$dpa \
		,  pda=$pda \
		,  range=$range \
		,  error_type=$error_type \
		,  error_flags=$error_flags \
		,  transaction_type=$transaction_type \
		  \
	WHERE system_timestamp=$system_timestamp ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		local_bind_fw_media_low_log_entry(p_stmt, p_fw_media_low_log_entry);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_low_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_media_low_log_entry \
				 WHERE system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_media_low_log_entry_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_low_log_entry_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_low_log_entry_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_low_log_entry_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_media_low_log_entry_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry,
	int history_id,
	int fw_media_low_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_low_log_entry, 0, sizeof (struct db_fw_media_low_log_entry) * fw_media_low_log_entry_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  system_timestamp,  dpa,  pda,  range,  error_type,  error_flags,  transaction_type  \
		FROM fw_media_low_log_entry_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_low_log_entry_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_media_low_log_entry(p_ps, p_stmt, &p_fw_media_low_log_entry[index]);
		local_get_fw_media_low_log_entry_relationships_history(p_ps, p_stmt, &p_fw_media_low_log_entry[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_low_log_entry_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_entry_history");
}

enum db_return_codes db_get_fw_media_low_log_entry_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_media_low_log_entry WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_low_log_entry_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_media_low_log_entry_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_low_log_entrys_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry,
	int fw_media_low_log_entry_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  dpa ,  pda ,  range ,  error_type ,  error_flags ,  transaction_type  \
		FROM fw_media_low_log_entry \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_low_log_entry_count)
		{
			local_row_to_fw_media_low_log_entry(p_ps, p_stmt, &p_fw_media_low_log_entry[index]);
			local_get_fw_media_low_log_entry_relationships(p_ps, p_stmt, &p_fw_media_low_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_low_log_entrys_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_media_low_log_entry *p_fw_media_low_log_entry,
	int fw_media_low_log_entry_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  dpa ,  pda ,  range ,  error_type ,  error_flags ,  transaction_type  \
		FROM fw_media_low_log_entry_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_low_log_entry_count)
		{
			local_row_to_fw_media_low_log_entry(p_ps, p_stmt, &p_fw_media_low_log_entry[index]);
			local_get_fw_media_low_log_entry_relationships(p_ps, p_stmt, &p_fw_media_low_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_low_log_entry_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_media_low_log_entry \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END fw_media_low_log_entry ----------------
 */
/*
 * --- fw_media_high_log_entry ----------------
 */
void local_bind_fw_media_high_log_entry(sqlite3_stmt *p_stmt, struct db_fw_media_high_log_entry *p_fw_media_high_log_entry)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_media_high_log_entry->device_handle);
	BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)p_fw_media_high_log_entry->system_timestamp);
	BIND_INTEGER(p_stmt, "$dpa", (unsigned long long)p_fw_media_high_log_entry->dpa);
	BIND_INTEGER(p_stmt, "$pda", (unsigned long long)p_fw_media_high_log_entry->pda);
	BIND_INTEGER(p_stmt, "$range", (unsigned int)p_fw_media_high_log_entry->range);
	BIND_INTEGER(p_stmt, "$error_type", (unsigned int)p_fw_media_high_log_entry->error_type);
	BIND_INTEGER(p_stmt, "$error_flags", (unsigned int)p_fw_media_high_log_entry->error_flags);
	BIND_INTEGER(p_stmt, "$transaction_type", (unsigned int)p_fw_media_high_log_entry->transaction_type);
}
void local_get_fw_media_high_log_entry_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_high_log_entry *p_fw_media_high_log_entry)
{
}

void local_get_fw_media_high_log_entry_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_high_log_entry *p_fw_media_high_log_entry,
	int history_id)
{
}

void local_row_to_fw_media_high_log_entry(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_high_log_entry *p_fw_media_high_log_entry)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_media_high_log_entry->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_media_high_log_entry->system_timestamp);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_media_high_log_entry->dpa);
	INTEGER_COLUMN(p_stmt,
		3,
		p_fw_media_high_log_entry->pda);
	INTEGER_COLUMN(p_stmt,
		4,
		p_fw_media_high_log_entry->range);
	INTEGER_COLUMN(p_stmt,
		5,
		p_fw_media_high_log_entry->error_type);
	INTEGER_COLUMN(p_stmt,
		6,
		p_fw_media_high_log_entry->error_flags);
	INTEGER_COLUMN(p_stmt,
		7,
		p_fw_media_high_log_entry->transaction_type);
}
void db_print_fw_media_high_log_entry(struct db_fw_media_high_log_entry *p_value)
{
	printf("fw_media_high_log_entry.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_media_high_log_entry.system_timestamp: unsigned %lld\n", p_value->system_timestamp);
	printf("fw_media_high_log_entry.dpa: unsigned %lld\n", p_value->dpa);
	printf("fw_media_high_log_entry.pda: unsigned %lld\n", p_value->pda);
	printf("fw_media_high_log_entry.range: unsigned %d\n", p_value->range);
	printf("fw_media_high_log_entry.error_type: unsigned %d\n", p_value->error_type);
	printf("fw_media_high_log_entry.error_flags: unsigned %d\n", p_value->error_flags);
	printf("fw_media_high_log_entry.transaction_type: unsigned %d\n", p_value->transaction_type);
}
enum db_return_codes db_add_fw_media_high_log_entry(const PersistentStore *p_ps,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_media_high_log_entry \
		(device_handle, system_timestamp, dpa, pda, range, error_type, error_flags, transaction_type)  \
		VALUES 		\
		($device_handle, \
		$system_timestamp, \
		$dpa, \
		$pda, \
		$range, \
		$error_type, \
		$error_flags, \
		$transaction_type) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_media_high_log_entry(p_stmt, p_fw_media_high_log_entry);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_high_log_entry_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_media_high_log_entry", p_count);
}
int db_get_fw_media_high_log_entrys(const PersistentStore *p_ps,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry,
	int fw_media_high_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_high_log_entry, 0, sizeof (struct db_fw_media_high_log_entry) * fw_media_high_log_entry_count);
	char *sql = "SELECT \
		device_handle \
		,  system_timestamp \
		,  dpa \
		,  pda \
		,  range \
		,  error_type \
		,  error_flags \
		,  transaction_type \
		  \
		FROM fw_media_high_log_entry \
		         \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_high_log_entry_count)
		{
			local_row_to_fw_media_high_log_entry(p_ps, p_stmt, &p_fw_media_high_log_entry[index]);
			local_get_fw_media_high_log_entry_relationships(p_ps, p_stmt, &p_fw_media_high_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_media_high_log_entrys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_entry");
}

enum db_return_codes db_save_fw_media_high_log_entry_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_media_high_log_entry temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_media_high_log_entry_by_system_timestamp(p_ps, p_fw_media_high_log_entry->system_timestamp, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_media_high_log_entry_by_system_timestamp(p_ps,
				p_fw_media_high_log_entry->system_timestamp,
				p_fw_media_high_log_entry);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_media_high_log_entry \
			( device_handle ,  system_timestamp ,  dpa ,  pda ,  range ,  error_type ,  error_flags ,  transaction_type )  \
			VALUES 		\
			($device_handle, \
			$system_timestamp, \
			$dpa, \
			$pda, \
			$range, \
			$error_type, \
			$error_flags, \
			$transaction_type) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_media_high_log_entry(p_stmt, p_fw_media_high_log_entry);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_media_high_log_entry_history \
			(history_id, \
				 device_handle,  system_timestamp,  dpa,  pda,  range,  error_type,  error_flags,  transaction_type)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $system_timestamp , \
				 $dpa , \
				 $pda , \
				 $range , \
				 $error_type , \
				 $error_flags , \
				 $transaction_type )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_media_high_log_entry(p_stmt, p_fw_media_high_log_entry);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_media_high_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry)
{
	memset(p_fw_media_high_log_entry, 0, sizeof (struct db_fw_media_high_log_entry));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  system_timestamp,  dpa,  pda,  range,  error_type,  error_flags,  transaction_type  \
		FROM fw_media_high_log_entry \
		WHERE  system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_media_high_log_entry(p_ps, p_stmt, p_fw_media_high_log_entry);
			local_get_fw_media_high_log_entry_relationships(p_ps, p_stmt, p_fw_media_high_log_entry);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_media_high_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_media_high_log_entry \
	SET \
	device_handle=$device_handle \
		,  system_timestamp=$system_timestamp \
		,  dpa=$dpa \
		,  pda=$pda \
		,  range=$range \
		,  error_type=$error_type \
		,  error_flags=$error_flags \
		,  transaction_type=$transaction_type \
		  \
	WHERE system_timestamp=$system_timestamp ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		local_bind_fw_media_high_log_entry(p_stmt, p_fw_media_high_log_entry);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_high_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_media_high_log_entry \
				 WHERE system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_media_high_log_entry_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_high_log_entry_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_high_log_entry_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_high_log_entry_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_media_high_log_entry_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry,
	int history_id,
	int fw_media_high_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_high_log_entry, 0, sizeof (struct db_fw_media_high_log_entry) * fw_media_high_log_entry_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  system_timestamp,  dpa,  pda,  range,  error_type,  error_flags,  transaction_type  \
		FROM fw_media_high_log_entry_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_high_log_entry_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_media_high_log_entry(p_ps, p_stmt, &p_fw_media_high_log_entry[index]);
		local_get_fw_media_high_log_entry_relationships_history(p_ps, p_stmt, &p_fw_media_high_log_entry[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_high_log_entry_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_entry_history");
}

enum db_return_codes db_get_fw_media_high_log_entry_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_media_high_log_entry WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_high_log_entry_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_media_high_log_entry_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_high_log_entrys_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry,
	int fw_media_high_log_entry_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  dpa ,  pda ,  range ,  error_type ,  error_flags ,  transaction_type  \
		FROM fw_media_high_log_entry \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_high_log_entry_count)
		{
			local_row_to_fw_media_high_log_entry(p_ps, p_stmt, &p_fw_media_high_log_entry[index]);
			local_get_fw_media_high_log_entry_relationships(p_ps, p_stmt, &p_fw_media_high_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_high_log_entrys_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_media_high_log_entry *p_fw_media_high_log_entry,
	int fw_media_high_log_entry_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  dpa ,  pda ,  range ,  error_type ,  error_flags ,  transaction_type  \
		FROM fw_media_high_log_entry_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_high_log_entry_count)
		{
			local_row_to_fw_media_high_log_entry(p_ps, p_stmt, &p_fw_media_high_log_entry[index]);
			local_get_fw_media_high_log_entry_relationships(p_ps, p_stmt, &p_fw_media_high_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_high_log_entry_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_media_high_log_entry \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END fw_media_high_log_entry ----------------
 */
/*
 * --- fw_thermal_low_log_entry ----------------
 */
void local_bind_fw_thermal_low_log_entry(sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_thermal_low_log_entry->device_handle);
	BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)p_fw_thermal_low_log_entry->system_timestamp);
	BIND_INTEGER(p_stmt, "$host_reported_temp_data", (unsigned int)p_fw_thermal_low_log_entry->host_reported_temp_data);
}
void local_get_fw_thermal_low_log_entry_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry)
{
}

void local_get_fw_thermal_low_log_entry_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry,
	int history_id)
{
}

void local_row_to_fw_thermal_low_log_entry(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_thermal_low_log_entry->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_thermal_low_log_entry->system_timestamp);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_thermal_low_log_entry->host_reported_temp_data);
}
void db_print_fw_thermal_low_log_entry(struct db_fw_thermal_low_log_entry *p_value)
{
	printf("fw_thermal_low_log_entry.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_thermal_low_log_entry.system_timestamp: unsigned %lld\n", p_value->system_timestamp);
	printf("fw_thermal_low_log_entry.host_reported_temp_data: unsigned %d\n", p_value->host_reported_temp_data);
}
enum db_return_codes db_add_fw_thermal_low_log_entry(const PersistentStore *p_ps,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_thermal_low_log_entry \
		(device_handle, system_timestamp, host_reported_temp_data)  \
		VALUES 		\
		($device_handle, \
		$system_timestamp, \
		$host_reported_temp_data) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_thermal_low_log_entry(p_stmt, p_fw_thermal_low_log_entry);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_low_log_entry_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_thermal_low_log_entry", p_count);
}
int db_get_fw_thermal_low_log_entrys(const PersistentStore *p_ps,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry,
	int fw_thermal_low_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_low_log_entry, 0, sizeof (struct db_fw_thermal_low_log_entry) * fw_thermal_low_log_entry_count);
	char *sql = "SELECT \
		device_handle \
		,  system_timestamp \
		,  host_reported_temp_data \
		  \
		FROM fw_thermal_low_log_entry \
		    \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_low_log_entry_count)
		{
			local_row_to_fw_thermal_low_log_entry(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index]);
			local_get_fw_thermal_low_log_entry_relationships(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_thermal_low_log_entrys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_entry");
}

enum db_return_codes db_save_fw_thermal_low_log_entry_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_thermal_low_log_entry temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_thermal_low_log_entry_by_system_timestamp(p_ps, p_fw_thermal_low_log_entry->system_timestamp, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_thermal_low_log_entry_by_system_timestamp(p_ps,
				p_fw_thermal_low_log_entry->system_timestamp,
				p_fw_thermal_low_log_entry);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_thermal_low_log_entry \
			( device_handle ,  system_timestamp ,  host_reported_temp_data )  \
			VALUES 		\
			($device_handle, \
			$system_timestamp, \
			$host_reported_temp_data) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_thermal_low_log_entry(p_stmt, p_fw_thermal_low_log_entry);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_thermal_low_log_entry_history \
			(history_id, \
				 device_handle,  system_timestamp,  host_reported_temp_data)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $system_timestamp , \
				 $host_reported_temp_data )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_thermal_low_log_entry(p_stmt, p_fw_thermal_low_log_entry);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_low_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry)
{
	memset(p_fw_thermal_low_log_entry, 0, sizeof (struct db_fw_thermal_low_log_entry));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  system_timestamp,  host_reported_temp_data  \
		FROM fw_thermal_low_log_entry \
		WHERE  system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_thermal_low_log_entry(p_ps, p_stmt, p_fw_thermal_low_log_entry);
			local_get_fw_thermal_low_log_entry_relationships(p_ps, p_stmt, p_fw_thermal_low_log_entry);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_thermal_low_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_thermal_low_log_entry \
	SET \
	device_handle=$device_handle \
		,  system_timestamp=$system_timestamp \
		,  host_reported_temp_data=$host_reported_temp_data \
		  \
	WHERE system_timestamp=$system_timestamp ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		local_bind_fw_thermal_low_log_entry(p_stmt, p_fw_thermal_low_log_entry);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_low_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_thermal_low_log_entry \
				 WHERE system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_low_log_entry_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_low_log_entry_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_low_log_entry_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_low_log_entry_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_thermal_low_log_entry_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry,
	int history_id,
	int fw_thermal_low_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_low_log_entry, 0, sizeof (struct db_fw_thermal_low_log_entry) * fw_thermal_low_log_entry_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  system_timestamp,  host_reported_temp_data  \
		FROM fw_thermal_low_log_entry_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_low_log_entry_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_thermal_low_log_entry(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index]);
		local_get_fw_thermal_low_log_entry_relationships_history(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_low_log_entry_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_entry_history");
}

enum db_return_codes db_get_fw_thermal_low_log_entry_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_thermal_low_log_entry WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_low_log_entry_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_thermal_low_log_entry_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_low_log_entrys_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry,
	int fw_thermal_low_log_entry_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  host_reported_temp_data  \
		FROM fw_thermal_low_log_entry \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_low_log_entry_count)
		{
			local_row_to_fw_thermal_low_log_entry(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index]);
			local_get_fw_thermal_low_log_entry_relationships(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_low_log_entrys_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_thermal_low_log_entry *p_fw_thermal_low_log_entry,
	int fw_thermal_low_log_entry_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  host_reported_temp_data  \
		FROM fw_thermal_low_log_entry_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_low_log_entry_count)
		{
			local_row_to_fw_thermal_low_log_entry(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index]);
			local_get_fw_thermal_low_log_entry_relationships(p_ps, p_stmt, &p_fw_thermal_low_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_low_log_entry_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_thermal_low_log_entry \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END fw_thermal_low_log_entry ----------------
 */
/*
 * --- fw_thermal_high_log_entry ----------------
 */
void local_bind_fw_thermal_high_log_entry(sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_thermal_high_log_entry->device_handle);
	BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)p_fw_thermal_high_log_entry->system_timestamp);
	BIND_INTEGER(p_stmt, "$host_reported_temp_data", (unsigned int)p_fw_thermal_high_log_entry->host_reported_temp_data);
}
void local_get_fw_thermal_high_log_entry_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry)
{
}

void local_get_fw_thermal_high_log_entry_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry,
	int history_id)
{
}

void local_row_to_fw_thermal_high_log_entry(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_thermal_high_log_entry->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_thermal_high_log_entry->system_timestamp);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_thermal_high_log_entry->host_reported_temp_data);
}
void db_print_fw_thermal_high_log_entry(struct db_fw_thermal_high_log_entry *p_value)
{
	printf("fw_thermal_high_log_entry.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_thermal_high_log_entry.system_timestamp: unsigned %lld\n", p_value->system_timestamp);
	printf("fw_thermal_high_log_entry.host_reported_temp_data: unsigned %d\n", p_value->host_reported_temp_data);
}
enum db_return_codes db_add_fw_thermal_high_log_entry(const PersistentStore *p_ps,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_thermal_high_log_entry \
		(device_handle, system_timestamp, host_reported_temp_data)  \
		VALUES 		\
		($device_handle, \
		$system_timestamp, \
		$host_reported_temp_data) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_thermal_high_log_entry(p_stmt, p_fw_thermal_high_log_entry);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_high_log_entry_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_thermal_high_log_entry", p_count);
}
int db_get_fw_thermal_high_log_entrys(const PersistentStore *p_ps,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry,
	int fw_thermal_high_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_high_log_entry, 0, sizeof (struct db_fw_thermal_high_log_entry) * fw_thermal_high_log_entry_count);
	char *sql = "SELECT \
		device_handle \
		,  system_timestamp \
		,  host_reported_temp_data \
		  \
		FROM fw_thermal_high_log_entry \
		    \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_high_log_entry_count)
		{
			local_row_to_fw_thermal_high_log_entry(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index]);
			local_get_fw_thermal_high_log_entry_relationships(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_thermal_high_log_entrys(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_entry");
}

enum db_return_codes db_save_fw_thermal_high_log_entry_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_thermal_high_log_entry temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_thermal_high_log_entry_by_system_timestamp(p_ps, p_fw_thermal_high_log_entry->system_timestamp, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_thermal_high_log_entry_by_system_timestamp(p_ps,
				p_fw_thermal_high_log_entry->system_timestamp,
				p_fw_thermal_high_log_entry);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_thermal_high_log_entry \
			( device_handle ,  system_timestamp ,  host_reported_temp_data )  \
			VALUES 		\
			($device_handle, \
			$system_timestamp, \
			$host_reported_temp_data) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_thermal_high_log_entry(p_stmt, p_fw_thermal_high_log_entry);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_thermal_high_log_entry_history \
			(history_id, \
				 device_handle,  system_timestamp,  host_reported_temp_data)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $system_timestamp , \
				 $host_reported_temp_data )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_thermal_high_log_entry(p_stmt, p_fw_thermal_high_log_entry);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_high_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry)
{
	memset(p_fw_thermal_high_log_entry, 0, sizeof (struct db_fw_thermal_high_log_entry));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  system_timestamp,  host_reported_temp_data  \
		FROM fw_thermal_high_log_entry \
		WHERE  system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_thermal_high_log_entry(p_ps, p_stmt, p_fw_thermal_high_log_entry);
			local_get_fw_thermal_high_log_entry_relationships(p_ps, p_stmt, p_fw_thermal_high_log_entry);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_thermal_high_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_thermal_high_log_entry \
	SET \
	device_handle=$device_handle \
		,  system_timestamp=$system_timestamp \
		,  host_reported_temp_data=$host_reported_temp_data \
		  \
	WHERE system_timestamp=$system_timestamp ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		local_bind_fw_thermal_high_log_entry(p_stmt, p_fw_thermal_high_log_entry);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_high_log_entry_by_system_timestamp(const PersistentStore *p_ps,
	const unsigned long long system_timestamp)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_thermal_high_log_entry \
				 WHERE system_timestamp = $system_timestamp";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$system_timestamp", (unsigned long long)system_timestamp);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_high_log_entry_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_high_log_entry_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_high_log_entry_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_high_log_entry_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_thermal_high_log_entry_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry,
	int history_id,
	int fw_thermal_high_log_entry_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_high_log_entry, 0, sizeof (struct db_fw_thermal_high_log_entry) * fw_thermal_high_log_entry_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  system_timestamp,  host_reported_temp_data  \
		FROM fw_thermal_high_log_entry_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_high_log_entry_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_thermal_high_log_entry(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index]);
		local_get_fw_thermal_high_log_entry_relationships_history(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_high_log_entry_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_entry_history");
}

enum db_return_codes db_get_fw_thermal_high_log_entry_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_thermal_high_log_entry WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_high_log_entry_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM fw_thermal_high_log_entry_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_high_log_entrys_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry,
	int fw_thermal_high_log_entry_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  host_reported_temp_data  \
		FROM fw_thermal_high_log_entry \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_high_log_entry_count)
		{
			local_row_to_fw_thermal_high_log_entry(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index]);
			local_get_fw_thermal_high_log_entry_relationships(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_high_log_entrys_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_fw_thermal_high_log_entry *p_fw_thermal_high_log_entry,
	int fw_thermal_high_log_entry_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  system_timestamp ,  host_reported_temp_data  \
		FROM fw_thermal_high_log_entry_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_high_log_entry_count)
		{
			local_row_to_fw_thermal_high_log_entry(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index]);
			local_get_fw_thermal_high_log_entry_relationships(p_ps, p_stmt, &p_fw_thermal_high_log_entry[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_high_log_entry_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_thermal_high_log_entry \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END fw_thermal_high_log_entry ----------------
 */
/*
 * --- fw_media_low_log_info ----------------
 */
void local_bind_fw_media_low_log_info(sqlite3_stmt *p_stmt, struct db_fw_media_low_log_info *p_fw_media_low_log_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_media_low_log_info->device_handle);
	BIND_INTEGER(p_stmt, "$max_log_entries", (unsigned int)p_fw_media_low_log_info->max_log_entries);
	BIND_INTEGER(p_stmt, "$new_log_entries", (unsigned int)p_fw_media_low_log_info->new_log_entries);
	BIND_INTEGER(p_stmt, "$newest_log_entry_timestamp", (unsigned long long)p_fw_media_low_log_info->newest_log_entry_timestamp);
	BIND_INTEGER(p_stmt, "$oldest_log_entry_timestamp", (unsigned long long)p_fw_media_low_log_info->oldest_log_entry_timestamp);
}
void local_get_fw_media_low_log_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_low_log_info *p_fw_media_low_log_info)
{
}

void local_get_fw_media_low_log_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_low_log_info *p_fw_media_low_log_info,
	int history_id)
{
}

void local_row_to_fw_media_low_log_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_low_log_info *p_fw_media_low_log_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_media_low_log_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_media_low_log_info->max_log_entries);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_media_low_log_info->new_log_entries);
	INTEGER_COLUMN(p_stmt,
		3,
		p_fw_media_low_log_info->newest_log_entry_timestamp);
	INTEGER_COLUMN(p_stmt,
		4,
		p_fw_media_low_log_info->oldest_log_entry_timestamp);
}
void db_print_fw_media_low_log_info(struct db_fw_media_low_log_info *p_value)
{
	printf("fw_media_low_log_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_media_low_log_info.max_log_entries: unsigned %d\n", p_value->max_log_entries);
	printf("fw_media_low_log_info.new_log_entries: unsigned %d\n", p_value->new_log_entries);
	printf("fw_media_low_log_info.newest_log_entry_timestamp: unsigned %lld\n", p_value->newest_log_entry_timestamp);
	printf("fw_media_low_log_info.oldest_log_entry_timestamp: unsigned %lld\n", p_value->oldest_log_entry_timestamp);
}
enum db_return_codes db_add_fw_media_low_log_info(const PersistentStore *p_ps,
	struct db_fw_media_low_log_info *p_fw_media_low_log_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_media_low_log_info \
		(device_handle, max_log_entries, new_log_entries, newest_log_entry_timestamp, oldest_log_entry_timestamp)  \
		VALUES 		\
		($device_handle, \
		$max_log_entries, \
		$new_log_entries, \
		$newest_log_entry_timestamp, \
		$oldest_log_entry_timestamp) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_media_low_log_info(p_stmt, p_fw_media_low_log_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_low_log_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_media_low_log_info", p_count);
}
int db_get_fw_media_low_log_infos(const PersistentStore *p_ps,
	struct db_fw_media_low_log_info *p_fw_media_low_log_info,
	int fw_media_low_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_low_log_info, 0, sizeof (struct db_fw_media_low_log_info) * fw_media_low_log_info_count);
	char *sql = "SELECT \
		device_handle \
		,  max_log_entries \
		,  new_log_entries \
		,  newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp \
		  \
		FROM fw_media_low_log_info \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_low_log_info_count)
		{
			local_row_to_fw_media_low_log_info(p_ps, p_stmt, &p_fw_media_low_log_info[index]);
			local_get_fw_media_low_log_info_relationships(p_ps, p_stmt, &p_fw_media_low_log_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_media_low_log_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_info");
}

enum db_return_codes db_save_fw_media_low_log_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_media_low_log_info *p_fw_media_low_log_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_media_low_log_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_media_low_log_info_by_device_handle(p_ps, p_fw_media_low_log_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_media_low_log_info_by_device_handle(p_ps,
				p_fw_media_low_log_info->device_handle,
				p_fw_media_low_log_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_media_low_log_info \
			( device_handle ,  max_log_entries ,  new_log_entries ,  newest_log_entry_timestamp ,  oldest_log_entry_timestamp )  \
			VALUES 		\
			($device_handle, \
			$max_log_entries, \
			$new_log_entries, \
			$newest_log_entry_timestamp, \
			$oldest_log_entry_timestamp) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_media_low_log_info(p_stmt, p_fw_media_low_log_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_media_low_log_info_history \
			(history_id, \
				 device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $max_log_entries , \
				 $new_log_entries , \
				 $newest_log_entry_timestamp , \
				 $oldest_log_entry_timestamp )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_media_low_log_info(p_stmt, p_fw_media_low_log_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_media_low_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_media_low_log_info *p_fw_media_low_log_info)
{
	memset(p_fw_media_low_log_info, 0, sizeof (struct db_fw_media_low_log_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_media_low_log_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_media_low_log_info(p_ps, p_stmt, p_fw_media_low_log_info);
			local_get_fw_media_low_log_info_relationships(p_ps, p_stmt, p_fw_media_low_log_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_media_low_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_media_low_log_info *p_fw_media_low_log_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_media_low_log_info \
	SET \
	device_handle=$device_handle \
		,  max_log_entries=$max_log_entries \
		,  new_log_entries=$new_log_entries \
		,  newest_log_entry_timestamp=$newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp=$oldest_log_entry_timestamp \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_fw_media_low_log_info(p_stmt, p_fw_media_low_log_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_low_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_media_low_log_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_media_low_log_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_low_log_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_low_log_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_low_log_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_media_low_log_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_media_low_log_info *p_fw_media_low_log_info,
	int history_id,
	int fw_media_low_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_low_log_info, 0, sizeof (struct db_fw_media_low_log_info) * fw_media_low_log_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_media_low_log_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_low_log_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_media_low_log_info(p_ps, p_stmt, &p_fw_media_low_log_info[index]);
		local_get_fw_media_low_log_info_relationships_history(p_ps, p_stmt, &p_fw_media_low_log_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_low_log_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_info_history");
}

/*
 * --- END fw_media_low_log_info ----------------
 */
/*
 * --- fw_media_high_log_info ----------------
 */
void local_bind_fw_media_high_log_info(sqlite3_stmt *p_stmt, struct db_fw_media_high_log_info *p_fw_media_high_log_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_media_high_log_info->device_handle);
	BIND_INTEGER(p_stmt, "$max_log_entries", (unsigned int)p_fw_media_high_log_info->max_log_entries);
	BIND_INTEGER(p_stmt, "$new_log_entries", (unsigned int)p_fw_media_high_log_info->new_log_entries);
	BIND_INTEGER(p_stmt, "$newest_log_entry_timestamp", (unsigned long long)p_fw_media_high_log_info->newest_log_entry_timestamp);
	BIND_INTEGER(p_stmt, "$oldest_log_entry_timestamp", (unsigned long long)p_fw_media_high_log_info->oldest_log_entry_timestamp);
}
void local_get_fw_media_high_log_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_high_log_info *p_fw_media_high_log_info)
{
}

void local_get_fw_media_high_log_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_high_log_info *p_fw_media_high_log_info,
	int history_id)
{
}

void local_row_to_fw_media_high_log_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_media_high_log_info *p_fw_media_high_log_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_media_high_log_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_media_high_log_info->max_log_entries);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_media_high_log_info->new_log_entries);
	INTEGER_COLUMN(p_stmt,
		3,
		p_fw_media_high_log_info->newest_log_entry_timestamp);
	INTEGER_COLUMN(p_stmt,
		4,
		p_fw_media_high_log_info->oldest_log_entry_timestamp);
}
void db_print_fw_media_high_log_info(struct db_fw_media_high_log_info *p_value)
{
	printf("fw_media_high_log_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_media_high_log_info.max_log_entries: unsigned %d\n", p_value->max_log_entries);
	printf("fw_media_high_log_info.new_log_entries: unsigned %d\n", p_value->new_log_entries);
	printf("fw_media_high_log_info.newest_log_entry_timestamp: unsigned %lld\n", p_value->newest_log_entry_timestamp);
	printf("fw_media_high_log_info.oldest_log_entry_timestamp: unsigned %lld\n", p_value->oldest_log_entry_timestamp);
}
enum db_return_codes db_add_fw_media_high_log_info(const PersistentStore *p_ps,
	struct db_fw_media_high_log_info *p_fw_media_high_log_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_media_high_log_info \
		(device_handle, max_log_entries, new_log_entries, newest_log_entry_timestamp, oldest_log_entry_timestamp)  \
		VALUES 		\
		($device_handle, \
		$max_log_entries, \
		$new_log_entries, \
		$newest_log_entry_timestamp, \
		$oldest_log_entry_timestamp) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_media_high_log_info(p_stmt, p_fw_media_high_log_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_high_log_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_media_high_log_info", p_count);
}
int db_get_fw_media_high_log_infos(const PersistentStore *p_ps,
	struct db_fw_media_high_log_info *p_fw_media_high_log_info,
	int fw_media_high_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_high_log_info, 0, sizeof (struct db_fw_media_high_log_info) * fw_media_high_log_info_count);
	char *sql = "SELECT \
		device_handle \
		,  max_log_entries \
		,  new_log_entries \
		,  newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp \
		  \
		FROM fw_media_high_log_info \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_high_log_info_count)
		{
			local_row_to_fw_media_high_log_info(p_ps, p_stmt, &p_fw_media_high_log_info[index]);
			local_get_fw_media_high_log_info_relationships(p_ps, p_stmt, &p_fw_media_high_log_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_media_high_log_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_info");
}

enum db_return_codes db_save_fw_media_high_log_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_media_high_log_info *p_fw_media_high_log_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_media_high_log_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_media_high_log_info_by_device_handle(p_ps, p_fw_media_high_log_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_media_high_log_info_by_device_handle(p_ps,
				p_fw_media_high_log_info->device_handle,
				p_fw_media_high_log_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_media_high_log_info \
			( device_handle ,  max_log_entries ,  new_log_entries ,  newest_log_entry_timestamp ,  oldest_log_entry_timestamp )  \
			VALUES 		\
			($device_handle, \
			$max_log_entries, \
			$new_log_entries, \
			$newest_log_entry_timestamp, \
			$oldest_log_entry_timestamp) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_media_high_log_info(p_stmt, p_fw_media_high_log_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_media_high_log_info_history \
			(history_id, \
				 device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $max_log_entries , \
				 $new_log_entries , \
				 $newest_log_entry_timestamp , \
				 $oldest_log_entry_timestamp )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_media_high_log_info(p_stmt, p_fw_media_high_log_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_media_high_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_media_high_log_info *p_fw_media_high_log_info)
{
	memset(p_fw_media_high_log_info, 0, sizeof (struct db_fw_media_high_log_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_media_high_log_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_media_high_log_info(p_ps, p_stmt, p_fw_media_high_log_info);
			local_get_fw_media_high_log_info_relationships(p_ps, p_stmt, p_fw_media_high_log_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_media_high_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_media_high_log_info *p_fw_media_high_log_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_media_high_log_info \
	SET \
	device_handle=$device_handle \
		,  max_log_entries=$max_log_entries \
		,  new_log_entries=$new_log_entries \
		,  newest_log_entry_timestamp=$newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp=$oldest_log_entry_timestamp \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_fw_media_high_log_info(p_stmt, p_fw_media_high_log_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_high_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_media_high_log_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_media_high_log_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_high_log_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_media_high_log_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_media_high_log_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_media_high_log_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_media_high_log_info *p_fw_media_high_log_info,
	int history_id,
	int fw_media_high_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_media_high_log_info, 0, sizeof (struct db_fw_media_high_log_info) * fw_media_high_log_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_media_high_log_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_media_high_log_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_media_high_log_info(p_ps, p_stmt, &p_fw_media_high_log_info[index]);
		local_get_fw_media_high_log_info_relationships_history(p_ps, p_stmt, &p_fw_media_high_log_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_media_high_log_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_info_history");
}

/*
 * --- END fw_media_high_log_info ----------------
 */
/*
 * --- fw_thermal_low_log_info ----------------
 */
void local_bind_fw_thermal_low_log_info(sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_thermal_low_log_info->device_handle);
	BIND_INTEGER(p_stmt, "$max_log_entries", (unsigned int)p_fw_thermal_low_log_info->max_log_entries);
	BIND_INTEGER(p_stmt, "$new_log_entries", (unsigned int)p_fw_thermal_low_log_info->new_log_entries);
	BIND_INTEGER(p_stmt, "$newest_log_entry_timestamp", (unsigned long long)p_fw_thermal_low_log_info->newest_log_entry_timestamp);
	BIND_INTEGER(p_stmt, "$oldest_log_entry_timestamp", (unsigned long long)p_fw_thermal_low_log_info->oldest_log_entry_timestamp);
}
void local_get_fw_thermal_low_log_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info)
{
}

void local_get_fw_thermal_low_log_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info,
	int history_id)
{
}

void local_row_to_fw_thermal_low_log_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_thermal_low_log_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_thermal_low_log_info->max_log_entries);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_thermal_low_log_info->new_log_entries);
	INTEGER_COLUMN(p_stmt,
		3,
		p_fw_thermal_low_log_info->newest_log_entry_timestamp);
	INTEGER_COLUMN(p_stmt,
		4,
		p_fw_thermal_low_log_info->oldest_log_entry_timestamp);
}
void db_print_fw_thermal_low_log_info(struct db_fw_thermal_low_log_info *p_value)
{
	printf("fw_thermal_low_log_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_thermal_low_log_info.max_log_entries: unsigned %d\n", p_value->max_log_entries);
	printf("fw_thermal_low_log_info.new_log_entries: unsigned %d\n", p_value->new_log_entries);
	printf("fw_thermal_low_log_info.newest_log_entry_timestamp: unsigned %lld\n", p_value->newest_log_entry_timestamp);
	printf("fw_thermal_low_log_info.oldest_log_entry_timestamp: unsigned %lld\n", p_value->oldest_log_entry_timestamp);
}
enum db_return_codes db_add_fw_thermal_low_log_info(const PersistentStore *p_ps,
	struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_thermal_low_log_info \
		(device_handle, max_log_entries, new_log_entries, newest_log_entry_timestamp, oldest_log_entry_timestamp)  \
		VALUES 		\
		($device_handle, \
		$max_log_entries, \
		$new_log_entries, \
		$newest_log_entry_timestamp, \
		$oldest_log_entry_timestamp) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_thermal_low_log_info(p_stmt, p_fw_thermal_low_log_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_low_log_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_thermal_low_log_info", p_count);
}
int db_get_fw_thermal_low_log_infos(const PersistentStore *p_ps,
	struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info,
	int fw_thermal_low_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_low_log_info, 0, sizeof (struct db_fw_thermal_low_log_info) * fw_thermal_low_log_info_count);
	char *sql = "SELECT \
		device_handle \
		,  max_log_entries \
		,  new_log_entries \
		,  newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp \
		  \
		FROM fw_thermal_low_log_info \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_low_log_info_count)
		{
			local_row_to_fw_thermal_low_log_info(p_ps, p_stmt, &p_fw_thermal_low_log_info[index]);
			local_get_fw_thermal_low_log_info_relationships(p_ps, p_stmt, &p_fw_thermal_low_log_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_thermal_low_log_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_info");
}

enum db_return_codes db_save_fw_thermal_low_log_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_thermal_low_log_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_thermal_low_log_info_by_device_handle(p_ps, p_fw_thermal_low_log_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_thermal_low_log_info_by_device_handle(p_ps,
				p_fw_thermal_low_log_info->device_handle,
				p_fw_thermal_low_log_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_thermal_low_log_info \
			( device_handle ,  max_log_entries ,  new_log_entries ,  newest_log_entry_timestamp ,  oldest_log_entry_timestamp )  \
			VALUES 		\
			($device_handle, \
			$max_log_entries, \
			$new_log_entries, \
			$newest_log_entry_timestamp, \
			$oldest_log_entry_timestamp) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_thermal_low_log_info(p_stmt, p_fw_thermal_low_log_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_thermal_low_log_info_history \
			(history_id, \
				 device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $max_log_entries , \
				 $new_log_entries , \
				 $newest_log_entry_timestamp , \
				 $oldest_log_entry_timestamp )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_thermal_low_log_info(p_stmt, p_fw_thermal_low_log_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_low_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info)
{
	memset(p_fw_thermal_low_log_info, 0, sizeof (struct db_fw_thermal_low_log_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_thermal_low_log_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_thermal_low_log_info(p_ps, p_stmt, p_fw_thermal_low_log_info);
			local_get_fw_thermal_low_log_info_relationships(p_ps, p_stmt, p_fw_thermal_low_log_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_thermal_low_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_thermal_low_log_info \
	SET \
	device_handle=$device_handle \
		,  max_log_entries=$max_log_entries \
		,  new_log_entries=$new_log_entries \
		,  newest_log_entry_timestamp=$newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp=$oldest_log_entry_timestamp \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_fw_thermal_low_log_info(p_stmt, p_fw_thermal_low_log_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_low_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_thermal_low_log_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_low_log_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_low_log_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_low_log_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_low_log_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_thermal_low_log_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_thermal_low_log_info *p_fw_thermal_low_log_info,
	int history_id,
	int fw_thermal_low_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_low_log_info, 0, sizeof (struct db_fw_thermal_low_log_info) * fw_thermal_low_log_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_thermal_low_log_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_low_log_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_thermal_low_log_info(p_ps, p_stmt, &p_fw_thermal_low_log_info[index]);
		local_get_fw_thermal_low_log_info_relationships_history(p_ps, p_stmt, &p_fw_thermal_low_log_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_low_log_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_info_history");
}

/*
 * --- END fw_thermal_low_log_info ----------------
 */
/*
 * --- fw_thermal_high_log_info ----------------
 */
void local_bind_fw_thermal_high_log_info(sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_fw_thermal_high_log_info->device_handle);
	BIND_INTEGER(p_stmt, "$max_log_entries", (unsigned int)p_fw_thermal_high_log_info->max_log_entries);
	BIND_INTEGER(p_stmt, "$new_log_entries", (unsigned int)p_fw_thermal_high_log_info->new_log_entries);
	BIND_INTEGER(p_stmt, "$newest_log_entry_timestamp", (unsigned long long)p_fw_thermal_high_log_info->newest_log_entry_timestamp);
	BIND_INTEGER(p_stmt, "$oldest_log_entry_timestamp", (unsigned long long)p_fw_thermal_high_log_info->oldest_log_entry_timestamp);
}
void local_get_fw_thermal_high_log_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info)
{
}

void local_get_fw_thermal_high_log_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info,
	int history_id)
{
}

void local_row_to_fw_thermal_high_log_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_fw_thermal_high_log_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_fw_thermal_high_log_info->max_log_entries);
	INTEGER_COLUMN(p_stmt,
		2,
		p_fw_thermal_high_log_info->new_log_entries);
	INTEGER_COLUMN(p_stmt,
		3,
		p_fw_thermal_high_log_info->newest_log_entry_timestamp);
	INTEGER_COLUMN(p_stmt,
		4,
		p_fw_thermal_high_log_info->oldest_log_entry_timestamp);
}
void db_print_fw_thermal_high_log_info(struct db_fw_thermal_high_log_info *p_value)
{
	printf("fw_thermal_high_log_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("fw_thermal_high_log_info.max_log_entries: unsigned %d\n", p_value->max_log_entries);
	printf("fw_thermal_high_log_info.new_log_entries: unsigned %d\n", p_value->new_log_entries);
	printf("fw_thermal_high_log_info.newest_log_entry_timestamp: unsigned %lld\n", p_value->newest_log_entry_timestamp);
	printf("fw_thermal_high_log_info.oldest_log_entry_timestamp: unsigned %lld\n", p_value->oldest_log_entry_timestamp);
}
enum db_return_codes db_add_fw_thermal_high_log_info(const PersistentStore *p_ps,
	struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO fw_thermal_high_log_info \
		(device_handle, max_log_entries, new_log_entries, newest_log_entry_timestamp, oldest_log_entry_timestamp)  \
		VALUES 		\
		($device_handle, \
		$max_log_entries, \
		$new_log_entries, \
		$newest_log_entry_timestamp, \
		$oldest_log_entry_timestamp) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_fw_thermal_high_log_info(p_stmt, p_fw_thermal_high_log_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_high_log_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "fw_thermal_high_log_info", p_count);
}
int db_get_fw_thermal_high_log_infos(const PersistentStore *p_ps,
	struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info,
	int fw_thermal_high_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_high_log_info, 0, sizeof (struct db_fw_thermal_high_log_info) * fw_thermal_high_log_info_count);
	char *sql = "SELECT \
		device_handle \
		,  max_log_entries \
		,  new_log_entries \
		,  newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp \
		  \
		FROM fw_thermal_high_log_info \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_high_log_info_count)
		{
			local_row_to_fw_thermal_high_log_info(p_ps, p_stmt, &p_fw_thermal_high_log_info[index]);
			local_get_fw_thermal_high_log_info_relationships(p_ps, p_stmt, &p_fw_thermal_high_log_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_fw_thermal_high_log_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_info");
}

enum db_return_codes db_save_fw_thermal_high_log_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_fw_thermal_high_log_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_fw_thermal_high_log_info_by_device_handle(p_ps, p_fw_thermal_high_log_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_fw_thermal_high_log_info_by_device_handle(p_ps,
				p_fw_thermal_high_log_info->device_handle,
				p_fw_thermal_high_log_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO fw_thermal_high_log_info \
			( device_handle ,  max_log_entries ,  new_log_entries ,  newest_log_entry_timestamp ,  oldest_log_entry_timestamp )  \
			VALUES 		\
			($device_handle, \
			$max_log_entries, \
			$new_log_entries, \
			$newest_log_entry_timestamp, \
			$oldest_log_entry_timestamp) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_fw_thermal_high_log_info(p_stmt, p_fw_thermal_high_log_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO fw_thermal_high_log_info_history \
			(history_id, \
				 device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $max_log_entries , \
				 $new_log_entries , \
				 $newest_log_entry_timestamp , \
				 $oldest_log_entry_timestamp )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_fw_thermal_high_log_info(p_stmt, p_fw_thermal_high_log_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_high_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info)
{
	memset(p_fw_thermal_high_log_info, 0, sizeof (struct db_fw_thermal_high_log_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_thermal_high_log_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_fw_thermal_high_log_info(p_ps, p_stmt, p_fw_thermal_high_log_info);
			local_get_fw_thermal_high_log_info_relationships(p_ps, p_stmt, p_fw_thermal_high_log_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_fw_thermal_high_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE fw_thermal_high_log_info \
	SET \
	device_handle=$device_handle \
		,  max_log_entries=$max_log_entries \
		,  new_log_entries=$new_log_entries \
		,  newest_log_entry_timestamp=$newest_log_entry_timestamp \
		,  oldest_log_entry_timestamp=$oldest_log_entry_timestamp \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_fw_thermal_high_log_info(p_stmt, p_fw_thermal_high_log_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_high_log_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM fw_thermal_high_log_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_fw_thermal_high_log_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_high_log_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_fw_thermal_high_log_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM fw_thermal_high_log_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_fw_thermal_high_log_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_fw_thermal_high_log_info *p_fw_thermal_high_log_info,
	int history_id,
	int fw_thermal_high_log_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_fw_thermal_high_log_info, 0, sizeof (struct db_fw_thermal_high_log_info) * fw_thermal_high_log_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  max_log_entries,  new_log_entries,  newest_log_entry_timestamp,  oldest_log_entry_timestamp  \
		FROM fw_thermal_high_log_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < fw_thermal_high_log_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_fw_thermal_high_log_info(p_ps, p_stmt, &p_fw_thermal_high_log_info[index]);
		local_get_fw_thermal_high_log_info_relationships_history(p_ps, p_stmt, &p_fw_thermal_high_log_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_fw_thermal_high_log_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_info_history");
}

/*
 * --- END fw_thermal_high_log_info ----------------
 */
/*
 * --- dimm_fw_log_level ----------------
 */
void local_bind_dimm_fw_log_level(sqlite3_stmt *p_stmt, struct db_dimm_fw_log_level *p_dimm_fw_log_level)
{
	BIND_INTEGER(p_stmt, "$device_handle", (int)p_dimm_fw_log_level->device_handle);
	BIND_INTEGER(p_stmt, "$log_level", (int)p_dimm_fw_log_level->log_level);
}
void local_get_dimm_fw_log_level_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_log_level *p_dimm_fw_log_level)
{
}

void local_get_dimm_fw_log_level_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_log_level *p_dimm_fw_log_level,
	int history_id)
{
}

void local_row_to_dimm_fw_log_level(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_log_level *p_dimm_fw_log_level)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_fw_log_level->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_fw_log_level->log_level);
}
void db_print_dimm_fw_log_level(struct db_dimm_fw_log_level *p_value)
{
	printf("dimm_fw_log_level.device_handle: %d\n", p_value->device_handle);
	printf("dimm_fw_log_level.log_level: %d\n", p_value->log_level);
}
enum db_return_codes db_add_dimm_fw_log_level(const PersistentStore *p_ps,
	struct db_dimm_fw_log_level *p_dimm_fw_log_level)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_fw_log_level \
		(device_handle, log_level)  \
		VALUES 		\
		($device_handle, \
		$log_level) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_fw_log_level(p_stmt, p_dimm_fw_log_level);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_log_level_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_fw_log_level", p_count);
}
int db_get_dimm_fw_log_levels(const PersistentStore *p_ps,
	struct db_dimm_fw_log_level *p_dimm_fw_log_level,
	int dimm_fw_log_level_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_log_level, 0, sizeof (struct db_dimm_fw_log_level) * dimm_fw_log_level_count);
	char *sql = "SELECT \
		device_handle \
		,  log_level \
		  \
		FROM dimm_fw_log_level \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_log_level_count)
		{
			local_row_to_dimm_fw_log_level(p_ps, p_stmt, &p_dimm_fw_log_level[index]);
			local_get_dimm_fw_log_level_relationships(p_ps, p_stmt, &p_dimm_fw_log_level[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_fw_log_levels(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_log_level");
}

enum db_return_codes db_save_dimm_fw_log_level_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_fw_log_level *p_dimm_fw_log_level)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_fw_log_level temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_fw_log_level_by_device_handle(p_ps, p_dimm_fw_log_level->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_fw_log_level_by_device_handle(p_ps,
				p_dimm_fw_log_level->device_handle,
				p_dimm_fw_log_level);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_fw_log_level \
			( device_handle ,  log_level )  \
			VALUES 		\
			($device_handle, \
			$log_level) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_fw_log_level(p_stmt, p_dimm_fw_log_level);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_fw_log_level_history \
			(history_id, \
				 device_handle,  log_level)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $log_level )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_fw_log_level(p_stmt, p_dimm_fw_log_level);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_log_level_by_device_handle(const PersistentStore *p_ps,
	const int device_handle,
	struct db_dimm_fw_log_level *p_dimm_fw_log_level)
{
	memset(p_dimm_fw_log_level, 0, sizeof (struct db_dimm_fw_log_level));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  log_level  \
		FROM dimm_fw_log_level \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_fw_log_level(p_ps, p_stmt, p_dimm_fw_log_level);
			local_get_dimm_fw_log_level_relationships(p_ps, p_stmt, p_dimm_fw_log_level);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_fw_log_level_by_device_handle(const PersistentStore *p_ps,
	const int device_handle,
	struct db_dimm_fw_log_level *p_dimm_fw_log_level)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_fw_log_level \
	SET \
	device_handle=$device_handle \
		,  log_level=$log_level \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (int)device_handle);
		local_bind_dimm_fw_log_level(p_stmt, p_dimm_fw_log_level);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_log_level_by_device_handle(const PersistentStore *p_ps,
	const int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_fw_log_level \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_log_level_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_log_level_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_log_level_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_log_level_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_fw_log_level_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_fw_log_level *p_dimm_fw_log_level,
	int history_id,
	int dimm_fw_log_level_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_log_level, 0, sizeof (struct db_dimm_fw_log_level) * dimm_fw_log_level_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  log_level  \
		FROM dimm_fw_log_level_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_log_level_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_fw_log_level(p_ps, p_stmt, &p_dimm_fw_log_level[index]);
		local_get_dimm_fw_log_level_relationships_history(p_ps, p_stmt, &p_dimm_fw_log_level[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_log_level_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_log_level_history");
}

/*
 * --- END dimm_fw_log_level ----------------
 */
/*
 * --- dimm_fw_time ----------------
 */
void local_bind_dimm_fw_time(sqlite3_stmt *p_stmt, struct db_dimm_fw_time *p_dimm_fw_time)
{
	BIND_INTEGER(p_stmt, "$device_handle", (int)p_dimm_fw_time->device_handle);
	BIND_INTEGER(p_stmt, "$time", (unsigned long long)p_dimm_fw_time->time);
}
void local_get_dimm_fw_time_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_time *p_dimm_fw_time)
{
}

void local_get_dimm_fw_time_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_time *p_dimm_fw_time,
	int history_id)
{
}

void local_row_to_dimm_fw_time(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_fw_time *p_dimm_fw_time)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_fw_time->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_fw_time->time);
}
void db_print_dimm_fw_time(struct db_dimm_fw_time *p_value)
{
	printf("dimm_fw_time.device_handle: %d\n", p_value->device_handle);
	printf("dimm_fw_time.time: unsigned %lld\n", p_value->time);
}
enum db_return_codes db_add_dimm_fw_time(const PersistentStore *p_ps,
	struct db_dimm_fw_time *p_dimm_fw_time)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_fw_time \
		(device_handle, time)  \
		VALUES 		\
		($device_handle, \
		$time) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_fw_time(p_stmt, p_dimm_fw_time);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_time_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_fw_time", p_count);
}
int db_get_dimm_fw_times(const PersistentStore *p_ps,
	struct db_dimm_fw_time *p_dimm_fw_time,
	int dimm_fw_time_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_time, 0, sizeof (struct db_dimm_fw_time) * dimm_fw_time_count);
	char *sql = "SELECT \
		device_handle \
		,  time \
		  \
		FROM dimm_fw_time \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_time_count)
		{
			local_row_to_dimm_fw_time(p_ps, p_stmt, &p_dimm_fw_time[index]);
			local_get_dimm_fw_time_relationships(p_ps, p_stmt, &p_dimm_fw_time[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_fw_times(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_time");
}

enum db_return_codes db_save_dimm_fw_time_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_fw_time *p_dimm_fw_time)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_fw_time temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_fw_time_by_device_handle(p_ps, p_dimm_fw_time->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_fw_time_by_device_handle(p_ps,
				p_dimm_fw_time->device_handle,
				p_dimm_fw_time);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_fw_time \
			( device_handle ,  time )  \
			VALUES 		\
			($device_handle, \
			$time) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_fw_time(p_stmt, p_dimm_fw_time);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_fw_time_history \
			(history_id, \
				 device_handle,  time)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $time )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_fw_time(p_stmt, p_dimm_fw_time);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_time_by_device_handle(const PersistentStore *p_ps,
	const int device_handle,
	struct db_dimm_fw_time *p_dimm_fw_time)
{
	memset(p_dimm_fw_time, 0, sizeof (struct db_dimm_fw_time));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  time  \
		FROM dimm_fw_time \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_fw_time(p_ps, p_stmt, p_dimm_fw_time);
			local_get_dimm_fw_time_relationships(p_ps, p_stmt, p_dimm_fw_time);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_fw_time_by_device_handle(const PersistentStore *p_ps,
	const int device_handle,
	struct db_dimm_fw_time *p_dimm_fw_time)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_fw_time \
	SET \
	device_handle=$device_handle \
		,  time=$time \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (int)device_handle);
		local_bind_dimm_fw_time(p_stmt, p_dimm_fw_time);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_time_by_device_handle(const PersistentStore *p_ps,
	const int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_fw_time \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_fw_time_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_time_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_fw_time_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_fw_time_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_fw_time_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_fw_time *p_dimm_fw_time,
	int history_id,
	int dimm_fw_time_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_fw_time, 0, sizeof (struct db_dimm_fw_time) * dimm_fw_time_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  time  \
		FROM dimm_fw_time_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_fw_time_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_fw_time(p_ps, p_stmt, &p_dimm_fw_time[index]);
		local_get_dimm_fw_time_relationships_history(p_ps, p_stmt, &p_dimm_fw_time[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_fw_time_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_time_history");
}

/*
 * --- END dimm_fw_time ----------------
 */
/*
 * --- dimm_platform_config ----------------
 */
void local_bind_dimm_platform_config(sqlite3_stmt *p_stmt, struct db_dimm_platform_config *p_dimm_platform_config)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_platform_config->device_handle);
	BIND_TEXT(p_stmt, "$signature", (char *)p_dimm_platform_config->signature);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_dimm_platform_config->length);
	BIND_INTEGER(p_stmt, "$revision", (unsigned int)p_dimm_platform_config->revision);
	BIND_INTEGER(p_stmt, "$checksum", (unsigned int)p_dimm_platform_config->checksum);
	BIND_TEXT(p_stmt, "$oem_id", (char *)p_dimm_platform_config->oem_id);
	BIND_TEXT(p_stmt, "$oem_table_id", (char *)p_dimm_platform_config->oem_table_id);
	BIND_INTEGER(p_stmt, "$oem_revision", (unsigned int)p_dimm_platform_config->oem_revision);
	BIND_INTEGER(p_stmt, "$creator_id", (unsigned int)p_dimm_platform_config->creator_id);
	BIND_INTEGER(p_stmt, "$creator_revision", (unsigned int)p_dimm_platform_config->creator_revision);
	BIND_INTEGER(p_stmt, "$current_config_size", (unsigned int)p_dimm_platform_config->current_config_size);
	BIND_INTEGER(p_stmt, "$current_config_offset", (unsigned int)p_dimm_platform_config->current_config_offset);
	BIND_INTEGER(p_stmt, "$config_input_size", (unsigned int)p_dimm_platform_config->config_input_size);
	BIND_INTEGER(p_stmt, "$config_input_offset", (unsigned int)p_dimm_platform_config->config_input_offset);
	BIND_INTEGER(p_stmt, "$config_output_size", (unsigned int)p_dimm_platform_config->config_output_size);
	BIND_INTEGER(p_stmt, "$config_output_offset", (unsigned int)p_dimm_platform_config->config_output_offset);
}
void local_get_dimm_platform_config_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_platform_config *p_dimm_platform_config)
{
}

void local_get_dimm_platform_config_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_platform_config *p_dimm_platform_config,
	int history_id)
{
}

void local_row_to_dimm_platform_config(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_platform_config *p_dimm_platform_config)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_platform_config->device_handle);
	TEXT_COLUMN(p_stmt,
		1,
		p_dimm_platform_config->signature,
		DIMM_PLATFORM_CONFIG_SIGNATURE_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_platform_config->length);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_platform_config->revision);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_platform_config->checksum);
	TEXT_COLUMN(p_stmt,
		5,
		p_dimm_platform_config->oem_id,
		DIMM_PLATFORM_CONFIG_OEM_ID_LEN);
	TEXT_COLUMN(p_stmt,
		6,
		p_dimm_platform_config->oem_table_id,
		DIMM_PLATFORM_CONFIG_OEM_TABLE_ID_LEN);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_platform_config->oem_revision);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_platform_config->creator_id);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_platform_config->creator_revision);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_platform_config->current_config_size);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_platform_config->current_config_offset);
	INTEGER_COLUMN(p_stmt,
		12,
		p_dimm_platform_config->config_input_size);
	INTEGER_COLUMN(p_stmt,
		13,
		p_dimm_platform_config->config_input_offset);
	INTEGER_COLUMN(p_stmt,
		14,
		p_dimm_platform_config->config_output_size);
	INTEGER_COLUMN(p_stmt,
		15,
		p_dimm_platform_config->config_output_offset);
}
void db_print_dimm_platform_config(struct db_dimm_platform_config *p_value)
{
	printf("dimm_platform_config.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_platform_config.signature: %s\n", p_value->signature);
	printf("dimm_platform_config.length: unsigned %d\n", p_value->length);
	printf("dimm_platform_config.revision: unsigned %d\n", p_value->revision);
	printf("dimm_platform_config.checksum: unsigned %d\n", p_value->checksum);
	printf("dimm_platform_config.oem_id: %s\n", p_value->oem_id);
	printf("dimm_platform_config.oem_table_id: %s\n", p_value->oem_table_id);
	printf("dimm_platform_config.oem_revision: unsigned %d\n", p_value->oem_revision);
	printf("dimm_platform_config.creator_id: unsigned %d\n", p_value->creator_id);
	printf("dimm_platform_config.creator_revision: unsigned %d\n", p_value->creator_revision);
	printf("dimm_platform_config.current_config_size: unsigned %d\n", p_value->current_config_size);
	printf("dimm_platform_config.current_config_offset: unsigned %d\n", p_value->current_config_offset);
	printf("dimm_platform_config.config_input_size: unsigned %d\n", p_value->config_input_size);
	printf("dimm_platform_config.config_input_offset: unsigned %d\n", p_value->config_input_offset);
	printf("dimm_platform_config.config_output_size: unsigned %d\n", p_value->config_output_size);
	printf("dimm_platform_config.config_output_offset: unsigned %d\n", p_value->config_output_offset);
}
enum db_return_codes db_add_dimm_platform_config(const PersistentStore *p_ps,
	struct db_dimm_platform_config *p_dimm_platform_config)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_platform_config \
		(device_handle, signature, length, revision, checksum, oem_id, oem_table_id, oem_revision, creator_id, creator_revision, current_config_size, current_config_offset, config_input_size, config_input_offset, config_output_size, config_output_offset)  \
		VALUES 		\
		($device_handle, \
		$signature, \
		$length, \
		$revision, \
		$checksum, \
		$oem_id, \
		$oem_table_id, \
		$oem_revision, \
		$creator_id, \
		$creator_revision, \
		$current_config_size, \
		$current_config_offset, \
		$config_input_size, \
		$config_input_offset, \
		$config_output_size, \
		$config_output_offset) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_platform_config(p_stmt, p_dimm_platform_config);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_platform_config_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_platform_config", p_count);
}
int db_get_dimm_platform_configs(const PersistentStore *p_ps,
	struct db_dimm_platform_config *p_dimm_platform_config,
	int dimm_platform_config_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_platform_config, 0, sizeof (struct db_dimm_platform_config) * dimm_platform_config_count);
	char *sql = "SELECT \
		device_handle \
		,  signature \
		,  length \
		,  revision \
		,  checksum \
		,  oem_id \
		,  oem_table_id \
		,  oem_revision \
		,  creator_id \
		,  creator_revision \
		,  current_config_size \
		,  current_config_offset \
		,  config_input_size \
		,  config_input_offset \
		,  config_output_size \
		,  config_output_offset \
		  \
		FROM dimm_platform_config \
		                 \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_platform_config_count)
		{
			local_row_to_dimm_platform_config(p_ps, p_stmt, &p_dimm_platform_config[index]);
			local_get_dimm_platform_config_relationships(p_ps, p_stmt, &p_dimm_platform_config[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_platform_configs(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_platform_config");
}

enum db_return_codes db_save_dimm_platform_config_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_platform_config *p_dimm_platform_config)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_platform_config temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_platform_config_by_device_handle(p_ps, p_dimm_platform_config->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_platform_config_by_device_handle(p_ps,
				p_dimm_platform_config->device_handle,
				p_dimm_platform_config);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_platform_config \
			( device_handle ,  signature ,  length ,  revision ,  checksum ,  oem_id ,  oem_table_id ,  oem_revision ,  creator_id ,  creator_revision ,  current_config_size ,  current_config_offset ,  config_input_size ,  config_input_offset ,  config_output_size ,  config_output_offset )  \
			VALUES 		\
			($device_handle, \
			$signature, \
			$length, \
			$revision, \
			$checksum, \
			$oem_id, \
			$oem_table_id, \
			$oem_revision, \
			$creator_id, \
			$creator_revision, \
			$current_config_size, \
			$current_config_offset, \
			$config_input_size, \
			$config_input_offset, \
			$config_output_size, \
			$config_output_offset) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_platform_config(p_stmt, p_dimm_platform_config);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_platform_config_history \
			(history_id, \
				 device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  current_config_size,  current_config_offset,  config_input_size,  config_input_offset,  config_output_size,  config_output_offset)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $signature , \
				 $length , \
				 $revision , \
				 $checksum , \
				 $oem_id , \
				 $oem_table_id , \
				 $oem_revision , \
				 $creator_id , \
				 $creator_revision , \
				 $current_config_size , \
				 $current_config_offset , \
				 $config_input_size , \
				 $config_input_offset , \
				 $config_output_size , \
				 $config_output_offset )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_platform_config(p_stmt, p_dimm_platform_config);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_platform_config_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_platform_config *p_dimm_platform_config)
{
	memset(p_dimm_platform_config, 0, sizeof (struct db_dimm_platform_config));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  current_config_size,  current_config_offset,  config_input_size,  config_input_offset,  config_output_size,  config_output_offset  \
		FROM dimm_platform_config \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_platform_config(p_ps, p_stmt, p_dimm_platform_config);
			local_get_dimm_platform_config_relationships(p_ps, p_stmt, p_dimm_platform_config);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_platform_config_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_platform_config *p_dimm_platform_config)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_platform_config \
	SET \
	device_handle=$device_handle \
		,  signature=$signature \
		,  length=$length \
		,  revision=$revision \
		,  checksum=$checksum \
		,  oem_id=$oem_id \
		,  oem_table_id=$oem_table_id \
		,  oem_revision=$oem_revision \
		,  creator_id=$creator_id \
		,  creator_revision=$creator_revision \
		,  current_config_size=$current_config_size \
		,  current_config_offset=$current_config_offset \
		,  config_input_size=$config_input_size \
		,  config_input_offset=$config_input_offset \
		,  config_output_size=$config_output_size \
		,  config_output_offset=$config_output_offset \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_platform_config(p_stmt, p_dimm_platform_config);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_platform_config_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_platform_config \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_platform_config_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_platform_config_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_platform_config_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_platform_config_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_platform_config_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_platform_config *p_dimm_platform_config,
	int history_id,
	int dimm_platform_config_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_platform_config, 0, sizeof (struct db_dimm_platform_config) * dimm_platform_config_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  current_config_size,  current_config_offset,  config_input_size,  config_input_offset,  config_output_size,  config_output_offset  \
		FROM dimm_platform_config_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_platform_config_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_platform_config(p_ps, p_stmt, &p_dimm_platform_config[index]);
		local_get_dimm_platform_config_relationships_history(p_ps, p_stmt, &p_dimm_platform_config[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_platform_config_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_platform_config_history");
}

/*
 * --- END dimm_platform_config ----------------
 */
/*
 * --- dimm_current_config ----------------
 */
void local_bind_dimm_current_config(sqlite3_stmt *p_stmt, struct db_dimm_current_config *p_dimm_current_config)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_current_config->device_handle);
	BIND_TEXT(p_stmt, "$signature", (char *)p_dimm_current_config->signature);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_dimm_current_config->length);
	BIND_INTEGER(p_stmt, "$revision", (unsigned int)p_dimm_current_config->revision);
	BIND_INTEGER(p_stmt, "$checksum", (unsigned int)p_dimm_current_config->checksum);
	BIND_TEXT(p_stmt, "$oem_id", (char *)p_dimm_current_config->oem_id);
	BIND_TEXT(p_stmt, "$oem_table_id", (char *)p_dimm_current_config->oem_table_id);
	BIND_INTEGER(p_stmt, "$oem_revision", (unsigned int)p_dimm_current_config->oem_revision);
	BIND_INTEGER(p_stmt, "$creator_id", (unsigned int)p_dimm_current_config->creator_id);
	BIND_INTEGER(p_stmt, "$creator_revision", (unsigned int)p_dimm_current_config->creator_revision);
	BIND_INTEGER(p_stmt, "$config_status", (unsigned int)p_dimm_current_config->config_status);
	BIND_INTEGER(p_stmt, "$mapped_memory_capacity", (unsigned long long)p_dimm_current_config->mapped_memory_capacity);
	BIND_INTEGER(p_stmt, "$mapped_app_direct_capacity", (unsigned long long)p_dimm_current_config->mapped_app_direct_capacity);
}
void local_get_dimm_current_config_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_current_config *p_dimm_current_config)
{
}

void local_get_dimm_current_config_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_current_config *p_dimm_current_config,
	int history_id)
{
}

void local_row_to_dimm_current_config(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_current_config *p_dimm_current_config)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_current_config->device_handle);
	TEXT_COLUMN(p_stmt,
		1,
		p_dimm_current_config->signature,
		DIMM_CURRENT_CONFIG_SIGNATURE_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_current_config->length);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_current_config->revision);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_current_config->checksum);
	TEXT_COLUMN(p_stmt,
		5,
		p_dimm_current_config->oem_id,
		DIMM_CURRENT_CONFIG_OEM_ID_LEN);
	TEXT_COLUMN(p_stmt,
		6,
		p_dimm_current_config->oem_table_id,
		DIMM_CURRENT_CONFIG_OEM_TABLE_ID_LEN);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_current_config->oem_revision);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_current_config->creator_id);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_current_config->creator_revision);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_current_config->config_status);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_current_config->mapped_memory_capacity);
	INTEGER_COLUMN(p_stmt,
		12,
		p_dimm_current_config->mapped_app_direct_capacity);
}
void db_print_dimm_current_config(struct db_dimm_current_config *p_value)
{
	printf("dimm_current_config.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_current_config.signature: %s\n", p_value->signature);
	printf("dimm_current_config.length: unsigned %d\n", p_value->length);
	printf("dimm_current_config.revision: unsigned %d\n", p_value->revision);
	printf("dimm_current_config.checksum: unsigned %d\n", p_value->checksum);
	printf("dimm_current_config.oem_id: %s\n", p_value->oem_id);
	printf("dimm_current_config.oem_table_id: %s\n", p_value->oem_table_id);
	printf("dimm_current_config.oem_revision: unsigned %d\n", p_value->oem_revision);
	printf("dimm_current_config.creator_id: unsigned %d\n", p_value->creator_id);
	printf("dimm_current_config.creator_revision: unsigned %d\n", p_value->creator_revision);
	printf("dimm_current_config.config_status: unsigned %d\n", p_value->config_status);
	printf("dimm_current_config.mapped_memory_capacity: unsigned %lld\n", p_value->mapped_memory_capacity);
	printf("dimm_current_config.mapped_app_direct_capacity: unsigned %lld\n", p_value->mapped_app_direct_capacity);
}
enum db_return_codes db_add_dimm_current_config(const PersistentStore *p_ps,
	struct db_dimm_current_config *p_dimm_current_config)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_current_config \
		(device_handle, signature, length, revision, checksum, oem_id, oem_table_id, oem_revision, creator_id, creator_revision, config_status, mapped_memory_capacity, mapped_app_direct_capacity)  \
		VALUES 		\
		($device_handle, \
		$signature, \
		$length, \
		$revision, \
		$checksum, \
		$oem_id, \
		$oem_table_id, \
		$oem_revision, \
		$creator_id, \
		$creator_revision, \
		$config_status, \
		$mapped_memory_capacity, \
		$mapped_app_direct_capacity) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_current_config(p_stmt, p_dimm_current_config);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_current_config_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_current_config", p_count);
}
int db_get_dimm_current_configs(const PersistentStore *p_ps,
	struct db_dimm_current_config *p_dimm_current_config,
	int dimm_current_config_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_current_config, 0, sizeof (struct db_dimm_current_config) * dimm_current_config_count);
	char *sql = "SELECT \
		device_handle \
		,  signature \
		,  length \
		,  revision \
		,  checksum \
		,  oem_id \
		,  oem_table_id \
		,  oem_revision \
		,  creator_id \
		,  creator_revision \
		,  config_status \
		,  mapped_memory_capacity \
		,  mapped_app_direct_capacity \
		  \
		FROM dimm_current_config \
		              \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_current_config_count)
		{
			local_row_to_dimm_current_config(p_ps, p_stmt, &p_dimm_current_config[index]);
			local_get_dimm_current_config_relationships(p_ps, p_stmt, &p_dimm_current_config[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_current_configs(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_current_config");
}

enum db_return_codes db_save_dimm_current_config_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_current_config *p_dimm_current_config)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_current_config temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_current_config_by_device_handle(p_ps, p_dimm_current_config->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_current_config_by_device_handle(p_ps,
				p_dimm_current_config->device_handle,
				p_dimm_current_config);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_current_config \
			( device_handle ,  signature ,  length ,  revision ,  checksum ,  oem_id ,  oem_table_id ,  oem_revision ,  creator_id ,  creator_revision ,  config_status ,  mapped_memory_capacity ,  mapped_app_direct_capacity )  \
			VALUES 		\
			($device_handle, \
			$signature, \
			$length, \
			$revision, \
			$checksum, \
			$oem_id, \
			$oem_table_id, \
			$oem_revision, \
			$creator_id, \
			$creator_revision, \
			$config_status, \
			$mapped_memory_capacity, \
			$mapped_app_direct_capacity) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_current_config(p_stmt, p_dimm_current_config);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_current_config_history \
			(history_id, \
				 device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  config_status,  mapped_memory_capacity,  mapped_app_direct_capacity)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $signature , \
				 $length , \
				 $revision , \
				 $checksum , \
				 $oem_id , \
				 $oem_table_id , \
				 $oem_revision , \
				 $creator_id , \
				 $creator_revision , \
				 $config_status , \
				 $mapped_memory_capacity , \
				 $mapped_app_direct_capacity )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_current_config(p_stmt, p_dimm_current_config);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_current_config_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_current_config *p_dimm_current_config)
{
	memset(p_dimm_current_config, 0, sizeof (struct db_dimm_current_config));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  config_status,  mapped_memory_capacity,  mapped_app_direct_capacity  \
		FROM dimm_current_config \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_current_config(p_ps, p_stmt, p_dimm_current_config);
			local_get_dimm_current_config_relationships(p_ps, p_stmt, p_dimm_current_config);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_current_config_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_current_config *p_dimm_current_config)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_current_config \
	SET \
	device_handle=$device_handle \
		,  signature=$signature \
		,  length=$length \
		,  revision=$revision \
		,  checksum=$checksum \
		,  oem_id=$oem_id \
		,  oem_table_id=$oem_table_id \
		,  oem_revision=$oem_revision \
		,  creator_id=$creator_id \
		,  creator_revision=$creator_revision \
		,  config_status=$config_status \
		,  mapped_memory_capacity=$mapped_memory_capacity \
		,  mapped_app_direct_capacity=$mapped_app_direct_capacity \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_current_config(p_stmt, p_dimm_current_config);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_current_config_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_current_config \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_current_config_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_current_config_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_current_config_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_current_config_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_current_config_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_current_config *p_dimm_current_config,
	int history_id,
	int dimm_current_config_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_current_config, 0, sizeof (struct db_dimm_current_config) * dimm_current_config_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  config_status,  mapped_memory_capacity,  mapped_app_direct_capacity  \
		FROM dimm_current_config_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_current_config_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_current_config(p_ps, p_stmt, &p_dimm_current_config[index]);
		local_get_dimm_current_config_relationships_history(p_ps, p_stmt, &p_dimm_current_config[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_current_config_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_current_config_history");
}

/*
 * --- END dimm_current_config ----------------
 */
/*
 * --- dimm_config_input ----------------
 */
void local_bind_dimm_config_input(sqlite3_stmt *p_stmt, struct db_dimm_config_input *p_dimm_config_input)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_config_input->device_handle);
	BIND_TEXT(p_stmt, "$signature", (char *)p_dimm_config_input->signature);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_dimm_config_input->length);
	BIND_INTEGER(p_stmt, "$revision", (unsigned int)p_dimm_config_input->revision);
	BIND_INTEGER(p_stmt, "$checksum", (unsigned int)p_dimm_config_input->checksum);
	BIND_TEXT(p_stmt, "$oem_id", (char *)p_dimm_config_input->oem_id);
	BIND_TEXT(p_stmt, "$oem_table_id", (char *)p_dimm_config_input->oem_table_id);
	BIND_INTEGER(p_stmt, "$oem_revision", (unsigned int)p_dimm_config_input->oem_revision);
	BIND_INTEGER(p_stmt, "$creator_id", (unsigned int)p_dimm_config_input->creator_id);
	BIND_INTEGER(p_stmt, "$creator_revision", (unsigned int)p_dimm_config_input->creator_revision);
	BIND_INTEGER(p_stmt, "$sequence_number", (unsigned int)p_dimm_config_input->sequence_number);
}
void local_get_dimm_config_input_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_config_input *p_dimm_config_input)
{
}

void local_get_dimm_config_input_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_config_input *p_dimm_config_input,
	int history_id)
{
}

void local_row_to_dimm_config_input(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_config_input *p_dimm_config_input)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_config_input->device_handle);
	TEXT_COLUMN(p_stmt,
		1,
		p_dimm_config_input->signature,
		DIMM_CONFIG_INPUT_SIGNATURE_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_config_input->length);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_config_input->revision);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_config_input->checksum);
	TEXT_COLUMN(p_stmt,
		5,
		p_dimm_config_input->oem_id,
		DIMM_CONFIG_INPUT_OEM_ID_LEN);
	TEXT_COLUMN(p_stmt,
		6,
		p_dimm_config_input->oem_table_id,
		DIMM_CONFIG_INPUT_OEM_TABLE_ID_LEN);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_config_input->oem_revision);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_config_input->creator_id);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_config_input->creator_revision);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_config_input->sequence_number);
}
void db_print_dimm_config_input(struct db_dimm_config_input *p_value)
{
	printf("dimm_config_input.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_config_input.signature: %s\n", p_value->signature);
	printf("dimm_config_input.length: unsigned %d\n", p_value->length);
	printf("dimm_config_input.revision: unsigned %d\n", p_value->revision);
	printf("dimm_config_input.checksum: unsigned %d\n", p_value->checksum);
	printf("dimm_config_input.oem_id: %s\n", p_value->oem_id);
	printf("dimm_config_input.oem_table_id: %s\n", p_value->oem_table_id);
	printf("dimm_config_input.oem_revision: unsigned %d\n", p_value->oem_revision);
	printf("dimm_config_input.creator_id: unsigned %d\n", p_value->creator_id);
	printf("dimm_config_input.creator_revision: unsigned %d\n", p_value->creator_revision);
	printf("dimm_config_input.sequence_number: unsigned %d\n", p_value->sequence_number);
}
enum db_return_codes db_add_dimm_config_input(const PersistentStore *p_ps,
	struct db_dimm_config_input *p_dimm_config_input)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_config_input \
		(device_handle, signature, length, revision, checksum, oem_id, oem_table_id, oem_revision, creator_id, creator_revision, sequence_number)  \
		VALUES 		\
		($device_handle, \
		$signature, \
		$length, \
		$revision, \
		$checksum, \
		$oem_id, \
		$oem_table_id, \
		$oem_revision, \
		$creator_id, \
		$creator_revision, \
		$sequence_number) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_config_input(p_stmt, p_dimm_config_input);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_config_input_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_config_input", p_count);
}
int db_get_dimm_config_inputs(const PersistentStore *p_ps,
	struct db_dimm_config_input *p_dimm_config_input,
	int dimm_config_input_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_config_input, 0, sizeof (struct db_dimm_config_input) * dimm_config_input_count);
	char *sql = "SELECT \
		device_handle \
		,  signature \
		,  length \
		,  revision \
		,  checksum \
		,  oem_id \
		,  oem_table_id \
		,  oem_revision \
		,  creator_id \
		,  creator_revision \
		,  sequence_number \
		  \
		FROM dimm_config_input \
		            \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_config_input_count)
		{
			local_row_to_dimm_config_input(p_ps, p_stmt, &p_dimm_config_input[index]);
			local_get_dimm_config_input_relationships(p_ps, p_stmt, &p_dimm_config_input[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_config_inputs(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_input");
}

enum db_return_codes db_save_dimm_config_input_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_config_input *p_dimm_config_input)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_config_input temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_config_input_by_device_handle(p_ps, p_dimm_config_input->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_config_input_by_device_handle(p_ps,
				p_dimm_config_input->device_handle,
				p_dimm_config_input);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_config_input \
			( device_handle ,  signature ,  length ,  revision ,  checksum ,  oem_id ,  oem_table_id ,  oem_revision ,  creator_id ,  creator_revision ,  sequence_number )  \
			VALUES 		\
			($device_handle, \
			$signature, \
			$length, \
			$revision, \
			$checksum, \
			$oem_id, \
			$oem_table_id, \
			$oem_revision, \
			$creator_id, \
			$creator_revision, \
			$sequence_number) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_config_input(p_stmt, p_dimm_config_input);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_config_input_history \
			(history_id, \
				 device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  sequence_number)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $signature , \
				 $length , \
				 $revision , \
				 $checksum , \
				 $oem_id , \
				 $oem_table_id , \
				 $oem_revision , \
				 $creator_id , \
				 $creator_revision , \
				 $sequence_number )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_config_input(p_stmt, p_dimm_config_input);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_config_input_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_config_input *p_dimm_config_input)
{
	memset(p_dimm_config_input, 0, sizeof (struct db_dimm_config_input));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  sequence_number  \
		FROM dimm_config_input \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_config_input(p_ps, p_stmt, p_dimm_config_input);
			local_get_dimm_config_input_relationships(p_ps, p_stmt, p_dimm_config_input);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_config_input_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_config_input *p_dimm_config_input)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_config_input \
	SET \
	device_handle=$device_handle \
		,  signature=$signature \
		,  length=$length \
		,  revision=$revision \
		,  checksum=$checksum \
		,  oem_id=$oem_id \
		,  oem_table_id=$oem_table_id \
		,  oem_revision=$oem_revision \
		,  creator_id=$creator_id \
		,  creator_revision=$creator_revision \
		,  sequence_number=$sequence_number \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_config_input(p_stmt, p_dimm_config_input);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_config_input_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_config_input \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_config_input_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_config_input_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_config_input_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_config_input_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_config_input_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_config_input *p_dimm_config_input,
	int history_id,
	int dimm_config_input_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_config_input, 0, sizeof (struct db_dimm_config_input) * dimm_config_input_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  sequence_number  \
		FROM dimm_config_input_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_config_input_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_config_input(p_ps, p_stmt, &p_dimm_config_input[index]);
		local_get_dimm_config_input_relationships_history(p_ps, p_stmt, &p_dimm_config_input[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_config_input_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_input_history");
}

/*
 * --- END dimm_config_input ----------------
 */
/*
 * --- dimm_config_output ----------------
 */
void local_bind_dimm_config_output(sqlite3_stmt *p_stmt, struct db_dimm_config_output *p_dimm_config_output)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_config_output->device_handle);
	BIND_TEXT(p_stmt, "$signature", (char *)p_dimm_config_output->signature);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_dimm_config_output->length);
	BIND_INTEGER(p_stmt, "$revision", (unsigned int)p_dimm_config_output->revision);
	BIND_INTEGER(p_stmt, "$checksum", (unsigned int)p_dimm_config_output->checksum);
	BIND_TEXT(p_stmt, "$oem_id", (char *)p_dimm_config_output->oem_id);
	BIND_TEXT(p_stmt, "$oem_table_id", (char *)p_dimm_config_output->oem_table_id);
	BIND_INTEGER(p_stmt, "$oem_revision", (unsigned int)p_dimm_config_output->oem_revision);
	BIND_INTEGER(p_stmt, "$creator_id", (unsigned int)p_dimm_config_output->creator_id);
	BIND_INTEGER(p_stmt, "$creator_revision", (unsigned int)p_dimm_config_output->creator_revision);
	BIND_INTEGER(p_stmt, "$sequence_number", (unsigned int)p_dimm_config_output->sequence_number);
	BIND_INTEGER(p_stmt, "$validation_status", (unsigned int)p_dimm_config_output->validation_status);
}
void local_get_dimm_config_output_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_config_output *p_dimm_config_output)
{
}

void local_get_dimm_config_output_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_config_output *p_dimm_config_output,
	int history_id)
{
}

void local_row_to_dimm_config_output(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_config_output *p_dimm_config_output)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_config_output->device_handle);
	TEXT_COLUMN(p_stmt,
		1,
		p_dimm_config_output->signature,
		DIMM_CONFIG_OUTPUT_SIGNATURE_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_config_output->length);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_config_output->revision);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_config_output->checksum);
	TEXT_COLUMN(p_stmt,
		5,
		p_dimm_config_output->oem_id,
		DIMM_CONFIG_OUTPUT_OEM_ID_LEN);
	TEXT_COLUMN(p_stmt,
		6,
		p_dimm_config_output->oem_table_id,
		DIMM_CONFIG_OUTPUT_OEM_TABLE_ID_LEN);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_config_output->oem_revision);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_config_output->creator_id);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_config_output->creator_revision);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_config_output->sequence_number);
	INTEGER_COLUMN(p_stmt,
		11,
		p_dimm_config_output->validation_status);
}
void db_print_dimm_config_output(struct db_dimm_config_output *p_value)
{
	printf("dimm_config_output.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_config_output.signature: %s\n", p_value->signature);
	printf("dimm_config_output.length: unsigned %d\n", p_value->length);
	printf("dimm_config_output.revision: unsigned %d\n", p_value->revision);
	printf("dimm_config_output.checksum: unsigned %d\n", p_value->checksum);
	printf("dimm_config_output.oem_id: %s\n", p_value->oem_id);
	printf("dimm_config_output.oem_table_id: %s\n", p_value->oem_table_id);
	printf("dimm_config_output.oem_revision: unsigned %d\n", p_value->oem_revision);
	printf("dimm_config_output.creator_id: unsigned %d\n", p_value->creator_id);
	printf("dimm_config_output.creator_revision: unsigned %d\n", p_value->creator_revision);
	printf("dimm_config_output.sequence_number: unsigned %d\n", p_value->sequence_number);
	printf("dimm_config_output.validation_status: unsigned %d\n", p_value->validation_status);
}
enum db_return_codes db_add_dimm_config_output(const PersistentStore *p_ps,
	struct db_dimm_config_output *p_dimm_config_output)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_config_output \
		(device_handle, signature, length, revision, checksum, oem_id, oem_table_id, oem_revision, creator_id, creator_revision, sequence_number, validation_status)  \
		VALUES 		\
		($device_handle, \
		$signature, \
		$length, \
		$revision, \
		$checksum, \
		$oem_id, \
		$oem_table_id, \
		$oem_revision, \
		$creator_id, \
		$creator_revision, \
		$sequence_number, \
		$validation_status) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_config_output(p_stmt, p_dimm_config_output);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_config_output_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_config_output", p_count);
}
int db_get_dimm_config_outputs(const PersistentStore *p_ps,
	struct db_dimm_config_output *p_dimm_config_output,
	int dimm_config_output_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_config_output, 0, sizeof (struct db_dimm_config_output) * dimm_config_output_count);
	char *sql = "SELECT \
		device_handle \
		,  signature \
		,  length \
		,  revision \
		,  checksum \
		,  oem_id \
		,  oem_table_id \
		,  oem_revision \
		,  creator_id \
		,  creator_revision \
		,  sequence_number \
		,  validation_status \
		  \
		FROM dimm_config_output \
		             \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_config_output_count)
		{
			local_row_to_dimm_config_output(p_ps, p_stmt, &p_dimm_config_output[index]);
			local_get_dimm_config_output_relationships(p_ps, p_stmt, &p_dimm_config_output[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_config_outputs(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_output");
}

enum db_return_codes db_save_dimm_config_output_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_config_output *p_dimm_config_output)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_config_output temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_config_output_by_device_handle(p_ps, p_dimm_config_output->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_config_output_by_device_handle(p_ps,
				p_dimm_config_output->device_handle,
				p_dimm_config_output);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_config_output \
			( device_handle ,  signature ,  length ,  revision ,  checksum ,  oem_id ,  oem_table_id ,  oem_revision ,  creator_id ,  creator_revision ,  sequence_number ,  validation_status )  \
			VALUES 		\
			($device_handle, \
			$signature, \
			$length, \
			$revision, \
			$checksum, \
			$oem_id, \
			$oem_table_id, \
			$oem_revision, \
			$creator_id, \
			$creator_revision, \
			$sequence_number, \
			$validation_status) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_config_output(p_stmt, p_dimm_config_output);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_config_output_history \
			(history_id, \
				 device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  sequence_number,  validation_status)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $signature , \
				 $length , \
				 $revision , \
				 $checksum , \
				 $oem_id , \
				 $oem_table_id , \
				 $oem_revision , \
				 $creator_id , \
				 $creator_revision , \
				 $sequence_number , \
				 $validation_status )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_config_output(p_stmt, p_dimm_config_output);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_config_output_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_config_output *p_dimm_config_output)
{
	memset(p_dimm_config_output, 0, sizeof (struct db_dimm_config_output));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  sequence_number,  validation_status  \
		FROM dimm_config_output \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_config_output(p_ps, p_stmt, p_dimm_config_output);
			local_get_dimm_config_output_relationships(p_ps, p_stmt, p_dimm_config_output);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_config_output_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_dimm_config_output *p_dimm_config_output)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_config_output \
	SET \
	device_handle=$device_handle \
		,  signature=$signature \
		,  length=$length \
		,  revision=$revision \
		,  checksum=$checksum \
		,  oem_id=$oem_id \
		,  oem_table_id=$oem_table_id \
		,  oem_revision=$oem_revision \
		,  creator_id=$creator_id \
		,  creator_revision=$creator_revision \
		,  sequence_number=$sequence_number \
		,  validation_status=$validation_status \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_dimm_config_output(p_stmt, p_dimm_config_output);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_config_output_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_config_output \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_config_output_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_config_output_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_config_output_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_config_output_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_config_output_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_config_output *p_dimm_config_output,
	int history_id,
	int dimm_config_output_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_config_output, 0, sizeof (struct db_dimm_config_output) * dimm_config_output_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  signature,  length,  revision,  checksum,  oem_id,  oem_table_id,  oem_revision,  creator_id,  creator_revision,  sequence_number,  validation_status  \
		FROM dimm_config_output_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_config_output_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_config_output(p_ps, p_stmt, &p_dimm_config_output[index]);
		local_get_dimm_config_output_relationships_history(p_ps, p_stmt, &p_dimm_config_output[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_config_output_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_output_history");
}

/*
 * --- END dimm_config_output ----------------
 */
/*
 * --- dimm_partition_change ----------------
 */
void local_bind_dimm_partition_change(sqlite3_stmt *p_stmt, struct db_dimm_partition_change *p_dimm_partition_change)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_partition_change->device_handle);
	BIND_INTEGER(p_stmt, "$id", (int)p_dimm_partition_change->id);
	BIND_INTEGER(p_stmt, "$config_table_type", (unsigned int)p_dimm_partition_change->config_table_type);
	BIND_INTEGER(p_stmt, "$extension_table_type", (unsigned int)p_dimm_partition_change->extension_table_type);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_dimm_partition_change->length);
	BIND_INTEGER(p_stmt, "$partition_size", (unsigned long long)p_dimm_partition_change->partition_size);
	BIND_INTEGER(p_stmt, "$status", (unsigned int)p_dimm_partition_change->status);
}
void local_get_dimm_partition_change_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_partition_change *p_dimm_partition_change)
{
}

void local_get_dimm_partition_change_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_partition_change *p_dimm_partition_change,
	int history_id)
{
}

void local_row_to_dimm_partition_change(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_partition_change *p_dimm_partition_change)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_partition_change->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_partition_change->id);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_partition_change->config_table_type);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_partition_change->extension_table_type);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_partition_change->length);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_partition_change->partition_size);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_partition_change->status);
}
void db_print_dimm_partition_change(struct db_dimm_partition_change *p_value)
{
	printf("dimm_partition_change.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_partition_change.id: %d\n", p_value->id);
	printf("dimm_partition_change.config_table_type: unsigned %d\n", p_value->config_table_type);
	printf("dimm_partition_change.extension_table_type: unsigned %d\n", p_value->extension_table_type);
	printf("dimm_partition_change.length: unsigned %d\n", p_value->length);
	printf("dimm_partition_change.partition_size: unsigned %lld\n", p_value->partition_size);
	printf("dimm_partition_change.status: unsigned %d\n", p_value->status);
}
enum db_return_codes db_add_dimm_partition_change(const PersistentStore *p_ps,
	struct db_dimm_partition_change *p_dimm_partition_change)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_partition_change \
		(device_handle, id, config_table_type, extension_table_type, length, partition_size, status)  \
		VALUES 		\
		($device_handle, \
		$id, \
		$config_table_type, \
		$extension_table_type, \
		$length, \
		$partition_size, \
		$status) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_partition_change(p_stmt, p_dimm_partition_change);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_partition_change_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_partition_change", p_count);
}
int db_get_dimm_partition_changes(const PersistentStore *p_ps,
	struct db_dimm_partition_change *p_dimm_partition_change,
	int dimm_partition_change_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_partition_change, 0, sizeof (struct db_dimm_partition_change) * dimm_partition_change_count);
	char *sql = "SELECT \
		device_handle \
		,  id \
		,  config_table_type \
		,  extension_table_type \
		,  length \
		,  partition_size \
		,  status \
		  \
		FROM dimm_partition_change \
		        \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_partition_change_count)
		{
			local_row_to_dimm_partition_change(p_ps, p_stmt, &p_dimm_partition_change[index]);
			local_get_dimm_partition_change_relationships(p_ps, p_stmt, &p_dimm_partition_change[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_partition_changes(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_change");
}

enum db_return_codes db_save_dimm_partition_change_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_partition_change *p_dimm_partition_change)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_partition_change temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_partition_change_by_id(p_ps, p_dimm_partition_change->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_partition_change_by_id(p_ps,
				p_dimm_partition_change->id,
				p_dimm_partition_change);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_partition_change \
			( device_handle ,  id ,  config_table_type ,  extension_table_type ,  length ,  partition_size ,  status )  \
			VALUES 		\
			($device_handle, \
			$id, \
			$config_table_type, \
			$extension_table_type, \
			$length, \
			$partition_size, \
			$status) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_partition_change(p_stmt, p_dimm_partition_change);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_partition_change_history \
			(history_id, \
				 device_handle,  id,  config_table_type,  extension_table_type,  length,  partition_size,  status)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $id , \
				 $config_table_type , \
				 $extension_table_type , \
				 $length , \
				 $partition_size , \
				 $status )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_partition_change(p_stmt, p_dimm_partition_change);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_partition_change_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_dimm_partition_change *p_dimm_partition_change)
{
	memset(p_dimm_partition_change, 0, sizeof (struct db_dimm_partition_change));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  id,  config_table_type,  extension_table_type,  length,  partition_size,  status  \
		FROM dimm_partition_change \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_partition_change(p_ps, p_stmt, p_dimm_partition_change);
			local_get_dimm_partition_change_relationships(p_ps, p_stmt, p_dimm_partition_change);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_partition_change_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_dimm_partition_change *p_dimm_partition_change)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_partition_change \
	SET \
	device_handle=$device_handle \
		,  id=$id \
		,  config_table_type=$config_table_type \
		,  extension_table_type=$extension_table_type \
		,  length=$length \
		,  partition_size=$partition_size \
		,  status=$status \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_dimm_partition_change(p_stmt, p_dimm_partition_change);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_partition_change_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_partition_change \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_partition_change_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_partition_change_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_partition_change_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_partition_change_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_partition_change_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_partition_change *p_dimm_partition_change,
	int history_id,
	int dimm_partition_change_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_partition_change, 0, sizeof (struct db_dimm_partition_change) * dimm_partition_change_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  id,  config_table_type,  extension_table_type,  length,  partition_size,  status  \
		FROM dimm_partition_change_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_partition_change_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_partition_change(p_ps, p_stmt, &p_dimm_partition_change[index]);
		local_get_dimm_partition_change_relationships_history(p_ps, p_stmt, &p_dimm_partition_change[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_partition_change_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_change_history");
}

enum db_return_codes db_get_dimm_partition_change_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM dimm_partition_change WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_partition_change_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM dimm_partition_change_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_partition_changes_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_dimm_partition_change *p_dimm_partition_change,
	int dimm_partition_change_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  id ,  config_table_type ,  extension_table_type ,  length ,  partition_size ,  status  \
		FROM dimm_partition_change \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_partition_change_count)
		{
			local_row_to_dimm_partition_change(p_ps, p_stmt, &p_dimm_partition_change[index]);
			local_get_dimm_partition_change_relationships(p_ps, p_stmt, &p_dimm_partition_change[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_partition_changes_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_dimm_partition_change *p_dimm_partition_change,
	int dimm_partition_change_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 device_handle ,  id ,  config_table_type ,  extension_table_type ,  length ,  partition_size ,  status  \
		FROM dimm_partition_change_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_partition_change_count)
		{
			local_row_to_dimm_partition_change(p_ps, p_stmt, &p_dimm_partition_change[index]);
			local_get_dimm_partition_change_relationships(p_ps, p_stmt, &p_dimm_partition_change[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_dimm_partition_change_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_partition_change \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Roll dimm_partition_changes by id to specified max.
 */
enum db_return_codes db_roll_dimm_partition_changes_by_id(const PersistentStore *p_ps,
	int max_rows)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char sql[1024];
	snprintf(sql, 1024,
				"DELETE FROM dimm_partition_change "
				"WHERE id NOT IN ("
				"SELECT id "
				"FROM dimm_partition_change "
				"ORDER BY id DESC "
				"LIMIT %d)", max_rows); 
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Get the max id in the dimm_partition_change table.
 */
enum db_return_codes db_get_next_dimm_partition_change_id(const PersistentStore *p_ps, int *p_max)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	if (p_max)
	{
		if ((rc = run_scalar_sql(p_ps, 
			"SELECT MAX(id) FROM dimm_partition_change", p_max)) 
			==  DB_SUCCESS)
		{
			(*p_max)++;
		}
	}
	return rc;
}
/*
 * --- END dimm_partition_change ----------------
 */
/*
 * --- dimm_interleave_set ----------------
 */
void local_bind_dimm_interleave_set(sqlite3_stmt *p_stmt, struct db_dimm_interleave_set *p_dimm_interleave_set)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_dimm_interleave_set->id);
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_dimm_interleave_set->device_handle);
	BIND_INTEGER(p_stmt, "$config_table_type", (unsigned int)p_dimm_interleave_set->config_table_type);
	BIND_INTEGER(p_stmt, "$extension_table_type", (unsigned int)p_dimm_interleave_set->extension_table_type);
	BIND_INTEGER(p_stmt, "$length", (unsigned int)p_dimm_interleave_set->length);
	BIND_INTEGER(p_stmt, "$index_id", (unsigned int)p_dimm_interleave_set->index_id);
	BIND_INTEGER(p_stmt, "$dimm_count", (unsigned int)p_dimm_interleave_set->dimm_count);
	BIND_INTEGER(p_stmt, "$memory_type", (unsigned int)p_dimm_interleave_set->memory_type);
	BIND_INTEGER(p_stmt, "$interleave_format", (unsigned int)p_dimm_interleave_set->interleave_format);
	BIND_INTEGER(p_stmt, "$mirror_enable", (unsigned int)p_dimm_interleave_set->mirror_enable);
	BIND_INTEGER(p_stmt, "$status", (unsigned int)p_dimm_interleave_set->status);
}
void local_get_dimm_interleave_set_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_interleave_set *p_dimm_interleave_set)
{
}

void local_get_dimm_interleave_set_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_interleave_set *p_dimm_interleave_set,
	int history_id)
{
}

void local_row_to_dimm_interleave_set(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_dimm_interleave_set *p_dimm_interleave_set)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_dimm_interleave_set->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_dimm_interleave_set->device_handle);
	INTEGER_COLUMN(p_stmt,
		2,
		p_dimm_interleave_set->config_table_type);
	INTEGER_COLUMN(p_stmt,
		3,
		p_dimm_interleave_set->extension_table_type);
	INTEGER_COLUMN(p_stmt,
		4,
		p_dimm_interleave_set->length);
	INTEGER_COLUMN(p_stmt,
		5,
		p_dimm_interleave_set->index_id);
	INTEGER_COLUMN(p_stmt,
		6,
		p_dimm_interleave_set->dimm_count);
	INTEGER_COLUMN(p_stmt,
		7,
		p_dimm_interleave_set->memory_type);
	INTEGER_COLUMN(p_stmt,
		8,
		p_dimm_interleave_set->interleave_format);
	INTEGER_COLUMN(p_stmt,
		9,
		p_dimm_interleave_set->mirror_enable);
	INTEGER_COLUMN(p_stmt,
		10,
		p_dimm_interleave_set->status);
}
void db_print_dimm_interleave_set(struct db_dimm_interleave_set *p_value)
{
	printf("dimm_interleave_set.id: %d\n", p_value->id);
	printf("dimm_interleave_set.device_handle: unsigned %d\n", p_value->device_handle);
	printf("dimm_interleave_set.config_table_type: unsigned %d\n", p_value->config_table_type);
	printf("dimm_interleave_set.extension_table_type: unsigned %d\n", p_value->extension_table_type);
	printf("dimm_interleave_set.length: unsigned %d\n", p_value->length);
	printf("dimm_interleave_set.index_id: unsigned %d\n", p_value->index_id);
	printf("dimm_interleave_set.dimm_count: unsigned %d\n", p_value->dimm_count);
	printf("dimm_interleave_set.memory_type: unsigned %d\n", p_value->memory_type);
	printf("dimm_interleave_set.interleave_format: unsigned %d\n", p_value->interleave_format);
	printf("dimm_interleave_set.mirror_enable: unsigned %d\n", p_value->mirror_enable);
	printf("dimm_interleave_set.status: unsigned %d\n", p_value->status);
}
enum db_return_codes db_add_dimm_interleave_set(const PersistentStore *p_ps,
	struct db_dimm_interleave_set *p_dimm_interleave_set)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO dimm_interleave_set \
		(id, device_handle, config_table_type, extension_table_type, length, index_id, dimm_count, memory_type, interleave_format, mirror_enable, status)  \
		VALUES 		\
		($id, \
		$device_handle, \
		$config_table_type, \
		$extension_table_type, \
		$length, \
		$index_id, \
		$dimm_count, \
		$memory_type, \
		$interleave_format, \
		$mirror_enable, \
		$status) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_dimm_interleave_set(p_stmt, p_dimm_interleave_set);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_interleave_set_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "dimm_interleave_set", p_count);
}
int db_get_dimm_interleave_sets(const PersistentStore *p_ps,
	struct db_dimm_interleave_set *p_dimm_interleave_set,
	int dimm_interleave_set_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_interleave_set, 0, sizeof (struct db_dimm_interleave_set) * dimm_interleave_set_count);
	char *sql = "SELECT \
		id \
		,  device_handle \
		,  config_table_type \
		,  extension_table_type \
		,  length \
		,  index_id \
		,  dimm_count \
		,  memory_type \
		,  interleave_format \
		,  mirror_enable \
		,  status \
		  \
		FROM dimm_interleave_set \
		            \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_interleave_set_count)
		{
			local_row_to_dimm_interleave_set(p_ps, p_stmt, &p_dimm_interleave_set[index]);
			local_get_dimm_interleave_set_relationships(p_ps, p_stmt, &p_dimm_interleave_set[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_dimm_interleave_sets(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_interleave_set");
}

enum db_return_codes db_save_dimm_interleave_set_state(const PersistentStore *p_ps,
	int history_id,
	struct db_dimm_interleave_set *p_dimm_interleave_set)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_dimm_interleave_set temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_dimm_interleave_set_by_id(p_ps, p_dimm_interleave_set->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_dimm_interleave_set_by_id(p_ps,
				p_dimm_interleave_set->id,
				p_dimm_interleave_set);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO dimm_interleave_set \
			( id ,  device_handle ,  config_table_type ,  extension_table_type ,  length ,  index_id ,  dimm_count ,  memory_type ,  interleave_format ,  mirror_enable ,  status )  \
			VALUES 		\
			($id, \
			$device_handle, \
			$config_table_type, \
			$extension_table_type, \
			$length, \
			$index_id, \
			$dimm_count, \
			$memory_type, \
			$interleave_format, \
			$mirror_enable, \
			$status) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_dimm_interleave_set(p_stmt, p_dimm_interleave_set);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO dimm_interleave_set_history \
			(history_id, \
				 id,  device_handle,  config_table_type,  extension_table_type,  length,  index_id,  dimm_count,  memory_type,  interleave_format,  mirror_enable,  status)  \
			VALUES 		($history_id, \
				 $id , \
				 $device_handle , \
				 $config_table_type , \
				 $extension_table_type , \
				 $length , \
				 $index_id , \
				 $dimm_count , \
				 $memory_type , \
				 $interleave_format , \
				 $mirror_enable , \
				 $status )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_dimm_interleave_set(p_stmt, p_dimm_interleave_set);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_dimm_interleave_set_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_dimm_interleave_set *p_dimm_interleave_set)
{
	memset(p_dimm_interleave_set, 0, sizeof (struct db_dimm_interleave_set));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  device_handle,  config_table_type,  extension_table_type,  length,  index_id,  dimm_count,  memory_type,  interleave_format,  mirror_enable,  status  \
		FROM dimm_interleave_set \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_dimm_interleave_set(p_ps, p_stmt, p_dimm_interleave_set);
			local_get_dimm_interleave_set_relationships(p_ps, p_stmt, p_dimm_interleave_set);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_dimm_interleave_set_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_dimm_interleave_set *p_dimm_interleave_set)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE dimm_interleave_set \
	SET \
	id=$id \
		,  device_handle=$device_handle \
		,  config_table_type=$config_table_type \
		,  extension_table_type=$extension_table_type \
		,  length=$length \
		,  index_id=$index_id \
		,  dimm_count=$dimm_count \
		,  memory_type=$memory_type \
		,  interleave_format=$interleave_format \
		,  mirror_enable=$mirror_enable \
		,  status=$status \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_dimm_interleave_set(p_stmt, p_dimm_interleave_set);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_interleave_set_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_interleave_set \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_dimm_interleave_set_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_interleave_set_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_interleave_set_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM dimm_interleave_set_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_dimm_interleave_set_history_by_history_id(const PersistentStore *p_ps,
	struct db_dimm_interleave_set *p_dimm_interleave_set,
	int history_id,
	int dimm_interleave_set_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_dimm_interleave_set, 0, sizeof (struct db_dimm_interleave_set) * dimm_interleave_set_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  device_handle,  config_table_type,  extension_table_type,  length,  index_id,  dimm_count,  memory_type,  interleave_format,  mirror_enable,  status  \
		FROM dimm_interleave_set_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_interleave_set_count)
		{
		rc = DB_SUCCESS;
		local_row_to_dimm_interleave_set(p_ps, p_stmt, &p_dimm_interleave_set[index]);
		local_get_dimm_interleave_set_relationships_history(p_ps, p_stmt, &p_dimm_interleave_set[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_dimm_interleave_set_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM dimm_interleave_set_history");
}

/*!
 * Roll dimm_interleave_sets by id to specified max.
 */
enum db_return_codes db_roll_dimm_interleave_sets_by_id(const PersistentStore *p_ps,
	int max_rows)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char sql[1024];
	snprintf(sql, 1024,
				"DELETE FROM dimm_interleave_set "
				"WHERE id NOT IN ("
				"SELECT id "
				"FROM dimm_interleave_set "
				"ORDER BY id DESC "
				"LIMIT %d)", max_rows); 
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Get the max id in the dimm_interleave_set table.
 */
enum db_return_codes db_get_next_dimm_interleave_set_id(const PersistentStore *p_ps, int *p_max)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	if (p_max)
	{
		if ((rc = run_scalar_sql(p_ps, 
			"SELECT MAX(id) FROM dimm_interleave_set", p_max)) 
			==  DB_SUCCESS)
		{
			(*p_max)++;
		}
	}
	return rc;
}
enum db_return_codes db_get_dimm_interleave_set_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM dimm_interleave_set WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_interleave_set_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM dimm_interleave_set_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_interleave_sets_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_dimm_interleave_set *p_dimm_interleave_set,
	int dimm_interleave_set_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  device_handle ,  config_table_type ,  extension_table_type ,  length ,  index_id ,  dimm_count ,  memory_type ,  interleave_format ,  mirror_enable ,  status  \
		FROM dimm_interleave_set \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_interleave_set_count)
		{
			local_row_to_dimm_interleave_set(p_ps, p_stmt, &p_dimm_interleave_set[index]);
			local_get_dimm_interleave_set_relationships(p_ps, p_stmt, &p_dimm_interleave_set[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_dimm_interleave_sets_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_dimm_interleave_set *p_dimm_interleave_set,
	int dimm_interleave_set_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  device_handle ,  config_table_type ,  extension_table_type ,  length ,  index_id ,  dimm_count ,  memory_type ,  interleave_format ,  mirror_enable ,  status  \
		FROM dimm_interleave_set_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < dimm_interleave_set_count)
		{
			local_row_to_dimm_interleave_set(p_ps, p_stmt, &p_dimm_interleave_set[index]);
			local_get_dimm_interleave_set_relationships(p_ps, p_stmt, &p_dimm_interleave_set[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_dimm_interleave_set_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM dimm_interleave_set \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END dimm_interleave_set ----------------
 */
/*
 * --- interleave_set_dimm_info ----------------
 */
void local_bind_interleave_set_dimm_info(sqlite3_stmt *p_stmt, struct db_interleave_set_dimm_info *p_interleave_set_dimm_info)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_interleave_set_dimm_info->id);
	BIND_INTEGER(p_stmt, "$config_table_type", (unsigned int)p_interleave_set_dimm_info->config_table_type);
	BIND_INTEGER(p_stmt, "$index_id", (unsigned int)p_interleave_set_dimm_info->index_id);
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_interleave_set_dimm_info->device_handle);
	BIND_INTEGER(p_stmt, "$manufacturer", (unsigned int)p_interleave_set_dimm_info->manufacturer);
	BIND_INTEGER(p_stmt, "$serial_num", (unsigned int)p_interleave_set_dimm_info->serial_num);
	BIND_TEXT(p_stmt, "$model_num", (char *)p_interleave_set_dimm_info->model_num);
	BIND_INTEGER(p_stmt, "$offset", (unsigned long long)p_interleave_set_dimm_info->offset);
	BIND_INTEGER(p_stmt, "$size", (unsigned long long)p_interleave_set_dimm_info->size);
}
void local_get_interleave_set_dimm_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_interleave_set_dimm_info *p_interleave_set_dimm_info)
{
}

void local_get_interleave_set_dimm_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_interleave_set_dimm_info *p_interleave_set_dimm_info,
	int history_id)
{
}

void local_row_to_interleave_set_dimm_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_interleave_set_dimm_info *p_interleave_set_dimm_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_interleave_set_dimm_info->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_interleave_set_dimm_info->config_table_type);
	INTEGER_COLUMN(p_stmt,
		2,
		p_interleave_set_dimm_info->index_id);
	INTEGER_COLUMN(p_stmt,
		3,
		p_interleave_set_dimm_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		4,
		p_interleave_set_dimm_info->manufacturer);
	INTEGER_COLUMN(p_stmt,
		5,
		p_interleave_set_dimm_info->serial_num);
	TEXT_COLUMN(p_stmt,
		6,
		p_interleave_set_dimm_info->model_num,
		INTERLEAVE_SET_DIMM_INFO_MODEL_NUM_LEN);
	INTEGER_COLUMN(p_stmt,
		7,
		p_interleave_set_dimm_info->offset);
	INTEGER_COLUMN(p_stmt,
		8,
		p_interleave_set_dimm_info->size);
}
void db_print_interleave_set_dimm_info(struct db_interleave_set_dimm_info *p_value)
{
	printf("interleave_set_dimm_info.id: %d\n", p_value->id);
	printf("interleave_set_dimm_info.config_table_type: unsigned %d\n", p_value->config_table_type);
	printf("interleave_set_dimm_info.index_id: unsigned %d\n", p_value->index_id);
	printf("interleave_set_dimm_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("interleave_set_dimm_info.manufacturer: unsigned %d\n", p_value->manufacturer);
	printf("interleave_set_dimm_info.serial_num: unsigned %d\n", p_value->serial_num);
	printf("interleave_set_dimm_info.model_num: %s\n", p_value->model_num);
	printf("interleave_set_dimm_info.offset: unsigned %lld\n", p_value->offset);
	printf("interleave_set_dimm_info.size: unsigned %lld\n", p_value->size);
}
enum db_return_codes db_add_interleave_set_dimm_info(const PersistentStore *p_ps,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO interleave_set_dimm_info \
		(id, config_table_type, index_id, device_handle, manufacturer, serial_num, model_num, offset, size)  \
		VALUES 		\
		($id, \
		$config_table_type, \
		$index_id, \
		$device_handle, \
		$manufacturer, \
		$serial_num, \
		$model_num, \
		$offset, \
		$size) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_interleave_set_dimm_info(p_stmt, p_interleave_set_dimm_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "interleave_set_dimm_info", p_count);
}
int db_get_interleave_set_dimm_infos(const PersistentStore *p_ps,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info,
	int interleave_set_dimm_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_interleave_set_dimm_info, 0, sizeof (struct db_interleave_set_dimm_info) * interleave_set_dimm_info_count);
	char *sql = "SELECT \
		id \
		,  config_table_type \
		,  index_id \
		,  device_handle \
		,  manufacturer \
		,  serial_num \
		,  model_num \
		,  offset \
		,  size \
		  \
		FROM interleave_set_dimm_info \
		          \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_set_dimm_info_count)
		{
			local_row_to_interleave_set_dimm_info(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			local_get_interleave_set_dimm_info_relationships(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_interleave_set_dimm_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM interleave_set_dimm_info");
}

enum db_return_codes db_save_interleave_set_dimm_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_interleave_set_dimm_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_interleave_set_dimm_info_by_id(p_ps, p_interleave_set_dimm_info->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_interleave_set_dimm_info_by_id(p_ps,
				p_interleave_set_dimm_info->id,
				p_interleave_set_dimm_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO interleave_set_dimm_info \
			( id ,  config_table_type ,  index_id ,  device_handle ,  manufacturer ,  serial_num ,  model_num ,  offset ,  size )  \
			VALUES 		\
			($id, \
			$config_table_type, \
			$index_id, \
			$device_handle, \
			$manufacturer, \
			$serial_num, \
			$model_num, \
			$offset, \
			$size) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_interleave_set_dimm_info(p_stmt, p_interleave_set_dimm_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO interleave_set_dimm_info_history \
			(history_id, \
				 id,  config_table_type,  index_id,  device_handle,  manufacturer,  serial_num,  model_num,  offset,  size)  \
			VALUES 		($history_id, \
				 $id , \
				 $config_table_type , \
				 $index_id , \
				 $device_handle , \
				 $manufacturer , \
				 $serial_num , \
				 $model_num , \
				 $offset , \
				 $size )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_interleave_set_dimm_info(p_stmt, p_interleave_set_dimm_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_interleave_set_dimm_info_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info)
{
	memset(p_interleave_set_dimm_info, 0, sizeof (struct db_interleave_set_dimm_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  config_table_type,  index_id,  device_handle,  manufacturer,  serial_num,  model_num,  offset,  size  \
		FROM interleave_set_dimm_info \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_interleave_set_dimm_info(p_ps, p_stmt, p_interleave_set_dimm_info);
			local_get_interleave_set_dimm_info_relationships(p_ps, p_stmt, p_interleave_set_dimm_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_interleave_set_dimm_info_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE interleave_set_dimm_info \
	SET \
	id=$id \
		,  config_table_type=$config_table_type \
		,  index_id=$index_id \
		,  device_handle=$device_handle \
		,  manufacturer=$manufacturer \
		,  serial_num=$serial_num \
		,  model_num=$model_num \
		,  offset=$offset \
		,  size=$size \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_interleave_set_dimm_info(p_stmt, p_interleave_set_dimm_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_interleave_set_dimm_info_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM interleave_set_dimm_info \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_interleave_set_dimm_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM interleave_set_dimm_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM interleave_set_dimm_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_interleave_set_dimm_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info,
	int history_id,
	int interleave_set_dimm_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_interleave_set_dimm_info, 0, sizeof (struct db_interleave_set_dimm_info) * interleave_set_dimm_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  config_table_type,  index_id,  device_handle,  manufacturer,  serial_num,  model_num,  offset,  size  \
		FROM interleave_set_dimm_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_set_dimm_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_interleave_set_dimm_info(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
		local_get_interleave_set_dimm_info_relationships_history(p_ps, p_stmt, &p_interleave_set_dimm_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_interleave_set_dimm_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM interleave_set_dimm_info_history");
}

/*!
 * Roll interleave_set_dimm_infos by id to specified max.
 */
enum db_return_codes db_roll_interleave_set_dimm_infos_by_id(const PersistentStore *p_ps,
	int max_rows)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char sql[1024];
	snprintf(sql, 1024,
				"DELETE FROM interleave_set_dimm_info "
				"WHERE id NOT IN ("
				"SELECT id "
				"FROM interleave_set_dimm_info "
				"ORDER BY id DESC "
				"LIMIT %d)", max_rows); 
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Get the max id in the interleave_set_dimm_info table.
 */
enum db_return_codes db_get_next_interleave_set_dimm_info_id(const PersistentStore *p_ps, int *p_max)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	if (p_max)
	{
		if ((rc = run_scalar_sql(p_ps, 
			"SELECT MAX(id) FROM interleave_set_dimm_info", p_max)) 
			==  DB_SUCCESS)
		{
			(*p_max)++;
		}
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_info_count_by_dimm_interleave_set_index_id(
	const PersistentStore *p_ps,
	const unsigned int index_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM interleave_set_dimm_info WHERE index_id = $index_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$index_id", (unsigned int)index_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_info_count_by_dimm_interleave_set_index_id_history(
	const PersistentStore *p_ps,
	const unsigned int index_id,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM interleave_set_dimm_info_history "
		"WHERE index_id = $index_id "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$index_id", (unsigned int)index_id);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_infos_by_dimm_interleave_set_index_id(const PersistentStore *p_ps,
	unsigned int index_id,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info,
	int interleave_set_dimm_info_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  config_table_type ,  index_id ,  device_handle ,  manufacturer ,  serial_num ,  model_num ,  offset ,  size  \
		FROM interleave_set_dimm_info \
		WHERE  index_id = $index_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$index_id", (unsigned int)index_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_set_dimm_info_count)
		{
			local_row_to_interleave_set_dimm_info(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			local_get_interleave_set_dimm_info_relationships(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_infos_by_dimm_interleave_set_index_id_history(const PersistentStore *p_ps,
	unsigned int index_id,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info,
	int interleave_set_dimm_info_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  config_table_type ,  index_id ,  device_handle ,  manufacturer ,  serial_num ,  model_num ,  offset ,  size  \
		FROM interleave_set_dimm_info_history \
		WHERE  index_id = $index_id AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$index_id", (unsigned int)index_id);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_set_dimm_info_count)
		{
			local_row_to_interleave_set_dimm_info(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			local_get_interleave_set_dimm_info_relationships(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_interleave_set_dimm_info_by_dimm_interleave_set_index_id(const PersistentStore *p_ps,
	unsigned int index_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM interleave_set_dimm_info \
				 WHERE index_id = $index_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$index_id", (unsigned int)index_id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_info_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM interleave_set_dimm_info WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_info_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM interleave_set_dimm_info_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_infos_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info,
	int interleave_set_dimm_info_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  config_table_type ,  index_id ,  device_handle ,  manufacturer ,  serial_num ,  model_num ,  offset ,  size  \
		FROM interleave_set_dimm_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_set_dimm_info_count)
		{
			local_row_to_interleave_set_dimm_info(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			local_get_interleave_set_dimm_info_relationships(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_interleave_set_dimm_infos_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_interleave_set_dimm_info *p_interleave_set_dimm_info,
	int interleave_set_dimm_info_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  config_table_type ,  index_id ,  device_handle ,  manufacturer ,  serial_num ,  model_num ,  offset ,  size  \
		FROM interleave_set_dimm_info_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < interleave_set_dimm_info_count)
		{
			local_row_to_interleave_set_dimm_info(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			local_get_interleave_set_dimm_info_relationships(p_ps, p_stmt, &p_interleave_set_dimm_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_interleave_set_dimm_info_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM interleave_set_dimm_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*!
 * Clear interleave_set_dimm_info.serial_num
 */
enum db_return_codes  db_clear_interleave_set_dimm_info_serial_num(PersistentStore *p_ps)
{
	enum db_return_codes rc =
		run_sql_no_results(p_ps->db, "UPDATE interleave_set_dimm_info SET serial_num=''");

	if (rc == DB_SUCCESS)
	{
		rc = run_sql_no_results(p_ps->db, "UPDATE interleave_set_dimm_info_history SET serial_num=''");
	}

	return rc;
}
/*
 * --- END interleave_set_dimm_info ----------------
 */
/*
 * --- enable_error_injection_info ----------------
 */
void local_bind_enable_error_injection_info(sqlite3_stmt *p_stmt, struct db_enable_error_injection_info *p_enable_error_injection_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_enable_error_injection_info->device_handle);
	BIND_INTEGER(p_stmt, "$enable", (unsigned int)p_enable_error_injection_info->enable);
}
void local_get_enable_error_injection_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_enable_error_injection_info *p_enable_error_injection_info)
{
}

void local_get_enable_error_injection_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_enable_error_injection_info *p_enable_error_injection_info,
	int history_id)
{
}

void local_row_to_enable_error_injection_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_enable_error_injection_info *p_enable_error_injection_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_enable_error_injection_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_enable_error_injection_info->enable);
}
void db_print_enable_error_injection_info(struct db_enable_error_injection_info *p_value)
{
	printf("enable_error_injection_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("enable_error_injection_info.enable: unsigned %d\n", p_value->enable);
}
enum db_return_codes db_add_enable_error_injection_info(const PersistentStore *p_ps,
	struct db_enable_error_injection_info *p_enable_error_injection_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO enable_error_injection_info \
		(device_handle, enable)  \
		VALUES 		\
		($device_handle, \
		$enable) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_enable_error_injection_info(p_stmt, p_enable_error_injection_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_enable_error_injection_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "enable_error_injection_info", p_count);
}
int db_get_enable_error_injection_infos(const PersistentStore *p_ps,
	struct db_enable_error_injection_info *p_enable_error_injection_info,
	int enable_error_injection_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_enable_error_injection_info, 0, sizeof (struct db_enable_error_injection_info) * enable_error_injection_info_count);
	char *sql = "SELECT \
		device_handle \
		,  enable \
		  \
		FROM enable_error_injection_info \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < enable_error_injection_info_count)
		{
			local_row_to_enable_error_injection_info(p_ps, p_stmt, &p_enable_error_injection_info[index]);
			local_get_enable_error_injection_info_relationships(p_ps, p_stmt, &p_enable_error_injection_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_enable_error_injection_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM enable_error_injection_info");
}

enum db_return_codes db_save_enable_error_injection_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_enable_error_injection_info *p_enable_error_injection_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_enable_error_injection_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_enable_error_injection_info_by_device_handle(p_ps, p_enable_error_injection_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_enable_error_injection_info_by_device_handle(p_ps,
				p_enable_error_injection_info->device_handle,
				p_enable_error_injection_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO enable_error_injection_info \
			( device_handle ,  enable )  \
			VALUES 		\
			($device_handle, \
			$enable) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_enable_error_injection_info(p_stmt, p_enable_error_injection_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO enable_error_injection_info_history \
			(history_id, \
				 device_handle,  enable)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $enable )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_enable_error_injection_info(p_stmt, p_enable_error_injection_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_enable_error_injection_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_enable_error_injection_info *p_enable_error_injection_info)
{
	memset(p_enable_error_injection_info, 0, sizeof (struct db_enable_error_injection_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  enable  \
		FROM enable_error_injection_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_enable_error_injection_info(p_ps, p_stmt, p_enable_error_injection_info);
			local_get_enable_error_injection_info_relationships(p_ps, p_stmt, p_enable_error_injection_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_enable_error_injection_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_enable_error_injection_info *p_enable_error_injection_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE enable_error_injection_info \
	SET \
	device_handle=$device_handle \
		,  enable=$enable \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_enable_error_injection_info(p_stmt, p_enable_error_injection_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_enable_error_injection_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM enable_error_injection_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_enable_error_injection_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM enable_error_injection_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_enable_error_injection_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM enable_error_injection_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_enable_error_injection_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_enable_error_injection_info *p_enable_error_injection_info,
	int history_id,
	int enable_error_injection_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_enable_error_injection_info, 0, sizeof (struct db_enable_error_injection_info) * enable_error_injection_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  enable  \
		FROM enable_error_injection_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < enable_error_injection_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_enable_error_injection_info(p_ps, p_stmt, &p_enable_error_injection_info[index]);
		local_get_enable_error_injection_info_relationships_history(p_ps, p_stmt, &p_enable_error_injection_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_enable_error_injection_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM enable_error_injection_info_history");
}

/*
 * --- END enable_error_injection_info ----------------
 */
/*
 * --- temperature_error_injection_info ----------------
 */
void local_bind_temperature_error_injection_info(sqlite3_stmt *p_stmt, struct db_temperature_error_injection_info *p_temperature_error_injection_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_temperature_error_injection_info->device_handle);
	BIND_INTEGER(p_stmt, "$temperature", (unsigned int)p_temperature_error_injection_info->temperature);
}
void local_get_temperature_error_injection_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_temperature_error_injection_info *p_temperature_error_injection_info)
{
}

void local_get_temperature_error_injection_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_temperature_error_injection_info *p_temperature_error_injection_info,
	int history_id)
{
}

void local_row_to_temperature_error_injection_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_temperature_error_injection_info *p_temperature_error_injection_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_temperature_error_injection_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_temperature_error_injection_info->temperature);
}
void db_print_temperature_error_injection_info(struct db_temperature_error_injection_info *p_value)
{
	printf("temperature_error_injection_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("temperature_error_injection_info.temperature: unsigned %d\n", p_value->temperature);
}
enum db_return_codes db_add_temperature_error_injection_info(const PersistentStore *p_ps,
	struct db_temperature_error_injection_info *p_temperature_error_injection_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO temperature_error_injection_info \
		(device_handle, temperature)  \
		VALUES 		\
		($device_handle, \
		$temperature) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_temperature_error_injection_info(p_stmt, p_temperature_error_injection_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_temperature_error_injection_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "temperature_error_injection_info", p_count);
}
int db_get_temperature_error_injection_infos(const PersistentStore *p_ps,
	struct db_temperature_error_injection_info *p_temperature_error_injection_info,
	int temperature_error_injection_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_temperature_error_injection_info, 0, sizeof (struct db_temperature_error_injection_info) * temperature_error_injection_info_count);
	char *sql = "SELECT \
		device_handle \
		,  temperature \
		  \
		FROM temperature_error_injection_info \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < temperature_error_injection_info_count)
		{
			local_row_to_temperature_error_injection_info(p_ps, p_stmt, &p_temperature_error_injection_info[index]);
			local_get_temperature_error_injection_info_relationships(p_ps, p_stmt, &p_temperature_error_injection_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_temperature_error_injection_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM temperature_error_injection_info");
}

enum db_return_codes db_save_temperature_error_injection_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_temperature_error_injection_info *p_temperature_error_injection_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_temperature_error_injection_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_temperature_error_injection_info_by_device_handle(p_ps, p_temperature_error_injection_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_temperature_error_injection_info_by_device_handle(p_ps,
				p_temperature_error_injection_info->device_handle,
				p_temperature_error_injection_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO temperature_error_injection_info \
			( device_handle ,  temperature )  \
			VALUES 		\
			($device_handle, \
			$temperature) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_temperature_error_injection_info(p_stmt, p_temperature_error_injection_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO temperature_error_injection_info_history \
			(history_id, \
				 device_handle,  temperature)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $temperature )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_temperature_error_injection_info(p_stmt, p_temperature_error_injection_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_temperature_error_injection_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_temperature_error_injection_info *p_temperature_error_injection_info)
{
	memset(p_temperature_error_injection_info, 0, sizeof (struct db_temperature_error_injection_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  temperature  \
		FROM temperature_error_injection_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_temperature_error_injection_info(p_ps, p_stmt, p_temperature_error_injection_info);
			local_get_temperature_error_injection_info_relationships(p_ps, p_stmt, p_temperature_error_injection_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_temperature_error_injection_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_temperature_error_injection_info *p_temperature_error_injection_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE temperature_error_injection_info \
	SET \
	device_handle=$device_handle \
		,  temperature=$temperature \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_temperature_error_injection_info(p_stmt, p_temperature_error_injection_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_temperature_error_injection_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM temperature_error_injection_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_temperature_error_injection_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM temperature_error_injection_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_temperature_error_injection_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM temperature_error_injection_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_temperature_error_injection_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_temperature_error_injection_info *p_temperature_error_injection_info,
	int history_id,
	int temperature_error_injection_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_temperature_error_injection_info, 0, sizeof (struct db_temperature_error_injection_info) * temperature_error_injection_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  temperature  \
		FROM temperature_error_injection_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < temperature_error_injection_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_temperature_error_injection_info(p_ps, p_stmt, &p_temperature_error_injection_info[index]);
		local_get_temperature_error_injection_info_relationships_history(p_ps, p_stmt, &p_temperature_error_injection_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_temperature_error_injection_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM temperature_error_injection_info_history");
}

/*
 * --- END temperature_error_injection_info ----------------
 */
/*
 * --- poison_error_injection_info ----------------
 */
void local_bind_poison_error_injection_info(sqlite3_stmt *p_stmt, struct db_poison_error_injection_info *p_poison_error_injection_info)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_poison_error_injection_info->id);
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_poison_error_injection_info->device_handle);
	BIND_INTEGER(p_stmt, "$dpa_address", (unsigned long long)p_poison_error_injection_info->dpa_address);
}
void local_get_poison_error_injection_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_poison_error_injection_info *p_poison_error_injection_info)
{
}

void local_get_poison_error_injection_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_poison_error_injection_info *p_poison_error_injection_info,
	int history_id)
{
}

void local_row_to_poison_error_injection_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_poison_error_injection_info *p_poison_error_injection_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_poison_error_injection_info->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_poison_error_injection_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		2,
		p_poison_error_injection_info->dpa_address);
}
void db_print_poison_error_injection_info(struct db_poison_error_injection_info *p_value)
{
	printf("poison_error_injection_info.id: %d\n", p_value->id);
	printf("poison_error_injection_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("poison_error_injection_info.dpa_address: unsigned %lld\n", p_value->dpa_address);
}
enum db_return_codes db_add_poison_error_injection_info(const PersistentStore *p_ps,
	struct db_poison_error_injection_info *p_poison_error_injection_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO poison_error_injection_info \
		(device_handle, dpa_address)  \
		VALUES 		\
		(\
		$device_handle, \
		$dpa_address) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_poison_error_injection_info(p_stmt, p_poison_error_injection_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_poison_error_injection_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "poison_error_injection_info", p_count);
}
int db_get_poison_error_injection_infos(const PersistentStore *p_ps,
	struct db_poison_error_injection_info *p_poison_error_injection_info,
	int poison_error_injection_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_poison_error_injection_info, 0, sizeof (struct db_poison_error_injection_info) * poison_error_injection_info_count);
	char *sql = "SELECT \
		id \
		,  device_handle \
		,  dpa_address \
		  \
		FROM poison_error_injection_info \
		    \
		 ORDER BY id DESC  \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < poison_error_injection_info_count)
		{
			local_row_to_poison_error_injection_info(p_ps, p_stmt, &p_poison_error_injection_info[index]);
			local_get_poison_error_injection_info_relationships(p_ps, p_stmt, &p_poison_error_injection_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_poison_error_injection_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM poison_error_injection_info");
}

enum db_return_codes db_save_poison_error_injection_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_poison_error_injection_info *p_poison_error_injection_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_poison_error_injection_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_poison_error_injection_info_by_id(p_ps, p_poison_error_injection_info->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_poison_error_injection_info_by_id(p_ps,
				p_poison_error_injection_info->id,
				p_poison_error_injection_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO poison_error_injection_info \
			( id ,  device_handle ,  dpa_address )  \
			VALUES 		\
			($id, \
			$device_handle, \
			$dpa_address) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_poison_error_injection_info(p_stmt, p_poison_error_injection_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO poison_error_injection_info_history \
			(history_id, \
				 id,  device_handle,  dpa_address)  \
			VALUES 		($history_id, \
				 $id , \
				 $device_handle , \
				 $dpa_address )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_poison_error_injection_info(p_stmt, p_poison_error_injection_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_poison_error_injection_info_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_poison_error_injection_info *p_poison_error_injection_info)
{
	memset(p_poison_error_injection_info, 0, sizeof (struct db_poison_error_injection_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  device_handle,  dpa_address  \
		FROM poison_error_injection_info \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_poison_error_injection_info(p_ps, p_stmt, p_poison_error_injection_info);
			local_get_poison_error_injection_info_relationships(p_ps, p_stmt, p_poison_error_injection_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_poison_error_injection_info_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_poison_error_injection_info *p_poison_error_injection_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE poison_error_injection_info \
	SET \
	id=$id \
		,  device_handle=$device_handle \
		,  dpa_address=$dpa_address \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_poison_error_injection_info(p_stmt, p_poison_error_injection_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_poison_error_injection_info_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM poison_error_injection_info \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_poison_error_injection_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM poison_error_injection_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_poison_error_injection_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM poison_error_injection_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_poison_error_injection_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_poison_error_injection_info *p_poison_error_injection_info,
	int history_id,
	int poison_error_injection_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_poison_error_injection_info, 0, sizeof (struct db_poison_error_injection_info) * poison_error_injection_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  device_handle,  dpa_address  \
		FROM poison_error_injection_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < poison_error_injection_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_poison_error_injection_info(p_ps, p_stmt, &p_poison_error_injection_info[index]);
		local_get_poison_error_injection_info_relationships_history(p_ps, p_stmt, &p_poison_error_injection_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_poison_error_injection_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM poison_error_injection_info_history");
}

enum db_return_codes db_get_poison_error_injection_info_count_by_dimm_topology_device_handle(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM poison_error_injection_info WHERE device_handle = $device_handle";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_poison_error_injection_info_count_by_dimm_topology_device_handle_history(
	const PersistentStore *p_ps,
	const unsigned int device_handle,
	int *p_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	const char *sql = "SELECT COUNT (*) FROM poison_error_injection_info_history "
		"WHERE device_handle = $device_handle "
			"AND history_id=$history_id";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_poison_error_injection_infos_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_poison_error_injection_info *p_poison_error_injection_info,
	int poison_error_injection_info_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  device_handle ,  dpa_address  \
		FROM poison_error_injection_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < poison_error_injection_info_count)
		{
			local_row_to_poison_error_injection_info(p_ps, p_stmt, &p_poison_error_injection_info[index]);
			local_get_poison_error_injection_info_relationships(p_ps, p_stmt, &p_poison_error_injection_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_poison_error_injection_infos_by_dimm_topology_device_handle_history(const PersistentStore *p_ps,
	unsigned int device_handle,
	struct db_poison_error_injection_info *p_poison_error_injection_info,
	int poison_error_injection_info_count, int history_id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		 id ,  device_handle ,  dpa_address  \
		FROM poison_error_injection_info_history \
		WHERE  device_handle = $device_handle AND history_id=$history_id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		rc = DB_SUCCESS;
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < poison_error_injection_info_count)
		{
			local_row_to_poison_error_injection_info(p_ps, p_stmt, &p_poison_error_injection_info[index]);
			local_get_poison_error_injection_info_relationships(p_ps, p_stmt, &p_poison_error_injection_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_delete_poison_error_injection_info_by_dimm_topology_device_handle(const PersistentStore *p_ps,
	unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM poison_error_injection_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
/*
 * --- END poison_error_injection_info ----------------
 */
/*
 * --- software_trigger_info ----------------
 */
void local_bind_software_trigger_info(sqlite3_stmt *p_stmt, struct db_software_trigger_info *p_software_trigger_info)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_software_trigger_info->device_handle);
	BIND_INTEGER(p_stmt, "$die_sparing_trigger", (unsigned int)p_software_trigger_info->die_sparing_trigger);
	BIND_INTEGER(p_stmt, "$user_spare_block_alarm_trip_trigger", (unsigned int)p_software_trigger_info->user_spare_block_alarm_trip_trigger);
	BIND_INTEGER(p_stmt, "$fatal_error_trigger", (unsigned int)p_software_trigger_info->fatal_error_trigger);
}
void local_get_software_trigger_info_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_software_trigger_info *p_software_trigger_info)
{
}

void local_get_software_trigger_info_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_software_trigger_info *p_software_trigger_info,
	int history_id)
{
}

void local_row_to_software_trigger_info(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_software_trigger_info *p_software_trigger_info)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_software_trigger_info->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_software_trigger_info->die_sparing_trigger);
	INTEGER_COLUMN(p_stmt,
		2,
		p_software_trigger_info->user_spare_block_alarm_trip_trigger);
	INTEGER_COLUMN(p_stmt,
		3,
		p_software_trigger_info->fatal_error_trigger);
}
void db_print_software_trigger_info(struct db_software_trigger_info *p_value)
{
	printf("software_trigger_info.device_handle: unsigned %d\n", p_value->device_handle);
	printf("software_trigger_info.die_sparing_trigger: unsigned %d\n", p_value->die_sparing_trigger);
	printf("software_trigger_info.user_spare_block_alarm_trip_trigger: unsigned %d\n", p_value->user_spare_block_alarm_trip_trigger);
	printf("software_trigger_info.fatal_error_trigger: unsigned %d\n", p_value->fatal_error_trigger);
}
enum db_return_codes db_add_software_trigger_info(const PersistentStore *p_ps,
	struct db_software_trigger_info *p_software_trigger_info)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO software_trigger_info \
		(device_handle, die_sparing_trigger, user_spare_block_alarm_trip_trigger, fatal_error_trigger)  \
		VALUES 		\
		($device_handle, \
		$die_sparing_trigger, \
		$user_spare_block_alarm_trip_trigger, \
		$fatal_error_trigger) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_software_trigger_info(p_stmt, p_software_trigger_info);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_software_trigger_info_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "software_trigger_info", p_count);
}
int db_get_software_trigger_infos(const PersistentStore *p_ps,
	struct db_software_trigger_info *p_software_trigger_info,
	int software_trigger_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_software_trigger_info, 0, sizeof (struct db_software_trigger_info) * software_trigger_info_count);
	char *sql = "SELECT \
		device_handle \
		,  die_sparing_trigger \
		,  user_spare_block_alarm_trip_trigger \
		,  fatal_error_trigger \
		  \
		FROM software_trigger_info \
		     \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < software_trigger_info_count)
		{
			local_row_to_software_trigger_info(p_ps, p_stmt, &p_software_trigger_info[index]);
			local_get_software_trigger_info_relationships(p_ps, p_stmt, &p_software_trigger_info[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_software_trigger_infos(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM software_trigger_info");
}

enum db_return_codes db_save_software_trigger_info_state(const PersistentStore *p_ps,
	int history_id,
	struct db_software_trigger_info *p_software_trigger_info)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_software_trigger_info temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_software_trigger_info_by_device_handle(p_ps, p_software_trigger_info->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_software_trigger_info_by_device_handle(p_ps,
				p_software_trigger_info->device_handle,
				p_software_trigger_info);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO software_trigger_info \
			( device_handle ,  die_sparing_trigger ,  user_spare_block_alarm_trip_trigger ,  fatal_error_trigger )  \
			VALUES 		\
			($device_handle, \
			$die_sparing_trigger, \
			$user_spare_block_alarm_trip_trigger, \
			$fatal_error_trigger) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_software_trigger_info(p_stmt, p_software_trigger_info);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO software_trigger_info_history \
			(history_id, \
				 device_handle,  die_sparing_trigger,  user_spare_block_alarm_trip_trigger,  fatal_error_trigger)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $die_sparing_trigger , \
				 $user_spare_block_alarm_trip_trigger , \
				 $fatal_error_trigger )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_software_trigger_info(p_stmt, p_software_trigger_info);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

enum db_return_codes db_get_software_trigger_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_software_trigger_info *p_software_trigger_info)
{
	memset(p_software_trigger_info, 0, sizeof (struct db_software_trigger_info));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  die_sparing_trigger,  user_spare_block_alarm_trip_trigger,  fatal_error_trigger  \
		FROM software_trigger_info \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_software_trigger_info(p_ps, p_stmt, p_software_trigger_info);
			local_get_software_trigger_info_relationships(p_ps, p_stmt, p_software_trigger_info);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_software_trigger_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_software_trigger_info *p_software_trigger_info)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE software_trigger_info \
	SET \
	device_handle=$device_handle \
		,  die_sparing_trigger=$die_sparing_trigger \
		,  user_spare_block_alarm_trip_trigger=$user_spare_block_alarm_trip_trigger \
		,  fatal_error_trigger=$fatal_error_trigger \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_software_trigger_info(p_stmt, p_software_trigger_info);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_software_trigger_info_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM software_trigger_info \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

enum db_return_codes db_get_software_trigger_info_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM software_trigger_info_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_software_trigger_info_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM software_trigger_info_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_software_trigger_info_history_by_history_id(const PersistentStore *p_ps,
	struct db_software_trigger_info *p_software_trigger_info,
	int history_id,
	int software_trigger_info_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_software_trigger_info, 0, sizeof (struct db_software_trigger_info) * software_trigger_info_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  die_sparing_trigger,  user_spare_block_alarm_trip_trigger,  fatal_error_trigger  \
		FROM software_trigger_info_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < software_trigger_info_count)
		{
		rc = DB_SUCCESS;
		local_row_to_software_trigger_info(p_ps, p_stmt, &p_software_trigger_info[index]);
		local_get_software_trigger_info_relationships_history(p_ps, p_stmt, &p_software_trigger_info[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_software_trigger_info_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM software_trigger_info_history");
}

/*
 * --- END software_trigger_info ----------------
 */
/*
 * --- performance ----------------
 */
void local_bind_performance(sqlite3_stmt *p_stmt, struct db_performance *p_performance)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_performance->id);
	BIND_TEXT(p_stmt, "$dimm_uid", (char *)p_performance->dimm_uid);
	BIND_INTEGER(p_stmt, "$time", (unsigned long long)p_performance->time);
	BIND_INTEGER(p_stmt, "$bytes_read", (unsigned long long)p_performance->bytes_read);
	BIND_INTEGER(p_stmt, "$bytes_written", (unsigned long long)p_performance->bytes_written);
	BIND_INTEGER(p_stmt, "$read_reqs", (unsigned long long)p_performance->read_reqs);
	BIND_INTEGER(p_stmt, "$host_write_cmds", (unsigned long long)p_performance->host_write_cmds);
	BIND_INTEGER(p_stmt, "$block_reads", (unsigned long long)p_performance->block_reads);
	BIND_INTEGER(p_stmt, "$block_writes", (unsigned long long)p_performance->block_writes);
}
void local_get_performance_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_performance *p_performance)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_performance_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_performance *p_performance,
	int history_id)
{
}

#endif

void local_row_to_performance(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_performance *p_performance)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_performance->id);
	TEXT_COLUMN(p_stmt,
		1,
		p_performance->dimm_uid,
		PERFORMANCE_DIMM_UID_LEN);
	INTEGER_COLUMN(p_stmt,
		2,
		p_performance->time);
	INTEGER_COLUMN(p_stmt,
		3,
		p_performance->bytes_read);
	INTEGER_COLUMN(p_stmt,
		4,
		p_performance->bytes_written);
	INTEGER_COLUMN(p_stmt,
		5,
		p_performance->read_reqs);
	INTEGER_COLUMN(p_stmt,
		6,
		p_performance->host_write_cmds);
	INTEGER_COLUMN(p_stmt,
		7,
		p_performance->block_reads);
	INTEGER_COLUMN(p_stmt,
		8,
		p_performance->block_writes);
}
void db_print_performance(struct db_performance *p_value)
{
	printf("performance.id: %d\n", p_value->id);
	printf("performance.dimm_uid: %s\n", p_value->dimm_uid);
	printf("performance.time: unsigned %lld\n", p_value->time);
	printf("performance.bytes_read: unsigned %lld\n", p_value->bytes_read);
	printf("performance.bytes_written: unsigned %lld\n", p_value->bytes_written);
	printf("performance.read_reqs: unsigned %lld\n", p_value->read_reqs);
	printf("performance.host_write_cmds: unsigned %lld\n", p_value->host_write_cmds);
	printf("performance.block_reads: unsigned %lld\n", p_value->block_reads);
	printf("performance.block_writes: unsigned %lld\n", p_value->block_writes);
}
enum db_return_codes db_add_performance(const PersistentStore *p_ps,
	struct db_performance *p_performance)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO performance \
		(dimm_uid, time, bytes_read, bytes_written, read_reqs, host_write_cmds, block_reads, block_writes)  \
		VALUES 		\
		(\
		$dimm_uid, \
		$time, \
		$bytes_read, \
		$bytes_written, \
		$read_reqs, \
		$host_write_cmds, \
		$block_reads, \
		$block_writes) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_performance(p_stmt, p_performance);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_performance_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "performance", p_count);
}
int db_get_performances(const PersistentStore *p_ps,
	struct db_performance *p_performance,
	int performance_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_performance, 0, sizeof (struct db_performance) * performance_count);
	char *sql = "SELECT \
		id \
		,  dimm_uid \
		,  time \
		,  bytes_read \
		,  bytes_written \
		,  read_reqs \
		,  host_write_cmds \
		,  block_reads \
		,  block_writes \
		  \
		FROM performance \
		          \
		 ORDER BY time DESC  \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < performance_count)
		{
			local_row_to_performance(p_ps, p_stmt, &p_performance[index]);
			local_get_performance_relationships(p_ps, p_stmt, &p_performance[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_performances(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM performance");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_performance_state(const PersistentStore *p_ps,
	int history_id,
	struct db_performance *p_performance)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_performance temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_performance_by_id(p_ps, p_performance->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_performance_by_id(p_ps,
				p_performance->id,
				p_performance);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO performance \
			( id ,  dimm_uid ,  time ,  bytes_read ,  bytes_written ,  read_reqs ,  host_write_cmds ,  block_reads ,  block_writes )  \
			VALUES 		\
			($id, \
			$dimm_uid, \
			$time, \
			$bytes_read, \
			$bytes_written, \
			$read_reqs, \
			$host_write_cmds, \
			$block_reads, \
			$block_writes) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_performance(p_stmt, p_performance);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO performance_history \
			(history_id, \
				 id,  dimm_uid,  time,  bytes_read,  bytes_written,  read_reqs,  host_write_cmds,  block_reads,  block_writes)  \
			VALUES 		($history_id, \
				 $id , \
				 $dimm_uid , \
				 $time , \
				 $bytes_read , \
				 $bytes_written , \
				 $read_reqs , \
				 $host_write_cmds , \
				 $block_reads , \
				 $block_writes )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_performance(p_stmt, p_performance);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_performance_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_performance *p_performance)
{
	memset(p_performance, 0, sizeof (struct db_performance));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  dimm_uid,  time,  bytes_read,  bytes_written,  read_reqs,  host_write_cmds,  block_reads,  block_writes  \
		FROM performance \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_performance(p_ps, p_stmt, p_performance);
			local_get_performance_relationships(p_ps, p_stmt, p_performance);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_performance_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_performance *p_performance)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE performance \
	SET \
	id=$id \
		,  dimm_uid=$dimm_uid \
		,  time=$time \
		,  bytes_read=$bytes_read \
		,  bytes_written=$bytes_written \
		,  read_reqs=$read_reqs \
		,  host_write_cmds=$host_write_cmds \
		,  block_reads=$block_reads \
		,  block_writes=$block_writes \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_performance(p_stmt, p_performance);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_performance_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM performance \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_performance_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM performance_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_performance_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM performance_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_performance_history_by_history_id(const PersistentStore *p_ps,
	struct db_performance *p_performance,
	int history_id,
	int performance_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_performance, 0, sizeof (struct db_performance) * performance_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  dimm_uid,  time,  bytes_read,  bytes_written,  read_reqs,  host_write_cmds,  block_reads,  block_writes  \
		FROM performance_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < performance_count)
		{
		rc = DB_SUCCESS;
		local_row_to_performance(p_ps, p_stmt, &p_performance[index]);
		local_get_performance_relationships_history(p_ps, p_stmt, &p_performance[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_performance_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM performance_history");
}

#endif

/*
 * --- END performance ----------------
 */
/*
 * --- driver_metadata_check_diag_result ----------------
 */
void local_bind_driver_metadata_check_diag_result(sqlite3_stmt *p_stmt, struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result)
{
	BIND_INTEGER(p_stmt, "$id", (int)p_driver_metadata_check_diag_result->id);
	BIND_INTEGER(p_stmt, "$result_type", (int)p_driver_metadata_check_diag_result->result_type);
	BIND_TEXT(p_stmt, "$ns_uid", (char *)p_driver_metadata_check_diag_result->ns_uid);
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_driver_metadata_check_diag_result->device_handle);
	BIND_INTEGER(p_stmt, "$health_flag", (unsigned int)p_driver_metadata_check_diag_result->health_flag);
}
void local_get_driver_metadata_check_diag_result_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_driver_metadata_check_diag_result_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result,
	int history_id)
{
}

#endif

void local_row_to_driver_metadata_check_diag_result(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_driver_metadata_check_diag_result->id);
	INTEGER_COLUMN(p_stmt,
		1,
		p_driver_metadata_check_diag_result->result_type);
	TEXT_COLUMN(p_stmt,
		2,
		p_driver_metadata_check_diag_result->ns_uid,
		DRIVER_METADATA_CHECK_DIAG_RESULT_NS_UID_LEN);
	INTEGER_COLUMN(p_stmt,
		3,
		p_driver_metadata_check_diag_result->device_handle);
	INTEGER_COLUMN(p_stmt,
		4,
		p_driver_metadata_check_diag_result->health_flag);
}
void db_print_driver_metadata_check_diag_result(struct db_driver_metadata_check_diag_result *p_value)
{
	printf("driver_metadata_check_diag_result.id: %d\n", p_value->id);
	printf("driver_metadata_check_diag_result.result_type: %d\n", p_value->result_type);
	printf("driver_metadata_check_diag_result.ns_uid: %s\n", p_value->ns_uid);
	printf("driver_metadata_check_diag_result.device_handle: unsigned %d\n", p_value->device_handle);
	printf("driver_metadata_check_diag_result.health_flag: unsigned %d\n", p_value->health_flag);
}
enum db_return_codes db_add_driver_metadata_check_diag_result(const PersistentStore *p_ps,
	struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO driver_metadata_check_diag_result \
		(result_type, ns_uid, device_handle, health_flag)  \
		VALUES 		\
		(\
		$result_type, \
		$ns_uid, \
		$device_handle, \
		$health_flag) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_driver_metadata_check_diag_result(p_stmt, p_driver_metadata_check_diag_result);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_driver_metadata_check_diag_result_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "driver_metadata_check_diag_result", p_count);
}
int db_get_driver_metadata_check_diag_results(const PersistentStore *p_ps,
	struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result,
	int driver_metadata_check_diag_result_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_driver_metadata_check_diag_result, 0, sizeof (struct db_driver_metadata_check_diag_result) * driver_metadata_check_diag_result_count);
	char *sql = "SELECT \
		id \
		,  result_type \
		,  ns_uid \
		,  device_handle \
		,  health_flag \
		  \
		FROM driver_metadata_check_diag_result \
		      \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < driver_metadata_check_diag_result_count)
		{
			local_row_to_driver_metadata_check_diag_result(p_ps, p_stmt, &p_driver_metadata_check_diag_result[index]);
			local_get_driver_metadata_check_diag_result_relationships(p_ps, p_stmt, &p_driver_metadata_check_diag_result[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_driver_metadata_check_diag_results(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM driver_metadata_check_diag_result");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_driver_metadata_check_diag_result_state(const PersistentStore *p_ps,
	int history_id,
	struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_driver_metadata_check_diag_result temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_driver_metadata_check_diag_result_by_id(p_ps, p_driver_metadata_check_diag_result->id, &temp) == DB_SUCCESS)
	{
		rc = db_update_driver_metadata_check_diag_result_by_id(p_ps,
				p_driver_metadata_check_diag_result->id,
				p_driver_metadata_check_diag_result);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO driver_metadata_check_diag_result \
			( id ,  result_type ,  ns_uid ,  device_handle ,  health_flag )  \
			VALUES 		\
			($id, \
			$result_type, \
			$ns_uid, \
			$device_handle, \
			$health_flag) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_driver_metadata_check_diag_result(p_stmt, p_driver_metadata_check_diag_result);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO driver_metadata_check_diag_result_history \
			(history_id, \
				 id,  result_type,  ns_uid,  device_handle,  health_flag)  \
			VALUES 		($history_id, \
				 $id , \
				 $result_type , \
				 $ns_uid , \
				 $device_handle , \
				 $health_flag )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_driver_metadata_check_diag_result(p_stmt, p_driver_metadata_check_diag_result);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_driver_metadata_check_diag_result_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result)
{
	memset(p_driver_metadata_check_diag_result, 0, sizeof (struct db_driver_metadata_check_diag_result));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		id,  result_type,  ns_uid,  device_handle,  health_flag  \
		FROM driver_metadata_check_diag_result \
		WHERE  id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_driver_metadata_check_diag_result(p_ps, p_stmt, p_driver_metadata_check_diag_result);
			local_get_driver_metadata_check_diag_result_relationships(p_ps, p_stmt, p_driver_metadata_check_diag_result);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_driver_metadata_check_diag_result_by_id(const PersistentStore *p_ps,
	const int id,
	struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE driver_metadata_check_diag_result \
	SET \
	id=$id \
		,  result_type=$result_type \
		,  ns_uid=$ns_uid \
		,  device_handle=$device_handle \
		,  health_flag=$health_flag \
		  \
	WHERE id=$id ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		local_bind_driver_metadata_check_diag_result(p_stmt, p_driver_metadata_check_diag_result);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_driver_metadata_check_diag_result_by_id(const PersistentStore *p_ps,
	const int id)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM driver_metadata_check_diag_result \
				 WHERE id = $id";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$id", (int)id);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_driver_metadata_check_diag_result_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM driver_metadata_check_diag_result_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_driver_metadata_check_diag_result_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM driver_metadata_check_diag_result_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_driver_metadata_check_diag_result_history_by_history_id(const PersistentStore *p_ps,
	struct db_driver_metadata_check_diag_result *p_driver_metadata_check_diag_result,
	int history_id,
	int driver_metadata_check_diag_result_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_driver_metadata_check_diag_result, 0, sizeof (struct db_driver_metadata_check_diag_result) * driver_metadata_check_diag_result_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		id,  result_type,  ns_uid,  device_handle,  health_flag  \
		FROM driver_metadata_check_diag_result_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < driver_metadata_check_diag_result_count)
		{
		rc = DB_SUCCESS;
		local_row_to_driver_metadata_check_diag_result(p_ps, p_stmt, &p_driver_metadata_check_diag_result[index]);
		local_get_driver_metadata_check_diag_result_relationships_history(p_ps, p_stmt, &p_driver_metadata_check_diag_result[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_driver_metadata_check_diag_result_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM driver_metadata_check_diag_result_history");
}

#endif

/*
 * --- END driver_metadata_check_diag_result ----------------
 */
/*
 * --- boot_status_register ----------------
 */
void local_bind_boot_status_register(sqlite3_stmt *p_stmt, struct db_boot_status_register *p_boot_status_register)
{
	BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)p_boot_status_register->device_handle);
	BIND_INTEGER(p_stmt, "$bsr", (unsigned long long)p_boot_status_register->bsr);
}
void local_get_boot_status_register_relationships(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_boot_status_register *p_boot_status_register)
{
}

#if 0
//NON-HISTORY TABLE

void local_get_boot_status_register_relationships_history(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_boot_status_register *p_boot_status_register,
	int history_id)
{
}

#endif

void local_row_to_boot_status_register(const PersistentStore *p_ps,
	sqlite3_stmt *p_stmt, struct db_boot_status_register *p_boot_status_register)
{
	INTEGER_COLUMN(p_stmt,
		0,
		p_boot_status_register->device_handle);
	INTEGER_COLUMN(p_stmt,
		1,
		p_boot_status_register->bsr);
}
void db_print_boot_status_register(struct db_boot_status_register *p_value)
{
	printf("boot_status_register.device_handle: unsigned %d\n", p_value->device_handle);
	printf("boot_status_register.bsr: unsigned %lld\n", p_value->bsr);
}
enum db_return_codes db_add_boot_status_register(const PersistentStore *p_ps,
	struct db_boot_status_register *p_boot_status_register)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = 	"INSERT INTO boot_status_register \
		(device_handle, bsr)  \
		VALUES 		\
		($device_handle, \
		$bsr) ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		local_bind_boot_status_register(p_stmt, p_boot_status_register);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_boot_status_register_count(const PersistentStore *p_ps, int *p_count)
{
	return table_row_count(p_ps, "boot_status_register", p_count);
}
int db_get_boot_status_registers(const PersistentStore *p_ps,
	struct db_boot_status_register *p_boot_status_register,
	int boot_status_register_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_boot_status_register, 0, sizeof (struct db_boot_status_register) * boot_status_register_count);
	char *sql = "SELECT \
		device_handle \
		,  bsr \
		  \
		FROM boot_status_register \
		   \
		 \
		";
	sqlite3_stmt *p_stmt;
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		int index = 0;
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < boot_status_register_count)
		{
			local_row_to_boot_status_register(p_ps, p_stmt, &p_boot_status_register[index]);
			local_get_boot_status_register_relationships(p_ps, p_stmt, &p_boot_status_register[index]);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_all_boot_status_registers(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM boot_status_register");
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_save_boot_status_register_state(const PersistentStore *p_ps,
	int history_id,
	struct db_boot_status_register *p_boot_status_register)
{
	enum db_return_codes rc = DB_SUCCESS;
	struct db_boot_status_register temp;
	/*
	 * Main table - Insert new or update existing
	 */
	if (db_get_boot_status_register_by_device_handle(p_ps, p_boot_status_register->device_handle, &temp) == DB_SUCCESS)
	{
		rc = db_update_boot_status_register_by_device_handle(p_ps,
				p_boot_status_register->device_handle,
				p_boot_status_register);
	}
	else
	{
		sqlite3_stmt *p_stmt;
		char *sql = 	"INSERT INTO boot_status_register \
			( device_handle ,  bsr )  \
			VALUES 		\
			($device_handle, \
			$bsr) ";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			local_bind_boot_status_register(p_stmt, p_boot_status_register);
			if (sqlite3_step(p_stmt) != SQLITE_DONE)
			{
				rc = DB_ERR_FAILURE;
			}
			sqlite3_finalize(p_stmt);
		}
	}
	/*
	 * Insert as a history
	 */
	if (rc == DB_SUCCESS)
	{
		sqlite3_stmt *p_stmt;
		char *sql = "INSERT INTO boot_status_register_history \
			(history_id, \
				 device_handle,  bsr)  \
			VALUES 		($history_id, \
				 $device_handle , \
				 $bsr )";
		if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
		{
			BIND_INTEGER(p_stmt, "$history_id", history_id);
			local_bind_boot_status_register(p_stmt, p_boot_status_register);
			rc = sqlite3_step(p_stmt) == SQLITE_DONE ? DB_SUCCESS : DB_ERR_FAILURE;
			sqlite3_finalize(p_stmt);
		}
		else
		{
			rc = DB_ERR_FAILURE;
		}
	}
	return rc;
}

#endif

enum db_return_codes db_get_boot_status_register_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_boot_status_register *p_boot_status_register)
{
	memset(p_boot_status_register, 0, sizeof (struct db_boot_status_register));
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "SELECT \
		device_handle,  bsr  \
		FROM boot_status_register \
		WHERE  device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			local_row_to_boot_status_register(p_ps, p_stmt, p_boot_status_register);
			local_get_boot_status_register_relationships(p_ps, p_stmt, p_boot_status_register);
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_update_boot_status_register_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle,
	struct db_boot_status_register *p_boot_status_register)
{
	sqlite3_stmt *p_stmt;
	enum db_return_codes rc = DB_SUCCESS;
	int sqlrc = 0;
	char *sql = "UPDATE boot_status_register \
	SET \
	device_handle=$device_handle \
		,  bsr=$bsr \
		  \
	WHERE device_handle=$device_handle ";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		local_bind_boot_status_register(p_stmt, p_boot_status_register);
		sqlrc = sqlite3_step(p_stmt) == SQLITE_OK;
		sqlite3_finalize(p_stmt);
		if (sqlrc != SQLITE_OK)
		{
			rc = DB_ERR_FAILURE;
		}
	}
	else
	{
		rc = DB_ERR_FAILURE;
	}
	return rc;
}
enum db_return_codes db_delete_boot_status_register_by_device_handle(const PersistentStore *p_ps,
	const unsigned int device_handle)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	sqlite3_stmt *p_stmt;
	char *sql = "DELETE FROM boot_status_register \
				 WHERE device_handle = $device_handle";
	if (SQLITE_PREPARE(p_ps->db, sql, p_stmt))
	{
		BIND_INTEGER(p_stmt, "$device_handle", (unsigned int)device_handle);
		if (sqlite3_step(p_stmt) == SQLITE_DONE)
		{
			rc = DB_SUCCESS;
		}
		sqlite3_finalize(p_stmt);
	}
	return rc;
}

#if 0
//NON-HISTORY TABLE

enum db_return_codes db_get_boot_status_register_history_by_history_id_count(const PersistentStore *p_ps, 
	int history_id,
	int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM boot_status_register_history WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
enum db_return_codes db_get_boot_status_register_history_count(const PersistentStore *p_ps, int *p_count)
{
	enum db_return_codes rc = DB_ERR_FAILURE;
	*p_count = 0;
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "select count(*) FROM boot_status_register_history");
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		if (sqlite3_step(p_stmt) == SQLITE_ROW)
		{
			*p_count = sqlite3_column_int(p_stmt, 0);
			rc = DB_SUCCESS;
		}
		// cleanup
		sqlite3_finalize(p_stmt);
	}
	return rc;
}
int db_get_boot_status_register_history_by_history_id(const PersistentStore *p_ps,
	struct db_boot_status_register *p_boot_status_register,
	int history_id,
	int boot_status_register_count)
{
	int rc = DB_ERR_FAILURE;
	memset(p_boot_status_register, 0, sizeof (struct db_boot_status_register) * boot_status_register_count);
	sqlite3_stmt *p_stmt;
	char buffer[1024];
	snprintf(buffer, 1024, "SELECT \
		device_handle,  bsr  \
		FROM boot_status_register_history \
		WHERE  history_id = '%d'", history_id);
	if (SQLITE_PREPARE(p_ps->db, buffer, p_stmt))
	{
		int index = 0;
		BIND_INTEGER(p_stmt, "$history_id", history_id);
		while (sqlite3_step(p_stmt) == SQLITE_ROW && index < boot_status_register_count)
		{
		rc = DB_SUCCESS;
		local_row_to_boot_status_register(p_ps, p_stmt, &p_boot_status_register[index]);
		local_get_boot_status_register_relationships_history(p_ps, p_stmt, &p_boot_status_register[index], history_id);
			index++;
		}
		sqlite3_finalize(p_stmt);
		rc = index;
	}
	return rc;
}
enum db_return_codes db_delete_boot_status_register_history(const PersistentStore *p_ps)
{
	return run_sql_no_results(p_ps->db, "DELETE FROM boot_status_register_history");
}

#endif

/*
 * --- END boot_status_register ----------------
 */
/*
 * Delete all histories
 */
enum db_return_codes db_clear_history(PersistentStore *p_ps)
{
	enum db_return_codes rc = DB_SUCCESS;
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM config_history"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM log_history"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM event_history"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM topology_state_history"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM host_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM sw_inventory_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM socket_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM runtime_config_validation_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM interleave_capability_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM platform_info_capability_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM platform_capabilities_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_capabilities_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_features_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_topology_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM namespace_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM identify_dimm_history"));
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM device_characteristics_history"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_smart_history"));
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_state_history"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM namespace_state_history"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_alarm_thresholds_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_power_management_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_die_sparing_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_optional_config_data_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_err_correction_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_erasure_coding_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_thermal_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_image_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_debug_log_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page0_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page1_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page2_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_ars_command_specific_data_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_long_op_status_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_details_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_security_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_sanitize_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_entry_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_entry_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_entry_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_entry_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_log_level_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_time_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_platform_config_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_current_config_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_input_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_output_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_change_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_interleave_set_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM interleave_set_dimm_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM enable_error_injection_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM temperature_error_injection_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM poison_error_injection_info_history"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM software_trigger_info_history"));
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM performance_history"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_metadata_check_diag_result_history"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM boot_status_register_history"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM history"));
	return rc;
}
/*
 * Delete all "simulation" tables and their history tables
 */
enum db_return_codes db_clear_state(PersistentStore *p_ps)
{
	enum db_return_codes rc = DB_SUCCESS;
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM config_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM config"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM log_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM log"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM event_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM event"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM topology_state_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM topology_state"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM host_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM host"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM sw_inventory_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM sw_inventory"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM socket_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM socket"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM runtime_config_validation_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM runtime_config_validation"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM interleave_capability_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM interleave_capability"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM platform_info_capability_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM platform_info_capability"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM platform_capabilities_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM platform_capabilities"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_capabilities_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_capabilities"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_features_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_features"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_topology_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_topology"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM namespace_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM namespace"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM identify_dimm_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM identify_dimm"));
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM device_characteristics_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM device_characteristics"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_smart_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_smart"));
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_state_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_state"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM namespace_state_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM namespace_state"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_alarm_thresholds_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_alarm_thresholds"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_power_management_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_power_management"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_die_sparing_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_die_sparing"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_optional_config_data_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_optional_config_data"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_err_correction_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_err_correction"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_erasure_coding_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_erasure_coding"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_thermal_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_thermal"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_image_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_image"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_debug_log_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_debug_log"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page0_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page0"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page1_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page1"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page2_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_memory_info_page2"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_ars_command_specific_data_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_ars_command_specific_data"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_long_op_status_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_long_op_status"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_details_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_details"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_security_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_security_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_sanitize_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_sanitize_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_entry_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_entry"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_entry_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_entry"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_entry_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_entry"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_entry_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_entry"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_low_log_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_media_high_log_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_low_log_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM fw_thermal_high_log_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_log_level_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_log_level"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_time_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_fw_time"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_platform_config_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_platform_config"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_current_config_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_current_config"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_input_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_input"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_output_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_config_output"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_change_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_partition_change"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_interleave_set_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM dimm_interleave_set"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM interleave_set_dimm_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM interleave_set_dimm_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM enable_error_injection_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM enable_error_injection_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM temperature_error_injection_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM temperature_error_injection_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM poison_error_injection_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM poison_error_injection_info"));
	
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM software_trigger_info_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM software_trigger_info"));
	
#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM performance_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM performance"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_metadata_check_diag_result_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM driver_metadata_check_diag_result"));
	
#endif

#if 0
//NON-HISTORY TABLE

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM boot_status_register_history"));
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM boot_status_register"));
	
#endif

	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, "DELETE FROM history"));
	return rc;
}
/*
 * Roll histories to a max count
 */
enum db_return_codes db_roll_history(PersistentStore *p_ps, int max)
{
	enum db_return_codes rc = DB_SUCCESS;
	db_begin_transaction(p_ps);
	char sql[1024];
	
#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM config_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM log_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM event_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM topology_state_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

	snprintf(sql, 1024,
				"DELETE FROM host_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM sw_inventory_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM socket_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM runtime_config_validation_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM interleave_capability_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM platform_info_capability_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM platform_capabilities_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM driver_capabilities_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM driver_features_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_topology_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM namespace_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM identify_dimm_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM device_characteristics_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

	snprintf(sql, 1024,
				"DELETE FROM dimm_partition_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_smart_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM dimm_state_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM namespace_state_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

	snprintf(sql, 1024,
				"DELETE FROM dimm_alarm_thresholds_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_power_management_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_die_sparing_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_optional_config_data_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_err_correction_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_erasure_coding_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_thermal_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_fw_image_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_fw_debug_log_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_memory_info_page0_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_memory_info_page1_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_memory_info_page2_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_ars_command_specific_data_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_long_op_status_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_details_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_security_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_sanitize_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_media_low_log_entry_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_media_high_log_entry_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_thermal_low_log_entry_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_thermal_high_log_entry_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_media_low_log_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_media_high_log_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_thermal_low_log_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM fw_thermal_high_log_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_fw_log_level_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_fw_time_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_platform_config_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_current_config_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_config_input_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_config_output_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_partition_change_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM dimm_interleave_set_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM interleave_set_dimm_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM enable_error_injection_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM temperature_error_injection_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM poison_error_injection_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
	snprintf(sql, 1024,
				"DELETE FROM software_trigger_info_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM performance_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM driver_metadata_check_diag_result_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

#if 0
//NON-HISTORY TABLE

	snprintf(sql, 1024,
				"DELETE FROM boot_status_register_history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	
#endif

	snprintf(sql, 1024,
				"DELETE FROM history "
				"WHERE history_id NOT IN "
				"(SELECT history_id FROM history ORDER BY ROWID DESC LIMIT %d)", max); 
	KEEP_DB_ERROR(rc, run_sql_no_results(p_ps->db, sql));
	db_end_transaction(p_ps);
	return rc;
}
#ifdef __cplusplus
}
#endif
