//! Cached exponents for basen values with 160-bit extended floats.
//!
//! Exact versions of base**n as an extended-precision float, with both
//! large and small powers. Use the large powers to minimize the amount
//! of compounded error.
//!
//! These values were calculated using Python, using the arbitrary-precision
//! integer to calculate exact extended-representation of each value.
//! These values are all normalized.
//!
//! This files takes ~ 44KB of storage.
//!
//! This file is mostly automatically generated, do not change values
//! manually, unless you know what you are doing. The script to generate
//! the values is as follows:
//!
//! ```text
//! import math
//! from collections import deque
//!
//! STEP_STR = "const BASE{0}_STEP: i32 = {1};"
//! SMALL_MANTISSA_STR = "const BASE{0}_SMALL_MANTISSA: [u128; {1}] = ["
//! SMALL_EXPONENT_STR = "const BASE{0}_SMALL_EXPONENT: [i32; {1}] = ["
//! LARGE_MANTISSA_STR = "const BASE{0}_LARGE_MANTISSA: [u128; {1}] = ["
//! LARGE_EXPONENT_STR = "const BASE{0}_LARGE_EXPONENT: [i32; {1}] = ["
//! SMALL_INT_STR = "const BASE{0}_SMALL_INT_POWERS: [u128; {1}] = {2};"
//! BIAS_STR = "const BASE{0}_BIAS: i32 = {1};"
//! EXP_STR = "// {}^{}"
//! POWER_STR = """pub(crate) const BASE{0}_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {{
//!     small: ExtendedFloatArray {{ mant: &BASE{0}_SMALL_MANTISSA, exp: &BASE{0}_SMALL_EXPONENT }},
//!     large: ExtendedFloatArray {{ mant: &BASE{0}_LARGE_MANTISSA, exp: &BASE{0}_LARGE_EXPONENT }},
//!     small_int: &BASE{0}_SMALL_INT_POWERS,
//!     step: BASE{0}_STEP,
//!     bias: BASE{0}_BIAS,
//! }};\n"""
//!
//! def calculate_bitshift(base, exponent):
//!     '''
//!     Calculate the bitshift required for a given base. The exponent
//!     is the absolute value of the max exponent (log distance from 1.)
//!     '''
//!
//!     return 127 + math.ceil(math.log2(base**exponent))
//!
//!
//! def next_fp(fp, base, step = 1):
//!     '''Generate the next extended-floating point value.'''
//!
//!     return (fp[0] * (base**step), fp[1])
//!
//!
//! def prev_fp(fp, base, step = 1):
//!     '''Generate the previous extended-floating point value.'''
//!
//!     return (fp[0] // (base**step), fp[1])
//!
//!
//! def normalize_fp(fp):
//!     '''Normalize a extended-float so the MSB is the 128th bit'''
//!
//!     while fp[0] >> 128 != 0:
//!         fp = (fp[0] >> 1, fp[1] + 1)
//!     return fp
//!
//!
//! def generate_small(base, count):
//!     '''Generate the small powers for a given base'''
//!
//!     bitshift = calculate_bitshift(base, count)
//!     fps = []
//!     fp = (1 << bitshift, -bitshift)
//!     for exp in range(count):
//!         fps.append((normalize_fp(fp), exp))
//!         fp = next_fp(fp, base)
//!
//!     # Print the small powers as integers.
//!     ints = [base**i for _, i in fps]
//!
//!     return fps, ints
//!
//!
//! def generate_large(base, step):
//!     '''Generate the large powers for a given base.'''
//!
//!     # Get our starting parameters
//!     min_exp = math.floor(math.log(5e-324, base) - math.log(0xFFFFFFFFFFFFFFFF, base))
//!     max_exp = math.ceil(math.log(1.7976931348623157e+308, base))
//!     bitshift = calculate_bitshift(base, abs(min_exp - step))
//!     fps = deque()
//!
//!     # Add negative exponents
//!     # We need to go below the minimum exponent, since we need
//!     # all resulting exponents to be positive.
//!     fp = (1 << bitshift, -bitshift)
//!     for exp in range(-step, min_exp-step, -step):
//!         fp = prev_fp(fp, base, step)
//!         fps.appendleft((normalize_fp(fp), exp))
//!
//!     # Add positive exponents
//!     fp = (1 << bitshift, -bitshift)
//!     fps.append((normalize_fp(fp), 0))
//!     for exp in range(step, max_exp, step):
//!         fp = next_fp(fp, base, step)
//!         fps.append((normalize_fp(fp), exp))
//!
//!     # Return the smallest exp, AKA, the bias
//!     return fps, -fps[0][1]
//!
//!
//! def print_array(base, string, fps, index):
//!     '''Print an entire array'''
//!
//!     print(string.format(base, len(fps)))
//!     for fp, exp in fps:
//!         value = "    {},".format(fp[index])
//!         exp = EXP_STR.format(base, exp)
//!         print(value.ljust(56, " ") + exp)
//!     print("];")
//!
//!
//! def generate_base(base):
//!     '''Generate all powers and variables.'''
//!
//!     step = math.floor(math.log(1e10, base))
//!     small, ints = generate_small(base, step)
//!     large, bias = generate_large(base, step)
//!
//!     print_array(base, SMALL_MANTISSA_STR, small, 0)
//!     print_array(base, SMALL_EXPONENT_STR, small, 1)
//!     print_array(base, LARGE_MANTISSA_STR, large, 0)
//!     print_array(base, LARGE_EXPONENT_STR, large, 1)
//!     print(SMALL_INT_STR.format(base, len(ints), ints))
//!     print(STEP_STR.format(base, step))
//!     print(BIAS_STR.format(base, bias))
//!
//!
//! def generate():
//!     '''Generate all bases.'''
//!
//!     bases = [
//!         3, 5, 6, 7, 9, 10, 11, 12, 13, 14, 15, 17, 18, 19, 20, 21,
//!         22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 33, 34, 35, 36
//!     ]
//!
//!     for base in bases:
//!         print("// BASE{}\n".format(base))
//!         generate_base(base)
//!         print("")
//!
//!     print("// HIGH LEVEL\n// ----------\n")
//!
//!     for base in bases:
//!         print(POWER_STR.format(base))
//!
//!
//! if __name__ == '__main__':
//!     generate()
//! ```

use crate::util::*;
use super::cached::{ExtendedFloatArray, ModeratePathPowers};

// LOW-LEVEL
// ---------

// BASE10

const BASE10_SMALL_MANTISSA: [u128; 10] = [
    170141183460469231731687303715884105728,            // 10^0
    212676479325586539664609129644855132160,            // 10^1
    265845599156983174580761412056068915200,            // 10^2
    332306998946228968225951765070086144000,            // 10^3
    207691874341393105141219853168803840000,            // 10^4
    259614842926741381426524816461004800000,            // 10^5
    324518553658426726783156020576256000000,            // 10^6
    202824096036516704239472512860160000000,            // 10^7
    253530120045645880299340641075200000000,            // 10^8
    316912650057057350374175801344000000000,            // 10^9
];
const BASE10_SMALL_EXPONENT: [i32; 10] = [
    -127,                                               // 10^0
    -124,                                               // 10^1
    -121,                                               // 10^2
    -118,                                               // 10^3
    -114,                                               // 10^4
    -111,                                               // 10^5
    -108,                                               // 10^6
    -104,                                               // 10^7
    -101,                                               // 10^8
    -98,                                                // 10^9
];
const BASE10_LARGE_MANTISSA: [u128; 66] = [
    213154451346726893197828921904416471830,            // 10^-350
    248144440523729302452212341484167231049,            // 10^-340
    288878149031346317441449898160257412877,            // 10^-330
    336298426882534191759128470626028036788,            // 10^-320
    195751447977110622310503659901458325789,            // 10^-310
    227884678143438210606695688214919443462,            // 10^-300
    265292681454958173686982700851419292695,            // 10^-290
    308841328899094571460716776609676066664,            // 10^-280
    179769313486231590772930519078902473361,            // 10^-270
    209279024841067836122739267394531603625,            // 10^-260
    243632850284999977008834559696879707771,            // 10^-250
    283625966735416996535885333662014114404,            // 10^-240
    330184081959790778970212365572822879074,            // 10^-230
    192192430817400325887261637005036975649,            // 10^-220
    223741436863085634409521749481834675708,            // 10^-210
    260469313784369307581244210575049132700,            // 10^-200
    303226189902482213896285056340332530323,            // 10^-190
    176500872419263593559319302637789241459,            // 10^-180
    205474058654233340126601167300005025998,            // 10^-170
    239203286653190548679094257880939433814,            // 10^-160
    278469275977917188637766821636980671685,            // 10^-150
    324180903818827574883781864350871964922,            // 10^-140
    188698121241077067612077729049413444545,            // 10^-130
    219673525124179510879420825570604582952,            // 10^-120
    255733641241886083594780445064656183766,            // 10^-110
    297713141471480582369003031710926657271,            // 10^-100
    173291855882550928723650886508942731464,            // 10^-90
    201738271725539733566868685312735302682,            // 10^-80
    234854258277383322788948059678933702737,            // 10^-70
    273406340597876490546562778389702670669,            // 10^-60
    318286871302263450979444638813965337664,            // 10^-50
    185267342779705912677713576013900652565,            // 10^-40
    215679573337205118357336120696157045389,            // 10^-30
    251084069415467230553431576928306656644,            // 10^-20
    292300327466180583640736966543256603931,            // 10^-10
    170141183460469231731687303715884105728,            // 10^0
    198070406285660843983859875840000000000,            // 10^10
    230584300921369395200000000000000000000,            // 10^20
    268435456000000000000000000000000000000,            // 10^30
    312500000000000000000000000000000000000,            // 10^40
    181898940354585647583007812500000000000,            // 10^50
    211758236813575084767080625169910490512,            // 10^60
    246519032881566189191165176650870696772,            // 10^70
    286985925493722536125179818657774823686,            // 10^80
    334095588761524455767567058393935234851,            // 10^90
    194469227433160678348252001680628882518,            // 10^100
    226391976970667809187727982272194794517,            // 10^110
    263554948580763080608714351281750475192,            // 10^120
    306818341581107909568485747186642227685,            // 10^130
    178591779887855465971216179422709524914,            // 10^140
    207908195312897984370608091613638127355,            // 10^150
    242036994678082392051126914580396990473,            // 10^160
    281768146294730706199918541335962934504,            // 10^170
    328021294314799255458543241647960309061,            // 10^180
    190933522718725292628248712075851106236,            // 10^190
    222275874948507748344271341427056009691,            // 10^200
    258763175164940474024358370140027266101,            // 10^210
    301239983137860514717593754339063617053,            // 10^220
    175344747920672243180215448571289666610,            // 10^230
    204128152598478183127259193653345185577,            // 10^240
    237636445786894977939384050729387888658,            // 10^250
    276645233140903266541874095249674153349,            // 10^260
    322057438479856665411351825168442625260,            // 10^270
    187462101736953869352205554703508169192,            // 10^280
    218234609040610805796698614376955862613,            // 10^290
    254058522452380049271391022923583936195,            // 10^300
];
const BASE10_LARGE_EXPONENT: [i32; 66] = [
    -1290,                                              // 10^-350
    -1257,                                              // 10^-340
    -1224,                                              // 10^-330
    -1191,                                              // 10^-320
    -1157,                                              // 10^-310
    -1124,                                              // 10^-300
    -1091,                                              // 10^-290
    -1058,                                              // 10^-280
    -1024,                                              // 10^-270
    -991,                                               // 10^-260
    -958,                                               // 10^-250
    -925,                                               // 10^-240
    -892,                                               // 10^-230
    -858,                                               // 10^-220
    -825,                                               // 10^-210
    -792,                                               // 10^-200
    -759,                                               // 10^-190
    -725,                                               // 10^-180
    -692,                                               // 10^-170
    -659,                                               // 10^-160
    -626,                                               // 10^-150
    -593,                                               // 10^-140
    -559,                                               // 10^-130
    -526,                                               // 10^-120
    -493,                                               // 10^-110
    -460,                                               // 10^-100
    -426,                                               // 10^-90
    -393,                                               // 10^-80
    -360,                                               // 10^-70
    -327,                                               // 10^-60
    -294,                                               // 10^-50
    -260,                                               // 10^-40
    -227,                                               // 10^-30
    -194,                                               // 10^-20
    -161,                                               // 10^-10
    -127,                                               // 10^0
    -94,                                                // 10^10
    -61,                                                // 10^20
    -28,                                                // 10^30
    5,                                                  // 10^40
    39,                                                 // 10^50
    72,                                                 // 10^60
    105,                                                // 10^70
    138,                                                // 10^80
    171,                                                // 10^90
    205,                                                // 10^100
    238,                                                // 10^110
    271,                                                // 10^120
    304,                                                // 10^130
    338,                                                // 10^140
    371,                                                // 10^150
    404,                                                // 10^160
    437,                                                // 10^170
    470,                                                // 10^180
    504,                                                // 10^190
    537,                                                // 10^200
    570,                                                // 10^210
    603,                                                // 10^220
    637,                                                // 10^230
    670,                                                // 10^240
    703,                                                // 10^250
    736,                                                // 10^260
    769,                                                // 10^270
    803,                                                // 10^280
    836,                                                // 10^290
    869,                                                // 10^300
];
const BASE10_SMALL_INT_POWERS: [u128; 10] = [1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000];
const BASE10_STEP: i32 = 10;
const BASE10_BIAS: i32 = 350;

cfg_if! {
if #[cfg(feature = "radix")] {
// BASE3

const BASE3_SMALL_MANTISSA: [u128; 20] = [
    170141183460469231731687303715884105728,            // 3^0
    255211775190703847597530955573826158592,            // 3^1
    191408831393027885698148216680369618944,            // 3^2
    287113247089541828547222325020554428416,            // 3^3
    215334935317156371410416743765415821312,            // 3^4
    323002402975734557115625115648123731968,            // 3^5
    242251802231800917836718836736092798976,            // 3^6
    181688851673850688377539127552069599232,            // 3^7
    272533277510776032566308691328104398848,            // 3^8
    204399958133082024424731518496078299136,            // 3^9
    306599937199623036637097277744117448704,            // 3^10
    229949952899717277477822958308088086528,            // 3^11
    172462464674787958108367218731066064896,            // 3^12
    258693697012181937162550828096599097344,            // 3^13
    194020272759136452871913121072449323008,            // 3^14
    291030409138704679307869681608673984512,            // 3^15
    218272806854028509480902261206505488384,            // 3^16
    327409210281042764221353391809758232576,            // 3^17
    245556907710782073166015043857318674432,            // 3^18
    184167680783086554874511282892989005824,            // 3^19
];
const BASE3_SMALL_EXPONENT: [i32; 20] = [
    -127,                                               // 3^0
    -126,                                               // 3^1
    -124,                                               // 3^2
    -123,                                               // 3^3
    -121,                                               // 3^4
    -120,                                               // 3^5
    -118,                                               // 3^6
    -116,                                               // 3^7
    -115,                                               // 3^8
    -113,                                               // 3^9
    -112,                                               // 3^10
    -110,                                               // 3^11
    -108,                                               // 3^12
    -107,                                               // 3^13
    -105,                                               // 3^14
    -104,                                               // 3^15
    -102,                                               // 3^16
    -101,                                               // 3^17
    -99,                                                // 3^18
    -97,                                                // 3^19
];
const BASE3_LARGE_MANTISSA: [u128; 69] = [
    301829093537629265639465570217176944359,            // 3^-720
    245033990385703656345786023933864839340,            // 3^-700
    198926007233479871031630637668169238011,            // 3^-680
    322988302900880006728964617948539328448,            // 3^-660
    262211676747596696167096696967233799204,            // 3^-640
    212871372756449173771443137071089544143,            // 3^-620
    172815421118085121562612771428651141606,            // 3^-600
    280593575260967566098415738074481154338,            // 3^-580
    227794354139073103116567345878808448350,            // 3^-560
    184930348919702200346046943747485274024,            // 3^-540
    300264105147079021545114594266031000970,            // 3^-520
    243763485459391712918376663011554847091,            // 3^-500
    197894572893436379626501802082900685163,            // 3^-480
    321313603691473325606249593990411338331,            // 3^-460
    260852105259086286749566195634740776863,            // 3^-440
    211767631486382365261996259087726574961,            // 3^-420
    171919370559843833352674924374427532806,            // 3^-400
    279138693352137745884317186629683060895,            // 3^-380
    226613236986043931067161987739751269180,            // 3^-360
    183971482278558945643179980616811190964,            // 3^-340
    298707231244876640116631457791747347925,            // 3^-320
    242499568120235502703106353919523432682,            // 3^-300
    196868486555962367745019627988939060464,            // 3^-280
    319647587822660709450189016904055940251,            // 3^-260
    259499583169196479959998361450291137700,            // 3^-240
    210669613131404954481085620515615417585,            // 3^-220
    171027966037226738058674240289082148799,            // 3^-200
    277691355027891684120101092281051616669,            // 3^-180
    225438243943221318560556485110109738278,            // 3^-160
    183017587375374702561553597022155160742,            // 3^-140
    297158429757277967604640789526650060843,            // 3^-120
    241242204211496523037749538228345943134,            // 3^-100
    195847720491584060106836777189641681162,            // 3^-80
    317990210271190550439415903835536554761,            // 3^-60
    258154073926689380540223575440483383976,            // 3^-40
    209577288018116110386327960504760073299,            // 3^-20
    170141183460469231731687303715884105728,            // 3^0
    276251521174629832311766924339483508736,            // 3^20
    224269343257001716702690972139746492416,            // 3^40
    182068638431613361423174859113151594496,            // 3^60
    295617658828691846632166420412766595202,            // 3^80
    239991359753539474232337032335634004651,            // 3^100
    194832247114605420104007752175098574688,            // 3^120
    316341426247257477645159711999449660471,            // 3^140
    256815541169845811576524073480007610450,            // 3^160
    208490626626972031635281014538153149532,            // 3^180
    338517997729425004575949331160209430911,            // 3^200
    274819152881557244028610584245948464515,            // 3^220
    223106503338424488684979682521025988628,            // 3^240
    181124609802400910077427551154104473922,            // 3^260
    294084876820548989626661915132664622178,            // 3^280
    238747000942913976797497733353022683918,            // 3^300
    193822038982362660063056049982127016523,            // 3^320
    314701191193291934781116205950433765545,            // 3^340
    255483948725482657093998355855298189652,            // 3^360
    207409599591488195571905341445445255582,            // 3^380
    336762776818711782198286065086981891498,            // 3^400
    273394211439632029990640781047045990695,            // 3^420
    221949692762318233808346663450192754968,            // 3^440
    180185475975832393914650652957737664335,            // 3^460
    292560042310176717160312096633717510967,            // 3^480
    237509094151441049982785534773499431992,            // 3^500
    192817068794482616882547252154136283242,            // 3^520
    313069460782756034010893203297842312622,            // 3^540
    254159260607975299744356396919078736707,            // 3^560
    206334177697445743564032291193028958152,            // 3^580
    335016656754825225194410391893304442626,            // 3^600
    271976658340519265432186039827268288213,            // 3^620
    220798880266451537830039115389735391778,            // 3^640
];
const BASE3_LARGE_EXPONENT: [i32; 69] = [
    -1269,                                              // 3^-720
    -1237,                                              // 3^-700
    -1205,                                              // 3^-680
    -1174,                                              // 3^-660
    -1142,                                              // 3^-640
    -1110,                                              // 3^-620
    -1078,                                              // 3^-600
    -1047,                                              // 3^-580
    -1015,                                              // 3^-560
    -983,                                               // 3^-540
    -952,                                               // 3^-520
    -920,                                               // 3^-500
    -888,                                               // 3^-480
    -857,                                               // 3^-460
    -825,                                               // 3^-440
    -793,                                               // 3^-420
    -761,                                               // 3^-400
    -730,                                               // 3^-380
    -698,                                               // 3^-360
    -666,                                               // 3^-340
    -635,                                               // 3^-320
    -603,                                               // 3^-300
    -571,                                               // 3^-280
    -540,                                               // 3^-260
    -508,                                               // 3^-240
    -476,                                               // 3^-220
    -444,                                               // 3^-200
    -413,                                               // 3^-180
    -381,                                               // 3^-160
    -349,                                               // 3^-140
    -318,                                               // 3^-120
    -286,                                               // 3^-100
    -254,                                               // 3^-80
    -223,                                               // 3^-60
    -191,                                               // 3^-40
    -159,                                               // 3^-20
    -127,                                               // 3^0
    -96,                                                // 3^20
    -64,                                                // 3^40
    -32,                                                // 3^60
    -1,                                                 // 3^80
    31,                                                 // 3^100
    63,                                                 // 3^120
    94,                                                 // 3^140
    126,                                                // 3^160
    158,                                                // 3^180
    189,                                                // 3^200
    221,                                                // 3^220
    253,                                                // 3^240
    285,                                                // 3^260
    316,                                                // 3^280
    348,                                                // 3^300
    380,                                                // 3^320
    411,                                                // 3^340
    443,                                                // 3^360
    475,                                                // 3^380
    506,                                                // 3^400
    538,                                                // 3^420
    570,                                                // 3^440
    602,                                                // 3^460
    633,                                                // 3^480
    665,                                                // 3^500
    697,                                                // 3^520
    728,                                                // 3^540
    760,                                                // 3^560
    792,                                                // 3^580
    823,                                                // 3^600
    855,                                                // 3^620
    887,                                                // 3^640
];
const BASE3_SMALL_INT_POWERS: [u128; 20] = [1, 3, 9, 27, 81, 243, 729, 2187, 6561, 19683, 59049, 177147, 531441, 1594323, 4782969, 14348907, 43046721, 129140163, 387420489, 1162261467];
const BASE3_STEP: i32 = 20;
const BASE3_BIAS: i32 = 720;

// BASE5

const BASE5_SMALL_MANTISSA: [u128; 14] = [
    170141183460469231731687303715884105728,            // 5^0
    212676479325586539664609129644855132160,            // 5^1
    265845599156983174580761412056068915200,            // 5^2
    332306998946228968225951765070086144000,            // 5^3
    207691874341393105141219853168803840000,            // 5^4
    259614842926741381426524816461004800000,            // 5^5
    324518553658426726783156020576256000000,            // 5^6
    202824096036516704239472512860160000000,            // 5^7
    253530120045645880299340641075200000000,            // 5^8
    316912650057057350374175801344000000000,            // 5^9
    198070406285660843983859875840000000000,            // 5^10
    247588007857076054979824844800000000000,            // 5^11
    309485009821345068724781056000000000000,            // 5^12
    193428131138340667952988160000000000000,            // 5^13
];
const BASE5_SMALL_EXPONENT: [i32; 14] = [
    -127,                                               // 5^0
    -125,                                               // 5^1
    -123,                                               // 5^2
    -121,                                               // 5^3
    -118,                                               // 5^4
    -116,                                               // 5^5
    -114,                                               // 5^6
    -111,                                               // 5^7
    -109,                                               // 5^8
    -107,                                               // 5^9
    -104,                                               // 5^10
    -102,                                               // 5^11
    -100,                                               // 5^12
    -97,                                                // 5^13
];
const BASE5_LARGE_MANTISSA: [u128; 68] = [
    285793394306920833441610418092098634655,            // 5^-504
    203068420253004570555511362849258201390,            // 5^-490
    288577581746103207017755725657449092679,            // 5^-476
    205046704412910121830119952091883627559,            // 5^-462
    291388892624283530821742192659774598780,            // 5^-448
    207044260935364498850036477975162511299,            // 5^-434
    294227591176883860910658765384315687611,            // 5^-420
    209061277570927374050781655074839937648,            // 5^-406
    297093944213496817569054052050375869453,            // 5^-392
    211097943899216614887176072592734406508,            // 5^-378
    299988221142963048588365030287739055137,            // 5^-364
    213154451346726893197828921904416471830,            // 5^-350
    302910693998692996157485768413290076965,            // 5^-336
    215230993204821882725842221200657943544,            // 5^-322
    305861637464235347360161968596028634045,            // 5^-308
    217327764647901735884376228537482684576,            // 5^-294
    308841328899094571460716776609676066664,            // 5^-280
    219444962751747547330237450047488370802,            // 5^-266
    311850048364799970571308236412006025948,            // 5^-252
    221582786512044528543660416923448526878,            // 5^-238
    314888078651228693933689466069052580904,            // 5^-224
    223741436863085634409521749481834675708,            // 5^-210
    317955705303185189918510999237120523316,            // 5^-196
    225921116696657399755928707376370229068,            // 5^-182
    321053216647239593947814323906257853121,            // 5^-168
    228122030881109760932058580285014566244,            // 5^-154
    324180903818827574883781864350871964922,            // 5^-140
    230344386280611654799899571593522271174,            // 5^-126
    327339060789614187001318969682759915221,            // 5^-112
    232588391774594204975783618524161450993,            // 5^-98
    330527984395124299475957654016385519914,            // 5^-84
    234854258277383322788948059678933702737,            // 5^-70
    333747974362642200374222141588992517906,            // 5^-56
    237142198758023568227473377297792835283,            // 5^-42
    336999333339382997433337688587745383420,            // 5^-28
    239452428260295134118491722992235809940,            // 5^-14
    170141183460469231731687303715884105728,            // 5^0
    241785163922925834941235200000000000000,            // 5^14
    171798691840000000000000000000000000000,            // 5^28
    244140625000000000000000000000000000000,            // 5^42
    173472347597680709441192448139190673828,            // 5^56
    246519032881566189191165176650870696772,            // 5^70
    175162308040602133865466197911239516410,            // 5^84
    248920611114445668285762562151204969623,            // 5^98
    176868732008334225927912486150152183216,            // 5^112
    251345585423243599518503524095297312920,            // 5^126
    178591779887855465971216179422709524914,            // 5^140
    253794183731564922327402455583054354682,            // 5^154
    180331613628627651967947866455016278082,            // 5^168
    256266636183436918326986907537468991453,            // 5^182
    182088396757817547443627082897044283139,            // 5^196
    258763175164940474024358370140027266101,            // 5^210
    183862294395666818064937594201088633455,            // 5^224
    261284035326052074402891767876281837538,            // 5^238
    185653473271011701515143789632334288014,            // 5^252
    263829453602698580304979415177988198613,            // 5^266
    187462101736953869352205554703508169192,            // 5^280
    266399669239026862544798113253119949479,            // 5^294
    189288349786683953755640255602884245064,            // 5^308
    268994923809890385876486015494726082500,            // 5^322
    191132389069459226417170338759437756337,            // 5^336
    271615461243554856334256923502490730495,            // 5^350
    192994392906736931318972184714148973580,            // 5^364
    274261527844625066050770363850331497104,            // 5^378
    194874536308464787773268059716493991903,            // 5^392
    276933372317195090450451374005771742621,            // 5^406
    196772995989530194869453349330805553038,            // 5^420
    279631245788224013707368483964622716141,            // 5^434
];
const BASE5_LARGE_EXPONENT: [i32; 68] = [
    -1298,                                              // 5^-504
    -1265,                                              // 5^-490
    -1233,                                              // 5^-476
    -1200,                                              // 5^-462
    -1168,                                              // 5^-448
    -1135,                                              // 5^-434
    -1103,                                              // 5^-420
    -1070,                                              // 5^-406
    -1038,                                              // 5^-392
    -1005,                                              // 5^-378
    -973,                                               // 5^-364
    -940,                                               // 5^-350
    -908,                                               // 5^-336
    -875,                                               // 5^-322
    -843,                                               // 5^-308
    -810,                                               // 5^-294
    -778,                                               // 5^-280
    -745,                                               // 5^-266
    -713,                                               // 5^-252
    -680,                                               // 5^-238
    -648,                                               // 5^-224
    -615,                                               // 5^-210
    -583,                                               // 5^-196
    -550,                                               // 5^-182
    -518,                                               // 5^-168
    -485,                                               // 5^-154
    -453,                                               // 5^-140
    -420,                                               // 5^-126
    -388,                                               // 5^-112
    -355,                                               // 5^-98
    -323,                                               // 5^-84
    -290,                                               // 5^-70
    -258,                                               // 5^-56
    -225,                                               // 5^-42
    -193,                                               // 5^-28
    -160,                                               // 5^-14
    -127,                                               // 5^0
    -95,                                                // 5^14
    -62,                                                // 5^28
    -30,                                                // 5^42
    3,                                                  // 5^56
    35,                                                 // 5^70
    68,                                                 // 5^84
    100,                                                // 5^98
    133,                                                // 5^112
    165,                                                // 5^126
    198,                                                // 5^140
    230,                                                // 5^154
    263,                                                // 5^168
    295,                                                // 5^182
    328,                                                // 5^196
    360,                                                // 5^210
    393,                                                // 5^224
    425,                                                // 5^238
    458,                                                // 5^252
    490,                                                // 5^266
    523,                                                // 5^280
    555,                                                // 5^294
    588,                                                // 5^308
    620,                                                // 5^322
    653,                                                // 5^336
    685,                                                // 5^350
    718,                                                // 5^364
    750,                                                // 5^378
    783,                                                // 5^392
    815,                                                // 5^406
    848,                                                // 5^420
    880,                                                // 5^434
];
const BASE5_SMALL_INT_POWERS: [u128; 14] = [1, 5, 25, 125, 625, 3125, 15625, 78125, 390625, 1953125, 9765625, 48828125, 244140625, 1220703125];
const BASE5_STEP: i32 = 14;
const BASE5_BIAS: i32 = 504;

// BASE6

const BASE6_SMALL_MANTISSA: [u128; 12] = [
    170141183460469231731687303715884105728,            // 6^0
    255211775190703847597530955573826158592,            // 6^1
    191408831393027885698148216680369618944,            // 6^2
    287113247089541828547222325020554428416,            // 6^3
    215334935317156371410416743765415821312,            // 6^4
    323002402975734557115625115648123731968,            // 6^5
    242251802231800917836718836736092798976,            // 6^6
    181688851673850688377539127552069599232,            // 6^7
    272533277510776032566308691328104398848,            // 6^8
    204399958133082024424731518496078299136,            // 6^9
    306599937199623036637097277744117448704,            // 6^10
    229949952899717277477822958308088086528,            // 6^11
];
const BASE6_SMALL_EXPONENT: [i32; 12] = [
    -127,                                               // 6^0
    -125,                                               // 6^1
    -122,                                               // 6^2
    -120,                                               // 6^3
    -117,                                               // 6^4
    -115,                                               // 6^5
    -112,                                               // 6^6
    -109,                                               // 6^7
    -107,                                               // 6^8
    -104,                                               // 6^9
    -102,                                               // 6^10
    -99,                                                // 6^11
];
const BASE6_LARGE_MANTISSA: [u128; 71] = [
    206105367118290399407064648402758144682,            // 6^-444
    208917317212507950117664039252872831665,            // 6^-432
    211767631486382365261996259087726574961,            // 6^-420
    214656833352574406771088703014069554755,            // 6^-408
    217585453364802351586979201161384846208,            // 6^-396
    220554029315269330081435801781477974040,            // 6^-384
    223563106333419891448609016293621894840,            // 6^-372
    226613236986043931067161987739751269180,            // 6^-360
    229704981378746362247969882824709232796,            // 6^-348
    232838907258801165579649662968151663564,            // 6^-336
    236015590119408703302029793810763336632,            // 6^-324
    239235613305375443823879271798297650114,            // 6^-312
    242499568120235502703106353919523432682,            // 6^-300
    245808053934833671173174941698733239342,            // 6^-288
    249161678297389871677290466673500998400,            // 6^-276
    252561057045065251911260457800735557729,            // 6^-264
    256006814417050404626793229969178591795,            // 6^-252
    259499583169196479959998361450291137700,            // 6^-240
    263040004690210240376322725691803307553,            // 6^-228
    266628729119434395515123988465075762881,            // 6^-216
    270266415466234845327287688358055741312,            // 6^-204
    273953731731016754981191818978678705632,            // 6^-192
    277691355027891684120101092281051616669,            // 6^-180
    281479971709018296242657937208050445965,            // 6^-168
    285320277490639481303204301467482637509,            // 6^-156
    289212977580839036146652597763405686112,            // 6^-144
    293158786809041363160730749526943361727,            // 6^-132
    297158429757277967604640789526650060843,            // 6^-120
    301212640893244858516269504216828222245,            // 6^-108
    305322164705175286969651759320250334279,            // 6^-96
    309487755838552588810803796052767101096,            // 6^-84
    313710179234688236904530296665341569850,            // 6^-72
    317990210271190550439415903835536554761,            // 6^-60
    322328634904349856025836233807108654402,            // 6^-48
    326726249813466247246220462666861782844,            // 6^-36
    331183862547146446042592332649497399781,            // 6^-24
    335702291671596630919115661345637412333,            // 6^-12
    170141183460469231731687303715884105728,            // 6^0
    172462464674787958108367218731066064896,            // 6^12
    174815415743320440759790006808579407872,            // 6^24
    177200468746272961345336076752392290304,            // 6^36
    179618061658836457920697688990341398528,            // 6^48
    182068638431613361423174859113151594496,            // 6^60
    184552649072141716781794491390137475072,            // 6^72
    187070549727531559196917812917453861026,            // 6^84
    189622802768228720381105803326920695033,            // 6^96
    192209876872921446586714266254161951235,            // 6^108
    194832247114605420104007752175098574688,            // 6^120
    197490395047822988635051696441052554380,            // 6^132
    200184808797092622572327630249651738267,            // 6^144
    202915983146544838776512848181734408257,            // 6^156
    205684419630781050995309380627725821797,            // 6^168
    208490626626972031635281014538153149532,            // 6^180
    211335119448212897232599978727666183358,            // 6^192
    214218420438151760708217936124820030498,            // 6^204
    217141059066909427380630585083218539864,            // 6^216
    220103572028307748788051030668660629356,            // 6^228
    223106503338424488684979682521025988628,            // 6^240
    226150404435492799169987273137391228527,            // 6^252
    229235834281163651816744244429413474808,            // 6^264
    232363359463149818964276081092475750857,            // 6^276
    235533554299270254021060647605641184828,            // 6^288
    238747000942913976797497733353022683918,            // 6^300
    242004289489942830549695955106475311593,            // 6^312
    245306018087052741642305313258629505287,            // 6^324
    248652793041613380567795520750960012282,            // 6^336
    252045228933006394543323172270604972624,            // 6^348
    255483948725482657093998355855298189652,            // 6^360
    258969583882559258973487053363982248701,            // 6^372
    262502774482977247520692697766891651596,            // 6^384
    266084169338241408156670471179837543899,            // 6^396
];
const BASE6_LARGE_EXPONENT: [i32; 71] = [
    -1275,                                              // 6^-444
    -1244,                                              // 6^-432
    -1213,                                              // 6^-420
    -1182,                                              // 6^-408
    -1151,                                              // 6^-396
    -1120,                                              // 6^-384
    -1089,                                              // 6^-372
    -1058,                                              // 6^-360
    -1027,                                              // 6^-348
    -996,                                               // 6^-336
    -965,                                               // 6^-324
    -934,                                               // 6^-312
    -903,                                               // 6^-300
    -872,                                               // 6^-288
    -841,                                               // 6^-276
    -810,                                               // 6^-264
    -779,                                               // 6^-252
    -748,                                               // 6^-240
    -717,                                               // 6^-228
    -686,                                               // 6^-216
    -655,                                               // 6^-204
    -624,                                               // 6^-192
    -593,                                               // 6^-180
    -562,                                               // 6^-168
    -531,                                               // 6^-156
    -500,                                               // 6^-144
    -469,                                               // 6^-132
    -438,                                               // 6^-120
    -407,                                               // 6^-108
    -376,                                               // 6^-96
    -345,                                               // 6^-84
    -314,                                               // 6^-72
    -283,                                               // 6^-60
    -252,                                               // 6^-48
    -221,                                               // 6^-36
    -190,                                               // 6^-24
    -159,                                               // 6^-12
    -127,                                               // 6^0
    -96,                                                // 6^12
    -65,                                                // 6^24
    -34,                                                // 6^36
    -3,                                                 // 6^48
    28,                                                 // 6^60
    59,                                                 // 6^72
    90,                                                 // 6^84
    121,                                                // 6^96
    152,                                                // 6^108
    183,                                                // 6^120
    214,                                                // 6^132
    245,                                                // 6^144
    276,                                                // 6^156
    307,                                                // 6^168
    338,                                                // 6^180
    369,                                                // 6^192
    400,                                                // 6^204
    431,                                                // 6^216
    462,                                                // 6^228
    493,                                                // 6^240
    524,                                                // 6^252
    555,                                                // 6^264
    586,                                                // 6^276
    617,                                                // 6^288
    648,                                                // 6^300
    679,                                                // 6^312
    710,                                                // 6^324
    741,                                                // 6^336
    772,                                                // 6^348
    803,                                                // 6^360
    834,                                                // 6^372
    865,                                                // 6^384
    896,                                                // 6^396
];
const BASE6_SMALL_INT_POWERS: [u128; 12] = [1, 6, 36, 216, 1296, 7776, 46656, 279936, 1679616, 10077696, 60466176, 362797056];
const BASE6_STEP: i32 = 12;
const BASE6_BIAS: i32 = 444;

// BASE7

const BASE7_SMALL_MANTISSA: [u128; 11] = [
    170141183460469231731687303715884105728,            // 7^0
    297747071055821155530452781502797185024,            // 7^1
    260528687173843511089146183814947536896,            // 7^2
    227962601277113072203002910838079094784,            // 7^3
    199467276117473938177627546983319207936,            // 7^4
    174533866602789695905424103610404306944,            // 7^5
    305434266554881967834492181318207537152,            // 7^6
    267254983235521721855180658653431595008,            // 7^7
    233848110331081506623283076321752645632,            // 7^8
    204617096539696318295372691781533564928,            // 7^9
    179039959472234278508451105308841869312,            // 7^10
];
const BASE7_SMALL_EXPONENT: [i32; 11] = [
    -127,                                               // 7^0
    -125,                                               // 7^1
    -122,                                               // 7^2
    -119,                                               // 7^3
    -116,                                               // 7^4
    -113,                                               // 7^5
    -111,                                               // 7^6
    -108,                                               // 7^7
    -105,                                               // 7^8
    -102,                                               // 7^9
    -99,                                                // 7^10
];
const BASE7_LARGE_MANTISSA: [u128; 71] = [
    225523710066019267908218449093892378358,            // 7^-407
    207654229874777697952558290622819921790,            // 7^-396
    191200646585071218824745087565327043379,            // 7^-385
    176050770921424471559828841966017113260,            // 7^-374
    324202605959679334675793064499391555083,            // 7^-363
    298514255748300407155832676761118149894,            // 7^-352
    274861334384351909098688755274300249465,            // 7^-341
    253082563679127241756740166823033329043,            // 7^-330
    233029444399168140266755890269514708731,            // 7^-319
    214565243719567885018434177936373702812,            // 7^-308
    197564062906901525077828647577582055243,            // 7^-297
    181909978874749832229630901118123515094,            // 7^-286
    334992507516972618831705765612473924060,            // 7^-275
    308449214239576126269380201889590576494,            // 7^-264
    284009091636748185128341566413187692045,            // 7^-253
    261505493963360706349361998265286260917,            // 7^-242
    240784979730461811097321743219099069136,            // 7^-231
    221706265459654420066135217688280855198,            // 7^-220
    204139262337252438224351496710052887294,            // 7^-209
    187964189199610581269006081054389519147,            // 7^-198
    173070755801490399767170008783447571912,            // 7^-187
    318714821597104302344272129046782646031,            // 7^-176
    293461298632634947256039303556899737616,            // 7^-165
    270208750768480083796084384984407980627,            // 7^-154
    248798629775241702400557051000361115726,            // 7^-143
    229084950069124576377435007616984162844,            // 7^-132
    210933293304638808346189699277966975327,            // 7^-121
    194219891838880796776419735762035423417,            // 7^-110
    178830784817964977889863278397948540205,            // 7^-99
    329322082262710237520775170056072850605,            // 7^-88
    303228087871569629902816732340606701122,            // 7^-77
    279201663740542055384000770694089408486,            // 7^-66
    257078984940548995242906668328695771951,            // 7^-55
    236709207289964795762477040448859922676,            // 7^-44
    217953438818817001855782665313698094789,            // 7^-33
    200683792729517998822275406364627986706,            // 7^-22
    184782515396710906443711214287193178833,            // 7^-11
    170141183460469231731687303715884105728,            // 7^0
    313319929076409987389789434290473271296,            // 7^11
    288493873028852398739253829029106548736,            // 7^22
    265634921533798919351224824788236107776,            // 7^33
    244587210111081219100242972308429222416,            // 7^44
    225207224277966141315155116349116687572,            // 7^55
    207362820991138609531788808643065835705,            // 7^66
    190932327625202079604864455739987836428,            // 7^77
    175803712344053086257499345217280929659,            // 7^88
    323747640416561983962207324433251030705,            // 7^99
    298095339619934405668456872884344580325,            // 7^110
    274475611277932187651307585787198545738,            // 7^121
    252727403529513497084111370284014563793,            // 7^132
    232702425535702904483240997815950541938,            // 7^143
    214264136353838934720254978302625752306,            // 7^154
    197286813928859418414994791325492154413,            // 7^165
    181654697853512422889189735564216996803,            // 7^176
    334522400104752565046325502667785120787,            // 7^187
    308016356015425756696586706818979868623,            // 7^198
    283610531143243549972387138852602501496,            // 7^209
    261138513603233655506120063924465915820,            // 7^220
    240447077236577495145565086308268824286,            // 7^231
    221395136833224081873557773886454798418,            // 7^242
    203852786137945162132478938839041683065,            // 7^253
    187700412080445632409314398956810353518,            // 7^264
    172827879189879361689316520165778889316,            // 7^275
    318267557265350256871483486812089934265,            // 7^286
    293049473506426909088112326597996634777,            // 7^297
    269829556805234360375792834335085449819,            // 7^308
    248449481428986202531286841797890645972,            // 7^319
    228763466660872229208763580066607974130,            // 7^330
    210637282789652967136592584117374228567,            // 7^341
    193947335860149215724544902551342320202,            // 7^352
    178579824943969003959847035500732297209,            // 7^363
];
const BASE7_LARGE_EXPONENT: [i32; 71] = [
    -1270,                                              // 7^-407
    -1239,                                              // 7^-396
    -1208,                                              // 7^-385
    -1177,                                              // 7^-374
    -1147,                                              // 7^-363
    -1116,                                              // 7^-352
    -1085,                                              // 7^-341
    -1054,                                              // 7^-330
    -1023,                                              // 7^-319
    -992,                                               // 7^-308
    -961,                                               // 7^-297
    -930,                                               // 7^-286
    -900,                                               // 7^-275
    -869,                                               // 7^-264
    -838,                                               // 7^-253
    -807,                                               // 7^-242
    -776,                                               // 7^-231
    -745,                                               // 7^-220
    -714,                                               // 7^-209
    -683,                                               // 7^-198
    -652,                                               // 7^-187
    -622,                                               // 7^-176
    -591,                                               // 7^-165
    -560,                                               // 7^-154
    -529,                                               // 7^-143
    -498,                                               // 7^-132
    -467,                                               // 7^-121
    -436,                                               // 7^-110
    -405,                                               // 7^-99
    -375,                                               // 7^-88
    -344,                                               // 7^-77
    -313,                                               // 7^-66
    -282,                                               // 7^-55
    -251,                                               // 7^-44
    -220,                                               // 7^-33
    -189,                                               // 7^-22
    -158,                                               // 7^-11
    -127,                                               // 7^0
    -97,                                                // 7^11
    -66,                                                // 7^22
    -35,                                                // 7^33
    -4,                                                 // 7^44
    27,                                                 // 7^55
    58,                                                 // 7^66
    89,                                                 // 7^77
    120,                                                // 7^88
    150,                                                // 7^99
    181,                                                // 7^110
    212,                                                // 7^121
    243,                                                // 7^132
    274,                                                // 7^143
    305,                                                // 7^154
    336,                                                // 7^165
    367,                                                // 7^176
    397,                                                // 7^187
    428,                                                // 7^198
    459,                                                // 7^209
    490,                                                // 7^220
    521,                                                // 7^231
    552,                                                // 7^242
    583,                                                // 7^253
    614,                                                // 7^264
    645,                                                // 7^275
    675,                                                // 7^286
    706,                                                // 7^297
    737,                                                // 7^308
    768,                                                // 7^319
    799,                                                // 7^330
    830,                                                // 7^341
    861,                                                // 7^352
    892,                                                // 7^363
];
const BASE7_SMALL_INT_POWERS: [u128; 11] = [1, 7, 49, 343, 2401, 16807, 117649, 823543, 5764801, 40353607, 282475249];
const BASE7_STEP: i32 = 11;
const BASE7_BIAS: i32 = 407;

// BASE9

const BASE9_SMALL_MANTISSA: [u128; 10] = [
    170141183460469231731687303715884105728,            // 9^0
    191408831393027885698148216680369618944,            // 9^1
    215334935317156371410416743765415821312,            // 9^2
    242251802231800917836718836736092798976,            // 9^3
    272533277510776032566308691328104398848,            // 9^4
    306599937199623036637097277744117448704,            // 9^5
    172462464674787958108367218731066064896,            // 9^6
    194020272759136452871913121072449323008,            // 9^7
    218272806854028509480902261206505488384,            // 9^8
    245556907710782073166015043857318674432,            // 9^9
];
const BASE9_SMALL_EXPONENT: [i32; 10] = [
    -127,                                               // 9^0
    -124,                                               // 9^1
    -121,                                               // 9^2
    -118,                                               // 9^3
    -115,                                               // 9^4
    -112,                                               // 9^5
    -108,                                               // 9^6
    -105,                                               // 9^7
    -102,                                               // 9^8
    -99,                                                // 9^9
];
const BASE9_LARGE_MANTISSA: [u128; 69] = [
    301829093537629265639465570217176944359,            // 9^-360
    245033990385703656345786023933864839340,            // 9^-350
    198926007233479871031630637668169238011,            // 9^-340
    322988302900880006728964617948539328448,            // 9^-330
    262211676747596696167096696967233799204,            // 9^-320
    212871372756449173771443137071089544143,            // 9^-310
    172815421118085121562612771428651141606,            // 9^-300
    280593575260967566098415738074481154338,            // 9^-290
    227794354139073103116567345878808448350,            // 9^-280
    184930348919702200346046943747485274024,            // 9^-270
    300264105147079021545114594266031000970,            // 9^-260
    243763485459391712918376663011554847091,            // 9^-250
    197894572893436379626501802082900685163,            // 9^-240
    321313603691473325606249593990411338331,            // 9^-230
    260852105259086286749566195634740776863,            // 9^-220
    211767631486382365261996259087726574961,            // 9^-210
    171919370559843833352674924374427532806,            // 9^-200
    279138693352137745884317186629683060895,            // 9^-190
    226613236986043931067161987739751269180,            // 9^-180
    183971482278558945643179980616811190964,            // 9^-170
    298707231244876640116631457791747347925,            // 9^-160
    242499568120235502703106353919523432682,            // 9^-150
    196868486555962367745019627988939060464,            // 9^-140
    319647587822660709450189016904055940251,            // 9^-130
    259499583169196479959998361450291137700,            // 9^-120
    210669613131404954481085620515615417585,            // 9^-110
    171027966037226738058674240289082148799,            // 9^-100
    277691355027891684120101092281051616669,            // 9^-90
    225438243943221318560556485110109738278,            // 9^-80
    183017587375374702561553597022155160742,            // 9^-70
    297158429757277967604640789526650060843,            // 9^-60
    241242204211496523037749538228345943134,            // 9^-50
    195847720491584060106836777189641681162,            // 9^-40
    317990210271190550439415903835536554761,            // 9^-30
    258154073926689380540223575440483383976,            // 9^-20
    209577288018116110386327960504760073299,            // 9^-10
    170141183460469231731687303715884105728,            // 9^0
    276251521174629832311766924339483508736,            // 9^10
    224269343257001716702690972139746492416,            // 9^20
    182068638431613361423174859113151594496,            // 9^30
    295617658828691846632166420412766595202,            // 9^40
    239991359753539474232337032335634004651,            // 9^50
    194832247114605420104007752175098574688,            // 9^60
    316341426247257477645159711999449660471,            // 9^70
    256815541169845811576524073480007610450,            // 9^80
    208490626626972031635281014538153149532,            // 9^90
    338517997729425004575949331160209430911,            // 9^100
    274819152881557244028610584245948464515,            // 9^110
    223106503338424488684979682521025988628,            // 9^120
    181124609802400910077427551154104473922,            // 9^130
    294084876820548989626661915132664622178,            // 9^140
    238747000942913976797497733353022683918,            // 9^150
    193822038982362660063056049982127016523,            // 9^160
    314701191193291934781116205950433765545,            // 9^170
    255483948725482657093998355855298189652,            // 9^180
    207409599591488195571905341445445255582,            // 9^190
    336762776818711782198286065086981891498,            // 9^200
    273394211439632029990640781047045990695,            // 9^210
    221949692762318233808346663450192754968,            // 9^220
    180185475975832393914650652957737664335,            // 9^230
    292560042310176717160312096633717510967,            // 9^240
    237509094151441049982785534773499431992,            // 9^250
    192817068794482616882547252154136283242,            // 9^260
    313069460782756034010893203297842312622,            // 9^270
    254159260607975299744356396919078736707,            // 9^280
    206334177697445743564032291193028958152,            // 9^290
    335016656754825225194410391893304442626,            // 9^300
    271976658340519265432186039827268288213,            // 9^310
    220798880266451537830039115389735391778,            // 9^320
];
const BASE9_LARGE_EXPONENT: [i32; 69] = [
    -1269,                                              // 9^-360
    -1237,                                              // 9^-350
    -1205,                                              // 9^-340
    -1174,                                              // 9^-330
    -1142,                                              // 9^-320
    -1110,                                              // 9^-310
    -1078,                                              // 9^-300
    -1047,                                              // 9^-290
    -1015,                                              // 9^-280
    -983,                                               // 9^-270
    -952,                                               // 9^-260
    -920,                                               // 9^-250
    -888,                                               // 9^-240
    -857,                                               // 9^-230
    -825,                                               // 9^-220
    -793,                                               // 9^-210
    -761,                                               // 9^-200
    -730,                                               // 9^-190
    -698,                                               // 9^-180
    -666,                                               // 9^-170
    -635,                                               // 9^-160
    -603,                                               // 9^-150
    -571,                                               // 9^-140
    -540,                                               // 9^-130
    -508,                                               // 9^-120
    -476,                                               // 9^-110
    -444,                                               // 9^-100
    -413,                                               // 9^-90
    -381,                                               // 9^-80
    -349,                                               // 9^-70
    -318,                                               // 9^-60
    -286,                                               // 9^-50
    -254,                                               // 9^-40
    -223,                                               // 9^-30
    -191,                                               // 9^-20
    -159,                                               // 9^-10
    -127,                                               // 9^0
    -96,                                                // 9^10
    -64,                                                // 9^20
    -32,                                                // 9^30
    -1,                                                 // 9^40
    31,                                                 // 9^50
    63,                                                 // 9^60
    94,                                                 // 9^70
    126,                                                // 9^80
    158,                                                // 9^90
    189,                                                // 9^100
    221,                                                // 9^110
    253,                                                // 9^120
    285,                                                // 9^130
    316,                                                // 9^140
    348,                                                // 9^150
    380,                                                // 9^160
    411,                                                // 9^170
    443,                                                // 9^180
    475,                                                // 9^190
    506,                                                // 9^200
    538,                                                // 9^210
    570,                                                // 9^220
    602,                                                // 9^230
    633,                                                // 9^240
    665,                                                // 9^250
    697,                                                // 9^260
    728,                                                // 9^270
    760,                                                // 9^280
    792,                                                // 9^290
    823,                                                // 9^300
    855,                                                // 9^310
    887,                                                // 9^320
];
const BASE9_SMALL_INT_POWERS: [u128; 10] = [1, 9, 81, 729, 6561, 59049, 531441, 4782969, 43046721, 387420489];
const BASE9_STEP: i32 = 10;
const BASE9_BIAS: i32 = 360;

// BASE11

const BASE11_SMALL_MANTISSA: [u128; 9] = [
    170141183460469231731687303715884105728,            // 11^0
    233944127258145193631070042609340645376,            // 11^1
    321673174979949641242721308587843387392,            // 11^2
    221150307798715378354370899654142328832,            // 11^3
    304081673223233645237259987024445702144,            // 11^4
    209056150340973131100616241079306420224,            // 11^5
    287452206718838055263347331484046327808,            // 11^6
    197623392119201162993551290395281850368,            // 11^7
    271732164163901599116133024293512544256,            // 11^8
];
const BASE11_SMALL_EXPONENT: [i32; 9] = [
    -127,                                               // 11^0
    -124,                                               // 11^1
    -121,                                               // 11^2
    -117,                                               // 11^3
    -114,                                               // 11^4
    -110,                                               // 11^5
    -107,                                               // 11^6
    -103,                                               // 11^7
    -100,                                               // 11^8
];
const BASE11_LARGE_MANTISSA: [u128; 70] = [
    171238059048456285069153007806767780751,            // 11^-333
    188020237695718721716314139814895141308,            // 11^-324
    206447153042951189194086232691622882083,            // 11^-315
    226679997440031906769564274339983654138,            // 11^-306
    248895760886189129917706060109685643694,            // 11^-297
    273288779277949488106534827711219187027,            // 11^-288
    300072434393060227248554937081761703289,            // 11^-279
    329481020481262984343182946079685486146,            // 11^-270
    180885896895108223589085363619768036079,            // 11^-261
    198613611477559667549559644399735817047,            // 11^-252
    218078729968836026458633966547967004969,            // 11^-243
    239451526564652757041971603611460905574,            // 11^-234
    262918963175987885652063848834117643272,            // 11^-225
    288686324907902004950333378158035100892,            // 11^-216
    316979015823390018300576675081611005238,            // 11^-207
    174022265350496153683469750680295263035,            // 11^-198
    191077310017213090920037889613280023761,            // 11^-189
    209803833606456768694276179860414245615,            // 11^-180
    230365649338482385703357683130616542302,            // 11^-171
    252942624941184287338239913752743474826,            // 11^-162
    277732256071429933018369293137137136134,            // 11^-153
    304951393939484349279174830456271141999,            // 11^-144
    334838142249192351062534286575243350001,            // 11^-135
    183826970023851061892407973678416422311,            // 11^-126
    201842924352393404728314593418040675821,            // 11^-117
    221624531513738305774870271734740961947,            // 11^-108
    243344834238186465986122993315325591576,            // 11^-99
    267193834301414676708834595107417746247,            // 11^-90
    293380154594991977782504639045547494542,            // 11^-81
    322132864088045607402818330486415493575,            // 11^-72
    176851740822108453297561032060281913714,            // 11^-63
    194184088111306107232610402385759498575,            // 11^-54
    213215091354676888577329040601887973466,            // 11^-45
    234111227256299197427917870802090967668,            // 11^-36
    257055288062508663991611489278928709768,            // 11^-27
    282247980565918687696808555402942563478,            // 11^-18
    309909681819761564465444461107912469729,            // 11^-9
    170141183460469231731687303715884105728,            // 11^0
    186815862862682349392341454201789874176,            // 11^9
    205124743505955904636591107127579246592,            // 11^18
    225227985212968648451224991661780107264,            // 11^27
    247301444262965381085386823495099626888,            // 11^36
    271538211722310826714720080747313199115,            // 11^45
    298150302539063592923933328180322755271,            // 11^54
    327370510177191550022527822209865447333,            // 11^63
    179727221507067050840782578187764330990,            // 11^72
    197341380157710189331417385894277071911,            // 11^81
    216681813672942089280666997109850212851,            // 11^90
    237917705546974217796728292841033536358,            // 11^99
    261234820095126400184969690376374338338,            // 11^108
    286837127456489808703442688252756740915,            // 11^117
    314948587854906379895896832699151980516,            // 11^126
    172907555363184572545974225978641470455,            // 11^135
    189853352925309785288573204303090400331,            // 11^144
    208459922652152508348644224820166488004,            // 11^153
    228890028448627145607277691781146666326,            // 11^162
    251322385889182188596856419989450117198,            // 11^171
    275953225560490099471194727985491113834,            // 11^180
    302998009619470597652934458975103615488,            // 11^189
    332693316582509542060143699364121908613,            // 11^198
    182649455416682266091504883729813020363,            // 11^207
    200550007476552106538373760555841065584,            // 11^216
    220204902374823007850602460837751774656,            // 11^225
    241786074406278472344858080843135119733,            // 11^234
    265482308278902675848717553283442644763,            // 11^243
    291500890537904924029114199837605959154,            // 11^252
    320069422837484936971208194402230692262,            // 11^261
    175718906461109937023963056381563419957,            // 11^270
    192940230367248485471498733602562118815,            // 11^279
    211849329385655768374338506351963365331,            // 11^288
];
const BASE11_LARGE_EXPONENT: [i32; 70] = [
    -1279,                                              // 11^-333
    -1248,                                              // 11^-324
    -1217,                                              // 11^-315
    -1186,                                              // 11^-306
    -1155,                                              // 11^-297
    -1124,                                              // 11^-288
    -1093,                                              // 11^-279
    -1062,                                              // 11^-270
    -1030,                                              // 11^-261
    -999,                                               // 11^-252
    -968,                                               // 11^-243
    -937,                                               // 11^-234
    -906,                                               // 11^-225
    -875,                                               // 11^-216
    -844,                                               // 11^-207
    -812,                                               // 11^-198
    -781,                                               // 11^-189
    -750,                                               // 11^-180
    -719,                                               // 11^-171
    -688,                                               // 11^-162
    -657,                                               // 11^-153
    -626,                                               // 11^-144
    -595,                                               // 11^-135
    -563,                                               // 11^-126
    -532,                                               // 11^-117
    -501,                                               // 11^-108
    -470,                                               // 11^-99
    -439,                                               // 11^-90
    -408,                                               // 11^-81
    -377,                                               // 11^-72
    -345,                                               // 11^-63
    -314,                                               // 11^-54
    -283,                                               // 11^-45
    -252,                                               // 11^-36
    -221,                                               // 11^-27
    -190,                                               // 11^-18
    -159,                                               // 11^-9
    -127,                                               // 11^0
    -96,                                                // 11^9
    -65,                                                // 11^18
    -34,                                                // 11^27
    -3,                                                 // 11^36
    28,                                                 // 11^45
    59,                                                 // 11^54
    90,                                                 // 11^63
    122,                                                // 11^72
    153,                                                // 11^81
    184,                                                // 11^90
    215,                                                // 11^99
    246,                                                // 11^108
    277,                                                // 11^117
    308,                                                // 11^126
    340,                                                // 11^135
    371,                                                // 11^144
    402,                                                // 11^153
    433,                                                // 11^162
    464,                                                // 11^171
    495,                                                // 11^180
    526,                                                // 11^189
    557,                                                // 11^198
    589,                                                // 11^207
    620,                                                // 11^216
    651,                                                // 11^225
    682,                                                // 11^234
    713,                                                // 11^243
    744,                                                // 11^252
    775,                                                // 11^261
    807,                                                // 11^270
    838,                                                // 11^279
    869,                                                // 11^288
];
const BASE11_SMALL_INT_POWERS: [u128; 9] = [1, 11, 121, 1331, 14641, 161051, 1771561, 19487171, 214358881];
const BASE11_STEP: i32 = 9;
const BASE11_BIAS: i32 = 333;

// BASE12

const BASE12_SMALL_MANTISSA: [u128; 9] = [
    170141183460469231731687303715884105728,            // 12^0
    255211775190703847597530955573826158592,            // 12^1
    191408831393027885698148216680369618944,            // 12^2
    287113247089541828547222325020554428416,            // 12^3
    215334935317156371410416743765415821312,            // 12^4
    323002402975734557115625115648123731968,            // 12^5
    242251802231800917836718836736092798976,            // 12^6
    181688851673850688377539127552069599232,            // 12^7
    272533277510776032566308691328104398848,            // 12^8
];
const BASE12_SMALL_EXPONENT: [i32; 9] = [
    -127,                                               // 12^0
    -124,                                               // 12^1
    -120,                                               // 12^2
    -117,                                               // 12^3
    -113,                                               // 12^4
    -110,                                               // 12^5
    -106,                                               // 12^6
    -102,                                               // 12^7
    -99,                                                // 12^8
];
const BASE12_LARGE_MANTISSA: [u128; 68] = [
    236015590119408703302029793810763336632,            // 12^-324
    283538504658222748235708766575760177913,            // 12^-315
    170315197362908885300398426895467760677,            // 12^-306
    204609010601448705405745986119597896326,            // 12^-297
    245808053934833671173174941698733239342,            // 12^-288
    295302729833943551617529441983408590696,            // 12^-279
    177381702616012906692133545122052956869,            // 12^-270
    213098391881773806300126011269370626834,            // 12^-261
    256006814417050404626793229969178591795,            // 12^-252
    307555061533862494767405465422567274311,            // 12^-243
    184741402471039290909022270993420155647,            // 12^-234
    221940003957364890317522299802459040748,            // 12^-225
    266628729119434395515123988465075762881,            // 12^-216
    320315751663685742610118741757695693407,            // 12^-207
    192406461791880080316008520325217417399,            // 12^-198
    231148461148045387015380597263260157877,            // 12^-189
    277691355027891684120101092281051616669,            // 12^-180
    333605892395873536287594592246578306329,            // 12^-171
    200389550171752283164939097875653100692,            // 12^-162
    240738984132727062349578629363188475398,            // 12^-153
    289212977580839036146652597763405686112,            // 12^-144
    173723725516468955947099703423373843986,            // 12^-135
    208703862874796048578293668364396201854,            // 12^-126
    250727425107703285166415666163110988836,            // 12^-117
    301212640893244858516269504216828222245,            // 12^-108
    180931653158622392278312153671259457350,            // 12^-99
    217363142646555453321168098187951653993,            // 12^-90
    261130293988778746809115702919522241550,            // 12^-81
    313710179234688236904530296665341569850,            // 12^-72
    188438643123668474334468683754392032451,            // 12^-63
    226381702429392491474935736226666160567,            // 12^-54
    271964785700545191021799322274747927151,            // 12^-45
    326726249813466247246220462666861782844,            // 12^-36
    196257103731642338395610271199702162833,            // 12^-27
    235774449020380624184618955567855082461,            // 12^-18
    283248808597909657338003839260381566656,            // 12^-9
    170141183460469231731687303715884105728,            // 12^0
    204399958133082024424731518496078299136,            // 12^9
    245556907710782073166015043857318674432,            // 12^18
    295001014066853243782145636489477750784,            // 12^27
    177200468746272961345336076752392290304,            // 12^36
    212880665669732098276382446210774990848,            // 12^45
    255745247947835503562868389206950936576,            // 12^54
    307240827353347547401607574753443315712,            // 12^63
    184552649072141716781794491390137475072,            // 12^72
    221713244121518884974124815309574946401,            // 12^81
    266356310061270520809673995345359110719,            // 12^90
    319988479671385965643116043114178672868,            // 12^99
    192209876872921446586714266254161951235,            // 12^108
    230912292876569386789935113689005718149,            // 12^117
    277407633098725295421526662764935275289,            // 12^126
    333265041643201293321649737744276185517,            // 12^135
    200184808797092622572327630249651738267,            // 12^144
    240493017062571660772163375622796335712,            // 12^153
    288917483816076538023589582665008561757,            // 12^162
    173546229063471511777292289904643662141,            // 12^171
    208490626626972031635281014538153149532,            // 12^180
    250471252679363433757155530343900661758,            // 12^189
    300904886870600004067510516586852827477,            // 12^198
    180746792244690548097558883605316900733,            // 12^207
    217141059066909427380630585083218539864,            // 12^216
    260863492774290665230282703014708894052,            // 12^225
    313389656266867868879861721401276560157,            // 12^234
    188246112191795662327951607127115677904,            // 12^243
    226150404435492799169987273137391228527,            // 12^252
    271686914703601365116141326731156710883,            // 12^261
    326392428107359965184387801150473482685,            // 12^270
    196056584547032659751107943421776414785,            // 12^279
];
const BASE12_LARGE_EXPONENT: [i32; 68] = [
    -1289,                                              // 12^-324
    -1257,                                              // 12^-315
    -1224,                                              // 12^-306
    -1192,                                              // 12^-297
    -1160,                                              // 12^-288
    -1128,                                              // 12^-279
    -1095,                                              // 12^-270
    -1063,                                              // 12^-261
    -1031,                                              // 12^-252
    -999,                                               // 12^-243
    -966,                                               // 12^-234
    -934,                                               // 12^-225
    -902,                                               // 12^-216
    -870,                                               // 12^-207
    -837,                                               // 12^-198
    -805,                                               // 12^-189
    -773,                                               // 12^-180
    -741,                                               // 12^-171
    -708,                                               // 12^-162
    -676,                                               // 12^-153
    -644,                                               // 12^-144
    -611,                                               // 12^-135
    -579,                                               // 12^-126
    -547,                                               // 12^-117
    -515,                                               // 12^-108
    -482,                                               // 12^-99
    -450,                                               // 12^-90
    -418,                                               // 12^-81
    -386,                                               // 12^-72
    -353,                                               // 12^-63
    -321,                                               // 12^-54
    -289,                                               // 12^-45
    -257,                                               // 12^-36
    -224,                                               // 12^-27
    -192,                                               // 12^-18
    -160,                                               // 12^-9
    -127,                                               // 12^0
    -95,                                                // 12^9
    -63,                                                // 12^18
    -31,                                                // 12^27
    2,                                                  // 12^36
    34,                                                 // 12^45
    66,                                                 // 12^54
    98,                                                 // 12^63
    131,                                                // 12^72
    163,                                                // 12^81
    195,                                                // 12^90
    227,                                                // 12^99
    260,                                                // 12^108
    292,                                                // 12^117
    324,                                                // 12^126
    356,                                                // 12^135
    389,                                                // 12^144
    421,                                                // 12^153
    453,                                                // 12^162
    486,                                                // 12^171
    518,                                                // 12^180
    550,                                                // 12^189
    582,                                                // 12^198
    615,                                                // 12^207
    647,                                                // 12^216
    679,                                                // 12^225
    711,                                                // 12^234
    744,                                                // 12^243
    776,                                                // 12^252
    808,                                                // 12^261
    840,                                                // 12^270
    873,                                                // 12^279
];
const BASE12_SMALL_INT_POWERS: [u128; 9] = [1, 12, 144, 1728, 20736, 248832, 2985984, 35831808, 429981696];
const BASE12_STEP: i32 = 9;
const BASE12_BIAS: i32 = 324;

// BASE13

const BASE13_SMALL_MANTISSA: [u128; 8] = [
    170141183460469231731687303715884105728,            // 13^0
    276479423123262501563991868538311671808,            // 13^1
    224639531287650782520743393187378233344,            // 13^2
    182519619171216260798104006964744814592,            // 13^3
    296594381153226423796919011317710323712,            // 13^4
    240982934686996469334996696695639638016,            // 13^5
    195798634433184631334684816065207205888,            // 13^6
    318172780953925025918862826105961709568,            // 13^7
];
const BASE13_SMALL_EXPONENT: [i32; 8] = [
    -127,                                               // 13^0
    -124,                                               // 13^1
    -120,                                               // 13^2
    -116,                                               // 13^3
    -113,                                               // 13^4
    -109,                                               // 13^5
    -105,                                               // 13^6
    -102,                                               // 13^7
];
const BASE13_LARGE_MANTISSA: [u128; 74] = [
    234492264952419818661296823055466035238,            // 13^-312
    178145751691013993109826848158654507171,            // 13^-304
    270677660536016922109887166011364690074,            // 13^-296
    205636101949623162244352604410701629545,            // 13^-288
    312446962496257587924386799659895562352,            // 13^-280
    237368592984352411701605597109952160960,            // 13^-272
    180330922359490147340520816471555077038,            // 13^-264
    273997845714729128559081401928980460360,            // 13^-256
    208158474636564731932432542641850208607,            // 13^-248
    316279498110609430339960234785638219567,            // 13^-240
    240280202619066059925234911435300433349,            // 13^-232
    182542896759209079117924981191278218615,            // 13^-224
    277358756934885281135534847026826478882,            // 13^-216
    210711787240726612911538723377341283448,            // 13^-208
    320159044357159213189440578296814412270,            // 13^-200
    243227526627608078053980201967728671175,            // 13^-192
    184782003669985976085437039910686134984,            // 13^-184
    280760893750083272032555777889582843511,            // 13^-176
    213296419277190995686832685182416132955,            // 13^-168
    324086177877525444638594829459150002632,            // 13^-160
    246211003089480177174216643591955743445,            // 13^-152
    187048575904513609075482455349961378053,            // 13^-144
    284204761841543230201034707437221523825,            // 13^-136
    215912754916246372204794564138901211197,            // 13^-128
    328061482386525623150638173263346085618,            // 13^-120
    249231075457753005677502945704601914895,            // 13^-112
    189342950357830398121128390896356453239,            // 13^-104
    287690873093270135400938070638952191684,            // 13^-96
    218561183040489207765436578950529361150,            // 13^-88
    332085548758937481126223466223271077691,            // 13^-80
    252288192624979397908801362020362227394,            // 13^-72
    191665468057395263450492571375008762286,            // 13^-64
    291219745668138391592456499213129019442,            // 13^-56
    221242097302626033221832363181059870585,            // 13^-48
    336158975117324458944453423021025623710,            // 13^-40
    255382808989916127627430380985009698372,            // 13^-32
    194016474213776704407345321083633835842,            // 13^-24
    294791904084909668600573533146696407742,            // 13^-16
    223955896183984548959338021657683010637,            // 13^-8
    170141183460469231731687303715884105728,            // 13^0
    258515384525064083559076046211093889024,            // 13^8
    196396318271963640537903969427202768896,            // 13^16
    298407879296195459704416306334760173568,            // 13^24
    226702983053742437531808354380178063872,            // 13^32
    172228168527856562581024588413877561828,            // 13^40
    261686384845036905964465455013477410965,            // 13^48
    198805353963305641500281734389939339052,            // 13^56
    302068208767375943221882620634577632307,            // 13^64
    229483766228882688509711411881831930515,            // 13^72
    174340752962680465097022527378326959423,            // 13^80
    264896281275768246289732905498826672107,            // 13^88
    201243939358090266003114541899396801267,            // 13^96
    305773436556486878301580275426022079145,            // 13^104
    232298659034884347081172590621418853088,            // 13^112
    176479250771793883849064971202097683281,            // 13^120
    268145550924567936613388404500729180677,            // 13^128
    203712436918765324677453955129179600235,            // 13^136
    309524113395086409019191949184662333574,            // 13^144
    235148079867157707257686510259099274415,            // 13^152
    178643979813719299400243488890704650366,            // 13^160
    271434676751037481783657313824389119896,            // 13^168
    206211213553813977645157972930830162165,            // 13^176
    313320796770113796525378889251061848013,            // 13^184
    238032452253233081187768481790039531230,            // 13^192
    180835261845894060934288352757379708280,            // 13^200
    274764147638855414866553566635921015415,            // 13^208
    208740640672290674581982064710037218783,            // 13^216
    317164051006752246428908041221997358677,            // 13^224
    240952204915712388090624317255539471706,            // 13^232
    183053422572495239382420468754129441919,            // 13^240
    278134458468443185818361944895170594350,            // 13^248
    211301094239026043298793364640604498188,            // 13^256
    321054447352308147843744474962905446408,            // 13^264
    243907771835992919704646129069158162971,            // 13^272
];
const BASE13_LARGE_EXPONENT: [i32; 74] = [
    -1282,                                              // 13^-312
    -1252,                                              // 13^-304
    -1223,                                              // 13^-296
    -1193,                                              // 13^-288
    -1164,                                              // 13^-280
    -1134,                                              // 13^-272
    -1104,                                              // 13^-264
    -1075,                                              // 13^-256
    -1045,                                              // 13^-248
    -1016,                                              // 13^-240
    -986,                                               // 13^-232
    -956,                                               // 13^-224
    -927,                                               // 13^-216
    -897,                                               // 13^-208
    -868,                                               // 13^-200
    -838,                                               // 13^-192
    -808,                                               // 13^-184
    -779,                                               // 13^-176
    -749,                                               // 13^-168
    -720,                                               // 13^-160
    -690,                                               // 13^-152
    -660,                                               // 13^-144
    -631,                                               // 13^-136
    -601,                                               // 13^-128
    -572,                                               // 13^-120
    -542,                                               // 13^-112
    -512,                                               // 13^-104
    -483,                                               // 13^-96
    -453,                                               // 13^-88
    -424,                                               // 13^-80
    -394,                                               // 13^-72
    -364,                                               // 13^-64
    -335,                                               // 13^-56
    -305,                                               // 13^-48
    -276,                                               // 13^-40
    -246,                                               // 13^-32
    -216,                                               // 13^-24
    -187,                                               // 13^-16
    -157,                                               // 13^-8
    -127,                                               // 13^0
    -98,                                                // 13^8
    -68,                                                // 13^16
    -39,                                                // 13^24
    -9,                                                 // 13^32
    21,                                                 // 13^40
    50,                                                 // 13^48
    80,                                                 // 13^56
    109,                                                // 13^64
    139,                                                // 13^72
    169,                                                // 13^80
    198,                                                // 13^88
    228,                                                // 13^96
    257,                                                // 13^104
    287,                                                // 13^112
    317,                                                // 13^120
    346,                                                // 13^128
    376,                                                // 13^136
    405,                                                // 13^144
    435,                                                // 13^152
    465,                                                // 13^160
    494,                                                // 13^168
    524,                                                // 13^176
    553,                                                // 13^184
    583,                                                // 13^192
    613,                                                // 13^200
    642,                                                // 13^208
    672,                                                // 13^216
    701,                                                // 13^224
    731,                                                // 13^232
    761,                                                // 13^240
    790,                                                // 13^248
    820,                                                // 13^256
    849,                                                // 13^264
    879,                                                // 13^272
];
const BASE13_SMALL_INT_POWERS: [u128; 8] = [1, 13, 169, 2197, 28561, 371293, 4826809, 62748517];
const BASE13_STEP: i32 = 8;
const BASE13_BIAS: i32 = 312;

// BASE14

const BASE14_SMALL_MANTISSA: [u128; 8] = [
    170141183460469231731687303715884105728,            // 14^0
    297747071055821155530452781502797185024,            // 14^1
    260528687173843511089146183814947536896,            // 14^2
    227962601277113072203002910838079094784,            // 14^3
    199467276117473938177627546983319207936,            // 14^4
    174533866602789695905424103610404306944,            // 14^5
    305434266554881967834492181318207537152,            // 14^6
    267254983235521721855180658653431595008,            // 14^7
];
const BASE14_SMALL_EXPONENT: [i32; 8] = [
    -127,                                               // 14^0
    -124,                                               // 14^1
    -120,                                               // 14^2
    -116,                                               // 14^3
    -112,                                               // 14^4
    -108,                                               // 14^5
    -105,                                               // 14^6
    -101,                                               // 14^7
];
const BASE14_LARGE_MANTISSA: [u128; 72] = [
    251548413169278560512334209582633427955,            // 14^-304
    172868555043538834443100066630384298338,            // 14^-296
    237596707101714066585640360644079693184,            // 14^-288
    326561387705008590881101164026582541311,            // 14^-280
    224418808746721891131396635947419054595,            // 14^-272
    308449214239576126269380201889590576494,            // 14^-264
    211971800172033630882960469392933087940,            // 14^-256
    291341601754078780972414349774563326428,            // 14^-248
    200215143815698040798730279921859793515,            // 14^-240
    275182833977670641254081992250398936156,            // 14^-232
    189110550462878905579110756278883461660,            // 14^-224
    259920284847963995399800125815215983072,            // 14^-216
    178621854545095883446307559621296276382,            // 14^-208
    245504247117858718392171207907774065824,            // 14^-200
    337429792711562885676838629607664070711,            // 14^-192
    231887770468403152813044191455947332680,            // 14^-184
    318714821597104302344272129046782646031,            // 14^-176
    219026508600450572879143037057044702127,            // 14^-168
    301037844611736789461197056572315300910,            // 14^-160
    206878574807117564156377107253687360168,            // 14^-152
    284341291171704802743493772571841275116,            // 14^-144
    195404405556671025581418948604574579552,            // 14^-136
    268570783748031302676865943988466296381,            // 14^-128
    184566631641558957302915986868221103411,            // 14^-120
    253674960769150428442670675281025859157,            // 14^-112
    174329956473941702727882427159635204541,            // 14^-104
    239605309345945263344621501916001364416,            // 14^-96
    329322082262710237520775170056072850605,            // 14^-88
    226316007274407653805017736090948441653,            // 14^-80
    311056791556242112413125050076207081650,            // 14^-72
    213763773801352511153375590062662203372,            // 14^-64
    293804554217770280277607620947080882230,            // 14^-56
    201907728667158642949418150287074186439,            // 14^-48
    277509182960549548157083678768414391770,            // 14^-40
    190709258966464876623809832149485832940,            // 14^-32
    262117606830390855604604612394616145902,            // 14^-24
    180131893393211845729384454981934494079,            // 14^-16
    247579698363561878555441197267606485702,            // 14^-8
    170141183460469231731687303715884105728,            // 14^0
    233848110331081506623283076321752645632,            // 14^8
    321409182616407632938244080939940446208,            // 14^16
    220878121537715117784741212850409701376,            // 14^24
    303582767467198764972828371186555551744,            // 14^32
    208627491173467029036775346642092064768,            // 14^40
    286745064197610355009611023687762970225,            // 14^48
    197056321243220373650760726805477101625,            // 14^56
    270841235580262673674173137819514560921,            // 14^64
    186126926626483659918254253754028720893,            // 14^72
    255819485841579348845580110620374327715,            // 14^80
    175803712344053086257499345217280929659,            // 14^88
    241630891972710985114650364591580944199,            // 14^96
    332106115263742508816700348007180790592,            // 14^104
    228229244396512279339912293302076319311,            // 14^112
    313686412889065357315780098042512623701,            // 14^120
    215570896471654994597359469530054904893,            // 14^128
    296288328063653274657691113306601058431,            // 14^136
    203614622343740041422835873088956724196,            // 14^144
    279855198502973302491761604313633635665,            // 14^152
    192321482680456518790460799730877816261,            // 14^160
    264333504599995236391965677440014878764,            // 14^168
    181654697853512422889189735564216996803,            // 14^176
    249672695121914450880056351845343996855,            // 14^184
    171579528154314464133715598246382584383,            // 14^192
    235825022574310344970824197408281461914,            // 14^200
    324126321306564057111299587260117144567,            // 14^208
    222745387696552489161405202381218740358,            // 14^216
    306149204668634768979587047598946613225,            // 14^224
    210391192582005308252369450519735221250,            // 14^232
    289169158312782474283830561810879736710,            // 14^240
    198722201944671478335130298919429936059,            // 14^248
    273130881427012463325938578210122255044,            // 14^256
    187700412080445632409314398956810353518,            // 14^264
];
const BASE14_LARGE_EXPONENT: [i32; 72] = [
    -1285,                                              // 14^-304
    -1254,                                              // 14^-296
    -1224,                                              // 14^-288
    -1194,                                              // 14^-280
    -1163,                                              // 14^-272
    -1133,                                              // 14^-264
    -1102,                                              // 14^-256
    -1072,                                              // 14^-248
    -1041,                                              // 14^-240
    -1011,                                              // 14^-232
    -980,                                               // 14^-224
    -950,                                               // 14^-216
    -919,                                               // 14^-208
    -889,                                               // 14^-200
    -859,                                               // 14^-192
    -828,                                               // 14^-184
    -798,                                               // 14^-176
    -767,                                               // 14^-168
    -737,                                               // 14^-160
    -706,                                               // 14^-152
    -676,                                               // 14^-144
    -645,                                               // 14^-136
    -615,                                               // 14^-128
    -584,                                               // 14^-120
    -554,                                               // 14^-112
    -523,                                               // 14^-104
    -493,                                               // 14^-96
    -463,                                               // 14^-88
    -432,                                               // 14^-80
    -402,                                               // 14^-72
    -371,                                               // 14^-64
    -341,                                               // 14^-56
    -310,                                               // 14^-48
    -280,                                               // 14^-40
    -249,                                               // 14^-32
    -219,                                               // 14^-24
    -188,                                               // 14^-16
    -158,                                               // 14^-8
    -127,                                               // 14^0
    -97,                                                // 14^8
    -67,                                                // 14^16
    -36,                                                // 14^24
    -6,                                                 // 14^32
    25,                                                 // 14^40
    55,                                                 // 14^48
    86,                                                 // 14^56
    116,                                                // 14^64
    147,                                                // 14^72
    177,                                                // 14^80
    208,                                                // 14^88
    238,                                                // 14^96
    268,                                                // 14^104
    299,                                                // 14^112
    329,                                                // 14^120
    360,                                                // 14^128
    390,                                                // 14^136
    421,                                                // 14^144
    451,                                                // 14^152
    482,                                                // 14^160
    512,                                                // 14^168
    543,                                                // 14^176
    573,                                                // 14^184
    604,                                                // 14^192
    634,                                                // 14^200
    664,                                                // 14^208
    695,                                                // 14^216
    725,                                                // 14^224
    756,                                                // 14^232
    786,                                                // 14^240
    817,                                                // 14^248
    847,                                                // 14^256
    878,                                                // 14^264
];
const BASE14_SMALL_INT_POWERS: [u128; 8] = [1, 14, 196, 2744, 38416, 537824, 7529536, 105413504];
const BASE14_STEP: i32 = 8;
const BASE14_BIAS: i32 = 304;

// BASE15

const BASE15_SMALL_MANTISSA: [u128; 8] = [
    170141183460469231731687303715884105728,            // 15^0
    319014718988379809496913694467282698240,            // 15^1
    299076299051606071403356588563077529600,            // 15^2
    280384030360880691940646801777885184000,            // 15^3
    262860028463325648694356376666767360000,            // 15^4
    246431276684367795650959103125094400000,            // 15^5
    231029321891594808422774159179776000000,            // 15^6
    216589989273370132896350774231040000000,            // 15^7
];
const BASE15_SMALL_EXPONENT: [i32; 8] = [
    -127,                                               // 15^0
    -124,                                               // 15^1
    -120,                                               // 15^2
    -116,                                               // 15^3
    -112,                                               // 15^4
    -108,                                               // 15^5
    -104,                                               // 15^6
    -100,                                               // 15^7
];
const BASE15_LARGE_MANTISSA: [u128; 70] = [
    250900630150587280377850342682701393565,            // 15^-296
    299434584015762656519487081521438784545,            // 15^-288
    178678447421354466262014715708770369510,            // 15^-280
    213241818261213966824485132704511196490,            // 15^-272
    254491091184140684406056053498886823977,            // 15^-264
    303719580053283976738045342609656736770,            // 15^-256
    181235388002241613446887230826356048960,            // 15^-248
    216293370737313530448778168911616746705,            // 15^-240
    258132932759965856034693127408739651425,            // 15^-232
    308065895631104646134849028190499821539,            // 15^-224
    183828919146951883093674448351327494412,            // 15^-216
    219388591817396682923135459647357436704,            // 15^-208
    261826890148575264294539329193196343516,            // 15^-200
    312474408249691315911417866816293319027,            // 15^-192
    186459564477102023710796186076704586060,            // 15^-184
    222528106411894691985532295229724475420,            // 15^-176
    265573709142416387133758214804806317848,            // 15^-168
    316946007966797681726272939583274683428,            // 15^-160
    189127855107486501747742000280219617523,            // 15^-152
    225712548373888955751123878842649157480,            // 15^-144
    269374146206443663446975509775551676690,            // 15^-136
    321481597577162915672598741300570427622,            // 15^-128
    191834329753307055373436045708607941890,            // 15^-120
    228942560627082928532238637834630336646,            // 15^-112
    273228968630845173556739192069109710925,            // 15^-104
    326082092794781635112624979076905003855,            // 15^-96
    194579534838936734374686465136944546456,            // 15^-88
    232218795295605362158733969001967701715,            // 15^-80
    277138954685954882938577351637270128950,            // 15^-72
    330748422437782207809506145002547658085,            // 15^-64
    197364024608240385117657735223630441823,            // 15^-56
    235541913835671069456466721167496749113,            // 15^-48
    281104893779381725235703590002221275596,            // 15^-40
    335481528615950719408183659471355674975,            // 15^-32
    200188361236473853754168248068850933442,            // 15^-24
    238912587169125791566529710613345919103,            // 15^-16
    285127586615387248178740525206042107172,            // 15^-8
    170141183460469231731687303715884105728,            // 15^0
    203053114943784499590328850841600000000,            // 15^8
    242331495818902131179520000000000000000,            // 15^16
    289207845356544000000000000000000000000,            // 15^24
    172575953309595678001642227172851562500,            // 15^32
    205958864110335933831952325359648803271,            // 15^40
    245799330046413899594233809090775284541,            // 15^48
    293346493787707294402529081970820087350,            // 15^56
    175045565423820113080636775231191731391,            // 15^64
    208906195393080226844550976655564036201,            // 15^72
    249316789990916616866725045444974174010,            // 15^80
    297544367482333459204270734183314844936,            // 15^88
    177550518406095745907734479894927853604,            // 15^96
    211895703844201159681805788312089434384,            // 15^104
    252884585810862301272632571838697691236,            // 15^112
    301802313971178147521594347169174498450,            // 15^120
    180091317994529147280371081412973024665,            // 15^128
    214927993031252433012289596290523334542,            // 15^136
    256503437827277086943219190518379691262,            // 15^144
    306121192913408770879770801728425641044,            // 15^152
    182668477164486370906358954938714362551,            // 15^160
    218003675159015088778073023304915283497,            // 15^168
    260174076669190616934963528529542450091,            // 15^176
    310501876270165601037714752361776484586,            // 15^184
    185282516232160242762660936045505469826,            // 15^192
    221123371193098747019724833334214306438,            // 15^200
    263897243421146573236258451562349376134,            // 15^208
    314945248480606581688897076801472259251,            // 15^216
    187933962959619728504626775329093970226,            // 15^224
    224287710985311612370215929865332425589,            // 15^232
    267673689772824125386430153566188172461,            // 15^240
    319452206640471392649554807130291119305,            // 15^248
    190623352661362587893902772482928415571,            // 15^256
];
const BASE15_LARGE_EXPONENT: [i32; 70] = [
    -1284,                                              // 15^-296
    -1253,                                              // 15^-288
    -1221,                                              // 15^-280
    -1190,                                              // 15^-272
    -1159,                                              // 15^-264
    -1128,                                              // 15^-256
    -1096,                                              // 15^-248
    -1065,                                              // 15^-240
    -1034,                                              // 15^-232
    -1003,                                              // 15^-224
    -971,                                               // 15^-216
    -940,                                               // 15^-208
    -909,                                               // 15^-200
    -878,                                               // 15^-192
    -846,                                               // 15^-184
    -815,                                               // 15^-176
    -784,                                               // 15^-168
    -753,                                               // 15^-160
    -721,                                               // 15^-152
    -690,                                               // 15^-144
    -659,                                               // 15^-136
    -628,                                               // 15^-128
    -596,                                               // 15^-120
    -565,                                               // 15^-112
    -534,                                               // 15^-104
    -503,                                               // 15^-96
    -471,                                               // 15^-88
    -440,                                               // 15^-80
    -409,                                               // 15^-72
    -378,                                               // 15^-64
    -346,                                               // 15^-56
    -315,                                               // 15^-48
    -284,                                               // 15^-40
    -253,                                               // 15^-32
    -221,                                               // 15^-24
    -190,                                               // 15^-16
    -159,                                               // 15^-8
    -127,                                               // 15^0
    -96,                                                // 15^8
    -65,                                                // 15^16
    -34,                                                // 15^24
    -2,                                                 // 15^32
    29,                                                 // 15^40
    60,                                                 // 15^48
    91,                                                 // 15^56
    123,                                                // 15^64
    154,                                                // 15^72
    185,                                                // 15^80
    216,                                                // 15^88
    248,                                                // 15^96
    279,                                                // 15^104
    310,                                                // 15^112
    341,                                                // 15^120
    373,                                                // 15^128
    404,                                                // 15^136
    435,                                                // 15^144
    466,                                                // 15^152
    498,                                                // 15^160
    529,                                                // 15^168
    560,                                                // 15^176
    591,                                                // 15^184
    623,                                                // 15^192
    654,                                                // 15^200
    685,                                                // 15^208
    716,                                                // 15^216
    748,                                                // 15^224
    779,                                                // 15^232
    810,                                                // 15^240
    841,                                                // 15^248
    873,                                                // 15^256
];
const BASE15_SMALL_INT_POWERS: [u128; 8] = [1, 15, 225, 3375, 50625, 759375, 11390625, 170859375];
const BASE15_STEP: i32 = 8;
const BASE15_BIAS: i32 = 296;

// BASE17

const BASE17_SMALL_MANTISSA: [u128; 8] = [
    170141183460469231731687303715884105728,            // 17^0
    180775007426748558714917760198126862336,            // 17^1
    192073445390920343634600120210509791232,            // 17^2
    204078035727852865111762627723666653184,            // 17^3
    216832912960843669181247791956395819008,            // 17^4
    230384970020896398505075778953670557696,            // 17^5
    244784030647202423411643015138274967552,            // 17^6
    260083032562652574874870703584417153024,            // 17^7
];
const BASE17_SMALL_EXPONENT: [i32; 8] = [
    -127,                                               // 17^0
    -123,                                               // 17^1
    -119,                                               // 17^2
    -115,                                               // 17^3
    -111,                                               // 17^4
    -107,                                               // 17^5
    -103,                                               // 17^6
    -99,                                                // 17^7
];
const BASE17_LARGE_MANTISSA: [u128; 67] = [
    242357519372662728375060373498171096966,            // 17^-280
    196814917627041423628769027913700117445,            // 17^-272
    319660903452112403161269325795427500295,            // 17^-264
    259591839957616255097162958982226744872,            // 17^-256
    210810651677570156300974624360660789801,            // 17^-248
    171196178077006380843936555487576333355,            // 17^-240
    278051712804343163076920967240438328697,            // 17^-232
    225801638394791143591435147984466155468,            // 17^-224
    183370134237042583472660416437030762483,            // 17^-216
    297824288336843871393537735572731133862,            // 17^-208
    241858651334916724628852435647948831754,            // 17^-200
    196409794352921881357331916371145369426,            // 17^-192
    319002914345514633220184430371226516661,            // 17^-184
    259057496842743054458415989246502258570,            // 17^-176
    210376719623757394611528557965027181599,            // 17^-168
    170843788344482468189688357828247614648,            // 17^-160
    277479371939006412580670873073737346072,            // 17^-152
    225336848935989032116058614991615739830,            // 17^-144
    182992685667322765043701960137104341962,            // 17^-136
    297211247657519026846228781764556419486,            // 17^-128
    241360810165739638162969982622013142483,            // 17^-120
    196005504983412884662857445298594989473,            // 17^-112
    318346279641847361755138868550752397781,            // 17^-104
    258524253618237975606418771469177669493,            // 17^-96
    209943680774466452613153056292584472116,            // 17^-88
    170492123969995067410761994182235561833,            // 17^-80
    276908209178500704519431941303594138118,            // 17^-72
    224873016198504574919954139464723650485,            // 17^-64
    182616014036679619640392724861134514431,            // 17^-56
    296599468859408121472112670217716316867,            // 17^-48
    240863993751428088431121063565557553364,            // 17^-40
    195602047802007523394399179976814437447,            // 17^-32
    317690996553211397504541477525504659745,            // 17^-24
    257992108020089771434235624403421929746,            // 17^-16
    209511533291127649202286079418781645629,            // 17^-8
    170141183460469231731687303715884105728,            // 17^0
    276338222097818360804550122558443225088,            // 17^8
    224410138213025296601555690180338253824,            // 17^16
    182240117745863932172015090234506084352,            // 17^24
    295988949345058405730513287165118905920,            // 17^32
    240368199982629535993820184805831872348,            // 17^40
    195199421095732140407812372336079928061,            // 17^48
    317037062297446153078380490386689107619,            // 17^56
    257461057788947429232455063849242347039,            // 17^64
    209080275338955809947349984471742343690,            // 17^72
    339581930651806711314806824645491109283,            // 17^80
    275769408276943332346622203534744539377,            // 17^88
    223948213014292349032159641795245781843,            // 17^96
    181864995198918377644202761532864575404,            // 17^104
    295379686522363719531147616048257012833,            // 17^112
    239873426754333326759263874720553852658,            // 17^120
    194797623155139058727680168785666503632,            // 17^128
    316384474098117832632118586933880356805,            // 17^136
    256931100670110578075784008065054807112,            // 17^144
    208649905086942477070838417411852870460,            // 17^152
    338882936158725697632383782036927673666,            // 17^160
    275201765300840924300371814765015192837,            // 17^168
    223487238641092167380922055560365365575,            // 17^176
    181490644803170745141604509031893001355,            // 17^184
    294771677804553486829405243638525556429,            // 17^192
    239379671965861754658715697694088165244,            // 17^200
    194396652274299323679629299288271234884,            // 17^208
    315733229184507643855046849844924656388,            // 17^216
    256402234413519915955050497805856036474,            // 17^224
    208220420707848135466354215936729966175,            // 17^232
    338185380473947645648707735110624890484,            // 17^240
    274635290759447542459052551448368297186,            // 17^248
];
const BASE17_LARGE_EXPONENT: [i32; 67] = [
    -1272,                                              // 17^-280
    -1239,                                              // 17^-272
    -1207,                                              // 17^-264
    -1174,                                              // 17^-256
    -1141,                                              // 17^-248
    -1108,                                              // 17^-240
    -1076,                                              // 17^-232
    -1043,                                              // 17^-224
    -1010,                                              // 17^-216
    -978,                                               // 17^-208
    -945,                                               // 17^-200
    -912,                                               // 17^-192
    -880,                                               // 17^-184
    -847,                                               // 17^-176
    -814,                                               // 17^-168
    -781,                                               // 17^-160
    -749,                                               // 17^-152
    -716,                                               // 17^-144
    -683,                                               // 17^-136
    -651,                                               // 17^-128
    -618,                                               // 17^-120
    -585,                                               // 17^-112
    -553,                                               // 17^-104
    -520,                                               // 17^-96
    -487,                                               // 17^-88
    -454,                                               // 17^-80
    -422,                                               // 17^-72
    -389,                                               // 17^-64
    -356,                                               // 17^-56
    -324,                                               // 17^-48
    -291,                                               // 17^-40
    -258,                                               // 17^-32
    -226,                                               // 17^-24
    -193,                                               // 17^-16
    -160,                                               // 17^-8
    -127,                                               // 17^0
    -95,                                                // 17^8
    -62,                                                // 17^16
    -29,                                                // 17^24
    3,                                                  // 17^32
    36,                                                 // 17^40
    69,                                                 // 17^48
    101,                                                // 17^56
    134,                                                // 17^64
    167,                                                // 17^72
    199,                                                // 17^80
    232,                                                // 17^88
    265,                                                // 17^96
    298,                                                // 17^104
    330,                                                // 17^112
    363,                                                // 17^120
    396,                                                // 17^128
    428,                                                // 17^136
    461,                                                // 17^144
    494,                                                // 17^152
    526,                                                // 17^160
    559,                                                // 17^168
    592,                                                // 17^176
    625,                                                // 17^184
    657,                                                // 17^192
    690,                                                // 17^200
    723,                                                // 17^208
    755,                                                // 17^216
    788,                                                // 17^224
    821,                                                // 17^232
    853,                                                // 17^240
    886,                                                // 17^248
];
const BASE17_SMALL_INT_POWERS: [u128; 8] = [1, 17, 289, 4913, 83521, 1419857, 24137569, 410338673];
const BASE17_STEP: i32 = 8;
const BASE17_BIAS: i32 = 280;

// BASE18

const BASE18_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 18^0
    191408831393027885698148216680369618944,            // 18^1
    215334935317156371410416743765415821312,            // 18^2
    242251802231800917836718836736092798976,            // 18^3
    272533277510776032566308691328104398848,            // 18^4
    306599937199623036637097277744117448704,            // 18^5
    172462464674787958108367218731066064896,            // 18^6
];
const BASE18_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 18^0
    -123,                                               // 18^1
    -119,                                               // 18^2
    -115,                                               // 18^3
    -111,                                               // 18^4
    -107,                                               // 18^5
    -102,                                               // 18^6
];
const BASE18_LARGE_MANTISSA: [u128; 75] = [
    259764989429046712145887613713888779974,            // 18^-273
    296222660952677279411722167462707735076,            // 18^-266
    337797118290463899238253918549284876092,            // 18^-259
    192603247770383575639211190527648274245,            // 18^-252
    219634857984796466920734002291401705412,            // 18^-245
    250460318818255417964791343261171179660,            // 18^-238
    285612092170198511649999639102587856294,            // 18^-231
    325697370261002112643262654266086944683,            // 18^-224
    185704281966673733437923590446998072591,            // 18^-217
    211767631486382365261996259087726574961,            // 18^-210
    241488937521646207617474790890828249100,            // 18^-203
    275381589414827976227270551469877695982,            // 18^-196
    314031030021154964119856834958393443507,            // 18^-189
    179052434161812488744424286609186187994,            // 18^-182
    204182205669996766442639895844185984708,            // 18^-175
    232838907258801165579649662968151663564,            // 18^-168
    265517538884334791214783518037108753711,            // 18^-161
    302782573089615796089597203369720463425,            // 18^-154
    172638852694972345186098175788410725025,            // 18^-147
    196868486555962367745019627988939060464,            // 18^-140
    224498717373391335032231518045098273537,            // 18^-133
    256006814417050404626793229969178591795,            // 18^-126
    291937031065346039954998156631577529912,            // 18^-119
    332910005936047335476283449703688561122,            // 18^-112
    189816741726628588213403698819375225722,            // 18^-105
    216457269515865090355509585365869594574,            // 18^-98
    246836760024792608106756526472045881483,            // 18^-91
    281479971709018296242657937208050445965,            // 18^-84
    320985312176969416466104839150917967197,            // 18^-77
    183017587375374702561553597022155160742,            // 18^-70
    208703862874796048578293668364396201854,            // 18^-63
    237995173051452727716558620615765508935,            // 18^-56
    271397479737933588417468230506889186025,            // 18^-49
    309487755838552588810803796052767101096,            // 18^-42
    176461975819512133258798291874254633040,            // 18^-35
    201228179937237770199942876645925476060,            // 18^-28
    229470287934835004924643178169318563534,            // 18^-21
    261676138308856451194147028266269550444,            // 18^-14
    298402037041419227483658365640566588740,            // 18^-7
    170141183460469231731687303715884105728,            // 18^0
    194020272759136452871913121072449323008,            // 18^7
    221250760550139932836609227367108313088,            // 18^14
    252303011164126931290527343657214803968,            // 18^21
    287713403941314941508226937857819803648,            // 18^28
    328093598350474163167634651360244400128,            // 18^35
    187070549727531559196917812917453861026,            // 18^42
    213325653114257310428744028742785781912,            // 18^49
    243265625417291205836310243227923719532,            // 18^56
    277407633098725295421526662764935275289,            // 18^63
    316341426247257477645159711999449660471,            // 18^70
    180369762796928745579122531717097251784,            // 18^77
    205684419630781050995309380627725821797,            // 18^84
    234551954955343535589691141355422293223,            // 18^91
    267471010551644448060009348077202513313,            // 18^98
    305010211912915299914630616083972269987,            // 18^105
    173908995182860443486855135343139262701,            // 18^112
    198316891856377323275537495574245323224,            // 18^119
    226150404435492799169987273137391228527,            // 18^126
    257890313566309108293837274983090159159,            // 18^133
    294084876820548989626661915132664622178,            // 18^140
    335359298992515654651080492391625827617,            // 18^147
    191213265769831372286179520084128641258,            // 18^154
    218049793855157992570938056229892893589,            // 18^161
    248652793041613380567795520750960012282,            // 18^168
    283550882549632193861238568804430594202,            // 18^175
    323346872605688987884591669129361771279,            // 18^182
    184364088525767284952804747951506893851,            // 18^189
    210239343674659878049714940191476577896,            // 18^196
    239746156543789930881397013133212694240,            // 18^203
    273394211439632029990640781047045990695,            // 18^210
    311764726184655516350818907233192566650,            // 18^217
    177760245875679923341865534533290364422,            // 18^224
    202708660472811443153981984577454872377,            // 18^231
    231158551948781603682221903513060749406,            // 18^238
    263601347936609267755115798860540547258,            // 18^245
];
const BASE18_LARGE_EXPONENT: [i32; 75] = [
    -1266,                                              // 18^-273
    -1237,                                              // 18^-266
    -1208,                                              // 18^-259
    -1178,                                              // 18^-252
    -1149,                                              // 18^-245
    -1120,                                              // 18^-238
    -1091,                                              // 18^-231
    -1062,                                              // 18^-224
    -1032,                                              // 18^-217
    -1003,                                              // 18^-210
    -974,                                               // 18^-203
    -945,                                               // 18^-196
    -916,                                               // 18^-189
    -886,                                               // 18^-182
    -857,                                               // 18^-175
    -828,                                               // 18^-168
    -799,                                               // 18^-161
    -770,                                               // 18^-154
    -740,                                               // 18^-147
    -711,                                               // 18^-140
    -682,                                               // 18^-133
    -653,                                               // 18^-126
    -624,                                               // 18^-119
    -595,                                               // 18^-112
    -565,                                               // 18^-105
    -536,                                               // 18^-98
    -507,                                               // 18^-91
    -478,                                               // 18^-84
    -449,                                               // 18^-77
    -419,                                               // 18^-70
    -390,                                               // 18^-63
    -361,                                               // 18^-56
    -332,                                               // 18^-49
    -303,                                               // 18^-42
    -273,                                               // 18^-35
    -244,                                               // 18^-28
    -215,                                               // 18^-21
    -186,                                               // 18^-14
    -157,                                               // 18^-7
    -127,                                               // 18^0
    -98,                                                // 18^7
    -69,                                                // 18^14
    -40,                                                // 18^21
    -11,                                                // 18^28
    18,                                                 // 18^35
    48,                                                 // 18^42
    77,                                                 // 18^49
    106,                                                // 18^56
    135,                                                // 18^63
    164,                                                // 18^70
    194,                                                // 18^77
    223,                                                // 18^84
    252,                                                // 18^91
    281,                                                // 18^98
    310,                                                // 18^105
    340,                                                // 18^112
    369,                                                // 18^119
    398,                                                // 18^126
    427,                                                // 18^133
    456,                                                // 18^140
    485,                                                // 18^147
    515,                                                // 18^154
    544,                                                // 18^161
    573,                                                // 18^168
    602,                                                // 18^175
    631,                                                // 18^182
    661,                                                // 18^189
    690,                                                // 18^196
    719,                                                // 18^203
    748,                                                // 18^210
    777,                                                // 18^217
    807,                                                // 18^224
    836,                                                // 18^231
    865,                                                // 18^238
    894,                                                // 18^245
];
const BASE18_SMALL_INT_POWERS: [u128; 7] = [1, 18, 324, 5832, 104976, 1889568, 34012224];
const BASE18_STEP: i32 = 7;
const BASE18_BIAS: i32 = 273;

// BASE19

const BASE19_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 19^0
    202042655359307212681378673162612375552,            // 19^1
    239925653239177315059137174380602195968,            // 19^2
    284911713221523061632725394576965107712,            // 19^3
    338332659450558635688861406060146065408,            // 19^4
    200885016548769189940261459848211726336,            // 19^5
    238550957151663413054060483569751425024,            // 19^6
];
const BASE19_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 19^0
    -123,                                               // 19^1
    -119,                                               // 19^2
    -115,                                               // 19^3
    -111,                                               // 19^4
    -106,                                               // 19^5
    -102,                                               // 19^6
];
const BASE19_LARGE_MANTISSA: [u128; 74] = [
    211773375714873706954526795446522416463,            // 19^-273
    176297720171655089222657736919708805072,            // 19^-266
    293529685049453215881543661471874663538,            // 19^-259
    244358452058655253685245151337508023797,            // 19^-252
    203424239979142604035433607238695466107,            // 19^-245
    338694414393807804581041366088196084612,            // 19^-238
    281957318246159342222500231430112814088,            // 19^-231
    234724654242834897063227749239967575385,            // 19^-224
    195404267752744776874588226122954622761,            // 19^-217
    325341434449269614607198466720898964379,            // 19^-210
    270841190386491955899443752473167688523,            // 19^-203
    225470667559284387835624717321131334248,            // 19^-196
    187700481810335462307725392489901825033,            // 19^-189
    312514893872556142260088491662230499613,            // 19^-182
    260163314313871975296322899252180920269,            // 19^-175
    216581518007204247993872678149555585108,            // 19^-168
    180300416551865148993646942230063074210,            // 19^-161
    300194037865176954165930472692882676095,            // 19^-154
    249906412012854191998973084780461202007,            // 19^-147
    208042821933683442841807340743863291908,            // 19^-140
    173192097831823827445985218112992713522,            // 19^-133
    288358929883670982310136287441393538062,            // 19^-126
    240053886651337192078493312543483732688,            // 19^-119
    199840762759316398351061941853830886312,            // 19^-112
    332728047167428932050765437002481330484,            // 19^-105
    276990419380016367557580152018464109110,            // 19^-98
    230589795725005243268941857960660823918,            // 19^-91
    191962068621409314298003076822671000161,            // 19^-84
    319610290416807823632571053703933063036,            // 19^-77
    266070110813870135824430347248350305665,            // 19^-70
    221498825260546806854707890357129976717,            // 19^-63
    184393990898599943457278294759382590034,            // 19^-56
    307009699392470911375170613720126268120,            // 19^-49
    255580333886495993526502175170654921405,            // 19^-42
    212766265035907553834903279232863277537,            // 19^-35
    177124283582141234157272719875613305569,            // 19^-28
    294905884907949220984741236292709278206,            // 19^-21
    245504114948215358970059027289947453449,            // 19^-14
    204377984777481442195434041001194890061,            // 19^-7
    170141183460469231731687303715884105728,            // 19^0
    283279261617600303001696824239079817216,            // 19^7
    235825149533115640143913831779140632576,            // 19^14
    196320411295724208786653789764804673536,            // 19^21
    326866782169983241283381922259938632192,            // 19^28
    272111016325296008481488740678554545334,            // 19^35
    226527775976573799542251126784141726274,            // 19^42
    188580506522192103284103254483314456872,            // 19^49
    313980104972594785419533005246072709649,            // 19^56
    261383077542535819285819131850454151532,            // 19^63
    217596950072905364683647533344696965413,            // 19^70
    181145746412467300065049161663171614429,            // 19^77
    301601483222404786925353682146130769038,            // 19^84
    251078086246727305139312855014591629283,            // 19^91
    209018220732132084019166621358085321302,            // 19^98
    174004100680832526758226172220060851297,            // 19^105
    289710886904423785696066284771374024143,            // 19^112
    241179367792317286239104434314982328918,            // 19^119
    200777706596478115292553836948931709029,            // 19^126
    334288026667806625723834320874258952107,            // 19^133
    278289075684203471850125042859239363909,            // 19^140
    231670904929322723497179168675130067798,            // 19^147
    192862073578757583112120353708241916963,            // 19^154
    321108767943438131876044073228689471164,            // 19^161
    267317567774791732969648063348207624890,            // 19^168
    222537311885602256946640972609673172416,            // 19^175
    185258513286308067365753274779077551187,            // 19^182
    308449099652072036761706748552096805527,            // 19^189
    256778610031103646805005837172547824357,            // 19^196
    213763809470930565948441961387708938512,            // 19^203
    177954722295557497937779873950595261616,            // 19^210
    296288537013515928119667265056482304979,            // 19^217
    246655149223317717362685108852956230915,            // 19^224
    205336201162591117777390249036062770913,            // 19^231
    170938882243688352586356584716130516258,            // 19^238
];
const BASE19_LARGE_EXPONENT: [i32; 74] = [
    -1287,                                              // 19^-273
    -1257,                                              // 19^-266
    -1228,                                              // 19^-259
    -1198,                                              // 19^-252
    -1168,                                              // 19^-245
    -1139,                                              // 19^-238
    -1109,                                              // 19^-231
    -1079,                                              // 19^-224
    -1049,                                              // 19^-217
    -1020,                                              // 19^-210
    -990,                                               // 19^-203
    -960,                                               // 19^-196
    -930,                                               // 19^-189
    -901,                                               // 19^-182
    -871,                                               // 19^-175
    -841,                                               // 19^-168
    -811,                                               // 19^-161
    -782,                                               // 19^-154
    -752,                                               // 19^-147
    -722,                                               // 19^-140
    -692,                                               // 19^-133
    -663,                                               // 19^-126
    -633,                                               // 19^-119
    -603,                                               // 19^-112
    -574,                                               // 19^-105
    -544,                                               // 19^-98
    -514,                                               // 19^-91
    -484,                                               // 19^-84
    -455,                                               // 19^-77
    -425,                                               // 19^-70
    -395,                                               // 19^-63
    -365,                                               // 19^-56
    -336,                                               // 19^-49
    -306,                                               // 19^-42
    -276,                                               // 19^-35
    -246,                                               // 19^-28
    -217,                                               // 19^-21
    -187,                                               // 19^-14
    -157,                                               // 19^-7
    -127,                                               // 19^0
    -98,                                                // 19^7
    -68,                                                // 19^14
    -38,                                                // 19^21
    -9,                                                 // 19^28
    21,                                                 // 19^35
    51,                                                 // 19^42
    81,                                                 // 19^49
    110,                                                // 19^56
    140,                                                // 19^63
    170,                                                // 19^70
    200,                                                // 19^77
    229,                                                // 19^84
    259,                                                // 19^91
    289,                                                // 19^98
    319,                                                // 19^105
    348,                                                // 19^112
    378,                                                // 19^119
    408,                                                // 19^126
    437,                                                // 19^133
    467,                                                // 19^140
    497,                                                // 19^147
    527,                                                // 19^154
    556,                                                // 19^161
    586,                                                // 19^168
    616,                                                // 19^175
    646,                                                // 19^182
    675,                                                // 19^189
    705,                                                // 19^196
    735,                                                // 19^203
    765,                                                // 19^210
    794,                                                // 19^217
    824,                                                // 19^224
    854,                                                // 19^231
    884,                                                // 19^238
];
const BASE19_SMALL_INT_POWERS: [u128; 7] = [1, 19, 361, 6859, 130321, 2476099, 47045881];
const BASE19_STEP: i32 = 7;
const BASE19_BIAS: i32 = 273;

// BASE20

const BASE20_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 20^0
    212676479325586539664609129644855132160,            // 20^1
    265845599156983174580761412056068915200,            // 20^2
    332306998946228968225951765070086144000,            // 20^3
    207691874341393105141219853168803840000,            // 20^4
    259614842926741381426524816461004800000,            // 20^5
    324518553658426726783156020576256000000,            // 20^6
];
const BASE20_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 20^0
    -123,                                               // 20^1
    -119,                                               // 20^2
    -115,                                               // 20^3
    -110,                                               // 20^4
    -106,                                               // 20^5
    -102,                                               // 20^6
];
const BASE20_LARGE_MANTISSA: [u128; 72] = [
    219444962751747547330237450047488370802,            // 20^-266
    261598781051334795153424084243164504531,            // 20^-259
    311850048364799970571308236412006025948,            // 20^-252
    185877113559722882849757812268737570016,            // 20^-245
    221582786512044528543660416923448526878,            // 20^-238
    264147265567832623176169892458258303259,            // 20^-231
    314888078651228693933689466069052580904,            // 20^-224
    187687920720117505749278942387731421532,            // 20^-217
    223741436863085634409521749481834675708,            // 20^-210
    266720577315194170963194071628850311885,            // 20^-203
    317955705303185189918510999237120523316,            // 20^-196
    189516368689051383685178160212707831452,            // 20^-189
    225921116696657399755928707376370229068,            // 20^-182
    269318958159276723570738682003462587676,            // 20^-175
    321053216647239593947814323906257853121,            // 20^-168
    191362629322552438943275406304751547051,            // 20^-161
    228122030881109760932058580285014566244,            // 20^-154
    271942652322184754529069161754863937192,            // 20^-147
    324180903818827574883781864350871964922,            // 20^-140
    193226876150862917234767594546599367214,            // 20^-133
    230344386280611654799899571593522271174,            // 20^-126
    274591906405224388599276031963255728690,            // 20^-119
    327339060789614187001318969682759915221,            // 20^-112
    195109284394749514461349826862072894109,            // 20^-105
    232588391774594204975783618524161450993,            // 20^-98
    277266969412081485957841418414308370343,            // 20^-91
    330527984395124299475957654016385519914,            // 20^-84
    197010030981972396061395200500718069025,            // 20^-77
    234854258277383322788948059678933702737,            // 20^-70
    279968092772225526319680285071055534765,            // 20^-63
    333747974362642200374222141588992517906,            // 20^-56
    198929294563914656862152899258728336040,            // 20^-49
    237142198758023568227473377297792835283,            // 20^-42
    282695530364541492733327600118866962532,            // 20^-35
    336999333339382997433337688587745383420,            // 20^-28
    200867255532373784442745261542645325315,            // 20^-21
    239452428260295134118491722992235809940,            // 20^-14
    285449538541191976211657193889899027276,            // 20^-7
    170141183460469231731687303715884105728,            // 20^0
    202824096036516704239472512860160000000,            // 20^7
    241785163922925834941235200000000000000,            // 20^14
    288230376151711744000000000000000000000,            // 20^21
    171798691840000000000000000000000000000,            // 20^28
    204800000000000000000000000000000000000,            // 20^35
    244140625000000000000000000000000000000,            // 20^42
    291038304567337036132812500000000000000,            // 20^49
    173472347597680709441192448139190673828,            // 20^56
    206795153138256918717852173017490713391,            // 20^63
    246519032881566189191165176650870696772,            // 20^70
    293873587705571876992184134305561419454,            // 20^77
    175162308040602133865466197911239516410,            // 20^84
    208809742975952784854729411496209521782,            // 20^91
    248920611114445668285762562151204969623,            // 20^98
    296736492054993710858538820923811161069,            // 20^105
    176868732008334225927912486150152183216,            // 20^112
    210843958864610464486971481025400380154,            // 20^119
    251345585423243599518503524095297312920,            // 20^126
    299627286700300692937974362486955300474,            // 20^133
    178591779887855465971216179422709524914,            // 20^140
    212897992000407535995502685812365442412,            // 20^147
    253794183731564922327402455583054354682,            // 20^154
    302546243347602990063908643225496238091,            // 20^161
    180331613628627651967947866455016278082,            // 20^168
    214972035442146840057310898846407268146,            // 20^175
    256266636183436918326986907537468991453,            // 20^182
    305493636349960468205197939321361769978,            // 20^189
    182088396757817547443627082897044283139,            // 20^196
    217066284129402097992452481862359384464,            // 20^203
    258763175164940474024358370140027266101,            // 20^210
    308469742733169167070816004443201143863,            // 20^217
    183862294395666818064937594201088633455,            // 20^224
    219180934900840303975269310714112083263,            // 20^231
];
const BASE20_LARGE_EXPONENT: [i32; 72] = [
    -1277,                                              // 20^-266
    -1247,                                              // 20^-259
    -1217,                                              // 20^-252
    -1186,                                              // 20^-245
    -1156,                                              // 20^-238
    -1126,                                              // 20^-231
    -1096,                                              // 20^-224
    -1065,                                              // 20^-217
    -1035,                                              // 20^-210
    -1005,                                              // 20^-203
    -975,                                               // 20^-196
    -944,                                               // 20^-189
    -914,                                               // 20^-182
    -884,                                               // 20^-175
    -854,                                               // 20^-168
    -823,                                               // 20^-161
    -793,                                               // 20^-154
    -763,                                               // 20^-147
    -733,                                               // 20^-140
    -702,                                               // 20^-133
    -672,                                               // 20^-126
    -642,                                               // 20^-119
    -612,                                               // 20^-112
    -581,                                               // 20^-105
    -551,                                               // 20^-98
    -521,                                               // 20^-91
    -491,                                               // 20^-84
    -460,                                               // 20^-77
    -430,                                               // 20^-70
    -400,                                               // 20^-63
    -370,                                               // 20^-56
    -339,                                               // 20^-49
    -309,                                               // 20^-42
    -279,                                               // 20^-35
    -249,                                               // 20^-28
    -218,                                               // 20^-21
    -188,                                               // 20^-14
    -158,                                               // 20^-7
    -127,                                               // 20^0
    -97,                                                // 20^7
    -67,                                                // 20^14
    -37,                                                // 20^21
    -6,                                                 // 20^28
    24,                                                 // 20^35
    54,                                                 // 20^42
    84,                                                 // 20^49
    115,                                                // 20^56
    145,                                                // 20^63
    175,                                                // 20^70
    205,                                                // 20^77
    236,                                                // 20^84
    266,                                                // 20^91
    296,                                                // 20^98
    326,                                                // 20^105
    357,                                                // 20^112
    387,                                                // 20^119
    417,                                                // 20^126
    447,                                                // 20^133
    478,                                                // 20^140
    508,                                                // 20^147
    538,                                                // 20^154
    568,                                                // 20^161
    599,                                                // 20^168
    629,                                                // 20^175
    659,                                                // 20^182
    689,                                                // 20^189
    720,                                                // 20^196
    750,                                                // 20^203
    780,                                                // 20^210
    810,                                                // 20^217
    841,                                                // 20^224
    871,                                                // 20^231
];
const BASE20_SMALL_INT_POWERS: [u128; 7] = [1, 20, 400, 8000, 160000, 3200000, 64000000];
const BASE20_STEP: i32 = 7;
const BASE20_BIAS: i32 = 266;

// BASE21

const BASE21_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 21^0
    223310303291865866647839586127097888768,            // 21^1
    293094773070573949975289456791815979008,            // 21^2
    192343444827564154671283706019629236224,            // 21^3
    252450771336177953006059864150763372544,            // 21^4
    331341637378733563320453571697876926464,            // 21^5
    217442949529793900929047656426731732992,            // 21^6
];
const BASE21_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 21^0
    -123,                                               // 21^1
    -119,                                               // 21^2
    -114,                                               // 21^3
    -110,                                               // 21^4
    -106,                                               // 21^5
    -101,                                               // 21^6
];
const BASE21_LARGE_MANTISSA: [u128; 72] = [
    265792046637109917415346664544311644566,            // 21^-266
    222919047571269915259553685343512655124,            // 21^-259
    186961582932271121624867982904669033286,            // 21^-252
    313608315425491608924946770674906010971,            // 21^-245
    263022418727710133477288744271513099136,            // 21^-238
    220596168374913987040298637494305910349,            // 21^-231
    185013390634471631571140317809259206000,            // 21^-224
    310340428541697166126551714455108139315,            // 21^-217
    260281651120390792657949798157903984485,            // 21^-210
    218297494233351046265168261721480533031,            // 21^-203
    183085499048559996047315841365348324923,            // 21^-196
    307106593958686861611740529249331599561,            // 21^-189
    257569443082684061423017949935163592863,            // 21^-182
    216022772923099798407562907443165238977,            // 21^-175
    181177696635406520735133253192775193924,            // 21^-168
    303906456841905544859103298218510174905,            // 21^-161
    254885497015839035366704878377106222377,            // 21^-154
    213771754848918589809660748287013123679,            // 21^-147
    179289774060178761532557823153782969422,            // 21^-140
    300739666054273966520895709937608788747,            // 21^-133
    252229518422167527105381200194678741672,            // 21^-126
    211544193016418411141172507362190098875,            // 21^-119
    177421524169372127003871154186445625381,            // 21^-112
    297605874117660039208334869188863219128,            // 21^-105
    249601215872730120248972738434931034589,            // 21^-98
    209339843004961281067680970866486691183,            // 21^-91
    175572741968079828414917198043029764862,            // 21^-84
    294504737174751578579986057113044461622,            // 21^-77
    247000300975358943781849899984000214119,            // 21^-70
    207158462940841036386486635842909509778,            // 21^-63
    173743224597499683284324187350601048284,            // 21^-56
    291435914951326341195521446658582144907,            // 21^-49
    244426488343013658131803347774089798564,            // 21^-42
    204999813470743585875109991683322467328,            // 21^-35
    171932771312675304370424739613867083896,            // 21^-28
    288399070718915219190771894363169343679,            // 21^-21
    241879495562467180767563428674995339169,            // 21^-14
    202863657735483715761934442388454879916,            // 21^-7
    170141183460469231731687303715884105728,            // 21^0
    285393871257854494969375049060085399552,            // 21^7
    239359043163317715346200180609446313984,            // 21^14
    200749761344015565073276401119215484928,            // 21^21
    336736528915531181897146582027467512352,            // 21^28
    282419986820723101796180638245767114566,            // 21^35
    236864854587323684235605162154192799609,            // 21^42
    198657892347713919139770232958888414539,            // 21^49
    333227639539799771559276379603105082665,            // 21^56
    279477091096160878422591085338554667498,            // 21^63
    234396656158058199668034426556911280906,            // 21^70
    196587821214923499260650393397155679946,            // 21^77
    329755313778627116894330266634709348105,            // 21^84
    276564861173063847678827937275753188756,            // 21^91
    231954177050879743842271364358265802922,            // 21^98
    194539320805773455930561882331086488568,            // 21^105
    326319170628861950820780822968026849107,            // 21^112
    273682977505152590337309001442167333935,            // 21^119
    229537149263215762988196018745023784859,            // 21^126
    192512166347254302133159603748230049238,            // 21^133
    322918833057513041780316846850363342758,            // 21^140
    270831123875909826489436864951365328546,            // 21^147
    227145307585155914742230901032159391581,            // 21^154
    190506135408554552002160365053046193282,            // 21^161
    319553927960379009120613001977483142041,            // 21^168
    268008987363883357189628558586004761583,            // 21^175
    224778389570351742159039695242233732277,            // 21^182
    188521007876654358646847770858984158583,            // 21^189
    316224086121109227441755855330741121363,            // 21^196
    265216258308350559209803685525363124313,            // 21^203
    222436135507219581307712678266362793773,            // 21^210
    186556565932173473138426942632517671964,            // 21^217
    312928942170691327838033505846831903751,            // 21^224
    262452630275340665419193194878653363790,            // 21^231
];
const BASE21_LARGE_EXPONENT: [i32; 72] = [
    -1296,                                              // 21^-266
    -1265,                                              // 21^-259
    -1234,                                              // 21^-252
    -1204,                                              // 21^-245
    -1173,                                              // 21^-238
    -1142,                                              // 21^-231
    -1111,                                              // 21^-224
    -1081,                                              // 21^-217
    -1050,                                              // 21^-210
    -1019,                                              // 21^-203
    -988,                                               // 21^-196
    -958,                                               // 21^-189
    -927,                                               // 21^-182
    -896,                                               // 21^-175
    -865,                                               // 21^-168
    -835,                                               // 21^-161
    -804,                                               // 21^-154
    -773,                                               // 21^-147
    -742,                                               // 21^-140
    -712,                                               // 21^-133
    -681,                                               // 21^-126
    -650,                                               // 21^-119
    -619,                                               // 21^-112
    -589,                                               // 21^-105
    -558,                                               // 21^-98
    -527,                                               // 21^-91
    -496,                                               // 21^-84
    -466,                                               // 21^-77
    -435,                                               // 21^-70
    -404,                                               // 21^-63
    -373,                                               // 21^-56
    -343,                                               // 21^-49
    -312,                                               // 21^-42
    -281,                                               // 21^-35
    -250,                                               // 21^-28
    -220,                                               // 21^-21
    -189,                                               // 21^-14
    -158,                                               // 21^-7
    -127,                                               // 21^0
    -97,                                                // 21^7
    -66,                                                // 21^14
    -35,                                                // 21^21
    -5,                                                 // 21^28
    26,                                                 // 21^35
    57,                                                 // 21^42
    88,                                                 // 21^49
    118,                                                // 21^56
    149,                                                // 21^63
    180,                                                // 21^70
    211,                                                // 21^77
    241,                                                // 21^84
    272,                                                // 21^91
    303,                                                // 21^98
    334,                                                // 21^105
    364,                                                // 21^112
    395,                                                // 21^119
    426,                                                // 21^126
    457,                                                // 21^133
    487,                                                // 21^140
    518,                                                // 21^147
    549,                                                // 21^154
    580,                                                // 21^161
    610,                                                // 21^168
    641,                                                // 21^175
    672,                                                // 21^182
    703,                                                // 21^189
    733,                                                // 21^196
    764,                                                // 21^203
    795,                                                // 21^210
    826,                                                // 21^217
    856,                                                // 21^224
    887,                                                // 21^231
];
const BASE21_SMALL_INT_POWERS: [u128; 7] = [1, 21, 441, 9261, 194481, 4084101, 85766121];
const BASE21_STEP: i32 = 7;
const BASE21_BIAS: i32 = 266;

// BASE22

const BASE22_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 22^0
    233944127258145193631070042609340645376,            // 22^1
    321673174979949641242721308587843387392,            // 22^2
    221150307798715378354370899654142328832,            // 22^3
    304081673223233645237259987024445702144,            // 22^4
    209056150340973131100616241079306420224,            // 22^5
    287452206718838055263347331484046327808,            // 22^6
];
const BASE22_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 22^0
    -123,                                               // 22^1
    -119,                                               // 22^2
    -114,                                               // 22^3
    -110,                                               // 22^4
    -105,                                               // 22^5
    -101,                                               // 22^6
];
const BASE22_LARGE_MANTISSA: [u128; 70] = [
    170993699408656992611557257796811971606,            // 22^-259
    198613611477559667549559644399735817047,            // 22^-252
    230694854843066209807480559654047740794,            // 22^-245
    267958050080955588510051414081700394572,            // 22^-238
    311240216657766425412953324616068813788,            // 22^-231
    180756787183491671385793896073455879749,            // 22^-224
    209953690842110549054787911387661891558,            // 22^-217
    243866650791248218437107825156701479612,            // 22^-210
    283257432291885574825362738029166670391,            // 22^-203
    329010845428281790557809997379933231677,            // 22^-196
    191077310017213090920037889613280023761,            // 22^-189
    221941245467987325560887198530024987097,            // 22^-182
    257790506028392555656294807759853932728,            // 22^-175
    299430350849140679073407301022315056449,            // 22^-168
    173898054647064197545039940705011138943,            // 22^-161
    201987095324676314922450395008827008097,            // 22^-154
    234613243722037545951821958217296505941,            // 22^-147
    272509360270263083361596599896866808472,            // 22^-140
    316526681344939644363801942778159729297,            // 22^-133
    183826970023851061892407973678416422311,            // 22^-126
    213519787744680626489456759979414929795,            // 22^-119
    248008764723795396613310192753363562635,            // 22^-112
    288068604926548520494484698905215058937,            // 22^-105
    334599147077506390730919115808572688418,            // 22^-98
    194322788582847037606425160078121997271,            // 22^-91
    225710952896522753756036626884027523975,            // 22^-84
    262169118861406100263284327408804984594,            // 22^-77
    304516223083230613247976703037939646270,            // 22^-70
    176851740822108453297561032060281913714,            // 22^-63
    205417878332621336576645714920472857997,            // 22^-56
    238598187060653380400994399373084663429,            // 22^-49
    277137975188549744820727493204211809261,            // 22^-42
    321902937477411396463387072352875080544,            // 22^-35
    186949300409097210175717446748311788922,            // 22^-28
    217146455371525722164937611369356262209,            // 22^-21
    252221233121680633945860233133208134270,            // 22^-14
    292961496095243353908740467141073506540,            // 22^-7
    170141183460469231731687303715884105728,            // 22^0
    197623392119201162993551290395281850368,            // 22^7
    229544689406569328704727047276706398208,            // 22^14
    266622103131276669014944105065242165248,            // 22^21
    309688479667831891620434363534947647488,            // 22^28
    179855595827611186243917689814617910464,            // 22^35
    208906933736774069538597567757512183241,            // 22^42
    242650815297018604365857953016443636978,            // 22^49
    281845207868958552685905724295461354592,            // 22^56
    327370510177191550022527822209865447333,            // 22^63
    190124664073591590972067520727552350138,            // 22^70
    220834722526051754798575401303998056567,            // 22^77
    256505250966711193359488809204377717494,            // 22^84
    297937493800295372760917121026044042670,            // 22^91
    173031058579617612943075122066206208260,            // 22^98
    200980056932689281584830175015332323410,            // 22^105
    233443542542281837469979323535186565406,            // 22^112
    271150722048713022289973166238934106646,            // 22^119
    314948587854906379895896832699151980516,            // 22^126
    182910471789153927982518189465009158829,            // 22^133
    212455251303071888680280266327561854998,            // 22^140
    246772278069909496307705991140052073087,            // 22^147
    286632393646709698979385809127728921005,            // 22^154
    332930950470730393610585733499849696474,            // 22^161
    193353961796034982060992467369794830962,            // 22^168
    224585635486054469192295827945894724481,            // 22^175
    260862033537650797156363766298784568546,            // 22^182
    302998009619470597652934458975103615488,            // 22^189
    175970018688269509849993361647138860754,            // 22^196
    204393735233038880379975139336743154541,            // 22^203
    237408618319925875163860351801153447367,            // 22^210
    275756260280259146490263920764281465178,            // 22^217
    320298039817924375055421761218548035863,            // 22^224
];
const BASE22_LARGE_EXPONENT: [i32; 70] = [
    -1282,                                              // 22^-259
    -1251,                                              // 22^-252
    -1220,                                              // 22^-245
    -1189,                                              // 22^-238
    -1158,                                              // 22^-231
    -1126,                                              // 22^-224
    -1095,                                              // 22^-217
    -1064,                                              // 22^-210
    -1033,                                              // 22^-203
    -1002,                                              // 22^-196
    -970,                                               // 22^-189
    -939,                                               // 22^-182
    -908,                                               // 22^-175
    -877,                                               // 22^-168
    -845,                                               // 22^-161
    -814,                                               // 22^-154
    -783,                                               // 22^-147
    -752,                                               // 22^-140
    -721,                                               // 22^-133
    -689,                                               // 22^-126
    -658,                                               // 22^-119
    -627,                                               // 22^-112
    -596,                                               // 22^-105
    -565,                                               // 22^-98
    -533,                                               // 22^-91
    -502,                                               // 22^-84
    -471,                                               // 22^-77
    -440,                                               // 22^-70
    -408,                                               // 22^-63
    -377,                                               // 22^-56
    -346,                                               // 22^-49
    -315,                                               // 22^-42
    -284,                                               // 22^-35
    -252,                                               // 22^-28
    -221,                                               // 22^-21
    -190,                                               // 22^-14
    -159,                                               // 22^-7
    -127,                                               // 22^0
    -96,                                                // 22^7
    -65,                                                // 22^14
    -34,                                                // 22^21
    -3,                                                 // 22^28
    29,                                                 // 22^35
    60,                                                 // 22^42
    91,                                                 // 22^49
    122,                                                // 22^56
    153,                                                // 22^63
    185,                                                // 22^70
    216,                                                // 22^77
    247,                                                // 22^84
    278,                                                // 22^91
    310,                                                // 22^98
    341,                                                // 22^105
    372,                                                // 22^112
    403,                                                // 22^119
    434,                                                // 22^126
    466,                                                // 22^133
    497,                                                // 22^140
    528,                                                // 22^147
    559,                                                // 22^154
    590,                                                // 22^161
    622,                                                // 22^168
    653,                                                // 22^175
    684,                                                // 22^182
    715,                                                // 22^189
    747,                                                // 22^196
    778,                                                // 22^203
    809,                                                // 22^210
    840,                                                // 22^217
    871,                                                // 22^224
];
const BASE22_SMALL_INT_POWERS: [u128; 7] = [1, 22, 484, 10648, 234256, 5153632, 113379904];
const BASE22_STEP: i32 = 7;
const BASE22_BIAS: i32 = 259;

// BASE23

const BASE23_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 23^0
    244577951224424520614300499091583401984,            // 23^1
    175790402442555124191528483722075570176,            // 23^2
    252698703511172991025322195350483632128,            // 23^3
    181627193148655587299450327908160110592,            // 23^4
    261089090151192406742959846367980158976,            // 23^5
    187657783546169542346502389576985739264,            // 23^6
];
const BASE23_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 23^0
    -123,                                               // 23^1
    -118,                                               // 23^2
    -114,                                               // 23^3
    -109,                                               // 23^4
    -105,                                               // 23^5
    -100,                                               // 23^6
];
const BASE23_LARGE_MANTISSA: [u128; 69] = [
    177660070384959299518117426820409879479,            // 23^-252
    281679224484842507661590842703493245235,            // 23^-245
    223300557447880794264720046366338636067,            // 23^-238
    177021003404592607294928857171705382014,            // 23^-231
    280665986726726659567026238400885338462,            // 23^-224
    222497315545222527698408099502330410230,            // 23^-217
    176384235233432227183413281705333529169,            // 23^-210
    279656393720034524921314175179274359595,            // 23^-203
    221696963010873772243653930801015609536,            // 23^-196
    175749757602354687973874432000535316950,            // 23^-189
    278650432354108872233513992865613552171,            // 23^-182
    220899489451391157505367921930602163472,            // 23^-175
    175117562271981659025201047697761937562,            // 23^-168
    277648089565453253577594524944387660343,            // 23^-161
    220104884510717915806439152820624263438,            // 23^-154
    174487641032572953025011241897191693675,            // 23^-147
    276649352337562360960754521131809259049,            // 23^-140
    219313137870049397588841701361789229875,            // 23^-133
    173859985703919913633146478806481352851,            // 23^-126
    275654207700752992922523689325318624601,            // 23^-119
    218524239247699070573788964393354746146,            // 23^-112
    173234588135239186624038395745277208596,            // 23^-105
    274662642731995629169562809047557585825,            // 23^-98
    217738178398965000940790150805824878955,            // 23^-91
    172611440205066873148451849294509762826,            // 23^-84
    273674644554746611058977369746265857416,            // 23^-77
    216954945115996814791722634413562231827,            // 23^-70
    171990533821153063740069798803230423676,            // 23^-63
    272690200338780925750826785583705218666,            // 23^-56
    216174529227663138172270874689475697712,            // 23^-49
    171371860920356751697330024074687407971,            // 23^-42
    271709297300025591858350535820526217684,            // 23^-35
    215396920599419513929297198496715326537,            // 23^-28
    170755413468541124475850272054650469150,            // 23^-21
    270731922700393644432243678371210997948,            // 23^-14
    214622109133176793688901966303396671549,            // 23^-7
    170141183460469231731687303715884105728,            // 23^0
    269758063847618717123097185016917000192,            // 23^7
    213850084767170003246100602438595641344,            // 23^14
    339058325839400057321133061640411938816,            // 23^21
    268787708095090219373873551177395072962,            // 23^28
    213080837475827679663192730864754774513,            // 23^35
    337838687796969586566185112723027994705,            // 23^42
    267820842841689106502015241773782572538,            // 23^49
    212314357269641678380024305538561269739,            // 23^56
    336623436955327832661614051077606366471,            // 23^63
    266857455531624240538482847903341038248,            // 23^70
    211550634195037448645447237257455979321,            // 23^77
    335412557533128124785597638278283462337,            // 23^84
    265897533654269339698691446779528819034,            // 23^91
    210789658334244775585362676204225495858,            // 23^98
    334206033805791401974785682123232789907,            // 23^105
    264941064744000514367957679290425962065,            // 23^112
    210031419805168987228793890184514974036,            // 23^119
    333003850105302012456495057986285588216,            // 23^126
    263988036380034387491686584861354377998,            // 23^133
    209275908761262624819472661494385533289,            // 23^140
    331805990820004247517955628678622148639,            // 23^147
    263038436186266797268116360982380732098,            // 23^154
    208523115391397574746439388866966551936,            // 23^161
    330612440394399607270379273154482344971,            // 23^168
    262092251831112080049001043331284413642,            // 23^175
    207773029919737660433151703156509162570,            // 23^182
    329423183328944795675128680336602321085,            // 23^189
    261149471027342931361145730086041639640,            // 23^196
    207025642605611692530569464171386341133,            // 23^203
    328238204179850442208732821518504518810,            // 23^210
    260210081531930842969216498623209392426,            // 23^217
    206280943743386975765635578931435897048,            // 23^224
];
const BASE23_LARGE_EXPONENT: [i32; 69] = [
    -1267,                                              // 23^-252
    -1236,                                              // 23^-245
    -1204,                                              // 23^-238
    -1172,                                              // 23^-231
    -1141,                                              // 23^-224
    -1109,                                              // 23^-217
    -1077,                                              // 23^-210
    -1046,                                              // 23^-203
    -1014,                                              // 23^-196
    -982,                                               // 23^-189
    -951,                                               // 23^-182
    -919,                                               // 23^-175
    -887,                                               // 23^-168
    -856,                                               // 23^-161
    -824,                                               // 23^-154
    -792,                                               // 23^-147
    -761,                                               // 23^-140
    -729,                                               // 23^-133
    -697,                                               // 23^-126
    -666,                                               // 23^-119
    -634,                                               // 23^-112
    -602,                                               // 23^-105
    -571,                                               // 23^-98
    -539,                                               // 23^-91
    -507,                                               // 23^-84
    -476,                                               // 23^-77
    -444,                                               // 23^-70
    -412,                                               // 23^-63
    -381,                                               // 23^-56
    -349,                                               // 23^-49
    -317,                                               // 23^-42
    -286,                                               // 23^-35
    -254,                                               // 23^-28
    -222,                                               // 23^-21
    -191,                                               // 23^-14
    -159,                                               // 23^-7
    -127,                                               // 23^0
    -96,                                                // 23^7
    -64,                                                // 23^14
    -33,                                                // 23^21
    -1,                                                 // 23^28
    31,                                                 // 23^35
    62,                                                 // 23^42
    94,                                                 // 23^49
    126,                                                // 23^56
    157,                                                // 23^63
    189,                                                // 23^70
    221,                                                // 23^77
    252,                                                // 23^84
    284,                                                // 23^91
    316,                                                // 23^98
    347,                                                // 23^105
    379,                                                // 23^112
    411,                                                // 23^119
    442,                                                // 23^126
    474,                                                // 23^133
    506,                                                // 23^140
    537,                                                // 23^147
    569,                                                // 23^154
    601,                                                // 23^161
    632,                                                // 23^168
    664,                                                // 23^175
    696,                                                // 23^182
    727,                                                // 23^189
    759,                                                // 23^196
    791,                                                // 23^203
    822,                                                // 23^210
    854,                                                // 23^217
    886,                                                // 23^224
];
const BASE23_SMALL_INT_POWERS: [u128; 7] = [1, 23, 529, 12167, 279841, 6436343, 148035889];
const BASE23_STEP: i32 = 7;
const BASE23_BIAS: i32 = 252;

// BASE24

const BASE24_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 24^0
    255211775190703847597530955573826158592,            // 24^1
    191408831393027885698148216680369618944,            // 24^2
    287113247089541828547222325020554428416,            // 24^3
    215334935317156371410416743765415821312,            // 24^4
    323002402975734557115625115648123731968,            // 24^5
    242251802231800917836718836736092798976,            // 24^6
];
const BASE24_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 24^0
    -123,                                               // 24^1
    -118,                                               // 24^2
    -114,                                               // 24^3
    -109,                                               // 24^4
    -105,                                               // 24^5
    -100,                                               // 24^6
];
const BASE24_LARGE_MANTISSA: [u128; 68] = [
    256006814417050404626793229969178591795,            // 24^-252
    273382276918988884237693747042282021609,            // 24^-245
    291937031065346039954998156631577529912,            // 24^-238
    311751116669878803408975082301396512655,            // 24^-231
    332910005936047335476283449703688561122,            // 24^-224
    177752486079622930343415992310050508587,            // 24^-217
    189816741726628588213403698819375225722,            // 24^-210
    202699811599676133995465766268541805984,            // 24^-203
    216457269515865090355509585365869594574,            // 24^-196
    231148461148045387015380597263260157877,            // 24^-189
    246836760024792608106756526472045881483,            // 24^-182
    263589840905381559535877208688654464260,            // 24^-175
    281479971709018296242657937208050445965,            // 24^-168
    300584325257628424747408646813479651038,            // 24^-161
    320985312176969416466104839150917967197,            // 24^-154
    171385468196052762160979317193129295473,            // 24^-147
    183017587375374702561553597022155160742,            // 24^-140
    195439191206027575440487166351295574484,            // 24^-133
    208703862874796048578293668364396201854,            // 24^-126
    222868822317958475703480592144987545632,            // 24^-119
    237995173051452727716558620615765508935,            // 24^-112
    254148165753675349373102394182948812519,            // 24^-105
    271397479737933588417468230506889186025,            // 24^-98
    289817523528740604428224130917268871991,            // 24^-91
    309487755838552588810803796052767101096,            // 24^-84
    330493028329548101430287061507520336961,            // 24^-77
    176461975819512133258798291874254633040,            // 24^-70
    188438643123668474334468683754392032451,            // 24^-63
    201228179937237770199942876645925476060,            // 24^-56
    214885756602899904017224155871405769601,            // 24^-49
    229470287934835004924643178169318563534,            // 24^-42
    245044687360099685434665347000146337133,            // 24^-35
    261676138308856451194147028266269550444,            // 24^-28
    279436384024154813848437280673013431065,            // 24^-21
    298402037041419227483658365640566588740,            // 24^-14
    318654909672648364505254319167929262488,            // 24^-7
    170141183460469231731687303715884105728,            // 24^0
    181688851673850688377539127552069599232,            // 24^7
    194020272759136452871913121072449323008,            // 24^14
    207188640880972374233825193254612631552,            // 24^21
    221250760550139932836609227367108313088,            // 24^28
    236267291661697281793781435669856387072,            // 24^35
    252303011164126931290527343657214803968,            // 24^42
    269427092488254686881046533485512097792,            // 24^49
    287713403941314941508226937857819803648,            // 24^56
    307240827353347547401607574753443315712,            // 24^63
    328093598350474163167634651360244400128,            // 24^70
    175180834861447020226468989874232056416,            // 24^77
    187070549727531559196917812917453861026,            // 24^84
    199767232545952890607255496509019333039,            // 24^91
    213325653114257310428744028742785781912,            // 24^98
    227804298516055047806476167412340090352,            // 24^105
    243265625417291205836310243227923719532,            // 24^112
    259776329486140560138677002900131432918,            // 24^119
    277407633098725295421526662764935275289,            // 24^126
    296235592571734482952577544661578831571,            // 24^133
    316341426247257477645159711999449660471,            // 24^140
    337811864845093800590802876046287308325,            // 24^147
    180369762796928745579122531717097251784,            // 24^154
    192611655877384358682393055110005707838,            // 24^161
    205684419630781050995309380627725821797,            // 24^168
    219644446158456132093135554410564634898,            // 24^175
    234551954955343535589691141355422293223,            // 24^182
    250471252679363433757155530343900661758,            // 24^189
    267471010551644448060009348077202513313,            // 24^196
    285624560584202347610957248166426707331,            // 24^203
    305010211912915299914630616083972269987,            // 24^210
    325711588600364141070945877624827809796,            // 24^217
];
const BASE24_LARGE_EXPONENT: [i32; 68] = [
    -1283,                                              // 24^-252
    -1251,                                              // 24^-245
    -1219,                                              // 24^-238
    -1187,                                              // 24^-231
    -1155,                                              // 24^-224
    -1122,                                              // 24^-217
    -1090,                                              // 24^-210
    -1058,                                              // 24^-203
    -1026,                                              // 24^-196
    -994,                                               // 24^-189
    -962,                                               // 24^-182
    -930,                                               // 24^-175
    -898,                                               // 24^-168
    -866,                                               // 24^-161
    -834,                                               // 24^-154
    -801,                                               // 24^-147
    -769,                                               // 24^-140
    -737,                                               // 24^-133
    -705,                                               // 24^-126
    -673,                                               // 24^-119
    -641,                                               // 24^-112
    -609,                                               // 24^-105
    -577,                                               // 24^-98
    -545,                                               // 24^-91
    -513,                                               // 24^-84
    -481,                                               // 24^-77
    -448,                                               // 24^-70
    -416,                                               // 24^-63
    -384,                                               // 24^-56
    -352,                                               // 24^-49
    -320,                                               // 24^-42
    -288,                                               // 24^-35
    -256,                                               // 24^-28
    -224,                                               // 24^-21
    -192,                                               // 24^-14
    -160,                                               // 24^-7
    -127,                                               // 24^0
    -95,                                                // 24^7
    -63,                                                // 24^14
    -31,                                                // 24^21
    1,                                                  // 24^28
    33,                                                 // 24^35
    65,                                                 // 24^42
    97,                                                 // 24^49
    129,                                                // 24^56
    161,                                                // 24^63
    193,                                                // 24^70
    226,                                                // 24^77
    258,                                                // 24^84
    290,                                                // 24^91
    322,                                                // 24^98
    354,                                                // 24^105
    386,                                                // 24^112
    418,                                                // 24^119
    450,                                                // 24^126
    482,                                                // 24^133
    514,                                                // 24^140
    546,                                                // 24^147
    579,                                                // 24^154
    611,                                                // 24^161
    643,                                                // 24^168
    675,                                                // 24^175
    707,                                                // 24^182
    739,                                                // 24^189
    771,                                                // 24^196
    803,                                                // 24^203
    835,                                                // 24^210
    867,                                                // 24^217
];
const BASE24_SMALL_INT_POWERS: [u128; 7] = [1, 24, 576, 13824, 331776, 7962624, 191102976];
const BASE24_STEP: i32 = 7;
const BASE24_BIAS: i32 = 252;

// BASE25

const BASE25_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 25^0
    265845599156983174580761412056068915200,            // 25^1
    207691874341393105141219853168803840000,            // 25^2
    324518553658426726783156020576256000000,            // 25^3
    253530120045645880299340641075200000000,            // 25^4
    198070406285660843983859875840000000000,            // 25^5
    309485009821345068724781056000000000000,            // 25^6
];
const BASE25_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 25^0
    -123,                                               // 25^1
    -118,                                               // 25^2
    -114,                                               // 25^3
    -109,                                               // 25^4
    -104,                                               // 25^5
    -100,                                               // 25^6
];
const BASE25_LARGE_MANTISSA: [u128; 68] = [
    285793394306920833441610418092098634655,            // 25^-252
    203068420253004570555511362849258201390,            // 25^-245
    288577581746103207017755725657449092679,            // 25^-238
    205046704412910121830119952091883627559,            // 25^-231
    291388892624283530821742192659774598780,            // 25^-224
    207044260935364498850036477975162511299,            // 25^-217
    294227591176883860910658765384315687611,            // 25^-210
    209061277570927374050781655074839937648,            // 25^-203
    297093944213496817569054052050375869453,            // 25^-196
    211097943899216614887176072592734406508,            // 25^-189
    299988221142963048588365030287739055137,            // 25^-182
    213154451346726893197828921904416471830,            // 25^-175
    302910693998692996157485768413290076965,            // 25^-168
    215230993204821882725842221200657943544,            // 25^-161
    305861637464235347360161968596028634045,            // 25^-154
    217327764647901735884376228537482684576,            // 25^-147
    308841328899094571460716776609676066664,            // 25^-140
    219444962751747547330237450047488370802,            // 25^-133
    311850048364799970571308236412006025948,            // 25^-126
    221582786512044528543660416923448526878,            // 25^-119
    314888078651228693933689466069052580904,            // 25^-112
    223741436863085634409521749481834675708,            // 25^-105
    317955705303185189918510999237120523316,            // 25^-98
    225921116696657399755928707376370229068,            // 25^-91
    321053216647239593947814323906257853121,            // 25^-84
    228122030881109760932058580285014566244,            // 25^-77
    324180903818827574883781864350871964922,            // 25^-70
    230344386280611654799899571593522271174,            // 25^-63
    327339060789614187001318969682759915221,            // 25^-56
    232588391774594204975783618524161450993,            // 25^-49
    330527984395124299475957654016385519914,            // 25^-42
    234854258277383322788948059678933702737,            // 25^-35
    333747974362642200374222141588992517906,            // 25^-28
    237142198758023568227473377297792835283,            // 25^-21
    336999333339382997433337688587745383420,            // 25^-14
    239452428260295134118491722992235809940,            // 25^-7
    170141183460469231731687303715884105728,            // 25^0
    241785163922925834941235200000000000000,            // 25^7
    171798691840000000000000000000000000000,            // 25^14
    244140625000000000000000000000000000000,            // 25^21
    173472347597680709441192448139190673828,            // 25^28
    246519032881566189191165176650870696772,            // 25^35
    175162308040602133865466197911239516410,            // 25^42
    248920611114445668285762562151204969623,            // 25^49
    176868732008334225927912486150152183216,            // 25^56
    251345585423243599518503524095297312920,            // 25^63
    178591779887855465971216179422709524914,            // 25^70
    253794183731564922327402455583054354682,            // 25^77
    180331613628627651967947866455016278082,            // 25^84
    256266636183436918326986907537468991453,            // 25^91
    182088396757817547443627082897044283139,            // 25^98
    258763175164940474024358370140027266101,            // 25^105
    183862294395666818064937594201088633455,            // 25^112
    261284035326052074402891767876281837538,            // 25^119
    185653473271011701515143789632334288014,            // 25^126
    263829453602698580304979415177988198613,            // 25^133
    187462101736953869352205554703508169192,            // 25^140
    266399669239026862544798113253119949479,            // 25^147
    189288349786683953755640255602884245064,            // 25^154
    268994923809890385876486015494726082500,            // 25^161
    191132389069459226417170338759437756337,            // 25^168
    271615461243554856334256923502490730495,            // 25^175
    192994392906736931318972184714148973580,            // 25^182
    274261527844625066050770363850331497104,            // 25^189
    194874536308464787773268059716493991903,            // 25^196
    276933372317195090450451374005771742621,            // 25^203
    196772995989530194869453349330805553038,            // 25^210
    279631245788224013707368483964622716141,            // 25^217
];
const BASE25_LARGE_EXPONENT: [i32; 68] = [
    -1298,                                              // 25^-252
    -1265,                                              // 25^-245
    -1233,                                              // 25^-238
    -1200,                                              // 25^-231
    -1168,                                              // 25^-224
    -1135,                                              // 25^-217
    -1103,                                              // 25^-210
    -1070,                                              // 25^-203
    -1038,                                              // 25^-196
    -1005,                                              // 25^-189
    -973,                                               // 25^-182
    -940,                                               // 25^-175
    -908,                                               // 25^-168
    -875,                                               // 25^-161
    -843,                                               // 25^-154
    -810,                                               // 25^-147
    -778,                                               // 25^-140
    -745,                                               // 25^-133
    -713,                                               // 25^-126
    -680,                                               // 25^-119
    -648,                                               // 25^-112
    -615,                                               // 25^-105
    -583,                                               // 25^-98
    -550,                                               // 25^-91
    -518,                                               // 25^-84
    -485,                                               // 25^-77
    -453,                                               // 25^-70
    -420,                                               // 25^-63
    -388,                                               // 25^-56
    -355,                                               // 25^-49
    -323,                                               // 25^-42
    -290,                                               // 25^-35
    -258,                                               // 25^-28
    -225,                                               // 25^-21
    -193,                                               // 25^-14
    -160,                                               // 25^-7
    -127,                                               // 25^0
    -95,                                                // 25^7
    -62,                                                // 25^14
    -30,                                                // 25^21
    3,                                                  // 25^28
    35,                                                 // 25^35
    68,                                                 // 25^42
    100,                                                // 25^49
    133,                                                // 25^56
    165,                                                // 25^63
    198,                                                // 25^70
    230,                                                // 25^77
    263,                                                // 25^84
    295,                                                // 25^91
    328,                                                // 25^98
    360,                                                // 25^105
    393,                                                // 25^112
    425,                                                // 25^119
    458,                                                // 25^126
    490,                                                // 25^133
    523,                                                // 25^140
    555,                                                // 25^147
    588,                                                // 25^154
    620,                                                // 25^161
    653,                                                // 25^168
    685,                                                // 25^175
    718,                                                // 25^182
    750,                                                // 25^189
    783,                                                // 25^196
    815,                                                // 25^203
    848,                                                // 25^210
    880,                                                // 25^217
];
const BASE25_SMALL_INT_POWERS: [u128; 7] = [1, 25, 625, 15625, 390625, 9765625, 244140625];
const BASE25_STEP: i32 = 7;
const BASE25_BIAS: i32 = 252;

// BASE26

const BASE26_SMALL_MANTISSA: [u128; 7] = [
    170141183460469231731687303715884105728,            // 26^0
    276479423123262501563991868538311671808,            // 26^1
    224639531287650782520743393187378233344,            // 26^2
    182519619171216260798104006964744814592,            // 26^3
    296594381153226423796919011317710323712,            // 26^4
    240982934686996469334996696695639638016,            // 26^5
    195798634433184631334684816065207205888,            // 26^6
];
const BASE26_SMALL_EXPONENT: [i32; 7] = [
    -127,                                               // 26^0
    -123,                                               // 26^1
    -118,                                               // 26^2
    -113,                                               // 26^3
    -109,                                               // 26^4
    -104,                                               // 26^5
    -99,                                                // 26^6
];
const BASE26_LARGE_MANTISSA: [u128; 67] = [
    223302816785416365261501121183664506010,            // 26^-245
    208793887424582006747864373745206480886,            // 26^-238
    195227664627991173689253365541181602096,            // 26^-231
    182542896759209079117924981191278218615,            // 26^-224
    170682311959929403775713752016508602388,            // 26^-217
    319184719133881733167774634228398630548,            // 26^-210
    298445936660656380797439373851559361096,            // 26^-203
    279054640682520272457995982566385389064,            // 26^-196
    260923279297292454826462160021044868443,            // 26^-189
    243969989220528359050303591757077787596,            // 26^-182
    228118226171942658526451569535286060353,            // 26^-175
    213296419277190995686832685182416132955,            // 26^-168
    199437647924631042854519895051784617004,            // 26^-161
    186479339618067826498814972661362929947,            // 26^-154
    174362987461285330626576271532417342855,            // 26^-147
    326067771997608226855765039505019953527,            // 26^-140
    304881768440366443726402955195105316329,            // 26^-133
    285072309225356538301642822964643005408,            // 26^-126
    266549954439052068809893515776468694564,            // 26^-119
    249231075457753005677502945704601914895,            // 26^-112
    233037477363483566918311865718592557833,            // 26^-105
    217896045893127676516686822732460831572,            // 26^-98
    203738417326773735777703283427712394918,            // 26^-91
    190500669824811165252338688092226229364,            // 26^-84
    178123034820162511238845936350714491622,            // 26^-77
    333099254325168111301464298292509503356,            // 26^-70
    311456385593267303107050428484389238716,            // 26^-63
    291219745668138391592456499213129019442,            // 26^-56
    272297965910924348580716754684381379181,            // 26^-49
    254605614290044560620651113294031241626,            // 26^-42
    238062809654687405271268739306918462570,            // 26^-35
    222594861070586991002412082254420837256,            // 26^-28
    208131930589681357218231284385128680344,            // 26^-21
    194608717931053648412097371491373800423,            // 26^-14
    181964165649487446029462142596867446143,            // 26^-7
    170141183460469231731687303715884105728,            // 26^0
    318172780953925025918862826105961709568,            // 26^7
    297499748388299952530789444812804194304,            // 26^14
    278169930267914565987548746187341299712,            // 26^21
    260096052263752396381159488684908609536,            // 26^28
    243196510629429753543644020787720315589,            // 26^35
    227395003759435617782759026890197718352,            // 26^42
    212620187686592486096128182198450396409,            // 26^49
    198805353963305641500281734389939339052,            // 26^56
    185888128472231349675004689621278542958,            // 26^63
    173810189806491030350550655153103473999,            // 26^70
    325034011895830307254166714675296871096,            // 26^77
    303915176108832810280521264920190203131,            // 26^84
    284168520489679119677023043643028417295,            // 26^91
    265704888683728554332237168601547339173,            // 26^98
    248440917202145588620033973783642939305,            // 26^105
    232298659034884347081172590621418853088,            // 26^112
    217205231719130933968079964556427917884,            // 26^119
    203092488274228969742684708734280089507,            // 26^126
    189896709517356115129460946793154219973,            // 26^133
    177558316370753675479515454481338457101,            // 26^140
    332043201723146894831623394706398020929,            // 26^147
    310468949199606660363664220576467010752,            // 26^154
    290296467197293861525395669447395056845,            // 26^161
    271434676751037481783657313824389119896,            // 26^168
    253798416681617203255579043607175584203,            // 26^175
    237308059092157195001470401326248026898,            // 26^182
    221889149847346996027560241559399617642,            // 26^189
    207472072412249451654170398205728730819,            // 26^196
    193991733532924439390933354984357040572,            // 26^203
    181387269339713144970498252974494732833,            // 26^210
    339203541092472346195720855153810480522,            // 26^217
];
const BASE26_LARGE_EXPONENT: [i32; 67] = [
    -1279,                                              // 26^-245
    -1246,                                              // 26^-238
    -1213,                                              // 26^-231
    -1180,                                              // 26^-224
    -1147,                                              // 26^-217
    -1115,                                              // 26^-210
    -1082,                                              // 26^-203
    -1049,                                              // 26^-196
    -1016,                                              // 26^-189
    -983,                                               // 26^-182
    -950,                                               // 26^-175
    -917,                                               // 26^-168
    -884,                                               // 26^-161
    -851,                                               // 26^-154
    -818,                                               // 26^-147
    -786,                                               // 26^-140
    -753,                                               // 26^-133
    -720,                                               // 26^-126
    -687,                                               // 26^-119
    -654,                                               // 26^-112
    -621,                                               // 26^-105
    -588,                                               // 26^-98
    -555,                                               // 26^-91
    -522,                                               // 26^-84
    -489,                                               // 26^-77
    -457,                                               // 26^-70
    -424,                                               // 26^-63
    -391,                                               // 26^-56
    -358,                                               // 26^-49
    -325,                                               // 26^-42
    -292,                                               // 26^-35
    -259,                                               // 26^-28
    -226,                                               // 26^-21
    -193,                                               // 26^-14
    -160,                                               // 26^-7
    -127,                                               // 26^0
    -95,                                                // 26^7
    -62,                                                // 26^14
    -29,                                                // 26^21
    4,                                                  // 26^28
    37,                                                 // 26^35
    70,                                                 // 26^42
    103,                                                // 26^49
    136,                                                // 26^56
    169,                                                // 26^63
    202,                                                // 26^70
    234,                                                // 26^77
    267,                                                // 26^84
    300,                                                // 26^91
    333,                                                // 26^98
    366,                                                // 26^105
    399,                                                // 26^112
    432,                                                // 26^119
    465,                                                // 26^126
    498,                                                // 26^133
    531,                                                // 26^140
    563,                                                // 26^147
    596,                                                // 26^154
    629,                                                // 26^161
    662,                                                // 26^168
    695,                                                // 26^175
    728,                                                // 26^182
    761,                                                // 26^189
    794,                                                // 26^196
    827,                                                // 26^203
    860,                                                // 26^210
    892,                                                // 26^217
];
const BASE26_SMALL_INT_POWERS: [u128; 7] = [1, 26, 676, 17576, 456976, 11881376, 308915776];
const BASE26_STEP: i32 = 7;
const BASE26_BIAS: i32 = 245;

// BASE27

const BASE27_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 27^0
    287113247089541828547222325020554428416,            // 27^1
    242251802231800917836718836736092798976,            // 27^2
    204399958133082024424731518496078299136,            // 27^3
    172462464674787958108367218731066064896,            // 27^4
    291030409138704679307869681608673984512,            // 27^5
];
const BASE27_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 27^0
    -123,                                               // 27^1
    -118,                                               // 27^2
    -113,                                               // 27^3
    -108,                                               // 27^4
    -104,                                               // 27^5
];
const BASE27_LARGE_MANTISSA: [u128; 76] = [
    301829093537629265639465570217176944359,            // 27^-240
    217807991453958805640698687941213190524,            // 27^-234
    314352208961548438173441007673156326734,            // 27^-228
    226845008347394462887832488874694288292,            // 27^-222
    327394917835133689224334116987590197085,            // 27^-216
    236256977848340413085493360278597848651,            // 27^-210
    170489389240119998671196096475026672256,            // 27^-204
    246059457021648542224892012444309926233,            // 27^-198
    177563121844831037921653709250257605479,            // 27^-192
    256268648406457241006138264113659504394,            // 27^-186
    184930348919702200346046943747485274024,            // 27^-180
    266901426797403574706768528236472000862,            // 27^-174
    192603247770383575639211190527648274245,            // 27^-168
    277975367137008028446553971650055283412,            // 27^-162
    200594500948068090486693848039128919647,            // 27^-156
    289508773565335211238455692680966173052,            // 27^-150
    208917317212507950117664039252872831665,            // 27^-144
    301520709674946766164053267333891939739,            // 27^-138
    217585453364802351586979201161384846208,            // 27^-132
    314031030021154964119856834958393443507,            // 27^-126
    226613236986043931067161987739751269180,            // 27^-120
    327060412939660347810097743318775450603,            // 27^-114
    236015590119408703302029793810763336632,            // 27^-108
    170315197362908885300398426895467760677,            // 27^-102
    245808053934833671173174941698733239342,            // 27^-96
    177381702616012906692133545122052956869,            // 27^-90
    256006814417050404626793229969178591795,            // 27^-84
    184741402471039290909022270993420155647,            // 27^-78
    266628729119434395515123988465075762881,            // 27^-72
    192406461791880080316008520325217417399,            // 27^-66
    277691355027891684120101092281051616669,            // 27^-60
    200389550171752283164939097875653100692,            // 27^-54
    289212977580839036146652597763405686112,            // 27^-48
    208703862874796048578293668364396201854,            // 27^-42
    301212640893244858516269504216828222245,            // 27^-36
    217363142646555453321168098187951653993,            // 27^-30
    313710179234688236904530296665341569850,            // 27^-24
    226381702429392491474935736226666160567,            // 27^-18
    326726249813466247246220462666861782844,            // 27^-12
    235774449020380624184618955567855082461,            // 27^-6
    170141183460469231731687303715884105728,            // 27^0
    245556907710782073166015043857318674432,            // 27^6
    177200468746272961345336076752392290304,            // 27^12
    255745247947835503562868389206950936576,            // 27^18
    184552649072141716781794491390137475072,            // 27^24
    266356310061270520809673995345359110719,            // 27^30
    192209876872921446586714266254161951235,            // 27^36
    277407633098725295421526662764935275289,            // 27^42
    200184808797092622572327630249651738267,            // 27^48
    288917483816076538023589582665008561757,            // 27^54
    208490626626972031635281014538153149532,            // 27^60
    300904886870600004067510516586852827477,            // 27^66
    217141059066909427380630585083218539864,            // 27^72
    313389656266867868879861721401276560157,            // 27^78
    226150404435492799169987273137391228527,            // 27^84
    326392428107359965184387801150473482685,            // 27^90
    235533554299270254021060647605641184828,            // 27^96
    339934694701922439619874702371784251126,            // 27^102
    245306018087052741642305313258629505287,            // 27^108
    177019420046226713314377865847118993119,            // 27^114
    255483948725482657093998355855298189652,            // 27^120
    184364088525767284952804747951506893851,            // 27^126
    266084169338241408156670471179837543899,            // 27^132
    192013492808081754945415747456910215687,            // 27^138
    277124201053027125645172361985060059244,            // 27^144
    199980276610139913759598726349951659975,            // 27^150
    288622291962264730584478255384696488209,            // 27^156
    208277608246209791806511248482402407710,            // 27^162
    300597447285417578884462942447710218615,            // 27^168
    216919202393792943992865658673403318648,            // 27^174
    313069460782756034010893203297842312622,            // 27^180
    225919342762644710883872352816958877073,            // 27^186
    326058947472506854027112756453181563499,            // 27^192
    235292905704349129354647627776916151748,            // 27^198
    339587377705461640362820917278613293254,            // 27^204
    245055384801472810432512717678228228007,            // 27^210
];
const BASE27_LARGE_EXPONENT: [i32; 76] = [
    -1269,                                              // 27^-240
    -1240,                                              // 27^-234
    -1212,                                              // 27^-228
    -1183,                                              // 27^-222
    -1155,                                              // 27^-216
    -1126,                                              // 27^-210
    -1097,                                              // 27^-204
    -1069,                                              // 27^-198
    -1040,                                              // 27^-192
    -1012,                                              // 27^-186
    -983,                                               // 27^-180
    -955,                                               // 27^-174
    -926,                                               // 27^-168
    -898,                                               // 27^-162
    -869,                                               // 27^-156
    -841,                                               // 27^-150
    -812,                                               // 27^-144
    -784,                                               // 27^-138
    -755,                                               // 27^-132
    -727,                                               // 27^-126
    -698,                                               // 27^-120
    -670,                                               // 27^-114
    -641,                                               // 27^-108
    -612,                                               // 27^-102
    -584,                                               // 27^-96
    -555,                                               // 27^-90
    -527,                                               // 27^-84
    -498,                                               // 27^-78
    -470,                                               // 27^-72
    -441,                                               // 27^-66
    -413,                                               // 27^-60
    -384,                                               // 27^-54
    -356,                                               // 27^-48
    -327,                                               // 27^-42
    -299,                                               // 27^-36
    -270,                                               // 27^-30
    -242,                                               // 27^-24
    -213,                                               // 27^-18
    -185,                                               // 27^-12
    -156,                                               // 27^-6
    -127,                                               // 27^0
    -99,                                                // 27^6
    -70,                                                // 27^12
    -42,                                                // 27^18
    -13,                                                // 27^24
    15,                                                 // 27^30
    44,                                                 // 27^36
    72,                                                 // 27^42
    101,                                                // 27^48
    129,                                                // 27^54
    158,                                                // 27^60
    186,                                                // 27^66
    215,                                                // 27^72
    243,                                                // 27^78
    272,                                                // 27^84
    300,                                                // 27^90
    329,                                                // 27^96
    357,                                                // 27^102
    386,                                                // 27^108
    415,                                                // 27^114
    443,                                                // 27^120
    472,                                                // 27^126
    500,                                                // 27^132
    529,                                                // 27^138
    557,                                                // 27^144
    586,                                                // 27^150
    614,                                                // 27^156
    643,                                                // 27^162
    671,                                                // 27^168
    700,                                                // 27^174
    728,                                                // 27^180
    757,                                                // 27^186
    785,                                                // 27^192
    814,                                                // 27^198
    842,                                                // 27^204
    871,                                                // 27^210
];
const BASE27_SMALL_INT_POWERS: [u128; 6] = [1, 27, 729, 19683, 531441, 14348907];
const BASE27_STEP: i32 = 6;
const BASE27_BIAS: i32 = 240;

// BASE28

const BASE28_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 28^0
    297747071055821155530452781502797185024,            // 28^1
    260528687173843511089146183814947536896,            // 28^2
    227962601277113072203002910838079094784,            // 28^3
    199467276117473938177627546983319207936,            // 28^4
    174533866602789695905424103610404306944,            // 28^5
];
const BASE28_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 28^0
    -123,                                               // 28^1
    -118,                                               // 28^2
    -113,                                               // 28^3
    -108,                                               // 28^4
    -103,                                               // 28^5
];
const BASE28_LARGE_MANTISSA: [u128; 76] = [
    200215143815698040798730279921859793515,            // 28^-240
    179711238516029806533278035755362570551,            // 28^-234
    322614250185735942212427179391214768413,            // 28^-228
    289575530396283324168013345552040300667,            // 28^-222
    259920284847963995399800125815215983072,            // 28^-216
    233302014099717072256401710525774728336,            // 28^-210
    209409703497448836012980688794302909957,            // 28^-204
    187964189199610581269006081054389519147,            // 28^-198
    337429792711562885676838629607664070711,            // 28^-192
    302873822652608199592547515371033250847,            // 28^-186
    271856707468083969679745656096547629768,            // 28^-180
    244016035285282981482333348801442963368,            // 28^-174
    219026508600450572879143037057044702127,            // 28^-168
    196596143419909740056291955312532441411,            // 28^-162
    176462857644721687377034700397980722042,            // 28^-156
    316782817673398770114452445482208770257,            // 28^-150
    284341291171704802743493772571841275116,            // 28^-144
    255222080727080523208383932871281083497,            // 28^-138
    229084950069124576377435007616984162844,            // 28^-132
    205624506307086466111975488366169508167,            // 28^-126
    184566631641558957302915986868221103411,            // 28^-120
    331330561004604641231243330979299081348,            // 28^-114
    297399209378286220063892720385616742108,            // 28^-108
    266942745850723232302069966588191406954,            // 28^-102
    239605309345945263344621501916001364416,            // 28^-96
    215067482294014848993159294730496555497,            // 28^-90
    193042558474796699288911421705232156812,            // 28^-84
    173273193069468359944466704194632400641,            // 28^-78
    311056791556242112413125050076207081650,            // 28^-72
    279201663740542055384000770694089408486,            // 28^-66
    250608799266136415663698628779517553855,            // 28^-60
    224944111822980370837543334787608800457,            // 28^-54
    201907728667158642949418150287074186439,            // 28^-48
    181230486831379296755646484093658378299,            // 28^-42
    325341576312636457580643512071759255196,            // 28^-36
    292023552792399342330208805478900181690,            // 28^-30
    262117606830390855604604612394616145902,            // 28^-24
    235274309738072614830216430996812400431,            // 28^-18
    211180017596241035935669959185363632952,            // 28^-12
    189553206559602063269009666658011215615,            // 28^-6
    170141183460469231731687303715884105728,            // 28^0
    305434266554881967834492181318207537152,            // 28^6
    274154937941858738966065755004164112384,            // 28^12
    246078905440687093968343124469641904128,            // 28^18
    220878121537715117784741212850409701376,            // 28^24
    198258133856129805696540977101424033792,            // 28^30
    177954644699400447924746318168681611264,            // 28^36
    319460845859371388212562249545703474176,            // 28^42
    286745064197610355009611023687762970225,            // 28^48
    257379684889104161503034418684704785797,            // 28^54
    231021595363755916570056887236303965333,            // 28^60
    207362820991138609531788808643065835705,            // 28^66
    186126926626483659918254253754028720893,            // 28^72
    334131573344103639308512797544978713750,            // 28^78
    299913371828921883094842698046640019943,            // 28^84
    269199434526831288331795872363961423540,            // 28^90
    241630891972710985114650364591580944199,            // 28^96
    216885626294688985349681859923056842835,            // 28^102
    194674507506895938395726937393857685110,            // 28^108
    174738015241079713869620349559402181972,            // 28^114
    313686412889065357315780098042512623701,            // 28^120
    281561987228284074576142980610683957411,            // 28^126
    252727403529513497084111370284014563793,            // 28^132
    226845751173734538409794758625366435361,            // 28^138
    203614622343740041422835873088956724196,            // 28^144
    182762578614186646525232068123189313087,            // 28^150
    328091958791815868790390435525895683218,            // 28^156
    294492270354449044397893099587906656150,            // 28^162
    264333504599995236391965677440014878764,            // 28^168
    237263278829077450304247817879793628477,            // 28^174
    212965297630021155936008083539885258459,            // 28^180
    191155657202715751455043144381591497592,            // 28^186
    171579528154314464133715598246382584383,            // 28^192
    308016356015425756696586706818979868623,            // 28^198
    276472597266073797985814891590470600613,            // 28^204
    248159214750338106225838746496027188809,            // 28^210
];
const BASE28_LARGE_EXPONENT: [i32; 76] = [
    -1281,                                              // 28^-240
    -1252,                                              // 28^-234
    -1224,                                              // 28^-228
    -1195,                                              // 28^-222
    -1166,                                              // 28^-216
    -1137,                                              // 28^-210
    -1108,                                              // 28^-204
    -1079,                                              // 28^-198
    -1051,                                              // 28^-192
    -1022,                                              // 28^-186
    -993,                                               // 28^-180
    -964,                                               // 28^-174
    -935,                                               // 28^-168
    -906,                                               // 28^-162
    -877,                                               // 28^-156
    -849,                                               // 28^-150
    -820,                                               // 28^-144
    -791,                                               // 28^-138
    -762,                                               // 28^-132
    -733,                                               // 28^-126
    -704,                                               // 28^-120
    -676,                                               // 28^-114
    -647,                                               // 28^-108
    -618,                                               // 28^-102
    -589,                                               // 28^-96
    -560,                                               // 28^-90
    -531,                                               // 28^-84
    -502,                                               // 28^-78
    -474,                                               // 28^-72
    -445,                                               // 28^-66
    -416,                                               // 28^-60
    -387,                                               // 28^-54
    -358,                                               // 28^-48
    -329,                                               // 28^-42
    -301,                                               // 28^-36
    -272,                                               // 28^-30
    -243,                                               // 28^-24
    -214,                                               // 28^-18
    -185,                                               // 28^-12
    -156,                                               // 28^-6
    -127,                                               // 28^0
    -99,                                                // 28^6
    -70,                                                // 28^12
    -41,                                                // 28^18
    -12,                                                // 28^24
    17,                                                 // 28^30
    46,                                                 // 28^36
    74,                                                 // 28^42
    103,                                                // 28^48
    132,                                                // 28^54
    161,                                                // 28^60
    190,                                                // 28^66
    219,                                                // 28^72
    247,                                                // 28^78
    276,                                                // 28^84
    305,                                                // 28^90
    334,                                                // 28^96
    363,                                                // 28^102
    392,                                                // 28^108
    421,                                                // 28^114
    449,                                                // 28^120
    478,                                                // 28^126
    507,                                                // 28^132
    536,                                                // 28^138
    565,                                                // 28^144
    594,                                                // 28^150
    622,                                                // 28^156
    651,                                                // 28^162
    680,                                                // 28^168
    709,                                                // 28^174
    738,                                                // 28^180
    767,                                                // 28^186
    796,                                                // 28^192
    824,                                                // 28^198
    853,                                                // 28^204
    882,                                                // 28^210
];
const BASE28_SMALL_INT_POWERS: [u128; 6] = [1, 28, 784, 21952, 614656, 17210368];
const BASE28_STEP: i32 = 6;
const BASE28_BIAS: i32 = 240;

// BASE29

const BASE29_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 29^0
    308380895022100482513683237985039941632,            // 29^1
    279470186113778562278025434423942447104,            // 29^2
    253269856165611822064460549946697842688,            // 29^3
    229525807150085713745917373389194919936,            // 29^4
    208007762729765178082237619633957896192,            // 29^5
];
const BASE29_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 29^0
    -123,                                               // 29^1
    -118,                                               // 29^2
    -113,                                               // 29^3
    -108,                                               // 29^4
    -103,                                               // 29^5
];
const BASE29_LARGE_MANTISSA: [u128; 76] = [
    180411782640948456163108621039484300353,            // 29^-240
    199886291656678749778798199538954577713,            // 29^-234
    221462972137332420374365077239613292335,            // 29^-228
    245368742505570013761609603669658845854,            // 29^-222
    271855015841791437163383377269432435982,            // 29^-216
    301200343954417841961395964144914808415,            // 29^-210
    333713346863741229059418106803065448261,            // 29^-204
    184867979254122350694373980738345610830,            // 29^-198
    204823511403978168182734748793342650457,            // 29^-192
    226933138952023660037574950865203175761,            // 29^-186
    251429385238135556008900174982942596722,            // 29^-180
    278569873281449391084528542386060643447,            // 29^-174
    308640035159552273337407148073177516540,            // 29^-168
    170978056925499484035797671737173759761,            // 29^-162
    189434244555519246827735852425496159000,            // 29^-156
    209882681179121373738148351480252166035,            // 29^-150
    232538419662693834656863635067192575556,            // 29^-144
    257639726705579546348788818074016917383,            // 29^-138
    285450588652016995453946272087193218261,            // 29^-132
    316263487792383250937533050996202736450,            // 29^-126
    175201239175822924661859658491633525197,            // 29^-120
    194113297257345719179697640426306620561,            // 29^-114
    215066813165088318898530949380694999931,            // 29^-108
    238282151601732438626757341123945812037,            // 29^-102
    264003464487880391063907148518553600721,            // 29^-96
    292501258667905961095191488633763572010,            // 29^-90
    324075240786231793266420435905585066661,            // 29^-84
    179528734626566103050967344518404222838,            // 29^-78
    198907923224385351394870557131224024014,            // 29^-72
    220378993946205681004466278772642662300,            // 29^-66
    244167754534112212293116837315364018708,            // 29^-60
    270524387495766270724774864712621178351,            // 29^-54
    299726081232954872012689080099923610541,            // 29^-48
    332079945186715557451765374774991672241,            // 29^-42
    183963119828570739334286542177638201257,            // 29^-36
    203820977132676910337518249049241331980,            // 29^-30
    225822386345870676314424417535913775896,            // 29^-24
    250198732693485636863600410972168425001,            // 29^-18
    277206378226597610088077970727216566225,            // 29^-12
    307129357939822377568583266911848900855,            // 29^-6
    170141183460469231731687303715884105728,            // 29^0
    188507034973849692637027842793274343424,            // 29^6
    208855384169162106382075974754544648192,            // 29^12
    231400231309666912826787170353923751936,            // 29^18
    256378676868573301455838684793773426688,            // 29^24
    284053415075970164879962990673826094314,            // 29^30
    314715496631115750914098978917481871688,            // 29^36
    174343694817539847357322592625772163155,            // 29^42
    193163185467532912371523116182520814756,            // 29^48
    214014141773314894388674508400179209532,            // 29^54
    237115849834993471687725998227739696222,            // 29^60
    262711266541086358656385267533109039869,            // 29^66
    291069574706415032602452081981147446763,            // 29^72
    322489014023779685975823216414806216349,            // 29^78
    178650008801035774670984867305268298071,            // 29^84
    197934343538640677425270715935647505996,            // 29^90
    219300321421789265371466533424126863110,            // 29^96
    242972644948356100526463281459541324974,            // 29^102
    269200272076455967884065094333741996545,            // 29^108
    298259034474605889482552078350958780837,            // 29^114
    330454538398307869859357548505383554011,            // 29^120
    183062689350548673356118481539553806163,            // 29^126
    202823349890643162760568057855283073783,            // 29^132
    224717070457148091679782830215989826885,            // 29^138
    248974103731497779640892318688061490897,            // 29^144
    275849556968674066644388728598920812956,            // 29^150
    305626074918518590345961283073160340970,            // 29^156
    338616812354378462414859289927404969896,            // 29^162
    187584363735341677353233362159059264795,            // 29^168
    207833115392789061354400140358968381064,            // 29^174
    230267613961761833964469003400183663813,            // 29^180
    255123799397575000468261630468177681638,            // 29^186
    282663080140582027146793786682106989268,            // 29^192
    313175082306023963280707662359655742783,            // 29^198
    173490347817277081445057491729890113081,            // 29^204
    192217724118601260264099717636971309833,            // 29^210
];
const BASE29_LARGE_EXPONENT: [i32; 76] = [
    -1293,                                              // 29^-240
    -1264,                                              // 29^-234
    -1235,                                              // 29^-228
    -1206,                                              // 29^-222
    -1177,                                              // 29^-216
    -1148,                                              // 29^-210
    -1119,                                              // 29^-204
    -1089,                                              // 29^-198
    -1060,                                              // 29^-192
    -1031,                                              // 29^-186
    -1002,                                              // 29^-180
    -973,                                               // 29^-174
    -944,                                               // 29^-168
    -914,                                               // 29^-162
    -885,                                               // 29^-156
    -856,                                               // 29^-150
    -827,                                               // 29^-144
    -798,                                               // 29^-138
    -769,                                               // 29^-132
    -740,                                               // 29^-126
    -710,                                               // 29^-120
    -681,                                               // 29^-114
    -652,                                               // 29^-108
    -623,                                               // 29^-102
    -594,                                               // 29^-96
    -565,                                               // 29^-90
    -536,                                               // 29^-84
    -506,                                               // 29^-78
    -477,                                               // 29^-72
    -448,                                               // 29^-66
    -419,                                               // 29^-60
    -390,                                               // 29^-54
    -361,                                               // 29^-48
    -332,                                               // 29^-42
    -302,                                               // 29^-36
    -273,                                               // 29^-30
    -244,                                               // 29^-24
    -215,                                               // 29^-18
    -186,                                               // 29^-12
    -157,                                               // 29^-6
    -127,                                               // 29^0
    -98,                                                // 29^6
    -69,                                                // 29^12
    -40,                                                // 29^18
    -11,                                                // 29^24
    18,                                                 // 29^30
    47,                                                 // 29^36
    77,                                                 // 29^42
    106,                                                // 29^48
    135,                                                // 29^54
    164,                                                // 29^60
    193,                                                // 29^66
    222,                                                // 29^72
    251,                                                // 29^78
    281,                                                // 29^84
    310,                                                // 29^90
    339,                                                // 29^96
    368,                                                // 29^102
    397,                                                // 29^108
    426,                                                // 29^114
    455,                                                // 29^120
    485,                                                // 29^126
    514,                                                // 29^132
    543,                                                // 29^138
    572,                                                // 29^144
    601,                                                // 29^150
    630,                                                // 29^156
    659,                                                // 29^162
    689,                                                // 29^168
    718,                                                // 29^174
    747,                                                // 29^180
    776,                                                // 29^186
    805,                                                // 29^192
    834,                                                // 29^198
    864,                                                // 29^204
    893,                                                // 29^210
];
const BASE29_SMALL_INT_POWERS: [u128; 6] = [1, 29, 841, 24389, 707281, 20511149];
const BASE29_STEP: i32 = 6;
const BASE29_BIAS: i32 = 240;

// BASE30

const BASE30_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 30^0
    319014718988379809496913694467282698240,            // 30^1
    299076299051606071403356588563077529600,            // 30^2
    280384030360880691940646801777885184000,            // 30^3
    262860028463325648694356376666767360000,            // 30^4
    246431276684367795650959103125094400000,            // 30^5
];
const BASE30_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 30^0
    -123,                                               // 30^1
    -118,                                               // 30^2
    -113,                                               // 30^3
    -108,                                               // 30^4
    -103,                                               // 30^5
];
const BASE30_LARGE_MANTISSA: [u128; 74] = [
    293697914606894485088361958296166003399,            // 30^-234
    199401546035358756435491021348075025228,            // 30^-228
    270761041082025567891957153683056483774,            // 30^-222
    183828919146951883093674448351327494412,            // 30^-216
    249615464467793559236989678532104461317,            // 30^-210
    338944929832632662139396137837142035924,            // 30^-204
    230121290169646228383872457298707723793,            // 30^-198
    312474408249691315911417866816293319027,            // 30^-192
    212149548916169413644283660602828329029,            // 30^-186
    288071150257973936473359891361485890787,            // 30^-180
    195581343526079319127909243854250027224,            // 30^-174
    265573709142416387133758214804806317848,            // 30^-168
    180307062310000458937613050074040708794,            // 30^-162
    244833246663194772909432488501026624276,            // 30^-156
    332451307806128616353452734867573546427,            // 30^-150
    225712548373888955751123878842649157480,            // 30^-144
    306487917461553679299669913344627685478,            // 30^-138
    208085115840167395965624010961720331305,            // 30^-132
    282552185370553343852869987471204477403,            // 30^-126
    191834329753307055373436045708607941890,            // 30^-120
    260485757869036576463347072380734960806,            // 30^-114
    176852678401887104140389725347567150767,            // 30^-108
    240142648210703765821352805529490956868,            // 30^-102
    326082092794781635112624979076905003855,            // 30^-96
    221388270750079128888532155889145795079,            // 30^-90
    300616117896153936087600778123712697276,            // 30^-84
    204098550552778150334824777443135675335,            // 30^-78
    277138954685954882938577351637270128950,            // 30^-72
    188159102542382767109407940268116001044,            // 30^-66
    255495283293346012426090218975127795266,            // 30^-60
    173464474753336275982316368848893943009,            // 30^-54
    235541913835671069456466721167496749113,            // 30^-48
    319834901366763207379289417958082873567,            // 30^-42
    217146839104937741700096042539405091508,            // 30^-36
    294856812260112936979848919457246198708,            // 30^-30
    200188361236473853754168248068850933442,            // 30^-24
    271829432512427567293473804075629134624,            // 30^-18
    184554286581985369843377175900278276895,            // 30^-12
    250600417923680198594596164731872945757,            // 30^-6
    170141183460469231731687303715884105728,            // 30^0
    231029321891594808422774159179776000000,            // 30^6
    313707395752840890251476992000000000000,            // 30^12
    212986666247081951232000000000000000000,            // 30^18
    289207845356544000000000000000000000000,            // 30^24
    196353084654473304748535156250000000000,            // 30^30
    266621631967110632288608940143603831529,            // 30^36
    181018532909474941844489348460628831000,            // 30^42
    245799330046413899594233809090775284541,            // 30^48
    333763121820680299409099755486799743829,            // 30^54
    226603183715861233202026897808422346833,            // 30^60
    307697282971558792524556831461079215336,            // 30^66
    208906195393080226844550976655564036201,            // 30^72
    283667103278554017412807162817392837984,            // 30^78
    192591285603182277357146536646298485706,            // 30^84
    261513604232519641878753574965804062932,            // 30^90
    177550518406095745907734479894927853604,            // 30^96
    241090223040513319460187919146199534233,            // 30^102
    327368774631124380764508606973252185538,            // 30^108
    222261842997828194477899721342605392688,            // 30^114
    301802313971178147521594347169174498450,            // 30^120
    204903899584886496461222208185427085841,            // 30^126
    278232512618195742220116760148063945729,            // 30^132
    188901556375124208702207652989717774500,            // 30^138
    256503437827277086943219190518379691262,            // 30^144
    174148945301850322989380722761059920238,            // 30^150
    236471334705220322048773145103478211637,            // 30^156
    321096932515698698858834100478208840422,            // 30^162
    218003675159015088778073023304915283497,            // 30^168
    296020282788056879712669614682501632104,            // 30^174
    200978281118435287206001718625001320433,            // 30^180
    272902039690575235287053981567967613406,            // 30^186
    185282516232160242762660936045505469826,            // 30^192
    251589257890814574386886921482483833102,            // 30^198
    170812540689859376035668482781137073721,            // 30^204
];
const BASE30_LARGE_EXPONENT: [i32; 74] = [
    -1276,                                              // 30^-234
    -1246,                                              // 30^-228
    -1217,                                              // 30^-222
    -1187,                                              // 30^-216
    -1158,                                              // 30^-210
    -1129,                                              // 30^-204
    -1099,                                              // 30^-198
    -1070,                                              // 30^-192
    -1040,                                              // 30^-186
    -1011,                                              // 30^-180
    -981,                                               // 30^-174
    -952,                                               // 30^-168
    -922,                                               // 30^-162
    -893,                                               // 30^-156
    -864,                                               // 30^-150
    -834,                                               // 30^-144
    -805,                                               // 30^-138
    -775,                                               // 30^-132
    -746,                                               // 30^-126
    -716,                                               // 30^-120
    -687,                                               // 30^-114
    -657,                                               // 30^-108
    -628,                                               // 30^-102
    -599,                                               // 30^-96
    -569,                                               // 30^-90
    -540,                                               // 30^-84
    -510,                                               // 30^-78
    -481,                                               // 30^-72
    -451,                                               // 30^-66
    -422,                                               // 30^-60
    -392,                                               // 30^-54
    -363,                                               // 30^-48
    -334,                                               // 30^-42
    -304,                                               // 30^-36
    -275,                                               // 30^-30
    -245,                                               // 30^-24
    -216,                                               // 30^-18
    -186,                                               // 30^-12
    -157,                                               // 30^-6
    -127,                                               // 30^0
    -98,                                                // 30^6
    -69,                                                // 30^12
    -39,                                                // 30^18
    -10,                                                // 30^24
    20,                                                 // 30^30
    49,                                                 // 30^36
    79,                                                 // 30^42
    108,                                                // 30^48
    137,                                                // 30^54
    167,                                                // 30^60
    196,                                                // 30^66
    226,                                                // 30^72
    255,                                                // 30^78
    285,                                                // 30^84
    314,                                                // 30^90
    344,                                                // 30^96
    373,                                                // 30^102
    402,                                                // 30^108
    432,                                                // 30^114
    461,                                                // 30^120
    491,                                                // 30^126
    520,                                                // 30^132
    550,                                                // 30^138
    579,                                                // 30^144
    609,                                                // 30^150
    638,                                                // 30^156
    667,                                                // 30^162
    697,                                                // 30^168
    726,                                                // 30^174
    756,                                                // 30^180
    785,                                                // 30^186
    815,                                                // 30^192
    844,                                                // 30^198
    874,                                                // 30^204
];
const BASE30_SMALL_INT_POWERS: [u128; 6] = [1, 30, 900, 27000, 810000, 24300000];
const BASE30_STEP: i32 = 6;
const BASE30_BIAS: i32 = 234;

// BASE31

const BASE31_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 31^0
    329648542954659136480144150949525454848,            // 31^1
    319347025987326038465139646232352784384,            // 31^2
    309367431425222099763104032287591759872,            // 31^3
    299699699193183909145507031278604517376,            // 31^4
    290334083593396911984709936551148126208,            // 31^5
];
const BASE31_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 31^0
    -123,                                               // 31^1
    -118,                                               // 31^2
    -113,                                               // 31^3
    -108,                                               // 31^4
    -103,                                               // 31^5
];
const BASE31_LARGE_MANTISSA: [u128; 74] = [
    279877549230888994466722529922465951128,            // 31^-234
    231333407733284590292354000091386191754,            // 31^-228
    191209140142019782313183919746595976478,            // 31^-222
    316088675925298444140273004034743674741,            // 31^-216
    261263794643914746305034655542823358647,            // 31^-210
    215948167684024592485185159022754788335,            // 31^-204
    178492435929157837038046463019461592108,            // 31^-198
    295066635902569136770934561228011804883,            // 31^-192
    243887980937787207068127465341221873783,            // 31^-186
    201586150409601608540557644579125879351,            // 31^-180
    333242957530806000015262416296943861408,            // 31^-174
    275442704070282164877682412643443854572,            // 31^-168
    227667776650720372831068985136241988697,            // 31^-162
    188179304658062925738530341045324738580,            // 31^-156
    311080041475689584245114254221154550706,            // 31^-150
    257123896754632869764136622554477725012,            // 31^-144
    212526325921342359594316086083656739425,            // 31^-138
    175664105047096554749814041961690625344,            // 31^-132
    290391109602281582919653498207198252857,            // 31^-126
    240023414326551697067636257878891518439,            // 31^-120
    198391884324143423870507706735824676989,            // 31^-114
    327962502126029666246881442269148110223,            // 31^-108
    271078131968920728852328597367236335349,            // 31^-102
    224060229920801636254083864004341099802,            // 31^-96
    185197478924335716894196793983277050828,            // 31^-90
    306150772156692797734166574102932652557,            // 31^-84
    253049598289706899575201136976026000492,            // 31^-78
    209158705508044155113481306826330986049,            // 31^-72
    172880590941369685133685946960988620150,            // 31^-66
    285789670113326703641484035227164931190,            // 31^-60
    236220084333189890382264913587684321799,            // 31^-54
    195248233500715771863466694882341429463,            // 31^-48
    322765719034956327944217102842371688313,            // 31^-42
    266782719403631527651786656524762917448,            // 31^-36
    220509847158485190531810693342620280466,            // 31^-30
    182262902194543738831736462131829646508,            // 31^-24
    301299610375241462753468658335350302125,            // 31^-18
    249039859782804352657429715423616325073,            // 31^-12
    205844447270932349829274233837837318735,            // 31^-6
    170141183460469231731687303715884105728,            // 31^0
    281261143481103258485187751033924747264,            // 31^6
    232477020622927971041480650211423420416,            // 31^12
    192154395906963839472434931204572053504,            // 31^18
    317651282414350176352409831205456445952,            // 31^24
    262555370495753687560337482407159416999,            // 31^30
    217015722562838525119352550468059930336,            // 31^36
    179374825776921533841673613411506136043,            // 31^42
    296525318465664137401900073705767585881,            // 31^48
    245093658238625334625948305994309525378,            // 31^54
    202582705650977753904076181059390370572,            // 31^60
    334890367411416504232175148268475073388,            // 31^66
    276804374353098300220638069068074602530,            // 31^72
    228793268236589371413606590094648720070,            // 31^78
    189109582219266748069833984779993889354,            // 31^84
    312617887429839722873414506299686714932,            // 31^90
    258395006731549634868442333657079980382,            // 31^96
    213576964685944914612871149421565862652,            // 31^102
    176532512843220607081571440296564925357,            // 31^108
    291826678373921783044839874892818001863,            // 31^114
    241209986871908117885122799335937441435,            // 31^120
    199372648487594099824157002892036768477,            // 31^126
    329583807705824908665978895106558273752,            // 31^132
    272418225684124763666228455375305707749,            // 31^138
    225167887346958248891898253497228074938,            // 31^144
    186113015621359245893427982047443315204,            // 31^150
    307664249923018427479642383336104189758,            // 31^156
    254300566687046383525258086411660098107,            // 31^162
    210192696205470376085866529648531617961,            // 31^168
    173735238240724141814430167095519087450,            // 31^174
    287202491346848457988114640516635531033,            // 31^180
    237387854850570349221700427782261535085,            // 31^186
    196213456806330839063706448987155946803,            // 31^192
    324361333953874416226256763524449662322,            // 31^198
    268101578446229760174200876130811505774,            // 31^204
];
const BASE31_LARGE_EXPONENT: [i32; 74] = [
    -1287,                                              // 31^-234
    -1257,                                              // 31^-228
    -1227,                                              // 31^-222
    -1198,                                              // 31^-216
    -1168,                                              // 31^-210
    -1138,                                              // 31^-204
    -1108,                                              // 31^-198
    -1079,                                              // 31^-192
    -1049,                                              // 31^-186
    -1019,                                              // 31^-180
    -990,                                               // 31^-174
    -960,                                               // 31^-168
    -930,                                               // 31^-162
    -900,                                               // 31^-156
    -871,                                               // 31^-150
    -841,                                               // 31^-144
    -811,                                               // 31^-138
    -781,                                               // 31^-132
    -752,                                               // 31^-126
    -722,                                               // 31^-120
    -692,                                               // 31^-114
    -663,                                               // 31^-108
    -633,                                               // 31^-102
    -603,                                               // 31^-96
    -573,                                               // 31^-90
    -544,                                               // 31^-84
    -514,                                               // 31^-78
    -484,                                               // 31^-72
    -454,                                               // 31^-66
    -425,                                               // 31^-60
    -395,                                               // 31^-54
    -365,                                               // 31^-48
    -336,                                               // 31^-42
    -306,                                               // 31^-36
    -276,                                               // 31^-30
    -246,                                               // 31^-24
    -217,                                               // 31^-18
    -187,                                               // 31^-12
    -157,                                               // 31^-6
    -127,                                               // 31^0
    -98,                                                // 31^6
    -68,                                                // 31^12
    -38,                                                // 31^18
    -9,                                                 // 31^24
    21,                                                 // 31^30
    51,                                                 // 31^36
    81,                                                 // 31^42
    110,                                                // 31^48
    140,                                                // 31^54
    170,                                                // 31^60
    199,                                                // 31^66
    229,                                                // 31^72
    259,                                                // 31^78
    289,                                                // 31^84
    318,                                                // 31^90
    348,                                                // 31^96
    378,                                                // 31^102
    408,                                                // 31^108
    437,                                                // 31^114
    467,                                                // 31^120
    497,                                                // 31^126
    526,                                                // 31^132
    556,                                                // 31^138
    586,                                                // 31^144
    616,                                                // 31^150
    645,                                                // 31^156
    675,                                                // 31^162
    705,                                                // 31^168
    735,                                                // 31^174
    764,                                                // 31^180
    794,                                                // 31^186
    824,                                                // 31^192
    853,                                                // 31^198
    883,                                                // 31^204
];
const BASE31_SMALL_INT_POWERS: [u128; 6] = [1, 31, 961, 29791, 923521, 28629151];
const BASE31_STEP: i32 = 6;
const BASE31_BIAS: i32 = 234;

// BASE33

const BASE33_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 33^0
    175458095443608895223302531957005484032,            // 33^1
    180941160926221673199030736080661905408,            // 33^2
    186595572205166100486500446583182589952,            // 33^3
    192426683836577541126703585538907045888,            // 33^4
    198440017706470589286913072586997891072,            // 33^5
];
const BASE33_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 33^0
    -122,                                               // 33^1
    -117,                                               // 33^2
    -112,                                               // 33^3
    -107,                                               // 33^4
    -102,                                               // 33^5
];
const BASE33_LARGE_MANTISSA: [u128; 72] = [
    312720366502509540124188550868577470271,            // 33^-228
    188065849520234218420139418711557486770,            // 33^-222
    226200577540468899968415353242546081790,            // 33^-216
    272068009211510778916881923005514361559,            // 33^-210
    327236130168906522048072439540872358087,            // 33^-204
    196795435814493025178219114364462772750,            // 33^-198
    236700290627603575579431376108202583063,            // 33^-192
    284696783496570703658001718555889571262,            // 33^-186
    171212841180688182412693440011648609110,            // 33^-180
    205930229524469868245369490973155168397,            // 33^-174
    247687376364759110421449812279701734754,            // 33^-168
    297911756579516503503766398038992340964,            // 33^-162
    179160149400574418614614090847719005026,            // 33^-156
    215489039450973655880961058312920063458,            // 33^-150
    259184457474862991304220337185522020620,            // 33^-144
    311740138466868713435648102307209093287,            // 33^-138
    187476353478425047886734580102095167043,            // 33^-132
    225491547456297727220699982909970598320,            // 33^-126
    271215206775862669602998925081523543654,            // 33^-120
    326210402191372960931717472229356291097,            // 33^-114
    196178576715647050727159488732861667611,            // 33^-108
    235958348989735718000162987087766864735,            // 33^-102
    283804395923732956473182174007605293342,            // 33^-96
    170676171219393275768862099029406252664,            // 33^-90
    205284737238107330416766226733647440005,            // 33^-84
    246910995494199118965634147066154669324,            // 33^-78
    296977946418022259559609845940558313794,            // 33^-72
    178598568447993152099773363289076713958,            // 33^-66
    214813584890064968045989415466080479007,            // 33^-60
    258372038781252962127704032685119793391,            // 33^-54
    310762982974959535779710715380461212334,            // 33^-48
    186888705223357570139042022172334279947,            // 33^-42
    224784739840635370899218738060963824868,            // 33^-36
    270365077465939099085308357449561310080,            // 33^-30
    325187889378018709806160016110270251585,            // 33^-24
    195563651173574149793648095659417101115,            // 33^-18
    235218732982278125085778976978720276897,            // 33^-12
    282914805556997697053159518336509943216,            // 33^-6
    170141183460469231731687303715884105728,            // 33^0
    204641268259797795202129106105341575168,            // 33^6
    246137048204490191229033862368809975808,            // 33^12
    296047063302535604636168085099824283648,            // 33^18
    178038747781766620403928207609098977344,            // 33^24
    214140247554538205717741625292909706188,            // 33^30
    257562166629747183314707096744136073023,            // 33^36
    309788890395834268833431845588643035745,            // 33^42
    186302898963108514077454910111746951397,            // 33^48
    224080147727111315915564635257325506176,            // 33^54
    269517612902775795573733407057825442248,            // 33^60
    324168581650849479410768181980047458277,            // 33^66
    194950653127503229444535496044715069916,            // 33^72
    234481435315497306894208542281087212757,            // 33^78
    282028003628468309677795696968404883305,            // 33^84
    339215745262040382698593524434347927396,            // 33^90
    203999816247443047277506123370388146319,            // 33^96
    245365526867526093228395853689144006062,            // 33^102
    295119098058174229856749108333084701428,            // 33^108
    177480681884243207276442522127875068624,            // 33^114
    213469020807909470054649976638820369879,            // 33^120
    256754833038160188521579955085771893751,            // 33^126
    308817851128733566186038756059234075639,            // 33^132
    185718928923909504089695656878457160369,            // 33^138
    223377764171191270264346565880758640128,            // 33^144
    268672804733672512805858396506074869210,            // 33^150
    323152468963460648455199258995519212913,            // 33^156
    194339576535660800930047023345366203227,            // 33^162
    233746448722509584063702393615930970798,            // 33^168
    281143981397731320107155010945727058593,            // 33^174
    338152466949356579395454697640846384307,            // 33^180
    203360374878824300807265157901557614739,            // 33^186
    244596423879111037469845073222731160599,            // 33^192
    294194041538814672349638971928150296782,            // 33^198
];
const BASE33_LARGE_EXPONENT: [i32; 72] = [
    -1278,                                              // 33^-228
    -1247,                                              // 33^-222
    -1217,                                              // 33^-216
    -1187,                                              // 33^-210
    -1157,                                              // 33^-204
    -1126,                                              // 33^-198
    -1096,                                              // 33^-192
    -1066,                                              // 33^-186
    -1035,                                              // 33^-180
    -1005,                                              // 33^-174
    -975,                                               // 33^-168
    -945,                                               // 33^-162
    -914,                                               // 33^-156
    -884,                                               // 33^-150
    -854,                                               // 33^-144
    -824,                                               // 33^-138
    -793,                                               // 33^-132
    -763,                                               // 33^-126
    -733,                                               // 33^-120
    -703,                                               // 33^-114
    -672,                                               // 33^-108
    -642,                                               // 33^-102
    -612,                                               // 33^-96
    -581,                                               // 33^-90
    -551,                                               // 33^-84
    -521,                                               // 33^-78
    -491,                                               // 33^-72
    -460,                                               // 33^-66
    -430,                                               // 33^-60
    -400,                                               // 33^-54
    -370,                                               // 33^-48
    -339,                                               // 33^-42
    -309,                                               // 33^-36
    -279,                                               // 33^-30
    -249,                                               // 33^-24
    -218,                                               // 33^-18
    -188,                                               // 33^-12
    -158,                                               // 33^-6
    -127,                                               // 33^0
    -97,                                                // 33^6
    -67,                                                // 33^12
    -37,                                                // 33^18
    -6,                                                 // 33^24
    24,                                                 // 33^30
    54,                                                 // 33^36
    84,                                                 // 33^42
    115,                                                // 33^48
    145,                                                // 33^54
    175,                                                // 33^60
    205,                                                // 33^66
    236,                                                // 33^72
    266,                                                // 33^78
    296,                                                // 33^84
    326,                                                // 33^90
    357,                                                // 33^96
    387,                                                // 33^102
    417,                                                // 33^108
    448,                                                // 33^114
    478,                                                // 33^120
    508,                                                // 33^126
    538,                                                // 33^132
    569,                                                // 33^138
    599,                                                // 33^144
    629,                                                // 33^150
    659,                                                // 33^156
    690,                                                // 33^162
    720,                                                // 33^168
    750,                                                // 33^174
    780,                                                // 33^180
    811,                                                // 33^186
    841,                                                // 33^192
    871,                                                // 33^198
];
const BASE33_SMALL_INT_POWERS: [u128; 6] = [1, 33, 1089, 35937, 1185921, 39135393];
const BASE33_STEP: i32 = 6;
const BASE33_BIAS: i32 = 228;

// BASE34

const BASE34_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 34^0
    180775007426748558714917760198126862336,            // 34^1
    192073445390920343634600120210509791232,            // 34^2
    204078035727852865111762627723666653184,            // 34^3
    216832912960843669181247791956395819008,            // 34^4
    230384970020896398505075778953670557696,            // 34^5
];
const BASE34_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 34^0
    -122,                                               // 34^1
    -117,                                               // 34^2
    -112,                                               // 34^3
    -107,                                               // 34^4
    -102,                                               // 34^5
];
const BASE34_LARGE_MANTISSA: [u128; 72] = [
    177178627816250193201809052313908765038,            // 34^-228
    254908880844119689445018585029338745822,            // 34^-222
    183370134237042583472660416437030762483,            // 34^-216
    263816670637481076390123392064426194701,            // 34^-210
    189778002824260993825223275854263893366,            // 34^-204
    273035743108558333663040444930692235847,            // 34^-198
    196409794352921881357331916371145369426,            // 34^-192
    282576976029244796208883094031259474966,            // 34^-186
    203273333809293576669182005378991774735,            // 34^-180
    292451627295127901322196175360309190338,            // 34^-174
    210376719623757394611528557965027181599,            // 34^-168
    302671348208910116654395983418902348443,            // 34^-162
    217728333226311038599715596529740433091,            // 34^-156
    313248197228018957952397977805898800839,            // 34^-150
    225336848935989032116058614991615739830,            // 34^-144
    324194654192627182361160566056046446660,            // 34^-138
    233211244195868906543198111154344051285,            // 34^-132
    335523635050871088900625460663893770435,            // 34^-126
    241360810165739638162969982622013142483,            // 34^-120
    173624253549320755964330470575918052959,            // 34^-114
    249795162684930840147804514904540284970,            // 34^-108
    179691552375964624115693619162436412029,            // 34^-102
    258524253618237975606418771469177669493,            // 34^-96
    185970872935167515117533503151908736546,            // 34^-90
    267558382598330880564928534158522761469,            // 34^-84
    192469624325502244069180532494789364726,            // 34^-78
    276908209178500704519431941303594138118,            // 34^-72
    199195474556460799929094324231727524305,            // 34^-66
    286584765410084542875511011996966308779,            // 34^-60
    206156359596095351859662099550265135432,            // 34^-54
    296599468859408121472112670217716316867,            // 34^-48
    213360492734829033350750838314127106094,            // 34^-42
    306964136079605489713373754120772284157,            // 34^-36
    220816374276485055706958449270700673524,            // 34^-30
    317690996553211397504541477525504659745,            // 34^-24
    228532801567968794073232940648013889325,            // 34^-18
    328792707121977505492535302517672775182,            // 34^-12
    236518879379437072732268269343858967136,            // 34^-6
    170141183460469231731687303715884105728,            // 34^0
    244784030647202423411643015138274967552,            // 34^6
    176086766417174433233313223161344425984,            // 34^12
    253338007592048088741599978367647481856,            // 34^18
    182240117745863932172015090234506084352,            // 34^24
    262190903226072497809728032921351003168,            // 34^30
    188608497911442742195268251474898499613,            // 34^36
    271353163261640374618023568003458637143,            // 34^42
    195199421095732140407812372336079928061,            // 34^48
    280835598436492094076350884217966821974,            // 34^54
    202020664063606263361049018383687293681,            // 34^60
    290649397269553933781951224423558743337,            // 34^66
    209080275338955809947349984471742343690,            // 34^72
    300806139262500062617960370620227418603,            // 34^78
    216386584701305758176605078134277823932,            // 34^84
    311317808562643058781928971959145202656,            // 34^90
    223948213014292349032159641795245781843,            // 34^96
    322196808103274080809106622508050735664,            // 34^102
    231774082397596158130210248504430880773,            // 34^108
    333455974238137167751959613429477643393,            // 34^114
    239873426754333326759263874720553852658,            // 34^120
    172554295943652591814138834633669982456,            // 34^126
    248255802666326436218178945574152405546,            // 34^132
    178584205106164167247426311765412935775,            // 34^138
    256931100670110578075784008065054807112,            // 34^144
    184824829419575343010846487395966556532,            // 34^150
    265909556926979410208640982981191461086,            // 34^156
    191283532323968216046404126970062690800,            // 34^162
    275201765300840924300371814765015192837,            // 34^168
    197967934574808286676526111499839216564,            // 34^174
    284818689858133833493377548255268489047,            // 34^180
    204885923234829474007514465274285458561,            // 34^186
    294771677804553486829405243638525556429,            // 34^192
    212045660980140516237478265085420658486,            // 34^198
];
const BASE34_LARGE_EXPONENT: [i32; 72] = [
    -1287,                                              // 34^-228
    -1257,                                              // 34^-222
    -1226,                                              // 34^-216
    -1196,                                              // 34^-210
    -1165,                                              // 34^-204
    -1135,                                              // 34^-198
    -1104,                                              // 34^-192
    -1074,                                              // 34^-186
    -1043,                                              // 34^-180
    -1013,                                              // 34^-174
    -982,                                               // 34^-168
    -952,                                               // 34^-162
    -921,                                               // 34^-156
    -891,                                               // 34^-150
    -860,                                               // 34^-144
    -830,                                               // 34^-138
    -799,                                               // 34^-132
    -769,                                               // 34^-126
    -738,                                               // 34^-120
    -707,                                               // 34^-114
    -677,                                               // 34^-108
    -646,                                               // 34^-102
    -616,                                               // 34^-96
    -585,                                               // 34^-90
    -555,                                               // 34^-84
    -524,                                               // 34^-78
    -494,                                               // 34^-72
    -463,                                               // 34^-66
    -433,                                               // 34^-60
    -402,                                               // 34^-54
    -372,                                               // 34^-48
    -341,                                               // 34^-42
    -311,                                               // 34^-36
    -280,                                               // 34^-30
    -250,                                               // 34^-24
    -219,                                               // 34^-18
    -189,                                               // 34^-12
    -158,                                               // 34^-6
    -127,                                               // 34^0
    -97,                                                // 34^6
    -66,                                                // 34^12
    -36,                                                // 34^18
    -5,                                                 // 34^24
    25,                                                 // 34^30
    56,                                                 // 34^36
    86,                                                 // 34^42
    117,                                                // 34^48
    147,                                                // 34^54
    178,                                                // 34^60
    208,                                                // 34^66
    239,                                                // 34^72
    269,                                                // 34^78
    300,                                                // 34^84
    330,                                                // 34^90
    361,                                                // 34^96
    391,                                                // 34^102
    422,                                                // 34^108
    452,                                                // 34^114
    483,                                                // 34^120
    514,                                                // 34^126
    544,                                                // 34^132
    575,                                                // 34^138
    605,                                                // 34^144
    636,                                                // 34^150
    666,                                                // 34^156
    697,                                                // 34^162
    727,                                                // 34^168
    758,                                                // 34^174
    788,                                                // 34^180
    819,                                                // 34^186
    849,                                                // 34^192
    880,                                                // 34^198
];
const BASE34_SMALL_INT_POWERS: [u128; 6] = [1, 34, 1156, 39304, 1336336, 45435424];
const BASE34_STEP: i32 = 6;
const BASE34_BIAS: i32 = 228;

// BASE35

const BASE35_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 35^0
    186091919409888222206532988439248240640,            // 35^1
    203538036854565243038395456105427763200,            // 35^2
    222619727809680734573245030115311616000,            // 35^3
    243490327291838303439486751688622080000,            // 35^4
    266317545475448144386938634659430400000,            // 35^5
];
const BASE35_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 35^0
    -122,                                               // 35^1
    -117,                                               // 35^2
    -112,                                               // 35^3
    -107,                                               // 35^4
    -102,                                               // 35^5
];
const BASE35_LARGE_MANTISSA: [u128; 71] = [
    209347927024496598038547144935237941328,            // 35^-222
    179203738418473223738234323205585013488,            // 35^-216
    306800075067366652359189609572260396844,            // 35^-210
    262623667597658781475957209306702595701,            // 35^-204
    224808259148244964176155115402974869951,            // 35^-198
    192437923982884034851744022831878031683,            // 35^-192
    329457242604437106875016361913696434196,            // 35^-186
    282018410035885035080273518999343445370,            // 35^-180
    241410335891937127541744115564170038883,            // 35^-174
    206649453374488156391879990784284316065,            // 35^-168
    176893820317164915550880607303262990177,            // 35^-162
    302845453064861578599747045789577351117,            // 35^-156
    259238475028744868618971960605253012775,            // 35^-150
    221910503377561727015952290044860359194,            // 35^-144
    189957418565367403947310019950147953923,            // 35^-138
    325210571999150993462493010696415973233,            // 35^-132
    278383221194439940371876875109029372360,            // 35^-126
    238298581027663491439386725539968731964,            // 35^-120
    203985762777473298078638682440591083627,            // 35^-114
    174613676827044081042213382537397413381,            // 35^-108
    298941805740831610317914779329181594758,            // 35^-102
    255896917250379178747915691007066092001,            // 35^-96
    219050099386294168656703063834650082992,            // 35^-90
    187508886612326915552639247886565481996,            // 35^-84
    321018640549353575448221161417409055741,            // 35^-78
    274794889570264049674062490664049677426,            // 35^-72
    235226936369523138128743288428339150410,            // 35^-66
    201356406883409564599485887013038683702,            // 35^-60
    172362924156377643998457025163202983813,            // 35^-54
    295088476036816041433113722500648405442,            // 35^-48
    252598431814515573631538400793788490787,            // 35^-42
    216226565713775509518577464877951003673,            // 35^-36
    185091915989032531210622402180350455682,            // 35^-30
    316880742672855387564028424008808371736,            // 35^-24
    271252811178556028546302550648486766486,            // 35^-18
    232194884901007304160476988045002650981,            // 35^-12
    198760943121441293094365080225590768425,            // 35^-6
    170141183460469231731687303715884105728,            // 35^0
    291284815363771407923214131658752000000,            // 35^6
    249342463523006462279286784000000000000,            // 35^12
    213439427105318371328000000000000000000,            // 35^18
    182706099873129286194801330566406250000,            // 35^24
    312796181882350172840701732490664710439,            // 35^30
    267756389819817671992755772413469185401,            // 35^36
    229201916269897841937750664282591869365,            // 35^42
    196198934625406481835921498002300001382,            // 35^48
    335896161555784742544555459149306885413,            // 35^54
    287530183492901554386658282699185698307,            // 35^60
    246128464334152340394473578644658105769,            // 35^66
    210688214432220328999897334535237907298,            // 35^72
    180351036686161310983217274120661055754,            // 35^78
    308764270668182755993104788402294107778,            // 35^84
    264305036979502132888830858069861274896,            // 35^90
    226247526702365186299998179464172492810,            // 35^96
    193669950160303864708173829536684422103,            // 35^102
    331566493935091266452974274160244175815,            // 35^108
    283823948447894890028153679891537227671,            // 35^114
    242955893270455896832007072848205740580,            // 35^120
    207972464612800582882401068589523181243,            // 35^126
    178026330025978501184229413909552478799,            // 35^132
    304784330382628027085206214062708598478,            // 35^138
    260898171728955674614898641290082042559,            // 35^144
    223331218918173589647399860035052277272,            // 35^150
    191173564049707817673529979461919468076,            // 35^156
    327292635293038256953486831534470016773,            // 35^162
    280165486398550695538293601153410149006,            // 35^168
    239824216327565160313568077123618791086,            // 35^174
    205291720534454460326603808102056989101,            // 35^180
    175731588600014458664842906141703880790,            // 35^186
    300855691125661557416025651977765954086,            // 35^192
    257535220627636274500677259708261311842,            // 35^198
];
const BASE35_LARGE_EXPONENT: [i32; 71] = [
    -1266,                                              // 35^-222
    -1235,                                              // 35^-216
    -1205,                                              // 35^-210
    -1174,                                              // 35^-204
    -1143,                                              // 35^-198
    -1112,                                              // 35^-192
    -1082,                                              // 35^-186
    -1051,                                              // 35^-180
    -1020,                                              // 35^-174
    -989,                                               // 35^-168
    -958,                                               // 35^-162
    -928,                                               // 35^-156
    -897,                                               // 35^-150
    -866,                                               // 35^-144
    -835,                                               // 35^-138
    -805,                                               // 35^-132
    -774,                                               // 35^-126
    -743,                                               // 35^-120
    -712,                                               // 35^-114
    -681,                                               // 35^-108
    -651,                                               // 35^-102
    -620,                                               // 35^-96
    -589,                                               // 35^-90
    -558,                                               // 35^-84
    -528,                                               // 35^-78
    -497,                                               // 35^-72
    -466,                                               // 35^-66
    -435,                                               // 35^-60
    -404,                                               // 35^-54
    -374,                                               // 35^-48
    -343,                                               // 35^-42
    -312,                                               // 35^-36
    -281,                                               // 35^-30
    -251,                                               // 35^-24
    -220,                                               // 35^-18
    -189,                                               // 35^-12
    -158,                                               // 35^-6
    -127,                                               // 35^0
    -97,                                                // 35^6
    -66,                                                // 35^12
    -35,                                                // 35^18
    -4,                                                 // 35^24
    26,                                                 // 35^30
    57,                                                 // 35^36
    88,                                                 // 35^42
    119,                                                // 35^48
    149,                                                // 35^54
    180,                                                // 35^60
    211,                                                // 35^66
    242,                                                // 35^72
    273,                                                // 35^78
    303,                                                // 35^84
    334,                                                // 35^90
    365,                                                // 35^96
    396,                                                // 35^102
    426,                                                // 35^108
    457,                                                // 35^114
    488,                                                // 35^120
    519,                                                // 35^126
    550,                                                // 35^132
    580,                                                // 35^138
    611,                                                // 35^144
    642,                                                // 35^150
    673,                                                // 35^156
    703,                                                // 35^162
    734,                                                // 35^168
    765,                                                // 35^174
    796,                                                // 35^180
    827,                                                // 35^186
    857,                                                // 35^192
    888,                                                // 35^198
];
const BASE35_SMALL_INT_POWERS: [u128; 6] = [1, 35, 1225, 42875, 1500625, 52521875];
const BASE35_STEP: i32 = 6;
const BASE35_BIAS: i32 = 222;

// BASE36

const BASE36_SMALL_MANTISSA: [u128; 6] = [
    170141183460469231731687303715884105728,            // 36^0
    191408831393027885698148216680369618944,            // 36^1
    215334935317156371410416743765415821312,            // 36^2
    242251802231800917836718836736092798976,            // 36^3
    272533277510776032566308691328104398848,            // 36^4
    306599937199623036637097277744117448704,            // 36^5
];
const BASE36_SMALL_EXPONENT: [i32; 6] = [
    -127,                                               // 36^0
    -122,                                               // 36^1
    -117,                                               // 36^2
    -112,                                               // 36^3
    -107,                                               // 36^4
    -102,                                               // 36^5
];
const BASE36_LARGE_MANTISSA: [u128; 71] = [
    206105367118290399407064648402758144682,            // 36^-222
    208917317212507950117664039252872831665,            // 36^-216
    211767631486382365261996259087726574961,            // 36^-210
    214656833352574406771088703014069554755,            // 36^-204
    217585453364802351586979201161384846208,            // 36^-198
    220554029315269330081435801781477974040,            // 36^-192
    223563106333419891448609016293621894840,            // 36^-186
    226613236986043931067161987739751269180,            // 36^-180
    229704981378746362247969882824709232796,            // 36^-174
    232838907258801165579649662968151663564,            // 36^-168
    236015590119408703302029793810763336632,            // 36^-162
    239235613305375443823879271798297650114,            // 36^-156
    242499568120235502703106353919523432682,            // 36^-150
    245808053934833671173174941698733239342,            // 36^-144
    249161678297389871677290466673500998400,            // 36^-138
    252561057045065251911260457800735557729,            // 36^-132
    256006814417050404626793229969178591795,            // 36^-126
    259499583169196479959998361450291137700,            // 36^-120
    263040004690210240376322725691803307553,            // 36^-114
    266628729119434395515123988465075762881,            // 36^-108
    270266415466234845327287688358055741312,            // 36^-102
    273953731731016754981191818978678705632,            // 36^-96
    277691355027891684120101092281051616669,            // 36^-90
    281479971709018296242657937208050445965,            // 36^-84
    285320277490639481303204301467482637509,            // 36^-78
    289212977580839036146652597763405686112,            // 36^-72
    293158786809041363160730749526943361727,            // 36^-66
    297158429757277967604640789526650060843,            // 36^-60
    301212640893244858516269504216828222245,            // 36^-54
    305322164705175286969651759320250334279,            // 36^-48
    309487755838552588810803796052767101096,            // 36^-42
    313710179234688236904530296665341569850,            // 36^-36
    317990210271190550439415903835536554761,            // 36^-30
    322328634904349856025836233807108654402,            // 36^-24
    326726249813466247246220462666861782844,            // 36^-18
    331183862547146446042592332649497399781,            // 36^-12
    335702291671596630919115661345637412333,            // 36^-6
    170141183460469231731687303715884105728,            // 36^0
    172462464674787958108367218731066064896,            // 36^6
    174815415743320440759790006808579407872,            // 36^12
    177200468746272961345336076752392290304,            // 36^18
    179618061658836457920697688990341398528,            // 36^24
    182068638431613361423174859113151594496,            // 36^30
    184552649072141716781794491390137475072,            // 36^36
    187070549727531559196917812917453861026,            // 36^42
    189622802768228720381105803326920695033,            // 36^48
    192209876872921446586714266254161951235,            // 36^54
    194832247114605420104007752175098574688,            // 36^60
    197490395047822988635051696441052554380,            // 36^66
    200184808797092622572327630249651738267,            // 36^72
    202915983146544838776512848181734408257,            // 36^78
    205684419630781050995309380627725821797,            // 36^84
    208490626626972031635281014538153149532,            // 36^90
    211335119448212897232599978727666183358,            // 36^96
    214218420438151760708217936124820030498,            // 36^102
    217141059066909427380630585083218539864,            // 36^108
    220103572028307748788051030668660629356,            // 36^114
    223106503338424488684979682521025988628,            // 36^120
    226150404435492799169987273137391228527,            // 36^126
    229235834281163651816744244429413474808,            // 36^132
    232363359463149818964276081092475750857,            // 36^138
    235533554299270254021060647605641184828,            // 36^144
    238747000942913976797497733353022683918,            // 36^150
    242004289489942830549695955106475311593,            // 36^156
    245306018087052741642305313258629505287,            // 36^162
    248652793041613380567795520750960012282,            // 36^168
    252045228933006394543323172270604972624,            // 36^174
    255483948725482657093998355855298189652,            // 36^180
    258969583882559258973487053363982248701,            // 36^186
    262502774482977247520692697766891651596,            // 36^192
    266084169338241408156670471179837543899,            // 36^198
];
const BASE36_LARGE_EXPONENT: [i32; 71] = [
    -1275,                                              // 36^-222
    -1244,                                              // 36^-216
    -1213,                                              // 36^-210
    -1182,                                              // 36^-204
    -1151,                                              // 36^-198
    -1120,                                              // 36^-192
    -1089,                                              // 36^-186
    -1058,                                              // 36^-180
    -1027,                                              // 36^-174
    -996,                                               // 36^-168
    -965,                                               // 36^-162
    -934,                                               // 36^-156
    -903,                                               // 36^-150
    -872,                                               // 36^-144
    -841,                                               // 36^-138
    -810,                                               // 36^-132
    -779,                                               // 36^-126
    -748,                                               // 36^-120
    -717,                                               // 36^-114
    -686,                                               // 36^-108
    -655,                                               // 36^-102
    -624,                                               // 36^-96
    -593,                                               // 36^-90
    -562,                                               // 36^-84
    -531,                                               // 36^-78
    -500,                                               // 36^-72
    -469,                                               // 36^-66
    -438,                                               // 36^-60
    -407,                                               // 36^-54
    -376,                                               // 36^-48
    -345,                                               // 36^-42
    -314,                                               // 36^-36
    -283,                                               // 36^-30
    -252,                                               // 36^-24
    -221,                                               // 36^-18
    -190,                                               // 36^-12
    -159,                                               // 36^-6
    -127,                                               // 36^0
    -96,                                                // 36^6
    -65,                                                // 36^12
    -34,                                                // 36^18
    -3,                                                 // 36^24
    28,                                                 // 36^30
    59,                                                 // 36^36
    90,                                                 // 36^42
    121,                                                // 36^48
    152,                                                // 36^54
    183,                                                // 36^60
    214,                                                // 36^66
    245,                                                // 36^72
    276,                                                // 36^78
    307,                                                // 36^84
    338,                                                // 36^90
    369,                                                // 36^96
    400,                                                // 36^102
    431,                                                // 36^108
    462,                                                // 36^114
    493,                                                // 36^120
    524,                                                // 36^126
    555,                                                // 36^132
    586,                                                // 36^138
    617,                                                // 36^144
    648,                                                // 36^150
    679,                                                // 36^156
    710,                                                // 36^162
    741,                                                // 36^168
    772,                                                // 36^174
    803,                                                // 36^180
    834,                                                // 36^186
    865,                                                // 36^192
    896,                                                // 36^198
];
const BASE36_SMALL_INT_POWERS: [u128; 6] = [1, 36, 1296, 46656, 1679616, 60466176];
const BASE36_STEP: i32 = 6;
const BASE36_BIAS: i32 = 222;

}}  // cfg_if

// HIGH LEVEL
// ----------

pub(crate) const BASE10_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE10_SMALL_MANTISSA, exp: &BASE10_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE10_LARGE_MANTISSA, exp: &BASE10_LARGE_EXPONENT },
    small_int: &BASE10_SMALL_INT_POWERS,
    step: BASE10_STEP,
    bias: BASE10_BIAS,
};

cfg_if! {
if #[cfg(feature = "radix")] {
pub(crate) const BASE3_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE3_SMALL_MANTISSA, exp: &BASE3_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE3_LARGE_MANTISSA, exp: &BASE3_LARGE_EXPONENT },
    small_int: &BASE3_SMALL_INT_POWERS,
    step: BASE3_STEP,
    bias: BASE3_BIAS,
};

pub(crate) const BASE5_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE5_SMALL_MANTISSA, exp: &BASE5_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE5_LARGE_MANTISSA, exp: &BASE5_LARGE_EXPONENT },
    small_int: &BASE5_SMALL_INT_POWERS,
    step: BASE5_STEP,
    bias: BASE5_BIAS,
};

pub(crate) const BASE6_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE6_SMALL_MANTISSA, exp: &BASE6_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE6_LARGE_MANTISSA, exp: &BASE6_LARGE_EXPONENT },
    small_int: &BASE6_SMALL_INT_POWERS,
    step: BASE6_STEP,
    bias: BASE6_BIAS,
};

pub(crate) const BASE7_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE7_SMALL_MANTISSA, exp: &BASE7_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE7_LARGE_MANTISSA, exp: &BASE7_LARGE_EXPONENT },
    small_int: &BASE7_SMALL_INT_POWERS,
    step: BASE7_STEP,
    bias: BASE7_BIAS,
};

pub(crate) const BASE9_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE9_SMALL_MANTISSA, exp: &BASE9_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE9_LARGE_MANTISSA, exp: &BASE9_LARGE_EXPONENT },
    small_int: &BASE9_SMALL_INT_POWERS,
    step: BASE9_STEP,
    bias: BASE9_BIAS,
};

pub(crate) const BASE11_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE11_SMALL_MANTISSA, exp: &BASE11_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE11_LARGE_MANTISSA, exp: &BASE11_LARGE_EXPONENT },
    small_int: &BASE11_SMALL_INT_POWERS,
    step: BASE11_STEP,
    bias: BASE11_BIAS,
};

pub(crate) const BASE12_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE12_SMALL_MANTISSA, exp: &BASE12_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE12_LARGE_MANTISSA, exp: &BASE12_LARGE_EXPONENT },
    small_int: &BASE12_SMALL_INT_POWERS,
    step: BASE12_STEP,
    bias: BASE12_BIAS,
};

pub(crate) const BASE13_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE13_SMALL_MANTISSA, exp: &BASE13_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE13_LARGE_MANTISSA, exp: &BASE13_LARGE_EXPONENT },
    small_int: &BASE13_SMALL_INT_POWERS,
    step: BASE13_STEP,
    bias: BASE13_BIAS,
};

pub(crate) const BASE14_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE14_SMALL_MANTISSA, exp: &BASE14_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE14_LARGE_MANTISSA, exp: &BASE14_LARGE_EXPONENT },
    small_int: &BASE14_SMALL_INT_POWERS,
    step: BASE14_STEP,
    bias: BASE14_BIAS,
};

pub(crate) const BASE15_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE15_SMALL_MANTISSA, exp: &BASE15_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE15_LARGE_MANTISSA, exp: &BASE15_LARGE_EXPONENT },
    small_int: &BASE15_SMALL_INT_POWERS,
    step: BASE15_STEP,
    bias: BASE15_BIAS,
};

pub(crate) const BASE17_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE17_SMALL_MANTISSA, exp: &BASE17_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE17_LARGE_MANTISSA, exp: &BASE17_LARGE_EXPONENT },
    small_int: &BASE17_SMALL_INT_POWERS,
    step: BASE17_STEP,
    bias: BASE17_BIAS,
};

pub(crate) const BASE18_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE18_SMALL_MANTISSA, exp: &BASE18_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE18_LARGE_MANTISSA, exp: &BASE18_LARGE_EXPONENT },
    small_int: &BASE18_SMALL_INT_POWERS,
    step: BASE18_STEP,
    bias: BASE18_BIAS,
};

pub(crate) const BASE19_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE19_SMALL_MANTISSA, exp: &BASE19_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE19_LARGE_MANTISSA, exp: &BASE19_LARGE_EXPONENT },
    small_int: &BASE19_SMALL_INT_POWERS,
    step: BASE19_STEP,
    bias: BASE19_BIAS,
};

pub(crate) const BASE20_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE20_SMALL_MANTISSA, exp: &BASE20_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE20_LARGE_MANTISSA, exp: &BASE20_LARGE_EXPONENT },
    small_int: &BASE20_SMALL_INT_POWERS,
    step: BASE20_STEP,
    bias: BASE20_BIAS,
};

pub(crate) const BASE21_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE21_SMALL_MANTISSA, exp: &BASE21_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE21_LARGE_MANTISSA, exp: &BASE21_LARGE_EXPONENT },
    small_int: &BASE21_SMALL_INT_POWERS,
    step: BASE21_STEP,
    bias: BASE21_BIAS,
};

pub(crate) const BASE22_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE22_SMALL_MANTISSA, exp: &BASE22_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE22_LARGE_MANTISSA, exp: &BASE22_LARGE_EXPONENT },
    small_int: &BASE22_SMALL_INT_POWERS,
    step: BASE22_STEP,
    bias: BASE22_BIAS,
};

pub(crate) const BASE23_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE23_SMALL_MANTISSA, exp: &BASE23_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE23_LARGE_MANTISSA, exp: &BASE23_LARGE_EXPONENT },
    small_int: &BASE23_SMALL_INT_POWERS,
    step: BASE23_STEP,
    bias: BASE23_BIAS,
};

pub(crate) const BASE24_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE24_SMALL_MANTISSA, exp: &BASE24_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE24_LARGE_MANTISSA, exp: &BASE24_LARGE_EXPONENT },
    small_int: &BASE24_SMALL_INT_POWERS,
    step: BASE24_STEP,
    bias: BASE24_BIAS,
};

pub(crate) const BASE25_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE25_SMALL_MANTISSA, exp: &BASE25_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE25_LARGE_MANTISSA, exp: &BASE25_LARGE_EXPONENT },
    small_int: &BASE25_SMALL_INT_POWERS,
    step: BASE25_STEP,
    bias: BASE25_BIAS,
};

pub(crate) const BASE26_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE26_SMALL_MANTISSA, exp: &BASE26_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE26_LARGE_MANTISSA, exp: &BASE26_LARGE_EXPONENT },
    small_int: &BASE26_SMALL_INT_POWERS,
    step: BASE26_STEP,
    bias: BASE26_BIAS,
};

pub(crate) const BASE27_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE27_SMALL_MANTISSA, exp: &BASE27_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE27_LARGE_MANTISSA, exp: &BASE27_LARGE_EXPONENT },
    small_int: &BASE27_SMALL_INT_POWERS,
    step: BASE27_STEP,
    bias: BASE27_BIAS,
};

pub(crate) const BASE28_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE28_SMALL_MANTISSA, exp: &BASE28_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE28_LARGE_MANTISSA, exp: &BASE28_LARGE_EXPONENT },
    small_int: &BASE28_SMALL_INT_POWERS,
    step: BASE28_STEP,
    bias: BASE28_BIAS,
};

pub(crate) const BASE29_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE29_SMALL_MANTISSA, exp: &BASE29_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE29_LARGE_MANTISSA, exp: &BASE29_LARGE_EXPONENT },
    small_int: &BASE29_SMALL_INT_POWERS,
    step: BASE29_STEP,
    bias: BASE29_BIAS,
};

pub(crate) const BASE30_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE30_SMALL_MANTISSA, exp: &BASE30_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE30_LARGE_MANTISSA, exp: &BASE30_LARGE_EXPONENT },
    small_int: &BASE30_SMALL_INT_POWERS,
    step: BASE30_STEP,
    bias: BASE30_BIAS,
};

pub(crate) const BASE31_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE31_SMALL_MANTISSA, exp: &BASE31_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE31_LARGE_MANTISSA, exp: &BASE31_LARGE_EXPONENT },
    small_int: &BASE31_SMALL_INT_POWERS,
    step: BASE31_STEP,
    bias: BASE31_BIAS,
};

pub(crate) const BASE33_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE33_SMALL_MANTISSA, exp: &BASE33_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE33_LARGE_MANTISSA, exp: &BASE33_LARGE_EXPONENT },
    small_int: &BASE33_SMALL_INT_POWERS,
    step: BASE33_STEP,
    bias: BASE33_BIAS,
};

pub(crate) const BASE34_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE34_SMALL_MANTISSA, exp: &BASE34_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE34_LARGE_MANTISSA, exp: &BASE34_LARGE_EXPONENT },
    small_int: &BASE34_SMALL_INT_POWERS,
    step: BASE34_STEP,
    bias: BASE34_BIAS,
};

pub(crate) const BASE35_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE35_SMALL_MANTISSA, exp: &BASE35_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE35_LARGE_MANTISSA, exp: &BASE35_LARGE_EXPONENT },
    small_int: &BASE35_SMALL_INT_POWERS,
    step: BASE35_STEP,
    bias: BASE35_BIAS,
};

pub(crate) const BASE36_POWERS: ModeratePathPowers<u128> = ModeratePathPowers {
    small: ExtendedFloatArray { mant: &BASE36_SMALL_MANTISSA, exp: &BASE36_SMALL_EXPONENT },
    large: ExtendedFloatArray { mant: &BASE36_LARGE_MANTISSA, exp: &BASE36_LARGE_EXPONENT },
    small_int: &BASE36_SMALL_INT_POWERS,
    step: BASE36_STEP,
    bias: BASE36_BIAS,
};

}}  // cfg_if

/// Get powers from radix.
pub(crate) fn get_powers(radix: u32)
    -> &'static ModeratePathPowers<u128>
{
    debug_assert_radix!(radix);

    #[cfg(not(feature = "radix"))] {
        &BASE10_POWERS
    }

    #[cfg(feature = "radix")] {
        match radix {
            3  => &BASE3_POWERS,
            5  => &BASE5_POWERS,
            6  => &BASE6_POWERS,
            7  => &BASE7_POWERS,
            9  => &BASE9_POWERS,
            10 => &BASE10_POWERS,
            11 => &BASE11_POWERS,
            12 => &BASE12_POWERS,
            13 => &BASE13_POWERS,
            14 => &BASE14_POWERS,
            15 => &BASE15_POWERS,
            17 => &BASE17_POWERS,
            18 => &BASE18_POWERS,
            19 => &BASE19_POWERS,
            20 => &BASE20_POWERS,
            21 => &BASE21_POWERS,
            22 => &BASE22_POWERS,
            23 => &BASE23_POWERS,
            24 => &BASE24_POWERS,
            25 => &BASE25_POWERS,
            26 => &BASE26_POWERS,
            27 => &BASE27_POWERS,
            28 => &BASE28_POWERS,
            29 => &BASE29_POWERS,
            30 => &BASE30_POWERS,
            31 => &BASE31_POWERS,
            33 => &BASE33_POWERS,
            34 => &BASE34_POWERS,
            35 => &BASE35_POWERS,
            36 => &BASE36_POWERS,
            // Powers of 2, and others, should already be handled by now.
            _  => unreachable!(),
        }
    }
}

// TESTS
// -----

#[cfg(test)]
mod tests {
    use crate::util::test::*;
    use super::*;

    #[test]
    fn normalization_test() {
        // Ensure each valid is normalized.
        for base in BASE_POWN.iter().cloned() {
            let powers = get_powers(base);
            for idx in 0..powers.small.len() {
                let fp = powers.get_small(idx);
                assert_eq!(fp.mant.leading_zeros(), 0);
            }
            for idx in 0..powers.large.len() {
                let fp = powers.get_large(idx);
                assert_eq!(fp.mant.leading_zeros(), 0);
            }
        }
    }

    #[cfg(feature = "radix")]
    #[test]
    #[should_panic]
    fn pow2_test() {
        for base in BASE_POW2.iter().cloned() {
            let _ = get_powers(base);
        }
    }
}
