/*
 * See the dyninst/COPYRIGHT file for copyright information.
 * 
 * We provide the Paradyn Tools (below described as "Paradyn")
 * on an AS IS basis, and do not warrant its validity or performance.
 * We reserve the right to update, modify, or discontinue this
 * software at any time.  We shall have no obligation to supply such
 * updates or modifications or any other form of support to you.
 * 
 * By your use of Paradyn, you understand and agree that we (or any
 * other person or entity with proprietary rights in Paradyn) are
 * under no obligation to provide either maintenance services,
 * update services, notices of latent defects, or correction of
 * defects for Paradyn.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

// groups.C: Doxygen file to collect group definitions and documentation, and ensure that they are found in the correct order.

/*! 
 * \defgroup moduleAbstractionIntro REMOVE
 * The Instruction API contains three major components: the top-level instruction representation,
 * the abstract syntax trees representing the operands of an instruction, and the decoder that creates
 * the entire representation.  We will present an overview of the features and uses of each of these three components,
 * followed by an example of how the Instruction API can be applied to binary analysis.
 */
/** \defgroup instruction Instruction Interface
 * The Instruction API represents a machine language instruction as an Instruction object,
 * which contains an Operation
 * and a collection of Operands.  The %Operation contains the following items:
 *   - The mnemonic for the machine language instruction represented by its associated %Instruction
 *   - The number of operands accepted by the %Operation
 *   - Which Operands are read and/or written by the associated machine operation
 *   - What other registers (if any) are affected by the underlying machine operation
 *
 * Each Operand contains
 * flags to indicate whether it is read, written, or both by the machine instruction
 * represented by its parent %Instruction, and contains a Expression abstract
 * syntax tree representing the operations required to compute the value of the operand.  Figure 1 depicts
 * these ownership relationships within an %Instruction.
 * \dotfile ownership_graph.dot "An Instruction and the objects it owns"
 *
 * Instruction objects provide two types of interfaces: direct read access to their components,
 * and common summary operations on those
 * components.  The first interface allows access to the %Operation and %Operand data members, and each
 * %Operand object in turn allows traversal of its abstract syntax tree.
 * More details about how to work with this abstract syntax tree can be found in \ref instruction_ast_module.
 * This interface would be used, for example, in a data flow analysis where a user wants to evaluate
 * the results of an effective address computation given a known register state.
 * 
 * The second interface allows
 * a user to get the sets of registers read and written by the instruction, information
 * about how the instruction accesses memory, and information about how the instruction affects
 * control flow, without having to manipulate the Operands directly.
 * For instance, a user could implement a register liveness analysis algorithm using just this second interface
 * (namely the \c getReadSet and \c getWriteSet functions).
 */
/** \defgroup instructiondecoder Instruction Decoding
 * An InstructionDecoder interprets a sequence of bytes according to a
 * given machine language and transforms them into an instruction
 * representation. It determines the opcode of the machine instruction,
 * translates that opcode to an Operation object, uses that %Operation to determine 
 * how to decode the instruction's Operands, and produces a decoded Instruction.
 *
 * \dotfile decoder_use.dot "The InstructionDecoder's inputs and outputs"
 * Instruction decoders are built from the following elements:
 *  - A function to find and extract an opcode given a pointer into a buffer that points to the beginning of a machine instruction
 *  - A table that, for a particular architecture,
 * maps opcodes to Operations and functions that decode Operands
 *
 * From these elements, it is possible to generalize the construction of Instructions
 * from Operations and Operands to an entirely platform-independent algorithm.  Likewise, much of
 * the construction of the ASTs representing each operand can be performed in a platform-independent manner.
 * 
 */
/** \defgroup instruction_ast_module InstructionAST Hierarchy
 * The AST representation of an operand encapsulates the operations performed
 * on registers and immediates to produce an operand for the machine
 * language instruction.
 *
 * The inheritance hierarchy of the AST classes is shown in Figure 3.
 * \dotfile full_inheritance_graph.dot "The InstructionAST inheritance hierarchy"
 * The grammar for these AST representations is simple: all leaves must be RegisterAST or Immediate nodes.
 * These nodes may be combined using a BinaryFunction node, which may be constructed as either
 * an addition or a multiplication.  Also, a single node may descend from a Dereference node, which
 * treats its child as a memory address.  Figure 4 shows the allowable parent/child relationships
 * within a given tree, and Figure 5 shows how an example IA32 instruction is represented using these objects.
 * \dotfile ast_ownership.dot "InstructionAST intermediate node types and the objects they own"
 * \dotfile instruction_representation.dot "The decomposition of \c mov \c %eax, (\c %esi)"
 * These ASTs may be searched for leaf elements or subtrees (via \c getUses and \c isUsed) and traversed
 * breadth-first or depth-first (via \c getChildren).
 *
 * Any node in these ASTs may be evaluated.  Evaluation attempts
 * to determine the value represented by a node.  If successful, it will return that value and cache it in the node.
 * The tree structure, combined with the evaluation mechanism, allows the substitution of known register and memory
 * values into an operand, regardless of whether those values are known at the time an instruction is decoded.
 * More details on this mechanism may be found in \ref Dyninst::InstructionAPI::Expression.
 *
 */

/** \defgroup liveness_example Example of Use: Register Liveness Analysis
/// Now that we've discussed the components of the API, we present an example that shows
/// how the API helps make analysis algorithms easy to implement.
///
/// Our first example will be a register liveness analysis algorithm over a basic block of code.
/// We'll elide the decoding of instructions and the parsing necessary to build a basic block,
/// and start from a representation of a basic block as a decoded sequence of Instruction objects.
///
/// \code
/// std::set<RegisterAST::Ptr> getLiveRegisters(const std::map<Address, Instruction>& basicBlock)
/// {
///   typedef std::map<Address, std::set<RegisterAST::Ptr> > LivenessSet;
///   LivenessSet liveRegsPost;
///   LivenessSet liveRegsPre;
///   for(std::map<Address, Instruction>::reverse_iterator ri = basicBlock.rbegin();
///       ri != basicBlock.rend();
///       ++ri)
///   {
///     Address curAddr = ri->first;
///     Instruction curInsn = ri->second;
///     std::set_difference(liveRegsPost[curAddr].begin(), liveRegsPost[curAddr].end(), 
///       curInsn->getRegsWritten().begin(), curInsn.getRegsWritten().end(), std::inserter(liveRegsPre[curAddr].begin());
///     std::set_union(liveRegsPre[curAddr].begin(), liveRegsPre[curAddr].end(), curInsn->getRegsRead().begin(), curInsn->getRegsRead().end(),
///       std::inserter(liveRegsPre[curAddr].begin()));
///     
///   }
*/
