// Boost.Geometry
// Unit Test

// Copyright (c) 2016-2020 Oracle and/or its affiliates.

// Contributed and/or modified by Adam Wulkiewicz, on behalf of Oracle

// Use, modification and distribution is subject to the Boost Software License,
// Version 1.0. (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_GEOMETRY_TEST_INVERSE_CASES_HPP
#define BOOST_GEOMETRY_TEST_INVERSE_CASES_HPP

struct coordinates
{
    double lon;
    double lat;
};

struct expected_result
{
    double distance;
    double azimuth;
    double reverse_azimuth;
    double reduced_length;
    double geodesic_scale;
};

struct expected_results
{
    coordinates p1;
    coordinates p2;
    expected_result reference; // karney or vincenty
    expected_result vincenty;
    expected_result thomas;
    expected_result andoyer;

};

expected_results expected[] =
{
    {
        { 0, 0 },{ 2, 0 },
        { 222638.98158654713188298047, 90.00000000000000000000, 90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 },
        { 222638.98158645280636847019, 90.00000000000000000000, 90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 },
        { 222638.98158654122380539775, 90.00000000000000000000, 90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 },
        { 222638.98158654125290922821, 90.00000000000000000000, 90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 }
    },{
        { 0, 0 },{ 2, 2 },
        { 313775.70942918420769274235, 45.17488858648466987233, 45.20980230803674970730, 313648.30600607168162241578, 0.99878199603375228399 },
        { 313775.70929574419278651476, 45.17488858646600391467, 45.20980230801805532792, 313648.30601466231746599078, 0.99878199603375328319 },
        { 313775.70869052573107182980, 45.17488750916628958976, 45.20980122940459722258, 313648.30008120078127831221, 0.99878199607986462016 },
        { 313772.17715740547282621264, 45.17456700769592004008, 45.20948072916769433505, 313646.54128309490624815226, 0.99878200978130526888 }
    },{
        { 0, 0 },{ 0, 2 },
        { 221149.45337213488528504968, 0.00000000000000000000, 0.00000000000000000000, 221104.84592454897938296199, 0.99939490135383823866 },
        { 221149.45318342180689796805, 0.00000000000000000000, 0.00000000000000000000, 221104.84593068063259124756, 0.99939490135383812763 },
        { 221149.45337045041378587484, 0.00000000000000000000, 0.00000000000000000000, 221104.84593068063259124756, 0.99939490135383812763 },
        { 221146.95820782837108708918, 0.00000000000000002350, 0.00000000000000002350, 221104.84893554644077084959, 0.99939490135383812763 }
    },{
        { 0, 0 },{ -2, 2 },
        { 313775.70942918420769274235, -45.17488858648466987233, -45.20980230803674970730, 313648.30600607168162241578, 0.99878199603375228399 },
        { 313775.70929574419278651476, -45.17488858646600391467, -45.20980230801805532792, 313648.30601466231746599078, 0.99878199603375328319 },
        { 313775.70869052573107182980, -45.17488750916629669518, -45.20980122940459722258, 313648.30008120078127831221, 0.99878199607986462016 },
        { 313772.17715740547282621264, -45.17456700769592004008, -45.20948072916767301876, 313646.54128309484804049134, 0.99878200978130526888 }
    },{
        { 0, 0 },{ -2, 0 },
        { 222638.98158654713188298047, -90.00000000000000000000, -90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 },
        { 222638.98158645280636847019, -90.00000000000000000000, -90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 },
        { 222638.98158654122380539775, -90.00000000000000000000, -90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 },
        { 222638.98158654125290922821, -90.00000000000000000000, -90.00000000000000000000, 222593.46652303214068524539, 0.99938672191946420487 }
    },{
        { 0, 0 },{ -2, -2 },
        { 313775.70942918420769274235, -134.82511141351534433852, -134.79019769196324318727, 313648.30600607168162241578, 0.99878199603375228399 },
        { 313775.70929574419278651476, -134.82511141353401740162, -134.79019769198197309379, 313648.30601466225925832987, 0.99878199603375317217 },
        { 313775.70869052573107182980, -134.82511249083370330482, -134.79019877059539567199, 313648.30008120083948597312, 0.99878199607986462016 },
        { 313772.17715740547282621264, -134.82543299230408706535, -134.79051927083230566495, 313646.54128309502266347408, 0.99878200978130537990 }
    },{
        { 0, 0 },{ 0, -2 },
        { 221149.45337213488528504968, -180.00000000000000000000, -180.00000000000000000000, 221104.84592454897938296199, 0.99939490135383823866 },
        { 221149.45318342180689796805, 180.00000000000000000000, 180.00000000000000000000, 221104.84593068063259124756, 0.99939490135383812763 },
        { 221149.45337045041378587484, 180.00000000000000000000, 180.00000000000000000000, 221104.84593068063259124756, 0.99939490135383812763 },
        { 221146.95820782837108708918, 180.00000000000000000000, 180.00000000000000000000, 221104.84893554644077084959, 0.99939490135383812763 }
    },{
        { 0, 0 },{ 2, -2 },
        { 313775.70942918420769274235, 134.82511141351534433852, 134.79019769196324318727, 313648.30600607168162241578, 0.99878199603375228399 },
        { 313775.70929574419278651476, 134.82511141353401740162, 134.79019769198197309379, 313648.30601466225925832987, 0.99878199603375317217 },
        { 313775.70869052573107182980, 134.82511249083370330482, 134.79019877059539567199, 313648.30008120083948597312, 0.99878199607986462016 },
        { 313772.17715740547282621264, 134.82543299230408706535, 134.79051927083230566495, 313646.54128309502266347408, 0.99878200978130537990 }
    },{
        { -1, 0 },{ 2, 0 },
        { 333958.47237982071237638593, 90.00000000000000000000, 90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 },
        { 333958.47237967920955270529, 90.00000000000000000000, 90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 },
        { 333958.47237982566002756357, 90.00000000000000000000, 90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 },
        { 333958.47237982566002756357, 90.00000000000000000000, 90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 }
    },{
        { -1, 0 },{ 2, 2 },
        { 400487.62166116386651992798, 56.46443555420994186989, 56.51681282834063324572, 400222.73679570353124290705, 0.99801604059094650712 },
        { 400487.62155657896073535085, 56.46443555419775606197, 56.51681282832842612152, 400222.73680666065774857998, 0.99801604059094761734 },
        { 400487.62116943788714706898, 56.46443456250134573793, 56.51681183466060787168, 400222.72635490150423720479, 0.99801604069467231284 },
        { 400484.31980028707766905427, 56.46436757836366382435, 56.51674485214859089410, 400222.02724612434394657612, 0.99801604768666130063 }
    },{
        { -1, 0 },{ 0, 2 },
        { 247576.47264764847932383418, 26.70874628270370010341, 26.72620180544743817563, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, 26.70874628267113948255, 26.72620180541484202763, 247513.88751566910650581121, 0.99924166327570307367 },
        { 247576.47204536388744600117, 26.70874541733553186873, 26.72620093942252594843, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, 26.70817834948789482041, 26.72563387223648589952, 247512.65741612159763462842, 0.99924167083675718981 }
    },{
        { -1, 0 },{ -2, 2 },
        { 247576.47264764847932383418, -26.70874628270370010341, -26.72620180544743817563, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, -26.70874628267113948255, -26.72620180541484202763, 247513.88751566910650581121, 0.99924166327570307367 },
        { 247576.47204536388744600117, -26.70874541733554607958, -26.72620093942252594843, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, -26.70817834948789482041, -26.72563387223651432123, 247512.65741612159763462842, 0.99924167083675718981 }
    },{
        { -1, 0 },{ -2, 0 },
        { 111319.49079327356594149023, -90.00000000000000000000, -90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 },
        { 111319.49079322640318423510, -90.00000000000000000000, -90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 },
        { 111319.49079326246283017099, -90.00000000000000000000, -90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 },
        { 111319.49079326246283017099, -90.00000000000000000000, -90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 }
    },{
        { -1, 0 },{ -2, -2 },
        { 247576.47264764847932383418, -153.29125371729628568573, -153.27379819455256892979, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, -153.29125371732885696474, -153.27379819458516863051, 247513.88751566907740198076, 0.99924166327570307367 },
        { 247576.47204536388744600117, -153.29125458266443615685, -153.27379906057745984072, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, -153.29182165051210517959, -153.27436612776349988962, 247512.65741612162673845887, 0.99924167083675718981 }
    },{
        { -1, 0 },{ 0, -2 },
        { 247576.47264764847932383418, 153.29125371729628568573, 153.27379819455256892979, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, 153.29125371732885696474, 153.27379819458516863051, 247513.88751566907740198076, 0.99924166327570307367 },
        { 247576.47204536388744600117, 153.29125458266446457856, 153.27379906057745984072, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, 153.29182165051210517959, 153.27436612776349988962, 247512.65741612162673845887, 0.99924167083675718981 }
    },{
        { -1, 0 },{ 2, -2 },
        { 400487.62166116386651992798, 123.53556444579005813011, 123.48318717165936675428, 400222.73679570353124290705, 0.99801604059094650712 },
        { 400487.62155657896073535085, 123.53556444580226525431, 123.48318717167158808934, 400222.73680666059954091907, 0.99801604059094761734 },
        { 400487.62116943788714706898, 123.53556543749867557835, 123.48318816533939923374, 400222.72635490150423720479, 0.99801604069467231284 },
        { 400484.31980028707766905427, 123.53563242163632196480, 123.48325514785142331675, 400222.02724612434394657612, 0.99801604768666130063 }
    },{
        { 1, 0 },{ 2, 0 },
        { 111319.49079327356594149023, 90.00000000000000000000, 90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 },
        { 111319.49079322640318423510, 90.00000000000000000000, 90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 },
        { 111319.49079326246283017099, 90.00000000000000000000, 90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 },
        { 111319.49079326246283017099, 90.00000000000000000000, 90.00000000000000000000, 111313.80114861292531713843, 0.99984666872462601983 }
    },{
        { 1, 0 },{ 2, 2 },
        { 247576.47264764847932383418, 26.70874628270370010341, 26.72620180544743817563, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, 26.70874628267113948255, 26.72620180541484202763, 247513.88751566910650581121, 0.99924166327570307367 },
        { 247576.47204536388744600117, 26.70874541733553186873, 26.72620093942252594843, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, 26.70817834948789482041, 26.72563387223648589952, 247512.65741612159763462842, 0.99924167083675718981 }
    },{
        { 1, 0 },{ 0, 2 },
        { 247576.47264764847932383418, -26.70874628270370010341, -26.72620180544743817563, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, -26.70874628267113948255, -26.72620180541484202763, 247513.88751566910650581121, 0.99924166327570307367 },
        { 247576.47204536388744600117, -26.70874541733554607958, -26.72620093942252594843, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, -26.70817834948789482041, -26.72563387223651432123, 247512.65741612159763462842, 0.99924167083675718981 }
    },{
        { 1, 0 },{ -2, 2 },
        { 400487.62166116386651992798, -56.46443555420994186989, -56.51681282834063324572, 400222.73679570353124290705, 0.99801604059094650712 },
        { 400487.62155657896073535085, -56.46443555419775606197, -56.51681282832842612152, 400222.73680666065774857998, 0.99801604059094761734 },
        { 400487.62116943788714706898, -56.46443456250133863250, -56.51681183466062918797, 400222.72635490167886018753, 0.99801604069467231284 },
        { 400484.31980028707766905427, -56.46436757836366382435, -56.51674485214855536697, 400222.02724612399470061064, 0.99801604768666130063 }
    },{
        { 1, 0 },{ -2, 0 },
        { 333958.47237982071237638593, -90.00000000000000000000, -90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 },
        { 333958.47237967920955270529, -90.00000000000000000000, -90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 },
        { 333958.47237982566002756357, -90.00000000000000000000, -90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 },
        { 333958.47237982566002756357, -90.00000000000000000000, -90.00000000000000000000, 333804.87081722758011892438, 0.99862030063297502203 }
    },{
        { 1, 0 },{ -2, -2 },
        { 400487.62166116386651992798, -123.53556444579005813011, -123.48318717165936675428, 400222.73679570353124290705, 0.99801604059094650712 },
        { 400487.62155657896073535085, -123.53556444580226525431, -123.48318717167158808934, 400222.73680666059954091907, 0.99801604059094761734 },
        { 400487.62116943788714706898, -123.53556543749867557835, -123.48318816533941344460, 400222.72635490121319890022, 0.99801604069467231284 },
        { 400484.31980028707766905427, -123.53563242163632196480, -123.48325514785139489504, 400222.02724612463498488069, 0.99801604768666130063 }
    },{
        { 1, 0 },{ 0, -2 },
        { 247576.47264764847932383418, -153.29125371729628568573, -153.27379819455256892979, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, -153.29125371732885696474, -153.27379819458516863051, 247513.88751566907740198076, 0.99924166327570307367 },
        { 247576.47204536388744600117, -153.29125458266443615685, -153.27379906057745984072, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, -153.29182165051210517959, -153.27436612776349988962, 247512.65741612162673845887, 0.99924167083675718981 }
    },{
        { 1, 0 },{ 2, -2 },
        { 247576.47264764847932383418, 153.29125371729628568573, 153.27379819455256892979, 247513.88750887679634615779, 0.99924166327570262958 },
        { 247576.47247877591871656477, 153.29125371732885696474, 153.27379819458516863051, 247513.88751566907740198076, 0.99924166327570307367 },
        { 247576.47204536388744600117, 153.29125458266446457856, 153.27379906057745984072, 247513.88563483985490165651, 0.99924166328723251773 },
        { 247573.34468778557493351400, 153.29182165051210517959, 153.27436612776349988962, 247512.65741612162673845887, 0.99924167083675718981 }
    },{
        { 0, -1 },{ 2, 0 },
        { 248575.56516798117081634700, 63.59904041930798257454, 63.58158489649751743400, 248512.21908866314333863556, 0.99923553066716652715 },
        { 248575.56515735003631561995, 63.59904041929794260568, 63.58158489648748457057, 248512.21909045605571009219, 0.99923553066715220528 },
        { 248575.56501938513247296214, 63.59903955892717419829, 63.58158403677443004653, 248512.21158473053947091103, 0.99923553071334769715 },
        { 248574.11260137573117390275, 63.59909262487922632090, 63.58163710212382113696, 248512.67536870032199658453, 0.99923552785888392069 }
    },{
        { 0, -1 },{ 2, 2 },
        { 399491.95531813317211344838, 33.86382331638844789268, 33.88128418166377286980, 399229.04036820423789322376, 0.99802588852441009859 },
        { 399491.95525925338733941317, 33.86382331636024645150, 33.88128418163555721776, 399229.04037646017968654633, 0.99802588850630413742 },
        { 399491.95420239900704473257, 33.86382231856556046523, 33.88128318318352683036, 399229.03571708954405039549, 0.99802588855242924115 },
        { 399486.91562757542124018073, 33.86329689027443379246, 33.88075775553439683563, 399226.58732164395041763783, 0.99802591281687313973 }
    },{
        { 0, -1 },{ 0, 2 },
        { 331723.84192993369651958346, 0.00000000000000000000, 0.00000000000000000000, 331573.30333569750655442476, 0.99863870153788791839 },
        { 331723.84185956208966672421, 0.00000000000000000000, 0.00000000000000000000, 331573.30334328504977747798, 0.99863870152539924163 },
        { 331723.84192803525365889072, 0.00000000000000000000, 0.00000000000000000000, 331573.30334328504977747798, 0.99863870152539924163 },
        { 331720.09633564797695726156, 0.00000000000000002351, 0.00000000000000002350, 331573.30672345001948997378, 0.99863870151614875237 }
    },{
        { 0, -1 },{ -2, 2 },
        { 399491.95531813317211344838, -33.86382331638844789268, -33.88128418166377286980, 399229.04036820423789322376, 0.99802588852441009859 },
        { 399491.95525925338733941317, -33.86382331636024645150, -33.88128418163555721776, 399229.04037646017968654633, 0.99802588850630413742 },
        { 399491.95420239900704473257, -33.86382231856555335980, -33.88128318318355525207, 399229.03571708960225805640, 0.99802588855242924115 },
        { 399486.91562757542124018073, -33.86329689027443379246, -33.88075775553439683563, 399226.58732164395041763783, 0.99802591281687313973 }
    },{
        { 0, -1 },{ -2, 0 },
        { 248575.56516798117081634700, -63.59904041930798257454, -63.58158489649751743400, 248512.21908866314333863556, 0.99923553066716652715 },
        { 248575.56515735003631561995, -63.59904041929794260568, -63.58158489648748457057, 248512.21909045605571009219, 0.99923553066715220528 },
        { 248575.56501938513247296214, -63.59903955892717419829, -63.58158403677445846824, 248512.21158473053947091103, 0.99923553071334769715 },
        { 248574.11260137573117390275, -63.59909262487922632090, -63.58163710212379271525, 248512.67536870032199658453, 0.99923552785888392069 }
    },{
        { 0, -1 },{ -2, -2 },
        { 248515.52773222429095767438, -116.44291541236518128244, -116.39055416434335654685, 248452.22805491264443844557, 0.99923590400821504787 },
        { 248515.52763759149820543826, -116.44291541237451781399, -116.39055416435273571096, 248452.22805559608968906105, 0.99923590407101092747 },
        { 248515.52758369332877919078, -116.44291626924928095832, -116.39055502319511958831, 248452.22057304141344502568, 0.99923590411705243142 },
        { 248514.07922688819235190749, -116.44286383106521043374, -116.39050258320843056481, 248452.68231273870333097875, 0.99923590134832673826 }
    },{
        { 0, -1 },{ 0, -2 },
        { 110575.06481433614681009203, -180.00000000000000000000, -180.00000000000000000000, 110569.48860416181560140103, 0.99984871368918792900 },
        { 110575.06460189135395921767, 180.00000000000000000000, 180.00000000000000000000, 110569.48860883565794210881, 0.99984871370166705784 },
        { 110575.06481286860071122646, 180.00000000000000000000, 180.00000000000000000000, 110569.48860883565794210881, 0.99984871370166705784 },
        { 110573.82008000080531928688, 180.00000000000000000000, 180.00000000000000000000, 110569.49123749321734067053, 0.99984871370886085895 }
    },{
        { 0, -1 },{ 2, -2 },
        { 248515.52773222429095767438, 116.44291541236518128244, 116.39055416434335654685, 248452.22805491264443844557, 0.99923590400821504787 },
        { 248515.52763759149820543826, 116.44291541237451781399, 116.39055416435273571096, 248452.22805559608968906105, 0.99923590407101092747 },
        { 248515.52758369332877919078, 116.44291626924930938003, 116.39055502319511958831, 248452.22057304164627566934, 0.99923590411705243142 },
        { 248514.07922688819235190749, 116.44286383106521043374, 116.39050258320843056481, 248452.68231273870333097875, 0.99923590134832673826 }
    },{
        { 0, 1 },{ 2, 0 },
        { 248575.56516798117081634700, 116.40095958069201742546, 116.41841510350248256600, 248512.21908866314333863556, 0.99923553066716652715 },
        { 248575.56515735003631561995, 116.40095958070207871060, 116.41841510351252964028, 248512.21909045602660626173, 0.99923553066715220528 },
        { 248575.56501938513247296214, 116.40096044107282580171, 116.41841596322554153176, 248512.21158473062678240240, 0.99923553071334780817 },
        { 248574.11260137573117390275, 116.40090737512078078453, 116.41836289787615044133, 248512.67536870026378892362, 0.99923552785888392069 }
    },{
        { 0, 1 },{ 2, 2 },
        { 248515.52773222429095767438, 63.55708458763482582299, 63.60944583565664345315, 248452.22805491264443844557, 0.99923590400821504787 },
        { 248515.52763759149820543826, 63.55708458762548218601, 63.60944583564726428904, 248452.22805559608968906105, 0.99923590407101092747 },
        { 248515.52758369332877919078, 63.55708373075069061997, 63.60944497680485909541, 248452.22057304115151055157, 0.99923590411705243142 },
        { 248514.07922688819235190749, 63.55713616893478246084, 63.60949741679156232976, 248452.68231273847050033510, 0.99923590134832684928 }
    },{
        { 0, 1 },{ 0, 2 },
        { 110575.06481433614681009203, 0.00000000000000000000, 0.00000000000000000000, 110569.48860416181560140103, 0.99984871368918792900 },
        { 110575.06460189135395921767, 0.00000000000000000000, 0.00000000000000000000, 110569.48860883565794210881, 0.99984871370166705784 },
        { 110575.06481286860071122646, 0.00000000000000000000, 0.00000000000000000000, 110569.48860883565794210881, 0.99984871370166705784 },
        { 110573.82008000080531928688, 0.00000000000000002350, 0.00000000000000002350, 110569.49123749321734067053, 0.99984871370886085895 }
    },{
        { 0, 1 },{ -2, 2 },
        { 248515.52773222429095767438, -63.55708458763482582299, -63.60944583565664345315, 248452.22805491264443844557, 0.99923590400821504787 },
        { 248515.52763759149820543826, -63.55708458762548218601, -63.60944583564726428904, 248452.22805559608968906105, 0.99923590407101092747 },
        { 248515.52758369332877919078, -63.55708373075070483083, -63.60944497680488041169, 248452.22057304150075651705, 0.99923590411705254244 },
        { 248514.07922688819235190749, -63.55713616893478246084, -63.60949741679156943519, 248452.68231273873243480921, 0.99923590134832684928 }
    },{
        { 0, 1 },{ -2, 0 },
        { 248575.56516798117081634700, -116.40095958069201742546, -116.41841510350248256600, 248512.21908866314333863556, 0.99923553066716652715 },
        { 248575.56515735003631561995, -116.40095958070207871060, -116.41841510351252964028, 248512.21909045602660626173, 0.99923553066715220528 },
        { 248575.56501938513247296214, -116.40096044107282580171, -116.41841596322554153176, 248512.21158473062678240240, 0.99923553071334780817 },
        { 248574.11260137573117390275, -116.40090737512078078453, -116.41836289787615044133, 248512.67536870026378892362, 0.99923552785888392069 }
    },{
        { 0, 1 },{ -2, -2 },
        { 399491.95531813317211344838, -146.13617668361155210732, -146.11871581833622713020, 399229.04036820423789322376, 0.99802588852441009859 },
        { 399491.95525925338733941317, -146.13617668363974644308, -146.11871581836444988767, 399229.04037646006327122450, 0.99802588850630413742 },
        { 399491.95420239900704473257, -146.13617768143444664020, -146.11871681681645895878, 399229.03571708971867337823, 0.99802588855242924115 },
        { 399486.91562757542124018073, -146.13670310972554489126, -146.11924224446559605894, 399226.58732164406683295965, 0.99802591281687325075 }
    },{
        { 0, 1 },{ 0, -2 },
        { 331723.84192993369651958346, -180.00000000000000000000, -180.00000000000000000000, 331573.30333569750655442476, 0.99863870153788791839 },
        { 331723.84185956208966672421, 180.00000000000000000000, 180.00000000000000000000, 331573.30334328504977747798, 0.99863870152539924163 },
        { 331723.84192803525365889072, 180.00000000000000000000, 180.00000000000000000000, 331573.30334328504977747798, 0.99863870152539924163 },
        { 331720.09633564797695726156, 180.00000000000000000000, 180.00000000000000000000, 331573.30672345001948997378, 0.99863870151614875237 }
    },{
        { 0, 1 },{ 2, -2 },
        { 399491.95531813317211344838, 146.13617668361155210732, 146.11871581833622713020, 399229.04036820423789322376, 0.99802588852441009859 },
        { 399491.95525925338733941317, 146.13617668363974644308, 146.11871581836444988767, 399229.04037646006327122450, 0.99802588850630413742 },
        { 399491.95420239900704473257, 146.13617768143444664020, 146.11871681681645895878, 399229.03571708971867337823, 0.99802588855242924115 },
        { 399486.91562757542124018073, 146.13670310972554489126, 146.11924224446559605894, 399226.58732164406683295965, 0.99802591281687325075 }
    },
    {
        { -1, -1 },{ 2, 0 },
        { 351772.23553088010521605611, 71.69677339271241578444, 71.67058676207953737958, 351592.72256732499226927757, 0.99846922689201267342 },
        { 351772.23552330053644254804, 71.69677339270522509196, 71.67058676207236089795, 351592.72256738768192008138, 0.99846922689200567902 },
        { 351772.23547190619865432382, 71.69677274795321864076, 71.67058611830732672843, 351592.71064292185474187136, 0.99846922699591911243 },
        { 351771.13000385620398446918, 71.69688960494077889507, 71.67070297450321447741, 351594.87308142206165939569, 0.99846920815167106156 }
    }, {
        { -1, -1 },{ 2, 2 },
        { 470675.27496387914288789034, 45.18799532476352709409, 45.21418997011227247640, 470245.32092262554215267301, 0.99726004843767979136 },
        { 470675.27491378918057307601, 45.18799532474472613330, 45.21418997009345730476, 470245.32093124865787103772, 0.99726004841255500022 },
        { 470675.27385581465205177665, 45.18799424651773222195, 45.21418889088008796762, 470245.31203990190988406539, 0.99726004851631100401 },
        { 470669.97387331607751548290, 45.18767381182056652733, 45.21386845703356982540, 470242.67608062457293272018, 0.99726007931332938394 }
    }, {
        { -1, -1 },{ 0, 2 },
        { 349898.53698544885264709592, 18.54804530050607525027, 18.55677506382081531910, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, 18.54804530047409016902, 18.55677506378881247429, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, 18.54804465014367664821, 18.55677441312981201804, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, 18.54754137243966738424, 18.55627113576925424354, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { -1, -1 },{ -2, 2 },
        { 349898.53698544885264709592, -18.54804530050607525027, -18.55677506382081531910, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, -18.54804530047409016902, -18.55677506378881247429, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, -18.54804465014367664821, -18.55677441312981201804, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, -18.54754137243966738424, -18.55627113576928266525, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { -1, -1 },{ -2, 0 },
        { 156899.56829134028521366417, -45.19676732164486310239, -45.18804022935886877121, 156883.63778222308610565960, 0.99969540695122460772 },
        { 156899.56827460310887545347, -45.19676732162573529195, -45.18804022933975517162, 156883.63778401838499121368, 0.99969540695121617002 },
        { 156899.56792193598812445998, -45.19676624219880523015, -45.18803915024161454994, 156883.63480984271154738963, 0.99969540696276681935 },
        { 156897.79947260793414898217, -45.19644594033049145310, -45.18771884804953486992, 156882.75281389255542308092, 0.99969541038810816325 }
    }, {
        { -1, -1 },{ -2, -2 },
        { 156876.14940188667969778180, -134.82952991582811819171, -134.80335129773999369718, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, -134.82952991584656388113, -134.80335129775846780831, 156860.22615983051946386695, 0.99969549954768643918 },
        { 156876.14903264911845326424, -134.82953099167991695140, -134.80335237457535413341, 156860.22319510785746388137, 0.99969549955919845274 },
        { 156874.38594904550700448453, -134.82985169207830722371, -134.80367307400518939176, 156859.34477911840076558292, 0.99969550299904519353 }
    }, {
        { -1, -1 },{ 0, -2 },
        { 156876.14940188667969778180, 134.82952991582811819171, 134.80335129773999369718, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, 134.82952991584656388113, 134.80335129775846780831, 156860.22615983051946386695, 0.99969549954768643918 },
        { 156876.14903264911845326424, 134.82953099167994537311, 134.80335237457538255512, 156860.22319510785746388137, 0.99969549955919856377 },
        { 156874.38594904550700448453, 134.82985169207830722371, 134.80367307400518939176, 156859.34477911840076558292, 0.99969550299904519353 }
    }, {
        { -1, -1 },{ 2, -2 },
        { 351676.50043935602298006415, 108.36087395536006283692, 108.28232205125480902552, 351497.13544799503870308399, 0.99847006808788085763 },
        { 351676.50037266127765178680, 108.36087395536665667350, 108.28232205126143128382, 351497.13543194788508117199, 0.99847006821303718738 },
        { 351676.50038040406070649624, 108.36087459655153963922, 108.28232269539883247944, 351497.12354394816793501377, 0.99847006831660412018 },
        { 351675.39839278126601129770, 108.36075825022818719390, 108.28220634670809374711, 351499.26940269087208434939, 0.99847004976658149111 }
    }, {
        { -1, 1 },{ 2, 0 },
        { 351772.23553088010521605611, 108.30322660728758421556, 108.32941323792046262042, 351592.72256732499226927757, 0.99846922689201267342 },
        { 351772.23552330053644254804, 108.30322660729477490804, 108.32941323792763910205, 351592.72256738756550475955, 0.99846922689200567902 },
        { 351772.23547190619865432382, 108.30322725204678135924, 108.32941388169267327157, 351592.71064292197115719318, 0.99846922699591911243 },
        { 351771.13000385620398446918, 108.30311039505922110493, 108.32929702549677131174, 351594.87308142217807471752, 0.99846920815167106156 }
    }, {
        { -1, 1 },{ 2, 2 },
        { 351676.50043935602298006415, 71.63912604463993716308, 71.71767794874519097448, 351497.13544799503870308399, 0.99847006808788085763 },
        { 351676.50037266127765178680, 71.63912604463332911564, 71.71767794873858292704, 351497.13543194829253479838, 0.99847006821303718738 },
        { 351676.50038040406070649624, 71.63912540344847457163, 71.71767730460115330970, 351497.12354394729482010007, 0.99847006831660423121 },
        { 351675.39839278126601129770, 71.63924174977181280610, 71.71779365329189204203, 351499.26940269029000774026, 0.99847004976658149111 }
    }, {
        { -1, 1 },{ 0, 2 },
        { 156876.14940188667969778180, 45.17047008417186759743, 45.19664870225999919739, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, 45.17047008415342190801, 45.19664870224154640255, 156860.22615983063587918878, 0.99969549954768632816 },
        { 156876.14903264911845326424, 45.17046900832006883775, 45.19664762542461744488, 156860.22319510774104855955, 0.99969549955919845274 },
        { 156874.38594904550700448453, 45.17014830792169988172, 45.19632692599481060824, 156859.34477911831345409155, 0.99969550299904519353 }
    }, {
        { -1, 1 },{ -2, 2 },
        { 156876.14940188667969778180, -45.17047008417186759743, -45.19664870225999919739, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, -45.17047008415342190801, -45.19664870224154640255, 156860.22615983063587918878, 0.99969549954768632816 },
        { 156876.14903264911845326424, -45.17046900832009015403, -45.19664762542463876116, 156860.22319510785746388137, 0.99969549955919856377 },
        { 156874.38594904550700448453, -45.17014830792169988172, -45.19632692599484613538, 156859.34477911848807707429, 0.99969550299904519353 }
    }, {
        { -1, 1 },{ -2, 0 },
        { 156899.56829134028521366417, -134.80323267835512979218, -134.81195977064112412336, 156883.63778222308610565960, 0.99969540695122460772 },
        { 156899.56827460310887545347, -134.80323267837425760263, -134.81195977066025193380, 156883.63778401838499121368, 0.99969540695121617002 },
        { 156899.56792193598812445998, -134.80323375780119476985, -134.81196084975837834463, 156883.63480984274065122008, 0.99969540696276681935 },
        { 156897.79947260793414898217, -134.80355405966952275776, -134.81228115195048644637, 156882.75281389255542308092, 0.99969541038810816325 }
    }, {
        { -1, 1 },{ -2, -2 },
        { 349898.53698544885264709592, -161.45195469949391053888, -161.44322493617917757547, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, -161.45195469952594180540, -161.44322493621118042029, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, -161.45195534985634822078, -161.44322558687019864010, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, -161.45245862756033261576, -161.44372886423073509832, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { -1, 1 },{ 0, -2 },
        { 349898.53698544885264709592, 161.45195469949391053888, 161.44322493617917757547, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, 161.45195469952594180540, 161.44322493621118042029, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, 161.45195534985634822078, 161.44322558687019864010, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, 161.45245862756033261576, 161.44372886423073509832, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { -1, 1 },{ 2, -2 },
        { 470675.27496387914288789034, 134.81200467523646580048, 134.78581002988772752360, 470245.32092262554215267301, 0.99726004843767979136 },
        { 470675.27491378918057307601, 134.81200467525528097212, 134.78581002990657111695, 470245.32093124854145571589, 0.99726004841255500022 },
        { 470675.27385581465205177665, 134.81200575348228198891, 134.78581110911991913781, 470245.31203990190988406539, 0.99726004851631100401 },
        { 470669.97387331607751548290, 134.81232618817941215639, 134.78613154296644438546, 470242.67608062474755570292, 0.99726007931332960599 }
    }, {
        { 1, 1 },{ 2, 0 },
        { 156899.56829134028521366417, 134.80323267835512979218, 134.81195977064112412336, 156883.63778222308610565960, 0.99969540695122460772 },
        { 156899.56827460310887545347, 134.80323267837425760263, 134.81195977066025193380, 156883.63778401838499121368, 0.99969540695121617002 },
        { 156899.56792193598812445998, 134.80323375780122319156, 134.81196084975840676634, 156883.63480984268244355917, 0.99969540696276681935 },
        { 156897.79947260793414898217, 134.80355405966952275776, 134.81228115195048644637, 156882.75281389255542308092, 0.99969541038810816325 }
    }, {
        { 1, 1 },{ 2, 2 },
        { 156876.14940188667969778180, 45.17047008417186759743, 45.19664870225999919739, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, 45.17047008415342190801, 45.19664870224154640255, 156860.22615983063587918878, 0.99969549954768632816 },
        { 156876.14903264911845326424, 45.17046900832006883775, 45.19664762542461744488, 156860.22319510774104855955, 0.99969549955919845274 },
        { 156874.38594904550700448453, 45.17014830792169988172, 45.19632692599481060824, 156859.34477911831345409155, 0.99969550299904519353 }
    }, {
        { 1, 1 },{ 0, 2 },
        { 156876.14940188667969778180, -45.17047008417186759743, -45.19664870225999919739, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, -45.17047008415342190801, -45.19664870224154640255, 156860.22615983063587918878, 0.99969549954768632816 },
        { 156876.14903264911845326424, -45.17046900832009015403, -45.19664762542463876116, 156860.22319510785746388137, 0.99969549955919856377 },
        { 156874.38594904550700448453, -45.17014830792169988172, -45.19632692599484613538, 156859.34477911848807707429, 0.99969550299904519353 }
    }, {
        { 1, 1 },{ -2, 2 },
        { 351676.50043935602298006415, -71.63912604463993716308, -71.71767794874519097448, 351497.13544799503870308399, 0.99847006808788085763 },
        { 351676.50037266127765178680, -71.63912604463332911564, -71.71767794873858292704, 351497.13543194829253479838, 0.99847006821303718738 },
        { 351676.50038040406070649624, -71.63912540344847457163, -71.71767730460118173141, 351497.12354394828435033560, 0.99847006831660412018 },
        { 351675.39839278126601129770, -71.63924174977181280610, -71.71779365329184940947, 351499.26940268895123153925, 0.99847004976658171316 }
    }, {
        { 1, 1 },{ -2, 0 },
        { 351772.23553088010521605611, -108.30322660728758421556, -108.32941323792046262042, 351592.72256732499226927757, 0.99846922689201267342 },
        { 351772.23552330053644254804, -108.30322660729477490804, -108.32941323792763910205, 351592.72256738756550475955, 0.99846922689200567902 },
        { 351772.23547190619865432382, -108.30322725204673872668, -108.32941388169263063901, 351592.71064292272785678506, 0.99846922699591933448 },
        { 351771.13000385620398446918, -108.30311039505922110493, -108.32929702549677131174, 351594.87308142217807471752, 0.99846920815167106156 }
    }, {
        { 1, 1 },{ -2, -2 },
        { 470675.27496387914288789034, -134.81200467523646580048, -134.78581002988772752360, 470245.32092262554215267301, 0.99726004843767979136 },
        { 470675.27491378918057307601, -134.81200467525528097212, -134.78581002990657111695, 470245.32093124854145571589, 0.99726004841255500022 },
        { 470675.27385581465205177665, -134.81200575348222514549, -134.78581110911991913781, 470245.31203990196809172630, 0.99726004851631100401 },
        { 470669.97387331607751548290, -134.81232618817941215639, -134.78613154296644438546, 470242.67608062474755570292, 0.99726007931332960599 }
    }, {
        { 1, 1 },{ 0, -2 },
        { 349898.53698544885264709592, -161.45195469949391053888, -161.44322493617917757547, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, -161.45195469952594180540, -161.44322493621118042029, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, -161.45195534985634822078, -161.44322558687019864010, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, -161.45245862756033261576, -161.44372886423073509832, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { 1, 1 },{ 2, -2 },
        { 349898.53698544885264709592, 161.45195469949391053888, 161.44322493617917757547, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, 161.45195469952594180540, 161.44322493621118042029, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, 161.45195534985634822078, 161.44322558687019864010, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, 161.45245862756033261576, 161.44372886423073509832, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { 1, -1 },{ 2, 0 },
        { 156899.56829134028521366417, 45.19676732164486310239, 45.18804022935886877121, 156883.63778222308610565960, 0.99969540695122460772 },
        { 156899.56827460310887545347, 45.19676732162573529195, 45.18804022933975517162, 156883.63778401838499121368, 0.99969540695121617002 },
        { 156899.56792193598812445998, 45.19676624219878391386, 45.18803915024158612823, 156883.63480984268244355917, 0.99969540696276681935 },
        { 156897.79947260793414898217, 45.19644594033049145310, 45.18771884804951355363, 156882.75281389255542308092, 0.99969541038810816325 }
    }, {
        { 1, -1 },{ 2, 2 },
        { 349898.53698544885264709592, 18.54804530050607525027, 18.55677506382081531910, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, 18.54804530047409016902, 18.55677506378881247429, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, 18.54804465014367664821, 18.55677441312981201804, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, 18.54754137243966738424, 18.55627113576925424354, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { 1, -1 },{ 0, 2 },
        { 349898.53698544885264709592, -18.54804530050607525027, -18.55677506382081531910, 349721.87727315956726670265, 0.99848548653821667109 },
        { 349898.53691851865733042359, -18.54804530047409016902, -18.55677506378881247429, 349721.87728093314217403531, 0.99848548652432367323 },
        { 349898.53644461580552160740, -18.54804465014367664821, -18.55677441312981201804, 349721.87595040485030040145, 0.99848548653585678103 },
        { 349894.27001174906035885215, -18.54754137243966738424, -18.55627113576928266525, 349720.85020991315832361579, 0.99848549544738174344 }
    }, {
        { 1, -1 },{ -2, 2 },
        { 470675.27496387914288789034, -45.18799532476352709409, -45.21418997011227247640, 470245.32092262554215267301, 0.99726004843767979136 },
        { 470675.27491378918057307601, -45.18799532474472613330, -45.21418997009345730476, 470245.32093124865787103772, 0.99726004841255500022 },
        { 470675.27385581465205177665, -45.18799424651773932737, -45.21418889088011638933, 470245.31203990196809172630, 0.99726004851631100401 },
        { 470669.97387331607751548290, -45.18767381182056652733, -45.21386845703356271997, 470242.67608062445651739836, 0.99726007931332938394 }
    }, {
        { 1, -1 },{ -2, 0 },
        { 351772.23553088010521605611, -71.69677339271241578444, -71.67058676207953737958, 351592.72256732499226927757, 0.99846922689201267342 },
        { 351772.23552330053644254804, -71.69677339270522509196, -71.67058676207236089795, 351592.72256738768192008138, 0.99846922689200567902 },
        { 351772.23547190619865432382, -71.69677274795323285161, -71.67058611830736936099, 351592.71064292214578017592, 0.99846922699591911243 },
        { 351771.13000385620398446918, -71.69688960494077889507, -71.67070297450322868826, 351594.87308142206165939569, 0.99846920815167106156 }
    }, {
        { 1, -1 },{ -2, -2 },
        { 351676.50043935602298006415, -108.36087395536006283692, -108.28232205125480902552, 351497.13544799503870308399, 0.99847006808788085763 },
        { 351676.50037266127765178680, -108.36087395536665667350, -108.28232205126143128382, 351497.13543194788508117199, 0.99847006821303718738 },
        { 351676.50038040406070649624, -108.36087459655152542837, -108.28232269539881826859, 351497.12354394845897331834, 0.99847006831660412018 },
        { 351675.39839278126601129770, -108.36075825022818719390, -108.28220634670809374711, 351499.26940269087208434939, 0.99847004976658149111 }
    }, {
        { 1, -1 },{ 0, -2 },
        { 156876.14940188667969778180, -134.82952991582811819171, -134.80335129773999369718, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, -134.82952991584656388113, -134.80335129775846780831, 156860.22615983051946386695, 0.99969549954768643918 },
        { 156876.14903264911845326424, -134.82953099167991695140, -134.80335237457535413341, 156860.22319510785746388137, 0.99969549955919845274 },
        { 156874.38594904550700448453, -134.82985169207830722371, -134.80367307400518939176, 156859.34477911840076558292, 0.99969550299904519353 }
    }, {
        { 1, -1 },{ 2, -2 },
        { 156876.14940188667969778180, 134.82952991582811819171, 134.80335129773999369718, 156860.22615479407249949872, 0.99969549952259184611 },
        { 156876.14925185780157335103, 134.82952991584656388113, 134.80335129775846780831, 156860.22615983051946386695, 0.99969549954768643918 },
        { 156876.14903264911845326424, 134.82953099167994537311, 134.80335237457538255512, 156860.22319510785746388137, 0.99969549955919856377 },
        { 156874.38594904550700448453, 134.82985169207830722371, 134.80367307400518939176, 156859.34477911840076558292, 0.99969550299904519353 }
    }, {
        // near pole
        {90, -45}, {90, -80},
        {3900195.49395913071930408478, -180.00000000000000000000, -180.00000000000000000000, 3662502.04478993499651551247, 0.81922976250863166481},
        {3900195.49395686946809291840, 180.00000000000000000000, 180.00000000000000000000, 3662502.06126507185399532318, 0.81922976508736944368},
        {3900195.49053949490189552307, 180.00000000000000000000, 180.00000000000000000000, 3662502.06126507185399532318, 0.81922976508736944368},
        {3900203.61653685756027698517, 180.00000000000000000000, 180.00000000000000000000, 3662507.91531699104234576225, 0.81923068138101373670}
    }, {
        // pole, same lon
        {90, -45}, {90, -90},
        {5017021.35133497882634401321, -180.00000000000000000000, -180.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.35133314039558172226, 180.00000000000000000000, 180.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.34791153203696012497, 180.00000000000000000000, 180.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017017.85355509258806705475, 180.00000000000000000000, 180.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172}
    }, {
        // pole
        {90, -45}, {0, -90},
        {5017021.35133497882634401321, -180.00000000000000000000, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.35133314039558172226, 180.00000000000000000000, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.34791153203696012497, 180.00000000000000000000, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017017.85355509258806705475, 180.00000000000000000000, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172}
    }, {
        // near pole
        {90, 45}, {90, 80},
        {3900195.49395913071930408478, 0.00000000000000000000, 0.00000000000000000000, 3662502.04478993499651551247, 0.81922976250863166481},
        {3900195.49395686946809291840, 0.00000000000000000000, 0.00000000000000000000, 3662502.06126507185399532318, 0.81922976508736944368},
        {3900195.49053949490189552307, 0.00000000000000000000, 0.00000000000000000000, 3662502.06126507185399532318, 0.81922976508736944368},
        {3900203.61653685756027698517, 0.00000000000000000076, 0.00000000000000000071, 3662507.91531699104234576225, 0.81923068138101373670}
    }, {
        // pole, same lon
        {90, 45}, {90, 90},
        {5017021.35133497882634401321, 0.00000000000000000000, 0.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.35133314039558172226, 0.00000000000000000000, 0.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.34791153203696012497, 0.00000000000000000000, 0.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017017.85355509258806705475, 0.00000000000000000000, 0.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172}
    }, {
        // pole
        {90, 45}, {0, 90},
        {5017021.35133497882634401321, -0.00000000000000000000, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.35133314039558172226, -0.00000000000000496992, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017021.34791153203696012497, 0.00000000000000000000, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172},
        {5017017.85355509258806705475, 0.00000000000000000000, -90.00000000000000000000, 4517590.87884893082082271576, 0.70710678118654746172}
    }, {
        // pole
        {0, -90}, {90, -45},
        {5017021.35133497882634401321, 90.00000000000000000000, 0.00000000000000000000, 4517590.87884893082082271576, 0.70778496454077310940},
        {5017021.35133314039558172226, 90.00000000000000000000, 0.00000000000000496992, 4517590.87884893082082271576, 0.70778494508412304054},
        {5017021.34791153203696012497, 90.00000000000000000000, 0.00000000000000000000, 4517590.87884893082082271576, 0.70778494508412304054},
        {5017017.85355509258806705475, 90.00000000000000000000, 0.00000000000000000000, 4517590.87884893082082271576, 0.70777827352623101653}
    }, {
        // pole
        {0, 90}, {90, 45},
        {5017021.35133497882634401321, 90.00000000000000000000, -180.00000000000000000000, 4517590.87884893082082271576, 0.70778496454077310940},
        {5017021.35133314039558172226, 90.00000000000000000000, 180.00000000000000000000, 4517590.87884893082082271576, 0.70778494508412304054},
        {5017021.34791153203696012497, 90.00000000000001421085, 180.00000000000000000000, 4517590.87884893082082271576, 0.70778494508412304054},
        {5017017.85355509258806705475, 90.00000000000000000000, 180.00000000000000000000, 4517590.87884893082082271576, 0.70777827352623101653}
    }, {
        {90, -45}, {90, 45},
        {9969888.75595548748970031738, 0.00000000000000000000, 0.00000000000000000000, 6361290.27546359039843082428, 0.00431189731318673553},
        {9969888.75595730356872081757, 0.00000000000000000000, 0.00000000000000000000, 6361290.36011654324829578400, 0.00431188406305459994},
        {9969888.73915978334844112396, 0.00000000000000000000, 0.00000000000000000000, 6361290.36011654324829578400, 0.00431188406305459994},
        {9969881.64984572120010852814, 0.00000000000000001848, 0.00000000000000001176, 6361314.54906367603689432144, 0.00430809793699618842}
    }, {
        {90, 45}, {90, -45},
        {9969888.75595548748970031738, -180.00000000000000000000, -180.00000000000000000000, 6361290.27546359039843082428, 0.00431189731318673553},
        {9969888.75595730356872081757, 180.00000000000000000000, 180.00000000000000000000, 6361290.36011654324829578400, 0.00431188406305459994},
        {9969888.73915978334844112396, 180.00000000000000000000, 180.00000000000000000000, 6361290.36011654324829578400, 0.00431188406305459994},
        {9969881.64984572120010852814, 180.00000000000000000000, 180.00000000000000000000, 6361314.54906367603689432144, 0.00430809793699618842}
    }, {
        // pole, same lon
        {90, 0}, {90, 90},
        {10001965.72931272350251674652, 0.00000000000000000000, 0.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.72931179217994213104, 0.00000000000000000000, 0.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.71749142371118068695, 0.00000000000000000000, 0.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001958.67847795411944389343, 0.00000000000000000000, 0.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000}
    }, {
        // pole
        {90, 0}, {0, 90},
        {10001965.72931272350251674652, -0.00000000000000000000, -90.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.72931179217994213104, -0.00000000000000352016, -90.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.71749142371118068695, 0.00000000000000000000, -90.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001958.67847795411944389343, 0.00000000000000000000, -90.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000}
    }, {
        // pole, different lon
        {0, 0}, {1, 90},
        {10001965.72931272350251674652, 0.00000000000000000000, 1.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.72931179217994213104, 0.00000000000000006144, 1.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.71749142371118068695, 0.00000000000000000000, 0.99999999999998867573, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001958.67847795411944389343, 0.00000000000000000000, 0.99999999999998867573, 6378136.99999999906867742538, 0.00000000000000000000}
    }, {
        // pole, different lon
        {0, 0}, {-1, 90},
        {10001965.72931272350251674652, -0.00000000000000000000, -1.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.72931179217994213104, -0.00000000000000006144, -1.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.71749142371118068695, 0.00000000000000000000, -1.00000000000001421085, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001958.67847795411944389343, 0.00000000000000000000, -0.99999999999998867573, 6378136.99999999906867742538, 0.00000000000000000000}
    }, {
        // pole, different lon
        {0, 0}, {1, -90},
        {10001965.72931272350251674652, -180.00000000000000000000, 179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.72931179217994213104, 180.00000000000000000000, 179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.71749142371118068695, 180.00000000000000000000, 179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001958.67847795411944389343, 180.00000000000000000000, 179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000}
    }, {
        // pole, different lon
        {0, 0}, {-1, -90},
        {10001965.72931272350251674652, -180.00000000000000000000, -179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.72931179217994213104, -180.00000000000000000000, -179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001965.71749142371118068695, 180.00000000000000000000, -179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000},
        {10001958.67847795411944389343, 180.00000000000000000000, -179.00000000000000000000, 6378136.99999999906867742538, 0.00000000000000000000}
    }
    /*, {
        // antipodal
        {-90, -45}, {90, 45},
        {20003931.45862544700503349304, -180.00000000000000000000, 0.00000000000000000000, 33675.52452803074265830219, -0.99472900658949181540},
        {19987083.06974205747246742249, 90.00000000000000000000, 90.00000000000000000000, 00000.00000000078240703623, -1.00000000000000000000},
        {00000000.00000000000000000000, 0.00000000000000000000, 0.00000000000000000000, 000000.00000000000000000000, 1.00000000000000000000},
        {20020712.84987257421016693115, 0.00000000000000000000, 0.00000000000000000000, 6361314.54906367603689432144, 0.00430809793699618842}
    },{
        // antipodal
        {-90, 45}, {90, -45},
        {20003931.45862544700503349304, 0.00000000000000000000, -180.00000000000000000000, 33675.52452803074265830219, -0.99472900658949181540},
        {19987083.06974205747246742249, 90.00000000000000000000, 90.00000000000000000000, -000000.00000000078240703623, -1.00000000000000000000},
        {00000000.00000000000000000000, 0.00000000000000000000, 0.00000000000000000000, 000000.00000000000000000000, 1.00000000000000000000},
        {20020712.84987257421016693115, 180.00000000000000000000, 0.00000000000000000000, 33590.79639541531651047990, -1.00525773151080621837}
    }
    // pi  = 3.1415926535897931
    // r2d = 57.295779513082323
    // commented out because of inconsistencies
    /*, {
        {0.26387320519564444 * 57.295779513082323, 0.84008767903244352 * 57.295779513082323},
        {0.26387320679785603 * 57.295779513082323, 0.84008767099694759 * 57.295779513082323},
        {0.051647212499179529, 172.39749230932208, 172.39749237768564, 0.051647212499179536, 1.0000000000000000},
        {0.051647209427812361, 172.39750897806516, 172.39750904523200, 0.051638407375539180, 0.99999999999999734},
        // TODO: This is wrong
        {0.00000000000000000, 0.00000000000000000, 0.00000000000000000, 0.00000000000000000, 1.0000000000000000},
        {0.00000000000000000, 0.00000000000000000, 0.00000000000000000, -0.051193490763866173, 1.0000000000000000}
    }, {
        {0.26387320519564444 * 57.295779513082323, 0.84008767903244352 * 57.295779513082323},
        {(0.26387320679785603 - 3.1415926535897931) * 57.295779513082323, -0.84008767099694759 * 57.295779513082323},
        {20003931.407432225, -1.3045327945402774e-05, -179.99998695467218, 30010.522036068141, -0.99475954491334551},
        {20003669.536724485, -7.5912543895164619, -172.40874567755344, 29487.136123182696, -0.99480541226332386},
        // TODO: This is wrong
        {0.00000000000000000, 0.00000000000000000, 0.00000000000000000, 0.00000000000000000, 1.0000000000000000},
        {20018879.436128609, 0.00000000000000000, 180.00000000000000, 29935.015234617360, -0.99477272997811228}
    }*/
};

size_t const expected_size = sizeof(expected) / sizeof(expected_results);

#endif // BOOST_GEOMETRY_TEST_INVERSE_CASES_HPP
