/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2021 the original author or authors.
 */
package org.assertj.core.internal.floats;

import static java.lang.String.format;
import static org.assertj.core.api.Assertions.assertThatExceptionOfType;
import static org.assertj.core.test.TestData.someInfo;

import org.assertj.core.internal.FloatsBaseTest;
import org.junit.jupiter.api.Test;

/**
 * Tests for <code>{@link Floats#assertIsNotNegative(AssertionInfo, Float))}</code>.
 *
 * @author Nicolas François
 */
class Floats_assertIsNotNegative_Test extends FloatsBaseTest {

  @Test
  void should_succeed_since_actual_is_not_negative() {
    floats.assertIsNotNegative(someInfo(), 6f);
  }

  @Test
  void should_succeed_since_actual_is_zero() {
    floats.assertIsNotNegative(someInfo(), 0f);
  }

  @Test
  void should_fail_since_actual_is_negative() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> floats.assertIsNotNegative(someInfo(), -6f))
                                                   .withMessage(format("%nExpecting:%n  -6.0f%nto be greater than or equal to:%n  0.0f%n"));
  }

  @Test
  void should_succeed_since_actual_is_not_negative_according_to_custom_comparison_strategy() {
    floatsWithAbsValueComparisonStrategy.assertIsNotNegative(someInfo(), -1f);
  }

  @Test
  void should_succeed_since_actual_positive_is_not_negative_according_to_custom_comparison_strategy() {
    floatsWithAbsValueComparisonStrategy.assertIsNotNegative(someInfo(), 1f);
  }

}
