/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2021 the original author or authors.
 */
package org.assertj.core.internal.files;

import static java.nio.file.Files.readAllBytes;
import static org.assertj.core.api.Assertions.catchThrowableOfType;
import static org.assertj.core.api.BDDAssertions.then;
import static org.assertj.core.error.ShouldBeFile.shouldBeFile;
import static org.assertj.core.error.ShouldHaveBinaryContent.shouldHaveBinaryContent;
import static org.assertj.core.internal.BinaryDiffResult.noDiff;
import static org.assertj.core.test.TestData.someInfo;
import static org.assertj.core.util.AssertionsUtil.expectAssertionError;
import static org.assertj.core.util.FailureMessages.actualIsNull;
import static org.mockito.BDDMockito.given;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;

import org.assertj.core.internal.BinaryDiffResult;
import org.assertj.core.internal.FilesBaseTest;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

class Files_assertHasSameBinaryContentAs_Test extends FilesBaseTest {

  private static File actual;
  private static File expected;
  private static byte[] expectedBytes;

  @BeforeAll
  static void setUpOnce() throws IOException {
    // Does not matter if the values differ, the actual comparison is mocked in this test
    actual = new File("src/test/resources/actual_file.txt");
    expected = new File("src/test/resources/expected_file.txt");
    expectedBytes = readAllBytes(expected.toPath());
  }

  @Test
  void should_pass_if_file_has_expected_binary_content() throws IOException {
    // GIVEN
    given(binaryDiff.diff(actual, expectedBytes)).willReturn(noDiff());
    // WHEN/THEN
    files.assertSameBinaryContentAs(someInfo(), actual, expected);
  }

  @Test
  void should_throw_error_if_expected_is_null() {
    // GIVEN
    File nullExpected = null;
    // WHEN
    NullPointerException npe = catchThrowableOfType(() -> files.assertSameBinaryContentAs(someInfo(), actual, nullExpected),
                                                    NullPointerException.class);
    // THEN
    then(npe).hasMessage("The file to compare to should not be null");
  }

  @Test
  void should_fail_if_actual_is_null() {
    // GIVEN
    File file = null;
    // WHEN
    AssertionError error = expectAssertionError(() -> files.assertSameBinaryContentAs(someInfo(), file, expected));
    // THEN
    then(error).hasMessage(actualIsNull());
  }

  @Test
  void should_fail_if_actual_is_not_a_file() {
    // GIVEN
    File notAFile = new File("xyz");
    // WHEN
    AssertionError error = expectAssertionError(() -> files.assertSameBinaryContentAs(someInfo(), notAFile, expected));
    // THEN
    then(error).hasMessage(shouldBeFile(notAFile).create());
  }

  @Test
  void should_fail_if_expected_is_not_a_file() {
    // GIVEN
    File notAFile = new File("xyz");
    // WHEN
    IllegalArgumentException iae = catchThrowableOfType(() -> files.assertSameBinaryContentAs(someInfo(), actual, notAFile),
                                                        IllegalArgumentException.class);
    // THEN
    then(iae).hasMessage("Expected file:<'%s'> should be an existing file", notAFile);
  }

  @Test
  void should_throw_error_wrapping_caught_IOException() throws IOException {
    // GIVEN
    IOException cause = new IOException();
    given(binaryDiff.diff(actual, expectedBytes)).willThrow(cause);
    // WHEN
    UncheckedIOException uioe = catchThrowableOfType(() -> files.assertSameBinaryContentAs(someInfo(), actual, expected),
                                                     UncheckedIOException.class);
    // THEN
    then(uioe).hasCause(cause);
  }

  @Test
  void should_fail_if_file_does_not_have_expected_binary_content() throws IOException {
    // GIVEN
    BinaryDiffResult diff = new BinaryDiffResult(15, (byte) 0xCA, (byte) 0xFE);
    when(binaryDiff.diff(actual, expectedBytes)).thenReturn(diff);
    // WHEN
    expectAssertionError(() -> files.assertSameBinaryContentAs(someInfo(), actual, expected));
    // THEN
    verify(failures).failure(someInfo(), shouldHaveBinaryContent(actual, diff));
  }
}
