/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2021 the original author or authors.
 */
package org.assertj.core.error;

import org.assertj.core.api.Condition;

/**
 * Creates an error message indicating that an assertion that verifies a map contains a value.
 * 
 * @author Nicolas François
 */
public class ShouldContainValue extends BasicErrorMessageFactory {

  /**
   * Creates a new <code>{@link ShouldContainValue}</code>.
   * @param actual the actual value in the failed assertion.
   * @param value the value not found
   * @return the created {@code ErrorMessageFactory}.
   */
  public static ErrorMessageFactory shouldContainValue(Object actual, Object value) {
    return new ShouldContainValue(actual, value);
  }

  /**
   * Creates a new <code>{@link ShouldContainValue}</code>.
   *
   * @param actual the actual map in the failed assertion.
   * @param valueCondition value condition.
   * @return the created {@code ErrorMessageFactory}.
   */
  public static ErrorMessageFactory shouldContainValue(Object actual, Condition<?> valueCondition) {
    return new ShouldContainValue(actual, valueCondition);
  }

  private ShouldContainValue(Object actual, Object value) {
    super("%nExpecting:%n  %s%nto contain value:%n  %s", actual, value);
  }

  private ShouldContainValue(Object actual, Condition<?> valueCondition) {
    super("%nExpecting:%n  %s%nto contain a value satisfying:%n  %s", actual, valueCondition);
  }
}
