# (c) 2020-2022, NetApp, Inc
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

''' Unit Tests NetApp ONTAP REST APIs Ansible module: na_ontap_rest_info '''

from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import pytest
import sys

from ansible_collections.netapp.ontap.tests.unit.compat import unittest
from ansible_collections.netapp.ontap.tests.unit.compat.mock import patch
from ansible_collections.netapp.ontap.tests.unit.plugins.module_utils.ansible_mocks import call_main, create_module, expect_and_capture_ansible_exception, \
    patch_ansible, create_and_apply
from ansible_collections.netapp.ontap.tests.unit.framework.mock_rest_and_zapi_requests import \
    patch_request_and_invoke, register_responses
from ansible_collections.netapp.ontap.tests.unit.framework.rest_factory import rest_responses

from ansible_collections.netapp.ontap.plugins.modules.na_ontap_rest_info \
    import NetAppONTAPGatherInfo as ontap_rest_info_module, main as my_main


if sys.version_info < (2, 7):
    pytestmark = pytest.mark.skip('Skipping Unit Tests on 2.6 as requests is not available')


# REST API canned responses when mocking send_request
SRR = rest_responses({
    # common responses
    'validate_ontap_version_pass': (200, {'version': 'ontap_version'}, None),
    'validate_ontap_version_fail': (200, None, 'API not found error'),
    'error_invalid_api': (500, None, {'code': 3, 'message': 'Invalid API'}),
    'error_user_is_not_authorized': (500, None, {'code': 6, 'message': 'user is not authorized'}),
    'error_no_processing': (500, None, {'code': 123, 'message': 'error reported as is'}),
    'error_no_aggr_recommendation': (500, None, {'code': 19726344, 'message': 'No recommendation can be made for this cluster'}),
    'get_subset_info': (200,
                        {'_links': {'self': {'href': 'dummy_href'}},
                         'num_records': 3,
                         'records': [{'name': 'dummy_vol1'},
                                     {'name': 'dummy_vol2'},
                                     {'name': 'dummy_vol3'}],
                         'version': 'ontap_version'}, None),
    'get_subset_info_with_next': (200,
                                  {'_links': {'self': {'href': 'dummy_href'},
                                              'next': {'href': '/api/next_record_api'}},
                                   'num_records': 3,
                                   'records': [{'name': 'dummy_vol1'},
                                               {'name': 'dummy_vol2'},
                                               {'name': 'dummy_vol3'}],
                                   'version': 'ontap_version'}, None),
    'get_next_record': (200,
                        {'_links': {'self': {'href': 'dummy_href'}},
                         'num_records': 2,
                         'records': [{'name': 'dummy_vol1'},
                                     {'name': 'dummy_vol2'}],
                         'version': 'ontap_version'}, None),
    'metrocluster_post': (200,
                          {'job': {
                              'uuid': 'fde79888-692a-11ea-80c2-005056b39fe7',
                              '_links': {
                                  'self': {
                                      'href': '/api/cluster/jobs/fde79888-692a-11ea-80c2-005056b39fe7'}}
                          }},
                          None),
    'metrocluster_return': (200,
                            {"_links": {
                                "self": {
                                    "href": "/api/cluster/metrocluster/diagnostics"
                                }
                            }, "aggregate": {
                                "state": "ok",
                                "summary": {
                                    "message": ""
                                }, "timestamp": "2020-07-22T16:42:51-07:00"
                            }}, None),
    'job': (200,
            {
                "uuid": "cca3d070-58c6-11ea-8c0c-005056826c14",
                "description": "POST /api/cluster/metrocluster",
                "state": "success",
                "message": "There are not enough disks in Pool1.",
                "code": 2432836,
                "start_time": "2020-02-26T10:35:44-08:00",
                "end_time": "2020-02-26T10:47:38-08:00",
                "_links": {
                    "self": {
                        "href": "/api/cluster/jobs/cca3d070-58c6-11ea-8c0c-005056826c14"
                    }
                }
            }, None),
    'get_private_cli_subset_info': (200,
                                    {
                                        'records': [
                                            {'node': 'node1', 'check_type': 'type'},
                                            {'node': 'node1', 'check_type': 'type'},
                                            {'node': 'node1', 'check_type': 'type'}],
                                        "num_records": 3}, None),
    'get_private_cli_vserver_security_file_directory_info': (
        200,
        {
            'records': [
                {'acls': ['junk', 'junk', 'DACL - ACEs', 'AT-user-0x123']},
                {'node': 'node1', 'check_type': 'type'},
                {'node': 'node1', 'check_type': 'type'}],
            "num_records": 3}, None)

})

ALL_SUBSETS = ['application/applications',
               'application/templates',
               'cloud/targets',
               'cluster',
               'cluster/chassis',
               'cluster/jobs',
               'cluster/licensing/licenses',
               'cluster/metrocluster',
               'cluster/metrocluster/diagnostics',
               'cluster/metrocluster/nodes',
               'cluster/metrics',
               'cluster/nodes',
               'cluster/ntp/servers',
               'cluster/peers',
               'cluster/schedules',
               'cluster/software',
               'cluster/software/download',
               'cluster/software/history',
               'cluster/software/packages',
               'name-services/dns',
               'name-services/ldap',
               'name-services/name-mappings',
               'name-services/nis',
               'network/ethernet/broadcast-domains',
               'network/ethernet/ports',
               'network/fc/logins',
               'network/fc/wwpn-aliases',
               'network/ip/interfaces',
               'network/ip/routes',
               'network/ip/service-policies',
               'network/ipspaces',
               'network/ethernet/switches',
               'private/support/alerts',
               'protocols/cifs/home-directory/search-paths',
               'protocols/cifs/services',
               'protocols/cifs/shares',
               'protocols/nfs/export-policies',
               'protocols/nfs/kerberos/realms',
               'protocols/nfs/services',
               'protocols/nvme/interfaces',
               'protocols/nvme/services',
               'protocols/nvme/subsystems',
               'protocols/san/fcp/services',
               'protocols/san/igroups',
               'protocols/san/iscsi/credentials',
               'protocols/san/iscsi/services',
               'protocols/san/lun-maps',
               'protocols/vscan/server-status',
               'protocols/vscan',
               'security/accounts',
               'security/audit/destinations',
               'security/roles',
               'snapmirror/policies',
               'snapmirror/relationships',
               'storage/aggregates',
               'storage/bridges',
               'storage/disks',
               'storage/flexcache/flexcaches',
               'storage/flexcache/origins',
               'storage/luns',
               'storage/namespaces',
               'storage/ports',
               'storage/qos/policies',
               'storage/qtrees',
               'storage/quota/reports',
               'storage/quota/rules',
               'storage/shelves',
               'storage/snapshot-policies',
               'storage/volumes',
               'storage/volume-efficiency-policies',
               'support/autosupport',
               'support/autosupport/check',
               'support/autosupport/messages',
               'support/ems',
               'support/ems/destinations',
               'support/ems/events',
               'support/ems/filters',
               'svm/peers',
               'svm/peer-permissions',
               'svm/svms']


# Super Important, Metrocluster doesn't call get_subset_info and has 3 api calls instead of 1!!!!
# The metrocluster calls need to be in the correct place. The Module return the keys in a sorted list.
ALL_RESPONSES = [
    ('GET', 'cluster', SRR['validate_ontap_version_pass']),
    ('GET', 'application/applications', SRR['get_subset_info']),
    ('GET', 'application/templates', SRR['get_subset_info']),
    ('GET', 'cloud/targets', SRR['get_subset_info']),
    ('GET', 'cluster', SRR['get_subset_info']),
    ('GET', 'cluster/chassis', SRR['get_subset_info']),
    ('GET', 'cluster/jobs', SRR['get_subset_info']),
    ('GET', 'cluster/licensing/licenses', SRR['get_subset_info']),
    ('GET', 'cluster/metrics', SRR['get_subset_info']),
    ('GET', 'cluster/metrocluster', SRR['get_subset_info']),
    # MCC DIAGs
    ('POST', 'cluster/metrocluster/diagnostics', SRR['metrocluster_post']),
    ('GET', 'cluster/jobs/fde79888-692a-11ea-80c2-005056b39fe7', SRR['job']),
    ('GET', 'cluster/metrocluster/diagnostics', SRR['metrocluster_return']),
    # Back to normal
    ('GET', 'cluster/metrocluster/nodes', SRR['get_subset_info']),
    ('GET', 'cluster/nodes', SRR['get_subset_info']),
    ('GET', 'cluster/ntp/servers', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', '*', SRR['get_subset_info']),
    ('GET', 'support/ems/filters', SRR['get_subset_info']),
    ('GET', 'svm/peer-permissions', SRR['get_subset_info']),
    ('GET', 'svm/peers', SRR['get_subset_info']),
    ('GET', 'svm/svms', SRR['get_private_cli_subset_info']),
    # ('GET', 'svm/svms', SRR['get_private_cli_subset_info']),
]


def set_default_args():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False
    })


def set_args_run_ontap_version_check():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 1024,
        'gather_subset': ['volume_info']
    })


def set_args_run_metrocluster_diag():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 1024,
        'gather_subset': ['cluster/metrocluster/diagnostics']
    })


def set_args_run_ontap_gather_facts_for_vserver_info():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 1024,
        'gather_subset': ['vserver_info']
    })


def set_args_run_ontap_gather_facts_for_volume_info():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 1024,
        'gather_subset': ['volume_info']
    })


def set_args_run_ontap_gather_facts_for_all_subsets():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 1024,
        'gather_subset': ['all']
    })


def set_args_run_ontap_gather_facts_for_all_subsets_with_fields_section_pass():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 1024,
        'fields': '*',
        'gather_subset': ['all']
    })


def set_args_run_ontap_gather_facts_for_all_subsets_with_fields_section_fail():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 1024,
        'fields': ['uuid', 'name', 'node'],
        'gather_subset': ['all']
    })


def set_args_run_ontap_gather_facts_for_aggregate_info_with_fields_section_pass():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'fields': ['uuid', 'name', 'node'],
        'validate_certs': False,
        'max_records': 1024,
        'gather_subset': ['aggregate_info']
    })


def set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass():
    return dict({
        'hostname': 'hostname',
        'username': 'username',
        'password': 'password',
        'https': True,
        'validate_certs': False,
        'max_records': 3,
        'gather_subset': ['volume_info']
    })


def test_run_ontap_version_check_for_9_6_pass():
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['get_subset_info']),
    ])
    assert not create_and_apply(ontap_rest_info_module, set_args_run_ontap_version_check())['changed']


def test_run_ontap_version_check_for_10_2_pass():
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['get_subset_info']),
    ])
    assert not create_and_apply(ontap_rest_info_module, set_args_run_ontap_version_check())['changed']


def test_run_ontap_version_check_for_9_2_fail():
    ''' Test for Checking the ONTAP version '''
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_fail']),
    ])
    assert create_and_apply(ontap_rest_info_module, set_args_run_ontap_version_check(), fail=True)['msg'] == SRR['validate_ontap_version_fail'][2]


# metrocluster/diagnostics doesn't call get_subset_info and has 3 api calls instead of 1
def test_run_metrocluster_pass():
    gather_subset = ['cluster/metrocluster/diagnostics']
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('POST', 'cluster/metrocluster/diagnostics', SRR['metrocluster_post']),
        ('GET', 'cluster/jobs/fde79888-692a-11ea-80c2-005056b39fe7', SRR['job']),
        ('GET', 'cluster/metrocluster/diagnostics', SRR['metrocluster_return']),
    ])
    assert set(create_and_apply(ontap_rest_info_module, set_args_run_metrocluster_diag())['ontap_info']) == set(gather_subset)


def test_run_ontap_gather_facts_for_vserver_info_pass():
    gather_subset = ['svm/svms']
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'svm/svms', SRR['get_subset_info']),
    ])
    assert set(create_and_apply(ontap_rest_info_module, set_args_run_ontap_gather_facts_for_vserver_info())['ontap_info']) == set(gather_subset)


def test_run_ontap_gather_facts_for_volume_info_pass():
    gather_subset = ['storage/volumes']
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['get_subset_info']),
    ])
    assert set(create_and_apply(ontap_rest_info_module, set_args_run_ontap_gather_facts_for_volume_info())['ontap_info']) == set(gather_subset)


def test_run_ontap_gather_facts_for_all_subsets_pass():
    gather_subset = ALL_SUBSETS
    register_responses(ALL_RESPONSES)
    assert set(create_and_apply(ontap_rest_info_module, set_args_run_ontap_gather_facts_for_all_subsets())['ontap_info']) == set(gather_subset)


def test_run_ontap_gather_facts_for_all_subsets_with_fields_section_pass():
    gather_subset = ALL_SUBSETS
    register_responses(ALL_RESPONSES)
    assert set(create_and_apply(ontap_rest_info_module, set_args_run_ontap_gather_facts_for_all_subsets_with_fields_section_pass()
                                )['ontap_info']) == set(gather_subset)


def test_run_ontap_gather_facts_for_all_subsets_with_fields_section_fail():
    error_message = "Error: fields: %s, only one subset will be allowed." \
                    % set_args_run_ontap_gather_facts_for_aggregate_info_with_fields_section_pass()['fields']
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
    ])
    assert create_and_apply(ontap_rest_info_module, set_args_run_ontap_gather_facts_for_all_subsets_with_fields_section_fail(), fail=True
                            )['msg'] == error_message


def test_run_ontap_gather_facts_for_aggregate_info_pass_with_fields_section_pass():
    gather_subset = ['storage/aggregates']
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/aggregates', SRR['get_subset_info']),
    ])
    assert set(create_and_apply(ontap_rest_info_module, set_args_run_ontap_gather_facts_for_aggregate_info_with_fields_section_pass()
                                )['ontap_info']) == set(gather_subset)


def test_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass():
    total_records = 5
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['get_subset_info_with_next']),
        ('GET', '/next_record_api', SRR['get_next_record']),
    ])
    assert create_and_apply(ontap_rest_info_module, set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
                            )['ontap_info']['storage/volumes']['num_records'] == total_records


def test_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass_python_keys():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['use_python_keys'] = True
    args['state'] = 'info'
    total_records = 5
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['get_subset_info_with_next']),
        ('GET', '/next_record_api', SRR['get_next_record']),
    ])
    assert create_and_apply(ontap_rest_info_module, args)['ontap_info']['storage_volumes']['num_records'] == total_records


def test_get_all_records_for_volume_info_with_parameters():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['use_python_keys'] = True
    args['parameters'] = {'fields': '*'}
    total_records = 5
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['get_subset_info_with_next']),
        ('GET', '/next_record_api', SRR['get_next_record']),
    ])
    assert create_and_apply(ontap_rest_info_module, args)['ontap_info']['storage_volumes']['num_records'] == total_records


def test_negative_error_on_get_next():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['use_python_keys'] = True
    args['parameters'] = {'fields': '*'}
    total_records = 5
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['get_subset_info_with_next']),
        ('GET', '/next_record_api', SRR['generic_error']),
    ])
    assert create_and_apply(ontap_rest_info_module, args, fail=True)['msg'] == 'Expected error'


def test_negative_bad_api():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['use_python_keys'] = True
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['error_invalid_api']),
    ])
    assert create_and_apply(ontap_rest_info_module, args)['ontap_info']['storage_volumes'] == 'Invalid API'


def test_negative_error_no_aggr_recommendation():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['use_python_keys'] = True
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['error_no_aggr_recommendation']),
    ])
    assert create_and_apply(ontap_rest_info_module, args)['ontap_info']['storage_volumes'] == 'No recommendation can be made for this cluster'


def test_negative_error_not_authorized():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['use_python_keys'] = True
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['error_user_is_not_authorized']),
    ])
    assert 'user is not authorized to make' in create_and_apply(ontap_rest_info_module, args, fail=True)['msg']


def test_negative_error_no_processing():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['use_python_keys'] = True
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/volumes', SRR['error_no_processing']),
    ])
    assert create_and_apply(ontap_rest_info_module, args, fail=True)['msg']['message'] == 'error reported as is'


def test_strip_dacls():
    record = {}
    response = {
        'records': [record]
    }
    assert ontap_rest_info_module.strip_dacls(response) is None
    record['acls'] = []
    assert ontap_rest_info_module.strip_dacls(response) is None
    record['acls'] = ['junk', 'junk', 'DACL - ACEs']
    assert ontap_rest_info_module.strip_dacls(response) == []
    record['acls'] = ['junk', 'junk', 'DACL - ACEs', 'AT-user-0x123']
    assert ontap_rest_info_module.strip_dacls(response) == [{'access_type': 'AT', 'user_or_group': 'user'}]
    record['acls'] = ['junk', 'junk', 'DACL - ACEs', 'AT-user-0x123', 'AT2-group-0xABC']
    assert ontap_rest_info_module.strip_dacls(response) == [{'access_type': 'AT', 'user_or_group': 'user'}, {'access_type': 'AT2', 'user_or_group': 'group'}]


def test_private_cli_vserver_security_file_directory():
    args = set_args_get_all_records_for_volume_info_to_check_next_api_call_functionality_pass()
    args['gather_subset'] = 'private/cli/vserver/security/file-directory'
    args['use_python_keys'] = True
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'private/cli/vserver/security/file-directory?fields=acls', SRR['get_private_cli_vserver_security_file_directory_info']),
    ])
    assert create_and_apply(ontap_rest_info_module, args)['ontap_info'] == \
        {'private_cli_vserver_security_file-directory': [{'access_type': 'AT', 'user_or_group': 'user'}]}


def test_get_ontap_subset_info_all_with_field():
    register_responses([
        ('GET', 'some/api', SRR['get_subset_info']),
    ])
    my_obj = create_module(ontap_rest_info_module, set_default_args())
    subset_info = {'subset': {'api_call': 'some/api'}}
    assert my_obj.get_ontap_subset_info_all('subset', 'fields', subset_info)['num_records'] == 3


def test_negative_get_ontap_subset_info_all_bad_subset():
    my_obj = create_module(ontap_rest_info_module, set_default_args())
    msg = 'Specified subset bad_subset is not found, supported subsets are []'
    assert expect_and_capture_ansible_exception(my_obj.get_ontap_subset_info_all, 'fail', 'bad_subset', None, {})['msg'] == msg


def test_demo_subset():
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'cluster/software', SRR['get_subset_info']),
        ('GET', 'svm/svms', SRR['get_subset_info']),
        ('GET', 'cluster/nodes', SRR['get_subset_info']),
    ])
    assert 'cluster/nodes' in call_main(my_main, set_default_args(), {'gather_subset': 'demo'})['ontap_info']


def test_subset_with_default_fields():
    register_responses([
        ('GET', 'cluster', SRR['validate_ontap_version_pass']),
        ('GET', 'storage/aggregates', SRR['get_subset_info']),
    ])
    assert 'storage/aggregates' in create_and_apply(ontap_rest_info_module, set_default_args(), {'gather_subset': 'aggr_efficiency_info'})['ontap_info']


def test_negative_error_on_post():
    register_responses([
        ('POST', 'api', SRR['generic_error']),
    ])
    assert create_module(ontap_rest_info_module, set_default_args()).run_post({'api_call': 'api'}) is None


@patch('time.sleep')
def test_negative_error_on_wait_after_post(sleep_mock):
    register_responses([
        ('POST', 'api', SRR['metrocluster_post']),
        ('GET', 'cluster/jobs/fde79888-692a-11ea-80c2-005056b39fe7', SRR['generic_error']),
        ('GET', 'cluster/jobs/fde79888-692a-11ea-80c2-005056b39fe7', SRR['generic_error']),     # retries
        ('GET', 'cluster/jobs/fde79888-692a-11ea-80c2-005056b39fe7', SRR['generic_error']),
        ('GET', 'cluster/jobs/fde79888-692a-11ea-80c2-005056b39fe7', SRR['generic_error']),
    ])
    my_obj = create_module(ontap_rest_info_module, set_default_args())
    assert expect_and_capture_ansible_exception(my_obj.run_post, 'fail', {'api_call': 'api'})['msg'] == 'Expected error'
