#include "config.h"

#include "aboot-utils.h"
#include <ctype.h>
#include <stddef.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/wait.h>
#if defined(__linux__)
#include <linux/fs.h>
#endif
#include <time.h>

void verbose(const char *format, ...)
{
	if (use_verbose) {
		va_list args;
		va_start(args, format);
		vprintf(format, args);
		va_end(args);
	}
}

void error(const char *format, ...)
{
	va_list args;

	va_start(args, format);
	vfprintf(stderr, format, args);
	va_end(args);
}

void fatal(const char *format, ...)
{
	va_list args;

	va_start(args, format);
	vfprintf(stderr, format, args);
	va_end(args);

	exit(1);
}

void oom()
{
	fatal("Out of memory");
}

void *xmalloc(size_t size)
{
	void *res = malloc(size);
	if (res == NULL)
		oom();
	return res;
}

void *xcalloc(size_t nmemb, size_t size)
{
	void *res = calloc(nmemb, size);
	if (res == NULL)
		oom();
	return res;
}

void *xrealloc(void *ptr, size_t size)
{
	void *res = realloc(ptr, size);
	if (res == NULL)
		oom();
	return res;
}

char *xasprintf(const char *fmt, ...)
{
	va_list ap;
	char *result;

	va_start(ap, fmt);
	int ret = vasprintf(&result, fmt, ap);
	va_end(ap);

	if (ret < 0) {
		oom();
	}

	return result;
}

bool str_has_prefix(const char *str, const char *prefix)
{
	return strncmp(str, prefix, strlen(prefix)) == 0;
}

char *join_path(const char *a, const char *b)
{
	if (a == NULL) {
		return xstrdup(b);
	}

	if (b == NULL) {
		return xstrdup(a);
	}

	size_t a_len = strlen(a);
	size_t b_len = strlen(b);

	if (a_len == 0) {
		return xstrdup(b);
	}

	if (b_len == 0) {
		return xstrdup(a);
	}

	/* Skip trailing slashes in a (except leading) */
	while (a_len > 1 && a[a_len - 1] == '/') {
		a_len--;
	}

	/* Skip initial slashes in b */
	while (b_len > 0 && b[0] == '/') {
		b_len--;
		b++;
	}

	char *res = xmalloc(a_len + 1 + b_len + 1);
	char *d = res;
	memcpy(d, a, a_len);
	d += a_len;
	if (b_len > 0 && a[a_len - 1] != '/') {
		memcpy(d, "/", 1);
		d += 1;
	}
	memcpy(d, b, b_len);
	d += b_len;
	*d = 0;

	return res;
}

char *join_3_path(const char *a, const char *b, const char *c)
{
	autofree char *ab = join_path(a, b);
	return join_path(ab, c);
}

char *xbasename(const char *path)
{
	size_t len = strlen(path);

	/* Skip any trailing slashes (except the initial one for absolute paths) */
	while (len > 1 && path[len - 1] == '/') {
		len--;
	}
	size_t end = len;

	/* Skip last path element */
	while (len > 0 && path[len - 1] != '/') {
		len--;
	}

	if (end == len) {
		/* No basename, this only happens if path is "/" or "", we return path in this case)*/
		return xstrndup(path, end);
	}
	return xstrndup(path + len, end - len);
}

char *xdirname(const char *path)
{
	size_t len = strlen(path);

	/* Skip any trailing slashes (except the initial one for absolute paths) */
	while (len > 1 && path[len - 1] == '/') {
		len--;
	}

	/* Skip last path element */
	while (len > 0 && path[len - 1] != '/') {
		len--;
	}

	if (len == 0) {
		/* This was a single-element relative path like "foo",
		   which has not "real" dirname */
		return xstrdup(".");
	}

	/* Skip any separator slashes (except the initial one for absolute paths) */
	while (len > 1 && path[len - 1] == '/') {
		len--;
	}

	return xstrndup(path, len);
}

char *resolve_relative_path(const char *dir, const char *relative)
{
	autofree char *dircopy = xstrdup(dir);
	while (true) {
		if (str_has_prefix(relative, "./")) {
			relative += 2;
			continue;
		}

		if (strcmp(relative, ".") == 0) {
			relative += 1;
			continue;
		}

		if (str_has_prefix(relative, "../")) {
			autofree char *old_dircopy = steal_ptr(&dircopy);
			dircopy = xdirname(old_dircopy);
			relative += 3;
			continue;
		}

		if (strcmp(relative, "..") == 0) {
			autofree char *old_dircopy = steal_ptr(&dircopy);
			dircopy = xdirname(old_dircopy);
			relative += 2;
			continue;
		}

		break;
	}

	if (strlen(relative) == 0) {
		return steal_ptr(&dircopy);
	}

	return join_path(dircopy, relative);
}

char *resolve_symlink_in_root(const char *rootdir, const char *symlink_path)
{
	struct stat st;
	autofree char *full_path = join_path(rootdir, symlink_path);

	if (lstat(full_path, &st) == -1) {
		return NULL;
	}

	if (!S_ISLNK(st.st_mode)) {
		/* Not a symlink, resolves to itself */
		return xstrdup(symlink_path);
	}

	size_t bufsize = st.st_size + 1;
	autofree char *target = xmalloc(bufsize);
	ssize_t len = readlink(full_path, target, (size_t)bufsize - 1);
	if (len == -1) {
		return NULL;
	}
	target[len] = '\0';

	if (target[0] == '/') {
		return steal_ptr(&target);
	} else {
		autofree char *dirname = xdirname(symlink_path);
		return resolve_relative_path(dirname, target);
	}
}

char *read_file(const char *filename)
{
	autoclose int fd = -1;
	size_t n_read;
	size_t buf_size;

	fd = open(filename, O_RDONLY);
	if (fd == -1)
		return NULL;

	buf_size = 4096;
	autofree char *buf = xmalloc(buf_size);

	n_read = 0;
	while (1) {
		if (n_read == buf_size) {
			buf_size *= 2;
			buf = xrealloc(buf, buf_size);
		}

		ssize_t r = read(fd, buf + n_read, buf_size - n_read);
		if (r < 0)
			return NULL;

		n_read += r;
		if (r == 0) /* early eof */
			break;
	}

	// Always null-terminate
	buf[n_read] = 0;

	return steal_ptr(&buf);
}

char *read_proc_cmdline(void)
{
	return read_cmdline_from_path("/proc/cmdline");
}

char *resolve_kernel_cmdline(const char *rootdir)
{
	autofree char *cmdline = NULL;
	const char *paths[] = { "/usr/etc/kernel/cmdline", "/etc/kernel/cmdline",
				"/usr/lib/kernel/cmdline", "/proc/cmdline", NULL };

	for (int i = 0; paths[i] != NULL; i++) {
		autofree char *path = join_path(rootdir, paths[i]);
		cmdline = read_cmdline_from_path(path);
		if (cmdline) {
			if (use_verbose)
				verbose("Reading cmdline from: %s\n", paths[i]);

			return steal_ptr(&cmdline);
		}
	}

	return NULL;
}

char *read_cmdline_from_path(const char *path)
{
	FILE *f = fopen(path, "r");
	autofree char *cmdline = NULL;
	size_t len;

	if (!f)
		goto out;

	/* Note that /proc/cmdline will not end in a newline, so getline
	 * will fail unless we provide a length.
	 */
	if (getline(&cmdline, &len, f) < 0)
		goto out;
	/* ... but the length will be the size of the malloc buffer, not
	 * strlen().  Fix that.
	 */
	len = strlen(cmdline);

	if (cmdline[len - 1] == '\n')
		cmdline[len - 1] = '\0';
out:
	if (f)
		fclose(f);
	return steal_ptr(&cmdline);
}

char *find_proc_cmdline_key(const char *cmdline, const char *key)
{
	const size_t key_len = strlen(key);
	for (const char *iter = cmdline; iter;) {
		const char *next = strchr(iter, ' ');
		if (strncmp(iter, key, key_len) == 0 && iter[key_len] == '=') {
			const char *start = iter + key_len + 1;
			if (next)
				return strndup(start, next - start);

			return xstrdup(start);
		}

		if (next)
			next += strspn(next, " ");

		iter = next;
	}

	return NULL;
}

char *trimwhitespace(char *str)
{
	char *end;

	// Trim leading space
	while (isspace((unsigned char)*str))
		str++;

	if (*str == 0) // All spaces?
		return str;

	// Trim trailing space
	end = str + strlen(str) - 1;
	while (end > str && isspace((unsigned char)*end))
		end--;

	// Write new null terminator character
	end[1] = '\0';

	return str;
}

char *remove_prefix(const char *full_path, const char *prefix)
{
	if (!full_path || !prefix) {
		return NULL;
	}

	char *start = strstr(full_path, prefix);

	if (start != full_path) {
		return NULL;
	}

	const char *tail = full_path + strlen(prefix);
	if (strstr(tail, "..") != NULL) {
		return NULL;
	}

	return xstrdup(tail);
}

int write_file_to_disk(const char *source, const char *destination)
{
	autoclose int src_fd = open(source, O_RDONLY);
	if (src_fd < 0) {
		error("Error: Failed to open source file '%s': %s\n", source,
		      strerror(errno));
		return -1;
	}

	char real_destination[PATH_MAX];
	if (!realpath_retry(destination, real_destination)) {
		error("Error: Failed to resolve destination '%s': %s\n",
		      destination, strerror(errno));
		return -1;
	}

	autoclose int dest_fd = open(destination, O_WRONLY);
	if (dest_fd < 0) {
		error("Error: Failed to open destination '%s': %s\n",
		      destination, strerror(errno));
		return -1;
	}

	const size_t buffer_size = 1024 * 1024;
	autofree char *buffer = xmalloc(buffer_size);

	ssize_t bytes_read;
	off_t total_written = 0;

	while (true) {
		bytes_read = read(src_fd, buffer, buffer_size);
		if (bytes_read <= 0) {
			if (bytes_read < 0 && errno == EINTR)
				continue;
			break;
		}
		char *write_ptr = buffer;
		ssize_t bytes_to_write = bytes_read;

		while (bytes_to_write > 0) {
			ssize_t bytes_written =
				write(dest_fd, write_ptr, bytes_to_write);
			if (bytes_written < 0) {
				if (errno == EINTR)
					continue;
				error("Error: Write failed to '%s': %s\n",
				      destination, strerror(errno));
				return -1;
			}
			write_ptr += bytes_written;
			bytes_to_write -= bytes_written;
			total_written += bytes_written;
		}

		verbose("Written %ld bytes\n", (long)total_written);
	}

	if (bytes_read < 0) {
		error("Error: Read failed from '%s': %s\n", source, strerror(errno));
		return -1;
	}

	if (fsync(dest_fd) < 0) {
		error("Error: Failed to sync '%s': %s\n", destination,
		      strerror(errno));
		return -1;
	}

	fprintf(stdout, "Success: Image '%s' successfully written to '%s'.\n",
		source, destination);
	return 0;
}

off_t get_image_size(const char *image)
{
	struct stat file_stat;
	if (stat(image, &file_stat) == -1) {
		error("Error: Failed to get image size: %s\n", image);
		return -1;
	}
	return file_stat.st_size;
}

off_t get_block_dev_size(const char *destination)
{
	autoclose int file = -1;
	file = open(destination, O_RDONLY);
	if (file < 0) {
		perror("Error: Failed to open device for size check");
		return -1;
	}
	off_t DEST_SIZE = 0;
	if (ioctl(file, BLKGETSIZE64, &DEST_SIZE) == -1) {
		error("Error: Failed to get device size for: %s\n", destination);
		return -1;
	}
	return DEST_SIZE;
}

int atomic_symlink(const char *target, const char *linkpath)
{
	struct timeval tv;
	gettimeofday(&tv, NULL);
	autofree char *temp_link = xasprintf("%s_%ld%ld.tmp", linkpath,
					     (long)tv.tv_sec, (long)tv.tv_usec);
	if (temp_link == NULL) {
		perror("Error: Failed to allocate temporary path memory");
		return -1;
	}

	if (symlink(target, temp_link) != 0) {
		perror("Error: Failed to create temporary symlink");
		return -1;
	}

	if (rename(temp_link, linkpath) != 0) {
		perror("Error: Failed to rename symlink atomically");
		remove(temp_link);
		return -1;
	}

	return 0;
}

char *xstrdup(const char *string)
{
	char *res = strdup(string);
	if (res == NULL)
		oom();
	return res;
}

char *xstrndup(const char *string, size_t n)
{
	char *res = strndup(string, n);
	if (res == NULL)
		oom();
	return res;
}

struct ConfigKey {
	const char *key;
	size_t offset;
};

static struct ConfigKey keys[] = {
	{ "BOOT_TYPE", offsetof(AbootConfig, boot_type) },
	{ "PARTITION_A", offsetof(AbootConfig, partition_a) },
	{ "PARTITION_B", offsetof(AbootConfig, partition_b) },
	{ "VBMETA_PARTITION_A", offsetof(AbootConfig, vbmeta_partition_a) },
	{ "VBMETA_PARTITION_B", offsetof(AbootConfig, vbmeta_partition_b) },
	{ "PAGESIZE", offsetof(AbootConfig, pagesize) },
	{ "BASE", offsetof(AbootConfig, base) },
	{ "KERNEL_OFFSET", offsetof(AbootConfig, kernel_offset) },
	{ "PARTITION_CTL", offsetof(AbootConfig, partition_ctl) },
	{ "RAMDISK_OFFSET", offsetof(AbootConfig, ramdisk_offset) },
	{ "TAGS_OFFSET", offsetof(AbootConfig, tags_offset) },
	{ "SECOND_OFFSET", offsetof(AbootConfig, second_offset) },
	{ "DTB_OFFSET", offsetof(AbootConfig, dtb_offset) },
	{ "DTB_FILE", offsetof(AbootConfig, dtb_file) },
	{ "CMDLINE", offsetof(AbootConfig, cmdline) },
	{ "COMPRESS_KERNEL", offsetof(AbootConfig, compress_kernel) },
};

static struct ConfigKey *find_config_key(const char *name)
{
	for (size_t i = 0; i < N_ELEMENTS(keys); i++) {
		if (strcmp(keys[i].key, name) == 0)
			return &keys[i];
	}
	return NULL;
}

int aboot_config_load(AbootConfig *cfg, const char *config_file)
{
	autofree char *config_content = read_file(config_file);
	if (!config_content) {
		perror("Failed to read config file");
		return -1;
	}

	char *line = strtok(config_content, "\n");
	while (line) {
		if (line[0] == '#' || line[0] == '\0') {
			line = strtok(NULL, "\n");
			continue;
		}
		char *equals = strchr(line, '=');
		if (!equals) {
			line = strtok(NULL, "\n");
			continue;
		}
		*equals = '\0';
		char *key = trimwhitespace(line);
		char *value = trimwhitespace(equals + 1);

		struct ConfigKey *cfg_key = find_config_key(key);
		if (cfg_key == NULL) {
			error("Unknown key %s\n", key);
			return -1;
		} else {
			*(char **)((char *)cfg + cfg_key->offset) = xstrdup(value);
		}
		line = strtok(NULL, "\n");
	}
	return 0;
}

AbootConfig *aboot_config_new(void)
{
	AbootConfig *cfg = xcalloc(1, sizeof(AbootConfig));

	cfg->vbmeta_partition_a = xstrdup("/dev/disk/by-partlabel/vbmeta_a");
	cfg->vbmeta_partition_b = xstrdup("/dev/disk/by-partlabel/vbmeta_b");
	cfg->boot_type = xstrdup("aboot");
	cfg->compress_kernel = xstrdup("true");
	cfg->pagesize = xstrdup("2048");
	cfg->base = xstrdup("0x10000000");
	cfg->kernel_offset = xstrdup("0x00008000");
	cfg->ramdisk_offset = xstrdup("0x01000000");
	cfg->tags_offset = xstrdup("0x00000100");
	cfg->second_offset = xstrdup("0x00f00000");
	cfg->dtb_offset = xstrdup("0x01f00000");
	cfg->dtb_file = xstrdup("");

	return cfg;
}

void aboot_config_free(AbootConfig *cfg)
{
	free(cfg->boot_type);
	free(cfg->partition_a);
	free(cfg->partition_b);
	free(cfg->vbmeta_partition_a);
	free(cfg->vbmeta_partition_b);
	free(cfg->bootlabel);
	free(cfg->pagesize);
	free(cfg->base);
	free(cfg->kernel_offset);
	free(cfg->partition_ctl);
	free(cfg->ramdisk_offset);
	free(cfg->tags_offset);
	free(cfg->second_offset);
	free(cfg->dtb_offset);
	free(cfg->dtb_file);
	free(cfg->cmdline);
	free(cfg->compress_kernel);
	free(cfg);
}

/*
 * Executes a command with optional output redirection.
 * argv: NULL-terminated array of the command and its arguments.
 * stdout_fd: The file descriptor to write output to, or -1 for standard stdout.
 */
int exec_command(const char *const argv[], int stdout_fd)
{
	if (!argv || !argv[0])
		return -1;
	pid_t pid = fork();

	if (pid == -1) {
		perror("Error: fork failed");
		return -1;
	} else if (pid == 0) {
		if (stdout_fd != -1) {
			if (dup2(stdout_fd, STDOUT_FILENO) == -1) {
				fprintf(stderr,
					"Error: Failed to redirect stdout: %s\n",
					strerror(errno));
				_exit(127);
			}
			if (stdout_fd != STDOUT_FILENO) {
				close(stdout_fd);
			}
		}
		execvp(argv[0], (char *const *)argv);
		if (errno == ENOENT) {
			// Error: The program is not installed
			_exit(127);
		} else {
			// Error: Failed to execute the program
			_exit(126);
		}
	} else {
		int status;
		if (waitpid(pid, &status, 0) == -1) {
			perror("Error: waitpid failed");
			return -1;
		}

		if (WIFEXITED(status)) {
			int exit_code = WEXITSTATUS(status);
			if (exit_code != 0) {
				if (exit_code == 127) {
					error("Error: The program '%s' is not installed\n",
					      argv[0]);
				} else if (exit_code == 126) {
					error("Error: Failed to execute '%s'\n",
					      argv[0]);
				} else {
					error("Error: Command '%s' failed with exit code %d\n",
					      argv[0], exit_code);
				}
				return exit_code;
			}
		} else if (WIFSIGNALED(status)) {
			error("Error: Command '%s' terminated by signal %d\n",
			      argv[0], WTERMSIG(status));
			return -1;
		}
	}
	return 0;
}

#define MSEC_TO_NSEC(ms) (ms * 1000000)
char *__realpath_retry(const char *path, char *resolved_path,
		       unsigned long timeout_ms, unsigned long rate_ms)
{
	struct timespec ts = {
		.tv_nsec = MSEC_TO_NSEC(rate_ms),
	};
	char *rp;

	while (timeout_ms > 0 && !(rp = realpath(path, resolved_path)) &&
	       errno == ENOENT) {
		nanosleep(&ts, NULL);
		timeout_ms -= rate_ms;
	}

	return rp;
}
