#include "aboot-update.h"
#include <errno.h>
#include <getopt.h>
#include <magic.h>
#include <sys/stat.h>

const struct option longopts[] = {
	{ "help", no_argument, NULL, 'h' },
	{ "root", required_argument, NULL, 'r' },
	{ "preptree", no_argument, NULL, 'p' },
	{ "cmdline", required_argument, NULL, 'c' },
	{ "verbose", no_argument, NULL, 'v' },
	{},
};

bool use_verbose = false;

void print_usage(void)
{
	fprintf(stderr,
		"Usage: aboot-update [OPTIONS] KERNEL_VERSION\n"
		"Options:\n"
		"  -h, --help                       - Show this help message\n"
		"  -r, --root PATH                  - The root directory of the image being updated (if not live)\n"
		"  -p, --preptree                   - Run after the preptree stage of ostree\n"
		"  -c, --cmdline CMDLINE            - Override kernel cmdline\n"
		"  -v, --verbose                    - Show verbose output\n");
}

char *remove_cmdline_key(const char *cmdline, const char *key)
{
	if (!cmdline || !key)
		return xstrdup(cmdline ? cmdline : "");

	autofree char *cmdline_copy = xstrdup(cmdline);
	autofree char *result = xstrdup("");
	const size_t key_len = strlen(key);
	bool first = true;

	char *saveptr;
	char *param = strtok_r(cmdline_copy, " ", &saveptr);
	while (param) {
		if (!(strncmp(param, key, key_len) == 0 && param[key_len] == '=')) {
			autofree char *old_result = steal_ptr(&result);
			result = xasprintf("%s%s%s", old_result,
					   first ? "" : " ", param);
			first = false;
		}
		param = strtok_r(NULL, " ", &saveptr);
	}

	return steal_ptr(&result);
}

int is_efi_application(const char *filename)
{
	magic_t cookie;
	int is_efi = 0;

	cookie = magic_open(MAGIC_NONE);
	if (cookie == NULL) {
		fprintf(stderr, "Error: unable to initialize magic library\n");
		return 0;
	}

	if (magic_load(cookie, NULL) != 0) {
		fprintf(stderr, "Error: cannot load magic database - %s\n",
			magic_error(cookie));
		magic_close(cookie);
		return 0;
	}

	const char *description = magic_file(cookie, filename);

	if (description != NULL && strstr(description, "EFI application") != NULL) {
		is_efi = 1;
	}

	magic_close(cookie);
	return is_efi;
}

int build_ukiboot(AbootConfig *cfg, char *dtb_path, char *kernel, char *initrd,
		  char *kernel_version, char *destination_boot)
{
	const char *ukify_args[32];
	int i = 0;

	ukify_args[i++] = "ukify";
	ukify_args[i++] = "build";
	ukify_args[i++] = "--linux";
	ukify_args[i++] = kernel;
	ukify_args[i++] = "--initrd";
	ukify_args[i++] = initrd;
	ukify_args[i++] = "--uname";
	ukify_args[i++] = kernel_version;
	ukify_args[i++] = "--cmdline";
	ukify_args[i++] = cfg->cmdline;
	ukify_args[i++] = "--output";
	ukify_args[i++] = destination_boot;

	if (dtb_path && dtb_path[0] != '\0') {
		ukify_args[i++] = "--devicetree";
		ukify_args[i++] = dtb_path;
	}
	ukify_args[i++] = NULL;

	if (exec_command(ukify_args, -1) != 0) {
		return -1;
	}
	return 0;
}

int build_aboot(AbootConfig *cfg, char *dtb_path, char *initrd, char *kernel,
		char *destination_boot)
{
	// cfg->boot_type == aboot
	/* The Android Boot Image v2 is limited to 511 bytes for the kernel command line.
	* Some kernel parameters are longer, so add them to the bootconfig file if they
	* are present. Note that not all kernel parameters can be added to the
	* bootconfig file, such as the ones that are early parameters, which there are
	* hundreds. So only add certain parameters to the bootconfig file.
	*/
	autofree char *boot_config_entries = xstrdup("");

	const char *keys[] = { "cgroup_disable",      "cgroup_no_v1",
			       "module.sig_enforce",  "no_console_suspend",
			       "rd.modules-load",     "systemd.random-seed",
			       "systemd.show_status", NULL };

	for (int i = 0; keys[i] != NULL; i++) {
		autofree char *cmdline_val =
			find_proc_cmdline_key(cfg->cmdline, keys[i]);

		if (cmdline_val && cmdline_val[0] != '\0') {
			autofree char *old_entries = steal_ptr(&boot_config_entries);

			boot_config_entries =
				xasprintf("%s%s = \"%s\"\n", old_entries,
					  keys[i], cmdline_val);

			char *new_cmdline =
				remove_cmdline_key(cfg->cmdline, keys[i]);
			free(cfg->cmdline);
			cfg->cmdline = new_cmdline;
		}
	}

	if (boot_config_entries[0] != '\0') {
		verbose("Creating bootconfig file with entries:\n%s",
			boot_config_entries);
		autounlink char *bootconfig_file =
			xasprintf("/tmp/aboot-bootconfig-XXXXXX");
		autoclose int bootconfig_fd = mkstemp(bootconfig_file);
		if (bootconfig_fd < 0) {
			error("Error: Failed to create bootconfig temp file: %s\n",
			      strerror(errno));
			return -1;
		}

		{
			autofclose FILE *fp = fdopen(bootconfig_fd, "w");
			if (!fp) {
				error("Error: Failed to open bootconfig file: %s\n",
				      strerror(errno));
				return -1;
			}
			bootconfig_fd = -1;

			fprintf(fp, "kernel {\n%s\n}\n", boot_config_entries);
		}

		autofree char *old_cmdline = steal_ptr(&cfg->cmdline);
		cfg->cmdline = xasprintf("%s bootconfig", old_cmdline);

		const char *bootconfig_args[] = { "bootconfig", "-a",
						  bootconfig_file, initrd, NULL };

		if (exec_command(bootconfig_args, -1) != 0) {
			error("Error: Failed to apply bootconfig to initrd\n");
			return -1;
		}
	}

	const char *mkbootimg_args[32];
	int i = 0;

	mkbootimg_args[i++] = "mkbootimg";
	mkbootimg_args[i++] = "--base";
	mkbootimg_args[i++] = cfg->base;
	mkbootimg_args[i++] = "--kernel_offset";
	mkbootimg_args[i++] = cfg->kernel_offset;
	mkbootimg_args[i++] = "--ramdisk_offset";
	mkbootimg_args[i++] = cfg->ramdisk_offset;
	mkbootimg_args[i++] = "--tags_offset";
	mkbootimg_args[i++] = cfg->tags_offset;
	mkbootimg_args[i++] = "--pagesize";
	mkbootimg_args[i++] = cfg->pagesize;
	mkbootimg_args[i++] = "--second_offset";
	mkbootimg_args[i++] = cfg->second_offset;
	mkbootimg_args[i++] = "--ramdisk";
	mkbootimg_args[i++] = initrd;
	mkbootimg_args[i++] = "--dtb_offset";
	mkbootimg_args[i++] = cfg->dtb_offset;

	if (dtb_path != NULL && dtb_path[0] != '\0') {
		mkbootimg_args[i++] = "--dtb";
		mkbootimg_args[i++] = dtb_path;
	}

	mkbootimg_args[i++] = "--kernel";
	mkbootimg_args[i++] = kernel;
	mkbootimg_args[i++] = "--cmdline";
	mkbootimg_args[i++] = cfg->cmdline;
	mkbootimg_args[i++] = "--header_version";
	mkbootimg_args[i++] = "2";
	mkbootimg_args[i++] = "-o";
	mkbootimg_args[i++] = destination_boot;

	mkbootimg_args[i++] = NULL;

	if (exec_command(mkbootimg_args, -1) != 0) {
		return -1;
	}

	return 0;
}

char *resolve_boot_file(const char *dir_path, const char *prefix)
{
	autofclosedir DIR *dir = opendir(dir_path);

	if (!dir)
		return NULL;

	struct dirent *file;

	while ((file = readdir(dir))) {
		if (str_has_prefix(file->d_name, prefix)) {
			return join_path(dir_path, file->d_name);
		}
	}

	return NULL;
}

int main(int argc, char *argv[])
{
	autofree char *cmdline_opt = NULL;
	autofree char *rootdir = NULL;
	bool preptree = false;
	autofreeconfig AbootConfig *cfg = aboot_config_new();
	autofree char *destination_boot = NULL;
	autofree char *boot_dir = NULL;
	autounlink char *kernel_cleanup = NULL;

	int opt;
	const char *optstring = "hr:pc:v";
	while ((opt = getopt_long(argc, argv, optstring, longopts, NULL)) != -1) {
		switch (opt) {
		case 'h':
			print_usage();
			return EXIT_FAILURE;
		case 'r':
			rootdir = xstrdup(optarg);
			break;
		case 'p':
			preptree = true;
			break;
		case 'c':
			cmdline_opt = xstrdup(optarg);
			break;
		case 'v':
			use_verbose = true;
			break;
		default:
			print_usage();
			return EXIT_FAILURE;
		}
	}

	argv += optind;
	argc -= optind;
	if (argc < 1) {
		print_usage();
		error("Error: No kernel_version argument provided.\n");
		return EXIT_FAILURE;
	}

	if (rootdir == NULL) {
		rootdir = xstrdup("/");
	}

	char *kernel_version = argv[0];

	if (preptree) {
		/* If we're running at image build time, after calling ostree
		* prep-tree, then all the files that will be in /boot have been
		* moved to /usr/lib/ostree-boot.
		*/
		boot_dir = join_path(rootdir, "/usr/lib/ostree-boot");
	} else {
		boot_dir = join_path(rootdir, "/boot");
	}

	autofree char *aboot_cfg = join_path(boot_dir, DEFAULT_ABOOT_CFG_NAME);

	struct stat st;
	if (stat(aboot_cfg, &st) == 0) {
		// Config defaults (from mkbootimg)
		if (aboot_config_load(cfg, aboot_cfg) != 0) {
			error("Error: Failed to load config file: %s\n", aboot_cfg);
			return EXIT_FAILURE;
		}
		verbose("Loaded config from %s\n", aboot_cfg);
	} else {
		verbose("No config file at %s, using defaults\n", aboot_cfg);
	}
	// --cmdline overrides config file
	if (cmdline_opt != NULL) {
		free(cfg->cmdline);
		cfg->cmdline = steal_ptr(&cmdline_opt);
	}
	//If CMDLINE was not defined in aboot.cfg, use the default from the os config
	if (cfg->cmdline == NULL) {
		cfg->cmdline = resolve_kernel_cmdline(rootdir);
	}

	autofree char *dtb_path = NULL;
	if (cfg->dtb_file[0] != '\0') {
		//Convert dtb path from config file to real path.
		if (cfg->dtb_file[0] == '/') {
			/* Absolute path: just use as is. This is useful for dtbs packaged separately
			* so they don't have to keep 'uname -r' version in sync, as the location of
			* these files change based on that.
			*/
			dtb_path = join_path(rootdir, cfg->dtb_file);
		} else {
			// Relative path, related to versioned dtb dir
			autofree char *dtb_name =
				xasprintf("dtb-%s", kernel_version);
			autofree char *ftd_dir = join_path(boot_dir, dtb_name);
			dtb_path = join_path(ftd_dir, cfg->dtb_file);
		}
		if (access(dtb_path, F_OK) != 0) {
			error("Error: DTB not found: '%s': %s\n", dtb_path,
			      strerror(errno));
			return -1;
		}
	}

	autofree char *initrd_name = xasprintf("initramfs-%s.img", kernel_version);
	autofree char *initrd = join_path(boot_dir, initrd_name);

	autofree char *kernel_name = xasprintf("vmlinuz-%s", kernel_version);
	autofree char *kernel = join_path(boot_dir, kernel_name);

	autofree char *module_dir =
		join_3_path(rootdir, "/usr/lib/modules/", kernel_version);

	if (preptree) {
		/* In addition to the different BOOT_DIR (/usr/lib/ostree-boot), ostree prep-tree
		* will be creating the initrd file (and also a vmlinuz copy) with a custom boot deploy
		* suffix. For example, we can have files like:
		* /usr/lib/ostree-boot/vmlinuz-5.14.0-230.194.el9iv.aarch64-217d248ed4e397b0129b14185eff75dc56568d0ef732b6e3a690144eab2a9003
		* /usr/lib/ostree-boot/initramfs-5.14.0-230.194.el9iv.aarch64.img-217d248ed4e397b0129b14185eff75dc56568d0ef732b6e3a690144eab2a9003
		*/
		autofree char *initrd_expanded =
			resolve_boot_file(boot_dir, initrd_name);
		autofree char *kernel_expanded =
			resolve_boot_file(boot_dir, kernel_name);

		if (!initrd_expanded || !kernel_expanded) {
			error("Error: Failed to find kernel or initrd files with suffix\n");
			return EXIT_FAILURE;
		}

		/* We need to use these when we build aboot.img
		* Also, when deploying an image, ostree will be looking for aboot.img in /usr/lib/modules/$KVER/aboot.img to
		* mark a commit as using aboot, which will be exported to /boot as aboot-$kver.img.
		*/
		free(initrd);
		initrd = steal_ptr(&initrd_expanded);
		free(kernel);
		kernel = steal_ptr(&kernel_expanded);

		destination_boot = join_path(module_dir, "/aboot.img");
	} else {
		autofree char *aboot_img = xasprintf("aboot-%s.img", kernel_version);
		destination_boot = join_path(boot_dir, aboot_img);
	}

	if (access(initrd, F_OK) != 0) {
		error("Error: Initrd not found: '%s': %s\n", initrd, strerror(errno));
		return -1;
	}

	if (access(kernel, F_OK) != 0) {
		error("Error: Kernel not found: '%s': %s\n", kernel, strerror(errno));
		return -1;
	}

	if (is_efi_application(kernel)) {
		/* The Fedora aarch64 kernel uses the vmlinuz.efi target. ABL won't boot
		* these binaries, so use unzboot (https://github.com/eballetbo/unzboot)
		* to extract the kernel.
		*/
		verbose("Kernel is an EFI application, extracting with unzboot\n");
		autounlink char *temp_unzboot =
			xasprintf("/tmp/aboot-unzboot-XXXXXX");

		int unzboot_fd = mkstemp(temp_unzboot);
		if (unzboot_fd < 0) {
			error("Error: Failed to create temporary file: %s\n",
			      strerror(errno));
			return EXIT_FAILURE;
		}
		close(unzboot_fd);
		const char *unzboot_args[] = { "unzboot", kernel, temp_unzboot, NULL };

		if (exec_command(unzboot_args, -1) != 0) {
			return EXIT_FAILURE;
		}

		kernel_cleanup = steal_ptr(&temp_unzboot);
		free(kernel);
		kernel = xstrdup(kernel_cleanup);
		/* The kernel returned by unzboot is not compressed. Compression is
		* enabled here.
		*/
		if (strcmp(cfg->compress_kernel, "true") == 0) {
			verbose("Compressing kernel: %s\n", kernel);
			autounlink char *temp_gzip =
				xasprintf("/tmp/aboot-gzip-XXXXXX");
			autoclose int gzip_fd = mkstemp(temp_gzip);
			if (gzip_fd < 0) {
				error("Error: Failed to create temp file: %s\n",
				      strerror(errno));
				return EXIT_FAILURE;
			}

			const char *gzip_args[] = { "gzip", "--stdout", kernel, NULL };

			if (exec_command(gzip_args, gzip_fd) != 0) {
				return EXIT_FAILURE;
			}

			if (kernel_cleanup) {
				cleanup_unlink(&kernel_cleanup);
			}

			kernel_cleanup = steal_ptr(&temp_gzip);
			free(kernel);
			kernel = xstrdup(kernel_cleanup);
		}
	} else {
		/* The kernel is gzip compressed. Booting an uncompressed kernel saves
		* ~100ms off the overall boot in the firmware.
		*/
		if (strcmp(cfg->compress_kernel, "false") == 0) {
			autounlink char *temp_gunzip =
				xasprintf("/tmp/aboot-gunzip-XXXXXX");
			autoclose int gunzip_fd = mkstemp(temp_gunzip);
			if (gunzip_fd < 0) {
				error("Error: Failed to create temp file: %s\n",
				      strerror(errno));
				return EXIT_FAILURE;
			}

			const char *gunzip_args[] = { "gunzip", "--stdout",
						      kernel, NULL };

			if (exec_command(gunzip_args, gunzip_fd) != 0) {
				return EXIT_FAILURE;
			}
			kernel_cleanup = steal_ptr(&temp_gunzip);
			free(kernel);
			kernel = xstrdup(kernel_cleanup);
		}
	}
	if (strcmp(cfg->boot_type, "ukiboot") == 0) {
		if (build_ukiboot(cfg, dtb_path, kernel, initrd, kernel_version,
				  destination_boot) != 0) {
			return EXIT_FAILURE;
		}
	} else {
		if (build_aboot(cfg, dtb_path, initrd, kernel, destination_boot) != 0) {
			return EXIT_FAILURE;
		}
	}

	return EXIT_SUCCESS;
}
