#include "config.h"

#include <libfdisk/libfdisk.h>
#include <getopt.h>
#include <linux/bsg.h>
#include <scsi/scsi_bsg_ufs.h>
#include <scsi/sg.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <dirent.h>
#include <fnmatch.h>
#include <limits.h>

#include "aboot-utils.h"

bool use_verbose = false;

typedef struct {
	uint8_t priority;
	bool active;
	uint8_t tries_remaining;
	bool successful;
	bool unbootable;
} BootSlotInfo;

static char *slot_a_dev, *slot_b_dev;
static uint32_t slot_a_part, slot_b_part;

static char *ufs_bsg_dev;

#define MAX_PRIO 3
#define MAX_TRIES 7

static inline void cleanup_fdisk(struct fdisk_context **ctx)
{
	if (ctx && *ctx) {
		fdisk_unref_context(*ctx);
	}
}

#define autofdisk __attribute__((cleanup(cleanup_fdisk)))

/*
 * include/ufs/ufs.h, not in uapi.
 */
/* UTP UPIU Transaction Codes Initiator to Target */
enum upiu_request_transaction {
	UPIU_TRANSACTION_QUERY_REQ = 0x16,
};
/* UPIU Query request function */
enum {
	UPIU_QUERY_FUNC_STANDARD_READ_REQUEST           = 0x01,
	UPIU_QUERY_FUNC_STANDARD_WRITE_REQUEST          = 0x81,
};
/* UTP QUERY Transaction Specific Fields OpCode */
enum query_opcode {
	UPIU_QUERY_OPCODE_NOP		= 0x0,
	UPIU_QUERY_OPCODE_READ_DESC	= 0x1,
	UPIU_QUERY_OPCODE_WRITE_DESC	= 0x2,
	UPIU_QUERY_OPCODE_READ_ATTR	= 0x3,
	UPIU_QUERY_OPCODE_WRITE_ATTR	= 0x4,
	UPIU_QUERY_OPCODE_READ_FLAG	= 0x5,
	UPIU_QUERY_OPCODE_SET_FLAG	= 0x6,
	UPIU_QUERY_OPCODE_CLEAR_FLAG	= 0x7,
	UPIU_QUERY_OPCODE_TOGGLE_FLAG	= 0x8,
};
/* Attribute idn for Query requests */
enum attr_idn {
	QUERY_ATTR_IDN_BOOT_LU_EN		= 0x00,
	QUERY_ATTR_IDN_MAX_HPB_SINGLE_CMD	= 0x01,
	QUERY_ATTR_IDN_POWER_MODE		= 0x02,
	QUERY_ATTR_IDN_ACTIVE_ICC_LVL		= 0x03,
	QUERY_ATTR_IDN_OOO_DATA_EN		= 0x04,
	QUERY_ATTR_IDN_BKOPS_STATUS		= 0x05,
	QUERY_ATTR_IDN_PURGE_STATUS		= 0x06,
	QUERY_ATTR_IDN_MAX_DATA_IN		= 0x07,
	QUERY_ATTR_IDN_MAX_DATA_OUT		= 0x08,
	QUERY_ATTR_IDN_DYN_CAP_NEEDED		= 0x09,
	QUERY_ATTR_IDN_REF_CLK_FREQ		= 0x0A,
	QUERY_ATTR_IDN_CONF_DESC_LOCK		= 0x0B,
	QUERY_ATTR_IDN_MAX_NUM_OF_RTT		= 0x0C,
	QUERY_ATTR_IDN_EE_CONTROL		= 0x0D,
	QUERY_ATTR_IDN_EE_STATUS		= 0x0E,
	QUERY_ATTR_IDN_SECONDS_PASSED		= 0x0F,
	QUERY_ATTR_IDN_CNTX_CONF		= 0x10,
	QUERY_ATTR_IDN_CORR_PRG_BLK_NUM		= 0x11,
	QUERY_ATTR_IDN_RESERVED2		= 0x12,
	QUERY_ATTR_IDN_RESERVED3		= 0x13,
	QUERY_ATTR_IDN_FFU_STATUS		= 0x14,
	QUERY_ATTR_IDN_PSA_STATE		= 0x15,
	QUERY_ATTR_IDN_PSA_DATA_SIZE		= 0x16,
	QUERY_ATTR_IDN_REF_CLK_GATING_WAIT_TIME	= 0x17,
	QUERY_ATTR_IDN_CASE_ROUGH_TEMP          = 0x18,
	QUERY_ATTR_IDN_HIGH_TEMP_BOUND          = 0x19,
	QUERY_ATTR_IDN_LOW_TEMP_BOUND           = 0x1A,
	QUERY_ATTR_IDN_WB_FLUSH_STATUS	        = 0x1C,
	QUERY_ATTR_IDN_AVAIL_WB_BUFF_SIZE       = 0x1D,
	QUERY_ATTR_IDN_WB_BUFF_LIFE_TIME_EST    = 0x1E,
	QUERY_ATTR_IDN_CURR_WB_BUFF_SIZE        = 0x1F,
	QUERY_ATTR_IDN_EXT_IID_EN		= 0x2A,
	QUERY_ATTR_IDN_TIMESTAMP		= 0x30
};

/* Query response result code */
enum {
	QUERY_RESULT_SUCCESS                    = 0x00,
	QUERY_RESULT_NOT_READABLE               = 0xF6,
	QUERY_RESULT_NOT_WRITEABLE              = 0xF7,
	QUERY_RESULT_ALREADY_WRITTEN            = 0xF8,
	QUERY_RESULT_INVALID_LENGTH             = 0xF9,
	QUERY_RESULT_INVALID_VALUE              = 0xFA,
	QUERY_RESULT_INVALID_SELECTOR           = 0xFB,
	QUERY_RESULT_INVALID_INDEX              = 0xFC,
	QUERY_RESULT_INVALID_IDN                = 0xFD,
	QUERY_RESULT_INVALID_OPCODE             = 0xFE,
	QUERY_RESULT_GENERAL_FAILURE            = 0xFF,
};

static inline const char *query_result_to_str(int rc)
{
	switch (rc) {
	case QUERY_RESULT_SUCCESS:
		return "Success";
	case QUERY_RESULT_NOT_READABLE:
		return "Readable";
	case QUERY_RESULT_NOT_WRITEABLE:
		return "Writeable";
	case QUERY_RESULT_ALREADY_WRITTEN:
		return "Written";
	case QUERY_RESULT_INVALID_LENGTH:
		return "Invalid Length";
	case QUERY_RESULT_INVALID_VALUE:
		return "Invalid Value";
	case QUERY_RESULT_INVALID_SELECTOR:
		return "Invalid Selector";
	case QUERY_RESULT_INVALID_INDEX:
		return "Invalid Index";
	case QUERY_RESULT_INVALID_IDN:
		return "Invalid Idn";
	case QUERY_RESULT_INVALID_OPCODE:
		return "Invalid Opcode";
	case QUERY_RESULT_GENERAL_FAILURE:
		return "General Failure";
	default:
		return "Unknown Failure";
	}
}

/*
 * struct utp_upiu_req query initializer (e.g, upiu_req in ufs_bsg_request)
 */
static inline void utp_upiu_query_req_init(struct utp_upiu_req *upiu_req,
					   uint8_t query_function, uint16_t data_len,
					   uint8_t opcode, uint8_t idn,
					   uint8_t index, uint8_t selector)
{
	upiu_req->header = (struct utp_upiu_header){
		.transaction_code = UPIU_TRANSACTION_QUERY_REQ,
		.query_function = query_function,
		.data_segment_length = data_len,
	};
	upiu_req->qr = (struct utp_upiu_query){
		.opcode = opcode,
		.idn = idn,
		.index = index,
		.selector = selector,
		.length = data_len,
	};
}

static inline void sg_io_v4_scsi_transport_init(struct sg_io_v4 *hdr,
						void *response, size_t response_len,
						void *request, size_t request_len)
{
	*hdr = (struct sg_io_v4){
		.guard = 'Q',
		.protocol = BSG_PROTOCOL_SCSI,
		.subprotocol = BSG_SUB_PROTOCOL_SCSI_TRANSPORT,
		.response = (uint64_t)response,
		.max_response_len = response_len,
		.request = (uint64_t)request,
		.request_len = request_len,
	};
}

static int sg_io_upiu_send(int fd, struct ufs_bsg_request *request,
			   struct ufs_bsg_reply *response)
{
	struct sg_io_v4 hdr;

	sg_io_v4_scsi_transport_init(&hdr, response, sizeof(*response), request,
				     sizeof(*request));
	while (ioctl(fd, SG_IO, &hdr) &&
	       ((errno == EINTR) || (errno == EAGAIN)))
		;

	if (hdr.info) {
		error("Command failed with status %#x.\n", hdr.info);
		return -1;
	}

	if (response->upiu_rsp.header.response) {
		verbose("UPIU query error: '%s' (%#x)\n",
			query_result_to_str(response->upiu_rsp.header.response),
			response->upiu_rsp.header.response);
		return -1;
	}

	return 0;
}

static int ufs_bsg_write_bootlun(const char *bsgdev, unsigned long id)
{
	struct ufs_bsg_request request;
	struct ufs_bsg_reply response = { 0 };
	autoclose int fd = -1;

	request.msgcode = UPIU_TRANSACTION_QUERY_REQ,
	utp_upiu_query_req_init(&request.upiu_req,
				UPIU_QUERY_FUNC_STANDARD_WRITE_REQUEST, 0,
				UPIU_QUERY_OPCODE_WRITE_ATTR,
				QUERY_ATTR_IDN_BOOT_LU_EN, 0, 0);
	request.upiu_req.qr.value = htobe32(id);

	fd = open(bsgdev, O_RDWR);
	if (fd < 0) {
		perror("Error: Failed to open bsg devnode.");
		return -1;
	}

	if (sg_io_upiu_send(fd, &request, &response)) {
		error("Error: Failed to write bootlun.\n");
		return -1;
	}

	return 0;
}

static int ufs_bsg_read_bootlun(const char *bsgdev, unsigned long *id)
{
	struct ufs_bsg_request request;
	struct ufs_bsg_reply response = { 0 };
	autoclose int fd = -1;

	request.msgcode = UPIU_TRANSACTION_QUERY_REQ,
	utp_upiu_query_req_init(&request.upiu_req,
				UPIU_QUERY_FUNC_STANDARD_READ_REQUEST, 0,
				UPIU_QUERY_OPCODE_READ_ATTR,
				QUERY_ATTR_IDN_BOOT_LU_EN, 0, 0);

	fd = open(bsgdev, O_RDWR);
	if (fd < 0) {
		perror("Error: Failed to open bsg devnode.");
		return -1;
	}

	if (sg_io_upiu_send(fd, &request, &response)) {
		error("Error: Failed to read bootlun.\n");
		return -1;
	}
	*id = be32toh(response.upiu_rsp.qr.value);

	return 0;
}

char *resolve_ufs_bsg(const char *devlnk)
{
	/* Canonicalize, we expect inputs like /dev/disk/by-uuid/... */
	char real[PATH_MAX];
	if (!realpath_retry(devlnk, real)) {
		error("Can't resolve partition device '%s': %s\n", devlnk,
		      strerror(errno));
		return NULL;
	}

	autofree char *devname = xbasename(real);
	autofree char *sys_cls_blk = join_path("/sys/class/block", devname);
	autofree char *sys_hst_dir = join_path(sys_cls_blk, "../../../../../");
	if (!realpath(sys_hst_dir, real)) {
		error("Can't resolve host sysfs dir '%s': %s\n", sys_hst_dir,
		      strerror(errno));
		return NULL;
	}

	DIR *d = opendir(sys_hst_dir);
	if (!d) {
		error("Can't open dir '%s': %s.\n", sys_hst_dir, strerror(errno));
		return NULL;
	}
	
	struct dirent *de;
	while ((de = readdir(d))) {
		if (de->d_type != DT_DIR)
			continue;
		if (fnmatch("ufs-bsg[0-9]", de->d_name, 0))
			continue;
		break;
	}
	int rc = errno;
	closedir(d);
	if (!de) {
		if (rc)
			error("Can't list directory '%s': %s.\n", sys_hst_dir,
			      strerror(rc));
		else
			error("'%s' does not appear to be a UFS device.\n",
			      sys_hst_dir);
		return NULL;
	}

	autofree char *bsgname = xbasename(de->d_name);
	return join_path("/dev/bsg", bsgname);
}

int resolve_boot_lun_id(const char *devlnk, unsigned long *id)
{
	/* Canonicalize, we expect inputs like /dev/disk/by-uuid/... */
	char real[PATH_MAX];
	if (!realpath_retry(devlnk, real)) {
		error("Can't resolve partition device '%s': %s\n", devlnk,
		      strerror(errno));
		return -1;
	}

	autofree char *devname = xbasename(real);
	autofree char *sys_cls_blk = join_path("/sys/class/block", devname);
	autofree char *boot_lun_id_path = join_path(sys_cls_blk, "../../../unit_descriptor/boot_lun_id");

	autofree char *boot_lun_id = read_file(boot_lun_id_path);
	if (!boot_lun_id) {
		error("Failed to read boot LUN id for '%s' (%s).\n", devlnk,
		      boot_lun_id_path);
		return -1;
	}

	char *end;
	*id = strtoul(boot_lun_id, &end, 0);
	if (boot_lun_id == end || *id == ULONG_MAX) {
		error("Invalid boot LUN id for '%s' (%lu): %s.\n", devlnk, *id,
		      strerror(errno));
		return -1;
	}
	return 0;
}

char *resolve_partition(const char *part_dev, uint32_t *partnum_out)
{
	/* Canonicalize in case we get an input like /dev/disk/by-uuid/... */
	char real[PATH_MAX];
	if (!realpath_retry(part_dev, real)) {
		error("Can't resolve partition device '%s': %s\n", part_dev,
		      strerror(errno));
		return NULL;
	}

	/* Compute sysfs path for partition device */
	autofree char *part_base = xbasename(real);
	autofree char *sys_part = join_path("/sys/class/block", part_base);

	/* Read "partition" file which contains partition nr */
	autofree char *part_file = join_path(sys_part, "partition");
	autofree char *part_data = read_file(part_file);
	if (part_data == NULL) {
		error("Device '%s' is not a partition\n", part_dev);
		return NULL;
	}
	char *end = NULL;
	unsigned pnum = strtol(part_data, &end, 10);
	if (end == part_data) {
		error("Error parsing partition number\n");
		return NULL;
	}

	/* Canonicalize sysfs path to find its parent dir, which is the disk */
	char sys_part_real[PATH_MAX];
	if (!realpath(sys_part, sys_part_real)) {
		error("Can't resolve disk device '%s': %s\n", sys_part,
		      strerror(errno));
		return NULL;
	}

	/* Disk device is the basename of the sysfs path */
	autofree char *sys_disk_path = xdirname(sys_part_real);
	autofree char *sys_disk_basename = xbasename(sys_disk_path);

	*partnum_out = pnum - 1;
	return join_path("/dev", sys_disk_basename);
}

int set_gpt_bits(const char *disk_dev, uint32_t partnum, uint64_t values, uint64_t mask)
{
	autofdisk struct fdisk_context *cxt = fdisk_new_context();
	if (!cxt) {
		error("fdisk_new_context failed\n");
		return -1;
	}

	if (fdisk_assign_device(cxt, disk_dev, /*readonly=*/0) != 0) {
		error("Failed to open disk %s: %s\n", disk_dev, strerror(errno));
		return -1;
	}

	if (!fdisk_is_labeltype(cxt, FDISK_DISKLABEL_GPT)) {
		error("Disk %s does not have a GPT label\n", disk_dev);
		return -1;
	}

	// Read current attrs
	uint64_t attrs = 0;
	if (fdisk_gpt_get_partition_attrs(cxt, partnum, &attrs) != 0) {
		error("Failed to read GPT attributes for %s partition %u\n",
		      disk_dev, partnum);
		return -1;
	}

	uint64_t new_attrs = (attrs & ~mask) | (values & mask);

	if (new_attrs == attrs) {
		verbose("Attributes already in desired state (0x%016llx)\n",
			(unsigned long long)attrs);
	} else {
		if (fdisk_gpt_set_partition_attrs(cxt, partnum, new_attrs) != 0) {
			error("Failed to set attrs for %s partition %u\n",
			      disk_dev, partnum);
			return -1;
		}
		if (fdisk_write_disklabel(cxt) != 0) {
			error("Failed to write GPT label to %s\n", disk_dev);
			return -1;
		}
		if (fdisk_reread_partition_table(cxt) != 0) {
			// Not fatal on some kernels; udev may still rescan.
			verbose("kernel re-read request did not succeed; you may need to trigger it manually.\n");
		}
		verbose("Updated %s partition %u: 0x%016llx -> 0x%016llx\n",
			disk_dev, partnum, (unsigned long long)attrs,
			(unsigned long long)new_attrs);
	}

	return 0;
}

int get_gpt_bits(const char *disk_dev, uint32_t partnum, uint64_t *out_attrs)
{
	autofdisk struct fdisk_context *cxt = fdisk_new_context();
	if (!cxt) {
		error("fdisk_new_context failed\n");
		return -1;
	}

	if (fdisk_assign_device(cxt, disk_dev, /*readonly=*/1) != 0) {
		error("Failed to open disk %s: %s\n", disk_dev, strerror(errno));
		return -1;
	}

	if (!fdisk_is_labeltype(cxt, FDISK_DISKLABEL_GPT)) {
		error("Disk %s does not have a GPT label\n", disk_dev);
		return -1;
	}

	uint64_t attrs = 0;
	if (fdisk_gpt_get_partition_attrs(cxt, partnum, &attrs) != 0) {
		error("Failed to read GPT attributes for %s partition %u\n",
		      disk_dev, partnum);
		return -1;
	}

	*out_attrs = attrs;
	return 0;
}

/* We currently support the legacy Qualcomm GPT attribute schema, used before chromeos
 * standardized this. This is the setup that is used by autosig-u-boot and ridesx4.
 * In the future we may also want to support the chromeos layout.
 *
 * Bit48-49: priority, lowest 0x1, highest 0x3
 * Bit50: active
 * Bit51-53: remaining tries
 * Bit54: successful
 * Bit55: unbootable
 */
int read_slot_info_dev(const char *disk_dev, uint32_t partnum, BootSlotInfo *info)
{
	uint64_t attrs;
	if (get_gpt_bits(disk_dev, partnum, &attrs) != 0) {
		return -1;
	}

	info->priority = (uint8_t)((attrs >> 48) & 0x3);
	info->active = ((attrs >> 50) & 0x1) != 0;
	info->tries_remaining = (uint8_t)((attrs >> 51) & 0x7);
	info->successful = ((attrs >> 54) & 0x1) != 0;
	info->unbootable = ((attrs >> 55) & 0x1) != 0;

	return 0;
}

int write_slot_info(BootSlotInfo *info, uint32_t slot)
{
	uint64_t attrs = ((uint64_t)(info->priority & 0x3)) << 48 |
			 ((uint64_t)(info->active & 0x1)) << 50 |
			 ((uint64_t)(info->tries_remaining & 0x7)) << 51 |
			 ((uint64_t)(info->successful & 0x1)) << 54 |
			 ((uint64_t)(info->unbootable & 0x1)) << 55;
	uint64_t mask = (uint64_t)0xff << 48;

	uint32_t partnum = slot == 0 ? slot_a_part : slot_b_part;
	char *slot_dev = slot == 0 ? slot_a_dev : slot_b_dev;

	if (set_gpt_bits(slot_dev, partnum, attrs, mask) != 0) {
		return -1;
	}

	return 0;
}

int read_slot_info(BootSlotInfo info[2])
{
	if (read_slot_info_dev(slot_a_dev, slot_a_part, &info[0]) ||
	    read_slot_info_dev(slot_b_dev, slot_b_part, &info[1]))
		return -1;
	return 0;
}

static bool slot_valid(BootSlotInfo *slot)
{
	return slot->successful != 0 || slot->tries_remaining > 0;
}

static int find_active_slot(BootSlotInfo slots[2], uint32_t *out_slot)
{
	int32_t active_slot = -1;
	for (int32_t i = 0; i < 2; i++) {
		if (slots[i].active) {
			if (active_slot == -1) {
				active_slot = i;
			} else {
				if (slots[i].priority > slots[active_slot].priority) {
					active_slot = i;
				}
			}
		}
	}

	if (active_slot < 0) {
		/* No active slots, fall back to valid slots in prio order */
		for (int32_t i = 0; i < 2; i++) {
			if (slot_valid(&slots[i])) {
				if (active_slot == -1) {
					active_slot = i;
				} else {
					if (slots[i].priority >
					    slots[active_slot].priority) {
						active_slot = i;
					}
				}
			}
		}
	}

	if (active_slot < 0) {
		return -1;
	}
	*out_slot = active_slot;
	return 0;
}

static uint32_t slot_other(uint32_t slot)
{
	if (slot == 0) {
		return 1;
	}

	return 0;
}

static int get_booted_slot(void)
{
	int slot = 0;

	autofree char *cmdline = read_proc_cmdline();
	if (cmdline == NULL)
		fatal("Failed to read /proc/cmdline\n");

	autofree char *slot_suffix =
		find_proc_cmdline_key(cmdline, "androidboot.slot_suffix");

	if (slot_suffix == NULL) {
		error("No slot suffix in command line, assuming slot 0\n");
	} else {
		if (strcmp(slot_suffix, "_a") == 0)
			slot = 0;
		else if (strcmp(slot_suffix, "_b") == 0)
			slot = 1;
		else
			fatal("androidboot.slot_suffix invalid: %s\n", slot_suffix);
	}

	return slot;
}

static int cmd_get_booted(int argc, arg_unused char **argv)
{
	if (argc != 1) {
		error("Unexpected arguments to get-booted\n");
		return 1;
	}

	int slot = get_booted_slot();
	printf("%d\n", slot);

	return 0;
}

static int parse_slot(const char *slot_s)
{
	char *slot_s_end;
	uint32_t slot = strtol(slot_s, &slot_s_end, 10);
	if (*slot_s == 0 || *slot_s_end != 0) {
		return -1;
	}
	if (slot != 0 && slot != 1)
		return -1;

	return slot;
}

static int cmd_prepare_switch(int argc, arg_unused char **argv)
{
	if (argc < 2) {
		error("No slot specified\n");
		return 1;
	}

	if (argc > 2) {
		error("Unexpected arguments to prepare-switch\n");
		return 1;
	}

	char *slot_s = argv[1];
	int32_t new_slot = parse_slot(slot_s);
	if (new_slot < 0) {
		error("Invalid slot %s\n", slot_s);
		return 1;
	}

	BootSlotInfo slots[2];
	if (read_slot_info(slots) != 0) {
		return 1;
	}

	uint32_t old_slot = slot_other(new_slot);

	// Ensure old slot is bootable
	slots[old_slot].active = 1;
	slots[old_slot].successful = 1;
	// TODO: The u-boot code always decreases the, so we need to re-set it.
	// but this doesn't seem correct.
	slots[old_slot].tries_remaining = MAX_TRIES;

	if (write_slot_info(&slots[old_slot], old_slot) != 0) {
		return 1;
	}

	// Mark new slot non-bootable
	slots[new_slot].priority = 0;
	slots[new_slot].active = 0;
	slots[new_slot].successful = 0;
	slots[new_slot].tries_remaining = 0;

	if (write_slot_info(&slots[new_slot], new_slot) != 0) {
		return 1;
	}

	return 0;
}

static int init_ufs_bsg(void)
{
	autofree char *ufs_bsg_dev_b = NULL;

	ufs_bsg_dev = resolve_ufs_bsg("/dev/disk/by-partlabel/xbl_a");
	ufs_bsg_dev_b = resolve_ufs_bsg("/dev/disk/by-partlabel/xbl_b");
	if (!ufs_bsg_dev ^ !ufs_bsg_dev_b) {
		error("A UFS BSG device exists only for one of the two slots.\n");
		return -1;
	}
	if (!ufs_bsg_dev && !ufs_bsg_dev_b) {
		verbose("No UFS BSG device available. Boot LUN id will not be updated.\n");
		return 0;
	}
	if (strcmp(ufs_bsg_dev, ufs_bsg_dev_b)) {
		error("Unsupported configuration: 2 UFS host controllers.\n");
		return -1;
	}
	return 0;
}

static int read_boot_lun_ids(unsigned long ids[2])
{
	if (resolve_boot_lun_id("/dev/disk/by-partlabel/xbl_a", &ids[0]) ||
	    resolve_boot_lun_id("/dev/disk/by-partlabel/xbl_b", &ids[1]))
		return -1;
	return 0;
}

/*
 * TODO: There is no rollback in case of mid-flight failure.
 */
static int cmd_finalize_switch(int argc, arg_unused char **argv)
{
	if (argc < 2) {
		error("No slot specified\n");
		return 1;
	}

	if (argc > 2) {
		error("Unexpected arguments to finalize-switch\n");
		return 1;
	}

	char *slot_s = argv[1];
	int32_t new_slot = parse_slot(slot_s);
	if (new_slot < 0) {
		error("Invalid slot %s\n", slot_s);
		return 1;
	}

	BootSlotInfo slots[2];
	if (read_slot_info(slots) != 0) {
		return 1;
	}

	uint32_t old_slot = slot_other(new_slot);

	// Mark new slot as active for 7 tries
	slots[new_slot].active = 1;
	slots[new_slot].successful = 0;
	slots[new_slot].tries_remaining = MAX_TRIES;
	slots[new_slot].priority = MAX_PRIO;

	if (write_slot_info(&slots[new_slot], new_slot) != 0) {
		return 1;
	}

	// Ensure old slot lower prio
	slots[old_slot].active = 0;
	slots[old_slot].priority = MAX_PRIO - 1;

	if (write_slot_info(&slots[old_slot], old_slot) != 0) {
		return 1;
	}

	if (init_ufs_bsg())
		return 1;

	if (ufs_bsg_dev) {
		unsigned long boot_lun_ids[2];
		if (read_boot_lun_ids(boot_lun_ids))
			return 1;
		if (ufs_bsg_write_bootlun(ufs_bsg_dev, boot_lun_ids[new_slot]))
			return 1;
	}

	return 0;
}

static int cmd_mark_successful(arg_unused int argc, arg_unused char **argv)
{
	int32_t slot = 0;

	if (argc < 2) {
		/* No slot specified, use the booted */
		slot = get_booted_slot();
	} else if (argc == 2) {
		char *slot_s = argv[1];
		slot = parse_slot(slot_s);
		if (slot < 0) {
			error("Invalid slot %s\n", slot_s);
			return 1;
		}
	} else {
		error("Unexpected arguments to mark-successful\n");
		return 1;
	}

	BootSlotInfo info;
	uint32_t partnum = slot == 0 ? slot_a_part : slot_b_part;
	char *slot_dev = slot == 0 ? slot_a_dev : slot_b_dev;
	if (read_slot_info_dev(slot_dev, partnum, &info) != 0) {
		return 1;
	}

	info.successful = 1;
	// TODO: The u-boot code always decreases the, so we need to re-set it.
	// but this doesn't seem correct.
	info.tries_remaining = MAX_TRIES;

	if (write_slot_info(&info, slot) != 0) {
		return 1;
	}

	return 0;
}

static int cmd_get_active(int argc, arg_unused char **argv)
{
	if (argc != 1) {
		error("Unexpected arguments to get-active\n");
		return 1;
	}

	BootSlotInfo slots[2];
	if (read_slot_info(slots) != 0) {
		return 1;
	}

	uint32_t slot;
	if (find_active_slot(slots, &slot) != 0) {
		error("No active slot\n");
		return 1;
	}

	printf("%d\n", slot);

	return 0;
}

static int cmd_dump(int argc, arg_unused char **argv)
{
	if (argc != 1) {
		error("Unexpected arguments to dump\n");
		return 1;
	}

	BootSlotInfo slots[2];
	if (read_slot_info(slots) != 0) {
		return 1;
	}

	for (int i = 0; i < 2; i++) {
		BootSlotInfo *slot = &slots[i];
		printf("Slot %d: prio: %d, tries_remaining: %d, successful: %d, active: %d, unbootable: %d\n",
		       i, slot->priority, slot->tries_remaining,
		       slot->successful, slot->active, slot->unbootable);
	}

	if (init_ufs_bsg())
		return 1;

	if (ufs_bsg_dev) {
		unsigned long boot_lun_id;

		if (ufs_bsg_read_bootlun(ufs_bsg_dev, &boot_lun_id))
			return 1;

		printf("UFS boot LUN ID: %#lx\n", boot_lun_id);
	}

	return 0;
}

static int init_slots(void)
{
	slot_a_dev = resolve_partition("/dev/disk/by-partlabel/boot_a", &slot_a_part);
	if (slot_a_dev == NULL) {
		error("Failed to read slot a info\n");
		return -1;
	}

	slot_b_dev = resolve_partition("/dev/disk/by-partlabel/boot_b", &slot_b_part);
	if (slot_b_dev == NULL) {
		error("Failed to read slot b info\n");
		return -1;
	}
	return 0;
}

typedef struct {
	char *name;
	int (*fn)(int argc, char **argv);
} CmdHandler;

CmdHandler commands[] = {
	{ "dump", cmd_dump },
	{ "get-active", cmd_get_active },
	{ "get-booted", cmd_get_booted },
	{ "prepare-switch", cmd_prepare_switch },
	{ "finalize-switch", cmd_finalize_switch },
	{ "mark-successful", cmd_mark_successful },
};

static CmdHandler *find_command_handler(const char *name)
{
	for (size_t i = 0; i < N_ELEMENTS(commands); i++) {
		CmdHandler *handler = &commands[i];
		if (strcmp(name, handler->name) == 0)
			return handler;
	}

	return NULL;
}

static void usage(const char *argv0)
{
	autofree char *bin = xbasename(argv0);
	fprintf(stderr,
		"Usage: %s [OPTIONS] COMMAND [...]\n"
		"Options:\n"
		"  --verbose               Show verbose output\n"
		"  --help                  Show help\n"
		"Commands:\n"
		"  dump                    Dump current boot options\n"
		"  get-active              Show current active slot (on disk)\n"
		"  get-booted              Show current booted slot\n"
		"  prepare-switch SLOT     Prepare to switch to SLOT (before update)\n"
		"  finalize-switch SLOT    Finalize switch to SLOT (after update)\n"
		"  mark-successful [SLOT]  Mark boot successful (after update & boot)\n",
		bin);
}

#define OPT_VERSION 102
#define OPT_VERBOSE 103
#define OPT_HELP 104

int main(int argc, char **argv)
{
	const char *bin = argv[0];
	const struct option longopts[] = {
		{ .name = "help", .has_arg = no_argument, .flag = NULL, .val = OPT_HELP },
		{ .name = "version", .has_arg = no_argument, .flag = NULL, .val = OPT_VERSION },
		{ .name = "verbose", .has_arg = no_argument, .flag = NULL, .val = OPT_VERBOSE },
		{},
	};
	int opt;

	while ((opt = getopt_long(argc, argv, "", longopts, NULL)) != -1) {
		switch (opt) {
		case OPT_HELP:
			usage(bin);
			exit(0);
			break;
		case OPT_VERSION:
			printf("aboot-gptctl %s\n", PACKAGE_VERSION);
			exit(0);
			break;
		case OPT_VERBOSE:
			use_verbose = true;
			break;
		case ':':
			error("option needs a value\n");
			exit(EXIT_FAILURE);
		default:
			usage(bin);
			exit(1);
		}
	}

	argv += optind;
	argc -= optind;

	if (argc < 1) {
		error("No command specified\n");
		usage(bin);
		exit(1);
	}
	const char *command = argv[0];

	if (init_slots() != 0) {
		return 1;
	}

	CmdHandler *handler = find_command_handler(command);

	if (handler == NULL) {
		error("Unknown command '%s'\n", command);
		usage(bin);
		return 1;
	}

	if (handler->fn(argc, argv)) {
		return 1;
	}

	return 0;
}
