#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <ctype.h>
#include <linux/limits.h>
#ifdef HAVE_SELINUX
#include <selinux/selinux.h>
#endif
#include "aboot-deploy.h"

const struct option longopts[] = {
	{ "help", no_argument, NULL, 'h' },
	{ "config", required_argument, NULL, 'c' },
	{ "local", no_argument, NULL, 'l' },
	{ "options", required_argument, NULL, 'o' },
	{ "root", required_argument, NULL, 'r' },
	{ "verbose", no_argument, NULL, 'v' },
	{},
};

bool use_verbose = false;

void print_usage(void)
{
	fprintf(stderr,
		"Usage: aboot-deploy [OPTIONS] aboot_img\n"
		"Options:\n"
		"  -h, --help                       - Show this help message\n"
		"  -c, --config PATH                - Config files to read env vars from\n"
		"  -l, --local                      - Flash image to correct slot\n"
		"  -o, --options                    - cmdline of next boot, used by OSTree\n"
		"  -r, --root PATH                  - Location for the root directory\n"
		"  -v, --verbose                    - Show verbose output\n");
}

char *get_partition_path(const char *boot_type, BootPartition partition)
{
	const char *bootlabel = "boot";
	if (strcmp(boot_type, "ukiboot") == 0) {
		bootlabel = "ukiboot";
	}

	const char *suffix;
	if (partition == PARTITION_A) {
		suffix = "_a";
	} else {
		suffix = "_b";
	}
	return xasprintf("/dev/disk/by-partlabel/%s%s", bootlabel, suffix);
}

int validate_destinations(const char *destination)
{
	if (!destination) {
		error("Error: No destination provided.\n");
		return -1;
	}
	struct stat dest_stat;
	if (stat(destination, &dest_stat) != 0) {
		error("Error: Cannot access destination '%s': %s\n",
		      destination, strerror(errno));
		return -1;
	}
	return 0;
}

char *get_vbmeta_img(const char *rootdir, const char *moduledir)
{
	/* The vbmeta.img file is not exported to the /boot deploy dir,
	   so we pick it up from the module dir */
	autofree char *vbmeta_path = join_3_path(rootdir, moduledir, "vbmeta.img");

	struct stat sb;
	if (stat(vbmeta_path, &sb) == 0 && S_ISREG(sb.st_mode)) {
		return steal_ptr(&vbmeta_path);
	}

	verbose("Info: No vbmeta image found at '%s', skipping vbmeta deploy.\n",
		vbmeta_path);
	return NULL;
}

int validate_image_sizes(const char *aboot_img, const char *destination,
			 const char *vbmeta_img, const char *vbmeta_destination)
{
	off_t aboot_img_size = get_image_size(aboot_img);
	if (aboot_img_size < 0) {
		perror("Error: Could not stat aboot image");
		return -1;
	}
	off_t dest_size = get_block_dev_size(destination);
	if (dest_size < 0) {
		perror("Error: Could not get size of aboot destination");
		return -1;
	}

	if (aboot_img_size > dest_size) {
		error("Error: Android Boot Image file too large, size: %lld dest size: %lld\n",
		      (long long)aboot_img_size, (long long)dest_size);
		return -1;
	}

	if (vbmeta_img != NULL) {
		if (access(vbmeta_destination, F_OK) != 0) {
			error("Error: Cannot access vbmeta destination '%s': %s\n",
			      vbmeta_destination, strerror(errno));
			return -1;
		}

		off_t vbmeta_img_size = get_image_size(vbmeta_img);
		if (vbmeta_img_size < 0) {
			return -1;
		}
		off_t vbmeta_dest_size = get_block_dev_size(vbmeta_destination);
		if (vbmeta_dest_size < 0) {
			return -1;
		}
		if (vbmeta_img_size > vbmeta_dest_size) {
			error("vbmeta file too large, size: %lld dest size: %lld\n",
			      (long long)vbmeta_img_size,
			      (long long)vbmeta_dest_size);
			return -1;
		}
	}

	return 0;
}

static const char *get_slot_number(BootPartition partition)
{
	return (partition == PARTITION_A) ? "0" : "1";
}

BootSwitcher detect_switcher(const char *boot_type)
{
	if (strcmp(boot_type, "ukiboot") == 0) {
		return BOOT_SWITCHER_UKIBOOT;
	} else if (strcmp(boot_type, "aboot-gptctl") == 0) {
		return BOOT_SWITCHER_ABOOT_GPTCTL;
	} else if (access("/usr/bin/abctl", X_OK) == 0) {
		return BOOT_SWITCHER_ABCTL;
	} else if (access("/usr/bin/qbootctl", X_OK) == 0) {
		return BOOT_SWITCHER_QBOOTCTL;
	} else {
		error("Warning: No valid ab switching executable, proceeding in single-slot mode with slot a.\n");
		return BOOT_SWITCHER_SINGLE_SLOT;
	}
}

BootPartition get_booted_slot(BootSwitcher switcher)
{
	if (switcher == BOOT_SWITCHER_SINGLE_SLOT) {
		return PARTITION_A;
	}

	autofree char *cmdline_content = read_proc_cmdline();
	if (cmdline_content == NULL) {
		verbose("Warning: Could not read /proc/cmdline, assuming slot A\n");
		return PARTITION_A;
	}

	autofree char *slot =
		find_proc_cmdline_key(cmdline_content, "androidboot.slot_suffix");
	if (slot == NULL) {
		verbose("Warning: Kernel arg 'androidboot.slot_suffix' not found. Assuming slot A.\n");
		return PARTITION_A;
	}

	if (strcmp(slot, "_b") == 0) {
		return PARTITION_B;
	}
	return PARTITION_A;
}

int prepare_switch(BootSwitcher switcher, BootPartition partition)
{
	const char *slot_num = get_slot_number(partition);

	switch (switcher) {
	case BOOT_SWITCHER_UKIBOOT: {
		const char *ukiboot_args[] = { "ukibootctl", "prepare-switch",
					       slot_num, NULL };
		return (exec_command(ukiboot_args, -1));
	}
	case BOOT_SWITCHER_ABOOT_GPTCTL: {
		const char *gptctl_args[] = { "aboot-gptctl", "prepare-switch",
					      slot_num, NULL };
		return (exec_command(gptctl_args, -1));
	}
	case BOOT_SWITCHER_ABCTL:
		return 0;
	case BOOT_SWITCHER_QBOOTCTL:
		return 0;
	case BOOT_SWITCHER_SINGLE_SLOT:
		return 0;
	}
	return -1;
}

int finalize_switch(BootSwitcher switcher, BootPartition partition)
{
	const char *slot_num = get_slot_number(partition);

	switch (switcher) {
	case BOOT_SWITCHER_UKIBOOT: {
		const char *ukiboot_args[] = { "ukibootctl", "finalize-switch",
					       slot_num, NULL };
		return (exec_command(ukiboot_args, -1));
	}
	case BOOT_SWITCHER_ABOOT_GPTCTL: {
		const char *gptctl_args[] = { "aboot-gptctl", "finalize-switch",
					      slot_num, NULL };
		return (exec_command(gptctl_args, -1));
	}
	case BOOT_SWITCHER_ABCTL: {
		const char *abctl_args[] = { "abctl", "--set_active", slot_num, NULL };
		return (exec_command(abctl_args, -1));
	}
	case BOOT_SWITCHER_QBOOTCTL: {
		const char *qbootctl_args[] = { "qbootctl", "-s", slot_num, NULL };
		return (exec_command(qbootctl_args, -1));
	}
	case BOOT_SWITCHER_SINGLE_SLOT:
		return 0;
	}
	return -1;
}

/*
 *  Normally, ostree finds the deploy directory that matches the
 *  current boot from the ostree=/path/to/deploy kernel commandline,
 *  which is read from the ostree BLS files in
 *  /boot/loader/entries. However, when using using aboot, these BLS
 *  files are not used, and we only get ostree=true on the real kernel
 *  commandline. In this case, ostree relies on the symlinks /ostree/root.[ab]
 *  to find the deploy directory that matches the currently booted slot.
 *  This function is what creates these symlinks.
 *
 * During deploy, ostree calls aboot-deploy with the full kernel
 * commandline, including the real path to the deploy, which is where
 * we get the info to update the symlink.
 *
 * These commandline options would look something like:
 *   ostree=/ostree/boot.1/default/a1ac6097df4f00852efd43e8fc508eccb4b75f5a7aa9101ee5cd1a3ee5e482ce/0
 *
 * With a matching /ostree like:
 *  /ostree/
 *    deploy/default/deploy/3e5d649eaeb3aaeeae8e65bd10e9d378416d63957846c49a6102d7065114cbf3.0/
 *      .ostree.cfs
 *    root.a -> deploy/default/deploy/3e5d649eaeb3aaeeae8e65bd10e9d378416d63957846c49a6102d7065114cbf3.0
 *    boot.1 -> boot.1.1
 *    boot.1.1/
 *      default/a1ac6097df4f00852efd43e8fc508eccb4b75f5a7aa9101ee5cd1a3ee5e482ce/
 *        0 -> ../../../deploy/default/deploy/3e5d649eaeb3aaeeae8e65bd10e9d378416d63957846c49a6102d7065114cbf3.0
 */
char *resolve_ostree_deploy_dir(const char *rootdir, const char *kernel_options)
{
	if (kernel_options == NULL) {
		error("Error: No kernel options\n");
		return NULL;
	}

	autofree char *ostree_opt = find_proc_cmdline_key(kernel_options, "ostree");
	if (ostree_opt == NULL || !str_has_prefix(ostree_opt, "/ostree")) {
		error("Error: No valid ostree= deploy dir option in kernel options\n");
		return NULL;
	}
	/* Resolve the deploy symlink relative to /ostree */
	if (!str_has_prefix(ostree_opt, "/ostree")) {
		error("Error: ostree deploy path not  in /ostree\n");
		return NULL;
	}
	autofree char *ostree_root = join_path(rootdir, "/ostree");
	autofree char *resolved =
		resolve_symlink_in_root(ostree_root, ostree_opt + strlen("/ostree"));

	return join_path("/ostree", resolved);
}

int create_boot_link(BootPartition slot, const char *rootdir,
		     const char *ostree_deploy_dir)
{
	/* skip initial "/ostree/" */

	const char *link_target = ostree_deploy_dir + strlen("/ostree/");

	autofree char *link =
		join_path(rootdir, slot == PARTITION_A ? "/ostree/root.a" :
							 "/ostree/root.b");

	verbose("Creating new link: %s -> %s\n", link, link_target);
	if (atomic_symlink(link_target, link) != 0) {
		return -1;
	}

#if defined(__linux__) && defined(HAVE_SELINUX)
	if (is_selinux_enabled() > 0) {
		if (lsetfilecon(link, "system_u:object_r:root_t:s0") != 0) {
			perror("Error: lsetfilecon failed");
			return -1;
		}
	}
#endif

	return 0;
}

int set_active_slot(BootSwitcher switcher, BootPartition slot,
		    const char *rootdir, const char *ostree_deploy_dir)
{
	if (ostree_deploy_dir) {
		if (create_boot_link(slot, rootdir, ostree_deploy_dir) != 0) {
			return -1;
		}
	}
	return finalize_switch(switcher, slot);
}

char *aboot_config_get_partition(AbootConfig *config, BootPartition slot)
{
	if (config == NULL) {
		return NULL;
	}

	if (slot == PARTITION_A && config->partition_a != NULL) {
		return xstrdup(config->partition_a);
	} else if (slot == PARTITION_B && config->partition_b != NULL) {
		return xstrdup(config->partition_b);
	} else {
		return get_partition_path(config->boot_type, slot);
	}
}

char *aboot_config_get_vbmeta_partition(AbootConfig *config, BootPartition slot)
{
	if (config == NULL) {
		return NULL;
	}

	if (slot == PARTITION_A) {
		return xstrdup(config->vbmeta_partition_a);
	} else if (slot == PARTITION_B) {
		return xstrdup(config->vbmeta_partition_b);
	} else {
		return NULL;
	}
}

/* Extracts the kernel version from aboot.img basename like aboot-6.12.0-154.el10iv.x86_64.img */
char *extract_kernel_version(const char *basename)
{
	size_t len = strlen(basename);

	/* Skip trailing extension, like ".img" */
	while (len > 0 && basename[len - 1] != '.') {
		len--;
	}
	if (len > 0 && basename[len - 1] == '.') {
		len--;
	}

	/* Skip prefix, like "aboot-" */
	size_t start = 0;
	while (start < len && basename[start] != '-') {
		start++;
	}
	if (start < len && basename[start] == '-') {
		start++;
	}

	return xstrndup(basename + start, len - start);
}

int main(int argc, char *argv[])
{
	autofree char *aboot_img = NULL;
	autofree char *destination = NULL;
	autofree char *vbmeta_destination = NULL;
	autofree char *options = NULL;
	autofree char *rootdir = NULL;
	char *aboot_cfg_default = DEFAULT_ABOOT_CFG_PATH;
	autofree char *aboot_cfg = NULL;
	bool local = false;
	autofreeconfig AbootConfig *cfg = aboot_config_new();

	autofree char *ostree_deploy_dir = NULL;
	autofree char *boot_path = NULL;

	int opt;
	const char *optstring = "hc:lo:r:v";
	while ((opt = getopt_long(argc, argv, optstring, longopts, NULL)) != -1) {
		switch (opt) {
		case 'h':
			print_usage();
			return EXIT_FAILURE;
		case 'c':
			aboot_cfg_default = optarg;
			break;
		case 'l':
			local = true;
			break;
		case 'o':
			options = xstrdup(optarg);
			break;
		case 'r':
			rootdir = xstrdup(optarg);
			break;
		case 'v':
			use_verbose = true;
			break;
		default:
			print_usage();
			return EXIT_FAILURE;
		}
	}

	argv += optind;
	argc -= optind;
	if (argc < 1) {
		print_usage();
		error("Error: No image argument provided.\n");
		return EXIT_FAILURE;
	}

	const char *aboot_img_arg = argv[0];
	autofree char *aboot_img_basename = xbasename(aboot_img_arg);
	autofree char *kernel_version = extract_kernel_version(aboot_img_basename);

	if (rootdir == NULL) {
		rootdir = xstrdup("/");
	}

	boot_path = join_path(rootdir, "/boot/");
	aboot_img = join_path(boot_path, aboot_img_arg);
	aboot_cfg = join_path(rootdir, aboot_cfg_default);

	if (aboot_config_load(cfg, aboot_cfg) != 0) {
		error("Error: Failed to load config file: %s\n", aboot_cfg);
		return EXIT_FAILURE;
	}

	if (!local) {
		ostree_deploy_dir = resolve_ostree_deploy_dir(rootdir, options);
		if (ostree_deploy_dir == NULL) {
			return EXIT_FAILURE;
		}

		verbose("In ostree mode, deploy dir at '%s'\n", ostree_deploy_dir);

		autofree char *root_a_path = join_path(rootdir, "/ostree/root.a");
		autofree char *root_b_path = join_path(rootdir, "/ostree/root.b");
		bool has_root_link = access(root_a_path, F_OK) == 0 ||
				     access(root_b_path, F_OK) == 0;

		if (!has_root_link) {
			/* If there are no root links we assume that this is
			 * the initial run during ostree deploy at image build time,
			 * and we do nothing but create the link. aboot.img will
			 * will be manually written to the partition by the image builder.
			*/
			if (create_boot_link(PARTITION_A, rootdir,
					     ostree_deploy_dir) != 0) {
				error("Error: Failed to create boot.\n");
				return EXIT_FAILURE;
			}
			verbose("No boot links, assuming image build time: Exiting without flashing.\n");
			return EXIT_SUCCESS;
		}
	}
	autofree char *moduledir =
		join_3_path(ostree_deploy_dir, "usr/lib/modules", kernel_version);

	BootSwitcher switcher = detect_switcher(cfg->boot_type);
	BootPartition current_slot = get_booted_slot(switcher);

	BootPartition target_slot;
	if (switcher == BOOT_SWITCHER_SINGLE_SLOT) {
		target_slot = PARTITION_A;
	} else {
		if (current_slot == PARTITION_A) {
			target_slot = PARTITION_B;
		} else {
			target_slot = PARTITION_A;
		}
	}

	if (destination == NULL) {
		destination = aboot_config_get_partition(cfg, target_slot);
		if (destination == NULL) {
			error("Error: Failed to get or allocate destination path for slot %d.\n",
			      target_slot);
			return EXIT_FAILURE;
		}
	}

	if (validate_destinations(destination) != 0) {
		return EXIT_FAILURE;
	}

	if (vbmeta_destination == NULL) {
		vbmeta_destination =
			aboot_config_get_vbmeta_partition(cfg, target_slot);
		if (vbmeta_destination == NULL) {
			error("Error: Failed to get or allocate vbmeta destination path for slot %d.\n",
			      target_slot);
			return EXIT_FAILURE;
		}
	}

	autofree char *vbmeta_img = get_vbmeta_img(rootdir, moduledir);
	if (validate_image_sizes(aboot_img, destination, vbmeta_img,
				 vbmeta_destination) != 0) {
		return EXIT_FAILURE;
	}

	if (prepare_switch(switcher, target_slot) != 0) {
		return EXIT_FAILURE;
	}

	if (write_file_to_disk(aboot_img, destination) != 0) {
		return EXIT_FAILURE;
	}

	if (vbmeta_img != NULL) {
		if (write_file_to_disk(vbmeta_img, vbmeta_destination) != 0) {
			return EXIT_FAILURE;
		}
	}

	if (set_active_slot(switcher, target_slot, rootdir, ostree_deploy_dir) != 0) {
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}
